/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Yseasstat  yseasrange      Multi-year seasonal range
      Yseasstat  yseasmin        Multi-year seasonal minimum
      Yseasstat  yseasmax        Multi-year seasonal maximum
      Yseasstat  yseassum        Multi-year seasonal sum
      Yseasstat  yseasmean       Multi-year seasonal mean
      Yseasstat  yseasavg        Multi-year seasonal average
      Yseasstat  yseasvar        Multi-year seasonal variance
      Yseasstat  yseasvar1       Multi-year seasonal variance [Normalize by (n-1)]
      Yseasstat  yseasstd        Multi-year seasonal standard deviation
      Yseasstat  yseasstd1       Multi-year seasonal standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>

#include "cdo_int.h"
#include "pstream_int.h"
#include "cdo_season.h"

#define NSEAS 4

struct date_time_t
{
  int64_t vdate;
  int vtime;
};

void
set_date(int64_t vdate_new, int vtime_new, date_time_t *datetime)
{
  int year, month, day;
  cdiDecodeDate(vdate_new, &year, &month, &day);
  if (month == 12) vdate_new = cdiEncodeDate(year - 1, month, day);

  if (vdate_new > datetime->vdate)
    {
      datetime->vdate = vdate_new;
      datetime->vtime = vtime_new;
    }
}

void *
Yseasstat(void *process)
{
  int year, month, day;
  int nrecs;
  int seas_nsets[NSEAS];
  size_t nmiss;
  date_time_t datetime[NSEAS];
  FieldVector2D vars1[NSEAS], vars2[NSEAS], samp1[NSEAS];

  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("yseasrange", func_range, 0, nullptr);
  cdoOperatorAdd("yseasmin",   func_min,   0, nullptr);
  cdoOperatorAdd("yseasmax",   func_max,   0, nullptr);
  cdoOperatorAdd("yseassum",   func_sum,   0, nullptr);
  cdoOperatorAdd("yseasmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("yseasavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("yseasvar",   func_var,   0, nullptr);
  cdoOperatorAdd("yseasvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("yseasstd",   func_std,   0, nullptr);
  cdoOperatorAdd("yseasstd1",  func_std1,  0, nullptr);

  const int operatorID = cdoOperatorID();
  const int operfunc = cdoOperatorF1(operatorID);

  const bool lrange  = operfunc == func_range;
  const bool lmean   = operfunc == func_mean || operfunc == func_avg;
  const bool lstd    = operfunc == func_std || operfunc == func_std1;
  const bool lvarstd = operfunc == func_std || operfunc == func_var || operfunc == func_std1 || operfunc == func_var1;
  const int  divisor = operfunc == func_std1 || operfunc == func_var1;
  const bool lvars2  = lvarstd || lrange;
  // clang-format on

  for (int seas = 0; seas < NSEAS; seas++)
    {
      seas_nsets[seas] = 0;
      datetime[seas].vdate = 0;
      datetime[seas].vtime = 0;
    }

  const int streamID1 = cdoStreamOpenRead(cdoStreamName(0));

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID2)) taxisDeleteBounds(taxisID2);
  vlistDefTaxis(vlistID2, taxisID2);

  const int streamID2 = cdoStreamOpenWrite(cdoStreamName(1));
  pstreamDefVlist(streamID2, vlistID2);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recinfo(maxrecs);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  fieldInit(field);
  field.ptr = (double *) Malloc(gridsizemax * sizeof(double));

  int tsID = 0;
  int otsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const int64_t vdate = taxisInqVdate(taxisID1);
      const int vtime = taxisInqVtime(taxisID1);
      cdiDecodeDate(vdate, &year, &month, &day);

      const int seas = month_to_season(month);

      set_date(vdate, vtime, &datetime[seas]);

      if (!vars1[seas].size())
        {
          fieldsFromVlist(vlistID1, samp1[seas], FIELD_NONE);
          fieldsFromVlist(vlistID1, vars1[seas], FIELD_PTR);
          if (lvars2) fieldsFromVlist(vlistID1, vars2[seas], FIELD_PTR);
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          pstreamInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recinfo[recID].varID = varID;
              recinfo[recID].levelID = levelID;
              recinfo[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          Field &rsamp1 = samp1[seas][varID][levelID];
          Field &rvars1 = vars1[seas][varID][levelID];
          const int nsets = seas_nsets[seas];

          size_t gridsize = rvars1.size;

          if (nsets == 0)
            {
              pstreamReadRecord(streamID1, rvars1.ptr, &nmiss);
              rvars1.nmiss = nmiss;
              if (lrange)
                {
                  vars2[seas][varID][levelID].nmiss = rvars1.nmiss;
                  for (size_t i = 0; i < gridsize; i++) vars2[seas][varID][levelID].ptr[i] = rvars1.ptr[i];
                }

              if (nmiss > 0 || rsamp1.ptr)
                {
                  if (rsamp1.ptr == nullptr) rsamp1.ptr = (double *) Malloc(gridsize * sizeof(double));

                  for (size_t i = 0; i < gridsize; i++) rsamp1.ptr[i] = !DBL_IS_EQUAL(rvars1.ptr[i], rvars1.missval);
                }
            }
          else
            {
              pstreamReadRecord(streamID1, field.ptr, &nmiss);
              field.nmiss = nmiss;
              field.grid = rvars1.grid;
              field.missval = rvars1.missval;

              if (field.nmiss > 0 || rsamp1.ptr)
                {
                  if (rsamp1.ptr == nullptr)
                    {
                      rsamp1.ptr = (double *) Malloc(gridsize * sizeof(double));
                      for (size_t i = 0; i < gridsize; i++) rsamp1.ptr[i] = nsets;
                    }

                  for (size_t i = 0; i < gridsize; i++)
                    if (!DBL_IS_EQUAL(field.ptr[i], rvars1.missval)) rsamp1.ptr[i]++;
                }

              if (lvarstd)
                {
                  farsumq(vars2[seas][varID][levelID], field);
                  farsum(rvars1, field);
                }
              else if (lrange)
                {
                  farmin(vars2[seas][varID][levelID], field);
                  farmax(rvars1, field);
                }
              else
                {
                  farfun(rvars1, field, operfunc);
                }
            }
        }

      if (seas_nsets[seas] == 0 && lvarstd)
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recinfo[recID].lconst) continue;

            const int varID = recinfo[recID].varID;
            const int levelID = recinfo[recID].levelID;

            farmoq(vars2[seas][varID][levelID], vars1[seas][varID][levelID]);
          }

      seas_nsets[seas]++;
      tsID++;
    }

  for (int seas = 0; seas < NSEAS; seas++)
    if (seas_nsets[seas])
      {
        const int nsets = seas_nsets[seas];
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recinfo[recID].lconst) continue;

            const int varID = recinfo[recID].varID;
            const int levelID = recinfo[recID].levelID;
            Field &rsamp1 = samp1[seas][varID][levelID];
            Field &rvars1 = vars1[seas][varID][levelID];

            if (lmean)
              {
                if (rsamp1.ptr)
                  fardiv(rvars1, rsamp1);
                else
                  farcdiv(rvars1, (double) nsets);
              }
            else if (lvarstd)
              {
                Field &rvars2 = vars2[seas][varID][levelID];
                if (rsamp1.ptr)
                  {
                    if (lstd)
                      farstd(rvars1, rvars2, rsamp1, divisor);
                    else
                      farvar(rvars1, rvars2, rsamp1, divisor);
                  }
                else
                  {
                    if (lstd)
                      farcstd(rvars1, rvars2, nsets, divisor);
                    else
                      farcvar(rvars1, rvars2, nsets, divisor);
                  }
              }
            else if (lrange)
              {
                Field &rvars2 = vars2[seas][varID][levelID];
                farsub(rvars1, rvars2);
              }
          }

        taxisDefVdate(taxisID2, datetime[seas].vdate);
        taxisDefVtime(taxisID2, datetime[seas].vtime);
        pstreamDefTimestep(streamID2, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recinfo[recID].lconst) continue;

            const int varID = recinfo[recID].varID;
            const int levelID = recinfo[recID].levelID;
            Field &rvars1 = vars1[seas][varID][levelID];

            pstreamDefRecord(streamID2, varID, levelID);
            pstreamWriteRecord(streamID2, rvars1.ptr, rvars1.nmiss);
          }

        otsID++;
      }

  for (int seas = 0; seas < NSEAS; seas++)
    {
      if (vars1[seas].size())
        {
          fieldsFree(vlistID1, samp1[seas]);
          fieldsFree(vlistID1, vars1[seas]);
          if (lvars2) fieldsFree(vlistID1, vars2[seas]);
        }
    }

  if (field.ptr) Free(field.ptr);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
