/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
using Gtk;

namespace Clinica {

	public class UserInterface : GLib.Object {
	
	    /* Error signal and associated callback passed by main () */
	    public signal void error (string message);
	    
	    /* true if the graphic has been started */
	    public bool started { get; private set; default = false; }
	
	    /* Various objects needed for coordination */
		internal Builder builder;	
		internal Window window;

        /* Gtk Objects */
		private Alignment main;
		
		/* Pages to be loaded */
		private StartPage start_page;
		private PatientListPage patient_list_page;
		private DoctorListPage  doctor_list_page;
		private MedicineSearchPage medicine_search_page;
		
		/* Windows */
		internal SettingsManager? settings_manager = null;
		internal CalendarWindow?  calendar_window  = null;
		internal GLib.HashTable<int, VisitWindow> visit_windows;
		
		private extern Peas.ExtensionSet setup_extension_set (ResourceManager resources, Peas.Engine engine);
		internal Peas.ExtensionSet extension_set;
		
		/* ResourceManager */
		public ResourceManager resource_manager;
			
		public UserInterface (ResourceManager rm) {
		    resource_manager = rm;
		    resource_manager.user_interface = this;
			
		    /* Keep resource_manager and connect callback signal */
		    error.connect ((t,l) => resource_manager.error_callback(t,l));
		    	    
		    /* Start building the interface */
			builder = new Builder ();
			
			try {
			    /* Load window-related objects and the button for the main
			     * window shown at startup */
    			builder.add_from_file (resource_manager.get_ui_file ("window.glade"));			
    			
    			/* Connect callback for destroy events */
	    		window = builder.get_object ("window") as Gtk.Window;

                /* Load the main alignement where all the pages will be put */
	            main = builder.get_object ("main") as Alignment;
	        } catch (Error e) {
	            error ("Error opening UI files, check your installation.");
	        }
	        
	        /* Create an empty HashTable to store the Visit windows */
	        visit_windows = new HashTable<int, VisitWindow> (null, null);
	        
	        /* Load pages and connect callbacks with errors */
            start_page = new StartPage (resource_manager);
	        
	        /* Show starting page */
	        show_start_page ();
	        
	        /* Load plugins */
	        load_plugins.begin ();
	        
	        /* Load other pages and then connect signals so the user can
	         * visit them */
	        load_pages.begin ();
		}
		
		internal async void load_pages () {
            patient_list_page = new PatientListPage (resource_manager);
            doctor_list_page = new DoctorListPage (resource_manager);
            medicine_search_page = new MedicineSearchPage (resource_manager);
	        builder.connect_signals (this);
		}
		
		internal async void load_plugins () {
		    extension_set = setup_extension_set (resource_manager,
		                                         resource_manager.plugin_engine);
		}
			
		public void error_callback (Page page, string message) {
			resource_manager.error_callback (page, message);
		}
		
		/**
		 * @brief Set active page in the main window.
		 * @param page the page to be set as active.
		 */
		private void set_page (Page page) {
		    if (main.get_child () != null) {
		        if (main.get_child () == (page as Widget))
		            return;
		        else {
		        	main.get_child ().hide ();	
		            main.remove (main.get_child ());
		        }
		    } 
	        main.add (page);
            page.show ();
            page.setup ();
		}
		
		/**
		 * @brief Show the button "Create new Patient"
		 * and nothing more on the main window. 
		 */
		public void show_start_page (string patient_name = "") {
		    set_page (start_page);
		    start_page.set_name (patient_name);
		}
		
		/**
		 * @brief Show patient list in the main page.
		 */
		public void show_patient_list_page () {
		    set_page (patient_list_page);
		}
		
		/**
		 * @brief Show doctor list in the main page.
		 */
		public void show_doctor_list_page (GLib.Object? source = null) {
		    set_page (doctor_list_page);
		}
		
		/**
		 * @brief Show the search medicines page.
		 */
		[CCode (instance_pos = -1)]
		public void show_medicines_search_page (GLib.Object? source = null) {
		    set_page (medicine_search_page);
		}
		
		/**
		 * @brief Show the calendar window.
		 */
		[CCode (instance_pos = -1)]
		public void show_calendar_window (GLib.Object? source = null) {
		    if (calendar_window == null) {
    		    calendar_window = new CalendarWindow (resource_manager);
    		    calendar_window.destroy.connect ((cw) => calendar_window = null);
    		}
		else {
                /* Bring the window to front if it's not there */		    
                calendar_window.present ();
            	}
            
		    calendar_window.show_all ();
		}
		
		/**
		 * @brief Show the VisitiWindow for the given page. If it
		 * is already open than bring it on top.
		 */
		public VisitWindow show_visit_window (Patient p) {
		    int p_id = p.get_id ();

            VisitWindow? win = visit_windows.lookup (p_id);
            if (win == null) {
                win = new VisitWindow (resource_manager, p);
                visit_windows.insert (p_id, win);
            }
            else {
                win.present ();
            }
            
            win.show_all ();
            return win;
		}
		
		/* CALLBACKS */
	    [CCode (instance_pos = -1)]
		public void on_window_destroy (GLib.Object w) {
			Gtk.main_quit ();
		}
		
		[CCode (instance_pos = -1)]
		public void on_browse_patient_list_requested (StartPage page) {
		    show_patient_list_page ();
		}
		
		[CCode (instance_pos = -1)]
		public void on_browse_doctor_list_requested (StartPage page) {
		    show_doctor_list_page ();
		}
		
		/* Menu item */
		
		[CCode (instance_pos = -1)]
		public void on_new_patient_menu_item_activate (MenuItem item) {
			var new_patient_dialog = new PatientEditor (resource_manager);
			new_patient_dialog.dialog.set_transient_for (window);
			new_patient_dialog.run ();
			new_patient_dialog.destroy ();
		}
		
		[CCode (instance_pos = -1)]
        	public void on_new_doctor_menu_item_activate (MenuItem item) {
        		var new_doc_dialog = new DoctorEditor (resource_manager);
        		new_doc_dialog.dialog.set_transient_for (window);
	        	new_doc_dialog.run ();
			new_doc_dialog.destroy ();
        	}
		
		[CCode (instance_pos = -1)]
		public void on_patient_menu_item_activate (MenuItem item) {
		    show_patient_list_page ();
		}
		
		[CCode (instance_pos = -1)]
		public void on_doctor_menu_item_activate (MenuItem item) {
		    show_doctor_list_page ();
		}
		
		[CCode (instance_pos = -1)]
		public void on_home_menu_item_activate (MenuItem item) {
		    show_start_page();
		}
		
		[CCode (instance_pos = -1)]
		public void on_search_medicines_menu_item_activate (MenuItem item) {
		    set_page (medicine_search_page);
        }
		
		[CCode (instance_pos = -1)]
		public void on_about_menu_item_activate (GLib.Object? source) {
		    var about_dialog = new AboutDialog (resource_manager);
		    about_dialog.set_transient_for (window);
		    about_dialog.run (); 
		    about_dialog.destroy ();
		}
		
		[CCode (instance_pos = -1)]
		public void on_settings_menu_item_activate (MenuItem item) {
		    if (settings_manager == null){
    		    settings_manager = new SettingsManager (resource_manager);
    		    /* Dereference the window on release to make it be freed and
		     * reconstructed when necessary */
		    settings_manager.destroy.connect ((sm) => settings_manager = null);
		}
    		    else {
                	/* Bring the window to front if it's not there */		    
                	settings_manager.present ();
                	}
		    settings_manager.show_all ();
		}
		
		[CCode (instance_pos = -1)]
		public void on_help_menu_item_activate (MenuItem item) {
		    try {
    		    Gtk.show_uri (this.window.get_screen (), "ghelp:clinica", Gdk.CURRENT_TIME);
    		} catch (GLib.Error e) {
    		    error (_("Cannot open the help: %s").printf (e));
    		}
		}
		
		/**
		 * @brief start application
		 */
		public void start () {
			window.show_all ();
			started = true;
			Gtk.main ();
		}
	}
}
