/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
 using Gee;
 using Sqlite;
 
 namespace Clinica {
 
    public class VisitIterator : SqlDataIterator {
    
        
		private ResourceManager resource_manager;

        public VisitIterator (ResourceManager resources) {
            base (resources.db, resources.visits_table, "date", true);
            resource_manager = resources;
        }
        
        public VisitIterator.with_day (ResourceManager resources, DateTime day, bool descending = true) {
            DateTime day_after = day.add_days (1).add_minutes (-1);
            base.with_like (resources.db, resources.visits_table, "date", descending, 
                "date BETWEEN '%s' AND '%s'".printf (SqlDataType.datetime_to_string (day),
                                                     SqlDataType.datetime_to_string (day_after)));
            resource_manager = resources;
        }
        
        public VisitIterator.with_patient (ResourceManager resources, Patient p, bool descending = true) {
            base.with_like (resources.db, resources.visits_table, "date", descending,
                "patient = %d".printf (p.get_id ()));
            resource_manager = resources;
        }
        
        public new VisitIterator iterator () { return this; }
        
        public new Visit get () { 
            return new Visit.with_id (resource_manager, base.get ()); 
        }
    }
 
    public class Visit : SqlDataType {
        
        /* FIELDS */
        public string anamnesis {
        	get { return get_text ("anamnesis"); }
        	set { set_text ("anamnesis", value); }
        }
        
        public string physical_examination {
        	get { return get_text ("physical_examination"); }
        	set { set_text ("physical_examination", value); }
        }
        
        public string laboratory_exam {
        	get { return get_text ("laboratory_exam"); }
        	set { set_text ("laboratory_exam", value); }
        }
        
        public string histopathology {
        	get { return get_text ("histopathology"); }
        	set { set_text ("histopathology", value); }
        }
        
        public string diagnosis {
        	get { return get_text ("diagnosis"); }
        	set { set_text ("diagnosis", value); }
        }
        
        public string topical_therapy {
        	get { return get_text ("topical_therapy"); }
        	set { set_text ("topical_therapy", value); }
        }
        
        public string systemic_therapy {
        	get { return get_text ("systemic_therapy"); }
        	set { set_text ("systemic_therapy", value); }
        }
        
        public string subsequent_checks {
        	get { return get_text ("subsequent_checks"); }
        	set { set_text ("subsequent_checks", value); }
        }
        
        private DateTime _date;
        public DateTime date {
        	get { 
        		_date = get_date ("date");
        		return _date;
        	}
        	set {
        		_date = value;
        		set_date ("date", _date);
        	}
        }
        
        private Patient _patient;
        public  Patient patient {
        	get {
        		_patient = new Patient.with_id (resource_manager, get_integer ("patient"));
        		return _patient;
        	}
        	set {
        		this._patient = value;
        		set_integer("patient", value.get_id ());
        	}
        }
        
		private ResourceManager resource_manager;
        
        /**
         * @brief Create a new visit to 
         * the passed patient.
         * @param patient The patient that has been
         * visited
         */
        public Visit (ResourceManager resources, Patient? patient) {
            base (resources.db);
            resource_manager = resources;
            table_name = resources.visits_table;
            error.connect ((t,l) => resources.error_callback(t,l));
            table_name = "visits";
            
            add_text_field ("anamnesis");
            add_text_field ("physical_examination");
            add_text_field ("laboratory_exam");
            add_text_field ("histopathology");
            add_text_field ("diagnosis");
            add_text_field ("topical_therapy");
            add_text_field ("systemic_therapy");
            add_text_field ("subsequent_checks");
            add_date_field ("date");
            
            add_integer_field ("patient");
            
            if (patient != null) {
                this.patient = patient;
            }
            
            init_resources ();
        }
        
        public Visit.with_id (ResourceManager resources, int ID) {
            this (resources, null);
            load (ID);
            this.patient = new Patient.with_id (resources, get_integer ("patient"));
        }
        
        public static new VisitIterator all (ResourceManager resources) {
            return new VisitIterator (resources);
        }
        
        public static new VisitIterator for_day (ResourceManager resources, DateTime date, bool descending = false) {
            return new VisitIterator.with_day (resources, date, descending);
        }
       
    }
 
 
 }
