/*
 * Copyright (C) 2003-2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "misc_pci_control.h"

#define COMP_(x) misc_pci_control_ ## x

struct cpssp {
	struct sig_std_logic *port_reg_out_xx0_0;
	struct sig_std_logic *port_reg_out_xx0_1;
	struct sig_std_logic *port_reg_out_xx0_2;
	struct sig_std_logic *port_reg_out_xx0_3;
	struct sig_std_logic *port_reg_out_xx0_4;
	struct sig_std_logic *port_reg_out_xx0_5;
	struct sig_std_logic *port_reg_out_xx0_6;
	struct sig_std_logic *port_reg_out_xx0_7;
	struct sig_std_logic *port_reg_out_xx1_0;
	struct sig_std_logic *port_reg_out_xx1_1;
	struct sig_std_logic *port_reg_out_xx1_2;
	struct sig_std_logic *port_reg_out_xx1_3;
	struct sig_std_logic *port_reg_out_xx1_4;
	struct sig_std_logic *port_reg_out_xx1_5;
	struct sig_std_logic *port_reg_out_xx1_6;
	struct sig_std_logic *port_reg_out_xx1_7;
	struct sig_pci_conn *port_pci;
	uint32_t state_conf_addr;
	unsigned char state_conf_status_io;
	unsigned char state_io_reg_xx0;
	unsigned char state_io_reg_xx1;
};

/* check if our io address is used.
 * @return -1 if io is disabled or out of range,
 * 	   otherwise the local address part is returned.
 */
static int
COMP_(pci_check_io_addr)(struct cpssp *cpssp, uint32_t addr)
{
	if (! cpssp->state_conf_status_io) {
		return -1;
	}

	if ((addr & (~0xF)) != cpssp->state_conf_addr) {
		return -1;
	}

	return addr & 0xF;
}

static int
COMP_(pci_ior)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int local_addr;

	local_addr = COMP_(pci_check_io_addr)(cpssp, addr);
	switch (local_addr) {
	case -1:
		return -1;

	case 0:
		*valp = cpssp->state_io_reg_xx0
		      | cpssp->state_io_reg_xx1 << 8;
		/* fall through */

	default:
		return 0;
	}

	/* not reached */
	return 0;
}

static int
COMP_(bool_2_std)(int bv)
{
	return bv ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0;
}

static void
COMP_(update_reg_out)(struct cpssp *cpssp)
{
	sig_std_logic_set(cpssp->port_reg_out_xx0_0, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx0 & 1 << 0));
	sig_std_logic_set(cpssp->port_reg_out_xx0_1, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx0 & 1 << 1));
	sig_std_logic_set(cpssp->port_reg_out_xx0_2, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx0 & 1 << 2));
	sig_std_logic_set(cpssp->port_reg_out_xx0_3, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx0 & 1 << 3));
	sig_std_logic_set(cpssp->port_reg_out_xx0_4, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx0 & 1 << 4));
	sig_std_logic_set(cpssp->port_reg_out_xx0_5, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx0 & 1 << 5));
	sig_std_logic_set(cpssp->port_reg_out_xx0_6, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx0 & 1 << 6));
	sig_std_logic_set(cpssp->port_reg_out_xx0_7, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx0 & 1 << 7));

	sig_std_logic_set(cpssp->port_reg_out_xx1_0, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx1 & 1 << 0));
	sig_std_logic_set(cpssp->port_reg_out_xx1_1, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx1 & 1 << 1));
	sig_std_logic_set(cpssp->port_reg_out_xx1_2, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx1 & 1 << 2));
	sig_std_logic_set(cpssp->port_reg_out_xx1_3, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx1 & 1 << 3));
	sig_std_logic_set(cpssp->port_reg_out_xx1_4, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx1 & 1 << 4));
	sig_std_logic_set(cpssp->port_reg_out_xx1_5, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx1 & 1 << 5));
	sig_std_logic_set(cpssp->port_reg_out_xx1_6, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx1 & 1 << 6));
	sig_std_logic_set(cpssp->port_reg_out_xx1_7, cpssp,
		COMP_(bool_2_std)(cpssp->state_io_reg_xx1 & 1 << 7));
}

static int
COMP_(pci_iow)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int local_addr;

	local_addr = COMP_(pci_check_io_addr)(cpssp, addr);
	switch (local_addr) {
	case -1:
		return -1;

	case 0:
		if ((bs & 1) == 1) {
			cpssp->state_io_reg_xx0 = val & 0xff;
		}
		if ((bs & 2) == 2) {
			cpssp->state_io_reg_xx1 = (val >> 8) & 0xff;

		}

		COMP_(update_reg_out)(cpssp);
		/* fall through */

	default:
		return 0;
	}

	/* not reached */
	return 0;
}

static int
COMP_(pci_c0r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;

	if ((addr & 3) != 0) {
		return -1;
	}

	switch (addr & 0xff) {
	case 0x0:
		*valp = 0xdeadbeaf;
		break;

	case 0x4:
		*valp = cpssp->state_conf_status_io || 1 << 25;
		break;

	case 0x8:
		*valp = 1 << 16 | 1 << 26;
		break;

	case 0x0c:
		*valp = 0;
		break;

	case 0x10:
		*valp = cpssp->state_conf_addr | 1 << 0;
		break;

	default:
		*valp = 0;
		break;
	}

	return 0;
}

static int
COMP_(pci_c0w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;

	if ((addr & 3) != 0) {
		return -1;
	}

	switch (addr & 0xff) {
	case 0x04:
		if (bs & 1) {
			cpssp->state_conf_status_io = val & 1;
		}
		break;

	case 0x10:
		/* bits 0-3 are hardwired */
		val &= ~0xf;

		if (bs & 1) {
			cpssp->state_conf_addr &= ~0xff;
			cpssp->state_conf_addr |= val & 0xff;
		}

		if (bs & 2) {
			cpssp->state_conf_addr &= ~0xff00;
			cpssp->state_conf_addr |= val & 0xff00;
		}

		if (bs & 4) {
			cpssp->state_conf_addr &= ~0xff0000;
			cpssp->state_conf_addr |= val & 0xff0000;
		}

		if (bs & 8) {
			cpssp->state_conf_addr &= ~0xff000000;
			cpssp->state_conf_addr |= val & 0xff000000;
		}
		break;

	default:
		break;
	}

	return 0;
}

static void
COMP_(pci_reset_n_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	if (! val) {
		cpssp->state_conf_addr = 0;
		cpssp->state_conf_status_io = 0;
		cpssp->state_io_reg_xx0 = 0;
		cpssp->state_io_reg_xx1 = 0;
		COMP_(update_reg_out)(cpssp);
	}
}

void *
COMP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_pci_conn *port_pci,
	struct sig_std_logic *port_reg_out_xx0_0,
	struct sig_std_logic *port_reg_out_xx0_1,
	struct sig_std_logic *port_reg_out_xx0_2,
	struct sig_std_logic *port_reg_out_xx0_3,
	struct sig_std_logic *port_reg_out_xx0_4,
	struct sig_std_logic *port_reg_out_xx0_5,
	struct sig_std_logic *port_reg_out_xx0_6,
	struct sig_std_logic *port_reg_out_xx0_7,
	struct sig_std_logic *port_reg_out_xx1_0,
	struct sig_std_logic *port_reg_out_xx1_1,
	struct sig_std_logic *port_reg_out_xx1_2,
	struct sig_std_logic *port_reg_out_xx1_3,
	struct sig_std_logic *port_reg_out_xx1_4,
	struct sig_std_logic *port_reg_out_xx1_5,
	struct sig_std_logic *port_reg_out_xx1_6,
	struct sig_std_logic *port_reg_out_xx1_7
)
{
	static const struct sig_pci_bus_funcs pci_main_funcs = {
		.ior = COMP_(pci_ior),
		.iow = COMP_(pci_iow)
	};
	static const struct sig_pci_idsel_funcs pci_idsel_funcs = {
		.c0r = COMP_(pci_c0r),
		.c0w = COMP_(pci_c0w)
	};
	static const struct sig_std_logic_funcs pci_n_reset_funcs = {
		.boolean_or_set = COMP_(pci_reset_n_set)
	};

	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Call */
	/* Out */
	cpssp->port_pci = port_pci;

	cpssp->port_reg_out_xx0_0 = port_reg_out_xx0_0;
	sig_std_logic_connect_out(port_reg_out_xx0_0, cpssp, 0);

	cpssp->port_reg_out_xx0_1 = port_reg_out_xx0_1;
	sig_std_logic_connect_out(port_reg_out_xx0_1, cpssp, 0);

	cpssp->port_reg_out_xx0_2 = port_reg_out_xx0_2;
	sig_std_logic_connect_out(port_reg_out_xx0_2, cpssp, 0);

	cpssp->port_reg_out_xx0_3 = port_reg_out_xx0_3;
	sig_std_logic_connect_out(port_reg_out_xx0_3, cpssp, 0);

	cpssp->port_reg_out_xx0_4 = port_reg_out_xx0_4;
	sig_std_logic_connect_out(port_reg_out_xx0_4, cpssp, 0);

	cpssp->port_reg_out_xx0_5 = port_reg_out_xx0_5;
	sig_std_logic_connect_out(port_reg_out_xx0_5, cpssp, 0);

	cpssp->port_reg_out_xx0_6 = port_reg_out_xx0_6;
	sig_std_logic_connect_out(port_reg_out_xx0_6, cpssp, 0);

	cpssp->port_reg_out_xx0_7 = port_reg_out_xx0_7;
	sig_std_logic_connect_out(port_reg_out_xx0_7, cpssp, 0);

	cpssp->port_reg_out_xx1_0 = port_reg_out_xx1_0;
	sig_std_logic_connect_out(port_reg_out_xx1_0, cpssp, 0);

	cpssp->port_reg_out_xx1_1 = port_reg_out_xx1_1;
	sig_std_logic_connect_out(port_reg_out_xx1_1, cpssp, 0);

	cpssp->port_reg_out_xx1_2 = port_reg_out_xx1_2;
	sig_std_logic_connect_out(port_reg_out_xx1_2, cpssp, 0);

	cpssp->port_reg_out_xx1_3 = port_reg_out_xx1_3;
	sig_std_logic_connect_out(port_reg_out_xx1_3, cpssp, 0);

	cpssp->port_reg_out_xx1_4 = port_reg_out_xx1_4;
	sig_std_logic_connect_out(port_reg_out_xx1_4, cpssp, 0);

	cpssp->port_reg_out_xx1_5 = port_reg_out_xx1_5;
	sig_std_logic_connect_out(port_reg_out_xx1_5, cpssp, 0);

	cpssp->port_reg_out_xx1_6 = port_reg_out_xx1_6;
	sig_std_logic_connect_out(port_reg_out_xx1_6, cpssp, 0);

	cpssp->port_reg_out_xx1_7 = port_reg_out_xx1_7;
	sig_std_logic_connect_out(port_reg_out_xx1_7, cpssp, 0);

	/* In */
	cpssp->state_conf_addr = 0;
	cpssp->state_conf_status_io = 0;
	cpssp->state_io_reg_xx0 = 0;
	cpssp->state_io_reg_xx1 = 0;

	sig_pci_bus_connect(port_pci->main, cpssp, &pci_main_funcs);
	sig_pci_idsel_connect(port_pci->idsel, cpssp, &pci_idsel_funcs);
	sig_std_logic_connect_in(port_pci->n_reset, cpssp, &pci_n_reset_funcs);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
COMP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
