/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "xml.h"
#include "xml-schem.h"

static void
bbox_null(
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp
)
{
	*minxp = 1e100;
	*minyp = 1e100;
	*maxxp = -1e100;
	*maxyp = -1e100;
}

static void
bbox_add(
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp,
	double x,
	double y
)
{
	if (x < *minxp) *minxp = x;
	if (*maxxp < x) *maxxp = x;
	if (y < *minyp) *minyp = y;
	if (*maxyp < y) *maxyp = y;
}

static struct xml_schem_ge *
ge_get(struct xml *x1)
{
	struct xml_schem_ge *ge;

	ge = malloc(sizeof(*ge));
	assert(ge);

	if (strcmp(x1->start.string, "line") == 0) {
		ge->type = GE_LINE;
		ge->line.x0 = atof(xml_attr_lookup(x1, "x0"));
		ge->line.y0 = atof(xml_attr_lookup(x1, "y0"));
		ge->line.x1 = atof(xml_attr_lookup(x1, "x1"));
		ge->line.y1 = atof(xml_attr_lookup(x1, "y1"));
	} else if (strcmp(x1->start.string, "box") == 0) {
		ge->type = GE_BOX;
		ge->box.x0 = atof(xml_attr_lookup(x1, "x0"));
		ge->box.y0 = atof(xml_attr_lookup(x1, "y0"));
		ge->box.x1 = atof(xml_attr_lookup(x1, "x1"));
		ge->box.y1 = atof(xml_attr_lookup(x1, "y1"));
	} else if (strcmp(x1->start.string, "circle") == 0) {
		ge->type = GE_CIRCLE;
		ge->circle.x = atof(xml_attr_lookup(x1, "x"));
		ge->circle.y = atof(xml_attr_lookup(x1, "y"));
		ge->circle.r = atof(xml_attr_lookup(x1, "r"));
	} else if (strcmp(x1->start.string, "text") == 0) {
		ge->type = GE_TEXT;
		ge->text.str = strdup(xml_attr_lookup(x1, "string"));
		assert(ge->text.str);
		ge->text.x = atof(xml_attr_lookup(x1, "x"));
		ge->text.y = atof(xml_attr_lookup(x1, "y"));
		ge->text.align = atoi(xml_attr_lookup(x1, "align"));
		ge->text.angle = atoi(xml_attr_lookup(x1, "angle"));
	} else {
		fprintf(stderr, "%s\n", x1->start.string);
		assert(0);
	}

	return ge;
}

static struct xml *
ge_put(struct xml_schem_ge *ge)
{
	struct xml *x;
	char buf[1024];

	switch (ge->type) {
	case GE_LINE:
		x = xml_alloc("line");
		sprintf(buf, "%f", ge->line.x0);
		xml_attr_set(x, "x0", buf);
		sprintf(buf, "%f", ge->line.y0);
		xml_attr_set(x, "y0", buf);
		sprintf(buf, "%f", ge->line.x1);
		xml_attr_set(x, "x1", buf);
		sprintf(buf, "%f", ge->line.y1);
		xml_attr_set(x, "y1", buf);
		break;
	case GE_BOX:
		x = xml_alloc("box");
		sprintf(buf, "%f", ge->box.x0);
		xml_attr_set(x, "x0", buf);
		sprintf(buf, "%f", ge->box.y0);
		xml_attr_set(x, "y0", buf);
		sprintf(buf, "%f", ge->box.x1);
		xml_attr_set(x, "x1", buf);
		sprintf(buf, "%f", ge->box.y1);
		xml_attr_set(x, "y1", buf);
		break;
	case GE_CIRCLE:
		x = xml_alloc("circle");
		sprintf(buf, "%f", ge->circle.x);
		xml_attr_set(x, "x", buf);
		sprintf(buf, "%f", ge->circle.y);
		xml_attr_set(x, "y", buf);
		sprintf(buf, "%f", ge->circle.r);
		xml_attr_set(x, "r", buf);
		break;
	case GE_TEXT:
		x = xml_alloc("text");
		xml_attr_set(x, "string", ge->text.str);
		sprintf(buf, "%f", ge->text.x);
		xml_attr_set(x, "x", buf);
		sprintf(buf, "%f", ge->text.y);
		xml_attr_set(x, "y", buf);
		sprintf(buf, "%u", ge->text.align);
		xml_attr_set(x, "align", buf);
		sprintf(buf, "%u", ge->text.angle);
		xml_attr_set(x, "angle", buf);
		break;
	default:
		assert(0);
	}

	return x;
}

static void
ge_bbox_add(
	struct xml_schem_ge *ge,
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp
)
{
	switch (ge->type) {
	case GE_LINE:
		bbox_add(minxp, minyp, maxxp, maxyp, ge->line.x0, ge->line.y0);
		bbox_add(minxp, minyp, maxxp, maxyp, ge->line.x1, ge->line.y1);
		break;
	case GE_BOX:
		bbox_add(minxp, minyp, maxxp, maxyp, ge->box.x0, ge->box.y0);
		bbox_add(minxp, minyp, maxxp, maxyp, ge->box.x1, ge->box.y1);
		break;
	case GE_CIRCLE:
		bbox_add(minxp, minyp, maxxp, maxyp,
				ge->circle.x - ge->circle.r,
				ge->circle.y - ge->circle.r);
		bbox_add(minxp, minyp, maxxp, maxyp,
				ge->circle.x + ge->circle.r,
				ge->circle.y + ge->circle.r);
		break;
	case GE_TEXT:
		/* FIXME */
		bbox_add(minxp, minyp, maxxp, maxyp,
				ge->text.x, ge->text.y);
		break;
	}
}

static void
ge_translate(struct xml_schem_ge *ge, double dx, double dy)
{
	switch (ge->type) {
	case GE_LINE:
		ge->line.x0 += dx;
		ge->line.y0 += dy;
		ge->line.x1 += dx;
		ge->line.y1 += dy;
		break;
	case GE_BOX:
		ge->box.x0 += dx;
		ge->box.y0 += dy;
		ge->box.x1 += dx;
		ge->box.y1 += dy;
		break;
	case GE_CIRCLE:
		ge->circle.x += dx;
		ge->circle.y += dy;
		break;
	case GE_TEXT:
		ge->text.x += dx;
		ge->text.y += dy;
		break;
	}
}

static struct xml_schem_generic *
generic_get(struct xml *x0)
{
	struct xml *x1;
	struct xml_schem_generic *gen;
	struct xml_schem_ge *ge;

	gen = malloc(sizeof(*gen));
	assert(gen);
	memset(gen, 0, sizeof(*gen));

	gen->seq = xml_attr_lookup(x0, "seq");
	gen->name = xml_attr_lookup(x0, "id");
	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		 && x1->type != XML_START_END) continue;

		if (strcmp(x1->start.string, "type") == 0) {
			gen->type = xml_attr_lookup(x1, "id");
		} else if (strcmp(x1->start.string, "value") == 0) {
			gen->value = x1->start.child_first->text.string;
		} else {
			ge = ge_get(x1);

			ge->prev = gen->ge_last;
			ge->next = NULL;
			if (ge->prev) {
				ge->prev->next = ge;
			} else {
				gen->ge_first = ge;
			}
			gen->ge_last = ge;
		}
	}

	return gen;
}

static struct xml *
generic_put(struct xml_schem_generic *gen)
{
	struct xml *x0;
	struct xml *x1;
	struct xml *x2;

	x0 = xml_alloc("generic");
	xml_attr_set(x0, "seq", gen->seq);
	xml_attr_set(x0, "id", gen->name);

	x1 = xml_alloc("type");
	xml_attr_set(x1, "id", gen->type);
	xml_append(x0, x1);

	x1 = xml_alloc("value");
	x2 = xml_alloc_text(gen->value);
	xml_append(x1, x2);
	xml_append(x0, x1);

	return x0;
}

static void
generic_bbox_add(
	struct xml_schem_generic *gen,
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp
)
{
	struct xml_schem_ge *ge;

	for (ge = gen->ge_first; ge; ge = ge->next) {
		ge_bbox_add(ge, minxp, minyp, maxxp, maxyp);
	}
}

static void
generic_translate(struct xml_schem_generic *gen, double dx, double dy)
{
	struct xml_schem_ge *ge;

	for (ge = gen->ge_first; ge; ge = ge->next) {
		ge_translate(ge, dx, dy);
	}
}

static struct xml_schem_port *
port_get(struct xml *x0)
{
	struct xml_schem_port *port;
	struct xml *x1;
	struct xml_schem_ge *ge;

	port = malloc(sizeof(*port));
	assert(port);
	memset(port, 0, sizeof(*port));

	port->seq = xml_attr_lookup(x0, "seq");
	port->name = xml_attr_lookup(x0, "id");
	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		 && x1->type != XML_START_END) continue;

		if (strcmp(x1->start.string, "signal") == 0) {
			port->signal = xml_attr_lookup(x1, "id");
		} else if (strcmp(x1->start.string, "inout") == 0) {
			port->inout = xml_attr_lookup(x1, "id");
		} else if (strcmp(x1->start.string, "line") == 0
			|| strcmp(x1->start.string, "box") == 0
			|| strcmp(x1->start.string, "circle") == 0
			|| strcmp(x1->start.string, "text") == 0) {
			ge = ge_get(x1);

			ge->prev = port->ge_last;
			ge->next = NULL;
			if (ge->prev) {
				ge->prev->next = ge;
			} else {
				port->ge_first = ge;
			}
			port->ge_last = ge;
		} else {
			assert(0);
		}
	}

	return port;
}

static struct xml *
port_put(struct xml_schem_port *port)
{
	struct xml *x0;
	struct xml *x1;
	struct xml_schem_ge *ge;

	x0 = xml_alloc("port");
	if (port->seq) {
		xml_attr_set(x0, "seq", port->seq);
	}
	xml_attr_set(x0, "id", port->name);

	x1 = xml_alloc("signal");
	xml_attr_set(x1, "id", port->signal);
	xml_append(x0, x1);

	x1 = xml_alloc("inout");
	xml_attr_set(x1, "id", port->inout);
	xml_append(x0, x1);

	for (ge = port->ge_first; ge; ge = ge->next) {
		x1 = ge_put(ge);
		xml_append(x0, x1);
	}

	return x0;
}

static void
port_bbox_add(
	struct xml_schem_port *port,
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp
)
{
	struct xml_schem_ge *ge;

	for (ge = port->ge_first; ge; ge = ge->next) {
		ge_bbox_add(ge, minxp, minyp, maxxp, maxyp);
	}
}

static void
port_translate(struct xml_schem_port *port, double dx, double dy)
{
	struct xml_schem_ge *ge;

	for (ge = port->ge_first; ge; ge = ge->next) {
		ge_translate(ge, dx, dy);
	}
}

static struct xml_schem_signal *
signal_get(struct xml *x0)
{
	struct xml_schem_signal *sig;
	struct xml *x1;
	struct xml_schem_ge *ge;

	sig = malloc(sizeof(*sig));
	assert(sig);
	memset(sig, 0, sizeof(*sig));

	sig->id = xml_attr_lookup(x0, "id");

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		 && x1->type != XML_START_END) continue;

		if (strcmp(x1->start.string, "name") == 0) {
			struct xml_schem_name *name;

			name = malloc(sizeof(*name));
			assert(name);
			name->name = strdup(xml_attr_lookup(x1, "id"));
			assert(name->name);

			name->prev = sig->name_last;
			name->next = NULL;
			if (name->prev) {
				name->prev->next = name;
			} else {
				sig->name_first = name;
			}
			sig->name_last = name;

		} else if (strcmp(x1->start.string, "type") == 0) {
			sig->type = strdup(xml_attr_lookup(x1, "id"));
			assert(sig->type);

		} else if (strcmp(x1->start.string, "line") == 0
			|| strcmp(x1->start.string, "box") == 0
			|| strcmp(x1->start.string, "circle") == 0
			|| strcmp(x1->start.string, "text") == 0) {
			ge = ge_get(x1);

			ge->prev = sig->ge_last;
			ge->next = NULL;
			if (ge->prev) {
				ge->prev->next = ge;
			} else {
				sig->ge_first = ge;
			}
			sig->ge_last = ge;

		} else {
			assert(0);
		}
	}

	return sig;
}

static struct xml *
signal_put(struct xml_schem_signal *sig)
{
	struct xml *x0;
	struct xml *x1;
	struct xml_schem_ge *ge;
	struct xml_schem_name *name;

	x0 = xml_alloc("signal");
	xml_attr_set(x0, "id", sig->id);

	for (name = sig->name_first; name; name = name->next) {
		x1 = xml_alloc("name");
		xml_attr_set(x1, "id", name->name);
		xml_append(x0, x1);
	}

	x1 = xml_alloc("type");
	xml_attr_set(x1, "id", sig->type);
	xml_append(x0, x1);

	for (ge = sig->ge_first; ge; ge = ge->next) {
		x1 = ge_put(ge);
		xml_append(x0, x1);
	}

	return x0;
}

static void
signal_bbox_add(
	struct xml_schem_signal *sig,
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp
)
{
	struct xml_schem_ge *ge;

	for (ge = sig->ge_first; ge; ge = ge->next) {
		ge_bbox_add(ge, minxp, minyp, maxxp, maxyp);
	}
}

static void
signal_translate(struct xml_schem_signal *sig, double dx, double dy)
{
	struct xml_schem_ge *ge;

	for (ge = sig->ge_first; ge; ge = ge->next) {
		ge_translate(ge, dx, dy);
	}
}

static struct xml_schem_component *
comp_get(struct xml *x0)
{
	struct xml_schem_component *comp;
	struct xml_schem_generic *gen;
	struct xml_schem_port *port;
	struct xml_schem_ge *ge;
	struct xml *x1;
	struct xml *x2;

	comp = malloc(sizeof(*comp));
	assert(comp);
	memset(comp, 0, sizeof(*comp));

	comp->name = xml_attr_lookup(x0, "id");
	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		 && x1->type != XML_START_END) continue;

		if (strcmp(x1->start.string, "type") == 0) {
			comp->type = xml_attr_lookup(x1, "id");
		} else if (strcmp(x1->start.string, "generic") == 0) {
			gen = malloc(sizeof(*gen));
			assert(gen);
			memset(gen, 0, sizeof(*gen));

			gen->name = xml_attr_lookup(x1, "id");
			for (x2 = x1->start.child_first; x2; x2 = x2->next) {
				if (x2->type != XML_START
				 && x2->type != XML_START_END) continue;
				
				if (strcmp(x2->start.string, "type") == 0) {
					gen->type = xml_attr_lookup(x2, "id");
				} else if (strcmp(x2->start.string, "value") == 0) {
					gen->value = xml_attr_lookup(x2, "id");
				} else {
					ge = ge_get(x2);

					ge->prev = gen->ge_last;
					ge->next = NULL;
					if (ge->prev) {
						ge->prev->next = ge;
					} else {
						gen->ge_first = ge;
					}
					gen->ge_last = ge;
				}
			}

			gen->prev = comp->gen_last;
			gen->next = NULL;
			if (gen->prev) {
				gen->prev->next = gen;
			} else {
				comp->gen_first = gen;
			}
			comp->gen_last = gen;

		} else if (strcmp(x1->start.string, "port") == 0) {
			port = malloc(sizeof(*port));
			assert(port);
			memset(port, 0, sizeof(*port));

			port->name = xml_attr_lookup(x1, "id");
			for (x2 = x1->start.child_first; x2; x2 = x2->next) {
				if (x2->type != XML_START
				 && x2->type != XML_START_END) continue;
				
				if (strcmp(x2->start.string, "inout") == 0) {
					port->inout = xml_attr_lookup(x2, "id");
				} else if (strcmp(x2->start.string, "signal") == 0) {
					port->signal = xml_attr_lookup(x2, "id");
				} else {
					ge = ge_get(x2);

					ge->prev = port->ge_last;
					ge->next = NULL;
					if (ge->prev) {
						ge->prev->next = ge;
					} else {
						port->ge_first = ge;
					}
					port->ge_last = ge;
				}
			}

			port->prev = comp->port_last;
			port->next = NULL;
			if (port->prev) {
				port->prev->next = port;
			} else {
				comp->port_first = port;
			}
			comp->port_last = port;

		} else {
			ge = ge_get(x1);

			ge->prev = comp->ge_last;
			ge->next = NULL;
			if (ge->prev) {
				ge->prev->next = ge;
			} else {
				comp->ge_first = ge;
			}
			comp->ge_last = ge;
		}
	}

	return comp;
}

static struct xml *
comp_put(struct xml_schem_component *comp)
{
	struct xml_schem_port *port;
	struct xml *x0;
	struct xml *x1;
	struct xml *x2;
	struct xml_schem_ge *ge;

	x0 = xml_alloc("comp");
	xml_attr_set(x0, "id", comp->name);

	x1 = xml_alloc("type");
	xml_attr_set(x1, "id", comp->type);
	xml_append(x0, x1);

	for (ge = comp->ge_first; ge; ge = ge->next) {
		x1 = ge_put(ge);
		xml_append(x0, x1);
	}

	for (port = comp->port_first; port; port = port->next) {
		x1 = xml_alloc("port");
		xml_attr_set(x1, "id", port->name);

		x2 = xml_alloc("inout");
		xml_attr_set(x2, "id", port->inout);
		xml_append(x1, x2);

		x2 = xml_alloc("signal");
		xml_attr_set(x2, "id", port->signal);
		xml_append(x1, x2);

		for (ge = port->ge_first; ge; ge = ge->next) {
			x2 = ge_put(ge);
			xml_append(x1, x2);
		}

		xml_append(x0, x1);
	}

	return x0;
}

static void
comp_bbox_add(
	struct xml_schem_component *comp,
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp
)
{
	struct xml_schem_port *port;
	struct xml_schem_ge *ge;

	for (ge = comp->ge_first; ge; ge = ge->next) {
		ge_bbox_add(ge, minxp, minyp, maxxp, maxyp);
	}
	for (port = comp->port_first; port; port = port->next) {
		for (ge = port->ge_first; ge; ge = ge->next) {
			ge_bbox_add(ge, minxp, minyp, maxxp, maxyp);
		}
	}
}

void
xml_schem_component_bbox(
	struct xml_schem_component *comp,
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp
)
{
	bbox_null(minxp, minyp, maxxp, maxyp);
	comp_bbox_add(comp, minxp, minyp, maxxp, maxyp);
}

static void
comp_translate(struct xml_schem_component *comp, double dx, double dy)
{
	struct xml_schem_port *port;
	struct xml_schem_ge *ge;

	for (ge = comp->ge_first; ge; ge = ge->next) {
		ge_translate(ge, dx, dy);
	}
	for (port = comp->port_first; port; port = port->next) {
		for (ge = port->ge_first; ge; ge = ge->next) {
			ge_translate(ge, dx, dy);
		}
	}
}

static struct xml_schem *
plan_get(struct xml *x0)
{
	struct xml_schem *plan;
	struct xml *x1;
	struct xml *x2;

	assert(x0->type == XML_START
	    || x0->type == XML_START_END);
	assert(strcmp(x0->start.string, "plan") == 0);

	plan = malloc(sizeof(*plan));
	assert(plan);
	memset(plan, 0, sizeof(*plan));

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (strcmp(x1->start.string, "generics") == 0) {
			for (x2 = x1->start.child_first; x2; x2 = x2->next) {
				struct xml_schem_generic *gen;
				
				gen = generic_get(x2);

				gen->prev = plan->gen_last;
				gen->next = NULL;
				if (gen->prev) {
					gen->prev->next = gen;
				} else {
					plan->gen_first = gen;
				}
				plan->gen_last = gen;
			}
		} else if (strcmp(x1->start.string, "ports") == 0) {
			for (x2 = x1->start.child_first; x2; x2 = x2->next) {
				struct xml_schem_port *port;

				port = port_get(x2);

				port->prev = plan->port_last;
				port->next = NULL;
				if (port->prev) {
					port->prev->next = port;
				} else {
					plan->port_first = port;
				}
				plan->port_last = port;
			}
		} else if (strcmp(x1->start.string, "signals") == 0) {
			for (x2 = x1->start.child_first; x2; x2 = x2->next) {
				struct xml_schem_signal *sig;

				sig = signal_get(x2);

				sig->prev = plan->sig_last;
				sig->next = NULL;
				if (sig->prev) {
					sig->prev->next = sig;
				} else {
					plan->sig_first = sig;
				}
				plan->sig_last = sig;
			}
		} else if (strcmp(x1->start.string, "comps") == 0) {
			for (x2 = x1->start.child_first; x2; x2 = x2->next) {
				struct xml_schem_component *comp;

				comp = comp_get(x2);

				if (comp->type) { /* FIXME */
					comp->prev = plan->comp_last;
					comp->next = NULL;
					if (comp->prev) {
						comp->prev->next = comp;
					} else {
						plan->comp_first = comp;
					}
					plan->comp_last = comp;
				}
			}
		} else {
			assert(0);
		}
	}

	return plan;
}

static struct xml *
plan_put(struct xml_schem *plan)
{
	struct xml *x0;
	struct xml *x1;
	struct xml *x2;
	struct xml_schem_generic *gen;
	struct xml_schem_port *port;
	struct xml_schem_signal *sig;
	struct xml_schem_component *comp;

	x0 = xml_alloc("plan");
	
	x1 = xml_alloc("generics");
	xml_append(x0, x1);
	for (gen = plan->gen_first; gen; gen = gen->next) {
		x2 = generic_put(gen);
		xml_append(x1, x2);
	}

	x1 = xml_alloc("ports");
	xml_append(x0, x1);
	for (port = plan->port_first; port; port = port->next) {
		x2 = port_put(port);
		xml_append(x1, x2);
	}

	x1 = xml_alloc("signals");
	xml_append(x0, x1);
	for (sig = plan->sig_first; sig; sig = sig->next) {
		x2 = signal_put(sig);
		xml_append(x1, x2);
	}

	x1 = xml_alloc("comps");
	xml_append(x0, x1);
	for (comp = plan->comp_first; comp; comp = comp->next) {
		x2 = comp_put(comp);
		xml_append(x1, x2);
	}

	return x0;
}

void
xml_schem_bbox(
	struct xml_schem *plan,
	double *minxp,
	double *minyp,
	double *maxxp,
	double *maxyp
)
{
	struct xml_schem_generic *gen;
	struct xml_schem_port *port;
	struct xml_schem_signal *sig;
	struct xml_schem_component *comp;

	bbox_null(minxp, minyp, maxxp, maxyp);

	for (gen = plan->gen_first; gen; gen = gen->next) {
		generic_bbox_add(gen, minxp, minyp, maxxp, maxyp);
	}
	for (port = plan->port_first; port; port = port->next) {
		port_bbox_add(port, minxp, minyp, maxxp, maxyp);
	}
	for (sig = plan->sig_first; sig; sig = sig->next) {
		signal_bbox_add(sig, minxp, minyp, maxxp, maxyp);
	}
	for (comp = plan->comp_first; comp; comp = comp->next) {
		comp_bbox_add(comp, minxp, minyp, maxxp, maxyp);
	}
}

void
xml_schem_translate(struct xml_schem *plan, double dx, double dy)
{
	struct xml_schem_generic *gen;
	struct xml_schem_port *port;
	struct xml_schem_signal *sig;
	struct xml_schem_component *comp;

	for (gen = plan->gen_first; gen; gen = gen->next) {
		generic_translate(gen, dx, dy);
	}
	for (port = plan->port_first; port; port = port->next) {
		port_translate(port, dx, dy);
	}
	for (sig = plan->sig_first; sig; sig = sig->next) {
		signal_translate(sig, dx, dy);
	}
	for (comp = plan->comp_first; comp; comp = comp->next) {
		comp_translate(comp, dx, dy);
	}
}

struct xml_schem *
xml_schem_read(FILE *fp)
{
	struct xml *xml;
	struct xml_schem *schem;

	xml = xml_parse(fp);
	assert(xml);

	schem = plan_get(xml);

	xml_free(xml);

	return schem;
}

int
xml_schem_write(FILE *fp, struct xml_schem *schem)
{
	struct xml *xml;

	xml = plan_put(schem);

	xml_write(fp, xml);

	xml_free(xml);

	return 0;
}
