/*
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

/* ===================== RUNTIME || INIT ======================== */
#if defined(RUNTIME_RM) || defined(INIT_RM)

#define FD_CHANGELINE_SUPPORT	1

#define IDE_ERR_STAT	0x01
#define IDE_INDEX_STAT	0x02
#define IDE_ECC_STAT	0x04
#define IDE_DRQ_STAT	0x08
#define IDE_SEEK_STAT	0x10
#define IDE_WRERR_STAT	0x20
#define IDE_READY_STAT	0x40
#define IDE_BUSY_STAT	0x80

#define WIN_RESTORE	0x10 /* Recalibrate Device */
#define WIN_IDENTIFY    0xec /* Identify ATA Device */
#define WIN_PIDENTIFY   0xa1 /* Identify ATAPI Device */

extern const uint16_t ide_port_table[];

#endif /* RUNTIME_RM || INIT_RM */
/* ==================== RUNTIME_RM ==================== */
#ifdef RUNTIME_RM

CODE16;

#include "assert.h"
#include "fixme.h"
#include "stdio.h"
#include "string.h"
#include "in.h"

#include "io.h"
#include "const.h"
#include "cmos.h"
#include "var.h"
#include "el_torito.h"
#include "ptrace.h"
#include "floppy.h"
#include "harddisk.h"
#include "disk.h"

struct Int13DPT {
	uint16_t size;
	uint16_t flags;
	uint32_t cyls;
	uint32_t heads;
	uint32_t secs;
	uint32_t total_low;	/* Should be long long */
	uint32_t total_high;
	uint16_t bytes_per_sec;
	/* ... */
} __attribute__((__packed__));


struct Int13Ext {
	uint8_t size;
	uint8_t reserved;
	uint16_t count;
	uint16_t offset;
	uint16_t segment;
#if 0	/* FIXME VOSSI */
	uint64_t long lba;
#else
	uint32_t lba;
	uint32_t lba2;
#endif
} __attribute__((__packed__));

CONST uint16_t ide_port_table[] = {
	/* hda/hdb */ 0x01f0,
	/* hdc/hdd */ 0x0170,
	/* hde/hdf */ 0x01e8,
	/* hdg/hdh */ 0x0168,
	/* hdi/hdj */ 0x01e0,
	/* hdk/hdl */ 0x0160,
};

static void
int13_fail(struct regs *regs, uint8_t status)
{
	if (DL < 2) {
		var_put(fd_res, status);
	} else {
		var_put(hd_res, status);
	}

	AH = status;
	F |= 1 << 0;	/* Set carry. */
}

static void
int13_success(struct regs *regs, uint8_t status)
{
	if (DL < 2) {
		var_put(fd_res, status);
	} else {
		var_put(hd_res, status);
	}

	AH = status;
	F &= ~(1 << 0);	/* Clear carry. */
}

static void
int13_eltorito(struct regs *regs)
{
	struct info {
		uint8_t size;
		uint8_t media;
		uint8_t emulated_drive;
		uint8_t controller_index;
		uint32_t ilba;
		uint16_t device_spec;
		uint16_t buffer_segment;
		uint16_t load_segment;
		uint16_t sector_count;
		uint8_t cylinders;
		uint8_t spt;
		uint8_t heads;
	} __attribute__((__packed__));

	if (AH == 0x4a		/* ElTorito - Initiate disk emu */
	 || AH == 0x4c		/* ElTorito - Initiate disk emu and boot */
	 || AH == 0x4d) {	/* ElTorito - Return Boot catalog */
		/* FIXME VOSSI */
		assert(0);
		int13_fail(regs, 0x01);

	} else if (AH == 0x4b) {
		/*
		 * ElTorito - Terminate disk emu
		 */
		if (ebda_get(cdemu.media) == 0x00) {
			/* No Emulation */
			/* Nothing to do... */

		} else if (0x01 <= ebda_get(cdemu.media)
			&& ebda_get(cdemu.media) <= 0x03) {
			/* Floppy Emulation */
			/* Decrement number of floppies. */
			uint16_t sys_conf;
			uint8_t num;

			sys_conf = var_get(sys_conf);

			assert((sys_conf >> 0) & 1);
			num = ((sys_conf >> 6) & 0x3) + 1;
			num--;

			if (! num) {
				sys_conf &= ~(1 << 0);
				sys_conf &= ~(3 << 6);
			} else {
				sys_conf |= 1 << 0;
				sys_conf &= ~(3 << 6);
				sys_conf |= (num - 1) << 6;
			}

			var_put(sys_conf, sys_conf);

		} else if (ebda_get(cdemu.media) == 0x04) {
			/* Harddisk */
			uint8_t hd_num;

			hd_num = var_get(hd_num);

			assert(hd_num);
			hd_num--;

			var_put(hd_num, hd_num);

		} else {
			assert(0);
		}

		put_byte(DS, SI + offsetof(struct info, size),
			0x13);
		put_byte(DS, SI + offsetof(struct info, media),
			ebda_get(cdemu.media));
		put_byte(DS, SI + offsetof(struct info, emulated_drive),
			ebda_get(cdemu.emulated_drive));
		put_byte(DS, SI + offsetof(struct info, controller_index),
			ebda_get(cdemu.controller_index));
		put_long(DS, SI + offsetof(struct info, ilba),
			ebda_get(cdemu.ilba));
		put_word(DS, SI + offsetof(struct info, device_spec),
			ebda_get(cdemu.device_spec));
		put_word(DS, SI + offsetof(struct info, buffer_segment),
			ebda_get(cdemu.buffer_segment));
		put_word(DS, SI + offsetof(struct info, load_segment),
			ebda_get(cdemu.load_segment));
		put_word(DS, SI + offsetof(struct info, sector_count),
			ebda_get(cdemu.sector_count));
		put_byte(DS, SI + offsetof(struct info, cylinders),
			ebda_get(cdemu.vdevice.cylinders));
		put_byte(DS, SI + offsetof(struct info, spt),
			ebda_get(cdemu.vdevice.spt));
		put_byte(DS, SI + offsetof(struct info, heads),
			ebda_get(cdemu.vdevice.heads));

		if (AL == 0x00) {
			/* Should be handled accordingly to spec */
			/* FIXME VOSSI */
			ebda_put(cdemu.active, 0);
		}

		var_put(hd_res, 0x00);
		int13_success(regs, 0x00);
	}
}

static void
int13_floppy(struct regs *regs)
{
	uint8_t ret;
#if 0
	assert(/* 0x00 <= DL && */ DL <= 0x01);
#endif

	if (AH == 0x00) {
		/*
		 * Floppy controller reset.
		 */
		uint8_t drive_type;

		if (2 <= DL) {
			int13_fail(regs, 0x01);	/* Invalid param. */
			return;
		}

		drive_type = cmos_get(fd_config);
		if (DL == 0) {
			drive_type >>= 4;
		} else { assert(DL == 1);
			drive_type &= 0xf;
		}

		if (drive_type == 0) {
			int13_fail(regs, 0x80);	/* Not responding. */
			return;
		}

		var_put(f_track[DL], 0);
		int13_success(regs, 0x00);

	} else if (AH == 0x01) {
		/*
		 * Read floppy status.
		 */
		if (var_get(fd_res) == 0) {
			int13_success(regs, 0x00);
		} else {
			int13_fail(regs, var_get(fd_res));
		}

	} else if (AH == 0x02) {
		/*
		 * Read floppy sectors.
		 */
		ret = floppy_read_chs(DL, AL, CH, DH, CL, ES, BX);
		if (ret == 0) {
			int13_success(regs, 0x00);
		} else {
			AL = 0;
			int13_fail(regs, ret);
		}

	} else if (AH == 0x03) {
		/*
		 * Write floppy sectors.
		 */
		ret = floppy_write_chs(DL, AL, CH, DH, CL, ES, BX);
		if (ret == 0) {
			int13_success(regs, 0x00);
		} else {
			AL = 0;
			int13_fail(regs, ret);
		}

	} else if (AH == 0x04) {
		/*
		 * Verify floppy sectors.
		 */
		ret = floppy_verify_chs(DL, AL, CH, DH, CL, ES, BX);
		if (ret == 0) {
			int13_success(regs, 0x00);
		} else {
			AL = 0;
			int13_fail(regs, ret);
		}

	} else if (AH == 0x05) {
		/*
		 * Format disk track.
		 */
		dprintf("int13_floppy: Unknown function %02x\n", AH);
		assert(0);

	} else if (AH == 0x08) {
		/*
		 * Read floppy drive parameters.
		 */
		static CONST struct {
			uint8_t max_track;
			uint8_t sectors;
			uint8_t max_head;

		} floppy_types[] = {
			/* 0 none         */	{ 0, 0, 0 },
			/* 1 360KB, 5.25" */	{ 40-1, 9, 2-1 },
			/* 2 1.2MB, 5.25" */	{ 80-1, 15, 2-1 },
			/* 3 720KB, 3.5"  */	{ 80-1, 9, 2-1 },
			/* 4 1.44MB, 3.5" */	{ 80-1, 18, 2-1 },
			/* 5 2.88MB, 3.5" */	{ 80-1, 36, 2-1 },
			/* 6 160KB, 5.25" */	{ 40-1, 8, 1-1 },
			/* 7 180KB, 5.25" */	{ 40-1, 9, 1-1 },
			/* 8 320KB, 5.25" */	{ 40-1, 8, 2-1 },
			/* 9 unknown      */	{ 0, 0, 0 },
			/* a unknown      */	{ 0, 0, 0 },
			/* b unknown      */	{ 0, 0, 0 },
			/* c unknown      */	{ 0, 0, 0 },
			/* d unknown      */	{ 0, 0, 0 },
			/* e unknown      */	{ 0, 0, 0 },
			/* f unknown      */	{ 0, 0, 0 }
		};

		uint8_t num_floppies;
		uint8_t drive_type;

		num_floppies = 0;
		drive_type = cmos_get(fd_config);
		if ((drive_type >> 0) & 0xf) {
			num_floppies++;
		}
		if ((drive_type >> 4) & 0xf) {
			num_floppies++;
		}

		if (2 <= DL) {
			AL = 0;
			BX = 0;
			CX = 0;
			DX = 0;
			ES = 0;
			DI = 0;
			DL = num_floppies;
			AH = 0x00;	/* no error! */
			F |= 1 << 0;	/* Set carry. */
			return;
		}

		if (DL == 0) {
			drive_type >>= 4;
		} else { assert(DL == 1);
			drive_type &= 0xf;
		}

		BH = 0;
		BL = drive_type;
		AL = 0;
		DL = num_floppies;
		CH = const_get(floppy_types[drive_type].max_track);
		CL = const_get(floppy_types[drive_type].sectors);
		DH = const_get(floppy_types[drive_type].max_head);

		/* Get diskette parameter table stored in INT vector 0x1e. */
		DI = get_word(0x0000, 0x1e * 4 + 0);
		ES = get_word(0x0000, 0x1e * 4 + 2);

		/* Disk status not changed upon success. */
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x15) {
		/*
		 * Read floppy drive type.
		 */
		uint8_t drive_type;

		if (2 <= DL) {
			/* var_put(fmot_stat, ...);	FIXME VOSSI */
			int13_fail(regs, 0x01);
			return;
		}

		drive_type = cmos_get(fd_config);
		if (DL == 0) {
			drive_type >>= 4;
		} else { assert(DL == 1);
			drive_type &= 0xf;
		}
		if (drive_type == 0) {
			int13_success(regs, 0x00);	/* Drive not present. */
		} else {
							/* Drive present. */
#if FD_CHANGELINE_SUPPORT
			int13_success(regs, 0x02);	/* Does support */
							/* change line. */
#else
			int13_success(regs, 0x01);	/* Does not support */
							/* change line. */
#endif
		}
	} else if (AH == 0x16) {
		/*
		 * Get floppy change line status.
		 */
#if FD_CHANGELINE_SUPPORT
		ret = floppy_check_changeline(DL);
		if(ret == 0) {
			/* disk not changed: return successfully */
			int13_success(regs,0x00);
		} else {
			/* other cases:
			 *  - disk changed
			 *  - no disk present,
			 *  - drive not ready,
			 *  - drive not present,
			 *  - ...
			 * => return as if failed
			 */
			int13_fail(regs,ret);
		}
#else
		dprintf("int13_floppy: Unknown function %02x\n", AH);
		assert(0);
#endif
	} else if (AH == 0x17) {
		/*
		 * Set floppy type for format (old).
		 */
		dprintf("int13_floppy: Unknown function %02x\n", AH);
		assert(0);

	} else if (AH == 0x18) {
		/*
		 * Set floppy type for format (new).
		 * AH:			0x18
		 * DL:			drive
		 * CH:			cylinder number (bits 0-7)
		 * CL (bit 6-7):	cylinder number (bits 8-9)
		 * CL (bit 0-5):	sectors per track
		 */
		dprintf("int13_floppy: Set floppy type called (%d,%d).\n",
				(((CL >> 6) & 3) << 8) | CH, (CL >> 0) & 0x3f);
		int13_success(regs, 0x00);	/* FIXME VOSSI */

	} else if (AH == 0x41) {
		/*
		 * BIOS Enhanced Disk Drive call:
		 * "Check Extensions Present".
		 */
		/* Not supported for floppy disks. FIXME VOSSI */
		int13_fail(regs, 0x01);

	} else {
		dprintf("int13_floppy: Drive 0x%02x: Unknown function %02x\n",
				DL, AH);
		int13_fail(regs, 0x01);
	}
}

static void
int13_harddisk(struct regs *regs)
{
	uint8_t device;

	if (DL < 0x80 || 0x80 + MAX_BIOS_HDS <= DL) {
		var_put(hd_res, 0x01);
		AH = 0x01;
		F |= 1 << 0; /* Set carry. */
		return;
	}
	device = ebda_get(hdidmap[DL - 0x80]);
	if (MAX_BIOS_HDS <= device) {
		var_put(hd_res, 0x01);
		AH = 0x01;
		F |= 1 << 0; /* Set carry. */
		return;
	}

	/* Clear completion flag. */
	var_put(hd_irq, 0);

	if (AH == 0x00) {
		/*
		 * Reset disk controller.
		 */
		/* ata_reset(device); */	/* FIXME VOSSI */
		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x02
		|| AH == 0x03
		|| AH == 0x04) {
		/*
		 * Read disk sectors.
		 * Write disk sectors.
		 * Verify disk sectors.
		 */
		uint8_t count;
		uint16_t cylinder;
		uint8_t sector;
		uint8_t head;
		uint16_t segment;
		uint16_t offset;
		uint8_t status;

		count = AL;
		cylinder = ((uint16_t) (CL & 0xc0) << 2) | CH;
		sector = CL & 0x3f;
		head = DH;

		segment = ES;
		offset = BX;

		/* Sanity check on count. */
		if (128 < count || count == 0) {
			var_put(hd_res, 0x01);
			AH = 0x01;
			F |= 1 << 0; /* Set carry. */
			return;
		}

		/* Sanity check on C/H/S. */
		if (ebda_get(devices[device].lchs_cylinders) <= cylinder
		 || ebda_get(devices[device].lchs_heads) <= head
		 || ebda_get(devices[device].lchs_spt) < sector) {
			var_put(hd_res, 0x01);
			AH = 0x01;
			F |= 1 << 0; /* Set carry. */
			return;
		}

		if (AH == 0x02) {
			status = ide_harddisk_read_chs(device,
					count, cylinder, head, sector,
					segment, offset);
		} else if (AH == 0x03) {
			status = ide_harddisk_write_chs(device,
					count, cylinder, head, sector,
					segment, offset);
		} else {
			/* Verify */
			status = 0; /* FIXME */
		}
		
		/* Set nb of sector transferred. */
		/* FIXME VOSSI */

		if (status != 0) {
			var_put(hd_res, 0x0c);
			AH = 0x0c;
			F |= 1 << 0; /* Set carry. */
			return;
		}

		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x05) {
		/*
		 * Format disk track.
		 */
		/* FIXME VOSSI */
		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x08) {
		/*
		 * Read disk drive parameters.
		 */
		uint16_t nlc;
		uint16_t nlh;
		uint16_t nlspt;
		uint16_t count;

		nlc = ebda_get(devices[device].lchs_cylinders);
		if (1024 < nlc) {
			nlc = 1024;	/* FIXME VOSSI */
		}
		nlh = ebda_get(devices[device].lchs_heads);
		nlspt = ebda_get(devices[device].lchs_spt);
		count = ebda_get(harddrives);

		nlc = nlc - 2; /* 0 based , last sector not used */

		AL = 0;
		CH = nlc & 0xff;
		CL = ((nlc >> 8) << 6) | (nlspt & 0x3f);
		DH = nlh - 1;
		DL = count;	/* FIXME returns 0, 1, or n hard drives */

		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x10) {
		/*
		 * Check drive ready.
		 */
		/* FIXME */

		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x11) {
		/*
		 * Recalibrate drive.
		 */
		/* FIXME */

		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x15) {
		/*
		 * Read disk drive size.
		 */
		uint16_t npc;
		uint16_t nph;
		uint16_t npspt;
		uint32_t lba;

		/* Get physical geometry from table. */
		/* FIXME VOSSI */
		npc = ebda_get(devices[device].pchs_cylinders);
		nph = ebda_get(devices[device].pchs_heads);
		npspt = ebda_get(devices[device].pchs_spt);

		/* Compute sector count seen by int13. */
		lba = (uint32_t) (npc - 1)
			* (uint32_t) nph
			* (uint32_t) npspt;

		CX = (lba >> 16) & 0xffff;
		DX = (lba >>  0) & 0xffff;

		var_put(hd_res, 0x00);
		AH = 0x03; /* Type: 3 (Harddisk) */
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x41) {
		/*
		 * BIOS Enhanced Disk Drive call:
		 * "Check Extensions Present".
		 */
		BX = 0xaa55;	/* Install check. */
		CX = 0x0007;	/* Options supported. */

		var_put(hd_res, 0x00);
		AH = 0x30; /* Version 3.0 */
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x42
		|| AH == 0x43
		|| AH == 0x44
		|| AH == 0x47) {
		/*
		 * Extended read.
		 * Extended write.
		 * Extended verify.
		 * Extended seek.
		 */
		uint16_t count;
		uint16_t segment;
		uint16_t offset;
#if 0
		uint64_t lba;
#else
		uint32_t lba;
#endif
		uint8_t status;


		count = get_word(DS, SI + 2);
		segment = get_word(DS, SI + 6);
		offset = get_word(DS, SI + 4);
#if 0
		lba = get_qword(DS, SI + 8);
#else
		lba = get_long(DS, SI + 8);
#endif

		/* Sanity check on lba. */
		if (ebda_get(devices[device].sectors) <= lba) {
			var_put(hd_res, 0x01);
			AH = 0x01;
			F |= 1 << 0; /* Set carry. */
			return;
		}

		if (AH == 0x44 || AH == 0x47) {
			/* Verify or seek. */
			var_put(hd_res, 0x00);
			AH = 0x00;
			F &= ~(1 << 0); /* Clear carry. */
			return;
		}

		if (AH == 0x42) {
			/* Read */
			status = ide_harddisk_read_lba(device / 2, device % 2,
					count, lba, segment, offset);
		} else { assert(AH == 0x43);
			/* Write */
			status = ide_harddisk_write_lba(device / 2, device % 2,
					count, lba, segment, offset);
		}

		if (status != 0) {
			var_put(hd_res, 0x0c);
			AH = 0x0c;
			F |= 1 << 0; /* Set carry. */
			return;
		}

		/* FIXME VOSSI */
		put_word(DS, SI + 2, count);

		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x45
		|| AH == 0x49) {
		/*
		 * IBM/MS lock/unlock drive.
		 * IBM/MS extended media change.
		 */
		/* All these functions return SUCCESS. */
		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x46) {
		/*
		 * IBM/MS eject media.
		 */
		/* This function returns FAILURE. */
		var_put(hd_res, 0xb2);
		AH = 0xb2;
		F |= 1 << 0; /* Set carry. */

	} else if (AH == 0x48) {
		/*
		 * BIOS Enhanced Disk Drive call:
		 * "Get Device Parameters".
		 */
		struct info {
			uint16_t size;
			uint16_t flags;
			uint32_t cyls;
			uint32_t heads;
			uint32_t secs;
			uint32_t total_low;	/* Should be long long */
			uint32_t total_high;
			uint16_t bytes_per_sec;
			/* ... */
		} __attribute__((__packed__));
		uint16_t size;

		size = get_word(DS, SI + offsetof(struct info, size));

		if (size < 0x1a) {
			/* Buffer is too small. */
			var_put(hd_res, 0x01);
			AH = 0x01;
			F |= 1 << 0; /* Set carry. */
			return;
		}

		if (0x1a <= size) {
			/* EDD 1.x */
			uint32_t cyls;
			uint32_t heads;
			uint32_t secs;
			uint32_t /* long */ total;	/* FIXME MARCEL */

			cyls = ebda_get(devices[device].pchs_cylinders);
			heads = ebda_get(devices[device].pchs_heads);
			secs = ebda_get(devices[device].pchs_spt);
			total = cyls * heads * secs;

			put_word(DS, SI + offsetof(struct info, size), 0x1a);
			put_word(DS, SI + offsetof(struct info, flags), 0x02);
			put_long(DS, SI + offsetof(struct info, cyls), cyls);
			put_long(DS, SI + offsetof(struct info, heads), heads);
			put_long(DS, SI + offsetof(struct info, secs), secs);
			put_long(DS, SI + offsetof(struct info, total_low), total);
			put_long(DS, SI + offsetof(struct info, total_high), 0L); /* FIXME MARCEL */
			put_word(DS, SI + offsetof(struct info, bytes_per_sec), 512);
		}
		if (0x1e <= size) {
			/* EDD 2.x */
			/* put_word(DS, SI + offsetof(struct info, size), 0x1e); */
			put_word(DS, SI + offsetof(struct info, size), 0x1a);

			/* Not supported yet. FIXME VOSSI */
		}
		if (0x42 <= size) {
			/* EDD 3.x */
			/* Not supported yet. FIXME VOSSI */
		}

		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x4e) {
		/*
		 * Set hardware configuration.
		 */
		assert(0);	/* FIXME VOSSI */

	} else if (AH == 0x09
		|| AH == 0x0c
		|| AH == 0x0d
		|| AH == 0x11
		|| AH == 0x14) {
		/*
		 * Initialize drive parameters.
		 * Seek to specified cylinder.
		 * Alternate disk reset.
		 * Recalibrate.
		 * Controller internal diagnostic.
		 */
		/* All these functions return SUCCESS. */
		var_put(hd_res, 0x00);
		AH = 0x00;
		F &= ~(1 << 0); /* Clear carry. */

	} else if (AH == 0x0a
		|| AH == 0x0b
		|| AH == 0x18
		|| AH == 0x50) {
		/*
		 * Read disk sectors with ECC.
		 * Write disk sectors with ECC.
		 * Set media type for format.
		 * IBM/MS send packet command.
		 */
		/* All these functions return UNSUPPORTED. */
		var_put(hd_res, 0x01);
		AH = 0x01;
		F |= 1 << 0; /* Set carry. */

	} else {
		dprintf("int13_harddisk: AH=0x%02x\n", AH);
		/* Return UNSUPPORTED. */
		var_put(hd_res, 0x01);
		AH = 0x01;
		F |= 1 << 0; /* Set carry. */
	}
}

static void
int13_cdrom(struct regs *regs)
{
	uint8_t device;

	if (DL < 0xe0 || 0xe0 + MAX_BIOS_HDS <= DL) {
		var_put(hd_res, 0x01);
		int13_fail(regs, 0x01);
		return;
	}
	device = ebda_get(cdidmap[DL - 0xe0]);
	if (device == MAX_BIOS_HDS) {
		var_put(hd_res, 0x01);
		int13_fail(regs, 0x01);
		return;
	}

	if (AH == 0x00
	 || AH == 0x09
	 || AH == 0x0c
	 || AH == 0x0d
	 || AH == 0x10
	 || AH == 0x11
	 || AH == 0x14
	 || AH == 0x16) {
		/*
		 * Disk controller reset.
		 * Initialize drive parameters.
		 * Seek to specified cylinder.
		 * Alternate disk reset.
		 * Check drive ready.
		 * Recalibrate.
		 * Controller internal diagnostic.
		 * Detect disk change.
		 */
		/* All these functions return SUCCESS. */
		var_put(hd_res, 0x00);
		int13_success(regs, 0x00);

	} else if (AH == 0x03
		|| AH == 0x05
		|| AH == 0x43) {
		/*
		 * Write disk sectors.
		 * Format disk track.
		 * Extended write.
		 */
		/* All these functions return DISK WRITE-PROTECTED. */
		var_put(hd_res, 0x03);
		int13_success(regs, 0x03);

	} else if (AH == 0x01) {
		/*
		 * Read disk status.
		 */
		uint8_t status;

		status = var_get(hd_res);
		var_put(hd_res, 0x00);
		if (status) {
			int13_fail(regs, status);
		} else {
			int13_success(regs, status);
		}

	} else if (AH == 0x015) {
		/*
		 * Read disk drive size.
		 */
		var_put(hd_res, 0x02);
		int13_fail(regs, 0x02);

	} else if (AH == 0x41) {
		/*
		 * BIOS Enhanced Disk Drive call:
		 * "Check Extensions Present".
		 */
		BX = 0xaa55;		/* Install check. */
		CX = 0x0007;		/* Ext disk access, removable and EDD.*/
		var_put(hd_res, 0x00);
		int13_success(regs, 0x01);	/* Version 1.0 */

	} else if (AH == 0x42
		|| AH == 0x44
		|| AH == 0x47) {
		/*
		 * Extended read.
		 * Extended verify sectors.
		 * Extended seek.
		 */
		uint16_t count;
		uint16_t segment;
		uint16_t offset;
#if 0	/* FIXME VOSSI */
		uint64_t lba;
#else
		uint32_t lba;
#endif
		uint8_t status;

		count = get_word(DS, SI + offsetof(struct Int13Ext, count));
		segment = get_word(DS, SI + offsetof(struct Int13Ext, segment));
		offset = get_word(DS, SI + offsetof(struct Int13Ext, offset));

#if 0	/* FIXME VOSSI */
		lba = get_longlong(DS, SI + offsetof(struct Int13Ext, lba));
#else
		lba = get_long(DS, SI + offsetof(struct Int13Ext, lba));
#endif

		if (AH == 0x44
		 || AH == 0x47) {
			/* Verify or seek. */
			var_put(hd_res, 0x00);
			int13_success(regs, 0x00);
			return;
		}

		status = read_sector_lba_atapi(device,
				count, lba, segment, offset, 2048);
		count = (status == 0) ? count : 0;	/* FIXME VOSSI */
		put_word(DS, SI + offsetof(struct Int13Ext, count), count);
		if (status != 0) {
			var_put(hd_res, 0x0c);
			int13_fail(regs, 0x0c);
			return;
		}

		var_put(hd_res, 0x00);
		int13_success(regs, 0x00);

	} else if (AH == 0x45) {
		/*
		 * Lock/unlock drive.
		 */
		assert(0);	/* FIXME VOSSI */

	} else if (AH == 0x46) {
		/*
		 * Eject media.
		 */
		assert(0);	/* FIXME VOSSI */

	} else if (AH == 0x48) {
		/*
		 * BIOS Enhanced Disk Drive call:
		 * "Get Device Parameters".
		 */
		uint16_t size;

		size = get_word(DS, SI + offsetof(struct Int13DPT, size));

		if (size < 0x1a) {
			/* Buffer is too small. */
			var_put(hd_res, 0x01);
			int13_fail(regs, 0x01);
			return;
		}
		if (0x1a <= size) {
			uint16_t blksize;

			blksize = 2048;	/* FIXME VOSSI */

			put_word(DS, SI + offsetof(struct Int13DPT, size),
					0x1a);

			/* removable, media change, lockable, max values */
			put_word(DS, SI + offsetof(struct Int13DPT, flags),
					0x74);

			put_long(DS, SI + offsetof(struct Int13DPT, cyls),
					0xffffffff);
			put_long(DS, SI + offsetof(struct Int13DPT, heads),
					0xffffffff);
			put_long(DS, SI + offsetof(struct Int13DPT, secs),
					0xffffffff);
			put_long(DS, SI + offsetof(struct Int13DPT, total_low),
					0xffffffff);
			put_long(DS, SI + offsetof(struct Int13DPT, total_high),
					0xffffffff);
			put_word(DS, SI + offsetof(struct Int13DPT, bytes_per_sec),
					blksize);
		}
		if (0x1e <= size) {
			/* EDD 2.x */
			put_word(DS, SI + offsetof(struct Int13DPT, size), 0x1e);

			/* FIXME VOSSI */
		}
		if (0x42 <= size) {
			/* EDD 3.x */
			/* Not supported yet. FIXME VOSSI */
		}

		var_put(hd_res, 0x00);
		int13_success(regs, 0x00);

	} else if (AH == 0x49) {
		/*
		 * Extended media change.
		 */
		/* Always send changed?? */
		var_put(hd_res, 0x06);
		int13_fail(regs, 0x06);

	} else if (AH == 0x4e) {
		/*
		 * Set hardware configuration.
		 */
		/* DMA, prefetch, PIO maximum not supported. */
		if (AL == 0x01
		 || AL == 0x03
		 || AL == 0x04
		 || AL == 0x06) {
			var_put(hd_res, 0x00);
			int13_success(regs, 0x00);
		} else {
			var_put(hd_res, 0x01);
			int13_success(regs, 0x01);
		}

	} else if (AH == 0x02
		|| AH == 0x04
		|| AH == 0x08
		|| AH == 0x0a
		|| AH == 0x0b
		|| AH == 0x18
		|| AH == 0x50) {
		/*
		 * Read sectors.
		 * Verify sectors.
		 * Read disk drive parameters.
		 * Read disk sectors with ECC.
		 * Write disk sectors with ECC.
		 * Set media type for format.
		 * ? - Send packet command.
		 */
		/* All these functions return UNIMPLEMENTED. */
		var_put(hd_res, 0x01);
		int13_fail(regs, 0x01);

	} else {
		dprintf("int13_cdrom: AH=0x%02x\n", AH);
		/* Return UNSUPPORTED. */
		var_put(hd_res, 0x01);
		int13_fail(regs, 0x01);
	}
}

static void
int13_cdemu(struct regs *regs)
{
	uint8_t device;

	/* Recompute the device number. */
	device  = ebda_get(cdemu.controller_index) * 2;
	device += ebda_get(cdemu.device_spec);

	var_put(hd_res, 0x00);

	/*
	 * Basic checks: emulation should be active, DL should equal the
	 * emulated drive.
	 */
	if (ebda_get(cdemu.active) == 0
	 || ebda_get(cdemu.emulated_drive) != DL) {
		assert(0);
	}

	if (AH == 0x00		/* Disk controller reset. */
	 || AH == 0x09		/* Initialize drive parameters. */
	 || AH == 0x0c		/* Seek to specified cylinder. */
	 || AH == 0x0d		/* Alternate disk reset. */
	 || AH == 0x10		/* Check drive ready. */
	 || AH == 0x11		/* Recalibrate. */
	 || AH == 0x14		/* Controller internal diagnostic. */
	 || AH == 0x16) {	/* Detect disk change. */
		/* All these functions return SUCCESS. */
		var_put(hd_res, 0x00);
		int13_success(regs, 0x00);

	} else if (AH == 0x03	/* Write disk sectors. */
		|| AH == 0x05) {/* Format disk track. */
		/* All these functions return DISK WRITE-PROTECTED. */
		var_put(hd_res, 0x03);
		int13_fail(regs, 0x03);
		
	} else if (AH == 0x01) {
		/* Read disk status. */
		uint8_t status;

		status = var_get(hd_res);

		var_put(hd_res, 0x00);
		if (status) {
			int13_fail(regs, status);
		} else {
			int13_success(regs, status);
		}

	} else if (AH == 0x02	/* Read disk sectors. */
		|| AH == 0x04) {/* Verify disk sectors. */
		uint16_t vspt;
		uint16_t vcylinders;
		uint16_t vheads;
		uint16_t sector;
		uint16_t cylinder;
		uint16_t head;
		uint16_t nbsectors;
		uint16_t segment;
		uint16_t offset;
		uint32_t ilba;
		uint32_t vlba;
		uint32_t slba;
		uint32_t before;
		uint32_t elba;
		uint8_t status;

		vspt = ebda_get(cdemu.vdevice.spt);
		vcylinders = ebda_get(cdemu.vdevice.cylinders);
		vheads = ebda_get(cdemu.vdevice.heads);

		ilba = ebda_get(cdemu.ilba);

		sector = CL & 0x3f;
		cylinder = (((uint16_t) CL & 0xc0) << 2) | CH;
		head = DH;
		nbsectors = AL;

		/* No sector to read? */
		if (nbsectors == 0) {
			var_put(hd_res, 0x00);
			int13_success(regs, 0x00);
			return;
		}

		/* Sanity checks. SCO OpenServer needs this! */
		if (vspt < sector
		 || vcylinders <= cylinder
		 || vheads <= head) {
			var_put(hd_res, 0x01);
			int13_fail(regs, 0x01);
			return;
		}

		if (AH == 0x04) {
			/* Verify disk sectors. */
			var_put(hd_res, 0x00);
			int13_success(regs, 0x00);
			return;
		}

		segment = ES + (BX / 16);
		offset = BX % 16;

		/* Calculate the virtual lba inside the image. */
		vlba = ((((uint32_t) cylinder * (uint32_t) vheads)
			+ (uint32_t) head) * (uint32_t) vspt)
			+ (uint32_t) sector - 1;

		/* In advance so we don't loose the count. */
		AL = nbsectors;

		/* Start lba on CD. */
		slba = (uint32_t) vlba / 4;
		before = (uint32_t) vlba % 4;

		/* End lba on CD. */
		elba = (uint32_t) (vlba + nbsectors - 1) / 4;

		/* Read command. */
		status = read_sector_lba_atapi(device,
				nbsectors, ilba * 4 + vlba,
				segment, offset, 512);
		if (status != 0) {
			AL = 0;
			var_put(hd_res, 0x02);
			int13_fail(regs, 0x02);
			return;
		}

		var_put(hd_res, 0x00);
		int13_success(regs, 0x00);

	} else if (AH == 0x08) {
		/* Read disk drive parameters. */
		uint16_t vspt;
		uint16_t vcylinders;
		uint16_t vheads;

		vspt = ebda_get(cdemu.vdevice.spt);
		vcylinders = ebda_get(cdemu.vdevice.cylinders) - 1;
		vheads = ebda_get(cdemu.vdevice.heads) - 1;

		AL = 0x00;
		BL = 0x00;
		CH = vcylinders & 0xff;
		CL = ((vcylinders >> 2) & 0xc0) | (vspt & 0x3f);
		DH = vheads;
		DL = 0x02;	/* FIXME ElTorito Various. should send the real count of drives 1 or 2 */
				/* FIXME ElTorito Harddisk. should send the HD count */
		if (ebda_get(cdemu.media) == 0x01) {
			BL = 0x02;
		} else if (ebda_get(cdemu.media) == 0x02) {
			BL = 0x04;
		} else if (ebda_get(cdemu.media) == 0x03) {
			BL = 0x06;
		}

		/* Get diskette parameter table stored in INT vector 0x1e. */
		DI = get_word(0x0000, 0x1e * 4 + 0);
		ES = get_word(0x0000, 0x1e * 4 + 2);

		var_put(hd_res, 0x00);
		int13_success(regs, 0x00);

	} else if (AH == 0x15) {
		/* Identify drive. */
		/* What to do...? - FIXME VOSSI */
		var_put(hd_res, 0x00);
		int13_success(regs, 0x03);

	} else if (AH == 0x0a	/* Read disk sectors with ECC */
		|| AH == 0x0b	/* Write disk sectors with ECC */
		|| AH == 0x18	/* Set media type for format */
		|| AH == 0x41	/* IBM/MS installation check */
		|| AH == 0x42	/* IBM/MS extended read */
		|| AH == 0x43	/* IBM/MS extended write */
		|| AH == 0x44	/* IBM/MS verify sectors */
		|| AH == 0x45	/* IBM/MS lock/unlock drive */
		|| AH == 0x46	/* IBM/MS eject media */
		|| AH == 0x47	/* IBM/MS extended seek */
		|| AH == 0x48	/* IBM/MS get drive parameters */
		|| AH == 0x49	/* IBM/MS extended media change */
		|| AH == 0x4e	/* ? - set hardware configuration */
		|| AH == 0x50	/* ? - send packet command */
		|| 1) {		/* default */
		dprintf("int13_cdemu: AH=0x%02x\n", AH);
		/* All these functions return UNIMPLEMENTED. */
		var_put(hd_res, 0x01);
		int13_fail(regs, 0x01);
	}
}

C_ENTRY void
bios_13_xxxx(struct regs *regs)
{
	if (0x4a <= AH && AH <= 0x4d) {
		/*
		 * El-torito functions.
		 */
		int13_eltorito(regs);

	} else if (ebda_get(cdemu.active)) {
		if (DL == ebda_get(cdemu.emulated_drive)) {
			/*
			 * Emulated drive.
			 */
			int13_cdemu(regs);

		} else if ((DL & 0xe0) == ebda_get(cdemu.emulated_drive)) {
			/*
			 * Drive in same class as emulated drive.
			 * -> change drive number
			 */
			DL--;

			/* int 13h,08h is special because it returns
			 * the number of drives in DL, so we mustn't
			 * change it before returning from this handler */
			if(AH == 0x08) goto int13_legacy;

			if (DL < 0x80) {
				int13_floppy(regs);
			} else if (DL < 0xe0) {
				int13_harddisk(regs);
			} else {
				int13_cdrom(regs);
			}

			/* change back drive number as some software
			 * seems to rely on it */
			DL++;
		} else {
			/*
			 * Unaffected drive.
			 */
			goto int13_legacy;
		}
	} else {
int13_legacy:
		/*
		 * Standard int13 functions.
		 */
		if (DL < 0x80) {
			int13_floppy(regs);
		} else if (DL < 0xe0) {
			int13_harddisk(regs);
		} else {
			int13_cdrom(regs);
		}
	}
}

C_ENTRY void
bios_0e(struct regs *regs)
{
	var_put(f_recal, var_get(f_recal) | (1 << 7));

	eoi();

#if 0
	asm volatile (
		"pushw %ax\n"
		"movb $0x91, %ah\n" /* Signal end-of-operation. */
		"movb $0x01, %al\n" /* Device type is floppy. */
		"int $0x15\n"
		"popw %ax\n"
	);
#endif
}

#endif /* RUNTIME_RM */
/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

#include "assert.h"
#include "stdio.h"
#include "string.h"
#include "in.h"
#include "io.h"
#include "var.h"
#include "cmos.h"
#include "const.h"
#include "video.h"

#include "el_torito.h"
#include "ptrace.h"
#include "floppy.h"
#include "harddisk.h"
#include "disk.h"

/* cut && paste of umide.h */
/* structure returned by HDIO_GET_IDENTITY, as per ANSI ATA2 rev.2f spec */
struct hd_driveid {
	uint16_t config;	/* lots of obsolete bit flags */
	uint16_t cyls;	/* "physical" cyls */
	uint16_t reserved2;	/* reserved (word 2) */
	uint16_t heads;	/* "physical" heads */
	uint16_t track_bytes;	/* unformatted bytes per track */
	uint16_t sector_bytes;	/* unformatted bytes per sector */
	uint16_t sectors;	/* "physical" sectors per track */
	uint16_t vendor0;	/* vendor unique */
	uint16_t vendor1;	/* vendor unique */
	uint16_t vendor2;	/* vendor unique */
	uint8_t serial_no[20];	/* 0 = not_specified */
	uint16_t buf_type;
	uint16_t buf_size;    /* 512 byte increments; 0 = not_specified */
	uint16_t ecc_bytes;   /* for r/w long cmds; 0 = not_specified */
	uint8_t fw_rev[8];	/* 0 = not_specified */
	uint8_t model[40];	/* 0 = not_specified */
	uint8_t max_multsect;	/* 0=not_implemented */
	uint8_t vendor3;		/* vendor unique */
	uint16_t dword_io;	/* 0=not_implemented; 1=implemented */
	uint8_t vendor4;		/* vendor unique */
	uint8_t capability;    /* bits 0:DMA 1:LBA 2:IORDYsw 3:IORDYsup */
	uint16_t reserved50;	/* reserved (word 50) */
	uint8_t vendor5;		/* vendor unique */
	uint8_t tPIO;		/* 0=slow, 1=medium, 2=fast */
	uint8_t vendor6;		/* vendor unique */
	uint8_t tDMA;		/* 0=slow, 1=medium, 2=fast */
	uint16_t field_valid;	/* bits 0:cur_ok 1:eide_ok */
	uint16_t cur_cyls;	/* logical cylinders */
	uint16_t cur_heads;	/* logical heads */
	uint16_t cur_sectors;	/* logical sectors per track */
	uint16_t cur_capacity0;	/* logical total sectors on drive */
	uint16_t cur_capacity1;	/*  (2 words, misaligned int)     */
	uint8_t multsect;		/* current multiple sector count */
	uint8_t multsect_valid;	/* when (bit0==1) multsect is ok */
	uint32_t lba_capacity;		/* total number of sectors */
	uint16_t dma_1word;	/* single-word dma info */
	uint16_t dma_mword;	/* multiple-word dma info */
	uint16_t eide_pio_modes;	/* bits 0:mode3 1:mode4 */
	uint16_t eide_dma_min;	/* min mword dma cycle time (ns) */
	uint16_t eide_dma_time;/* recommended mword dma cycle time (ns) */
	uint16_t eide_pio;	/* min cycle time (ns), no IORDY  */
	uint16_t eide_pio_iordy;	/* min cycle time (ns), with IORDY */
	uint16_t words69_70[2];	/* reserved words 69-70 */
	/* HDIO_GET_IDENTITY currently returns only words 0 through 70 */
	uint16_t words71_74[4];	/* reserved words 71-74 */
	uint16_t queue_depth;	/*  */
	uint16_t words76_79[4];	/* reserved words 76-79 */
	uint16_t major_rev_num;	/*  */
	uint16_t minor_rev_num;	/*  */
	/* bits 0:Smart 1:Security 2:Removable 3:PM */
	uint16_t command_set_1;
	uint16_t command_set_2;	/* bits 14:Smart Enabled 13:0 zero */
	uint16_t cfsse;	/* command set-feature supported extensions */
	uint16_t cfs_enable_1;	/* command set-feature enabled */
	uint16_t cfs_enable_2;	/* command set-feature enabled */
	uint16_t csf_default;	/* command set-feature default */
	uint16_t dma_ultra;	/*  */
	uint16_t word89;	/* reserved (word 89) */
	uint16_t word90;	/* reserved (word 90) */
	uint16_t CurAPMvalues;	/* current APM values */
	uint16_t word92;	/* reserved (word 92) */
	uint16_t hw_config;	/* hardware config */
	uint16_t words94_125[32];	/* reserved words 94-125 */
	uint16_t last_lun;	/* reserved (word 126) */
	uint16_t word127;	/* reserved (word 127) */
	uint16_t dlf;	/* device lock function
				 * 15:9        reserved
				 * 8   security level 1:max 0:high
				 * 7:6 reserved
				 * 5   enhanced erase
				 * 4   expire
				 * 3   frozen
				 * 2   locked
				 * 1   en/disabled
				 * 0   capability
				 */
	uint16_t csfo;	/* current set features options
				 * 15:4        reserved
				 * 3   auto reassign
				 * 2   reverting
				 * 1   read-look-ahead
				 * 0   write cache
				 */
	uint16_t words130_155[26]; /* reserved vendor words 130-155 */
	uint16_t word156;
	uint16_t words157_159[3];  /* reserved vendor words 157-159 */
	uint16_t words160_255[95]; /* reserved words 160-255 */
} __attribute__((__packed__));


static uint8_t
get_identify_ide(
	uint16_t cmd, /* ATA COMMAND */
	uint8_t contr,
	uint8_t drive,
	uint16_t *buf
)
{
	uint16_t port;
	uint8_t status;
	uint16_t i;

	assert(cmd == WIN_IDENTIFY || cmd == WIN_PIDENTIFY);
	assert(/* 0 <= contr && */ contr < 6);
	assert(/* 0 <= drive && */ drive < 2);

	port = const_get(ide_port_table[contr]);

	/* Wait while busy. */
	for (i = 0; ; i++) {
		if (i == 0xffff) {
			return 1;
		}
		status = inb(port + 0x206);
		if (! (status & IDE_BUSY_STAT)) {
			break;
		}
	}

	/* Send command. */
	outb(drive << 4, port + 6);	/* unit */
	outb(cmd, port + 7);		/* command */

	/* Wait while controller busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);

	if (! (status & IDE_DRQ_STAT)
	 || (status & IDE_ERR_STAT)) {
		/* Strange response -> command not recognized. */
		return 1;
	}

	/* Read info. */
	for (i = 0; i < 512 / sizeof(uint16_t); i++) {
		buf[i] = inw(port);
	}

	return 0;
}

static void
detecting(uint16_t contr, uint16_t unit)
{
	bprintf("  Detecting IDE ");
	if (contr == 0) {
		if (unit == 0) {
			bprintf("Primary Master  ");
		} else {
			bprintf("Primary Slave   ");
		}
	} else if (contr == 1) {
		if (unit == 0) {
			bprintf("Secondary Master");
		} else {
			bprintf("Secondary Slave ");
		}
	} else {
		bprintf("%d/%d             ", contr, unit);
	}
	bprintf("... ");
}


void
disk_init(void)
{
	uint8_t hd;
	uint8_t cd;
	uint16_t contr;
	uint16_t unit;
	uint16_t i;

	hd = 0x80;
	cd = 0xe0;
	ebda_put(harddrives, 0);
	ebda_put(cdcount, 0);
	var_put(hd_num, 0);
	cmos_put(hd_config, 0x00);
	cmos_put(hd_extconfig0, 0);
	cmos_put(hd_extconfig1, 0);
	cmos_put(hd_extconfig2, 0);
	cmos_put(hd_extconfig3, 0);

	for (contr = 0; contr < 6; contr++) {
		for (unit = 0; unit < 2; unit++) {
			const uint16_t nr = contr * 2 + unit;
			uint16_t buffer[512 / sizeof(uint16_t)];
			struct hd_driveid *id = (struct hd_driveid *) buffer;
			int type; /* none == -1; disk == 0; cdrom == 1*/

			if (contr < 2) {
				detecting(contr, unit);
			}

			/* First try disk. */
			if (! get_identify_ide(WIN_IDENTIFY,
					contr, unit, buffer)) {
				/* Harddisk */
				type = 0;
				ebda_put(devices[nr].drive, hd);
				ebda_put(devices[nr].pchs_cylinders, id->cyls);
				ebda_put(devices[nr].pchs_heads, id->heads);
				ebda_put(devices[nr].pchs_spt, id->sectors);
				ebda_put(devices[nr].lchs_cylinders, id->cur_cyls);
				ebda_put(devices[nr].lchs_heads, id->cur_heads);
				ebda_put(devices[nr].lchs_spt, id->cur_sectors);

				ebda_put(devices[nr].sectors, id->lba_capacity);
				ebda_put(devices[nr].type, 0);

				if (hd - 0x80 < 2) {
					ebda_put(hdpt[hd - 0x80].cyls, id->cyls);
					ebda_put(hdpt[hd - 0x80].heads, id->heads);
					ebda_put(hdpt[hd - 0x80].reserved0, 0);
					ebda_put(hdpt[hd - 0x80].precomp, -1);
					ebda_put(hdpt[hd - 0x80].reserved1, 0);
					ebda_put(hdpt[hd - 0x80].control, (8 < id->heads) << 3);
					ebda_put(hdpt[hd - 0x80].reserved2, 0);
					ebda_put(hdpt[hd - 0x80].reserved3, 0);
					ebda_put(hdpt[hd - 0x80].reserved4, 0);
					ebda_put(hdpt[hd - 0x80].landing, id->cyls - 1);
					ebda_put(hdpt[hd - 0x80].secs, id->sectors);
					ebda_put(hdpt[hd - 0x80].reserved5, 0);
				}
				if (hd == 0x80) {
					cmos_put(hd_config,
						0xf0 | cmos_get(hd_config));
					cmos_put(hd_extconfig0, 0x2f);
				} else if (hd == 0x81) {
					cmos_put(hd_config,
						0x0f | cmos_get(hd_config));
					cmos_put(hd_extconfig1, 0x2f);
				} else if (hd == 0x82) {
					cmos_put(hd_extconfig2, 0x2f);
				} else if (hd == 0x83) {
					cmos_put(hd_extconfig2, 0x2f);
				}
				if (nr == 0) {
					/* Primary Master */
					cmos_put(hd_config_user0.cyls,
							id->cyls);
					cmos_put(hd_config_user0.heads,
							id->heads);
					cmos_put(hd_config_user0.precomp,
							-1);
					cmos_put(hd_config_user0.landing,
							id->cyls - 1);
					cmos_put(hd_config_user0.secs,
							id->sectors);
				} else if (nr == 1) {
					/* Primary Slave */
					cmos_put(hd_config_user1.cyls,
							id->cyls);
					cmos_put(hd_config_user1.heads,
							id->heads);
					cmos_put(hd_config_user1.precomp,
							-1);
					cmos_put(hd_config_user1.landing,
							id->cyls - 1);
					cmos_put(hd_config_user1.secs,
							id->sectors);
				} else if (nr == 2) {
					/* Secondary Master */
					cmos_put(hd_config_user2.cyls,
							id->cyls);
					cmos_put(hd_config_user2.heads,
							id->heads);
					cmos_put(hd_config_user2.precomp,
							-1);
					cmos_put(hd_config_user2.landing,
							id->cyls - 1);
					cmos_put(hd_config_user2.secs,
							id->sectors);
				} else if (nr == 3) {
					/* Secondary Slave */
					cmos_put(hd_config_user3.cyls,
							id->cyls);
					cmos_put(hd_config_user3.heads,
							id->heads);
					cmos_put(hd_config_user3.precomp,
							-1);
					cmos_put(hd_config_user3.landing,
							id->cyls - 1);
					cmos_put(hd_config_user3.secs,
							id->sectors);
				}

				ebda_put(hdidmap[hd - 0x80], nr);

				hd++;

				ebda_put(harddrives, ebda_get(harddrives) + 1);
				var_put(hd_num, var_get(hd_num) + 1);

			} else if (! get_identify_ide(WIN_PIDENTIFY,
					contr, unit, buffer)) {
				/* CD-ROM */
				type = 1;
				ebda_put(devices[nr].drive, cd);
				ebda_put(devices[nr].type, 1);

				ebda_put(cdidmap[cd - 0xe0], nr);

				cd++;

				ebda_put(cdcount, ebda_get(cdcount) + 1);

			} else {
				/* nothing */
				ebda_put(devices[nr].drive, 0xff);
				ebda_put(devices[nr].type, 0xfe);
				type = -1;
			}

			if (2 <= contr && type != -1) {
				detecting(contr, unit);
			}

			if (contr < 2 && type == -1) {
				bprintf("None\n");
			} else if (type != -1) {
				/* display device name */
				for (i = 0; i < 40; i++) {
					if (i % 2) {
						putchar(id->model[i-1]);
					} else {
						putchar(id->model[i+1]);
					}
				}
				putchar('\n');
			}
		}
	}

	for ( ; hd < 0x80 + MAX_BIOS_HDS; hd++) {
		ebda_put(hdidmap[hd - 0x80], MAX_BIOS_HDS);
	}
	for ( ; cd < 0xe0 + MAX_BIOS_HDS; cd++) {
		ebda_put(cdidmap[cd - 0xe0], MAX_BIOS_HDS);
	}

	/* int $0x41 vector is harddisk0_param_table. */
	put_word(0x0000, 0x41 * 4 + 0, offsetof(struct ebda, hdpt[0]));
	put_word(0x0000, 0x41 * 4 + 2, EBDA_SEG);

	/* int $0x46 vector is harddisk1_param_table. */
	put_word(0x0000, 0x46 * 4 + 0, offsetof(struct ebda, hdpt[1]));
	put_word(0x0000, 0x46 * 4 + 2, EBDA_SEG);

	/*
	 * WARNING: Minix3 needs HDs accessed last.
	 * So just recalibrate HDs.
	 */
	for (hd = 0x80; hd < 0x80 + MAX_BIOS_HDS; hd++) {
		uint16_t nr;

		nr = ebda_get(hdidmap[hd - 0x80]);

		if (nr != MAX_BIOS_HDS) {
			recalibrate(nr / 2, nr % 2);
		}
	}
}

#endif /* INIT_RM */
