/*
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

CODE16;

#include "assert.h"
#include "stdio.h"

/* ==================== RUNTIME_RM ==================== */
#ifdef RUNTIME_RM

#include "ptrace.h"
#include "io.h"
#include "segment.h"
#include "var.h"
#include "const.h"
#include "mouse.h"

#define CMD_RSTSCL	0xe6
#define CMD_SETSCL	0xe7
#define CMD_SETRES	0xe8
#define CMD_STSREQ	0xe9
#define CMD_RESET	0xff

#define REPLY_ACK	0xfa
#define REPLY_RESEND	0xfe


static uint8_t
mouse_send_kbc_command(uint8_t cmd, uint8_t data)
{
	uint32_t retry;
	uint8_t val;

	for (retry = 0; ; retry++) {

		asm volatile ( "cli\n" );

		if (retry == 0x10000) {
			return 4;	/* timeout */
		}

		val = inb_p(0x64);

		if (((val >> 5) & 1) == 1) {
			/* old mouse data available
			 * -> read it and reset receive buffer index */
			inb(0x60);
			ebda_put(mouse_flags_1, ebda_get(mouse_flags_1) & ~0x7);

		} else if (((val >> 0) & 1) == 1) {
			/* keyboard data available 
			 * -> give kbd driver a chance to handle it
			 *    and try again */
			asm volatile ( "sti\n" );

		} else if (((val >> 1) & 1) == 0) {
			/* neither mouse nor keyboard data available
			 * nor controller command busy
			 * -> send our command and data byte */

			outb_p(cmd, 0x64);
			outb(data, 0x60);
			return 0;
		}
	}
}

/* convenience macro to send one byte directly to the mouse */
#define mouse_send(c) mouse_send_kbc_command(0xd4, c)

static uint8_t
mouse_wait_ack(void)
{
	uint32_t timeout;
	uint8_t val;

	for (timeout = 0; ; timeout++) {
		if (timeout == 0x120000) {
			return 4;	/* Timeout. */
		}

		val = inb_p(0x64);	/* Keyboard status register. */

		if (((val >> 5) & 1) == 0) {
			continue;
		}

		val = inb(0x60);	/* Keyboard data register. */
		if (val == REPLY_ACK) {
			return 0;	/* Got ACK. */

		} else if (val == REPLY_RESEND) {
			return 3;	/* Got RESEND. */

		} else {
			/* What error code should be sent in this case? */
			/* FIXME VOSSI */
			return 3;	/* Unknown code. */
		}
	}
}

static uint8_t
mouse_recv(uint8_t *valp)
{
	uint32_t timeout;
	uint8_t val;

	for (timeout = 0; ; timeout++) {
		if (timeout == 0x10000) {
			return 3;
		}

		val = inb_p(0x64);	/* Keyboard status register. */
		if (((val >> 5) & 1) == 0) {
			continue;
		}

		val = inb(0x60);	/* Keyboard data register. */
		*valp = val;
		return 0;
	}
}

static ALWAYS_INLINE void 
mouse_success(struct regs *regs)
{
	AH = 0;	
	F &= ~(1 << 0);	/* Clear carry. */
}

static ALWAYS_INLINE void
mouse_error(struct regs *regs, uint8_t err)
{
	AH = err;	
	F |= (1 << 0);	/* Set carry. */
}

/*
 * Disable/enable mouse.
 *
 * BH:	0: disable mouse
 *	1: enable mouse
 */
static ALWAYS_INLINE void
bios_15_c200(struct regs *regs)
{
	uint8_t err;

	if (BH == 0) {
		/* Disable mouse. */
		err = mouse_send(0xf5);
		if (err) {
			mouse_error(regs, err);
			return;
		}

		err = mouse_wait_ack();
		if (err) {
			mouse_error(regs, err);
			return;
		}

	} else if (BH == 1) {
		/* Enable mouse. */
		if ((ebda_get(mouse_flags_2) & 0x80) != 0x80) {
			mouse_error(regs, 5);	/* no handler present */
			return;
		}

		err = mouse_send(0xf4);
		if (err) {
			mouse_error(regs, err);
			return;
		}

		err = mouse_wait_ack();
		if (err) {
			mouse_error(regs, err);
			return;
		}

	} else {
		mouse_error(regs, 1); /* Invalid subfunction. */
		return;
	}

	mouse_success(regs);
}

/*
 * Reset mouse.
 */
static ALWAYS_INLINE void
bios_15_c201(struct regs *regs)
{
	uint8_t err;

	err = mouse_send(CMD_RESET);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_wait_ack();
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_recv(&BL);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_recv(&BH);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	mouse_success(regs);
}

/*
 * Set mouse sample rate.
 *
 * BH:	0:  10 reports/sec
 *	1:  20 reports/sec
 *	2:  40 reports/sec
 *	3:  60 reports/sec
 *	4:  80 reports/sec
 *	5: 100 reports/sec (default)
 *	6: 200 reports/sec
 */
static ALWAYS_INLINE void
bios_15_c202(struct regs *regs)
{
	static const uint8_t const sample_rate[] = {
		10,
		20,
		40,
		60,
		80,
		100,
		200
	};
	uint8_t err;

	/* Check argument. */
	if (/* BH < 0 || */ 6 < BH) {
		mouse_error(regs, 2);	/* Invalid argument. */
		return;
	}

	err = mouse_send(0xf3);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_wait_ack();
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_send(sample_rate[BH]);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_wait_ack();
	if (err) {
		mouse_error(regs, err);
		return;
	}

	mouse_success(regs);
}

/*
 * Set mouse resolution.
 *
 * BH:	0:   25 dpi, 1 count  per millimeter
 *	1:   50 dpi, 2 counts per millimeter
 *	2:  100 dpi, 4 counts per millimeter
 *	3:  200 dpi, 8 counts per millimeter
 */
static ALWAYS_INLINE void
bios_15_c203(struct regs *regs)
{
	uint8_t err;

	/* Check argument. */
	if (/* BH < 0 || */ 3 < BH) {
		mouse_error(regs, 2);	/* Invalid argument. */
		return;
	}

	err = mouse_send(CMD_SETRES);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_wait_ack();
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_send(BH);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_wait_ack();
	if (err) {
		mouse_error(regs, err);
		return;
	}

	mouse_success(regs);
}

/*
 * Get Device ID
 */
static ALWAYS_INLINE void
bios_15_c204(struct regs *regs)
{
	uint8_t err;

	err = mouse_send(0xf2);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_wait_ack();
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_recv(&BH);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	mouse_success(regs);
}

/*
 * Initialize mouse.
 */
static ALWAYS_INLINE void
bios_15_c205(struct regs *regs)
{
	uint8_t val;
	uint8_t err;

	/* Check argument. */
	if (BH <= 0 || 8 < BH) {
		mouse_error(regs, 2);	/* Invalid argument. */
		return;
	}

	/* Initialize package size field. */
	val = ebda_get(mouse_flags_2);
	val &= ~(7 << 0);
	val |= BH - 1;
	ebda_put(mouse_flags_2, val);

	/* Send mouse reset command. */
	err = mouse_send(CMD_RESET);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_wait_ack();
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_recv(&val);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	err = mouse_recv(&val);
	if (err) {
		mouse_error(regs, err);
		return;
	}
	
	mouse_success(regs);
}

/*
 * Return status and set scaling factor.
 *
 * In:	BH:	0:	Return status
 * 		1:	Set scaling factor to 1:1
 * 		2:	Set scaling factor to 2:1
 *
 * Out:	AH:	Error code
 * 	BL:	Bit 7:	Reserved
 * 		Bit 6:	0: Stream
 * 			1: Requests
 * 		Bit 5:	Transmission enabled
 * 		Bit 4:	0: Scaling factor 1:1
 * 			1: Scaling factor 2:1
 * 		Bit 3:	Reserved
 * 		Bit 2:	Button pressed (left)
 * 		Bit 1:	Button pressed (middle)
 * 		Bit 0:	Button pressed (right)
 * 	CL:	0:	1 count  per millimeter
 * 		1:	2 counts per millimeter
 * 		2:	4 counts per millimeter
 * 		3:	8 counts per millimeter
 * 	DL:	# reports per second
 * 	F:	C:	0: Success
 * 			1: Error
 */
static ALWAYS_INLINE void
bios_15_c206(struct regs *regs)
{
	uint8_t err;

	if (BH == 0) {
		/* Return status. */
		err = mouse_send(CMD_STSREQ);
		if (err) {
			mouse_error(regs, err);
			return;
		}

		err = mouse_wait_ack();
		if (err) {
			mouse_error(regs, err);
			return;
		}

		err = mouse_recv(&BL);
		if (err) {
			mouse_error(regs, err);
			return;
		}

		err = mouse_recv(&CL);
		if (err) {
			mouse_error(regs, err);
			return;
		}

		err = mouse_recv(&DL);
		if (err) {
			mouse_error(regs, err);
			return;
		}

	} else if (BH == 1) {
		/* Set scaling factor to 1:1. */
		err = mouse_send(CMD_RSTSCL);
		if (err) {
			mouse_error(regs, err);
			return;
		}

		err = mouse_wait_ack();
		if (err) {
			mouse_error(regs, err);
			return;
		}

	} else if (BH == 2) {
		/* Set scaling factor to 2:1. */
		err = mouse_send(CMD_SETSCL);
		if (err) {
			mouse_error(regs, err);
			return;
		}

		err = mouse_wait_ack();
		if (err) {
			mouse_error(regs, err);
			return;
		}

	} else {
		mouse_error(regs, 1);	/* Invalid subfunction. */
		return;
	}

	mouse_success(regs);
}

/*
 * Set mouse handler.
 */
static ALWAYS_INLINE void
bios_15_c207(struct regs *regs)
{
	ebda_put(mouse_driver_offset, BX);
	ebda_put(mouse_driver_seg, ES);

	if (BX == 0x0000 && ES == 0x0000) {
		ebda_put(mouse_flags_2, ebda_get(mouse_flags_2) & ~0x80);
	} else {
		ebda_put(mouse_flags_2, ebda_get(mouse_flags_2) | 0x80);
	}

	mouse_success(regs);
}

void
bios_15_c2xx(struct regs *regs)
{
	uint8_t err;

	if (AL > 0x07) {
		mouse_error(regs, 1);
		return;
	}

	/* disable AUXINT */
	err =  mouse_send_kbc_command(0x60, 0x41);
	if (err) {
		mouse_error(regs, err);
		return;
	}

	if (AL == 0x00) {
		bios_15_c200(regs);
	} else if (AL == 0x01) {
		bios_15_c201(regs);
	} else if (AL == 0x02) {
		bios_15_c202(regs);
	} else if (AL == 0x03) {
		bios_15_c203(regs);
	} else if (AL == 0x04) {
		bios_15_c204(regs);
	} else if (AL == 0x05) {
		bios_15_c205(regs);
	} else if (AL == 0x06) {
		bios_15_c206(regs);
	} else if (AL == 0x07) {
		bios_15_c207(regs);
	}

	/* re-enable AUXINT */
	err = mouse_send_kbc_command(0x60, 0x43);
	if (err) {
		mouse_error(regs, err);
		return;
	}
}

/*
 * Interrupt 12 generated by PS/2 mouse.
 */
void
bios_74(void)
{
	uint8_t count_act;
	uint8_t count_max;
	uint8_t val;

	val = inb(0x60);	/* Get mouse data. */

	count_act = ebda_get(mouse_flags_1) & 0x7;
	count_max = ebda_get(mouse_flags_2) & 0x7;

	ebda_put(mouse_flags_1, ebda_get(mouse_flags_1) & ~0x7);
	ebda_put(mouse_data[count_act], val);

	if (count_act < count_max) {
		/*
		 * Wait for more data bytes.
		 */
		count_act++;
		ebda_put(mouse_flags_1,
				(ebda_get(mouse_flags_1) & ~0x7) | count_act);

	} else if (ebda_get(mouse_flags_2) & 0x80) {
		/*
		 * Got "max" + 1 data bytes.
		 * Far call enabled.
		 * Call far routine.
		 */
		static const uint8_t const pos[8][8] = {
			{ 6, },
			{ 6, 7, },
			{ 6, 4, 2, },
			{ 6, 7, 4, 2, },
			{ 6, 4, 2, 5, 3, },
			{ 6, 7, 4, 2, 5, 3, },
			{ 6, 4, 2, 0, 5, 3, 1, },
			{ 6, 7, 4, 2, 0, 5, 3, 1 }
		};
		uint32_t data[2];
		uint16_t n;

		data[0] = 0;
		data[1] = 0;
		for (n = 0; n <= count_max; n++) {
			uint8_t v;
			uint8_t p;

			v = ebda_get(mouse_data[n]);
			p = const_get(pos[count_max][n]);
			data[p / 4] |= v << ((p % 4) * 8);
		}

		asm volatile (
			"pushw %%ds\n"
			"mov %0, %%ds\n"
			"pushl %1\n"
			"pushl %2\n"
			"lcallw *(%3)\n"
			"addw $8, %%sp\n"
			"popw %%ds\n"
			: /* No output. */
			: "r" (EBDA_SEG),
			  "r" (data[1]),
			  "r" (data[0]),
			  "r" ((unsigned long)
				  &((struct ebda *) 0)->mouse_driver_offset)
		);
	}

	eoi();
}

#endif /* RUNTIME_RM */
/* =================== REAL-MODE INIT ========================= */
#ifdef INIT_RM

CODE16;

#include "ptrace.h"
#include "io.h"
#include "var.h"
#include "mouse.h"

/*
 * Detect mouse and initialize variables.
 */
void
mouse_init(void)
{
	uint8_t val;
	uint16_t equipment;

	/*
	 * Check if we have a mouse.
	 */
	val = 1;	/* FIXME VOSSI */

	if (! val) {
		return;
	}

	/*
	 * We have a mouse.
	 */
	/* Set PS/2 bit in sys_conf variable. */
	equipment = var_get(sys_conf);
	equipment |= (1 << 2);
	var_put(sys_conf, equipment);

	/* initialize mouse flags in ebda */
	ebda_put(mouse_flags_1, 0);
	ebda_put(mouse_flags_2, 0);

	/* Enable interrupt 12 (Bit 4 in slave PIC). */
	val = inb_p(0xa1);
	val &= ~(1 << 4);
	outb_p(val, 0xa1);

	/* Enable mouse port on keyboard controller.
	 * NOTE: If this is enabled and IRQ12 is used by anything else
	 * (PCI/USB for instance), then neither mouse NOR KEYBOARD (!)
	 * will work under windows. */
	outb(0x60,0x64);
	outb(0x47,0x60);
}

#endif /* INIT_RM */
