/*
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

/* ==================== RUNTIME_RM || ==================== */
#if defined(RUNTIME_RM)

CODE16;

#include "ptrace.h"
#include "assert.h"
#include "stdio.h"
#include "traps.h"

void
bios_exception(struct regs *regs)
{
	dprintf("Exception at %04x:%04x\n",
			regs->cs, regs->ip);
	dprintf("ds=%04x es=%04x\n",
			regs->ds, regs->es);
	dprintf("eax=%08lx ebx=%08lx ecx=%08lx edx=%08lx\n",
			regs->a.eax, regs->b.ebx, regs->c.ecx, regs->d.edx);
	dprintf(" di=    %04x  si=    %04x  bp=    %04x esp=%08lx\n",
			regs->di, regs->si, regs->bp, (unsigned long) regs);
	assert(0);
}

void
bios_dummy(struct regs *regs)
{
	/* Nothing to do... */
}

#endif /* RUNTIME_RM */
/* ==================== INIT_RM ==================== */
#if defined(INIT_RM)

CODE16;

#include "io.h"
#include "segment.h"
#include "entry.h"
#include "traps.h"
#include "ptrace.h"
#include "assert.h"

void
pic_init(uint8_t pic0_vec_start, uint8_t pic1_vec_start)
{
	/*
	 * Setup interrupt controller.
	 * (See linux-2.4.19/arch/i386/kernel/i8259.c.)
	 */

	/* Setup master. */
	outb_p(0x11, 0x20);	/* ICW1: select 8259A-1 init */
	outb_p(pic0_vec_start, 0x21);	
				/* ICW2: 8259A-1 IR0-7 mapped to 0x08-0x0f */
	outb_p(0x04, 0x21);	/* 8259A-1 (the master) has a slave on IR2 */
	outb_p(0x01, 0x21);	/* master expects normal EOI */

	/* Setup slave. */
	outb_p(0x11, 0xA0);	/* ICW1: select 8259A-2 init */
	outb_p(pic1_vec_start, 0xA1);
				/* ICW2: 8259A-2 IR0-7 mapped to 0x70-0x77 */
	outb_p(0x02, 0xA1);	/* 8259A-2 is a slave on master's IR2 */
	outb_p(0x01, 0xA1);	/* slave expects normal EOI */

#if 0
	udelay(100);		/* wait for 8259A to initialize */
#endif

	outb(0xff, 0x21);	/* disable master IRQs */
	outb(0xff, 0xA1);	/* disable slave IRQs */

	/*
	 * WARNING:
	 * Interrupts 4 (and 3?) should be edge triggered.
	 * At least minix3 needs it edge triggered.
	 */
	outb((1 << ( 7-0))  /* Interrupt 7 is level triggered. */
	   | (0 << ( 6-0))  /* Interrupt 6 is edge  triggered. */
	   | (1 << ( 5-0))  /* Interrupt 5 is level triggered. */
	   | (0 << ( 4-0))  /* Interrupt 4 is edge triggered. */
	   | (0 << ( 3-0))  /* Interrupt 3 is edge triggered. */
	   | (0 << ( 2-0))  /* Interrupt 2 is edge  triggered. */
	   | (0 << ( 1-0))  /* Interrupt 1 is edge  triggered. */
	   | (0 << ( 0-0)), /* Interrupt 0 is edge  triggered. */
		0x04d0);
	/*
	 * WARNING:
	 * Interrupts 15/14 should be edge triggered.
	 * At least ubcd-disk-samsung1 experiment needs it edge triggered.
	 */
	outb((0 << (15-8))  /* Interrupt 15 is edge triggered. */
	   | (0 << (14-8))  /* Interrupt 14 is edge triggered. */
	   | (0 << (13-8))  /* Interrupt 13 is edge  triggered. */
	   | (1 << (12-8))  /* Interrupt 12 is level triggered. */
	   | (1 << (11-8))  /* Interrupt 11 is level triggered. */
	   | (1 << (10-8))  /* Interrupt 10 is level triggered. */
	   | (1 << ( 9-8))  /* Interrupt  9 is level triggered. */
	   | (0 << ( 8-8)), /* Interrupt  8 is edge  triggered. */
		0x04d1);
	
	/*
	 * Enable kbd, timer, floppy and IDE interrupts.
	 */
	outb_p((1 << 7) /* Disable parallel. */
	     | (0 << 6) /* Enable floppy. */
	     | (1 << 5) /* ? */
	     | (1 << 4) /* Disable serial-1 */
	     | (1 << 3) /* Disable serial-0 */
	     | (0 << 2) /* Enable cascade. */
	     | (0 << 1) /* Enable KBD. */
	     | (0 << 0), /* Enable timer. */
		0x21);
	outb_p((0 << (15-8)) /* Enable ide-1. */
	     | (0 << (14-8)) /* Enable ide-0. */
	     | (1 << (13-8)) /* Disable FPU. */
	     | (1 << (12-8)) /* Disable mouse. */
	     | (1 << (11-8)) /* Disable ?. */
	     | (1 << (10-8)) /* Disable ?. */
	     | (1 << ( 9-8)) /* Disable ?. */
	     | (1 << ( 8-8)), /* Disable RTC. */
		0xa1);
}


static void
set_RM_gate(unsigned int n, void *addr)
{
	/*
	 * ATTENTION!
	 * Adresses in interrupt vector table must be
	 * calculated by special macros due to flaws
	 * in the boot programs of some DOSoid systems.
	 */
	put_word(0x0000, n * 4 + 0, PTR_OFF_VEC(addr));
	put_word(0x0000, n * 4 + 2, PTR_SEG_VEC(addr));
}

static void
set_RM_exception_gate(unsigned int n, void *addr)
{
	set_RM_gate(n, addr);
}

static void
set_RM_interrupt_gate(unsigned int n, void *addr)
{
	set_RM_gate(n, addr);
}

static void
set_RM_system_gate(unsigned int n, void *addr)
{
	set_RM_gate(n, addr);
}

static void
setup_RM_idt(void)
{
	int irq;

	set_RM_exception_gate(0x00, irq00);	/* Exception 0x00 - 0x07 */
	set_RM_exception_gate(0x01, irq01);
	set_RM_exception_gate(0x02, irq02);
	set_RM_exception_gate(0x03, irq03);
	set_RM_exception_gate(0x04, irq04);
	set_RM_exception_gate(0x05, irq05);
	set_RM_exception_gate(0x06, irq06);
	set_RM_exception_gate(0x07, irq07);

	set_RM_interrupt_gate(0x08, irq08);	/* Interrupt 0x00 - 0x07 */
	set_RM_interrupt_gate(0x09, irq09);
	/* irq02: not used - cascade */
	set_RM_interrupt_gate(0x0b, irq0b);
	set_RM_interrupt_gate(0x0c, irq0c);
	set_RM_interrupt_gate(0x0d, irq0d);
	set_RM_interrupt_gate(0x0e, irq0e);
	set_RM_interrupt_gate(0x0f, irq0f);

	set_RM_system_gate(0x10, irqdummy);	/* System calls 0x10 - 0x1f */
	set_RM_system_gate(0x11, irq11);
	set_RM_system_gate(0x12, irq12);
	set_RM_system_gate(0x13, irq13);
	set_RM_system_gate(0x14, irq14);
	set_RM_system_gate(0x15, irq15);
	set_RM_system_gate(0x16, irq16);
	set_RM_system_gate(0x17, irq17);
	set_RM_system_gate(0x18, irq18);
	set_RM_system_gate(0x19, irq19);
	set_RM_system_gate(0x1a, irq1a);
	set_RM_system_gate(0x1b, irqdummy);
	set_RM_system_gate(0x1c, irq1c);

	for (irq = 0x1d; irq < 0x70; irq++) {
		set_RM_system_gate(irq, irqdummy);
	}

	set_RM_interrupt_gate(0x70, irq70);	/* Interrupt 0x08 - 0x0f */
	set_RM_interrupt_gate(0x71, irq71);
	set_RM_interrupt_gate(0x72, irq72);
	set_RM_interrupt_gate(0x73, irq73);
	set_RM_interrupt_gate(0x74, irq74);
	set_RM_interrupt_gate(0x75, irq75);
	set_RM_interrupt_gate(0x76, irq76);
	set_RM_interrupt_gate(0x77, irq77);

	for (irq = 0x78; irq < 0xff; irq++) {	/* System calls. */
		set_RM_system_gate(irq, irqdummy);
	}

#ifdef CONFIG_SMP_SUPPORT
	set_RM_interrupt_gate(0xff, irqff);
#endif
}

void
trap_init(void)
{
	/* Setup RM IDT. */
	setup_RM_idt();

	/* Initialize PIC for RM */
	pic_init(0x08, 0x70);

	/*
	 * Setup clock.
	 * (ch 0: timer interrupt, ch 1: legacy RAM refresh timer.)
	 */
	/* 1193180 Hz divided by 0xffff will lead to 18.2067 HZ. */
	outb_p(0x34, 0x43);     /* binary, mode 2, LSB/MSB, ch 0 */
	outb_p(0xff, 0x40);     /* LSB */
	outb_p(0xff, 0x40);     /* MSB */

	/* 1193180 Hz divided by 0x12 (=18) will lead to a refresh
	 * delay of approx. 15 us */ 
	outb_p(0x74, 0x43);	/* binary, mode 2, LSB/MSB, ch 1 */
	outb_p(0x12, 0x41);	/* LSB */
	outb_p(0x00, 0x41);	/* MSB */
}

#endif /* INIT_RM */
