#include "EncyclopediaDetailPanel.h"

#include "CUIControls.h"
#include "DesignWnd.h"
#include "Encyclopedia.h"
#include "GraphControl.h"
#include "../universe/Condition.h"
#include "../universe/Universe.h"
#include "../universe/Tech.h"
#include "../universe/ShipDesign.h"
#include "../universe/Building.h"
#include "../universe/Planet.h"
#include "../universe/System.h"
#include "../universe/Ship.h"
#include "../universe/Fleet.h"
#include "../universe/Special.h"
#include "../universe/Species.h"
#include "../universe/Field.h"
#include "../universe/Effect.h"
#include "../universe/Predicates.h"
#include "../Empire/Empire.h"
#include "../Empire/EmpireManager.h"
#include "../util/EnumText.h"
#include "../util/i18n.h"
#include "../util/Logger.h"
#include "../util/OptionsDB.h"
#include "../util/Directories.h"
#include "../client/human/HumanClientApp.h"
#include "../combat/CombatLogManager.h"
#include "../combat/CombatEvents.h"
#include "../parse/Parse.h"

#include <GG/DrawUtil.h>
#include <GG/StaticGraphic.h>
#include <GG/GUI.h>

namespace {
    const GG::X TEXT_MARGIN_X(3);
    const GG::Y TEXT_MARGIN_Y(3);
    void    AddOptions(OptionsDB& db)
    { db.Add("UI.autogenerated-effects-descriptions", UserStringNop("OPTIONS_DB_AUTO_EFFECT_DESC"),  false,  Validator<bool>()); }
    bool temp_bool = RegisterOptions(&AddOptions);

    const std::string EMPTY_STRING;
    const std::string INCOMPLETE_DESIGN = "incomplete design";
    const std::string UNIVERSE_OBJECT = "universe object";
    const std::string PLANET_SUITABILITY_REPORT = "planet suitability report";
    const std::string COMBAT_LOG = "combat log";
    const std::string GRAPH = "data graph";
}

namespace {
    const Encyclopedia& GetEncyclopedia() {
        static Encyclopedia encyclopedia;
        return encyclopedia;
    }

    std::string LinkTaggedText(const std::string& tag, const std::string& stringtable_entry)
    { return "<" + tag + " " + stringtable_entry + ">" + UserString(stringtable_entry) + "</" + tag + ">"; }

    std::string LinkTaggedIDText(const std::string& tag, int id, const std::string& text)
    { return "<" + tag + " " + boost::lexical_cast<std::string>(id) + ">" + text + "</" + tag + ">"; }

    std::string PediaDirText(const std::string& dir_name) {
        std::string retval;
        const Encyclopedia& encyclopedia = GetEncyclopedia();
        const Universe& universe = GetUniverse();
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();
        const ObjectMap& objects = Objects();

        std::multimap<std::string, std::string> sorted_entries_list;

        if (dir_name == "ENC_INDEX") {
            sorted_entries_list.insert(std::make_pair(UserString("ENC_SHIP_PART"),      LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_SHIP_PART") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_SHIP_HULL"),      LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_SHIP_HULL") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_TECH"),           LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_TECH") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_BUILDING_TYPE"),  LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_BUILDING_TYPE") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_SPECIAL"),        LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_SPECIAL") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_SPECIES"),        LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_SPECIES") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_FIELD_TYPE"),     LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_FIELD_TYPE") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_EMPIRE"),         LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_EMPIRE") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_SHIP_DESIGN"),    LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_SHIP_DESIGN") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_SHIP"),           LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_SHIP") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_MONSTER"),        LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_MONSTER") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_FLEET"),          LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_FLEET") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_PLANET"),         LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_PLANET") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_BUILDING"),       LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_BUILDING") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_SYSTEM"),         LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_SYSTEM") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_FIELD"),          LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_FIELD") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_GRAPH"),          LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_GRAPH") + "\n"));
            sorted_entries_list.insert(std::make_pair(UserString("ENC_GALAXY_SETUP"),   LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, "ENC_GALAXY_SETUP") + "\n"));

            for (std::map<std::string, std::vector<EncyclopediaArticle> >::const_iterator it = encyclopedia.articles.begin();
                 it != encyclopedia.articles.end(); ++it)
            { sorted_entries_list.insert(std::make_pair(UserString(it->first),  LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, it->first) + "\n")); }

        } else if (dir_name == "ENC_SHIP_PART") {
            const PartTypeManager& part_type_manager = GetPartTypeManager();
            for (PartTypeManager::iterator it = part_type_manager.begin(); it != part_type_manager.end(); ++it)
                sorted_entries_list.insert(std::make_pair(UserString(it->first),  LinkTaggedText(VarText::SHIP_PART_TAG, it->first) + "\n"));

        } else if (dir_name == "ENC_SHIP_HULL") {
            const HullTypeManager& hull_type_manager = GetHullTypeManager();
            for (HullTypeManager::iterator it = hull_type_manager.begin(); it != hull_type_manager.end(); ++it)
                sorted_entries_list.insert(std::make_pair(UserString(it->first),  LinkTaggedText(VarText::SHIP_HULL_TAG, it->first) + "\n"));

        } else if (dir_name == "ENC_TECH") {
            std::vector<std::string> tech_names = GetTechManager().TechNames();
            std::map<std::string, std::string> userstring_tech_names;
            // sort tech names by user-visible name, so names are shown alphabetically in UI
            for (std::vector<std::string>::const_iterator it = tech_names.begin(); it != tech_names.end(); ++it)
                userstring_tech_names[UserString(*it)] = *it;
            for (std::map<std::string, std::string>::const_iterator it = userstring_tech_names.begin(); it != userstring_tech_names.end(); ++it)
                sorted_entries_list.insert(std::make_pair(UserString(it->first),  LinkTaggedText(VarText::TECH_TAG, it->second) + "\n"));

        } else if (dir_name == "ENC_BUILDING_TYPE") {
            const BuildingTypeManager& building_type_manager = GetBuildingTypeManager();
            for (BuildingTypeManager::iterator it = building_type_manager.begin(); it != building_type_manager.end(); ++it)
                sorted_entries_list.insert(std::make_pair(UserString(it->first),  LinkTaggedText(VarText::BUILDING_TYPE_TAG, it->first) + "\n"));

        } else if (dir_name == "ENC_SPECIAL") {
            const std::vector<std::string> special_names = SpecialNames();
            for (std::vector<std::string>::const_iterator it = special_names.begin(); it != special_names.end(); ++it)
                sorted_entries_list.insert(std::make_pair(UserString(*it),  LinkTaggedText(VarText::SPECIAL_TAG, *it) + "\n"));

        } else if (dir_name == "ENC_SPECIES") {
            const SpeciesManager& species_manager = GetSpeciesManager();
            for (SpeciesManager::iterator it = species_manager.begin(); it != species_manager.end(); ++it)
                sorted_entries_list.insert(std::make_pair(UserString(it->first),  LinkTaggedText(VarText::SPECIES_TAG, it->first) + "\n"));

        } else if (dir_name == "ENC_FIELD_TYPE") {
            const FieldTypeManager& fields_manager = GetFieldTypeManager();
            for (FieldTypeManager::iterator it = fields_manager.begin(); it != fields_manager.end(); ++it)
                sorted_entries_list.insert(std::make_pair(UserString(it->first),  LinkTaggedText(VarText::FIELD_TYPE_TAG, it->first) + "\n"));

        } else if (dir_name == "ENC_EMPIRE") {
            const EmpireManager& empire_manager = Empires();
            for (EmpireManager::const_iterator it = empire_manager.begin(); it != empire_manager.end(); ++it)
                sorted_entries_list.insert(std::make_pair(UserString(it->second->Name()),  LinkTaggedIDText(VarText::EMPIRE_ID_TAG, it->first, it->second->Name()) + "\n"));

        } else if (dir_name == "ENC_SHIP_DESIGN") {
            for (Universe::ship_design_iterator it = universe.beginShipDesigns(); it != universe.endShipDesigns(); ++it)
                if (!it->second->IsMonster())
                    sorted_entries_list.insert(std::make_pair(UserString(it->second->Name()),  LinkTaggedIDText(VarText::DESIGN_ID_TAG, it->first, it->second->Name()) + "\n"));

        } else if (dir_name == "ENC_SHIP") {
            for (ObjectMap::const_iterator<Ship> ship_it = objects.const_begin<Ship>();
                 ship_it != objects.const_end<Ship>(); ++ship_it)
            {
                TemporaryPtr<const Ship> ship = *ship_it;
                const std::string& ship_name = ship->PublicName(client_empire_id);
                sorted_entries_list.insert(std::make_pair(ship_name,  LinkTaggedIDText(VarText::SHIP_ID_TAG, ship->ID(), ship_name) + "  "));
            }

        } else if (dir_name == "ENC_MONSTER") {
            // monster objects
            std::vector<TemporaryPtr<const Ship> > monsters;
            for (ObjectMap::const_iterator<Ship> ship_it = objects.const_begin<Ship>();
                 ship_it != objects.const_end<Ship>(); ++ship_it)
            {
                if ((*ship_it)->IsMonster())
                    monsters.push_back(*ship_it);
            }
            if (!monsters.empty()) {
                retval += UserString("MONSTER_OBJECTS");
                for (std::vector<TemporaryPtr<const Ship> >::const_iterator ship_it = monsters.begin(); ship_it != monsters.end(); ++ship_it) {
                    TemporaryPtr<const Ship> ship = *ship_it;
                    const std::string& ship_name = ship->PublicName(client_empire_id);
                    retval += LinkTaggedIDText(VarText::SHIP_ID_TAG, ship->ID(), ship_name) + "  ";
                }
            } else {
                retval += UserString("NO_MONSTER_OBJECTS");
            }

            // monster types
            retval += "\n\n" + UserString("MONSTER_TYPES") + "\n";
            for (Universe::ship_design_iterator it = universe.beginShipDesigns(); it != universe.endShipDesigns(); ++it)
                if (it->second->IsMonster())
                    sorted_entries_list.insert(std::make_pair(UserString(it->second->Name()),  LinkTaggedIDText(VarText::DESIGN_ID_TAG, it->first, it->second->Name()) + "\n"));

        } else if (dir_name == "ENC_FLEET") {
            for (ObjectMap::const_iterator<Fleet> fleet_it = objects.const_begin<Fleet>();
                 fleet_it != objects.const_end<Fleet>(); ++fleet_it)
            {
                TemporaryPtr<const Fleet> fleet = *fleet_it;
                const std::string& flt_name = fleet->PublicName(client_empire_id);
                sorted_entries_list.insert(std::make_pair(flt_name,  LinkTaggedIDText(VarText::FLEET_ID_TAG, fleet->ID(), flt_name) + "  "));
            }

        } else if (dir_name == "ENC_PLANET") {
            for (ObjectMap::const_iterator<Planet> planet_it = objects.const_begin<Planet>();
                 planet_it != objects.const_end<Planet>(); ++planet_it)
            {
                TemporaryPtr<const Planet> planet = *planet_it;
                const std::string& plt_name = planet->PublicName(client_empire_id);
                sorted_entries_list.insert(std::make_pair(plt_name,  LinkTaggedIDText(VarText::PLANET_ID_TAG, planet->ID(), plt_name) + "  "));
            }

        } else if (dir_name == "ENC_BUILDING") {
            for (ObjectMap::const_iterator<Building> building_it = objects.const_begin<Building>();
                 building_it != objects.const_end<Building>(); ++building_it)
            {
                TemporaryPtr<const Building> building = *building_it;
                const std::string& bld_name = building->PublicName(client_empire_id);
                sorted_entries_list.insert(std::make_pair(bld_name,  LinkTaggedIDText(VarText::BUILDING_ID_TAG, building->ID(), bld_name) + "  "));
            }

        } else if (dir_name == "ENC_SYSTEM") {
            for (ObjectMap::const_iterator<System> system_it = objects.const_begin<System>();
                 system_it != objects.const_end<System>(); ++system_it)
            {
                TemporaryPtr<const System> system = *system_it;
                const std::string& sys_name = system->ApparentName(client_empire_id);
                sorted_entries_list.insert(std::make_pair(sys_name,  LinkTaggedIDText(VarText::SYSTEM_ID_TAG, system->ID(), sys_name) + "  "));
            }

        } else if (dir_name == "ENC_FIELD") {
            for (ObjectMap::const_iterator<Field> field_it = objects.const_begin<Field>();
                 field_it != objects.const_end<Field>(); ++field_it)
            {
                const std::string& field_name = field_it->Name();
                sorted_entries_list.insert(std::make_pair(field_name,
                    LinkTaggedIDText(VarText::FIELD_ID_TAG, field_it->ID(), field_name) + "  "));
            }

        } else if (dir_name == "ENC_GRAPH") {
            const std::map<std::string, std::map<int, std::map<int, double> > >&
                stat_records = universe.GetStatRecords();
            for (std::map<std::string, std::map<int, std::map<int, double> > >::const_iterator
                 it = stat_records.begin(); it != stat_records.end(); ++it)
            { sorted_entries_list.insert(std::make_pair(UserString(it->first),   LinkTaggedText(TextLinker::GRAPH_TAG, it->first) + "\n")); }

        } else {
            // list categories
            std::map<std::string, std::vector<EncyclopediaArticle> >::const_iterator category_it =
                encyclopedia.articles.find(dir_name);
            if (category_it != encyclopedia.articles.end()) {
                const std::vector<EncyclopediaArticle>& articles = category_it->second;
                for (std::vector<EncyclopediaArticle>::const_iterator article_it = articles.begin();
                     article_it != articles.end(); ++article_it)
                { sorted_entries_list.insert(std::make_pair(UserString(article_it->name),  LinkTaggedText(TextLinker::ENCYCLOPEDIA_TAG, article_it->name) + "\n")); }
            }
        }

        // add sorted entries
        for (std::multimap<std::string, std::string>::const_iterator it = sorted_entries_list.begin();
             it != sorted_entries_list.end(); ++it)
        { retval += it->second; }

        return retval;
    }

}

namespace {
    struct DescriptionVisitor : public boost::static_visitor<>
    {
        DescriptionVisitor(ShipPartClass part_class, std::string& description) :
            m_class(part_class),
            m_description(description)
        {}
        void operator()(const float& d) const {
            std::string desc_string;

            switch(m_class){
            case PC_FUEL:
            case PC_TROOPS:
            case PC_COLONY:
                desc_string += UserString("PART_DESC_CAPACITY");
                break;
            case PC_SHIELD:
                desc_string = UserString("PART_DESC_SHIELD_STRENGTH");
                break;
            case PC_DETECTION:
                desc_string = UserString("PART_DESC_DETECTION");
                break;
            default:
                desc_string = UserString("PART_DESC_STRENGTH");
                break;
            }
            m_description +=
                str(FlexibleFormat(desc_string) % d);
        }
        void operator()(const DirectFireStats& stats) const {
            m_description +=
                str(FlexibleFormat(UserString("PART_DESC_DIRECT_FIRE_STATS"))
                    % stats.m_damage
                    % stats.m_ROF
                    % stats.m_range);
        }
        void operator()(const LRStats& stats) const {
            m_description +=
                str(FlexibleFormat(UserString("PART_DESC_LR_STATS"))
                    % stats.m_damage
                    % stats.m_ROF
                    % stats.m_range
                    % stats.m_speed
                    % stats.m_structure
                    % stats.m_stealth
                    % stats.m_capacity);
        }
        void operator()(const FighterStats& stats) const {
            m_description +=
                str(FlexibleFormat(UserString("PART_DESC_FIGHTER_STATS"))
                    % (stats.m_type == BOMBER ? UserString("BOMBER") : UserString("INTERCEPTOR"))
                    % stats.m_anti_ship_damage
                    % stats.m_anti_fighter_damage
                    % stats.m_launch_rate
                    % stats.m_speed
                    % stats.m_stealth
                    % stats.m_structure
                    % stats.m_detection
                    % stats.m_capacity);
        }

        const ShipPartClass m_class;
        std::string& m_description;
    };

    
    class ColorByOwner: public LinkDecorator{
    public:
        
        virtual std::string Decorate ( const std::string& href, const std::string& content ) const{
            GG::Clr color = ClientUI::DefaultLinkColor();
            Empire* empire = HrefToEmpire(href);
            if ( empire ) {
                color = empire->Color();
}
            return GG::RgbaTag(color) + content + "</rgba>";
        }

    private:
        Empire* HrefToEmpire ( const std::string& target ) const{
            int id = try_to_int(target);
            if ( id ) {
                TemporaryPtr<const UniverseObject> object = Objects().Object(id);
                if ( object && object->Owner() != ALL_EMPIRES ) {
                    return Empires().Lookup(object->Owner());
                }
            }
            return NULL;
        }
    };
}

std::list <std::pair<std::string, std::string> >            EncyclopediaDetailPanel::m_items = std::list<std::pair<std::string, std::string> >(0);
std::list <std::pair<std::string, std::string> >::iterator  EncyclopediaDetailPanel::m_items_it = m_items.begin();

EncyclopediaDetailPanel::EncyclopediaDetailPanel(GG::X w, GG::Y h, GG::Flags<GG::WndFlag> flags) :
    CUIWnd("", GG::X1, GG::Y1, w - 1, h - 1, flags),
    m_name_text(0),
    m_cost_text(0),
    m_summary_text(0),
    m_description_box(0),
    m_icon(0),
    m_other_icon(0),
    m_graph(0)
{
    const int PTS = ClientUI::Pts();
    const int NAME_PTS = PTS*3/2;
    const int COST_PTS = PTS;
    const int SUMMARY_PTS = PTS*4/3;
    const int ICON_WIDTH(32);
    boost::shared_ptr<GG::Font> font = ClientUI::GetFont();

    m_name_text =       new GG::TextControl(GG::X0, GG::Y0, GG::X(10), GG::Y(10), "", ClientUI::GetBoldFont(NAME_PTS),  ClientUI::TextColor());
    m_cost_text =       new GG::TextControl(GG::X0, GG::Y0, GG::X(10), GG::Y(10), "", ClientUI::GetFont(COST_PTS),      ClientUI::TextColor());
    m_summary_text =    new GG::TextControl(GG::X0, GG::Y0, GG::X(10), GG::Y(10), "", ClientUI::GetFont(SUMMARY_PTS),   ClientUI::TextColor());

    m_index_button =    new GG::Button(GG::X0, GG::Y0, GG::X(ICON_WIDTH), GG::Y(ICON_WIDTH), "", font, GG::CLR_WHITE, GG::CLR_ZERO);
    m_back_button =     new GG::Button(GG::X0, GG::Y0, GG::X(ICON_WIDTH), GG::Y(ICON_WIDTH), "", font, GG::CLR_WHITE, GG::CLR_ZERO);
    m_next_button =     new GG::Button(GG::X0, GG::Y0, GG::X(ICON_WIDTH), GG::Y(ICON_WIDTH), "", font, GG::CLR_WHITE, GG::CLR_ZERO);
    m_back_button->Disable();
    m_next_button->Disable();

    m_index_button->SetUnpressedGraphic(GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "uparrownormal.png"      )));
    m_index_button->SetPressedGraphic  (GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "uparrowclicked.png"     )));
    m_index_button->SetRolloverGraphic (GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "uparrowmouseover.png"   )));
    m_back_button->SetUnpressedGraphic (GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "leftarrownormal.png"    )));
    m_back_button->SetPressedGraphic   (GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "leftarrowclicked.png"   )));
    m_back_button->SetRolloverGraphic  (GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "leftarrowmouseover.png" )));
    m_next_button->SetUnpressedGraphic (GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "rightarrownormal.png"   )));
    m_next_button->SetPressedGraphic   (GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "rightarrowclicked.png"  )));
    m_next_button->SetRolloverGraphic  (GG::SubTexture(ClientUI::GetTexture( ClientUI::ArtDir() / "icons" / "buttons" / "rightarrowmouseover.png")));

    CUILinkTextMultiEdit* desc_box = new CUILinkTextMultiEdit(GG::X0, GG::Y0, GG::X(10), GG::Y(10), "", GG::MULTI_WORDBREAK | GG::MULTI_READ_ONLY);
    GG::Connect(desc_box->LinkClickedSignal,        &EncyclopediaDetailPanel::HandleLinkClick,          this);
    GG::Connect(desc_box->LinkDoubleClickedSignal,  &EncyclopediaDetailPanel::HandleLinkDoubleClick,    this);
    GG::Connect(desc_box->LinkRightClickedSignal,   &EncyclopediaDetailPanel::HandleLinkDoubleClick,    this);
    GG::Connect(m_index_button->LeftClickedSignal,  &EncyclopediaDetailPanel::OnIndex,                  this);
    GG::Connect(m_back_button->LeftClickedSignal,   &EncyclopediaDetailPanel::OnBack,                   this);
    GG::Connect(m_next_button->LeftClickedSignal,   &EncyclopediaDetailPanel::OnNext,                   this);
    desc_box->SetDecorator(VarText::SHIP_ID_TAG, new ColorByOwner());
    desc_box->SetDecorator(VarText::PLANET_ID_TAG, new ColorByOwner());
    m_description_box = desc_box;
    m_description_box->SetColor(GG::CLR_ZERO);
    m_description_box->SetInteriorColor(ClientUI::CtrlColor());

    m_graph = new GraphControl(GG::X0, GG::Y0, GG::X1, GG::Y1);
    m_graph->ShowPoints(false);
    AttachChild(m_graph);

    AttachChild(m_name_text);
    AttachChild(m_cost_text);
    AttachChild(m_summary_text);
    AttachChild(m_description_box);
    AttachChild(m_index_button);
    AttachChild(m_back_button);
    AttachChild(m_next_button);

    SetChildClippingMode(ClipToWindow);

    DoLayout();

    MoveChildUp(m_graph);

    AddItem(TextLinker::ENCYCLOPEDIA_TAG, "ENC_INDEX");
}

void EncyclopediaDetailPanel::DoLayout() {
    const int PTS = ClientUI::Pts();
    const int NAME_PTS = PTS*3/2;
    const int COST_PTS = PTS;
    const int SUMMARY_PTS = PTS*4/3;

    const int ICON_SIZE = 12 + NAME_PTS + COST_PTS + SUMMARY_PTS;

    const int BTN_WIDTH = 24;

    // name
    GG::Pt ul = GG::Pt();
    GG::Pt lr = ul + GG::Pt(Width() - 1, GG::Y(NAME_PTS + 4));
    m_name_text->SizeMove(ul, lr);

    // cost / turns
    ul += GG::Pt(GG::X0, m_name_text->Height());
    lr = ul + GG::Pt(Width(), GG::Y(COST_PTS + 4));
    m_cost_text->SizeMove(ul, lr);

    // one line summary
    ul += GG::Pt(GG::X0, m_cost_text->Height());
    lr = ul + GG::Pt(Width(), GG::Y(SUMMARY_PTS + 4));
    m_summary_text->SizeMove(ul, lr);

    // main verbose description (fluff, effects, unlocks, ...)
    ul = GG::Pt(BORDER_LEFT, ICON_SIZE + TEXT_MARGIN_Y + 1);
    lr = GG::Pt(Width() - BORDER_RIGHT, Height() - BORDER_BOTTOM*3 - PTS - 4);
    m_description_box->SizeMove(ul, lr);

    // graph
    m_graph->SizeMove(ul + GG::Pt(GG::X1, GG::Y1), lr - GG::Pt(GG::X1, GG::Y1));

    // "back" button
    ul = GG::Pt(Width() - BORDER_RIGHT*3 - BTN_WIDTH * 3 - 8, Height() - BORDER_BOTTOM*2 - PTS);
    lr = GG::Pt(Width() - BORDER_RIGHT*3 - BTN_WIDTH * 2 - 8, Height() - BORDER_BOTTOM*2);
    m_back_button->SizeMove(ul, lr);

    // "up" button
    ul = GG::Pt(Width() - BORDER_RIGHT*3 - BTN_WIDTH * 2 - 4, Height() - BORDER_BOTTOM*3 - PTS);
    lr = GG::Pt(Width() - BORDER_RIGHT*3 - BTN_WIDTH - 4, Height() - BORDER_BOTTOM*3);
    m_index_button->SizeMove(ul, lr);

    // "next" button
    ul = GG::Pt(Width() - BORDER_RIGHT*3 - BTN_WIDTH, Height() - BORDER_BOTTOM*2 - PTS);
    lr = GG::Pt(Width() - BORDER_RIGHT*3, Height() - BORDER_BOTTOM*2);
    m_next_button->SizeMove(ul, lr);

    // icon
    if (m_icon) {
        ul = GG::Pt(GG::X1, GG::Y1);
        lr = ul + GG::Pt(GG::X(ICON_SIZE), GG::Y(ICON_SIZE));
        m_icon->SizeMove(ul, lr);
    }
    // other icon
    if (m_other_icon) {
        lr = GG::Pt(Width() - BORDER_RIGHT, GG::Y(ICON_SIZE + 1));
        ul = lr - GG::Pt(GG::X(ICON_SIZE), GG::Y(ICON_SIZE));
        m_other_icon->SizeMove(ul, lr);
    }

    PositionButtons();
    MoveChildUp(m_close_button);    // so it's over top of the top-right icon
}

void EncyclopediaDetailPanel::SizeMove(const GG::Pt& ul, const GG::Pt& lr) {
    GG::Pt old_size = GG::Wnd::Size();

    CUIWnd::SizeMove(ul, lr);

    if (old_size != GG::Wnd::Size())
        DoLayout();
}

GG::Pt EncyclopediaDetailPanel::ClientUpperLeft() const
{ return GG::Wnd::UpperLeft(); }

void EncyclopediaDetailPanel::Render() {
    GG::Pt ul = UpperLeft();
    GG::Pt lr = LowerRight();
    const GG::Y ICON_SIZE = m_summary_text->Bottom() - m_name_text->Top();
    GG::Pt cl_ul = ul + GG::Pt(BORDER_LEFT, ICON_SIZE + BORDER_BOTTOM); // BORDER_BOTTOM is the size of the border at the bottom of a standard CUIWnd
    GG::Pt cl_lr = lr - GG::Pt(BORDER_RIGHT, BORDER_BOTTOM);

    // draw background and outer border
    AngledCornerRectangle(ul, lr, ClientUI::WndColor(), ClientUI::WndOuterBorderColor(),
                          OUTER_EDGE_ANGLE_OFFSET, 1, false, !m_resizable); // show notched bottom-right corner if not resizable, pointed corner if resizable

    // use GL to draw the lines
    glDisable(GL_TEXTURE_2D);
    // draw inner border, including extra resize-tab lines

    glBegin(GL_LINE_STRIP);
        glColor(ClientUI::WndInnerBorderColor());
        glVertex(cl_ul.x, cl_ul.y);
        glVertex(cl_lr.x, cl_ul.y);
        if (m_resizable) {
            glVertex(cl_lr.x, cl_lr.y - INNER_BORDER_ANGLE_OFFSET);
            glVertex(cl_lr.x - INNER_BORDER_ANGLE_OFFSET, cl_lr.y);
        } else {
            glVertex(cl_lr.x, cl_lr.y);
        }
        glVertex(cl_ul.x, cl_lr.y);
        glVertex(cl_ul.x, cl_ul.y);
    glEnd();
    if (m_resizable) {
        glBegin(GL_LINES);
            // draw the extra lines of the resize tab
            GG::Clr tab_lines_colour = m_mouse_in_resize_tab ? ClientUI::WndInnerBorderColor() : ClientUI::WndOuterBorderColor();
            glColor(tab_lines_colour);

            glVertex(cl_lr.x, cl_lr.y - RESIZE_HASHMARK1_OFFSET);
            glVertex(cl_lr.x - RESIZE_HASHMARK1_OFFSET, cl_lr.y);

            glVertex(cl_lr.x, cl_lr.y - RESIZE_HASHMARK2_OFFSET);
            glVertex(cl_lr.x - RESIZE_HASHMARK2_OFFSET, cl_lr.y);
        glEnd();
    }
    glEnable(GL_TEXTURE_2D);
}

void EncyclopediaDetailPanel::HandleLinkClick(const std::string& link_type, const std::string& data) {
    using boost::lexical_cast;
    try {
        if (link_type == VarText::PLANET_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToPlanet(lexical_cast<int>(data));
            this->SetPlanet(lexical_cast<int>(data));

        } else if (link_type == VarText::SYSTEM_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToSystem(lexical_cast<int>(data));
        } else if (link_type == VarText::FLEET_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToFleet(lexical_cast<int>(data));
        } else if (link_type == VarText::SHIP_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToShip(lexical_cast<int>(data));
        } else if (link_type == VarText::BUILDING_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToBuilding(lexical_cast<int>(data));
        } else if (link_type == VarText::FIELD_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToField(lexical_cast<int>(data));

        } else if (link_type == VarText::COMBAT_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToCombatLog(lexical_cast<int>(data));

        } else if (link_type == VarText::EMPIRE_ID_TAG) {
            this->SetEmpire(lexical_cast<int>(data));
        } else if (link_type == VarText::DESIGN_ID_TAG) {
            this->SetDesign(lexical_cast<int>(data));
        } else if (link_type == VarText::PREDEFINED_DESIGN_TAG) {
            if (const ShipDesign* design = GetPredefinedShipDesign(data))
                this->SetDesign(design->ID());

        } else if (link_type == VarText::TECH_TAG) {
            this->SetTech(data);
        } else if (link_type == VarText::BUILDING_TYPE_TAG) {
            this->SetBuildingType(data);
        } else if (link_type == VarText::FIELD_TYPE_TAG) {
            this->SetFieldType(data);
        } else if (link_type == VarText::SPECIAL_TAG) {
            this->SetSpecial(data);
        } else if (link_type == VarText::SHIP_HULL_TAG) {
            this->SetHullType(data);
        } else if (link_type == VarText::SHIP_PART_TAG) {
            this->SetPartType(data);
        } else if (link_type == VarText::SPECIES_TAG) {
            this->SetSpecies(data);
        } else if (link_type == TextLinker::ENCYCLOPEDIA_TAG) {
            this->SetText(data, false);
        } else if (link_type == TextLinker::GRAPH_TAG) {
            this->SetGraph(data);
        }

    } catch (const boost::bad_lexical_cast&) {
        Logger().errorStream() << "EncyclopediaDetailPanel::HandleLinkClick caught lexical cast exception for link type: " << link_type << " and data: " << data;
    }
}

void EncyclopediaDetailPanel::HandleLinkDoubleClick(const std::string& link_type, const std::string& data) {
    using boost::lexical_cast;
    try {
        if (link_type == VarText::PLANET_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToPlanet(lexical_cast<int>(data));
        } else if (link_type == VarText::SYSTEM_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToSystem(lexical_cast<int>(data));
        } else if (link_type == VarText::FLEET_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToFleet(lexical_cast<int>(data));
        } else if (link_type == VarText::SHIP_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToShip(lexical_cast<int>(data));
        } else if (link_type == VarText::BUILDING_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToBuilding(lexical_cast<int>(data));

        } else if (link_type == VarText::EMPIRE_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToEmpire(lexical_cast<int>(data));
        } else if (link_type == VarText::DESIGN_ID_TAG) {
            ClientUI::GetClientUI()->ZoomToShipDesign(lexical_cast<int>(data));
        } else if (link_type == VarText::PREDEFINED_DESIGN_TAG) {
            if (const ShipDesign* design = GetPredefinedShipDesign(data))
                ClientUI::GetClientUI()->ZoomToShipDesign(design->ID());

        } else if (link_type == VarText::TECH_TAG) {
            ClientUI::GetClientUI()->ZoomToTech(data);
        } else if (link_type == VarText::BUILDING_TYPE_TAG) {
            ClientUI::GetClientUI()->ZoomToBuildingType(data);
        } else if (link_type == VarText::SPECIAL_TAG) {
            ClientUI::GetClientUI()->ZoomToSpecial(data);
        } else if (link_type == VarText::SHIP_HULL_TAG) {
            ClientUI::GetClientUI()->ZoomToShipHull(data);
        } else if (link_type == VarText::SHIP_PART_TAG) {
            ClientUI::GetClientUI()->ZoomToShipPart(data);
        } else if (link_type == VarText::SPECIES_TAG) {
            ClientUI::GetClientUI()->ZoomToSpecies(data);

        } else if (link_type == TextLinker::ENCYCLOPEDIA_TAG) {
            this->SetText(data, false);
        } else if (link_type == TextLinker::GRAPH_TAG) {
            this->SetGraph(data);
        }
    } catch (const boost::bad_lexical_cast&) {
        Logger().errorStream() << "EncyclopediaDetailPanel::HandleLinkDoubleClick caught lexical cast exception for link type: " << link_type << " and data: " << data;
    }
}

namespace {
    int DefaultLocationForEmpire(int empire_id) {
        const Empire* empire = Empires().Lookup(empire_id);
        if (!empire) {
            Logger().debugStream() << "DefaultLocationForEmpire: Unable to get empire with ID: " << empire_id;
            return INVALID_OBJECT_ID;
        }
        // get a location where the empire might build something.
        TemporaryPtr<const UniverseObject> location = GetUniverseObject(empire->CapitalID());
        // no capital?  scan through all objects to find one owned by this empire
        // TODO: only loop over planets?
        // TODO: pass in a location condition, and pick a location that matches it if possible
        if (!location) {
            for (ObjectMap::const_iterator<> obj_it = Objects().const_begin(); obj_it != Objects().const_end(); ++obj_it) {
                if (obj_it->OwnedBy(empire_id)) {
                    location = *obj_it;
                    break;
                }
            }
        }
        return location ? location->ID() : INVALID_OBJECT_ID;
    }

    std::vector<std::string> TechsThatUnlockItem(const ItemSpec& item) {
        std::vector<std::string> retval;

        const TechManager& tm = GetTechManager();
        for (TechManager::iterator it = tm.begin(); it != tm.end(); ++it) {
            const Tech* tech = *it;
            if (!tech) continue;
            const std::string& tech_name = tech->Name();

            const std::vector<ItemSpec>& unlocked_items = tech->UnlockedItems();
            bool found_item = false;
            for (std::vector<ItemSpec>::const_iterator item_it = unlocked_items.begin();
                 item_it != unlocked_items.end(); ++item_it)
            {
                if (*item_it == item) {
                    found_item = true;
                    break;
                }
            }
            if (found_item)
                retval.push_back(tech_name);
        }

        return retval;
    }

    const std::string& GeneralTypeOfObject(UniverseObjectType obj_type) {
        switch (obj_type) {
        case OBJ_SHIP:          return UserString("ENC_SHIP");          break;
        case OBJ_FLEET:         return UserString("ENC_FLEET");         break;
        case OBJ_PLANET:        return UserString("ENC_PLANET");        break;
        case OBJ_BUILDING:      return UserString("ENC_BUILDING");      break;
        case OBJ_SYSTEM:        return UserString("ENC_SYSTEM");        break;
        case OBJ_FIELD:         return UserString("END_FIELD");         break;
        case OBJ_POP_CENTER:    return UserString("ENC_POP_CENTER");    break;
        case OBJ_PROD_CENTER:   return UserString("END_PROD_CENTER");   break;
        default:                return EMPTY_STRING;
        }
    }

    const std::string& LinkTag(UniverseObjectType obj_type) {
        switch (obj_type) {
        case OBJ_SHIP:          return VarText::SHIP_ID_TAG;        break;
        case OBJ_FLEET:         return VarText::FLEET_ID_TAG;       break;
        case OBJ_PLANET:        return VarText::PLANET_ID_TAG;      break;
        case OBJ_BUILDING:      return VarText::BUILDING_ID_TAG;    break;
        case OBJ_SYSTEM:        return VarText::SYSTEM_ID_TAG;      break;
        case OBJ_FIELD:
        default:                return EMPTY_STRING;
        }
    }

    /// Creates a link tag of the appropriate type for object_id,
    /// with the content being the public name from the point of view of client_empire_id.
    /// Returns not_found if object_id is not found.
    std::string PublicNameLink ( int client_empire_id, int object_id, std::string not_found ) {
        TemporaryPtr<const UniverseObject> object = GetUniverseObject ( object_id );
        if (object) {
            const std::string& name = object->PublicName(client_empire_id);
            const std::string& tag = LinkTag(object->ObjectType());
            return LinkTaggedIDText(tag, object_id, name);
        } else {
            return not_found;
        }
    }

    void RefreshDetailPanelPediaTag(        const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        detailed_description = PediaDirText(item_name);
        if (!detailed_description.empty()) {
            // attempt to treat as a directory
            name = UserString(item_name);

        } else if (item_name == "ENC_GALAXY_SETUP") {
            const GalaxySetupData& gsd = ClientApp::GetApp()->GetGalaxySetupData();

            name = UserString("ENC_GALAXY_SETUP");

            detailed_description += str(FlexibleFormat(UserString("ENC_GALAXY_SETUP_SETTINGS"))
                % gsd.m_seed
                % boost::lexical_cast<std::string>(gsd.m_size)
                % TextForGalaxyShape(gsd.m_shape)
                % TextForGalaxySetupSetting(gsd.m_age)
                % TextForGalaxySetupSetting(gsd.m_starlane_freq)
                % TextForGalaxySetupSetting(gsd.m_planet_density)
                % TextForGalaxySetupSetting(gsd.m_specials_freq)
                % TextForGalaxySetupSetting(gsd.m_monster_freq)
                % TextForGalaxySetupSetting(gsd.m_native_freq)
                % TextForAIAggression(gsd.m_ai_aggr));

        } else {
            // couldn't find a directory; look up in custom encyclopedia entries
            const Encyclopedia& encyclopedia = GetEncyclopedia();

            for (std::map<std::string, std::vector<EncyclopediaArticle> >::const_iterator
                     category_it = encyclopedia.articles.begin();
                 category_it != encyclopedia.articles.end(); ++category_it)
            {
                const std::vector<EncyclopediaArticle>& articles = category_it->second;
                for (std::vector<EncyclopediaArticle>::const_iterator article_it = articles.begin();
                     article_it != articles.end(); ++article_it)
                {
                    if (article_it->name != item_name)
                        continue;
                    name = UserString(article_it->name);
                    detailed_description = UserString(article_it->description);
                    general_type = UserString(article_it->category);
                    specific_type = UserString(article_it->short_description);
                    texture = ClientUI::GetTexture(ClientUI::ArtDir() / article_it->icon, true);
                    break;
                }
            }
        }
    }

    void RefreshDetailPanelShipPartTag(     const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        const PartType* part = GetPartType(item_name);
        if (!part) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find part with name " << item_name;
            return;
        }
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        // Ship Parts
        name = UserString(item_name);
        texture = ClientUI::PartIcon(item_name);
        int default_location_id = DefaultLocationForEmpire(client_empire_id);
        turns = part->ProductionTime(client_empire_id, default_location_id);
        cost = part->ProductionCost(client_empire_id, default_location_id);
        cost_units = UserString("ENC_PP");
        general_type = UserString("ENC_SHIP_PART");
        specific_type = UserString(boost::lexical_cast<std::string>(part->Class()));

        std::string stat_description;
        boost::apply_visitor(DescriptionVisitor(part->Class(), stat_description), part->Stats());
        detailed_description += UserString(part->Description()) + "\n\n" + stat_description;

        std::string slot_types_list;
        if (part->CanMountInSlotType(SL_EXTERNAL))
            slot_types_list += UserString("SL_EXTERNAL") + "   ";
        if (part->CanMountInSlotType(SL_INTERNAL))
            slot_types_list += UserString("SL_INTERNAL") + "   ";
        if (part->CanMountInSlotType(SL_CORE))
            slot_types_list += UserString("SL_CORE");
        if (!slot_types_list.empty())
            detailed_description += "\n\n" +UserString("ENC_SHIP_PART_CAN_MOUNT_IN_SLOT_TYPES") + slot_types_list;

        std::vector<std::string> unlocked_by_techs = TechsThatUnlockItem(ItemSpec(UIT_SHIP_PART, item_name));
        if (!unlocked_by_techs.empty()) {
            detailed_description += "\n\n" + UserString("ENC_UNLOCKED_BY");
            for (std::vector<std::string>::const_iterator unlock_tech_it = unlocked_by_techs.begin();
                 unlock_tech_it != unlocked_by_techs.end(); ++unlock_tech_it)
            { detailed_description += LinkTaggedText(VarText::TECH_TAG, *unlock_tech_it) + "  "; }
            detailed_description += "\n\n";
        }

        if (GetOptionsDB().Get<bool>("UI.autogenerated-effects-descriptions")) {
            if (part->Location())
                detailed_description += str(FlexibleFormat(UserString("ENC_LOCATION_CONDITION_STR")) % part->Location()->Description());
            if (!part->Effects().empty())
                detailed_description += str(FlexibleFormat(UserString("ENC_EFFECTS_STR")) % EffectsDescription(part->Effects()));
        }
    }

    void RefreshDetailPanelShipHullTag(     const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        const HullType* hull = GetHullType(item_name);
        if (!hull) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find hull with name " << item_name;
            return;
        }
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        // Ship Hulls
        name = UserString(item_name);
        texture = ClientUI::HullTexture(item_name);
        int default_location_id = DefaultLocationForEmpire(client_empire_id);
        turns = hull->ProductionTime(client_empire_id, default_location_id);
        cost = hull->ProductionCost(client_empire_id, default_location_id);
        cost_units = UserString("ENC_PP");
        general_type = UserString("ENC_SHIP_HULL");

        detailed_description += UserString(hull->Description()) + "\n\n" + str(FlexibleFormat(UserString("HULL_DESC"))
            % hull->StarlaneSpeed()
            % hull->Fuel()
            % hull->BattleSpeed()
            % hull->Structure());

        std::vector<std::string> unlocked_by_techs = TechsThatUnlockItem(ItemSpec(UIT_SHIP_HULL, item_name));
        if (!unlocked_by_techs.empty()) {
            detailed_description += "\n\n" + UserString("ENC_UNLOCKED_BY");
            for (std::vector<std::string>::const_iterator unlock_tech_it = unlocked_by_techs.begin();
                 unlock_tech_it != unlocked_by_techs.end(); ++unlock_tech_it)
            { detailed_description += LinkTaggedText(VarText::TECH_TAG, *unlock_tech_it) + "  "; }
            detailed_description += "\n\n";
        }

        if (GetOptionsDB().Get<bool>("UI.autogenerated-effects-descriptions")) {
            if (hull->Location())
                detailed_description += str(FlexibleFormat(UserString("ENC_LOCATION_CONDITION_STR")) % hull->Location()->Description());
            if (!hull->Effects().empty())
                detailed_description += str(FlexibleFormat(UserString("ENC_EFFECTS_STR")) % EffectsDescription(hull->Effects()));
        }
    }

    void RefreshDetailPanelTechTag(         const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        const Tech* tech = GetTech(item_name);
        if (!tech) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find tech with name " << item_name;
            return;
        }
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        // Technologies
        name = UserString(item_name);
        texture = ClientUI::TechIcon(item_name);
        other_texture = ClientUI::CategoryIcon(tech->Category()); 
        color = ClientUI::CategoryColor(tech->Category());
        turns = tech->ResearchTime(client_empire_id);
        cost = tech->ResearchCost(client_empire_id);
        cost_units = UserString("ENC_RP");
        general_type = str(FlexibleFormat(UserString("ENC_TECH_DETAIL_TYPE_STR"))
            % UserString(tech->Category())
            % UserString(boost::lexical_cast<std::string>(tech->Type()))
            % UserString(tech->ShortDescription()));

        const std::set<std::string>& unlocked_techs = tech->UnlockedTechs();
        const std::vector<ItemSpec>& unlocked_items = tech->UnlockedItems();
        if (!unlocked_techs.empty() || !unlocked_items.empty())
            detailed_description += UserString("ENC_UNLOCKS");

        if (!unlocked_techs.empty()) {
            for (std::set<std::string>::const_iterator it = unlocked_techs.begin();
                 it != unlocked_techs.end(); ++it)
            {
                std::string link_text = LinkTaggedText(VarText::TECH_TAG, *it);
                detailed_description += str(FlexibleFormat(UserString("ENC_TECH_DETAIL_UNLOCKED_ITEM_STR"))
                    % UserString("UIT_TECH")
                    % link_text);
            }
        }

        if (!unlocked_items.empty()) {
            for (unsigned int i = 0; i < unlocked_items.size(); ++i) {
                const ItemSpec& item = unlocked_items[i];

                std::string TAG;
                switch (item.type) {
                case UIT_BUILDING:      TAG = VarText::BUILDING_TYPE_TAG;       break;
                case UIT_SHIP_PART:     TAG = VarText::SHIP_PART_TAG;           break;
                case UIT_SHIP_HULL:     TAG = VarText::SHIP_HULL_TAG;           break;
                case UIT_SHIP_DESIGN:   TAG = VarText::PREDEFINED_DESIGN_TAG;   break;
                case UIT_TECH:          TAG = VarText::TECH_TAG;                break;
                default: break;
                }

                std::string link_text;
                if (!TAG.empty())
                    link_text = LinkTaggedText(TAG, item.name);
                else
                    link_text = UserString(item.name);

                detailed_description += str(FlexibleFormat(UserString("ENC_TECH_DETAIL_UNLOCKED_ITEM_STR"))
                    % UserString(boost::lexical_cast<std::string>(unlocked_items[i].type))
                    % link_text);
            }
        }

        if (!unlocked_techs.empty() || !unlocked_items.empty())
            detailed_description += "\n";

        detailed_description += UserString(tech->Description());

        if (GetOptionsDB().Get<bool>("UI.autogenerated-effects-descriptions") && !tech->Effects().empty()) {
            detailed_description += str(FlexibleFormat(UserString("ENC_EFFECTS_STR"))
                % EffectsDescription(tech->Effects()));
        }

        const std::set<std::string>& unlocked_by_techs = tech->Prerequisites();
        if (!unlocked_by_techs.empty()) {
            detailed_description += "\n\n" + UserString("ENC_UNLOCKED_BY");
            for (std::set<std::string>::const_iterator it = unlocked_by_techs.begin();
                 it != unlocked_by_techs.end(); ++it)
            { detailed_description += LinkTaggedText(VarText::TECH_TAG, *it) + "  "; }
            detailed_description += "\n\n";
        }
    }

    void RefreshDetailPanelBuildingTypeTag( const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        const BuildingType* building_type = GetBuildingType(item_name);
        if (!building_type) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find building type with name " << item_name;
            return;
        }
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        // Building types
        name = UserString(item_name);
        texture = ClientUI::BuildingIcon(item_name);
        int default_location_id = DefaultLocationForEmpire(client_empire_id);
        turns = building_type->ProductionTime(client_empire_id, default_location_id);
        cost = building_type->ProductionCost(client_empire_id, default_location_id);
        cost_units = UserString("ENC_PP");
        general_type = UserString("ENC_BUILDING_TYPE");

        detailed_description += UserString(building_type->Description());

        if (GetOptionsDB().Get<bool>("UI.autogenerated-effects-descriptions")) {
            if (building_type->Location())
                detailed_description += str(FlexibleFormat(UserString("ENC_LOCATION_CONDITION_STR")) % building_type->Location()->Description());
            if (!building_type->Effects().empty())
                detailed_description += str(FlexibleFormat(UserString("ENC_EFFECTS_STR")) % EffectsDescription(building_type->Effects()));
        }

        std::vector<std::string> unlocked_by_techs = TechsThatUnlockItem(ItemSpec(UIT_BUILDING, item_name));
        if (!unlocked_by_techs.empty()) {
            detailed_description += "\n\n" + UserString("ENC_UNLOCKED_BY");
            for (std::vector<std::string>::const_iterator unlock_tech_it = unlocked_by_techs.begin();
                 unlock_tech_it != unlocked_by_techs.end(); ++unlock_tech_it)
            { detailed_description += LinkTaggedText(VarText::TECH_TAG, *unlock_tech_it) + "  "; }
            detailed_description += "\n\n";
        }
    }

    void RefreshDetailPanelSpecialTag(      const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        const Special* special = GetSpecial(item_name);
        if (!special) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find special with name " << item_name;
            return;
        }
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        // Specials
        name = UserString(item_name);
        texture = ClientUI::SpecialIcon(item_name);
        detailed_description = UserString(special->Description());
        general_type = UserString("ENC_SPECIAL");

        // objects that have special
        std::vector<TemporaryPtr<const UniverseObject> > objects_with_special;
        for (ObjectMap::const_iterator<> obj_it = Objects().const_begin(); obj_it != Objects().const_end(); ++obj_it)
            if (obj_it->Specials().find(item_name) != obj_it->Specials().end())
                objects_with_special.push_back(*obj_it);

        if (!objects_with_special.empty()) {
            detailed_description += "\n\n" + UserString("OBJECTS_WITH_SPECIAL");
            for (std::vector<TemporaryPtr<const UniverseObject> >::const_iterator obj_it = objects_with_special.begin();
                 obj_it != objects_with_special.end(); ++obj_it)
            {
                TemporaryPtr<const UniverseObject> obj = *obj_it;

                if (TemporaryPtr<const Ship> ship = boost::dynamic_pointer_cast<const Ship>(obj))
                    detailed_description += LinkTaggedIDText(VarText::SHIP_ID_TAG, ship->ID(), ship->PublicName(client_empire_id)) + "  ";

                else if (TemporaryPtr<const Fleet> fleet = boost::dynamic_pointer_cast<const Fleet>(obj))
                    detailed_description += LinkTaggedIDText(VarText::FLEET_ID_TAG, fleet->ID(), fleet->PublicName(client_empire_id)) + "  ";

                else if (TemporaryPtr<const Planet> planet = boost::dynamic_pointer_cast<const Planet>(obj))
                    detailed_description += LinkTaggedIDText(VarText::PLANET_ID_TAG, planet->ID(), planet->PublicName(client_empire_id)) + "  ";

                else if (TemporaryPtr<const Building> building = boost::dynamic_pointer_cast<const Building>(obj))
                    detailed_description += LinkTaggedIDText(VarText::BUILDING_ID_TAG, building->ID(), building->PublicName(client_empire_id)) + "  ";

                else if (TemporaryPtr<const System> system = boost::dynamic_pointer_cast<const System>(obj))
                    detailed_description += LinkTaggedIDText(VarText::SYSTEM_ID_TAG, system->ID(), system->PublicName(client_empire_id)) + "  ";

                else
                    detailed_description += obj->PublicName(client_empire_id) + "  ";
            }
            detailed_description += "\n";
        }

        if (GetOptionsDB().Get<bool>("UI.autogenerated-effects-descriptions")) {
            if (special->Location()) {
                const std::string& loc_cond = UserString("ENC_LOCATION_CONDITION_STR");
                std::string desc = special->Location()->Description();
                detailed_description += str(FlexibleFormat(loc_cond) % desc);
            }
            if (!special->Effects().empty())
                detailed_description += str(FlexibleFormat(UserString("ENC_EFFECTS_STR")) % EffectsDescription(special->Effects()));
        }
    }

    void RefreshDetailPanelEmpireTag(       const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        int empire_id = ALL_EMPIRES;
        try {
            empire_id = boost::lexical_cast<int>(item_name);
        } catch(...)
        {}
        Empire* empire = Empires().Lookup(empire_id);
        if (!empire) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find empire with id " << item_name;
            return;
        }

        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        // Empires
        name = empire->Name();
        TemporaryPtr<const Planet> capital = GetPlanet(empire->CapitalID());
        if (capital)
            detailed_description += UserString("EMPIRE_CAPITAL") +
                LinkTaggedIDText(VarText::PLANET_ID_TAG, capital->ID(), capital->Name());
        else
            detailed_description += UserString("NO_CAPITAL");

        // Empire meters
        detailed_description += "\n\n" + UserString("EMPIRE_METERS") + "\n";
        for (std::map<std::string, Meter>::const_iterator meter_it = empire->meter_begin();
             meter_it != empire->meter_end(); ++meter_it)
        {
            detailed_description += UserString(meter_it->first) + ": " + DoubleToString(meter_it->second.Initial(), 3, false) + "\n";
        }

        // Planets
        std::vector<TemporaryPtr<UniverseObject> > empire_planets = Objects().FindObjects(OwnedVisitor<Planet>(empire_id));
        if (!empire_planets.empty()) {
            detailed_description += "\n\n" + UserString("OWNED_PLANETS");
            for (std::vector<TemporaryPtr<UniverseObject> >::const_iterator planet_it = empire_planets.begin();
                 planet_it != empire_planets.end(); ++planet_it)
            {
                TemporaryPtr<const UniverseObject> obj = *planet_it;
                detailed_description += LinkTaggedIDText(VarText::PLANET_ID_TAG, obj->ID(), obj->PublicName(client_empire_id)) + "  ";
            }
        } else {
            detailed_description += "\n\n" + UserString("NO_OWNED_PLANETS_KNOWN");
        }

        // Fleets
        std::vector<TemporaryPtr<UniverseObject> > empire_fleets = Objects().FindObjects(OwnedVisitor<Fleet>(empire_id));
        if (!empire_fleets.empty()) {
            detailed_description += "\n\n" + UserString("OWNED_FLEETS") + "\n";
            for (std::vector<TemporaryPtr<UniverseObject> >::const_iterator fleet_it = empire_fleets.begin();
                 fleet_it != empire_fleets.end(); ++fleet_it)
            {
                TemporaryPtr<const UniverseObject> obj = *fleet_it;
                std::string fleet_link = LinkTaggedIDText(VarText::FLEET_ID_TAG, obj->ID(), obj->PublicName(client_empire_id));
                std::string system_link;
                if (TemporaryPtr<const System> system = GetSystem(obj->SystemID())) {
                    std::string sys_name = system->ApparentName(client_empire_id);
                    system_link = LinkTaggedIDText(VarText::SYSTEM_ID_TAG, system->ID(), sys_name);
                    detailed_description += str(FlexibleFormat(UserString("OWNED_FLEET_AT_SYSTEM"))
                                            % fleet_link % system_link);
                } else {
                    detailed_description += fleet_link;
                }
                detailed_description += "\n";
            }
        } else {
            detailed_description += "\n\n" + UserString("NO_OWNED_FLEETS_KNOWN");
        }

        // Misc. Statistics

        // empire destroyed ships...
        const std::map<int, int>&           empire_ships_destroyed = empire->EmpireShipsDestroyed();
        if (!empire_ships_destroyed.empty())
            detailed_description += "\n\n" + UserString("EMPIRE_SHIPS_DESTROYED");
        for (std::map<int, int>::const_iterator it = empire_ships_destroyed.begin();
             it != empire_ships_destroyed.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            const Empire* target_empire = Empires().Lookup(it->first);
            std::string target_empire_name;
            if (target_empire)
                target_empire_name = target_empire->Name();
            else
                target_empire_name = UserString("UNOWNED");

            detailed_description += "\n" + target_empire_name + " : " + num_str;
        }


        // ship designs destroyed
        const std::map<int, int>&           empire_designs_destroyed = empire->ShipDesignsDestroyed();
        if (!empire_designs_destroyed.empty())
            detailed_description += "\n\n" + UserString("SHIP_DESIGNS_DESTROYED");
        for (std::map<int, int>::const_iterator it = empire_designs_destroyed.begin();
             it != empire_designs_destroyed.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            const ShipDesign* design = GetShipDesign(it->first);
            std::string design_name;
            if (design)
                design_name = design->Name();
            else
                design_name = UserString("UNKNOWN");

            detailed_description += "\n" + design_name + " : " + num_str;
        }


        // species ships destroyed
        const std::map<std::string, int>&   species_ships_destroyed = empire->SpeciesShipsDestroyed();
        if (!species_ships_destroyed.empty())
            detailed_description += "\n\n" + UserString("SPECIES_SHIPS_DESTROYED");
        for (std::map<std::string, int>::const_iterator it = species_ships_destroyed.begin();
             it != species_ships_destroyed.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string species_name;
            if (it->first.empty())
                species_name = UserString("NONE");
            else
                species_name = UserString(it->first);
            detailed_description += "\n" + species_name + " : " + num_str;;
        }


        // species planets invaded
        const std::map<std::string, int>&   species_planets_invaded = empire->SpeciesPlanetsInvaded();
        if (!species_planets_invaded.empty())
            detailed_description += "\n\n" + UserString("SPECIES_PLANETS_INVADED");
        for (std::map<std::string, int>::const_iterator it = species_planets_invaded.begin();
             it != species_planets_invaded.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string species_name;
            if (it->first.empty())
                species_name = UserString("NONE");
            else
                species_name = UserString(it->first);
            detailed_description += "\n" + species_name + " : " + num_str;
        }


        // species ships produced
        const std::map<std::string, int>&   species_ships_produced = empire->SpeciesShipsProduced();
        if (!species_ships_produced.empty())
            detailed_description += "\n\n" + UserString("SPECIES_SHIPS_PRODUCED");
        for (std::map<std::string, int>::const_iterator it = species_ships_produced.begin();
             it != species_ships_produced.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string species_name;
            if (it->first.empty())
                species_name = UserString("NONE");
            else
                species_name = UserString(it->first);
            detailed_description += "\n" + species_name + " : " + num_str;
        }


        // ship designs produced
        const std::map<int, int>&           ship_designs_produced = empire->ShipDesignsProduced();
        if (!ship_designs_produced.empty())
            detailed_description += "\n\n" + UserString("SHIP_DESIGNS_PRODUCED");
        for (std::map<int, int>::const_iterator it = ship_designs_produced.begin();
             it != ship_designs_produced.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            const ShipDesign* design = GetShipDesign(it->first);
            std::string design_name;
            if (design)
                design_name = design->Name();
            else
                design_name = UserString("UNKNOWN");

            detailed_description += "\n" + design_name + " : " + num_str;
        }


        // species ships lost
        const std::map<std::string, int>&   species_ships_lost = empire->SpeciesShipsLost();
        if (!species_ships_lost.empty())
            detailed_description += "\n\n" + UserString("SPECIES_SHIPS_LOST");
        for (std::map<std::string, int>::const_iterator it = species_ships_lost.begin();
             it != species_ships_lost.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string species_name;
            if (it->first.empty())
                species_name = UserString("NONE");
            else
                species_name = UserString(it->first);
            detailed_description += "\n" + species_name + " : " + num_str;
        }


        // ship designs lost
        const std::map<int, int>&           ship_designs_lost = empire->ShipDesignsLost();
        if (!ship_designs_lost.empty())
            detailed_description += "\n\n" + UserString("SHIP_DESIGNS_LOST");
        for (std::map<int, int>::const_iterator it = ship_designs_lost.begin();
             it != ship_designs_lost.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            const ShipDesign* design = GetShipDesign(it->first);
            std::string design_name;
            if (design)
                design_name = design->Name();
            else
                design_name = UserString("UNKNOWN");

            detailed_description += "\n" + design_name + " : " + num_str;
        }


        // species ships scrapped
        const std::map<std::string, int>&   species_ships_scrapped = empire->SpeciesShipsScrapped();
        if (!species_ships_scrapped.empty())
            detailed_description += "\n\n" + UserString("SPECIES_SHIPS_SCRAPPED");
        for (std::map<std::string, int>::const_iterator it = species_ships_scrapped.begin();
             it != species_ships_scrapped.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string species_name;
            if (it->first.empty())
                species_name = UserString("NONE");
            else
                species_name = UserString(it->first);
            detailed_description += "\n" + species_name + " : " + num_str;
        }


        // ship designs scrapped
        const std::map<int, int>&           ship_designs_scrapped = empire->ShipDesignsScrapped();
        if (!ship_designs_scrapped.empty())
            detailed_description += "\n\n" + UserString("SHIP_DESIGNS_SCRAPPED");
        for (std::map<int, int>::const_iterator it = ship_designs_scrapped.begin();
             it != ship_designs_scrapped.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            const ShipDesign* design = GetShipDesign(it->first);
            std::string design_name;
            if (design)
                design_name = design->Name();
            else
                design_name = UserString("UNKNOWN");

            detailed_description += "\n" + design_name + " : " + num_str;
        }


        // species planets depopulated
        const std::map<std::string, int>&   species_planets_depoped = empire->SpeciesPlanetsDepoped();
        if (!species_planets_depoped.empty())
            detailed_description += "\n\n" + UserString("SPECIES_SHIPS_DEPOPED");
        for (std::map<std::string, int>::const_iterator it = species_planets_depoped.begin();
             it != species_planets_depoped.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string species_name;
            if (it->first.empty())
                species_name = UserString("NONE");
            else
                species_name = UserString(it->first);
            detailed_description += "\n" + species_name + " : " + num_str;
        }


        // species planets bombed
        const std::map<std::string, int>&   species_planets_bombed = empire->SpeciesPlanetsBombed();
        if (!species_planets_bombed.empty())
            detailed_description += "\n\n" + UserString("SPECIES_SHIPS_BOMBED");
        for (std::map<std::string, int>::const_iterator it = species_planets_bombed.begin();
             it != species_planets_bombed.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string species_name;
            if (it->first.empty())
                species_name = UserString("NONE");
            else
                species_name = UserString(it->first);
            detailed_description += "\n" + species_name + " : " + num_str;
        }


        // buildings produced
        const std::map<std::string, int>&   building_types_produced = empire->BuildingTypesProduced();
        if (!building_types_produced.empty())
            detailed_description += "\n\n" + UserString("BUILDING_TYPES_PRODUCED");
        for (std::map<std::string, int>::const_iterator it = building_types_produced.begin();
             it != building_types_produced.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string building_type_name;
            if (it->first.empty())
                building_type_name = UserString("NONE");
            else
                building_type_name = UserString(it->first);
            detailed_description += "\n" + building_type_name + " : " + num_str;
        }


        // buildings scrapped
        const std::map<std::string, int>&   building_types_scrapped = empire->BuildingTypesScrapped();
        if (!building_types_scrapped.empty())
            detailed_description += "\n\n" + UserString("BUILDING_TYPES_SCRAPPED");
        for (std::map<std::string, int>::const_iterator it = building_types_scrapped.begin();
             it != building_types_scrapped.end(); ++it)
        {
            std::string num_str = boost::lexical_cast<std::string>(it->second);
            std::string building_type_name;
            if (it->first.empty())
                building_type_name = UserString("NONE");
            else
                building_type_name = UserString(it->first);
            detailed_description += "\n" + building_type_name + " : " + num_str;
        }
    }

    void RefreshDetailPanelSpeciesTag(      const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        const Species* species = GetSpecies(item_name);
        if (!species) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find species with name " << item_name;
            return;
        }
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        // Species
        name = UserString(item_name);
        texture = ClientUI::SpeciesIcon(item_name);
        general_type = UserString("ENC_SPECIES");
        detailed_description = UserString(species->GameplayDescription());

        // inherent species limitations
        detailed_description += "\n";
        if (species->CanProduceShips())
            detailed_description += UserString("CAN_PRODUCE_SHIPS");
        else
            detailed_description += UserString("CANNOT_PRODUCE_SHIPS");
        detailed_description += "\n";
        if (species->CanColonize())
            detailed_description += UserString("CAN_COLONIZE");
        else
            detailed_description += UserString("CANNNOT_COLONIZE");

        // focus preference
        if (!species->PreferredFocus().empty()) {
            detailed_description += "\n\n";
            detailed_description += UserString("FOCUS_PREFERENCE");
            detailed_description += UserString(species->PreferredFocus());
        }

        // environmental preferences
        detailed_description += "\n\n";
        const std::map<PlanetType, PlanetEnvironment>& pt_env_map = species->PlanetEnvironments();
        if (!pt_env_map.empty()) {
            detailed_description += UserString("ENVIRONMENTAL_PREFERENCES") + "\n";
            for (std::map<PlanetType, PlanetEnvironment>::const_iterator pt_env_it = pt_env_map.begin();
                 pt_env_it != pt_env_map.end(); ++pt_env_it)
            {
                detailed_description += UserString(boost::lexical_cast<std::string>(pt_env_it->first)) + " : " +
                                        UserString(boost::lexical_cast<std::string>(pt_env_it->second)) + "\n";
            }
        } else {
            detailed_description += "\n";
        }

        if (GetOptionsDB().Get<bool>("UI.autogenerated-effects-descriptions") && !species->Effects().empty()) {
            detailed_description += str(FlexibleFormat(UserString("ENC_EFFECTS_STR")) % EffectsDescription(species->Effects()));
        }

        // Long description
        detailed_description += "\n";
        detailed_description += UserString(species->Description());

        // homeworld
        detailed_description += "\n\n";
        if (species->Homeworlds().empty()) {
            detailed_description += UserString("NO_HOMEWORLD") + "\n";
        } else {
            detailed_description += UserString("HOMEWORLD") + "\n";
            for (std::set<int>::const_iterator hw_it = species->Homeworlds().begin();
                 hw_it != species->Homeworlds().end(); ++hw_it)
            {
                if (TemporaryPtr<const Planet> homeworld = GetPlanet(*hw_it))
                    detailed_description += LinkTaggedIDText(VarText::PLANET_ID_TAG, *hw_it,
                                                             homeworld->PublicName(client_empire_id)) + "\n";
                else
                    detailed_description += UserString("UNKNOWN_PLANET") + "\n";
            }
        }

        // occupied planets
        std::vector<TemporaryPtr<Planet> > planets = Objects().FindObjects<Planet>();
        std::vector<TemporaryPtr<const Planet> > species_occupied_planets;
        for (std::vector<TemporaryPtr<Planet> >::const_iterator planet_it = planets.begin();
             planet_it != planets.end(); ++planet_it)
        {
            TemporaryPtr<const Planet> planet = *planet_it;
            if (planet->SpeciesName() == item_name)
                species_occupied_planets.push_back(planet);
        }
        if (!species_occupied_planets.empty()) {
            detailed_description += "\n" + UserString("OCCUPIED_PLANETS") + "\n";
            for (std::vector<TemporaryPtr<const Planet> >::const_iterator planet_it =
                     species_occupied_planets.begin();
                 planet_it != species_occupied_planets.end(); ++planet_it)
            {
                TemporaryPtr<const Planet> planet = *planet_it;
                detailed_description += LinkTaggedIDText(VarText::PLANET_ID_TAG, planet->ID(),
                                                         planet->PublicName(client_empire_id)) + "  ";
            }
            detailed_description += "\n";
        }
    }

    void RefreshDetailPanelFieldTypeTag(    const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        const FieldType* field_type = GetFieldType(item_name);
        if (!field_type) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find fiedl type with name " << item_name;
            return;
        }

        // Field types
        name = UserString(item_name);
        texture = ClientUI::FieldTexture(item_name);
        general_type = UserString("ENC_FIELD_TYPE");

        detailed_description += UserString(field_type->Description());

        if (GetOptionsDB().Get<bool>("UI.autogenerated-effects-descriptions")) {
            if (!field_type->Effects().empty())
                detailed_description += str(FlexibleFormat(UserString("ENC_EFFECTS_STR")) % EffectsDescription(field_type->Effects()));
        }
    }

    namespace {
        std::string GetDetailedDescription(const TemporaryPtr<Ship> ship, const ShipDesign* design) {
            GetUniverse().UpdateMeterEstimates(ship->ID());

            std::string hull_link;
            if (!design->Hull().empty())
                 hull_link = LinkTaggedText(VarText::SHIP_HULL_TAG, design->Hull());

            std::string parts_list;
            const std::vector<std::string>& parts = design->Parts();
            std::vector<std::string> non_empty_parts;
            for (std::vector<std::string>::const_iterator part_it = parts.begin();
                 part_it != parts.end(); ++part_it)
            {
                if (!part_it->empty())
                    non_empty_parts.push_back(*part_it);
            }
            for (std::vector<std::string>::const_iterator part_it = non_empty_parts.begin();
                    part_it != non_empty_parts.end(); ++part_it)
            {
                if (part_it != non_empty_parts.begin())
                    parts_list += ", ";
                parts_list += LinkTaggedText(VarText::SHIP_PART_TAG, *part_it);
            }

            return str(FlexibleFormat(UserString("ENC_SHIP_DESIGN_DESCRIPTION_STR"))
            % design->Description()
            % hull_link
            % parts_list
            % static_cast<int>(design->SRWeapons().size())
            % static_cast<int>(design->LRWeapons().size())
            % static_cast<int>(design->FWeapons().size())
            % static_cast<int>(design->PDWeapons().size())
            % ship->CurrentMeterValue(METER_MAX_STRUCTURE)
            % ship->CurrentMeterValue(METER_MAX_SHIELD)
            % ship->CurrentMeterValue(METER_DETECTION)
            % ship->CurrentMeterValue(METER_STEALTH)
            % ship->CurrentMeterValue(METER_BATTLE_SPEED)
            % ship->CurrentMeterValue(METER_STARLANE_SPEED)
            % ship->CurrentMeterValue(METER_MAX_FUEL)
            % design->ColonyCapacity()
            % design->TroopCapacity()
            % design->Attack());
        }
    }

    void RefreshDetailPanelShipDesignTag(   const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        int design_id = boost::lexical_cast<int>(item_name);
        const ShipDesign* design = GetShipDesign(boost::lexical_cast<int>(item_name));
        if (!design) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find ShipDesign with id " << item_name;
            return;
        }
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        // Ship Designs
        name = design->Name();
        texture = ClientUI::ShipDesignIcon(design_id);
        int default_location_id = DefaultLocationForEmpire(client_empire_id);
        turns = design->ProductionTime(client_empire_id, default_location_id);
        cost = design->ProductionCost(client_empire_id, default_location_id);
        cost_units = UserString("ENC_PP");
        general_type = design->IsMonster() ? UserString("ENC_MONSTER") : UserString("ENC_SHIP_DESIGN");

        TemporaryPtr<Ship> temp = GetUniverse().CreateShip(client_empire_id, design_id, "",
                                                           client_empire_id, TEMPORARY_OBJECT_ID);

        detailed_description = GetDetailedDescription(temp, design);

        GetUniverse().Delete(TEMPORARY_OBJECT_ID);

        // ships of this design
        std::vector<TemporaryPtr<const Ship> > design_ships;
        for (std::map<int, TemporaryPtr<UniverseObject> >::iterator
             ship_it = Objects().ExistingShipsBegin();
             ship_it != Objects().ExistingShipsEnd(); ++ship_it)
        {
            TemporaryPtr<const Ship> ship = boost::dynamic_pointer_cast<const Ship>(ship_it->second);
            if (ship && ship->DesignID() == design_id)
                design_ships.push_back(ship);
        }
        if (!design_ships.empty()) {
            detailed_description += "\n\n" + UserString("SHIPS_OF_DESIGN");
            for (std::vector<TemporaryPtr<const Ship> >::const_iterator ship_it = design_ships.begin();
                 ship_it != design_ships.end(); ++ship_it)
            {
                detailed_description += LinkTaggedIDText(VarText::SHIP_ID_TAG, (*ship_it)->ID(),
                                                         (*ship_it)->PublicName(client_empire_id)) + "  ";
            }
        } else {
            detailed_description += "\n\n" + UserString("NO_SHIPS_OF_DESIGN");
        }
    }

    void RefreshDetailPanelIncomplDesignTag(const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color, boost::weak_ptr<const ShipDesign>& inc_design)
    {
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        boost::shared_ptr<const ShipDesign> incomplete_design = inc_design.lock();
        if (incomplete_design) {
            // incomplete design.  not yet in game universe; being created on design screen
            name = incomplete_design->Name();

            const std::string& design_icon = incomplete_design->Icon();
            if (design_icon.empty())
                texture = ClientUI::HullIcon(incomplete_design->Hull());
            else
                texture = ClientUI::GetTexture(ClientUI::ArtDir() / design_icon, true);

            int default_location_id = DefaultLocationForEmpire(client_empire_id);
            turns = incomplete_design->ProductionTime(client_empire_id, default_location_id);
            cost = incomplete_design->ProductionCost(client_empire_id, default_location_id);
            cost_units = UserString("ENC_PP");

            GetUniverse().InsertShipDesignID(new ShipDesign(*incomplete_design), TEMPORARY_OBJECT_ID);

            TemporaryPtr<Ship> temp = GetUniverse().CreateShip(client_empire_id, TEMPORARY_OBJECT_ID, "",
                                                               client_empire_id, TEMPORARY_OBJECT_ID);
            GetUniverse().UpdateMeterEstimates(TEMPORARY_OBJECT_ID);

            detailed_description = GetDetailedDescription(temp, incomplete_design.get());

            GetUniverse().Delete(TEMPORARY_OBJECT_ID);
            GetUniverse().DeleteShipDesign(TEMPORARY_OBJECT_ID);
        }

        general_type = UserString("ENC_INCOMPETE_SHIP_DESIGN");
    }

    void RefreshDetailPanelObjectTag(       const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        int id = boost::lexical_cast<int>(item_name);
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        if (id != INVALID_OBJECT_ID) {
            TemporaryPtr<const UniverseObject> obj = GetUniverseObject(id);
            if (!obj) {
                Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find UniverseObject with id " << item_name;
                return;
            }

            detailed_description = obj->Dump();
            name = obj->PublicName(client_empire_id);
            general_type = GeneralTypeOfObject(obj->ObjectType());
            if (general_type.empty()) {
                Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't interpret object: " << obj->Name() << " (" << item_name << ")";
                return;
            }
        }
    }

    void RefreshDetailPanelSuitabilityTag(  const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        general_type = UserString("SP_PLANET_SUITABILITY");

        int planet_id = boost::lexical_cast<int>(item_name);
        TemporaryPtr<Planet> planet = GetPlanet(planet_id);

        std::string original_planet_species = planet->SpeciesName();
        int original_owner_id = planet->Owner();
        float orig_initial_target_pop = planet->GetMeter(METER_TARGET_POPULATION)->Initial();
        name = planet->PublicName(planet_id);

        int empire_id = HumanClientApp::GetApp()->EmpireID();
        Empire* empire = HumanClientApp::GetApp()->Empires().Lookup(empire_id);
        if (!empire) {
            return;
        }
        const std::vector<int> pop_center_ids = empire->GetPopulationPool().PopCenterIDs();

        std::set<std::string> species_names;
        std::map<std::string, std::pair<PlanetEnvironment, float> > population_counts;

        // Collect species colonizing/environment hospitality information
        for (std::vector<int>::const_iterator it = pop_center_ids.begin(); it != pop_center_ids.end(); it++) {
            TemporaryPtr<const UniverseObject> obj = GetUniverseObject(*it);
            TemporaryPtr<const PopCenter> pc = boost::dynamic_pointer_cast<const PopCenter>(obj);
            if (!pc)
                continue;

            const std::string& species_name = pc->SpeciesName();
            if (species_name.empty())
                continue;

            const Species* species = GetSpecies(species_name);
            if (!species)
                continue;

            // Exclude species that can't colonize this planet (either by virtue
            // of "can't produce ships" or "cannot colonize" traits) UNLESS they
            // are already here (aka: it's their home planet). Showing them on
            // their own planet allows comparison vs other races, which might
            // be better suited to this planet. 
            if (!species->CanProduceShips() || !species->CanColonize()) {
                if (species_name != planet->SpeciesName())
                    continue;
            }
            species_names.insert(species_name);
        }

        boost::shared_ptr<GG::Font> font = ClientUI::GetFont();
        GG::X max_species_name_column1_width(0);

        GetUniverse().InhibitUniverseObjectSignals(true);

        for (std::set<std::string>::const_iterator it = species_names.begin();
             it != species_names.end(); it++)
        {
            std::string species_name = *it;

            std::string species_name_column1 = str(FlexibleFormat(UserString("ENC_SPECIES_PLANET_TYPE_SUITABILITY_COLUMN1")) % UserString(species_name)); 
            max_species_name_column1_width = std::max(font->TextExtent(species_name_column1).x, max_species_name_column1_width);

            // Setting the planet's species allows all of it meters to reflect
            // species (and empire) properties, such as environment type
            // preferences and tech.
            // @see also: MapWnd::UpdateMeterEstimates()
            planet->SetSpecies(species_name);
            planet->SetOwner(empire_id);
            planet->GetMeter(METER_TARGET_POPULATION)->Set(0.0, 0.0);
            GetUniverse().UpdateMeterEstimates(planet_id);

            const Species* species = GetSpecies(species_name);
            PlanetEnvironment planet_environment = PE_UNINHABITABLE;
            if (species)
                planet_environment = species->GetPlanetEnvironment(planet->Type());

            double planet_capacity = ((planet_environment == PE_UNINHABITABLE) ? 0 : planet->CurrentMeterValue(METER_TARGET_POPULATION));

            population_counts[species_name].first = planet_environment;
            population_counts[species_name].second = planet_capacity;
        }

        std::multimap<float, std::pair<std::string, PlanetEnvironment> > target_population_species;
        for (std::map<std::string, std::pair<PlanetEnvironment, float> >::const_iterator
             it = population_counts.begin(); it != population_counts.end(); ++it)
        {
            target_population_species.insert(std::make_pair(
                it->second.second, std::make_pair(it->first, it->second.first)));
        }

        bool positive_header_placed = false;
        bool negative_header_placed = false;

        for (std::multimap<float, std::pair<std::string, PlanetEnvironment> >::const_reverse_iterator
             it = target_population_species.rbegin(); it != target_population_species.rend(); it++)
        {
            std::string user_species_name = UserString(it->second.first);
            std::string species_name_column1 = str(FlexibleFormat(UserString("ENC_SPECIES_PLANET_TYPE_SUITABILITY_COLUMN1")) % LinkTaggedText(VarText::SPECIES_TAG, it->second.first));

            while (font->TextExtent(species_name_column1).x < max_species_name_column1_width)
            { species_name_column1 += "\t"; }

            if (it->first > 0) {
                if (!positive_header_placed) {
                    detailed_description += str(FlexibleFormat(UserString("ENC_SUITABILITY_REPORT_POSITIVE_HEADER")) % planet->PublicName(planet_id));                    
                    positive_header_placed = true;
                }

                detailed_description += str(FlexibleFormat(UserString("ENC_SPECIES_PLANET_TYPE_SUITABILITY"))
                    % species_name_column1
                    % UserString(boost::lexical_cast<std::string>(it->second.second))
                    % (GG::RgbaTag(ClientUI::StatIncrColor()) + DoubleToString(it->first, 2, true) + "</rgba>"));

            } else if (it->first <= 0) {
                if (!negative_header_placed) {
                    if (positive_header_placed)
                        detailed_description += "\n\n";

                    detailed_description += str(FlexibleFormat(UserString("ENC_SUITABILITY_REPORT_NEGATIVE_HEADER")) % planet->PublicName(planet_id));                    
                    negative_header_placed = true;
                }

                detailed_description += str(FlexibleFormat(UserString("ENC_SPECIES_PLANET_TYPE_SUITABILITY"))
                    % species_name_column1
                    % UserString(boost::lexical_cast<std::string>(it->second.second))
                    % (GG::RgbaTag(ClientUI::StatDecrColor()) + DoubleToString(it->first, 2, true) + "</rgba>"));
            }

            detailed_description += "\n";
        }

        planet->SetSpecies(original_planet_species);
        planet->SetOwner(original_owner_id);
        planet->GetMeter(METER_TARGET_POPULATION)->Set(orig_initial_target_pop, orig_initial_target_pop);

        GetUniverse().InhibitUniverseObjectSignals(false);

        GetUniverse().UpdateMeterEstimates(planet_id);
    }

    std::map<int, int> CountByOwner(const std::set<int>& objects) {
        std::map<int, int> objects_per_owner;
        for (std::set<int>::const_iterator it = objects.begin(); it != objects.end(); ++it) {
            TemporaryPtr<const UniverseObject> object = Objects().Object(*it);
            if (object && (
                    object->ObjectType() == OBJ_SHIP || (
                        object->GetMeter(METER_POPULATION) &&
                        object->CurrentMeterValue(METER_POPULATION) > 0.0)))
            {
                int owner_id = object->Owner();
                if (objects_per_owner.find(owner_id) == objects_per_owner.end())
                    objects_per_owner[owner_id] = 0;
                ++objects_per_owner[owner_id];
            }
        }
        return objects_per_owner;
    }

    std::string CountsToText(const std::map<int, int>& count_per_empire, std::string delimiter = ", ") {
        std::stringstream ss;
        for (std::map<int,int>::const_iterator it = count_per_empire.begin(); it != count_per_empire.end(); ) {
            std::string owner_string = UserString("NEUTRAL");
            if (const Empire* owner = Empires().Lookup(it->first))
                owner_string = GG::RgbaTag(owner->Color()) + owner->Name() + "</rgba>";
            ss << owner_string << ": " << it->second;
            ++it;
            if (it != count_per_empire.end())
                ss << delimiter;
        }
        return ss.str();
    }

    void RefreshDetailPanelCombatTag(       const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color)
    {
        int log_id = boost::lexical_cast<int>(item_name);
        bool available = CombatLogAvailable(log_id);
        if (!available) {
            Logger().errorStream() << "EncyclopediaDetailPanel::Refresh couldn't find combat log with id: " << item_name;
            return;
        }
        const CombatLog& log = GetCombatLog(log_id);
        int client_empire_id = HumanClientApp::GetApp()->EmpireID();

        name = UserString("ENC_COMBAT_LOG");
        texture = ClientUI::GetTexture(ClientUI::ArtDir() / "/icons/sitrep/combat.png", true);
        general_type = UserString("ENC_COMBAT_LOG");

        TemporaryPtr<const System> system = GetSystem(log.system_id);
        const std::string& sys_name = (system ? system->PublicName(client_empire_id) : UserString("ERROR"));

        detailed_description = str(FlexibleFormat(UserString("ENC_COMBAT_LOG_DESCRIPTION_STR"))
                                   % LinkTaggedIDText(VarText::SYSTEM_ID_TAG, log.system_id, sys_name)
                                   % log.turn) + "\n";

        detailed_description += "\n"+ UserString("COMBAT_INITIAL_FORCES") + "\n" + CountsToText(CountByOwner(log.object_ids))+"\n";

        for ( std::vector<CombatEventPtr>::const_iterator it = log.combat_events.begin();
              it != log.combat_events.end(); ++it )
        {
            const BoutBeginEvent* bout_begin = dynamic_cast<BoutBeginEvent*>(it->get());
            const AttackEvent* attack = dynamic_cast<AttackEvent*>(it->get());
            const IncapacitationEvent* incapacitation = dynamic_cast<IncapacitationEvent*>(it->get());
            if ( bout_begin ) {
                detailed_description += str(FlexibleFormat(UserString("ENC_ROUND_BEGIN")) % bout_begin->bout) + "\n";
            } else if ( attack ) {
                std::string attacker_link = PublicNameLink(client_empire_id, attack->attacker_id, UserString("ENC_COMBAT_UNKNOWN_OBJECT"));
                std::string target_link = PublicNameLink(client_empire_id, attack->target_id, UserString("ENC_COMBAT_UNKNOWN_OBJECT"));

                const std::string& template_str = UserString("ENC_COMBAT_ATTACK_STR");

            detailed_description += str(FlexibleFormat(template_str)
                                        % attacker_link
                                        % target_link
                                            % attack->damage
                                            % attack->bout
                                            % attack->round) + "\n";
            } else if ( incapacitation ) {
                TemporaryPtr<const UniverseObject> object = GetUniverseObject(incapacitation->object_id);
                std::string  template_str;
                if ( !object ) {
                    template_str = UserString("ENC_COMBAT_UNKNOWN_DESTROYED_STR");
                } else if ( object->ObjectType() == OBJ_PLANET ) {
                    template_str = UserString("ENC_COMBAT_PLANET_INCAPACITATED_STR");
                } else {
                    template_str = UserString("ENC_COMBAT_DESTROYED_STR");
        }
                const std::string object_link = PublicNameLink ( client_empire_id, incapacitation->object_id, UserString ( "ENC_COMBAT_UNKNOWN_OBJECT" ) );
                
                int owner_id = object?object->Owner():ALL_EMPIRES;
                std::string owner_string = " ";
                if ( owner_id != ALL_EMPIRES ) {
                    Empire* owner = Empires().Lookup(owner_id);
                    if ( owner ) {
                        owner_string += owner->Name() + " ";
    }
                }

                detailed_description += str(FlexibleFormat(template_str) % owner_string % object_link) + "\n";
            }
        }

        detailed_description += "\n" + UserString("COMBAT_SUMMARY_DESTROYED") + "\n" + CountsToText(CountByOwner(log.destroyed_object_ids));
    }

    void GetRefreshDetailPanelInfo(         const std::string& item_type, const std::string& item_name,
                                            std::string& name, boost::shared_ptr<GG::Texture>& texture,
                                            boost::shared_ptr<GG::Texture>& other_texture, int& turns,
                                            float& cost, std::string& cost_units, std::string& general_type,
                                            std::string& specific_type, std::string& detailed_description,
                                            GG::Clr& color, boost::weak_ptr<const ShipDesign>& incomplete_design)
    {
        if (item_type == TextLinker::ENCYCLOPEDIA_TAG) {
            RefreshDetailPanelPediaTag(         item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_SHIP_PART") {
            RefreshDetailPanelShipPartTag(      item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_SHIP_HULL") {
            RefreshDetailPanelShipHullTag(      item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_TECH") {
            RefreshDetailPanelTechTag(          item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_BUILDING_TYPE") {
            RefreshDetailPanelBuildingTypeTag(  item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_SPECIAL") {
            RefreshDetailPanelSpecialTag(       item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_EMPIRE") {
            RefreshDetailPanelEmpireTag(        item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_SPECIES") {
            RefreshDetailPanelSpeciesTag(       item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_FIELD_TYPE") {
            RefreshDetailPanelFieldTypeTag(     item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == "ENC_SHIP_DESIGN") {
            RefreshDetailPanelShipDesignTag(    item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == INCOMPLETE_DESIGN) {
            RefreshDetailPanelIncomplDesignTag( item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color,
                                                incomplete_design);
        } else if (item_type == UNIVERSE_OBJECT) {
            RefreshDetailPanelObjectTag(        item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == PLANET_SUITABILITY_REPORT) {
            RefreshDetailPanelSuitabilityTag(   item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == COMBAT_LOG) {
            RefreshDetailPanelCombatTag(        item_type, item_name,
                                                name, texture, other_texture, turns, cost, cost_units,
                                                general_type, specific_type, detailed_description, color);
        } else if (item_type == TextLinker::GRAPH_TAG) {
            // should be handled externally...
        }
    }
}

void EncyclopediaDetailPanel::Refresh() {
    if (m_icon) {
        DeleteChild(m_icon);
        m_icon = 0;
    }
    if (m_other_icon) {
        DeleteChild(m_other_icon);
        m_other_icon = 0;
    }
    m_name_text->Clear();
    m_summary_text->Clear();
    m_cost_text->Clear();

    GG::Pt initial_scroll_pos = m_description_box->ScrollPosition();
    m_description_box->Clear();

    DetachChild(m_graph);

    // get details of item as applicable in order to set summary, cost, description TextControls
    std::string name;
    boost::shared_ptr<GG::Texture> texture;
    boost::shared_ptr<GG::Texture> other_texture;
    int turns = -1;
    float cost = 0.0f;
    std::string cost_units;             // "PP" or "RP" or empty string, depending on whether and what something costs
    std::string general_type;           // general type of thing being shown, eg. "Building" or "Ship Part"
    std::string specific_type;          // specific type of thing; thing's purpose.  eg. "Farming" or "Colonization".  May be left blank for things without specific types (eg. specials)
    std::string detailed_description;
    GG::Clr color(GG::CLR_ZERO);

    using boost::io::str;
    if (m_items.empty())
        return;

    GetRefreshDetailPanelInfo(m_items_it->first, m_items_it->second,
                              name, texture, other_texture, turns, cost, cost_units,
                              general_type, specific_type, detailed_description, color,
                              m_incomplete_design);

    if (m_items_it->first == TextLinker::GRAPH_TAG) {
        const std::string& graph_id = m_items_it->second;

        const std::map<std::string, std::map<int, std::map<int, double> > >&
            stat_records = GetUniverse().GetStatRecords();

        std::map<std::string, std::map<int, std::map<int, double> > >::const_iterator
            stat_name_it = stat_records.find(graph_id);
        if (stat_name_it != stat_records.end()) {
            const std::map<int, std::map<int, double> >& empire_lines = stat_name_it->second;
            m_graph->Clear();

            // add lines for each empire
            for (std::map<int, std::map<int, double> >::const_iterator empire_it = empire_lines.begin();
                 empire_it != empire_lines.end(); ++empire_it)
            {
                int empire_id = empire_it->first;

                GG::Clr empire_clr = GG::CLR_WHITE;
                if (const Empire* empire = Empires().Lookup(empire_id))
                    empire_clr = empire->Color();

                const std::map<int, double>& empire_line = empire_it->second;
                // convert formats...
                std::vector<std::pair<double, double> > line_data_pts;
                for (std::map<int, double>::const_iterator line_it = empire_line.begin();
                     line_it != empire_line.end(); ++line_it)
                { line_data_pts.push_back(std::make_pair(line_it->first, line_it->second)); }

                m_graph->AddSeries(line_data_pts, empire_clr);
            }

            m_graph->AutoSetRange();
            AttachChild(m_graph);
            m_graph->Show();
        }

        name = UserString(graph_id);
        general_type = UserString("ENC_GRAPH");
    }

    // Create Icons
    if (texture) {
        m_icon =        new GG::StaticGraphic(GG::X0, GG::Y0, GG::X(10), GG::Y(10), texture,        GG::GRAPHIC_FITGRAPHIC | GG::GRAPHIC_PROPSCALE);
        if (color != GG::CLR_ZERO)
            m_icon->SetColor(color);
    }
    if (other_texture) {
        m_other_icon =  new GG::StaticGraphic(GG::X0, GG::Y0, GG::X(10), GG::Y(10), other_texture,  GG::GRAPHIC_FITGRAPHIC | GG::GRAPHIC_PROPSCALE);
        if (color != GG::CLR_ZERO)
            m_other_icon->SetColor(color);
    }

    if (m_icon) {
        m_icon->Show();
        AttachChild(m_icon);
    }
    if (m_other_icon) {
        m_other_icon->Show();
        AttachChild(m_other_icon);
    }

    // Set Text
    if (!name.empty())
        m_name_text->SetText(name);

    m_summary_text->SetText(str(FlexibleFormat(UserString("ENC_DETAIL_TYPE_STR"))
        % specific_type
        % general_type));

    if (color != GG::CLR_ZERO)
        m_summary_text->SetColor(color);

    if (cost != 0.0 && turns != -1) {
        m_cost_text->SetText(str(FlexibleFormat(UserString("ENC_COST_AND_TURNS_STR"))
            % DoubleToString(cost, 3, false)
            % cost_units
            % turns));
    }

    if (!detailed_description.empty())
        m_description_box->SetText(detailed_description);

    DoLayout();

    m_description_box->SetScrollPosition(initial_scroll_pos);
}

void EncyclopediaDetailPanel::AddItem(const std::string& type, const std::string& name) {
    // if the actual item is not the last one, all aubsequented items are deleted
    if (!m_items.empty()) {
        if (m_items_it->first == type && m_items_it->second == name)
            return;
        std::list<std::pair <std::string, std::string> >::iterator end = m_items.end();
        end--;
        if (m_items_it != end) {
            std::list<std::pair <std::string, std::string> >::iterator i = m_items_it;
            ++i;
            m_items.erase(i, m_items.end());
        }
    }

    m_items.push_back(std::pair<std::string, std::string>(type, name));
    if (m_items.size() == 1)
        m_items_it = m_items.begin();
    else
        ++m_items_it;

    if (m_back_button->Disabled() && m_items.size() > 1) // enable Back button
        m_back_button->Disable(false); 

    if (!m_next_button->Disabled())                      // disable Next button
        m_next_button->Disable(true);

    Refresh();
    m_description_box->SetScrollPosition(GG::Pt(GG::X0, GG::Y0));   // revert to top for new screen
}

void EncyclopediaDetailPanel::PopItem() {
    if (!m_items.empty()) {
        m_items.pop_back();
        if (m_items_it == m_items.end() && m_items_it != m_items.begin())
            m_items_it--;
        Refresh();
        m_description_box->SetScrollPosition(GG::Pt(GG::X0, GG::Y0));   // revert to top for new screen
    }
}

void EncyclopediaDetailPanel::ClearItems() {
    m_items.clear();
    m_items_it = m_items.end();
    Refresh();
    m_description_box->SetScrollPosition(GG::Pt(GG::X0, GG::Y0));   // revert to top for new screen
}

void EncyclopediaDetailPanel::SetText(const std::string& text, bool lookup_in_stringtable) {
    if (m_items_it != m_items.end() && text == m_items_it->second)
        return;
    if (text == "ENC_INDEX")
        SetIndex();
    else
        AddItem(TextLinker::ENCYCLOPEDIA_TAG, (text.empty() || !lookup_in_stringtable) ? text : UserString(text));
}

void EncyclopediaDetailPanel::SetPlanet(int planet_id) {
    int current_item_id = INVALID_OBJECT_ID;
    if (m_items_it != m_items.end()) {
        try {
            current_item_id = boost::lexical_cast<int>(m_items_it->second);
        } catch (...) {
        }
    }
    if (planet_id == current_item_id)
        return;

    AddItem(PLANET_SUITABILITY_REPORT, boost::lexical_cast<std::string>(planet_id));
}

void EncyclopediaDetailPanel::SetCombatLog(int log_id) {
    int current_item_id = -1;
    if (m_items_it != m_items.end()) {
        try {
            current_item_id = boost::lexical_cast<int>(m_items_it->second);
        } catch (...) {
        }
    }
    if (log_id == current_item_id)
        return;

    AddItem(COMBAT_LOG, boost::lexical_cast<std::string>(log_id));
}

void EncyclopediaDetailPanel::SetTech(const std::string& tech_name) {
    if (m_items_it != m_items.end() && tech_name == m_items_it->second)
        return;
    AddItem("ENC_TECH", tech_name);
}

void EncyclopediaDetailPanel::SetPartType(const std::string& part_name) {
    if (m_items_it != m_items.end() && part_name == m_items_it->second)
        return;
    AddItem("ENC_SHIP_PART", part_name);
}

void EncyclopediaDetailPanel::SetHullType(const std::string& hull_name) {
    if (m_items_it != m_items.end() && hull_name == m_items_it->second)
        return;
    AddItem("ENC_SHIP_HULL", hull_name);
}

void EncyclopediaDetailPanel::SetBuildingType(const std::string& building_name) {
    if (m_items_it != m_items.end() && building_name == m_items_it->second)
        return;
    AddItem("ENC_BUILDING_TYPE", building_name);
}

void EncyclopediaDetailPanel::SetSpecial(const std::string& special_name) {
    if (m_items_it != m_items.end() && special_name == m_items_it->second)
        return;
    AddItem("ENC_SPECIAL", special_name);
}

void EncyclopediaDetailPanel::SetSpecies(const std::string& species_name) {
    if (m_items_it != m_items.end() && species_name == m_items_it->second)
        return;
    AddItem("ENC_SPECIES", species_name);
}

void EncyclopediaDetailPanel::SetFieldType(const std::string& field_type_name) {
    if (m_items_it != m_items.end() && field_type_name == m_items_it->second)
        return;
    AddItem("ENC_FIELD_TYPE", field_type_name);
}

void EncyclopediaDetailPanel::SetObject(int object_id) {
    int current_item_id = INVALID_OBJECT_ID;
    if (m_items_it != m_items.end()) {
        try {
            current_item_id = boost::lexical_cast<int>(m_items_it->second);
        } catch (...) {
        }
    }
    if (object_id == current_item_id)
        return;
    AddItem(UNIVERSE_OBJECT, boost::lexical_cast<std::string>(object_id));
}

void EncyclopediaDetailPanel::SetObject(const std::string& object_id) {
    if (m_items_it != m_items.end() && object_id == m_items_it->second)
        return;
    AddItem(UNIVERSE_OBJECT, object_id);
}

void EncyclopediaDetailPanel::SetEmpire(int empire_id) {
    int current_item_id = ALL_EMPIRES;
    if (m_items_it != m_items.end()) {
        try {
            current_item_id = boost::lexical_cast<int>(m_items_it->second);
        } catch (...) {
        }
    }
    if (empire_id == current_item_id)
        return;
    AddItem("ENC_EMPIRE", boost::lexical_cast<std::string>(empire_id));
}

void EncyclopediaDetailPanel::SetEmpire(const std::string& empire_id) {
    if (m_items_it != m_items.end() && empire_id == m_items_it->second)
        return;
    AddItem("ENC_EMPIRE", empire_id);
}

void EncyclopediaDetailPanel::SetDesign(int design_id) {
    int current_item_id = ShipDesign::INVALID_DESIGN_ID;
    if (m_items_it != m_items.end()) {
        try {
            current_item_id = boost::lexical_cast<int>(m_items_it->second);
        } catch (...) {
        }
    }
    if (design_id == current_item_id)
        return;
    AddItem("ENC_SHIP_DESIGN", boost::lexical_cast<std::string>(design_id));
}

void EncyclopediaDetailPanel::SetDesign(const std::string& design_id) {
    if (m_items_it != m_items.end() && design_id == m_items_it->second)
        return;
    AddItem("ENC_SHIP_DESIGN", design_id);
}

void EncyclopediaDetailPanel::SetIncompleteDesign(boost::weak_ptr<const ShipDesign> incomplete_design) {
    m_incomplete_design = incomplete_design;

    if (m_items_it == m_items.end() ||
        m_items_it->first != INCOMPLETE_DESIGN) {
        AddItem(INCOMPLETE_DESIGN, EMPTY_STRING);
    } else {
        Refresh();
    }
}

void EncyclopediaDetailPanel::SetGraph(const std::string& graph_id)
{ AddItem(TextLinker::GRAPH_TAG, graph_id); }

void EncyclopediaDetailPanel::SetIndex()
{ AddItem(TextLinker::ENCYCLOPEDIA_TAG, "ENC_INDEX"); }

void EncyclopediaDetailPanel::SetItem(TemporaryPtr<const Planet> planet)
{ SetPlanet(planet ? planet->ID() : INVALID_OBJECT_ID); }

void EncyclopediaDetailPanel::SetItem(const Tech* tech)
{ SetTech(tech ? tech->Name() : EMPTY_STRING); }

void EncyclopediaDetailPanel::SetItem(const PartType* part)
{ SetPartType(part ? part->Name() : EMPTY_STRING); }

void EncyclopediaDetailPanel::SetItem(const HullType* hull_type)
{ SetHullType(hull_type ? hull_type->Name() : EMPTY_STRING); }

void EncyclopediaDetailPanel::SetItem(const BuildingType* building_type)
{ SetBuildingType(building_type ? building_type->Name() : EMPTY_STRING); }

void EncyclopediaDetailPanel::SetItem(const Special* special)
{ SetSpecial(special ? special->Name() : EMPTY_STRING); }

void EncyclopediaDetailPanel::SetItem(const Species* species)
{ SetSpecies(species ? species->Name() : EMPTY_STRING); }

void EncyclopediaDetailPanel::SetItem(const FieldType* field_type)
{ SetFieldType(field_type ? field_type->Name() : EMPTY_STRING); }

void EncyclopediaDetailPanel::SetItem(TemporaryPtr<const UniverseObject> obj)
{ SetObject(obj ? obj->ID() : INVALID_OBJECT_ID); }

void EncyclopediaDetailPanel::SetItem(const Empire* empire)
{ SetEmpire(empire ? empire->EmpireID() : ALL_EMPIRES); }

void EncyclopediaDetailPanel::SetItem(const ShipDesign* design)
{ SetDesign(design ? design->ID() : ShipDesign::INVALID_DESIGN_ID); }

void EncyclopediaDetailPanel::OnIndex()
{ AddItem(TextLinker::ENCYCLOPEDIA_TAG, "ENC_INDEX"); }

void EncyclopediaDetailPanel::OnBack() {
    if (m_items_it != m_items.begin())
        m_items_it--;

    if (m_items_it == m_items.begin())              // disable Back button, if the beginning is reached
        m_back_button->Disable(true);
    if (m_next_button->Disabled())                  // enable Next button
        m_next_button->Disable(false);

    Refresh();
    m_description_box->SetScrollPosition(GG::Pt(GG::X0, GG::Y0));   // revert to top for new screen
}

void EncyclopediaDetailPanel::OnNext() {
    std::list<std::pair <std::string, std::string> >::iterator end = m_items.end();
    end--;
    if (m_items_it != end && !m_items.empty())
        m_items_it++;

    if (m_items_it == end)                          // disable Next button, if the end is reached;
        m_next_button->Disable(true);
    if (m_back_button->Disabled())                  // enable Back button
        m_back_button->Disable(false);

    Refresh();
    m_description_box->SetScrollPosition(GG::Pt(GG::X0, GG::Y0));   // revert to top for new screen
}

void EncyclopediaDetailPanel::CloseClicked()
{ ClosingSignal(); }

Encyclopedia::Encyclopedia() :
    articles()
{
    parse::encyclopedia_articles(GetResourceDir() / "encyclopedia.txt", *this);
    if (GetOptionsDB().Get<bool>("verbose-logging")) {
        Logger().debugStream() << "(Category) Encyclopedia Articles:";
        for (std::map<std::string, std::vector<EncyclopediaArticle> >::const_iterator
             category_it = articles.begin(); category_it != articles.end(); ++category_it)
        {
            const std::string& category = category_it->first;
            const std::vector<EncyclopediaArticle>& article_vec = category_it->second;
            for (std::vector<EncyclopediaArticle>::const_iterator article_it = article_vec.begin();
                 article_it != article_vec.end(); ++article_it)
            { Logger().debugStream() << "(" << UserString(category) << ") : " << UserString(article_it->name); }
        }
    }
}
