;
; Emacs mode for Julia
;

; USAGE
; =====

; Put the following code in your .emacs, site-load.el, or other relevant file
; (add-to-list 'load-path "path-to-julia-mode")
; (require 'julia-mode)


(defvar julia-mode-hook nil)

(add-to-list 'auto-mode-alist '("\\.jl\\'" . julia-mode))

;; define ignore-errors macro if it isn't present
;; (necessary for emacs 22 compatibility)
(when (not (fboundp 'ignore-errors))
  (defmacro ignore-errors (body) `(condition-case nil ,body (error nil))))

(defvar julia-mode-syntax-table
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?_ "w" table)   ; underscores in words
    (modify-syntax-entry ?@ "w" table)
    (modify-syntax-entry ?. "_" table)
    (modify-syntax-entry ?# "< 14" table)  ; # single-line and multiline start
    (modify-syntax-entry ?= ". 23bn" table)
    (modify-syntax-entry ?\n ">" table)  ; \n single-line comment end
    (modify-syntax-entry ?\{ "(} " table)
    (modify-syntax-entry ?\} "){ " table)
    (modify-syntax-entry ?\[ "(] " table)
    (modify-syntax-entry ?\] ")[ " table)
    (modify-syntax-entry ?\( "() " table)
    (modify-syntax-entry ?\) ")( " table)
    ;(modify-syntax-entry ?\\ "." table)  ; \ is an operator outside quotes
    (modify-syntax-entry ?'  "." table)  ; character quote or transpose
    (modify-syntax-entry ?\" "\"" table)
    (modify-syntax-entry ?` "\"" table)
    ;; (modify-syntax-entry ?\" "." table)
    (modify-syntax-entry ?? "." table)
    (modify-syntax-entry ?$ "." table)
    (modify-syntax-entry ?& "." table)
    (modify-syntax-entry ?* "." table)
    (modify-syntax-entry ?+ "." table)
    (modify-syntax-entry ?- "." table)
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)
    (modify-syntax-entry ?% "." table)
    table)
  "Syntax table for `julia-mode'.")

;; syntax table that holds within strings
(defvar julia-mode-string-syntax-table
  (let ((table (make-syntax-table)))
    table)
  "Syntax table for `julia-mode'.")

;; disable " inside char quote
(defvar julia-mode-char-syntax-table
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?\" "." table)
    table)
  "Syntax table for `julia-mode'.")

(defconst julia-string-regex
  "\"[^\"]*?\\(\\(\\\\\\\\\\)*\\\\\"[^\"]*?\\)*\"")

(defconst julia-char-regex
  "\\(\\s(\\|\\s-\\|-\\|[,%=<>\\+*/?&|$!\\^~\\\\;:]\\|^\\)\\('\\(\\([^']*?[^\\\\]\\)\\|\\(\\\\\\\\\\)\\)'\\)")

(defconst julia-unquote-regex
  "\\(\\s(\\|\\s-\\|-\\|[,%=<>\\+*/?&|!\\^~\\\\;:]\\|^\\)\\($[a-zA-Z0-9_]+\\)")

(defconst julia-forloop-in-regex
  "for +.*[^
].* \\(in\\)\\(\\s-\\|$\\)+")

(defconst julia-font-lock-keywords
  (list '("\\<\\(\\|Uint\\(8\\|16\\|32\\|64\\|128\\)\\|Int\\(8\\|16\\|32\\|64\\|128\\)\\|BigInt\\|Integer\\|BigFloat\\|FloatingPoint\\|Float16\\|Float32\\|Float64\\|Complex128\\|Complex64\\|ComplexPair\\|Bool\\|Char\\|DataType\\|Number\\|Real\\|Int\\|Uint\\|Array\\|DArray\\|AbstractArray\\|AbstractVector\\|AbstractMatrix\\|AbstractSparseMatrix\\|SubArray\\|StridedArray\\|StridedVector\\|StridedMatrix\\|VecOrMat\\|StridedVecOrMat\\|DenseArray\\|Range\\|OrdinalRange\\|StepRange\\|UnitRange\\|FloatRange\\|SparseMatrixCSC\\|Tuple\\|NTuple\\|Symbol\\|Function\\|Vector\\|Matrix\\|Union\\|Type\\|Any\\|Complex\\|None\\|String\\|Ptr\\|Void\\|Exception\\|Task\\|Signed\\|Unsigned\\|Associative\\|Dict\\|IO\\|IOStream\\|Rational\\|Regex\\|RegexMatch\\|Set\\|IntSet\\|ASCIIString\\|UTF8String\\|ByteString\\|Expr\\|WeakRef\\|Nothing\\|ObjectIdDict\\|SubString\\)\\>" .
      font-lock-type-face)
    (cons
     (concat "\\<\\("
         (mapconcat
          'identity
          '("if" "else" "elseif" "while" "for" "begin" "end" "quote"
            "try" "catch" "return" "local" "abstract" "function" "macro" "ccall"
	    "finally" "typealias" "break" "continue" "type" "global" "@\\w+"
	    "module" "using" "import" "export" "const" "let" "bitstype" "do"
	    "baremodule" "importall" "immutable")
          "\\|") "\\)\\>")
     'font-lock-keyword-face)
    '("\\<\\(true\\|false\\|C_NULL\\|Inf\\|NaN\\|Inf32\\|NaN32\\|nothing\\)\\>" . font-lock-constant-face)
    (list julia-unquote-regex 2 'font-lock-constant-face)
    (list julia-char-regex 2 'font-lock-string-face)
    (list julia-forloop-in-regex 1 'font-lock-keyword-face)
    ;(list julia-string-regex 0 'font-lock-string-face)
))

(defconst julia-block-start-keywords
  (list "if" "while" "for" "begin" "try" "function" "type" "let" "macro"
	"quote" "do" "immutable"))

(defconst julia-block-other-keywords
  (list "else" "elseif"))

(defconst julia-block-end-keywords
  (list "end" "else" "elseif" "catch" "finally"))

(defun julia-member (item lst)
  (if (null lst)
      nil
    (or (equal item (car lst))
	(julia-member item (cdr lst)))))

(if (not (fboundp 'evenp))
    (defun evenp (x) (zerop (% x 2))))

(defun julia-find-comment-open (p0)
  (if (< (point) p0)
      nil
    (if (and (equal (char-after (point)) ?#)
	     (evenp (julia-strcount
		     (buffer-substring p0 (point)) ?\")))
	t
      (if (= (point) p0)
	  nil
	(progn (backward-char 1)
	       (julia-find-comment-open p0))))))

(defun julia-in-comment ()
  (save-excursion
    (julia-find-comment-open (line-beginning-position))))

(defun julia-strcount (str chr)
  (let ((i 0)
	(c 0))
    (while (< i (length str))
      (if (equal (elt str i) chr)
	  (setq c (+ c 1)))
      (setq i (+ i 1)))
    c))

(defun julia-in-brackets ()
  (let ((before (buffer-substring (line-beginning-position) (point))))
    (> (julia-strcount before ?[)
       (julia-strcount before ?]))))

(defun julia-at-keyword (kw-list)
  "Return the word at point if it matches any keyword in KW-LIST.
KW-LIST is a list of strings.  The word at point is not considered
a keyword if used as a field name, X.word, or quoted, :word."
  (and (or (= (point) 1)
	   (and (not (equal (char-before (point)) ?.))
		(not (equal (char-before (point)) ?:))))
       (not (julia-in-comment))
       (not (julia-in-brackets))
       (julia-member (current-word t) kw-list)))

;; if backward-sexp gives an error, move back 1 char to move over the '('
(defun julia-safe-backward-sexp ()
  (if (condition-case nil (backward-sexp) (error t))
      (ignore-errors (backward-char))))

(defun julia-last-open-block-pos (min)
  "Move back and return the position of the last open block, if one found.
Do not move back beyond position MIN."
  (let ((count 0))
    (while (not (or (> count 0) (<= (point) min)))
      (julia-safe-backward-sexp)
      (setq count
	    (cond ((julia-at-keyword julia-block-start-keywords)
		   (+ count 1))
		  ((and (equal (current-word t) "end")
			(not (julia-in-comment)) (not (julia-in-brackets)))
		   (- count 1))
		  (t count))))
    (if (> count 0)
	(point)
      nil)))

(defun julia-last-open-block (min)
  "Move back and return indentation level for last open block.
Do not move back beyond MIN."
  (let ((pos (julia-last-open-block-pos min)))
    (and pos
	 (progn
	   (goto-char pos)
	   (+ julia-basic-offset (current-indentation))))))

(defun julia-paren-indent ()
  "Return indent by last opening paren."
  (let* ((p (parse-partial-sexp
             (save-excursion
               ;; only indent by paren if the last open
               ;; paren is closer than the last open
               ;; block
               (or (julia-last-open-block-pos (point-min))
                   (point-min)))
             (progn (beginning-of-line)
                    (point))))
         (pos (cadr p)))
    (if (or (= 0 (car p)) (null pos))
        nil
      (progn (goto-char pos) (+ 1 (current-column))))))

(defun julia-indent-line ()
  "Indent current line of julia code."
  (interactive)
;  (save-excursion
    (end-of-line)
    (indent-line-to
     (or (save-excursion (ignore-errors (julia-paren-indent)))
         (save-excursion
           (let ((endtok (progn
                           (beginning-of-line)
                           (forward-to-indentation 0)
                           (julia-at-keyword julia-block-end-keywords))))
             (ignore-errors (+ (julia-last-open-block (point-min))
                           (if endtok (- julia-basic-offset) 0)))))
	 ;; previous line ends in =
	 (save-excursion
	   (if (and (not (equal (point-min) (line-beginning-position)))
		    (progn
		      (forward-line -1)
		      (end-of-line) (backward-char 1)
		      (equal (char-after (point)) ?=)))
	       (+ julia-basic-offset (current-indentation))
	     nil))
	 ;; take same indentation as previous line
	 (save-excursion (forward-line -1)
			 (current-indentation))
         0))
    (when (julia-at-keyword julia-block-end-keywords)
      (forward-word 1)))

(defalias 'julia-mode-prog-mode
  (if (fboundp 'prog-mode)
      'prog-mode
    'fundamental-mode))

;;;###autoload
(define-derived-mode julia-mode julia-mode-prog-mode "Julia"
  "Major mode for editing julia code."
  (set-syntax-table julia-mode-syntax-table)
  (set (make-local-variable 'comment-start) "# ")
  (set (make-local-variable 'comment-start-skip) "#+\\s-*")
  (set (make-local-variable 'font-lock-defaults) '(julia-font-lock-keywords))
  (set (make-local-variable 'font-lock-syntactic-keywords)
       (list
	(list "\\(\\\\\\)\\s-*\".*?\"" 1 julia-mode-char-syntax-table)))
  (set (make-local-variable 'font-lock-syntactic-keywords)
        (list
 	(list julia-char-regex 2
 	      julia-mode-char-syntax-table)
        (list julia-string-regex 0
              julia-mode-string-syntax-table)
	))
  (set (make-local-variable 'indent-line-function) 'julia-indent-line)
  (set (make-local-variable 'julia-basic-offset) 4)
  (setq indent-tabs-mode nil)
  (setq imenu-generic-expression julia-imenu-generic-expression)
  (imenu-add-to-menubar "Imenu"))

;;; IMENU
(defvar julia-imenu-generic-expression
  ;; don't use syntax classes, screws egrep
  '(("Function (_)" "[ \t]*function[ \t]+\\(_[^ \t\n]*\\)" 1)
    ("Function" "^[ \t]*function[ \t]+\\([^_][^\t\n]*\\)" 1)
    ("Const" "[ \t]*const \\([^ \t\n]*\\)" 1)
    ("Type"  "^[ \t]*[a-zA-Z0-9_]*type[a-zA-Z0-9_]* \\([^ \t\n]*\\)" 1)
    ("Require"      " *\\(\\brequire\\)(\\([^ \t\n)]*\\)" 2)
    ("Include"      " *\\(\\binclude\\)(\\([^ \t\n)]*\\)" 2)
    ;; ("Classes" "^.*setClass(\\(.*\\)," 1)
    ;; ("Coercions" "^.*setAs(\\([^,]+,[^,]*\\)," 1) ; show from and to
    ;; ("Generics" "^.*setGeneric(\\([^,]*\\)," 1)
    ;; ("Methods" "^.*set\\(Group\\|Replace\\)?Method(\"\\(.+\\)\"," 2)
    ;; ;;[ ]*\\(signature=\\)?(\\(.*,?\\)*\\)," 1)
    ;; ;;
    ;; ;;("Other" "^\\(.+\\)\\s-*<-[ \t\n]*[^\\(function\\|read\\|.*data\.frame\\)]" 1)
    ;; ("Package" "^.*\\(library\\|require\\)(\\(.*\\)," 2)
    ;; ("Data" "^\\(.+\\)\\s-*<-[ \t\n]*\\(read\\|.*data\.frame\\).*(" 1)))
    ))

(setq latexsubs (make-hash-table :test 'equal))
(defun latexsub (arg)
  "Perform a LaTeX-like Unicode symbol substitution."
  (interactive "*i")
  (let ((orig-pt (point)))
    (while (not (or (bobp) (= ?\\ (char-before))
		    (= ?\s (char-syntax (char-before)))))
      (backward-char))
    (if (and (not (bobp)) (= ?\\ (char-before)))
        (progn
          (backward-char)
          (let ((sub (gethash (buffer-substring (point) orig-pt) latexsubs)))
            (if sub
                (progn
                  (delete-region (point) orig-pt)
                  (insert sub))
              (goto-char orig-pt))))
      (goto-char orig-pt))))

(defun latexsub-or-indent (arg)
  "Either indent according to mode or perform a LaTeX-like symbol substution"
  (interactive "*i")
  (if (latexsub arg)
      (indent-according-to-mode)))
(define-key julia-mode-map (kbd "TAB") 'latexsub-or-indent)

; LaTeX-like symbol substitutions, equivalent to those in the Julia REPL,
; generated by:
; for (k,v) in sort!(collect(Base.REPLCompletions.latex_symbols), by=x->x[2])
;    println("(puthash \"$(escape_string(k))\" \"$(escape_string(v))\" latexsubs)")
; end
(puthash "\\textexclamdown" "¡" latexsubs)
(puthash "\\sterling" "£" latexsubs)
(puthash "\\yen" "¥" latexsubs)
(puthash "\\textbrokenbar" "¦" latexsubs)
(puthash "\\S" "§" latexsubs)
(puthash "\\textasciidieresis" "¨" latexsubs)
(puthash "\\copyright" "©" latexsubs)
(puthash "\\textordfeminine" "ª" latexsubs)
(puthash "\\neg" "¬" latexsubs)
(puthash "\\circledR" "®" latexsubs)
(puthash "\\textasciimacron" "¯" latexsubs)
(puthash "\\degree" "°" latexsubs)
(puthash "\\pm" "±" latexsubs)
(puthash "\\^2" "²" latexsubs)
(puthash "\\^3" "³" latexsubs)
(puthash "\\textasciiacute" "´" latexsubs)
(puthash "\\P" "¶" latexsubs)
(puthash "\\cdotp" "·" latexsubs)
(puthash "\\^1" "¹" latexsubs)
(puthash "\\textordmasculine" "º" latexsubs)
(puthash "\\textonequarter" "¼" latexsubs)
(puthash "\\textonehalf" "½" latexsubs)
(puthash "\\textthreequarters" "¾" latexsubs)
(puthash "\\textquestiondown" "¿" latexsubs)
(puthash "\\AA" "Å" latexsubs)
(puthash "\\AE" "Æ" latexsubs)
(puthash "\\DH" "Ð" latexsubs)
(puthash "\\times" "×" latexsubs)
(puthash "\\O" "Ø" latexsubs)
(puthash "\\TH" "Þ" latexsubs)
(puthash "\\ss" "ß" latexsubs)
(puthash "\\aa" "å" latexsubs)
(puthash "\\ae" "æ" latexsubs)
(puthash "\\eth" "ð" latexsubs)
(puthash "\\div" "÷" latexsubs)
(puthash "\\o" "ø" latexsubs)
(puthash "\\th" "þ" latexsubs)
(puthash "\\DJ" "Đ" latexsubs)
(puthash "\\dj" "đ" latexsubs)
(puthash "\\Elzxh" "ħ" latexsubs)
(puthash "\\hbar" "ħ" latexsubs)
(puthash "\\imath" "ı" latexsubs)
(puthash "\\L" "Ł" latexsubs)
(puthash "\\l" "ł" latexsubs)
(puthash "\\NG" "Ŋ" latexsubs)
(puthash "\\ng" "ŋ" latexsubs)
(puthash "\\OE" "Œ" latexsubs)
(puthash "\\oe" "œ" latexsubs)
(puthash "\\texthvlig" "ƕ" latexsubs)
(puthash "\\textnrleg" "ƞ" latexsubs)
(puthash "\\textdoublepipe" "ǂ" latexsubs)
(puthash "\\Elztrna" "ɐ" latexsubs)
(puthash "\\Elztrnsa" "ɒ" latexsubs)
(puthash "\\Elzopeno" "ɔ" latexsubs)
(puthash "\\Elzrtld" "ɖ" latexsubs)
(puthash "\\Elzschwa" "ə" latexsubs)
(puthash "\\varepsilon" "ɛ" latexsubs)
(puthash "\\Elzpgamma" "ɣ" latexsubs)
(puthash "\\Elzpbgam" "ɤ" latexsubs)
(puthash "\\Elztrnh" "ɥ" latexsubs)
(puthash "\\Elzbtdl" "ɬ" latexsubs)
(puthash "\\Elzrtll" "ɭ" latexsubs)
(puthash "\\Elztrnm" "ɯ" latexsubs)
(puthash "\\Elztrnmlr" "ɰ" latexsubs)
(puthash "\\Elzltlmr" "ɱ" latexsubs)
(puthash "\\Elzltln" "ɲ" latexsubs)
(puthash "\\Elzrtln" "ɳ" latexsubs)
(puthash "\\Elzclomeg" "ɷ" latexsubs)
(puthash "\\textphi" "ɸ" latexsubs)
(puthash "\\Elztrnr" "ɹ" latexsubs)
(puthash "\\Elztrnrl" "ɺ" latexsubs)
(puthash "\\Elzrttrnr" "ɻ" latexsubs)
(puthash "\\Elzrl" "ɼ" latexsubs)
(puthash "\\Elzrtlr" "ɽ" latexsubs)
(puthash "\\Elzfhr" "ɾ" latexsubs)
(puthash "\\Elzrtls" "ʂ" latexsubs)
(puthash "\\Elzesh" "ʃ" latexsubs)
(puthash "\\Elztrnt" "ʇ" latexsubs)
(puthash "\\Elzrtlt" "ʈ" latexsubs)
(puthash "\\Elzpupsil" "ʊ" latexsubs)
(puthash "\\Elzpscrv" "ʋ" latexsubs)
(puthash "\\Elzinvv" "ʌ" latexsubs)
(puthash "\\Elzinvw" "ʍ" latexsubs)
(puthash "\\Elztrny" "ʎ" latexsubs)
(puthash "\\Elzrtlz" "ʐ" latexsubs)
(puthash "\\Elzyogh" "ʒ" latexsubs)
(puthash "\\Elzglst" "ʔ" latexsubs)
(puthash "\\Elzreglst" "ʕ" latexsubs)
(puthash "\\Elzinglst" "ʖ" latexsubs)
(puthash "\\textturnk" "ʞ" latexsubs)
(puthash "\\Elzdyogh" "ʤ" latexsubs)
(puthash "\\Elztesh" "ʧ" latexsubs)
(puthash "\\^h" "ʰ" latexsubs)
(puthash "\\^j" "ʲ" latexsubs)
(puthash "\\^r" "ʳ" latexsubs)
(puthash "\\^w" "ʷ" latexsubs)
(puthash "\\^y" "ʸ" latexsubs)
(puthash "\\rasp" "ʼ" latexsubs)
(puthash "\\textasciicaron" "ˇ" latexsubs)
(puthash "\\Elzverts" "ˈ" latexsubs)
(puthash "\\Elzverti" "ˌ" latexsubs)
(puthash "\\Elzlmrk" "ː" latexsubs)
(puthash "\\Elzhlmrk" "ˑ" latexsubs)
(puthash "\\Elzsbrhr" "˒" latexsubs)
(puthash "\\Elzsblhr" "˓" latexsubs)
(puthash "\\Elzrais" "˔" latexsubs)
(puthash "\\Elzlow" "˕" latexsubs)
(puthash "\\u" "˘" latexsubs)
(puthash "\\texttildelow" "˜" latexsubs)
(puthash "\\^l" "ˡ" latexsubs)
(puthash "\\^s" "ˢ" latexsubs)
(puthash "\\^x" "ˣ" latexsubs)
(puthash "\\grave" "̀" latexsubs)
(puthash "\\acute" "́" latexsubs)
(puthash "\\hat" "̂" latexsubs)
(puthash "\\tilde" "̃" latexsubs)
(puthash "\\bar" "̄" latexsubs)
(puthash "\\breve" "̆" latexsubs)
(puthash "\\dot" "̇" latexsubs)
(puthash "\\ddot" "̈" latexsubs)
(puthash "\\ocirc" "̊" latexsubs)
(puthash "\\H" "̋" latexsubs)
(puthash "\\check" "̌" latexsubs)
(puthash "\\Elzpalh" "̡" latexsubs)
(puthash "\\Elzrh" "̢" latexsubs)
(puthash "\\c" "̧" latexsubs)
(puthash "\\k" "̨" latexsubs)
(puthash "\\Elzsbbrg" "̪" latexsubs)
(puthash "\\Elzxl" "̵" latexsubs)
(puthash "\\Elzbar" "̶" latexsubs)
(puthash "\\sout" "̶" latexsubs)
(puthash "\\Alpha" "Α" latexsubs)
(puthash "\\Beta" "Β" latexsubs)
(puthash "\\Gamma" "Γ" latexsubs)
(puthash "\\Delta" "Δ" latexsubs)
(puthash "\\Epsilon" "Ε" latexsubs)
(puthash "\\Zeta" "Ζ" latexsubs)
(puthash "\\Eta" "Η" latexsubs)
(puthash "\\Theta" "Θ" latexsubs)
(puthash "\\Iota" "Ι" latexsubs)
(puthash "\\Kappa" "Κ" latexsubs)
(puthash "\\Lambda" "Λ" latexsubs)
(puthash "\\Xi" "Ξ" latexsubs)
(puthash "\\Pi" "Π" latexsubs)
(puthash "\\Rho" "Ρ" latexsubs)
(puthash "\\Sigma" "Σ" latexsubs)
(puthash "\\Tau" "Τ" latexsubs)
(puthash "\\Upsilon" "Υ" latexsubs)
(puthash "\\Phi" "Φ" latexsubs)
(puthash "\\Chi" "Χ" latexsubs)
(puthash "\\Psi" "Ψ" latexsubs)
(puthash "\\Omega" "Ω" latexsubs)
(puthash "\\alpha" "α" latexsubs)
(puthash "\\beta" "β" latexsubs)
(puthash "\\gamma" "γ" latexsubs)
(puthash "\\delta" "δ" latexsubs)
(puthash "\\zeta" "ζ" latexsubs)
(puthash "\\eta" "η" latexsubs)
(puthash "\\theta" "θ" latexsubs)
(puthash "\\iota" "ι" latexsubs)
(puthash "\\kappa" "κ" latexsubs)
(puthash "\\lambda" "λ" latexsubs)
(puthash "\\mu" "μ" latexsubs)
(puthash "\\nu" "ν" latexsubs)
(puthash "\\xi" "ξ" latexsubs)
(puthash "\\pi" "π" latexsubs)
(puthash "\\rho" "ρ" latexsubs)
(puthash "\\varsigma" "ς" latexsubs)
(puthash "\\sigma" "σ" latexsubs)
(puthash "\\tau" "τ" latexsubs)
(puthash "\\upsilon" "υ" latexsubs)
(puthash "\\varphi" "φ" latexsubs)
(puthash "\\chi" "χ" latexsubs)
(puthash "\\psi" "ψ" latexsubs)
(puthash "\\omega" "ω" latexsubs)
(puthash "\\vartheta" "ϑ" latexsubs)
(puthash "\\phi" "ϕ" latexsubs)
(puthash "\\varpi" "ϖ" latexsubs)
(puthash "\\Stigma" "Ϛ" latexsubs)
(puthash "\\Digamma" "Ϝ" latexsubs)
(puthash "\\digamma" "ϝ" latexsubs)
(puthash "\\Koppa" "Ϟ" latexsubs)
(puthash "\\Sampi" "Ϡ" latexsubs)
(puthash "\\varkappa" "ϰ" latexsubs)
(puthash "\\varrho" "ϱ" latexsubs)
(puthash "\\textTheta" "ϴ" latexsubs)
(puthash "\\epsilon" "ϵ" latexsubs)
(puthash "\\backepsilon" "϶" latexsubs)
(puthash "\\^A" "\u1d2c" latexsubs)
(puthash "\\^B" "\u1d2e" latexsubs)
(puthash "\\^D" "\u1d30" latexsubs)
(puthash "\\^E" "\u1d31" latexsubs)
(puthash "\\^G" "\u1d33" latexsubs)
(puthash "\\^H" "\u1d34" latexsubs)
(puthash "\\^I" "\u1d35" latexsubs)
(puthash "\\^J" "\u1d36" latexsubs)
(puthash "\\^K" "\u1d37" latexsubs)
(puthash "\\^L" "\u1d38" latexsubs)
(puthash "\\^M" "\u1d39" latexsubs)
(puthash "\\^N" "\u1d3a" latexsubs)
(puthash "\\^O" "\u1d3c" latexsubs)
(puthash "\\^P" "\u1d3e" latexsubs)
(puthash "\\^R" "\u1d3f" latexsubs)
(puthash "\\^T" "\u1d40" latexsubs)
(puthash "\\^U" "\u1d41" latexsubs)
(puthash "\\^W" "\u1d42" latexsubs)
(puthash "\\^a" "\u1d43" latexsubs)
(puthash "\\^alpha" "\u1d45" latexsubs)
(puthash "\\^b" "\u1d47" latexsubs)
(puthash "\\^d" "\u1d48" latexsubs)
(puthash "\\^e" "\u1d49" latexsubs)
(puthash "\\^epsilon" "\u1d4b" latexsubs)
(puthash "\\^g" "\u1d4d" latexsubs)
(puthash "\\^k" "\u1d4f" latexsubs)
(puthash "\\^m" "\u1d50" latexsubs)
(puthash "\\^o" "\u1d52" latexsubs)
(puthash "\\^p" "\u1d56" latexsubs)
(puthash "\\^t" "\u1d57" latexsubs)
(puthash "\\^u" "\u1d58" latexsubs)
(puthash "\\^v" "\u1d5b" latexsubs)
(puthash "\\^beta" "\u1d5d" latexsubs)
(puthash "\\^gamma" "\u1d5e" latexsubs)
(puthash "\\^delta" "\u1d5f" latexsubs)
(puthash "\\^phi" "\u1d60" latexsubs)
(puthash "\\^chi" "\u1d61" latexsubs)
(puthash "\\_i" "\u1d62" latexsubs)
(puthash "\\_r" "\u1d63" latexsubs)
(puthash "\\_u" "\u1d64" latexsubs)
(puthash "\\_v" "\u1d65" latexsubs)
(puthash "\\_beta" "\u1d66" latexsubs)
(puthash "\\_gamma" "\u1d67" latexsubs)
(puthash "\\_rho" "\u1d68" latexsubs)
(puthash "\\_phi" "\u1d69" latexsubs)
(puthash "\\_chi" "\u1d6a" latexsubs)
(puthash "\\^c" "\u1d9c" latexsubs)
(puthash "\\^f" "\u1da0" latexsubs)
(puthash "\\^iota" "\u1da5" latexsubs)
(puthash "\\^Phi" "\u1db2" latexsubs)
(puthash "\\^z" "\u1dbb" latexsubs)
(puthash "\\^theta" "\u1dbf" latexsubs)
(puthash "\\enspace" " " latexsubs)
(puthash "\\quad" " " latexsubs)
(puthash "\\thickspace" " " latexsubs)
(puthash "\\thinspace" " " latexsubs)
(puthash "\\hspace" " " latexsubs)
(puthash "\\endash" "–" latexsubs)
(puthash "\\emdash" "—" latexsubs)
(puthash "\\Vert" "‖" latexsubs)
(puthash "\\lq" "‘" latexsubs)
(puthash "\\rq" "’" latexsubs)
(puthash "\\Elzreapos" "‛" latexsubs)
(puthash "\\textquotedblleft" "“" latexsubs)
(puthash "\\textquotedblright" "”" latexsubs)
(puthash "\\dagger" "†" latexsubs)
(puthash "\\ddagger" "‡" latexsubs)
(puthash "\\bullet" "•" latexsubs)
(puthash "\\dots" "…" latexsubs)
(puthash "\\textperthousand" "‰" latexsubs)
(puthash "\\textpertenthousand" "‱" latexsubs)
(puthash "\\prime" "′" latexsubs)
(puthash "\\backprime" "‵" latexsubs)
(puthash "\\guilsinglleft" "‹" latexsubs)
(puthash "\\guilsinglright" "›" latexsubs)
(puthash "\\nolinebreak" "\u2060" latexsubs)
(puthash "\\^0" "⁰" latexsubs)
(puthash "\\^i" "ⁱ" latexsubs)
(puthash "\\^4" "⁴" latexsubs)
(puthash "\\^5" "⁵" latexsubs)
(puthash "\\^6" "⁶" latexsubs)
(puthash "\\^7" "⁷" latexsubs)
(puthash "\\^8" "⁸" latexsubs)
(puthash "\\^9" "⁹" latexsubs)
(puthash "\\^+" "⁺" latexsubs)
(puthash "\\^-" "⁻" latexsubs)
(puthash "\\^=" "⁼" latexsubs)
(puthash "\\^(" "⁽" latexsubs)
(puthash "\\^)" "⁾" latexsubs)
(puthash "\\^n" "ⁿ" latexsubs)
(puthash "\\_0" "₀" latexsubs)
(puthash "\\_1" "₁" latexsubs)
(puthash "\\_2" "₂" latexsubs)
(puthash "\\_3" "₃" latexsubs)
(puthash "\\_4" "₄" latexsubs)
(puthash "\\_5" "₅" latexsubs)
(puthash "\\_6" "₆" latexsubs)
(puthash "\\_7" "₇" latexsubs)
(puthash "\\_8" "₈" latexsubs)
(puthash "\\_9" "₉" latexsubs)
(puthash "\\_+" "₊" latexsubs)
(puthash "\\_-" "₋" latexsubs)
(puthash "\\_=" "₌" latexsubs)
(puthash "\\_(" "₍" latexsubs)
(puthash "\\_)" "₎" latexsubs)
(puthash "\\_a" "\u2090" latexsubs)
(puthash "\\_e" "\u2091" latexsubs)
(puthash "\\_o" "\u2092" latexsubs)
(puthash "\\_x" "\u2093" latexsubs)
(puthash "\\_schwa" "\u2094" latexsubs)
(puthash "\\_h" "\u2095" latexsubs)
(puthash "\\_k" "\u2096" latexsubs)
(puthash "\\_l" "\u2097" latexsubs)
(puthash "\\_m" "\u2098" latexsubs)
(puthash "\\_n" "\u2099" latexsubs)
(puthash "\\_p" "\u209a" latexsubs)
(puthash "\\_s" "\u209b" latexsubs)
(puthash "\\_t" "\u209c" latexsubs)
(puthash "\\Elzpes" "₧" latexsubs)
(puthash "\\euro" "€" latexsubs)
(puthash "\\dddot" "⃛" latexsubs)
(puthash "\\ddddot" "⃜" latexsubs)
(puthash "\\hslash" "ℏ" latexsubs)
(puthash "\\Im" "ℑ" latexsubs)
(puthash "\\ell" "ℓ" latexsubs)
(puthash "\\textnumero" "№" latexsubs)
(puthash "\\wp" "℘" latexsubs)
(puthash "\\Re" "ℜ" latexsubs)
(puthash "\\Elzxrat" "℞" latexsubs)
(puthash "\\texttrademark" "™" latexsubs)
(puthash "\\mho" "℧" latexsubs)
(puthash "\\aleph" "ℵ" latexsubs)
(puthash "\\beth" "ℶ" latexsubs)
(puthash "\\gimel" "ℷ" latexsubs)
(puthash "\\daleth" "ℸ" latexsubs)
(puthash "\\BbbPi" "ℿ" latexsubs)
(puthash "\\bbsum" "⅀" latexsubs)
(puthash "\\Game" "⅁" latexsubs)
(puthash "\\leftarrow" "←" latexsubs)
(puthash "\\uparrow" "↑" latexsubs)
(puthash "\\rightarrow" "→" latexsubs)
(puthash "\\downarrow" "↓" latexsubs)
(puthash "\\leftrightarrow" "↔" latexsubs)
(puthash "\\updownarrow" "↕" latexsubs)
(puthash "\\nwarrow" "↖" latexsubs)
(puthash "\\nearrow" "↗" latexsubs)
(puthash "\\searrow" "↘" latexsubs)
(puthash "\\swarrow" "↙" latexsubs)
(puthash "\\nleftarrow" "↚" latexsubs)
(puthash "\\nrightarrow" "↛" latexsubs)
(puthash "\\leftsquigarrow" "↜" latexsubs)
(puthash "\\rightsquigarrow" "↝" latexsubs)
(puthash "\\twoheadleftarrow" "↞" latexsubs)
(puthash "\\twoheadrightarrow" "↠" latexsubs)
(puthash "\\leftarrowtail" "↢" latexsubs)
(puthash "\\rightarrowtail" "↣" latexsubs)
(puthash "\\mapsto" "↦" latexsubs)
(puthash "\\hookleftarrow" "↩" latexsubs)
(puthash "\\hookrightarrow" "↪" latexsubs)
(puthash "\\looparrowleft" "↫" latexsubs)
(puthash "\\looparrowright" "↬" latexsubs)
(puthash "\\leftrightsquigarrow" "↭" latexsubs)
(puthash "\\nleftrightarrow" "↮" latexsubs)
(puthash "\\Lsh" "↰" latexsubs)
(puthash "\\Rsh" "↱" latexsubs)
(puthash "\\curvearrowleft" "↶" latexsubs)
(puthash "\\curvearrowright" "↷" latexsubs)
(puthash "\\circlearrowleft" "↺" latexsubs)
(puthash "\\circlearrowright" "↻" latexsubs)
(puthash "\\leftharpoonup" "↼" latexsubs)
(puthash "\\leftharpoondown" "↽" latexsubs)
(puthash "\\upharpoonleft" "↾" latexsubs)
(puthash "\\upharpoonright" "↿" latexsubs)
(puthash "\\rightharpoonup" "⇀" latexsubs)
(puthash "\\rightharpoondown" "⇁" latexsubs)
(puthash "\\downharpoonright" "⇂" latexsubs)
(puthash "\\downharpoonleft" "⇃" latexsubs)
(puthash "\\rightleftarrows" "⇄" latexsubs)
(puthash "\\dblarrowupdown" "⇅" latexsubs)
(puthash "\\leftrightarrows" "⇆" latexsubs)
(puthash "\\leftleftarrows" "⇇" latexsubs)
(puthash "\\upuparrows" "⇈" latexsubs)
(puthash "\\rightrightarrows" "⇉" latexsubs)
(puthash "\\downdownarrows" "⇊" latexsubs)
(puthash "\\leftrightharpoons" "⇋" latexsubs)
(puthash "\\rightleftharpoons" "⇌" latexsubs)
(puthash "\\nLeftarrow" "⇍" latexsubs)
(puthash "\\nRightarrow" "⇏" latexsubs)
(puthash "\\Leftarrow" "⇐" latexsubs)
(puthash "\\Uparrow" "⇑" latexsubs)
(puthash "\\Rightarrow" "⇒" latexsubs)
(puthash "\\Downarrow" "⇓" latexsubs)
(puthash "\\Leftrightarrow" "⇔" latexsubs)
(puthash "\\Updownarrow" "⇕" latexsubs)
(puthash "\\Lleftarrow" "⇚" latexsubs)
(puthash "\\Rrightarrow" "⇛" latexsubs)
(puthash "\\DownArrowUpArrow" "⇵" latexsubs)
(puthash "\\leftarrowtriangle" "⇽" latexsubs)
(puthash "\\rightarrowtriangle" "⇾" latexsubs)
(puthash "\\forall" "∀" latexsubs)
(puthash "\\complement" "∁" latexsubs)
(puthash "\\partial" "∂" latexsubs)
(puthash "\\exists" "∃" latexsubs)
(puthash "\\nexists" "∄" latexsubs)
(puthash "\\varnothing" "∅" latexsubs)
(puthash "\\del" "∇" latexsubs)
(puthash "\\nabla" "∇" latexsubs)
(puthash "\\in" "∈" latexsubs)
(puthash "\\notin" "∉" latexsubs)
(puthash "\\ni" "∋" latexsubs)
(puthash "\\prod" "∏" latexsubs)
(puthash "\\coprod" "∐" latexsubs)
(puthash "\\sum" "∑" latexsubs)
(puthash "\\minus" "−" latexsubs)
(puthash "\\mp" "∓" latexsubs)
(puthash "\\dotplus" "∔" latexsubs)
(puthash "\\setminus" "∖" latexsubs)
(puthash "\\ast" "∗" latexsubs)
(puthash "\\circ" "∘" latexsubs)
(puthash "\\surd" "√" latexsubs)
(puthash "\\sqrt" "√" latexsubs)
(puthash "\\cbrt" "∛" latexsubs)
(puthash "\\propto" "∝" latexsubs)
(puthash "\\infty" "∞" latexsubs)
(puthash "\\rightangle" "∟" latexsubs)
(puthash "\\angle" "∠" latexsubs)
(puthash "\\measuredangle" "∡" latexsubs)
(puthash "\\sphericalangle" "∢" latexsubs)
(puthash "\\mid" "∣" latexsubs)
(puthash "\\nmid" "∤" latexsubs)
(puthash "\\parallel" "∥" latexsubs)
(puthash "\\nparallel" "∦" latexsubs)
(puthash "\\wedge" "∧" latexsubs)
(puthash "\\vee" "∨" latexsubs)
(puthash "\\cap" "∩" latexsubs)
(puthash "\\cup" "∪" latexsubs)
(puthash "\\int" "∫" latexsubs)
(puthash "\\iint" "∬" latexsubs)
(puthash "\\iiint" "∭" latexsubs)
(puthash "\\oint" "∮" latexsubs)
(puthash "\\oiint" "∯" latexsubs)
(puthash "\\oiiint" "∰" latexsubs)
(puthash "\\clwintegral" "∱" latexsubs)
(puthash "\\therefore" "∴" latexsubs)
(puthash "\\because" "∵" latexsubs)
(puthash "\\Colon" "∷" latexsubs)
(puthash "\\dotminus" "∸" latexsubs)
(puthash "\\kernelcontraction" "∻" latexsubs)
(puthash "\\sim" "∼" latexsubs)
(puthash "\\backsim" "∽" latexsubs)
(puthash "\\lazysinv" "∾" latexsubs)
(puthash "\\wr" "≀" latexsubs)
(puthash "\\nsim" "≁" latexsubs)
(puthash "\\eqsim" "≂" latexsubs)
(puthash "\\neqsim" "≂̸" latexsubs)
(puthash "\\simeq" "≃" latexsubs)
(puthash "\\nsime" "≄" latexsubs)
(puthash "\\cong" "≅" latexsubs)
(puthash "\\approxnotequal" "≆" latexsubs)
(puthash "\\ncong" "≇" latexsubs)
(puthash "\\approx" "≈" latexsubs)
(puthash "\\napprox" "≉" latexsubs)
(puthash "\\approxeq" "≊" latexsubs)
(puthash "\\tildetrpl" "≋" latexsubs)
(puthash "\\allequal" "≌" latexsubs)
(puthash "\\asymp" "≍" latexsubs)
(puthash "\\Bumpeq" "≎" latexsubs)
(puthash "\\nBumpeq" "≎̸" latexsubs)
(puthash "\\bumpeq" "≏" latexsubs)
(puthash "\\nbumpeq" "≏̸" latexsubs)
(puthash "\\doteq" "≐" latexsubs)
(puthash "\\Doteq" "≑" latexsubs)
(puthash "\\fallingdotseq" "≒" latexsubs)
(puthash "\\risingdotseq" "≓" latexsubs)
(puthash "\\coloneq" "≔" latexsubs)
(puthash "\\eqcolon" "≕" latexsubs)
(puthash "\\eqcirc" "≖" latexsubs)
(puthash "\\circeq" "≗" latexsubs)
(puthash "\\wedgeq" "≙" latexsubs)
(puthash "\\starequal" "≛" latexsubs)
(puthash "\\triangleq" "≜" latexsubs)
(puthash "\\questeq" "≟" latexsubs)
(puthash "\\ne" "≠" latexsubs)
(puthash "\\equiv" "≡" latexsubs)
(puthash "\\nequiv" "≢" latexsubs)
(puthash "\\le" "≤" latexsubs)
(puthash "\\ge" "≥" latexsubs)
(puthash "\\leqq" "≦" latexsubs)
(puthash "\\geqq" "≧" latexsubs)
(puthash "\\lneqq" "≨" latexsubs)
(puthash "\\lvertneqq" "≨︀" latexsubs)
(puthash "\\gneqq" "≩" latexsubs)
(puthash "\\gvertneqq" "≩︀" latexsubs)
(puthash "\\ll" "≪" latexsubs)
(puthash "\\NotLessLess" "≪̸" latexsubs)
(puthash "\\gg" "≫" latexsubs)
(puthash "\\NotGreaterGreater" "≫̸" latexsubs)
(puthash "\\between" "≬" latexsubs)
(puthash "\\nless" "≮" latexsubs)
(puthash "\\ngtr" "≯" latexsubs)
(puthash "\\nleq" "≰" latexsubs)
(puthash "\\ngeq" "≱" latexsubs)
(puthash "\\lesssim" "≲" latexsubs)
(puthash "\\gtrsim" "≳" latexsubs)
(puthash "\\lessgtr" "≶" latexsubs)
(puthash "\\gtrless" "≷" latexsubs)
(puthash "\\notlessgreater" "≸" latexsubs)
(puthash "\\notgreaterless" "≹" latexsubs)
(puthash "\\prec" "≺" latexsubs)
(puthash "\\succ" "≻" latexsubs)
(puthash "\\preccurlyeq" "≼" latexsubs)
(puthash "\\succcurlyeq" "≽" latexsubs)
(puthash "\\precsim" "≾" latexsubs)
(puthash "\\nprecsim" "≾̸" latexsubs)
(puthash "\\succsim" "≿" latexsubs)
(puthash "\\nsuccsim" "≿̸" latexsubs)
(puthash "\\nprec" "⊀" latexsubs)
(puthash "\\nsucc" "⊁" latexsubs)
(puthash "\\subset" "⊂" latexsubs)
(puthash "\\supset" "⊃" latexsubs)
(puthash "\\nsubset" "⊄" latexsubs)
(puthash "\\nsupset" "⊅" latexsubs)
(puthash "\\subseteq" "⊆" latexsubs)
(puthash "\\supseteq" "⊇" latexsubs)
(puthash "\\nsubseteq" "⊈" latexsubs)
(puthash "\\nsupseteq" "⊉" latexsubs)
(puthash "\\subsetneq" "⊊" latexsubs)
(puthash "\\varsubsetneqq" "⊊︀" latexsubs)
(puthash "\\supsetneq" "⊋" latexsubs)
(puthash "\\varsupsetneq" "⊋︀" latexsubs)
(puthash "\\cupdot" "⊍" latexsubs)
(puthash "\\uplus" "⊎" latexsubs)
(puthash "\\sqsubset" "⊏" latexsubs)
(puthash "\\NotSquareSubset" "⊏̸" latexsubs)
(puthash "\\sqsupset" "⊐" latexsubs)
(puthash "\\NotSquareSuperset" "⊐̸" latexsubs)
(puthash "\\sqsubseteq" "⊑" latexsubs)
(puthash "\\sqsupseteq" "⊒" latexsubs)
(puthash "\\sqcap" "⊓" latexsubs)
(puthash "\\sqcup" "⊔" latexsubs)
(puthash "\\oplus" "⊕" latexsubs)
(puthash "\\ominus" "⊖" latexsubs)
(puthash "\\otimes" "⊗" latexsubs)
(puthash "\\oslash" "⊘" latexsubs)
(puthash "\\odot" "⊙" latexsubs)
(puthash "\\circledcirc" "⊚" latexsubs)
(puthash "\\circledast" "⊛" latexsubs)
(puthash "\\circleddash" "⊝" latexsubs)
(puthash "\\boxplus" "⊞" latexsubs)
(puthash "\\boxminus" "⊟" latexsubs)
(puthash "\\boxtimes" "⊠" latexsubs)
(puthash "\\boxdot" "⊡" latexsubs)
(puthash "\\vdash" "⊢" latexsubs)
(puthash "\\dashv" "⊣" latexsubs)
(puthash "\\top" "⊤" latexsubs)
(puthash "\\perp" "⊥" latexsubs)
(puthash "\\models" "⊧" latexsubs)
(puthash "\\vDash" "⊨" latexsubs)
(puthash "\\Vdash" "⊩" latexsubs)
(puthash "\\Vvdash" "⊪" latexsubs)
(puthash "\\VDash" "⊫" latexsubs)
(puthash "\\nvdash" "⊬" latexsubs)
(puthash "\\nvDash" "⊭" latexsubs)
(puthash "\\nVdash" "⊮" latexsubs)
(puthash "\\nVDash" "⊯" latexsubs)
(puthash "\\vartriangleleft" "⊲" latexsubs)
(puthash "\\vartriangleright" "⊳" latexsubs)
(puthash "\\trianglelefteq" "⊴" latexsubs)
(puthash "\\trianglerighteq" "⊵" latexsubs)
(puthash "\\original" "⊶" latexsubs)
(puthash "\\image" "⊷" latexsubs)
(puthash "\\multimap" "⊸" latexsubs)
(puthash "\\hermitconjmatrix" "⊹" latexsubs)
(puthash "\\intercal" "⊺" latexsubs)
(puthash "\\veebar" "⊻" latexsubs)
(puthash "\\rightanglearc" "⊾" latexsubs)
(puthash "\\bigwedge" "⋀" latexsubs)
(puthash "\\bigvee" "⋁" latexsubs)
(puthash "\\bigcap" "⋂" latexsubs)
(puthash "\\bigcup" "⋃" latexsubs)
(puthash "\\diamond" "⋄" latexsubs)
(puthash "\\cdot" "⋅" latexsubs)
(puthash "\\star" "⋆" latexsubs)
(puthash "\\divideontimes" "⋇" latexsubs)
(puthash "\\bowtie" "⋈" latexsubs)
(puthash "\\ltimes" "⋉" latexsubs)
(puthash "\\rtimes" "⋊" latexsubs)
(puthash "\\leftthreetimes" "⋋" latexsubs)
(puthash "\\rightthreetimes" "⋌" latexsubs)
(puthash "\\backsimeq" "⋍" latexsubs)
(puthash "\\curlyvee" "⋎" latexsubs)
(puthash "\\curlywedge" "⋏" latexsubs)
(puthash "\\Subset" "⋐" latexsubs)
(puthash "\\Supset" "⋑" latexsubs)
(puthash "\\Cap" "⋒" latexsubs)
(puthash "\\Cup" "⋓" latexsubs)
(puthash "\\pitchfork" "⋔" latexsubs)
(puthash "\\lessdot" "⋖" latexsubs)
(puthash "\\gtrdot" "⋗" latexsubs)
(puthash "\\verymuchless" "⋘" latexsubs)
(puthash "\\ggg" "⋙" latexsubs)
(puthash "\\lesseqgtr" "⋚" latexsubs)
(puthash "\\gtreqless" "⋛" latexsubs)
(puthash "\\curlyeqprec" "⋞" latexsubs)
(puthash "\\curlyeqsucc" "⋟" latexsubs)
(puthash "\\Elzsqspne" "⋥" latexsubs)
(puthash "\\lnsim" "⋦" latexsubs)
(puthash "\\gnsim" "⋧" latexsubs)
(puthash "\\precnsim" "⋨" latexsubs)
(puthash "\\succnsim" "⋩" latexsubs)
(puthash "\\ntriangleleft" "⋪" latexsubs)
(puthash "\\ntriangleright" "⋫" latexsubs)
(puthash "\\ntrianglelefteq" "⋬" latexsubs)
(puthash "\\ntrianglerighteq" "⋭" latexsubs)
(puthash "\\vdots" "⋮" latexsubs)
(puthash "\\cdots" "⋯" latexsubs)
(puthash "\\adots" "⋰" latexsubs)
(puthash "\\ddots" "⋱" latexsubs)
(puthash "\\barwedge" "⌅" latexsubs)
(puthash "\\lceil" "⌈" latexsubs)
(puthash "\\rceil" "⌉" latexsubs)
(puthash "\\lfloor" "⌊" latexsubs)
(puthash "\\rfloor" "⌋" latexsubs)
(puthash "\\recorder" "⌕" latexsubs)
(puthash "\\ulcorner" "⌜" latexsubs)
(puthash "\\urcorner" "⌝" latexsubs)
(puthash "\\llcorner" "⌞" latexsubs)
(puthash "\\lrcorner" "⌟" latexsubs)
(puthash "\\frown" "⌢" latexsubs)
(puthash "\\smile" "⌣" latexsubs)
(puthash "\\obar" "⌽" latexsubs)
(puthash "\\Elzdlcorn" "⎣" latexsubs)
(puthash "\\lmoustache" "⎰" latexsubs)
(puthash "\\rmoustache" "⎱" latexsubs)
(puthash "\\textvisiblespace" "␣" latexsubs)
(puthash "\\circledS" "Ⓢ" latexsubs)
(puthash "\\Elzdshfnc" "┆" latexsubs)
(puthash "\\Elzsqfnw" "┙" latexsubs)
(puthash "\\diagup" "╱" latexsubs)
(puthash "\\diagdown" "╲" latexsubs)
(puthash "\\blacksquare" "■" latexsubs)
(puthash "\\square" "□" latexsubs)
(puthash "\\Elzvrecto" "▯" latexsubs)
(puthash "\\bigtriangleup" "△" latexsubs)
(puthash "\\blacktriangle" "▴" latexsubs)
(puthash "\\vartriangle" "▵" latexsubs)
(puthash "\\blacktriangleright" "▸" latexsubs)
(puthash "\\triangleright" "▹" latexsubs)
(puthash "\\bigtriangledown" "▽" latexsubs)
(puthash "\\blacktriangledown" "▾" latexsubs)
(puthash "\\triangledown" "▿" latexsubs)
(puthash "\\blacktriangleleft" "◂" latexsubs)
(puthash "\\triangleleft" "◃" latexsubs)
(puthash "\\lozenge" "◊" latexsubs)
(puthash "\\bigcirc" "○" latexsubs)
(puthash "\\Elzcirfl" "◐" latexsubs)
(puthash "\\Elzcirfr" "◑" latexsubs)
(puthash "\\Elzcirfb" "◒" latexsubs)
(puthash "\\Elzrvbull" "◘" latexsubs)
(puthash "\\Elzsqfl" "◧" latexsubs)
(puthash "\\Elzsqfr" "◨" latexsubs)
(puthash "\\Elzsqfse" "◪" latexsubs)
(puthash "\\bigstar" "★" latexsubs)
(puthash "\\rightmoon" "☾" latexsubs)
(puthash "\\mercury" "☿" latexsubs)
(puthash "\\venus" "♀" latexsubs)
(puthash "\\female" "♀" latexsubs)
(puthash "\\male" "♂" latexsubs)
(puthash "\\mars" "♂" latexsubs)
(puthash "\\jupiter" "♃" latexsubs)
(puthash "\\saturn" "♄" latexsubs)
(puthash "\\uranus" "♅" latexsubs)
(puthash "\\neptune" "♆" latexsubs)
(puthash "\\pluto" "♇" latexsubs)
(puthash "\\aries" "♈" latexsubs)
(puthash "\\taurus" "♉" latexsubs)
(puthash "\\gemini" "♊" latexsubs)
(puthash "\\cancer" "♋" latexsubs)
(puthash "\\leo" "♌" latexsubs)
(puthash "\\virgo" "♍" latexsubs)
(puthash "\\libra" "♎" latexsubs)
(puthash "\\scorpio" "♏" latexsubs)
(puthash "\\sagittarius" "♐" latexsubs)
(puthash "\\capricornus" "♑" latexsubs)
(puthash "\\aquarius" "♒" latexsubs)
(puthash "\\pisces" "♓" latexsubs)
(puthash "\\spadesuit" "♠" latexsubs)
(puthash "\\heartsuit" "♡" latexsubs)
(puthash "\\diamondsuit" "♢" latexsubs)
(puthash "\\clubsuit" "♣" latexsubs)
(puthash "\\quarternote" "♩" latexsubs)
(puthash "\\eighthnote" "♪" latexsubs)
(puthash "\\flat" "♭" latexsubs)
(puthash "\\natural" "♮" latexsubs)
(puthash "\\sharp" "♯" latexsubs)
(puthash "\\checkmark" "✓" latexsubs)
(puthash "\\maltese" "✠" latexsubs)
(puthash "\\langle" "⟨" latexsubs)
(puthash "\\rangle" "⟩" latexsubs)
(puthash "\\longleftarrow" "⟵" latexsubs)
(puthash "\\longrightarrow" "⟶" latexsubs)
(puthash "\\longleftrightarrow" "⟷" latexsubs)
(puthash "\\Longleftarrow" "⟸" latexsubs)
(puthash "\\Longrightarrow" "⟹" latexsubs)
(puthash "\\Longleftrightarrow" "⟺" latexsubs)
(puthash "\\longmapsto" "⟼" latexsubs)
(puthash "\\Mapsfrom" "⤆" latexsubs)
(puthash "\\Mapsto" "⤇" latexsubs)
(puthash "\\Uuparrow" "⤊" latexsubs)
(puthash "\\Ddownarrow" "⤋" latexsubs)
(puthash "\\bkarow" "⤍" latexsubs)
(puthash "\\dbkarow" "⤏" latexsubs)
(puthash "\\drbkarrow" "⤐" latexsubs)
(puthash "\\UpArrowBar" "⤒" latexsubs)
(puthash "\\DownArrowBar" "⤓" latexsubs)
(puthash "\\twoheadrightarrowtail" "⤖" latexsubs)
(puthash "\\hksearow" "⤥" latexsubs)
(puthash "\\hkswarow" "⤦" latexsubs)
(puthash "\\tona" "⤧" latexsubs)
(puthash "\\toea" "⤨" latexsubs)
(puthash "\\tosa" "⤩" latexsubs)
(puthash "\\towa" "⤪" latexsubs)
(puthash "\\rdiagovfdiag" "⤫" latexsubs)
(puthash "\\fdiagovrdiag" "⤬" latexsubs)
(puthash "\\seovnearrow" "⤭" latexsubs)
(puthash "\\neovsearrow" "⤮" latexsubs)
(puthash "\\fdiagovnearrow" "⤯" latexsubs)
(puthash "\\rdiagovsearrow" "⤰" latexsubs)
(puthash "\\neovnwarrow" "⤱" latexsubs)
(puthash "\\nwovnearrow" "⤲" latexsubs)
(puthash "\\ElzRlarr" "⥂" latexsubs)
(puthash "\\ElzrLarr" "⥄" latexsubs)
(puthash "\\Elzrarrx" "⥇" latexsubs)
(puthash "\\LeftRightVector" "⥎" latexsubs)
(puthash "\\RightUpDownVector" "⥏" latexsubs)
(puthash "\\DownLeftRightVector" "⥐" latexsubs)
(puthash "\\LeftUpDownVector" "⥑" latexsubs)
(puthash "\\LeftVectorBar" "⥒" latexsubs)
(puthash "\\RightVectorBar" "⥓" latexsubs)
(puthash "\\RightUpVectorBar" "⥔" latexsubs)
(puthash "\\RightDownVectorBar" "⥕" latexsubs)
(puthash "\\DownLeftVectorBar" "⥖" latexsubs)
(puthash "\\DownRightVectorBar" "⥗" latexsubs)
(puthash "\\LeftUpVectorBar" "⥘" latexsubs)
(puthash "\\LeftDownVectorBar" "⥙" latexsubs)
(puthash "\\LeftTeeVector" "⥚" latexsubs)
(puthash "\\RightTeeVector" "⥛" latexsubs)
(puthash "\\RightUpTeeVector" "⥜" latexsubs)
(puthash "\\RightDownTeeVector" "⥝" latexsubs)
(puthash "\\DownLeftTeeVector" "⥞" latexsubs)
(puthash "\\DownRightTeeVector" "⥟" latexsubs)
(puthash "\\LeftUpTeeVector" "⥠" latexsubs)
(puthash "\\LeftDownTeeVector" "⥡" latexsubs)
(puthash "\\UpEquilibrium" "⥮" latexsubs)
(puthash "\\ReverseUpEquilibrium" "⥯" latexsubs)
(puthash "\\RoundImplies" "⥰" latexsubs)
(puthash "\\Vvert" "⦀" latexsubs)
(puthash "\\Elroang" "⦆" latexsubs)
(puthash "\\Elzddfnc" "⦙" latexsubs)
(puthash "\\Angle" "⦜" latexsubs)
(puthash "\\Elzlpargt" "⦠" latexsubs)
(puthash "\\obslash" "⦸" latexsubs)
(puthash "\\boxdiag" "⧄" latexsubs)
(puthash "\\boxbslash" "⧅" latexsubs)
(puthash "\\boxast" "⧆" latexsubs)
(puthash "\\boxcircle" "⧇" latexsubs)
(puthash "\\ElzLap" "⧊" latexsubs)
(puthash "\\Elzdefas" "⧋" latexsubs)
(puthash "\\LeftTriangleBar" "⧏" latexsubs)
(puthash "\\NotLeftTriangleBar" "⧏̸" latexsubs)
(puthash "\\RightTriangleBar" "⧐" latexsubs)
(puthash "\\NotRightTriangleBar" "⧐̸" latexsubs)
(puthash "\\dualmap" "⧟" latexsubs)
(puthash "\\shuffle" "⧢" latexsubs)
(puthash "\\blacklozenge" "⧫" latexsubs)
(puthash "\\RuleDelayed" "⧴" latexsubs)
(puthash "\\bigodot" "⨀" latexsubs)
(puthash "\\bigoplus" "⨁" latexsubs)
(puthash "\\bigotimes" "⨂" latexsubs)
(puthash "\\bigcupdot" "⨃" latexsubs)
(puthash "\\biguplus" "⨄" latexsubs)
(puthash "\\bigsqcap" "⨅" latexsubs)
(puthash "\\bigsqcup" "⨆" latexsubs)
(puthash "\\conjquant" "⨇" latexsubs)
(puthash "\\disjquant" "⨈" latexsubs)
(puthash "\\bigtimes" "⨉" latexsubs)
(puthash "\\iiiint" "⨌" latexsubs)
(puthash "\\intbar" "⨍" latexsubs)
(puthash "\\intBar" "⨎" latexsubs)
(puthash "\\clockoint" "⨏" latexsubs)
(puthash "\\sqrint" "⨖" latexsubs)
(puthash "\\intx" "⨘" latexsubs)
(puthash "\\intcap" "⨙" latexsubs)
(puthash "\\intcup" "⨚" latexsubs)
(puthash "\\upint" "⨛" latexsubs)
(puthash "\\lowint" "⨜" latexsubs)
(puthash "\\plusdot" "⨥" latexsubs)
(puthash "\\minusdot" "⨪" latexsubs)
(puthash "\\ElzTimes" "⨯" latexsubs)
(puthash "\\btimes" "⨲" latexsubs)
(puthash "\\intprod" "⨼" latexsubs)
(puthash "\\intprodr" "⨽" latexsubs)
(puthash "\\amalg" "⨿" latexsubs)
(puthash "\\ElzAnd" "⩓" latexsubs)
(puthash "\\ElzOr" "⩔" latexsubs)
(puthash "\\ElOr" "⩖" latexsubs)
(puthash "\\perspcorrespond" "⩞" latexsubs)
(puthash "\\Elzminhat" "⩟" latexsubs)
(puthash "\\Equal" "⩵" latexsubs)
(puthash "\\ddotseq" "⩷" latexsubs)
(puthash "\\leqslant" "⩽" latexsubs)
(puthash "\\nleqslant" "⩽̸" latexsubs)
(puthash "\\geqslant" "⩾" latexsubs)
(puthash "\\ngeqslant" "⩾̸" latexsubs)
(puthash "\\lessapprox" "⪅" latexsubs)
(puthash "\\gtrapprox" "⪆" latexsubs)
(puthash "\\lneq" "⪇" latexsubs)
(puthash "\\gneq" "⪈" latexsubs)
(puthash "\\lnapprox" "⪉" latexsubs)
(puthash "\\gnapprox" "⪊" latexsubs)
(puthash "\\lesseqqgtr" "⪋" latexsubs)
(puthash "\\gtreqqless" "⪌" latexsubs)
(puthash "\\eqslantless" "⪕" latexsubs)
(puthash "\\eqslantgtr" "⪖" latexsubs)
(puthash "\\NestedLessLess" "⪡" latexsubs)
(puthash "\\NotNestedLessLess" "⪡̸" latexsubs)
(puthash "\\NestedGreaterGreater" "⪢" latexsubs)
(puthash "\\NotNestedGreaterGreater" "⪢̸" latexsubs)
(puthash "\\partialmeetcontraction" "⪣" latexsubs)
(puthash "\\bumpeqq" "⪮" latexsubs)
(puthash "\\preceq" "⪯" latexsubs)
(puthash "\\npreceq" "⪯̸" latexsubs)
(puthash "\\succeq" "⪰" latexsubs)
(puthash "\\nsucceq" "⪰̸" latexsubs)
(puthash "\\precneqq" "⪵" latexsubs)
(puthash "\\succneqq" "⪶" latexsubs)
(puthash "\\precapprox" "⪷" latexsubs)
(puthash "\\succapprox" "⪸" latexsubs)
(puthash "\\precnapprox" "⪹" latexsubs)
(puthash "\\succnapprox" "⪺" latexsubs)
(puthash "\\subseteqq" "⫅" latexsubs)
(puthash "\\nsubseteqq" "⫅̸" latexsubs)
(puthash "\\supseteqq" "⫆" latexsubs)
(puthash "\\nsupseteqq" "⫆̸" latexsubs)
(puthash "\\subsetneqq" "⫋" latexsubs)
(puthash "\\supsetneqq" "⫌" latexsubs)
(puthash "\\mlcp" "⫛" latexsubs)
(puthash "\\forks" "⫝̸" latexsubs)
(puthash "\\forksnot" "⫝" latexsubs)
(puthash "\\dashV" "⫣" latexsubs)
(puthash "\\Dashv" "⫤" latexsubs)
(puthash "\\interleave" "⫴" latexsubs)
(puthash "\\Elztdcol" "⫶" latexsubs)
(puthash "\\_j" "\u2c7c" latexsubs)
(puthash "\\^V" "\u2c7d" latexsubs)
(puthash "\\openbracketleft" "〚" latexsubs)
(puthash "\\openbracketright" "〛" latexsubs)
(puthash "\\overbrace" "︷" latexsubs)
(puthash "\\underbrace" "︸" latexsubs)

(provide 'julia-mode)
