/***************************************************************************
 *   Copyright (C) 2005-2010 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QFile>

#include <KApplication>
#include <kio/netaccess.h>
#include <KLocale>
#include <KMessageBox>
#include <KTemporaryFile>

#include <kmdcodec.h>

#include "komparatorcomparejob.h"
#include "komparatorwidget.h"

#include "kfileitemext.h"

#include <version.h>

KFileItemExt::KFileItemExt( const KIO::UDSEntry &_entry, const KUrl &_url, bool _determineMimeTypeOnDemand, bool _urlIsDirectory ) :
	KFileItem( _entry, _url, _determineMimeTypeOnDemand, _urlIsDirectory )
{
	dir = 0;
	hasdupes_size = 0;
	hasdupes_path = 0;
	isdupe_size = 0;
	isdupe_path = 0;
	duplicates_size = NULL;
	duplicates_path = NULL;
	dup_size_parent = NULL;
	dup_path_parent = NULL;
	next = NULL;
	m_md5 = NULL;
	m_tmp_file = NULL;
	parent_path = "";
	virtual_parent_path = "";
}

KFileItemExt::KFileItemExt( mode_t _mode, mode_t _permissions, const KUrl &_url, bool _determineMimeTypeOnDemand ) :
	KFileItem( _mode, _permissions, _url, _determineMimeTypeOnDemand )
{
	dir = 0;
	hasdupes_size = 0;
	hasdupes_path = 0;
	isdupe_size = 0;
	isdupe_path = 0;
	duplicates_size = NULL;
	duplicates_path = NULL;
	dup_size_parent = NULL;
	dup_path_parent = NULL;
	next = NULL;
	m_md5 = NULL;
	m_tmp_file = NULL;
	parent_path = "";
	virtual_parent_path = "";
}

KFileItemExt::KFileItemExt( const KUrl &url, const QString &mimeType, mode_t mode ) :
	KFileItem( url, mimeType, mode )
{
	dir = 0;
	hasdupes_size = 0;
	hasdupes_path = 0;
	isdupe_size = 0;
	isdupe_path = 0;
	duplicates_size = NULL;
	duplicates_path = NULL;
	dup_size_parent = NULL;
	dup_path_parent = NULL;
	next = NULL;
	m_md5 = NULL;
	m_tmp_file = NULL;
	parent_path = "";
	virtual_parent_path = "";
}

KFileItemExt::KFileItemExt( const KFileItem &item ) :
	KFileItem( item )
{
	dir = 0;
	hasdupes_size = 0;
	hasdupes_path = 0;
	isdupe_size = 0;
	isdupe_path = 0;
	duplicates_size = NULL;
	duplicates_path = NULL;
	dup_size_parent = NULL;
	dup_path_parent = NULL;
	next = NULL;
	m_md5 = NULL;
	m_tmp_file = NULL;
	parent_path = "";
	virtual_parent_path = "";
}

KFileItemExt::KFileItemExt( const KFileItemExt &item ) :
	QObject(), KFileItem( item )
{
	dir = item.dir;
	hasdupes_size = item.hasdupes_size;
	hasdupes_path = item.hasdupes_path;
	isdupe_size = item.isdupe_size;
	isdupe_path = item.isdupe_path;
	duplicates_size = item.duplicates_size;
	duplicates_path = item.duplicates_path;
	dup_size_parent = item.dup_size_parent;
	dup_path_parent = item.dup_path_parent;
	next = item.next;
	m_md5 = item.m_md5;
	m_tmp_file = item.m_tmp_file;
	parent_path = item.parent_path;
	virtual_parent_path = item.virtual_parent_path;
}

KFileItemExt::~KFileItemExt()
{
	delete m_md5;
	m_md5 = NULL;

	// We don't need to delete the duplicates, as duplicates also appera in "next".
	delete next;
	next = NULL;
}

class compatSleep : public QThread
{
	public:
		static void sleep( unsigned long secs )
		{
			QThread::sleep( secs );
		}
		static void msleep( unsigned long msecs )
		{
			QThread::msleep( msecs );
		}
		static void usleep( unsigned long usecs )
		{
			QThread::usleep( usecs );
		}
};

QByteArray KFileItemExt::MD5( KomparatorWidget *parent, bool force_recalculate )
{
	connect( this, SIGNAL( emitMessage( const QString &, const QString & ) ), parent, SLOT( slotInteractionMessageError( const QString &, const QString & ) ) );
	connect( this, SIGNAL( emitDownloadFile( KFileItemExt * ) ), parent, SLOT( slotInteractionDownloadFile( KFileItemExt * ) ) );

	if ( force_recalculate )
	{
		delete m_md5;
		m_md5 = NULL;
	}

	if ( !m_md5 )
	{
		const int BUFFER_SIZE = 512*1024;
		char buffer[BUFFER_SIZE];
		int read_data;

		if ( isLocalFile() )
		{
			QFile file( url().directory( KUrl::AppendTrailingSlash )+url().fileName() );
			if ( !file.open( QIODevice::ReadOnly ) )
			{
				emit emitMessage( i18n( "Error getting md5 sum of local file %1." ).arg( url().directory( KUrl::AppendTrailingSlash )+url().fileName() ),
					i18n( "Error while calculating md5 checksum" ) );
				return QByteArray( "d41d8cd98f00b204e9800998ecf8427e" ); // checksum of empty file.
			}

			m_md5 = new KMD5();
			while ( ( read_data = file.read( buffer, BUFFER_SIZE ) ) > 0 )
			{
				m_md5->update( buffer, read_data );

				if ( parent->compare_job->isCanceled() ) break; // m_md5 sum will be invalid, but cancel will delete it anyway.
			}

			file.close();
		}
		else
		{
			m_tmp_file = NULL;
			m_download_error = -1;

			m_mutex.lock();

			emit emitDownloadFile( this );

			while ( m_download_error == -1 )
			{
				m_mutex.unlock();
				compatSleep::msleep( 10 );
				m_mutex.lock();
			}
			compatSleep::msleep( 10 );

			m_mutex.unlock();

			if ( parent->compare_job->isCanceled() )
			{
				m_tmp_file->close();

				delete m_tmp_file;
				m_tmp_file = NULL;

				return QByteArray( "d41d8cd98f00b204e9800998ecf8427e" ); // checksum of empty file.
			}

			m_md5 = new KMD5();

			if ( m_download_error == 0 ) // everything went well.
			{
				QFile file( m_tmp_file->fileName() );
				m_tmp_file->close();
				if ( !file.open( QIODevice::ReadOnly ) )
				{
					emit emitMessage( i18n( "Opening temporary file %1 failed. MD5 checksum won't be correct." ).arg( m_tmp_file->fileName() ),
						i18n( "Error while calculating md5 checksum" ) );
				}
				else
				{
					while ( ( read_data = file.read( buffer, BUFFER_SIZE ) ) > 0 )
					{
						m_md5->update( buffer, read_data );

						if ( parent->compare_job->isCanceled() ) break; // m_md5 sum will be invalid, but cancel will delete it anyway.
					}

					file.close();
				}
			}
			else // download errors are handled by komparatorwidget.
			{
				m_tmp_file->close();
			}

			delete m_tmp_file;
			m_tmp_file = NULL;
		}
	}

	return m_md5->hexDigest();
}

QByteArray KFileItemExt::MD5( bool force_recalculate )
{
	if ( force_recalculate )
	{
		delete m_md5;
		m_md5 = NULL;
	}

	if ( !m_md5 )
	{
		const int BUFFER_SIZE = 512*1024;
		char buffer[BUFFER_SIZE];
		int read_data;

		if ( isLocalFile() )
		{
			QFile file( url().directory( KUrl::AppendTrailingSlash )+url().fileName() );
			if ( !file.open( QIODevice::ReadOnly ) )
			{
				KMessageBox::error( 0,
					i18n( "Error getting md5 sum of local file %1." ).arg( url().directory( KUrl::AppendTrailingSlash )+url().fileName() ),
					i18n( "Error while calculating md5 sum" ) );
				return QByteArray( "d41d8cd98f00b204e9800998ecf8427e" ); // checksum of empty file.
			}

			m_md5 = new KMD5();
			while ( ( read_data = file.read( buffer, BUFFER_SIZE ) ) > 0 )
			{
				m_md5->update( buffer, read_data );
			}

			file.close();
		}
		else
		{
			QString m_tmp_file_str;

			m_md5 = new KMD5();

			if ( KIO::NetAccess::download( this->url(), m_tmp_file_str, NULL ) ) // everything went well.
			{
				QFile file( m_tmp_file_str );
				if ( !file.open( QIODevice::ReadOnly ) )
				{
					KMessageBox::sorry( NULL, i18n( "Opening temporary file %1 failed. MD5 checksum won't be correct." ).arg( m_tmp_file_str ) );
				}
				else
				{
					while ( ( read_data = file.read( buffer, BUFFER_SIZE ) ) > 0 )
					{
						m_md5->update( buffer, read_data );
					}

					file.close();
				}
			}
			else // download errors are handled by komparatorwidget.
			{
				KMessageBox::sorry( NULL, i18n( "Download of %1 to %2 failed." ).arg( this->url().url() ).arg( m_tmp_file_str ) );
			}

			KIO::NetAccess::removeTempFile( m_tmp_file_str );
		}
	}

	return m_md5->hexDigest();
}

QString KFileItemExt::getFile( KomparatorWidget *parent )
{
	connect( this, SIGNAL( emitMessage( const QString &, const QString & ) ), parent, SLOT( slotInteractionMessageError( const QString &, const QString & ) ) );
	connect( this, SIGNAL( emitDownloadFile( KFileItemExt * ) ), parent, SLOT( slotInteractionDownloadFile( KFileItemExt * ) ) );

	QString ret = "";

	if ( isLocalFile() )
	{
		ret = url().directory( KUrl::AppendTrailingSlash ) + url().fileName();
	}
	else
	{
		m_tmp_file = NULL;
		m_download_error = -1;

		m_mutex.lock();

		emit emitDownloadFile( this );

		while ( m_download_error == -1 ) // wait until the KIO::Job has finished (in komparatorwidget).
		{
			m_mutex.unlock();
			compatSleep::msleep( 10 );
			m_mutex.lock();
		}
		compatSleep::msleep( 10 );

		m_mutex.unlock();

		if ( parent->compare_job->isCanceled() )
		{
			if ( m_tmp_file )
			{
				m_tmp_file->close();

				delete m_tmp_file;
			}

			m_tmp_file = NULL;

			return "";
		}

		if ( m_download_error == 0 ) // everything went well. otherwise will fail in komparejob.
		{
			if ( m_tmp_file ) ret = m_tmp_file->fileName();
		}

		if ( m_tmp_file ) m_tmp_file->close();
	}

	return ret;
}

void KFileItemExt::deleteFile( KomparatorWidget *parent )
{
	connect( this, SIGNAL( emitMessage( const QString &, const QString & ) ), parent, SLOT( slotInteractionMessageError( const QString &, const QString & ) ) );
	connect( this, SIGNAL( emitDownloadFile( KFileItemExt * ) ), parent, SLOT( slotInteractionDownloadFile( KFileItemExt * ) ) );

	if ( m_tmp_file )
	{
		m_tmp_file->close();

		delete m_tmp_file;
		m_tmp_file = NULL;
	}
}

void KFileItemExt::setTmpFile( KTemporaryFile *_tmp_file )
{
	m_mutex.lock();
	m_tmp_file = _tmp_file;
	m_mutex.unlock();
}

void KFileItemExt::setDownloadError( int _download_error )
{
	m_mutex.lock();
	m_download_error = _download_error;
	m_mutex.unlock();
}
