/* -*- Mode: C++; c-default-style: "k&r"; indent-tabs-mode: nil; tab-width: 2; c-basic-offset: 2 -*- */

/* libmwaw
* Version: MPL 2.0 / LGPLv2+
*
* The contents of this file are subject to the Mozilla Public License Version
* 2.0 (the "License"); you may not use this file except in compliance with
* the License or as specified alternatively below. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
*
* Software distributed under the License is distributed on an "AS IS" basis,
* WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
* for the specific language governing rights and limitations under the
* License.
*
* Major Contributor(s):
* Copyright (C) 2002 William Lachance (wrlach@gmail.com)
* Copyright (C) 2002,2004 Marc Maurer (uwog@uwog.net)
* Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
* Copyright (C) 2006, 2007 Andrew Ziem
* Copyright (C) 2011, 2012 Alonso Laurent (alonso@loria.fr)
*
*
* All Rights Reserved.
*
* For minor contributions see the git repository.
*
* Alternatively, the contents of this file may be used under the terms of
* the GNU Lesser General Public License Version 2 or later (the "LGPLv2+"),
* in which case the provisions of the LGPLv2+ are applicable
* instead of those above.
*/

#include <iomanip>
#include <iostream>
#include <limits>
#include <map>
#include <sstream>

#include <librevenge/librevenge.h>

#include "MWAWEntry.hxx"
#include "MWAWFontConverter.hxx"
#include "MWAWParser.hxx"

#include "ClarisWksDocument.hxx"
#include "ClarisWksText.hxx"

#include "ClarisWksStyleManager.hxx"

/** Internal: the structures of a ClarisWksStyleManagerInternal */
namespace ClarisWksStyleManagerInternal
{
////////////////////////////////////////
//! Internal: the pattern of a ClarisWksStyleManager
struct Pattern : public MWAWGraphicStyle::Pattern {
  //! constructor ( 4 int by patterns )
  explicit Pattern(uint16_t const *pat=0) : MWAWGraphicStyle::Pattern(), m_percent(0)
  {
    if (!pat) return;
    m_colors[0]=MWAWColor::white();
    m_colors[1]=MWAWColor::black();
    m_dim=MWAWVec2i(8,8);
    m_data.resize(8);
    for (size_t i=0; i < 4; ++i) {
      uint16_t val=pat[i];
      m_data[2*i]=static_cast<unsigned char>(val>>8);
      m_data[2*i+1]=static_cast<unsigned char>(val&0xFF);
    }
    int numOnes=0;
    for (size_t j=0; j < 8; ++j) {
      uint8_t val=static_cast<uint8_t>(m_data[j]);
      for (int b=0; b < 8; b++) {
        if (val&1) ++numOnes;
        val = uint8_t(val>>1);
      }
    }
    m_percent=float(numOnes)/64.f;
  }
  //! destructor
  virtual ~Pattern();
  //! the percentage
  float m_percent;
};

Pattern::~Pattern()
{
}

////////////////////////////////////////
//! Internal: the gradient of a ClarisWksStyleManager
struct Gradient {
  //! construtor
  Gradient(int type=0, int nColor=0, int angle=0, float decal=0) :
    m_type(type), m_numColors(nColor), m_angle(angle), m_decal(decal), m_box()
  {
    m_colors[0]=MWAWColor::black();
    m_colors[1]=MWAWColor::white();
  }
  //! check if the gradient is valid
  bool ok() const
  {
    return m_type>=0 && m_type<=2 && m_numColors>=2 && m_numColors<=4;
  }
  //! update the style
  bool update(MWAWGraphicStyle &style) const;
  //! operator<<
  friend std::ostream &operator<<(std::ostream &o, Gradient const &gr)
  {
    switch (gr.m_type) {
    case 0:
      o << "linear,";
      break;
    case 1:
      o << "radial,";
      break;
    case 2:
      o << "rectangle,";
      break;
    default:
      o << "#type=" << gr.m_type << ",";
      break;
    }
    if (gr.m_angle)
      o << "angle=" << gr.m_angle << ",";
    if (gr.m_decal>0)
      o << "decal=" << gr.m_decal << ",";
    o << "col=[";
    for (int c=0; c < gr.m_numColors; ++c) {
      if (c>=4) break;
      o << gr.m_colors[c] << ",";
    }
    o << "],";
    if (gr.m_box!=MWAWBox2i(MWAWVec2i(0,0),MWAWVec2i(0,0)))
      o << "center=" << gr.m_box << ",";
    return o;
  }
  //! the type
  int m_type;
  //! the number of color
  int m_numColors;
  //! the color
  MWAWColor m_colors[4];
  //! the angle
  int m_angle;
  //! the decal
  float m_decal;
  //! the center bdbox
  MWAWBox2i m_box;
};

bool Gradient::update(MWAWGraphicStyle &style) const
{
  if (!ok()) return false;
  style.m_gradientStopList.resize(0);
  if (m_type==1 || m_type==2) {
    style.m_gradientType=m_type==1 ? MWAWGraphicStyle::G_Radial : MWAWGraphicStyle::G_Rectangular;
    for (int c=0; c < m_numColors; ++c)
      style.m_gradientStopList.push_back(MWAWGraphicStyle::GradientStop(float(c)/float(m_numColors-1), m_colors[m_numColors-1-c]));
    style.m_gradientPercentCenter=MWAWVec2f(float(m_box.center()[1])/100.f, float(m_box.center()[0])/100.f);
    return true;
  }
  style.m_gradientAngle=float(m_angle+90);
  if (m_decal<=0.05f) {
    style.m_gradientType= MWAWGraphicStyle::G_Axial;
    for (int c=0; c < m_numColors; ++c)
      style.m_gradientStopList.push_back(MWAWGraphicStyle::GradientStop(float(c)/float(m_numColors-1), m_colors[m_numColors-1-c]));
    return true;
  }
  style.m_gradientType= MWAWGraphicStyle::G_Linear;
  if (m_decal >=0.95f)  {
    for (int c=0; c < m_numColors; ++c)
      style.m_gradientStopList.push_back(MWAWGraphicStyle::GradientStop(float(c)/float(m_numColors-1), m_colors[c]));
    return true;
  }
  for (int c=-m_numColors+1; c<m_numColors; ++c) {
    // checkme: look almost good
    float pos=float(c)/float(m_numColors-1)+(1-m_decal)/2.f;
    if (pos < 0) {
      if (c!=m_numColors-1 && float(c+1)/float(m_numColors-1)+(1-m_decal)/2.f>=0)
        continue;
      pos=0;
    }
    style.m_gradientStopList.push_back(MWAWGraphicStyle::GradientStop(pos>1?1:pos, m_colors[c<0?-c:c]));
    if (pos>=1)
      break;
  }
  return true;
}

//! Internal: the state of a ClarisWksStyleManager
struct State {
  //! constructor
  State() : m_version(-1), m_localFIdMap(), m_stylesMap(), m_lookupMap(),
    m_fontList(), m_cellFormatList(), m_graphList(), m_ksenList(), m_nameList(),
    m_colorList(), m_patternList(), m_gradientList(), m_wallpaperList()
  {
  }
  //! set the default color map
  void setDefaultColorList(int version);
  //! set the default pattern map
  void setDefaultPatternList(int version);
  //! set the default pattern map
  void setDefaultGradientList(int version);
  //! set the default pattern map
  void setDefaultWallPaperList(int version);
  //! return a mac font id corresponding to a local id
  int getFontId(int localId) const
  {
    if (m_localFIdMap.find(localId)==m_localFIdMap.end())
      return localId;
    return m_localFIdMap.find(localId)->second;
  }

  //! the version
  int m_version;
  //! a map local fontId->fontId
  std::map<int, int> m_localFIdMap;
  //! the styles map id->style
  std::map<int, ClarisWksStyleManager::Style> m_stylesMap;
  //! the style lookupMap
  std::map<int, int> m_lookupMap;
  //! the list of fonts
  std::vector<MWAWFont> m_fontList;
  //! the list of format
  std::vector<ClarisWksStyleManager::CellFormat> m_cellFormatList;
  //! the Graphic list
  std::vector<MWAWGraphicStyle> m_graphList;
  //! the KSEN list
  std::vector<ClarisWksStyleManager::KSEN> m_ksenList;
  //! the style name list
  std::vector<std::string> m_nameList;
  //! a list colorId -> color
  std::vector<MWAWColor> m_colorList;
  //! a list patternId -> pattern
  std::vector<Pattern> m_patternList;
  //! a list gradientId -> gradient
  std::vector<Gradient> m_gradientList;
  //! a list wallPaperId -> pattern
  std::vector<MWAWGraphicStyle::Pattern> m_wallpaperList;
};


void State::setDefaultColorList(int version)
{
  if (!m_colorList.empty()) return;
  if (version==1) {
    uint32_t const defCol[81] = {
      0xffffff,0x000000,0x222222,0x444444,0x555555,0x888888,0xbbbbbb,0xdddddd,
      0xeeeeee,0x440000,0x663300,0x996600,0x002200,0x003333,0x003399,0x000055,
      0x330066,0x660066,0x770000,0x993300,0xcc9900,0x004400,0x336666,0x0033ff,
      0x000077,0x660099,0x990066,0xaa0000,0xcc3300,0xffcc00,0x006600,0x006666,
      0x0066ff,0x0000aa,0x663399,0xcc0099,0xdd0000,0xff3300,0xffff00,0x008800,
      0x009999,0x0099ff,0x0000dd,0x9900cc,0xff0099,0xff3333,0xff6600,0xffff33,
      0x00ee00,0x00cccc,0x00ccff,0x3366ff,0x9933ff,0xff33cc,0xff6666,0xff6633,
      0xffff66,0x66ff66,0x66cccc,0x66ffff,0x3399ff,0x9966ff,0xff66ff,0xff9999,
      0xff9966,0xffff99,0x99ff99,0x66ffcc,0x99ffff,0x66ccff,0x9999ff,0xff99ff,
      0xffcccc,0xffcc99,0xffffcc,0xccffcc,0x99ffcc,0xccffff,0x99ccff,0xccccff,
      0xffccff
    };
    m_colorList.resize(81);
    for (size_t i = 0; i < 81; i++)
      m_colorList[i] = defCol[i];
  }
  else {
    uint32_t const defCol[256] = {
      0xffffff,0x0,0x777777,0x555555,0xffff00,0xff6600,0xdd0000,0xff0099,
      0x660099,0xdd,0x99ff,0xee00,0x6600,0x663300,0x996633,0xbbbbbb,
      0xffffcc,0xffff99,0xffff66,0xffff33,0xffccff,0xffcccc,0xffcc99,0xffcc66,
      0xffcc33,0xffcc00,0xff99ff,0xff99cc,0xff9999,0xff9966,0xff9933,0xff9900,
      0xff66ff,0xff66cc,0xff6699,0xff6666,0xff6633,0xff33ff,0xff33cc,0xff3399,
      0xff3366,0xff3333,0xff3300,0xff00ff,0xff00cc,0xff0066,0xff0033,0xff0000,
      0xccffff,0xccffcc,0xccff99,0xccff66,0xccff33,0xccff00,0xccccff,0xcccccc,
      0xcccc99,0xcccc66,0xcccc33,0xcccc00,0xcc99ff,0xcc99cc,0xcc9999,0xcc9966,
      0xcc9933,0xcc9900,0xcc66ff,0xcc66cc,0xcc6699,0xcc6666,0xcc6633,0xcc6600,
      0xcc33ff,0xcc33cc,0xcc3399,0xcc3366,0xcc3333,0xcc3300,0xcc00ff,0xcc00cc,
      0xcc0099,0xcc0066,0xcc0033,0xcc0000,0x99ffff,0x99ffcc,0x99ff99,0x99ff66,
      0x99ff33,0x99ff00,0x99ccff,0x99cccc,0x99cc99,0x99cc66,0x99cc33,0x99cc00,
      0x9999ff,0x9999cc,0x999999,0x999966,0x999933,0x999900,0x9966ff,0x9966cc,
      0x996699,0x996666,0x996600,0x9933ff,0x9933cc,0x993399,0x993366,0x993333,
      0x993300,0x9900ff,0x9900cc,0x990099,0x990066,0x990033,0x990000,0x66ffff,
      0x66ffcc,0x66ff99,0x66ff66,0x66ff33,0x66ff00,0x66ccff,0x66cccc,0x66cc99,
      0x66cc66,0x66cc33,0x66cc00,0x6699ff,0x6699cc,0x669999,0x669966,0x669933,
      0x669900,0x6666ff,0x6666cc,0x666699,0x666666,0x666633,0x666600,0x6633ff,
      0x6633cc,0x663399,0x663366,0x663333,0x6600ff,0x6600cc,0x660066,0x660033,
      0x660000,0x33ffff,0x33ffcc,0x33ff99,0x33ff66,0x33ff33,0x33ff00,0x33ccff,
      0x33cccc,0x33cc99,0x33cc66,0x33cc33,0x33cc00,0x3399ff,0x3399cc,0x339999,
      0x339966,0x339933,0x339900,0x3366ff,0x3366cc,0x336699,0x336666,0x336633,
      0x336600,0x3333ff,0x3333cc,0x333399,0x333366,0x333333,0x333300,0x3300ff,
      0x3300cc,0x330099,0x330066,0x330033,0x330000,0xffff,0xffcc,0xff99,
      0xff66,0xff33,0xff00,0xccff,0xcccc,0xcc99,0xcc66,0xcc33,
      0xcc00,0x99cc,0x9999,0x9966,0x9933,0x9900,0x66ff,0x66cc,
      0x6699,0x6666,0x6633,0x33ff,0x33cc,0x3399,0x3366,0x3333,
      0x3300,0xff,0xcc,0x99,0x66,0x33,0xdd0000,0xbb0000,
      0xaa0000,0x880000,0x770000,0x550000,0x440000,0x220000,0x110000,0xdd00,
      0xbb00,0xaa00,0x8800,0x7700,0x5500,0x4400,0x2200,0x1100,
      0xee,0xbb,0xaa,0x88,0x77,0x55,0x44,0x22,
      0x11,0xeeeeee,0xdddddd,0xaaaaaa,0x888888,0x444444,0x222222,0x111111,
    };
    m_colorList.resize(256);
    for (size_t i = 0; i < 256; i++)
      m_colorList[i] = defCol[i];
  }
}

void State::setDefaultPatternList(int)
{
  if (m_patternList.size()) return;
  static uint16_t const(s_pattern[4*64]) = {
    0x0000, 0x0000, 0x0000, 0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0x7fff, 0xffff, 0xf7ff, 0xffff, 0x7fff, 0xf7ff, 0x7fff, 0xf7ff,
    0xffee, 0xffbb, 0xffee, 0xffbb, 0x77dd, 0x77dd, 0x77dd, 0x77dd, 0xaa55, 0xaa55, 0xaa55, 0xaa55, 0x8822, 0x8822, 0x8822, 0x8822,
    0xaa00, 0xaa00, 0xaa00, 0xaa00, 0xaa00, 0x4400, 0xaa00, 0x1100, 0x8800, 0xaa00, 0x8800, 0xaa00, 0x8800, 0x2200, 0x8800, 0x2200,
    0x8000, 0x0800, 0x8000, 0x0800, 0x8800, 0x0000, 0x8800, 0x0000, 0x8000, 0x0000, 0x0800, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001,
    0xeedd, 0xbb77, 0xeedd, 0xbb77, 0x3366, 0xcc99, 0x3366, 0xcc99, 0x1122, 0x4488, 0x1122, 0x4488, 0x8307, 0x0e1c, 0x3870, 0xe0c1,
    0x0306, 0x0c18, 0x3060, 0xc081, 0x0102, 0x0408, 0x1020, 0x4080, 0xffff, 0x0000, 0x0000, 0x0000, 0xff00, 0x0000, 0x0000, 0x0000,
    0x77bb, 0xddee, 0x77bb, 0xddee, 0x99cc, 0x6633, 0x99cc, 0x6633, 0x8844, 0x2211, 0x8844, 0x2211, 0xe070, 0x381c, 0x0e07, 0x83c1,
    0xc060, 0x3018, 0x0c06, 0x0381, 0x8040, 0x2010, 0x0804, 0x0201, 0xc0c0, 0xc0c0, 0xc0c0, 0xc0c0, 0x8080, 0x8080, 0x8080, 0x8080,
    0xffaa, 0xffaa, 0xffaa, 0xffaa, 0xe4e4, 0xe4e4, 0xe4e4, 0xe4e4, 0xffff, 0xff00, 0x00ff, 0x0000, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
    0xff00, 0xff00, 0xff00, 0xff00, 0xff00, 0x0000, 0xff00, 0x0000, 0x8888, 0x8888, 0x8888, 0x8888, 0xff80, 0x8080, 0x8080, 0x8080,
    0x4ecf, 0xfce4, 0x273f, 0xf372, 0x6006, 0x36b1, 0x8118, 0x1b63, 0x2004, 0x4002, 0x1080, 0x0801, 0x9060, 0x0609, 0x9060, 0x0609,
    0x8814, 0x2241, 0x8800, 0xaa00, 0x2050, 0x8888, 0x8888, 0x0502, 0xaa00, 0x8000, 0x8800, 0x8000, 0x2040, 0x8000, 0x0804, 0x0200,
    0xf0f0, 0xf0f0, 0x0f0f, 0x0f0f, 0x0077, 0x7777, 0x0077, 0x7777, 0xff88, 0x8888, 0xff88, 0x8888, 0xaa44, 0xaa11, 0xaa44, 0xaa11,
    0x8244, 0x2810, 0x2844, 0x8201, 0x8080, 0x413e, 0x0808, 0x14e3, 0x8142, 0x2418, 0x1020, 0x4080, 0x40a0, 0x0000, 0x040a, 0x0000,
    0x7789, 0x8f8f, 0x7798, 0xf8f8, 0xf1f8, 0x6cc6, 0x8f1f, 0x3663, 0xbf00, 0xbfbf, 0xb0b0, 0xb0b0, 0xff80, 0x8080, 0xff08, 0x0808,
    0x1020, 0x54aa, 0xff02, 0x0408, 0x0008, 0x142a, 0x552a, 0x1408, 0x55a0, 0x4040, 0x550a, 0x0404, 0x8244, 0x3944, 0x8201, 0x0101
  };
  m_patternList.resize(64);
  for (size_t i = 0; i < 64; i++)
    m_patternList[i]=Pattern(&s_pattern[i*4]);
}

void State::setDefaultGradientList(int)
{
  if (!m_gradientList.empty()) return;
  Gradient grad;
  // grad0
  grad=Gradient(0,2,0,1);
  m_gradientList.push_back(grad);
  // grad1
  grad=Gradient(0,2,180,1);
  m_gradientList.push_back(grad);
  // grad2
  grad=Gradient(0,2,90,0);
  m_gradientList.push_back(grad);
  // grad3
  grad=Gradient(0,2,315,1);
  m_gradientList.push_back(grad);
  // grad4
  grad=Gradient(0,2,225,1);
  m_gradientList.push_back(grad);
  // grad5
  grad=Gradient(2,2,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(79,80),MWAWVec2i(79,80));
  m_gradientList.push_back(grad);
  // grad6
  grad=Gradient(2,2,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(81,20),MWAWVec2i(81,20));
  m_gradientList.push_back(grad);
  // grad7
  grad=Gradient(2,2,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(50,50),MWAWVec2i(50,50));
  m_gradientList.push_back(grad);
  // grad8
  grad=Gradient(0,2,90,1);
  m_gradientList.push_back(grad);
  // grad9
  grad=Gradient(0,2,270,0.979172f);
  m_gradientList.push_back(grad);
  // grad10
  grad=Gradient(0,2,0,0);
  m_gradientList.push_back(grad);
  // grad11
  grad=Gradient(0,2,45,1);
  m_gradientList.push_back(grad);
  // grad12
  grad=Gradient(0,2,135,1);
  m_gradientList.push_back(grad);
  // grad13
  grad=Gradient(2,2,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(22,77),MWAWVec2i(23,77));
  m_gradientList.push_back(grad);
  // grad14
  grad=Gradient(2,2,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(22,22),MWAWVec2i(22,22));
  m_gradientList.push_back(grad);
  // grad15
  grad=Gradient(1,2,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(51,50),MWAWVec2i(51,50));
  m_gradientList.push_back(grad);
  // grad16
  grad=Gradient(2,3,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(79,15),MWAWVec2i(86,22));
  grad.m_colors[1]=MWAWColor(0xaa0000);
  grad.m_colors[2]=MWAWColor(0xcc3300);
  m_gradientList.push_back(grad);
  // grad17
  grad=Gradient(0,4,0,1);
  grad.m_colors[0]=MWAWColor(0xff33cc);
  grad.m_colors[1]=MWAWColor(0xdd0000);
  grad.m_colors[2]=MWAWColor(0xdd0000);
  grad.m_colors[3]=MWAWColor(0x000000);
  m_gradientList.push_back(grad);
  // grad18
  grad=Gradient(0,3,112,0.80835f);
  grad.m_colors[0]=MWAWColor(0x0000dd);
  grad.m_colors[1]=MWAWColor(0x000077);
  grad.m_colors[2]=MWAWColor(0xff3333);
  m_gradientList.push_back(grad);
  // grad19
  grad=Gradient(1,4,332,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(77,71),MWAWVec2i(77,71));
  grad.m_colors[0]=MWAWColor(0xffff00);
  grad.m_colors[1]=MWAWColor(0xff3300);
  grad.m_colors[2]=MWAWColor(0x9900cc);
  grad.m_colors[3]=MWAWColor(0x0000dd);
  m_gradientList.push_back(grad);
  // grad20
  grad=Gradient(0,3,270,0.625f);
  grad.m_colors[1]=MWAWColor(0x0000dd);
  grad.m_colors[2]=MWAWColor(0x00cccc);
  m_gradientList.push_back(grad);
  // grad21
  grad=Gradient(0,2,270,0.229172f);
  grad.m_colors[0]=MWAWColor(0x0000aa);
  grad.m_colors[1]=MWAWColor(0xdddddd);
  m_gradientList.push_back(grad);
  // grad22
  grad=Gradient(0,3,90,0.729172f);
  grad.m_colors[0]=MWAWColor(0xffffff);
  grad.m_colors[2]=MWAWColor(0x9999ff);
  m_gradientList.push_back(grad);
  // grad23
  grad=Gradient(2,4,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(41,40),MWAWVec2i(62,62));
  grad.m_colors[0]=MWAWColor(0xffffff);
  grad.m_colors[1]=MWAWColor(0xccffff);
  grad.m_colors[2]=MWAWColor(0x99ffff);
  grad.m_colors[3]=MWAWColor(0x66ffff);
  m_gradientList.push_back(grad);
  // grad24
  grad=Gradient(0,3,90,0.854172f);
  grad.m_colors[1]=MWAWColor(0xdd0000);
  grad.m_colors[2]=MWAWColor(0xffcc00);
  m_gradientList.push_back(grad);
  // grad25
  grad=Gradient(0,4,315,0.633453f);
  grad.m_colors[0]=MWAWColor(0xcc3300);
  grad.m_colors[1]=MWAWColor(0xff6600);
  grad.m_colors[2]=MWAWColor(0xffcc00);
  grad.m_colors[3]=MWAWColor(0xffff00);
  m_gradientList.push_back(grad);
  // grad26
  grad=Gradient(0,3,45,0.721832f);
  grad.m_colors[1]=MWAWColor(0x0000dd);
  grad.m_colors[2]=MWAWColor(0xff0099);
  m_gradientList.push_back(grad);
  // grad27
  grad=Gradient(0,3,180,1);
  grad.m_colors[1]=MWAWColor(0x0000dd);
  grad.m_colors[2]=MWAWColor(0x9900cc);
  m_gradientList.push_back(grad);
  // grad28
  grad=Gradient(0,4,90,0.81987f);
  grad.m_colors[1]=MWAWColor(0x9900cc);
  grad.m_colors[2]=MWAWColor(0x9933ff);
  grad.m_colors[3]=MWAWColor(0x66ffff);
  m_gradientList.push_back(grad);
  // grad29
  grad=Gradient(0,4,270,0.916672f);
  grad.m_colors[0]=MWAWColor(0x0066ff);
  grad.m_colors[1]=MWAWColor(0x00ccff);
  grad.m_colors[2]=MWAWColor(0xffffcc);
  grad.m_colors[3]=MWAWColor(0xff6633);
  m_gradientList.push_back(grad);
  // grad30
  grad=Gradient(2,2,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(0,88),MWAWVec2i(12,100));
  grad.m_colors[0]=MWAWColor(0xff6600);
  grad.m_colors[1]=MWAWColor(0xffff00);
  m_gradientList.push_back(grad);
  // grad31
  grad=Gradient(2,4,0,0);
  grad.m_box=MWAWBox2i(MWAWVec2i(99,52),MWAWVec2i(100,54));
  grad.m_colors[0]=MWAWColor(0xffffff);
  grad.m_colors[1]=MWAWColor(0xffffcc);
  grad.m_colors[2]=MWAWColor(0xffff66);
  grad.m_colors[3]=MWAWColor(0xffcc00);
  m_gradientList.push_back(grad);
}

void State::setDefaultWallPaperList(int version)
{
  // checkme: does ClarisWork v4 version has wallpaper?
  if (version <= 2 || m_wallpaperList.size())
    return;
  librevenge::RVNGBinaryData data;
  std::string mime("image/pict");
  uint32_t const defCol[20] = {
    0xdcdcdc, 0x0000cd, 0xeeeeee, 0xeedd8e, 0xc71585,
    0xc9c9c9, 0xcd853f, 0x696969, 0xfa8072, 0x6495ed,
    0x4682b4, 0xdaa520, 0xcd5c5c, 0xb22222, 0x8b8682,
    0xb03060, 0xeeeee0, 0x4682b4, 0xfa8072, 0x505050
  };
  // 20 ppm contents corresponding to the default wallpaper ( after reducing each image by 50% in h and v )
  static unsigned char wall0[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,232,229,200,228,227,202,231,231,211,236,236,211,233,233,221,234,234,214,223,
    223,208,219,225,204,235,229,208,234,234,217,233,233,222,215,215,204,227,227,204,238,238,219,233,233,213,232,232,213,230,229,206,
    226,223,216,235,235,223,229,221,201,223,220,208,233,233,213,220,220,205,231,231,214,232,232,217,225,225,207,222,222,209,234,234,
    215,219,219,201,232,232,219,233,233,219,230,221,207,229,229,218,239,232,220,216,216,207,233,230,213,232,229,214,226,218,205,232,
    229,217,232,224,210,221,221,209,230,230,210,231,229,214,224,216,204,232,232,212,234,234,213,220,220,203,235,232,212,235,234,218,
    223,223,200,229,229,216,228,228,209,226,224,203,230,230,215,232,231,221,224,222,212,228,228,217,229,227,212,226,217,205,234,232,
    212,228,228,216,221,218,206,230,229,210,236,237,211,236,235,215,222,220,206,235,235,216,228,228,210,224,219,213,225,231,210,222,
    222,197,223,223,205,232,234,217,229,226,203,231,229,216,229,230,211,225,226,211,226,218,208,229,227,214,226,227,212,221,221,217,
    237,229,214,228,226,213,224,224,203,238,235,220,218,218,206,234,229,212,240,237,223,228,226,208,228,228,210,234,231,211,224,219,
    207,223,224,199,229,230,217,228,228,211,225,224,205,219,226,207,228,228,205,225,227,210,237,237,217,227,226,209,233,231,218,232,
    222,218,225,227,204,228,220,211,229,231,216,228,222,204,224,227,209,239,238,225,219,217,202,227,219,206,236,233,219,229,227,220,
    236,234,215,233,226,210,223,224,210,223,216,206,232,232,218,216,216,202,225,228,205,228,229,216,223,223,204,236,234,213,235,234,
    219,215,220,205,230,224,211,231,228,218,224,215,198,232,226,213,227,227,213,230,224,212,226,226,211,238,239,216,219,219,201,229,
    229,208,236,239,221,219,217,207,235,228,211,229,229,215,219,219,205,228,228,211,236,229,216,222,223,205,231,229,211,218,228,205,
    226,221,203,232,230,213,233,231,218,226,226,211,227,226,212,236,232,217,224,221,207,220,218,208,238,233,222,224,224,207,233,225,
    214,230,230,215,222,228,209,232,229,210,231,230,217,233,224,205,226,229,213,227,229,212,224,224,204,221,221,204,236,236,219,226,
    223,200,234,227,213,230,229,219,216,211,197,230,230,210,223,221,214,230,223,206,227,228,213,237,230,218,231,228,211,237,237,214,
    226,232,213,227,224,204,230,225,212,234,232,221,216,216,204,226,226,217,227,229,219,223,222,201,232,235,213,226,226,216,224,224,
    207,227,225,213,236,235,215,220,221,203,225,225,211,223,216,210,231,229,207,235,224,219,225,222,209,221,223,206,232,232,214,236,
    228,213,223,218,201,224,221,212,229,234,215,223,224,211,222,221,205,233,234,211,222,223,206,227,226,211,240,240,218,229,225,209,
    232,229,211,228,228,213,226,224,206,233,233,216,234,234,214,223,221,202,228,227,213,232,239,217,221,216,210,215,225,209,235,230,
    221,220,223,207,227,227,214,233,230,211,223,223,200,229,232,206,238,231,214,222,222,205,226,226,208,231,231,215,224,224,211,226,
    227,218,229,228,208,226,221,206,229,227,214,228,227,216,220,221,201,229,227,210,232,231,214,226,226,208,229,229,215,245,239,214,
    231,223,207,228,226,207,236,236,220,217,217,195,228,220,209,230,227,215,226,225,203,234,231,217,235,236,227,226,217,199,233,230,
    219,233,224,207,222,219,203,226,226,208,238,238,222
  };
  data=librevenge::RVNGBinaryData(wall0,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[0])));

  static unsigned char wall1[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,67,80,149,62,66,138,64,70,147,60,71,131,67,68,150,53,68,125,69,
    73,145,65,69,128,70,69,144,60,72,137,65,69,144,67,68,134,60,70,140,60,67,132,64,70,152,72,65,133,66,69,132,
    65,69,144,65,64,132,59,74,138,67,70,142,64,73,137,66,66,136,66,66,144,62,72,135,62,69,140,66,71,137,65,69,
    137,66,71,136,62,66,139,65,73,136,61,68,137,64,72,146,62,73,135,62,75,142,70,68,134,60,72,141,66,67,136,60,
    73,143,69,71,136,62,69,139,64,72,135,61,71,136,60,69,140,64,71,139,66,65,132,65,71,141,68,76,137,70,66,129,
    61,70,142,64,68,136,65,69,142,64,68,135,64,73,138,67,69,139,62,68,137,64,70,133,65,69,145,66,74,136,62,69,
    137,66,68,141,70,72,143,63,67,136,61,65,134,64,70,147,69,68,137,58,75,144,67,68,134,64,71,139,64,68,138,64,
    67,138,59,73,137,68,68,140,63,66,134,62,73,139,64,73,139,66,66,139,60,71,142,61,69,140,65,74,144,57,68,134,
    65,74,135,65,69,137,64,71,139,64,71,140,62,68,139,67,73,138,61,72,143,65,71,136,65,71,143,64,65,137,64,73,
    137,69,70,138,60,75,136,61,68,133,57,66,138,66,72,145,58,66,136,66,73,137,67,67,140,64,65,140,61,74,137,62,
    64,134,62,71,137,67,68,139,59,74,139,66,73,140,64,70,137,64,67,139,58,74,138,68,68,140,64,74,136,66,66,135,
    67,68,140,65,74,139,59,69,137,63,67,140,65,73,137,69,72,132,66,65,144,64,71,140,60,69,137,65,68,135,64,70,
    140,68,68,138,59,68,138,65,73,136,67,68,137,61,79,146,66,72,138,63,67,140,65,70,136,62,71,138,61,65,138,68,
    73,141,65,71,134,66,73,138,64,69,139,66,70,139,58,68,135,67,72,141,69,71,138,64,66,143,64,73,138,62,65,128,
    63,73,138,68,70,131,62,71,142,68,68,134,62,70,144,66,71,137,60,69,143,64,69,134,67,67,143,64,75,134,65,70,
    142,63,72,134,65,70,142,62,73,133,55,73,131,68,73,147,66,67,142,66,70,143,58,66,134,63,72,143,65,73,134,66,
    69,138,62,73,140,64,70,142,66,66,134,62,72,144,59,68,136,64,71,143,65,68,136,65,66,140,69,75,136,56,74,128,
    64,67,139,69,73,137,65,71,143,62,65,134,66,66,142,67,70,139,62,72,142,68,69,134,64,74,142,62,70,133,65,71,
    140,59,68,135,67,78,144,61,66,137,64,71,141,62,76,149,60,67,135,63,71,139,67,67,139,65,75,138,67,70,133,61,
    71,139,60,70,136,64,67,137,69,71,139,64,66,137,58,74,139,66,65,132,66,71,142,65,72,137,67,75,143,65,71,128,
    63,67,144,70,71,132,62,66,139,61,73,138,67,73,139,63,68,137,65,66,136,65,70,139,65,69,139,61,68,139,63,75,
    140,65,70,139,65,70,142,62,69,136,58,71,133,66,76,148,64,70,134,62,72,139,60,68,139,61,72,142,65,73,137,64,
    71,141,66,69,135,64,74,143,68,65,133,60,72,142,66,68,133,66,69,138,67,68,142,64,72,139,67,70,137,61,74,129,
    67,70,143,64,66,141,69,73,133,66,67,145,63,63,135,60,77,137,61,69,132,71,66,143,60,72,133,65,70,137,63,67,
    134,60,67,141,65,70,133,67,71,137,58,58,135
  };
  data=librevenge::RVNGBinaryData(wall1,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[1])));

  static unsigned char wall2[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,214,214,214,216,216,216,215,215,215,223,223,223,220,220,220,217,217,217,214,
    214,214,218,218,218,218,218,218,215,215,215,211,211,211,220,220,220,224,224,224,219,219,219,217,217,217,213,213,213,218,218,218,
    221,221,221,222,222,222,218,218,218,217,217,217,216,216,216,213,213,213,214,214,214,215,215,215,217,217,217,219,219,219,222,222,
    222,219,219,219,209,209,209,214,214,214,217,217,217,222,222,222,218,218,218,216,216,216,221,221,221,212,212,212,214,214,214,217,
    217,217,217,217,217,217,217,217,223,223,223,221,221,221,216,216,216,212,212,212,218,218,218,219,219,219,221,221,221,210,210,210,
    214,214,214,218,218,218,208,208,208,213,213,213,218,218,218,218,218,218,217,217,217,224,224,224,217,217,217,212,212,212,211,211,
    211,220,220,220,219,219,219,216,216,216,216,216,216,218,218,218,218,218,218,210,210,210,215,215,215,216,216,216,215,215,215,219,
    219,219,222,222,222,215,215,215,208,208,208,216,216,216,223,223,223,214,214,214,216,216,216,218,218,218,218,218,218,212,212,212,
    210,210,210,216,216,216,220,220,220,219,219,219,218,218,218,224,224,224,215,215,215,210,210,210,217,217,217,222,222,222,213,213,
    213,216,216,216,220,220,220,218,218,218,222,222,222,224,224,224,221,221,221,213,213,213,218,218,218,223,223,223,219,219,219,214,
    214,214,213,213,213,215,215,215,218,218,218,216,216,216,217,217,217,217,217,217,216,216,216,221,221,221,215,215,215,222,222,222,
    218,218,218,219,219,219,221,221,221,218,218,218,210,210,210,218,218,218,213,213,213,215,215,215,222,222,222,219,219,219,216,216,
    216,216,216,216,223,223,223,214,214,214,221,221,221,216,216,216,219,219,219,216,216,216,215,215,215,211,211,211,219,219,219,214,
    214,214,215,215,215,222,222,222,219,219,219,208,208,208,215,215,215,220,220,220,219,219,219,218,218,218,213,213,213,216,216,216,
    215,215,215,212,212,212,215,215,215,215,215,215,214,214,214,217,217,217,225,225,225,215,215,215,211,211,211,218,218,218,223,223,
    223,221,221,221,215,215,215,213,213,213,215,215,215,218,218,218,211,211,211,218,218,218,216,216,216,213,213,213,220,220,220,223,
    223,223,212,212,212,212,212,212,220,220,220,222,222,222,215,215,215,213,213,213,215,215,215,219,219,219,222,222,222,219,219,219,
    218,218,218,219,219,219,214,214,214,221,221,221,223,223,223,211,211,211,214,214,214,220,220,220,220,220,220,213,213,213,214,214,
    214,215,215,215,218,218,218,216,216,216,214,214,214,214,214,214,219,219,219,215,215,215,219,219,219,218,218,218,211,211,211,218,
    218,218,218,218,218,217,217,217,218,218,218,217,217,217,218,218,218,219,219,219,213,213,213,216,216,216,215,215,215,219,219,219,
    217,217,217,219,219,219,217,217,217,212,212,212,223,223,223,216,216,216,214,214,214,218,218,218,217,217,217,220,220,220,218,218,
    218,213,213,213,215,215,215,217,217,217,224,224,224,219,219,219,225,225,225,216,216,216,212,212,212,226,226,226,216,216,216,214,
    214,214,222,222,222,217,217,217,218,218,218,213,213,213,210,210,210,216,216,216,217,217,217,227,227,227,221,221,221,218,218,218,
    215,215,215,210,210,210,225,225,225,214,214,214,215,215,215,223,223,223,216,216,216,218,218,218,216,216,216,209,209,209,213,213,
    213,220,220,220,224,224,224,215,215,215,204,204,204
  };
  data=librevenge::RVNGBinaryData(wall2,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[2])));

  static unsigned char wall3[3085]= {
    80,54,10,51,50,32,51,50,10,50,53,53,10,255,255,153,255,255,153,255,254,152,255,255,154,242,242,150,216,216,149,217,
    217,154,191,191,143,241,241,146,255,255,154,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,
    255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,254,152,255,255,156,249,245,149,222,220,170,149,150,247,144,
    144,234,207,207,198,158,158,156,218,218,146,255,255,154,254,254,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,
    255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,152,255,255,156,254,254,161,253,189,33,228,173,81,148,155,245,158,
    156,234,230,230,214,223,223,230,159,159,156,244,244,148,255,255,154,254,254,153,255,255,153,255,255,153,255,255,153,255,255,153,
    255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,154,255,252,150,255,198,42,248,119,0,218,92,8,193,185,177,255,
    255,255,227,226,228,194,194,191,182,182,189,198,198,145,255,255,154,253,253,153,255,255,153,255,255,153,255,255,153,255,255,153,
    255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,165,255,199,83,246,107,0,244,169,81,248,234,142,199,199,144,178,
    178,153,154,154,154,217,217,214,188,188,197,179,179,129,255,255,157,252,252,152,255,255,153,255,255,153,255,255,153,255,255,153,
    255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,154,255,249,148,252,241,147,254,254,164,255,255,156,254,254,155,255,
    255,150,194,194,146,240,240,246,179,179,185,168,168,117,255,255,159,252,252,152,255,255,153,255,255,153,255,255,153,255,255,153,
    255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,154,255,255,155,255,252,150,255,254,152,251,251,153,255,
    255,154,199,199,149,219,219,225,172,172,178,167,167,115,255,255,159,252,252,152,255,255,153,255,255,153,255,255,153,255,255,153,
    255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,254,152,255,255,153,255,255,153,253,253,153,255,
    255,152,201,201,151,240,240,246,145,145,153,166,166,119,255,255,159,252,252,152,255,255,153,254,254,153,253,253,153,255,255,155,
    255,255,155,255,255,156,255,255,155,255,255,155,255,255,152,254,254,152,254,254,154,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,253,253,153,255,
    255,155,195,195,145,211,211,219,148,148,147,214,214,138,255,255,155,251,251,152,254,254,153,255,255,154,254,254,153,240,240,144,
    239,239,143,239,239,142,240,240,144,241,241,144,250,250,157,255,255,159,255,255,151,254,254,152,253,253,153,253,253,153,254,254,
    153,255,255,154,254,254,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,155,243,
    243,143,196,196,168,231,231,236,132,132,131,215,215,142,255,255,163,254,254,158,254,254,153,221,221,147,195,195,150,197,197,189,
    201,201,196,202,202,197,194,194,189,181,181,175,168,168,165,192,192,167,234,234,157,255,255,156,255,255,151,255,255,153,255,255,
    153,246,246,148,254,254,151,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,154,254,254,151,188,
    188,159,219,219,227,255,255,252,131,131,136,103,103,85,202,202,127,187,187,127,166,166,138,187,187,193,228,228,236,253,253,254,
    250,250,251,255,255,255,248,248,249,252,252,253,230,230,231,217,217,222,184,184,190,188,188,160,215,215,161,207,207,156,203,203,
    154,202,202,196,237,237,165,255,255,150,254,254,153,255,255,153,255,255,153,255,255,153,254,254,153,255,255,155,215,215,142,205,
    205,210,255,255,255,238,238,238,206,206,205,122,122,127,127,127,130,140,140,147,213,213,219,250,250,249,252,252,251,252,252,252,
    254,254,254,230,230,229,232,232,232,248,248,247,240,240,240,227,227,226,246,246,245,210,210,213,136,136,146,122,122,130,229,229,
    235,222,222,232,205,205,155,255,255,152,253,253,153,255,255,153,255,255,153,255,255,153,253,253,153,255,255,151,202,202,159,238,
    238,249,255,255,255,246,246,246,229,229,229,229,229,228,246,246,246,255,255,255,253,253,253,245,245,244,250,250,250,255,255,255,
    255,255,255,252,252,252,255,255,255,237,237,237,226,226,226,243,243,244,219,219,216,217,217,205,133,133,133,173,173,171,255,255,
    255,198,198,193,239,239,146,255,255,154,254,254,153,255,255,153,255,255,153,255,255,153,254,254,153,255,255,152,233,233,159,228,
    228,230,255,255,255,239,239,239,247,247,247,253,253,253,253,253,253,251,251,251,237,237,237,246,246,246,253,253,253,248,248,248,
    253,253,253,252,252,252,240,240,240,232,232,232,230,230,230,243,243,242,247,247,247,144,144,146,146,146,145,231,231,230,222,222,
    220,173,173,180,212,212,161,255,255,152,254,254,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,251,251,153,163,
    163,155,218,219,222,241,241,241,255,255,255,253,253,253,245,245,245,249,249,249,248,248,248,248,248,248,249,249,249,254,254,254,
    248,248,248,238,238,238,239,239,239,254,254,254,250,250,250,241,241,241,172,172,172,153,153,152,240,240,242,242,242,243,209,209,
    209,152,152,154,167,167,159,244,244,155,254,254,152,255,255,153,255,255,153,255,255,153,255,255,153,253,253,153,255,255,154,199,
    199,149,229,229,233,252,251,248,252,252,252,255,255,255,229,229,229,242,242,242,255,255,255,245,245,245,249,249,249,255,255,255,
    255,255,255,253,253,253,240,240,240,242,242,242,214,214,214,176,176,176,163,163,163,236,236,238,232,232,222,196,195,191,184,181,
    178,227,227,224,178,178,184,165,165,136,255,255,154,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,243,
    245,151,177,167,153,223,222,224,245,247,247,255,255,255,248,248,248,216,216,216,203,203,203,181,181,181,192,192,192,200,200,200,
    219,219,219,203,203,203,195,195,195,172,172,172,153,153,153,161,161,161,239,239,239,216,216,216,182,183,178,105,102,97,212,202,
    194,177,179,189,94,94,97,213,213,139,255,255,156,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,254,254,153,255,
    255,153,214,217,148,192,182,170,224,218,215,237,237,235,241,242,241,253,253,253,230,230,230,222,222,222,210,210,210,175,175,175,
    178,178,178,154,154,154,165,165,165,180,180,180,197,197,197,194,194,193,194,194,193,113,113,118,73,73,83,138,138,141,178,180,
    172,178,178,133,222,222,148,255,255,157,253,253,152,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,253,253,153,249,248,148,174,163,122,182,179,188,202,203,205,223,223,223,251,251,250,212,212,212,242,242,242,238,238,239,
    229,229,229,237,237,237,206,206,206,235,235,234,233,233,231,192,192,194,101,101,106,142,142,112,199,199,149,222,222,149,248,248,
    146,254,254,156,255,255,154,254,254,152,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,253,253,153,255,255,156,242,244,148,191,191,141,164,164,155,140,140,142,172,171,180,192,195,196,207,206,206,207,203,203,
    195,193,191,185,184,181,172,172,172,141,141,149,89,89,96,99,99,96,198,198,126,255,255,161,253,253,154,255,255,154,255,255,
    154,253,253,153,254,254,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,254,254,153,254,254,154,253,253,154,253,253,152,209,209,135,153,158,110,111,95,89,110,116,116,98,117,119,
    96,105,116,88,93,104,106,105,104,166,166,117,171,171,117,240,240,147,255,255,158,254,254,152,252,252,153,254,254,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,153,253,253,153,254,253,152,255,255,162,252,195,125,113,11,0,197,117,27,165,52,47,
    158,98,29,166,134,86,233,239,163,255,255,158,254,254,159,255,255,155,253,253,152,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,153,255,255,153,254,247,149,255,255,166,245,139,95,162,22,31,171,21,7,94,11,0,
    249,164,0,174,68,1,211,173,106,255,255,158,251,249,150,254,254,152,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,153,255,255,155,255,250,157,254,253,148,255,116,68,218,0,0,186,25,27,163,43,40,
    212,89,15,170,36,14,182,25,27,255,243,145,255,255,157,255,254,154,255,255,153,255,254,151,255,254,151,255,254,151,255,254,
    151,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,155,255,245,145,255,156,105,251,78,70,255,99,97,199,3,4,182,0,0,255,82,82,
    242,43,55,250,35,41,198,25,18,239,237,138,255,255,150,255,251,146,255,254,154,255,255,162,255,255,161,255,255,161,255,255,
    161,255,255,155,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,254,152,255,255,156,255,240,142,245,24,25,253,12,17,255,20,19,179,0,0,179,16,16,255,47,47,
    253,0,0,249,0,1,164,12,3,240,195,54,255,210,58,254,204,53,255,218,79,255,231,106,255,229,101,255,230,102,255,230,
    103,255,250,144,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,254,253,152,255,255,159,203,137,85,187,0,0,182,2,2,124,0,0,218,24,24,255,53,53,
    254,0,0,251,0,3,154,13,3,242,196,45,255,206,50,254,203,50,255,202,46,255,201,44,255,200,43,255,198,38,255,202,
    46,255,212,66,255,253,148,255,255,154,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,153,255,254,153,255,255,161,204,165,101,157,57,29,214,21,21,255,109,109,251,35,35,
    255,0,0,231,0,1,155,9,5,229,177,62,255,210,54,254,203,51,255,204,50,255,201,45,255,202,47,255,221,85,255,254,
    150,255,252,146,255,255,152,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,153,255,254,153,255,252,151,255,255,162,255,101,73,255,56,58,255,20,17,255,0,0,
    252,2,1,167,0,0,173,65,40,222,157,73,255,202,42,255,202,49,255,209,61,255,226,96,255,241,124,255,255,158,255,255,
    155,255,255,154,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,153,255,255,153,254,253,151,255,255,160,234,92,66,195,0,0,199,4,3,202,0,0,
    178,16,7,184,103,63,234,195,101,253,217,78,255,230,101,255,240,124,255,254,152,255,254,160,255,255,158,255,255,152,255,255,
    152,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,254,152,255,255,155,246,243,148,187,137,83,181,126,76,201,149,86,
    229,198,117,254,255,156,255,255,149,255,254,153,255,255,163,255,255,158,255,255,154,255,254,151,255,254,152,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,
    255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,153,255,255,152,255,254,154,255,255,166,255,255,167,255,255,165,
    255,255,160,255,255,153,254,254,154,255,255,153,255,254,150,255,254,152,255,255,153,255,255,153,255,255,153,255,255,153,255,255,
    153,255,255,153,255,255,153,255,255,153,255,255,153
  };
  data=librevenge::RVNGBinaryData(wall3,3085);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(32,32),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[3])));
  static unsigned char wall4[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,189,16,230,207,20,255,213,16,255,209,20,255,207,16,255,207,14,255,207,
    14,255,214,20,255,212,16,255,215,14,255,207,18,255,208,14,255,207,16,255,209,20,255,177,23,221,151,6,183,158,11,209,
    194,10,238,217,19,255,217,14,255,217,17,255,217,18,255,217,14,255,214,14,255,214,17,255,216,18,255,216,17,255,215,15,
    255,216,16,255,215,14,254,149,12,195,136,9,182,152,9,203,159,10,208,200,13,238,216,14,255,216,15,255,216,14,255,215,
    18,255,219,13,255,216,15,255,218,14,255,216,20,255,212,15,255,220,17,255,206,11,238,142,0,173,137,3,177,153,11,204,
    153,10,203,172,10,211,210,16,248,214,21,255,214,14,255,217,15,255,215,15,255,216,21,255,216,13,255,217,15,255,217,16,
    255,212,19,255,177,12,216,124,12,164,137,14,183,153,11,204,153,11,201,152,10,203,169,13,221,207,13,250,217,15,255,217,
    20,255,210,15,255,219,14,255,214,15,255,214,13,255,216,16,255,210,15,247,153,5,188,127,6,164,138,15,183,153,12,204,
    153,5,196,153,9,201,152,6,204,180,8,222,218,18,253,214,17,255,215,15,255,217,13,255,216,21,255,220,15,255,212,19,
    255,188,12,226,140,10,175,132,5,161,137,8,187,153,8,204,153,11,204,153,9,204,153,10,201,156,11,205,184,12,229,215,
    18,252,217,15,255,217,17,255,217,13,255,212,15,255,211,14,253,165,8,199,130,7,168,127,2,164,145,9,177,153,9,204,
    153,8,196,153,11,202,153,10,196,153,11,201,157,7,206,190,13,237,218,16,255,215,20,255,216,10,255,217,19,255,192,16,
    234,144,6,178,135,5,175,130,3,160,146,4,184,153,11,204,153,11,201,153,8,203,153,11,204,153,5,202,152,10,203,160,
    8,208,202,14,238,217,16,255,219,18,255,214,14,253,176,12,211,136,3,170,132,8,172,117,10,165,135,3,177,153,5,204,
    153,8,205,153,10,196,153,10,202,153,11,196,153,7,201,153,10,204,169,10,211,207,13,248,217,15,255,202,11,247,148,11,
    186,129,5,173,136,10,177,122,8,170,136,11,183,153,11,203,153,10,202,153,10,202,153,7,202,153,11,202,153,10,204,153,
    10,201,153,8,203,169,13,221,210,16,251,187,12,220,134,3,169,136,4,175,132,6,171,125,12,164,138,12,181,153,11,201,
    153,11,196,153,5,205,153,9,202,153,7,196,153,8,205,153,10,197,153,10,202,153,7,202,174,12,212,150,10,191,135,7,
    170,132,10,177,136,5,170,127,13,170,138,11,186,153,4,204,153,8,205,153,11,202,153,10,197,154,9,206,151,8,197,150,
    8,195,140,5,181,130,1,168,131,3,157,128,2,163,129,2,168,136,8,171,132,10,173,131,3,161,140,11,186,153,8,202,
    154,10,197,150,5,199,147,11,194,140,5,181,129,4,174,122,6,159,121,7,153,120,2,152,122,8,153,118,2,153,122,8,
    153,121,2,163,134,3,168,132,7,167,146,11,186,145,8,186,132,2,173,126,10,161,119,6,158,119,0,153,117,8,152,120,
    3,153,116,5,153,117,7,153,117,3,153,123,11,153,120,2,153,122,7,153,118,3,153,120,2,157,135,5,181,110,3,155,
    119,4,152,118,17,152,119,3,153,120,0,153,119,0,153,116,12,153,103,13,153,116,2,153,115,12,153,121,14,153,117,0,
    153,119,3,153,124,1,153,128,0,153,121,10,155
  };
  data=librevenge::RVNGBinaryData(wall4,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[4])));

  static unsigned char wall5[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,194,194,181,244,245,220,195,194,180,208,206,189,192,189,177,199,198,179,209,
    206,197,219,218,199,178,175,166,219,210,193,208,209,191,205,202,181,209,200,192,199,197,177,217,209,197,221,220,196,203,203,186,
    246,239,211,195,190,175,204,197,176,206,205,185,214,212,190,214,207,192,209,207,180,176,176,166,221,219,198,199,199,176,204,203,
    179,201,199,184,210,212,189,222,220,192,219,216,195,220,215,209,223,212,200,194,194,182,216,216,198,227,226,207,229,219,207,193,
    194,179,219,211,192,217,212,198,234,231,208,201,193,182,220,220,200,196,195,180,225,220,204,218,211,203,216,208,194,202,201,189,
    212,207,180,209,201,189,225,223,199,218,216,195,194,185,167,187,185,162,224,222,198,213,204,197,225,217,192,180,180,162,221,214,
    198,214,212,190,232,225,203,209,205,187,192,180,160,185,187,172,212,210,188,211,203,192,228,223,202,201,201,193,193,191,172,225,
    216,206,227,228,203,209,210,193,214,212,191,185,183,173,240,237,211,200,197,187,207,204,185,193,196,182,181,180,165,204,204,186,
    232,225,203,205,209,187,209,201,179,185,183,165,229,221,197,228,226,208,212,209,184,180,182,161,218,216,194,213,206,192,233,231,
    206,193,191,178,215,205,184,209,201,183,196,195,173,201,208,192,221,221,200,196,194,175,202,194,178,207,205,189,223,222,199,211,
    209,199,211,204,189,192,183,175,241,237,218,207,200,190,217,215,195,183,180,170,220,222,200,214,213,200,185,182,169,193,187,176,
    229,214,202,183,182,165,208,206,180,221,216,200,195,191,169,201,199,182,214,214,192,219,214,196,232,224,197,181,179,161,207,207,
    183,222,219,199,242,227,204,210,205,191,210,201,181,184,183,171,213,212,195,204,207,192,240,232,208,224,220,211,198,186,176,215,
    215,198,239,239,214,203,202,189,208,208,185,203,199,192,236,227,206,219,224,208,217,211,200,203,202,184,221,218,202,176,170,156,
    225,222,194,222,215,192,209,204,186,204,196,180,205,200,178,235,227,211,211,211,182,177,176,166,224,222,191,211,207,189,225,218,
    199,196,195,173,195,196,173,217,209,191,217,214,194,189,186,179,230,222,206,188,183,170,216,209,197,213,212,196,216,214,197,219,
    219,205,193,189,170,207,201,197,235,229,212,203,198,183,223,216,202,198,199,188,218,219,197,232,231,213,202,196,185,203,205,186,
    220,210,188,185,182,170,214,209,190,204,201,183,230,222,194,207,204,190,202,200,177,215,216,196,224,220,199,206,201,184,209,202,
    180,214,211,189,225,217,192,218,216,195,184,180,161,199,195,183,220,216,200,197,199,186,234,232,205,218,217,199,218,216,200,188,
    187,174,228,225,205,221,217,205,230,226,203,195,199,185,210,205,190,232,230,216,212,210,192,205,196,189,207,203,189,170,161,145,
    212,213,185,203,204,188,229,227,199,203,201,181,204,197,177,211,208,192,234,226,197,183,179,162,208,201,181,194,194,174,230,223,
    198,216,212,193,192,192,171,200,197,178,227,224,200,187,188,178,237,236,214,211,208,200,208,200,186,192,189,181,214,212,199,214,
    213,205,213,210,190,184,185,170,230,222,205,212,208,199,229,227,199,208,200,192,206,202,188,219,216,203,227,224,198,214,205,188,
    221,212,187,195,191,178,201,198,173,223,215,201,213,204,179,210,205,185,213,201,182,192,199,173,234,226,194,215,211,193,220,215,
    190,186,193,177,227,223,190,236,225,205,231,221,195
  };
  data=librevenge::RVNGBinaryData(wall5,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[5])));

  static unsigned char wall6[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,164,114,35,175,126,42,211,147,51,200,149,48,146,110,40,196,144,48,225,
    179,59,189,146,48,164,121,41,195,152,50,193,156,48,191,149,50,201,157,49,164,116,42,176,137,49,170,116,39,161,118,39,
    159,122,44,160,111,41,127,91,35,126,95,33,183,137,43,172,133,47,107,80,23,116,83,30,211,155,56,204,169,59,128,95,
    33,126,93,31,144,109,36,154,116,41,153,107,32,198,149,52,176,118,38,168,125,42,223,190,63,201,153,47,170,122,38,152,
    107,42,163,126,44,179,144,48,249,208,75,203,162,55,127,84,37,194,144,48,209,145,48,187,153,53,110,82,22,150,121,39,
    162,122,42,215,176,58,241,205,70,197,139,52,163,112,41,191,149,52,219,165,58,212,162,57,222,178,68,169,122,40,145,98,
    30,207,144,49,210,156,53,158,110,42,156,108,35,117,98,35,124,99,35,189,140,51,176,129,41,128,92,30,142,106,39,204,
    158,51,180,155,54,127,91,28,163,125,40,158,112,41,137,104,35,184,133,44,183,147,56,115,88,28,169,124,39,135,98,29,
    133,91,31,207,163,58,139,104,34,122,92,27,159,115,38,206,150,50,116,89,30,125,87,30,197,153,48,175,131,41,141,112,
    38,159,119,42,187,142,51,166,120,40,182,143,48,173,116,44,143,105,35,185,144,52,175,121,42,203,164,59,225,182,63,210,
    160,56,140,104,39,216,165,56,244,197,69,216,167,57,180,134,47,189,149,48,231,166,59,188,135,47,198,144,46,125,79,28,
    121,90,31,180,140,44,207,182,61,183,157,54,165,122,42,181,130,39,172,134,46,218,180,59,215,163,51,144,106,41,128,96,
    36,149,106,37,228,175,54,177,122,38,143,94,34,114,89,27,126,104,35,204,158,56,198,166,56,106,75,21,142,115,40,162,
    116,42,174,128,47,147,116,38,128,94,34,136,104,34,151,105,32,186,139,52,175,142,51,116,81,31,139,113,45,129,104,38,
    174,144,49,232,179,54,193,141,45,120,79,29,208,152,48,217,166,55,201,157,56,162,113,39,194,143,47,229,185,65,209,150,
    52,191,150,52,184,147,48,195,170,58,189,173,69,208,171,61,161,132,48,177,126,44,149,103,33,143,106,39,239,192,65,248,
    199,67,198,152,58,199,151,51,249,212,73,213,180,65,161,125,46,186,139,46,217,169,55,190,155,56,195,165,53,154,119,47,
    120,87,26,159,121,40,165,128,46,172,148,57,202,167,61,182,142,48,134,106,31,160,121,42,218,166,61,159,119,35,109,81,
    24,185,143,48,235,177,59,189,150,51,151,116,43,144,98,29,134,87,34,203,157,56,203,148,46,157,117,36,141,99,33,170,
    124,41,147,108,40,189,133,41,202,141,48,149,100,35,125,89,30,199,146,51,217,160,52,142,98,38,144,102,36,157,114,43,
    188,140,49,243,221,81,222,174,60,155,105,41,178,131,46,243,192,65,235,180,65,217,159,56,211,152,52,206,151,52,198,165,
    58,232,198,70,204,152,56,176,126,46,193,142,42,180,134,39,207,142,57,188,146,51,161,111,38,150,104,30,226,179,59,250,
    226,81,193,150,53,161,112,39,205,151,54,191,142,48,180,138,46,157,122,42,192,140,46,191,150,50,189,156,51,166,128,41,
    161,122,48,186,147,42,162,126,43,141,104,39,206,154,53,200,155,52,115,87,28,136,94,36,161,113,40,144,104,37,122,93,
    29,128,93,27,219,164,55,168,131,49,113,82,25
  };
  data=librevenge::RVNGBinaryData(wall6,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[6])));

  static unsigned char wall7[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,61,61,61,53,53,53,61,61,61,59,59,59,51,51,51,61,61,61,63,
    63,63,52,52,52,58,58,58,66,66,66,60,60,60,63,63,63,63,63,63,62,62,62,51,51,51,69,69,69,56,56,56,
    57,57,57,61,61,61,61,61,61,58,58,58,56,56,56,55,55,55,57,57,57,58,58,58,65,65,65,65,65,65,59,59,
    59,66,66,66,58,58,58,57,57,57,69,69,69,59,59,59,61,61,61,64,64,64,61,61,61,54,54,54,62,62,62,66,
    66,66,61,61,61,63,63,63,64,64,64,60,60,60,63,63,63,67,67,67,60,60,60,60,60,60,65,65,65,58,58,58,
    64,64,64,60,60,60,59,59,59,60,60,60,69,69,69,66,66,66,67,67,67,65,65,65,63,63,63,61,61,61,65,65,
    65,60,60,60,56,56,56,65,65,65,65,65,65,66,66,66,59,59,59,60,60,60,60,60,60,58,58,58,73,73,73,64,
    64,64,65,65,65,58,58,58,55,55,55,60,60,60,65,65,65,59,59,59,63,63,63,70,70,70,63,63,63,58,58,58,
    59,59,59,64,64,64,64,64,64,62,62,62,66,66,66,60,60,60,69,69,69,57,57,57,60,60,60,64,64,64,63,63,
    63,61,61,61,63,63,63,63,63,63,58,58,58,60,60,60,62,62,62,66,66,66,65,65,65,66,66,66,62,62,62,68,
    68,68,59,59,59,65,65,65,61,61,61,59,59,59,60,60,60,54,54,54,55,55,55,70,70,70,63,63,63,60,60,60,
    65,65,65,58,58,58,59,59,59,60,60,60,61,61,61,60,60,60,59,59,59,63,63,63,64,64,64,64,64,64,66,66,
    66,64,64,64,70,70,70,69,69,69,60,60,60,69,69,69,63,63,63,60,60,60,62,62,62,64,64,64,64,64,64,57,
    57,57,57,57,57,58,58,58,61,61,61,62,62,62,58,58,58,58,58,58,65,65,65,68,68,68,61,61,61,61,61,61,
    57,57,57,60,60,60,58,58,58,64,64,64,55,55,55,55,55,55,56,56,56,58,58,58,60,60,60,62,62,62,56,56,
    56,68,68,68,64,64,64,70,70,70,56,56,56,57,57,57,63,63,63,63,63,63,59,59,59,65,65,65,56,56,56,67,
    67,67,61,61,61,70,70,70,69,69,69,67,67,67,65,65,65,71,71,71,54,54,54,56,56,56,56,56,56,59,59,59,
    69,69,69,60,60,60,62,62,62,65,65,65,56,56,56,61,61,61,59,59,59,64,64,64,64,64,64,63,63,63,62,62,
    62,56,56,56,55,55,55,60,60,60,57,57,57,68,68,68,67,67,67,62,62,62,61,61,61,60,60,60,54,54,54,67,
    67,67,59,59,59,56,56,56,67,67,67,60,60,60,59,59,59,66,66,66,68,68,68,64,64,64,73,73,73,57,57,57,
    62,62,62,58,58,58,59,59,59,61,61,61,68,68,68,66,66,66,60,60,60,62,62,62,62,62,62,65,65,65,70,70,
    70,64,64,64,54,54,54,63,63,63,63,63,63,59,59,59,65,65,65,61,61,61,66,66,66,68,68,68,64,64,64,57,
    57,57,59,59,59,63,63,63,67,67,67,63,63,63,62,62,62,59,59,59,69,69,69,64,64,64,61,61,61,60,60,60,
    57,57,57,71,71,71,62,62,62,65,65,65,56,56,56,63,63,63,67,67,67,69,69,69,62,62,62,58,58,58,57,57,
    57,62,62,62,63,63,63,55,55,55,52,52,52
  };
  data=librevenge::RVNGBinaryData(wall7,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[7])));

  static unsigned char wall8[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,239,239,238,240,
    240,238,239,239,238,238,238,238,238,238,238,238,238,238,239,239,238,239,239,238,239,239,238,239,239,238,239,239,238,239,239,238,
    239,239,238,239,239,238,239,239,238,238,238,238,225,225,233,188,188,234,213,213,238,239,239,238,238,238,238,238,238,234,223,223,
    240,227,227,245,221,221,246,211,211,247,211,211,247,211,211,247,212,212,247,204,204,246,202,202,247,215,215,253,204,204,239,145,
    145,224,182,182,240,236,236,238,239,239,238,238,238,224,153,153,249,178,178,253,216,216,254,200,200,255,195,195,255,190,190,255,
    185,185,255,171,171,255,183,183,255,213,213,250,186,186,233,133,133,222,126,126,240,183,183,228,208,208,238,239,239,252,183,183,
    254,214,214,255,199,199,255,180,180,255,169,169,255,165,165,254,156,156,251,148,148,255,156,156,251,155,155,222,122,122,204,102,
    102,195,92,92,181,98,98,215,187,187,239,240,240,255,175,175,255,195,195,255,174,174,248,150,150,233,130,130,231,128,128,225,
    123,123,211,109,109,228,126,126,219,116,116,189,86,86,175,82,82,197,146,146,222,209,209,236,237,237,238,238,238,255,168,168,
    255,184,184,240,142,142,208,106,106,180,78,78,177,74,74,166,64,64,170,67,67,201,99,99,199,96,96,164,66,66,207,172,
    172,239,241,241,239,240,240,238,238,238,238,238,238,254,172,172,255,171,171,227,124,124,185,83,83,162,70,70,161,70,70,161,
    70,70,166,74,74,199,98,98,196,93,93,171,84,84,230,222,222,239,239,239,238,238,238,238,238,238,238,238,238,240,183,183,
    236,135,135,191,88,88,185,117,117,225,211,211,229,218,218,229,218,218,229,210,210,216,128,128,178,73,73,192,136,136,238,239,
    239,238,238,238,238,238,238,238,238,238,238,238,238,236,209,209,208,112,112,197,124,124,230,222,222,239,241,241,239,239,239,239,
    239,239,239,240,240,233,182,182,191,86,86,198,148,148,240,242,242,238,238,238,238,238,238,238,238,238,238,238,238,234,224,224,
    213,125,125,238,181,181,239,240,240,238,238,238,238,238,238,238,238,238,238,239,239,240,232,232,218,149,149,185,111,111,232,222,
    222,238,239,239,238,238,238,238,238,238,238,238,238,238,237,237,228,198,198,234,192,192,239,235,235,238,238,238,238,238,238,238,
    238,238,238,238,238,238,239,239,234,225,225,210,163,163,227,206,206,238,239,239,238,238,238,238,238,238,238,238,238,238,238,238,
    238,240,240,238,240,240,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,239,239,239,241,241,238,239,
    239,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238
  };
  data=librevenge::RVNGBinaryData(wall8,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[8])));

  static unsigned char wall9[3085]= {
    80,54,10,51,50,32,51,50,10,50,53,53,10,205,135,154,165,109,132,147,116,109,143,115,116,129,117,117,121,127,126,159,
    138,142,138,182,164,103,193,204,136,220,213,173,233,238,155,237,213,97,201,199,102,189,172,101,156,162,122,138,137,122,116,127,
    125,126,128,128,142,139,145,111,126,146,119,141,157,156,153,179,179,177,188,199,200,207,215,219,208,230,219,212,243,240,221,243,
    243,225,226,226,224,226,224,206,204,204,196,161,173,195,169,168,164,143,146,143,134,123,132,120,121,121,123,123,116,115,117,119,
    126,116,109,144,143,127,162,185,145,208,208,166,221,230,151,241,224,109,211,218,105,204,205,104,180,197,124,166,160,129,138,142,
    128,125,130,149,120,141,174,144,167,143,108,126,162,138,138,207,117,156,194,183,170,198,190,177,219,214,217,234,232,232,235,241,
    241,239,239,240,233,236,233,217,227,216,198,196,196,188,204,204,170,181,180,134,146,154,123,132,134,122,131,130,120,120,119,124,
    115,111,80,136,125,88,158,154,115,188,187,110,211,226,127,217,237,117,229,234,102,222,236,102,214,221,107,200,191,150,173,182,
    151,144,166,161,137,153,173,130,157,175,112,156,206,124,166,206,127,155,201,157,154,196,173,164,195,191,194,209,206,208,225,226,
    224,232,232,231,228,227,229,224,223,224,221,221,221,197,217,214,170,198,201,125,176,183,96,162,150,125,152,153,110,132,139,105,
    130,143,78,106,131,69,132,148,91,156,159,102,199,196,103,197,196,103,213,205,100,233,217,103,217,216,132,206,205,175,200,192,
    181,182,180,169,170,167,186,151,152,198,136,159,204,146,161,206,150,156,206,167,169,198,156,173,212,154,156,205,169,164,200,190,
    191,205,205,209,219,222,217,224,224,224,218,220,220,190,215,224,148,201,208,110,206,208,104,192,170,126,164,184,78,155,160,64,
    168,152,75,124,148,103,145,140,76,140,136,91,147,160,104,167,156,96,195,201,109,213,205,126,229,229,145,206,220,188,214,215,
    210,210,212,202,203,203,202,184,187,199,149,161,202,153,162,201,152,185,206,170,173,202,161,166,209,177,162,206,178,174,204,171,
    153,201,164,177,200,178,189,212,210,214,204,218,217,184,203,208,164,205,207,118,232,232,101,234,217,107,199,203,108,185,185,131,
    176,182,74,158,153,82,150,154,94,110,131,110,131,133,103,149,153,120,157,156,143,176,175,157,189,195,168,203,215,194,214,227,
    204,218,220,227,225,224,211,210,216,199,184,198,192,178,185,204,195,180,202,169,173,205,166,172,203,157,178,204,164,153,204,167,
    153,204,147,150,205,145,160,192,165,177,179,194,191,161,170,172,116,207,192,102,214,221,105,242,231,98,222,214,127,213,226,156,
    211,213,101,185,182,108,154,183,107,146,160,132,127,121,120,131,126,149,140,144,169,155,163,175,172,169,185,185,183,202,211,211,
    219,221,222,221,220,220,223,223,222,221,223,220,208,208,208,211,179,188,201,193,189,202,155,188,205,151,161,204,150,163,203,157,
    158,206,106,153,209,115,151,172,148,156,164,169,169,147,153,153,99,161,157,99,181,201,102,222,203,130,251,219,132,230,222,127,
    214,220,106,230,228,136,209,191,114,153,177,130,143,134,120,127,120,140,119,123,144,114,146,160,133,154,177,173,167,197,176,186,
    214,211,206,224,227,227,228,227,227,230,232,233,221,230,230,215,217,215,209,212,209,207,183,188,200,170,152,203,142,165,208,102,
    157,196,120,150,167,110,153,145,138,140,146,121,136,123,121,122,127,142,141,112,158,164,109,184,196,140,215,211,116,219,209,136,
    233,238,136,228,228,153,219,209,109,186,201,145,156,159,133,129,129,151,126,120,167,109,135,169,128,143,191,113,148,201,159,161,
    195,182,173,207,197,202,222,225,224,238,239,239,235,240,240,235,232,232,229,230,230,218,220,218,196,193,197,200,167,178,191,127,
    147,159,106,139,150,114,146,161,102,104,139,120,119,116,116,127,133,128,137,124,138,149,84,159,164,91,204,186,117,202,209,133,
    229,214,139,230,241,139,214,228,121,213,227,175,182,176,158,165,170,176,125,125,189,97,113,198,104,139,204,100,156,204,150,151,
    206,149,163,196,169,175,207,202,205,230,226,229,238,235,235,233,244,241,224,237,225,229,225,231,213,209,222,196,191,193,183,179,
    161,144,130,142,145,129,130,157,103,107,122,137,107,114,114,112,110,115,122,100,117,126,80,143,140,92,172,155,138,197,196,110,
    213,205,116,216,235,180,219,242,152,218,238,160,201,206,161,168,175,155,157,159,166,108,129,206,98,148,204,132,155,204,155,147,
    205,152,176,204,158,176,196,180,165,207,210,209,224,222,222,225,234,234,236,242,242,233,238,230,184,207,245,176,209,220,191,187,
    187,164,166,166,144,133,136,133,120,131,118,121,120,112,111,111,133,125,124,119,122,118,127,132,127,81,141,155,137,174,178,106,
    191,196,99,224,223,117,248,230,154,224,241,164,214,219,167,188,185,167,161,164,196,99,155,205,122,154,204,158,151,201,154,159,
    197,173,177,212,167,178,205,170,166,194,178,187,210,210,214,225,224,222,214,228,238,191,244,247,171,242,220,174,224,234,195,200,
    205,149,170,184,155,168,166,130,133,132,113,112,121,119,113,113,125,118,118,109,119,117,94,113,125,85,137,144,122,148,145,121,
    164,162,96,207,198,116,223,226,150,234,218,173,215,228,209,216,224,196,184,181,199,136,158,205,145,153,203,154,153,204,168,173,
    203,158,168,200,162,175,205,163,171,202,156,162,192,178,191,215,214,210,193,219,230,154,241,226,152,246,244,140,232,234,142,213,
    211,121,218,205,125,177,180,138,150,146,129,128,133,101,125,137,122,120,121,98,107,112,101,112,118,122,111,127,126,128,129,118,
    146,147,128,183,185,134,207,219,143,214,240,188,208,227,199,216,220,207,213,214,204,162,174,205,151,164,202,155,162,206,162,171,
    208,150,180,201,158,164,203,155,156,205,140,149,203,156,156,188,188,188,185,207,202,156,223,219,151,231,252,148,243,209,148,229,
    213,99,244,233,120,205,207,156,172,185,128,155,162,117,160,159,90,123,152,111,119,124,106,120,111,115,123,130,114,113,114,128,
    123,128,140,158,164,157,187,195,151,206,217,171,229,215,216,217,218,219,224,222,209,194,208,196,172,163,203,165,162,197,174,173,
    201,154,174,201,162,168,204,140,158,206,114,161,189,162,172,173,166,165,180,177,177,159,201,211,142,214,247,136,227,221,160,245,
    210,143,249,244,119,239,225,101,196,208,99,159,189,147,172,180,77,152,151,102,133,126,116,114,116,116,116,114,100,104,101,124,
    111,136,134,136,142,166,165,170,188,203,196,209,222,241,227,224,229,226,226,224,219,221,221,198,195,194,212,153,162,209,152,151,
    203,161,148,204,154,153,204,128,164,206,105,167,180,120,142,148,132,122,165,162,164,154,189,180,106,215,220,112,223,248,141,227,
    212,177,239,222,144,230,227,100,206,219,103,201,200,98,195,205,110,154,162,112,151,148,115,134,134,131,123,122,110,106,114,126,
    114,121,137,132,134,161,158,163,184,184,182,209,207,214,224,225,227,234,232,232,223,234,234,217,222,222,201,175,181,204,157,168,
    205,160,170,203,129,155,205,127,152,181,114,163,159,139,137,143,111,116,141,143,143,133,164,157,145,194,189,95,219,225,122,222,
    239,165,244,205,184,246,243,139,212,227,101,201,208,115,216,202,123,180,202,151,173,176,97,149,148,131,129,128,111,107,119,153,
    104,101,148,122,117,159,131,154,184,164,165,201,196,195,222,222,223,233,232,232,240,240,240,232,232,233,219,224,221,203,198,198,
    199,148,158,210,123,151,191,102,149,156,154,147,138,143,120,138,113,115,137,132,120,116,144,144,123,161,163,99,176,189,131,226,
    213,137,237,218,170,233,239,164,240,226,137,211,247,145,212,222,98,229,205,95,193,195,138,160,160,141,137,138,122,128,124,151,
    124,113,156,112,127,168,127,129,187,132,140,195,179,177,210,204,208,234,235,234,239,238,239,238,238,238,230,227,228,219,223,222,
    195,187,178,190,162,166,169,143,150,148,128,136,150,104,123,138,114,104,114,133,125,120,131,121,110,185,139,107,145,150,126,187,
    185,129,211,218,192,223,229,162,243,216,155,245,221,152,237,227,117,236,220,105,213,202,141,187,189,132,152,164,125,145,137,147,
    131,132,158,106,130,171,125,143,195,120,146,202,157,174,198,188,189,219,219,221,233,234,233,240,239,239,232,237,236,222,223,230,
    219,220,211,185,184,188,168,166,166,142,148,146,145,111,127,123,123,117,113,129,140,114,118,120,115,136,121,99,132,151,142,163,
    172,111,188,197,191,223,220,136,226,226,148,234,220,151,239,226,157,227,225,115,230,233,125,211,219,114,176,188,151,151,160,143,
    134,147,168,145,140,182,150,150,203,111,164,206,148,158,197,161,166,208,199,200,228,224,224,235,237,237,225,246,245,224,225,233,
    207,213,225,191,210,214,182,178,179,162,165,160,133,127,132,122,116,131,113,112,119,121,119,112,101,110,126,71,113,124,103,148,
    160,91,167,172,98,197,194,96,231,225,124,248,232,174,219,226,144,249,237,149,230,228,132,217,221,110,203,212,142,195,188,163,
    162,163,175,141,132,198,132,151,204,126,162,204,158,161,205,153,173,194,176,188,214,216,214,236,235,236,238,240,241,211,250,241,
    182,233,242,183,211,229,176,208,204,175,172,187,140,159,149,114,125,137,107,104,131,124,126,107,101,117,117,79,114,101,74,155,
    127,79,146,146,107,172,177,97,189,203,133,208,215,120,223,233,126,216,236,158,240,227,143,243,251,128,215,216,144,203,204,182,
    189,185,177,149,172,204,107,154,204,151,154,204,153,159,203,161,161,199,169,160,197,189,189,219,223,217,230,224,230,208,241,213,
    172,252,227,174,226,233,143,209,209,123,198,197,122,194,162,108,152,154,111,120,148,110,106,127,95,105,115,107,110,109,91,140,
    118,109,126,131,124,187,160,102,162,178,134,192,206,125,202,196,106,221,225,134,234,219,157,251,226,137,236,234,162,210,242,216,
    217,218,187,175,176,204,136,153,204,148,151,203,148,163,208,177,169,206,160,164,201,156,157,196,186,176,221,215,216,205,224,224,
    145,255,230,178,247,227,152,217,223,102,222,227,107,200,196,120,158,164,92,141,153,93,118,158,89,109,131,110,118,109,94,114,
    116,106,112,139,128,136,136,110,137,150,178,180,179,115,166,194,105,211,216,106,226,211,156,222,235,149,239,225,179,244,224,210,
    229,220,181,201,197,199,166,167,204,143,150,204,150,191,204,154,161,204,152,167,205,152,161,194,169,178,200,180,184,194,210,213,
    171,229,213,179,250,224,132,247,246,143,214,253,98,217,238,105,202,210,114,158,168,91,156,149,105,118,139,105,126,116,106,141,
    104,106,99,111,118,116,118,131,128,127,138,155,166,105,151,176,136,184,187,107,196,204,114,228,234,140,254,230,168,252,234,193,
    226,238,217,215,221,206,184,202,201,165,152,205,157,183,204,152,180,203,155,163,203,152,158,205,138,153,198,159,167,182,196,199,
    188,210,213,181,235,217,168,225,239,171,242,243,101,236,245,164,236,221,171,192,202,140,161,171,100,140,145,106,126,122,116,126,
    121,105,132,108,124,135,124,121,118,120,123,133,136,124,124,124,131,147,148,142,187,187,130,209,213,149,223,214,141,236,233,181,
    235,237,234,223,229,208,213,200,196,175,161,204,151,170,205,152,164,198,169,162,203,135,154,204,100,152,183,150,149,180,175,171,
    165,192,195,180,210,220,194,231,232,190,240,234,140,234,239,177,227,231,141,205,239,142,195,200,96,154,165,110,147,141,114,121,
    120,110,145,109,113,125,112,117,112,115,125,133,121,123,123,121,140,144,139,159,157,161,174,176,184,177,204,212,175,220,237,185,
    248,224,220,233,223,224,225,227,206,195,205,202,153,165,204,152,150,206,126,159,203,111,156,174,108,147,149,123,140,155,144,150,
    146,174,173,152,192,195,145,211,218,173,208,236,139,240,232,161,242,230,123,225,252,138,210,211,96,179,199,117,156,180,130,141,
    140,131,135,129,114,116,115,108,109,117,117,127,122,121,127,129,120,132,122,141,140,140,167,167,163,184,187,187,195,220,213,189,
    231,223,207,224,236,215,218,232,221,213,213,202,189,171,193,168,171,176,114,146,171,117,157,162,122,131,156,118,111,136,130,145,
    125,147,148,133,163,170,115,198,185,127,214,228,131,226,227,170,244,236,162,245,216,153,218,232,100,201,220,99,172,207,121,170,
    176,148,147,141,121,122,125,113,119,131,121,144,128,132,111,121,142,121,119,147,115,123,147,140,152,171,169,171,167,194,188,187,
    219,221,190,229,250,164,222,233,183,212,215,174,207,206,180,182,181,154,149,151,138,112,130,160,127,136,146,142,127,117,118,120,
    128,125,129,116,144,141,121,163,163,130,205,201,103,207,231,192,230,246,174,237,229,154,234,241,108,208,241,137,206,212,156,179,
    197,158,164,163,148,134,147,138,119,109,117,116,132,150,116,126,155,101,110,150,107,116,141,122,138,152,141,155,174,163,170,187,
    189,188,177,205,212,159,213,231,166,234,209,145,218,205,156,191,193,149,164,171,143,139,136,135,112,121,127,129,128,120,125,129,
    122,121,125,127,126,127,120,152,157,119,180,187,93,198,202,123,210,232,177,233,241,155,243,236,149,226,245,162,220,218,157,220,
    207,175,187,189,184,170,174,200,111,153,167,111,126,175,103,133,155,101,128,152,102,105,148,115,117,135,117,123,146,138,152,168,
    167,176,151,196,197,140,214,228,150,212,242,167,216,225,152,225,208,97,192,185,121,155,166,122,131,124,117,114,124,108,117,132,
    117,132,123,126,126,123,131,138,137,108,148,160,120,183,173,132,200,211,147,215,224,157,228,239,151,240,251,165,243,242,206,231,
    216,215,211,213,193,186,185,202,157,161,206,149,152
  };
  data=librevenge::RVNGBinaryData(wall9,3085);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(32,32),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[9])));
  static unsigned char wall10[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,128,131,223,119,126,219,114,128,203,129,135,223,102,114,199,116,128,209,108,
    119,201,99,107,182,103,118,200,111,125,202,120,132,217,114,128,206,123,136,205,110,129,206,110,122,200,118,121,204,116,128,204,
    112,117,196,119,130,213,120,137,216,117,128,202,128,134,218,111,116,194,121,133,206,126,134,217,108,117,196,121,128,210,117,130,
    207,119,123,207,118,130,212,101,114,193,107,113,199,100,103,177,106,118,196,111,116,193,128,138,218,121,131,220,111,119,193,114,
    126,207,106,119,192,107,124,200,101,110,181,128,136,215,131,139,223,118,129,209,114,127,209,108,118,200,105,114,188,124,137,214,
    117,125,208,126,137,216,120,132,217,109,123,210,118,124,200,116,134,213,110,121,202,121,132,212,123,136,219,120,126,204,105,118,
    193,114,118,200,104,115,192,108,121,200,117,130,207,134,142,226,118,129,206,115,126,204,117,132,204,111,123,201,105,113,195,115,
    125,204,120,130,210,122,135,217,119,132,212,105,115,191,107,117,198,104,119,188,108,127,199,127,136,219,126,145,218,118,136,212,
    105,119,196,112,116,201,107,118,197,115,119,206,126,134,216,113,126,204,109,119,204,118,131,212,108,119,200,109,121,203,117,125,
    201,131,138,228,127,136,224,122,130,212,113,122,204,127,144,223,122,133,214,119,130,207,125,132,215,126,141,225,119,135,210,117,
    130,206,123,140,221,130,141,219,112,123,201,109,122,201,120,133,216,129,141,222,121,124,205,124,130,212,112,123,200,116,127,204,
    105,112,195,109,122,199,120,126,211,125,133,220,114,127,203,113,119,200,110,131,206,123,128,214,110,116,204,120,124,203,108,117,
    185,106,116,196,121,127,211,116,125,210,99,110,181,122,136,217,119,136,214,119,129,213,122,137,212,121,137,217,110,119,196,120,
    130,210,114,132,212,116,120,199,103,117,190,109,118,202,107,120,194,128,138,219,120,132,204,111,127,203,115,128,200,127,136,219,
    122,131,218,102,107,186,111,128,199,115,127,211,110,124,204,113,118,203,113,123,202,95,105,180,106,118,194,111,120,200,109,127,
    206,116,136,216,121,133,211,121,126,212,132,141,214,110,121,202,107,114,190,107,113,188,119,129,218,124,131,213,111,120,203,112,
    129,204,120,127,208,107,114,192,123,132,216,111,119,206,133,143,219,132,143,224,103,109,196,120,129,209,119,130,206,111,117,193,
    117,132,213,121,131,215,116,131,207,129,134,222,112,127,211,127,137,221,108,122,195,118,127,205,118,133,210,133,146,227,131,143,
    228,116,129,211,116,129,207,112,120,197,107,110,198,107,116,195,124,138,211,110,121,202,118,133,209,129,140,215,119,126,211,111,
    124,203,112,117,198,124,134,216,111,121,204,115,131,207,114,131,209,117,120,199,101,109,190,99,116,188,123,134,213,114,117,204,
    112,122,203,117,127,208,113,124,205,122,132,212,112,122,200,118,127,212,122,136,212,110,120,202,101,107,188,110,120,202,124,137,
    218,95,110,183,105,117,195,124,135,219,124,130,218,127,129,215,123,132,213,125,135,213,114,124,211,122,131,210,123,138,212,117,
    123,209,111,121,198,112,123,201,119,129,208,123,139,216,131,136,222,103,114,186,119,129,207,118,123,203,125,137,216,125,135,218,
    119,123,202,115,123,204,111,118,193,105,123,197,105,115,197,104,119,201,107,119,192,97,114,193,98,109,189,98,108,188,107,125,
    204,111,116,199,103,105,185,125,135,219,131,143,222
  };
  data=librevenge::RVNGBinaryData(wall10,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[10])));

  static unsigned char wall11[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,200,105,51,194,113,50,205,117,59,202,122,53,204,120,61,205,116,57,195,
    114,52,205,117,54,185,109,54,215,126,55,213,117,59,203,121,59,199,114,54,188,109,53,205,120,52,193,115,51,189,114,56,
    209,128,61,213,122,62,196,114,54,201,118,56,190,111,59,194,114,55,202,123,59,202,121,58,197,115,54,189,109,53,196,114,
    53,190,112,54,212,128,66,187,110,53,187,107,54,219,128,69,209,117,56,182,110,56,177,106,58,199,115,53,190,115,54,218,
    130,69,207,124,54,195,111,58,198,116,56,190,120,59,193,111,56,203,120,59,194,114,53,183,108,53,213,122,60,193,115,53,
    176,105,52,185,107,53,204,120,58,204,120,61,199,114,55,218,121,63,186,107,53,186,112,53,194,115,55,202,116,57,204,116,
    59,206,116,61,191,114,54,198,116,61,207,121,62,200,121,59,202,117,59,204,121,59,197,115,53,194,111,51,199,123,61,202,
    117,61,188,110,50,193,109,53,205,119,60,202,118,56,204,118,53,203,125,61,190,112,57,212,123,60,204,118,56,204,124,56,
    204,121,59,199,111,54,195,117,53,191,114,53,201,121,61,199,112,54,191,115,52,199,121,64,210,125,63,199,114,59,204,116,
    54,184,114,54,208,117,61,211,128,58,192,115,60,199,114,60,200,114,55,198,118,61,197,114,58,197,113,54,205,119,57,202,
    120,59,206,116,59,205,118,60,204,129,61,200,115,56,190,109,58,199,119,61,215,124,61,198,117,58,183,104,53,203,116,54,
    198,116,60,191,115,53,196,116,59,205,120,64,202,121,61,195,115,55,207,116,59,199,117,56,204,121,54,185,110,52,189,106,
    53,212,125,54,206,120,59,181,108,53,198,112,60,193,113,51,186,112,50,193,111,53,204,122,57,207,122,59,195,117,53,202,
    120,60,201,114,54,204,122,61,191,113,53,185,112,51,207,127,64,203,120,61,195,111,53,186,102,52,204,114,54,193,110,52,
    199,113,61,198,117,61,202,115,56,191,112,54,202,116,60,199,112,53,198,120,61,209,122,61,188,106,51,193,117,56,205,118,
    59,196,115,56,193,113,54,204,123,58,210,124,59,202,121,54,194,110,61,194,114,54,206,120,60,205,118,64,205,122,56,193,
    114,59,201,117,56,192,114,52,186,113,56,205,119,61,204,121,51,206,118,58,184,107,54,207,121,56,203,121,63,207,119,54,
    199,122,50,198,118,61,218,127,66,208,123,59,201,114,51,194,113,56,199,121,59,196,117,56,204,122,61,212,123,59,203,123,
    58,197,114,59,191,110,52,197,115,54,195,112,53,199,114,60,188,110,51,211,125,62,214,126,63,206,117,61,181,107,53,194,
    118,58,204,116,58,202,118,60,208,120,61,208,121,56,199,109,57,189,114,56,205,119,56,204,120,61,210,129,59,186,109,53,
    197,122,61,214,123,61,197,112,51,192,111,56,177,102,52,191,109,54,202,120,62,196,118,55,212,124,61,206,119,59,196,112,
    53,193,108,53,204,121,58,203,125,59,201,116,57,201,123,55,197,112,54,204,123,61,200,116,55,198,116,59,183,106,56,201,
    123,61,203,118,58,202,117,58,205,122,61,189,114,53,182,113,53,206,123,59,203,115,61,202,117,57,195,110,54,200,124,61,
    207,124,53,219,130,65,213,125,69,191,116,53,190,106,52,213,122,54,204,120,53,196,119,63,193,113,51,189,105,51,208,125,
    61,202,116,56,201,116,56,204,126,63,204,114,60
  };
  data=librevenge::RVNGBinaryData(wall11,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[11])));

  static unsigned char wall12[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,171,76,63,193,89,73,198,94,75,154,70,58,173,84,67,193,90,72,204,
    99,73,212,103,80,210,97,84,185,87,65,172,75,59,213,104,83,187,82,73,198,91,71,223,103,84,205,99,68,180,86,71,
    200,100,76,206,99,80,173,76,65,182,90,72,171,79,64,164,73,64,205,102,82,208,99,78,194,91,76,197,94,73,209,104,
    86,188,89,74,190,94,79,187,88,69,177,81,59,175,85,62,174,88,64,212,100,83,186,87,76,205,104,80,189,93,78,177,
    84,70,174,81,66,196,94,80,203,98,86,174,93,73,191,89,67,196,96,75,168,80,71,180,89,74,217,105,79,221,103,86,
    173,78,64,191,85,72,202,95,79,189,92,72,200,100,76,200,95,77,187,90,69,183,87,71,179,87,72,209,108,87,195,90,
    66,192,87,71,189,92,69,155,71,54,197,96,74,208,107,88,229,119,92,161,79,58,195,92,72,191,93,78,183,84,68,203,
    102,86,170,80,66,172,91,76,179,87,74,158,70,52,196,91,78,207,109,89,224,106,85,213,98,79,184,79,71,188,85,71,
    179,83,69,197,91,69,190,91,81,209,96,76,188,88,72,169,81,66,193,94,76,179,85,68,204,100,79,186,89,67,182,86,
    70,183,89,77,205,92,77,198,102,78,176,83,75,204,105,79,175,87,75,195,93,78,191,87,66,208,92,79,215,95,83,194,
    90,76,214,105,80,207,94,73,216,102,76,209,91,79,183,85,71,179,84,66,178,83,75,178,81,68,191,108,85,147,76,65,
    188,102,79,178,93,73,189,94,70,195,91,73,211,98,76,189,91,70,201,97,78,232,107,94,218,94,78,209,98,79,200,96,
    79,218,109,85,188,92,75,197,97,74,170,83,66,182,88,71,185,85,74,189,84,71,203,100,77,217,104,84,197,97,79,208,
    99,81,189,87,66,215,105,82,226,111,83,177,83,71,192,102,76,177,79,73,203,96,79,175,80,63,208,95,78,189,89,70,
    184,89,78,181,84,71,196,88,73,201,92,78,187,81,67,184,84,67,186,91,70,182,78,69,192,87,77,196,94,81,181,84,
    61,203,96,74,207,99,78,187,84,69,195,93,74,204,93,77,192,92,68,202,96,80,194,97,80,190,85,70,194,103,83,187,
    90,65,207,110,90,177,78,70,172,85,67,207,104,80,206,91,75,198,92,77,214,106,82,188,87,76,181,78,68,173,87,59,
    176,83,72,183,87,76,194,87,69,179,88,65,191,97,79,182,90,71,169,89,72,223,112,90,169,76,66,183,94,71,223,98,
    82,194,92,72,192,88,70,207,95,84,215,102,78,196,83,72,198,97,77,204,95,76,199,88,77,210,97,78,193,84,72,193,
    86,71,177,86,67,189,88,67,205,98,82,215,103,82,206,90,81,204,105,78,215,102,82,200,98,78,203,109,81,195,91,76,
    196,93,75,199,93,73,178,91,69,194,98,76,222,103,88,212,97,83,180,94,71,207,98,80,186,87,71,198,92,77,216,103,
    83,204,101,79,184,89,73,171,87,71,184,92,81,202,100,81,178,85,74,201,97,81,199,97,74,193,91,74,206,98,85,210,
    102,78,190,87,75,202,87,80,220,100,79,173,83,65,164,78,63,195,89,74,177,83,70,208,101,85,223,98,83,217,108,87,
    209,83,73,204,96,77,218,110,89,193,91,73,162,82,67,212,101,79,205,96,76,188,88,62,211,102,86,208,96,82,190,85,
    71,175,87,67,161,82,60,195,96,74,179,79,64
  };
  data=librevenge::RVNGBinaryData(wall12,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[12])));

  static unsigned char wall13[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,
    255,153,204,255,153,202,253,151,159,212,114,153,208,113,206,255,154,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,
    204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,205,255,154,194,246,145,107,162,70,115,171,84,207,255,
    156,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,
    255,153,204,255,153,206,255,155,137,190,92,99,157,68,203,254,152,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,
    204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,207,255,155,135,189,92,73,131,43,193,244,
    142,205,255,154,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,255,153,204,
    255,153,205,255,154,192,243,141,97,154,64,90,150,64,181,234,136,205,255,154,204,255,153,204,255,153,204,255,153,204,255,153,
    204,255,153,203,255,153,203,255,154,203,255,156,203,255,154,205,255,154,202,252,151,120,178,83,128,184,91,123,179,92,178,231,
    134,206,255,154,204,255,153,204,255,153,204,255,153,204,255,153,204,255,154,219,229,157,230,172,122,230,134,84,229,192,137,200,
    235,143,130,183,85,110,135,55,190,225,130,108,168,80,178,232,136,205,255,156,203,255,153,204,255,153,204,255,153,204,255,153,
    211,237,149,248,141,128,255,122,124,251,17,17,239,30,31,125,87,38,77,62,10,214,61,50,192,101,68,82,66,17,191,128,
    80,219,147,97,227,214,157,207,254,157,204,255,153,203,255,154,227,190,134,255,50,50,243,30,30,235,31,31,223,105,102,154,
    62,39,184,6,0,148,19,20,180,32,32,204,46,44,234,121,123,238,54,56,208,67,64,211,221,151,204,255,153,203,255,156,
    230,147,96,243,1,3,241,15,15,254,91,91,255,140,140,245,47,47,190,0,0,140,0,0,147,4,4,236,47,47,232,57,
    56,208,9,9,152,0,0,174,137,86,205,255,156,203,255,156,226,128,77,236,1,3,252,52,52,251,59,59,238,10,10,211,
    2,2,171,0,0,144,0,0,126,0,0,204,0,0,204,0,0,193,0,0,129,0,0,148,128,77,206,255,156,203,255,155,
    223,152,101,234,18,21,254,32,31,240,19,19,215,0,0,187,0,0,165,0,0,140,0,0,131,0,0,204,0,0,194,0,
    0,169,0,0,129,3,5,195,152,102,205,255,155,203,255,154,217,217,146,225,32,29,239,0,0,220,0,0,197,0,0,175,
    0,0,153,0,0,128,0,0,158,0,0,196,0,0,169,0,0,140,0,0,141,27,23,210,215,144,204,255,154,204,255,153,
    207,252,155,224,153,118,200,18,18,180,0,0,169,0,0,153,0,0,132,0,0,114,0,0,168,0,0,162,0,0,133,0,
    0,116,8,9,186,121,86,208,248,150,204,255,153,204,255,153,204,255,153,208,251,159,219,160,124,199,48,44,163,22,23,146,
    23,23,141,38,28,183,76,64,156,34,22,124,28,16,143,54,39,181,150,99,209,241,149,204,255,154,204,255,153,204,255,153,
    204,255,153,204,255,153,207,253,157,219,222,155,219,200,157,214,204,154,201,225,140,212,241,151,197,225,136,187,223,134,200,233,
    142,207,253,155,204,255,153,204,255,153,204,255,153
  };
  data=librevenge::RVNGBinaryData(wall13,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[13])));

  static unsigned char wall14[3085]= {
    80,54,10,51,50,32,51,50,10,50,53,53,10,89,55,81,116,93,202,152,158,144,102,124,146,83,106,199,144,112,107,50,
    71,142,49,86,157,33,131,173,67,89,155,111,55,135,113,62,131,84,94,152,66,158,161,55,82,137,55,87,87,68,118,62,
    38,136,38,102,68,62,116,133,69,104,175,76,131,56,60,134,27,31,77,63,168,69,76,50,59,17,59,78,51,75,74,135,
    22,152,41,56,154,67,101,134,57,99,104,34,105,92,181,110,70,191,157,43,166,190,117,88,158,121,99,149,50,115,179,67,
    61,142,35,71,149,57,92,154,101,94,159,127,102,136,123,18,126,145,96,129,127,114,128,26,137,166,47,115,88,92,142,26,
    98,119,50,112,28,48,110,35,50,101,103,30,117,144,89,111,76,109,115,114,78,125,97,43,75,49,37,59,199,64,48,224,
    71,104,155,96,154,83,81,156,56,66,94,192,98,101,185,84,75,173,195,62,149,192,22,22,171,114,112,159,70,130,155,140,
    66,152,48,17,163,51,51,196,97,108,69,121,108,51,160,90,99,180,32,134,192,0,162,15,62,195,10,80,98,65,17,25,
    107,62,55,130,67,72,120,68,55,88,49,57,37,61,73,83,3,58,109,87,60,114,109,66,111,38,64,115,96,67,99,143,
    75,74,195,89,124,178,134,99,178,94,100,179,92,95,127,109,73,78,109,89,97,30,85,118,130,63,124,155,25,108,181,70,
    81,185,45,62,186,94,102,136,75,60,49,63,37,80,130,87,76,94,82,147,20,11,167,18,40,163,35,35,37,71,82,47,
    114,91,49,123,117,74,157,134,66,122,120,56,73,141,57,116,24,40,125,109,47,88,142,44,79,47,68,177,22,70,197,114,
    72,144,144,102,143,159,87,77,128,61,53,110,99,110,161,25,68,83,55,71,13,55,109,35,95,113,60,92,79,73,203,53,
    89,225,65,89,193,93,126,143,104,43,103,88,58,66,100,81,113,150,141,168,50,29,181,14,16,73,46,43,36,88,73,71,
    88,89,54,100,114,52,122,105,61,142,87,63,155,94,45,168,193,38,115,174,55,124,139,49,138,86,55,144,96,68,153,65,
    59,182,146,92,142,137,73,156,103,77,97,134,69,155,95,80,94,96,110,64,84,172,92,56,39,93,53,0,70,68,28,100,
    35,106,87,65,200,56,76,161,109,104,71,151,132,55,73,44,144,93,47,132,173,20,139,22,19,48,41,32,43,53,38,58,
    60,63,66,88,97,82,116,76,81,106,50,59,157,0,46,198,129,52,84,165,48,39,85,61,121,113,75,115,77,54,146,88,
    65,165,108,100,88,201,77,152,123,76,166,87,78,147,47,82,74,134,87,54,139,108,108,90,56,113,75,54,88,26,15,118,
    41,44,88,12,85,88,55,99,155,123,75,173,157,62,167,56,128,73,0,176,93,9,167,23,22,86,37,50,56,49,28,59,
    53,65,64,95,90,93,129,86,108,118,61,69,150,93,72,155,149,48,122,159,58,132,100,66,160,100,51,166,67,60,178,12,
    80,173,64,101,196,118,85,181,153,59,156,121,74,79,81,76,43,79,79,74,70,70,116,135,58,132,127,101,109,85,101,95,
    69,92,103,29,68,98,18,41,145,82,74,175,115,90,123,65,151,64,21,193,1,36,155,22,61,63,28,64,49,25,39,74,
    57,76,70,79,48,74,95,54,72,61,78,105,7,172,74,23,208,21,22,179,32,55,137,46,99,97,66,133,50,87,64,55,
    88,144,33,76,216,34,94,176,96,80,115,103,80,56,166,60,52,112,128,36,69,189,94,91,165,131,24,128,150,76,147,104,
    76,90,82,74,86,85,68,83,85,73,104,147,126,145,52,113,186,5,15,177,44,32,43,32,68,50,33,39,47,24,40,108,
    2,104,104,43,139,80,29,112,50,0,98,111,0,194,58,3,206,7,0,204,24,8,117,73,61,39,66,118,33,74,67,25,
    86,129,22,88,35,52,99,122,93,78,74,95,79,86,171,107,63,107,86,49,48,153,9,87,187,112,122,132,156,167,175,170,
    201,156,117,149,138,142,157,157,177,166,172,154,190,186,64,48,142,32,6,96,122,31,41,30,69,86,77,44,47,75,103,64,
    3,154,104,110,175,150,43,184,150,11,184,137,17,194,140,30,221,79,14,184,151,33,121,108,40,39,51,31,39,75,54,38,
    89,160,55,85,25,49,68,35,25,69,47,159,74,157,159,105,90,61,75,80,73,57,88,71,113,117,132,141,102,132,132,118,
    142,127,119,136,122,143,183,146,150,194,211,67,133,203,99,24,170,67,55,125,107,32,43,39,20,99,87,16,86,115,113,80,
    29,201,173,44,194,223,37,211,173,18,147,177,18,127,115,17,152,45,55,176,138,38,122,154,10,97,102,45,56,102,40,45,
    56,162,35,89,109,41,100,20,38,91,20,92,79,94,147,91,45,60,72,30,80,84,151,42,79,160,67,112,67,174,95,95,
    109,102,104,85,109,159,113,98,87,63,168,61,88,147,94,102,159,98,67,172,53,41,81,95,48,69,65,21,161,71,206,187,
    87,162,186,72,114,164,100,160,148,54,172,175,88,95,77,51,151,53,9,169,78,2,130,77,0,106,102,30,70,139,48,45,
    93,80,42,95,51,47,89,86,36,84,48,74,81,131,119,87,117,71,73,50,85,66,88,92,106,105,174,78,119,185,114,152,
    146,132,142,95,106,166,67,135,155,90,141,112,144,128,69,117,86,61,23,117,41,20,159,84,16,52,55,36,87,16,160,101,
    108,118,110,104,170,179,71,160,182,44,155,102,45,212,80,40,122,32,16,170,65,79,145,122,10,108,139,28,69,112,40,54,
    114,59,42,107,66,51,85,46,53,94,68,56,81,146,171,100,73,68,86,19,37,78,34,75,152,66,103,147,156,54,122,195,
    158,140,124,139,96,115,50,109,143,47,153,140,54,117,89,59,97,37,52,144,37,0,97,41,134,71,52,217,127,36,38,143,
    11,197,132,111,186,126,153,196,187,0,220,215,32,220,108,73,140,45,55,187,52,72,137,153,10,89,153,42,88,130,58,66,
    113,45,46,91,77,51,79,51,55,102,38,34,98,109,117,113,51,94,88,123,76,94,87,66,139,26,29,173,67,107,108,97,
    83,157,97,28,99,107,38,46,134,34,94,120,36,99,48,52,86,21,88,66,31,29,82,31,8,103,13,82,84,3,211,146,
    0,189,166,42,192,123,54,215,173,0,208,127,2,197,105,21,174,196,15,164,133,1,125,185,23,100,142,33,82,116,51,68,
    100,61,61,82,42,42,70,59,44,74,6,52,108,81,120,96,39,69,86,105,97,97,135,155,186,79,6,81,46,103,121,39,
    109,188,58,56,86,90,54,59,101,45,63,74,36,46,46,61,93,33,80,157,10,126,176,10,136,73,21,40,0,20,96,40,
    9,166,79,0,244,107,9,216,95,4,219,122,0,168,92,1,147,85,4,141,37,0,129,120,39,124,145,73,78,105,32,63,
    104,60,49,78,24,47,82,44,48,80,130,38,94,112,114,92,73,88,86,46,87,80,18,58,130,52,44,115,185,88,175,147,
    60,206,121,60,187,94,39,118,127,106,125,118,148,92,94,28,86,20,105,206,26,156,163,24,120,133,15,178,93,32,105,60,
    18,38,95,4,41,86,27,237,61,6,237,125,0,194,62,11,184,72,5,135,60,83,124,156,68,109,169,108,76,111,63,60,
    85,50,52,94,75,30,84,48,50,96,124,31,83,81,101,101,84,84,80,78,78,57,17,63,22,15,45,52,29,0,147,45,
    62,178,75,37,80,103,67,57,155,102,86,165,126,129,42,106,186,16,109,196,5,164,170,18,183,119,24,242,119,20,237,116,
    14,116,163,17,69,171,20,98,140,23,89,81,7,182,15,1,165,78,3,136,113,19,102,191,36,100,166,63,72,119,45,58,
    92,41,51,78,91,56,88,92,46,86,71,51,101,72,76,97,47,72,89,45,117,87,66,78,55,50,32,112,23,5,204,64,
    45,68,49,83,60,8,81,59,86,122,92,125,119,100,88,131,131,12,156,167,36,190,132,4,236,82,11,195,132,83,106,90,
    83,34,86,56,94,113,44,67,94,66,39,43,20,84,127,0,181,93,4,140,94,0,99,149,0,104,151,19,79,174,47,51,
    84,92,50,80,171,49,98,123,41,101,76,49,98,86,66,99,45,31,88,51,25,95,91,111,92,108,70,76,54,92,211,100,
    93,103,53,62,93,58,71,64,216,124,83,180,171,122,64,169,119,83,99,174,62,226,102,6,242,58,15,231,50,55,99,71,
    84,10,144,91,28,121,61,54,162,30,45,139,8,78,161,24,129,115,16,137,43,0,113,140,15,100,153,39,73,137,18,55,
    104,143,47,105,197,40,94,119,57,95,90,51,95,64,97,96,54,46,97,75,60,64,104,77,31,99,51,61,68,77,97,166,
    88,181,64,78,94,48,106,60,59,150,62,138,75,118,75,141,120,79,149,130,88,119,42,64,81,6,3,244,77,32,149,106,
    83,100,69,65,13,57,71,33,60,27,58,73,46,40,68,11,145,34,0,159,89,4,116,176,8,97,136,0,81,102,78,64,
    96,169,49,111,128,53,110,113,52,91,68,19,71,58,98,91,37,30,91,75,85,91,115,190,28,123,93,47,89,41,89,69,
    42,51,39,70,42,27,53,74,15,88,77,55,108,129,122,7,158,98,172,87,110,120,38,103,13,11,66,59,34,62,49,58,
    104,65,47,83,55,78,62,21,50,53,15,47,19,111,57,1,193,51,9,147,85,0,126,123,0,102,110,69,84,122,65,66,
    100,104,50,85,103,51,92,95,59,93,73,60,86,37,91,106,46,27,66,106,102,42,129,132,23,121,38,21,159,93,55,165,
    89,84,136,97,89,77,48,121,45,7,113,31,76,169,27,80,166,38,124,117,66,143,156,59,189,112,70,84,36,81,0,53,
    82,40,76,88,25,135,131,75,59,109,42,48,23,142,73,5,176,133,34,163,43,72,119,43,86,88,145,212,86,127,103,69,
    116,147,37,86,104,54,88,60,113,94,50,89,98,79,102,91,52,66,45,94,135,44,127,120,61,118,92,33,175,154,64,213,
    116,109,198,146,115,136,141,125,74,101,112,55,43,110,49,29,119,14,79,154,61,99,142,52,130,163,44,127,91,71,11,79,
    67,12,90,66,7,78,101,0,99,159,51,48,108,144,76,102,201,109,117,161,97,127,144,107,137,90,115,145,81,131,155,58,
    105,177,67,82,129,113,92,86,137,84,65,97,92,89,98,32,64,86,17,111,58,29,148,82,62,148,136,106,173,123,105,181,
    37,92,168,70,85,150,127,101,94,147,92,63,112,88,6,4,95,11,25,171,104,39,142,28,38,116,14,18,144,35,52,200,
    27,3,131,57,5,41,33,10,126,55,48,60,83,188,111,115,172,115,161,151,95,145,108,91,105,108,79,74,76,104,53,63,
    95,46,108,111,73,132,129,95,102,87,118,108,96,125,107,47,99,94,92,95,80,131,122,50,106,160,102,75,157,98,94,166,
    70,98,117,58,69,100,53,165,66,69,146,35,118,101,0,82,95,35,55,115,40,32,145,44,26,93,52,87,108,14,30,139,
    12,107,153,30,87,120,52,25,24,61,34,79,60,133,118,92,156,80,132,94,61,138,65,75,133,144,138,111,103,106,59,64,
    89,0,125,106,0,102,125,31,55,114,94,96,78,83,87,43,45,45,129,81,40,125,80,74,137,115,99,83,106,90,82,133,
    46,107,67,53,165,61,47,134,46,42,138,23,41,124,36,93,80,38,57,129,79,14,152,21,6,133,45,106,86,73,49,97,
    21,228,98,41,212,140,33,131,98,45,65,172,38,82,170,78,63,73,71,50,79,54,82,109,108,143,103,108,90,94,97,46,
    79,57,125,99,30,191,111,13,173,217,79,70,119,73,25,147,40,58,90,72,88,106,95,18,165,36,41,162,18,38,189,35,
    29,168,33,53,171,54,59,130,50,40,88,44,42,96,85,44,89,84,81,120,114,145,83,51,69,103,31,26,107,44,201,119,
    7,217,105,15,212,108,16,154,181,36,16,138,49,66,99,51,98,144,73,37,82,2,39,127,23,113,129,35,40,131,47,93,
    115,64,127,72,86,87,103,86,129,65,74,143,99,40,131,212,83,174,171,105,85,170,88,84,130,72,91,92,48,91,165,74,
    30,155,45,64,140,65,0,109,59,58,122,62,40,140,63,29,92,132,93,110,152,57,111,111,2,161,97,134,101,126,122,86,
    28,211,132,7,81,158,36,48,164,53,42,176,42,86,168,74,24,208,56,42,173,53,58,163,10,58,135,38,79,86,134,152,
    170,176,160,141,94,95,121,54,72,156,63,89,187,35,226,203,115,161,135,116,126,132,108,111,102,87,123,126,71,98,202,40,
    88,203,82,42,171,66,51,139,33,71,139,30,70,134,46,62,50,126,57,139,143,23,181,10,33,159,98,154,87,46,110,85,
    34,156,81,34,48,47,74,120,117,106,73,186,41,119,222,79,41,118,52,4,185,120,64,211,63,109,196,78,72,126,108,138,
    203,41,55,179,125,149,107,57,124,199,84,190,222,71,143,164,97,112,164,152,117,71,116,123,61,64,100,166,60,100,145,84,
    148,108,27,123,166,60,107,107,134,136,138,113,118,117,48,48,87,83,76,111,111,195,142,11,179,129,21,102,112,6,207,128,
    10,152,143,10,90,180,48,13,246,59,44,129,31,62,66,62,66,70,72,79,119,67,40,190,58,91,237,17,59,192,20,21,
    221,29,42,151,45,32,115,45,43,110,63,159,157,62,59,73,28,143,138,69,157,188,72,75,150,81,114,98,132,120,79,126,
    69,68,63,82,97,93,85,101,54,66,140,44,95,131,44,52,138,26,109,125,40,109,107,35,105,137,35,80,123,53,73,120,
    18,120,123,43,95,112,103,70,155,74,17,97,40,0,45,42,146,141,47,135,190,39,112,108,75,63,72,36,60,216,40,29,
    183,69,46,24,118,78,87,88,58,46,61,59,45
  };
  data=librevenge::RVNGBinaryData(wall14,3085);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(32,32),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[14])));
  static unsigned char wall15[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,202,182,208,223,200,218,222,195,220,229,195,223,226,199,226,220,199,220,208,
    177,208,212,194,212,220,191,211,240,215,235,201,172,200,209,184,207,219,197,219,227,204,229,217,188,215,240,213,222,209,183,206,
    226,202,224,211,191,213,209,181,209,208,190,208,216,187,216,220,194,219,216,198,213,209,188,209,213,196,213,206,182,204,218,202,
    212,214,190,213,221,194,219,210,187,209,227,207,221,222,199,213,208,188,208,213,198,212,216,192,213,220,193,220,225,196,217,208,
    193,204,208,185,200,226,207,225,205,183,205,221,199,221,211,188,210,221,197,219,217,190,210,220,201,215,205,179,197,225,200,225,
    207,185,206,214,190,210,213,188,208,220,197,220,212,194,211,216,195,217,209,185,210,221,194,219,211,187,203,224,198,218,218,193,
    218,199,188,202,217,200,215,219,193,216,219,202,218,213,188,211,214,192,212,225,203,218,218,195,210,209,189,206,203,185,209,216,
    189,203,217,199,214,215,192,216,225,200,223,206,192,208,216,194,215,217,207,217,205,179,205,218,196,211,211,187,208,210,183,201,
    225,202,227,215,187,212,218,194,215,214,192,214,214,197,213,224,196,220,224,198,224,219,193,216,215,194,214,218,189,211,216,189,
    206,206,183,203,217,191,216,220,194,218,220,196,220,214,193,214,219,194,217,217,195,212,223,202,223,212,190,210,220,191,217,215,
    188,207,215,191,215,212,188,204,219,191,214,214,185,211,223,197,222,221,198,220,201,188,198,221,197,213,226,197,227,220,195,218,
    217,191,216,217,194,218,211,183,211,220,193,212,211,192,209,213,193,211,212,184,213,212,192,210,225,202,224,223,194,223,217,196,
    210,204,175,201,217,201,217,208,182,207,213,197,210,215,190,209,216,188,210,224,199,217,211,190,210,224,204,224,206,183,206,221,
    195,213,220,199,218,219,194,212,215,193,215,205,192,205,211,189,211,222,200,222,216,192,216,220,193,217,227,198,219,212,189,212,
    221,196,220,210,185,209,223,201,222,211,191,208,219,193,220,218,196,218,209,184,207,222,201,221,214,192,214,223,199,223,206,187,
    201,222,196,219,220,200,217,215,188,216,214,196,215,200,184,207,217,192,211,209,189,209,214,188,208,214,189,209,218,192,217,216,
    194,216,221,196,219,212,191,208,207,186,206,209,187,206,214,189,213,217,200,216,218,191,217,217,190,213,231,212,229,214,193,214,
    210,190,209,222,199,214,224,203,221,210,185,204,213,193,210,214,193,212,209,187,208,219,194,217,217,193,207,210,187,203,224,196,
    225,219,187,211,217,195,214,208,186,203,210,187,209,217,191,208,230,201,227,205,183,205,212,185,212,214,193,215,220,205,220,212,
    186,202,218,193,215,207,194,209,217,198,216,210,194,210,209,186,209,215,198,213,217,193,214,222,196,217,220,198,222,226,201,226,
    209,186,208,214,189,209,230,202,222,214,184,214,211,186,208,220,195,213,213,192,213,217,201,217,210,193,210,213,190,213,220,198,
    212,218,195,216,212,194,213,214,191,209,220,194,218,208,184,208,222,207,219,211,188,202,223,200,218,214,198,211,216,193,217,214,
    192,209,216,193,209,213,194,209,217,194,216,208,184,208,220,197,218,212,186,209,218,193,210,222,198,223,217,190,217,228,200,228,
    222,193,219,217,192,215,208,182,201,222,196,221,203,181,203,217,198,215,226,206,223,229,200,231,210,184,207,226,204,212,220,190,
    211,207,181,207,216,187,210,223,201,215,208,193,208
  };
  data=librevenge::RVNGBinaryData(wall15,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[15])));

  static unsigned char wall16[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,214,214,211,225,225,216,218,218,215,221,224,218,231,224,221,229,229,227,226,
    226,225,227,228,225,213,204,204,239,236,236,226,227,218,232,232,226,213,213,204,204,205,205,231,231,228,231,231,230,194,194,193,
    220,220,210,232,232,221,222,216,219,228,224,224,222,222,214,222,222,220,231,231,223,228,226,226,228,227,223,230,230,225,219,219,
    218,213,213,211,220,212,213,226,223,216,219,219,217,220,220,221,220,220,215,214,214,213,224,224,222,236,236,233,231,231,224,226,
    226,218,228,229,224,226,227,217,227,227,217,226,226,219,226,226,224,225,225,224,221,219,218,232,226,224,236,234,234,229,229,220,
    229,229,219,231,228,227,230,230,222,217,217,214,226,226,216,222,223,218,227,219,218,219,217,215,227,227,227,236,234,229,232,232,
    223,222,222,218,228,230,220,219,225,217,234,225,222,208,208,205,210,210,207,232,224,217,232,230,228,210,210,210,224,224,222,229,
    229,221,230,227,224,233,232,230,228,225,217,227,224,219,235,230,227,225,225,217,212,210,208,231,224,221,233,230,222,218,218,209,
    227,227,227,237,238,233,230,230,220,220,220,212,197,197,195,217,217,215,217,210,207,234,231,221,222,223,219,232,234,231,237,235,
    234,222,219,217,214,212,213,215,213,213,228,228,220,224,224,221,219,216,217,213,205,204,223,220,220,215,214,210,227,227,219,226,
    226,219,237,234,232,241,240,233,219,216,214,219,216,208,224,216,214,230,223,221,219,217,209,229,229,230,229,229,226,195,192,192,
    207,199,198,235,233,231,236,229,229,234,229,229,237,229,230,230,231,228,213,213,210,219,219,212,222,214,214,232,229,226,224,222,
    221,229,229,221,222,219,215,217,216,214,223,223,215,230,229,227,235,233,223,233,233,225,234,234,227,221,220,218,230,228,220,227,
    225,223,221,213,212,220,213,210,208,209,209,204,209,201,224,226,224,222,222,219,233,225,222,229,227,225,208,208,203,218,218,214,
    226,226,223,224,224,225,229,229,224,235,234,225,224,222,219,212,213,210,210,215,207,220,215,212,233,225,223,230,225,223,232,230,
    227,231,231,221,229,229,222,218,218,214,216,216,206,243,243,235,229,227,224,223,218,216,211,211,211,219,216,214,230,222,220,232,
    233,225,228,225,223,213,211,208,227,227,218,237,235,235,230,230,220,224,224,220,224,224,223,224,224,216,226,226,224,206,200,198,
    216,215,212,218,220,213,225,225,222,232,232,225,221,221,222,215,210,208,219,211,210,213,217,209,237,237,230,242,237,234,228,228,
    223,233,233,230,237,237,221,222,222,220,214,214,214,214,220,210,218,218,213,217,209,210,208,208,208,223,223,221,234,234,227,236,
    241,231,229,230,228,231,230,227,222,220,217,233,234,234,227,227,220,226,226,222,222,222,220,220,218,218,228,227,227,220,220,218,
    217,217,210,209,209,209,221,221,219,220,220,213,225,225,220,219,219,217,219,217,206,227,226,216,232,231,224,237,234,235,231,231,
    226,229,229,221,225,226,226,219,211,210,229,226,223,229,226,226,212,212,212,200,200,197,202,202,194,211,211,209,238,230,229,234,
    232,231,239,233,229,237,237,234,234,233,230,229,226,219,233,233,225,226,226,224,232,232,225,225,225,212,227,227,216,220,220,210,
    211,214,208,229,228,220,228,228,228,225,225,225,232,229,229,231,228,228,234,225,226,230,230,227,232,230,223,231,226,231,239,240,
    237,223,220,220,225,216,213,206,206,202,196,196,196
  };
  data=librevenge::RVNGBinaryData(wall16,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[16])));

  static unsigned char wall17[781]= {
    80,54,10,49,54,32,49,54,10,50,53,53,10,80,99,182,88,88,184,73,86,155,82,87,174,62,65,125,61,69,128,55,
    54,123,52,59,120,55,62,123,70,79,154,81,83,160,74,85,165,76,87,175,77,81,163,65,77,151,62,62,135,67,70,146,
    66,74,146,66,71,147,58,61,126,50,56,119,71,76,159,72,82,153,66,73,144,67,79,146,84,89,183,58,73,132,69,76,
    160,73,82,151,68,75,153,64,69,145,76,79,153,62,71,143,74,78,162,65,70,124,61,70,141,75,76,153,72,75,157,61,
    64,129,79,84,169,67,77,150,63,66,135,64,65,131,71,76,157,75,77,158,73,81,158,66,70,147,57,62,134,75,88,168,
    64,70,154,72,74,154,65,73,142,70,76,150,70,75,154,69,76,142,77,83,167,58,59,121,60,76,148,76,82,163,85,94,
    174,61,70,131,62,70,140,63,71,134,67,75,154,69,69,141,72,80,153,66,74,148,53,58,121,73,75,149,73,79,163,71,
    75,154,50,61,115,74,75,153,66,73,144,63,73,147,59,59,127,53,59,123,82,89,177,84,96,189,83,91,176,76,83,168,
    70,74,156,73,77,156,63,71,141,81,91,181,64,72,138,59,63,129,60,67,138,67,74,147,70,72,142,65,72,141,63,72,
    143,73,87,160,75,82,167,69,74,154,68,62,133,84,96,191,65,74,145,60,67,125,70,75,147,61,65,139,55,64,127,71,
    78,153,72,77,148,67,73,150,83,90,173,77,80,164,64,75,140,70,74,144,66,69,140,62,77,144,70,73,158,65,71,138,
    66,70,136,57,64,131,69,77,148,64,69,135,71,79,155,63,70,151,73,83,160,73,84,161,72,81,163,68,82,157,56,60,
    128,70,78,154,69,69,142,68,74,145,77,83,160,71,77,158,81,87,171,53,57,123,74,83,159,70,77,152,68,70,148,67,
    76,151,70,77,154,74,82,149,59,67,132,55,55,120,46,57,114,71,80,151,75,78,160,75,87,164,57,66,133,63,59,131,
    52,62,125,73,80,155,77,82,165,58,59,124,73,82,169,67,70,138,63,65,134,72,77,152,63,71,134,68,64,138,68,78,
    156,87,91,179,72,85,167,79,85,162,62,64,132,54,53,117,66,64,130,77,78,157,56,66,123,61,77,144,74,76,162,73,
    77,152,60,75,145,73,78,162,71,81,157,61,70,141,87,92,181,76,83,165,64,70,139,52,63,121,42,49,94,66,84,160,
    67,75,152,67,69,140,68,71,143,77,84,153,74,84,156,73,82,167,66,71,137,70,68,152,66,74,149,67,71,145,84,92,
    179,62,66,129,58,63,125,71,82,152,62,65,132,72,77,156,60,70,129,78,85,169,68,79,153,79,84,171,66,68,134,56,
    65,122,61,72,136,69,75,152,74,81,154,66,76,150,60,62,131,70,79,150,70,75,157,67,69,139,55,63,125,51,52,118,
    72,81,156,66,71,145,83,89,167,75,83,165,61,62,124,62,65,129,65,78,154,64,68,140,63,71,139,61,63,127,65,74,
    145,71,73,151,68,76,154,68,68,140,78,84,168,60,68,121,71,75,158,75,93,178,64,68,149,53,56,106,60,70,139,67,
    70,145,69,74,137,58,66,133,66,68,146,92,100,190,74,78,152,61,63,131,82,89,172,51,62,116,63,69,138,78,81,160,
    78,83,165,71,82,159,46,48,97,45,57,111,73,82,152,94,91,181,77,81,161,76,87,180,63,72,143,77,85,162,65,66,
    132,66,76,159,54,59,115,45,59,111,69,80,167
  };
  data=librevenge::RVNGBinaryData(wall17,781);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(16,16),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[17])));

  static unsigned char wall18[3085]= {
    80,54,10,51,50,32,51,50,10,50,53,53,10,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,236,236,238,
    241,241,239,243,243,238,237,237,237,235,235,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,238,237,236,237,237,236,238,237,237,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,239,239,239,249,249,237,
    224,224,234,220,220,238,244,244,243,252,252,241,240,240,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,238,238,245,239,242,246,240,242,247,239,243,241,240,241,236,237,
    237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,239,239,239,234,234,234,171,171,241,
    167,167,247,155,155,239,134,134,206,166,166,221,227,227,244,243,243,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,237,237,237,243,240,241,197,232,217,177,223,204,180,228,204,226,233,228,245,241,
    244,237,238,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,237,237,245,245,242,185,185,247,183,183,255,
    235,235,232,107,107,252,141,141,205,60,60,132,122,122,220,222,222,241,240,240,237,237,237,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,236,237,237,249,246,247,155,225,189,190,229,216,164,246,218,107,245,189,69,189,131,186,207,
    195,248,243,245,236,237,236,238,238,238,238,238,238,238,238,238,238,237,237,239,243,243,232,217,217,235,140,140,244,150,150,228,
    119,119,230,78,78,237,120,120,242,88,88,144,67,67,179,190,190,242,239,239,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,236,237,237,245,241,243,189,217,203,174,241,224,215,253,249,126,221,181,89,195,145,118,242,193,44,140,
    98,212,206,212,244,244,243,237,237,237,238,238,238,238,238,238,238,238,238,238,239,239,239,238,238,254,133,133,244,122,122,236,
    109,108,248,116,116,242,119,119,232,77,77,147,76,76,181,196,196,241,237,237,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,235,237,236,250,240,245,169,222,197,81,242,173,87,184,139,83,209,151,68,229,150,94,236,161,30,173,
    116,148,146,151,243,242,241,237,238,238,238,238,238,238,238,238,238,236,236,238,247,247,236,185,185,241,91,91,242,124,125,244,
    100,100,219,71,70,214,69,69,210,51,51,183,61,61,180,181,181,242,242,242,238,237,237,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,235,238,237,252,236,244,143,246,199,113,247,183,138,249,193,89,220,156,96,229,163,81,191,145,29,134,
    106,132,139,143,228,226,225,239,240,240,237,237,237,238,238,238,237,235,235,245,251,251,195,157,157,173,53,53,183,27,25,175,
    28,25,162,13,17,178,28,29,181,34,34,130,48,48,156,151,151,239,240,240,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,237,238,237,241,240,240,229,230,230,61,171,115,48,179,120,39,194,130,24,170,117,26,163,115,28,175,123,19,155,
    112,75,116,118,229,222,222,240,242,242,237,237,237,238,238,238,237,237,237,242,240,240,217,227,227,151,152,151,117,60,17,128,
    72,24,110,54,27,56,1,0,66,28,29,162,174,174,188,189,189,221,220,220,241,241,241,237,237,237,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,236,236,236,165,171,170,76,118,89,53,76,46,49,77,57,32,59,55,25,60,52,62,78,
    78,139,141,141,194,194,194,241,241,241,237,237,237,238,238,238,238,238,238,237,238,238,240,238,238,245,245,246,164,129,92,166,
    114,58,155,107,63,77,24,1,84,63,62,196,199,199,240,239,239,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,237,236,252,255,255,198,162,139,148,80,22,174,111,60,119,53,26,52,0,0,123,114,
    114,195,196,195,220,220,220,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,236,235,234,247,251,255,193,170,148,139,
    84,28,108,57,18,67,17,2,116,111,113,191,192,191,238,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,237,237,237,240,241,242,226,224,221,142,95,48,158,110,60,93,42,11,70,36,23,142,145,
    147,216,216,215,241,241,241,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,237,239,240,241,233,228,224,109,
    59,15,85,31,0,74,38,25,144,145,148,216,215,215,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,236,236,236,245,246,246,162,125,91,103,50,3,79,25,0,84,63,62,160,165,
    165,221,220,220,241,241,241,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,236,236,236,247,248,248,156,
    120,89,60,1,0,92,68,66,177,182,183,223,222,222,241,241,241,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,236,235,234,246,250,255,196,174,151,82,27,0,75,27,6,109,105,104,186,187,
    187,231,231,231,239,239,239,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,236,235,234,246,250,255,199,
    177,152,69,13,0,92,78,82,191,194,193,238,237,237,238,238,238,238,238,238,238,238,238,238,237,236,238,237,235,238,237,235,
    238,238,237,238,238,238,238,238,238,238,238,238,238,237,237,239,240,241,235,230,225,104,54,14,56,10,2,138,136,138,203,203,
    203,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,237,238,239,241,239,
    234,225,85,44,21,129,128,131,204,205,204,237,237,237,238,238,238,238,238,237,238,238,237,238,242,249,238,242,251,238,243,251,
    239,241,242,237,236,235,238,238,238,238,238,238,238,238,238,236,236,236,248,250,250,148,107,79,61,36,32,150,154,155,222,222,
    221,240,240,240,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,237,237,242,
    243,243,200,197,196,155,154,154,221,221,221,241,241,241,237,237,237,238,237,237,238,241,245,238,217,179,235,204,146,237,205,157,
    235,228,221,242,246,248,239,238,238,238,238,238,238,238,238,236,236,235,246,249,251,199,186,177,105,90,86,173,176,177,223,223,
    222,240,240,240,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,
    237,237,246,246,247,229,229,229,227,227,227,240,240,240,237,236,236,242,247,249,237,189,123,242,216,161,251,218,130,250,188,75,
    224,133,35,207,182,157,234,240,245,239,238,236,238,238,238,238,238,238,236,236,236,246,248,248,218,220,221,198,197,197,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,236,236,236,240,240,240,240,240,240,237,237,235,239,243,248,230,203,172,248,224,123,254,249,182,238,181,85,224,145,45,
    253,195,87,171,85,22,196,200,203,246,246,245,237,236,236,238,238,238,238,238,238,236,236,235,241,240,240,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,237,237,237,238,238,238,238,236,234,238,245,254,234,197,144,249,173,61,219,138,32,232,151,47,241,149,48,
    249,165,57,202,101,24,144,138,139,232,234,234,239,239,239,237,237,237,238,238,238,238,238,238,237,237,237,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,237,235,237,244,251,247,197,144,251,183,74,252,197,96,236,155,55,241,163,64,
    216,133,42,197,102,20,150,141,137,221,224,225,241,240,240,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,237,239,240,242,234,230,226,211,121,31,222,117,12,225,118,19,209,99,10,206,97,7,
    211,103,15,203,94,4,144,99,76,218,225,229,243,241,240,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,239,239,239,235,235,235,143,137,129,137,74,26,133,53,0,125,50,6,107,33,4,
    111,30,0,102,62,48,127,129,130,196,196,195,241,241,241,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,237,237,237,240,239,238,224,230,237,164,146,126,139,88,34,160,115,67,106,57,33,
    40,0,2,105,106,108,179,180,180,213,213,213,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,237,237,245,246,246,233,227,222,144,93,46,161,109,59,96,41,9,
    74,36,22,140,140,142,203,203,202,238,238,238,239,239,239,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,235,235,235,244,245,246,162,125,91,103,50,3,79,25,0,
    84,63,61,162,166,166,220,219,219,240,240,240,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,236,235,234,246,250,255,196,174,151,82,27,0,75,27,6,
    109,105,104,186,187,187,231,231,231,239,239,239,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,237,239,240,241,235,230,225,104,54,14,56,10,2,
    138,136,138,203,203,203,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,236,236,236,248,250,250,148,107,79,61,36,32,
    150,154,155,222,222,221,240,240,240,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,236,236,235,246,249,251,199,186,177,105,90,86,
    173,176,177,223,223,222,240,240,240,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,236,236,236,246,248,248,218,220,221,
    197,197,197,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,236,236,236,241,240,240,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,237,237,237,
    238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
    238,238,238,238,238,238,238,238,238,238,238,238,238
  };
  data=librevenge::RVNGBinaryData(wall18,3085);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(32,32),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[18])));
  static unsigned char wall19[3085]= {
    80,54,10,51,50,32,51,50,10,50,53,53,10,0,1,76,0,0,31,0,0,0,0,0,1,1,1,0,0,4,0,22,
    0,0,77,0,0,104,0,0,87,1,0,49,0,0,7,20,0,1,87,0,0,50,2,6,21,0,16,0,48,1,0,174,
    0,0,172,0,0,162,0,0,143,0,0,144,0,0,164,0,0,181,4,0,132,1,8,65,0,7,7,0,6,0,0,1,
    0,0,2,0,0,7,0,0,4,0,0,4,15,0,6,19,0,0,16,0,0,1,0,0,0,1,0,0,0,0,0,33,
    0,0,84,0,0,96,0,0,85,1,0,37,0,0,14,38,0,0,70,0,2,49,3,33,6,0,13,0,93,0,0,177,
    0,0,164,0,0,159,0,0,150,0,0,160,0,0,169,0,0,168,1,20,49,0,46,0,0,38,2,0,55,0,0,62,
    0,0,66,0,0,56,0,0,33,0,0,18,0,0,23,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,31,
    0,10,81,0,3,96,0,0,67,0,0,33,3,0,4,64,0,0,49,1,3,59,0,31,6,10,4,0,152,0,0,149,
    0,0,154,0,0,151,0,0,150,0,1,172,3,0,167,7,18,18,0,55,0,0,66,3,0,91,0,0,100,0,1,112,
    0,0,111,0,0,100,0,0,91,0,0,52,0,0,63,1,0,14,0,0,0,0,0,1,0,0,0,1,2,0,0,41,
    0,28,78,0,19,99,0,0,51,0,1,28,7,0,3,73,0,0,35,3,11,35,0,18,2,79,0,0,184,0,0,158,
    0,0,139,0,0,136,0,0,161,0,1,175,7,0,57,1,42,0,0,75,3,0,92,0,0,89,0,0,39,0,0,8,
    0,0,8,0,0,38,0,0,95,0,0,100,0,0,99,0,0,22,0,0,0,0,0,1,0,0,0,0,0,0,4,41,
    0,51,85,0,18,89,1,0,60,0,1,15,40,0,0,88,0,0,40,3,21,16,0,21,0,86,0,1,187,1,0,150,
    0,0,127,0,0,138,0,0,160,2,0,162,15,0,9,1,47,0,0,84,1,0,65,0,0,0,0,7,0,0,12,0,
    0,6,1,0,1,0,0,0,14,0,0,116,0,0,85,0,0,14,0,0,1,0,0,0,1,0,0,0,0,0,30,34,
    0,53,89,0,2,91,2,0,58,0,0,13,44,0,0,89,0,0,37,3,12,17,0,15,0,87,0,1,163,0,0,131,
    0,0,130,0,0,125,0,0,145,8,0,82,7,7,0,0,29,2,0,82,0,0,26,0,6,0,0,37,3,0,43,0,
    0,28,1,0,8,3,0,0,0,0,0,62,0,0,80,0,0,13,0,0,0,0,0,0,0,0,0,0,0,0,42,31,
    0,56,83,0,22,91,0,0,54,0,0,27,9,0,2,78,0,0,40,3,10,14,0,9,0,75,0,1,153,0,0,138,
    0,0,131,0,0,116,0,0,137,3,0,70,7,7,0,0,16,3,1,59,0,0,5,0,18,0,0,47,0,0,52,0,
    0,37,0,0,15,2,0,1,0,0,0,50,0,0,78,0,0,6,0,0,0,1,0,0,0,0,0,17,0,0,65,34,
    0,53,83,0,25,100,1,0,59,0,0,15,37,0,0,88,0,0,19,2,10,4,0,10,0,60,0,0,135,0,0,131,
    0,0,134,0,0,131,0,0,149,4,0,70,2,0,0,0,16,3,0,67,0,0,16,0,10,0,0,30,1,0,32,0,
    0,24,0,0,10,1,0,0,0,0,0,38,0,0,45,0,0,0,0,0,1,1,0,0,0,0,0,47,1,0,75,34,
    0,57,81,0,15,99,2,0,65,0,1,17,45,0,0,85,0,0,11,2,10,1,0,10,0,63,0,0,131,0,0,119,
    0,0,132,0,0,132,0,0,159,0,0,72,0,2,0,0,12,3,0,54,0,0,42,0,0,0,0,5,2,0,16,0,
    0,9,0,0,0,1,0,0,0,0,0,57,0,0,10,0,3,0,1,1,0,0,0,0,18,0,0,70,3,0,69,47,
    0,51,91,0,7,119,0,0,74,0,0,31,11,0,2,66,0,0,8,3,10,0,0,10,0,70,0,0,133,0,0,118,
    0,0,124,0,0,135,0,0,141,0,0,120,0,0,5,0,1,0,0,7,0,0,49,0,0,28,0,0,0,0,2,0,
    0,2,0,0,0,0,0,0,50,0,0,67,0,1,0,0,15,0,1,2,0,0,0,0,51,0,0,76,11,0,60,52,
    0,46,120,0,3,119,0,0,60,0,0,30,5,0,3,63,0,0,8,1,3,0,0,15,1,18,1,0,118,0,0,119,
    0,0,118,0,0,125,0,0,122,0,0,134,0,0,51,0,0,0,0,0,3,0,9,0,0,59,0,0,54,0,0,62,
    0,0,66,0,0,69,0,0,31,0,0,2,0,11,0,1,7,0,0,0,0,36,0,0,74,0,7,66,3,2,55,40,
    0,25,104,0,14,107,0,0,71,0,0,37,0,0,6,61,0,0,31,0,0,0,0,14,4,1,4,0,99,0,0,123,
    0,0,118,0,0,118,0,0,117,0,0,120,0,0,111,0,0,20,0,1,0,0,0,2,0,11,1,0,23,1,0,27,
    1,0,29,0,0,25,0,0,1,0,6,0,0,4,0,1,0,0,3,0,0,75,0,0,76,1,12,53,0,4,47,18,
    0,38,85,0,3,118,0,0,90,1,0,43,0,0,21,31,0,0,65,0,0,16,2,8,13,0,12,0,49,0,0,134,
    0,0,116,0,0,119,0,0,120,0,0,118,0,0,120,0,0,109,0,0,26,0,0,1,0,1,0,0,2,0,0,8,
    0,0,1,0,0,0,0,10,1,0,14,0,0,0,0,0,0,0,49,0,1,103,0,0,73,0,18,63,0,22,55,10,
    0,27,69,1,0,109,0,0,112,0,0,64,0,0,30,0,0,7,74,0,0,42,0,1,12,0,15,6,3,4,0,114,
    0,0,132,0,0,116,0,0,120,0,0,106,0,0,98,0,0,97,0,0,92,0,0,58,0,0,24,0,0,19,0,0,
    14,0,0,7,0,1,5,4,0,5,0,0,1,0,0,28,0,0,143,0,1,101,0,0,71,0,19,57,0,56,33,13,
    4,12,65,0,0,103,0,0,106,0,0,72,1,0,32,0,0,11,35,0,0,75,0,0,16,1,3,14,0,7,1,16,
    0,0,122,0,0,129,0,0,114,0,0,106,0,0,88,0,0,82,0,0,71,0,0,83,0,0,75,0,0,73,0,0,
    65,0,0,45,0,0,28,0,0,9,0,0,0,0,0,98,0,0,168,0,0,98,0,0,68,1,9,24,0,64,5,26,
    14,7,59,0,0,87,1,0,75,0,0,70,0,0,48,0,0,9,1,0,4,70,0,0,53,0,0,20,1,0,6,0,
    0,0,17,0,1,117,0,0,131,0,0,118,0,0,102,0,0,84,0,0,84,0,0,73,0,0,66,0,0,67,0,0,
    64,0,0,40,0,0,29,0,0,5,0,0,16,0,0,137,0,0,155,0,1,119,0,0,47,0,13,7,0,73,0,9,
    64,5,31,0,1,42,1,0,47,0,0,67,0,0,71,0,0,26,0,0,0,10,0,3,77,0,0,66,0,0,30,2,
    0,4,0,0,0,18,0,0,99,0,0,126,0,0,117,0,0,101,0,0,86,0,0,78,0,0,71,0,0,67,0,0,
    63,0,0,46,0,0,31,0,0,19,0,0,92,0,0,143,0,0,149,0,0,113,0,5,21,0,22,1,0,60,1,0,
    111,0,12,38,0,7,0,0,21,1,0,50,0,0,64,0,0,44,1,0,11,0,0,0,13,0,1,104,0,0,80,0,
    0,35,1,0,6,2,0,0,0,0,1,65,0,1,134,0,0,129,0,0,119,0,0,107,0,0,86,0,0,76,0,0,
    72,0,0,66,0,0,50,0,0,81,0,0,132,4,1,143,0,1,81,0,0,10,0,5,0,0,17,1,0,48,0,0,
    57,0,2,74,0,3,13,0,5,0,0,26,1,0,53,0,0,58,0,0,36,1,0,5,0,0,0,20,0,1,103,0,
    0,83,0,0,27,1,0,6,1,3,2,1,2,0,41,2,0,102,3,0,106,0,1,132,0,0,127,0,0,117,0,0,
    99,0,0,93,0,0,95,0,0,136,0,1,145,10,0,19,0,0,5,0,2,0,0,12,1,0,28,0,0,41,0,0,
    21,0,0,52,0,1,50,0,0,0,0,10,1,0,23,0,0,56,0,0,53,0,0,51,2,0,7,0,0,1,47,0,
    3,122,0,0,57,0,0,21,0,3,37,1,7,22,0,12,7,0,14,4,1,12,0,43,5,0,105,4,0,102,4,0,
    97,5,0,95,12,0,43,13,0,44,13,0,42,0,21,1,0,8,1,0,8,1,0,28,0,0,23,0,0,10,0,0,
    0,0,0,19,0,0,71,0,0,16,0,0,0,0,9,1,0,42,0,0,65,0,0,70,0,0,61,2,0,18,0,0,
    12,21,0,7,87,0,0,96,0,0,63,0,1,40,2,0,34,1,3,23,1,6,21,0,10,15,0,15,5,0,14,6,
    0,15,4,0,15,5,0,7,18,0,6,22,0,0,55,1,0,55,0,0,27,0,0,22,0,0,5,0,0,0,0,0,
    3,0,0,8,0,0,39,0,0,68,0,0,8,0,0,0,0,13,0,0,42,0,0,67,0,0,70,0,0,61,1,0,
    47,0,0,32,2,0,22,35,0,7,65,0,0,70,0,0,58,0,0,47,0,0,39,2,0,31,1,0,17,1,0,14,
    1,0,8,1,0,13,2,0,27,2,0,37,2,0,30,0,0,24,0,0,2,0,0,0,0,0,0,0,0,1,1,0,
    0,2,7,5,0,1,26,0,0,63,0,0,76,0,0,9,0,0,0,0,20,1,0,69,0,0,72,0,0,83,0,0,
    82,1,0,69,1,0,64,0,0,48,1,0,24,22,0,16,33,0,11,33,0,1,51,0,0,60,0,0,42,0,0,38,
    0,0,24,0,0,16,0,0,35,0,0,39,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,
    1,3,6,0,0,1,8,0,0,63,0,0,71,0,0,20,0,0,0,0,7,3,0,40,0,0,68,0,0,80,0,0,
    85,0,0,90,0,0,81,0,0,76,0,0,44,0,1,33,1,0,23,0,0,5,15,0,1,32,0,1,32,0,1,33,
    1,1,14,0,1,0,0,0,1,0,0,0,0,0,1,0,0,1,0,0,1,0,0,2,0,0,0,5,0,0,4,0,
    1,0,0,0,0,0,14,0,0,44,0,0,14,0,0,0,0,0,1,0,4,3,0,28,0,0,57,0,0,61,0,0,
    69,0,0,74,1,1,46,8,7,44,6,6,31,6,5,19,9,9,5,1,1,0,0,2,0,0,2,0,0,2,0,0,
    0,0,0,2,0,0,2,0,0,1,0,0,1,2,0,1,0,0,1,0,1,0,0,4,0,0,4,0,0,0,0,0,
    0,0,0,1,0,0,28,0,0,11,0,3,0,0,1,1,0,0,0,0,2,0,0,18,0,0,35,1,0,23,1,0,
    30,0,0,15,16,3,0,25,4,3,6,4,1,17,4,5,16,2,8,9,0,30,5,0,48,1,0,35,1,1,21,1,
    11,23,1,1,23,0,0,16,0,0,7,0,0,3,0,3,0,0,0,0,0,0,0,0,0,0,6,0,0,7,0,0,
    0,0,0,38,0,0,26,0,0,0,0,17,1,0,22,0,0,7,0,0,1,0,0,6,1,0,2,0,0,2,4,0,
    1,16,0,0,12,0,0,4,0,0,9,2,18,18,0,49,6,7,53,0,20,31,1,72,31,0,70,23,0,72,4,0,
    111,16,0,70,13,0,50,8,0,45,8,0,11,0,0,53,0,0,35,0,0,10,0,1,1,0,6,0,0,19,0,0,
    31,0,2,64,0,3,3,0,16,0,0,34,0,0,44,0,0,69,0,0,34,1,0,0,0,0,1,6,0,0,49,0,
    1,39,0,0,15,2,4,1,2,20,4,1,35,0,11,11,0,107,0,0,144,0,0,142,0,0,129,0,0,127,0,0,
    123,0,0,117,0,0,103,0,0,94,0,0,78,0,0,83,0,0,76,0,0,68,0,0,3,0,4,0,0,19,0,0,
    75,0,7,40,0,26,0,0,66,1,0,68,0,0,65,0,0,65,1,0,33,0,0,9,23,0,0,75,0,0,28,0,
    1,10,2,11,14,0,18,1,9,11,0,88,1,0,146,0,0,160,0,0,147,1,0,137,1,0,130,1,0,128,0,0,
    120,1,0,113,0,0,113,0,0,91,0,0,84,0,0,79,0,0,74,0,0,33,0,1,0,0,0,1,0,38,0,2,
    69,0,16,3,0,70,0,0,81,0,0,72,0,0,72,1,0,36,0,0,8,19,0,1,73,0,0,46,0,0,20,2,
    14,2,0,16,0,47,7,0,127,0,0,161,0,0,145,0,0,160,0,0,146,0,0,134,0,0,139,0,0,129,0,0,
    116,0,0,123,0,0,104,0,0,87,0,0,73,0,0,76,0,0,64,0,0,4,0,0,0,1,0,0,0,32,0,14,
    33,0,51,0,0,87,1,0,90,0,0,75,0,0,50,0,0,15,1,0,5,58,0,0,61,0,0,33,1,12,20,0,
    14,0,52,0,1,175,0,0,161,0,0,133,0,0,145,0,0,146,0,0,141,0,0,136,0,0,135,0,0,125,0,0,
    119,0,0,120,0,0,113,0,0,92,0,0,77,0,0,73,0,0,43,0,0,0,0,0,1,0,1,0,1,16,0,13,
    3,0,59,0,0,100,0,0,90,0,0,63,2,0,21,0,0,8,43,0,0,87,0,0,36,2,7,26,0,23,1,44,
    0,0,175,0,0,143,0,0,134,0,0,148,0,0,148,0,0,147,0,0,141,0,0,142,0,0,144,0,0,133,0,0,
    124,0,0,118,0,0,119,0,0,95,0,0,81
  };
  data=librevenge::RVNGBinaryData(wall19,3085);
  m_wallpaperList.push_back(MWAWGraphicStyle::Pattern(MWAWVec2i(32,32),MWAWEmbeddedObject(data,mime),MWAWColor(defCol[19])));

}

}

////////////////////////////////////////////////////
// KSEN function
////////////////////////////////////////////////////
std::ostream &operator<<(std::ostream &o, ClarisWksStyleManager::KSEN const &ksen)
{
  switch (ksen.m_valign) {
  case 0:
    break;
  case 1:
    o<<"yCenter,";
    break;
  case 2:
    o<<"yBottom,";
    break;
  default:
    o << "valign=#" << ksen.m_valign << ",";
    break;
  }
  switch (ksen.m_lineType) {
  case MWAWBorder::None:
    o << "lType=none,";
    break;
  case MWAWBorder::Simple:
    break;
  case MWAWBorder::Dot:
    o << "dotted,";
    break;
  case MWAWBorder::LargeDot:
    o << "dotted[large],";
    break;
  case MWAWBorder::Dash:
    o << "dash,";
    break;
#if !defined(__clang__)
  default:
    o << "lType=#" << int(ksen.m_lineType) << ",";
    break;
#endif
  }
  switch (ksen.m_lineRepeat) {
  case MWAWBorder::Single:
    break;
  case MWAWBorder::Double:
    o << "double,";
    break;
  case MWAWBorder::Triple:
    o << "triple,";
    break;
#if !defined(__clang__)
  default:
    o << "lRepeat=#" << int(ksen.m_lineRepeat) << ",";
    break;
#endif
  }
  switch (ksen.m_lines) {
  case 0:
    break;
  case 1:
    o << "lines=LT<->RB,";
    break;
  case 2:
    o << "lines=LB<->RT,";
    break;
  case 3:
    o << "cross,";
    break;
  default:
    o << "lines=#" << ksen.m_lines << ",";
    break;
  }
  o << ksen.m_extra;
  return o;
}

////////////////////////////////////////////////////
// Style function
////////////////////////////////////////////////////
std::ostream &operator<<(std::ostream &o, ClarisWksStyleManager::Style const &style)
{
  if (style.m_styleId != -1) {
    o << "styleId=[" << style.m_styleId ;
    if (style.m_localStyleId != -1 && style.m_localStyleId != style.m_styleId)
      o << ",lId=" << style.m_localStyleId;
    o << "],";
  }
  if (style.m_fontId != -1)
    o << "font=" << style.m_fontId << ",";
  if (style.m_cellFormatId != -1)
    o << "cellStyle=" << style.m_cellFormatId << ",";
  if (style.m_rulerId != -1)
    o << "ruler=" << style.m_rulerId << ",";
  if (style.m_rulerPId != -1)
    o << "ruler[parent]=LK" << style.m_rulerPId << ",";
  if (style.m_nameId != -1)
    o << "name=" << style.m_nameId << ",";
  if (style.m_ksenId != -1)
    o << "ksenId=" << style.m_ksenId << ",";
  if (style.m_graphicId != -1)
    o << "graphicId=" << style.m_graphicId << ",";
  o << style.m_extra;
  return o;
}

////////////////////////////////////////////////////
// StyleManager function
////////////////////////////////////////////////////
ClarisWksStyleManager::ClarisWksStyleManager(MWAWParserStatePtr parserState, ClarisWksDocument *document) :
  m_document(document), m_parserState(parserState), m_state()
{
  m_state.reset(new ClarisWksStyleManagerInternal::State);
}

ClarisWksStyleManager::~ClarisWksStyleManager()
{
}

int ClarisWksStyleManager::version() const
{
  if (m_state->m_version <= 0) m_state->m_version = m_parserState->m_version;
  return m_state->m_version;
}

bool ClarisWksStyleManager::getRulerName(int id, std::string &name) const
{
  Style style, parentStyle;
  if (!get(id, style) || style.m_rulerPId < 0 ||
      !get(style.m_rulerPId, parentStyle) || parentStyle.m_nameId < 0) return false;
  /* CHANGE: no sure how to recognize the basic style from a modified
     paragraph based on a style, so use a hack to avoid potential problem...
   */
  if (style.m_rulerId!=parentStyle.m_rulerId+1) return false;
  if (parentStyle.m_nameId >= static_cast<int>(m_state->m_nameList.size())) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::getRulerName: oops something look bad\n"));
    return false;
  }
  name=m_state->m_nameList[size_t(parentStyle.m_nameId)];
  return true;
}

bool ClarisWksStyleManager::get(int id, MWAWFont &ft) const
{
  if (id < 0 || id >= int(m_state->m_fontList.size())) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::get: can not find font %d\n", id));
    return false;
  }
  ft=m_state->m_fontList[size_t(id)];
  return true;
}

bool ClarisWksStyleManager::getColor(int id, MWAWColor &col) const
{
  int numColor = static_cast<int>(m_state->m_colorList.size());
  if (!numColor) {
    m_state->setDefaultColorList(version());
    numColor = int(m_state->m_colorList.size());
  }
  if (id < 0 || id >= numColor)
    return false;
  col = m_state->m_colorList[size_t(id)];
  return true;
}

bool ClarisWksStyleManager::getPattern(int id, MWAWGraphicStyle::Pattern &pattern, float &percent) const
{
  if (m_state->m_patternList.empty())
    m_state->setDefaultPatternList(version());
  if (id <= 0 || id > int(m_state->m_patternList.size()))
    return false;
  ClarisWksStyleManagerInternal::Pattern const &pat=m_state->m_patternList[size_t(id-1)];
  pattern = pat;
  percent = pat.m_percent;
  return true;
}

// accessor
int ClarisWksStyleManager::getFontId(int localId) const
{
  return m_state->getFontId(localId);
}

bool ClarisWksStyleManager::get(int styleId, ClarisWksStyleManager::Style &style) const
{
  style = Style();
  if (version() <= 2 || m_state->m_lookupMap.find(styleId) == m_state->m_lookupMap.end())
    return false;
  int id = m_state->m_lookupMap.find(styleId)->second;
  if (id < 0 ||m_state-> m_stylesMap.find(id) ==m_state-> m_stylesMap.end())
    return false;
  style =m_state-> m_stylesMap.find(id)->second;
  return true;
}

bool ClarisWksStyleManager::get(int formatId, ClarisWksStyleManager::CellFormat &format) const
{
  format = CellFormat();
  if (formatId<0||formatId>=int(m_state->m_cellFormatList.size())) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::get: can not find format %d\n", formatId));
    return false;
  }
  format = m_state->m_cellFormatList[size_t(formatId)];
  return true;
}

bool ClarisWksStyleManager::get(int ksenId, ClarisWksStyleManager::KSEN &ksen) const
{
  ksen = KSEN();
  if (ksenId < 0) return false;
  if (ksenId >= int(m_state->m_ksenList.size())) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::get: can not find ksen %d\n", ksenId));
    return false;
  }
  ksen = m_state->m_ksenList[size_t(ksenId)];
  return true;
}

bool ClarisWksStyleManager::get(int graphId, MWAWGraphicStyle &graph) const
{
  graph = MWAWGraphicStyle();
  if (graphId < 0) return false;
  if (graphId >= int(m_state->m_graphList.size())) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::get: can not find graph %d\n", graphId));
    return false;
  }
  graph = m_state->m_graphList[size_t(graphId)];
  return true;
}

bool ClarisWksStyleManager::updateGradient(int id, MWAWGraphicStyle &style) const
{
  if (m_state->m_gradientList.empty())
    m_state->setDefaultGradientList(version());
  if (id < 0 || id>=int(m_state->m_gradientList.size())) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::updateGradiant: called with id=%d\n", id));
    return false;
  }
  if (!m_state->m_gradientList[size_t(id)].update(style)) return false;
  // try to update the color list
  size_t numColors=style.m_gradientStopList.size();
  if (numColors<=1) return true;
  float f=1.f/float(numColors);
  MWAWColor col=MWAWColor::barycenter(f,style.m_gradientStopList[0].m_color,f,style.m_gradientStopList[1].m_color);
  for (size_t c=2; c<numColors; c++)
    col=MWAWColor::barycenter(1,col,f,style.m_gradientStopList[c].m_color);
  style.setSurfaceColor(col);
  return true;
}

bool ClarisWksStyleManager::updateWallPaper(int id, MWAWGraphicStyle &style) const
{
  int numWallpaper = static_cast<int>(m_state->m_wallpaperList.size());
  if (!numWallpaper) {
    m_state->setDefaultWallPaperList(version());
    numWallpaper = int(m_state->m_wallpaperList.size());
  }
  if (id <= 0 || id > numWallpaper) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::updateWallPaper: can not find wall paper %d\n", int(id)));
    return false;
  }
  MWAWGraphicStyle::Pattern const &pat=m_state->m_wallpaperList[size_t(id-1)];
  style.setPattern(pat);
  MWAWColor col;
  if (pat.getAverageColor(col))
    style.setSurfaceColor(col);
  return true;
}

////////////////////////////////////////////////////////////
// read file main structure
////////////////////////////////////////////////////////////
bool ClarisWksStyleManager::readPatternList(long endPos)
{
  int vers=version();
  MWAWInputStreamPtr &input= m_parserState->m_input;
  long pos=input->tell();
  long sz=long(input->readULong(4));
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  f << "Entries(PatternList):";
  if (sz<0 || (sz && sz < 140) || (endPos>0 && pos+sz+4>endPos) ||
      (endPos<=0 && !input->checkPosition(pos+sz+4))) {
    f << "###";
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readPatternList: can read pattern size\n"));
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos,librevenge::RVNG_SEEK_SET);
    return false;
  }
  if (sz==0) {
    ascFile.addPos(pos);
    ascFile.addNote("_");
    return true;
  }
  int N=static_cast<int>(input->readLong(2));
  f << "N=" << N << ",";
  long val;
  static long const expectedVal[]= {-1,0,8,0x80,1}; // type, ?, fSz, ?, ?
  for (int i=0; i<5; ++i) {
    val= input->readLong(2);
    if (i==2 && val!=8) {
      input->seek(pos,librevenge::RVNG_SEEK_SET);
      return false;
    }
    if (val!=expectedVal[i]) f << "f" << i << "=" << val << ",";
  }
  for (int i=0; i < 64; ++i) {
    val= input->readLong(2);
    if (val!=i) f << "pat" << i << "=" << val << ",";
  }
  if (140+8*N!=sz) {
    f << "###";
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readPatternList: unexpected pattern size\n"));
    ascFile.addDelimiter(input->tell(),'|');
  }
  else {
    m_state->setDefaultPatternList(vers);
    for (int i=0; i < N; ++i) {
      uint16_t pat[4];
      for (int j=0; j<4; ++j) pat[j]=static_cast<uint16_t>(input->readULong(2));
      ClarisWksStyleManagerInternal::Pattern pattern(pat);
      m_state->m_patternList.push_back(pattern);
      f << "pat" << i+64 << "=[" << pattern << "],";
    }
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  input->seek(pos+4+sz,librevenge::RVNG_SEEK_SET);
  return true;
}

bool ClarisWksStyleManager::readGradientList(long endPos)
{
  MWAWInputStreamPtr &input= m_parserState->m_input;
  long pos=input->tell();
  long sz=long(input->readULong(4));
  long finalPos=pos+sz+4;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  f << "Entries(GradientList):";
  if (sz<0 || (sz && sz < 76) || (endPos>0 && finalPos>endPos) ||
      (endPos<0 && !input->checkPosition(finalPos))) {
    f << "###";
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readGradientList: can read zone size\n"));
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos,librevenge::RVNG_SEEK_SET);
    return false;
  }
  if (sz==0) {
    ascFile.addPos(pos);
    ascFile.addNote("_");
    return true;
  }
  int N=static_cast<int>(input->readLong(2));
  f << "N=" << N << ",";
  long val;
  static long const expectedVal[]= {-1,0,0x28,0x40,1}; // type, ?, fSz, ?, ?
  for (int i=0; i<5; ++i) {
    val= input->readLong(2);
    if (i==2 && val!=0x28) { // fSz
      input->seek(pos,librevenge::RVNG_SEEK_SET);
      return false;
    }
    if (val!=expectedVal[i]) f << "f" << i << "=" << val << ",";
  }
  int const numDefGrad=32;
  if (12+2*numDefGrad+40*N!=sz) {
    f << "###";
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readGradientList: unexpected fieldSize\n"));
    ascFile.addDelimiter(input->tell(),'|');
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(finalPos,librevenge::RVNG_SEEK_SET);
    return true;
  }

  for (int i=0; i < numDefGrad; ++i) {
    val= input->readLong(2);
    if (val!=i) f << "grad" << i << "=" << val << ",";
  }

  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  m_state->setDefaultGradientList(version());
  for (int i=0; i < N; ++i) {
    pos=input->tell();
    f.str("");
    f << "GradientList-" << numDefGrad+i << ":";
    ClarisWksStyleManagerInternal::Gradient grad;
    for (int j=0; j<4; ++j) {
      unsigned char color[3];
      for (int c=0; c < 3; c++) color[c] = static_cast<unsigned char>(input->readULong(2)/256);
      grad.m_colors[j]= MWAWColor(color[0], color[1],color[2]);
    }
    grad.m_numColors=static_cast<int>(input->readLong(1));
    grad.m_type=static_cast<int>(input->readLong(1));
    grad.m_angle=static_cast<int>(input->readLong(2));
    grad.m_decal=float(input->readLong(4))/65536.f;
    int dim[4];
    for (int j=0; j<4; ++j) dim[j]=static_cast<int>(input->readLong(2));
    grad.m_box=MWAWBox2i(MWAWVec2i(dim[0],dim[1]),MWAWVec2i(dim[2],dim[3]));
    f << grad;
    if (!grad.ok()) {
      f << "##";
      MWAW_DEBUG_MSG(("ClarisWksStyleManager::readGradientList: can read the number of color or type\n"));
      ascFile.addPos(pos);
      ascFile.addNote(f.str().c_str());
      input->seek(finalPos,librevenge::RVNG_SEEK_SET);
      return true;
    }
    m_state->m_gradientList.push_back(grad);
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+40, librevenge::RVNG_SEEK_SET);
  }
  input->seek(finalPos,librevenge::RVNG_SEEK_SET);
  return true;
}

bool ClarisWksStyleManager::readColorList(MWAWEntry const &entry)
{
  if (!entry.valid()) return false;
  long pos = entry.begin();
  MWAWInputStreamPtr &input= m_parserState->m_input;
  input->seek(pos+4, librevenge::RVNG_SEEK_SET); // avoid header
  if (entry.length() == 4) return true;

  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  f << "Entries(ColorList):";
  int N = static_cast<int>(input->readULong(2));
  f << "N=" << N << ",";
  for (int i = 0; i < 2; i++) {
    int val = static_cast<int>(input->readLong(2));
    if (val) f << "f" << i << "=" << val << ",";
  }

  int const fSz = 16;
  if (pos+10+N*fSz > entry.end()) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readColorList: can not read data\n"));
    input->seek(pos, librevenge::RVNG_SEEK_SET);
    return false;
  }

  ascFile.addDelimiter(input->tell(),'|');
  input->seek(entry.end()-N*fSz, librevenge::RVNG_SEEK_SET);
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  m_state->m_colorList.resize(size_t(N));
  for (int i = 0; i < N; i++) {
    pos = input->tell();
    unsigned char color[3];
    for (int c=0; c < 3; c++) color[c] = static_cast<unsigned char>(input->readULong(2)/256);
    m_state->m_colorList[size_t(i)]= MWAWColor(color[0], color[1],color[2]);

    f.str("");
    f << "ColorList[" << i << "]:";
    ascFile.addDelimiter(input->tell(),'|');
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
  }

  input->seek(entry.end(), librevenge::RVNG_SEEK_SET);
  return true;
}

bool ClarisWksStyleManager::readStyles(MWAWEntry const &entry)
{
  if (!entry.valid() || entry.type() != "STYL")
    return false;
  int vers=version();
  long pos = entry.begin();
  MWAWInputStreamPtr &input= m_parserState->m_input;
  input->seek(pos+4, librevenge::RVNG_SEEK_SET); // skip header
  long sz = long(input->readULong(4));
  if (sz > entry.length()) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readStyles: pb with entry length"));
    input->seek(pos, librevenge::RVNG_SEEK_SET);
    return false;
  }

  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  f << "Entries(STYL):";
  if (vers <= 3) {
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(entry.end(), librevenge::RVNG_SEEK_SET);
    return true;
  }
  bool limitSet = true;
  if (vers <= 4) {
    // version 4 does not contents total length fields
    input->seek(-4, librevenge::RVNG_SEEK_CUR);
    limitSet = false;
  }
  else
    input->pushLimit(entry.end());
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  int id = 0;
  while (long(input->tell()) < entry.end()) {
    pos = input->tell();
    if (!readGenStyle(id)) {
      input->seek(pos, librevenge::RVNG_SEEK_SET);
      if (limitSet) input->popLimit();
      return false;
    }
    id++;
  }
  if (limitSet) input->popLimit();

  return true;
}

bool ClarisWksStyleManager::readGenStyle(int id)
{
  MWAWInputStreamPtr &input= m_parserState->m_input;
  long pos = input->tell();
  long sz = long(input->readULong(4));
  long endPos = pos+4+sz;
  if (!input->checkPosition(endPos)) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readGenStyle: pb with sub zone: %d", id));
    return false;
  }
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  f << "STYL-" << id << ":";
  if (sz < 16) {
    if (sz) f << "#";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(endPos, librevenge::RVNG_SEEK_SET);
    return true;
  }

  std::string name("");
  int N = static_cast<int>(input->readLong(2));
  int type = static_cast<int>(input->readLong(2));
  int val = static_cast<int>(input->readLong(2));
  int fSz = static_cast<int>(input->readLong(2));
  f << "N=" << N << ", type?=" << type <<", fSz=" << fSz << ",";
  if (val) f << "unkn=" << val << ",";
  int unkn[2];
  for (int i = 0; i < 2; i++) {
    unkn[i] = static_cast<int>(input->readLong(2));
    if (unkn[i])  f << "f" << i << "=" << val << ",";
  }
  // check here for gradient definition...
  if (version()>4 && type==-1 && val==0 && fSz==0x28 && unkn[0]==0x40 && unkn[1]==1) {
    input->seek(pos, librevenge::RVNG_SEEK_SET);
    if (readGradientList(endPos)) return true;
    input->seek(pos+16, librevenge::RVNG_SEEK_SET);
  }
  for (int i = 0; i < 4; i++)
    name += char(input->readULong(1));
  f << name;

  long actPos = input->tell();
  if (actPos != pos && actPos != endPos - N*fSz)
    ascFile.addDelimiter(input->tell(), '|');

  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  long numRemain = endPos - actPos;
  if (N > 0 && fSz > 0 && numRemain >= N*fSz) {
    input->seek(endPos-N*fSz, librevenge::RVNG_SEEK_SET);

    bool ok = false;
    if (name == "CHAR")
      ok = readStyleFonts(N, fSz);
    else if (name == "CELL")
      ok = readCellStyles(N, fSz);
    else if (name == "FNTM")
      ok = readFontNames(N, fSz);
    else if (name == "GRPH")
      ok = readGraphStyles(N, fSz);
    else if (name == "KSEN")
      ok = readKSEN(N, fSz);
    else if (name == "LKUP")
      ok = readLookUp(N, fSz);
    else if (name == "NAME")
      ok = readStyleNames(N, fSz);
    else if (name == "RULR")
      ok = m_document && m_document->getTextParser() && m_document->getTextParser()->readSTYL_RULR(N, fSz);
    else if (name == "STYL")
      ok = readStylesDef(N, fSz);
    if (!ok) {
      input->seek(endPos-N*fSz, librevenge::RVNG_SEEK_SET);
      for (int i = 0; i < N; i++) {
        pos = input->tell();
        f.str("");
        f << "STYL-" << id << "/" << name << "-" << i << ":";
        ascFile.addPos(pos);
        ascFile.addNote(f.str().c_str());
        input->seek(fSz, librevenge::RVNG_SEEK_CUR);
      }
    }
  }

  input->seek(endPos, librevenge::RVNG_SEEK_SET);
  if (name=="NAME") {
    if (!readPatternList()) {
      MWAW_DEBUG_MSG(("ClarisWksStyleManager::readGenStyle: can not find the pattern list\n"));
      input->seek(endPos, librevenge::RVNG_SEEK_SET);
    }
    else if (version()==4) {
      endPos=input->tell();
      if (!readGradientList()) {
        MWAW_DEBUG_MSG(("ClarisWksStyleManager::readGenStyle: can not find the gradient list\n"));
        input->seek(endPos, librevenge::RVNG_SEEK_SET);
      }
    }
  }

  return true;
}

bool ClarisWksStyleManager::readStylesDef(int N, int fSz)
{
  m_state->m_stylesMap.clear();
  if (fSz == 0 || N== 0)
    return true;
  if (fSz < 28) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readStylesDef: Find old data size %d\n", fSz));
    return false;
  }
  MWAWInputStreamPtr &input= m_parserState->m_input;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;

  for (int i = 0; i < N; i++) {
    long pos = input->tell();
    Style style;
    f.str("");
    int val = static_cast<int>(input->readLong(2));
    if (val != -1) f << "f0=" << val << ",";
    val = static_cast<int>(input->readLong(2));
    if (val) f << "f1=" << val << ",";
    f << "unkn?=" << input->readLong(2) << ",";
    style.m_localStyleId = static_cast<int>(input->readLong(2));
    if (i != style.m_localStyleId && style.m_localStyleId != -1) f << "#styleId,";
    style.m_styleId = static_cast<int>(input->readLong(2));
    for (int j = 0; j < 2; j++) {
      // unknown : hash, dataId ?
      f << "g" << j << "=" << input->readLong(1) << ",";
    }
    f << "g=" << input->readLong(2) << ",";
    style.m_rulerPId=static_cast<int>(input->readLong(2));
    int lookupId2 = static_cast<int>(input->readLong(2));
    f << "lookupId2=" << lookupId2 << ",";
    style.m_fontId = static_cast<int>(input->readLong(2));
    style.m_cellFormatId = static_cast<int>(input->readLong(2));
    style.m_graphicId = static_cast<int>(input->readLong(2));
    style.m_rulerId = static_cast<int>(input->readLong(2));
    if (fSz >= 30)
      style.m_ksenId = static_cast<int>(input->readLong(2));
    style.m_nameId = static_cast<int>(input->readLong(2));
    style.m_extra = f.str();
    if (m_state->m_stylesMap.find(i)==m_state->m_stylesMap.end())
      m_state->m_stylesMap[i] = style;
    else {
      MWAW_DEBUG_MSG(("ClarisWksStyleManager::readStylesDef: style %d already exists\n", i));
    }
    if (long(input->tell()) != pos+fSz)
      ascFile.addDelimiter(input->tell(), '|');

    f.str("");
    if (!i)
      f << "Entries(Style)-0:" << style;
    else
      f << "Style-" << i << ":" << style;
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
  }
  return true;
}

bool ClarisWksStyleManager::readLookUp(int N, int fSz)
{
  m_state->m_lookupMap.clear();

  if (fSz == 0 || N== 0) return true;
  if (fSz < 2) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readLookUp: the field size seems bad\n"));
    return false;
  }
  MWAWInputStreamPtr &input= m_parserState->m_input;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;

  for (int i = 0; i < N; i++) {
    long pos = input->tell();
    f.str("");
    if (i == 0) f << "Entries(StylLookUp): StylLookUp-LK0:";
    else f << "StylLookUp-LK" << i << ":";
    int val = static_cast<int>(input->readLong(2));
    if (m_state->m_stylesMap.find(val)!=m_state->m_stylesMap.end() &&
        m_state->m_stylesMap.find(val)->second.m_localStyleId != val &&
        m_state->m_stylesMap.find(val)->second.m_localStyleId != -1) {
      MWAW_DEBUG_MSG(("ClarisWksStyleManager::readLookUp: find some incoherence between style and lookup\n"));
      f << "##";
    }
    m_state->m_lookupMap[i] = val;
    f << "styleId=" << val;
    if (fSz != 2) {
      ascFile.addDelimiter(input->tell(), '|');
      input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
    }
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
  }

  return true;
}

////////////////////////////////////////////////////////////
// small structure
////////////////////////////////////////////////////////////
bool ClarisWksStyleManager::readFontNames()
{
  MWAWInputStreamPtr &input= m_parserState->m_input;
  long pos=input->tell();
  if (!input->checkPosition(pos+8)) return false;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  f << "Entries(FNTM):";
  if (input->readULong(4) != 0x464e544d)
    return false;
  long sz = long(input->readULong(4));
  if (sz==0) {
    ascFile.addPos(pos);
    ascFile.addNote("_");
    return true;
  }
  long endPos=pos+4+sz;
  if (!input->checkPosition(endPos)) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readFontNames: the data size seems bad\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }

  int N = static_cast<int>(input->readLong(2));
  f << "N=" << N << ",";
  int type = static_cast<int>(input->readLong(2));
  if (type != -1)
    f << "#type=" << type << ",";
  int val = static_cast<int>(input->readLong(2));
  if (val) f << "#unkn=" << val << ",";
  int fSz = static_cast<int>(input->readULong(2));
  int hSz = static_cast<int>(input->readULong(2));
  if (!fSz || N*fSz+hSz+12 != sz) {
    input->seek(pos, librevenge::RVNG_SEEK_SET);
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readFontNames: unexpected field/header size\n"));
    f << "###";
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    return false;
  }

  if (long(input->tell()) != pos+8+12+hSz) {
    ascFile.addDelimiter(input->tell(), '|');
    input->seek(pos+8+12+hSz, librevenge::RVNG_SEEK_SET);
  }
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());

  pos=input->tell();
  if (fSz!=72) {
    input->seek(endPos, librevenge::RVNG_SEEK_SET);
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readFontNames: no sure how to read the data\n"));
    ascFile.addPos(pos);
    ascFile.addNote("FNTM-data:###");
    return true;
  }

  for (int i=0; i<N; ++i) {
    pos=input->tell();
    f.str("");
    f << "FNTM-" << i << ":";
    int fId=static_cast<int>(input->readULong(2));
    f << "fId=" << fId << ",";
    val=static_cast<int>(input->readULong(2)); // always fId?
    if (val!=fId) f << "fId2=" << val << ",";
    for (int j=0; j<2; ++j) { // always 0?
      val=static_cast<int>(input->readLong(2));
      if (val) f << "f" << j << "=" << val << ",";
    }
    int sSz=static_cast<int>(input->readULong(1));
    if (!sSz || sSz+9>fSz) {
      MWAW_DEBUG_MSG(("ClarisWksStyleManager::readFontNames: the string size seems bad\n"));
      f << "###";
    }
    else {
      std::string name("");
      for (int s=0; s<sSz; ++s) name+=char(input->readULong(1));
      f << "'" << name << "'";
      m_parserState->m_fontConverter->setCorrespondance(fId, name);
    }
    input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
  }
  return true;
}

bool ClarisWksStyleManager::readFontNames(int N, int fSz)
{
  if (fSz == 0 || N== 0) return true;
  if (fSz < 16) return false;

  MWAWInputStreamPtr &input= m_parserState->m_input;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  for (int i = 0; i < N; i++) {
    long pos = input->tell();
    f.str("");
    if (i == 0) f << "Entries(FntNames): FntNames-0:";
    else f << "FntNames-" << i << ":";
    int fontEncoding = static_cast<int>(input->readULong(2));
    f << "nameEncoding=" << fontEncoding << ",";
    f << "type?=" << input->readLong(2) << ",";

    int nChar = static_cast<int>(input->readULong(1));
    if (5+nChar > fSz) {
      static bool first = true;
      if (first) {
        MWAW_DEBUG_MSG(("ClarisWksStyleManager::readFontNames: pb with name field %d", i));
        first = false;
      }
      f << "#";
    }
    else {
      std::string name("");
      bool ok = true;
      for (int c = 0; c < nChar; c++) {
        char ch = char(input->readULong(1));
        if (ch == '\0') {
          MWAW_DEBUG_MSG(("ClarisWksStyleManager::readFontNames: pb with name field %d\n", i));
          ok = false;
          break;
        }
        else if (ch & 0x80) {
          static bool first = true;
          if (first) {
            MWAW_DEBUG_MSG(("ClarisWksStyleManager::readFontNames: find odd font\n"));
            first = false;
          }
          if (fontEncoding!=0x4000)
            ok = false;
        }
        name += ch;
      }
      f << "'" << name << "'";
      if (name.length() && ok) {
        std::string family = fontEncoding==0x4000 ? "Osaka" : "";
        m_state->m_localFIdMap[i]=m_parserState->m_fontConverter->getId(name, family);
      }
    }
    if (long(input->tell()) != pos+fSz) {
      ascFile.addDelimiter(input->tell(), '|');
      input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
    }

    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
  }
  return true;
}

bool ClarisWksStyleManager::readFont(int id, int fontSize, MWAWFont &font)
{
  MWAWInputStreamPtr &input= m_parserState->m_input;
  long pos = input->tell();

  input->seek(pos, librevenge::RVNG_SEEK_SET);
  font = MWAWFont();
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  if (id == 0)
    f << "Entries(CHAR)-0:";
  else
    f << "CHAR-" << id << ":";

  int val = static_cast<int>(input->readLong(2));
  if (val != -1) f << "f0=" << val << ",";
  f << "flags=[";
  for (int i = 0; i < 6; i++) {
    val  = static_cast<int>(input->readLong(2));
    if (val) {
      if (i == 3)
        f << "f" << i << "=" << std::hex << val << std::dec << ",";
      else
        f << "f" << i << "=" << val << ",";
    }
  }
  font.setId(getFontId(static_cast<int>(input->readULong(2))));
  int flag =static_cast<int>(input->readULong(2));
  uint32_t flags=0;
  if (flag&0x1) flags |= MWAWFont::boldBit;
  if (flag&0x2) flags |= MWAWFont::italicBit;
  if (flag&0x4) font.setUnderlineStyle(MWAWFont::Line::Simple);
  if (flag&0x8) flags |= MWAWFont::embossBit;
  if (flag&0x10) flags |= MWAWFont::shadowBit;
  if (flag&0x20) font.setDeltaLetterSpacing(-1);
  if (flag&0x40) font.setDeltaLetterSpacing(1);
  if (flag&0x80) font.setStrikeOutStyle(MWAWFont::Line::Simple);
  if (flag&0x100) font.set(MWAWFont::Script::super100());
  if (flag&0x200) font.set(MWAWFont::Script::sub100());
  if (flag&0x400) font.set(MWAWFont::Script::super());
  if (flag&0x800) font.set(MWAWFont::Script::sub());
  if (flag&0x2000) {
    font.setUnderlineStyle(MWAWFont::Line::Simple);
    font.setUnderlineType(MWAWFont::Line::Double);
  }
  font.setSize(float(input->readULong(1)));

  int colId = static_cast<int>(input->readULong(1));
  MWAWColor color(MWAWColor::black());
  if (colId!=1 && !getColor(colId, color))
    f << "#col=" << std::hex << colId << std::dec << ",";
  font.setColor(color);
  if (fontSize >= 12 && version()==6) {
    flag = static_cast<int>(input->readULong(2));
    if (flag & 0x1)
      font.setUnderlineStyle(MWAWFont::Line::Simple);
    if (flag & 0x2) {
      font.setUnderlineStyle(MWAWFont::Line::Simple);
      font.setUnderlineType(MWAWFont::Line::Double);
    }
    if (flag & 0x20)
      font.setStrikeOutStyle(MWAWFont::Line::Simple);
    flag &= 0xFFDC;
    if (flag)
      f << "#flag2=" << std::hex << flag << std::dec << ",";
  }
  font.setFlags(flags);
  f << font.getDebugString(m_parserState->m_fontConverter);
  if (long(input->tell()) != pos+fontSize)
    ascFile.addDelimiter(input->tell(), '|');
  input->seek(pos+fontSize, librevenge::RVNG_SEEK_SET);
  ascFile.addPos(pos);
  ascFile.addNote(f.str().c_str());
  return true;
}

bool ClarisWksStyleManager::readStyleFonts(int N, int fSz)
{
  if (fSz == 0 || N== 0) return true;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  if (m_state->m_fontList.size()) {
    MWAW_DEBUG_MSG(("ClarisWksText::readStyleFonts: font list already exists!!!\n"));
  }
  m_state->m_fontList.resize(size_t(N));
  MWAWInputStreamPtr &input= m_parserState->m_input;
  for (int i = 0; i < N; i++) {
    long pos = input->tell();
    MWAWFont font;
    if (readFont(i, fSz, font))
      m_state->m_fontList[size_t(i)] = font;
    else {
      f.str("");
      if (!i)
        f << "Entries(Font)-F0:#";
      else
        f << "FontF-" << i << ":#";
      ascFile.addPos(pos);
      ascFile.addNote(f.str().c_str());
    }
    input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
  }
  return true;
}

bool ClarisWksStyleManager::readStyleNames(int N, int fSz)
{
  if (fSz == 0 || N== 0) return true;
  if (fSz < 2) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readStyleNames: the field size seems bad\n"));
    return false;
  }
  MWAWInputStreamPtr &input= m_parserState->m_input;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  m_state->m_nameList.clear();
  for (int i = 0; i < N; i++) {
    long pos = input->tell();
    f.str("");
    if (i == 0) f << "Entries(StylName): StylName-0:";
    else f << "StylName-" << i << ":";
    f << "LK" << input->readLong(2) << ",";
    std::string name("");
    if (fSz > 4) {
      int nChar = static_cast<int>(input->readULong(1));
      if (3+nChar > fSz) {
        static bool first = true;
        if (first) {
          MWAW_DEBUG_MSG(("ClarisWksStyleManager::readStyleNames: pb with name field %d\n", i));
          first = false;
        }
        f << "#";
      }
      else {
        for (int c = 0; c < nChar; c++)
          name += char(input->readULong(1));
        f << "'" << name << "'";
      }
      m_state->m_nameList.push_back(name);
    }
    if (long(input->tell()) != pos+fSz) {
      ascFile.addDelimiter(input->tell(), '|');
      input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
    }

    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
  }
  return true;
}

bool ClarisWksStyleManager::readCellStyles(int N, int fSz)
{
  if (fSz == 0 || N== 0) return true;
  if (fSz < 18) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readCellStyles: Find old ruler size %d\n", fSz));
    return false;
  }
  MWAWInputStreamPtr &input= m_parserState->m_input;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  int val;
  for (int i = 0; i < N; i++) {
    long pos = input->tell();
    CellFormat format;
    f.str("");
    // 3 int, id or color ?
    for (int j = 0; j < 3; j++) {
      val = static_cast<int>(input->readLong(2));
      if (val == -1) continue;
      f << "f" << j << "=" << val << ",";
    }
    /* g0=0|4|8|c|1f, g1:number which frequently with 8|c|d
       g2=0|4|c|13|17|1f, g3:number which frequently with 8|c|d
     */
    for (int j = 0; j < 4; j++) {
      val = static_cast<int>(input->readULong(1));
      if (val)
        f << "g" << j << "=" << std::hex << val << std::dec << ",";
    }
    format.m_fileFormat=static_cast<int>(input->readULong(1));
    format.m_digits=static_cast<int>(input->readULong(1));
    val = static_cast<int>(input->readULong(1));
    switch (val) {
    case 0: // default
      break;
    case 1:
      format.m_hAlign=MWAWCell::HALIGN_LEFT;
      break;
    case 2:
      format.m_hAlign=MWAWCell::HALIGN_CENTER;
      break;
    case 3:
      format.m_hAlign=MWAWCell::HALIGN_RIGHT;
      break;
    default:
      f << "#hAlign=" << val << ",";
      break;
    }

    format.m_borders=static_cast<int>(input->readULong(1)); // 0-f
    val = static_cast<int>(input->readULong(1));
    if (val==1) format.m_thousandHasSeparator=true;
    else if (val) f << "#separateThousand=" << val << ",";
    val = static_cast<int>(input->readULong(1));
    if (val==1) format.m_parenthesesForNegative=true;
    else if (val) f << "#parenthesesForNegative=" << val << ",";
    val = static_cast<int>(input->readULong(1));
    if (val==1) format.m_wrap=true;
    else if (val) f << "#wrap=" << val << ",";
    val = static_cast<int>(input->readULong(1));
    if (val==1) f << "lock,";
    else if (val) f << "#lock=" << val << ",";
    format.m_extra=f.str();
    m_state->m_cellFormatList.push_back(format);

    f.str("");
    if (!i)
      f << "Entries(CellStyle)-0:"<<format;
    else
      f << "CellStyle-" << i << ":"<<format;
    if (long(input->tell()) != pos+fSz)
      ascFile.addDelimiter(input->tell(), '|');
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
  }
  return true;
}

bool ClarisWksStyleManager::readGraphStyles(int N, int fSz)
{
  if (fSz == 0 || N== 0) return true;
  int const vers = version();
  if ((vers <= 4 && fSz < 24) || (vers >= 5 && fSz < 28)) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readGraphStyles: Find old ruler size %d\n", fSz));
    return false;
  }
  MWAWInputStreamPtr &input= m_parserState->m_input;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;
  int val;
  std::vector<int16_t> values16; // some values can be small or little endian, so...
  std::vector<int32_t> values32;
  for (int i = 0; i < N; i++) {
    long pos = input->tell();
    f.str("");
    MWAWGraphicStyle graph;
    // 3 int, id (find either f0=<small number> or f1=0, f2=small number
    for (int j = 0; j < 3; j++) {
      val = static_cast<int>(input->readLong(2));
      if (val == -1) continue;
      f << "f" << j << "=" << val << ",";
    }

    values16.resize(0);
    values32.resize(0);
    // 2 two dim
    for (int j = 0; j < 2; j++)
      values16.push_back(static_cast<int16_t>(input->readLong(2)));
    graph.m_lineWidth=float(input->readULong(1));
    val = static_cast<int>(input->readULong(1)); // 0|1|4|80
    if (val)
      f << "f3=" << std::hex << val << std::dec << ",";
    int col[2];
    for (int j = 0; j < 2; j++)
      col[j] = static_cast<int>(input->readULong(1));
    for (int j = 0; j < 3; j++)
      values16.push_back(static_cast<int16_t>(input->readLong(2)));

    if (m_document) m_document->checkOrdering(values16, values32);
    if (values16[0] || values16[1])
      f << "dim=" << values16[0] << "x" << values16[1] << ",";
    for (size_t j = 0; j < 2; ++j) {
      if (values16[j+2]==1) {
        if (j==0) graph.m_lineOpacity=0;
        else graph.m_surfaceOpacity=0;
        continue;
      }
      MWAWColor color;
      if (!getColor(col[j], color)) {
        f << "#col" << j << "=" << col << ",";
        continue;
      }
      MWAWGraphicStyle::Pattern pattern;
      float percent;
      if (values16[j+2] && getPattern(values16[j+2], pattern, percent)) {
        pattern.m_colors[1]=color;
        if (!pattern.getUniqueColor(color)) {
          if (j) graph.setPattern(pattern);
          pattern.getAverageColor(color);
        }
      }
      else if (values16[j+2])
        f << "###pat" << j << "=" << values16[j+2];

      if (j==0) graph.m_lineColor = color;
      else graph.setSurfaceColor(color);
    }
    if (values16[4])
      f << "g0=" << values16[4] << ",";

    val = static_cast<int>(input->readULong(1)); // 0|1|2
    if (val) f << "g1=" << val << ",";
    val = static_cast<int>(input->readULong(2)); // 0|1
    if (val) f << "g2=" << val << ",";

    graph.m_extra = f.str();
    m_state->m_graphList.push_back(graph);
    f.str("");
    if (!i)
      f << "Entries(GrphStyle)-0:" << graph;
    else
      f << "GrphStyle-" << i << ":" << graph;
    if (long(input->tell()) != pos+fSz)
      ascFile.addDelimiter(input->tell(), '|');
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());
    input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
  }
  return true;
}


bool ClarisWksStyleManager::readKSEN(int N, int fSz)
{
  if (fSz == 0 || N== 0) return true;
  m_state->m_ksenList.resize(0);
  if (fSz != 14) {
    MWAW_DEBUG_MSG(("ClarisWksStyleManager::readKSEN: Find odd ksen size %d\n", fSz));
  }
  MWAWInputStreamPtr &input= m_parserState->m_input;
  libmwaw::DebugFile &ascFile = m_parserState->m_asciiFile;
  libmwaw::DebugStream f;

  for (int i = 0; i < N; i++) {
    long pos = input->tell();
    KSEN ksen;
    f.str("");
    long val = input->readLong(2); // always -1
    if (val != -1)
      f << "unkn=" << val << ",";
    val = input->readLong(4); // a big number
    if (val != -1)
      f << "f0=" << val << ",";
    for (int j = 0; j < 2; j++) { // fl0=[0|1|2|4|8|f]: a pos?, fl1=[0|8|9|b|d|f] ?
      val = input->readLong(2);
      if (val) f << "fl" << j << "=" << std::hex << val << std::dec << ",";
    }
    val = input->readLong(1); // 0-5
    switch (val) {
    case 0:
      break;
    case 1:
      ksen.m_lineType = MWAWBorder::Dash;
      break;
    case 2:
      ksen.m_lineType = MWAWBorder::Dot;
      break;
    case 3:
      ksen.m_lineRepeat = MWAWBorder::Double;
      break;
    case 4:
      ksen.m_lineRepeat = MWAWBorder::Double;
      f << "w[external]=2,";
      break;
    case 5:
      ksen.m_lineRepeat = MWAWBorder::Double;
      f << "w[internal]=2,";
      break;
    default:
      f << "#lineType=" << val << ",";
      break;
    }
    ksen.m_valign = static_cast<int>(input->readLong(1));
    ksen.m_lines = static_cast<int>(input->readLong(1));
    val = input->readLong(1); // 0-18
    if (val) f << "g1=" << val << ",";
    ksen.m_extra = f.str();
    m_state->m_ksenList.push_back(ksen);
    f.str("");
    if (!i)
      f << "Entries(Ksen)-0:";
    else
      f << "Ksen-" << i << ":";
    f << ksen;
    ascFile.addPos(pos);
    ascFile.addNote(f.str().c_str());

    input->seek(pos+fSz, librevenge::RVNG_SEEK_SET);
  }
  return true;
}

ClarisWksStyleManager::CellFormat::~CellFormat()
{
}

std::ostream &operator<<(std::ostream &o, ClarisWksStyleManager::CellFormat const &form)
{
  o << static_cast<MWAWCell::Format const &>(form) << ",";
  if (form.m_fileFormat >= 0 && form.m_fileFormat < 16) {
    static char const *(wh[16])= {
      "general", "currency", "percent", "scientific", "fixed",
      "date[m/d/y]", "date[d B y]", "date[B d Y]", "date[a, b d Y]", "date[A, B d Y]",
      "form10", "form11",
      "time[H:M]", "time[H:M:S]", "time[I:M p]", "time[I:M:S p]"
    };
    o << wh[form.m_fileFormat] << ",";
  }
  else if (form.m_fileFormat > 0)
    o << "#format=" << form.m_fileFormat << ",";
  if (form.m_borders) o << "borders=" << form.m_borders << ",";
  if (form.m_wrap) o << "wrap[content],";
  o << form.m_extra;
  return o;
}

// vim: set filetype=cpp tabstop=2 shiftwidth=2 cindent autoindent smartindent noexpandtab:
