/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <QAction>
#include <QApplication>
#include <QColor>
#include <QComboBox>
#include <QCursor>
#include <QDebug>
#include <QFile>
#include <QHBoxLayout>
#include <QHeaderView>
#include <QInputDialog>
#include <QListWidget>
#include <QMenu>
#include <QMenuBar>
#include <QMessageBox>
#include <QPushButton>
#include <QScrollArea>
#include <QSettings>
#include <QSortFilterProxyModel>
#include <QSplitter>
#include <QStackedWidget>
#include <QString>
#include <QStringList>
#include <QToolBar>
#include <QTextBrowser>
#include <QTreeView>
#include <QTreeWidget>
#include <QVBoxLayout>

#include <iostream>
#include <sstream>
#include <fstream>
#include <cstdio>
#include <stdio.h>
#include <stdlib.h>

#include "MvFileInfo.h"

#include "MvQObstatCurveToolBase.h"
#include "MvKeyProfile.h"

#include "MvQFileInfoLabel.h"
#include "MvQKeyManager.h"
#include "MvQKeyProfileModel.h"
//#include "MvQPlotWidget.h"
#include "MvQTreeView.h"

#include "mars.h"

MvQObstatCurveToolBase::MvQObstatCurveToolBase(QWidget *)
{
	//File info label        
	fileInfoLabel_=new MvQFileInfoLabel;

	//Set up grib message group box
	setupItemBox();
	
	//Set up grib dump group box
	setupPlotBox();

	//----------------------
	// Init
	//----------------------

	currentItemNo_=-1;

	//----------------------------
	// 
	//----------------------------

	setupViewActions();
	setupSettingsActions();

	//----------------------------
	// Signals and slots
	//----------------------------

	//Message list selection	
	connect(itemTree_,SIGNAL(clicked(QModelIndex)),
		this,SLOT(slotSelectItem(QModelIndex)));

	connect(itemTree_,SIGNAL(activated(QModelIndex)),
		this,SLOT(slotSelectItem(QModelIndex)));


	char *mvbin=getenv("METVIEW_BIN");
	if (mvbin == 0)  
	{	
		qDebug() << "MvQObstatCurveToolBase::MvQObstatCurveToolBase() --> No METVIEW_BIN found!";
			//marslog(LOG_EXIT|LOG_EROR,"%s\n",sbuff.str().c_str());
			//setError(13,"Environment variable METVIEW_USER_DIRECTORY is not defined!");	
		return;	
	}

	magmlScript_=QString(mvbin);
	magmlScript_+="/ObstatCurveKind1.magml";
	
}
 
MvQObstatCurveToolBase::~MvQObstatCurveToolBase()
{	
	if(itemModel_) delete itemModel_;	
}

void MvQObstatCurveToolBase::setupItemBox()
{
	QWidget *w;
	QLabel *label;
	QVBoxLayout *vb;

	//--------------------------------
	// Message tree (key profile)
	//--------------------------------

	itemPanel_=new QWidget;
	QVBoxLayout *itemLayout = new QVBoxLayout;
	itemLayout->setContentsMargins(0,0,0,0);
	itemPanel_->setLayout(itemLayout);
	//messageSplitter_->addWidget(messageWidget);	
	

	//Label
	label=new QLabel(tr("Items"));
	label->setFrameShape(QFrame::StyledPanel);
	label->setProperty("mvStyle","panel");
	itemLayout->addWidget(label);

	//Message tree
	itemModel_= new MvQKeyProfileModel;
	itemSortModel_= new QSortFilterProxyModel;
        itemSortModel_->setSourceModel(itemModel_);
	itemSortModel_->setDynamicSortFilter(true);
	itemSortModel_->setFilterRole(Qt::UserRole);
	//messageSortModel_->setFilterRegExp(QRegExp("[1]"));
	itemSortModel_->setFilterFixedString("1");
	itemSortModel_->setFilterKeyColumn(0);


	itemTree_=new MvQTreeView;
	itemTree_->setObjectName("itemTree");
	itemTree_->setProperty("mvStyle",0);
	itemTree_->setSortingEnabled(true); 
	itemTree_->sortByColumn (0,Qt::AscendingOrder); 
	itemTree_->setAlternatingRowColors(true);
        itemTree_->setAllColumnsShowFocus(true);
	itemTree_->setModel(itemSortModel_);	
	//messageTree_->header()->setContextMenuPolicy(Qt::CustomContextMenu);
	itemTree_->setActvatedByKeyNavigation(true);
	itemTree_->setAcceptDrops(true);
	itemTree_->setDragDropMode(QAbstractItemView::DropOnly);
	itemTree_->setRootIsDecorated(false);

	//Set header
	QHeaderView *header=itemTree_->header();
	header->setContextMenuPolicy(Qt::CustomContextMenu);

	itemLayout->addWidget(itemTree_);	
}

void MvQObstatCurveToolBase::setupPlotBox()
{
//	QWidget *w;
	QLabel *label;

	//--------------------------------
	// Message tree (key profile)
	//--------------------------------

	plotPanel_=new QWidget;
	QVBoxLayout *plotLayout = new QVBoxLayout;
	plotLayout->setContentsMargins(0,0,0,0);
	plotPanel_->setLayout(plotLayout);
	//messageSplitter_->addWidget(messageWidget);	

	//Label
	label=new QLabel(tr("Plot"));
	label->setFrameShape(QFrame::StyledPanel);
	label->setProperty("mvStyle","panel");
	plotLayout->addWidget(label);


	QScrollArea *area=new QScrollArea;
	
	//plot area
	plotWidget_= new QLabel;
	plotWidget_->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
     	plotWidget_->setScaledContents(true);

	//plotWidget_->setMinimumSize(800,700);
	//plotWidget_->setSize(800,600);
	//plotLayout->addWidget(plotWidget_);

	area->setWidget(plotWidget_);
	plotLayout->addWidget(area);
	
}

void MvQObstatCurveToolBase::setupViewActions()
{
	//
	QAction* actionFileInfo = new QAction(this);
        actionFileInfo->setObjectName(QString::fromUtf8("actionFileInfo"));
	actionFileInfo->setText(tr("File info"));
	actionFileInfo->setCheckable(true);
	actionFileInfo->setChecked(true);
	actionFileInfo->setToolTip(tr("Show/hide file info"));
	QIcon icon;
	icon.addPixmap(QPixmap(QString::fromUtf8(":/examinerBase/fileInfo.png")), QIcon::Normal, QIcon::Off);
 	actionFileInfo->setIcon(icon);	
	
	//Define routines
	connect(actionFileInfo,SIGNAL(triggered(bool)),
		fileInfoLabel_,SLOT(setVisible(bool)));

	MvQMainWindow::MenuType menuType=MvQMainWindow::ViewMenu;
	menuItems_[menuType].push_back(new MvQMenuItem(actionFileInfo));	
}

void MvQObstatCurveToolBase::setupSettingsActions()
{
	//
	QAction* actionKeyManager = new QAction(this);
        actionKeyManager->setObjectName(QString::fromUtf8("actionKeyManager"));
	actionKeyManager->setText(tr("Key manager"));
	actionKeyManager->setToolTip(tr("Key profile manager"));
	QIcon icon;
	icon.addPixmap(QPixmap(QString::fromUtf8(":/examinerBase/keyManager.png")), QIcon::Normal, QIcon::Off);
 	actionKeyManager->setIcon(icon);
	actionKeyManager->setToolTip(tr("Key manager"));
  
	//Key profile management
	connect(actionKeyManager, SIGNAL(triggered()), 
		this, SLOT(slotManageKeyProfiles()));

	MvQMainWindow::MenuType menuType=MvQMainWindow::SettingsMenu;
	menuItems_[menuType].push_back(new MvQMenuItem(actionKeyManager));	
}


void MvQObstatCurveToolBase::init(MvQObstatCurveData *data)
{	
	data_=data;
	
	//----------------------------------
	// Initilaize the key profile list
	//---------------------------------
	
	MvKeyProfile *prof=new MvKeyProfile("Metview default");
	loadKeyProfile(prof);

	//disconnect(keyCombo_,SIGNAL(currentIndexChanged(int)),this,SLOT(loadKeyProfile(int)));
	
	//Set tree focus
	//ssageTree_->setFocus(Qt::OtherFocusReason);
	
	//Fileinfo label
	updateFileInfoLabel();
}


void MvQObstatCurveToolBase::loadKeyProfile(MvKeyProfile *prof)
{	
	//Get the current profile from the model (this will be the previous)
	MvKeyProfile* prevProfile = itemModel_->keyProfile();

	//Update grib metadata with the new key profile
	data_->loadKeyProfile(prof);

	//---------------------
	// Update message tree
	//---------------------
	
	itemModel_->setKeyProfile(prof);

	for(int i=0;i < itemModel_->columnCount()-1; i++)
	{ 
		itemTree_->resizeColumnToContents(i);
	}

	itemTree_->setCurrentIndex(itemSortModel_->index(0,0));
	slotSelectItem(itemTree_->currentIndex());

	//Clear the previous provile
	if(prevProfile != 0 && prof  != prevProfile)
		prevProfile->clearKeyData();	

	/*
	//Get the current profile from the model (this will be the previous)
	MvKeyProfile* prevProfile = messageModel_->keyProfile();

	//Update grib metadata with the new key profile
	data_->loadKeyProfile(prof);

	//---------------------
	// Update message tree
	//---------------------
	
	messageModel_->setKeyProfile(prof);

	for(int i=0;i < messageModel_->columnCount()-1; i++)
	{ 
		messageTree_->resizeColumnToContents(i);
	}

	messageTree_->setCurrentIndex(messageSortModel_->index(0,0));
	slotSelectMessage(messageTree_->currentIndex());

	//Clear the previous provile
	if(prevProfile != 0 && prof  != prevProfile)
		prevProfile->clearKeyData();
	*/	
}

void MvQObstatCurveToolBase::slotSelectItem(QModelIndex index)
{	
	int cnt=itemSortModel_->mapToSource(index).row(); 

	string fdump;

	qDebug() << "slotSelectItem" << currentItemNo_ << cnt ;

	if(currentItemNo_ == cnt)
	{
		return;
	}

	currentItemNo_=cnt;

createPlot();

	//Override cursor
	QApplication::setOverrideCursor(QCursor(Qt::BusyCursor));

	//Generate and read grib dumps
	//createPlot();

	//emit itemSelected(currentItemNo_);

	//loadItems(currentItemNo_);

	//Preview plot
	//createPlot();

	//Log
	//updateLogOutput();

	//Restore cursor
	QApplication::restoreOverrideCursor();
}

void MvQObstatCurveToolBase::updateFileInfoLabel()
{
	fileInfoLabel_->setObstatScatterTextLabel(data_->fileName(),data_->itemNum());
}


void MvQObstatCurveToolBase::createPlot()
{	
	//currentItemNo_

	string itemId="1";

	QList<MvQObstatCurveItem*> items;

	data_->getStatItem(items);

	QVector<float> meanX,meanY;
	QStringList tmpStr,tmpStr1;

	
	QString yStr,yLabelStr;
	QStringList rmsStr,avgStr;
	float  yMin, yMax;
	float avgMin=999999, avgMax=-999999;
	float rmsMin=999999, rmsMax=-999999;

	foreach(MvQObstatCurveItem* item, items)
	{		
		tmpStr.clear();
		int yPos=0;
		foreach(float v,item->data["y"])
		{
			tmpStr << QString::number(v);
			tmpStr1 << QString::number(yPos);
			yPos++;
		}
		yLabelStr=tmpStr.join("/");
		yStr=tmpStr1.join("/");

		yMin=0;
		yMax=yPos-1;
	
		tmpStr.clear();
		foreach(float v,item->data["rms"])
		{
			tmpStr << QString::number(v);
			if(rmsMin > v)
				rmsMin=v;
			if(rmsMax< v)
				rmsMax=v;

		}
		qDebug() << "minmax:" << rmsMin << rmsMax;

		rmsStr << tmpStr.join("/");
		

		tmpStr.clear();
		foreach(float v,item->data["avg"])
		{
			tmpStr << QString::number(v);
			if(avgMin > v)
				avgMin=v;
			if(avgMax< v)
				avgMax=v;
		}
		avgStr << tmpStr.join("/");
	}

	QString paramX, paramY;
	float  startX, endX, startY, endY;


	/*if(swapAxes_ == false)
	{		
		paramX=item->titleX();
		paramY=item->titleY();
		startX=item->startX();
		endX=item->endX();
		startY=item->startY();
		endY=item->endY();

	}
	else
	{
		paramX=item->titleY();
		paramY=item->titleX();
		startX=item->startY();
		endX=item->endY();
		startY=item->startX();
		endY=item->endX();
	}*/

	QString resFile=QString::fromStdString(marstmp());

	QString cmd="/var/tmp/cgr/PERFORCE/metview_4/bin/magmlx " + magmlScript_ +
	" -blockTitle=\"" + "" + "\"" +
	" -startDate=" + "" +
	" -endDate=" + "" +
	" -area=" + "" +
	" -flag=" + "" +
	" -x_param=\"" + paramX.simplified() + "\"" +
	" -y_param=\"" + paramY.simplified() + "\"" +
	" -start_x=" +  QString::number(rmsMin)+
	" -end_x=" + QString::number(rmsMax) +
	" -start_y=" + QString::number(yMin) +
	" -end_y=" + QString::number(yMax) +
	" -y_label=" + yLabelStr +
	" -p1c1_x_lst=" + rmsStr[0] +
	" -p1c1_y_lst=" + yStr +
	" -p1c2_x_lst=" + rmsStr[1] +
	" -p1c2_y_lst=" + yStr +
	" -x_min=" + QString::number(items[0]->min) +
	" -x_max=" + QString::number(items[0]->max) +
	" -y_min=" + "100" +
	" -y_max=" + "1000" +
	" -source_num=" + "" +
	" -source_num=" + "" +
 	" -path=" + resFile;

	qDebug()  << cmd;
	
	system(cmd.toStdString().c_str());

	QString resFileWithSuffix=resFile + ".png";

	QPixmap pixmap(resFileWithSuffix,"PNG");
 	plotWidget_->setPixmap(pixmap);
	plotWidget_->adjustSize();

	QFile fRes(resFile);
	if(!fRes.remove())
	{
		qDebug() << "MvQObstatScatterToolBase::createPlot() ---> Res file" << resFile << "could not be deleted!";
	}

	QFile fResSuffix(resFileWithSuffix);
	if(!fResSuffix.remove())
	{
		qDebug() << "MvQObstatScatterToolBase::createPlot() ---> Res file" << resFileWithSuffix << "could not be deleted!";
	}

}
