/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

// 	$Id$	


#ifndef lint
static char vcid[] = "$Id$";
#endif /* lint */


#include "assembly.H"

// MIRA uses sleep(3)
#include <unistd.h>

// MIRA uses stat (2)
#include <sys/types.h>
#include <sys/stat.h>

// for perror (3C) and some routines that use the quick C I/O
//  instead of the dog slow C++ streams
#include <cstdio>
//we use errno
#include <errno.h>

// for getting username etc
#include <pwd.h>

// for time measurements
#include <sys/times.h>
#include <sys/time.h>
#include <limits.h>
#include <unistd.h>

// BOOST
#include <boost/algorithm/string.hpp> 

// for checking time restrictions
#include <time.h>
#include "mira/timerestrict.H"

#include "errorhandling/errorhandling.H"
#include "mira/dynamic.H"
#include "mira/ads.H"
#include "util/progressindic.H"
#include "util/dptools.H"
#include "util/misc.H"
#include "caf/caf.H"
#include "mira/structs.H"
#include "mira/contig.H"


#if 0
#include <valgrind/memcheck.h>
#define VALGRIND_LEAKCHECK
#endif

// cs1 for normal clocking ('user compatible' as is does not disturb)
//  cs2 for extensive clocking output, more for analysis of MIRA behaviour
//  during development

#ifndef PUBLICQUIET
#define CLOCK_STEPS2
#endif
#define CLOCK_STEPS2


#if DEVELOPMENTVERSION != 0 
//#define TRACKMEMUSAGE 1
#define TRACKMEMUSAGE 0
#else
#define TRACKMEMUSAGE 0
#endif


//#define FUNCSTART(bla)  static const char * THISFUNC = bla"  "; {cout << THISFUNC << "enter.\n"; cout.flush();}
//#define FUNCTRACE(bla) { cout << THISFUNC << bla; cout.flush();}
//#define FUNCEND() {cout << THISFUNC << "exit.\n"; cout.flush();}
//
//#define CEBUG(bla)
//#define CEBUGF(bla)



void Assembly::test()
{
  if(AS_readpool[2].getLSClipoff()>0){
    Read::setCoutType(Read::AS_TEXT);
    cout << AS_readpool[2];
    exit(10);
  }

//  uint32 s=5000000;
//  AS_readpool.reserve(s);
//  for(uint32 i=0;i<s;i++) AS_readpool.addNewEmptyRead();
//  AS_permanent_overlap_bans.resize(s);
//  AS_istroublemaker.resize(s,0);
//  loadAlignmentsFromFile(-1,"miratmp","", ".ads_pass");
}


void Assembly::foolCompiler()
{
#include "stdinc/foolcompiler.C"
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

Assembly::Assembly(vector<MIRAParameters> & params, bool resumeassembly): AS_readpool(&params)
{
  FUNCSTART("Assembly::Assembly(MIRAParameters * params)");

  AS_miraparams=params;
  AS_resumeasembly=resumeassembly;

  init();

#ifdef TIMERESTRICTED
  cout << "Compiled on " << __DATE__ << ". Will run until "<< TR_OUT_MAXDAY << "." << TR_OUT_MAXMONTH << "." << TR_OUT_MAXYEAR << " (dd.mm.yyyy)\n";
#endif

  setExtendedLog(AS_miraparams[0].getSpecialParams().mi_extended_log);

  // honour user supplied working directory
  // will probably already be done by the parameter parsing already, but just in case
  //  this was done programatically ...
  if(!AS_miraparams[0].getDirectoryParams().dir_cwd.empty()){
    int ret=chdir(AS_miraparams[0].getDirectoryParams().dir_cwd.c_str());
    if(ret){
      MIRANOTIFY(Notify::FATAL,"Changing working directory to '" << AS_miraparams[0].getDirectoryParams().dir_cwd << "' failed, system message is: " << strerror(errno) << endl);
    }
  }
  // purge the directories if we're not resuming
  ensureStandardDirectories(!resumeassembly);

  setContigBuiltCallback();

  AS_assemblyinfo.setLargeContigSize(500);

  AS_systemmemory=grepMemSizeFromProcFS("/proc/meminfo","MemTotal:");

  //makeTmpDir();

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Assembly::init()
{
  AS_steps.resize(ASNUMOFSTEPS);

  for(int8 i=ASNUMOFSTEPS-1; i>=0;i--){
    AS_steps[i]=0;
  }

  // initialise some other variables

  zeroVars();
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Assembly::zeroVars()
{
  AS_num_reads_valid=0;
  AS_num_reads_too_small=0;
  AS_numADSFacts_fromalignments=0;
  AS_numADSFacts_fromshreds=0;
  AS_seqtypespresent.clear();
  AS_454dosimpleedit=false;
  AS_needsskimfornastyrepeats=false;

  AS_doneskimchimera=false;

#ifdef TIMERESTRICTED
  AS_timesup=false;
#endif

  AS_shouldrun_nfs_check=true;

  //TODO: rest
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

Assembly::~Assembly()
{
  FUNCSTART("Assembly::~Assembly()");

  dumpMemInfo();

  cout << "Dynamic allocs: " << Dynamic::DYN_alloccount << endl;
  cout << "Align allocs: " << Align::AL_alloccount << endl;

  discard();
  
  if(AS_miraparams[0].getAssemblyParams().as_output_removetmpdir){
    string system_rmdir = static_cast<string>("rm -rf ")+AS_miraparams[0].getDirectoryParams().dir_tmp;
    int tmp=system(system_rmdir.c_str()); 
    // don't complain about unused variable
    (void) tmp;
  }

  // TODO: scandir on result and remove all if no results?

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

// TODO not complete

void Assembly::discard()
{
  FUNCSTART("Assembly::discard()");

  AS_readpool.discard();
  AS_contigs.clear();
  AS_bbcontigs.clear();
  //AS_ok_for_assembly.clear();

  nukeSTLContainer(AS_adsfacts);
  nukeSTLContainer(AS_confirmed_edges);

  nukeSTLContainer(AS_used_ids);
  nukeSTLContainer(AS_multicopies);
  nukeSTLContainer(AS_hasmcoverlaps);
  nukeSTLContainer(AS_maxcoveragereached);
  nukeSTLContainer(AS_steps);
  nukeSTLContainer(AS_istroublemaker);
  nukeSTLContainer(AS_allrmbsok);
  nukeSTLContainer(AS_probablermbsnotok);
  nukeSTLContainer(AS_weakrmbsnotok);

  AS_permanent_overlap_bans.nuke();

  nukeSTLContainer(AS_readhitmiss);
  nukeSTLContainer(AS_readhmcovered);
  nukeSTLContainer(AS_count_rhm);
  nukeSTLContainer(AS_clipleft);
  nukeSTLContainer(AS_clipright);

  init();

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

// TODO not complete

void Assembly::dmi_dumpALine(ostream & ostr, const char * desc, size_t numelem, size_t bytes_size, size_t freecapacity, size_t lostbyalign)
{

  ostr << setw(30) << desc
       << setw(10) << numelem;

  {
    ostringstream ostrstr;
    byteToHumanReadableSize(static_cast<double>(bytes_size), ostrstr);
    ostr << setw(12) << ostrstr.str();
  }
  {
    ostringstream ostrstr;
    byteToHumanReadableSize(static_cast<double>(freecapacity), ostrstr);
    ostr << setw(12) << ostrstr.str();
  }
  {
    ostringstream ostrstr;
    byteToHumanReadableSize(static_cast<double>(lostbyalign), ostrstr);
    ostr << setw(12) << ostrstr.str();
  }

  ostr << '\n';
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::dumpMemInfo()
{
  FUNCSTART("Assembly::dumpMemInfo()");

  cout << "\n\n========================== Memory self assessment ==============================\n";

  size_t bytes_size = 0;
  size_t tmp_bytes_size=0;

  size_t numelem=0;
  size_t tmp_numelem=0;

  size_t freecapacity=0;
  size_t tmp_freecapacity=0;

  size_t lostbyalign=0;
  size_t tmp_lostbyalign=0;

  // we currently do not use these
  (void) numelem;
  (void) freecapacity;
  (void) lostbyalign;


  if(sizeof(size_t) == sizeof(int32)){
    cout << "Running in 32 bit mode.\n\n";
  }else{
    cout << "Running in 64 bit mode.\n\n";
  }

  dumpFile("/proc/meminfo",cout);
  cout << '\n';
  dumpFile("/proc/self/status",cout);

  cout << "\nInformation on current assembly object:\n\n";
  
  cout << "AS_readpool: " << AS_readpool.size() << " reads.\n";
  cout << "AS_contigs: " << AS_contigs.size() << " contigs.\n";
  cout << "AS_bbcontigs: " << AS_bbcontigs.size() << " contigs.\n";

  bytes_size+=AS_readpool.estimateMemoryUsage();
  cout << "Mem used for reads: " << bytes_size  << " (";
  byteToHumanReadableSize(static_cast<double>(bytes_size), cout);
  cout << ")\n\nMemory used in assembly structures:\n"
       << setw(52) << "Eff. Size" << setw(12) << "Free cap." << setw(12) << "LostByAlign" << endl;
  

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_writtenskimhitsperid,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_writtenskimhitsperid: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_skim_edges,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_skim_edges: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_adsfacts,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_adsfacts: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_confirmed_edges,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_confirmed_edges: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_permanent_overlap_bans,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_permanent_overlap_bans: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_readhitmiss,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_readhitmiss: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_readhmcovered,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_readhmcovered: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_count_rhm,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_count_rhm: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_clipleft,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_clipleft: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_clipright,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_clipright: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_used_ids,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_used_ids: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_multicopies,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_multicopies: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_hasmcoverlaps,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_hasmcoverlaps: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_maxcoveragereached,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_maxcoveragereached: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_coverageperseqtype,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);   
  dmi_dumpALine(cout,"AS_coverageperseqtype: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);   
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_istroublemaker,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_istroublemaker: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_isdebris,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_isdebris: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_needalloverlaps,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_needalloverlaps: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_readsforrepeatresolve,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_readsforrepeatresolve: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_allrmbsok,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_allrmbsok: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_probablermbsnotok,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_probablermbsnotok: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_weakrmbsnotok,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_weakrmbsnotok: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_readmaytakeskim,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_readmaytakeskim: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_skimstaken,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_skimstaken: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_numskimoverlaps,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_numskimoverlaps: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_numleftextendskims,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_numleftextendskims: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_numrightextendskims,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_rightextendskims: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_skimleftextendratio,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_skimleftextendratio: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_skimrightextendratio,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_skimrightextendratio: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;

  tmp_bytes_size=estimateMemoryUsageOfContainer(AS_usedtmpfiles,true,tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  dmi_dumpALine(cout,"AS_usedtmpfiles: ",tmp_numelem,tmp_bytes_size,tmp_freecapacity,tmp_lostbyalign);
  bytes_size+=tmp_bytes_size;


  cout << "Total: " << bytes_size << " (";
  byteToHumanReadableSize(static_cast<double>(bytes_size), cout);
  cout << ")";
  cout << "\n\n================================================================================\n";

  FUNCEND();
}

//void Assembly::dmiAddBytesWithCapacity(size_t & bcap, size_t & bsize)
//{
//}

/*************************************************************************
 *
 * go through list of previous files and delete old with same base name
 *  but different file name
 *
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush(); }

uint32 Assembly::cleanupOldFile(const string & basename, const string & filename)
{
  FUNCSTART("uint32 Assembly::cleanupOldFile(const string & basename, const string & filename)");

  CEBUG("\nCOF: ###" << basename << "### and ###"<<filename<<"###\n");

  uint32 numdeleted=0;
  list<usedtmpfiles_t>::iterator ulfI=AS_usedtmpfiles.begin();
  while(ulfI != AS_usedtmpfiles.end()){
    //cout << "cOF: " << ulfI->basename << '\t' << ulfI->filename << endl;
    if(ulfI->basename == basename
       && ulfI->filename != filename) {
      numdeleted++;

      // the following two checks seem pointless as normally the list would
      //  arrive at an end sometime
      // however, I've seen one case where this loop became endless,
      //  hence this foolguard
      if(numdeleted>100) {
	cerr << "\n\nOUCH! something strange ... tried more than 100 deletes of " << basename << " ... list size is " << AS_usedtmpfiles.size() << '\n';
      }
      if(numdeleted>120) {
	cerr << "\n\nOUCH! something wierd ... tried more than 120 deletes of " << basename << " ... list size is " << AS_usedtmpfiles.size() << '\n';
	cerr << "We'll stop that here.\n";
	return numdeleted;
      }

      /* 

      string system_rmfile = static_cast<string>("rm -f ")+ulfI->filename;

      //cout << "RMFILE: " << system_rmfile << endl;

      // on my system at home, the delete sometime fails for no apparent
      //  reason at all (file can be removed later by hand without
      //  problem etc.)
      // as I've also had this report once from another system, I'll put
      //  the system call in a loop with 10 tries, always waiting a
      //  second between each try

      bool filestillthere=true;
      for(uint32 i=0; i<10 && filestillthere; i++){
	filestillthere=system(system_rmfile.c_str());
	if(filestillthere) sleep(1);
      }
      */

      // Let's try something different ... using remove() and the rm -rf system call
      int filestillthere=1;
      string system_rmfile = static_cast<string>("rm -f ")+ulfI->filename;
      for(uint32 i=0; i<10 && filestillthere; i++){
	CEBUG("COF: removing with remove: " << ulfI->filename.c_str() << endl);
	filestillthere=remove(ulfI->filename.c_str());
	if(filestillthere) {
	  int errsv=errno;
	  cout << "Errno: " << errsv << endl;
	  filestillthere=system(system_rmfile.c_str());
	  if(filestillthere) sleep(1);
	}
      }

      if(filestillthere) {
	cerr << "WARNING: Could not delete old file " + ulfI->filename
	     << "\nThis can have a number of different reasons, none of them"
	     << "\nwarranting an abort, but this is strange anyway.\n\n";
	ulfI++;
      }else{
	ulfI=AS_usedtmpfiles.erase(ulfI);
      }
    }else{
      ulfI++;
    }
  }

  FUNCEND();
  return numdeleted;
}
//#define CEBUG(bla)


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

string Assembly::buildFileName(int32 version, const string & prefix, const string & postfix, const string & basename, const string & suffix, const string & dirname, bool removeold)
{
  FUNCSTART("string Assembly::buildFileName(int32 version, const string & prefix, const string & postfix, const string & basename, const string & suffix, const string & dirname, bool removeold)");

  ostringstream ostr;

  if(version>=0){
    ostr << AS_miraparams[0].getDirectoryParams().dir_tmp << "/";
  } else if(!dirname.empty()){
    ostr << dirname << "/";
  }

  ostr << prefix << basename << postfix ;

  if(version>=0){
    ostr << "." << version;
  }
  ostr << suffix;

  string filename=ostr.str();
  string newbasename(basename+suffix);

  if(removeold && AS_miraparams[0].getAssemblyParams().as_output_removerollovertmps) {
    cleanupOldFile(newbasename,filename);
  }

  bool mustadd=true;
  list<usedtmpfiles_t>::const_iterator ulfI=AS_usedtmpfiles.begin();
  for(;ulfI != AS_usedtmpfiles.end(); ulfI++){
    if(ulfI->basename == newbasename
       && ulfI->filename == filename) {
      mustadd=false;
      break;
    }
  }

  if(mustadd){
    usedtmpfiles_t ulf;

    AS_usedtmpfiles.push_back(ulf);
    AS_usedtmpfiles.back().basename=newbasename;
    AS_usedtmpfiles.back().filename=filename;
  }

  //{
  //  cout << "IVEGOT: " << AS_usedtmpfiles.size() << endl;
  //  //cout << "IVEGOT: " << filename << "IVEGOT" << endl;
  //  list<usedtmpfiles_t>::const_iterator ulfI=AS_usedtmpfiles.begin();
  //  for(;ulfI != AS_usedtmpfiles.end(); ulfI++){
  //    cout << "ulf: " << ulfI->basename << '\t' << ulfI->filename << endl;
  //  }
  //}

  FUNCEND();
  return filename;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::ensureStandardDirectories(bool purge){
  FUNCSTART("void Assembly::ensureStandardDirectories(bool purge)");

  if(ensureDirectory(AS_miraparams[0].getDirectoryParams().dir_top, purge)
     || ensureDirectory(AS_miraparams[0].getDirectoryParams().dir_tmp, purge)
     || ensureDirectory(AS_miraparams[0].getDirectoryParams().dir_results, purge)
     || ensureDirectory(AS_miraparams[0].getDirectoryParams().dir_info, purge)
     || ensureDirectory(AS_miraparams[0].getDirectoryParams().dir_checkpoint, purge)){

    MIRANOTIFY(Notify::FATAL, "Could not make sure that a needed directory exists, aborting MIRA.");
  }

  if(AS_miraparams[0].getDirectoryParams().dir_tmp_symlink.size()){
    struct stat st;
    int rc=lstat(AS_miraparams[0].getDirectoryParams().dir_tmp_symlink.c_str(),&st);

    if(rc){
      string system_lntmp = 
	static_cast<string>("ln -s ")
	+AS_miraparams[0].getDirectoryParams().dir_tmp
	+" "
	+AS_miraparams[0].getDirectoryParams().dir_tmp_symlink;
      
      //cout << "LINKING: " << system_lntmp << endl;
      
      if(system(system_lntmp.c_str())) {
	perror((static_cast<string>("Could not link "+AS_miraparams[0].getDirectoryParams().dir_tmp_symlink+" to "+AS_miraparams[0].getDirectoryParams().dir_tmp)).c_str()) ;
      } 
    }
  }

  checkForNFSMountOnTmpDir();

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::checkForNFSMountOnTmpDir()
{
  FUNCSTART("void Assembly::checkForNFSMountOnTmpDir()");

  if(!AS_shouldrun_nfs_check) return;

  string cmd="stat -f -L -c %T " + AS_miraparams[0].getDirectoryParams().dir_tmp;
  string result;

  cout << '\n';
  if(checkRunabilityOfCommand(cmd) && getSTDOUTFromCommand(cmd,result)){
    //cout << "result from NFS check: " << result << endl;
    boost::to_lower(result);
    if(boost::find_first(result,"nfs")){
      cout << "\n\n\n\n\nWARNING WARNING WARNING!\n\n"
	"It looks like the tmp directory is on a NFS (Network File System) mount. This\n"
	"will slow down MIRA *considerably* ... by about a factor of 10!\n"
	"If you don't want that, you have three possibilities:\n\n"
	"1) RECOMMENDED! Use -DI:trt to redirect the tmp directory somewhere else on a\n"
	"   local disk or even SSD.\n"
	"2) POSSIBLE: put the whole project somewhere else and restart MIRA.\n"
	"3) ABSOLUTELY NOT RECOMMENDED AT ALL: use \"-MI:sonfs=no\" to tell MIRA not\n"
	"   to stop when it finds the tmp directory on NFS.\n\n"
	"If you do not know what NFS is and which directory to use in \"-DI:trt\", ask\n"
	"your local system administrator to guide you.\n\n";

      if(AS_miraparams[0].getSpecialParams().mi_stop_on_nfs){
	MIRANOTIFY(Notify::FATAL,"Tmp directory is on a NFS mount ... but we don't want that.");
      }
    }else{
      cout << "Tmp directory is not on a NFS mount, good.\n\n";
    }
  }else{
    cout << "Could not perform NFS check for directory " 
	 <<  AS_miraparams[0].getDirectoryParams().dir_tmp
	 << "\nMake sure it is *NOT* on a NFS mount or else MIRA will run *very* slowly.\n"
      "\nFor a check to run smoothly, please make sure the Unix 'stat' command is available"
      "\nand understands the following call: " << cmd 
	 << "\nNote that this might not be the case on BSD systems and derivatives (e.g. MacOSX),"
      "\nin which case no check for NFS is possible.\n\n";
  }

  AS_shouldrun_nfs_check=false;

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::preassembleTasks(bool usereadextension, bool clipvectorleftovers)
{
  assembly_parameters const & as_fixparams= AS_miraparams[0].getAssemblyParams();

  if(as_fixparams.as_clip_proposeendclips){
    string tmpfname;
    tmpfname=buildFileName(0,"","", 
			  as_fixparams.as_tmpf_clippings,
			  ".txt");
      
    // doing it twice catches a few outliers missed the first time
    string logprefix="proposed cutback 1a: ";
    uint64 numclipped=performNewProposedCutbackClips(tmpfname,logprefix);
    if(numclipped>0){
      logprefix="proposed cutback 1b: ";
      performNewProposedCutbackClips(tmpfname,logprefix);
    }else{
      cout << "No bases clipped in first pec round, skipping second round.\n";
    }
    dumpSomeStatistics();
  }

  //performSnapshot(0);

  //performHashEditing();

#if TRACKMEMUSAGE 
  cout << "\ndmi pre 00\n";
  dumpMemInfo();
#endif
    
  // find out whether there are SRMr or CRMr tags that need attention
  bool initialrepeatmarkerspresent=false;
  for(uint32 i=0; i< AS_readpool.size() && !initialrepeatmarkerspresent; i++){
    if(AS_readpool[i].isUsedInAssembly()){
      if(AS_readpool[i].hasTag(Read::REA_tagentry_idSRMr)){
	initialrepeatmarkerspresent=true;
      }else if(AS_readpool[i].hasTag(Read::REA_tagentry_idCRMr)){
	initialrepeatmarkerspresent=true;
      }
    }
  }

  if(initialrepeatmarkerspresent){
    performHashAnalysis(0,"","");
    
    //performSnapshot(0);

#if TRACKMEMUSAGE 
    cout << "\ndmi pre 10\n";
    dumpMemInfo();
#endif

    cout << "Repeat markers found in data loaded, performing an initial step of marker propagation.\nInitial marker propagation 1/2\n";
    findPossibleOverlaps(0, "", "_initialRMpropagation");
      
#if TRACKMEMUSAGE 
    cout << "\ndmi pre 20\n";
    dumpMemInfo();
#endif
      
    AS_steps[ASADSLISTOK]=0;
    makeAlignments(Assembly::ma_needSRMrOrTwoCRMr, true, false, 0, "", "", "initialRMpropagation1");
#if TRACKMEMUSAGE 
    cout << "\ndmi pre 30\n";
    dumpMemInfo();
#endif
      
    AS_steps[ASADSLISTOK]=0;
    cout << "Initial marker propagation 2/2\n";
    makeAlignments(Assembly::ma_needSRMrOrTwoCRMr, true, false, 0, "", "", "initialRMpropagation2");
    AS_steps[ASADSLISTOK]=0;
  }
    
#if TRACKMEMUSAGE 
  cout << "\ndmi pre 50\n";
  dumpMemInfo();
#endif

  //if(AS_454dosimpleedit) editSimple454Overcalls(0);
 
  if(clipvectorleftovers
	 || (usereadextension && as_fixparams.as_readextension_firstpassnum == 0)){
    cout << "Pre-assembly alignment search for read extension and / or vector clipping:\n";

    findPossibleOverlaps(0, "", "_preassembly");
    
#if TRACKMEMUSAGE 
    cout << "\ndmi pre 61\n";
    dumpMemInfo();
#endif

    // do not use the 100% trans rule for read extension and clipping!
    // needed vectors would not get filled
    makeAlignments(Assembly::ma_takeall, false, false, 0, "", "_preassembly1");
      
#if TRACKMEMUSAGE 
    cout << "\ndmi pre 62a\n";
    dumpMemInfo();
#endif
      
    loadAlignmentsFromFile(0, "", "_preassembly1");
      
#if TRACKMEMUSAGE 
    cout << "\ndmi pre 62b\n";
    dumpMemInfo();
#endif
      
    if(usereadextension) {
      cout << "Pre-assembly read extension:\n";
      extendADS(0, "", "_preassembly1");
      AS_needsskimfornastyrepeats=true;
#if TRACKMEMUSAGE 
      cout << "\ndmi pre 62c\n";
      dumpMemInfo();
#endif
    }
    if(clipvectorleftovers) {
      cout << "Pre-assembly vector clipping\n";
      performSeqVectorClippings();
      AS_needsskimfornastyrepeats=true;
#if TRACKMEMUSAGE 
      cout << "\ndmi pre 62d\n";
      dumpMemInfo();
#endif
    }
      
    dumpSomeStatistics();
      
    // we need to throw away all permbans that might have appeared in
    //  this pre-assembly sequence massage
      
    AS_permanent_overlap_bans.nuke();
    AS_permanent_overlap_bans.resize(AS_readpool.size());
      
#if TRACKMEMUSAGE 
    cout << "\ndmi pre 63\n";
    dumpMemInfo();
#endif
      
    performHashAnalysis(0, "", "_preassembly2");
      
#if TRACKMEMUSAGE 
    cout << "\ndmi pre 64\n";
    dumpMemInfo();
#endif
      
    if(as_fixparams.as_clip_proposeendclips){
      string tmpfname;
      tmpfname=buildFileName(0,"","", 
			    as_fixparams.as_tmpf_clippings,
			    ".txt");
      string logprefix="proposed cutback: ";
	
      performNewProposedCutbackClips(tmpfname,logprefix);
      dumpSomeStatistics();
#if TRACKMEMUSAGE 
      cout << "\ndmi pre 64b\n";
      dumpMemInfo();
#endif
    }
      
    //nukeSTLContainer(AS_adsfacts);
    //nukeSTLContainer(AS_confirmed_edges);
    AS_adsfacts.clear();
    AS_confirmed_edges.clear();
      
#if TRACKMEMUSAGE 
    cout << "\ndmi pre 65a\n";
    dumpMemInfo();
#endif
//    findPossibleOverlaps(0, "", "_preassembly2");
//#if TRACKMEMUSAGE 
//    cout << "\ndmi pre 65b\n";
//    dumpMemInfo();
//#endif
  }
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

bool Assembly::checkTerminationRequest()
{
  struct stat st;
  string fname=AS_miraparams[0].getDirectoryParams().dir_checkpoint+"/terminate";
  int rc=stat(fname.c_str(),&st);
  if(rc==0) {
    string command="mv "
      +AS_miraparams[0].getDirectoryParams().dir_checkpoint+"/terminate"
      +" "
      +AS_miraparams[0].getDirectoryParams().dir_checkpoint+"/terminate_acknowledged";
    system(command.c_str());
    return true;
  }
  return false;
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::assemble()
{
  FUNCSTART("Assembly::assemble()");

  uint32 startpass=1;

  ensureStandardDirectories(false);

  assembly_parameters const & as_fixparams= AS_miraparams[0].getAssemblyParams();
  //directory_parameters const & dir_params= AS_miraparams->getDirectoryParams();
  edit_parameters const & ed_params= AS_miraparams[0].getEditParams();

  if (as_fixparams.as_numrmbbreakloops <1) {
    AS_miraparams[0].setAssemblyNumRMBBreakLoops(1);
    cout << "Number of RMB break loops <1, setting to 1\n";
  }
  if (as_fixparams.as_loadbackbone
      && as_fixparams.as_startbackboneusage_inpass > static_cast<int32>(as_fixparams.as_numpasses)) {
    AS_miraparams[0].setAssemblyStartBackboneUsageInPass(as_fixparams.as_numpasses);
    cout << "Start of backbone usage > number of passes, correcting start to " << as_fixparams.as_numpasses << endl;
  }

  if(AS_systemmemory==0 && as_fixparams.as_automemmanagement){
    cout << "Can't find info about system or process memory, switching off automatic"
      "\nmemory management.\n";
    AS_miraparams[0].getNonConstAssemblyParams().as_automemmanagement=false;
  }

  AS_assemblyinfo.setLargeContigSize(AS_miraparams[0].getSpecialParams().mi_as_largecontigsize);
  AS_assemblyinfo.setLargeContigSizeForStats(AS_miraparams[0].getSpecialParams().mi_as_largecontigsize4stats);


  // find out whether we have 454 reads in the data set
  // also find out whether there are SRMr or CRMr tags that need attention
  AS_seqtypespresent.clear();
  AS_seqtypespresent.resize(Read::SEQTYPE_END,false);
  AS_454dosimpleedit=false;
  for(uint32 i=0; i< AS_readpool.size(); i++){
    if(AS_readpool[i].isUsedInAssembly()){
      if(!AS_readpool[i].isBackbone() && !AS_readpool[i].isRail()){
	AS_seqtypespresent[AS_readpool[i].getSequencingType()]=true;
      }
      if(AS_readpool[i].getSequencingType()==Read::SEQTYPE_454GS20){
	AS_454dosimpleedit=true;
      }
    }
  }

  if((AS_seqtypespresent[Read::SEQTYPE_454GS20]
      || AS_seqtypespresent[Read::SEQTYPE_SOLEXA]
      || AS_seqtypespresent[Read::SEQTYPE_IONTORRENT])
     && (as_fixparams.as_output_gap4da
	 || as_fixparams.as_output_tmp_gap4da
	 || as_fixparams.as_output_exttmp_gap4da)){
    cout << "454 type data present, switching off GAP4DA type output results (you *DO NOT* want millions of files in a directory, really.)\n";
    MIRAParameters::parseQuickmode("-OUT:org=no:otg=no:oetg=no", 
				   "", AS_miraparams);
  }

  // allocate or reserve memory that is quite static from the size,
  //  reducing memory fragmentations at least a bit
  {
#if TRACKMEMUSAGE 
    cout << "\ndmi as_init 00\n";
    dumpMemInfo();
#endif
    AS_multicopies.reserve(AS_readpool.size());  // do NOT init multicopies!
    AS_hasmcoverlaps.reserve(AS_readpool.size()); // does not need init

    AS_permanent_overlap_bans.nuke();
    AS_permanent_overlap_bans.resize(AS_readpool.size());

    AS_used_ids.resize(AS_readpool.size(),0);
    AS_clipleft.resize(AS_readpool.size(),0);
    AS_clipright.resize(AS_readpool.size(),0);
    AS_istroublemaker.resize(AS_readpool.size(),0);
    AS_isdebris.resize(AS_readpool.size(),0);
    AS_needalloverlaps.resize(AS_readpool.size(),false);
    AS_maxcoveragereached.resize(AS_readpool.size(),0);

    //AS_allowquickoverlap.resize(AS_readpool.size(),false);

    // these 3 only temp filled, but block anyway to reduce fragmentation
    AS_allrmbsok.reserve(AS_readpool.size());
    AS_probablermbsnotok.reserve(AS_readpool.size());
    AS_weakrmbsnotok.reserve(AS_readpool.size());
#if TRACKMEMUSAGE 
    cout << "\ndmi as_init 10\n";
    dumpMemInfo();
#endif
  }

  AS_needsskimfornastyrepeats=true;

  bool clipvectorleftovers=false;
  bool usereadextension=false;
  for(uint32 i=0; i<Read::SEQTYPE_END; i++){
    if(AS_seqtypespresent[i]){
      if(AS_miraparams[i].getAssemblyParams().as_clip_possible_vectors){
	clipvectorleftovers=true;
      }
      if(AS_miraparams[i].getAssemblyParams().as_use_read_extension){
	usereadextension=true;
      }
    }
  }

  //performSnapshot(0);

  if(AS_resumeasembly){
    // startpass=...;
    //performHashAnalysis(startpass,"","");
  }else{
    preassembleTasks(usereadextension,clipvectorleftovers);
  }

  EDITParameters eparams;
#ifdef MIRA_HAS_EDIT
  //  eparams.setDoEval();
  eparams.setVerbose(0);
  eparams.setShowProgress(true);
#else
#endif

  uint32 actpass=startpass;
  if(as_fixparams.as_numpasses==0){
    // 0 passes? Then the user does not want an assembly
    // Just calculate some statistics and the hash analysis 

    cout << "You have selected to have 0 passes on the assembly, therefore"
      "\njust running a couple of hash statistics and info for read repeats.\n";
    dumpSomeStatistics();
    performHashAnalysis(0, "", "_pass");
  }else{
    bool rerunSKIM=true;
    for(; actpass<=as_fixparams.as_numpasses; actpass++){

      //dumpMemInfo();
      dumpFile("/proc/self/status",cout);
      cout << "\n\nPass: " << actpass << " / " << as_fixparams.as_numpasses << endl;

#ifdef VALGRIND_LEAKCHECK
      cout << "\n==MEMTRACK1 debugging start\n";
      dumpMemInfo();
      VALGRIND_DO_LEAK_CHECK
	cout << "\n==MEMTRACK1 debugging end\n";

      if(actpass==20) {
	cout << "\n==MEMTRACK exiting\n";
	dumpMemInfo();
	exit(0);
      }
#endif

      performSnapshot(actpass);

      AS_contigs.clear();

#ifdef MIRA_HAS_EDIT
      if(actpass<as_fixparams.as_numpasses) {
	// eventually strict editing in first passes
	//  eparams.setStrictEvaluation(false);
	eparams.setStrictEvaluation(true);
	eparams.setConfirmationThreshold(static_cast<float>(0.8));
      } else {
	// eventually lazy editing in last passes
	eparams.setStrictEvaluation(ed_params.ed_strict_editing_mode);
	eparams.setConfirmationThreshold(static_cast<float>(ed_params.ed_confirmation_threshold/100.0));
      }
#endif

      if(as_fixparams.as_skimeachpass
	 || rerunSKIM                  // forced by new SRMs in previous pass
	 || usereadextension
	 || clipvectorleftovers
	 || AS_seqtypespresent[Read::SEQTYPE_SOLEXA]){

	dumpSomeStatistics();
      
#if TRACKMEMUSAGE 
	cout << "\ndmi 20\n";
	dumpMemInfo();
#endif
	// as something might have changed in the reads, redo
	//  the hash analysis
	performHashAnalysis(actpass, "", "_pass");
      
#if TRACKMEMUSAGE 
	cout << "\ndmi 30\n";
	dumpMemInfo();
#endif

//      for(uint32 i=0; i<AS_readpool.size();++i){
//	if(AS_readpool[i].isBackbone()) continue;
//	if(AS_readpool[i].isRail()) continue;
//	if(1) {
//	//if(i==16226603) {
//	  Read::setCoutType(Read::AS_TEXT);
//	  cout << "########\n" << AS_readpool[i] << endl;
//	  continue;
//	}
//	AS_readpool[i].setValidData(false);
//      }

	findPossibleOverlaps(actpass, "", "_pass");
	//if(AS_454dosimpleedit) editSimple454Overcalls(actpass);
      }

      rerunSKIM=false;

      // TODO: setting that here is messy
      AS_steps[ASVECTORSCLIPPED]=0;

#if TRACKMEMUSAGE 
      cout << "\ndmi 40\n";
      dumpMemInfo();
#endif

      makeAlignments(Assembly::ma_takeall, false, true, actpass, "", "_pass");

#if TRACKMEMUSAGE 
      cout << "\ndmi 50\n";
      dumpMemInfo();
#endif

      loadAlignmentsFromFile(actpass, "", "_pass");

#if TRACKMEMUSAGE 
      cout << "\ndmi 60\n";
      dumpMemInfo();
#endif

      // count all SRMr tags in reads
      // idea: after building contigs, run the repeat resolver with
      //  all reads that have new tags

      vector<uint32> xrmrcount(AS_readpool.size(),0);
      for(uint32 rnr=0; rnr<AS_readpool.size(); rnr++){
	xrmrcount[rnr]=AS_readpool.getRead(rnr).countTags(Read::REA_tagentry_idSRMr);
      }

#if TRACKMEMUSAGE 
      cout << "\ndmi 70\n";
      dumpMemInfo();
#endif

#ifdef VALGRIND_LEAKCHECK
      cout << "\n==MEMTRACK2 debugging start\n";
      dumpMemInfo();
      VALGRIND_DO_LEAK_CHECK
	cout << "\n==MEMTRACK2 debugging end\n";
#endif

      if(as_fixparams.as_dateoutput) dateStamp(cout);

      bool foundrepeats=buildFirstContigs(actpass,
					  eparams, 
					  (actpass==as_fixparams.as_numpasses));

#ifdef VALGRIND_LEAKCHECK
      cout << "\n==MEMTRACK3 debugging start\n";
      dumpMemInfo();
      VALGRIND_DO_LEAK_CHECK
	cout << "\n==MEMTRACK3 debugging end\n";
#endif

#if TRACKMEMUSAGE 
      cout << "\ndmi 80\n";
      dumpMemInfo();
#endif

      // try to guess a good value for minimum coverage
      //  for large contigs
      AS_assemblyinfo.calcCurrentInfo();
      if(AS_assemblyinfo.ASI_avgcoverage[1]>0){
	AS_assemblyinfo.setLargeTotalCov(
	  static_cast<uint32>(.5+AS_assemblyinfo.ASI_avgcoverage[1]/3)
	  );
      }else{
	AS_assemblyinfo.setLargeTotalCov(
	  static_cast<uint32>(.5+AS_assemblyinfo.ASI_avgcoverage[0]/3)
	  );
      }
      for(uint8 st=0; st<Read::SEQTYPE_END; st++){
	AS_assemblyinfo.setLargeContigCovPerST(
	  static_cast<uint32>(.5+AS_assemblyinfo.ASI_avgcoverage_perst[0][st]/3),
	  st
	  );
      }
      saveAssemblyInfo();

#if TRACKMEMUSAGE 
      cout << "\ndmi 90\n";
      dumpMemInfo();
#endif

      // some things are worth doing only if it's not the last pass
      if(actpass!=as_fixparams.as_numpasses){

	if(foundrepeats){
	  cout << "Repeats found during contig building, adding additional alignment iteration\nfor quick repeat resolving.\n";
	  AS_steps[ASADSLISTOK]=0;
	  //makeAlignments(Assembly::ma_needSRMrOrTwoCRMr, true, actpass, "", "", "repeat_resolve");

	  // Step 1 of repeat resolving
	  // define which reads should be taken into the alignments phase
	  //  (== those who got additional SRMr tags)
	  AS_readsforrepeatresolve.clear();
	  AS_readsforrepeatresolve.resize(AS_readpool.size(),false);
	  for(uint32 rnr=0; rnr<AS_readpool.size(); rnr++){
	    if(AS_readpool.getRead(rnr).countTags(Read::REA_tagentry_idSRMr) != xrmrcount[rnr]) AS_readsforrepeatresolve[rnr]=true;
	  }

	  // prepare for second round of repeat resolve
	  for(uint32 rnr=0; rnr<AS_readpool.size(); rnr++){
	    xrmrcount[rnr]=AS_readpool.getRead(rnr).countTags(Read::REA_tagentry_idCRMr);
	  }
	
#if TRACKMEMUSAGE 
	  cout << "\ndmi a0\n";
	  dumpMemInfo();
#endif
	  makeAlignments(Assembly::ma_needRRFlag, true, false, actpass, "", "", "repeat_resolve");
	  AS_steps[ASADSLISTOK]=0;

	  // Step 2 of repeat resolving
	  // define which reads should be taken into the alignments phase
	  //  (== those who got additional CRMr tags in alignment phase above)

	  AS_readsforrepeatresolve.clear();
	  AS_readsforrepeatresolve.resize(AS_readpool.size(),false);
	  for(uint32 rnr=0; rnr<AS_readpool.size(); rnr++){
	    if(AS_readpool.getRead(rnr).countTags(Read::REA_tagentry_idCRMr) != xrmrcount[rnr]) AS_readsforrepeatresolve[rnr]=true;
	  }

#if TRACKMEMUSAGE 
	  cout << "\ndmi b0\n";
	  dumpMemInfo();
#endif
	  makeAlignments(Assembly::ma_needRRFlagAndBothCRMr, true, false, actpass, "", "", "repeat_resolve");
	
#if TRACKMEMUSAGE 
	  cout << "\ndmi c0\n";
	  dumpMemInfo();
#endif
	  nukeSTLContainer(AS_readsforrepeatresolve);
	
#if TRACKMEMUSAGE 
	  cout << "\ndmi d0\n";
	  dumpMemInfo();
#endif
	  rerunSKIM=true;
	}

	if(usereadextension
	   && actpass >= as_fixparams.as_readextension_firstpassnum
	   && actpass <= as_fixparams.as_readextension_lastpassnum) {
	  extendADS(actpass, "", "_pass");
	  AS_needsskimfornastyrepeats=true;
	  rerunSKIM=true;
	}
	if(clipvectorleftovers) {
	  performSeqVectorClippings();
	  AS_needsskimfornastyrepeats=true;
	  rerunSKIM=true;
	}
	if(AS_seqtypespresent[Read::SEQTYPE_SOLEXA]
	   || AS_seqtypespresent[Read::SEQTYPE_ABISOLID]){
	  performSRMRightClippings();
	}
      }

      if(checkTerminationRequest()){
	cout << "Seen termination request by user\n";
	if(actpass + 2 < as_fixparams.as_numpasses){
	  assembly_parameters & ap=const_cast<assembly_parameters &>(AS_miraparams[0].getAssemblyParams());
	  cout << "Changing number of passes from " << ap.as_numpasses << " to ";
	  ap.as_numpasses=actpass+2;
	  cout << ap.as_numpasses << endl;
	}else{
	  cout << "No further action necessary, will terminate anyway in 2 passes max.\n";
	}
      }

#if TRACKMEMUSAGE
      cout << "\ndmi e0\n";
      dumpMemInfo();
#endif
    }
  }

  performSnapshot(actpass);

  FUNCEND();
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::performSnapshot(uint32 actpass)
{
  FUNCSTART("void Assembly::performSnapshot(uint32 actpass)");

  assembly_parameters const & as_fixparams= AS_miraparams[0].getAssemblyParams();
  cout << "Performing snapshot " << actpass << endl;
  if(as_fixparams.as_dateoutput) dateStamp(cout);
  {
    ofstream fout(buildDefaultCheckpointFileName(as_fixparams.as_infile_chkptMAF).c_str(),
		  ios::out|ios::trunc);
    
    AS_readpool.dumpAs(fout,Read::AS_MAF,true);
    fout.close();
  }

  {
    ofstream fout(buildDefaultCheckpointFileName("passInfo.txt").c_str(),
		  ios::out|ios::trunc);
    fout << actpass << endl;
		  
    fout.close();
  }
  if(as_fixparams.as_dateoutput) dateStamp(cout);

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::saveResults()
{
  FUNCSTART("void Assembly::saveResults()");

  assembly_parameters const & as_fixparams= AS_miraparams[0].getAssemblyParams();

  saveDebrisList();
  if(as_fixparams.as_dateoutput) dateStamp(cout);
  saveStatistics();
  if(as_fixparams.as_dateoutput) dateStamp(cout);
  saveReadTagList();
  if(as_fixparams.as_dateoutput) dateStamp(cout);
  saveConsensusTagList();
  if(as_fixparams.as_dateoutput) dateStamp(cout);
  saveContigReadList();
  if(as_fixparams.as_dateoutput) dateStamp(cout);
  if(as_fixparams.as_output_caf){
    saveAsCAF();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }
  if(as_fixparams.as_output_maf){
    saveAsMAF();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }
  if(as_fixparams.as_output_wiggle){
    saveAsWiggle();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }
  if(as_fixparams.as_output_gap4da){
    saveAsGAP4DA();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }
  if(as_fixparams.as_output_fasta) {
    saveAsFASTA();
    if(AS_readpool.getNumOfStrainInReadpool()>1) saveStrainsAsFASTAQUAL();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }
  if(as_fixparams.as_output_tcs) {
    saveAsTCS();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }
  saveSNPList();
  //saveFeatureAnalysis();
  if(as_fixparams.as_output_txt){
    saveAsTXT();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }
  if(as_fixparams.as_output_ace){
    saveAsACE();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }
  if(as_fixparams.as_output_html){
    saveAsHTML();
    if(as_fixparams.as_dateoutput) dateStamp(cout);
  }

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::makeNewReadPoolFromContigs()
{
  FUNCSTART("void Assembly::makeNewReadPoolFromContigs()");

  list<Contig>::iterator I = AS_contigs.begin();
  int32 ccounter=0;

  while(I!=AS_contigs.end()){
    //const vector<Contig::contigread_t> & cr = I->getContigReads();
    //vector<Contig::contigread_t>::iterator crI = const_cast<vector<Contig::contigread_t>::iterator>(cr.begin());
    const vector<Contig::contigread_t> & cr = I->getContigReads();
    vector<Contig::contigread_t>::const_iterator crI = cr.begin();


    for(;crI!=cr.end();crI++){
      if(crI->orpid>=0){
	if(crI->read.checkRead()){
	  cout << "Precheck failed: " << endl;
	  cout << crI->read;
	  throw Notify(Notify::FATAL, THISFUNC, crI->read.checkRead()) ;
	}
	
	AS_readpool.getRead(crI->orpid)=crI->read;
	
	if(AS_readpool.getRead(crI->orpid).checkRead()){
	  cout << "Postcheck1 failed: " << endl;
	  cout << AS_readpool.getRead(crI->orpid);
	  throw Notify(Notify::FATAL, THISFUNC, AS_readpool.getRead(crI->orpid).checkRead()) ;
	}
	
	const_cast<Read &>(crI->read).discard();
	
	if(AS_readpool.getRead(crI->orpid).checkRead()){
	  cout << "Postcheck2 failed: " << endl;
	  cout << AS_readpool.getRead(crI->orpid);
	  throw Notify(Notify::FATAL, THISFUNC, AS_readpool.getRead(crI->orpid).checkRead()) ;
	}
      }
    }

    I++;ccounter++;
  }
  AS_contigs.clear();

  FUNCEND();
}



/*************************************************************************
 *
 * returns whether new strong repeat markers (SRMs) were found for
 *  any contig built in any stage
 *
 *************************************************************************/

#define CEBUG(bla)   {cout << bla; cout.flush(); }

bool Assembly::buildFirstContigs(const int32 passnr, const EDITParameters & eparams, const bool lastpass)
{
  FUNCSTART("void Assembly::buildFirstContigs()");

  CEBUG("BFC: " << passnr << "\t" << lastpass << endl);

  assembly_parameters const & as_fixparams= AS_miraparams[0].getAssemblyParams();
  directory_parameters const & dir_params= AS_miraparams[0].getDirectoryParams();
  edit_parameters const & ed_params= AS_miraparams[0].getEditParams();

  AS_deleteoldresultfiles=true;

  AS_contigs.clear();
  Contig::setIDCounter(1);
  Contig::resetCERNumbering();

  AS_assemblyinfo.zeroInfo();

  vector<Align> aligncache;
  setupAlignCache(aligncache);

  // AS_hasmcoverlaps will be initialised by Pathfinder
  AS_hasmcoverlaps.clear();

  // when using short reads (Solexa, SOLiD), the contig might propose
  //  new right clips to squeeze the data in. This vector holds
  //  the info.
  // -1 says: no proposed new right clip
  AS_clipright.clear();
  AS_clipright.resize(AS_readpool.size(),-1);

  // Keep track of coverage information for contigs so to be able to make
  //  average contig coverage predictions
  vector<vector<uint32> > covperstpercon(Read::SEQTYPE_END);

  // initially, overlaps of every read can be reduced
  if(AS_needalloverlaps.size()==0){
    AS_needalloverlaps.resize(AS_readpool.size(),false);
  }

  // filter out all reads that have no overlap at this stage
  // for this, first set all reads to "used" and "singlet", then
  //   re-allow those that have at least one SW ovelap

  AS_used_ids.clear();
  AS_used_ids.resize(AS_readpool.size(),1);
  AS_isdebris.clear();
  AS_isdebris.resize(AS_readpool.size(),1);

  // take back all reads with overlaps
  {
    vector<newedges_t>::const_iterator ceI=AS_confirmed_edges.begin();
    for(;ceI!=AS_confirmed_edges.end();ceI++){
      AS_isdebris[ceI->rid1]=0;
      AS_used_ids[ceI->rid1]=0;
    }
  }

  // Now, go through all the singlets and take them back into
  //  assembly if they have special MIRA tags attached
  // Also remove backbones & rails from the debris list
  {
    for(uint32 i=0; i< AS_readpool.size(); i++){
      if(AS_readpool[i].hasTag(Read::REA_tagentry_idSRMr)
	 || AS_readpool[i].hasTag(Read::REA_tagentry_idCRMr)
	 || AS_readpool[i].hasTag(Read::REA_tagentry_idWRMr)
	 || AS_readpool[i].hasTag(Read::REA_tagentry_idSAOr)
	 || AS_readpool[i].hasTag(Read::REA_tagentry_idSROr)
	 || AS_readpool[i].hasTag(Read::REA_tagentry_idSIOr)) {
	AS_isdebris[i]=0;
	AS_used_ids[i]=0;
      }
      if(AS_readpool[i].isBackbone()
	 || AS_readpool[i].isRail()) {
	AS_isdebris[i]=0;
      }
    }
  }

  // get out reads that are
  //  - invalid or not used in assembly
  //  - or backbone or rail
  // additionally set flag if we have 454 reads that are used

  bool used454readsinpool=false;
  {
    for(uint32 i=0; i< AS_readpool.size(); i++){
      if(AS_readpool[i].hasValidData()== false
	|| AS_readpool[i].isUsedInAssembly()==false){
	AS_used_ids[i]=-1;
	if(AS_readpool[i].isSequencingType(Read::SEQTYPE_454GS20)){
	  used454readsinpool=true;
	}
//      } else if(AS_readpool[i].isBackbone()
//		|| AS_readpool[i].isRail()) {
//	AS_used_ids[i]=1;
//      } 
      }
    }
  }

  //AS_miraparams[0].setAssemblyLogUnusedIDs("unused.debug");


  // get total length of backbones (if any)
  uint32 totalbblen=0;
  for(list<Contig>::const_iterator cI=AS_bbcontigs.begin(); cI!=AS_bbcontigs.end(); cI++){
    totalbblen+=cI->getContigLength();
  }

  ofstream fout;
  if(as_fixparams.as_tmpf_unused_ids.size()!=0){
    fout.open((dir_params.dir_tmp+"/"+as_fixparams.as_tmpf_unused_ids).c_str(), ios::out);
    fout.close();
  }

  // outside precomputed lowerbound of oedges, for PathFinder
  // is used lateron in constructStepByStep() of the Pathfinder
  //
  // however, every once in a while (every 10k to 100k reads used from
  //  the pool), the overlap edges vector will be compressed (unecessary
  //  edges thrown out) and therefore this vector will be emptied (to be reconstructed
  //  by pathfinder)
  //
  // On mapping 8m Solexas to 70 contigs (4MB), time goes down from 234 minutes to
  //  221 minutes (Core i940)
  //
  // effect of compression not as big as hoped for, but every little bit helps

  vector<vector<newedges_t>::iterator> tmp_lowerbound_oedges;

  // PathFinder object can be created ouside loop and re-used
  //  (constructor needs to do a lot of lower_bound() searches.
  // Saves a lot of time in the endgame (singlets) of big projects
  // TODO: check wrt multiple MIRAparams
  Pathfinder paf(&AS_miraparams,
		 AS_readpool,
		 AS_confirmed_edges,
		 AS_adsfacts);

  // this vector will hold the read IDs added by pathfinder to contig
  // + backbone + rail IDs
  // used after call to *_constructStepByStep() to set AS_used_ids[] elements
  vector<int32> pf_ids_in_contig;

  uint32 overlapcompressstepping=AS_readpool.size()/5;
  if(overlapcompressstepping<10000) overlapcompressstepping=10000;
  //if(overlapcompressstepping<100) overlapcompressstepping=100;
  if(overlapcompressstepping>100000) overlapcompressstepping=100000;
  if(overlapcompressstepping>AS_readpool.size()) overlapcompressstepping=AS_readpool.size();
  uint32 nextoverlapcompress=overlapcompressstepping;
  cout << "overlapcompressstepping: " << overlapcompressstepping
       << "\nnextoverlapcompress: " << nextoverlapcompress << endl;

  bool foundSRMs=false;
  uint32 numsingletssincecleanup=0;

  bool maykillintermediatesinmglets=true;
  bool shouldmovesmallclusterstodebris=false;
  for(uint32 st=0;st < AS_miraparams.size(); st++){
    if(AS_miraparams[st].getAssemblyParams().as_savesimplesingletsinproject) maykillintermediatesinmglets=false;
    if(AS_miraparams[st].getAssemblyParams().as_minimum_readspercontig>1) shouldmovesmallclusterstodebris=true;
  }

  if(shouldmovesmallclusterstodebris) bfc_moveSmallClustersToDebris();


#ifdef CLOCK_STEPS2
  timeval tv;
  timeval tvloop;
  timeval tvtotal;
  gettimeofday(&tvtotal,NULL);
#endif

  uint32 unused=AS_used_ids.size();
  uint32 numcontigs=1;
  // bug: if someone specifically sets as_maxcontigsperpass to 2^32-1, then
  //  this loop never runs.
  for(;unused>0; ++numcontigs){
    CEBUG("bfc 1\n");
    if(as_fixparams.as_dateoutput) dateStamp(cout);
    cout << '\n';

#ifdef CLOCK_STEPS2
    gettimeofday(&tv,NULL);
#endif

    if(as_fixparams.as_tmpf_unused_ids.size()!=0){
      fout.open((dir_params.dir_tmp+"/"+as_fixparams.as_tmpf_unused_ids).c_str(), ios::out);
      fout << "\nUnused for contig" << numcontigs << endl;
    }
    unused=0;
    for(uint32 i=0; i<AS_used_ids.size(); i++){
      if(AS_used_ids[i]==0){
	unused++;
	//cout << "nused: " << AS_readpool[i].getName() <<endl;
	if(as_fixparams.as_tmpf_unused_ids.size()!=0){
	  fout << AS_readpool[i].getName();
	  fout << "\t" << AS_readpool[i].getSCFName() << endl;;
	}
      }else{
	//cout << "ASui["<<i<<"]: " << static_cast<int16>(AS_used_ids[i]) << endl;
      }
    }
    if(as_fixparams.as_tmpf_unused_ids.c_str()!=0){
      fout.close();
    }

#ifdef CLOCK_STEPS2
    cout << "Timing BFC unused: " << diffsuseconds(tv) << endl;
    cout << "Unused: " << unused << endl;
    cout << "AS_used_ids.size(): " << AS_used_ids.size() << endl;
#endif

    // jump out of for loop if max number of contigs was reached
    if(as_fixparams.as_maxcontigsperpass>0 && numcontigs==as_fixparams.as_maxcontigsperpass+1) break;

    CEBUG("bfc 2\n");
    if(unused>0){

#ifdef CLOCK_STEPS2
      gettimeofday(&tv,NULL);
      tvloop=tv;
#endif

      //// compress the overlap edges when needed
      //if(AS_readpool.size()-unused > nextoverlapcompress){
      //	if(AS_miraparams[0].getAssemblyParams().as_dateoutput) dateStamp(cout);
      //	cout << "Compressing overlap edges ..."; cout.flush();
      //	vector<newedges_t>::const_iterator srcI=AS_confirmed_edges.begin();
      //	vector<newedges_t>::iterator dstI=AS_confirmed_edges.begin();
      //	for(; srcI != AS_confirmed_edges.end(); ++srcI){
      //	  if(AS_used_ids[srcI->rid1] == 0
      //	     || AS_used_ids[srcI->linked_with] == 0){
      //	    *dstI=*srcI;
      //	    ++dstI;
      //	  }
      //	}
      //	AS_confirmed_edges.resize(dstI-AS_confirmed_edges.begin());
      //	nextoverlapcompress=AS_readpool.size()-unused+overlapcompressstepping;
      //	cout << "done.\n";
      //	if(AS_miraparams[0].getAssemblyParams().as_dateoutput) dateStamp(cout);
      //}

      Contig::setIDCounter(numcontigs);
      // TODO: change wrt multiple MIRAparams
      Contig buildcon(&AS_miraparams, AS_readpool);

      //vector<int8> tmpused=AS_used_ids;

#ifdef CLOCK_STEPS2
      cout << "Timing BFC prelim1: " << diffsuseconds(tv) << endl;
#endif

      CEBUG("bfc 3\n");
      list<Contig>::const_iterator bbContigI=AS_bbcontigs.begin();
      if(numcontigs <= AS_bbcontigs.size()){
	advance(bbContigI,numcontigs-1);
      } else {
	bbContigI=AS_bbcontigs.end();
      }

      CEBUG("bfc 4\n");
      // set newreptmarked to true just to get into the for pass
      //  value is changed within pass
      bool newreptmarked=true;
      bool wasovercalledited=false;
      bool wasmajorovercalledited=false;
      bool mayeditovercalls=true;

      bool mastermayeditovercalls=AS_miraparams[Read::SEQTYPE_454GS20].getEditParams().ed_automatic_contic_editing;

      // build a contig, repetitively until
      //  ... maximum number of iterations has passed
      //  ... or no now repeats were marked
      //  ... or no 454 edits happened in last iteration

      // Note: if were in last pass, iterate at least once if
      //  454 edits were made! (disregarding as_fixparams.as_numrmbbreakloops,
      //  maxiter gets adapted in loop then

      uint32 maxiter=as_fixparams.as_numrmbbreakloops;

      // Note: assemblies with a lot of passes (>=4) will
      //  get only one loop in the first pass. The reason: first pass already
      //  discovers a lot of repeats that are dealt with later on
      //  in makeAlignments(). It's faster to let makeAlignments() deal
      //  with "discovering" and marking reads than to loop here.

      if(as_fixparams.as_mark_repeats){
	if(as_fixparams.as_numpasses >= 4 && passnr==1){
	  maxiter=1;
	}
	// same thing: passes >= 6, pass 2 leads to maxiter = 2 
	if(as_fixparams.as_numpasses >= 6 && passnr==2
	  && maxiter > 2){
	  maxiter=2;
	}
      }

      bool markrepeatsduringstore=true;

      bool contignotok=false;

      list<Contig::pbdse_t> pbdsev;

      CEBUG("bfc 5\n");
      for(uint32 iter=0; 
	  iter < maxiter 
	    && (newreptmarked || wasmajorovercalledited || contignotok);
	  iter++){

#ifdef CLOCK_STEPS2
	gettimeofday(&tv,NULL);
#endif

	//AS_used_ids=tmpused;

	// if pf_ids_in_contig entries, they're from the previous iteration
	// which means that AS_used_ids is also not clean
	// get that out again
	{
	  vector<int32>::const_iterator iibcI=pf_ids_in_contig.begin();
	  for(;iibcI!=pf_ids_in_contig.end(); ++iibcI){
	    AS_used_ids[*iibcI]=0;
	    //cout << "deusing " << AS_readpool[*iibcI].getName() << endl;
	  }
	}
	
#ifdef CLOCK_STEPS2
	cout << "Timing BFC setup AS_used_ids: " << diffsuseconds(tv) << endl;
#endif

	CEBUG("bfc 6/"<<iter << '\n');
	if(as_fixparams.as_loadbackbone 
	   && passnr>=as_fixparams.as_startbackboneusage_inpass
	   && bbContigI != AS_bbcontigs.end()) {

#ifdef CLOCK_STEPS2
	  gettimeofday(&tv,NULL);
#endif
	  // let's try to pre-allocate memory for the most crucial
	  //  parts of the contig structure: number of reads and contig len
	  
	  {
	    double cfraction=100.0/static_cast<double>(totalbblen)*static_cast<double>(bbContigI->getContigLength());
	    uint32 expectedmaxlen=static_cast<uint32>(
	      cfraction*static_cast<double>(bbContigI->getContigLength())
	      );
	    // add 12.5% for the expected maxlen (worst case: 454 reads: 1 indel 
	    //  all 8 bases.
	    expectedmaxlen+=expectedmaxlen/8;

	    uint32 expectedmaxreads=bbContigI->getNumReadsInContig()+static_cast<uint32>(
	      cfraction*static_cast<double>(AS_readpool.size())
	      );
	    // add 20% to expected maxreads
	    expectedmaxreads+=expectedmaxreads/5;

	    //cout << "expectedmaxreads: " << expectedmaxreads << endl;

	    if(expectedmaxreads>AS_readpool.size()) expectedmaxreads=AS_readpool.size();

	    //cout << "rps: " << AS_readpool.size() << endl;
	    //cout << "allocate: " << expectedmaxreads << endl;
	    

	    buildcon.reserveCapacity(expectedmaxlen, expectedmaxreads);
	  }

#ifdef CLOCK_STEPS2
	  cout << "Timing BFC prealloc con: " << diffsuseconds(tv) << endl;
	  gettimeofday(&tv,NULL);
#endif
	  buildcon=*bbContigI;

#ifdef CLOCK_STEPS2
	  cout << "Timing BFC copy bbcon: " << diffsuseconds(tv) << endl;
	  gettimeofday(&tv,NULL);
#endif

	  // re-initialising the baselocks is necessary as reads might
	  //  have got new SRMc/WRMc tags in previous iterations
	  //  these are not known in the initial backbone contig, so
	  //  they must be made known
	  buildcon.initialiseBaseLocks();
	  buildcon.setContigID(numcontigs);

	  // tell contig to use backbone characters when possible for 
	  //  tmp consensus
	  // TODO: make configurable?
	  buildcon.useBackbone4TmpConsensus(true);

	  // and by default try to merge short reads (Solexa, SOLiD)
	  buildcon.mergeNewSRReads(true);
#ifdef CLOCK_STEPS2
	  cout << "Timing BFC bbsetup remain: " << diffsuseconds(tv) << endl;
#endif

	} else {
#ifdef CLOCK_STEPS2
	  gettimeofday(&tv,NULL);
#endif
	  buildcon.discard();
#ifdef CLOCK_STEPS2
	  cout << "Timing BFC discard con: " << diffsuseconds(tv) << endl;
#endif
	}

	CEBUG("bfc 7/"<<iter << '\n');

	buildcon.setContigNamePrefix(
	  AS_miraparams[0].getContigParams().con_nameprefix);

	if(iter!=0){
	  if(newreptmarked){
	    cout << "\nIdentified misassembled reads in contig.\n";
	  } else {
	    cout << "\nLooping because of edits in reads.\n";
	  }
	  cout << "Rebuilding contig " << numcontigs << " now.\n";
	}else{
	  cout << "Building new contig " << numcontigs << endl;
	}

	//Contig::setCoutType(Contig::AS_DEBUG);
	//cout << con;

	if(!AS_coverageperseqtype.empty()){
	  cout << "Setting contig coverage targets to: ";
	  for(uint8 ii=0; ii<AS_coverageperseqtype.size(); ii++){
	    cout << '\t' << AS_coverageperseqtype[ii];
	  }
	  cout << endl;
	  buildcon.setContigCoverageTarget(AS_coverageperseqtype);
	}

	if(as_fixparams.as_dateoutput) dateStamp(cout);
	
	cout << "Unused reads: " << unused << endl;
	cout.flush();
	
	//cerr << "paf.constructStepByStep()\n";
	//exit(1);

	CEBUG("bfc 8/"<<iter << '\n');

	// in which mode are we? de-novo or mapping?
	bool assemblymode_mapping=false;
	if(as_fixparams.as_loadbackbone
	   && passnr >= as_fixparams.as_startbackboneusage_inpass
	   && ! as_fixparams.as_backbone_alsobuildnewcontigs){
	  assemblymode_mapping=true;
	}

	CEBUG("assemblymode_mapping: " << assemblymode_mapping << '\n');

	// if we have Solexa or Solid mapping, first round shall be
	//  without any error, then slowly increase until
	if(assemblymode_mapping
	   && (AS_seqtypespresent[Read::SEQTYPE_SOLEXA]
	       || AS_seqtypespresent[Read::SEQTYPE_ABISOLID])){

	  if(1){
	    cout << "Gogo: 100% mapping\n";
	    buildcon.setSpecialSRAddConditions(0,0,0);
	    paf.constructStepByStep(aligncache,
				    &AS_used_ids,
				    &pf_ids_in_contig,
				    &AS_multicopies,
				    &AS_hasmcoverlaps,
				    &AS_istroublemaker,
				    &tmp_lowerbound_oedges,
				    &AS_clipright,
				    buildcon);
	    buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	    cout << "\nbfc debug checkpoint" << endl;
	    //assout::saveAsCAF(con, "bfcdebug_m0.caf", true);
	    cout << "done" << endl;

	    if(AS_miraparams[0].getAlignParams().al_solexahack_maxerrors>0 && paf.getReadAddAttempts()>0) {
	      cout << "Gogo: mapping 1 mismatch\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	      //  6,-1,Read::SEQTYPE_SOLEXA);
	      buildcon.setSpecialSRAddConditions(1,0,1);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      //buildcon.allowedRefIDs_forbidAll();
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m1.caf", true);
	      cout << "done" << endl;
	    
	      cout << "Gogo: mapping 1 gap\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	      //  6,-1,Read::SEQTYPE_SOLEXA);
	      buildcon.setSpecialSRAddConditions(1,-1,0);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      //buildcon.allowedRefIDs_forbidAll();
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m1g.caf", true);
	      cout << "done" << endl;
	    }

	    if(AS_miraparams[0].getAlignParams().al_solexahack_maxerrors>1 && paf.getReadAddAttempts()>0) {
	      cout << "Gogo: mapping 2 mismatches\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	      //  8,-1,Read::SEQTYPE_SOLEXA);
	      buildcon.setSpecialSRAddConditions(2,0,2);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      //buildcon.allowedRefIDs_forbidAll();
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m2.caf", true);
	      cout << "done" << endl;
	    
	      cout << "Gogo: mapping 1 gap, 1 mismatch\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	      //  8,-1,Read::SEQTYPE_SOLEXA);
	      buildcon.setSpecialSRAddConditions(2,1,1);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      //buildcon.allowedRefIDs_forbidAll();
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m2m1g1.caf", true);
	      cout << "done" << endl;
	    
	      cout << "Gogo: mapping 2 errors (==remaining 2 gaps)\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	      //  8,-1,Read::SEQTYPE_SOLEXA);
	      buildcon.setSpecialSRAddConditions(2,-1,-1);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      //buildcon.allowedRefIDs_forbidAll();
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m2g2.caf", true);
	      cout << "done" << endl;
	    }

	    for(uint32 numerr=3; numerr<=AS_miraparams[0].getAlignParams().al_solexahack_maxerrors; numerr++){
	      if(paf.getReadAddAttempts()==0) break;
	      cout << "Gogo: mapping all " << numerr << " errors\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      buildcon.setSpecialSRAddConditions(numerr,-1,-1);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m2g2.caf", true);
	      cout << "done" << endl;
	    }


	  }else{
	    cout << "Gogo: mapping 1 gap\n";
	    if(as_fixparams.as_dateoutput) dateStamp(cout);

	    buildcon.setSpecialSRAddConditions(1,1,0);
	    paf.constructStepByStep(aligncache,
				    &AS_used_ids,
				    &pf_ids_in_contig,
				    &AS_multicopies,
				    &AS_hasmcoverlaps,
				    &AS_istroublemaker,
				    &tmp_lowerbound_oedges,
				    &AS_clipright,
				    buildcon);
	    buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	    cout << "\nbfc debug checkpoint" << endl;
	    //assout::saveAsCAF(con, "bfcdebug_m1g.caf", true);
	    cout << "done" << endl;

	    cout << "Gogo: mapping 2 gaps\n";
	    buildcon.setSpecialSRAddConditions(2,2,0);
	    paf.constructStepByStep(aligncache,
				    &AS_used_ids,
				    &pf_ids_in_contig,
				    &AS_multicopies,
				    &AS_hasmcoverlaps,
				    &AS_istroublemaker,
				    &tmp_lowerbound_oedges,
				    &AS_clipright,
				    buildcon);
	    buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	    cout << "\nbfc debug checkpoint" << endl;
	    //assout::saveAsCAF(con, "bfcdebug_m1g.caf", true);
	    cout << "done" << endl;

	    cout << "Gogo: mapping 3 gaps\n";
	    buildcon.setSpecialSRAddConditions(3,3,0);
	    paf.constructStepByStep(aligncache,
				    &AS_used_ids,
				    &pf_ids_in_contig,
				    &AS_multicopies,
				    &AS_hasmcoverlaps,
				    &AS_istroublemaker,
				    &tmp_lowerbound_oedges,
				    &AS_clipright,
				    buildcon);
	    buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	    cout << "\nbfc debug checkpoint" << endl;
	    //assout::saveAsCAF(con, "bfcdebug_m1g.caf", true);
	    cout << "done" << endl;

	    cout << "Gogo: mapping 1 gap, 1 mismatch\n";
	    if(as_fixparams.as_dateoutput) dateStamp(cout);
	    //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	    //  8,-1,Read::SEQTYPE_SOLEXA);
	    buildcon.setSpecialSRAddConditions(2,1,1);
	    paf.constructStepByStep(aligncache,
				    &AS_used_ids,
				    &pf_ids_in_contig,
				    &AS_multicopies,
				    &AS_hasmcoverlaps,
				    &AS_istroublemaker,
				    &tmp_lowerbound_oedges,
				    &AS_clipright,
				    buildcon);
	    //buildcon.allowedRefIDs_forbidAll();
	    buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	    cout << "\nbfc debug checkpoint" << endl;
	    //assout::saveAsCAF(con, "bfcdebug_m2m1g1.caf", true);
	    cout << "done" << endl;


	    cout << "Gogo: mapping 2 gaps, 1 mismatch\n";
	    if(as_fixparams.as_dateoutput) dateStamp(cout);
	    //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	    //  8,-1,Read::SEQTYPE_SOLEXA);
	    buildcon.setSpecialSRAddConditions(3,2,1);
	    paf.constructStepByStep(aligncache,
				    &AS_used_ids,
				    &pf_ids_in_contig,
				    &AS_multicopies,
				    &AS_hasmcoverlaps,
				    &AS_istroublemaker,
				    &tmp_lowerbound_oedges,
				    &AS_clipright,
				    buildcon);
	    //buildcon.allowedRefIDs_forbidAll();
	    buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	    cout << "\nbfc debug checkpoint" << endl;
	    //assout::saveAsCAF(con, "bfcdebug_m2m1g1.caf", true);
	    cout << "done" << endl;

	    cout << "Gogo: mapping 1 gap, 2 mismatches\n";
	    if(as_fixparams.as_dateoutput) dateStamp(cout);
	    //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	    //  8,-1,Read::SEQTYPE_SOLEXA);
	    buildcon.setSpecialSRAddConditions(3,1,2);
	    paf.constructStepByStep(aligncache,
				    &AS_used_ids,
				    &pf_ids_in_contig,
				    &AS_multicopies,
				    &AS_hasmcoverlaps,
				    &AS_istroublemaker,
				    &tmp_lowerbound_oedges,
				    &AS_clipright,
				    buildcon);
	    //buildcon.allowedRefIDs_forbidAll();
	    buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	    cout << "\nbfc debug checkpoint" << endl;
	    //assout::saveAsCAF(con, "bfcdebug_m2m1g1.caf", true);
	    cout << "done" << endl;

	    // ----------------------------------------------------

	    cout << "Gogo: 100% mapping\n";
	    buildcon.setSpecialSRAddConditions(0,0,0);
	    paf.constructStepByStep(aligncache,
				    &AS_used_ids,
				    &pf_ids_in_contig,
				    &AS_multicopies,
				    &AS_hasmcoverlaps,
				    &AS_istroublemaker,
				    &tmp_lowerbound_oedges,
				    &AS_clipright,
				    buildcon);
	    buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	    cout << "\nbfc debug checkpoint" << endl;
	    //assout::saveAsCAF(con, "bfcdebug_m0.caf", true);
	    cout << "done" << endl;

	    if(AS_miraparams[0].getAlignParams().al_solexahack_maxerrors>0) {
	      cout << "Gogo: mapping 1 mismatch\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	      //  6,-1,Read::SEQTYPE_SOLEXA);
	      buildcon.setSpecialSRAddConditions(1,0,1);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      //buildcon.allowedRefIDs_forbidAll();
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m1.caf", true);
	      cout << "done" << endl;
	    
	    }

	    if(AS_miraparams[0].getAlignParams().al_solexahack_maxerrors>1) {
	      cout << "Gogo: mapping 2 mismatches\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	      //  8,-1,Read::SEQTYPE_SOLEXA);
	      buildcon.setSpecialSRAddConditions(2,0,2);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      //buildcon.allowedRefIDs_forbidAll();
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m2.caf", true);
	      cout << "done" << endl;
	    
	    
	      cout << "Gogo: mapping 2 errors (==remaining 2 gaps)\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      //buildcon.allowedRefIDs_allowSomeRailsOnShortReadCoverage(
	      //  8,-1,Read::SEQTYPE_SOLEXA);
	      buildcon.setSpecialSRAddConditions(2,-1,-1);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      //buildcon.allowedRefIDs_forbidAll();
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m2g2.caf", true);
	      cout << "done" << endl;
	    }

	    for(uint32 numerr=3; numerr<=AS_miraparams[0].getAlignParams().al_solexahack_maxerrors; numerr++){
	      cout << "Gogo: mapping all " << numerr << " errors\n";
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      buildcon.setSpecialSRAddConditions(numerr,-1,-1);
	      paf.constructStepByStep(aligncache,
				      &AS_used_ids,
				      &pf_ids_in_contig,
				      &AS_multicopies,
				      &AS_hasmcoverlaps,
				      &AS_istroublemaker,
				      &tmp_lowerbound_oedges,
				      &AS_clipright,
				      buildcon);
	      buildcon.setCoutType(Contig::AS_TEXT);buildcon.stats(cout);
	      cout << "\nbfc debug checkpoint" << endl;
	      //assout::saveAsCAF(con, "bfcdebug_m2g2.caf", true);
	      cout << "done" << endl;
	    }
	  }




	  // TODO: hack until there's a routine that only clears tags set by the
	  //  makeIntelligentConsensus() functions.
	  buildcon.clearConsensusTags();

	}else{
#ifdef CLOCK_STEPS2
	  gettimeofday(&tv,NULL);
#endif

	  CEBUG("use genomic pathfinder: " << AS_miraparams[0].getPathfinderParams().paf_use_genomic_algorithms << '\n');
	  if(AS_miraparams[0].getPathfinderParams().paf_use_genomic_algorithms){
	    paf.n4_constructStepByStep(aligncache,
				       &AS_used_ids,
				       &pf_ids_in_contig,
				       &AS_multicopies,
				       &AS_hasmcoverlaps,
				       &AS_istroublemaker,
				       &tmp_lowerbound_oedges,
				       &AS_clipright,
				       &AS_wellconnected,
				       buildcon);
	  }else{
	    // EST routines also use new version
	    paf.n4_constructStepByStep(aligncache,
				       &AS_used_ids,
				       &pf_ids_in_contig,
				       &AS_multicopies,
				       &AS_hasmcoverlaps,
				       &AS_istroublemaker,
				       &tmp_lowerbound_oedges,
				       &AS_clipright,
				       &AS_wellconnected,
				       buildcon);
	  }

#ifdef CLOCK_STEPS2
	  cout << "Timing BFC paf construct: " << diffsuseconds(tv) << endl;
#endif

	  // as n4_* will completely use start cache for starting ID of contig build, ask the pathfinder
	  //  whether there are any left in cache. Trigger moving small clusters to debris if not
	  if(shouldmovesmallclusterstodebris && paf.n4_getNumElementsInStartCache()==0){
	    cout << "Triggering additional cluster check.\n";
	    bfc_moveSmallClustersToDebris();
	  }
	}

	CEBUG("bfc 9/"<<iter << '\n');

	bool contigok=bfc_checkIfContigMeetsRequirements(buildcon);
	if(!contigok){
	  newreptmarked=false;

	  // this contig won't be taken, but contig numbering would
	  //  go up in next loop. 
	  // Countermeasure: decrease contig number by one to have the
	  //  loop increase it afterwards, effectivly re-using same contig number
	  --numcontigs;

	  // break out of the iter loop
	  break;
	}

	if(buildcon.getContigLength()>5 && buildcon.getNumReadsInContig()>1){
	  if(as_fixparams.as_put_asswithmira_tags){
	    buildcon.addTagToConsensus(0, 
				       4,
				       '=', 
				       "MIRA", 
				       "Assembled with MIRA",
				       false);
	  }
	}

	cout << "\n\nFinished building." << endl;

	try {
	  if(buildcon.getNumReadsInContig()>1){
	    if(as_fixparams.as_dateoutput) dateStamp(cout);
	    if(buildcon.getContigLength()>100000){
	      cout << "Calculating statistics (this may take a while)." << endl;
	    }

#ifdef CLOCK_STEPS2
	    gettimeofday(&tv,NULL);
#endif
	    // see whether we can define new multicopies
	    if(as_fixparams.as_automatic_repeat_detection){
	      buildcon.analyseReadCoverage(AS_maxcoveragereached,
				      AS_multicopies,
				      AS_coverageperseqtype);
	    }
#ifdef CLOCK_STEPS2
	    cout << "Timing BFC analysereadcov: " << diffsuseconds(tv) << endl;
	    gettimeofday(&tv,NULL);
#endif

	    buildcon.setCoutType(Contig::AS_TEXT);
	    buildcon.stats(cout);

#ifdef CLOCK_STEPS2
	    cout << "Timing BFC cout constats: " << diffsuseconds(tv) << endl;
#endif
	    
	    if(as_fixparams.as_dateoutput) dateStamp(cout);
//	  }else{
//	    cout << "Is singlet.\n";
//	    numsingletssincecleanup++;
//	    if(maykillintermediatesinmglets && numsingletssincecleanup>=20){
//#ifdef CLOCK_STEPS2
//	      gettimeofday(&tv,NULL);
//#endif
//	      cout << "Converted " << bfc_killIntermediateSinglets() << " intermediate singlets into debris.\n";
//#ifdef CLOCK_STEPS2
//	      cout << "Timing BFC kill singlets: " << diffsuseconds(tv) << endl;
//#endif
//	      numsingletssincecleanup=0;
//	    }
	  }
	}
	catch (Notify n) {
	  n.setGravity(Notify::FATAL);
	  n.handleError(n.tif);
	}
	catch (...) {
	  cerr << "Darn, error with that contig. See darn.fasta.\n";
	  vector<Contig::contigread_t>::const_iterator I=buildcon.getContigReads().begin(); 
	  while(I != buildcon.getContigReads().end()) {
	    Read::setCoutType(Read::AS_CLIPPEDFASTA);
	    cout << I->read;
	    I++;
	  }
	  abort();
	}

	CEBUG("bfc 10/"<<iter << '\n');

	string basename_forextsave;
	{
	  ostringstream ostr;
	  ostr << dir_params.dir_tmp << "/miratmp.pass_" << passnr << "_cb" << numcontigs << "_i" << iter << "_";
	  
	  basename_forextsave=ostr.str();
	}
	// saving pre-edit
	saveExtTmpContig(buildcon,(basename_forextsave+"pre"));

	bool wasedited=false;
	newreptmarked=false;

#ifdef CLOCK_STEPS2
	gettimeofday(&tv,NULL);
#endif


// doesn't work really well
//	if(buildcon.hasPacBioData() && iter < 1){
//	  if(mastermayeditovercalls){
//	    buildcon.deleteStarOnlyColumns(0, buildcon.getContigLength()-1,true,2);
//	    wasovercalledited=true;
//	    contignotok=true;
//	    
//	    // another iteration is needed to get a valid contig
//	    if(maxiter<2) maxiter=2;
//	  }
//	}else{

	bool goforit=true;
	//if(buildcon.hasPacBioData() && iter < 1){
	pbdsev.clear();
	if(buildcon.hasPacBioData() && !(lastpass && iter==maxiter-1)){
	  uint32 maxcorrect=buildcon.createPacBioDarkStrobeEdits(pbdsev);

	  CEBUG("PacBio maxcorrect: " << maxcorrect << endl);

	  if(maxcorrect>5) {
	    goforit=false;
	    wasmajorovercalledited=true;
	  }

	  // another iteration is needed to get a valid contig
	  //if(maxiter<2) maxiter=2;
	}

	if(goforit){
	  // handling of 454 / Solexa data
	  mayeditovercalls=buildcon.hasEditableOvercallData() & mastermayeditovercalls;


	  vector<bool> readsmarkedsrm;

	  // if 454 data present, mark areas with tricky overcalls
	  // in those areas, no repeat marker may be set
	  // do this always
	  if(mayeditovercalls && buildcon.getNumReadsInContig() >1){
	    cout << "\nMarking tricky 454 / Solexa overcalls in temporary contig.\n";
	    cout << "Marked " << buildcon.editTrickyOvercalls(true,false,readsmarkedsrm) << " reads.\n";
	  }

#ifdef CLOCK_STEPS2
	  cout << "Timing BFC edit tricky1: " << diffsuseconds(tv) << endl;
	  gettimeofday(&tv,NULL);
#endif

	  if(buildcon.getNumReadsInContig() <= 8) mayeditovercalls=false;

	  markrepeatsduringstore=true;
	  if(as_fixparams.as_mark_repeats
	     && !as_fixparams.as_mark_repeats_onlyinresult
	     && buildcon.getNumReadsInContig() >1){
	    newreptmarked=markRepeats(buildcon, readsmarkedsrm);
	    foundSRMs|=newreptmarked;
	    markrepeatsduringstore=false;
	  }
	  
#ifdef CLOCK_STEPS2
	  cout << "Timing BFC mark reps: " << diffsuseconds(tv) << endl;
	  gettimeofday(&tv,NULL);
#endif

	  CEBUG("bfc 11/"<<iter << '\n');

	  // edit only when no misassembled repeats found
	  //  this is to prevent the editor to try something foolish on
	  //  misassembled things
	  // same applies to overcall editing
	  //

	  wasovercalledited=false;

	  if(!newreptmarked && mayeditovercalls && buildcon.getNumReadsInContig() >1){
	    cout << "Editing tricky 454 / Solexa overcalls";
	    //if(newreptmarked){
	    //  cout << ", but not when reads with SRMs are involved";
	    //}
	    cout << ".\n";
	    //uint32 nummarks=buildcon.edit454TrickyOvercalls(false,newreptmarked);
	    uint32 nummarks=buildcon.editTrickyOvercalls(false,false,readsmarkedsrm);
	    cout << "Edited " << nummarks << " reads.\n";
	    wasovercalledited=nummarks>0;

	    // major edit is edits in >=5% of the reads
	    wasmajorovercalledited=(nummarks >= buildcon.getNumReadsInContig()/20);

//#if DEVELOPMENTVERSION == 0
	    if(wasovercalledited){
	      AS_needsskimfornastyrepeats=true;
	      cout << "Deleting superfluous gap columns ... ";
	      cout.flush();
	      buildcon.deleteStarOnlyColumns(0, buildcon.getContigLength()-1);
	      cout << "done.\n";
	    }
//#endif
#ifdef CLOCK_STEPS2
	    cout << "Timing BFC edit tricky2: " << diffsuseconds(tv) << endl;
	    gettimeofday(&tv,NULL);
#endif
	  }

	  // BaCh 29.04.2011
	  // However, editSingleDiscrepancyNoHAFTag() is conservative enough to allow
	  //  that kind of edits all the time
	  //

	  if(buildcon.getNumReadsInContig() >4){
	    uint32 numedits=buildcon.editSingleDiscrepancyNoHAFTag(readsmarkedsrm);
	    cout << "\nEdited " << numedits << " positions.\n";
	    if(numedits>0){
	      AS_needsskimfornastyrepeats=true;
	      cout << "Deleting superfluous gap columns ... ";
	      cout.flush();
	      buildcon.deleteStarOnlyColumns(0, buildcon.getContigLength()-1);
	      cout << "done.\n";
	    }
#ifdef CLOCK_STEPS2
	    cout << "Timing BFC edit single discrepancy, no HAF: " << diffsuseconds(tv) << endl;
	    gettimeofday(&tv,NULL);
#endif
	  }

	  // if we're in last pass and 454 / pacbio edits were made, loop at least
	  //  once (switching off the 454 / pacbio edits for next pass)
	  if(lastpass 
	     && wasovercalledited
	     && iter==maxiter-1){
	    if(maxiter>=1) {
	      mastermayeditovercalls=false;
	    }
	    maxiter++;
	  }

	  CEBUG("bfc 12/"<<iter << '\n');

#ifdef CLOCK_STEPS2
	  gettimeofday(&tv,NULL);
#endif
	  // get rid of all PSHP tags
	  buildcon.deleteTagsInReads(Read::REA_tagentry_idPSHP);
#ifdef CLOCK_STEPS2
	  cout << "Timing BFC delPSHP: " << diffsuseconds(tv) << endl;
#endif

	  CEBUG("bfc 13/"<<iter << '\n');

#ifdef MIRA_HAS_EDIT
	  if(!newreptmarked && ed_params.ed_automatic_contic_editing!=0){
	    cout << "Editing temporary contig: ";
	    if (buildcon.getNumReadsInContig() >1) {
	      cout << endl;
#ifdef CLOCK_STEPS2
	      gettimeofday(&tv,NULL);
#endif
	      wasedited=true;

	      editContigBack(buildcon, const_cast<EDITParameters &>(eparams));

	      //ScfBuffer::statistics();
	      //ScfBuffer::show();
	      //ScfBuffer::discard();
	      //ScfBuffer::statistics();
	      //ScfBuffer::show();
	    
	      cout << "done.\nDeleting superfluous gap columns ... ";
	      cout.flush();
	      buildcon.deleteStarOnlyColumns(0, buildcon.getContigLength()-1);
	      cout << "done.\n";
	    
#ifdef CLOCK_STEPS2
	      cout << "Timing BFC editconback and more: " << diffsuseconds(tv) << endl;
#endif
	      Contig::setCoutType(Contig::AS_TEXT);
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	      buildcon.stats(cout);
	      if(as_fixparams.as_dateoutput) dateStamp(cout);
	    } else {
	      cout << "(only 1 read in contig, no editing)\n";
	    }
	  }
#endif


	  CEBUG("bfc 14/"<<iter << '\n');
	  
	  // saving again if rept-marked or edited
	  if(newreptmarked || wasedited || wasovercalledited) {
	    if(wasovercalledited) {
	      saveExtTmpContig(buildcon,(basename_forextsave+"post454"));
	    } else {
	      saveExtTmpContig(buildcon,(basename_forextsave+"post"));
	    }
	  } else {
	    if ( as_fixparams.as_output_exttmp_fasta
		 || as_fixparams.as_output_exttmp_ace
		 || as_fixparams.as_output_exttmp_gap4da
		 || as_fixparams.as_output_exttmp_caf) {
	      cout << "No edit and no new repeat found, not saving extra temporary contig again.\n";
	    }
	  }
	}


	CEBUG("bfc 15/"<<iter << '\n');

	CEBUG("bfc 16/"<<iter << '\n');

	// Transfer all the reads fron the new contig into readpool
	//  if no misassembly was detected (or 454 editing happened)
#ifdef CLOCK_STEPS2
	gettimeofday(&tv,NULL);
#endif
	if(!newreptmarked || wasovercalledited || !pbdsev.empty()) {
	  transferContigReadsToReadpool(buildcon, pbdsev, passnr);

#ifdef CLOCK_STEPS2
	  cout << "Timing BFC rp transfer: " << diffsuseconds(tv) << endl;
#endif
	} else {
	  cout << " Transfering read tags to readpool." << endl;
	  transferContigReadTagsToReadpool(buildcon, bbContigI);
#ifdef CLOCK_STEPS2
	  cout << "Timing BFC crtag2rp transfer: " << diffsuseconds(tv) << endl;
#endif
	}
	cout << "Done." << endl;

	CEBUG("bfc 17/"<<iter << '\n');

	//cout << "\nbfc debug checkpoint" << endl;
	//assout::saveAsCAF(con, "bfcdebug.caf", true);
	//cout << "done" << endl;
	//
	//abort();
	
      }

      {
	const vector<Contig::contigread_t> & cr=buildcon.getContigReads();
	vector<Contig::contigread_t>::const_iterator crI=cr.begin();
	
	for(; crI!=cr.end(); crI++){
	  if(crI->orpid>=0){
	    AS_used_ids[crI->orpid]=1;
	  }
	}
      }
      pf_ids_in_contig.clear();

      if(pbdsev.empty()
	 && lastpass
	 && as_fixparams.as_mark_repeats
	 && as_fixparams.as_mark_repeats_onlyinresult
	 && buildcon.getNumReadsInContig() >1){
	CEBUG("bfc 18\n");
#ifdef CLOCK_STEPS2
	gettimeofday(&tv,NULL);
#endif
	vector<bool> dummy;
	newreptmarked=markRepeats(buildcon, dummy);
	foundSRMs|=newreptmarked;
	markrepeatsduringstore=false;
#ifdef CLOCK_STEPS2
	cout << "Timing BFC markrep during store: " << diffsuseconds(tv) << endl;
#endif
      }

      if(buildcon.getNumReadsInContig() >0){
	if(newreptmarked){
	  cout << "\nAccepting probably misassembled contig because of too many iterations.\n";
	}

	CEBUG("bfc 19\n");

#ifdef CLOCK_STEPS2
	gettimeofday(&tv,NULL);
#endif
	bfc_storeContig(buildcon,numcontigs,markrepeatsduringstore,passnr,lastpass);
#ifdef CLOCK_STEPS2
	cout << "Timing BFC store con: " << diffsuseconds(tv) << endl;
#endif
	
	{
	  if(buildcon.getContigLength()>=5000){
	    cout << "Contig coverage analysis ";
	    if(buildcon.getContigLength()>=100000){
	      cout << "(this may take a while) ";
	    }
	    cout << "... "; cout.flush();
	    const Contig::constats_t & cs=buildcon.getStats();
	    for(uint32 st=0; st<Read::SEQTYPE_END; st++){
	      covperstpercon[st].push_back(static_cast<uint32>(cs.avg_covperst[st]+.5));
	      //cout << "cccccccccccc: " << covperstpercon[st].back() << endl;
	    }
	  }
	}

	// did the caller to assemble() ask for a callback for each contig built?
	// if we're on the last past, then call it
	if(lastpass && AS_contigbuilt_callbackfunc!=NULL){
	  (*AS_contigbuilt_callbackfunc)(buildcon, AS_readpool);
	}

	if(as_fixparams.as_spoilerdetection) {
	  //if(as_fixparams.as_keepcontigsinmem) {
	  //	if(!as_fixparams.as_spdetect_lastpassonly
	  //	   || actpass==as_fixparams.as_numpasses-1) {
	  //	  huntPossibleContigJoinGrinches(actpass);
	  //	}
	  //}
	  
	  if(!as_fixparams.as_spdetect_lastpassonly
	     || passnr==as_fixparams.as_numpasses-1) {
	    huntSpoilSports(buildcon);
	  }
	}
      }

      // if we were in mapping mode, on the last contig and not also building 
      //  new contigs, move all remaining reads to debris

      if( !as_fixparams.as_backbone_alsobuildnewcontigs
	  && bbContigI != AS_bbcontigs.end()){
	++bbContigI;  // don't bother to decrease after using this ... will be re-init in next loop anyway
	if(bbContigI == AS_bbcontigs.end()){
	  uint32 adddebris=0;
	  for(size_t uid=0; uid<AS_used_ids.size(); ++uid){
	    if(AS_used_ids[uid]==0){
	      ++adddebris;
	      AS_isdebris[uid]=1;
	      AS_used_ids[uid]=1;
	    }
	  }
	  cout << "Last backbone mapped, not building new ones.\nMoved " << adddebris << " remaining reads to debris.\n"; 
	}
      }  
    }
    //unused=0;
#ifdef CLOCK_STEPS2
    cout << "Timing BFC loop total: " << diffsuseconds(tvloop) << endl;
#endif
  }   //while(unused>0...

  if(unused>0){
    // unused reads? we jumped out of contig creation
    // define unused reads as debris
    for(size_t uid=0; uid<AS_used_ids.size(); ++uid){
      if(AS_used_ids[uid]==0){
	AS_isdebris[uid]=1;
	AS_used_ids[uid]=1;
      }
    }
  }

  if(lastpass) {
    saveDebrisList();
  }else{
    saveDebrisList(passnr, "", "_pass");
  }

  // reads that are debris or singlets apparently need every chance
  //  they can get to align, therefore subsequent passes should not
  //  reduce the overlaps
  // Change: for Solexa reads ... sorry, we'll just miss out
  {
    for(uint32 i=0; i<AS_needalloverlaps.size(); i++){
      if(AS_isdebris[i] && AS_readpool[i].getSequencingType() != Read::SEQTYPE_SOLEXA) AS_needalloverlaps[i]=true;
    }

    list<Contig>::const_iterator clI=AS_contigs.begin();
    vector<int32> contigids;
    for(; clI!=AS_contigs.end(); clI++){
      if(clI->getNumReadsInContig()==1){
	clI->getReadIDsAtContigPosition(contigids,0,0);
	AS_needalloverlaps[contigids[0]]=true;
      }
    }
  }

  // calculate median of average contig coverage
  // TODO: this is calculated only once because of problems
  //  with decreasing average/median in subsequent passes
  //  see if it can be improved
  if(as_fixparams.as_uniform_read_distribution
     && passnr+1>=as_fixparams.as_urd_startinpass
     && !covperstpercon[0].empty() && AS_coverageperseqtype.empty()){
    AS_coverageperseqtype.clear();

    cout << "Setting coverage analysis values for uniform read distribution:\n";
    for(uint32 st=0; st<Read::SEQTYPE_END; st++){
      sort(covperstpercon[st].begin(),covperstpercon[st].end());
      AS_coverageperseqtype.push_back(covperstpercon[st][covperstpercon[st].size()/2]);
      cout << '\t' << Read::getNameOfSequencingType(st) << " coverage:\t" << AS_coverageperseqtype.back() << '\n';
    }
  }

  AS_steps[ASCONTIGSOK]=1;

  AS_used_ids.clear();

  //  saveAsCAF();

  FUNCEND();

  return foundSRMs;
}

#define CEBUG(bla)



/*************************************************************************
 *
 *
 *
 *************************************************************************/

uint32 Assembly::bfc_killIntermediateSinglets()
{
  uint32 counter=0;

  int32 lastrid1=-1;
  int32 numopenoverlaps=-1;

  vector<newedges_t>::const_iterator ceI=AS_confirmed_edges.begin();
  for(;ceI!=AS_confirmed_edges.end();ceI++){
    if(ceI->rid1 != lastrid1){
      if(numopenoverlaps==0 && AS_used_ids[lastrid1]==0){
	AS_isdebris[lastrid1]=1;
	AS_used_ids[lastrid1]=1;
	counter++;
      }
      lastrid1=ceI->rid1;
      numopenoverlaps=0;
    }
    //if(AS_isdebris[ceI->rid1] == 0
    //   && AS_used_ids[ceI->rid1] == 0
    //   && AS_isdebris[ceI->linked_with] == 0
    //   && AS_used_ids[ceI->linked_with] == 0){
    //  numopenoverlaps++;
    //}
    if(AS_used_ids[ceI->rid1] == 0
       && AS_used_ids[ceI->linked_with] == 0){
      numopenoverlaps++;
    }
  }
  if(numopenoverlaps==0){
    AS_isdebris[lastrid1]=1;
    AS_used_ids[lastrid1]=1;
    counter++;
  }

  return counter;
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Assembly::bfc_moveSmallClustersToDebris()
{

  cout << "Moving small clusters to debris:\n";

  uint32 totaldebris=0;

  vector<uint32> numreadsperst(Read::getNumSequencingTypes(),0);
  vector<int32> clusteridperread;
  vector<list<int32> > readinclusterlist;

  clusterUnassembledReads(clusteridperread,readinclusterlist, AS_used_ids);

  uint32 clustercount=0;

  for(size_t ricli=0; ricli<readinclusterlist.size(); ricli++){
    if(!readinclusterlist[ricli].empty()){
      uint32 totalreadsincluster=0;
      numreadsperst.clear();
      numreadsperst.resize(Read::getNumSequencingTypes(),0);
      {
	list<int32>::const_iterator rI=readinclusterlist[ricli].begin();
	for(; rI != readinclusterlist[ricli].end(); ++rI) {
	  numreadsperst[AS_readpool[*rI].getSequencingType()]++;
	  totalreadsincluster++;
	}
      }
      bool takecluster=false;
      for(size_t st=0; st<numreadsperst.size(); ++st){
	if(numreadsperst[st]>0
	   && totalreadsincluster>=AS_miraparams[st].getAssemblyParams().as_minimum_readspercontig){
	  takecluster=true;
	}
      }
      // the above also kill mapping reads
      // therefore, first check whether an unsued rail is part of that cluster
      //  if yes, then don't kill cluster
      if(!takecluster){
	list<int32>::const_iterator rI=readinclusterlist[ricli].begin();
	for(; rI != readinclusterlist[ricli].end(); ++rI) {
	  if(AS_readpool[*rI].isRail()){

	    // should have this ... need to rework how as_usedids is filled (only after contig is made)
	    //&& !AS_used_ids[*rI]){

	    takecluster=false;
	    break;
	  }
	}
      }
      if(!takecluster){
	CEBUG("Killing cluster: " << ricli);
	list<int32>::const_iterator rI=readinclusterlist[ricli].begin();
	for(; rI != readinclusterlist[ricli].end(); ++rI) {
	  CEBUG(" " << *rI);
	  AS_isdebris[*rI]=1;
	  AS_used_ids[*rI]=1;
	  ++totaldebris;
	}
	CEBUG('\n');
      }
    }
  }

  // clusterUnassembledReads() did not return orphans as cluster list
  // therefore, look for unused read ids with no cluster number and also 
  //  put them into debris

  for(size_t uid=0; uid<AS_used_ids.size(); ++uid){
    if(AS_used_ids[uid]==0 && clusteridperread[uid]==-1){
      CEBUG("Killing orphan: " << uid << endl);
      AS_isdebris[uid]=1;
      AS_used_ids[uid]=1;
      ++totaldebris;
    }
  }

  cout << "\nDone. " << totaldebris << " reads moved to debris.\n";

  return;
}



/*************************************************************************
 *
 * Checks if Contig meets specified requirements (atm: num of reads)
 * 
 * If not, mark the reads in the contig as debris and empty the contig
 *
 *************************************************************************/

bool Assembly::bfc_checkIfContigMeetsRequirements(Contig & con)
{
  bool contigok=false;

  vector<uint32> numreadsperst(Read::getNumSequencingTypes(),0);
  const vector<Contig::contigread_t> & conreads=con.getContigReads();
  uint32 totalreadsincon=0;
  {
    vector<Contig::contigread_t>::const_iterator crI = conreads.begin();
    for(;crI != conreads.end(); crI++){
      if(crI->orpid >= 0){
	if(crI->read.isBackbone()){
	  contigok=true;
	  break;
	}
	++numreadsperst[crI->read.getSequencingType()];
	++totalreadsincon;
      }
    }
  }

  if(!contigok){
    for(size_t st=0; st<numreadsperst.size(); ++st){
      if(numreadsperst[st]>0
	 && totalreadsincon>=AS_miraparams[st].getAssemblyParams().as_minimum_readspercontig){
	contigok=true;
      }
    }
  }

  if(!contigok){
    cout << "\nContig does not meet requirement of minimum reads per contig."
      "\nMoving " << totalreadsincon << " reads to debris.\n";
    vector<Contig::contigread_t>::const_iterator crI = conreads.begin();
    for(;crI != conreads.end(); crI++){
      if(crI->orpid >= 0){
	AS_isdebris[crI->orpid]=1;
	AS_used_ids[crI->orpid]=1;
      }
    }
    con.discard();
  }


  return contigok;
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Assembly::bfc_markRepReads(Contig & con)
{
  const vector<Contig::contigread_t> & conreads=con.getContigReads();
  vector<Contig::contigread_t>::const_iterator crI = conreads.begin();

  for(;crI != conreads.end(); crI++){
    if(crI->orpid >= 0
      && AS_multicopies[crI->orpid]) {
      cout << "xxxxxxxxxxxx mark " << crI->orpid << endl;
      Read & nonconstread = const_cast<Read &>(crI->read);
      int32 rc=nonconstread.getRightClipoff()-1;
      if(rc<nonconstread.getLeftClipoff()) rc=nonconstread.getLeftClipoff();
      nonconstread.addTag(nonconstread.getLeftClipoff(),
			  rc,
			  "tstR",
			  "");
    }
  }
}


/*************************************************************************
 *
 *
 *
 *************************************************************************/

void Assembly::bfc_storeContig(Contig & con, uint32 & numcontigs, const bool mustmarkrepeats, const int32 passnr, const bool lastpass)
{
  assembly_parameters const & as_fixparams= AS_miraparams[0].getAssemblyParams();
  const vector<Contig::contigread_t> & conreads=con.getContigReads();

  // look whether we store singlets in contig or not

  bool storecontig=true;

  if(con.getNumReadsInContig()==1){
    const assembly_parameters & as_rt_params = AS_miraparams[conreads[0].read.getSequencingType()].getAssemblyParams();

    storecontig=as_rt_params.as_savesimplesingletsinproject;
    if(conreads[0].read.hasTag(Read::REA_tagentry_idSRMr)
       || conreads[0].read.hasTag(Read::REA_tagentry_idCRMr)
       || conreads[0].read.hasTag(Read::REA_tagentry_idWRMr)
       || conreads[0].read.hasTag(Read::REA_tagentry_idSROr)
       || conreads[0].read.hasTag(Read::REA_tagentry_idSAOr)
       || conreads[0].read.hasTag(Read::REA_tagentry_idSIOr)) {
      storecontig|=as_fixparams.as_savetaggedsingletsinproject;
      if(conreads[0].orpid>=0) AS_needalloverlaps[conreads[0].orpid]=true;
     }
  }

  // TODO: U13 hat 0 werte im ASCII?
  //Contig::setCoutType(Contig::AS_DEBUG);
  //cout << "Debug in bfc_storeContig()\n" << con;

  if(storecontig){
    if(as_fixparams.as_mark_repeats&&mustmarkrepeats){
      vector<bool> dummy;
      markRepeats(con, dummy);
    }

    //bfc_markRepReads(con);

    cout << "Storing contig ... "; cout.flush();
    cout << as_fixparams.as_mark_repeats << mustmarkrepeats;
    if(as_fixparams.as_keepcontigsinmem){
      con.markFeaturesByConsensus(true, true, true);
      con.updateStatsFromConsensusTags(true,true,true,true,true);
      // store the contig information
      AS_assemblyinfo.storeContigStats(con.getStats());
      con.saveMem();
      AS_contigs.push_back(con);
    }else{
      if(as_fixparams.as_loadbackbone){
	con.removeRails();

	// TODO: ask contig whether it has mappings
	//if(as_fixparams.as_loadSOLEXA || as_fixparams.as_loadSOLID){
	if(AS_miraparams[Read::SEQTYPE_SOLEXA].getAssemblyParams().as_load_sequencedata 
	   || AS_miraparams[Read::SEQTYPE_ABISOLID].getAssemblyParams().as_load_sequencedata) {
	  cout << "Transforming CER mappings." << endl;
	  con.transformCERMappingsToCoverageReads();
	  cout << "done transforming CER mappings." << endl;
	  //assout::saveAsMAF(con, getMAFFilename()+".bla", AS_deleteoldresultfiles);
	}
      }

      con.markFeaturesByConsensus(true, true, true);
      // transfer important tags to readpool
      transferContigReadTagsToReadpool(con,AS_bbcontigs.end());

      con.updateStatsFromConsensusTags(true,true,true,true,true);
      // store the contig information
      AS_assemblyinfo.storeContigStats(con.getStats());


      if(lastpass) {
	assout::saveStatistics(con,
			       getStatisticsFilename(),
			       AS_deleteoldresultfiles);
	assout::saveReadTagList(con,
				getReadTagListFilename(),
				AS_deleteoldresultfiles);
	assout::saveConsensusTagList(con,
				     getConsensusTagListFilename(),
				     AS_deleteoldresultfiles);
	assout::saveContigReadList(con,
				   getContigReadListFilename(),
				   AS_deleteoldresultfiles);
	if(as_fixparams.as_output_gff3){
	  // store sequence for later
	  AS_gff3defer_names.push_back(con.getContigName());
//TODO: weiterhier	  
//	  assout::saveTagsAsGFF3(con, getGFF3Filename(), AS_deleteoldresultfiles);
	}
	if(as_fixparams.as_output_caf){
	  cout << "Saving CAF ... "; cout.flush();
	  assout::saveAsCAF(con, getCAFFilename(), AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_maf){
	  cout << "Saving MAF ... "; cout.flush();
	  assout::saveAsMAF(con, getMAFFilename(), AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_gap4da){
	  cout << "Saving gap4 direct assembly ... "; cout.flush();
	  assout::saveAsGAP4DA(con,getGAP4DAFilename(),AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_fasta) {
	  cout << "Saving FASTA ... "; cout.flush();
	  assout::saveAsFASTA(con,
			      getFASTAFilename(),
			      getFASTAPaddedFilename(),
			      AS_deleteoldresultfiles);

	  cout << "done.\n";
	  
	  // TODO: enable these functions for incremental write
	  //if(AS_readpool.getNumOfStrainInReadpool()>1){
	  //  saveStrainsAsFASTAQUAL();
	  //}
	}
	if(as_fixparams.as_output_tcs) {
	  cout << "Saving TCS ... "; cout.flush();
	  assout::saveAsTCS(con, getTCSFilename(),AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_wiggle) {
	  cout << "Saving Wiggle ... "; cout.flush();
	  assout::saveAsWiggle(con, getWiggleFilename(),AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	// TODO: enable these functions for incremental write
	//saveSNPAnalysis();
	//saveFeatureAnalysis();
	if(as_fixparams.as_output_txt){
	  cout << "Saving text ... "; cout.flush();
	  assout::saveAsTXT(con,getTXTFilename(),AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_ace){
	  cout << "Saving ACE ... "; cout.flush();
	  assout::saveAsACE(con,getACEFilename(),AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_html) {
	  cout << "Saving HTML ... "; cout.flush();
	  assout::dumpContigAsHTML(con,
				   getHTMLFilename(),
				   AS_deleteoldresultfiles,
				   AS_miraparams[0].getAssemblyParams().as_projectname_out);
	  cout << "done.\n";
	}
      }else{
	assout::saveStatistics(con,
			       getStatisticsFilename(passnr, "", "_pass"),
			       AS_deleteoldresultfiles);
	assout::saveReadTagList(con, 
				getReadTagListFilename(passnr),
				AS_deleteoldresultfiles);
	assout::saveConsensusTagList(con,getConsensusTagListFilename(passnr),
				     AS_deleteoldresultfiles);
	assout::saveContigReadList(con,
				   getContigReadListFilename(passnr, "", "_pass"),
				   AS_deleteoldresultfiles);
	    
	if(as_fixparams.as_output_tmp_caf) {
	  cout << "Saving temp CAF ... "; cout.flush();
	  assout::saveAsCAF(con, 
			    getCAFFilename(passnr, "", "_pass"),
			    AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_tmp_maf) {
	  cout << "Saving temp MAF ... "; cout.flush();
	  assout::saveAsMAF(con, 
			    getMAFFilename(passnr, "", "_pass"),
			    AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_tmp_gap4da) {
	  cout << "Saving temp gap4 direct assembly ... "; cout.flush();
	  assout::saveAsGAP4DA(con,
			       getGAP4DAFilename(passnr, "", "_pass"),
			       AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_tmp_fasta){
	  cout << "Saving temp FASTA ... "; cout.flush();
	  assout::saveAsFASTA(con,
			      getFASTAFilename(passnr, "", "_pass"),
			      getFASTAPaddedFilename(passnr, "", "_pass"),
			      AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_tmp_txt){
	  cout << "Saving temp text ... "; cout.flush();
	  assout::saveAsTXT(con,
			    getTXTFilename(passnr, "", "_pass"),
			    AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_tmp_ace) {
	  cout << "Saving temp ACE ... "; cout.flush();
	  assout::saveAsACE(con,
			    getACEFilename(passnr, "", "_pass"),
			    AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	if(as_fixparams.as_output_tmp_tcs) {
	  cout << "Saving temp TCS ... "; cout.flush();
	  assout::saveAsTCS(con,
			    getTCSFilename(passnr, "", "_pass"),
			    AS_deleteoldresultfiles);
	  cout << "done.\n";
	}
	//if(as_fixparams.as_output_tmp_html) saveAsHTML(passnr, "", "_pass");
	if(as_fixparams.as_output_tmp_html) {
	  cout << "Saving temp HTML ... "; cout.flush();
	  assout::dumpContigAsHTML(con,
				   getHTMLFilename(passnr, "", "_pass"),
				   AS_deleteoldresultfiles,
				   AS_miraparams[0].getAssemblyParams().as_projectname_out);
	  cout << "done.\n";
	}
      }
    }
    cout << "done." << endl;
  }else{
    // store the contig information
    AS_assemblyinfo.storeContigStats(con.getStats());

    if(conreads[0].orpid>=0) AS_isdebris[conreads[0].orpid]=1;
    numcontigs--;
  }

  AS_deleteoldresultfiles=false;
}




/*************************************************************************
 *
 *
 *
 *************************************************************************/


bool Assembly::markRepeats(Contig & con, vector<bool> & readsmarkedsrm)
{
  FUNCSTART("void Assembly::markRepeats(Contig & con)");

  cout << "Marking possibly misassembled repeats: ";
  cout.flush();
  
  Contig::repeatmarker_stats_t repstats;
  con.newMarkPossibleRepeats(repstats, readsmarkedsrm);
  cout << "done step 1, starting step 2:";
  con.codonSingleBaseRepeatMarker(6,repstats, readsmarkedsrm);
  if(repstats.numSRMs>0 || repstats.numWRMs>0 || repstats.numSNPs>0){
    cout << "\nFound\n";
    cout << " - " << repstats.numSRMs << " new Strong RMB (SRMc)\n";
    cout << " - " << repstats.numWRMs << " new Weak RMB (WRMc)\n";
    cout << " - " << repstats.numSNPs << " SNP\npositions tagged.";
    cout.flush();
  }else{
    cout << "done. Found none." << endl;
  }

  FUNCEND();

  return repstats.numSRMs>0;
}





/*************************************************************************
 *
 *
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush(); }
void Assembly::transferContigReadsToReadpool(const Contig & buildcon, list<Contig::pbdse_t> & pbdsev, int32 passnr)
{
  FUNCSTART("void Assembly::transferContigReadsToReadpool(const Contig & buildcon, vector<Contig::pbdse_t> & pbdsev)");

  cout << "Transfering reads to readpool." << endl;

  const vector<Contig::contigread_t> & cr = buildcon.getContigReads();

  // split one list pbdsev into sublists for each contigread if needed

  CEBUG("tCRTR 1" << endl);
  vector<list<Contig::pbdse_t> > vpbdsev;
  if(!pbdsev.empty()){
    vpbdsev.resize(AS_readpool.size());
    for(; pbdsev.begin() != pbdsev.end(); ){
      //cout << "ls: " << pbdsev.size() << "\tSplicing: " << pbdsev.front();
      size_t trid=pbdsev.front().rid;
      vpbdsev[trid].splice(vpbdsev[trid].begin(),pbdsev,pbdsev.begin());
      //cout << "ts: " << vpbdsev[trid].size() << endl;
    }
    
    //for(uint32 vi=0;vi<vpbdsev.size(); ++vi){
    //  cout << "vi: " << vi << endl;
    //  list<Contig::pbdse_t>::const_iterator pI=vpbdsev[vi].begin();
    //  for(; pI!=vpbdsev[vi].end(); ++pI){
    //	cout << "\t" << *pI;
    //  }
    //}
  }

  CEBUG("tCRTR 2" << endl);
  // create a temporary read with enough capacity to hold the
  //  largest of the reads to transfer (to prevent re-allocation)
  //
  // used for PacBio where elastic N stretches need to
  //  be corrected
  // also used to remove the gaps from the reads
  vector<Contig::contigread_t>::const_iterator crI = cr.begin();
  Read tmpr;
  {
    uint32 reservelen=0;
    for(;crI!=cr.end();crI++){
      if(crI->orpid>=0){
	reservelen=max(reservelen,crI->read.getLenSeq());
      }
    }
    tmpr.reserve(reservelen);
  }

  CEBUG("tCRTR 3" << endl);
  // now copy all contig reads to read pool, one by one, using the tmp read
  //  and performing edits on this tmp read, leaving contig reads untouched
  crI = cr.begin();
  for(;crI!=cr.end();crI++){
    if(crI->orpid>=0){
      tmpr=crI->read;

      // carefull, maybe there is nothing in the vpbdsev vector!
      //  then one would get segfaults
      if(tmpr.isSequencingType(Read::SEQTYPE_PACBIO) && !vpbdsev.empty()){
	list<Contig::pbdse_t>::const_iterator pRI=vpbdsev[crI->orpid].begin();
	for(; pRI != vpbdsev[crI->orpid].end(); ++pRI){
	  CEBUG("Apply: " << *pRI);
	  BUGIFTHROW(pRI->rid >= AS_readpool.size(), "pRI->rid (" << pRI->rid << ") >= AS_readpool.size() ?");
	  //Read::setCoutType(Read::AS_TEXT);
	  //cout << CON_reads[pRI->cri].read;
	  tmpr.correctNStretch(pRI->rposs,
			       pRI->rpose,
			       pRI->changeestim);
	}

	if(passnr==4){
	  tmpr.transformGapsToNs();
	}
      }

      tmpr.removeGapsFromRead();
      AS_readpool[crI->orpid]=tmpr;
    }
  }

  FUNCEND();
}
//#define CEBUG(bla)


/*************************************************************************
 *
 *
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush(); }
void Assembly::transferContigReadTagsToReadpool(const Contig & con, const list<Contig>::const_iterator bbContigI)
{
  FUNCSTART("void Assembly::transferContigReadTagsToReadpool(const Contig & con, const list<Contig>::const_iterator bbContigI)");

  cout << "Transfering tags to readpool.\n";
  
  const vector<Contig::contigread_t> & cr = con.getContigReads();
  vector<Contig::contigread_t>::const_iterator crI = cr.begin();
  uint32 tagnumber=0;

  try{
    // Transfer the contigread RMB tags into the readpool only!
    // Go through all the contigreads, if they have SRMr or WRMr tags,
    //  check if they are at an edited place.
    // If not, transfer the tag to the readpool (if not already present
    //  there)
    for(;crI!=cr.end(); crI++){
      if(crI->orpid==-1) continue;
      uint32 numtags=crI->read.getNumOfTags();
      
      //CEBUGF(crI->read.getName() << " has " << numtags << " tags.\n");
      
      for(tagnumber=0; tagnumber < numtags; tagnumber++) {
	const multitag_t & acttag=crI->read.getTag(tagnumber);
	if(acttag.identifier==Read::REA_tagentry_idSRMr
	   ||acttag.identifier==Read::REA_tagentry_idWRMr
	   ||acttag.identifier==Read::REA_tagentry_idSROr
	   ||acttag.identifier==Read::REA_tagentry_idSIOr
	   ||acttag.identifier==Read::REA_tagentry_idSAOr
	  ) {
	  
	  //CEBUGF("Tag " << tagnumber << " at " << acttag.from << " is " << acttag.identifier << endl);
	  
	  bool foundedit=false;
	  for(uint32 i=0; i<numtags; i++) {
	    if(crI->read.getTag(i).from==acttag.from
	       && (crI->read.getTag(i).identifier==Read::REA_tagentry_idED_C
		   || crI->read.getTag(i).identifier==Read::REA_tagentry_idED_I
		   || crI->read.getTag(i).identifier==Read::REA_tagentry_idED_D
		 )
	      ) {
	      //CEBUGF("Found " << crI->read.getTag(i).identifier << " at that position, skipping!");
	      foundedit=true;
	      break;
	    }
	  }
	  if(foundedit) continue;
	  
	  
	  // the next if is true for tags at gap bases;
	  //if(adj_tagpos < 0 ) {
	  //  cout << "Tag at gap base!!!!!\n";
	  //  continue;
	  //}

	  if(bbContigI!=AS_bbcontigs.end()){
	    if(crI->read.isBackbone() || crI->read.isRail()) {
	      //cout << "IsBackbone or read\n";
	      
	      int32 adj_tagposl=crI->read.getAdjustmentPosOfReadPos(acttag.from);
	      int32 adj_tagposu=crI->read.getAdjustmentPosOfReadPos(acttag.to);
	      // don't do that for gaps in backbones or reads
	      if(adj_tagposl < 0 && adj_tagposu < 0) continue;
	      if(adj_tagposl < 0) {
		adj_tagposl=adj_tagposu-(acttag.to-acttag.from);
		if(adj_tagposl<0) adj_tagposl=0;
	      }else if(adj_tagposu < 0) {
		adj_tagposu=adj_tagposl+(acttag.to-acttag.from);
		if(adj_tagposu > static_cast<int32>(crI->read.getLenSeq()-1)) adj_tagposu=crI->read.getLenSeq()-1;
	      }
	      
	      // This is an ugly, ugly hack: we set the P|WRMB tag in
	      //  the read of the backbonecontig. 
	      // This should be a no no, but it's needed to work
	      //  with backbones
	      Read & nonconstread = const_cast<Read &>(bbContigI->getReadWithID(crI->orpid));
	      
	      //int32 adj_tagposl=crI->read.getAdjustmentPosOfReadPos(acttag.from);
	      //int32 adj_tagposu=crI->read.getAdjustmentPosOfReadPos(acttag.to);
	      int32 bbcrtagposl=nonconstread.getReadPosOfAdjustmentPos(adj_tagposl);
	      int32 bbcrtagposu=nonconstread.getReadPosOfAdjustmentPos(adj_tagposu);
	      if(bbcrtagposl >= 0 && bbcrtagposu>=0) {
		try {
		  nonconstread.addTag(bbcrtagposl, 
				      bbcrtagposu,
				      acttag.identifier,
				      acttag.comment);
		}
		catch (Notify n) {
		  cout << "Tried to transfer tags to bbackbone contig read from:\n";
		  Read::setCoutType(Read::AS_TEXT);
		  cout << crI->read;
		  cout << "Exiting.\n";
		  n.handleError(THISFUNC);
		}
	      }
	    }
	  }
	    
	  Read & rpr=AS_readpool.getRead(crI->orpid);
		  
	  // TODO: FIXME
	  // this is wrong for reads without adjustments. 
	  //  -> no way to detect deletions in those reads!
	  int32 adj_tagposl=crI->read.getLowerNonGapAdjustmentPosOfReadPos(acttag.from);
	  int32 adj_tagposu=crI->read.getUpperNonGapAdjustmentPosOfReadPos(acttag.to);
	  int32 rprtagposl=rpr.getReadPosOfAdjustmentPos(adj_tagposl);
	  int32 rprtagposu=rpr.getReadPosOfAdjustmentPos(adj_tagposu);

	  try {
	    //cout << "Transfering tag for " << rpr.getName()
	    //	 << "\t" << rprtagposl << " " << rprtagposu 
	    //	 << "\tadj " << adj_tagposl << " " << adj_tagposu << endl;
	    //CEBUGF("Transfering tag for " << rpr.getName() << "\t" << rprtagposl << " " << rprtagposu << endl);
	    //Read::setCoutType(Read::AS_TEXT);
	    //CEBUGF("Before:\n" << rpr << endl);
		    
	    // add tag only when both positions are >=0 (i.e. not starting/stopping
	    //  on an inserted base, else addTag() would understandably barf
	    if(rprtagposl>=0 && rprtagposu>=0){
	      rpr.addTag(rprtagposl,
			 rprtagposu, 
			 acttag.identifier, 
			 acttag.comment);
	    }
	    //CEBUGF("After:\n" << rpr << endl);
	  }
	  catch (Notify n) {
	    // care about errors only if we have adjustments in the read
	    //  if not, the whole thing is more or less wrong anyway
	    if(rpr.usesAdjustments()){
	      cout << "Tried to transfer tags to readpool read from:\n";
	      Read::setCoutType(Read::AS_TEXT);
	      cout << crI->read;
	      cout << "Exiting." << endl;
	      n.handleError(THISFUNC);
	    }else{
	      cout << "Dbg: wrong tag transfer for " << rpr.getName() << '\n';
	    }
	  }
	}
      }
    }
  }
  catch(Notify n) {
    cout << "General error while transfering tag " << tagnumber << " to readpool read from:\n";
    Read::setCoutType(Read::AS_TEXT);
    cout << crI->read;
    cout << "Exiting.\n";
    n.handleError(THISFUNC);
  }

  FUNCEND();
}

//#define CEBUG(bla)



/*************************************************************************
 *
 * New version to search for spoil sports
 * Works with one contig at a time
 *
 * Look through all reads. If end of read with (500) bases of end of contig
 *  and has permbans and no freq >4:
 *     throw out (later version perhaps cut back)
 *
 *
 *************************************************************************/
// idea: keep info whether read has been totally embedded (>1000 bases left, right)
// if true, then less likely to be an error

void Assembly::huntSpoilSports(Contig & chkcon)
{
  cout << "Hunting join spoiler" << endl;

  const vector<Contig::contigread_t> & cr=chkcon.getContigReads();

  const uint32 endrange=30;

  vector<Contig::contigread_t>::const_iterator crI=cr.begin();
  for(; crI != cr.end() ; crI++){
    const Read & actread=crI->read;
    bool atfront=false;
    bool atback=false;
    if(crI->offset<=endrange) atfront=true;
    if(crI->offset+actread.getLenClippedSeq() > chkcon.getContigLength()-endrange) atback=true;
    if(atfront || atback){
      cout << "HSS At end: " << atfront << ' ' << atback << '\t' << actread.getName() << endl;

      if(crI->orpid>=0){
	// currently: always remove

	// if it has >2 permbans, remove
	//cout << "HSS Permbans: " << AS_permanent_overlap_bans[crI->orpid].size() << endl;
	//if(AS_permanent_overlap_bans[crI->orpid].size()>3) {
	//  if(!actread.hasTag(Read::REA_tagentry_idSRMr)
	//     && !actread.hasTag(Read::REA_tagentry_idCRMr)){
	//    //if(true){
	//    cout << "HSS remove " << actread.getName() << endl;
	//
	//    const vector<Read::bposhashstat_t> & bposhashstats=actread.getBPosHashStats();
	//    if(!bposhashstats.empty()){
	//      bool clipfront=false;
	//      bool clipback=false;
	//      if(atfront){
	//	if(crI->direction>0){
	//	  clipfront=true;
	//	}else{
	//	  clipback=true;
	//	}
	//      }
	//      if(atback){
	//	if(crI->direction>0){
	//	  clipback=true;
	//	}else{
	//	  clipfront=true;
	//	}
	//      }
	//      cout << "HSS clip front back " << clipfront << ' ' << clipback << endl;
	//      if(clipfront){
	//	uint32 bposfrom=actread.getLeftClipoff();
	//	uint32 maxcheck=max(actread.getLenSeq()/4,static_cast<uint32>(50));
	//	uint32 maxto=min(maxcheck,actread.getLenSeq()-1);
	//	bool foundinv=false;
	//	for(; bposfrom<maxto; bposfrom++){
	//	  if(!bposhashstats[bposfrom].fwd.isValid()){
	//	    foundinv=true;
	//	    break;
	//	  }else{
	//	    if(bposhashstats[bposfrom].fwd.getFrequency()>=4) break;
	//	  }
	//	}
	//	if(foundinv){
	//	  if(bposfrom-actread.getLeftClipoff()<50){
	//	    bposfrom=actread.getLeftClipoff()+50;
	//	    if(bposfrom>=actread.getLenSeq()) bposfrom=actread.getLenSeq()-1;
	//	  }
	//	  cout << "HSS moving left " <<  AS_readpool[crI->orpid].getLQClipoff();
	//	  AS_readpool[crI->orpid].setLQClipoff(bposfrom);
	//	  cout << " to " << AS_readpool[crI->orpid].getLQClipoff() 
	//	       << "\t" << actread.getName() << endl;
	//	}
	//      }
	//      if(clipback){
	//	uint32 bposto=actread.getRightClipoff();
	//	uint32 maxcheck=max(actread.getLenSeq()/4,static_cast<uint32>(50));
	//	bool foundinv=false;
	//	for(; bposto>0 && maxcheck>0; --bposto, --maxcheck){
	//	  if(!bposhashstats[bposto].rev.isValid()){
	//	    foundinv=true;
	//	    break;
	//	  }
	//	  if(bposhashstats[bposto].rev.getFrequency()>=4) break;
	//	}
	//	if(foundinv){
	//	  if(actread.getRightClipoff()-bposto<50){
	//	    if(actread.getRightClipoff()<50){
	//	      bposto=0;
	//	    }else{
	//	      bposto=actread.getRightClipoff()-50;
	//	    }
	//	  }
	//	  cout << "HSS moving right " <<  AS_readpool[crI->orpid].getRQClipoff();
	//	  AS_readpool[crI->orpid].setRQClipoff(bposto);
	//	  cout << " to " << AS_readpool[crI->orpid].getRQClipoff() 
	//	       << "\t" << actread.getName() << endl;
	//	}
	//      }
	//    }
	//  }
	//}
	AS_istroublemaker[crI->orpid]=true;
      }
    }
  }
}


/*************************************************************************
 *
 * This should catch chimeras as well as vector leftovers that were
 *  too long to be clipped away earlier
 *  
 * go through each contig; look at reads aligning with end reads 
 *  and see whether those have edges to reads in other contigs to 
 *  which the end read has no edges
 *
 * Catches situations like these (grinch is 1st read)
 *
 * Contig1
 * ...................................XXXXmismatchesXXXX
 * ........................
 *
 * Contig2
 *                ......................................................
 *                     .................................................
 *
 *************************************************************************/

//#define CEBUG(bla)   {cout << bla; cout.flush(); }
//#define CEBUGF(bla)  {cout << bla; cout.flush(); }

void Assembly::huntPossibleContigJoinGrinches(int32 version, const string prefix, const string postfix, const string tmpfname)
{
  FUNCSTART("void Assembly::huntPossibleContigJoinGrinches(int32 version, const string prefix, const string postfix, const string tmpfname)");

  cout << "\n\n";
  if(AS_miraparams[0].getAssemblyParams().as_dateoutput) dateStamp(cout);
  cout << "Hunting contig join spoiler ... "; cout.flush();

  string filename;
  if(tmpfname.size()){
    filename=buildFileName(version, prefix, postfix, tmpfname, ".txt");
  }else{
    filename=buildFileName(version, prefix, postfix, 
			   AS_miraparams[0].getAssemblyParams().as_tmpf_spoiler,
			   ".txt");
  }

  ofstream logout(filename.c_str(), ios::out | ios::trunc);


  list<Contig>::iterator I=AS_contigs.begin();
  vector<int32> contigidofendreads(AS_readpool.size(),-1);

  //TODO: +/- fehler rausholen, aufrumen, konfigurierbar machen

  {
    vector<int32> tmpreadids;
    vector<int32>::const_iterator J;
    uint32 contigid;
    int32 tmplen=0;
    for(;I!=AS_contigs.end();I++){
      //      if(I->getNumReadsInContig()==1
      //	 || I->getContigLength() <= 500) continue;
      if(I->getNumReadsInContig()==1) continue;

      contigid=I->getContigID();
      tmplen=I->getContigLength();
      if(tmplen>500) tmplen=500;
      I->getReadIDsAtContigPosition(tmpreadids,0,tmplen);
      for(J=tmpreadids.begin();J!=tmpreadids.end();J++){
	contigidofendreads[*J]=contigid;
      }

      tmplen=I->getContigLength()-500;
      if(tmplen<0) tmplen=0;
      I->getReadIDsAtContigPosition(tmpreadids,tmplen,I->getContigLength());
      for(J=tmpreadids.begin();J!=tmpreadids.end();J++){
	contigidofendreads[*J]=contigid;
      }
    }
  }

  for(uint32 i=0; i<contigidofendreads.size(); i++){
    CEBUG("CEndread " << i << " :\t");
    if(contigidofendreads[i]>=0) {
      CEBUG(contigidofendreads[i] << '\n');
    }else{
      CEBUG("none\n");
    }
  }

  //for(uint32 i=0; i<contigidofendreads.size(); i++){
  //  cout << "read " << AS_readpool[i].getName() << " " << i << " in " << contigidofendreads[i] << endl;
  //}

  uint32 actcontigid;
  for(I=AS_contigs.begin();I!=AS_contigs.end();I++){
    if(I->getNumReadsInContig()==1) continue;
    CEBUG("Checking " << I->getContigName() << " " << endl);
    actcontigid=I->getContigID();
    vector<int32> endreads;
    int32 xfirstcov=0;
    I->getReadIDsAtContigPosition(endreads,xfirstcov,xfirstcov);
    if(endreads.size()==1) {
      xfirstcov++;
      int32 endreadid=endreads[0];
      CEBUG("readid at front end: " << endreadid << endl);
      while(endreads.size()==1) {
	I->getReadIDsAtContigPosition(endreads,xfirstcov,xfirstcov);
	xfirstcov++;
      }
      int32 xendcov=xfirstcov+300;
      CEBUG("range after 1 cov at front: " << xfirstcov << " " << xendcov<<endl);
      if(xendcov >= static_cast<int32>(I->getContigLength())) xendcov=I->getContigLength()-1;
      CEBUG("adjusted range after 1 cov at front: " << xfirstcov << " " << xendcov<<endl);
      I->getReadIDsAtContigPosition(endreads,xfirstcov,xendcov);
      
      if(isReadGrinch(endreadid, actcontigid, endreads, contigidofendreads)){
	CEBUG("Looks like a grinch.\n");
	const vector<Contig::contigread_t> & creads= I->getContigReads();
	vector<Contig::contigread_t>::const_iterator crI=creads.begin();
	int32 direction=0;
	for(;crI!=creads.end();crI++){
	  if(crI->orpid == endreadid) {
	    direction=crI->direction;
	    break;
	  }
	}
	// add another 20 bases as safety net
	xfirstcov+=20;
	BUGIFTHROW(direction==0, "direction == 0???");
	logout << "Spoiler " << AS_readpool.getRead(endreadid).getName() << " " << xfirstcov << " ";
	uint32 tagstart=0;
	uint32 tagend=0;
	if(direction>0){
	  try{
	    AS_readpool.getRead(endreadid).setLQClipoff(AS_readpool.getRead(endreadid).getLQClipoff()+xfirstcov-1);
	    logout << "on left side.\n";
	    tagstart=AS_readpool.getRead(endreadid).getLQClipoff()-(xfirstcov-1);
	    tagend=AS_readpool.getRead(endreadid).getLQClipoff();
	    AS_readpool.getRead(endreadid).addTag(tagstart, tagend,
						  "CJSP","");
	  }
	  catch(...) {
	  }
	} else {
	  try{
	    AS_readpool.getRead(endreadid).setRQClipoff(AS_readpool.getRead(endreadid).getRQClipoff()-xfirstcov+1);
	    logout << "on right side.\n";
	    tagstart=AS_readpool.getRead(endreadid).getRQClipoff();
	    tagend=AS_readpool.getRead(endreadid).getRQClipoff()+(xfirstcov-1);
	    AS_readpool.getRead(endreadid).addTag(tagstart, tagend,
						  "CJSP","");
	  }
	  catch(...) {
	  }
	}
      }
    }




    xfirstcov=I->getContigLength()-1;
    I->getReadIDsAtContigPosition(endreads,xfirstcov,xfirstcov);
    if(endreads.size()==1) {
      xfirstcov--;
      int32 endreadid=endreads[0];
      CEBUG("readid at back end: " << endreadid << endl);
      while(endreads.size()==1) {
	I->getReadIDsAtContigPosition(endreads,xfirstcov,xfirstcov);
	xfirstcov--;
      }
      int32 xendcov=xfirstcov-300;
      if(xendcov <0) xendcov=0;
      CEBUG("range after 1 cov at front: " << xendcov << " " << xfirstcov << endl);
      I->getReadIDsAtContigPosition(endreads,xendcov,xfirstcov);
      
      if(isReadGrinch(endreadid, actcontigid, endreads, contigidofendreads)){
	CEBUG("Looks like a grinch.\n");

	const vector<Contig::contigread_t> & creads= I->getContigReads();
	vector<Contig::contigread_t>::const_iterator crI=creads.begin();
	int32 direction=0;
	for(;crI!=creads.end();crI++){
	  if(crI->orpid == endreadid) {
	    direction=crI->direction;
	    break;
	  }
	}
	// add another 20 bases as safety net
	xfirstcov-=20;
	BUGIFTHROW(direction==0, "direction == 0???");
	logout << "Spoiler " << AS_readpool.getRead(endreadid).getName() << " " << (I->getContigLength()-xfirstcov) << " ";
	uint32 tagstart=0;
	uint32 tagend=0;
	if(direction>0){
	  try{
	    AS_readpool.getRead(endreadid).setRQClipoff(AS_readpool.getRead(endreadid).getRQClipoff()-(I->getContigLength()-xfirstcov+1));
	    logout << "on right side.\n";
	    tagstart=AS_readpool.getRead(endreadid).getRQClipoff();
	    tagend=AS_readpool.getRead(endreadid).getRQClipoff()+(I->getContigLength()-xfirstcov)-1;
	    AS_readpool.getRead(endreadid).addTag(tagstart, tagend,
						  "CJSP","");
	  }
	  catch(...) {
	  }
	} else {
	  try{
	    AS_readpool.getRead(endreadid).setLQClipoff(AS_readpool.getRead(endreadid).getLQClipoff()+(I->getContigLength()-xfirstcov-1));
	    logout << "on left side.\n";
	    tagstart=AS_readpool.getRead(endreadid).getLQClipoff()-(I->getContigLength()-xfirstcov)+1;
	    tagend=AS_readpool.getRead(endreadid).getLQClipoff()-1;
	    AS_readpool.getRead(endreadid).addTag(tagstart, tagend,
						  "CJSP","");
	  }
	  catch(...) {
	  }
	}
      }
    }
  }

  logout.close();

  cout << "done.\n";
  if(AS_miraparams[0].getAssemblyParams().as_dateoutput) dateStamp(cout);

  FUNCEND();
  return;
}
//#define CEBUG(bla)
//#define CEBUGF(bla)


/*************************************************************************
 *
 * thereadid is an endread (the one having the end-coverage of 1
 * incontigid is the id of the contig it is in
 * readstocheck are the read at the beginning of a contig that overlap
 *  "thereadid"
 * contigidofendreads is a vector in which each read (that is an endread)
 *  has the id of the contig it is in
 *
 * returns whether "thereadid" matches more than one other read in 
 *  another contig
 *
 *************************************************************************/

bool Assembly__compareNewEdges_t_(const newedges_t & a, 
				    const newedges_t & b);
bool Assembly__compareNewEdges_t_(const newedges_t & a, const newedges_t & b)
{
  return a.rid1 < b.rid1;
}

bool Assembly::isReadGrinch(int32 thereadid, int32 incontigid, vector<int32> & readstocheck, vector<int32> & contigidofendreads)
{
  CEBUG("Checking if " << thereadid << " (" << AS_readpool.getRead(thereadid).getName() << ") in " << incontigid << " is grinch.\n");
  vector<uint32> possiblecontigmate(AS_readpool.size(),0);
  vector<int32>::const_iterator J=readstocheck.begin();
  newedges_t tmp;
  for(;J!=readstocheck.end(); J++){
    CEBUG("Read to check: " << *J << "   ( " << AS_readpool.getRead(*J).getName() << " )\n");
    //overlap_edges_t::iterator Mrun=AS_edges_forward.lower_bound(*J);
    tmp.rid1=*J;
    vector<newedges_t>::iterator Mrun=lower_bound(AS_confirmed_edges.begin(),
						  AS_confirmed_edges.begin(),
						  tmp,
						  Assembly__compareNewEdges_t_);
    for(;Mrun != AS_confirmed_edges.end() && Mrun->rid1 == *J; Mrun++) {
      if(contigidofendreads[Mrun->linked_with] != incontigid
	 && contigidofendreads[Mrun->linked_with] >= 0) {
	possiblecontigmate[contigidofendreads[Mrun->linked_with]]++;
	CEBUG("  Hit with " << Mrun->linked_with << "  ( " << AS_readpool.getRead(Mrun->linked_with).getName() << " ) in contig " << contigidofendreads[Mrun->linked_with] << endl);
      }
    }
  }
  
  for(uint32 i=0; i<possiblecontigmate.size(); i++){
    if(possiblecontigmate[i]>1 && static_cast<int32>(i) != incontigid) {
      CEBUG("Possible contig mate " << i << endl);
      return true;
    }
  }

  return false;
}


//#undef CEBUG





/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::findPossibleOverlaps(int32 version, const string prefix, const string postfix, const string tmpfname)
{
  FUNCSTART("void Assembly::findPossibleOverlaps(int32 version, const string prefix, const string postfix, const string tmpfname)");

  //directory_parameters const & dir_params= AS_miraparams->getDirectoryParams();
  assembly_parameters const & as_fixparams= AS_miraparams[0].getAssemblyParams();
  skim_parameters const & skim_params= AS_miraparams[0].getSkimParams();

  if(as_fixparams.as_dateoutput) dateStamp(cout);
  cout << '\n';

//  if(AS_needsskimfornastyrepeats && skim_params.sk_masknastyrepeats){
//    AS_needsskimfornastyrepeats=false;
//    markNastyReapeatsWithSkim(version, prefix, postfix, tmpfname);
//
//    // test strategy: reads with MRMr must get all overlaps
//    // BaCh 14.03.2009: BAD STRATEGY 
//    //  for high coverage (100x +), too many reads may get that set,
//    //  leading to long run times and high memory usage
//    //for(uint32 i=0; i<AS_readpool.size(); i++) {
//    //  if(AS_readpool[i].hasTag(Read::REA_tagMRMr)) AS_needalloverlaps[i]=1;
//    //}
//  }

  cout << "\nSearching for possible overlaps";


#if TRACKMEMUSAGE 
    cout << "\ndmi fpo 00\n";
    dumpMemInfo();
#endif

  // save memory for this step, those structures will have to be recomputed anyway
  //nukeSTLContainer(AS_adsfacts);
  //nukeSTLContainer(AS_confirmed_edges);
  AS_adsfacts.clear();
  AS_confirmed_edges.clear();

  nukeSTLContainer(AS_readhitmiss);
  nukeSTLContainer(AS_readhmcovered);
  nukeSTLContainer(AS_count_rhm);

#if TRACKMEMUSAGE 
    cout << "\ndmi fpo 05\n";
    dumpMemInfo();
#endif

  {
    vector<uint32> overlapcounter(AS_readpool.size(),0);

    //vector<uint32> rawhashhitcounter;
    //rawhashhitcounter.resize(AS_readpool.size(),0);

    bool onlyagainstrails=false;
    

    // very first call will be with version=0 ... pre_assembly
    //  don't set to true there as this analysis is then needed
    //  for multicopy analysis
    if(as_fixparams.as_loadbackbone
       && version >= as_fixparams.as_startbackboneusage_inpass
       && ! as_fixparams.as_backbone_alsobuildnewcontigs){
      onlyagainstrails=true;
    }

    if(onlyagainstrails) cout << " (only against backbone, the progress bar will be skewed)";
    cout << ":\n";

    //string rawhitsfilename;
    string megahubtmpfname;
    {
      if(tmpfname.size()){
	AS_posfmatch_filename=buildFileName(version, 
					    prefix,
					    postfix, 
					    tmpfname+"f", 
					    ".bin");
	AS_poscmatch_filename=buildFileName(version, 
					    prefix,
					    postfix,
					    tmpfname+"c",
					    ".bin");
	// the following two only to also automatically remove
	//  older versions of the reduced skim files
	buildFileName(version,
		      prefix,
		      postfix, 
		      tmpfname+"f", 
		      ".bin.reduced");
	buildFileName(version, 
		      prefix,
		      postfix,
		      tmpfname+"c",
		      ".bin.reduced");
	//rawhitsfilename=buildFileName(version, 
	//			      prefix,
	//			      postfix,
	//			      tmpfname+"_rawhashhits",
	//			      ".lst");
	megahubtmpfname=buildFileName(version, 
				     prefix,
				     postfix,
				     tmpfname+"_megahubs",
				     ".lst");
      }else{
	AS_posfmatch_filename=buildFileName(version, 
					    prefix,
					    postfix, 
					    as_fixparams.as_tmpf_posmatch+"f",
					    ".bin");
	AS_poscmatch_filename=buildFileName(version,
					    prefix,
					    postfix, 
					    as_fixparams.as_tmpf_posmatch+"c",
					    ".bin");
	// the following two only to also automatically remove
	//  older versions of the reduced skim files
	buildFileName(version, 
		      prefix,
		      postfix, 
		      as_fixparams.as_tmpf_posmatch+"f",
		      ".bin.reduced");
	buildFileName(version,
		      prefix,
		      postfix, 
		      as_fixparams.as_tmpf_posmatch+"c",
		      ".bin.reduced");
	//rawhitsfilename=buildFileName(version,
	//			      prefix,
	//			      postfix, 
	//			      AS_miraparams[0].getAssemblyParams().as_tmpf_posmatch+"_rawhashhits",
	//			      ".lst");
	megahubtmpfname=buildFileName(version,
				     prefix,
				     postfix, 
				     as_fixparams.as_tmpf_posmatch+"_megahubs",
				     ".lst");
      }
      AS_posfmatch_full_filename=AS_posfmatch_filename;
      AS_poscmatch_full_filename=AS_poscmatch_filename;
      
      //cout << "Only against rails? " << onlyagainstrails << endl;

      if(!onlyagainstrails) {
	if(skim_params.sk_basesperhash <=12
	   || (skim_params.sk_basesperhash <=14
	       && skim_params.sk_hashsavestepping <3)){
	  cout << "\n\nWARNING!!!!!!\nYou are not performing a 'mapping only' assembly and the parameters"
	    "\n -SK:bph=" << static_cast<uint16>(skim_params.sk_basesperhash) << " and -SK:hss="
	       << static_cast<uint16>(skim_params.sk_hashsavestepping)
	       << "\nare quite low. If SKIM takes ages, stop this assembly and restart while"
	    "\nincreasing these parameters.\n\n";
	}
      }

      vector<int32> overlaplenrequired;
      vector<int32> prrequired;
      for(uint32 i=0;i<Read::getNumSequencingTypes(); i++){
	overlaplenrequired.push_back(AS_miraparams[i].getAlignParams().al_min_overlap);
	prrequired.push_back(AS_miraparams[i].getSkimParams().sk_percentrequired);
      }

      vector<int32> chuntleftcut;
      vector<int32> chuntrightcut;

      // trigger chimera search in pre-assembly skim;
      if(!AS_doneskimchimera &&
	 (as_fixparams.as_clip_skimchimeradetection || as_fixparams.as_clip_skimjunkdetection)){
	chuntleftcut.resize(1);
	chuntrightcut.resize(1);
	AS_doneskimchimera=true;
      }

      AS_chimeracutflag.clear();

      Skim s2;
      s2.setExtendedLog(AS_miraparams[0].getSpecialParams().mi_extended_log);
      uint32 nummegahubs=s2.skimGo(AS_readpool,
				   AS_posfmatch_filename,
				   AS_poscmatch_filename,
				   megahubtmpfname,
				   AS_permanent_overlap_bans,
				   overlapcounter,
				   AS_writtenskimhitsperid,
				   chuntleftcut,
				   chuntrightcut,
				   AS_overlapcritlevell,
				   AS_overlapcritlevelr,
				   skim_params.sk_numthreads,
				   skim_params.sk_maxhashesinmem,
				   onlyagainstrails,
				   skim_params.sk_alsoskimrevcomp,
				   static_cast<uint8>(skim_params.sk_basesperhash),
				   static_cast<uint8>(skim_params.sk_hashsavestepping),
				   prrequired,
				   overlaplenrequired,
				   skim_params.sk_maxhitsperread
	);

      cout << "Total megahubs: " << nummegahubs << endl;

#if TRACKMEMUSAGE 
    cout << "\ndmi fpo 10\n";
    dumpMemInfo();
#endif

      if(nummegahubs){
	cout << "\n\nMIRA has detected megahubs in your data."
	  "This may not be a problem, but most probably is, especially for eukaryotes.\n\n";
	if(100.0/AS_num_reads_valid*nummegahubs > skim_params.sk_maxmegahubratio){
	  cout << "\n\nYou have more than " << skim_params.sk_maxmegahubratio << "% of your reads found to be megahubs."
	    "\n\nYou should sheck the following:\n\n"
	    "\t1) for Sanger sequences: are all the sequencing vectors masked / clipped?\n"
	    "\t2) for 454 sequences: are all the adaptors masked / clipped?\n\n";
	  if(skim_params.sk_masknastyrepeats){
	    cout << "You will find in the info directory a file called\n"
	      "    '*_info_readrepeats.lst',\n"
	      "consult the MIRA manual on how to extract repeat information from there.\n\n";
	  }else{
	    cout << "To learn more on the types of repeats you have and how MIRA\n"
	      " can help you to find them, please consult the manual on the\n"
	      " usage of -SK:mnr and the tmp files they create.\n";
	  }
	  cout << "*ONLY* when you are sure that no (or only a very negligible number) of sequencing"
	    "\nvector / adaptor sequence is remaining, try this:\n\n"
	    "\t3) for organisms with complex repeats (eukaryots & some bacteria):\n";
	  if(!skim_params.sk_masknastyrepeats) cout << "\t\t- use -SK:mnr=yes\n";
	  cout << "\t\t- reduce the -SK:nrr parameter (divide by 2)\n"
	    "4) for EST projects, -SK:nrr will not really work, use -SK:nrr (start at\n"
	    "   10 and increase in steps of of 10)\n"
	    "\n"
	    "*ONLY* if the above fails, try increasing the -SK:mmhr parameter\n"
	    "Note that the number of present megahubs will increase computation time in\n"
	    "an exponential way, so be careful when changing -SK:mmhr.\n";
	}

	if(100.0/AS_readpool.size()*nummegahubs >= skim_params.sk_maxmegahubratio){
	  cout << "\n\nYou have " << 100.0/AS_readpool.size()*nummegahubs
	       << "% of your reads as megahubs.\n"
	       << "You have set a maximum allowed ratio of: " << skim_params.sk_maxmegahubratio
	       << "\n\nEnding the assembly because the maximum ratio has been reached/surpassed.\n";
	  exit(10);
	}
      }

      if(chuntleftcut.size()){
	string cltmpfname;
	cltmpfname=buildFileName(0,"","", 
				as_fixparams.as_tmpf_clippings,
				".txt");
	string logprefix="skim detect: ";
	AS_chimeracutflag.resize(1);
	cutBackPossibleChimeras(cltmpfname, logprefix, chuntleftcut,chuntrightcut,AS_chimeracutflag);
	performSnapshot(0);
      }

    }

    // in mapping assemblies, correct the matches not being 100%
    if(AS_miraparams[0].getSkimParams().sk_swcheckonbackbones
       && as_fixparams.as_loadbackbone){
      recalcNonPerfectSkimMappingsBySW(version);
    }

    // initialiase well connected with overlap criterion levels
    // log overlap criterion levels if wanted

    AS_wellconnected.clear();
    AS_wellconnected.resize(AS_readpool.size(),false);

    {
      ofstream fout;

      if(AS_logflag_oclevel){
	string filename=buildFileName(version, "", "",
				      "elog.oclevel",
				      ".lst");
	fout.open(filename.c_str(), ios::out);
      }

      for(uint32 i=0; i<AS_readpool.size();++i){
	if(AS_logflag_oclevel){
	  fout << AS_readpool[i].getName() 
	       << '\t' << static_cast<uint16>(AS_overlapcritlevell[i])
	       << '\t' << static_cast<uint16>(AS_overlapcritlevelr[i])
	       << '\n';
	}
	if(AS_overlapcritlevell[i] == 0 && AS_overlapcritlevelr[i] == 0 ){
	  AS_wellconnected[i]=true;
	}
      }
    }


    //// log the raw hash hits
    //{
    //  ofstream ofs;
    //  ofs.open(rawhitsfilename.c_str(), ios::out| ios::trunc);
    //  for(size_t rhhci=0; rhhci < rawhashhitcounter.size(); rhhci++){
    //	ofs << rhhci << '\t' << rawhashhitcounter[rhhci] << '\n';
    //  }
    //  ofs.close();
    //}

    // Do this only once
    // TODO: check if ok to do more, i.e. if skim can be adapted to 
    //       still count banned overlaps.
    if(AS_multicopies.size()==0){
      string filename;
      if(tmpfname.size()){
	filename=buildFileName(version, prefix, postfix, tmpfname+"_multicopystat", ".txt");
      }else{
	filename=buildFileName(version, prefix, postfix, 
			       AS_miraparams[0].getAssemblyParams().as_tmpf_posmatch+"_multicopystat",
			       ".txt");
      }
      ofstream fout;
      fout.open(filename.c_str(), ios::out);
      fout.close();
      //flagMulticopyReads(overlapcounter, filename);
    }
    
    //if(AS_miraparams[0].getAssemblyParams().as_dateoutput) dateStamp(cout);
    //cout << '\n';
    //exit(0);
  }


// TODO: adapt to data in files instead of the old posXmatch multimaps
#if 0
  {
    vector<int32> cluster;
    cluster.resize(AS_readpool.size(),-1);
    uint32 clustercount=0;

    possible_overlaps_t::const_iterator I=AS_posfmatch.begin();
    while(I!=AS_posfmatch.end()){
      int32 cnum1=cluster[I->first];
      int32 cnum2=cluster[I->second.otherid];
      if(cnum1==-1 && cnum2==-1) {
	cluster[I->first]=clustercount;
	cluster[I->second.otherid]=clustercount;
	clustercount++;
      } else if(cnum1==-1) {
	cluster[I->first]=cluster[I->second.otherid];
      } else if(cnum2==-1) {
	cluster[I->second.otherid]=cluster[I->first];
      } else {
	if (cnum1!=cnum2) {
	  // uh oh ... we have to merge both these clusters
	  // simply change all cnum1 into cnum2 in cluster vector
	  for(uint32 j=0; j<AS_readpool.size(); j++) {
	    if(cluster[j]==cnum1) cluster[j]=cnum2; 
	  }
	}
      }

      I++;
    }

    I=AS_poscmatch.begin();
    while(I!=AS_poscmatch.end()){
      int32 cnum1=cluster[I->first];
      int32 cnum2=cluster[I->second.otherid];
      if(cnum1==-1 && cnum2==-1) {
	cluster[I->first]=clustercount;
	cluster[I->second.otherid]=clustercount;
	clustercount++;
      } else if(cnum1==-1) {
	cluster[I->first]=cluster[I->second.otherid];
      } else if(cnum2==-1) {
	cluster[I->second.otherid]=cluster[I->first];
      } else {
	if (cnum1!=cnum2) {
	  // uh oh ... we have to merge both these clusters
	  // simply change all cnum1 into cnum2 in cluster vector
	  for(uint32 j=0; j<AS_readpool.size(); j++) {
	    if(cluster[j]==cnum1) cluster[j]=cnum2; 
	  }
	}
      }

      I++;
    }

    string filename;
    if(tmpfname.size()){
      filename=buildFileName(version, prefix, postfix, tmpfname+"_pcluster", ".lst");
    }else{
      filename=buildFileName(version, prefix, postfix, 
			     AS_miraparams->getAssemblyParams().as_tmpf_posmatch+"_pcluster",
			     ".lst");
    }

    ofstream fout;
    fout.open(filename.c_str(), ios::out);
    uint32 outputcount=0;
    for(uint32 i=0; i<clustercount; i++) {
      bool found=false;
      for(uint32 j=0; j<AS_readpool.size(); j++) {
	if(cluster[j]==static_cast<int32>(i)) {
	  found=true;
	  fout << outputcount << " " << AS_readpool.getRead(j).getName() << endl;
	}
      }
      if(found) outputcount++;
    }
    for(uint32 j=0; j<AS_readpool.size(); j++) {
      if(cluster[j]==-1) {
	fout << "-1 " << AS_readpool.getRead(j).getName() << endl;
      }
    }
  }
#endif

#if TRACKMEMUSAGE 
    cout << "\ndmi fpo 20\n";
    dumpMemInfo();
#endif

  AS_steps[ASADSLISTOK]=0;

  reduceSkimHits4(version, prefix, postfix, tmpfname);

#if TRACKMEMUSAGE 
    cout << "\ndmi fpo 30\n";
    dumpMemInfo();
#endif

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::flagMulticopyReads(const vector<uint32> & overlapcounter, const string & tmpfilename)
{
  AS_multicopies.clear();
  AS_multicopies.resize(AS_readpool.size(),0);
    
  ofstream fout;
  fout.open(tmpfilename.c_str(), ios::out);

  //cout << "Searching for multicopy reads:\n";

  for(uint8 actseqtype=0; actseqtype<Read::SEQTYPE_END; actseqtype++){
    vector<uint32> sortedoverlapcounter=overlapcounter;

    // set overlapcounter of backbones and railreads and read that are
    //  not of currently analysed sequencing type to 0 so as not to
    //  count them
    uint32 numreadsinactseqtype=0;
    for(uint32 i=0; i<overlapcounter.size(); i++){
      if(AS_readpool[i].isRail() 
	 || AS_readpool[i].isBackbone()
	 || AS_readpool[i].getSequencingType()!=actseqtype) {
	sortedoverlapcounter[i]=0;
      }else{
	numreadsinactseqtype++;
      }
    }

    if(numreadsinactseqtype==0) continue;

    //cout << numreadsinactseqtype << " in sequencing type " << static_cast<uint16>(actseqtype) << endl;
    sort(sortedoverlapcounter.begin(), sortedoverlapcounter.end());

    // 5% quantil
    uint32 quantilnumber=5*numreadsinactseqtype/100;
    uint32 ifrom=0;
    // well, start the 5% quantil only at reads which have at least
    //  1 overlap
    while(ifrom<sortedoverlapcounter.size() 
	  && sortedoverlapcounter[ifrom]==0) ifrom++;

    ifrom+=quantilnumber;
    uint32 ito=static_cast<uint32>(sortedoverlapcounter.size())-quantilnumber;

    if(ito<=ifrom || ifrom>=sortedoverlapcounter.size()){
      ifrom=0;
      ito=static_cast<uint32>(sortedoverlapcounter.size());
    }

    uint32 nonsinglets=0;
    uint32 totaloverlaps=0;
    for(uint32 i=ifrom; i<ito; i++){
      totaloverlaps+=sortedoverlapcounter[i];
      nonsinglets++;
    }

      
    if(nonsinglets==0) nonsinglets=1;
    uint32 avgoverlaps=static_cast<uint32>(.5+static_cast<double>(totaloverlaps)/static_cast<double>(nonsinglets));
    // strictly speaking, this is not median. But close enough.
    uint32 medianoverlaps=sortedoverlapcounter[ifrom+((ito-ifrom)/2)];
    //uint32 multicopythreshold=avgoverlaps*2;
    uint32 multicopythreshold=medianoverlaps*2;
      
    fout << "Hitstatistics (" << Read::getShortNameOfSequencingType(actseqtype) << "): nonsinglets:" << nonsinglets << "\ttotaloverlaps: " << totaloverlaps << "\tavgoverlaps: " << avgoverlaps << "\tmedianoverlaps: " << medianoverlaps << endl;

    // now set multicopy flags for affected reads of this sequencing type
    for(uint32 i=0; i<overlapcounter.size(); i++){
      if(AS_readpool[i].isRail() 
	 || AS_readpool[i].isBackbone()
	 || AS_readpool[i].getSequencingType()!=actseqtype) continue; 
      if(overlapcounter[i]>multicopythreshold) {
	AS_multicopies[i]=static_cast<uint8>(actseqtype+1);
	if(overlapcounter[i]>multicopythreshold*10) {
	  AS_multicopies[i]=static_cast<uint8>(actseqtype+1+100);
	}
      }
    }
  }

  for(uint32 i=0; i<overlapcounter.size(); i++){
    if(AS_readpool[i].isRail() 
       || AS_readpool[i].isBackbone()) continue; 
    fout << i << "\t" << AS_readpool[i].getName() << "\t" << overlapcounter[i];
    if(AS_multicopies[i]>100) {
      fout << "\tst: " << static_cast<uint16>(AS_multicopies[i]-101);
      fout << "\tmulticopy / insane (forgotten clone vector?)";
    }else if(AS_multicopies[i]>0) {
      fout << "\tst: " << static_cast<uint16>(AS_multicopies[i]-1);
      fout << "\tmulticopy";
    }
    fout << '\n';
  }
  fout.close();
}






/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

const Read & Assembly::getRead(uint32 index)
{
  FUNCSTART("const Read & Assembly::getRead(uint32 index)");

  if(index>=AS_readpool.size()){
    MIRANOTIFY(Notify::INTERNAL,"index: " << index << " greater than AS_readpool.size():" << AS_readpool.size() << "  (out of bounds)");
  }

  FUNCEND();

  return AS_readpool.getRead(static_cast<int32>(index));
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void Assembly::refreshContigAndReadpoolValuesAfterLoading(ReadPool & rp, list<Contig> & contigs)
{
  rp.makeTemplateIDs();
  rp.makeStrainIDs();

  list<Contig>::iterator I=contigs.begin();
  for(; I!=contigs.end(); I++){
    I->recalcTemplateAndStrainIDsPresent();
  }
}






/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////        Obsolete         ///////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////


/*************************************************************************
 *
 * dead code, not used anymore?
 *
 *
 *************************************************************************/

//void Assembly::banReadPairGroups(const vector<int32> & g1, const vector<int32> & g2)
//{
//  FUNCSTART("void Assembly::banReadPairGroups(const vector<int32> & g1, const vector<int32> & g2)");
//
//  int32 id1, id2;
//  newedges_t tmpegde;
//  for(uint32 i1=0; i1 < g1.size(); i1++) {
//    id1=g1[i1];
//    for(uint32 i2=0; i2 < g2.size(); i2++) {
//      id2=g2[i2];
//
//      // skip it if there's alreaddy a permban on it
//      if(AS_permanent_overlap_bans.checkIfBanned(id1,id2)) continue;
//
//      if(AS_readpool.getRead(id1).getTemplatePartnerID()==id2) {
//#ifndef PUBLICQUIET
//	cout << "Do not ban template partners "  << id1 << ": " << AS_readpool.getRead(id1).getName() << "\t" << id2 << ": " << AS_readpool.getRead(id2).getName() << endl;
//#endif
//	  continue;
//      }
//
//#ifndef PUBLICQUIET
//      cout << "Banning: " << id1 << ": " << AS_readpool.getRead(id1).getName() << "\t" << id2 << ": " << AS_readpool.getRead(id2).getName() << endl;
//#endif
//
//      // put both ids in permanent overlap banlist
//      AS_permanent_overlap_bans.insertBan(id1,id2);
//
//      // now remove overlap edges between these reads
//      // first in one direction ...
//      //overlap_edges_t::iterator Irun=AS_edges_forward.lower_bound(id1);
//      tmpegde.rid1=id1;
//      vector<newedges_t>::iterator Irun=lower_bound(AS_confirmed_edges.begin(),
//						    AS_confirmed_edges.begin(),
//						    tmpegde,
//						    Assembly__compareNewEdges_t_);
//      while(Irun != AS_confirmed_edges.end()
//	    && Irun->rid1 == id1) {
//	if(Irun->linked_with==id2) {
//	  // erase() doesn't give back an iterator as advertised?
//	  AS_confirmed_edges.erase(Irun);
//	  //Irun=AS_edges_forward.lower_bound(id1);
//	  Irun=lower_bound(AS_confirmed_edges.begin(),
//			   AS_confirmed_edges.begin(),
//			   tmpegde,
//			   Assembly__compareNewEdges_t_);
//	  // Irun points to element after (or end) now, no need to increment
//	  continue;
//	}
//	Irun++;
//      }
// 
//      // .. then in the other one
//      //Irun=AS_edges_forward.lower_bound(id2);
//      tmpegde.rid1=id2;
//      Irun=lower_bound(AS_confirmed_edges.begin(),
//		       AS_confirmed_edges.begin(),
//		       tmpegde,
//		       Assembly__compareNewEdges_t_);
//      while(Irun != AS_confirmed_edges.end()
//	    && Irun->rid1 == id2) {
//	if(Irun->linked_with==id1) {
//	  AS_confirmed_edges.erase(Irun);
//	  //Irun=AS_edges_forward.lower_bound(id2);
//	  Irun=lower_bound(AS_confirmed_edges.begin(),
//			   AS_confirmed_edges.begin(),
//			   tmpegde,
//			   Assembly__compareNewEdges_t_);
//	  // Irun points to element after (or end) now, no need to increment
//	  continue;
//	}
//	Irun++;
//      }
//    }
//  }
//
//  FUNCEND();
//}



/////////////////////////////////////////////////////////////////////////
////////////////////        Dead since a while         //////////////////
/////////////////////////////////////////////////////////////////////////


/////*************************************************************************
//// *
//// *
//// *
//// *
//// *************************************************************************/
////
////void Assembly::flagMulticopyReads(const vector<uint32> & overlapcounter, const string & tmpfilename)
////{
////  AS_multicopies.clear();
////  AS_multicopies.resize(AS_readpool.size(),0);
////    
////  ofstream fout;
////  fout.open(tmpfilename.c_str(), ios::out);
////
////
////  for(uint32 actid=0; actid<AS_readpool.size(); actid++){
////    Read & r=AS_readpool.getRead(actid);
////    
////    //Read::setCoutType(Read::AS_TEXT);
////    //cout << r;
////
////    if(r.hasValidData()
////       && r.hasBaseFlags()){
////
////      int32 lpos=r.getLeftClipoff();
////      for(; lpos<static_cast<int32>(r.getLenSeq());lpos++){
////	if(r.hasTheseBaseFlagsSet(Read::BFLAGS_PROBABLE_MULTICOPY,lpos)){
////	  AS_multicopies[actid]=1;
////	  break;
////	}
////      }
////    }
////  }
////
////  for(uint32 i=0; i<overlapcounter.size(); i++){
////    if(AS_readpool[i].isRail() 
////       || AS_readpool[i].isBackbone()) continue; 
////    fout << i << "\t" << AS_readpool[i].getName() << "\t" << overlapcounter[i];
////    if(AS_multicopies[i]>100) {
////      fout << "\tst: " << static_cast<uint16>(AS_multicopies[i]-101);
////      fout << "\tmulticopy / insane (forgotten clone vector?)";
////    }else if(AS_multicopies[i]>0) {
////      fout << "\tst: " << static_cast<uint16>(AS_multicopies[i]-1);
////      fout << "\tmulticopy";
////    }
////    fout << '\n';
////  }
////  fout.close();
////}
//
//
//
/////*************************************************************************
//// *
//// *
//// *
//// *************************************************************************/
////
//////#define CEBUG(bla)   {cout << bla; cout.flush(); }
////void Assembly::transferContigReadsToReadpool(const Contig & buildcon, vector<Contig::pbdse_t> & pbdsev, int32 passnr)
////{
////  FUNCSTART("void Assembly::transferContigReadsToReadpool(const Contig & buildcon, vector<Contig::pbdse_t> & pbdsev)");
////
////  cout << "Transfering reads to readpool." << endl;
////
////  const vector<Contig::contigread_t> & cr = buildcon.getContigReads();
////  vector<Contig::contigread_t>::const_iterator crI = cr.begin();
////  
////  for(;crI!=cr.end();crI++){
////    if(crI->orpid>=0){
////      AS_readpool[crI->orpid]=crI->read;
////      //AS_readpool[crI->orpid].removeGapsFromRead();
////    }
////  }
////  
////  vector<Contig::pbdse_t>::const_reverse_iterator pRI=pbdsev.rbegin();
////  
////  for(; pRI != pbdsev.rend(); pRI++){
////    CEBUG("Apply: " << *pRI);
////    BUGIFTHROW(pRI->rid >= AS_readpool.size(), "pRI->rid (" << pRI->rid << ") >= AS_readpool.size() ?");
////    //Read::setCoutType(Read::AS_TEXT);
////    //cout << CON_reads[pRI->cri].read;
////    AS_readpool[pRI->rid].correctNStretch(pRI->rposs,
////					  pRI->rpose,
////					  pRI->changeestim);
////  }
////  
////  crI = cr.begin();
////  for(;crI!=cr.end();crI++){
////    if(crI->orpid>=0){
////      if(passnr==4 && AS_readpool[crI->orpid].isSequencingType(Read::SEQTYPE_PACBIO)){
////	AS_readpool[crI->orpid].transformGapsToNs();
////      }
////      AS_readpool[crI->orpid].removeGapsFromRead();
////    }
////  }
////
////  FUNCEND();
////}
//
//
//
/////*************************************************************************
//// *
//// *
//// *
//// *
//// *************************************************************************/
////
////void Assembly::markNastyReapeatsWithSkim(int32 version, const string prefix, const string postfix, const string tmpfname)
////{
////  assembly_parameters const & as_fixparams= AS_miraparams[0].getAssemblyParams();
////  skim_parameters const & skim_params= AS_miraparams[0].getSkimParams();
////
////  string filename;
////
////  {
////    if(tmpfname.size()){
////      filename=buildFileName(version, prefix, postfix, tmpfname+"_hist", "");
////    }else{
////      filename=buildFileName(version, prefix, postfix, 
////			     as_fixparams.as_tmpf_skimmarknastyrepeats+"_hist",
////			     ".lst");
////    }
////
////    ofstream fout;
////    fout.open(filename.c_str(), ios::out|ios::trunc);
////    Skim s2;
////    s2.maskNastyRepeats(AS_readpool, 
////			skim_params.sk_nastyrepeatsthreshold,
////			&fout);
////    fout.close();
////    if(as_fixparams.as_dateoutput) {
////      dateStamp(cout);
////      cout << '\n';
////    }
////  }
////
////  {
////    if(tmpfname.size()){
////      filename=buildFileName(version, prefix, postfix, tmpfname+"_nastyseq", "");
////    }else{
////      filename=buildFileName(version, prefix, postfix, 
////			     as_fixparams.as_tmpf_skimmarknastyrepeats+"_nastyseq",
////			     ".lst");
////    }
////
////    ofstream fout;
////    fout.open(filename.c_str(), ios::out|ios::trunc);
////    for(uint32 rpi=0; rpi<AS_readpool.size(); rpi++){
////      Read & actread= AS_readpool.getRead(rpi);
////      if(!actread.hasValidData()
////	 || !actread.isUsedInAssembly()) continue;
////      if(actread.hasTag(Read::REA_tagMRMr,-1)){
////	for(uint32 tn=0; tn<actread.getNumOfTags(); tn++){
////	  const tag_t & acttag=actread.getTag(tn);
////	  if(acttag.identifier==Read::REA_tagMRMr){
////	    fout << actread.getName() << '\t';
////	    for(uint32 readpos=acttag.from; readpos<=acttag.to; readpos++){
////	      fout << actread.getBaseInSequence(readpos);
////	    }
////	    fout << '\n';
////	  }
////	}
////      }
////    }
////  }
////      
////}
