/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

#ifndef lint
static char vcid[] = "$Id$";
#endif /* lint */


#include "mira/parameters_tokens.h"
#include <mira/parameters.H>
#include <mira/read.H>

// for chdir() in cwd of parameter parsing
#include <unistd.h>
// for strerror() in cwd of parameter parsing
#include <errno.h>


//#define CEBUG(bla)   {cout << bla; cout.flush(); }




string MIRAParameters::MP_currentparametersection;
bool   MIRAParameters::MP_errorinparams;
vector<string> MIRAParameters::MP_loadfilename;


void MIRAParameters::foolCompiler()
{
  (void) vcid;

#ifdef __sgi
  //(void) iostream_init;
#endif
}


MIRAParameters::MIRAParameters()
{
  MP_errorinparams=false;
  MP_parsedsomesettings=false;
  MP_loadfilename.clear();
  MP_loadfilename.reserve(12);

  // start step 0 is not a valid start step for miraSearchESTSNPs
  // but this at least catches errors if they occur
  mp_special_params.sp_est_startstep=0;
  mp_special_params.mi_stop_on_nfs=true;
  mp_special_params.mi_extended_log=false;
  mp_special_params.mi_as_largecontigsize=500;
  mp_special_params.mi_as_largecontigsize4stats=5000;
  mp_special_params.mi_stop_readnamelength=40;


  mp_assembly_params.as_numthreads=2;
  mp_assembly_params.as_automemmanagement=true;
  mp_assembly_params.as_amm_keeppercentfree=15;   // use all system mem minus 15%
  mp_assembly_params.as_amm_maxprocesssize=0;  // 0 = unlimited, use keep percent free

  mp_skim_params.sk_numthreads=mp_assembly_params.as_numthreads;
  mp_skim_params.sk_basesperhash=17;
  mp_skim_params.sk_hashsavestepping=4;
  mp_skim_params.sk_percentrequired=50;
  mp_skim_params.sk_maxhitsperread=2000;
  mp_skim_params.sk_maxhashesinmem=15000000;
  mp_skim_params.sk_memcaphitreduction=1024;
  mp_skim_params.sk_maxmegahubratio=0.0;  
  mp_skim_params.sk_masknastyrepeats=true;
  mp_skim_params.sk_alsoskimrevcomp=true;
  mp_skim_params.sk_swcheckonbackbones=true;

  mp_skim_params.sk_freqest_minnormal=0.4;
  mp_skim_params.sk_freqest_maxnormal=1.6;
  mp_skim_params.sk_freqest_repeat=2.1;
  mp_skim_params.sk_freqest_heavyrepeat=8;
  mp_skim_params.sk_freqest_crazyrepeat=20;
  mp_skim_params.sk_nastyrepeatratio=100;
  mp_skim_params.sk_repeatlevel_in_infofile=6;

  mp_pathfinder_params.paf_use_genomic_algorithms=true;
  mp_pathfinder_params.paf_use_emergency_blacklist=true;
  mp_pathfinder_params.paf_use_emergency_search_stop=true;
  mp_pathfinder_params.paf_use_max_contig_buildtime=false;
  mp_pathfinder_params.paf_ess_depth=500;
  mp_pathfinder_params.paf_max_contig_buildtime=10000;
  mp_pathfinder_params.paf_max_startcache_filltime=6;
  mp_pathfinder_params.paf_use_quick_rule=true;
  mp_pathfinder_params.paf_quickrule_minlen1=200;
  mp_pathfinder_params.paf_quickrule_minsim1=90;
  mp_pathfinder_params.paf_quickrule_minlen2=100;
  mp_pathfinder_params.paf_quickrule_minsim2=95;
  mp_pathfinder_params.paf_bbquickoverlap_minlen=150;

  mp_pathfinder_params.paf_skipwholecontigscan=5;

  setPathfinderMaxContigTime(mp_pathfinder_params.paf_max_contig_buildtime);
  setPathfinderNextReadMaxTimeToESS(1);

  mp_edit_params.ed_automatic_contic_editing=true;
  mp_edit_params.ed_strict_editing_mode=false;
  mp_edit_params.ed_confirmation_threshold=50;

  mp_assembly_params.as_dateoutput=true;

  mp_assembly_params.as_minimum_readlength=80;
  mp_assembly_params.as_minimum_readspercontig=1;

  mp_assembly_params.as_buntify_reads=true; 

  mp_assembly_params.as_loadbackbone=false; 
  mp_assembly_params.as_backbone_strainname="ReferenceStrain";
  mp_assembly_params.as_backbone_rail_fromstrain="";
  mp_assembly_params.as_backbone_strainname_forceforall=false;
  mp_assembly_params.as_infile_backbone_which="fasta";
  mp_assembly_params.as_backbone_raillength=0;
  mp_assembly_params.as_backbone_railoverlap=0;
  mp_assembly_params.as_backbone_outlen=30000;
  mp_assembly_params.as_backbone_basequals=30;
  mp_assembly_params.as_backbone_alsobuildnewcontigs=true;

  mp_assembly_params.as_loadstraindata=false;
  mp_assembly_params.as_assigndefaultstrain=false;
  mp_assembly_params.as_default_strainname="StrainX";

  mp_assembly_params.as_mergexmltraceinfo=false;

  mp_assembly_params.as_load_sequencedata=false;
  mp_assembly_params.as_enforce_qualsinreads=true;
  mp_assembly_params.as_wants_qualityfile=true;
  mp_assembly_params.as_infile_which="fasta";


  mp_assembly_params.as_fastq_qualoffset=0;
  mp_assembly_params.as_loadSOLEXA_solexa_scores_in_qual_files=false;


  mp_assembly_params.as_cleanup_tmpfiles=0;
  mp_assembly_params.as_spoilerdetection=true;
  mp_assembly_params.as_spdetect_lastpassonly=true;
  mp_assembly_params.as_keep_long_repeats_separated=true;
  mp_assembly_params.as_automatic_repeat_detection=true;
  mp_assembly_params.as_ard_multicopythreshold=2.0;
  mp_assembly_params.as_ard_multicopyminlen=400;
  mp_assembly_params.as_ard_multicopygrace=40;
  mp_assembly_params.as_uniform_read_distribution=true;
  mp_assembly_params.as_urd_startinpass=3;
  mp_assembly_params.as_urd_cutoffmultiplier=1.5;
  mp_assembly_params.as_numpasses=3;
  mp_assembly_params.as_maxcontigsperpass=0;
  mp_assembly_params.as_numrmbbreakloops=2;
  mp_assembly_params.as_startbackboneusage_inpass=3;
  mp_assembly_params.as_use_read_extension=true;
  mp_assembly_params.as_readextension_window_len=30;
  mp_assembly_params.as_readextension_window_maxerrors=2;
  mp_assembly_params.as_readextension_firstpassnum=0;
  mp_assembly_params.as_readextension_lastpassnum=0;
  mp_assembly_params.as_skimeachpass=true;
  mp_assembly_params.as_mark_repeats=true;
  mp_assembly_params.as_mark_repeats_onlyinresult=false;
  mp_assembly_params.as_readnaming_scheme=0;   // should be sanger scheme

  mp_assembly_params.as_basedefaultqual=10;
  
  //mp_assembly_params.as_projectname="mira";
  mp_assembly_params.as_projectname_in="mira";
  mp_assembly_params.as_projectname_out=mp_assembly_params.as_projectname_in;

  mp_assembly_params.as_put_asswithmira_tags=true;

  mp_assembly_params.as_filecheck_only=false;

  mp_assembly_params.as_keepcontigsinmem=false;
  mp_assembly_params.as_savesimplesingletsinproject=false;
  mp_assembly_params.as_savetaggedsingletsinproject=true;

  mp_assembly_params.as_output_caf=true;
  mp_assembly_params.as_output_maf=true;
  mp_assembly_params.as_output_fasta=true;
  mp_assembly_params.as_output_gap4da=true;
  mp_assembly_params.as_output_ace=true;
  mp_assembly_params.as_output_gff3=false;
  mp_assembly_params.as_output_wiggle=true;
  mp_assembly_params.as_output_html=false;
  mp_assembly_params.as_output_tcs=true;
  mp_assembly_params.as_output_txt=true;

  mp_assembly_params.as_output_tmp_html=false;
  mp_assembly_params.as_output_tmp_tcs=false;
  mp_assembly_params.as_output_tmp_txt=false;
  mp_assembly_params.as_output_tmp_ace=false;
  mp_assembly_params.as_output_tmp_fasta=false;
  mp_assembly_params.as_output_tmp_gap4da=false;
  mp_assembly_params.as_output_tmp_caf=true;
  mp_assembly_params.as_output_tmp_maf=false;

  mp_assembly_params.as_output_exttmp_html=false;
  mp_assembly_params.as_output_exttmp_ace=false;
  mp_assembly_params.as_output_exttmp_fasta=false;
  mp_assembly_params.as_output_exttmp_gap4da=false;
  mp_assembly_params.as_output_exttmp_caf=false;
  mp_assembly_params.as_output_exttmp_alsosinglets=false;

  mp_assembly_params.as_output_removerollovertmps=true;
  mp_assembly_params.as_output_removetmpdir=false;

  mp_assembly_params.as_clip_skimchimeradetection=true;
  mp_assembly_params.as_clip_skimjunkdetection=true;
  mp_assembly_params.as_clip_proposeendclips=true;
  mp_assembly_params.as_clip_pec_sxaggcxg=true;
  mp_assembly_params.as_clip_pec_basesperhash=17;

  mp_assembly_params.as_clip_badsolexaends=false;
  mp_assembly_params.as_clip_knownadaptorsright=false;

  mp_assembly_params.as_clip_lowercase=false;

  mp_assembly_params.as_clip_polyat=false;
  mp_assembly_params.as_clip_polyat_keeppolysignal=false;
  mp_assembly_params.as_clip_polyat_len=12;
  mp_assembly_params.as_clip_polyat_maxerrors=1;
  mp_assembly_params.as_clip_polyat_maxgap=9;

  mp_assembly_params.as_clip_3ppolybase=false;
  mp_assembly_params.as_clip_3ppolybase_len=12;
  mp_assembly_params.as_clip_3ppolybase_maxerrors=2;
  mp_assembly_params.as_clip_3ppolybase_maxgap=9;

  mp_assembly_params.as_clip_possible_vectors=true;
  mp_assembly_params.as_clip_quality=true;
  mp_assembly_params.as_clip_badstretchquality=true;
  mp_assembly_params.as_clip_maskedbases=true;
  mp_assembly_params.as_clip_mergessahavectorscreen=false;

  mp_assembly_params.as_clip_quality_minthreshold=0;
  mp_assembly_params.as_clip_quality_numminthreshold=0;

  mp_assembly_params.as_clip_quality_minqual=20;
  mp_assembly_params.as_clip_quality_winlen=30;
  mp_assembly_params.as_clip_badstretchquality_minqual=5;
  mp_assembly_params.as_clip_badstretchquality_winlen=20;
  mp_assembly_params.as_clip_vector_maxlenallowed=18;
  mp_assembly_params.as_clip_maskedbase_gapsize=20;
  mp_assembly_params.as_clip_maskedbase_maxfrontgap=40;
  mp_assembly_params.as_clip_maskedbase_maxendgap=60;
  mp_assembly_params.as_clip_ssahamerge_gapsize=10;
  mp_assembly_params.as_clip_ssahamerge_maxfrontgap=60;
  mp_assembly_params.as_clip_ssahamerge_maxendgap=120;
  mp_assembly_params.as_clip_ssahamerge_strictfrontclip=0;
  mp_assembly_params.as_clip_ssahamerge_strictendclip=0;
  mp_assembly_params.as_clip_ensureminimumleftclipoff=true;
  mp_assembly_params.as_clip_ensureminimumrightclipoff=false;
  mp_assembly_params.as_clip_minslrequired=25;
  mp_assembly_params.as_clip_minqlsetto=30;
  mp_assembly_params.as_clip_minsrrequired=10;
  mp_assembly_params.as_clip_minqrsetto=20;

  mp_assembly_params.as_external_qual=E_QUAL_SCF;
  mp_assembly_params.as_external_qual_override=0;
  mp_assembly_params.as_discard_read_on_eq_error=false;

// Initialise the scoring scheme for the similarity matrix
  mp_dynamic_params.dyn_score_multiplier=10;
  mp_dynamic_params.dyn_score_match=1*mp_dynamic_params.dyn_score_multiplier;
  mp_dynamic_params.dyn_score_mismatch=-1*mp_dynamic_params.dyn_score_multiplier;
  mp_dynamic_params.dyn_score_halfmatch=static_cast<int32>(.5*mp_dynamic_params.dyn_score_multiplier);
  mp_dynamic_params.dyn_score_halfmismatch=static_cast<int32>(-.5*mp_dynamic_params.dyn_score_multiplier);
  mp_dynamic_params.dyn_score_nmatch=0;
  mp_dynamic_params.dyn_score_npenaltymatch=-1;
  mp_dynamic_params.dyn_score_gap=-2*mp_dynamic_params.dyn_score_multiplier;
  mp_dynamic_params.dyn_score_oldgap=static_cast<int32>((-1.5)*mp_dynamic_params.dyn_score_multiplier);
  mp_dynamic_params.dyn_score_oldgapmatch=static_cast<int32>((-1.1)*mp_dynamic_params.dyn_score_multiplier);
  //mp_dynamic_params.dyn_score_gap=-5;
  //mp_dynamic_params.dyn_score_oldgap=-4;
  mp_dynamic_params.dyn_score_ltermgap=0;      // DO NOT TOUCH
  mp_dynamic_params.dyn_score_rtermgap=0;      // DO NOT TOUCH
  mp_dynamic_params.dyn_score_termgap=min(mp_dynamic_params.dyn_score_ltermgap,
					  mp_dynamic_params.dyn_score_rtermgap);       // 0  DO NOT CHANGE THIS ONE

  mp_align_params.al_max_cutoff=1;
  mp_align_params.al_min_score=15*mp_dynamic_params.dyn_score_multiplier;
  mp_align_params.al_min_relscore=65;
  mp_align_params.al_min_overlap=17;
  mp_align_params.al_kmin=25;
  mp_align_params.al_kmax=100;
  mp_align_params.al_kpercent=15;
  mp_align_params.al_solexahack_maxerrors=0;
  mp_align_params.ads_enforce_clean_ends=false;
  mp_align_params.ads_clean_end_distance=0;

  mp_align_params.ads_extra_gap_penalty=false;
  setAlignGapPenaltyLevel(0);
  mp_align_params.ads_max_gppercent=100;
  mp_align_params.ads_enforce_clean_ends=false;
  mp_align_params.ads_clean_end_distance=0;
  //mp_align_params.ads_extra_mismatch_penalty=true;
  //mp_align_params.ads_emp_windowlen=30;
  //mp_align_params.ads_emp_maxmismatches=15;

  mp_contig_params.con_reject_on_drop_in_relscore=15;
  mp_contig_params.con_danger_analyse_mode=ANALYSE_SIGNAL;
  mp_contig_params.con_danger_max_error_rate=1;         // 1% error rate
  mp_contig_params.con_insertsize_minimum=-1;
  mp_contig_params.con_insertsize_maximum=-1;
  mp_contig_params.con_template_build_direction=-1;

  mp_contig_params.con_assume_snp_insteadof_rmb=false;
  mp_contig_params.con_disregard_spurious_rmb_mismatches=true;
  mp_contig_params.con_also_mark_gap_bases=true;
  mp_contig_params.con_also_mark_gap_bases_needbothstrands=true;
  mp_contig_params.con_also_mark_gap_bases_evenmc=true;
  mp_contig_params.con_shorttagcomments=true;
  mp_contig_params.con_minrmbneighbourqual=20;
  mp_contig_params.con_mingroupqualforrmbtagging=30;
  //mp_contig_params.con_min_groupqual_srmbwrmb_change=45;
  //mp_contig_params.con_rmb_numzone_trigger=1;
  mp_contig_params.con_endreadmarkexclusionarea=25;
  mp_contig_params.con_emea_setzero_on_clipping_pec=true;
  mp_contig_params.con_minreadspergroup=2;
  mp_contig_params.con_gap_override_ratio=66;

  mp_contig_params.con_force_nonIUPACconsensus_perseqtype=false;
  mp_contig_params.con_force_nonIUPACconsensus_amongseqtypes=false;
  mp_contig_params.con_mergeshortreads=false;
  mp_contig_params.con_msr_keependsunmapped=-1;

  mp_contig_params.con_output_text_cpl=60;
  mp_contig_params.con_output_html_cpl=60;
  mp_contig_params.con_output_text_gapfill=' ';
  mp_contig_params.con_output_html_gapfill=' ';

  mp_directory_params.dir_tmp_redirectedto="";
  mp_directory_params.dir_scf="";
  mp_directory_params.dir_exp="";
  mp_directory_params.dir_fasta="";

  setAssemblyUseTemplateInformation(true);
}


MIRAParameters::~MIRAParameters()
{
}



void MIRAParameters::setupStdMIRAParameters(vector<MIRAParameters> & Pv, bool verbose){
  Pv.clear();
  Pv.resize(Read::SEQTYPE_END);

  generateProjectNames(Pv);

  const char modestring[] = 
    "\n--job=denovo,genome,accurate,sanger,454,iontor,pacbio,solexa,solid -SB:lb=no"
    ;

  // "\n\t-OUT:sssip=no:stsip=no:org=no"
  
  try {
    parseQuickmodeNoTechSettingsChange(modestring, 
				       "Default settings",
				       Pv, verbose, NULL);
  }
  catch (Notify n) {
    cout << "\nError while parsing MIRA-internal standard parameters!"
      "\nThere is nothing you can do about it, this is a blunder by the author."
      "\nPlease immediately file a bug report.\n";

    n.setGravity(Notify::INTERNAL);
    n.handleError(n.tif);
  }

  for(uint32 i=0; i<Read::SEQTYPE_END; i++){
    Pv[i].MP_parsedsomesettings=false;
    Pv[i].mp_assembly_params.as_load_sequencedata=false;
  }

}


// adapts a few values automatically after parsing
void MIRAParameters::postParsingChanges(vector<MIRAParameters> & Pv)
{
  FUNCSTART("void MIRAParameters::postParsingChanges(vector<MIRAParameters> & Pv)");
  BUGIFTHROW(Pv.empty(), "Empty MIRAParameters vector???");

  if(Pv[0].mp_assembly_params.as_clip_proposeendclips
     && Pv[0].mp_contig_params.con_emea_setzero_on_clipping_pec){
    cout << "-CL:pec and -CO:emeas1clpec are set, setting -CO:emea values to 1.\n";
    for(uint32 i=0; i<Read::SEQTYPE_END; i++){
      Pv[i].mp_contig_params.con_endreadmarkexclusionarea=1;
    }
  }

  correctTmpDirectory(Pv);

  FUNCEND();
}

// the "intelligent" version of dumpAllParams()
// looks which data are to be loaded or where settings have been parsed for
//  and sets the vector with the needed data accordingly
void MIRAParameters::dumpAllParams(vector<MIRAParameters> & Pv, ostream & ostr)
{
  FUNCSTART("void MIRAParameters::dumpAllParams(vector<MIRAParameters> & Pv, ostream & ostr)");
  BUGIFTHROW(Pv.empty(), "Empty MIRAParameters vector???");

  vector<int> indexesInPv;
  indexesInPv.push_back(0);

  for(uint32 i=1; i<Read::SEQTYPE_END; i++){
    if(Pv[i].mp_assembly_params.as_load_sequencedata
       || Pv[i].MP_parsedsomesettings){
      indexesInPv.push_back(i);
    }
  }

  dumpAllParams(Pv, indexesInPv, ostr);

  FUNCEND();
}

void MIRAParameters::dumpAllParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  FUNCSTART("void MIRAParameters::dumpAllParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)");
  
  BUGIFTHROW(indexesInPv.empty(), "Trying to dump nothing?\n");
  
  cout << "------------------------------------------------------------------------------\nParameter settings seen for:\n";

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

  for(uint32 i=0; i<indexesInPv.size(); i++){
    if(i>0) cout << ", ";
    switch(indexesInPv[i]){
    case Read::SEQTYPE_SANGER:{
      cout << "Sanger data (also common parameters)";
      break;
    }
    case Read::SEQTYPE_454GS20:{
      cout << "454 data";
      break;
    }
    case Read::SEQTYPE_IONTORRENT:{
      cout << "IonTorrent data";
      break;
    }
    case Read::SEQTYPE_PACBIO:{
      cout << "PacBio data";
      break;
    }
    case Read::SEQTYPE_SOLEXA:{
      cout << "Solexa data";
      break;
    }
    case Read::SEQTYPE_ABISOLID:{
      cout << "SOLiD data";
      break;
    }
    default: {
      // intentionally do nothing
    }
    }
  }

  ostr << "\n\nUsed parameter settings:\n";
  dumpGeneralParams(Pv, indexesInPv, ostr);
  dumpLoadParams(Pv, indexesInPv, ostr);
  dumpAssemblyParams(Pv, indexesInPv, ostr);
  dumpStrainBackboneParams(Pv, indexesInPv, ostr);
  dumpDataProcessingParams(Pv, indexesInPv, ostr);
  dumpClippingParams(Pv, indexesInPv, ostr);
  dumpSkimParams(Pv, indexesInPv, ostr);
  dumpPathfinderParams(Pv, indexesInPv, ostr);
  dumpAlignParams(Pv, indexesInPv, ostr);
  dumpContigParams(Pv, indexesInPv, ostr);
  dumpEditParams(Pv, indexesInPv, ostr);
  dumpMiscParams(Pv, indexesInPv, ostr);
  dumpDirectoryParams(Pv, indexesInPv, ostr);
  dumpFileInParams(Pv, indexesInPv, ostr);
  dumpFileOutParams(Pv, indexesInPv, ostr);
  dumpFileTempParams(Pv, indexesInPv, ostr);
  dumpOutputCustomisationParams(Pv, indexesInPv, ostr);
  dumpFileDirectoryOutNamesParams(Pv, indexesInPv, ostr);

  ostr << "------------------------------------------------------------------------------\n";

  FUNCEND();
}


#define OUTSTRING(o,xx) {if(xx.size()!=0){o<<(xx)<<"\n";}else{o<<"(none)\n";}}

// static func
void MIRAParameters::dumpFileDirectoryOutNamesParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  ostr << "\n    File / directory output names:\n";
  ostr << "\tCAF             : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_CAF+".caf"));
  ostr << "\tMAF             : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_MAF+".maf"));
  ostr << "\tFASTA           : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_FASTA+".fasta"));
  ostr << "\tFASTA quality   : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_FASTA+".fasta.qual"));
  ostr << "\tFASTA (padded)  : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_FASTAPADDED+".fasta"));
  ostr << "\tFASTA qual.(pad): ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_FASTAPADDED+".fasta.qual"));
  ostr << "\tGAP4 (directory): ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outdir_GAP4DA+".gap4da"));
  ostr << "\tACE             : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_ACE+".ace"));
  ostr << "\tHTML            : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_HTML+".html"));
  ostr << "\tSimple text     : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_TXT+".txt"));
  ostr << "\tTCS overview    : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_TCS+".tcs"));
  ostr << "\tWiggle          : ";
  OUTSTRING(ostr, (Pv[0].mp_assembly_params.as_outfile_WIGGLE+".wig"));
}


// static func
void MIRAParameters::dumpOutputCustomisationParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=46;

  vector<int> singlePvIndex;

  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }


  ostr << "\n    Alignment output customisation:\n";

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_contig_params.con_output_text_cpl,
		  "\t", "TEXT characters per line (tcpl)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_contig_params.con_output_html_cpl,
		  "\t", "HTML characters per line (hcpl)",
		  fieldlength);

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_contig_params.con_output_text_gapfill,
		  "\t", "TEXT end gap fill character (tegfc)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_contig_params.con_output_html_gapfill,
		  "\t", "HTML end gap fill character (hegfc)",
		  fieldlength);

}


// static func
void MIRAParameters::dumpFileTempParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=46;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n    Temporary result files:\n";

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tmp_caf,
		      "\t", "Saved as CAF                       (otc)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tmp_maf,
		      "\t", "Saved as MAF                       (otm)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tmp_fasta,
		      "\t", "Saved as FASTA                     (otf)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tmp_gap4da,
		      "\t", "Saved as GAP4 (directed assembly)  (otg)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tmp_ace,
		      "\t", "Saved as phrap ACE                 (ota)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tmp_html,
		      "\t", "Saved as HTML                      (oth)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tmp_tcs,
		      "\t", "Saved as Transposed Contig Summary (ots)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tmp_txt,
		      "\t", "Saved as simple text format        (ott)",
		      fieldlength);

  ostr << "\n    Extended temporary result files:\n";

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_exttmp_caf,
		      "\t", "Saved as CAF                      (oetc)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_exttmp_fasta,
		      "\t", "Saved as FASTA                    (oetf)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_exttmp_gap4da,
		      "\t", "Saved as GAP4 (directed assembly) (oetg)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_exttmp_ace,
		      "\t", "Saved as phrap ACE                (oeta)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_exttmp_html,
		      "\t", "Saved as HTML                     (oeth)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_exttmp_alsosinglets,
		      "\t", "Save also singlets               (oetas)",
		      fieldlength);
}


// static func
void MIRAParameters::dumpFileOutParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=46;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Output files (-OUTPUT/-OUT):\n";

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_savesimplesingletsinproject,
		      "\t", "Save simple singlets in project (sssip)",
		      fieldlength);
  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_savetaggedsingletsinproject,
		      "\t", "Save tagged singlets in project (stsip)",
		      fieldlength);

  ostr << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_removerollovertmps,
		      "\t", "Remove rollover tmps (rrot)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_removetmpdir,
		      "\t", "Remove tmp directory (rtd)",
		      fieldlength);


  ostr << "\n    Result files:\n";

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_caf,
		      "\t", "Saved as CAF                       (orc)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_maf,
		      "\t", "Saved as MAF                       (orm)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_fasta,
		      "\t", "Saved as FASTA                     (orf)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_gap4da,
		      "\t", "Saved as GAP4 (directed assembly)  (org)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_ace,
		      "\t", "Saved as phrap ACE                 (ora)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_gff3,
		      "\t", "Saved as GFF3                     (org3)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_html,
		      "\t", "Saved as HTML                      (orh)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_tcs,
		      "\t", "Saved as Transposed Contig Summary (ors)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_txt,
		      "\t", "Saved as simple text format        (ort)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_output_wiggle,
		      "\t", "Saved as wiggle                    (orw)",
		      fieldlength);
}


// static func
void MIRAParameters::dumpFileInParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=46;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  File names (-FN):\n";

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_infile_FASTA,
		  "\t", "When loading sequences from FASTA",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_infile_FASTAQUAL,
		  "\t", "When loading qualities from FASTA quality",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_infile_FASTQ,
		  "\t", "When loading sequences from FASTQ",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_CAF,
		  "\t", "When loading project from CAF",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_MAF,
		  "\t", "When loading project from MAF (disabled)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_FOFNEXP,
		  "\t", "When loading EXP fofn",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_PHD,
		  "\t", "When loading project from PHD",
		  fieldlength);

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_straindata,
		  "\t", "When loading strain data",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_infile_xmltraceinfo,
		  "\t", "When loading XML trace info files",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_ssahavectorscreen,
		  "\t", "When loading SSAHA2 vector screen results",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_smaltvectorscreen,
		  "\t", "When loading SMALT vector screen results",
		  fieldlength);

  ostr << '\n';

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_backbone_MAF,
		  "\t", "When loading backbone from MAF",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_backbone_CAF,
		  "\t", "When loading backbone from CAF",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_backbone_GBF,
		  "\t", "When loading backbone from GenBank",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_backbone_GFF3,
		  "\t", "When loading backbone from GFF3",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_backbone_FASTA,
		  "\t", "When loading backbone from FASTA",
		  fieldlength);

  ostr << '\n';

}


// static func
void MIRAParameters::dumpDirectoryParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=35;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Directories (-DI):\n";

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_cwd,
		  "\t", "Working directory",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_exp,
		  "\t", "When loading EXP files",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_scf,
		  "\t", "When loading SCF files",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_top,
		  "\t", "Top directory for writing files",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_results,
		  "\t", "For writing result files",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_info,
		  "\t", "For writing result info files",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_tmp,
		  "\t", "For writing tmp files",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_tmp_redirectedto,
		  "\t", "Tmp redirected to (trt)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_directory_params.dir_checkpoint,
		  "\t", "For writing checkpoint files",
		  fieldlength);

  // TODO: hide?
  //ostr << "\tFor writing gap4 DA res.: " << mp_assembly_params.as_outdir_GAP4DA << endl;

}


// static func
void MIRAParameters::dumpEditParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Edit options (-ED):\n";

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_edit_params.ed_automatic_contic_editing,
		      "\t", "Automatic contig editing (ace)",
		      fieldlength);

  ostr << "     Sanger only:\n";

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_edit_params.ed_strict_editing_mode,
		      "\t", "Strict editing mode (sem)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_edit_params.ed_confirmation_threshold,
		  "\t", "Confirmation threshold in percent (ct)",
		  fieldlength);
}


// static func
void MIRAParameters::dumpContigParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=58;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Contig parameters (-CO):\n";

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_contig_params.con_nameprefix,
		  "\t", "Name prefix (np)",
		  fieldlength);

// TODO: deprecate?
//  ostr << "\tError analysis (an)                                      : ";
//  switch(mp_contig_params.con_danger_analyse_mode){
//  case ANALYSE_SIGNAL:{
//    ostr << "SCF signal (signal)\n";
//    break;
//  }
//  case ANALYSE_TEXT:{
//    ostr << "text only (text)\n";
//    break;
//  }
//  case ANALYSE_NONE:{
//    ostr << "none (none)\n";
//    break;
//  }
//  default:{
//    ostr << "Unknown??? (please contact the authors)\n";
//  }
//  }


  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_contig_params.con_reject_on_drop_in_relscore,
		  "\t", "Reject on drop in relative alignment score in % (rodirs)",
		  fieldlength);

// TODO: deprecate?
//  ostr << "\tMax. error rate in dangerous zones in % (dmer)           : " << mp_contig_params.con_danger_max_error_rate << "\n";


  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_mark_repeats,
		      "\t", "Mark repeats (mr)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_mark_repeats_onlyinresult,
		      "\t    ", "Only in result (mroir)",
		      fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_contig_params.con_assume_snp_insteadof_rmb,
		      "\t    ", "Assume SNP instead of repeats (asir)",
		      fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_contig_params.con_minreadspergroup,
		  "\t    ", "Minimum reads per group needed for tagging (mrpg)",
		  fieldlength-4);
  multiParamPrintNumericChar(
    Pv, indexesInPv, ostr, 
    Pv[0].mp_contig_params.con_minrmbneighbourqual,
    "\t    ", "Minimum neighbour quality needed for tagging (mnq)",
    fieldlength-4);
  multiParamPrintNumericChar(
    Pv, indexesInPv, ostr, 
    Pv[0].mp_contig_params.con_mingroupqualforrmbtagging,
    "\t    ", "Minimum Group Quality needed for RMB Tagging (mgqrt)",
    fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_contig_params.con_endreadmarkexclusionarea,
		  "\t    ", "End-read Marking Exclusion Area in bases (emea)",
		  fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_contig_params.con_emea_setzero_on_clipping_pec,
		      "\t        ", "Set to 1 on clipping PEC (emeas1clpec)",
		      fieldlength-8);
  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_contig_params.con_also_mark_gap_bases,
		      "\t    ", "Also mark gap bases (amgb)",
		      fieldlength-4);
  multiParamPrintBool(
    Pv, indexesInPv, ostr, 
    Pv[0].mp_contig_params.con_also_mark_gap_bases_evenmc,
    "\t        ", "Also mark gap bases - even multicolumn (amgbemc)",
    fieldlength-8);
  multiParamPrintBool(
    Pv, indexesInPv, ostr, 
    Pv[0].mp_contig_params.con_also_mark_gap_bases_needbothstrands,
    "\t        ", "Also mark gap bases - need both strands (amgbnbs)",
    fieldlength-8);

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_contig_params.con_force_nonIUPACconsensus_perseqtype,
		      "\t", "Force non-IUPAC consensus per sequencing type (fnicpst)",
		      fieldlength);

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_contig_params.con_mergeshortreads,
		      "\t", "Merge short reads (msr)",
		      fieldlength);

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_contig_params.con_msr_keependsunmapped,
		  "\t    ", "Keep ends unmerged (msrkeu)",
		  fieldlength-4);

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_contig_params.con_gap_override_ratio,
		  "\t", "Gap override ratio (gor)",
		  fieldlength);

  // Not implemented yet
  //multiParamPrintBool(Pv, singlePvIndex, ostr, 
  //		      Pv[0].mp_contig_params.con_force_nonIUPACconsensus_amongseqtypes,
  //		      "\t", "Force non-IUPAC consensus among sequncing types (fnicast)",
  //		      fieldlength);
}



// static func
void MIRAParameters::dumpPathfinderParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Pathfinder options (-PF):\n";

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_pathfinder_params.paf_use_quick_rule,
		      "\t", "Use quick rule (uqr)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_pathfinder_params.paf_quickrule_minlen1,
		  "\t    ", "Quick rule min len 1 (qrml1)",
		  fieldlength-4);
  multiParamPrintNumericChar(Pv, indexesInPv, ostr, 
			     Pv[0].mp_pathfinder_params.paf_quickrule_minsim1,
			     "\t    ", "Quick rule min sim 1 (qrms1)",
			     fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_pathfinder_params.paf_quickrule_minlen2,
		  "\t    ", "Quick rule min len 2 (qrml2)",
		  fieldlength-4);
  multiParamPrintNumericChar(Pv, indexesInPv, ostr, 
			     Pv[0].mp_pathfinder_params.paf_quickrule_minsim2,
			     "\t    ", "Quick rule min sim 2 (qrms2)",
			     fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_pathfinder_params.paf_bbquickoverlap_minlen,
		  "\t", "Backbone quick overlap min len (bqoml)",
		  fieldlength);

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_pathfinder_params.paf_max_startcache_filltime,
		  "\t", "Max. start cache fill time (mscft)",
		  fieldlength);
}

// static func
void MIRAParameters::dumpSkimParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Parameters for SKIM algorithm (-SK):\n";

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_numthreads,
		  "\t", "Number of threads (not)",
		  fieldlength);
  ostr << '\n';
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_skim_params.sk_alsoskimrevcomp,
		      "\t", "Also compute reverse complements (acrc)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_basesperhash,
		  "\t", "Bases per hash (bph)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_hashsavestepping,
		  "\t", "Hash save stepping (hss)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_skim_params.sk_percentrequired,
		  "\t", "Percent required (pr)",
		  fieldlength);
  ostr << '\n';
  multiParamPrint(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_skim_params.sk_maxhitsperread,
		  "\t", "Max hits per read (mhpr)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_maxmegahubratio,
		  "\t", "Max megahub ratio (mmhr)",
		  fieldlength);

  ostr << '\n';
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_skim_params.sk_swcheckonbackbones,
		      "\t", "SW check on backbones (swcob)",
		      fieldlength);

  ostr << '\n';
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_freqest_minnormal,
		  "\t", "Freq. est. min normal (fenn)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_freqest_maxnormal,
		  "\t", "Freq. est. max normal (fexn)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_freqest_repeat,
		  "\t", "Freq. est. repeat (fer)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_freqest_heavyrepeat,
		  "\t", "Freq. est. heavy repeat (fehr)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_freqest_crazyrepeat,
		  "\t", "Freq. est. crazy (fecr)",
		  fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_skim_params.sk_masknastyrepeats,
		      "\t", "Mask nasty repeats (mnr)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_nastyrepeatratio,
		  "\t    ", "Nasty repeat ratio (nrr)",
		  fieldlength-4);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_repeatlevel_in_infofile,
		  "\t", "Repeat level in info file (rliif)",
		  fieldlength);

  ostr << '\n';
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_maxhashesinmem,
		  "\t", "Max hashes in memory (mhim)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_skim_params.sk_memcaphitreduction,
		  "\t", "MemCap: hit reduction (mchr)",
		  fieldlength);

}

// static func
void MIRAParameters::dumpClippingParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Clipping options (-CL):\n";

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_mergessahavectorscreen,
		      "\t", "Merge with SSAHA2/SMALT vector screen (msvs)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_ssahamerge_gapsize,
		  "\t    ", "Gap size (msvsgs)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_ssahamerge_maxfrontgap,
		  "\t    ", "Max front gap (msvsmfg)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_ssahamerge_maxendgap,
		  "\t    ", "Max end gap (msvsmeg)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_ssahamerge_strictfrontclip,
		  "\t    ", "Strict front clip (msvssfc)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_ssahamerge_strictendclip,
		  "\t    ", "Strict end clip (msvssec)",
		  fieldlength-4);

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_possible_vectors,
		      "\t", "Possible vector leftover clip (pvlc)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_vector_maxlenallowed,
		  "\t    ", "maximum len allowed (pvcmla)",
		  fieldlength-4);

  multiParamPrintNumericChar(Pv, indexesInPv, ostr, 
			     Pv[0].mp_assembly_params.as_clip_quality_minthreshold,
			     "\t", "Min qual. threshold for entire read (mqtfer)",
			     fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_quality_numminthreshold,
		  "\t    ", "Number of bases (mqtfernob)",
		  fieldlength-4);

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_quality,
		      "\t", "Quality clip (qc)",
		      fieldlength);
  multiParamPrintNumericChar(Pv, indexesInPv, ostr, 
			     Pv[0].mp_assembly_params.as_clip_quality_minqual,
			     "\t    ", "Minimum quality (qcmq)",
			     fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_quality_winlen,
		  "\t    ", "Window length (qcwl)",
		  fieldlength-4);


  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_badstretchquality,
		      "\t", "Bad stretch quality clip (bsqc)",
		      fieldlength);
  multiParamPrintNumericChar(Pv, indexesInPv, ostr, 
			     Pv[0].mp_assembly_params.as_clip_badstretchquality_minqual,
			     "\t    ", "Minimum quality (bsqcmq)",
			     fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_badstretchquality_winlen,
		  "\t    ", "Window length (bsqcwl)",
		  fieldlength-4);



  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_maskedbases,
		      "\t", "Masked bases clip (mbc)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_maskedbase_gapsize,
		  "\t    ", "Gap size (mbcgs)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_maskedbase_maxfrontgap,
		  "\t    ", "Max front gap (mbcmfg)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_maskedbase_maxendgap,
		  "\t    ", "Max end gap (mbcmeg)",
		  fieldlength-4);


  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_lowercase,
		      "\t", "Lower case clip (lcc)",
		      fieldlength);

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_polyat,
		      "\t", "Clip poly A/T at ends (cpat)",
		      fieldlength);
  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_polyat_keeppolysignal,
		      "\t    ", "Keep poly-a signal (cpkps)",
		      fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_polyat_len,
		  "\t    ", "Minimum signal length (cpmsl)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_polyat_maxerrors,
		  "\t    ", "Max errors allowed (cpmea)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_polyat_maxgap,
		  "\t    ", "Max gap from ends (cpmgfe)",
		  fieldlength-4);


  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_3ppolybase,
		      "\t", "Clip 3 prime polybase (c3pp)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_3ppolybase_len,
		  "\t    ", "Minimum signal length (c3ppmsl)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_3ppolybase_maxerrors,
		  "\t    ", "Max errors allowed (c3ppmea)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_3ppolybase_maxgap,
		  "\t    ", "Max gap from ends (c3ppmgfe)",
		  fieldlength-4);

  multiParamPrintBool(Pv, indexesInPv, ostr,
		      Pv[0].mp_assembly_params.as_clip_knownadaptorsright,
		      "\t", "Clip known adaptors right (ckar)",
		      fieldlength);

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_ensureminimumleftclipoff,
		      "\t", "Ensure minimum left clip (emlc)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_minslrequired,
		  "\t    ", "Minimum left clip req. (mlcr)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_minqlsetto,
		  "\t    ", "Set minimum left clip to (smlc)",
		  fieldlength-4);
  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_clip_ensureminimumrightclipoff,
		      "\t", "Ensure minimum right clip (emrc)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_clip_minsrrequired,
		  "\t    ", "Minimum right clip req. (mrcr)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr,
		  Pv[0].mp_assembly_params.as_clip_minqrsetto,
		  "\t    ", "Set minimum right clip to (smrc)",
		  fieldlength-4);

  cout << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr,
		      Pv[0].mp_assembly_params.as_clip_skimchimeradetection,
		      "\t", "Apply SKIM chimera detection clip (ascdc)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr,
		      Pv[0].mp_assembly_params.as_clip_skimjunkdetection,
		      "\t", "Apply SKIM junk detection clip (asjdc)",
		      fieldlength);
  cout << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr,
		      Pv[0].mp_assembly_params.as_clip_proposeendclips,
		      "\t", "Propose end clips (pec)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_clip_pec_basesperhash,
		  "\t    ", "Bases per hash (pecbph)",
		  fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_clip_pec_sxaggcxg,
		      "\t    ", "Handle Solexa GGCxG problem (pechsgp)",
		      fieldlength-4);

  cout << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr,
		      Pv[0].mp_assembly_params.as_clip_badsolexaends,
		      "\t", "Clip bad solexa ends (cbse)",
		      fieldlength);
}


// static func
void MIRAParameters::dumpDataProcessingParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
    if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Dataprocessing options (-DP):\n";

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_use_read_extension,
		      "\t", "Use read extensions (ure)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_readextension_window_len,
		  "\t    ", "Read extension window length (rewl)",
		  fieldlength-4);

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_readextension_window_maxerrors,
		  "\t    ", "Read extension w. maxerrors (rewme)",
		  fieldlength-4);

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_readextension_firstpassnum,
		  "\t    ", "First extension in pass (feip)",
		  fieldlength-4);

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_readextension_lastpassnum,
		  "\t    ", "Last extension in pass (leip)",
		  fieldlength-4);
}

// static func
void MIRAParameters::dumpStrainBackboneParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
    if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Strain and backbone options (-SB):\n";

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_loadstraindata,
		      "\t", "Load straindata (lsd)",
		      fieldlength);
  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_assigndefaultstrain,
		      "\t", "Assign default strain (ads)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_default_strainname,
		  "\t    ","Default strain name (dsn)",
		  fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_loadbackbone,
		      "\t", "Load backbone (lb)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_startbackboneusage_inpass,
		  "\t    ","Start backbone usage in pass (sbuip)",
		  fieldlength-4);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_infile_backbone_which,
		  "\t    ", "Backbone file type (bft)",
		  fieldlength-4);
  multiParamPrintNumericChar(Pv, singlePvIndex, ostr, 
			     Pv[0].mp_assembly_params.as_backbone_basequals,
			     "\t    ", "Backbone base quality (bbq)",
			     fieldlength-4);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_backbone_strainname,
		  "\t    ", "Backbone strain name (bsn)",
		  fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_backbone_strainname_forceforall,
		      "\t        ", "Force for all (bsnffa)",
		      fieldlength-8);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_backbone_rail_fromstrain,
		  "\t    ", "Backbone rail from strain (brfs)",
		  fieldlength-4);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_backbone_raillength,
		  "\t    ", "Backbone rail length (brl)",
		  fieldlength-4);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_backbone_railoverlap,
		  "\t    ", "Backbone rail overlap (bro)",
		  fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_backbone_alsobuildnewcontigs,
		      "\t    ", "Also build new contigs (abnc)",
		      fieldlength-4);

}

// static func
void MIRAParameters::dumpAssemblyParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
    if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Assembly options (-AS):\n";

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_numpasses,
		  "\t", "Number of passes (nop)",
		  fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_skimeachpass,
		      "\t    ", "Skim each pass (sep)",
		      fieldlength-4);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_numrmbbreakloops,
		  "\t", "Maximum number of RMB break loops (rbl)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_maxcontigsperpass,
		  "\t", "Maximum contigs per pass (mcpp)",
		  fieldlength);

  ostr << '\n';

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_minimum_readlength,
		  "\t", "Minimum read length (mrl)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_minimum_readspercontig,
		  "\t", "Minimum reads per contig (mrpc)",
		  fieldlength);
  multiParamPrintNumericChar(Pv, indexesInPv, ostr, 
			     Pv[0].mp_assembly_params.as_basedefaultqual,
			     "\t", "Base default quality (bdq)",
			     fieldlength);
  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_enforce_qualsinreads,
		      "\t", "Enforce presence of qualities (epoq)",
		      fieldlength);


  ostr << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_automatic_repeat_detection,
		  "\t", "Automatic repeat detection (ard)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_ard_multicopythreshold,
		  "\t    ", "Coverage threshold (ardct)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_ard_multicopyminlen,
		  "\t    ", "Minimum length (ardml)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_ard_multicopygrace,
		  "\t    ", "Grace length (ardgl)",
		  fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_uniform_read_distribution,
		      "\t    ", "Use uniform read distribution (urd)",
		      fieldlength-4);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_urd_startinpass,
		  "\t      ", "Start in pass (urdsip)",
		  fieldlength-6);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_urd_cutoffmultiplier,
		  "\t      ", "Cutoff multiplier (urdcm)",
		  fieldlength-6);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_keep_long_repeats_separated,
		  "\t", "Keep long repeats separated (klrs)",
		  fieldlength);

  ostr << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_spoilerdetection,
		      "\t", "Spoiler detection (sd)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_spdetect_lastpassonly,
		      "\t    ", "Last pass only (sdlpo)",
		      fieldlength-4);

  ostr << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_pathfinder_params.paf_use_genomic_algorithms,
		      "\t", "Use genomic pathfinder (ugpf)",
		      fieldlength);

  ostr << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_pathfinder_params.paf_use_emergency_search_stop,
		      "\t", "Use emergency search stop (uess)",
		      fieldlength);

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_pathfinder_params.paf_ess_depth,
		  "\t    ", "ESS partner depth (esspd)",
		  fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_pathfinder_params.paf_use_emergency_blacklist,
		      "\t", "Use emergency blacklist (uebl)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_pathfinder_params.paf_use_max_contig_buildtime,
		      "\t", "Use max. contig build time (umcbt)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_pathfinder_params.paf_max_contig_buildtime,
		  "\t    ", "Build time in seconds (bts)",
		  fieldlength-4);

}



// static func
void MIRAParameters::dumpLoadParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
    if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Load reads options (-LR):\n";

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_load_sequencedata,
		      "\t", "Load sequence data (lsd)",
		      fieldlength);

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_assembly_params.as_infile_which,
		  "\t    ", "File type (ft)",
		  fieldlength-4);


//  multiParamPrintBool(Pv, singlePvIndex, ostr, 
//		      Pv[0].mp_assembly_params.as_loadSANGER,
//		      "\t", "Load Sanger data (lsand)",
//		      fieldlength);


//  multiParamPrint(Pv, singlePvIndex, ostr, 
//		  Pv[0].mp_assembly_params.as_infile_which,
//		  "\t    ", "Sanger file type (sanft)",
//		  fieldlength-4);


  ostr << "\t    External quality (eq)                   : ";
  switch(Pv[0].mp_assembly_params.as_external_qual){
  case E_QUAL_SCF:{
    ostr << "from SCF (scf)\n";
    break;
  }
  case E_QUAL_NONE:{
    ostr << "none (none)\n";
    break;
  }
  default:{
    ostr << "Unknown??? (please contact the authors)\n";
  }
  }

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_external_qual_override,
		      "\t        ", 
		      "Ext. qual. override (eqo)",
		      fieldlength-8);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_discard_read_on_eq_error,
		      "\t        ", 
		      "Discard reads on e.q. error (droeqe)",
		      fieldlength-8);

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_loadSOLEXA_solexa_scores_in_qual_files,
		      "\t    ", "Solexa scores in qual file (ssiqf)",
		      fieldlength-4);

  multiParamPrintNumericChar(Pv, indexesInPv, ostr, 
			     Pv[0].mp_assembly_params.as_fastq_qualoffset,
			     "\t    ", "FASTQ qual offset (fqqo)",
			     fieldlength-4);

  ostr << '\n';

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_wants_qualityfile,
		      "\t", "Wants quality file (wqf)",
		      fieldlength);

  ostr << '\n';

  ostr << "\tRead naming scheme (rns)                    :";
  for(uint32 ipvindex=0; ipvindex<indexesInPv.size(); ipvindex++){
    if(ipvindex>0) {
      ostr << "\t";
      for(int32 pad=fieldlength;pad>0;pad--) ostr << ' ';
    }
    if(indexesInPv[ipvindex] >= static_cast<int>(Pv.size())) {
      cerr << "BARF! MIRAParameters broken, wanted to read elem " << indexesInPv[ipvindex] << " for vector of size " << Pv.size() << "\n";
      abort();
    }

    ostr << "  [" << Read::getShortNameOfSequencingType(indexesInPv[ipvindex]) << "] ";

    switch(Pv[indexesInPv[ipvindex]].mp_assembly_params.as_readnaming_scheme){
    case SCHEME_SANGER:{
      ostr << "Sanger Institute (sanger)\n";
      break;
    }
    case SCHEME_TIGR:{
      ostr << "TIGR (tigr)\n";
      break;
    }
    case SCHEME_FR:{
      ostr << "forward/reverse (fr)\n";
      break;
    }
    case SCHEME_SOLEXA:{
      ostr << "Solexa (solexa)\n";
      break;
    }
    case SCHEME_STLOUIS:{
      ostr << "St. Louis (stlouis)\n";
      break;
    }
    default:{
      ostr << "Unknown??? (please contact the authors)\n";
    }
    }
  }

  ostr << '\n';

  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_assembly_params.as_mergexmltraceinfo,
		      "\t", 
		      "Merge with XML trace info (mxti)",
		      fieldlength);

  ostr << '\n';

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_filecheck_only,
		      "\t", 
		      "Filecheck only (fo)",
		      fieldlength);

}

void MIRAParameters::dumpGeneralParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "  General (-GE):\n";

  multiParamPrint(Pv, singlePvIndex, ostr, Pv[0].mp_assembly_params.as_projectname_in,
		  "\t", "Project name in (proin)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, Pv[0].mp_assembly_params.as_projectname_out,
		  "\t", "Project name out (proout)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_numthreads,
		  "\t", "Number of threads (not)",
		  fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_automemmanagement,
		      "\t", "Automatic memory management (amm)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_amm_keeppercentfree,
		  "\t    ", "Keep percent memory free (kpmf)",
		  fieldlength-4);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_assembly_params.as_amm_maxprocesssize,
		  "\t    ", "Max. process size (mps)",
		  fieldlength-4);
//  multiParamPrintBool(Pv, singlePvIndex, ostr, 
//		      Pv[0].mp_assembly_params.as_keepcontigsinmem,
//		      "\t", 
//		      "Keep contigs in memory (kcim)",
//		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_special_params.sp_est_startstep,
		  "\t", 
		  "EST SNP pipeline step (esps)",
		  fieldlength);
  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_contig_params.con_use_template_information,
		      "\t", "Use template information (uti)",
		      fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_contig_params.con_insertsize_minimum,
		  "\t    ", "Template insert size minimum (tismin)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_contig_params.con_insertsize_maximum,
		  "\t    ", "Template insert size maximum (tismax)",
		  fieldlength-4);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_contig_params.con_template_build_direction,
		  "\t    ", "Template partner build direction (tpbd)",
		  fieldlength-4);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_assembly_params.as_buntify_reads,
		      "\t", 
		      "Colour reads by hash frequency (crhf)",
		      fieldlength);
}

void MIRAParameters::dumpMiscParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=45;

  vector<int> singlePvIndex;
  if(indexesInPv.size()==1){
    singlePvIndex.push_back(indexesInPv.front());
  }else{
    singlePvIndex.push_back(0);
  }

  ostr << "\n  Misc (-MI):\n";

  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_special_params.mi_stop_on_nfs,
		      "\t", "Stop on NFS (sonfs)",
		      fieldlength);
  multiParamPrintBool(Pv, singlePvIndex, ostr, 
		      Pv[0].mp_special_params.mi_extended_log,
		      "\t", "Extended log (el)",
		      fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_special_params.mi_as_largecontigsize,
		  "\t", "Large contig size (lcs)",
		  fieldlength);
  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_special_params.mi_as_largecontigsize4stats,
		  "\t", "Large contig size for stats(lcs4s)",
		  fieldlength);

  multiParamPrint(Pv, singlePvIndex, ostr, 
		  Pv[0].mp_special_params.mi_stop_readnamelength,
		  "\t", "Stop on max read name length (somrnl)",
		  fieldlength);
}

// static func
void MIRAParameters::dumpAlignParams(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr)
{
  const int32 fieldlength=40;

  ostr << "\n  Align parameters for Smith-Waterman align (-AL):\n";
  
  multiParamPrint(Pv, indexesInPv, ostr, Pv[0].mp_align_params.al_kpercent,
		  "\t", "Bandwidth in percent (bip)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_align_params.al_kmax,
		  "\t", 
		  "Bandwidth max (bmax)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_align_params.al_kmin,
		  "\t", 
		  "Bandwidth min (bmin)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_align_params.al_min_score,
		  "\t", 
		  "Minimum score (ms)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_align_params.al_min_overlap,
		  "\t",
		  "Minimum overlap (mo)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_align_params.al_min_relscore,
		  "\t", 
		  "Minimum relative score in % (mrs)",
		  fieldlength);
  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_align_params.al_solexahack_maxerrors,
		  "\t", 
		  "Solexa_hack_max_errors (shme)",
		  fieldlength);
  multiParamPrintBool(Pv, indexesInPv, ostr, 
		      Pv[0].mp_align_params.ads_extra_gap_penalty,
		      "\t", 
		      "Extra gap penalty (egp)",
		      fieldlength);

  ostr << "\t    extra gap penalty level (egpl)     :";
  for(uint32 ipvindex=0; ipvindex<indexesInPv.size(); ipvindex++){
    if(ipvindex>0) {
      ostr << '\t';
      for(int32 pad=fieldlength;pad>0;pad--) ostr << ' ';
    }
    if(indexesInPv[ipvindex] >= static_cast<int>(Pv.size())) {
      cerr << "BARF! MIRAParameters broken, wanted to read elem " << indexesInPv[ipvindex] << " for vector of size " << Pv.size() << "\n";
      abort();
    }

    ostr << "  [" << Read::getShortNameOfSequencingType(indexesInPv[ipvindex]) << "] ";

    switch(Pv[indexesInPv[ipvindex]].mp_align_params.ads_gp_level) {
    case 0: {
      ostr << "low\n";
      break;
    }
    case 1: {
      ostr << "medium\n";
      break;
    }
    case 2: {
      ostr << "high\n";
      break;
    }
    case 10: {
      ostr << "reject_codongaps\n";
      break;
    }
    default: {
      ostr << "Unknown??? (please contact the authors)\n";
      break;
    }
    }
  }
    

  multiParamPrint(Pv, indexesInPv, ostr, 
		  Pv[0].mp_align_params.ads_max_gppercent,
		  "\t    ", 
		  "Max. egp in percent (megpp)",
		  fieldlength-4);
}

void MIRAParameters::consistencyCheck()
{
  checkMin("-GE:not",mp_assembly_params.as_numthreads,
	   static_cast<uint32>(1),
	   static_cast<uint32>(1));
  checkMax("-GE:not",mp_assembly_params.as_numthreads,
	   static_cast<uint32>(256),
	   static_cast<uint32>(256));
  checkMin("-AS:mrl",mp_assembly_params.as_minimum_readlength,
	   static_cast<uint32>(20),
	   static_cast<uint32>(20));
  checkMin("-AS:bts",mp_pathfinder_params.paf_max_contig_buildtime,1000,2000);
  checkMax("-AS:bdq",mp_assembly_params.as_basedefaultqual,
	     static_cast<base_quality_t>(100),
	     static_cast<base_quality_t>(100));
  checkMin("-AS:rbl",mp_assembly_params.as_numrmbbreakloops,
	   static_cast<uint32>(1),
	   static_cast<uint32>(1));
  
  checkMin("-CL:pecbph",mp_assembly_params.as_clip_pec_basesperhash,
	   static_cast<uint32>(10),
	   static_cast<uint32>(10));

  if(sizeof(uint64) == 4) {
    checkMax("-SB:bph",mp_skim_params.sk_basesperhash,
	     static_cast<uint32>(16),
	     static_cast<uint32>(16));
  }
  if(sizeof(uint64) == 8) {
    checkMax("-SK:bph",mp_skim_params.sk_basesperhash,
	     static_cast<uint32>(32),
	     static_cast<uint32>(32));
  }
//  if(sizeof(unsigned long) == 4) {
//    if(mp_skim_params.sk_basesperhash>=16) {
//      cout << "We're on a 32 bit machine, must adapt -SK:bph.\n";
//    }
//    checkMax("-SK:bph",mp_skim_params.sk_basesperhash,
//	     static_cast<uint32>(15),
//	     static_cast<uint32>(15));
//  }
  checkMax("-SB:sbuip",mp_assembly_params.as_startbackboneusage_inpass,
	   static_cast<int32>(mp_assembly_params.as_numpasses),
	   static_cast<int32>(mp_assembly_params.as_numpasses));

  // TODO: more consistency checks down the parameter list

  checkMin("-OUT:tcpl",mp_contig_params.con_output_text_cpl,10,60);
  checkMin("-OUT:hcpl",mp_contig_params.con_output_html_cpl,10,60);
}

void MIRAParameters::generateProjectNames(vector<MIRAParameters> & Pv, string name) {
  generateProjectInNames(Pv, name);
  generateProjectOutNames(Pv, name);

  return;
}

void MIRAParameters::generateProjectInNames(vector<MIRAParameters> & Pv, string name) {
  if(name.empty()) name=Pv[0].mp_assembly_params.as_projectname_in;
  Pv[0].mp_assembly_params.as_infile_backbone_MAF=name+"_backbone_in.maf";
  Pv[0].mp_assembly_params.as_infile_backbone_CAF=name+"_backbone_in.caf";
  Pv[0].mp_assembly_params.as_infile_backbone_GBF=name+"_backbone_in.gbf";
  Pv[0].mp_assembly_params.as_infile_backbone_GFF3=name+"_backbone_in.gff3";
  Pv[0].mp_assembly_params.as_infile_backbone_FASTA=name+"_backbone_in.fasta";
  Pv[0].mp_assembly_params.as_infile_backbone_FASTAQUAL=name+"_backbone_in.fasta.qual";
  Pv[0].mp_assembly_params.as_infile_FOFNPHD=name+"_in.fofnphd";
  Pv[0].mp_assembly_params.as_infile_PHD=name+"_in.phd.1";
  Pv[0].mp_assembly_params.as_infile_straindata=name+"_straindata_in.txt";
  Pv[0].mp_assembly_params.as_infile_ssahavectorscreen=name+"_ssaha2vectorscreen_in.txt";
  Pv[0].mp_assembly_params.as_infile_smaltvectorscreen=name+"_smaltvectorscreen_in.txt";

  Pv[0].mp_assembly_params.as_infile_chkptMAF="readpool.maf";

  for(uint32 i=0; i<Read::SEQTYPE_END; i++){
    string sname=Read::getNameOfSequencingType(i);
    boost::to_lower(sname);
    Pv[i].mp_assembly_params.as_infile_xmltraceinfo=name+"_traceinfo_in."+sname+".xml";
    Pv[i].mp_assembly_params.as_infile_FASTQ=name+"_in."+sname+".fastq";
    Pv[i].mp_assembly_params.as_infile_FASTA=name+"_in."+sname+".fasta";
    Pv[i].mp_assembly_params.as_infile_FASTAQUAL=name+"_in."+sname+".fasta.qual";
    Pv[i].mp_assembly_params.as_infile_CAF=name+"_in."+sname+".caf";
    Pv[i].mp_assembly_params.as_infile_MAF=name+"_in."+sname+".maf";
    Pv[i].mp_assembly_params.as_infile_FOFNEXP=name+"_in."+sname+".fofn";
  }

  return;
}

void MIRAParameters::generateProjectOutNames(vector<MIRAParameters> & Pv, string name) {
  if(name.empty()) name=Pv[0].mp_assembly_params.as_projectname_out;
  Pv[0].mp_assembly_params.as_outfile_FASTA=name+"_out.unpadded";
  Pv[0].mp_assembly_params.as_outfile_FASTAQUAL=name+"_out.unpadded";
  Pv[0].mp_assembly_params.as_outfile_FASTAPADDED=name+"_out.padded";
  Pv[0].mp_assembly_params.as_outfile_FASTAPADDEDQUAL=name+"_out.padded";
  Pv[0].mp_assembly_params.as_outfile_CAF=name+"_out";
  Pv[0].mp_assembly_params.as_outfile_MAF=name+"_out";
  Pv[0].mp_assembly_params.as_outfile_HTML=name+"_out";
  Pv[0].mp_assembly_params.as_outfile_TXT=name+"_out";
  Pv[0].mp_assembly_params.as_outfile_ACE=name+"_out";
  Pv[0].mp_assembly_params.as_outfile_WIGGLE=name+"_out";
  Pv[0].mp_assembly_params.as_outfile_TCS=name+"_out";
  Pv[0].mp_assembly_params.as_outdir_GAP4DA=name+"_out";

  Pv[0].mp_assembly_params.as_outfile_callparams=name+"_info_callparameters.txt";

  Pv[0].mp_assembly_params.as_outfile_stats_reads_invalid=name+"_info_reads_invalid";
  Pv[0].mp_assembly_params.as_outfile_stats_reads_tooshort=name+"_info_reads_tooshort";
  Pv[0].mp_assembly_params.as_outfile_stats_contigstats=name+"_info_contigstats";
  Pv[0].mp_assembly_params.as_outfile_stats_info=name+"_info_assembly";
  Pv[0].mp_assembly_params.as_outfile_stats_debrislist=name+"_info_debrislist";
  Pv[0].mp_assembly_params.as_outfile_stats_crlist=name+"_info_contigreadlist";
  Pv[0].mp_assembly_params.as_outfile_stats_readtags=name+"_info_readtaglist";
  Pv[0].mp_assembly_params.as_outfile_stats_contigtags=name+"_info_consensustaglist";
  Pv[0].mp_assembly_params.as_outfile_stats_snpanalysis=name+"_info_snplist";
  Pv[0].mp_assembly_params.as_outfile_stats_snpenvironment=name+"_info_snpenvironment";
  Pv[0].mp_assembly_params.as_outfile_stats_featureanalysis=name+"_info_featureanalysis";
  Pv[0].mp_assembly_params.as_outfile_stats_featuresummary=name+"_info_featuresummary";
  Pv[0].mp_assembly_params.as_outfile_stats_featuresequences=name+"_info_featuresequences";
  Pv[0].mp_assembly_params.as_outfile_stats_readrepeats=name+"_info_readrepeats";

  Pv[0].mp_assembly_params.as_tmpf_spoiler=name+"_int_contigjoinspoiler";
  Pv[0].mp_assembly_params.as_tmpf_adsextend=name+"_int_alignextends";
  Pv[0].mp_assembly_params.as_tmpf_vectorclip=name+"_int_vectorclip";
  Pv[0].mp_assembly_params.as_tmpf_posmatch=name+"_int_posmatch";
  //Pv[0].mp_assembly_params.as_tmpf_overlap_criterion_levels=name+"_int_oclevel";
  Pv[0].mp_assembly_params.as_tmpf_normalisedskim=name+"_int_normalisedskims";
  Pv[0].mp_assembly_params.as_tmpf_clippings=name+"_int_clippings";
  Pv[0].mp_assembly_params.as_tmpf_ads=name+"_int_ads";
  Pv[0].mp_assembly_params.as_tmpf_poolinfo=name+"_log_readpoolinfo";

  string topdir=name+"_assembly";
  Pv[0].mp_directory_params.dir_top=topdir;
  Pv[0].mp_directory_params.dir_tmp_symlink.clear();
  Pv[0].mp_directory_params.dir_tmp=topdir+"/"+name+"_d_tmp";
  Pv[0].mp_directory_params.dir_results=topdir+"/"+name+"_d_results";
  Pv[0].mp_directory_params.dir_info=topdir+"/"+name+"_d_info";
  Pv[0].mp_directory_params.dir_checkpoint=topdir+"/"+name+"_d_chkpt";

  //Pv[0].mp_assembly_params.as_tmpf_unused_ids="miratmp.unused_ids";
  Pv[0].mp_assembly_params.as_tmpf_unused_ids="";

  Pv[0].mp_assembly_params.as_tmpf_scfreadfail=name+"_info_scfreadfail";
  Pv[0].mp_assembly_params.as_tmpf_scfreadfatallywrong=name+"_error_scfreadfatallywrong";

  Pv[0].mp_contig_params.con_nameprefix=name;

  return;
}

void MIRAParameters::correctTmpDirectory(vector<MIRAParameters> & Pv) 
{
  if(Pv[0].mp_directory_params.dir_tmp_redirectedto.size()){
    Pv[0].mp_directory_params.dir_tmp_symlink=Pv[0].mp_directory_params.dir_tmp;
    Pv[0].mp_directory_params.dir_tmp=
      Pv[0].mp_directory_params.dir_tmp_redirectedto
      +"/"
      +Pv[0].mp_assembly_params.as_projectname_out
      +"_d_tmp";
  }
}

//#define OUTYESNO(o, yn) { if(yn){o << "Yes\n";}else{ostr << "No\n";}}
//#define OUTCHARSTRING(o,xx) {o<<"'"<<(xx)<<"'\n";}


ostream & operator<<(ostream &ostr, MIRAParameters const  &mp)
{

  vector<MIRAParameters> Pv;
  Pv.push_back(mp);
  vector<int> bla;
  bla.push_back(0);
  
  MIRAParameters::dumpAllParams(
    Pv,
    bla,
    ostr);

  return ostr;
}


//void MIRAParameters::setBorgMode(vector<MIRAParameters> & Pv,bool verbose)
//{
//  cout << "\nWe are MIRA of borg. You will be assembled. Resistance is futile.\n\n";
//
//  // TODO: check the following
//  Pv[0].mp_align_params.al_max_cutoff=1;
//  Pv[0].mp_edit_params.ed_automatic_contic_editing=true;
//
//  const char modestring[] = " -GE:eq=scf:eqo=no:uti=yes"
//    " -ED:ace=yes -AS:mrl=32:nop=4:rbl=3:sd=yes:sdlpo=no:sep=yes"
//    " -DP:ure=yes:rewl=30:rewme=4:feip=0:leip=2"
//    " -AL:ms=15:mrs=55:mo=15:bmin=25:bmax=150:bip=20:egp=no:egpl=low"
//    " -CO:rodirs=30:analysis=signal:dmer=10:mr=yes";
//
//  if(verbose){
//    parseQuickmode(modestring, "-borg", Pv);
//  }else{
//    parseQuickmode(modestring, "", Pv);
//  }
//}
//
//void MIRAParameters::setQuickmodeGenomeDraft(vector<MIRAParameters> & Pv,bool verbose)
//{
//  const char modestring[] =	"\n\t-GE:uti=yes"
//    "\n\t-AS:mrl=40:nop=2:sep=no:rbl=2:sd=no:ugpf=yes"
//    "\n\t-DP:ure=no:tpae=no"
//    "\n\t-CL:pvlc=no:qc=no:mbc=no:emlc=yes:mlcr=25:smlc=30"
//    "\n\t-SK:bph=16:hss=8:pr=60:mhpr=100"
//    "\n\t-AL:bip=15:bmin=25:bmax=70:mo=15:ms=30:mrs=65:egp=yes:egpl=low"
//    "\n\t-CO:rodirs=15:mr=yes:asir=no:mrpg=2:emea=25"
//    "\n\t    amgb=yes:amgbemc=yes:amgbnbs=yes"
//    "\n\t-ED:ace=no";
//  
//  if(verbose){
//    parseQuickmode(modestring, "-genomedraft", Pv);
//  }else{
//    parseQuickmode(modestring, "", Pv);
//  }
//}
//
//void MIRAParameters::setQuickmodeGenomeNormal(vector<MIRAParameters> & Pv,bool verbose)
//{
//  const char modestring[] =	"\n\t-GE:uti=yes"
//    "\n\t-AS:mrl=40:nop=3:sep=no:rbl=3:sd=yes:sdlpo=yes:ugpf=yes"
//    "\n\t-DP:ure=yes:rewl=30:rewme=2:feip=0;leip=0:tpae=no"
//    "\n\t-CL:pvlc=yes:pvcmla=18:qc=no:mbc=no:emlc=yes:mlcr=25:smlc=30"
//    "\n\t-SK:bph=16:hss=4:pr=50:mhpr=200"
//    "\n\t-AL:bip=15:bmin=25:bmax=100:mo=15:ms=30:mrs=65:egp=yes:egpl=low"
//    "\n\t-CO:rodirs=20:mr=yes:asir=no:mrpg=2:emea=25"
//    "\n\t    amgb=yes:amgbemc=yes:amgbnbs=yes"
//    "\n\t-ED:ace=no";
//  
//  if(verbose){
//    parseQuickmode(modestring, "-genomenormal", Pv);
//  }else{
//    parseQuickmode(modestring, "", Pv);
//  }
//}
//void MIRAParameters::setQuickmodeGenomeAccurate(vector<MIRAParameters> & Pv,bool verbose)
//{
//  const char modestring[] =	"\n\t-GE:uti=yes"
//    "\n\t-AS:mrl=40:nop=4:sep=yes:rbl=4:sd=yes:sdlpo=yes:ugpf=yes"
//    "\n\t-DP:ure=yes:rewl=30:rewme=2:feip=0;leip=0:tpae=no"
//    "\n\t-CL:pvlc=yes:pvcmla=18:qc=no:mbc=no:emlc=yes:mlcr=25:smlc=30"
//    "\n\t-SK:bph=16:hss=4:pr=45:mhpr=200"
//    "\n\t-AL:bip=20:bmin=25:bmax=130:mo=15:ms=30:mrs=65:egp=yes:egpl=low"
//    "\n\t-CO:rodirs=25:mr=yes:asir=no:mrpg=2:emea=25"
//    "\n\t    amgb=yes:amgbemc=yes:amgbnbs=yes"
//    "\n\t-ED:ace=no";
//  
//  if(verbose){
//    parseQuickmode(modestring, "-genomeaccurate", Pv);
//  }else{
//    parseQuickmode(modestring, "", Pv);
//  }
//}



// return 0 if ok
int32 MIRAParameters::setSkimParams(skim_parameters & sp)
{
  mp_skim_params=sp;
  return 0;
}

int32 MIRAParameters::setAlignParams(align_parameters & ap)
{
  mp_align_params=ap;
  return 0;
}

int32 MIRAParameters::setDynamicParams(dynamic_parameters & dp)
{
  mp_dynamic_params=dp;
  return 0;
}
int32 MIRAParameters::setAssemblyParams(assembly_parameters & ap)
{
  mp_assembly_params=ap;
  return 0;
}
int32 MIRAParameters::setPathfinderParams(pathfinder_parameters & pp)
{
  mp_pathfinder_params=pp;
  return 0;
}
int32 MIRAParameters::setContigParams(contig_parameters & cp)
{
  mp_contig_params=cp;
  return 0;
}
int32 MIRAParameters::setDirectoryParams(directory_parameters & cp)
{
  mp_directory_params=cp;
  return 0;
}
int32 MIRAParameters::setEditParams(edit_parameters & cp)
{
  mp_edit_params=cp;
  return 0;
}

void MIRAParameters::setPathfinderuseGenomicAlgorithms(bool b)
{
  mp_pathfinder_params.paf_use_genomic_algorithms=b;
}

void MIRAParameters::setPathfinderMaxContigTime(uint32 t)
{
  mp_pathfinder_params.paf_maxcontigclockticks=t*sysconf(_SC_CLK_TCK);
}

void MIRAParameters::setPathfinderNextReadMaxTimeToESS(uint32 t)
{
  mp_pathfinder_params.paf_nextread_maxcttoess=t*sysconf(_SC_CLK_TCK);
}

void MIRAParameters::setContigRODIRS(uint32 r)
{
  mp_contig_params.con_reject_on_drop_in_relscore=r;
}

void MIRAParameters::setContigMinReadsPerGroup(uint32 r)
{
  mp_contig_params.con_minreadspergroup=r;
}

void MIRAParameters::setContigEndReadMarkExclusionArea(uint32 r)
{
  mp_contig_params.con_endreadmarkexclusionarea=r;
}

void MIRAParameters::setContigAssumeSNPInsteadofRepeats(bool b)
{
  mp_contig_params.con_assume_snp_insteadof_rmb=b;
}

void MIRAParameters::setContigDisregardSpuriousRMBMismatches(bool b)
{
  mp_contig_params.con_disregard_spurious_rmb_mismatches=b;
}



void MIRAParameters::setContigMinNeighbourQuality(base_quality_t q)
{
  mp_contig_params.con_minrmbneighbourqual=q;
}

void MIRAParameters::setContigMinGroupQuality(base_quality_t q)
{
  mp_contig_params.con_mingroupqualforrmbtagging=q;;
}

void MIRAParameters::setContigMarkGapBases(bool b)
{
  mp_contig_params.con_also_mark_gap_bases=b;
}

void MIRAParameters::setContigMarkMulticolumnGapBases(bool b)
{
  mp_contig_params.con_also_mark_gap_bases_evenmc=b;
}



void MIRAParameters::setContigAlignmentOutputTextLineLen(int32 l)
{
  mp_contig_params.con_output_text_cpl=l;
}

void MIRAParameters::setContigAlignmentOutputHTMLLineLen(int32 l)
{
  mp_contig_params.con_output_html_cpl=l;
}

void MIRAParameters::setContigAlignmentOutputTextGapPad(char c)
{
  mp_contig_params.con_output_text_gapfill=c;
}

void MIRAParameters::setContigAlignmentOutputHTMLGapPad(char c)
{
  mp_contig_params.con_output_html_gapfill=c;
}

void MIRAParameters::setContigForceNonIUPAC(bool perseq, bool amongseq)
{
  mp_contig_params.con_force_nonIUPACconsensus_perseqtype=false;
  mp_contig_params.con_force_nonIUPACconsensus_amongseqtypes=false;
}

void MIRAParameters::setAlignGapPenaltyLevel(uint32 level)
{
  mp_align_params.ads_gp_level=level;
  mp_align_params.ads_gp_function.clear();
  // 0 stars: no additional penalty
  mp_align_params.ads_gp_function.push_back(0);
  
  switch(level){
  case 0:{
    mp_align_params.ads_gp_function.push_back(0);     // 1 gap
    mp_align_params.ads_gp_function.push_back(5);     // 2 gaps
    mp_align_params.ads_gp_function.push_back(10);    // 3 gaps etc.
    mp_align_params.ads_gp_function.push_back(20);
    mp_align_params.ads_gp_function.push_back(40);
    mp_align_params.ads_gp_function.push_back(80);
    mp_align_params.ads_gp_function.push_back(100);
    break;
  }
  case 1:{
    mp_align_params.ads_gp_function.push_back(0);
    mp_align_params.ads_gp_function.push_back(10);
    mp_align_params.ads_gp_function.push_back(25);
    mp_align_params.ads_gp_function.push_back(50);
    mp_align_params.ads_gp_function.push_back(100);
    break;
  }
  case 2:{
    mp_align_params.ads_gp_function.push_back(0);
    mp_align_params.ads_gp_function.push_back(10);
    mp_align_params.ads_gp_function.push_back(50);
    mp_align_params.ads_gp_function.push_back(100);
    break;
  }
  case 10:{
    // jump function for codon sized gaps
    mp_align_params.ads_gp_function.push_back(0);     // 1 gap
    mp_align_params.ads_gp_function.push_back(5);     // 2 gaps
    mp_align_params.ads_gp_function.push_back(100);   // 3 or more gaps
    break;
  }
  default:{
    mp_align_params.ads_gp_function.push_back(0);
    mp_align_params.ads_gp_function.push_back(10);
    mp_align_params.ads_gp_function.push_back(50);
    mp_align_params.ads_gp_function.push_back(100);
  }
  }
}

void MIRAParameters::setAlignMaxCutoff(uint32 mco)
{
  mp_align_params.al_max_cutoff=mco;
}
void MIRAParameters::setAlignMinScore(uint32 ms)
{
  mp_align_params.al_min_score=ms;
}
void MIRAParameters::setAlignMinRelScore(uint32 mrs)
{
  mp_align_params.al_min_relscore=mrs;
}
void MIRAParameters::setAlignMinOverlap(uint32 mo)
{
  mp_align_params.al_min_overlap=mo;
}
void MIRAParameters::setAlignBandwidthInPercent(uint32 bip)
{
  mp_align_params.al_kpercent=bip;
}
void MIRAParameters::setAlignBandwidthMin(uint32 bm)
{
  mp_align_params.al_kmin=bm;
}
void MIRAParameters::setAlignBandwidthMax(uint32 bm)
{
  mp_align_params.al_kmax=bm;
}


//void MIRAParameters::setAssemblyPatternLength(uint32 pl)
//{
//  mp_assembly_params.as_pattern_length=pl;
//}
//
//void MIRAParameters::setAssemblyNumEndErrors(uint32 nee)
//{
//  mp_assembly_params.as_num_end_errors=nee;
//}
//
//void MIRAParameters::setAssemblyNumMiddleErrors(uint32 nme)
//{
//  mp_assembly_params.as_num_middle_errors=nme;
//}
//
//void MIRAParameters::setAssemblyNumMiddleRuns(uint32 nmr)
//{
//  mp_assembly_params.as_num_middle_runs=nmr;
//}


void MIRAParameters::setAssemblyUseTemplateInformation(bool b)
{
  mp_pathfinder_params.paf_use_template_information=b;
  mp_contig_params.con_use_template_information=b;
}

void MIRAParameters::setEditAutomaticContigEditing(bool b)
{
  mp_edit_params.ed_automatic_contic_editing=b;
}

void MIRAParameters::setEditStrictEditingMode(bool b)
{
  mp_edit_params.ed_strict_editing_mode=b;
}

void MIRAParameters::setEditConfirmationThreshold(uint32 r)
{
  mp_edit_params.ed_confirmation_threshold=r;
}

void MIRAParameters::setAssemblyPutAssembledWithMIRATags(bool b)
{
  mp_assembly_params.as_put_asswithmira_tags=b;
}

void MIRAParameters::setAssemblyReadExtension(bool b)
{
  mp_assembly_params.as_use_read_extension=b;
}

void MIRAParameters::setAssemblyClipPossibleVectors(bool b)
{
  mp_assembly_params.as_clip_possible_vectors=b;
}

void MIRAParameters::setAssemblyLoadStrainData(bool b)
{
  mp_assembly_params.as_loadstraindata=b;
}

void MIRAParameters::setAssemblyNumRMBBreakLoops(int32 i)
{
  mp_assembly_params.as_numrmbbreakloops=i;
}

void MIRAParameters::setAssemblyStartBackboneUsageInPass(int32 i)
{
  mp_assembly_params.as_startbackboneusage_inpass=i;
}

void MIRAParameters::setAssemblyLoadBackbone(bool b)
{
  mp_assembly_params.as_loadbackbone=b;
}

void MIRAParameters::setAssemblyNumPasses(int32 i)
{
  mp_assembly_params.as_numpasses=i;
}

void MIRAParameters::setAssemblyMarkRepeats(bool b)
{
  mp_assembly_params.as_mark_repeats=b;
}

void MIRAParameters::setAssemblyMinLength(uint32 r)
{
  mp_assembly_params.as_minimum_readlength=r;
}

void MIRAParameters::setAssemblyInfileWhich(string & iw)
{
  mp_assembly_params.as_infile_which=iw;
}

void MIRAParameters::setAssemblyLogUnusedIDs( const string & s)
{
  mp_assembly_params.as_tmpf_unused_ids=s;
}

void MIRAParameters::setAssemblyLogADS( const string & s)
{
  mp_assembly_params.as_tmpf_ads=s;
}

//void MIRAParameters::setAssemblyProjectName( const string & s)
//{
//  mp_assembly_params.as_projectname=s;
//  generateProjectNames();
//}

void MIRAParameters::setAssemblyInfileFOFNEXP( const string & s)
{
  mp_assembly_params.as_infile_FOFNEXP=s;
}

void MIRAParameters::setAssemblyInfileFASTA( const string & s)
{
  mp_assembly_params.as_infile_FASTA=s;
}
void MIRAParameters::setAssemblyInfileFASTAQUAL( const string & s)
{
  mp_assembly_params.as_infile_FASTAQUAL=s;
}

void MIRAParameters::setAssemblyInfileCAF( const string & s)
{
  mp_assembly_params.as_infile_CAF=s;
}

void MIRAParameters::setAssemblyInfileStrainData( const string & s)
{
  mp_assembly_params.as_infile_straindata=s;
}
void MIRAParameters::setAssemblyOutfileCAF( const string & s)
{
  mp_assembly_params.as_outfile_CAF=s;
}
void MIRAParameters::setAssemblyOutdirGAP4DA( const string & s)
{
  mp_assembly_params.as_outdir_GAP4DA=s;
}
void MIRAParameters::setAssemblyFASTQQualOffset(base_quality_t b)
{
  mp_assembly_params.as_fastq_qualoffset=b;
}





void MIRAParameters::loadParams(const string & pfile, vector<MIRAParameters> & Pv)
{
  FUNCSTART("void MIRAParameters::loadParams(const string & pfile)");

  cout << "Loading parameters from file: " << pfile << endl;

  if(MP_loadfilename.size()>=10) {
    MIRANOTIFY(Notify::FATAL, "Already loading 10 other files ... there's something unusual about that ... really: " << pfile);
  }

  for(uint32 i=0; i<MP_loadfilename.size(); i++) {
    if(MP_loadfilename[i] == pfile) {
      MIRANOTIFY(Notify::FATAL, "Already loading that file, recursion, sorry: " << pfile);
    }
  }

  MP_loadfilename.push_back(pfile);

  ifstream fin;

  fin.open(pfile.c_str(), ios::in);
  if(!fin){
    MIRANOTIFY(Notify::FATAL, "File not found: " << pfile);
  }
  
  parse(fin, Pv, NULL);

  fin.close();

  MP_loadfilename.pop_back();

  Pv[0].consistencyCheck();

  FUNCEND();
  return;
}

void MIRAParameters::parse(int argc, char **argv, vector<MIRAParameters> & Pv)
{
  stringstream tss;

  cout << "Parsing parameters:";
  for(int32 i=1; i<argc; i++){
    cout << " " << argv[i];
    //    cout << "Doing " << i << "\t" << argv[i] << endl;
    tss << argv[i] << "  *=BEGIN0=*";
  }
  cout << "\n\n";

  parse(tss, Pv, NULL);
}

void MIRAParameters::parse(const char * params, vector<MIRAParameters> & Pv)
{
  stringstream tss;
  tss << params;
  parse(tss, Pv, NULL);
}

void MIRAParameters::parseQuickmode(const char * params, const char * qm, vector<MIRAParameters> & Pv, bool verbose,MIRAParameters * mp)
{
  stringstream tss;

  //verbose=true;
  
  if(verbose){
    if(strlen(qm)>0) {
      cout << "Using quickmode switch " << qm << " : ";
    }
    cout << params << endl;
  }
  tss << params;
  parse(tss, Pv, mp);
}

void MIRAParameters::parseQuickmodeNoTechSettingsChange(const char * params, const char * qm, vector<MIRAParameters> & Pv, bool verbose,MIRAParameters * mp)
{
  vector<bool> save;
  saveParsedSettingsValues(Pv, mp, save);
  parseQuickmode(params, qm, Pv, verbose, mp);
  restoreParsedSettingsValues(Pv, mp, save);
}

int32 MIRAParameters::gimmeAnInt(FlexLexer * lexer, stringstream & errstream)
{
  string currenttoken=lexer->YYText();
  if(lexer->yylex() != MP_INT){
    errstream << "* Parameter section: '" << MP_currentparametersection << "'\t\tToken '" << currenttoken << "'\n*\tExpected a number after this, not '" << lexer->YYText() <<"'\n\n";
    MP_errorinparams=true;
  }

  CEBUG("\t\tInt: " << lexer->YYText() << endl);
  return atoi(lexer->YYText());
}

double MIRAParameters::gimmeADouble(FlexLexer * lexer, stringstream & errstream)
{
  string currenttoken=lexer->YYText();
  int yyretcode=lexer->yylex();
  if(yyretcode != MP_INT && yyretcode != MP_FLOAT ){
    errstream << "* Parameter section: '" << MP_currentparametersection << "'\t\tToken '" << currenttoken << "'\n*\tExpected a number (int or float) after this, not '" << lexer->YYText() <<"'\n\n";
    MP_errorinparams=true;
  }
  CEBUG("\t\tDouble: " << lexer->YYText() << endl);
  return atof(lexer->YYText());
}

int32 MIRAParameters::getFixedStringMode(FlexLexer * lexer, stringstream & errstream)
{
  string currenttoken=lexer->YYText();
  int32 tmp=lexer->yylex();
  if(tmp==MP_UNRECOGNISED_STRING){
    errstream << "* Parameter section: '" << MP_currentparametersection << "'\t\tToken '" << currenttoken << "'\n*\tNon recognised string '" << lexer->YYText() << "', probably expected something\n*\tlike yes|no|on|off|1|0 or other fixed strings.\n\n";
    MP_errorinparams=true;
  }
  CEBUG("\t\tFixed string: " << lexer->YYText() << endl);
  return tmp;
}

//void MIRAParameters::checkCOMMON(MIRAParameters * actpar, vector<MIRAParameters> & Pv, FlexLexer * lexer, stringstream & errstream, const string & laststset){
//  if(!Pv.empty()){
//    if(actpar!=&Pv[0]){
//      errstream << "* Parameter section: '" << MP_currentparametersection << "'\n*\tParameter '" << lexer->YYText() << "' can only be set as COMMON_SETTINGS, not individually\n*\tfor a specific sequencing type (" << laststset << ").\n\n";
//      MP_errorinparams=true;
//    }
//  }
//}

void MIRAParameters::checkCOMMON(const string & currentst, FlexLexer * lexer, stringstream & errstream){
  if(currentst != "COMMON_SETTINGS"
     && currentst != "_COMMON_SETTINGS"){
    errstream << "* Parameter section: '" << MP_currentparametersection << "'\n*\tParameter '" << lexer->YYText() << "' can only be set as COMMON_SETTINGS, not individually\n*\tfor a specific sequencing type (" << currentst << ").\n\n";

    CEBUG("Argh!!! " << MP_currentparametersection << " " << lexer->YYText() << " not COMMON but " << currentst << endl);

    MP_errorinparams=true;
  }
}
void MIRAParameters::checkNONCOMMON(const string & currentst, FlexLexer * lexer, stringstream & errstream){
  if(currentst == "COMMON_SETTINGS"
     || currentst == "_COMMON_SETTINGS"){
    errstream << "* Parameter section: '" << MP_currentparametersection << "'\n*\tParameter '" << lexer->YYText() << "' can only be set as sequencing type specific"
      "\n*\tparameter (SANGER_SETTINGS, 454_SETTINGS, etc.pp)\n*\tand not for " << currentst << ".\n\n";

    CEBUG("Argh!!! " << MP_currentparametersection << " " << lexer->YYText() << " not NONCOMMON but " << currentst << endl);

    MP_errorinparams=true;
  }
}


static const string noclipping_string = "-CL:msvs=no:pvlc=no:qc=no:bsqc=no:mbc=no:lcc=no:emlc=no:emrc=no:c3pp=no:cpat=no:mqtfer=0:ckar=no";
static const string noquality_string = "-LR:wqf=no -AS:bdq=20:epoq=no -CL:qc=no:bsqc=no";


// parses either into MIRAParameters Pv vector
//  or into single MIRAParameters object 
void MIRAParameters::parse(istream & is, vector<MIRAParameters> & Pv, MIRAParameters * singlemp)
{
  FUNCSTART("void MIRAParameters::parse(istream & is, vector<MIRAParameters> & Pv, MIRAParameters * singlemp)");

  stringstream errstream;

  FlexLexer* lexer = new MPFlexLexer(&is);

  MP_errorinparams=false;

  string currentseqtypesettings="COMMON_SETTINGS";

  MP_currentparametersection="(none)";

  MIRAParameters * actpar;

  if(!Pv.empty()){
    actpar=&Pv[Read::SEQTYPE_SANGER];
  }else if(singlemp != NULL){
    actpar=singlemp;
  }else{
    MIRANOTIFY(Notify::INTERNAL, "No parameter object to parse into? Not good.");
  }

  actpar->MP_parsedsomesettings=true;

  // vector to hold quickswitch --job definitions
  vector<vector<uint32> > jobdefs(4);

  int yyretcode=-1;
  while(yyretcode!=0){
    yyretcode=lexer->yylex();
    CEBUG("******: " << yyretcode << "\t" << lexer->YYText() << endl);
    switch(yyretcode){
    case 0: {break;}                              // do nothing, eof
    case MP_PARSING_SECTIONCHANGE: {
      MP_currentparametersection=lexer->YYText();
      break;
    }
    case MP_PARSING_SECTIONRESET: {
      MP_currentparametersection="(none)";
      break;
    }
    case MP_as_projectname: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR Projectname: projectname not found?\n";
	MP_errorinparams=true;
	break;
      }
      Pv[0].mp_assembly_params.as_projectname_in=lexer->YYText();
      Pv[0].mp_assembly_params.as_projectname_out=lexer->YYText();
      generateProjectNames(Pv);
      break;
    }
    case MP_as_projectname_in: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR Projectname in: projectname not found?\n";
	MP_errorinparams=true;
	break;
      }
      Pv[0].mp_assembly_params.as_projectname_in=lexer->YYText();
      generateProjectInNames(Pv);
      break;
    }
    case MP_as_projectname_out: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR Projectname out: projectname not found?\n";
	MP_errorinparams=true;
	break;
      }
      Pv[0].mp_assembly_params.as_projectname_out=lexer->YYText();
      generateProjectOutNames(Pv);
      break;
    }
    case MP_as_numthreads:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_numthreads=gimmeAnInt(lexer,errstream);
      actpar->mp_skim_params.sk_numthreads=actpar->mp_assembly_params.as_numthreads;
      break;
    }
    case MP_as_automemmanagement:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_automemmanagement=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_amm_keeppercentfree:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_amm_keeppercentfree=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_amm_maxprocesssize:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_amm_maxprocesssize=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_ed_automatic_contig_editing:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_edit_params.ed_automatic_contic_editing=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_ed_strict:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_edit_params.ed_strict_editing_mode=getFixedStringMode(lexer,errstream);;
      break;
    }
    case MP_ed_confirmation_threshold:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_edit_params.ed_confirmation_threshold=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sp_est_startstep:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_special_params.sp_est_startstep=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_mi_stoponnfs:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_special_params.mi_stop_on_nfs=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_mi_extendedlog:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_special_params.mi_extended_log=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_mi_stop_maxreadnamelength:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_special_params.mi_stop_readnamelength=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_mi_as_largecontigsize:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_special_params.mi_as_largecontigsize=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_mi_as_largecontigsize4stats:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_special_params.mi_as_largecontigsize4stats=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_load_sequencedata:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_load_sequencedata=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_infile_loadjob:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR file type not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_which=lexer->YYText();
      std::transform(actpar->mp_assembly_params.as_infile_which.begin(),
		     actpar->mp_assembly_params.as_infile_which.end(),
		     actpar->mp_assembly_params.as_infile_which.begin(), 
		     (int(*)(int))std::tolower); // ugly
      break;
    }
    case MP_as_readnaming_scheme:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_readnaming_scheme=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_external_quality:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_external_qual=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_extend_reads:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_use_read_extension=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_external_quality_override:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_external_qual_override=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_discard_read_on_eq_error:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_discard_read_on_eq_error=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_cleanup_tmp_files:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_cleanup_tmpfiles=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_mergessahavectorscreen:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_mergessahavectorscreen=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_ssahamerge_gapsize:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_ssahamerge_gapsize=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_ssahamerge_maxfrontgap:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_ssahamerge_maxfrontgap=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_ssahamerge_maxendgap:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_ssahamerge_maxendgap=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_ssahamerge_strictfrontclip:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_ssahamerge_strictfrontclip=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_ssahamerge_strictendclip:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_ssahamerge_strictendclip=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_possible_vectors: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_possible_vectors=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_vector_maxlenallowed: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_vector_maxlenallowed=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_ensureminimumleftclip: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_ensureminimumleftclipoff=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_minimumleftcliprequired: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_minslrequired=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_setminimumleftclip: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_minqlsetto=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_ensureminimumrightclip: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_ensureminimumrightclipoff=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_minimumrightcliprequired: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_minsrrequired=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_setminimumrightclip: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_minqrsetto=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_quality_minthreshold:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_quality_minthreshold=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_quality_numminthreshold:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_quality_numminthreshold=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_quality: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_quality=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_quality_minqual:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_quality_minqual=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_quality_winlen:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_quality_winlen=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_badstretchquality: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_badstretchquality=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_badstretchquality_minqual:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_badstretchquality_minqual=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_badstretchquality_winlen:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_badstretchquality_winlen=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_maskedbases: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_maskedbases=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_maskedbases_gapsize:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_maskedbase_gapsize=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_maskedbases_maxfrontgap:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_maskedbase_maxfrontgap=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_maskedbases_maxendgap:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_maskedbase_maxendgap=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_lowercase: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_lowercase=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_skimchimeradetection: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_skimchimeradetection=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_skimjunkdetection: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_skimjunkdetection=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_proposeendclips: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_proposeendclips=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_pec_sxaggcxg: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_pec_sxaggcxg=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_pec_basesperhash:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_pec_basesperhash=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_badsolexaends: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_badsolexaends=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_knownadaptorsright: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_knownadaptorsright=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_basedefaultqual:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_basedefaultqual=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_name_prefix: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR -CO:name_prefix: string not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_contig_params.con_nameprefix=lexer->YYText();
      break;
    }
    case MP_as_mark_repeats:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_mark_repeats=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_mark_repeats_only_in_result:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_mark_repeats_onlyinresult=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_assume_snp_insteadof_rmb:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_assume_snp_insteadof_rmb=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_spoiler_detection:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_spoilerdetection=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_sd_lastpassonly:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_spdetect_lastpassonly=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_uniform_read_distribution:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_uniform_read_distribution=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_urd_startinpass:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_urd_startinpass=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_urd_cutoffmultiplier:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_urd_cutoffmultiplier=gimmeADouble(lexer,errstream);
      break;
    }
    case MP_as_automatic_repeat_detection:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_automatic_repeat_detection=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_ard_multicopythreshold:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_ard_multicopythreshold=gimmeADouble(lexer,errstream);
      break;
    }
    case MP_as_ard_multicopyminlen:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_ard_multicopyminlen=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_ard_multicopygrace:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_ard_multicopygrace=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_keep_long_repeats_separated:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_keep_long_repeats_separated=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_polyat:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_polyat=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_polyat_keeppolysignal:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_polyat_keeppolysignal=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_polyat_len:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_polyat_len=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_polyat_maxerrors:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_polyat_maxerrors=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_polyat_maxgap:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_polyat_maxgap=gimmeAnInt(lexer,errstream);
      break;
    }

    case MP_as_clip_3ppolybase:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_3ppolybase=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_clip_3ppolybase_len:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_3ppolybase_len=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_3ppolybase_maxerrors:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_3ppolybase_maxerrors=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_clip_3ppolybase_maxgap:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_clip_3ppolybase_maxgap=gimmeAnInt(lexer,errstream);
      break;
    }

    case MP_as_readextension_window_len:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_readextension_window_len=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_readextension_window_maxerrors:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_readextension_window_maxerrors=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_readextension_firstpassnum:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_readextension_firstpassnum=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_readextension_lastpassnum:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_readextension_lastpassnum=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_filecheck_only:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_filecheck_only=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_loadstraindata:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_loadstraindata=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_assign_defaultstrain:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_assigndefaultstrain=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_default_strainname:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR default strainname: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_default_strainname=lexer->YYText();
      break;
    }
    case MP_as_loadbackbone:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_loadbackbone=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_backbone_alsobuildnewcontigs:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_backbone_alsobuildnewcontigs=getFixedStringMode(lexer,errstream);
      break;
    }
    //case MP_as_backbone_loadjob:{
    //  actpar->mp_assembly_params.as_infile_backbone_which=getFixedStringMode(lexer,errstream);
    //  break;
    //}
    case MP_as_backbone_loadjob:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR backbone file type not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_backbone_which=lexer->YYText();
      std::transform(actpar->mp_assembly_params.as_infile_backbone_which.begin(),
		     actpar->mp_assembly_params.as_infile_backbone_which.end(),
		     actpar->mp_assembly_params.as_infile_backbone_which.begin(), 
		     (int(*)(int))std::tolower); // ugly
      break;
    }
    case MP_as_backbone_strainname: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR backbone strainname: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_backbone_strainname=lexer->YYText();
      break;
    }
    case MP_as_backbone_strainname_forceforall: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_backbone_strainname_forceforall=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_backbone_rail_fromstrain: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR backbone rail from strain: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_backbone_rail_fromstrain=lexer->YYText();
      break;
    }
    case MP_as_startbackboneusage_inpass:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_startbackboneusage_inpass=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_backbone_raillength:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_backbone_raillength=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_backbone_railoverlap:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_backbone_railoverlap=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_backbone_outlen:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_backbone_outlen=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_backbone_basequals:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_backbone_basequals=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_merge_xmltraceinfo:{
      actpar->mp_assembly_params.as_mergexmltraceinfo=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_keepcontigsinmem:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_keepcontigsinmem=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_buntify_reads:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_buntify_reads=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_savesimplesingletsinproject:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_savesimplesingletsinproject=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_savetaggedsingletsinproject:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_savetaggedsingletsinproject=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_loadSOLEXA_solexa_scores_in_qual_files:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_loadSOLEXA_solexa_scores_in_qual_files=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_fastq_qualoffset:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_fastq_qualoffset=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_enforce_qualsinreads:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_enforce_qualsinreads=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_wants_qualityfile:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_wants_qualityfile=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_output_html_cpl:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_output_html_cpl=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_output_text_cpl:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_output_text_cpl=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_output_html_gapfill: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "-OUT:hegfc: character not found?\n";
	MP_errorinparams=true;
	break;
      }
      string tmp=lexer->YYText();
      if(tmp.size()>1) tmp=tmp[tmp.size()-2];
      actpar->mp_contig_params.con_output_html_gapfill=tmp[0];
      break;
    }
    case MP_con_output_text_gapfill: {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "-OUT:tegfc: character not found?\n";
	MP_errorinparams=true;
	break;
      }
      string tmp=lexer->YYText();
      if(tmp.size()>1) tmp=tmp[tmp.size()-2];
      actpar->mp_contig_params.con_output_text_gapfill=tmp[0];
      break;
    }
    case MP_as_output_removerollovertmps:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_removerollovertmps=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_removetmpdir:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_removetmpdir=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tmp_html:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tmp_html=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tmp_tcs:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tmp_tcs=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tmp_text:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tmp_txt=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tmp_caf:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tmp_caf=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tmp_maf:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tmp_maf=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tmp_ace:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tmp_ace=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tmp_gap4da:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tmp_gap4da=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tmp_fasta:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tmp_fasta=getFixedStringMode(lexer,errstream);
      break;
    }

    case MP_as_output_exttmp_html:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_exttmp_html=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_exttmp_caf:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_exttmp_caf=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_exttmp_ace:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_exttmp_ace=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_exttmp_gap4da:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_exttmp_gap4da=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_exttmp_fasta:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_exttmp_fasta=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_exttmp_alsosinglets:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_exttmp_alsosinglets=getFixedStringMode(lexer,errstream);
      break;
    }

    case MP_as_output_html:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_html=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_text:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_txt=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_ace:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_ace=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_gff3:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_gff3=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_wiggle:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_wiggle=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_tcs:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_tcs=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_gap4da:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_gap4da=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_caf:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_caf=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_maf:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_maf=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_output_fasta:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_output_fasta=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_numpasses:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_numpasses=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_skimeachpass:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_skimeachpass=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_as_numrmbbreakloops:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_numrmbbreakloops=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_maxcontigsperpass:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_maxcontigsperpass=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_minimum_readlength:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_minimum_readlength=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_as_minimum_readspercontig:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_minimum_readspercontig=gimmeAnInt(lexer,errstream);
      break;
    }
//    case MP_ads_extra_mismatch_penalty:{
//      int32 tmp=getFixedStringMode(lexer,errstream);
//      if(tmp==MP_UNRECOGNISED_STRING){
//	errstream << "-AL:emp" << endl;
//	break;
//      } 
//      actpar->mp_align_params.ads_extra_mismatch_penalty=(tmp>0);
//      break;
//    }
//    case MP_ads_emp_windowlen:{
//      actpar->mp_align_params.ads_emp_windowlen=gimmeAnInt(lexer,errstream);
//      break;
//    }
//    case MP_ads_emp_maxmismatches:{
//      actpar->mp_align_params.ads_emp_maxmismatches=gimmeAnInt(lexer,errstream);
//      break;
//    }
    case MP_ads_extra_gap_penalty:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_align_params.ads_extra_gap_penalty=(getFixedStringMode(lexer,errstream)>0);
      break;
    }
    case MP_ads_max_gppercent:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_align_params.ads_max_gppercent=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_ads_gp_level:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->setAlignGapPenaltyLevel(getFixedStringMode(lexer,errstream));
      break;
    }
    case MP_sk_numthreads:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_numthreads=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_alsoskimrevcomp:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_alsoskimrevcomp=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_sk_swcheckonbackbones:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_swcheckonbackbones=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_sk_basesperhash:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_basesperhash=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_hashsavestepping:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_hashsavestepping=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_percentrequired:{
      actpar->mp_skim_params.sk_percentrequired=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_maxhitsperread:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_maxhitsperread=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_maxhashesinmemory:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_maxhashesinmem=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_memcaphitreduction:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_memcaphitreduction=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_masknastyrepeats:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_masknastyrepeats=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_sk_nastyrepeatratio:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_nastyrepeatratio=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_repeatlevel_in_infofile:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_repeatlevel_in_infofile=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_sk_freqest_minnormal:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_freqest_minnormal=gimmeADouble(lexer,errstream);
      break;
    }
    case MP_sk_freqest_maxnormal:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_freqest_maxnormal=gimmeADouble(lexer,errstream);
      break;
    }
    case MP_sk_freqest_repeat:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_freqest_repeat=gimmeADouble(lexer,errstream);
      break;
    }
    case MP_sk_freqest_heavyrepeat:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_freqest_heavyrepeat=gimmeADouble(lexer,errstream);
      break;
    }
    case MP_sk_freqest_crazyrepeat:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_freqest_crazyrepeat=gimmeADouble(lexer,errstream);
      break;
    }
    case MP_sk_maxmegahubratio:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_skim_params.sk_maxmegahubratio=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_al_bip:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_align_params.al_kpercent=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_al_bmin:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_align_params.al_kmin=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_al_bmax:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_align_params.al_kmax=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_al_min_overlap:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_align_params.al_min_overlap=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_al_min_score:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_align_params.al_min_score=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_al_min_relscore:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_align_params.al_min_relscore=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_al_solexahack_maxerrors:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      int32 tmp=gimmeAnInt(lexer,errstream);
      for(uint32 p=0; p< Pv.size(); p++) {
	Pv[p].mp_align_params.al_solexahack_maxerrors=tmp;
      }
      break;
    }
    case MP_as_use_template_information:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->setAssemblyUseTemplateInformation(getFixedStringMode(lexer,errstream));
      break;
    }
    case MP_con_analyse_mode:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_danger_analyse_mode=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_rodirs:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_reject_on_drop_in_relscore=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_also_mark_gap_bases:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_also_mark_gap_bases=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_also_mark_gap_bases_evenmc:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_also_mark_gap_bases_evenmc=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_also_mark_gap_bases_needbothstrands:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_also_mark_gap_bases_needbothstrands=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_min_groupqual_for_rmb_tagging:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_mingroupqualforrmbtagging=gimmeAnInt(lexer,errstream);
      break;
    }
    //case MP_con_min_groupqual_for_wrmbsrmb_change:{
    //  actpar->mp_contig_params.con_min_groupqual_srmbwrmb_change=gimmeAnInt(lexer,errstream);
    //  break;
    //}
    //case MP_con_num_rmb_zone_trigger:{
    //  actpar->mp_contig_params.con_rmb_numzone_trigger=gimmeAnInt(lexer,errstream);
    //  break;
    //}
    case MP_con_min_readspergroup:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_minreadspergroup=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_min_rmb_neighbourqual:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_minrmbneighbourqual=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_endread_mark_exclusion_area:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_endreadmarkexclusionarea=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_emea_setzero_on_clipping_pec:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_emea_setzero_on_clipping_pec=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_gap_override_ratio:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_gap_override_ratio=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_danger_max_error_rate:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_danger_max_error_rate=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_insertsize_minimum:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_insertsize_minimum=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_insertsize_maximum:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_insertsize_maximum=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_template_build_direction:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_template_build_direction=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_con_force_nonIUPACconsensus_perseqtype:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_force_nonIUPACconsensus_perseqtype=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_force_nonIUPACconsensus_amongseqtypes:{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_force_nonIUPACconsensus_amongseqtypes=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_mergeshortreads:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_mergeshortreads=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_con_msr_keependsunmapped:{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_contig_params.con_msr_keependsunmapped=gimmeAnInt(lexer,errstream);
      break;
    }

    case MP_paf_use_genomic_pathfinder :{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->setPathfinderuseGenomicAlgorithms(getFixedStringMode(lexer,errstream));
      break;
    }
    case MP_paf_use_emergency_blacklist :{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_use_emergency_blacklist=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_paf_use_emergency_search_stop :{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_use_emergency_search_stop=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_paf_ess_partnerdepth :{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_ess_depth=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_paf_use_max_contig_buildtime :{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_use_max_contig_buildtime=getFixedStringMode(lexer,errstream);
      break;
    }
    case MP_paf_buildtime_inseconds :{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_max_contig_buildtime=gimmeAnInt(lexer,errstream);
      actpar->setPathfinderMaxContigTime(actpar->mp_pathfinder_params.paf_max_contig_buildtime); 
      break;
    }
    case MP_paf_max_startcache_filltime :{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_max_startcache_filltime=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_paf_skip_whole_contig_scan :{
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_skipwholecontigscan=static_cast<int16>(getFixedStringMode(lexer,errstream));
      break;
    }
    case MP_paf_use_quick_rule :{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_use_quick_rule=static_cast<int16>(getFixedStringMode(lexer,errstream));
      break;
    }
    case MP_paf_quickrule_minlen1 :{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_quickrule_minlen1=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_paf_quickrule_minsim1 :{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_quickrule_minsim1=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_paf_quickrule_minlen2 :{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_quickrule_minlen2=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_paf_quickrule_minsim2 :{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_quickrule_minsim2=gimmeAnInt(lexer,errstream);
      break;
    }
    case MP_paf_bbquickoverlap_minlen :{
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_pathfinder_params.paf_bbquickoverlap_minlen=gimmeAnInt(lexer,errstream);
      break;
    }

    case MP_fn_straindatain: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename straindatain: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_straindata=lexer->YYText();
      break;
    }
    case MP_fn_xmltraceinfoin: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename xmltraceinfoin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_xmltraceinfo=lexer->YYText();
      break;
    }
    case MP_fn_ssahavectorscreenin: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename SSAHA2 vectorscreenin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_ssahavectorscreen=lexer->YYText();
      break;
    }
    case MP_fn_smaltvectorscreenin: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename SMALT vectorscreenin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_smaltvectorscreen=lexer->YYText();
      break;
    }
    case MP_fn_fastaqualin: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename fastaqualin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_FASTAQUAL=lexer->YYText();
      break;
    }
    case MP_fn_fastain: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename fastain: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_FASTA=lexer->YYText();
      break;
    }
    case MP_fn_fastqin: {
      checkNONCOMMON(currentseqtypesettings, lexer, errstream);
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename fastqin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_FASTQ=lexer->YYText();
      break;
    }
    case MP_fn_phdin: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename phdin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_PHD=lexer->YYText();
      break;
    }
    case MP_fn_backbonein: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename cafin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_backbone_MAF=lexer->YYText();
      actpar->mp_assembly_params.as_infile_backbone_CAF=lexer->YYText();
      actpar->mp_assembly_params.as_infile_backbone_GBF=lexer->YYText();
      actpar->mp_assembly_params.as_infile_backbone_GFF3=lexer->YYText();
      actpar->mp_assembly_params.as_infile_backbone_FASTA=lexer->YYText();
      actpar->mp_assembly_params.as_infile_backbone_FASTAQUAL=actpar->mp_assembly_params.as_infile_backbone_FASTA+".qual";
      break;
    }
    case MP_fn_cafin: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename cafin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      size_t slen=strlen(lexer->YYText());
      if(slen<=4
	 || lexer->YYText()[slen-4]!='.'
	 || lexer->YYText()[slen-3]!='c'
	 || lexer->YYText()[slen-2]!='a'
	 || lexer->YYText()[slen-1]!='f'
	 ){
	errstream << "ERROR: Filename cafout: filename does not end in '.caf'\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_CAF=lexer->YYText();
      break;
    }
    case MP_fn_cafout: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename cafout: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_outfile_CAF=lexer->YYText();
      break;
    }
    case MP_fn_fofnexpin: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename fofnin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_FOFNEXP=lexer->YYText();
      break;
    }
    case MP_fn_fofnphdin: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Filename phdin: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_FOFNPHD=lexer->YYText();
      break;
    }
    case MP_dir_cwd: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Directory name cwd: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_directory_params.dir_cwd=lexer->YYText();
      if(!actpar->mp_directory_params.dir_cwd.empty()){
	int ret=chdir(actpar->mp_directory_params.dir_cwd.c_str());
	if(ret){
	  MIRANOTIFY(Notify::FATAL,"Changing working directory to '" << actpar->mp_directory_params.dir_cwd << "' failed, system message is: " << strerror(errno) << endl);
	}
      }
      break;
    }
    case MP_dir_tmp: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Directory name tmp: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_directory_params.dir_tmp=lexer->YYText();
      break;
    }
    case MP_dir_tmp_redirectedto: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Directory name tmp_redirected_to: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_directory_params.dir_tmp_redirectedto=lexer->YYText();
      break;
    }
    case MP_dir_gap4da: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Directory name tmp: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_outdir_GAP4DA=lexer->YYText();
      break;
    }
    case MP_dir_exp: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Directory name exp: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_directory_params.dir_exp=lexer->YYText();
      break;
    }
    case MP_dir_scf: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Directory name scf: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_directory_params.dir_scf=lexer->YYText();
      break;
    }
    case MP_dir_fasta: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR Directory name tmp: name not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_directory_params.dir_fasta=lexer->YYText();
      break;
    }
    case MP_quickmode_fastaname: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR --fasta=: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_FASTA=lexer->YYText();
      actpar->mp_assembly_params.as_infile_FASTAQUAL=actpar->mp_assembly_params.as_infile_FASTA+".qual";
      if(!Pv.empty()){
	for(uint32 i=0; i<Pv.size(); i++){
	  Pv[i].mp_assembly_params.as_infile_FASTA=lexer->YYText();
	  Pv[i].mp_assembly_params.as_infile_FASTAQUAL=actpar->mp_assembly_params.as_infile_FASTA+".qual";
	}
      }
      // FALL THROUGH!
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_fasta: {
      static const char modestring[] = 
	"SANGER_SETTINGS -LR:ft=fasta"
	"\n454_SETTINGS -LR:ft=fasta"
	"\nIONTOR_SETTINGS -LR:ft=fasta"
	"\nPACBIO_SETTINGS -LR:ft=fasta"
	"\nSOLEXA_SETTINGS -LR:ft=fasta"
	"\nSOLID_SETTINGS -LR:ft=fasta";
      parseQuickmodeNoTechSettingsChange(modestring, "fasta", Pv,false);
      break;
    }
    case MP_quickmode_fastqname: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR --fastq=: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_FASTQ=lexer->YYText();
      if(!Pv.empty()){
	for(uint32 i=0; i<Pv.size(); i++){
	  Pv[i].mp_assembly_params.as_infile_FASTQ=lexer->YYText();
	}
      }
      // FALL THROUGH!
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_fastq: {
      static const char modestring[] = 
	"SANGER_SETTINGS -LR:ft=fastq"
	"\n454_SETTINGS -LR:ft=fastq"
	"\nIONTOR_SETTINGS -LR:ft=fastq"
	"\nPACBIO_SETTINGS -LR:ft=fastq"
	"\nSOLEXA_SETTINGS -LR:ft=fastq"
	"\nSOLID_SETTINGS -LR:ft=fastq";
      parseQuickmodeNoTechSettingsChange(modestring, "fastq", Pv,false);
      break;
    }
    case MP_quickmode_phdname: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR --phd=: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_infile_PHD=lexer->YYText();
      // FALL THROUGH!
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_phd: {
      static const char modestring[] = 
	"SANGER_SETTINGS -LR:ft=phd"
	"\n454_SETTINGS -LR:ft=phd"
	"\nIONTOR_SETTINGS -LR:ft=phd"
	"\nPACBIO_SETTINGS -LR:ft=phd"
	"\nSOLEXA_SETTINGS -LR:ft=phd"
	"\nSOLID_SETTINGS -LR:ft=phd";
      parseQuickmodeNoTechSettingsChange(modestring, "phd", Pv);
      break;
    }
    case MP_quickmode_cafname: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR --caf=: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      actpar->mp_assembly_params.as_infile_CAF=lexer->YYText();
      // FALL THROUGH!
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_caf: {
      static const char modestring[] = 
	"SANGER_SETTINGS -LR:ft=caf"
	"\n454_SETTINGS -LR:ft=caf"
	"\nIONTOR_SETTINGS -LR:ft=caf"
	"\nPACBIO_SETTINGS -LR:ft=caf"
	"\nSOLEXA_SETTINGS -LR:ft=caf"
	"\nSOLID_SETTINGS -LR:ft=caf";
      parseQuickmodeNoTechSettingsChange(modestring, "caf", Pv);
      break;
    }
    case MP_quickmode_loadparam: {
      if(lexer->YYText()==NULL){
	errstream << "ERROR --params=: filename not found?\n";
	MP_errorinparams=true;
	break;
      }
      loadParams(lexer->YYText(), Pv);
      break;
    }
    case MP_quickmode_clippinglight: {
      static const char modestring[] =	
	"\nSANGER_SETTINGS"
	"\n\t-CL:pvlc=yes:pvcmla=18:qc=yes:qcmq=20:qcwl=25"
	"\n\t    mbc=yes:mbcgs=20:mbcmfg=40:mbcmeg=60"
	"\n\t    emlc=yes:mlcr=15:smlc=20";

      parseQuickmode(modestring, "-clippinglight", Pv);
      break;
    }
    case MP_quickmode_clippingmedium: {
      static const char modestring[] =	
	"\nSANGER_SETTINGS"
	"\n\t-CL:pvlc=yes:pvcmla=18:qc=yes:qcmq=20:qcwl=30"
	"\n\t    mbc=yes:mbcgs=20:mbcmfg=40:mbcmeg=60"
	"\n\t    emlc=yes:mlcr=25:smlc=30";

      parseQuickmode(modestring, "-clippingmedium", Pv);
      break;
    }
    case MP_quickmode_clippingheavy: {
      static const char modestring[] =	
	"\nSANGER_SETTINGS"
	"\n\t-CL:pvlc=yes:pvcmla=25:qc=yes:qcmq=25:qcwl=30"
	"\n\t    mbc=yes:mbcgs=30:mbcmfg=50:mbcmeg=80"
	"\n\t    emlc=yes:mlcr=35:smlc=40";

      parseQuickmodeNoTechSettingsChange(modestring, "-clippingheavy", Pv);
      break;
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_noquality_all: {
      string modestring =
	"\nSANGER_SETTINGS\n\t    " + noquality_string
	+ "\n454_SETTINGS\n\t    " + noquality_string
	+ "\nIONTOR_SETTINGS\n\t    " + noquality_string
	+ "\nPACBIO_SETTINGS\n\t    " + noquality_string
	+ "\nSOLEXA_SETTINGS\n\t    " + noquality_string
	+ "\nSOLID_SETTINGS\n\t    " + noquality_string;

      parseQuickmodeNoTechSettingsChange(modestring.c_str(), "-noquality", Pv);
      cout << "Adjusting -CO:mrpg to a minimum of 3\n";
      if(!Pv.empty()) {
	for(uint32 i=0; i<Pv.size(); i++){
	  if(Pv[i].mp_contig_params.con_minreadspergroup<3) {
	    Pv[i].mp_contig_params.con_minreadspergroup=3;
	  }
	}
      }else{
	if(actpar->mp_contig_params.con_minreadspergroup<3) actpar->mp_contig_params.con_minreadspergroup=3;
      }
      break;


      break;
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_noclipping_all: {
      string modestring =
	"\nSANGER_SETTINGS\n\t    " + noclipping_string
	+ "\n454_SETTINGS\n\t    " + noclipping_string
	+ "\nIONTOR_SETTINGS\n\t    " + noclipping_string
	+ "\nPACBIO_SETTINGS\n\t    " + noclipping_string
	+ "\nSOLEXA_SETTINGS\n\t    " + noclipping_string
	+ "\nSOLID_SETTINGS\n\t    " + noclipping_string
	+ "\nCOMMON_SETTINGS -CL:ascdc=no:asjdc=no:cbse=no ";

      parseQuickmodeNoTechSettingsChange(modestring.c_str(), "-noclipping", Pv);
      break;
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_noclipping_sanger: {
      string modestring =
	"\nSANGER_SETTINGS\n\t    " + noclipping_string
	+ "\nCOMMON_SETTINGS -CL:ascdc=no:asjdc=no ";

      parseQuickmodeNoTechSettingsChange(modestring.c_str(), "-noclipping", Pv);
      break;
    }
    case MP_quickmode_noclipping_454: {
      string modestring =
	"\n454_SETTINGS\n\t    " + noclipping_string
	+ "\nCOMMON_SETTINGS -CL:ascdc=no:asjdc=no ";

      parseQuickmodeNoTechSettingsChange(modestring.c_str(), "-noclipping", Pv);
      break;
    }
    case MP_quickmode_noclipping_iontor: {
      string modestring =
	"\nIONTOR_SETTINGS\n\t    " + noclipping_string
	+ "\nCOMMON_SETTINGS -CL:ascdc=no:asjdc=no ";

      parseQuickmodeNoTechSettingsChange(modestring.c_str(), "-noclipping", Pv);
      break;
    }
    case MP_quickmode_noclipping_pacbio: {
      string modestring =
	"\nPACBIO_SETTINGS\n\t    " + noclipping_string
	+ "\nCOMMON_SETTINGS -CL:ascdc=no:asjdc=no ";

      parseQuickmodeNoTechSettingsChange(modestring.c_str(), "-noclipping", Pv);
      break;
    }
    case MP_quickmode_noclipping_solexa: {
      string modestring =
	"\nSOLEXA_SETTINGS\n\t    " + noclipping_string
	+ "\nCOMMON_SETTINGS -CL:ascdc=no:asjdc=no ";

      parseQuickmodeNoTechSettingsChange(modestring.c_str(), "-noclipping", Pv);
      break;
    }
    case MP_quickmode_noclipping_solid: {
      string modestring =
	"\nSOLID_SETTINGS\n\t    " + noclipping_string
	+ "\nCOMMON_SETTINGS -CL:ascdc=no:asjdc=no ";

      parseQuickmodeNoTechSettingsChange(modestring.c_str(), "-noclipping", Pv);
      break;
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_notraceinfo: {
      static const char modestring[] =
	"\nSANGER_SETTINGS"
	"\n\t    -LR:mxti=no"
	"\n454_SETTINGS"
	"\n\t    -LR:mxti=no"
	"\nIONTOR_SETTINGS"
	"\n\t    -LR:mxti=no"
	"\nPACBIO_SETTINGS"
	"\n\t    -LR:mxti=no"
	"\nSOLEXA_SETTINGS"
	"\n\t    -LR:mxti=no"
	"\nSOLID_SETTINGS"
	"\n\t    -LR:mxti=no";

      parseQuickmodeNoTechSettingsChange(modestring, "-notraceinfo", Pv);
      break;
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_quickmode_highlyrepetitive: {
      cout << "Changing some parameters for highly repetitive data ...";
      static const char modestring[] =	
	"\n_COMMON_SETTINGS"
	"\n\t-AS:sep=yes"
	"\n\t-CO:mr=yes:mroir=false"
	"\n\t-SK:mnr=yes:nrr=10"
	"\n"
	"\n_SANGER_SETTINGS"
	"\n\t-GE:uti=yes"
	"\n\t-AS:urdcm=1.2"
	"\n\t-CL:pvlc=yes:pvcmla=10"
	"\n\t-DP:ure=yes:feip=0:leip=0"
	"\n\t-CO:emea=15:amgb=yes:amgbemc=yes:amgbnbs=yes"
	"\n"
	"\n_454_SETTINGS"
	"\n\t-DP:ure=no"
	"\n\t-AS:urdcm=1.4"
	"\n\t-CL:pvlc=yes:pvcmla=10"
	"\n\t-CO:emea=5:amgb=no"
	"\n"
	// TODO: what for IonTorrent? atm like 454
	"\n_IONTOR_SETTINGS"
	"\n\t-DP:ure=no"
	"\n\t-AS:urdcm=1.4"
	"\n\t-CL:pvlc=yes:pvcmla=10"
	"\n\t-CO:emea=5:amgb=no"
	"\n"
	// TODO: what for PacBio? atm as 454
	// new: switch off -AL:egp (for short strobing)
	//        better would be to change egp that it strikes only between two bases, 
	//        not when neighbouring N
	//        won't help:  ....atcaaatgtt***************t--------------....
	"\n_PACBIO_SETTINGS"
	"\n\t-DP:ure=no"
	"\n\t-AS:urdcm=1.4"
	"\n\t-CL:pvlc=yes:pvcmla=10"
	"\n\t-AL:egp=no"
	"\n\t-CO:emea=5:amgb=no"
	"\n"
	"\n_SOLEXA_SETTINGS"
	"\n\t-DP:ure=no"
	"\n\t-AS:urdcm=1.9"
	"\n\t-CL:pvlc=no"
	"\n\t-CO:emea=5:amgb=yes:amgbemc=yes:amgbnbs=yes"
	;
      parseQuickmode(modestring, "", Pv,false);
      cout << "  - fixed settings\n";

      MIRAParameters * tmpactpar;
      if(!Pv.empty()) {
	tmpactpar=&Pv[0];
      }else{
	tmpactpar=actpar;
      }

      if(tmpactpar->mp_assembly_params.as_numpasses<6){
	cout << "  - increassing number of passes (-AS:nop) ";
	tmpactpar->mp_assembly_params.as_numpasses++;
	if(tmpactpar->mp_assembly_params.as_numpasses<=5){
	  tmpactpar->mp_assembly_params.as_numpasses++;
	  cout << "by two.\n";
	}else{
	  cout << "by one.\n";
	}
      }
      if(tmpactpar->mp_assembly_params.as_numrmbbreakloops<3){
	cout << "  - increasing maximum of RMB break loop (-AS:rbl).\n";
	tmpactpar->mp_assembly_params.as_numrmbbreakloops++;
      }
      cout << "Done\n";
      break;
    }
    case MP_quickmode_lowqualitydata: {
      cout << "Adjusting parameters for low quality data (-lowqualitydata):\n"
	"  Increassing (-CO:mrpg) by 1.\n"
	"  Switching on -CO:amgbnbs=yes\n";
      //"  Switching on read extension (-DP:ure=yes)"

      if(!Pv.empty()) {
	for(uint32 i=0; i<Pv.size(); i++){
	  Pv[i].mp_contig_params.con_minreadspergroup+=1;
	  Pv[i].mp_contig_params.con_also_mark_gap_bases_needbothstrands=true;
	  Pv[i].mp_contig_params.con_reject_on_drop_in_relscore-=10;
	  if(Pv[i].mp_contig_params.con_reject_on_drop_in_relscore<10){
	    Pv[i].mp_contig_params.con_reject_on_drop_in_relscore=10;
	  }
	  Pv[i].mp_contig_params.con_mingroupqualforrmbtagging-=5;

	  Pv[i].mp_align_params.ads_extra_gap_penalty=true;
	  Pv[i].setAlignGapPenaltyLevel(0);
	  Pv[i].mp_align_params.ads_max_gppercent=100;
	}
      }else{
	actpar->mp_contig_params.con_minreadspergroup+=1;
	actpar->mp_contig_params.con_also_mark_gap_bases_needbothstrands=true;
	if(actpar->mp_contig_params.con_reject_on_drop_in_relscore<10){
	  actpar->mp_contig_params.con_reject_on_drop_in_relscore=10;
	}
	actpar->mp_contig_params.con_mingroupqualforrmbtagging-=5;

	actpar->mp_align_params.ads_extra_gap_penalty=true;
	actpar->setAlignGapPenaltyLevel(0);
	actpar->mp_align_params.ads_max_gppercent=100;
      }
      break;
    }
    case MP_quickmode_highqualitydata: {
      cout << "Adjusting parameters for high quality data (-highqualitydata):\n"
	"  Increassing (-CL:qcmq) by 4.\n"
	"  Increassing (-CO:mnq) by 4.\n"
	"  Increassing (-CO:mgqrt) by 2.\n";
      // why does gcc 4.3.2 warn
      //  warning: conversion to ‘base_quality_t’ from ‘int’ may alter its value [-Wconversion]
      // in the next 6 lines with += ???
      if(!Pv.empty()) {
	for(uint32 i=0; i<Pv.size(); i++){
	  Pv[i].mp_assembly_params.as_clip_quality_minqual+=4;
	  Pv[i].mp_contig_params.con_minrmbneighbourqual+=4;
	  Pv[i].mp_contig_params.con_mingroupqualforrmbtagging+=2;
	}
      }else{
	actpar->mp_assembly_params.as_clip_quality_minqual+=4;
	actpar->mp_contig_params.con_minrmbneighbourqual+=4;
	actpar->mp_contig_params.con_mingroupqualforrmbtagging+=2;
      }
      break;
    }
    case MP_as_nodateoutput : {
      checkCOMMON(currentseqtypesettings, lexer, errstream);
      actpar->mp_assembly_params.as_dateoutput=getFixedStringMode(lexer,errstream);
      break;
    }

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_silentparams_for_common :
    case MP_silentparams_for_sanger :
    case MP_silentparams_for_454 : 
    case MP_silentparams_for_iontor : 
    case MP_silentparams_for_pacbio : 
    case MP_silentparams_for_solexa : 
    case MP_silentparams_for_solid : 
    case MP_params_for_common :
    case MP_params_for_sanger :
    case MP_params_for_454 : 
    case MP_params_for_iontor : 
    case MP_params_for_pacbio : 
    case MP_params_for_solexa : 
    case MP_params_for_solid : {
      BUGIFTHROW(Pv.empty(),"Current parsing mode does not allow for changing the sequencing type.\n");
      currentseqtypesettings=lexer->YYText();
      switch(yyretcode){
      case MP_silentparams_for_common :
      case MP_params_for_common :
      case MP_silentparams_for_sanger :
      case MP_params_for_sanger : {
      	actpar=&Pv[Read::SEQTYPE_SANGER];
      	break;
      }
      case MP_silentparams_for_454 : 
      case MP_params_for_454 : {
      	BUGIFTHROW(Pv.size() < 1+Read::SEQTYPE_454GS20,"Trying to switch to 454 settings, but not enough parameter objects exist?");
      	actpar=&Pv[Read::SEQTYPE_454GS20];
      	break;
      }
      case MP_silentparams_for_iontor : 
      case MP_params_for_iontor : {
      	BUGIFTHROW(Pv.size() < 1+Read::SEQTYPE_IONTORRENT,"Trying to switch to IonTorrent settings, but not enough parameter objects exist?");
      	actpar=&Pv[Read::SEQTYPE_IONTORRENT];
      	break;
      }
      case MP_silentparams_for_pacbio : 
      case MP_params_for_pacbio : {
      	BUGIFTHROW(Pv.size() < 1+Read::SEQTYPE_PACBIO,"Trying to switch to PacBio settings, but not enough parameter objects exist?");
      	actpar=&Pv[Read::SEQTYPE_PACBIO];
      	break;
      }
      case MP_silentparams_for_solexa :
      case MP_params_for_solexa : {
      	BUGIFTHROW(Pv.size() < 1+Read::SEQTYPE_SOLEXA,"Trying to switch to SOLEXA settings, but not enough parameter objects exist?");
      	actpar=&Pv[Read::SEQTYPE_SOLEXA];
      	break;
      }
      case MP_silentparams_for_solid :
      case MP_params_for_solid : {
      	BUGIFTHROW(Pv.size() < 1+Read::SEQTYPE_ABISOLID,"Trying to switch to SOLiD settings, but not enough parameter objects exist?");
      	actpar=&Pv[Read::SEQTYPE_ABISOLID];
      	break;
      }
      default: {
	// do nothing
      }
      }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
      switch(yyretcode){
      case MP_params_for_sanger :
      case MP_params_for_454 :
      case MP_params_for_iontor :
      case MP_params_for_pacbio :
      case MP_params_for_solexa :
      case MP_params_for_solid : {
	actpar->MP_parsedsomesettings=true;
	break;
      }
      default : {
	// do nothing
      }
      }
      break;
    }
    case MP_jobdefstart : {
      MP_currentparametersection=lexer->YYText();
      // do nothing else?
      break;
    }
    case MP_jobdef_draft:
    case MP_jobdef_normal:
    case MP_jobdef_accurate: {
      jobdefs[JA_QUALITY].push_back(yyretcode);
      if(yyretcode==MP_jobdef_normal){
	cout << "--job=normal is deprecated and will be removed in later versions of MIRA"
	  "\nPlease use only 'draft' or 'accurate'\n";
      }
      break;
    }
    case MP_jobdef_genome:
    case MP_jobdef_est: 
    case MP_jobdef_estsnppipeline1: 
    case MP_jobdef_estsnppipeline2: 
    case MP_jobdef_estsnppipeline3: {
      jobdefs[JA_TYPE].push_back(yyretcode);
      break;
    }
    case MP_jobdef_denovo:
    case MP_jobdef_mapping: {
      jobdefs[JA_METHOD].push_back(yyretcode);
      break;
    }
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
    case MP_jobdef_sanger:
    case MP_jobdef_454:
    case MP_jobdef_iontor:
    case MP_jobdef_pacbio:
    case MP_jobdef_solexa:
    case MP_jobdef_solid: {
      jobdefs[JA_TECH].push_back(yyretcode);
      break;
    }
    case MP_jobdefend : {
      interpretJobDefs(Pv, jobdefs, errstream);

      // clean the jobdefs array
      {
	size_t tmp=jobdefs.size();
	jobdefs.clear();
	jobdefs.resize(tmp);
      }
      MP_currentparametersection="(none)";
      break;
    }
    case MP_ERROR : {
      errstream << "* Parameter section: '" << MP_currentparametersection 
		<< "'\n*\tunrecognised string or unexpected character: " << lexer->YYText();
      if(MP_errorinparams){
	errstream << "\n*\t(may be due to previous errors)";
      }
      errstream <<"\n\n";
      MP_errorinparams=true;
      break;
    }
    case MP_ERROR_DASHES : {
      errstream << "* Parameter section: '" << MP_currentparametersection << "'\n";
      errstream <<  "*\tone or several dashes with a blank too much behind them: " << lexer->YYText();
      if(MP_errorinparams){
	errstream << "\n*\t(may be due to previous errors)";
      }
      errstream <<"\n\n";;
      MP_errorinparams=true;
      break;
    }
    case MP_UNRECOGNISED_STRING : {
      errstream << "* Parameter section: '" << MP_currentparametersection << "'\n";
      errstream << "*\texpected something else than: " << lexer->YYText();
      if(MP_errorinparams){
	errstream << "\n*\t(may be due to previous errors)";
      }
      errstream <<"\n\n";;
      MP_errorinparams=true;
      break;
    }
    case MP_FLOAT : 
    case MP_INT : 
    case MP_ANID : {
      errstream << "* Parameter section: '" << MP_currentparametersection << "'\n";
      errstream << "*\tunrecognised string or unexpected character: " << lexer->YYText();
      if(MP_errorinparams){
	errstream << "\n*\t(may be due to previous errors)";
      }
      errstream <<"\n\n";;
      MP_errorinparams=true;
      break;
    }
    default:{
      errstream << "* Oooooops, internal error!\n";
      errstream << "* Parameter section: '" << MP_currentparametersection << "'\n";
      errstream <<  "*\tparsed valid parameter '" << lexer->YYText() << "' which is not handled by Parameter object\n";
      errstream << "*\tyyretcode: " << yyretcode << "\n\n";
      MP_errorinparams=true;
    }
    }
  }

  delete lexer;

  if(errstream.str().size()){
    vector<int> bla;
    for(uint32 i=0; i<Pv.size(); i++) bla.push_back(i);
    
    cout << "\n\n";

    //MIRAParameters::dumpAllParams(
    //  Pv,
    //  bla,
    //  cerr);

    cout << "========================= Parameter parsing error(s) ==========================\n\n";
    cout << errstream.str();
    cout << "===============================================================================\n";
    MIRANOTIFY(Notify::FATAL, "Error while parsing parameters, sorry.");
  }

  cout << endl;
  Pv[0].consistencyCheck();

  for(uint32 st=1; st<Pv.size(); ++st){
    if(Pv[st].MP_parsedsomesettings
       && Pv[st].mp_skim_params.sk_basesperhash>Pv[st].mp_align_params.al_min_overlap){
      cout << "WARNING: -SK:bph=" << Pv[st].mp_skim_params.sk_basesperhash 
	   << " is larger than -AL:mo=" << Pv[st].mp_align_params.al_min_overlap
	   << " for " << Read::getShortNameOfSequencingType(st) << ". Some overlaps will not be found.\n";
    }
  }

  FUNCEND();
}




 
void MIRAParameters::interpretJobDefs(vector<MIRAParameters> & Pv, vector<vector<uint32> > & jobdefs, stringstream & errstream)
{
  FUNCSTART("void MIRAParameters::interpretJobDefs(vector<vector<uint16> > & jobdefs, stringstream & errstream)");

  if(jobdefs[JA_QUALITY].size()==0){
    cout << "Seen no assembly quality in job definition, assuming 'accurate'.\n";
    jobdefs[JA_QUALITY].push_back(MP_jobdef_accurate);
  }else if(jobdefs[JA_QUALITY].size()>1){
    errstream << "* Parameter section: '" << MP_currentparametersection << "'\n";
    errstream <<  "*\tSeen multiple assembly qualities in job definition, pick only one\n";
    MP_errorinparams=true;
  }
  if(jobdefs[JA_TYPE].size()==0){
    cout << "Seen no assembly type in job definition, assuming 'genome'.\n";
    jobdefs[JA_TYPE].push_back(MP_jobdef_genome);
  }else if(jobdefs[JA_TYPE].size()>1){
    errstream << "* Parameter section: '" << MP_currentparametersection << "'\n";
    errstream <<  "*\tSeen multiple assembly types in job definition, pick only one\n";
    MP_errorinparams=true;
  }
  if(jobdefs[JA_METHOD].size()==0){
    cout << "Seen no assembly method in job definition, assuming 'denovo'.\n";
    jobdefs[JA_METHOD].push_back(MP_jobdef_denovo);
  }else if(jobdefs[JA_METHOD].size()>1){
    errstream << "* Parameter section: '" << MP_currentparametersection << "'\n";
    errstream <<  "*\tSeen multiple assembly methods in job definition, pick only one\n";
    MP_errorinparams=true;
  }
  
#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif
  bool hasSANGER=false;
  bool has454=false;
  bool hasIONTOR=false;
  bool hasPACBIO=false;
  bool hasSOLEXA=false;
  bool hasSOLID=false;

  {
    string multiple;
    for(uint32 i=0; i<jobdefs[JA_TECH].size(); i++){
      multiple.clear();
      switch(jobdefs[JA_TECH][i]){
      case MP_jobdef_sanger : {
	if(hasSANGER) {
	  multiple=Read::getNameOfSequencingType(Read::SEQTYPE_SANGER);
	}
	hasSANGER=true;
	break;
      }
      case MP_jobdef_454 : {
	if(has454) {
	  multiple=Read::getNameOfSequencingType(Read::SEQTYPE_454GS20);
	}
	has454=true;
	break;
      }
      case MP_jobdef_iontor : {
	if(hasIONTOR) {
	  multiple=Read::getNameOfSequencingType(Read::SEQTYPE_IONTORRENT);
	}
	hasIONTOR=true;
	break;
      }
      case MP_jobdef_pacbio : {
	if(hasPACBIO) {
	  multiple=Read::getNameOfSequencingType(Read::SEQTYPE_PACBIO);
	}
	hasPACBIO=true;
	break;
      }
      case MP_jobdef_solexa : {
	if(hasSOLEXA) {
	  multiple=Read::getNameOfSequencingType(Read::SEQTYPE_SOLEXA);
	}
	hasSOLEXA=true;
	break;
      }
      case MP_jobdef_solid : {
	if(hasSOLID) {
	  multiple=Read::getNameOfSequencingType(Read::SEQTYPE_ABISOLID);
	}
	hasSOLID=true;
	break;
      }
      default: {
	// do nothing
      }
      }
      if(!multiple.empty()){
	errstream << "* Parameter section: '" << MP_currentparametersection << "'\n";
	errstream <<  "*\tSeen '" << multiple << "' more than once, did you mean a different sequencing\n\ttechnology?\n";
	MP_errorinparams=true;
      }
    }
  }

  bool hasSHORTREADS=hasSOLEXA | hasSOLID;

#if CPP_READ_SEQTYPE_END != 6
#error "This code is made for 6 sequencing types, adapt!"
#endif

  CEBUG("hasSANGER: " << hasSANGER << endl);
  CEBUG("has454: " << has454 << endl);
  CEBUG("hasIONTOR: " << hasIONTOR << endl);
  CEBUG("hasPACBIO: " << hasPACBIO << endl);
  CEBUG("hasSOLEXA: " << hasSOLEXA << endl);
  CEBUG("hasSOLID: " << hasSOLID << endl);
  CEBUG("hasSHORTREADS: " << hasSHORTREADS << endl);

  // complete: 1.6 2.1

  ///////////////////////////////////////////////////////////////////////////////////
  ////  The following qual settings are for:  genome denovo

  string modestring=
    "\nCOMMON_SETTINGS"
    "\n\t-GE:not=2:crhf=yes:amm=yes:kpmf=15:mps=0"
    "\n\t-MI:sonfs=yes:el=false:lcs=500:lcs4s=5000:somrnl=40"
    "\n\t-LR:fo=no"
    "\n\t-AS:ugpf=yes:urd=no:urdsip=3:klrs=no:mcpp=0"
    "\n\t-SB:bsnffa=no:bbq=30:bsn=ReferenceStrain"
    "\n\t-CL:pec=yes:pecbph=17:pechsgp=yes:ascdc=yes:asjdc=no:cbse=yes"
    "\n\t-CO:mr=yes:mroir=false:asir=no"
    "\n\t    emeas1clpec=yes:fnicast=no"
    "\n\t-SK:acrc=yes:bph=17:pr=80:swcob=no:mnr=yes:nrr=100:mmhr=0:mchr=2048"
    "\n\t    fenn=0.4:fexn=1.6:fer=1.9:fehr=8:fecr=20"
    "\n\t-PF:mscft=5"
    "\n\t"
    "\n\t-OUT:ors=yes:otc=yes:orm=yes:otm=no:rrot=yes:rtd=no"
    "\n\t"
    "\nSANGER_SETTINGS"
    "\n\t-GE:uti=yes"
    "\n\t-LR:lsd=no"
    "\n\t-AS:mrl=80:mrpc=2"
    "\n\t    urdcm=1.5:ardct=2.0:ardml=400:ardgl=40"
    "\n\t-SB:ads=no:dsn=StrainX"
    "\n\t-DP:ure=yes:rewl=30:rewme=2:feip=0;leip=0"
    "\n\t-CL:bsqc=yes:bsqcmq=20:bsqcwl=30:mbc=yes:mbcgs=20:mbcmfg=40:mbcmeg=60:mqtfer=0:mqtfernob=0"
    "\n\t    emlc=yes:mlcr=25:smlc=30:qc=no:cpat=no:c3pp=no:lcc=no:ckar=no"
    "\n\t-CO:mrpg=2:emea=25"
    "\n\t    amgb=yes:amgbemc=yes:amgbnbs=yes"
    "\n\t    fnicpst=no"
    "\n\t-ED:ace=no"
    "\n\t-AL:bip=15:bmin=25:bmax=70:mo=17:ms=30:mrs=65:egp=no"
    "\n\t-PF:uqr=yes:qrml1=200:qrms1=90:qrml2:100:qrms2=95:bqoml=150"
    ;

  if(hasSANGER){
    modestring+=
      "\n\t"
      "\nSANGER_SETTINGS"
      "\n\t-LR:lsd=yes"
      ;
  }

  if(has454){
    modestring+=
      "\n\t"
      "\nCOMMON_SETTINGS"
      "\n\t-CL:pec=yes:pecbph=27"
      "\n\t-OUT:org=no"
      "\n\t-SK:fenn=0.4:fexn=1.6:fer=1.9:fehr=8:fecr=20"
      "\n\t"
      "\n454_SETTINGS"
      "\n\t-AL:egp=yes:egpl=reject_codongaps"
      "\n\t-LR:lsd=yes:rns=fr:mxti=yes:ft=fastq"
      "\n\t-GE:uti=yes"
      "\n\t-AS:mrl=40:mrpc=5"
      "\n\t    urdcm=1.5:ardct=2.0:ardml=200:ardgl=20"
      "\n\t-SB:ads=no:dsn=StrainX"
      "\n\t-DP:ure=no:rewl=15:rewme=2:feip=0;leip=0"
      "\n\t-CL:pvlc=no:qc=no:bsqc=no:emlc=no:mlcr=4:smlc=4:emrc=no:mrcr=10:smrc=15:mbc=yes:mbcgs=5:mbcmfg=12:mbcmeg=12:mqtfer=0:mqtfernob=0"
      "\n\t    msvs=no:msvsgs=8:msvsmfg=8:msvsmeg=12:msvssfc=0:msvssec=0:c3pp=no:cpat=no:lcc=yes:ckar=yes"
      "\n\t-AL:ms=15:mo=20:mrs=70:bip=20:bmin=20:bmax=80:egp=yes:egpl=reject_codongaps"
      "\n\t-CO:rodirs=30:mrpg=4:mnq=20:mgqrt=25:emea=10:amgb=no"
      "\n\t    fnicpst=no:gor=66"
      "\n\t-ED:ace=yes"
      "\n\t-PF:uqr=yes:qrml1=80:qrms1=90:qrml2=60:qrms2=95:bqoml=80"
      "\n\t-SK:pr=80"
      ;
  }

  if(hasIONTOR){
    modestring+=
      "\n\t"
      "\nCOMMON_SETTINGS"
      "\n\t-CL:pec=yes:pecbph=27"
      "\n\t-OUT:org=no"
      "\n\t-SK:fenn=0.4:fexn=1.6:fer=1.9:fehr=8:fecr=20"
      "\n\t"
      "\nIONTOR_SETTINGS"
      "\n\t-AL:egp=yes:egpl=reject_codongaps"
      "\n\t-LR:lsd=yes:ft=fastq:rns=fr:mxti=yes"
      "\n\t-GE:uti=yes"
      "\n\t-AS:mrl=40:mrpc=5"
      "\n\t    urdcm=1.5:ardct=2.0:ardml=200:ardgl=20"
      "\n\t-SB:ads=no:dsn=StrainX"
      "\n\t-DP:ure=no:rewl=15:rewme=2:feip=0;leip=0"
      "\n\t-CL:pvlc=no:qc=no:bsqc=no:emlc=no:mlcr=4:smlc=4:emrc=no:mrcr=10:smrc=15:mbc=yes:mbcgs=5:mbcmfg=12:mbcmeg=12:mqtfer=0:mqtfernob=0"
      "\n\t    msvs=no:msvsgs=8:msvsmfg=8:msvsmeg=12:msvssfc=0:msvssec=0:c3pp=no:cpat=no:lcc=yes:ckar=yes"
      "\n\t-AL:ms=15:mo=20:mrs=70:bip=20:bmin=20:bmax=80:egp=yes:egpl=reject_codongaps"
      "\n\t-CO:rodirs=25:mrpg=4:mnq=20:mgqrt=25:emea=10:amgb=no"
      "\n\t    fnicpst=no:gor=66"
      "\n\t-ED:ace=yes"
      "\n\t-PF:uqr=yes:qrml1=80:qrms1=90:qrml2=60:qrms2=95:bqoml=80"
      "\n\t-SK:pr=80"
      ;
  }

  // TODO: adapt once PacBio data analysed
  // template from 454
  // mrpg=3 for less coverage expected.
  if(hasPACBIO){
    modestring+=
      "\n\t"
      "\nCOMMON_SETTINGS"
      "\n\t-CL:pec=yes:pecbph=27"
      "\n\t-OUT:org=no"
      "\n\t-SK:fenn=0.4:fexn=1.6:fer=1.9:fehr=8:fecr=20"
      "\n\t"
      "\nPACBIO_SETTINGS"
      "\n\t-AL:egp=yes:egpl=reject_codongaps"
      "\n\t-LR:lsd=yes:rns=fr:mxti=no:ft=fastq"
      "\n\t-GE:uti=yes"
      "\n\t-AS:mrl=40:mrpc=5"
      "\n\t    urdcm=1.5:ardct=2.0:ardml=200:ardgl=20"
      "\n\t-SB:ads=no:dsn=StrainX"
      "\n\t-DP:ure=no:rewl=15:rewme=2:feip=0;leip=0"
      "\n\t-CL:pvlc=no:qc=no:bsqc=no:emlc=no:mlcr=4:smlc=4:emrc=no:mrcr=10:smrc=15:mbc=yes:mbcgs=5:mbcmfg=12:mbcmeg=12"
      "\n\t    msvs=no:msvsgs=8:msvsmfg=8:msvsmeg=12:msvssfc=0:msvssec=0:c3pp=no:cpat=no:lcc=yes:ckar=no"
      "\n\t-AL:ms=15:mo=20:mrs=70:bip=20:bmin=20:bmax=200:egp=yes:egpl=reject_codongaps"
      "\n\t-CO:rodirs=30:mrpg=3:mnq=20:mgqrt=25:emea=10:amgb=no"
      "\n\t    fnicpst=no:gor=66"
      "\n\t-ED:ace=yes"
      "\n\t-PF:uqr=yes:qrml1=80:qrms1=90:qrml2=60:qrms2=95:bqoml=80"
      "\n\t-SK:pr=80"
      ;
  }

  if(hasSOLEXA){
    modestring+=
      "\n\t"
      "\nCOMMON_SETTINGS"
      "\n\t-AS:nop=1:sd=no:rbl=1:urd=no:ard=yes"
      "\n\t-OUT:otc=yes:ort=no:org=no"
      "\n\t-SK:bph=16:hss=1:mchr=4096"
      "\n\t-CO:mr=yes"
      "\n\t-PF:swcs=12"
      "\n\t-CL:pecbph=31"
      "\n\t-SK:fenn=0.4:fexn=1.6:fer=1.9:fehr=8:fecr=20"
      "\n\t"
      "\nSOLEXA_SETTINGS"
      "\n\t-OUT:sssip=no:stsip=no"
      "\n\t-LR:lsd=yes:rns=solexa:ft=fastq"
      "\n\t-AS:mrl=20:mrpc=10"
      "\n\t    urdcm=1.5:ardct=2.5:ardml=300:ardgl=20"
      "\n\t-SB:ads=no:dsn=StrainX"
      "\n\t-GE:uti=yes"
      "\n\t-DP:ure=no"
      "\n\t-CL:pvlc=no:qc=no:bsqc=no:emlc=no:mlcr=0:smlc=0:mbc=no:mbcgs=5:mbcmfg=12:mbcmeg=12:mqtfer=5:mqtfernob=15"
      "\n\t    msvsgs=1:msvsmfg=2:msvsmeg=2:msvssfc=0:msvssec=0:cpat=no:lcc=no:ckar=yes"
      "\n\t    c3pp=yes:c3ppmsl=12:c3ppmea=2:c3ppmgfe=9"
      "\n\t-AL:ms=15:mo=25:mrs=90:bip=20:bmin=20:bmax=80:egp=no"
      "\n\t-CO:rodirs=30:mrpg=4:mnq=20:mgqrt=30:emea=4:amgb=yes"
      "\n\t    fnicpst=no"
      "\n\t    msr=yes:msrkceu=-1"
      "\n\t-ED:ace=no"
      "\n\t-PF:uqr=yes:qrml1=-95:qrms1=100:qrml2=-85:qrms2=100:bqoml=20"
      "\n\t-SK:pr=90"
      ;
    if(has454){
      modestring+=
	"\n\t"
	"\nCOMMON_SETTINGS"
	"\n\t-CL:pec=yes:pecbph=27"
	;
    }
    if(hasIONTOR){
      modestring+=
	"\n\t"
	"\nCOMMON_SETTINGS"
	"\n\t-CL:pec=yes:pecbph=27"
	;
    }
    if(hasPACBIO){
      modestring+=
	"\n\t"
	"\nCOMMON_SETTINGS"
	"\n\t-CL:pec=yes:pecbph=27"
	;
    }
  }

  if(hasSOLID){
    modestring+=
      "\n\t"
      "\nCOMMON_SETTINGS"
      "\n\t"
      "\nSOLID_SETTINGS"
      "\n\t-LR:lsd=yes:rns=fr"
      "\n\t-DP:ure=no"
      "\n\t-AS:mrl=30"
      "\n\t    urdcm=1.5:ardct=2.0:ardml=200:ardgl=20"
      ;
  }


  if(jobdefs[JA_QUALITY].front()==MP_jobdef_draft){
    CEBUG("MP_jobdef_draft\n");
    static const char g_draftonly[] =	
      "\nCOMMON_SETTINGS"
      "\n\t-AS:nop=2:sep=no:rbl=1:sd=no:ard=yes:urd=no"
      "\n\t-SK:bph=17:hss=8:pr=70:mhpr=200"
      "\n\t"
      "\n_SANGER_SETTINGS"
      "\n\t-DP:ure=no"
      "\n\t-CL:pvlc=no"
      "\n\t-AL:mo=17:ms=30:mrs=65"
      "\n\t-CO:rodirs=15"
      ;
    modestring+=g_draftonly;

    if(has454) {
      modestring+=
	"\nCOMMON_SETTINGS"
	"\n\t-AS:nop=3:rbl=2:sep=yes"
	"\n"
	"\n_454_SETTINGS"
	"\n\t-ED:ace=yes"
	"\n\t-AL:mrs=70"
	"\n\t-DP:ure=no"
	"\n\t-CL:emrc=no"
	"\n\t-SK:pr=80"
	;
    }

    if(hasIONTOR) {
      modestring+=
	"\nCOMMON_SETTINGS"
	"\n\t-AS:nop=3:rbl=2:sep=yes"
	"\n\t-SK:bph=21"
	"\n"
	"\n_IONTOR_SETTINGS"
	"\n\t-ED:ace=yes"
	"\n\t-AL:mrs=70:mo=21"
	"\n\t-DP:ure=no"
	"\n\t-CL:emrc=no"
	"\n\t-SK:pr=70"
	;
    }

  // TODO: adapt once PacBio data analysed
  // template from 454
    if(hasPACBIO) {
      modestring+=
	"\nCOMMON_SETTINGS"
	"\n\t-AS:nop=3:rbl=2:sep=yes"
	"\n"
	"\n_PACBIO_SETTINGS"
	"\n\t-ED:ace=yes"
	"\n\t-AL:mrs=70"
	"\n\t-DP:ure=no"
	"\n\t-CL:emrc=no"
	"\n\t-SK:pr=90"
	;
    }

    if(hasSOLEXA) {
      if(hasSANGER || has454 || hasPACBIO || hasIONTOR) {
	modestring+=
	  "\nCOMMON_SETTINGS"
	  "\n\t-SK:pr=85:hss=4"
	  "\n_454_SETTINGS"
	  "\n\t-SK:pr=85"
	  "\n_IONTOR_SETTINGS"
	  "\n\t-SK:pr=85"
	  "\n_PACBIO_SETTINGS"
	  "\n\t-SK:pr=85"
	  ;
      }else{
	modestring+=
	  "\nCOMMON_SETTINGS"
	  "\n\t-AS:nop=1:rbl=1:sep=yes"
	  "\n\t-SK:hss=4"
	  "\n_SOLEXA_SETTINGS"
	  "\n\t-SK:pr=90"
	  ;
      }
    }

  }else{
    CEBUG("not MP_jobdef_draft\n");
    static const char g_level23[]=
      "\nCOMMON_SETTINGS"
      "\n\t-AS:sep=yes:sd=yes:sdlpo=yes"
      "\n\t"
      "\n_SANGER_SETTINGS"
      "\n\t-DP:ure=yes:feip=0;leip=0"
      "\n\t-CL:pvlc=yes:pvcmla=18"
      ;    
    modestring+=g_level23;

    if(jobdefs[JA_QUALITY].front()==MP_jobdef_normal){
      CEBUG("MP_jobdef_normal\n");
      static const char g_normalonly[] =	
	"\nCOMMON_SETTINGS"
	"\n\t-AS:nop=3:rbl=2:urdsip=3"
	"\n\t-SK:bph=17:hss=4:pr=70:mhpr=2000"
	"\n\t"
	"\n_SANGER_SETTINGS"
	"\n\t-AL:bip=15:bmin=25:bmax=100"
	"\n\t-CO:rodirs=20"
	;
      modestring+=g_normalonly;

      if(has454) {
	modestring+=
	  "\nCOMMON_SETTINGS"
	  "\n\t-AS:nop=4:rbl=2:sep=yes:urdsip=3"
	  "\n"
	  "\n_SANGER_SETTINGS"
	  "\n\t-AL:mrs=70"
	  "\n"
	  "\n_454_SETTINGS"
	  "\n\t-ED:ace=yes"
	  "\n\t-AL:mrs=70"
	  "\n\t-SK:pr=80"
	  ;
      }

      if(hasIONTOR) {
	modestring+=
	  "\nCOMMON_SETTINGS"
	  "\n\t-AS:nop=4:rbl=2:sep=yes:urdsip=3"
	  "\n\t-SK:bph=19"
	  "\n"
	  "\n_SANGER_SETTINGS"
	  "\n\t-AL:mrs=70"
	  "\n"
	  "\n_IONTOR_SETTINGS"
	  "\n\t-ED:ace=yes"
	  "\n\t-AL:mrs=70:mo=19"
	  "\n\t-SK:pr=50"
	  ;
      }


      // TODO: adapt once PacBio data analysed
      // template from 454
      if(hasPACBIO) {
	modestring+=
	  "\nCOMMON_SETTINGS"
	  "\n\t-AS:nop=4:rbl=2:sep=yes:urdsip=3"
	  "\n"
	  "\n_SANGER_SETTINGS"
	  "\n\t-AL:mrs=70"
	  "\n"
	  "\n_PACBIO_SETTINGS"
	  "\n\t-ED:ace=yes"
	  "\n\t-AL:mrs=70"
	  "\n\t-SK:pr=80"
	  ;
      }

      if(hasSOLEXA) {
	if(hasSANGER || has454 || hasPACBIO) {
	  modestring+=
	    "\nCOMMON_SETTINGS"
	    "\n\t-SK:pr=85:hss=1"
	    "\n_454_SETTINGS"
	    "\n\t-SK:pr=85"
	    "\n_PACBIO_SETTINGS"
	    "\n\t-SK:pr=85"
	    ;
	}else{
	  modestring+=
	    "\nCOMMON_SETTINGS"
	    "\n\t-AS:nop=2:rbl=1:sep=yes"
	    "\n\t-SK:bph=17:hss=1:mhpr=100"
	    "\n"
	    "\n_SOLEXA_SETTINGS"
	    "\n\t-SK:pr=90"
	    ;
	}
      }

    }else{
      CEBUG("not MP_jobdef_draft ie MP_jobdef_accurate\n");
      static const char g_accurateonly[] =
	"\nCOMMON_SETTINGS"
	"\n\t-AS:nop=4:rbl=2:urdsip=3"
	"\n\t-SK:bph=21:hss=1:pr=65:mhpr=2000"
	"\n_SANGER_SETTINGS"
	"\n\t-AL:bip=20:bmin=25:bmax=130"
	"\n\t-CO:rodirs=25"
	;
      
      modestring+=g_accurateonly;

      if(has454) {
	modestring+=
	  "\nCOMMON_SETTINGS"
	  "\n\t-AS:nop=5:rbl=3:sep=yes:urdsip=4"
	  "\n"
	  "\n_SANGER_SETTINGS"
	  "\n\t-AL:mrs=70"
	  "\n\t-SK:pr=70"
	  "\n"
	  "\n_454_SETTINGS"
	  "\n\t-ED:ace=yes"
	  "\n\t-AL:mrs=70"
	  "\n\t-SK:pr=80"
	  ;
      }


      if(hasIONTOR) {
	modestring+=
	  "\nCOMMON_SETTINGS"
	  "\n\t-AS:nop=5:rbl=3:sep=yes:urdsip=4"
	  "\n\t-SK:bph=19"
	  "\n"
	  "\n_SANGER_SETTINGS"
	  "\n\t-AL:mrs=70"
	  "\n\t-SK:pr=70"
	  "\n"
	  "\n_IONTOR_SETTINGS"
	  "\n\t-ED:ace=yes"
	  "\n\t-AL:mrs=70:mo=19"
	  "\n\t-SK:pr=50"
	  ;
      }

      // TODO: adapt once PacBio data analysed
      // template from 454
      if(hasPACBIO) {
	modestring+=
	  "\nCOMMON_SETTINGS"
	  "\n\t-AS:nop=5:rbl=3:sep=yes:urdsip=4"
	  "\n"
	  "\n_SANGER_SETTINGS"
	  "\n\t-AL:mrs=70"
	  "\n\t-SK:pr=70"
	  "\n"
	  "\n_PACBIO_SETTINGS"
	  "\n\t-ED:ace=yes"
	  "\n\t-AL:mrs=70"
	  "\n\t-SK:pr=80"
	  ;
      }

      if(hasSOLEXA) {
	if(hasSANGER || has454 || hasPACBIO || hasIONTOR) {
	  modestring+=
	    "\nCOMMON_SETTINGS"
	    "\n\t-SK:pr=85:bph=19:hss=1"
	    "\n_454_SETTINGS"
	    "\n\t-SK:pr=85"
	    "\n_IONTOR_SETTINGS"
	    "\n\t-SK:pr=85"
	    "\n_PACBIO_SETTINGS"
	    "\n\t-SK:pr=85"
	    ;
	}else{
	  modestring+=
	    "\nCOMMON_SETTINGS"
	    "\n\t-AS:nop=3:rbl=1:sep=yes"
	    "\n\t-SK:bph=17:hss=1:mhpr=2000"
	    "\n_SOLEXA_SETTINGS"
	    "\n\t-SK:pr=95"
	    ;
	}
      }
    }
  }

  if(!jobdefs[JA_METHOD].empty() && jobdefs[JA_METHOD].front()==MP_jobdef_mapping){
    // modifier for mapping assembly

    CEBUG("MP_jobdef_mapping\n");

    modestring+=
      "\n\t"
      "\nCOMMON_SETTINGS"
      "\n\t-OUT:otc=yes:ort=no"
      "\n\t-AS:ard=yes:urd=no:nop=1:rbl=1"
      "\n\t-CL:ascdc=no"
      "\n\t-SB:lb=yes:bft=fasta:sbuip=1:brl=0:bro=0:abnc=no"
      "\n\t-SK:bph=16:hss=4:pr=60:mnr=no:mhpr=1000"
      "\n\t-CO:mr=yes"
      "\n\t-PF:swcs=12"
      ;

    if(jobdefs[JA_QUALITY].front()==MP_jobdef_accurate){
      modestring+=
	"\n\t"
	"\nCOMMON_SETTINGS"
	"\n\t-SK:swcob=yes";
    }

    if(hasSANGER){
      modestring+=
	"\nSANGER_SETTINGS"
	"\n\t-SB:ads=yes";
    }
    if(has454){
      modestring+=
	"\n454_SETTINGS"
	"\n\t-SB:ads=yes";
    }
    if(hasIONTOR){
      modestring+=
	"\nIONTOR_SETTINGS"
	"\n\t-SB:ads=yes";
    }
    if(hasSOLEXA){
      modestring+=
	"\n\t"
	"\nCOMMON_SETTINGS"
	"\n\t-SK:bph=17:hss=1:pr=60"
	"\n\t-CO:mr=yes"
	"\n\t-PF:swcs=12"
	"\n\t"
	"\nSOLEXA_SETTINGS"
	"\n\t-LR:rns=solexa"
	"\n\t-AS:mrl=20"
	"\n\t    urdcm=1.5:ardct=2.0:ardml=200:ardgl=20"
	"\n\t-GE:uti=no"
	"\n\t-DP:ure=no"
	"\n\t-SB:ads=yes"
	"\n\t-CL:pvlc=no:qc=no:bsqc=no:emlc=no:mlcr=0:smlc=0:mbc=no:mbcgs=5:mbcmfg=12:mbcmeg=12"
	"\n\t    msvsgs=1:msvsmfg=2:msvsmeg=2:msvssfc=0:msvssec=0:cpat=no:mqtfer=0"
	"\n\t-AL:ms=15:mo=20:mrs=80:bip=20:bmin=20:bmax=80:egp=no"
	"\n\t-CO:rodirs=30:mrpg=3:mnq=20:mgqrt=30:emea=4:amgb=yes"
	"\n\t    fnicpst=no"
	"\n\t-ED:ace=no"
	"\n\t-PF:uqr=yes:qrml1=-90:qrms1=100:qrml2=-80:qrms2=100:bqoml=20"
	"\n\t-SK:pr=60"
	;
      // moved to general mapping above.
      //if(!hasSANGER && !has454 && !hasPACBIO){
      //	modestring+=
      //	  "\nCOMMON_SETTINGS"
      //	  "\n\t-AS:nop=1"
      //	  ;
      //}
      if(hasPACBIO){
	modestring+=
	  "\nPACBIO_SETTINGS"
	  "\n\t-SB:ads=yes";
      }
    }

    modestring+=
      "\nCOMMON_SETTINGS";

    // also take care of brl by checking which sequencing types are used
    if(hasSHORTREADS){
      // TODO: check good brl length
      modestring+=
	"\nCOMMON_SETTINGS"
	"\n\t-SB:sbuip=0:brl=0:abnc=no"
	"\n\t-CO:mroir=yes";
      if(hasSOLEXA){
	if(jobdefs[JA_QUALITY].front()==MP_jobdef_draft){
	  modestring+=
	    "\nCOMMON_SETTINGS"
	    "\n\t-SB:bro=0"
	    "\n\t-SK:mhpr=1000"
	    "\n_SANGER_SETTINGS"
	    "\n\t-SK:pr=70"
	    "\n_454_SETTINGS"
	    "\n\t-SK:pr=70"
	    "\n_IONTOR_SETTINGS"
	    "\n\t-SK:pr=70"
	    "\n_PACBIO_SETTINGS"
	    "\n\t-SK:pr=70"
	    "\n_SOLEXA_SETTINGS"
	    "\n\t-AL:mrs=70"
	    "\n\t-CO:msr=yes"
	    "\n\t-SK:pr=90"
	    ;
	}else if(jobdefs[JA_QUALITY].front()==MP_jobdef_normal){
	  if(!hasSANGER && !has454 && !hasIONTOR) modestring+="\nCOMMON_SETTINGS\n-SK:bph=12:hss=1:mhpr=1500";
	  modestring+=
	    "\nCOMMON_SETTINGS"
	    "\n\t-SB:brl=0:bro=0"
	    "\n_SANGER_SETTINGS"
	    "\n\t-SK:pr=60"
	    "\n_454_SETTINGS"
	    "\n\t-SK:pr=60"
	    "\n_IONTOR_SETTINGS"
	    "\n\t-SK:pr=60"
	    "\n_PACBIO_SETTINGS"
	    "\n\t-SK:pr=60"
	    "\n_SOLEXA_SETTINGS"
	    "\n\t-AL:mrs=75"
	    "\n\t-CO:msr=yes"
	    "\n\t-SK:pr=75"
	    ;
	}else{
	  if(!hasSANGER && !has454 && !hasPACBIO) modestring+="\nCOMMON_SETTINGS\n-SK:bph=10:hss=1:mhpr=2000";
	  modestring+=
	    "\nCOMMON_SETTINGS"
	    "\n\t-SB:brl=0:bro=0"
	    "\n_SANGER_SETTINGS"
	    "\n\t-SK:pr=60"
	    "\n_454_SETTINGS"
	    "\n\t-SK:pr=60"
	    "\n_IONTOR_SETTINGS"
	    "\n\t-SK:pr=60"
	    "\n_PACBIO_SETTINGS"
	    "\n\t-SK:pr=60"
	    "\n_SOLEXA_SETTINGS"
	    "\n\t-AL:mrs=60"
	    "\n\t-CO:msr=yes"
	    "\n\t-SK:pr=60"
	    ;
	}
      }else{
	if(jobdefs[JA_QUALITY].front()==MP_jobdef_draft){
	  modestring+="\n\t-SB:bro=20";
	}else if(jobdefs[JA_QUALITY].front()==MP_jobdef_normal){
	  modestring+="\n\t-SB:bro=35";
	  if(!hasSANGER && !has454 && !hasPACBIO && !hasIONTOR) modestring+="\n\n-SK:bph=12";
	}else{
	  modestring+="\n\t-SB:bro=40";
	  if(!hasSANGER && !has454 && !hasPACBIO && !hasIONTOR) modestring+="\n\n-SK:bph=10";
	}
      }
    }else if(has454 || hasPACBIO){
      // dumb idea to set abnc=yes automatically ... users don't expect that
      //if(jobdefs[JA_QUALITY].front()==MP_jobdef_draft){
      //	modestring+="\n\t-SB:sbuip=0:abnc=no";
      //}else if(jobdefs[JA_QUALITY].front()==MP_jobdef_normal){
      //	modestring+="\n\t-SB:sbuip=0:abnc=no";
      //}else{
      //	modestring+="\n\t-SB:sbuip=1:abnc=yes";
      //}
      modestring+="\n\t-SB:brl=0:bro=0"
	"\n\t-SB:sbuip=0;abnc=no";
    }else{
      modestring+="\n\t-SB:brl=0:bro=0"
	"\n\t-SB:sbuip=0;abnc=no";
      //if(jobdefs[JA_QUALITY].front()==MP_jobdef_draft){
      //	modestring+="\n\t-SB:sbuip=0;abnc=no";
      //}else if(jobdefs[JA_QUALITY].front()==MP_jobdef_normal){
      //	modestring+="\n\t-SB:sbuip=0;abnc=no";
      //}else{
      //	modestring+="\n\t-SB:sbuip=0;abnc=no";
      //}
    }
  }

  string estbasesettings=
    "\nCOMMON_SETTINGS"
    "\n\t-GE:crhf=no"
    "\n\t-MI:lcs=500:lcs4s=1000"
    "\n\t-AS:sd=no:ard=no:urd=no:klrs=no:ugpf=no:uess=yes:esspd=500:umcbt=yes:bts=3600"
    "\n\t-CL:pec=no:pecbph=17:ascdc=no:asjdc=no"
    "\n\t-SK:mhpr=30:mnr=yes"
    "\n\t-OUT:orw=no"
    ;
  if(hasSANGER){
    estbasesettings+=
      "\nSANGER_SETTINGS"
      "\n\t-AS:mrpc=2"
      "\n\t-AL:mrs=85:egp=yes:egpl=reject_codongaps"
      "\n\t-DP:ure=no"
      "\n\t-CL:msvs=no:pvlc=no:qc=yes:bsqc=no:mbc=yes:c3pp=no:emlc=no:emrc=no:mqtfer=0:mqtfernob=0"
      "\n\t    cpat=yes:cpkps=no:cpmsl=12:cpmea=1:cpmgfe=20000"
      "\n\t-CO:rodirs=10"
      ;
  }
  if(has454) {
    estbasesettings+=
      "\n454_SETTINGS"
      "\n\t-AS:mrpc=2"
      "\n\t-AL:mrs=80:egp=yes:egpl=reject_codongaps"
      "\n\t-CL:msvs=no:pvlc=no:qc=no:bsqc=no:mbc=yes:c3pp=no:emlc=no:emrc=no:mqtfer=0:mqtfernob=0"
      "\n\t    cpat=yes:cpkps=no:cpmsl=12:cpmea=1:cpmgfe=20000"
      "\n\t-CO:rodirs=15"
      ;
  }
  if(hasIONTOR) {
    estbasesettings+=
      "\nIONTOR_SETTINGS"
      "\n\t-AS:mrpc=2"
      "\n\t-AL:mrs=80:egp=yes:egpl=reject_codongaps"
      "\n\t-CL:msvs=no:pvlc=no:qc=no:bsqc=no:mbc=yes:c3pp=no:emlc=no:emrc=no:mqtfer=0:mqtfernob=0"
      "\n\t    cpat=yes:cpkps=no:cpmsl=12:cpmea=1:cpmgfe=20000"
      "\n\t-CO:rodirs=15"
      ;
  }
  if(hasSOLEXA) {
    estbasesettings+=
      "\nCOMMON_SETTINGS"
      "\n\t-AS:nop=3:rbl=1"
      "\n\t-CL:pec=yes:pecbph=31:ascdc=yes"
      "\nSOLEXA_SETTINGS"
      "\n\t-AS:mrpc=4"
      "\n\t-AL:mrs=90:egp=yes:egpl=reject_codongaps"
      "\n\t-CL:msvs=no:pvlc=no:qc=no:bsqc=no:mbc=yes:emlc=no:emrc=no:mqtfer=5:mqtfernob=15"
      "\n\t    cpat=yes:cpkps=no:cpmsl=15:cpmea=1:cpmgfe=20000"
      "\n\t    c3pp=yes:c3ppmsl=12:c3ppmea=2:c3ppmgfe=9"
      "\n\t-CO:rodirs=15"
      ;
  }
  if(hasPACBIO) {
    estbasesettings+=
      "\nPACBIO_SETTINGS"
      "\n\t-AS:mrpc=2"
      "\n\t-AL:mrs=85:egp=yes:egpl=reject_codongaps"
      "\n\t-DP:ure=no"
      "\n\t-CL:msvs=no:pvlc=no:qc=yes:bsqc=no:mbc=yes:c3pp=no:cpat=yes:cpkps=no:cpmsl=12:cpmea=1:cpmgfe=20000:emlc=no:emrc=no:mqtfer=0:mqtfernob=0"
      "\n\t-CO:rodirs=10"
      ;
  }

  switch(jobdefs[JA_TYPE].front()){
    case MP_jobdef_genome : {
      // genome is handled in the whole part above
      // no need to set anything here
      break;
    }
    case MP_jobdef_est : {
      CEBUG("MP_jobdef_est\n");
      modestring+=estbasesettings;
      break;
    }
    case MP_jobdef_estsnppipeline1 : {
      CEBUG("MP_jobdef_estsnppipeline1\n");
      modestring+=estbasesettings;
      modestring+=
	"\nCOMMON_SETTINGS"
	"\n\t-GE:proout=step1:esps=1"
	"\n\t-SB:lsd=yes:lb=no"
	"\n\t-AS:nop=2:rbl=4"
	"\n\t-CO:mr=yes:mroir=no:asir=no"
	"\n\t-OUT:orc=yes:orf=yes:org=no:ora=no:ort=no:orh=no"
	;

      if(hasSANGER){
	modestring+=
	  "\nSANGER_SETTINGS"
	  "\n\t-AL:ms=15:mo=17:mrs=65:egpl=low"
	  "\n\t-CO:emea=15:rodirs=20:amgb=yes:amgbemc=yes:amgbnbs=yes"
	  "\n\t-ED:ace=no"
	  ;
      }
      if(hasPACBIO) {
	modestring+=
	  "\nPACBIO_SETTINGS"
	  "\n\t-AL:ms=15:mo=17:mrs=65:egpl=low"
	  "\n\t-CO:emea=15:rodirs=20:amgb=yes:amgbemc=yes:amgbnbs=yes"
	  "\n\t-ED:ace=no"
	  ;
      }
      if(has454) {
	modestring+=
	  "\n454_SETTINGS"
	  "\n\t-AL:ms=15:mo=17:mrs=65:egpl=low"
	  "\n\t-CO:emea=5:rodirs=25:amgb=no"
	  "\n\t-ED:ace=no"
	  ;
      }
      if(hasIONTOR) {
	modestring+=
	  "\nIONTOR_SETTINGS"
	  "\n\t-AL:ms=15:mo=17:mrs=65:egpl=low"
	  "\n\t-CO:emea=5:rodirs=25:amgb=no"
	  "\n\t-ED:ace=no"
	  ;
      }
      break;
    }
    case MP_jobdef_estsnppipeline2 : {
      CEBUG("MP_jobdef_estsnppipeline2\n");
      modestring+=estbasesettings;
      modestring+=
	"\nCOMMON_SETTINGS"
	"\n\t-GE:proout=step2:esps=2"
	"\n\t-SB:lsd=yes:lb=no"
	"\n\t-AS:nop=3:rbl=7"
	"\n\t-CO:mr=yes:mroir=no:asir=no"
	"\n\t-OUT:orc=yes:orf=yes:org=no:ora=no:ort=no:orh=no"
	"\nSANGER_SETTINGS"
	"\n\t-LR:lsd=yes:ft=caf" 
	;
      if(hasSANGER){
	modestring+=
	  "\nSANGER_SETTINGS"
	  "\n\t" + noclipping_string +
	  "\n\t-AL:ms=30:mo=30:mrs=75:egpl=reject_codongaps"
	  "\n\t-CO:emea=15:rodirs=10:amgb=yes:amgbemc=yes:amgbnbs=yes:fnicpst=yes"
	  "\n\t-ED:ace=yes"
	  ;
      }
      if(hasPACBIO){
	modestring+=
	  "\nPACBIO_SETTINGS"
	  "\n\t-LR:lsd=no" 
	  "\n\t" + noclipping_string +
	  "\n\t-AL:ms=30:mo=30:mrs=75:egpl=reject_codongaps"
	  "\n\t-CO:emea=15:rodirs=10:amgb=yes:amgbemc=yes:amgbnbs=yes:fnicpst=yes"
	  "\n\t-ED:ace=yes"
	  ;
      }
      if(has454){
	modestring+=
	  "\n454_SETTINGS"
	  "\n\t-LR:lsd=no"
	  "\n\t" + noclipping_string +
	  "\n\t-AL:ms=30:mo=30:mrs=75:egpl=reject_codongaps"
	  "\n\t-CO:emea=5:rodirs=15:amgb=no:fnicpst=yes"
	  "\n\t-ED:ace=yes"
	  ;
      }
      if(hasIONTOR){
	modestring+=
	  "\nIONTOR_SETTINGS"
	  "\n\t-LR:lsd=no"
	  "\n\t" + noclipping_string +
	  "\n\t-AL:ms=30:mo=30:mrs=75:egpl=reject_codongaps"
	  "\n\t-CO:emea=5:rodirs=15:amgb=no:fnicpst=yes"
	  "\n\t-ED:ace=yes"
	  ;
      }
      break;
    }
    case MP_jobdef_estsnppipeline3 : {
      CEBUG("MP_jobdef_estsnppipeline3\n");
      modestring+=estbasesettings;
      modestring+=
	"\nCOMMON_SETTINGS"
	"\n\t-GE:proout=step3:esps=3"
	"\n\t-FN:cafin=step2_reads.caf:straindatain=step2_straindata_in.txt"
	"\n\t-SB:lsd=yes:lb=no"
	"\n\t-AS:nop=1:rbl=1"
	"\n\t-CO:mr=yes:mroir=no:asir=yes"
	"\n\t-OUT:orc=yes:orf=yes:org=no:ora=no:ort=no:orh=no"
	"\nSANGER_SETTINGS"
	"\n\t-LR:lsd=yes:ft=caf"
	;
	if(hasSANGER){
	  modestring+=
	    "\nSANGER_SETTINGS"
	    "\n\t" + noclipping_string +
	    "\n\t-AL:ms=30:mo=30:mrs=70:egpl=low"
	    "\n\t-CO:rodirs=12:mrpg=1:emea=3:amgb=yes:amgbemc=yes:amgbnbs=yes"
	    "\n\t-ED:ace=no"
	    ;
	}
	if(hasPACBIO){
	  modestring+=
	    "\nPACBIO_SETTINGS"
	    "\n\t" + noclipping_string +
	    "\n\t-AL:ms=30:mo=30:mrs=70:egpl=low"
	    "\n\t-CO:rodirs=12:mrpg=1:emea=3:amgb=yes:amgbemc=yes:amgbnbs=yes"
	    "\n\t-LR:lsd=no"
	    "\n\t-ED:ace=no"
	    ;
	}
	if(has454){
	  modestring+=
	    "\n454_SETTINGS"
	    "\n\t" + noclipping_string +
	    "\n\t-AL:ms=30:mo=30:mrs=70:egpl=low"
	    "\n\t-CO:rodirs=12:mrpg=1:emea=3:amgb=yes:amgbemc=yes:amgbnbs=yes"
	    "\n\t-LR:lsd=no"
	    "\n\t-ED:ace=no"
	    ;
	}
	if(hasIONTOR){
	  modestring+=
	    "\nIONTOR_SETTINGS"
	    "\n\t" + noclipping_string +
	    "\n\t-AL:ms=30:mo=30:mrs=70:egpl=low"
	    "\n\t-CO:rodirs=12:mrpg=1:emea=3:amgb=yes:amgbemc=yes:amgbnbs=yes"
	    "\n\t-LR:lsd=no"
	    "\n\t-ED:ace=no"
	    ;
	}
      break;
    }
  default : {
  }
  }
  
  CEBUG("Standard settings after quality, method and type lookup:" << modestring << endl);

  parseQuickmode(modestring.c_str(), "JobDefs", Pv, false);

  FUNCEND();
}



void MIRAParameters::saveParsedSettingsValues(vector<MIRAParameters> & Pv, MIRAParameters * actpar, vector<bool> & saved)
{
  saved.clear();
  if(!Pv.empty()) {
    saved.resize(Pv.size());
    for(uint32 i=0; i<Pv.size(); i++){
      saved[i]=Pv[i].MP_parsedsomesettings;
    }
  }else if(actpar!= NULL){
    saved.resize(1, actpar->MP_parsedsomesettings);
  }
}

void MIRAParameters::restoreParsedSettingsValues(vector<MIRAParameters> & Pv, MIRAParameters * actpar, vector<bool> & saved)
{
  if(saved.size()==Pv.size()) {
    for(uint32 i=0; i<saved.size(); i++){
      Pv[i].MP_parsedsomesettings=saved[i];
    }
  }else if(actpar!= NULL && !saved.empty()){
    actpar->MP_parsedsomesettings=saved[0];
  }
}
