#include <string>

#include "Dictionary.h"
#include "DicomFile.h"
#include "SeriesHandler.h"
#include "HandlerFactory.h"

using namespace jcs;
using std::string;

enum { 
	STANDARD, 
	NUMARIS_VOLUME, 
	NUMARIS_SLICE, 
	NUMARIS_MOSAIC,
	SYNGO_VOLUME, 
	SYNGO_SLICE, 
	SYNGO_MOSAIC,
	GE_EPI,
	GE_DTI,
	GE_DTI2,
	ENHANCED_MR,
	ACHIEVA
};

const char* EnhancedMrSopClass = "1.2.840.10008.5.1.4.1.1.4.1";

SeriesHandler*
HandlerFactory::CreateHandler(const char* filename)
{
	string series_uid;
	DicomFile dfile(filename);
	dfile.Find("SeriesInstanceUid", series_uid);

	// Fix problems with termination
	string temp(series_uid.c_str());
	series_uid.clear();
	series_uid = temp;

	SeriesHandler* handler;

	switch (GetSeriesType(dfile)) {

		case SYNGO_VOLUME :
		case SYNGO_SLICE :
			handler = new SyngoHandler(series_uid);
			break;

		case NUMARIS_MOSAIC :
			handler = new NumarisMosaicHandler(series_uid);
			break;

		case SYNGO_MOSAIC :
			handler = new SyngoMosaicHandler(series_uid);
			break;

		case GE_EPI :
			handler = new GeEpiHandler(series_uid);
			break;

		case GE_DTI :
			handler = new GeDtiRbHandler(series_uid);
			break;

		case GE_DTI2 :
			handler = new GeDti2Handler(series_uid);
			break;

		case ENHANCED_MR :
			handler = new EnhancedMrHandler(series_uid);
			break;

		case ACHIEVA :
			handler = new AchievaDtiHandler(series_uid);
			break;

		default :
			handler = new SeriesHandler(series_uid);
		}

	return handler;
}

int
HandlerFactory::GetSeriesType(DicomFile& dfile)
{
	string s;

	dfile.Find("SopClass", s);
	if (s.find(EnhancedMrSopClass) != string::npos) return ENHANCED_MR;

	dfile.Find("Manufacturer", s);

	if (s.find("SIEMENS") != string::npos) {

		string software;
		dfile.Find("SoftwareVersion", software);

		if (software.find("VA13") != string::npos) {

			Dictionary* Numaris = Numaris_Dictionary::Instance();
			int x = 0;
			if (dfile.Find(Numaris->Lookup("BASE_RAW_MATRIX_SIZE"), x) &&
				(x == 64)) return NUMARIS_MOSAIC;

			dfile.Find("MrAcquisitionType", s);
			if (s.find("3D") == string::npos) return NUMARIS_SLICE;
			else return NUMARIS_VOLUME;
		}

		else if (software.find("syngo") != string::npos) {
			dfile.Find("ImageType", s);
			if ((s.find("MOSAIC") != string::npos) &&
				(s.find("DUMMY IMAGE") == string::npos)) return SYNGO_MOSAIC;
			dfile.Find("MrAcquisitionType", s);
			if (s.find("3D") == string::npos) return SYNGO_SLICE;
			else return SYNGO_VOLUME;
		}

		else return STANDARD;
	}

	if (s.find("GE") != string::npos) {
		string sequence;
		Dictionary* Excite = Excite_Dictionary::Instance();
		dfile.Find(Excite->Lookup("Pulse_seq_Name"), sequence);
		if (sequence.find("dti_epi") != string::npos) return GE_DTI;
		if (sequence.find("epi2") != string::npos) return GE_DTI2;
		if (sequence.find("epi") != string::npos) return GE_EPI;
		string imageType;
		dfile.Find("ImageType", imageType);
		if (imageType.find("EPI") != string::npos) return GE_EPI;
		else return STANDARD;
	}

	if (s.find("Philips") != string::npos) {
		string description;
		dfile.Find("SeriesDescription", description);
		if (description.find("DTI") != string::npos) return ACHIEVA;
		dfile.Find("ProtocolName", description);
		if (description.find("DTI") != string::npos) return ACHIEVA;
	
	}

	return STANDARD;

}

int
HandlerFactory::GetSeriesType(const char* filename)
{
	DicomFile dfile(filename);
	return GetSeriesType(dfile);
}
