/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2012 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

#include "internal.h"

struct _nacore_avl_tree
  {
	nacore_avl_tree_elem	root;
	size_t			count;
	nacore_cmp_cb		cmp_cb;
	nacore_get_size_cb	gs_cb;
  };

struct _nacore_avl_tree_elem
  {
	nacore_avl_tree_elem	 parent;
	nacore_avl_tree_elem	 left;
	nacore_avl_tree_elem	 right;
	size_t			 height;
	void			*value;
  };

_NACORE_DEF nacore_avl_tree
nacore_avl_tree_new(nacore_cmp_cb cmp_cb, nacore_get_size_cb gs_cb)
{
	nacore_avl_tree ret;

	ret = malloc(sizeof(struct _nacore_avl_tree));
	if (ret == NULL)
		return NULL;

	ret->root	= NULL;
	ret->count	= 0;
	ret->cmp_cb	= cmp_cb;
	ret->gs_cb	= gs_cb;

	return ret;
}

static void
dfs(nacore_avl_tree_elem elem, nacore_op_cb op_cb, void *op_opaque)
{
	if (elem->left != NULL)
		dfs(elem->left, op_cb, op_opaque);
	if (elem->right != NULL)
		dfs(elem->right, op_cb, op_opaque);

	op_cb(elem, op_opaque);
}

typedef struct
  {
	nacore_avl_tree	 tree;
	nacore_op_cb	 free_cb;
	void		*free_opaque;
  } free_elem_data_t;

static void
free_elem(void *value, void *opaque)
{
	nacore_avl_tree_elem e;
	free_elem_data_t *d;

	e = (nacore_avl_tree_elem)value;
	d = (free_elem_data_t *)opaque;

	if (d->free_cb != NULL)
		d->free_cb(e->value, d->free_opaque);

	if (d->tree->gs_cb != NULL)
		free(e->value);

	free(e);
}

_NACORE_DEF void
nacore_avl_tree_free(nacore_avl_tree tree, nacore_op_cb free_cb,
		     void *free_opaque)
{
	free_elem_data_t d;

	d.tree		= tree;
	d.free_cb	= free_cb;
	d.free_opaque	= free_opaque;

	if (tree->root != NULL)
		dfs(tree->root, free_elem, &d);

	free(tree);
}

_NACORE_DEF size_t
nacore_avl_tree_get_n_elems(nacore_avl_tree tree)
{
	return tree->count;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_get_first(nacore_avl_tree tree)
{
	nacore_avl_tree_elem ret;

	if (tree->root == NULL)
		ret = NULL;
	else
		for (ret = tree->root; ret->left != NULL; ret = ret->left) ;

	return ret;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_get_last(nacore_avl_tree tree)
{
	nacore_avl_tree_elem ret;

	if (tree->root == NULL)
		ret = NULL;
	else
		for (ret = tree->root; ret->right != NULL; ret = ret->right) ;

	return ret;
}

static nacore_avl_tree_elem
lookup(nacore_avl_tree tree, void *value, int *side, void *cmp_opaque)
{
	nacore_avl_tree_elem ret;

	if (tree->root == NULL)
	  {
		*side = 0;
		return NULL;
	  }

	ret = tree->root;
	while (1)
	  {
		*side = tree->cmp_cb(value, ret->value, cmp_opaque);
	
		if (*side == 0)
			break;
		else if (*side < 0)
		  {
			if (ret->left == NULL)
				break;
			ret = ret->left;
			continue;
		  }

		if (ret->right == NULL)
			break;
		ret = ret->right;
	  }

	return ret;
}

#define MAX(a, b)	(((a) > (b)) ? (a) : (b))

#define LHEIGHT(e)	(((e)->left != NULL) ? (e)->left->height : 0)
#define RHEIGHT(e)	(((e)->right != NULL) ? (e)->right->height : 0)

static void
adjust_up(nacore_avl_tree tree, nacore_avl_tree_elem elem)
{
	nacore_avl_tree_elem p, e;
	size_t lh, rh;

	while(1)
	  {
		lh = LHEIGHT(elem);
		rh = RHEIGHT(elem);

		if (lh == rh + 2)
		  {
			lh = LHEIGHT(elem->left);
			rh = RHEIGHT(elem->left);

			if (lh > rh)
			  {
				/* left-left */
				p = elem->left;

				elem->left = p->right;
				if (p->right != NULL)
					p->right->parent = elem;
				p->parent = elem->parent;
				if (p->parent != NULL)
				  {
					if (p->parent->left == elem)
						p->parent->left = p;
					else
						p->parent->right = p;
				  }
				p->right = elem;
				elem->parent = p;
			  }
			else
			  {
				/* left-right */
				p = elem->left;
				e = p->right;

				p->right = e->left;
				if (e->left != NULL)
					e->left->parent = p;
				elem->left = e->right;
				if (e->right != NULL)
					e->right->parent = elem;
				e->parent = elem->parent;
				if (e->parent != NULL)
				  {
					if (e->parent->left == elem)
						e->parent->left = e;
					else
						e->parent->right = e;
				  }
				e->left = p;
				p->parent = e;
				e->right = elem;
				elem->parent = e;

				lh = LHEIGHT(p);
				rh = RHEIGHT(p);
				p->height = MAX(lh, rh) + 1;
			  }

			lh = LHEIGHT(elem);
			rh = RHEIGHT(elem);
		  }
		else if (rh == lh + 2)
		  {
			lh = LHEIGHT(elem->right);
			rh = RHEIGHT(elem->right);

			if (rh > lh)
			  {
				/* right-right */
				p = elem->right;

				elem->right = p->left;
				if (p->left != NULL)
					p->left->parent = elem;
				p->parent = elem->parent;
				if (p->parent != NULL)
				  {
					if (p->parent->left == elem)
						p->parent->left = p;
					else
						p->parent->right = p;
				  }
				p->left = elem;
				elem->parent = p;
			  }
			else
			  {
				/* right-left */
				p = elem->right;
				e = p->left;

				p->left = e->right;
				if (e->right != NULL)
					e->right->parent = p;
				elem->right = e->left;
				if (e->left != NULL)
					e->left->parent = elem;
				e->parent = elem->parent;
				if (e->parent != NULL)
				  {
					if (e->parent->left == elem)
						e->parent->left = e;
					else
						e->parent->right = e;
				  }
				e->left = elem;
				e->right = p;
				elem->parent = e;
				p->parent = e;

				lh = LHEIGHT(p);
				rh = RHEIGHT(p);
				p->height = MAX(lh, rh) + 1;
			  }

			lh = LHEIGHT(elem);
			rh = RHEIGHT(elem);
		  }
		
		elem->height = MAX(lh, rh) + 1;

		if (elem->parent == NULL)
		  {
			tree->root = elem;
			break;
		  }

		elem = elem->parent;
	  }
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_insert(nacore_avl_tree tree, void *cmp_opaque, void *gs_opaque,
		       void *value)
{
	nacore_avl_tree_elem ret;
	size_t size;

	ret = malloc(sizeof(struct _nacore_avl_tree_elem));
	if (ret == NULL)
		return NULL;

	if (tree->gs_cb != NULL)
	  {
		size = tree->gs_cb(value, gs_opaque);
		ret->value = malloc(size);
		if (ret->value == NULL)
		  {
			free(ret);
			return NULL;
		  }
		memcpy(ret->value, value, size);
	  }
	else
		ret->value = value;

	nacore_avl_tree_elem_insert(tree, ret, cmp_opaque);

	return ret;
}

_NACORE_DEF void *
nacore_avl_tree_pop(nacore_avl_tree tree, nacore_avl_tree_elem elem)
{
	void *ret;

	nacore_avl_tree_elem_pop(tree, elem);

	ret = elem->value;
	free(elem);

	return ret;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_find(nacore_avl_tree tree, void *cmp_opaque,
		     nacore_filter_cb filter_cb, void *filter_opaque,
		     void *value)
{
	nacore_avl_tree_elem ret, tmp;
	int side;

	ret = lookup(tree, value, &side, cmp_opaque);
	if (side != 0)
		return NULL;

	if (filter_cb != NULL)
		if (filter_cb(ret->value, filter_opaque) == 0)
		  {
			tmp = ret;
			ret = nacore_avl_tree_find_prev(tree, tmp, cmp_opaque,
							filter_cb,
							filter_opaque);
			if (ret == NULL)
				ret = nacore_avl_tree_find_next(tree, tmp,
					cmp_opaque, filter_cb, filter_opaque);
		  }

	return ret;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_find_first(nacore_avl_tree tree, void *cmp_opaque,
			   nacore_filter_cb filter_cb, void *filter_opaque,
			   void *value)
{
	nacore_avl_tree_elem ret, prev;

	ret = nacore_avl_tree_find(tree, cmp_opaque, filter_cb, filter_opaque,
				   value);
	if (ret == NULL)
		return NULL;

	for (prev = ret; prev != NULL; ret = prev)
		prev = nacore_avl_tree_find_prev(tree, ret, cmp_opaque,
						 filter_cb, filter_opaque);

	return ret;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_find_last(nacore_avl_tree tree, void *cmp_opaque,
			  nacore_filter_cb filter_cb, void *filter_opaque,
			  void *value)
{
	nacore_avl_tree_elem ret, next;

	ret = nacore_avl_tree_find(tree, cmp_opaque, filter_cb, filter_opaque,
				   value);
	if (ret == NULL)
		return NULL;

	for (next = ret; next != NULL; ret = next)
		next = nacore_avl_tree_find_next(tree, ret, cmp_opaque,
						 filter_cb, filter_opaque);

	return ret;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_find_prev(nacore_avl_tree tree, nacore_avl_tree_elem elem,
			  void *cmp_opaque, nacore_filter_cb filter_cb,
			  void *filter_opaque)
{
	nacore_avl_tree_elem ret;

	ret = nacore_avl_tree_elem_get_prev(tree, elem);
	if (ret != NULL)
	  {
		if (tree->cmp_cb(ret->value, elem->value, cmp_opaque) != 0)
			ret = NULL;
		else if (filter_cb != NULL)
		  {
			while (filter_cb(ret->value, filter_opaque) == 0)
			  {
				ret = nacore_avl_tree_elem_get_prev(tree, ret);
				if (ret == NULL)
					break;

				if (tree->cmp_cb(ret->value, elem->value,
						 cmp_opaque) != 0)
				  {
					ret = NULL;
					break;
				  }
			  }
		  }
	  }

	return ret;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_find_next(nacore_avl_tree tree, nacore_avl_tree_elem elem,
			  void *cmp_opaque, nacore_filter_cb filter_cb,
			  void *filter_opaque)
{
	nacore_avl_tree_elem ret;

	ret = nacore_avl_tree_elem_get_next(tree, elem);
	if (ret != NULL)
	  {
		if (tree->cmp_cb(ret->value, elem->value, cmp_opaque) != 0)
			ret = NULL;
		else if (filter_cb != NULL)
		  {
			while (filter_cb(ret->value, filter_opaque) == 0)
			  {
				ret = nacore_avl_tree_elem_get_next(tree, ret);
				if (ret == NULL)
					break;

				if (tree->cmp_cb(ret->value, elem->value,
						 cmp_opaque) != 0)
				  {
					ret = NULL;
					break;
				  }
			  }
		  }
	  }

	return ret;
}

_NACORE_DEF nacore_avl_tree
nacore_avl_tree_dup(nacore_avl_tree tree, void *cmp_opaque,
		    nacore_get_size_cb gs_cb, void *gs_opaque,
		    nacore_filter_cb filter_cb, void *filter_opaque,
		    nacore_op_cb dup_cb, void *dup_opaque)
{
	nacore_avl_tree ret;
	nacore_avl_tree_elem elem;

	ret = nacore_avl_tree_new(tree->cmp_cb, gs_cb);
	if (ret == NULL)
		return NULL;

	for (elem = nacore_avl_tree_get_first(tree); elem != NULL;
	     elem = nacore_avl_tree_elem_get_next(tree, elem))
	  {
		if (filter_cb != NULL)
			if (filter_cb(elem->value, filter_opaque) == 0)
				continue;

		if (nacore_avl_tree_insert(ret, cmp_opaque, gs_opaque,
					   elem->value) == NULL)
		  {
			  nacore_avl_tree_free(ret, NULL, NULL);
			  ret = NULL;
			  break;
		  }
	  }

	if ((dup_cb != NULL) && (ret != NULL))
		for (elem = nacore_avl_tree_get_first(ret); elem != NULL;
		     elem = nacore_avl_tree_elem_get_next(ret, elem))
			dup_cb(elem->value, dup_opaque);

	return ret;
}

typedef struct
  {
	nacore_avl_tree	 dest;
	void		*cmp_opaque;
  } merge_data_t;

static void
merge(void *value, void *opaque)
{
	nacore_avl_tree_elem elem;
	merge_data_t *d;

	elem = (nacore_avl_tree_elem)value;
	d = (merge_data_t *)opaque;

	nacore_avl_tree_elem_insert(d->dest, elem, d->cmp_opaque);
}

_NACORE_DEF nacore_avl_tree
nacore_avl_tree_merge(nacore_avl_tree dest, nacore_avl_tree src,
		      void *cmp_opaque)
{
	merge_data_t d;

	if (dest->count == 0)
	  {
		nacore_avl_tree_free(dest, NULL, NULL);
		return src;
	  }

	if (src->count == 0)
	  {
		nacore_avl_tree_free(src, NULL, NULL);
		return dest;
	  }

	d.dest = dest;
	d.cmp_opaque = cmp_opaque;

	dfs(dest->root, merge, &d);

	src->root = NULL;
	nacore_avl_tree_free(src, NULL, NULL);

	return dest;
}

_NACORE_DEF void
nacore_avl_tree_dump(nacore_avl_tree tree, nacore_to_string_cb to_string_cb,
		     void *to_string_opaque)
{
	nacore_avl_tree_elem elem;
	char *desc;

	fprintf(stderr, "AVL tree: %p, root: %p, # elems: %"
		NACORE_LIBC_SIZE_FORMAT_LM "u\n", (void *)tree,
		(void *)tree->root, (NACORE_LIBC_SIZE_FORMAT_TYPE)tree->count);

	for (elem = nacore_avl_tree_get_first(tree); elem != NULL;
	     elem = nacore_avl_tree_elem_get_next(tree, elem))
	  {
		fprintf(stderr, "  Elem: %p, value ptr: %p, parent: %p, "
			"left: %p, right: %p, height: %"
			NACORE_LIBC_SIZE_FORMAT_LM "u", (void *)elem,
			(void *)elem->value, (void *)elem->parent,
			(void *)elem->left, (void *)elem->right,
			(NACORE_LIBC_SIZE_FORMAT_TYPE)elem->height);

		if (to_string_cb != NULL)
		  {
			desc = to_string_cb(elem->value, to_string_opaque);
			if (desc != NULL)
			  {
				fprintf(stderr, ", value: %s", desc);
				free(desc);
			  }
		  }

		fprintf(stderr, "\n");
	  }

	fprintf(stderr, "AVL tree dump end\n");
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_elem_new(void *value)
{
	nacore_avl_tree_elem ret;

	ret = malloc(sizeof(struct _nacore_avl_tree_elem));
	if (ret == NULL)
		return NULL;

	ret->parent = NULL;
	ret->left = NULL;
	ret->right = NULL;
	ret->height = 0;
	ret->value = value;

	return ret;
}

_NACORE_DEF void
nacore_avl_tree_elem_free(nacore_avl_tree_elem elem)
{
	free(elem);
}

_NACORE_DEF void
nacore_avl_tree_elem_insert(nacore_avl_tree tree, nacore_avl_tree_elem elem,
			    void *cmp_opaque)
{
	int side;

	elem->left   = NULL;
	elem->right  = NULL;
	elem->height = 1;

	elem->parent = lookup(tree, elem->value, &side, cmp_opaque);

	if (elem->parent == NULL)
	  {
		tree->root = elem;
		tree->count++;
		return;
	  }

	if (side == 0)
		while (elem->parent->left != NULL)
			elem->parent = elem->parent->left;

	if (side <= 0)
		elem->parent->left = elem;
	else
		elem->parent->right = elem;

	adjust_up(tree, elem);

	tree->count++;
}

_NACORE_DEF void
nacore_avl_tree_elem_pop(nacore_avl_tree tree, nacore_avl_tree_elem elem)
{
	nacore_avl_tree_elem p, e;

	if ((elem->left == NULL) && (elem->right == NULL))
	  {
		if (elem->parent == NULL)
		  {
			tree->root = NULL;
			tree->count--;
			return;
		  }

		e = elem->parent;

		if (e->left == elem)
			e->left = NULL;
		else
			e->right = NULL;
	  }
	else if ((elem->left != NULL) && (elem->right == NULL))
	  {
		if (elem->parent == NULL)
		  {
			tree->root = elem->left;
			elem->left->parent = NULL;
			tree->count--;
			return;
		  }

		e = elem->parent;

		elem->left->parent = e;
		if (e->left == elem)
			e->left = elem->left;
		else
			e->right = elem->left;
	  }
	else if ((elem->left == NULL) && (elem->right != NULL))
	  {
		if (elem->parent == NULL)
		  {
			tree->root = elem->right;
			elem->right->parent = NULL;
			tree->count--;
			return;
		  }

		e = elem->parent;

		elem->right->parent = e;
		if (e->left == elem)
			e->left = elem->right;
		else
			e->right = elem->right;
	  }
	else
	  {
		for (e = elem->right; e->left != NULL; e = e->left) ;
		p = e->parent;

		e->left = elem->left;
		e->left->parent = e;
		if (elem->right != e)
		  {
			p->left = e->right;
			e->right = elem->right;
			e->right->parent = e;
		  }
		else
			p = e;
		e->parent = elem->parent;
		if (e->parent != NULL)
		  {
			if (e->parent->left == elem)
				e->parent->left = e;
			else
				e->parent->right = e;
		  }
		else
			tree->root = e;

		e = p;
	  }

	adjust_up(tree, e);

	tree->count--;
}

_NACORE_DEF void *
nacore_avl_tree_elem_get_value(nacore_avl_tree tree, nacore_avl_tree_elem elem)
{
	return elem->value;
}

_NACORE_DEF int
nacore_avl_tree_elem_set_value(nacore_avl_tree tree, nacore_avl_tree_elem elem,
			       nacore_op_cb free_cb, void *free_opaque,
			       void *cmp_opaque, void *gs_opaque, void *value)
{
	void *new_val;
	size_t size;

	if (tree == NULL)
	  {
		elem->value = value;
		return 0;
	  }

	new_val = NULL;  /* make gcc happy */

	if (tree->gs_cb != NULL)
	  {
		size = tree->gs_cb(value, gs_opaque);
		new_val = malloc(size);
		if (new_val == NULL)
			return ENOMEM;

		memcpy(new_val, value, size);
	  }

	nacore_avl_tree_elem_pop(tree, elem);

	if (tree->gs_cb == NULL)
		elem->value = value;
	else
	  {
		if (free_cb != NULL)
			free_cb(elem->value, free_opaque);
		free(elem->value);
		elem->value = new_val;
	  }

	nacore_avl_tree_elem_insert(tree, elem, cmp_opaque);

	return 0;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_elem_get_prev(nacore_avl_tree tree, nacore_avl_tree_elem elem)
{
	nacore_avl_tree_elem ret;

	ret = NULL;

	if (elem->left != NULL)
	  {
		for (ret = elem->left, elem = ret->right; elem != NULL;
		     ret = elem, elem = ret->right) ;
		return ret;
	  }

	for (ret = elem->parent; ret != NULL; elem = ret, ret = elem->parent)
		if (ret->right == elem)
			break;

	return ret;
}

_NACORE_DEF nacore_avl_tree_elem
nacore_avl_tree_elem_get_next(nacore_avl_tree tree, nacore_avl_tree_elem elem)
{
	nacore_avl_tree_elem ret;

	ret = NULL;

	if (elem->right != NULL)
	  {
		for (ret = elem->right, elem = ret->left; elem != NULL;
		     ret = elem, elem = ret->left) ;
		return ret;
	  }

	for (ret = elem->parent; ret != NULL; elem = ret, ret = elem->parent)
		if (ret->left == elem)
			break;

	return ret;
}
