/* -*- Mode:C++; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * Copyright (c) 2009 IITP RAS
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Pavel Boyko <boyko@iitp.ru>
 */

#include "ns3/test.h"
#include "ns3/packet.h"
#include "ns3/simulator.h"
#include "flame-header.h"
#include "flame-rtable.h"

namespace ns3 {
namespace flame {

/// Built-in self test for FlameHeader
struct FlameHeaderTest : public TestCase
{
  FlameHeaderTest () :
    TestCase ("FlameHeader roundtrip serialization")
  {
  }
  bool DoRun ();
};

bool
FlameHeaderTest::DoRun ()
{
  FlameHeader a;
  a.AddCost (123);
  a.SetSeqno (456);
  a.SetOrigDst (Mac48Address ("11:22:33:44:55:66"));
  a.SetOrigSrc (Mac48Address ("00:11:22:33:44:55"));
  a.SetProtocol (0x806);
  Ptr<Packet> packet = Create<Packet> ();
  packet->AddHeader (a);
  FlameHeader b;
  packet->RemoveHeader (b);
  NS_TEST_ASSERT_MSG_EQ (b, a, "FlameHeader roundtrip serialization works");
  return false;
}

//-----------------------------------------------------------------------------

/// Unit test for FlameRtable
class FlameRtableTest : public TestCase
{
public:
  FlameRtableTest ();
  bool DoRun ();

private:
  /// Test Add apth and lookup path;
  void TestLookup ();
  /**
   * \name Test add path and try to lookup after entry has expired
   * \{
   */
  void TestAddPath ();
  void TestExpire ();
  ///\}
private:
  bool error;

  Mac48Address dst;
  Mac48Address hop;
  uint32_t iface;
  uint8_t cost;
  uint16_t seqnum;
  Ptr<FlameRtable> table;
};

/// Test instance
static FlameRtableTest g_FlameRtableTest;

FlameRtableTest::FlameRtableTest () :
  TestCase ("Mesh/Flame/FlameRtable"), 
  error (false), 
  dst ("01:00:00:01:00:01"), 
  hop ("01:00:00:01:00:03"),
  iface (8010), 
  cost (10), 
  seqnum (1)
{
}

void
FlameRtableTest::TestLookup ()
{
  FlameRtable::LookupResult correct (hop, iface, cost, seqnum);

  table->AddPath (dst, hop, iface, cost, seqnum);
  NS_TEST_EXPECT_MSG_EQ ((table->Lookup (dst) == correct), true, "Routing table lookup works");
}

void
FlameRtableTest::TestAddPath ()
{
  table->AddPath (dst, hop, iface, cost, seqnum);
}

void
FlameRtableTest::TestExpire ()
{
  // this is assumed to be called when path records are already expired
  FlameRtable::LookupResult correct (hop, iface, cost, seqnum);
  NS_TEST_EXPECT_MSG_EQ (table->Lookup (dst).IsValid (), false, "Routing table records expirations works");
}

bool
FlameRtableTest::DoRun ()
{
  table = CreateObject<FlameRtable> ();

  Simulator::Schedule (Seconds (0), &FlameRtableTest::TestLookup, this);
  Simulator::Schedule (Seconds (1), &FlameRtableTest::TestAddPath, this);
  Simulator::Schedule (Seconds (122), &FlameRtableTest::TestExpire, this);

  Simulator::Run ();
  Simulator::Destroy ();

  return GetErrorStatus ();
}


//-----------------------------------------------------------------------------

class FlameTestSuite : public TestSuite
{
public:
  FlameTestSuite ();
};

FlameTestSuite::FlameTestSuite ()
  : TestSuite ("devices-mesh-flame", UNIT)
{
  AddTestCase (new FlameHeaderTest);
  AddTestCase (new FlameRtableTest);
}

FlameTestSuite g_flameTestSuite;

}
}
