//                                               -*- C++ -*-
/**
 *  @file  MarginalTransformationGradient.cxx
 *  @brief Class for the Nataf transformationGradient evaluation for elliptical
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: MarginalTransformationGradient.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include "MarginalTransformationGradient.hxx"
#include "PersistentObjectFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS

CLASSNAMEINIT(MarginalTransformationGradient);

static Factory<MarginalTransformationGradient> RegisteredFactory("MarginalTransformationGradient");

/* ParameterDefault constructor */
MarginalTransformationGradient::MarginalTransformationGradient():
  NumericalMathGradientImplementation(),
  evaluation_()
{
  // Nothing to do
}

/* Parameter constructor */
MarginalTransformationGradient::MarginalTransformationGradient(const MarginalTransformationEvaluation & evaluation):
  NumericalMathGradientImplementation(),
  evaluation_(evaluation)
{
  // Nothing to do
}

/* Virtual constructor */
MarginalTransformationGradient * MarginalTransformationGradient::clone() const
{
  return new MarginalTransformationGradient(*this);
}

/* Gradient */
Matrix MarginalTransformationGradient::gradient(const NumericalPoint & inP) const
{
  const UnsignedLong dimension(getOutputDimension());
  Matrix result(dimension, dimension);
  // (G^{-1} o F)' = F' . G^{-1}' o F = F' / (G' o G^{-1} o F)
  for (UnsignedLong i = 0; i < dimension; ++i)
    {
      const NumericalScalar inputPDF(evaluation_.inputDistributionCollection_[i].computePDF(inP[i]));
      // Quick rejection step: if the input PDF is zero, the result will be zero, so continue only if the value is > 0
      if (inputPDF > 0.0)
        {
          NumericalScalar inputCDF(evaluation_.inputDistributionCollection_[i].computeCDF(inP[i]));
          // For accuracy reason, check if we are in the upper tail of the distribution
          const Bool upperTail(inputCDF > 0.5);
          if (upperTail) inputCDF = evaluation_.inputDistributionCollection_[i].computeCDF(inP[i], upperTail);
          // The upper tail CDF is defined by CDF(x, upper) = P(X>x)
          // The upper tail quantile is defined by Quantile(CDF(x, upper), upper) = x
          const NumericalPoint  outputQuantile(evaluation_.outputDistributionCollection_[i].computeQuantile(inputCDF, upperTail));
          const NumericalScalar outputPDF(evaluation_.outputDistributionCollection_[i].computePDF(outputQuantile));
          // The output PDF should never be zero here, be it can occure due to some strange rounding error
          if (outputPDF > 0.0) result(i, i) = inputPDF / outputPDF;
        }
    }
  return result;
}

/* Accessor for input point dimension */
UnsignedLong MarginalTransformationGradient::getInputDimension() const
{
  return evaluation_.inputDistributionCollection_.getSize();
}

/* Accessor for output point dimension */
UnsignedLong MarginalTransformationGradient::getOutputDimension() const
{
  return evaluation_.outputDistributionCollection_.getSize();
}

/* String converter */
String MarginalTransformationGradient::__repr__() const
{
  OSS oss;
  oss << "class=" << MarginalTransformationGradient::GetClassName()
      << " evaluation=" << evaluation_;
  return oss;
}

String MarginalTransformationGradient::__str__(const String & offset) const
{
  return OSS() << offset << "Gradient of " << evaluation_.getName();
}

/* Method save() stores the object through the StorageManager */
void MarginalTransformationGradient::save(Advocate & adv) const
{
  NumericalMathGradientImplementation::save(adv);
  adv.saveAttribute( "evaluation_", evaluation_ );
}

/* Method load() reloads the object from the StorageManager */
void MarginalTransformationGradient::load(Advocate & adv)
{
  NumericalMathGradientImplementation::load(adv);
  adv.loadAttribute( "evaluation_", evaluation_ );
}

END_NAMESPACE_OPENTURNS

