///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/input/XFormManager.h>

#include "DeformationGradientDataChannel.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(DeformationGradientDataChannel, DataChannel)

/******************************************************************************
* Serialization constructor.
******************************************************************************/
DeformationGradientDataChannel::DeformationGradientDataChannel(bool isLoading) : DataChannel(isLoading)
{
	if(!isLoading) {
		setVisible(false);
	}
}

/******************************************************************************
* Special constructor to create a standard data channel.
******************************************************************************/
DeformationGradientDataChannel::DeformationGradientDataChannel(DataChannelIdentifier which) : DataChannel(which)
{
	setVisible(false);
}

/******************************************************************************
* Render the orientation tripods into the viewport.
******************************************************************************/
void DeformationGradientDataChannel::render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode)
{
	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel) return;

	// Disable z-buffer.
	vp->setDepthTest(false);
	vp->setLightingEnabled(false);

	const Point3* p = posChannel->constDataPoint3();
	const Tensor2* F = constDataTensor2();
	for(size_t i = size(); i != 0; --i, ++p, ++F) {
		AffineTransformation tm(*F);
		tm.setTranslation(*p - ORIGIN);
		XFORM_MANAGER.renderTripod(vp, tm, false);
	}

	// Re-enable z-buffer.
	vp->setDepthTest(true);
}

/******************************************************************************
* Renders the channel's contents in high-quality mode to an offscreen buffer.
******************************************************************************/
void DeformationGradientDataChannel::renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext)
{
}

IMPLEMENT_PLUGIN_CLASS(DeformationGradientDataChannelEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void DeformationGradientDataChannelEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Atomic Deformation Gradients"), rolloutParams);

	QVBoxLayout* mainLayout = new QVBoxLayout(rollout);
	mainLayout->setContentsMargins(4,4,4,4);

	BooleanPropertyUI* showTensorsUI = new BooleanPropertyUI(this, "isVisible", tr("Show deformation tensors"));
	mainLayout->addWidget(showTensorsUI->checkBox());
}

};	// End of namespace AtomViz
