///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/utilities/ProgressIndicator.h>

#include "IMDAtomFileParser.h"
#include "IMDParserSettingsDialog.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>
#include "../ColumnChannelMapping.h"
#include "../CompressedTextParserStream.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(IMDAtomFileParser, MultiFileParser)

/******************************************************************************
* Checks if the given file has format that can be read by this importer.
******************************************************************************/
bool IMDAtomFileParser::checkFileFormat(const QString& filepath)
{
	// Open the input file for reading.
	CompressedTextParserStream stream(filepath);

	// Read first line.
	char buffer[1024];
	int count = stream.getline(buffer, sizeof(buffer)/sizeof(buffer[0]));
	if(count<5) return false;
	buffer[5] = '\0';

	return strcmp(buffer, "#F A ") == 0;
}

/******************************************************************************
* Opens the settings dialog for this parser.
******************************************************************************/
bool IMDAtomFileParser::showSettingsDialog(QWidget* parent)
{
	IMDParserSettingsDialog dialog(this, parent);
	if(dialog.exec() != QDialog::Accepted)
		return false;
	return true;
}

/******************************************************************************
* Parses the header of the given file and returns the number of data columns contained in the file.
******************************************************************************/
bool IMDAtomFileParser::inspectFileHeader(const QString& filename, int& numberOfColumns, QStringList& columnNames)
{
	// Open the input file for reading.
	CompressedTextParserStream stream(filename);

	// Read first header line.
	stream.readline();
	if(stream.line().compare(0, 2, "#F") != 0)
		throw Exception(tr("The file %1 is not an IMD atom file.").arg(filename));
	QStringList tokens = QString(stream.line().c_str()).split(QRegExp("\\s+"));
	if(tokens.size() < 2 || tokens[1] != "A")
		throw Exception(tr("The file %1 is not an IMD atom file in ASCII format.").arg(filename));

	// Read remaining header lines
	while(true) {
		stream.readline();
		if(stream.line().empty() || stream.line().at(0) != '#') throw Exception(tr("Invalid header in IMD atom file %1 (line %2).").arg(filename).arg(stream.lineNumber()));
		if(stream.line().at(1) == '#') continue;
		else if(stream.line().at(1) == 'E') break;
		else if(stream.line().at(1) == 'C') {
			columnNames = QString(stream.line().c_str()).split(QRegExp("\\s+"));
			columnNames.pop_front();
			numberOfColumns = columnNames.size();
		}
		else if(stream.line().at(1) == 'X') {}
		else if(stream.line().at(1) == 'Y') {}
		else if(stream.line().at(1) == 'Z') {}
		else throw Exception(tr("Invalid header line key in IMD atom file %1 (line %2).").arg(filename).arg(stream.lineNumber()));
	}
	return true;
}

/******************************************************************************
* Reads an atomic data set from the input file.
******************************************************************************/
EvaluationStatus IMDAtomFileParser::loadTimeStep(AtomsObject* destination, int movieFrame, const QString& filename, streampos byteOffset, int lineNumber, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);

	// Show the progress indicator.
	ProgressIndicator progress(tr("Opening IMD atoms file '%1'").arg(inputFile()), 0, suppressDialogs);

	// Open the input file for reading.
	CompressedTextParserStream stream(filename);

	// Seek to the byte offset where the requested movie frame is stored.
	if(byteOffset != streampos(0))
		stream.seek(byteOffset);

	setlocale(LC_NUMERIC, "C");

	// Read first header line.
	stream.readline();
	if(stream.line().compare(0, 2, "#F") != 0)
		throw Exception(tr("The file %1 is not an IMD atom file.").arg(filename));
	QStringList tokens = QString(stream.line().c_str()).split(QRegExp("\\s+"));
	if(tokens.size() < 2 || tokens[1] != "A")
		throw Exception(tr("The file %1 is not an IMD atom file in ASCII format.").arg(filename));

	ColumnChannelMapping columnMapping;
	Vector3 cellVectors[3];

#ifdef USE_DOUBLE_PRECISION_FP
	#define FLOAT_SCANF_STRING_3   "%lg %lg %lg"
#else
	#define FLOAT_SCANF_STRING_3   "%g %g %g"
#endif

	// Read remaining header lines
	while(true) {
		stream.readline();
		if(stream.line().empty() || stream.line().at(0) != '#') throw Exception(tr("Invalid header in IMD atom file %1 (line %2).").arg(filename).arg(stream.lineNumber()));
		if(stream.line().at(1) == '#') continue;
		else if(stream.line().at(1) == 'E') break;
		else if(stream.line().at(1) == 'C') {
			QStringList tokens = QString(stream.line().c_str()).split(QRegExp("\\s+"));
			int columnIndex = 0;
			for(int t=1; t<tokens.size(); t++) {
				const QString& token = tokens[t];
				if(token == "mass") columnMapping.defineStandardColumn(columnIndex, DataChannel::MassChannel);
				else if(token == "type") columnMapping.defineStandardColumn(columnIndex, DataChannel::AtomTypeChannel);
				//else if(token == "number") columnMapping.defineStandardColumn(columnIndex, DataChannel::AtomIndexChannel);
				else if(token == "x") columnMapping.defineStandardColumn(columnIndex, DataChannel::PositionChannel, 0);
				else if(token == "y") columnMapping.defineStandardColumn(columnIndex, DataChannel::PositionChannel, 1);
				else if(token == "z") columnMapping.defineStandardColumn(columnIndex, DataChannel::PositionChannel, 2);
				else if(token == "vx") columnMapping.defineStandardColumn(columnIndex, DataChannel::VelocityChannel, 0);
				else if(token == "vy") columnMapping.defineStandardColumn(columnIndex, DataChannel::VelocityChannel, 1);
				else if(token == "vz") columnMapping.defineStandardColumn(columnIndex, DataChannel::VelocityChannel, 2);
				else if(token == "Epot") columnMapping.defineStandardColumn(columnIndex, DataChannel::PotentialEnergyChannel);
				else {
					bool isStandardChannel = false;
					QMap<QString, DataChannel::DataChannelIdentifier> standardChannelList = DataChannel::standardChannelList();
					Q_FOREACH(DataChannel::DataChannelIdentifier id, standardChannelList) {
						for(size_t component = 0; component < DataChannel::standardChannelComponentCount(id); component++) {
							QString columnName = DataChannel::standardChannelName(id);
							columnName.remove(QRegExp("[^A-Za-z\\d_]"));
							QStringList componentNames = DataChannel::standardChannelComponentNames(id);
							if(!componentNames.empty()) {
								QString componentName = componentNames[component];
								componentName.remove(QRegExp("[^A-Za-z\\d_]"));
								columnName += componentName;
							}
							if(columnName == token) {
								columnMapping.defineStandardColumn(columnIndex, id, component);
								isStandardChannel = true;
								break;
							}
						}
						if(isStandardChannel) break;
					}
					if(!isStandardChannel) columnMapping.defineColumn(columnIndex, DataChannel::UserDataChannel, QString(token), qMetaTypeId<FloatType>());
				}
				columnIndex++;
			}
		}
		else if(stream.line().at(1) == 'X') {
			if(sscanf(stream.line().c_str()+2, FLOAT_SCANF_STRING_3, &cellVectors[0].X, &cellVectors[0].Y, &cellVectors[0].Z) != 3)
				throw Exception(tr("Invalid simulation cell bounds in line %1 of IMD file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
		}
		else if(stream.line().at(1) == 'Y') {
			if(sscanf(stream.line().c_str()+2, FLOAT_SCANF_STRING_3, &cellVectors[1].X, &cellVectors[1].Y, &cellVectors[1].Z) != 3)
				throw Exception(tr("Invalid simulation cell bounds in line %1 of IMD file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
		}
		else if(stream.line().at(1) == 'Z') {
			if(sscanf(stream.line().c_str()+2, FLOAT_SCANF_STRING_3, &cellVectors[2].X, &cellVectors[2].Y, &cellVectors[2].Z) != 3)
				throw Exception(tr("Invalid simulation cell bounds in line %1 of IMD file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
		}
		else throw Exception(tr("Invalid header line key in IMD atom file %1 (line %2).").arg(filename).arg(stream.lineNumber()));
	}
	destination->simulationCell()->setCellShape(ORIGIN, cellVectors[0], cellVectors[1], cellVectors[2]);

	// Save file position.
	streampos headerOffset = stream.byteOffset();

	// Count the number of atoms (lines) in the input file.
	int numAtoms = 0;
	while(!stream.eof()) {
		stream.readline();
		if(stream.line().empty()) break;
		numAtoms++;
	}
	stream.clearError();

	progress.setLabelText(tr("Loading IMD atoms file (%1 atoms)").arg(numAtoms));
	progress.setMaximum(numAtoms);

	// Jump back to beginning of atom list.
	stream.seek(headerOffset);

	// Resize destination atoms array.
	destination->setAtomsCount(numAtoms);

	// Prepare the mapping between input file columns and data channels.
	DataRecordParserHelper recordParser(&columnMapping, destination);

	// Parse one atom per line.
	for(int i = 0; i < numAtoms; i++) {

		// Update progress indicator.
		if((i % 1000) == 0) {
			progress.setValue(i);
			if(progress.isCanceled()) return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
		}

		stream.readline();
		try {
			recordParser.storeAtom(i, (char*)stream.line().c_str());
		}
		catch(Exception& ex) {
			throw ex.prependGeneralMessage(tr("Parsing error in line %1 of IMD atoms file.").arg(stream.lineNumber() - numAtoms));
		}
	}

	destination->invalidate();

	QString statusMessage = tr("Number of atoms: %1").arg(numAtoms);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, statusMessage);
}

};	// End of namespace AtomViz
