///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/IntegerPropertyUI.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/AnimManager.h>
#include <core/data/units/ParameterUnit.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(IntegerPropertyUI, NumericalPropertyUI)

/******************************************************************************
* Constructor for a Qt property.
******************************************************************************/
IntegerPropertyUI::IntegerPropertyUI(PropertiesEditor* parentEditor, const char* propertyName, const QString& labelText, ParameterUnit* parameterUnit) :  
	NumericalPropertyUI(parentEditor, propertyName, parameterUnit ? parameterUnit : UNITS_MANAGER.integerIdentity(), labelText)
{
}

/******************************************************************************
* Constructor for a PropertyField property.
******************************************************************************/
IntegerPropertyUI::IntegerPropertyUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField) :  
	NumericalPropertyUI(parentEditor, propField, UNITS_MANAGER.integerIdentity())
{
}

/******************************************************************************
* Takes the value entered by the user and stores it in the property field 
* this property UI is bound to.
******************************************************************************/
void IntegerPropertyUI::updatePropertyValue()
{
	if(editObject() && spinner()) {
		if(propertyName()) {
			if(!editObject()->setProperty(propertyName(), spinner()->intValue())) {
				OVITO_ASSERT_MSG(false, "IntegerPropertyUI::updatePropertyValue()", QString("The value of property %1 of object class %2 could not be set.").arg(QString(propertyName()), editObject()->metaObject()->className()).toLocal8Bit().constData());
			}
		}
		else if(propertyField()) {
			editObject()->setPropertyFieldValue(*propertyField(), spinner()->intValue());						
		}
	}
}

/******************************************************************************
* This method updates the displayed value of the parameter UI.
******************************************************************************/
void IntegerPropertyUI::updateUI()
{
	if(editObject() && spinner() && !spinner()->isDragging()) {
		QVariant val(0);
		if(propertyName()) {
			val = editObject()->property(propertyName());
			OVITO_ASSERT_MSG(val.isValid() && val.canConvert(QVariant::Int), "IntegerPropertyUI::updateUI()", QString("The object class %1 does not define a property with the name %2 that can be cast to integer type.").arg(editObject()->metaObject()->className(), QString(propertyName())).toLocal8Bit().constData());
			if(!val.isValid() || !val.canConvert(QVariant::Int)) {
				throw Exception(tr("The object class %1 does not define a property with the name %2 that can be cast to integer type.").arg(editObject()->metaObject()->className(), QString(propertyName())));
			}
		}
		else if(propertyField()) {
			val = editObject()->getPropertyFieldValue(*propertyField());
			OVITO_ASSERT(val.isValid());
		}
		spinner()->setIntValue(val.toInt());
	}
}

/******************************************************************************
* Gets the minimum value to be entered.
* This value is in native controller units.
******************************************************************************/
int IntegerPropertyUI::minValue() const
{
	return (spinner() ? (int)spinner()->minValue() : numeric_limits<int>::min());
}

/******************************************************************************
* Sets the minimum value to be entered.
* This value must be specified in native controller units.
******************************************************************************/
void IntegerPropertyUI::setMinValue(int minValue)
{
	if(spinner()) spinner()->setMinValue(minValue);
}

/******************************************************************************
* Gets the maximum value to be entered.
* This value is in native controller units.
******************************************************************************/
int IntegerPropertyUI::maxValue() const
{
	return (spinner() ? (int)spinner()->maxValue() : numeric_limits<int>::max());
}

/******************************************************************************
* Sets the maximum value to be entered.
* This value must be specified in native controller units.
******************************************************************************/
void IntegerPropertyUI::setMaxValue(int maxValue)
{
	if(spinner()) spinner()->setMaxValue(maxValue);
}

};

