///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file GroupNode.h 
 * \brief Contains the definition of the Core::GroupNode class. 
 */
 
#ifndef __OVITO_GROUPNODE_H
#define __OVITO_GROUPNODE_H

#include <core/Core.h>
#include "SceneNode.h"

namespace Core {

/**
 * \brief Scene node that groups all child nodes together.
 * 
 * \author Alexander Stukowski
 * \sa ObjectNode
 */
class CORE_DLLEXPORT GroupNode : public SceneNode
{
public:

	/// \brief Constructs an empty group node that is in the closed state.
	/// \param isLoading Indicates whether the object is being loaded from a file. 
	///                  This parameter is only used by the object serialization system.
	GroupNode(bool isLoading = false);

	/// \brief Returns whether this is an instance of the GroupNode class.
	/// \return Returns always \c true.
	virtual bool isGroupNode() const { return true; }

	/// \brief Returns whether this group is currently open.
	/// \return \c true if the group is in the open state.
	/// 
	/// When the group is open then child nodes the group can be edited individually.
	/// \sa setGroupOpen()
	bool isGroupOpen() const { return _isGroupOpen; }

	/// \brief Opens the group for editing or closes it.
	/// \param open Controls whether the group should be opened or closed.
	/// \undoable
	/// \sa isGroupOpen()
	void setGroupOpen(bool open) { _isGroupOpen = open; }

	/// \brief Returns the bounding box of the group.
	/// \param time The time at which the bounding box should be computed.
	/// \return An axis-aligned box in the node's local coordinate system that contains
	///         the bounding boxes of all child nodes.
	virtual Box3 localBoundingBox(TimeTicks time);
		
public:

	Q_PROPERTY(bool isGroupOpen READ isGroupOpen WRITE setGroupOpen)

protected:

	/// Is called when a reference target has been removed from a list reference field of this RefMaker.
	virtual void onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex);

private:

	/// Indicates if this group of nodes is currently open.
	PropertyField<bool, bool, GROUPNODE_OPENED_OR_CLOSED> _isGroupOpen;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(GroupNode)
	DECLARE_PROPERTY_FIELD(_isGroupOpen)
};

};

#endif // __OVITO_GROUPNODE_H
