/***************************************************************************
    smb4ksudowriterinterface  -  This class provides an interface to the
    smb4k_sudowriter utility program that writes entries to the sudoers
    file.
                             -------------------
    begin                : Sa Aug 2 2008
    copyright            : (C) 2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// KDE includes
#include <kglobal.h>

// system includes
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>

// application specific includes
#include <smb4ksudowriterinterface.h>
#include <smb4kcoremessage.h>
#include <smb4ksettings.h>

class Smb4KSudoWriterInterfacePrivate
{
  public:
    /**
     * The Smb4KSudoWriterInterface instance
     */
    Smb4KSudoWriterInterface instance;
};

K_GLOBAL_STATIC( Smb4KSudoWriterInterfacePrivate, priv );


Smb4KSudoWriterInterface::Smb4KSudoWriterInterface()
: QObject()
{
  m_proc = new KProcess( this );
  m_process_error = (QProcess::ProcessError)(-1);
  m_operation = Unknown;

  connect( m_proc, SIGNAL( finished( int, QProcess::ExitStatus ) ),
           this,   SLOT( slotProcessFinished( int, QProcess::ExitStatus ) ) );

  connect( m_proc, SIGNAL( error( QProcess::ProcessError ) ),
           this,   SLOT( slotProcessError( QProcess::ProcessError ) ) );
}


Smb4KSudoWriterInterface::~Smb4KSudoWriterInterface()
{
}


Smb4KSudoWriterInterface *Smb4KSudoWriterInterface::self()
{
  return &priv->instance;
}


void Smb4KSudoWriterInterface::addUser()
{
  // Get the name of the current user.
  uid_t user_id = getuid();
  struct passwd *pw = getpwuid( user_id );
  QString user( pw->pw_name );

  m_operation = AddUser;

  m_proc->setShellCommand( Smb4KSettings::kdesu()+" -t -c 'smb4k_sudowriter --adduser="+user+"'" );
  m_proc->setOutputChannelMode( KProcess::SeparateChannels );
  m_proc->start();
}


void Smb4KSudoWriterInterface::removeUser()
{
  // Get the name of the current user.
  uid_t user_id = getuid();
  struct passwd *pw = getpwuid( user_id );
  QString user( pw->pw_name );

  m_operation = RemoveUser;

  m_proc->setShellCommand( Smb4KSettings::kdesu()+" -t -c 'smb4k_sudowriter --removeuser="+user+"'" );
  m_proc->setOutputChannelMode( KProcess::SeparateChannels );
  m_proc->start();
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KSudoWriterInterface::slotProcessFinished( int /*exitCode*/, QProcess::ExitStatus exitStatus )
{
  if ( exitStatus == QProcess::NormalExit )
  {
    // smb4k_sudowriter won't generate any output if it successfully could
    // write to the sudoers file. So, just check stderr for output.

    QString stderr_output = QString::fromLocal8Bit( m_proc->readAllStandardError(), -1 ).trimmed();

    // Only report an error if smb4k_sudowriter complained.
    // Note: If kdesu is not installed or could not be found, Smb4K won't 
    // start up, so we do not need to test for it here.
    if ( !stderr_output.isEmpty() && stderr_output.contains( "smb4k_sudowriter" ) )
    {
      Smb4KCoreMessage::error( ERROR_SUDOWRITER, QString(), stderr_output );
      emit failed( m_operation );
    }
    else
    {
      // Do nothing
    }

    emit finished( m_operation );
  }
  else
  {
    // Something went wrong. Through an error.
    if ( m_process_error != -1 )
    {
      Smb4KCoreMessage::processError( ERROR_PROCESS_ERROR, m_process_error );
    }
    else
    {
      Smb4KCoreMessage::processError( ERROR_PROCESS_EXIT, m_process_error );
    }

    emit failed( m_operation );
    emit finished( m_operation );
  }

  m_operation = Unknown;
  m_proc->clearProgram();
  m_process_error = (QProcess::ProcessError)(-1);
}


void Smb4KSudoWriterInterface::slotProcessError( QProcess::ProcessError errorCode )
{
  m_process_error = errorCode;
}

#include "smb4ksudowriterinterface.moc"
