/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#ifndef SYNAPTIKSAPPLET_H
#define SYNAPTIKSAPPLET_H

/**
 * @file
 *
 * Provides the SynaptiksApplet.
 */

#include <Plasma/Applet>
#include <Plasma/Svg>
#include <QtDBus/QDBusInterface>


namespace Plasma {
    class ConfigLoader;
}



namespace synaptiks {
    /**
     * @brief @b synaptiks plasma applet.
     *
     * This class provides a simple plasma applet to show the touchpad state
     * and access touchpad configuration conveniently.
     */
    class SynaptiksApplet: public Plasma::Applet {
        Q_OBJECT

    public:
        /**
         * @brief Create a new applet.
         *
         * @param parent the parent widget
         * @param args required by KPlugin signature, ignored
         */
        SynaptiksApplet(QObject *parent, const QVariantList &args);

        /**
         * @brief Destroy this applet.
         */
        ~SynaptiksApplet();

        /**
         * @brief Called to draw the applet contents.
         *
         * If invoked, this applet draws itself using the given @p painter
         * within the given boundaries (@p contentsRect).  If the touchpad
         * is on, the touchpad icon is drawn unmodified.  If the touchpad is
         * off, a red cross will appear on the top of the icon.  If the
         * touchpad is not available, a dark cross will appear, and the
         * whole icon will be grayed out.
         *
         * @param painter the painter to use for drawing
         * @param option ignored (required by Plasma::Applet signature)
         * @param contentsRect the area to draw within
         */
        virtual void paintInterface(QPainter *painter,
                                    const QStyleOptionGraphicsItem *option,
                                    const QRect &contentsRect);

        /**
         * @brief Called to create a configuration interface within the
         * given @p parent dialog.
         *
         * If invoked, this applet will add its configuration pages to the
         * given dialog.
         *
         * @param parent the dialog to add the interface to
         */
        virtual void createConfigurationInterface(KConfigDialog *parent);

        /**
         * @brief Called to perform applet initialization.
         *
         * If invoked, this applet will load its configuration, setup its
         * actions and test the connectivity with the synaptiks daemon.
         *
         * If the connectivity check fails, a error message will be shown,
         * and the applet will be unusable.
         */
        virtual void init();

    protected:
        /**
         * @brief Called on double clicks into the applet area.
         *
         * Depending on the applet configuration, this function either shows
         * the touchpad configuration module or switches the touchpad.
         *
         * @param event the event (ignored)
         */
        virtual void mouseDoubleClickEvent(QGraphicsSceneMouseEvent *event);

    private Q_SLOTS:
        /**
         * @brief Called, if the touchpad was switched.
         *
         * Sets an informative tooltip and calls update() to issue a
         * repaint.
         *
         * @param on is the touchpad on?
         * @param reason the reason for the touchpad switch
         * @param closure additional information about this switch
         */
        void touchpadSwitched(bool on, const QString &reason,
                              const QDBusVariant &closure);

        /**
         * @brief Show the touchpad configuration module in a separate
         * dialog.
         */
        void showTouchpadConfiguration() const;

        /**
         * @brief Switch the touchpad on, if its off, or off otherwise.
         */
        void switchTouchpad();

    private:
        /**
         * @brief Show the given touchpad @p error.
         *
         * The touchpad contents will remain unmodified, but the tooltip
         * will contain the error message.
         */
        void showTouchpadError(const QDBusError &error);

        /** The SVG image of this applet */
        Plasma::Svg m_svg;

        /** The touchpad manager object from the synaptiks daemon */
        QPointer<QDBusInterface> touchpadManager;

        /** The applet configuration */
        QPointer<Plasma::ConfigLoader> m_config;
    };

}
#endif /* SYNAPTIKSAPPLET_H */
