/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include "touchpad.h"
#include "touchpadadaptor.h"
#include <KDebug>
#include <KLocalizedString>
#include <algorithm>


static const char OFF[] = "Synaptics Off";
static const char MOVE_SPEED[] = "Synaptics Move Speed";
static const char EDGE_MOTION_ALWAYS[] = "Synaptics Edge Motion Always";
static const char GESTURES[] = "Synaptics Gestures";
static const char LOCKED_DRAGS[] = "Synaptics Locked Drags";
static const char LOCKED_DRAGS_TIMEOUT[] = "Synaptics Locked Drags Timeout";
static const char CIRCULAR_SCROLLING[] = "Synaptics Circular Scrolling";
static const char CIRCULAR_SCROLLING_TRIGGER[] =
    "Synaptics Circular Scrolling Trigger";
static const char CIRCULAR_SCROLLING_DISTANCE[] =
    "Synaptics Circular Scrolling Distance";
static const char TWO_FINGER_SCROLLING[] = "Synaptics Two-Finger Scrolling";
static const char EDGE_SCROLLING[] = "Synaptics Edge Scrolling";
static const char COASTING_SPEED[] = "Synaptics Coasting Speed";
static const char SCROLLING_DISTANCE[] = "Synaptics Scrolling Distance";
static const char TAP_ACTION[] = "Synaptics Tap Action";
static const char FAST_TAP[] = "Synaptics Tap FastTap";
static const char CIRCULAR_PAD[] = "Synaptics Circular Pad";
static const char CAPABILITIES[] = "Synaptics Capabilities";

enum MoveSpeedItem {
    MinimumItem = 0,
    MaximumItem = 1,
    AccelerationItem = 2,
};

enum EdgeSrollingItem {
    VerticalEdgeItem = 0,
    HorizontalEdgeItem = 1,
    CornerCoastingItem = 2
};

enum TapActionItem {
    RightTopItem = 0,
    RightBottomItem = 1,
    LeftTopItem = 2,
    LeftBottomItem = 3
};

enum TwoFingerScrollingItem {
    VerticalScrollingItem = 0,
    HorizontalScrollingItem = 1
};

enum ScrollingDistanceItem {
    VerticalDistanceItem = 0,
    HorizontalDistanceItem = 1,
};


using namespace synaptiks;


namespace synaptiks {
    class TouchpadPrivate {
    public:
        QSharedPointer<QXDevice> device;
    };
}

bool Touchpad::isSupported() {
    return (QXDevice::isSupported() && QXDevice::isPropertyDefined(OFF));
}

Touchpad *Touchpad::findTouchpad(QObject *parent) {
    if (!Touchpad::isSupported()) {
        kWarning() << "device properties unsupported";
        return 0;
    }
    QXDevice::List devices = QXDevice::findDevicesWithProperty(OFF);
    if (devices.empty()) {
        kWarning() << "no touchpad found";
        return 0;
    } else {
        if (devices.size() > 1) {
            kWarning() << "multiple matching devices found, taking first";
        }
        return new Touchpad(devices.at(0), parent);
    }
}

Touchpad::Touchpad(QSharedPointer<QXDevice> device, QObject *parent):
    QObject(parent), d_ptr(new TouchpadPrivate) {
    Q_D(Touchpad);
    d->device = device;
    new TouchpadAdaptor(this);
}

Touchpad::~Touchpad() {
    delete this->d_ptr;
}

QString Touchpad::name() const {
    Q_D(const Touchpad);
    return QString::fromLocal8Bit(d->device->name());
}

bool Touchpad::isOn() const {
    Q_D(const Touchpad);
    return !d->device->property<bool>(OFF, 0);
}

void Touchpad::setOn(bool on) {
    Q_D(Touchpad);
    d->device->setProperty(OFF, !on);
}

float Touchpad::minimumSpeed() const {
    Q_D(const Touchpad);
    return d->device->property<float>(MOVE_SPEED, MinimumItem);
}

void Touchpad::setMinimumSpeed(float speed) {
    Q_D(Touchpad);
    d->device->setProperty(MOVE_SPEED, speed, MinimumItem);
}

float Touchpad::maximumSpeed() const {
    Q_D(const Touchpad);
    return d->device->property<float>(MOVE_SPEED, MaximumItem);
}

void Touchpad::setMaximumSpeed(float speed) {
    Q_D(Touchpad);
    d->device->setProperty(MOVE_SPEED, speed, MaximumItem);
}

float Touchpad::accelerationFactor() const {
    Q_D(const Touchpad);
    return d->device->property<float>(MOVE_SPEED, AccelerationItem);
}

void Touchpad::setAccelerationFactor(float accel) {
    Q_D(Touchpad);
    d->device->setProperty(MOVE_SPEED, accel, AccelerationItem);
}

bool Touchpad::edgeMotionAlways() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(EDGE_MOTION_ALWAYS, 0);
}

void Touchpad::setEdgeMotionAlways(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(EDGE_MOTION_ALWAYS, enabled);
}

bool Touchpad::tapAndDragGesture() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(GESTURES, 0);
}

void Touchpad::setTapAndDragGesture(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(GESTURES, enabled);
}

bool Touchpad::lockedDrags() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(LOCKED_DRAGS, 0);
}

void Touchpad::setLockedDrags(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(LOCKED_DRAGS, enabled);
}

int Touchpad::lockedDragsTimeout() const {
    Q_D(const Touchpad);
    return d->device->property<int>(LOCKED_DRAGS_TIMEOUT, 0);
}

void Touchpad::setLockedDragsTimeout(int timeout) {
    Q_D(Touchpad);
    d->device->setProperty(LOCKED_DRAGS_TIMEOUT, timeout);
}

bool Touchpad::circularScrolling() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(CIRCULAR_SCROLLING, 0);
}

void Touchpad::setCircularScrolling(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(CIRCULAR_SCROLLING, enabled);
}

Touchpad::CircularScrollingTrigger Touchpad::circularScrollingTrigger() const {
    Q_D(const Touchpad);
    return static_cast<CircularScrollingTrigger>(
        d->device->property<uchar>(CIRCULAR_SCROLLING_TRIGGER, 0));
}

float Touchpad::circularScrollingDistance() const {
    Q_D(const Touchpad);
    return d->device->property<float>(CIRCULAR_SCROLLING_DISTANCE, 0);
}

void Touchpad::setCircularScrollingDistance(float angle) {
    Q_D(Touchpad);
    d->device->setProperty(CIRCULAR_SCROLLING_DISTANCE, angle);
}

bool Touchpad::horizontalTwoFingerScrolling() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(
        TWO_FINGER_SCROLLING, HorizontalScrollingItem);
}

void Touchpad::setHorizontalTwoFingerScrolling(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(TWO_FINGER_SCROLLING, enabled,
                           HorizontalScrollingItem);
}

bool Touchpad::verticalTwoFingerScrolling() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(
        TWO_FINGER_SCROLLING, VerticalScrollingItem);
}

void Touchpad::setVerticalTwoFingerScrolling(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(TWO_FINGER_SCROLLING, enabled,
                           VerticalScrollingItem);
}

void Touchpad::setCircularScrollingTrigger(
    CircularScrollingTrigger trigger) {
    Q_D(Touchpad);
    d->device->setProperty(CIRCULAR_SCROLLING_TRIGGER,
                           static_cast<uchar>(trigger), 0);
}

bool Touchpad::horizontalEdgeScrolling() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(EDGE_SCROLLING, HorizontalEdgeItem);
}

void Touchpad::setHorizontalEdgeScrolling(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(EDGE_SCROLLING, enabled, HorizontalEdgeItem);
}

bool Touchpad::verticalEdgeScrolling() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(EDGE_SCROLLING, VerticalEdgeItem);
}

void Touchpad::setVerticalEdgeScrolling(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(EDGE_SCROLLING, enabled, VerticalEdgeItem);
}

bool Touchpad::cornerCoasting() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(EDGE_SCROLLING, CornerCoastingItem);
}

void Touchpad::setCornerCoasting(bool enabled) {
    Q_D(Touchpad);
    return d->device->setProperty(EDGE_SCROLLING, enabled,
                                  CornerCoastingItem);
}

float Touchpad::coastingSpeed() const {
    Q_D(const Touchpad);
    return d->device->property<float>(COASTING_SPEED, 0);
}

void Touchpad::setCoastingSpeed(float speed) {
    Q_D(Touchpad);
    return d->device->setProperty(COASTING_SPEED, speed);
}

int Touchpad::horizontalScrollingDistance() const {
    Q_D(const Touchpad);
    return d->device->property<int>(SCROLLING_DISTANCE,
                                    HorizontalDistanceItem);
}

void Touchpad::setHorizontalScrollingDistance(int distance) {
    Q_D(Touchpad);
    d->device->setProperty<int>(SCROLLING_DISTANCE, distance,
                                HorizontalDistanceItem);
}

int Touchpad::verticalScrollingDistance() const {
    Q_D(const Touchpad);
    return d->device->property<int>(SCROLLING_DISTANCE,
                                    VerticalDistanceItem);
}

void Touchpad::setVerticalScrollingDistance(int distance) {
    Q_D(Touchpad);
    d->device->setProperty<int>(SCROLLING_DISTANCE, distance,
                                VerticalDistanceItem);
}

QByteArray Touchpad::cornerButtons() const {
    Q_D(const Touchpad);
    QByteArray buttons;
    QList<uchar> values = d->device->property<uchar>(TAP_ACTION);
    std::copy(values.begin(), values.begin()+4,
              std::back_inserter(buttons));
    return buttons;
}

void Touchpad::setCornerButtons(const QByteArray &buttons) {
    Q_D(Touchpad);
    Q_ASSERT(buttons.size() >= 4);
    QList<uchar> values = d->device->property<uchar>(TAP_ACTION);
    std::copy(buttons.begin(), buttons.begin()+4, values.begin());
    d->device->setProperty(TAP_ACTION, values);
}

QByteArray Touchpad::fingerButtons() const {
    Q_D(const Touchpad);
    QByteArray buttons;
    QList<uchar> values = d->device->property<uchar>(TAP_ACTION);
    std::copy(values.begin()+4, values.begin()+7,
              std::back_inserter(buttons));
    return buttons;
}

void Touchpad::setFingerButtons(const QByteArray &buttons) {
    Q_D(Touchpad);
    Q_ASSERT(buttons.size() >= 3);
    QList<uchar> values = d->device->property<uchar>(TAP_ACTION);
    std::copy(buttons.begin(), buttons.begin()+3, values.begin()+4);
    return d->device->setProperty(TAP_ACTION, values);
}

bool Touchpad::fastTaps() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(FAST_TAP, 0);
}

void Touchpad::setFastTaps(bool enabled) {
    Q_D(Touchpad);
    d->device->setProperty(FAST_TAP, enabled);
}

bool Touchpad::circularTouchpad() const {
    Q_D(const Touchpad);
    return d->device->property<bool>(CIRCULAR_PAD, 0);
}

void Touchpad::setCircularTouchpad(bool circular) {
    Q_D(Touchpad);
    d->device->setProperty(CIRCULAR_PAD, circular);
}

Touchpad::Capabilities Touchpad::capabilities() const {
    Q_D(const Touchpad);
    Capabilities current_capabilities;
    QList<bool> values = d->device->property<bool>(CAPABILITIES);
    if (values.length() < 5) {
        return current_capabilities;
    }
    Capability capabilities[] = {
        LeftButtonCapability, MiddleButtonCapability,
        RightButtonCapability,
        TwoFingersCapability, ThreeFingersCapability};
    for (int i=0; i<5; i++) {
        if (values.at(i)) {
            current_capabilities |= capabilities[i];
        }
    }
    return current_capabilities;
}

bool Touchpad::hasLeftButton() const {
    return this->capabilities().testFlag(LeftButtonCapability);
}

bool Touchpad::hasMiddleButton() const {
    return this->capabilities().testFlag(MiddleButtonCapability);
}

bool Touchpad::hasRightButton() const {
    return this->capabilities().testFlag(RightButtonCapability);
}

int Touchpad::fingerDetection() const {
    Capabilities capabilities = this->capabilities();
    if (capabilities.testFlag(ThreeFingersCapability))
        return 3;
    else if (capabilities.testFlag(TwoFingersCapability))
        return 2;
    else
        return 1;
}

#include "moc_touchpad.cpp"
