% $Id: mp.w 2037 2014-09-02 14:59:07Z luigi $
%
% This file is part of MetaPost;
% the MetaPost program is in the public domain.
% See the <Show version...> code in mpost.w for more info. 

% Here is TeX material that gets inserted after \input webmac
\def\hang{\hangindent 3em\noindent\ignorespaces}
\def\textindent#1{\hangindent2.5em\noindent\hbox to2.5em{\hss#1 }\ignorespaces}
\def\ps{PostScript}
\def\psqrt#1{\sqrt{\mathstrut#1}}
\def\k{_{k+1}}
\def\pct!{{\char`\%}} % percent sign in ordinary text
\font\tenlogo=logo10 % font used for the METAFONT logo
\font\logos=logosl10
\def\MF{{\tenlogo META}\-{\tenlogo FONT}}
\def\MP{{\tenlogo META}\-{\tenlogo POST}}
\def\<#1>{$\langle#1\rangle$}
\def\section{\mathhexbox278}
\let\swap=\leftrightarrow
\def\round{\mathop{\rm round}\nolimits}
\mathchardef\vbv="026A % synonym for `\|'
\def\vb{\relax\ifmmode\vbv\else$\vbv$\fi}

\def\(#1){} % this is used to make section names sort themselves better
\def\9#1{} % this is used for sort keys in the index via @@:sort key}{entry@@>
\def\title{MetaPost}
\pdfoutput=1
\pageno=3

@* Introduction.

This is \MP\ by John Hobby, a graphics-language processor based on D. E. Knuth's \MF.

Much of the original Pascal version of this program was copied with
permission from MF.web Version 1.9. It interprets a language very
similar to D.E. Knuth's METAFONT, but with changes designed to make it
more suitable for PostScript output.

The main purpose of the following program is to explain the algorithms of \MP\
as clearly as possible. However, the program has been written so that it
can be tuned to run efficiently in a wide variety of operating environments
by making comparatively few changes. Such flexibility is possible because
the documentation that follows is written in the \.{WEB} language, which is
at a higher level than C.

A large piece of software like \MP\ has inherent complexity that cannot
be reduced below a certain level of difficulty, although each individual
part is fairly simple by itself. The \.{WEB} language is intended to make
the algorithms as readable as possible, by reflecting the way the
individual program pieces fit together and by providing the
cross-references that connect different parts. Detailed comments about
what is going on, and about why things were done in certain ways, have
been liberally sprinkled throughout the program.  These comments explain
features of the implementation, but they rarely attempt to explain the
\MP\ language itself, since the reader is supposed to be familiar with
{\sl The {\logos METAFONT\/}book} as well as the manual
@.WEB@>
@:METAFONTbook}{\sl The {\logos METAFONT\/}book@>
{\sl A User's Manual for MetaPost}, Computing Science Technical Report 162,
AT\AM T Bell Laboratories.

@ The present implementation is a preliminary version, but the possibilities
for new features are limited by the desire to remain as nearly compatible
with \MF\ as possible.

On the other hand, the \.{WEB} description can be extended without changing
the core of the program, and it has been designed so that such
extensions are not extremely difficult to make.
The |banner| string defined here should be changed whenever \MP\
undergoes any modifications, so that it will be clear which version of
\MP\ might be the guilty party when a problem arises.
@^extensions to \MP@>
@^system dependencies@>

@d default_banner "This is MetaPost, Version 1.999" /* printed when \MP\ starts */
@d true 1
@d false 0

@<Metapost version header@>=
#define metapost_version "1.999"

@ The external library header for \MP\ is |mplib.h|. It contains a
few typedefs and the header defintions for the externally used
fuctions.

The most important of the typedefs is the definition of the structure 
|MP_options|, that acts as a small, configurable front-end to the fairly 
large |MP_instance| structure.
 
@(mplib.h@>=
#ifndef MPLIB_H
#define MPLIB_H 1
#include <stdlib.h>
#ifndef HAVE_BOOLEAN
typedef int boolean;
#endif
@<Metapost version header@>
typedef struct MP_instance *MP;
@<Exported types@>
typedef struct MP_options {
  @<Option variables@>
} MP_options;
@<Exported function headers@>
@<MPlib header stuff@>
#endif 

@ The internal header file is much longer: it not only lists the complete
|MP_instance|, but also a lot of functions that have to be available to
the \ps\ backend, that is defined in a separate \.{WEB} file. 

The variables from |MP_options| are included inside the |MP_instance| 
wholesale.

@(mpmp.h@>=
#ifndef MPMP_H
#define MPMP_H 1
#include "avl.h"
#include "mplib.h"
#include <setjmp.h>
typedef struct psout_data_struct *psout_data;
typedef struct svgout_data_struct *svgout_data;
typedef struct pngout_data_struct *pngout_data;
#ifndef HAVE_BOOLEAN
typedef int boolean;
#endif
#ifndef INTEGER_TYPE
typedef int integer;
#endif
@<Declare helpers@>;
@<Enumeration types@>;
@<Types in the outer block@>;
@<Constants in the outer block@>;
typedef struct MP_instance {
  @<Option variables@>
  @<Global variables@>
} MP_instance;
@<Internal library declarations@>
@<MPlib internal header stuff@>
#endif

@ @c
#define KPATHSEA_DEBUG_H 1
#include <w2c/config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>
#include <math.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>           /* for access */
#endif
#include <time.h>               /* for struct tm \& co */
#include <zlib.h>               /* for |ZLIB_VERSION|, zlibVersion() */
#include <png.h>                /* for |PNG_LIBPNG_VER_STRING|, |png_libpng_ver| */
#include <pixman.h>             /* for |PIXMAN_VERSION_STRING|, |pixman_version_string()| */
#include <cairo.h>              /* for |CAIRO_VERSION_STRING|, |cairo_version_string()| */
#include <gmp.h>                /* for |gmp_version| */
#include <mpfr.h>               /* for |MPFR_VERSION_STRING|, |mpfr_get_version()| */
#include "mplib.h"
#include "mplibps.h"            /* external header */
#include "mplibsvg.h"           /* external header */
#include "mplibpng.h"           /* external header */
#include "mpmp.h"               /* internal header */
#include "mppsout.h"            /* internal header */
#include "mpsvgout.h"           /* internal header */
#include "mppngout.h"           /* internal header */
#include "mpmath.h"             /* internal header */
#include "mpmathdouble.h"       /* internal header */
#include "mpmathdecimal.h"      /* internal header */
#include "mpmathbinary.h"       /* internal header */
#include "mpstrings.h"          /* internal header */
extern font_number mp_read_font_info (MP mp, char *fname);      /* tfmin.w */
@h @<Declarations@>;
@<Basic printing procedures@>;
@<Error handling procedures@>
 
@ Some debugging support for development. The trick with the variadic macros
probably only works in gcc, as this preprocessor feature was not formalized 
until the c99 standard (and that is too new for us). Lets' hope that at least
most compilers understand the non-debug version.
@^system dependencies@>

@<MPlib internal header stuff@>=
#define DEBUG 0
#if DEBUG
#define debug_number(A) printf("%d: %s=%.32f (%d)\n", __LINE__, #A, number_to_double(A), number_to_scaled(A))
#else
#define debug_number(A)
#endif
#if DEBUG>1
void do_debug_printf(MP mp, const char *prefix, const char *fmt, ...);
#  define debug_printf(a1,a2,a3) do_debug_printf(mp, "", a1,a2,a3)
#  define FUNCTION_TRACE1(a1) do_debug_printf(mp, "FTRACE: ", a1)
#  define FUNCTION_TRACE2(a1,a2) do_debug_printf(mp, "FTRACE: ", a1,a2)
#  define FUNCTION_TRACE3(a1,a2,a3) do_debug_printf(mp, "FTRACE: ", a1,a2,a3)
#  define FUNCTION_TRACE3X(a1,a2,a3) (void)mp
#  define FUNCTION_TRACE4(a1,a2,a3,a4) do_debug_printf(mp, "FTRACE: ", a1,a2,a3,a4)
#else
#  define debug_printf(a1,a2,a3)
#  define FUNCTION_TRACE1(a1) (void)mp
#  define FUNCTION_TRACE2(a1,a2) (void)mp
#  define FUNCTION_TRACE3(a1,a2,a3) (void)mp
#  define FUNCTION_TRACE3X(a1,a2,a3) (void)mp
#  define FUNCTION_TRACE4(a1,a2,a3,a4) (void)mp
#endif

@ This function occasionally crashes (if something is written after the
log file is already closed), but that is not so important while debugging.

@c
#if DEBUG
void do_debug_printf(MP mp, const char *prefix, const char *fmt, ...) ;
void do_debug_printf(MP mp, const char *prefix, const char *fmt, ...) {
  va_list ap;
#if 0
  va_start (ap, fmt); 
  if (mp->log_file && !ferror((FILE *)mp->log_file)) {
    fputs(prefix, mp->log_file);
    vfprintf(mp->log_file, fmt, ap);
  }
  va_end(ap);
#endif
  va_start (ap, fmt);
#if 0
  if (mp->term_out  && !ferror((FILE *)mp->term_out)) {
#else
  if (false) {
#endif
    fputs(prefix, mp->term_out);
    vfprintf(mp->term_out, fmt, ap);
  } else {
    fputs(prefix, stdout);
    vfprintf(stdout, fmt, ap);
  }
  va_end(ap);
}
#endif

@ Here are the functions that set up the \MP\ instance.

@<Declarations@>=
MP_options *mp_options (void);
MP mp_initialize (MP_options * opt);

@ @c
MP_options *mp_options (void) {
  MP_options *opt;
  size_t l = sizeof (MP_options);
  opt = malloc (l);
  if (opt != NULL) {
    memset (opt, 0, l);
  }
  return opt;
}


@ @<Internal library declarations@>=
@<Declare subroutines for parsing file names@>
 

@ The whole instance structure is initialized with zeroes,
this greatly reduces the number of statements needed in 
the |Allocate or initialize variables| block.

@d set_callback_option(A) do { mp->A = mp_##A;
  if (opt->A!=NULL) mp->A = opt->A;
} while (0)

@c
static MP mp_do_new (jmp_buf * buf) {
  MP mp = malloc (sizeof (MP_instance));
  if (mp == NULL) {
    xfree (buf);
    return NULL;
  }
  memset (mp, 0, sizeof (MP_instance));
  mp->jump_buf = buf;
  return mp;
}


@ @c
static void mp_free (MP mp) {
  int k;        /* loop variable */
  @<Dealloc variables@>;
  if (mp->noninteractive) {
    @<Finish non-interactive use@>;
  }
  xfree (mp->jump_buf);
  @<Free table entries@>;
  free_math();
  xfree (mp);
}


@ @c
static void mp_do_initialize (MP mp) {
  @<Local variables for initialization@>;
  @<Set initial values of key variables@>;
}

@ For the retargetable math library, we need to have a pointer, at least.

@<Global variables@>=
void *math;

@ @<Exported types@>=
typedef enum {
  mp_nan_type = 0,
  mp_scaled_type,
  mp_fraction_type,
  mp_angle_type,
  mp_double_type,
  mp_binary_type,
  mp_decimal_type
} mp_number_type;
typedef union {
  void *num;
  double dval;
  int val;
} mp_number_store;
typedef struct mp_number_data {
  mp_number_store data;
  mp_number_type type;
} mp_number_data;
typedef struct mp_number_data mp_number;
#define is_number(A) ((A).type != mp_nan_type)

typedef void (*convert_func) (mp_number *r);
typedef void (*m_log_func) (MP mp, mp_number *r, mp_number a);
typedef void (*m_exp_func) (MP mp, mp_number *r, mp_number a);
typedef void (*pyth_add_func) (MP mp, mp_number *r, mp_number a, mp_number b);
typedef void (*pyth_sub_func) (MP mp, mp_number *r, mp_number a, mp_number b);
typedef void (*n_arg_func) (MP mp, mp_number *r, mp_number a, mp_number b);
typedef void (*velocity_func) (MP mp, mp_number *r, mp_number a, mp_number b, mp_number c, mp_number d, mp_number e);
typedef void (*ab_vs_cd_func) (MP mp, mp_number *r, mp_number a, mp_number b, mp_number c, mp_number d);
typedef void (*crossing_point_func) (MP mp, mp_number *r, mp_number a, mp_number b, mp_number c);
typedef void (*number_from_int_func) (mp_number *A, int B);
typedef void (*number_from_boolean_func) (mp_number *A, int B);
typedef void (*number_from_scaled_func) (mp_number *A, int B);
typedef void (*number_from_double_func) (mp_number *A, double B);
typedef void (*number_from_addition_func) (mp_number *A, mp_number B, mp_number C);
typedef void (*number_from_substraction_func) (mp_number *A, mp_number B, mp_number C);
typedef void (*number_from_div_func) (mp_number *A, mp_number B, mp_number C);
typedef void (*number_from_mul_func) (mp_number *A, mp_number B, mp_number C);
typedef void (*number_from_int_div_func) (mp_number *A, mp_number B, int C);
typedef void (*number_from_int_mul_func) (mp_number *A, mp_number B, int C);
typedef void (*number_from_oftheway_func) (MP mp, mp_number *A, mp_number t, mp_number B, mp_number C);
typedef void (*number_negate_func) (mp_number *A);
typedef void (*number_add_func) (mp_number *A, mp_number B);
typedef void (*number_substract_func) (mp_number *A, mp_number B);
typedef void (*number_modulo_func) (mp_number *A, mp_number B);
typedef void (*number_half_func) (mp_number *A);
typedef void (*number_halfp_func) (mp_number *A);
typedef void (*number_double_func) (mp_number *A);
typedef void (*number_abs_func) (mp_number *A);
typedef void (*number_clone_func) (mp_number *A, mp_number B);
typedef void (*number_swap_func) (mp_number *A, mp_number *B);
typedef void (*number_add_scaled_func) (mp_number *A, int b);
typedef void (*number_multiply_int_func) (mp_number *A, int b);
typedef void (*number_divide_int_func) (mp_number *A, int b);
typedef int (*number_to_int_func) (mp_number A);
typedef int (*number_to_boolean_func) (mp_number A);
typedef int (*number_to_scaled_func) (mp_number A);
typedef int (*number_round_func) (mp_number A);
typedef void (*number_floor_func) (mp_number *A);
typedef double (*number_to_double_func) (mp_number A);
typedef int (*number_odd_func) (mp_number A);
typedef int (*number_equal_func) (mp_number A, mp_number B);
typedef int (*number_less_func) (mp_number A, mp_number B);
typedef int (*number_greater_func) (mp_number A, mp_number B);
typedef int (*number_nonequalabs_func) (mp_number A, mp_number B);
typedef void (*make_scaled_func) (MP mp, mp_number *ret, mp_number A, mp_number B);
typedef void (*make_fraction_func) (MP mp, mp_number *ret, mp_number A, mp_number B);
typedef void (*take_fraction_func) (MP mp, mp_number *ret, mp_number A, mp_number B);
typedef void (*take_scaled_func) (MP mp, mp_number *ret, mp_number A, mp_number B);
typedef void (*sin_cos_func) (MP mp, mp_number A, mp_number *S, mp_number *C);
typedef void (*slow_add_func) (MP mp, mp_number *A, mp_number S, mp_number C);
typedef void (*sqrt_func) (MP mp, mp_number *ret, mp_number A);
typedef void (*init_randoms_func) (MP mp, int seed);
typedef void (*new_number_func) (MP mp, mp_number *A, mp_number_type t);
typedef void (*free_number_func) (MP mp, mp_number *n);
typedef void (*fraction_to_round_scaled_func) (mp_number *n);
typedef void (*print_func) (MP mp, mp_number A);
typedef char * (*tostring_func) (MP mp, mp_number A);
typedef void (*scan_func) (MP mp, int A);
typedef void (*mp_free_func) (MP mp);
typedef void (*set_precision_func) (MP mp);

typedef struct math_data {
  mp_number precision_default;
  mp_number precision_max;
  mp_number precision_min;
  mp_number epsilon_t;
  mp_number inf_t;
  mp_number one_third_inf_t;
  mp_number zero_t;
  mp_number unity_t;
  mp_number two_t;
  mp_number three_t;
  mp_number half_unit_t;
  mp_number three_quarter_unit_t;
  mp_number fraction_one_t;
  mp_number fraction_half_t;
  mp_number fraction_three_t;
  mp_number fraction_four_t;
  mp_number one_eighty_deg_t;
  mp_number three_sixty_deg_t;
  mp_number one_k;
  mp_number sqrt_8_e_k;
  mp_number twelve_ln_2_k;
  mp_number coef_bound_k;
  mp_number coef_bound_minus_1;
  mp_number twelvebits_3;
  mp_number arc_tol_k;
  mp_number twentysixbits_sqrt2_t;
  mp_number twentyeightbits_d_t;
  mp_number twentysevenbits_sqrt2_d_t;
  mp_number fraction_threshold_t;
  mp_number half_fraction_threshold_t;
  mp_number scaled_threshold_t;
  mp_number half_scaled_threshold_t;
  mp_number near_zero_angle_t;
  mp_number p_over_v_threshold_t;
  mp_number equation_threshold_t;
  mp_number tfm_warn_threshold_t;
  mp_number warning_limit_t;
  new_number_func allocate;
  free_number_func free;
  number_from_int_func from_int;
  number_from_boolean_func from_boolean;
  number_from_scaled_func from_scaled;
  number_from_double_func from_double;
  number_from_addition_func from_addition;
  number_from_substraction_func from_substraction;
  number_from_div_func from_div;
  number_from_mul_func from_mul;
  number_from_int_div_func from_int_div;
  number_from_int_mul_func from_int_mul;
  number_from_oftheway_func from_oftheway;
  number_negate_func negate;
  number_add_func add;
  number_substract_func substract;
  number_half_func half;
  number_modulo_func modulo;
  number_halfp_func halfp;
  number_double_func do_double;
  number_abs_func abs;
  number_clone_func clone;
  number_swap_func swap;
  number_add_scaled_func add_scaled;
  number_multiply_int_func multiply_int;
  number_divide_int_func divide_int;
  number_to_int_func to_int;
  number_to_boolean_func to_boolean;
  number_to_scaled_func to_scaled;
  number_to_double_func to_double;
  number_odd_func odd;
  number_equal_func equal;
  number_less_func less;
  number_greater_func greater;
  number_nonequalabs_func nonequalabs;
  number_round_func round_unscaled;
  number_floor_func floor_scaled;
  make_scaled_func make_scaled;
  make_fraction_func make_fraction;
  take_fraction_func take_fraction;
  take_scaled_func take_scaled;
  velocity_func velocity;
  ab_vs_cd_func ab_vs_cd;
  crossing_point_func crossing_point;
  n_arg_func n_arg;
  m_log_func m_log;
  m_exp_func m_exp;
  pyth_add_func pyth_add;
  pyth_sub_func pyth_sub;
  fraction_to_round_scaled_func fraction_to_round_scaled;
  convert_func fraction_to_scaled;
  convert_func scaled_to_fraction;
  convert_func scaled_to_angle;
  convert_func angle_to_scaled;
  init_randoms_func init_randoms;
  sin_cos_func sin_cos;
  sqrt_func sqrt;
  slow_add_func slow_add;
  print_func print;
  tostring_func tostring;
  scan_func scan_numeric;
  scan_func scan_fractional;
  mp_free_func free_math;
  set_precision_func set_precision;
} math_data;



@ This procedure gets things started properly.
@c
MP mp_initialize (MP_options * opt) {
  MP mp;
  jmp_buf *buf = malloc (sizeof (jmp_buf));
  if (buf == NULL || setjmp (*buf) != 0)
    return NULL;
  mp = mp_do_new (buf);
  if (mp == NULL)
    return NULL;
  mp->userdata = opt->userdata;
  mp->noninteractive = opt->noninteractive;
  set_callback_option (find_file);
  set_callback_option (open_file);
  set_callback_option (read_ascii_file);
  set_callback_option (read_binary_file);
  set_callback_option (close_file);
  set_callback_option (eof_file);
  set_callback_option (flush_file);
  set_callback_option (write_ascii_file);
  set_callback_option (write_binary_file);
  set_callback_option (shipout_backend);
  set_callback_option (run_script);
  if (opt->banner && *(opt->banner)) {
    mp->banner = xstrdup (opt->banner);
  } else {
    mp->banner = xstrdup (default_banner);
  }
  if (opt->command_line && *(opt->command_line))
    mp->command_line = xstrdup (opt->command_line);
  if (mp->noninteractive) {
    @<Prepare function pointers for non-interactive use@>;
  }
  /* open the terminal for output */
  t_open_out();
#if DEBUG
  setvbuf(stdout, (char *) NULL, _IONBF, 0);
  setvbuf(mp->term_out, (char *) NULL, _IONBF, 0);
#endif
  if (opt->math_mode == mp_math_scaled_mode) {
    mp->math = mp_initialize_scaled_math(mp);
  } else if (opt->math_mode == mp_math_decimal_mode) {
    mp->math = mp_initialize_decimal_math(mp);
  } else if (opt->math_mode == mp_math_binary_mode) {
    mp->math = mp_initialize_binary_math(mp);
  } else {
    mp->math = mp_initialize_double_math(mp);
  }
  @<Find and load preload file, if required@>;
  @<Allocate or initialize variables@>;
  mp_reallocate_paths (mp, 1000);
  mp_reallocate_fonts (mp, 8);
  mp->history = mp_fatal_error_stop;    /* in case we quit during initialization */
  @<Check the ``constant'' values...@>;
  if (mp->bad > 0) {
    char ss[256];
    mp_snprintf (ss, 256, "Ouch---my internal constants have been clobbered!\n"
                 "---case %i", (int) mp->bad);
    mp_fputs ((char *) ss, mp->err_out);
@.Ouch...clobbered@>;
    return mp;
  }
  mp_do_initialize (mp);        /* erase preloaded mem */
  mp_init_tab (mp);             /* initialize the tables */
  if (opt->math_mode == mp_math_scaled_mode) {
    set_internal_string (mp_number_system, mp_intern (mp, "scaled"));
  } else if (opt->math_mode == mp_math_decimal_mode) {
    set_internal_string (mp_number_system, mp_intern (mp, "decimal"));
  } else if (opt->math_mode == mp_math_binary_mode) {
    set_internal_string (mp_number_system, mp_intern (mp, "binary"));
  } else {
    set_internal_string (mp_number_system, mp_intern (mp, "double"));
  }
  mp_init_prim (mp);            /* call |primitive| for each primitive */
  mp_fix_date_and_time (mp);
  if (!mp->noninteractive) {
    @<Initialize the output routines@>;
    @<Get the first line of input and prepare to start@>;
    @<Initializations after first line is read@>;
    @<Fix up |mp->internal[mp_job_name]|@>;
  } else {
    mp->history = mp_spotless;
  }
  set_precision();
  return mp;
}


@ @<Initializations after first line is read@>=
mp_open_log_file (mp);
mp_set_job_id (mp);
mp_init_map_file (mp, mp->troff_mode);
mp->history = mp_spotless;      /* ready to go! */
if (mp->troff_mode) {
  number_clone (internal_value (mp_gtroffmode), unity_t);
  number_clone (internal_value (mp_prologues), unity_t);
}
if (mp->start_sym != NULL) {    /* insert the `\&{everyjob}' symbol */
  set_cur_sym (mp->start_sym);
  mp_back_input (mp);
}

@ @<Exported function headers@>=
extern MP_options *mp_options (void);
extern MP mp_initialize (MP_options * opt);
extern int mp_status (MP mp);
extern void *mp_userdata (MP mp);

@ @c
int mp_status (MP mp) {
  return mp->history;
}


@ @c
void *mp_userdata (MP mp) {
  return mp->userdata;
}


@ The overall \MP\ program begins with the heading just shown, after which
comes a bunch of procedure declarations and function declarations.
Finally we will get to the main program, which begins with the
comment `|start_here|'. If you want to skip down to the
main program now, you can look up `|start_here|' in the index.
But the author suggests that the best way to understand this program
is to follow pretty much the order of \MP's components as they appear in the
\.{WEB} description you are now reading, since the present ordering is
intended to combine the advantages of the ``bottom up'' and ``top down''
approaches to the problem of understanding a somewhat complicated system.

@ Some of the code below is intended to be used only when diagnosing the
strange behavior that sometimes occurs when \MP\ is being installed or
when system wizards are fooling around with \MP\ without quite knowing
what they are doing. Such code will not normally be compiled; it is
delimited by the preprocessor test `|#ifdef DEBUG .. #endif|'.

@ The following parameters can be changed at compile time to extend or
reduce \MP's capacity. 
@^system dependencies@>

@<Constants...@>=
#define bistack_size 1500       /* size of stack for bisection algorithms;
                                   should probably be left at this value */

@ Like the preceding parameters, the following quantities can be changed
to extend or reduce \MP's capacity. 

@ @<Glob...@>=
int pool_size;  /* maximum number of characters in strings, including all
                   error messages and help texts, and the names of all identifiers */
int max_in_open;        /* maximum number of input files and error insertions that
                           can be going on simultaneously */
int param_size; /* maximum number of simultaneous macro parameters */

@ @<Option variables@>=
int error_line; /* width of context lines on terminal error messages */
int half_error_line;    /* width of first lines of contexts in terminal
                           error messages; should be between 30 and |error_line-15| */
int halt_on_error;      /* do we quit at the first error? */
int max_print_line;     /* width of longest text lines output; should be at least 60 */
void *userdata; /* this allows the calling application to setup local */
char *banner;   /* the banner that is printed to the screen and log */
int ini_version;

@ @<Dealloc variables@>=
xfree (mp->banner);

@ 
@d set_lower_limited_value(a,b,c) do { a=c; if (b>c) a=b; } while (0)

@<Allocate or ...@>=
mp->param_size = 4;
mp->max_in_open = 0;
mp->pool_size = 10000;
set_lower_limited_value (mp->error_line, opt->error_line, 79);
set_lower_limited_value (mp->half_error_line, opt->half_error_line, 50);
if (mp->half_error_line > mp->error_line - 15)
  mp->half_error_line = mp->error_line - 15;
mp->max_print_line = 100;
set_lower_limited_value (mp->max_print_line, opt->max_print_line, 79);
mp->halt_on_error = (opt->halt_on_error ? true : false);
mp->ini_version = (opt->ini_version ? true : false);

@ In case somebody has inadvertently made bad settings of the ``constants,''
\MP\ checks them using a global variable called |bad|.

This is the second of many sections of \MP\ where global variables are
defined.

@<Glob...@>=
integer bad;    /* is some ``constant'' wrong? */

@ Later on we will say `|if ( int_packets+17*int_increment>bistack_size )mp->bad=19;|',
or something similar.

In case you are wondering about the non-consequtive values of |bad|: most
of the things that used to be WEB constants are now runtime variables
with checking at assignment time.

@<Check the ``constant'' values for consistency@>=
mp->bad = 0;

@ Here are some macros for common programming idioms.

@d incr(A)   (A)=(A)+1 /* increase a variable by unity */
@d decr(A)   (A)=(A)-1 /* decrease a variable by unity */
@d negate(A) (A)=-(A) /* change the sign of a variable */
@d double(A) (A)=(A)+(A)
@d odd(A)   (abs(A)%2==1)

@* The character set.
In order to make \MP\ readily portable to a wide variety of
computers, all of its input text is converted to an internal eight-bit
code that includes standard ASCII, the ``American Standard Code for
Information Interchange.''  This conversion is done immediately when each
character is read in. Conversely, characters are converted from ASCII to
the user's external representation just before they are output to a
text file.
@^ASCII code@>

Such an internal code is relevant to users of \MP\ only with respect to
the \&{char} and \&{ASCII} operations, and the comparison of strings.

@ Characters of text that have been converted to \MP's internal form
are said to be of type |ASCII_code|, which is a subrange of the integers.

@<Types...@>=
typedef unsigned char ASCII_code;       /* eight-bit numbers */

@ The present specification of \MP\ has been written under the assumption
that the character set contains at least the letters and symbols associated
with ASCII codes 040 through 0176; all of these characters are now
available on most computer terminals.

@<Types...@>=
typedef unsigned char text_char;        /* the data type of characters in text files */

@ @<Local variables for init...@>=
integer i;

@ The \MP\ processor converts between ASCII code and
the user's external character set by means of arrays |xord| and |xchr|
that are analogous to Pascal's |ord| and |chr| functions.

@<MPlib internal header stuff@>=
#define xchr(A) mp->xchr[(A)]
#define xord(A) mp->xord[(A)]

@ @<Glob...@>=
ASCII_code xord[256];   /* specifies conversion of input characters */
text_char xchr[256];    /* specifies conversion of output characters */

@ The core system assumes all 8-bit is acceptable.  If it is not,
a change file has to alter the below section.
@^system dependencies@>

Additionally, people with extended character sets can
assign codes arbitrarily, giving an |xchr| equivalent to whatever
characters the users of \MP\ are allowed to have in their input files.
Appropriate changes to \MP's |char_class| table should then be made.
(Unlike \TeX, each installation of \MP\ has a fixed assignment of category
codes, called the |char_class|.) Such changes make portability of programs
more difficult, so they should be introduced cautiously if at all.
@^character set dependencies@>
@^system dependencies@>

@<Set initial ...@>=
for (i = 0; i <= 0377; i++) {
  xchr (i) = (text_char) i;
}


@ The following system-independent code makes the |xord| array contain a
suitable inverse to the information in |xchr|. Note that if |xchr[i]=xchr[j]|
where |i<j<0177|, the value of |xord[xchr[i]]| will turn out to be
|j| or more; hence, standard ASCII code numbers will be used instead of
codes below 040 in case there is a coincidence.

@<Set initial ...@>=
for (i = 0; i <= 255; i++) {
  xord (xchr (i)) = 0177;
}
for (i = 0200; i <= 0377; i++) {
  xord (xchr (i)) = (ASCII_code) i;
}
for (i = 0; i <= 0176; i++) {
  xord (xchr (i)) = (ASCII_code) i;
}


@* Input and output.
The bane of portability is the fact that different operating systems treat
input and output quite differently, perhaps because computer scientists
have not given sufficient attention to this problem. People have felt somehow
that input and output are not part of ``real'' programming. Well, it is true
that some kinds of programming are more fun than others. With existing
input/output conventions being so diverse and so messy, the only sources of
joy in such parts of the code are the rare occasions when one can find a
way to make the program a little less bad than it might have been. We have
two choices, either to attack I/O now and get it over with, or to postpone
I/O until near the end. Neither prospect is very attractive, so let's
get it over with.

The basic operations we need to do are (1)~inputting and outputting of
text, to or from a file or the user's terminal; (2)~inputting and
outputting of eight-bit bytes, to or from a file; (3)~instructing the
operating system to initiate (``open'') or to terminate (``close'') input or
output from a specified file; (4)~testing whether the end of an input
file has been reached; (5)~display of bits on the user's screen.
The bit-display operation will be discussed in a later section; we shall
deal here only with more traditional kinds of I/O.

@ Finding files happens in a slightly roundabout fashion: the \MP\
instance object contains a field that holds a function pointer that finds a
file, and returns its name, or NULL. For this, it receives three
parameters: the non-qualified name |fname|, the intended |fopen|
operation type |fmode|, and the type of the file |ftype|.

The file types that are passed on in |ftype| can be  used to 
differentiate file searches if a library like kpathsea is used,
the fopen mode is passed along for the same reason.

@<Types...@>=
typedef unsigned char eight_bits;       /* unsigned one-byte quantity */

@ @<Exported types@>=
enum mp_filetype {
  mp_filetype_terminal = 0,     /* the terminal */
  mp_filetype_error,            /* the terminal */
  mp_filetype_program,          /* \MP\ language input */
  mp_filetype_log,              /* the log file */
  mp_filetype_postscript,       /* the postscript output */
  mp_filetype_bitmap,           /* the bitmap output file */
  mp_filetype_memfile,          /* memory dumps, obsolete */
  mp_filetype_metrics,          /* TeX font metric files */
  mp_filetype_fontmap,          /* PostScript font mapping files */
  mp_filetype_font,             /*  PostScript type1 font programs */
  mp_filetype_encoding,         /*  PostScript font encoding files */
  mp_filetype_text              /* first text file for readfrom and writeto primitives */
};
typedef char *(*mp_file_finder) (MP, const char *, const char *, int);
typedef char *(*mp_script_runner) (MP, const char *);
typedef void *(*mp_file_opener) (MP, const char *, const char *, int);
typedef char *(*mp_file_reader) (MP, void *, size_t *);
typedef void (*mp_binfile_reader) (MP, void *, void **, size_t *);
typedef void (*mp_file_closer) (MP, void *);
typedef int (*mp_file_eoftest) (MP, void *);
typedef void (*mp_file_flush) (MP, void *);
typedef void (*mp_file_writer) (MP, void *, const char *);
typedef void (*mp_binfile_writer) (MP, void *, void *, size_t);

@ @<Option variables@>=
mp_file_finder find_file;
mp_file_opener open_file;
mp_script_runner run_script;
mp_file_reader read_ascii_file;
mp_binfile_reader read_binary_file;
mp_file_closer close_file;
mp_file_eoftest eof_file;
mp_file_flush flush_file;
mp_file_writer write_ascii_file;
mp_binfile_writer write_binary_file;

@ The default function for finding files is |mp_find_file|. It is 
pretty stupid: it will only find files in the current directory.

@c
static char *mp_find_file (MP mp, const char *fname, const char *fmode,
                           int ftype) {
  (void) mp;
  if (fmode[0] != 'r' || (!access (fname, R_OK)) || ftype) {
    return mp_strdup (fname);
  }
  return NULL;
}

@ @c
static char *mp_run_script (MP mp, const char *str) {
  (void) mp;
  return mp_strdup (str);
}


@ Because |mp_find_file| is used so early, it has to be in the helpers
section.

@<Declarations@>=
static char *mp_find_file (MP mp, const char *fname, const char *fmode,
                           int ftype);
static void *mp_open_file (MP mp, const char *fname, const char *fmode,
                           int ftype);
static char *mp_read_ascii_file (MP mp, void *f, size_t * size);
static void mp_read_binary_file (MP mp, void *f, void **d, size_t * size);
static void mp_close_file (MP mp, void *f);
static int mp_eof_file (MP mp, void *f);
static void mp_flush_file (MP mp, void *f);
static void mp_write_ascii_file (MP mp, void *f, const char *s);
static void mp_write_binary_file (MP mp, void *f, void *s, size_t t);
static char *mp_run_script (MP mp, const char *str);

@ The function to open files can now be very short.

@c
void *mp_open_file (MP mp, const char *fname, const char *fmode, int ftype) {
  char realmode[3];
  (void) mp;
  realmode[0] = *fmode;
  realmode[1] = 'b';
  realmode[2] = 0;
  if (ftype == mp_filetype_terminal) {
    return (fmode[0] == 'r' ? stdin : stdout);
  } else if (ftype == mp_filetype_error) {
    return stderr;
  } else if (fname != NULL && (fmode[0] != 'r' || (!access (fname, R_OK)))) {
    return (void *) fopen (fname, realmode);
  }
  return NULL;
}


@ (Almost) all file names pass through |name_of_file|.

@<Glob...@>=
char *name_of_file;     /* the name of a system file */

@ If this parameter is true, the terminal and log will report the found
file names for input files instead of the requested ones. 
It is off by default because it creates an extra filename lookup.

@<Option variables@>=
int print_found_names;  /* configuration parameter */

@ @<Allocate or initialize ...@>=
mp->print_found_names = (opt->print_found_names > 0 ? true : false);

@ The |file_line_error_style| parameter makes \MP\ use a more
standard compiler error message format instead of the Knuthian 
exclamation mark. It needs the actual version of the current input 
file name, that will be saved by |open_in| in the |long_name|.

TODO: currently these long strings cause memory leaks, because they cannot
be safely freed as they may appear in the |input_stack| multiple times.
In fact, the current implementation is just a quick hack in response 
to a bug report for metapost 1.205.

@d long_name mp->cur_input.long_name_field /* long name of the current file */

@<Option variables@>=
int file_line_error_style;      /* configuration parameter */

@ @<Allocate or initialize ...@>=
mp->file_line_error_style = (opt->file_line_error_style > 0 ? true : false);

@ \MP's file-opening procedures return |false| if no file identified by
|name_of_file| could be opened.

The |do_open_file| function takes care of the |print_found_names| parameter.

@c
static boolean mp_do_open_file (MP mp, void **f, int ftype, const char *mode) {
  if (mp->print_found_names || mp->file_line_error_style) {
    char *s = (mp->find_file)(mp,mp->name_of_file,mode,ftype);
    if (s!=NULL) {
      *f = (mp->open_file)(mp,mp->name_of_file,mode, ftype); 
      if (mp->print_found_names) {
        xfree(mp->name_of_file);
        mp->name_of_file = xstrdup(s);
      }
      if ((*mode == 'r') && (ftype == mp_filetype_program)) {
        long_name = xstrdup(s);
      }
      xfree(s);
    } else {
      *f = NULL;
    }
  } else {
    *f = (mp->open_file)(mp,mp->name_of_file,mode, ftype); 
  }
  return (*f ? true : false);
}
@#
static boolean mp_open_in (MP mp, void **f, int ftype) {
  /* open a file for input */
  return mp_do_open_file (mp, f, ftype, "r");
}
@#
static boolean mp_open_out (MP mp, void **f, int ftype) {
  /* open a file for output */
  return mp_do_open_file (mp, f, ftype, "w");
}


@ @c
static char *mp_read_ascii_file (MP mp, void *ff, size_t * size) {
  int c;
  size_t len = 0, lim = 128;
  char *s = NULL;
  FILE *f = (FILE *) ff;
  *size = 0;
  (void) mp;                    /* for -Wunused */
  if (f == NULL)
    return NULL;
  c = fgetc (f);
  if (c == EOF)
    return NULL;
  s = malloc (lim);
  if (s == NULL)
    return NULL;
  while (c != EOF && c != '\n' && c != '\r') {
    if ((len + 1) == lim) {
      s = realloc (s, (lim + (lim >> 2)));
      if (s == NULL)
        return NULL;
      lim += (lim >> 2);
    }
    s[len++] = (char) c;
    c = fgetc (f);
  }
  if (c == '\r') {
    c = fgetc (f);
    if (c != EOF && c != '\n')
      ungetc (c, f);
  }
  s[len] = 0;
  *size = len;
  return s;
}


@ @c
void mp_write_ascii_file (MP mp, void *f, const char *s) {
  (void) mp;
  if (f != NULL) {
    fputs (s, (FILE *) f);
  }
}


@ @c
void mp_read_binary_file (MP mp, void *f, void **data, size_t * size) {
  size_t len = 0;
  (void) mp;
  if (f != NULL)
    len = fread (*data, 1, *size, (FILE *) f);
  *size = len;
}


@ @c
void mp_write_binary_file (MP mp, void *f, void *s, size_t size) {
  (void) mp;
  if (f != NULL)
    (void) fwrite (s, size, 1, (FILE *) f);
}


@ @c
void mp_close_file (MP mp, void *f) {
  (void) mp;
  if (f != NULL)
    fclose ((FILE *) f);
}


@ @c
int mp_eof_file (MP mp, void *f) {
  (void) mp;
  if (f != NULL)
    return feof ((FILE *) f);
  else
    return 1;
}


@ @c
void mp_flush_file (MP mp, void *f) {
  (void) mp;
  if (f != NULL)
    fflush ((FILE *) f);
}


@ Input from text files is read one line at a time, using a routine called
|input_ln|. This function is defined in terms of global variables called
|buffer|, |first|, and |last| that will be described in detail later; for
now, it suffices for us to know that |buffer| is an array of |ASCII_code|
values, and that |first| and |last| are indices into this array
representing the beginning and ending of a line of text.

@<Glob...@>=
size_t buf_size;        /* maximum number of characters simultaneously present in
                           current lines of open files */
ASCII_code *buffer;     /* lines of characters being read */
size_t first;   /* the first unused position in |buffer| */
size_t last;    /* end of the line just input to |buffer| */
size_t max_buf_stack;   /* largest index used in |buffer| */

@ @<Allocate or initialize ...@>=
mp->buf_size = 200;
mp->buffer = xmalloc ((mp->buf_size + 1), sizeof (ASCII_code));

@ @<Dealloc variables@>=
xfree (mp->buffer);

@ @c
static void mp_reallocate_buffer (MP mp, size_t l) {
  ASCII_code *buffer;
  if (l > max_halfword) {
    mp_confusion (mp, "buffer size");   /* can't happen (I hope) */
  }
  buffer = xmalloc ((l + 1), sizeof (ASCII_code));
  (void) memcpy (buffer, mp->buffer, (mp->buf_size + 1));
  xfree (mp->buffer);
  mp->buffer = buffer;
  mp->buf_size = l;
}


@ The |input_ln| function brings the next line of input from the specified
field into available positions of the buffer array and returns the value
|true|, unless the file has already been entirely read, in which case it
returns |false| and sets |last:=first|.  In general, the |ASCII_code|
numbers that represent the next line of the file are input into
|buffer[first]|, |buffer[first+1]|, \dots, |buffer[last-1]|; and the
global variable |last| is set equal to |first| plus the length of the
line. Trailing blanks are removed from the line; thus, either |last=first|
(in which case the line was entirely blank) or |buffer[last-1]<>" "|.
@^inner loop@>

The variable |max_buf_stack|, which is used to keep track of how large
the |buf_size| parameter must be to accommodate the present job, is
also kept up to date by |input_ln|.

@c
static boolean mp_input_ln (MP mp, void *f) {
  /* inputs the next line or returns |false| */
  char *s;
  size_t size = 0;
  mp->last = mp->first;         /* cf.\ Matthew 19\thinspace:\thinspace30 */
  s = (mp->read_ascii_file) (mp, f, &size);
  if (s == NULL)
    return false;
  if (size > 0) {
    mp->last = mp->first + size;
    if (mp->last >= mp->max_buf_stack) {
      mp->max_buf_stack = mp->last + 1;
      while (mp->max_buf_stack > mp->buf_size) {
        mp_reallocate_buffer (mp, (mp->buf_size + (mp->buf_size >> 2)));
      }
    }
    (void) memcpy ((mp->buffer + mp->first), s, size);
  }
  free (s);
  return true;
}


@ The user's terminal acts essentially like other files of text, except
that it is used both for input and for output. When the terminal is
considered an input file, the file variable is called |term_in|, and when it
is considered an output file the file variable is |term_out|.
@^system dependencies@>

@<Glob...@>=
void *term_in;  /* the terminal as an input file */
void *term_out; /* the terminal as an output file */
void *err_out;  /* the terminal as an output file */

@ Here is how to open the terminal files. In the default configuration,
nothing happens except that the command line (if there is one) is copied
to the input buffer.  The variable |command_line| will be filled by the 
|main| procedure. 

@d t_open_out()  do {/* open the terminal for text output */
    mp->term_out = (mp->open_file)(mp,"terminal", "w", mp_filetype_terminal);
    mp->err_out = (mp->open_file)(mp,"error", "w", mp_filetype_error);
} while (0)
@d t_open_in()  do { /* open the terminal for text input */
    mp->term_in = (mp->open_file)(mp,"terminal", "r", mp_filetype_terminal);
    if (mp->command_line!=NULL) {
      mp->last = strlen(mp->command_line);
      (void)memcpy((void *)mp->buffer,(void *)mp->command_line,mp->last);
      xfree(mp->command_line);
    } else {
	  mp->last = 0;
    }
} while (0)

@<Option variables@>=
char *command_line;

@ Sometimes it is necessary to synchronize the input/output mixture that
happens on the user's terminal, and three system-dependent
procedures are used for this
purpose. The first of these, |update_terminal|, is called when we want
to make sure that everything we have output to the terminal so far has
actually left the computer's internal buffers and been sent.
The second, |clear_terminal|, is called when we wish to cancel any
input that the user may have typed ahead (since we are about to
issue an unexpected error message). The third, |wake_up_terminal|,
is supposed to revive the terminal if the user has disabled it by
some instruction to the operating system.  The following macros show how
these operations can be specified:
@^system dependencies@>

@<MPlib internal header stuff@>=
#define update_terminal()  (mp->flush_file)(mp,mp->term_out)      /* empty the terminal output buffer */
#define clear_terminal()          /* clear the terminal input buffer */
#define wake_up_terminal() (mp->flush_file)(mp,mp->term_out)
                    /* cancel the user's cancellation of output */

@ We need a special routine to read the first line of \MP\ input from
the user's terminal. This line is different because it is read before we
have opened the transcript file; there is sort of a ``chicken and
egg'' problem here. If the user types `\.{input cmr10}' on the first
line, or if some macro invoked by that line does such an \.{input},
the transcript file will be named `\.{cmr10.log}'; but if no \.{input}
commands are performed during the first line of terminal input, the transcript
file will acquire its default name `\.{mpout.log}'. (The transcript file
will not contain error messages generated by the first line before the
first \.{input} command.)

The first line is even more special. It's nice to let the user start
running a \MP\ job by typing a command line like `\.{MP cmr10}'; in
such a case, \MP\ will operate as if the first line of input were
`\.{cmr10}', i.e., the first line will consist of the remainder of the
command line, after the part that invoked \MP.

@ Different systems have different ways to get started. But regardless of
what conventions are adopted, the routine that initializes the terminal
should satisfy the following specifications:

\yskip\textindent{1)}It should open file |term_in| for input from the
  terminal. (The file |term_out| will already be open for output to the
  terminal.)

\textindent{2)}If the user has given a command line, this line should be
  considered the first line of terminal input. Otherwise the
  user should be prompted with `\.{**}', and the first line of input
  should be whatever is typed in response.

\textindent{3)}The first line of input, which might or might not be a
  command line, should appear in locations |first| to |last-1| of the
  |buffer| array.

\textindent{4)}The global variable |loc| should be set so that the
  character to be read next by \MP\ is in |buffer[loc]|. This
  character should not be blank, and we should have |loc<last|.

\yskip\noindent(It may be necessary to prompt the user several times
before a non-blank line comes in. The prompt is `\.{**}' instead of the
later `\.*' because the meaning is slightly different: `\.{input}' need
not be typed immediately after~`\.{**}'.)

@d loc mp->cur_input.loc_field /* location of first unread character in |buffer| */

@c
boolean mp_init_terminal (MP mp) {                               /* gets the terminal input started */
  t_open_in();
  if (mp->last != 0) {
    loc = 0;
    mp->first = 0;
    return true;
  }
  while (1) {
    if (!mp->noninteractive) {
      wake_up_terminal();
      mp_fputs ("**", mp->term_out);
@.**@>;
      update_terminal();
    }
    if (!mp_input_ln (mp, mp->term_in)) {       /* this shouldn't happen */
      mp_fputs ("\n! End of file on the terminal... why?", mp->term_out);
@.End of file on the terminal@>;
      return false;
    }
    loc = (halfword) mp->first;
    while ((loc < (int) mp->last) && (mp->buffer[loc] == ' '))
      incr (loc);
    if (loc < (int) mp->last) {
      return true;              /* return unless the line was all blank */
    }
    if (!mp->noninteractive) {
      mp_fputs ("Please type the name of your input file.\n", mp->term_out);
    }
  }
}


@ @<Declarations@>=
static boolean mp_init_terminal (MP mp);

@* Globals for strings.

@ Symbolic token names and diagnostic messages are variable-length strings
of eight-bit characters. Many strings \MP\ uses are simply literals
in the compiled source, like the error messages and the names of the
internal parameters. Other strings are used or defined from the \MP\ input 
language, and these have to be interned.

\MP\ uses strings more extensively than \MF\ does, but the necessary
operations can still be handled with a fairly simple data structure.
The avl tree |strings| contains all of the known string structures.

Each structure contains an |unsigned char| pointer containing the eight-bit
data, a |size_t| that holds the length of that data, and an |int| that 
indicates how often this string is referenced (this will be explained below).
Such strings are referred to by structure pointers called |mp_string|.

Besides the avl tree, there is a set of three variables called |cur_string|,
|cur_length| and |cur_string_size| that are used for strings while they are
being built.

@<Exported types...@>=
typedef struct {
  unsigned char *str;   /* the string value */
  size_t len;   /* its length */
  int refs;     /* number of references */
} mp_lstring;
typedef mp_lstring *mp_string; /* for pointers to string values */

@ The string handling functions are in \.{mpstrings.w}, but strings
need a bunch of globals and those are defined here in the main file.

@<Glob...@>=
avl_tree strings;       /* string avl tree */
unsigned char *cur_string;      /*  current string buffer */
size_t cur_length;      /* current index in that buffer */
size_t cur_string_size; /*  malloced size of |cur_string| */

@ @<Allocate or initialize ...@>=
mp_initialize_strings(mp);

@ @<Dealloc variables@>=
mp_dealloc_strings(mp);

@ The next four variables are for keeping track of string memory usage.

@<Glob...@>=
integer pool_in_use;    /* total number of string bytes actually in use */
integer max_pl_used;    /* maximum |pool_in_use| so far */
integer strs_in_use;    /* total number of strings actually in use */
integer max_strs_used;  /* maximum |strs_in_use| so far */


@* On-line and off-line printing.
Messages that are sent to a user's terminal and to the transcript-log file
are produced by several `|print|' procedures. These procedures will
direct their output to a variety of places, based on the setting of
the global variable |selector|, which has the following possible
values:

\yskip
\hang |term_and_log|, the normal setting, prints on the terminal and on the
  transcript file.

\hang |log_only|, prints only on the transcript file.

\hang |term_only|, prints only on the terminal.

\hang |no_print|, doesn't print at all. This is used only in rare cases
  before the transcript file is open.

\hang |pseudo|, puts output into a cyclic buffer that is used
  by the |show_context| routine; when we get to that routine we shall discuss
  the reasoning behind this curious mode.

\hang |new_string|, appends the output to the current string in the
  string pool.

\hang |>=write_file| prints on one of the files used for the \&{write}
@:write_}{\&{write} primitive@>
  command.

\yskip
\noindent The symbolic names `|term_and_log|', etc., have been assigned
numeric codes that satisfy the convenient relations |no_print+1=term_only|,
|no_print+2=log_only|, |term_only+2=log_only+1=term_and_log|.  These
relations are not used when |selector| could be |pseudo|, or |new_string|.
We need not check for unprintable characters when |selector<pseudo|.

Three additional global variables, |tally|, |term_offset| and |file_offset|
record the number of characters that have been printed
since they were most recently cleared to zero. We use |tally| to record
the length of (possibly very long) stretches of printing; |term_offset|,
and |file_offset|, on the other hand, keep track of how many
characters have appeared so far on the current line that has been output
to the terminal, the transcript file, or the \ps\ output file, respectively.

@d new_string 0 /* printing is deflected to the string pool */
@d pseudo 2 /* special |selector| setting for |show_context| */
@d no_print 3 /* |selector| setting that makes data disappear */
@d term_only 4 /* printing is destined for the terminal only */
@d log_only 5 /* printing is destined for the transcript file only */
@d term_and_log 6 /* normal |selector| setting */
@d write_file 7 /* first write file selector */

@<Glob...@>=
void *log_file; /* transcript of \MP\ session */
void *output_file;      /* the generic font output goes here */
unsigned int selector;  /* where to print a message */
integer tally;  /* the number of characters recently printed */
unsigned int term_offset;
  /* the number of characters on the current terminal line */
unsigned int file_offset;
  /* the number of characters on the current file line */
ASCII_code *trick_buf;  /* circular buffer for pseudoprinting */
integer trick_count;    /* threshold for pseudoprinting, explained later */
integer first_count;    /* another variable for pseudoprinting */

@ The first 128 strings will contain 95 standard ASCII characters, and the
other 33 characters will be printed in three-symbol form like `\.{\^\^A}'
unless a system-dependent change is made here. Installations that have
an extended character set, where for example |xchr[032]=@t\.{'^^Z'}@>|,
would like string 032 to be printed as the single character 032 instead
of the three characters 0136, 0136, 0132 (\.{\^\^Z}). On the other hand,
even people with an extended character set will want to represent string
015 by \.{\^\^M}, since 015 is ASCII's ``carriage return'' code; the idea is
to produce visible strings instead of tabs or line-feeds or carriage-returns
or bell-rings or characters that are treated anomalously in text files.

The boolean expression defined here should be |true| unless \MP\ internal
code number~|k| corresponds to a non-troublesome visible symbol in the
local character set.
If character |k| cannot be printed, and |k<0200|, then character |k+0100| or
|k-0100| must be printable; moreover, ASCII codes |[060..071, 0141..0146]|
must be printable.
@^character set dependencies@>
@^system dependencies@>

@<Character |k| cannot be printed@>=
(k < ' ') || (k == 127)
 
@ @<Allocate or initialize ...@>=
mp->trick_buf = xmalloc ((mp->error_line + 1), sizeof (ASCII_code));

@ @<Dealloc variables@>=
xfree (mp->trick_buf);

@ @<Initialize the output routines@>=
mp->selector = term_only;
mp->tally = 0;
mp->term_offset = 0;
mp->file_offset = 0;

@ Macro abbreviations for output to the terminal and to the log file are
defined here for convenience. Some systems need special conventions
for terminal output, and it is possible to adhere to those conventions
by changing |wterm|, |wterm_ln|, and |wterm_cr| here.
@^system dependencies@>

@<MPlib internal header stuff@>=
#define mp_fputs(b,f) (mp->write_ascii_file)(mp,f,b)
#define wterm(A)     mp_fputs((A), mp->term_out)
#define wterm_chr(A) { unsigned char ss[2]; ss[0]=(A); ss[1]='\0'; wterm((char *)ss);}
#define wterm_cr     mp_fputs("\n", mp->term_out)
#define wterm_ln(A)  { wterm_cr; mp_fputs((A), mp->term_out); }
#define wlog(A)        mp_fputs((A), mp->log_file)
#define wlog_chr(A)  { unsigned char ss[2]; ss[0]=(A); ss[1]='\0'; wlog((char *)ss);}
#define wlog_cr      mp_fputs("\n", mp->log_file)
#define wlog_ln(A)   { wlog_cr; mp_fputs((A), mp->log_file); }


@ To end a line of text output, we call |print_ln|.  Cases |0..max_write_files|
use an array |wr_file| that will be declared later.

@d mp_print_text(A) mp_print_str(mp,text((A)))

@<Internal library ...@>=
void mp_print (MP mp, const char *s);
void mp_printf (MP mp, const char *ss, ...);
void mp_print_ln (MP mp);
void mp_print_char (MP mp, ASCII_code k);
void mp_print_str (MP mp, mp_string s);
void mp_print_nl (MP mp, const char *s);
void mp_print_two (MP mp, mp_number x, mp_number y);

@ @<Declarations@>=
static void mp_print_visible_char (MP mp, ASCII_code s);

@ @<Basic print...@>=
void mp_print_ln (MP mp) {                               /* prints an end-of-line */
  switch (mp->selector) {
  case term_and_log:
    wterm_cr;
    wlog_cr;
    mp->term_offset = 0;
    mp->file_offset = 0;
    break;
  case log_only:
    wlog_cr;
    mp->file_offset = 0;
    break;
  case term_only:
    wterm_cr;
    mp->term_offset = 0;
    break;
  case no_print:
  case pseudo:
  case new_string:
    break;
  default:
    mp_fputs ("\n", mp->wr_file[(mp->selector - write_file)]);
  }
}                               /* note that |tally| is not affected */


@ The |print_visible_char| procedure sends one character to the desired
destination, using the |xchr| array to map it into an external character
compatible with |input_ln|.  (It assumes that it is always called with
a visible ASCII character.)  All printing comes through |print_ln| or
|print_char|, which ultimately calls |print_visible_char|, hence these
routines are the ones that limit lines to at most |max_print_line| characters.
But we must make an exception for the \ps\ output file since it is not safe
to cut up lines arbitrarily in \ps.

@<Basic printing...@>=
static void mp_print_visible_char (MP mp, ASCII_code s) {                               /* prints a single character */
  switch (mp->selector) {
  case term_and_log:
    wterm_chr (xchr (s));
    wlog_chr (xchr (s));
    incr (mp->term_offset);
    incr (mp->file_offset);
    if (mp->term_offset == (unsigned) mp->max_print_line) {
      wterm_cr;
      mp->term_offset = 0;
    };
    if (mp->file_offset == (unsigned) mp->max_print_line) {
      wlog_cr;
      mp->file_offset = 0;
    };
    break;
  case log_only:
    wlog_chr (xchr (s));
    incr (mp->file_offset);
    if (mp->file_offset == (unsigned) mp->max_print_line)
      mp_print_ln (mp);
    break;
  case term_only:
    wterm_chr (xchr (s));
    incr (mp->term_offset);
    if (mp->term_offset == (unsigned) mp->max_print_line)
      mp_print_ln (mp);
    break;
  case no_print:
    break;
  case pseudo:
    if (mp->tally < mp->trick_count)
      mp->trick_buf[mp->tally % mp->error_line] = s;
    break;
  case new_string:
    append_char (s);
    break;
  default:
    {
      text_char ss[2] = {0,0};
      ss[0] = xchr (s);
      mp_fputs ((char *) ss, mp->wr_file[(mp->selector - write_file)]);
    }
  }
  incr (mp->tally);
}


@ The |print_char| procedure sends one character to the desired destination.
File names and string expressions might contain |ASCII_code| values that
can't be printed using |print_visible_char|.  These characters will be
printed in three- or four-symbol form like `\.{\^\^A}' or `\.{\^\^e4}'.
(This procedure assumes that it is safe to bypass all checks for unprintable
characters when |selector| is in the range |0..max_write_files-1|.
The user might want to write unprintable characters.

@<Basic printing...@>=
void mp_print_char (MP mp, ASCII_code k) {                               /* prints a single character */
  if (mp->selector < pseudo || mp->selector >= write_file) {
    mp_print_visible_char (mp, k);
  } else if (@<Character |k| cannot be printed@>) {
    mp_print (mp, "^^");
    if (k < 0100) {
      mp_print_visible_char (mp, (ASCII_code) (k + 0100));
    } else if (k < 0200) {
      mp_print_visible_char (mp, (ASCII_code) (k - 0100));
    } else {
      int l;    /* small index or counter */
      l = (k / 16);
      mp_print_visible_char (mp, xord (l < 10 ? l + '0' : l - 10 + 'a'));
      l = (k % 16);
      mp_print_visible_char (mp, xord (l < 10 ? l + '0' : l - 10 + 'a'));
    }
  } else {
    mp_print_visible_char (mp, k);
  }
}


@ An entire string is output by calling |print|. Note that if we are outputting
the single standard ASCII character \.c, we could call |print("c")|, since
|"c"=99| is the number of a single-character string, as explained above. But
|print_char("c")| is quicker, so \MP\ goes directly to the |print_char|
routine when it knows that this is safe. (The present implementation
assumes that it is always safe to print a visible ASCII character.)
@^system dependencies@>

@<Basic print...@>=
static void mp_do_print (MP mp, const char *ss, size_t len) {                               /* prints string |s| */
  if (len==0)
    return;
  if (mp->selector == new_string) {
    str_room (len);
    memcpy((mp->cur_string+mp->cur_length), ss, len);
    mp->cur_length += len;
  } else {
    size_t j = 0;
    while (j < len) {
      /* this was |xord((int)ss[j])| but that doesnt work */
      mp_print_char (mp, (ASCII_code) ss[j]);
      j++;
    }
  }
}


@ 
@<Basic print...@>=
void mp_print (MP mp, const char *ss) {
  assert (ss != NULL);
  mp_do_print (mp, ss, strlen (ss));
}
void mp_printf (MP mp, const char *ss, ...) {
  va_list ap;
  char pval[256];
  assert (ss != NULL);
  va_start(ap, ss);
  vsnprintf (pval, 256, ss, ap);
  mp_do_print (mp, pval, strlen (pval));
  va_end(ap);
}

void mp_print_str (MP mp, mp_string s) {
  assert (s != NULL);
  mp_do_print (mp, (const char *) s->str, s->len);
}


@ Here is the very first thing that \MP\ prints: a headline that identifies
the version number and base name. The |term_offset| variable is temporarily
incorrect, but the discrepancy is not serious since we assume that the banner
and mem identifier together will occupy at most |max_print_line|
character positions.

@<Initialize the output...@>=
wterm (mp->banner);
mp_print_ln (mp);
update_terminal();

@ The procedure |print_nl| is like |print|, but it makes sure that the
string appears at the beginning of a new line.

@<Basic print...@>=
void mp_print_nl (MP mp, const char *s) {                               /* prints string |s| at beginning of line */
  switch (mp->selector) {
  case term_and_log:
    if ((mp->term_offset > 0) || (mp->file_offset > 0))
      mp_print_ln (mp);
    break;
  case log_only:
    if (mp->file_offset > 0)
      mp_print_ln (mp);
    break;
  case term_only:
    if (mp->term_offset > 0)
      mp_print_ln (mp);
    break;
  case no_print:
  case pseudo:
  case new_string:
    break;
  }                             /* there are no other cases */
  mp_print (mp, s);
}


@ The following procedure, which prints out the decimal representation of a
given integer |n|, assumes that all integers fit nicely into a |int|.
@^system dependencies@>

@<Basic print...@>=
void mp_print_int (MP mp, integer n) {                               /* prints an integer in decimal form */
  char s[12];
  mp_snprintf (s, 12, "%d", (int) n);
  mp_print (mp, s);
}
void mp_print_pointer (MP mp, void *n) {                               /* prints an pointer in hexadecimal form */
  char s[12];
  mp_snprintf (s, 12, "%p", n);
  mp_print (mp, s);
}

@ @<Internal library ...@>=
void mp_print_int (MP mp, integer n);
void mp_print_pointer (MP mp, void *n);

@ \MP\ also makes use of a trivial procedure to print two digits. The
following subroutine is usually called with a parameter in the range |0<=n<=99|.

@c
static void mp_print_dd (MP mp, integer n) {                               /* prints two least significant digits */
  n = abs (n) % 100;
  mp_print_char (mp, xord ('0' + (n / 10)));
  mp_print_char (mp, xord ('0' + (n % 10)));
}


@ @<Declarations@>=
static void mp_print_dd (MP mp, integer n);

@ Here is a procedure that asks the user to type a line of input,
assuming that the |selector| setting is either |term_only| or |term_and_log|.
The input is placed into locations |first| through |last-1| of the
|buffer| array, and echoed on the transcript file if appropriate.

This procedure is never called when |interaction<mp_scroll_mode|.

@d prompt_input(A) do { 
    if (!mp->noninteractive) {
      wake_up_terminal();
      mp_print(mp, (A)); 
    }
    mp_term_input(mp);
  } while (0) /* prints a string and gets a line of input */

@c
void mp_term_input (MP mp) {                               /* gets a line from the terminal */
  size_t k;     /* index into |buffer| */
  if (mp->noninteractive) {
    if (!mp_input_ln (mp, mp->term_in))
      longjmp (*(mp->jump_buf), 1);     /* chunk finished */
    mp->buffer[mp->last] = xord ('%');
  } else {
    update_terminal();            /* Now the user sees the prompt for sure */
    if (!mp_input_ln (mp, mp->term_in)) {
      mp_fatal_error (mp, "End of file on the terminal!");
@.End of file on the terminal@>
    }
    mp->term_offset = 0;        /* the user's line ended with \<\rm return> */
    decr (mp->selector);        /* prepare to echo the input */
    if (mp->last != mp->first) {
      for (k = mp->first; k < mp->last; k++) {
        mp_print_char (mp, mp->buffer[k]);
      }
    }
    mp_print_ln (mp);
    mp->buffer[mp->last] = xord ('%');
    incr (mp->selector);        /* restore previous status */
  }
}


@* Reporting errors.

The |print_err| procedure supplies a `\.!' before the official message,
and makes sure that the terminal is awake if a stop is going to occur.
The |error| procedure supplies a `\..' after the official message, then it
shows the location of the error; and if |interaction=error_stop_mode|,
it also enters into a dialog with the user, during which time the help
message may be printed.
@^system dependencies@>

@ The global variable |interaction| has four settings, representing increasing
amounts of user interaction:

@<Exported types@>=
enum mp_interaction_mode {
  mp_unspecified_mode = 0,      /* extra value for command-line switch */
  mp_batch_mode,                /* omits all stops and omits terminal output */
  mp_nonstop_mode,              /* omits all stops */
  mp_scroll_mode,               /* omits error stops */
  mp_error_stop_mode            /* stops at every opportunity to interact */
};

@ @<Option variables@>=
int interaction;        /* current level of interaction */
int noninteractive;     /* do we have a terminal? */

@ Set it here so it can be overwritten by the commandline

@<Allocate or initialize ...@>=
mp->interaction = opt->interaction;
if (mp->interaction == mp_unspecified_mode
    || mp->interaction > mp_error_stop_mode)
  mp->interaction = mp_error_stop_mode;
if (mp->interaction < mp_unspecified_mode)
  mp->interaction = mp_batch_mode;

@ |print_err| is not merged in |error| because it is also used in |prompt_file_name|,
where |error| is not called at all.

@<Declarations@>=
static void mp_print_err (MP mp, const char *A);

@ @c
static void mp_print_err (MP mp, const char *A) {
  if (mp->interaction == mp_error_stop_mode)
    wake_up_terminal();
  if (mp->file_line_error_style && file_state && !terminal_input) {
    mp_print_nl (mp, "");
    if (long_name != NULL) {
      mp_print (mp, long_name);
    } else {
      mp_print (mp, mp_str (mp, name));
    }
    mp_print (mp, ":");
    mp_print_int (mp, line);
    mp_print (mp, ": ");
  } else {
    mp_print_nl (mp, "! ");
  }
  mp_print (mp, A);
@.!\relax@>
}


@ \MP\ is careful not to call |error| when the print |selector| setting
might be unusual. The only possible values of |selector| at the time of
error messages are

\yskip\hang|no_print| (when |interaction=mp_batch_mode|
  and |log_file| not yet open);

\hang|term_only| (when |interaction>mp_batch_mode| and |log_file| not yet open);

\hang|log_only| (when |interaction=mp_batch_mode| and |log_file| is open);

\hang|term_and_log| (when |interaction>mp_batch_mode| and |log_file| is open).

@d initialize_print_selector() mp->selector = (mp->interaction == mp_batch_mode ? no_print : term_only);

@ The global variable |history| records the worst level of error that
has been detected. It has four possible values: |spotless|, |warning_issued|,
|error_message_issued|, and |fatal_error_stop|.

Another global variable, |error_count|, is increased by one when an
|error| occurs without an interactive dialog, and it is reset to zero at
the end of every statement.  If |error_count| reaches 100, \MP\ decides
that there is no point in continuing further.

@<Exported types@>=
enum mp_history_state {
  mp_spotless = 0,      /* |history| value when nothing has been amiss yet */
  mp_warning_issued,            /* |history| value when |begin_diagnostic| has been called */
  mp_error_message_issued,      /* |history| value when |error| has been called */
  mp_fatal_error_stop,          /* |history| value when termination was premature */
  mp_system_error_stop          /* |history| value when termination was due to disaster */
};

@ @<Glob...@>=
int history;    /* has the source input been clean so far? */
int error_count;        /* the number of scrolled errors since the last statement ended */

@ The value of |history| is initially |fatal_error_stop|, but it will
be changed to |spotless| if \MP\ survives the initialization process.

@ Since errors can be detected almost anywhere in \MP, we want to declare the
error procedures near the beginning of the program. But the error procedures
in turn use some other procedures, which need to be declared |forward|
before we get to |error| itself.

It is possible for |error| to be called recursively if some error arises
when |get_next| is being used to delete a token, and/or if some fatal error
occurs while \MP\ is trying to fix a non-fatal one. But such recursion
@^recursion@>
is never more than two levels deep.

@<Declarations@>=
static void mp_get_next (MP mp);
static void mp_term_input (MP mp);
static void mp_show_context (MP mp);
static void mp_begin_file_reading (MP mp);
static void mp_open_log_file (MP mp);
static void mp_clear_for_error_prompt (MP mp);

@ @<Internal ...@>=
void mp_normalize_selector (MP mp);

@ @<Glob...@>=
boolean use_err_help;   /* should the |err_help| string be shown? */
mp_string err_help;    /* a string set up by \&{errhelp} */

@ @<Allocate or ...@>=
mp->use_err_help = false;

@ The |jump_out| procedure just cuts across all active procedure levels and
goes to |end_of_MP|. This is the only nonlocal |goto| statement in the
whole program. It is used when there is no recovery from a particular error.

The program uses a |jump_buf| to handle this, this is initialized at three
spots: the start of |mp_new|, the start of |mp_initialize|, and the start 
of |mp_run|. Those are the only library enty points.
@^system dependencies@>

@<Glob...@>=
jmp_buf *jump_buf;

@ If the array of internals is still |NULL| when |jump_out| is called, a
crash occured during initialization, and it is not safe to run the normal
cleanup routine.

@<Error hand...@>=
void mp_jump_out (MP mp) {
  if (mp->internal != NULL && mp->history < mp_system_error_stop)
    mp_close_files_and_terminate (mp);
  longjmp (*(mp->jump_buf), 1);
}

@ @<Internal ...@>=
void mp_jump_out (MP mp);

@ 

@<Error hand...@>=
void mp_warn (MP mp, const char *msg) {
  unsigned saved_selector = mp->selector;
  mp_normalize_selector (mp);
  mp_print_nl (mp, "Warning: ");
  mp_print (mp, msg);
  mp_print_ln (mp);
  mp->selector = saved_selector;
}

@ Here now is the general |error| routine.

The argument |deletions_allowed| is set |false| if the |get_next|
routine is active when |error| is called; this ensures that |get_next|
will never be called recursively.
@^recursion@>

Individual lines of help are recorded in the array |help_line|, which
contains entries in positions |0..(help_ptr-1)|. They should be printed
in reverse order, i.e., with |help_line[0]| appearing last.

@c
void mp_error (MP mp, const char *msg, const char **hlp, boolean deletions_allowed) {
  ASCII_code c; /* what the user types */
  integer s1, s2;       /* used to save global variables when deleting tokens */
  mp_sym s3;    /* likewise */
  int i = 0;
  const char *help_line[6];       /* helps for the next |error| */
  unsigned int help_ptr;  /* the number of help lines present */
  const char **cnt = NULL;
  mp_print_err(mp, msg);
  if (hlp) {
    cnt = hlp;
    while (*cnt) {
      i++; cnt++;
    }
    cnt = hlp;
  }
  help_ptr=i;
  while (i>0) {
    help_line[--i]= *cnt++;
  }
  if (mp->history < mp_error_message_issued)
    mp->history = mp_error_message_issued;
  mp_print_char (mp, xord ('.'));
  mp_show_context (mp);
  if (mp->halt_on_error) {
    mp->history = mp_fatal_error_stop;
    mp_jump_out (mp);
  }
  if ((!mp->noninteractive) && (mp->interaction == mp_error_stop_mode)) {
    @<Get user's advice and |return|@>;
  }
  incr (mp->error_count);
  if (mp->error_count == 100) {
    mp_print_nl (mp, "(That makes 100 errors; please try again.)");
@.That makes 100 errors...@>;
    mp->history = mp_fatal_error_stop;
    mp_jump_out (mp);
  }
  @<Put help message on the transcript file@>;
}


@ @<Exported function ...@>=
extern void mp_error (MP mp, const char *msg, const char **hlp, boolean deletions_allowed);
extern void mp_warn (MP mp, const char *msg);


@ @<Get user's advice...@>=
while (true) {
CONTINUE:
  mp_clear_for_error_prompt (mp);
  prompt_input ("? ");
@.?\relax@>;
  if (mp->last == mp->first)
    return;
  c = mp->buffer[mp->first];
  if (c >= 'a')
    c = (ASCII_code) (c + 'A' - 'a');   /* convert to uppercase */
  @<Interpret code |c| and |return| if done@>;
}


@ It is desirable to provide an `\.E' option here that gives the user
an easy way to return from \MP\ to the system editor, with the offending
line ready to be edited. But such an extension requires some system
wizardry, so the present implementation simply types out the name of the
file that should be
edited and the relevant line number.
@^system dependencies@>

@<Exported types@>=
typedef void (*mp_editor_cmd) (MP, char *, int);

@ @<Option variables@>=
mp_editor_cmd run_editor;

@ @<Allocate or initialize ...@>=
set_callback_option (run_editor);

@ @<Declarations@>=
static void mp_run_editor (MP mp, char *fname, int fline);

@ @c
void mp_run_editor (MP mp, char *fname, int fline) {
  char *s = xmalloc (256, 1);
  mp_snprintf (s, 256, "You want to edit file %s at line %d\n", fname, fline);
  wterm_ln (s);
@.You want to edit file x@>
}


@ 

@<Interpret code |c| and |return| if done@>=
switch (c) {
case '0':
case '1':
case '2':
case '3':
case '4':
case '5':
case '6':
case '7':
case '8':
case '9':
  if (deletions_allowed) {
    @<Delete tokens and |continue|@>;
  }
  break;
case 'E':
  if (mp->file_ptr > 0) {
    mp->interaction = mp_scroll_mode;
    mp_close_files_and_terminate (mp);
    (mp->run_editor) (mp,
                      mp_str (mp, mp->input_stack[mp->file_ptr].name_field),
                      mp_true_line (mp));
    mp_jump_out (mp);
  }
  break;
case 'H':
  @<Print the help information and |continue|@>;
  /* |break;| */
case 'I':
  @<Introduce new material from the terminal and |return|@>;
  /* |break;| */
case 'Q':
case 'R':
case 'S':
  @<Change the interaction level and |return|@>;
  /* |break;| */
case 'X':
  mp->interaction = mp_scroll_mode;
  mp_jump_out (mp);
  break;
default:
  break;
}
@<Print the menu of available options@>
 

@ @<Print the menu...@>=
{
  mp_print (mp, "Type <return> to proceed, S to scroll future error messages,");
@.Type <return> to proceed...@>;
  mp_print_nl (mp, "R to run without stopping, Q to run quietly,");
  mp_print_nl (mp, "I to insert something, ");
  if (mp->file_ptr > 0)
    mp_print (mp, "E to edit your file,");
  if (deletions_allowed)
    mp_print_nl (mp,
                 "1 or ... or 9 to ignore the next 1 to 9 tokens of input,");
  mp_print_nl (mp, "H for help, X to quit.");
}


@ @<Change the interaction...@>=
{
  mp->error_count = 0;
  mp_print (mp, "OK, entering ");
  switch (c) {
  case 'Q':
    mp->interaction = mp_batch_mode;
    mp_print (mp, "batchmode");
    decr (mp->selector);
    break;
  case 'R':
    mp->interaction = mp_nonstop_mode;
    mp_print (mp, "nonstopmode");
    break;
  case 'S':
    mp->interaction = mp_scroll_mode;
    mp_print (mp, "scrollmode");
    break;
  }                             /* there are no other cases */
  mp_print (mp, "...");
  mp_print_ln (mp);
  update_terminal();
  return;
}


@ When the following code is executed, |buffer[(first+1)..(last-1)]| may
contain the material inserted by the user; otherwise another prompt will
be given. In order to understand this part of the program fully, you need
to be familiar with \MP's input stacks.

@<Introduce new material...@>=
{
  mp_begin_file_reading (mp);   /* enter a new syntactic level for terminal input */
  if (mp->last > mp->first + 1) {
    loc = (halfword) (mp->first + 1);
    mp->buffer[mp->first] = xord (' ');
  } else {
    prompt_input ("insert>");
    loc = (halfword) mp->first;
@.insert>@>
  }
  mp->first = mp->last + 1;
  mp->cur_input.limit_field = (halfword) mp->last;
  return;
}


@ We allow deletion of up to 99 tokens at a time.

@<Delete tokens...@>=
{
  s1 = cur_cmd();
  s2 = cur_mod();
  s3 = cur_sym();
  mp->OK_to_interrupt = false;
  if ((mp->last > mp->first + 1) && (mp->buffer[mp->first + 1] >= '0')
      && (mp->buffer[mp->first + 1] <= '9'))
    c = xord (c * 10 + mp->buffer[mp->first + 1] - '0' * 11);
  else
    c = (ASCII_code) (c - '0');
  while (c > 0) {
    mp_get_next (mp);           /* one-level recursive call of |error| is possible */
    @<Decrease the string reference count, if the current token is a string@>;
    c--;
  };
  set_cur_cmd (s1);
  set_cur_mod (s2);
  set_cur_sym (s3);
  mp->OK_to_interrupt = true;
  help_ptr = 2;
  help_line[1] = "I have just deleted some text, as you asked.";
  help_line[0] = "You can now delete more, or insert, or whatever.";
  mp_show_context (mp);
  goto CONTINUE;
}


@ Some wriggling with |help_line| is done here to avoid giving no
information whatsoever, or presenting the same information twice
in a row.

@<Print the help info...@>=
{
  if (mp->use_err_help) {
    @<Print the string |err_help|, possibly on several lines@>;
    mp->use_err_help = false;
  } else {
    if (help_ptr == 0) {
      help_ptr=2; 
      help_line[1] = "Sorry, I don't know how to help in this situation.";
      help_line[0] = "Maybe you should try asking a human?";
    }
    do {
      decr (help_ptr);
      mp_print (mp, help_line[help_ptr]);
      mp_print_ln (mp);
    } while (help_ptr != 0);
  };
  help_ptr=4; 
  help_line[3] = "Sorry, I already gave what help I could...";
  help_line[2] = "Maybe you should try asking a human?";
  help_line[1] = "An error might have occurred before I noticed any problems.";
  help_line[0] = "``If all else fails, read the instructions.''";
  goto CONTINUE;
}


@ @<Print the string |err_help|, possibly on several lines@>=
{
  size_t j = 0;
  while (j < mp->err_help->len) {
    if (*(mp->err_help->str + j) != '%')
      mp_print (mp, (const char *) (mp->err_help->str + j));
    else if (j + 1 == mp->err_help->len)
      mp_print_ln (mp);
    else if (*(mp->err_help->str + j) != '%')
      mp_print_ln (mp);
    else {
      j++;
      mp_print_char (mp, xord ('%'));
    };
    j++;
  }
}


@ @<Put help message on the transcript file@>=
if (mp->interaction > mp_batch_mode)
  decr (mp->selector);          /* avoid terminal output */
if (mp->use_err_help) {
  mp_print_nl (mp, "");
  @<Print the string |err_help|, possibly on several lines@>;
} else {
  while (help_ptr > 0) {
    decr (help_ptr);
    mp_print_nl (mp, help_line[help_ptr]);
  };
  mp_print_ln (mp);
  if (mp->interaction > mp_batch_mode)
    incr (mp->selector);        /* re-enable terminal output */
  mp_print_ln (mp);
}


@ In anomalous cases, the print selector might be in an unknown state;
the following subroutine is called to fix things just enough to keep
running a bit longer.

@c
void mp_normalize_selector (MP mp) {
  if (mp->log_opened)
    mp->selector = term_and_log;
  else
    mp->selector = term_only;
  if (mp->job_name == NULL)
    mp_open_log_file (mp);
  if (mp->interaction == mp_batch_mode)
    decr (mp->selector);
}


@ The following procedure prints \MP's last words before dying.

@<Error hand...@>=
void mp_fatal_error (MP mp, const char *s) {                               /* prints |s|, and that's it */
  const char *hlp[] = {s, NULL} ;
  mp_normalize_selector (mp);
  if ( mp->interaction==mp_error_stop_mode )
    mp->interaction=mp_scroll_mode; /* no more interaction */
  if ( mp->log_opened ) 
    mp_error(mp, "Emergency stop", hlp, true);
  mp->history=mp_fatal_error_stop; 
  mp_jump_out(mp); /* irrecoverable error */
@.Emergency stop@>
}


@ @<Exported function ...@>=
extern void mp_fatal_error (MP mp, const char *s);


@ @<Internal library declarations@>=
void mp_overflow (MP mp, const char *s, integer n);


@ @<Error hand...@>=
void mp_overflow (MP mp, const char *s, integer n) {                               /* stop due to finiteness */
  char msg[256];
  const char *hlp[] = {
         "If you really absolutely need more capacity,",
         "you can ask a wizard to enlarge me.",
         NULL };
  mp_normalize_selector (mp);
  mp_snprintf (msg, 256, "MetaPost capacity exceeded, sorry [%s=%d]", s, (int) n);
@.MetaPost capacity exceeded ...@>;
  if ( mp->interaction==mp_error_stop_mode )
    mp->interaction=mp_scroll_mode; /* no more interaction */
  if ( mp->log_opened ) 
    mp_error(mp, msg, hlp, true);
  mp->history=mp_fatal_error_stop; 
  mp_jump_out(mp); /* irrecoverable error */
}


@ The program might sometime run completely amok, at which point there is
no choice but to stop. If no previous error has been detected, that's bad
news; a message is printed that is really intended for the \MP\
maintenance person instead of the user (unless the user has been
particularly diabolical).  The index entries for `this can't happen' may
help to pinpoint the problem.
@^dry rot@>

@<Internal library ...@>=
void mp_confusion (MP mp, const char *s);

@ Consistency check violated; |s| tells where.
@<Error hand...@>=
void mp_confusion (MP mp, const char *s) {
  char msg[256];
  const char *hlp[] = { 
           "One of your faux pas seems to have wounded me deeply...",
           "in fact, I'm barely conscious. Please fix it and try again.",
           NULL };
  mp_normalize_selector (mp);
  if (mp->history < mp_error_message_issued) {
    mp_snprintf (msg, 256, "This can't happen (%s)", s);
@.This can't happen@>;
    hlp[0] = "I'm broken. Please show this to someone who can fix can fix";
    hlp[1] = NULL;
  } else {
    mp_snprintf (msg, 256, "I can\'t go on meeting you like this");
@.I can't go on...@>;
  }
  if ( mp->interaction==mp_error_stop_mode )
    mp->interaction=mp_scroll_mode; /* no more interaction */
  if ( mp->log_opened ) 
    mp_error(mp, msg, hlp, true);
  mp->history=mp_fatal_error_stop; 
  mp_jump_out(mp); /* irrecoverable error */
}


@ Users occasionally want to interrupt \MP\ while it's running.
If the runtime system allows this, one can implement
a routine that sets the global variable |interrupt| to some nonzero value
when such an interrupt is signaled. Otherwise there is probably at least
a way to make |interrupt| nonzero using the C debugger.
@^system dependencies@>
@^debugging@>

@d check_interrupt { if ( mp->interrupt!=0 )
   mp_pause_for_instructions(mp); }

@<Global...@>=
integer interrupt;      /* should \MP\ pause for instructions? */
boolean OK_to_interrupt;        /* should interrupts be observed? */
integer run_state;      /* are we processing input ? */
boolean finished;       /* set true by |close_files_and_terminate| */
boolean reading_preload;

@ @<Allocate or ...@>=
mp->OK_to_interrupt = true;
mp->finished = false;

@ When an interrupt has been detected, the program goes into its
highest interaction level and lets the user have the full flexibility of
the |error| routine.  \MP\ checks for interrupts only at times when it is
safe to do this.

@c
static void mp_pause_for_instructions (MP mp) {
  const char *hlp[] = { "You rang?",
	           "Try to insert some instructions for me (e.g.,`I show x'),",
        	   "unless you just want to quit by typing `X'.", 
	           NULL } ;
  if (mp->OK_to_interrupt) {
    mp->interaction = mp_error_stop_mode;
    if ((mp->selector == log_only) || (mp->selector == no_print))
      incr (mp->selector);
@.Interruption@>;
    mp_error (mp, "Interruption", hlp, false);
    mp->interrupt = 0;
  }
}


@* Arithmetic with scaled numbers.
The principal computations performed by \MP\ are done entirely in terms of
integers less than $2^{31}$ in magnitude; thus, the arithmetic specified in this
program can be carried out in exactly the same way on a wide variety of
computers, including some small ones.
@^small computers@>

But C does not rigidly define the |/| operation in the case of negative
dividends; for example, the result of |(-2*n-1) / 2| is |-(n+1)| on some
computers and |-n| on others (is this true ?).  There are two principal
types of arithmetic: ``translation-preserving,'' in which the identity
|(a+q*b)/b=(a/b)+q| is valid; and ``negation-preserving,'' in which
|(-a)/b=-(a/b)|. This leads to two \MP s, which can produce
different results, although the differences should be negligible when the
language is being used properly.  The \TeX\ processor has been defined
carefully so that both varieties of arithmetic will produce identical
output, but it would be too inefficient to constrain \MP\ in a similar way.

@d inf_t  ((math_data *)mp->math)->inf_t

@ A single computation might use several subroutine calls, and it is
desirable to avoid producing multiple error messages in case of arithmetic
overflow. So the routines below set the global variable |arith_error| to |true|
instead of reporting errors directly to the user.
@^overflow in arithmetic@>

@<Glob...@>=
boolean arith_error;    /* has arithmetic overflow occurred recently? */

@ @<Allocate or ...@>=
mp->arith_error = false;

@ At crucial points the program will say |check_arith|, to test if
an arithmetic error has been detected.

@d check_arith() do { 
  if ( mp->arith_error ) 
    mp_clear_arith(mp); 
} while (0)

@c
static void mp_clear_arith (MP mp) {
  const char *hlp[] = {
         "Uh, oh. A little while ago one of the quantities that I was",
         "computing got too large, so I'm afraid your answers will be",
         "somewhat askew. You'll probably have to adopt different",
         "tactics next time. But I shall try to carry on anyway.",
         NULL };
  mp_error (mp, "Arithmetic overflow", hlp, true);
@.Arithmetic overflow@>;
  mp->arith_error = false;
}


@ The definitions of these are set up by the math initialization.

@d arc_tol_k ((math_data *)mp->math)->arc_tol_k
@d coef_bound_k ((math_data *)mp->math)->coef_bound_k
@d coef_bound_minus_1 ((math_data *)mp->math)->coef_bound_minus_1
@d sqrt_8_e_k ((math_data *)mp->math)->sqrt_8_e_k
@d twelve_ln_2_k ((math_data *)mp->math)->twelve_ln_2_k
@d twelvebits_3 ((math_data *)mp->math)->twelvebits_3
@d one_k  ((math_data *)mp->math)->one_k
@d epsilon_t  ((math_data *)mp->math)->epsilon_t
@d unity_t  ((math_data *)mp->math)->unity_t
@d zero_t  ((math_data *)mp->math)->zero_t
@d two_t ((math_data *)mp->math)->two_t
@d three_t  ((math_data *)mp->math)->three_t
@d half_unit_t ((math_data *)mp->math)->half_unit_t
@d three_quarter_unit_t ((math_data *)mp->math)->three_quarter_unit_t
@d twentysixbits_sqrt2_t ((math_data *)mp->math)->twentysixbits_sqrt2_t
@d twentyeightbits_d_t ((math_data *)mp->math)->twentyeightbits_d_t
@d twentysevenbits_sqrt2_d_t ((math_data *)mp->math)->twentysevenbits_sqrt2_d_t
@d warning_limit_t ((math_data *)mp->math)->warning_limit_t
@d precision_default  ((math_data *)mp->math)->precision_default
@d precision_max  ((math_data *)mp->math)->precision_max
@d precision_min  ((math_data *)mp->math)->precision_min

@ In fact, the two sorts of scaling discussed above aren't quite
sufficient; \MP\ has yet another, used internally to keep track of angles.

@ We often want to print two scaled quantities in parentheses,
separated by a comma.

@<Basic printing...@>=
void mp_print_two (MP mp, mp_number x, mp_number y) {                               /* prints `|(x,y)|' */
  mp_print_char (mp, xord ('('));
  print_number (x);
  mp_print_char (mp, xord (','));
  print_number (y);
  mp_print_char (mp, xord (')'));
}


@ 
@d fraction_one_t ((math_data *)mp->math)->fraction_one_t
@d fraction_half_t ((math_data *)mp->math)->fraction_half_t
@d fraction_three_t ((math_data *)mp->math)->fraction_three_t
@d fraction_four_t ((math_data *)mp->math)->fraction_four_t

@d one_eighty_deg_t ((math_data *)mp->math)->one_eighty_deg_t
@d three_sixty_deg_t ((math_data *)mp->math)->three_sixty_deg_t

@ @<Local variables for initialization@>=
integer k;  /* all-purpose loop index */

@ And now let's complete our collection of numeric utility routines
by considering random number generation.
\MP\ generates pseudo-random numbers with the additive scheme recommended
in Section 3.6 of {\sl The Art of Computer Programming}; however, the
results are random fractions between 0 and |fraction_one-1|, inclusive.

There's an auxiliary array |randoms| that contains 55 pseudo-random
fractions. Using the recurrence $x_n=(x_{n-55}-x_{n-31})\bmod 2^{28}$,
we generate batches of 55 new $x_n$'s at a time by calling |new_randoms|.
The global variable |j_random| tells which element has most recently
been consumed.
The global variable |random_seed| was introduced in version 0.9,
for the sole reason of stressing the fact that the initial value of the
random seed is system-dependant. The initialization code below will initialize
this variable to |(internal[mp_time] div unity)+internal[mp_day]|, but this 
is not good enough on modern fast machines that are capable of running
multiple MetaPost processes within the same second.
@^system dependencies@>

@<Glob...@>=
mp_number randoms[55];   /* the last 55 random values generated */
int j_random;   /* the number of unused |randoms| */

@ @<Option variables@>=
int random_seed;        /* the default random seed */

@ @<Allocate or initialize ...@>=
mp->random_seed = opt->random_seed;
{
  int i;
  for (i=0;i<55;i++) {
     new_fraction (mp->randoms[i]);
  }
}

@ @<Dealloc...@>=
{
  int i;
  for (i=0;i<55;i++) {
     free_number (mp->randoms[i]);
  }
}

@ @<Internal library ...@>=
void mp_new_randoms (MP mp);

@ @c
void mp_new_randoms (MP mp) {
  int k;        /* index into |randoms| */
  mp_number x;   /* accumulator */
  new_number (x);
  for (k = 0; k <= 23; k++) {
    set_number_from_substraction(x, mp->randoms[k], mp->randoms[k + 31]);
    if (number_negative(x))
      number_add (x, fraction_one_t);
    number_clone (mp->randoms[k], x);
  }
  for (k = 24; k <= 54; k++) {
    set_number_from_substraction(x, mp->randoms[k], mp->randoms[k - 24]);
    if (number_negative(x))
      number_add (x, fraction_one_t);
    number_clone (mp->randoms[k], x);
  }
  free_number (x);
  mp->j_random = 54;
}

@ To consume a random fraction, the program below will say `|next_random|'.

@c 
static void mp_next_random (MP mp, mp_number *ret) { 
  if ( mp->j_random==0 ) 
    mp_new_randoms(mp);
  else 
    decr(mp->j_random); 
  number_clone (*ret, mp->randoms[mp->j_random]);
}


@ To produce a uniform random number in the range |0<=u<x| or |0>=u>x|
or |0=u=x|, given a |scaled| value~|x|, we proceed as shown here.

Note that the call of |take_fraction| will produce the values 0 and~|x|
with about half the probability that it will produce any other particular
values between 0 and~|x|, because it rounds its answers.

@c
static void mp_unif_rand (MP mp, mp_number *ret, mp_number x_orig) {
  mp_number y;     /* trial value */
  mp_number x, abs_x;
  mp_number u;
  new_fraction (y);
  new_number (x);
  new_number (abs_x);
  new_number (u);
  number_clone (x, x_orig);
  number_clone (abs_x, x);
  number_abs (abs_x);
  mp_next_random(mp, &u);
  take_fraction (y, abs_x, u);
  free_number (u);
  if (number_equal(y, abs_x)) {
    set_number_to_zero(*ret);
  } else if (number_positive(x)) {
    number_clone (*ret, y);
  } else {
    number_clone (*ret, y);
    number_negate (*ret);
  }
  free_number (abs_x);
  free_number (x);
  free_number (y);
}


@ Finally, a normal deviate with mean zero and unit standard deviation
can readily be obtained with the ratio method (Algorithm 3.4.1R in
{\sl The Art of Computer Programming\/}).

@c
static void mp_norm_rand (MP mp, mp_number *ret) {
  mp_number ab_vs_cd; 
  mp_number abs_x;
  mp_number u;
  mp_number r;
  mp_number la, xa;
  new_number (ab_vs_cd);
  new_number (la);
  new_number (xa);
  new_number (abs_x);
  new_number (u);
  new_number (r);
  do {
    do {
      mp_number v;
      new_number (v);
      mp_next_random(mp, &v);
      number_substract (v, fraction_half_t);
      take_fraction (xa, sqrt_8_e_k, v); 
      free_number (v);
      mp_next_random(mp, &u);
      number_clone (abs_x, xa);
      number_abs (abs_x);
    } while (number_greaterequal (abs_x, u));
    make_fraction (r, xa, u);
    number_clone (xa, r);
    m_log (la, u);
    set_number_from_substraction(la, twelve_ln_2_k, la);
    ab_vs_cd (ab_vs_cd, one_k, la, xa, xa);
  } while (number_negative(ab_vs_cd));
  number_clone (*ret, xa);
  free_number (ab_vs_cd);
  free_number (r);
  free_number (abs_x);
  free_number (la);
  free_number (xa);
  free_number (u);
}


@* Packed data.

@d max_quarterword 0x3FFF /* largest allowable value in a |quarterword| */
@d max_halfword 0xFFFFFFF /* largest allowable value in a |halfword| */

@ The macros |qi| and |qo| are used for input to and output 
from quarterwords. These are legacy macros.
@^system dependencies@>

@d qo(A) (A) /* to read eight bits from a quarterword */
@d qi(A) (quarterword)(A) /* to store eight bits in a quarterword */

@ The reader should study the following definitions closely:
@^system dependencies@>

@<Types...@>=
typedef struct mp_value_node_data *mp_value_node;
typedef struct mp_node_data *mp_node;
typedef struct mp_symbol_entry *mp_sym;
typedef short quarterword;      /* 1/4 of a word */
typedef int halfword;   /* 1/2 of a word */
typedef struct {
  integer scale; /* only for |indep_scale|, used together with |serial| */
  integer serial; /* only for |indep_value|, used together with |scale| */
} mp_independent_data;
typedef struct {
  mp_independent_data indep;
  mp_number n;
  mp_string str;
  mp_sym sym;
  mp_node node;
  mp_knot p;
} mp_value_data;
typedef struct {
  mp_variable_type type;
  mp_value_data data;
} mp_value;
typedef struct {
  quarterword b0, b1, b2, b3;
} four_quarters;
typedef union {
  integer sc;
  four_quarters qqqq;
} font_data;


@ The global variable |math_mode| has four settings, representing the
math value type that will be used in this run.

the typedef for |mp_number| is here because it has to come very early.
 
@<Exported types@>=
typedef enum {
  mp_math_scaled_mode = 0,
  mp_math_double_mode = 1,
  mp_math_binary_mode = 2,
  mp_math_decimal_mode = 3
} mp_math_mode;

@ @<Option variables@>=
int math_mode;               /* math mode */

@ @<Allocate or initialize ...@>=
mp->math_mode = opt->math_mode;

@ 
@d xfree(A) do { mp_xfree(A); A=NULL; } while (0)
@d xrealloc(P,A,B) mp_xrealloc(mp,P,(size_t)A,B)
@d xmalloc(A,B)  mp_xmalloc(mp,(size_t)A,B)
@d xstrdup(A)  mp_xstrdup(mp,A)
@d XREALLOC(a,b,c) a = xrealloc(a,(b+1),sizeof(c));

@<Declare helpers@>=
extern void mp_xfree (void *x);
extern void *mp_xrealloc (MP mp, void *p, size_t nmem, size_t size);
extern void *mp_xmalloc (MP mp, size_t nmem, size_t size);
extern void mp_do_snprintf (char *str, int size, const char *fmt, ...);
extern void *do_alloc_node(MP mp, size_t size);

@ This is an attempt to spend less time in |malloc()|:

@d max_num_token_nodes 1000
@d max_num_pair_nodes 1000
@d max_num_knot_nodes 1000
@d max_num_value_nodes 1000
@d max_num_symbolic_nodes 1000

@<Global ...@>=
mp_node token_nodes;
int num_token_nodes;
mp_node pair_nodes;
int num_pair_nodes;
mp_knot knot_nodes;
int num_knot_nodes;
mp_node value_nodes;
int num_value_nodes;
mp_node symbolic_nodes;
int num_symbolic_nodes;

@ @<Allocate or initialize ...@>=
mp->token_nodes = NULL;
mp->num_token_nodes = 0;
mp->pair_nodes = NULL;
mp->num_pair_nodes = 0;
mp->knot_nodes = NULL;
mp->num_knot_nodes = 0;
mp->value_nodes = NULL;
mp->num_value_nodes = 0;
mp->symbolic_nodes = NULL;
mp->num_symbolic_nodes = 0;

@ @<Dealloc ...@>=
while (mp->value_nodes) {
      mp_node p = mp->value_nodes;
      mp->value_nodes = p->link;
      mp_free_node(mp,p,value_node_size);
}
while (mp->symbolic_nodes) {
      mp_node p = mp->symbolic_nodes;
      mp->symbolic_nodes = p->link;
      mp_free_node(mp,p,symbolic_node_size);
}
while (mp->pair_nodes) {
      mp_node p = mp->pair_nodes;
      mp->pair_nodes = p->link;
      mp_free_node(mp,p,pair_node_size);
}
while (mp->token_nodes) {
      mp_node p = mp->token_nodes;
      mp->token_nodes = p->link;
      mp_free_node(mp,p,token_node_size);
}
while (mp->knot_nodes) {
      mp_knot p = mp->knot_nodes;
      mp->knot_nodes = p->next;
      mp_free_knot(mp,p);
}

@ This is a nicer way of allocating nodes.

@d malloc_node(A) do_alloc_node(mp,(A))

@ 
@c
void *do_alloc_node (MP mp, size_t size) {
    void *p;
    p = xmalloc(1,size);
    add_var_used (size);
    ((mp_node)p)->link = NULL;
    ((mp_node)p)->has_number = 0;
    return p;
}


@ The |max_size_test| guards against overflow, on the assumption that
|size_t| is at least 31bits wide.

@d max_size_test 0x7FFFFFFF

@c
void mp_xfree (void *x) {
  if (x != NULL)
    free (x);
}
void *mp_xrealloc (MP mp, void *p, size_t nmem, size_t size) {
  void *w;
  if ((max_size_test / size) < nmem) {
    mp_fputs ("Memory size overflow!\n", mp->err_out);
    mp->history = mp_fatal_error_stop;
    mp_jump_out (mp);
  }
  w = realloc (p, (nmem * size));
  if (w == NULL) {
    mp_fputs ("Out of memory!\n", mp->err_out);
    mp->history = mp_system_error_stop;
    mp_jump_out (mp);
  }
  return w;
}
void *mp_xmalloc (MP mp, size_t nmem, size_t size) {
  void *w;
#if DEBUG
  if ((max_size_test / size) < nmem) {
    mp_fputs ("Memory size overflow!\n", mp->err_out);
    mp->history = mp_fatal_error_stop;
    mp_jump_out (mp);
  }
#endif
  w = malloc (nmem * size);
  if (w == NULL) {
    mp_fputs ("Out of memory!\n", mp->err_out);
    mp->history = mp_system_error_stop;
    mp_jump_out (mp);
  }
  return w;
}

@ @<Internal library declarations@>=
#  define mp_snprintf (void)snprintf

@* Dynamic memory allocation.

The \MP\ system does nearly all of its own memory allocation, so that it
can readily be transported into environments that do not have automatic
facilities for strings, garbage collection, etc., and so that it can be in
control of what error messages the user receives. 

@d MP_VOID (mp_node)(1) /* |NULL+1|, a |NULL| pointer different from |NULL| */

@d mp_link(A)      (A)->link /* the |link| field of a node */
@d set_mp_link(A,B) do {
   mp_node d = (B);
   /* |printf("set link    of %p to %p on line %d\n", (A), d, __LINE__);| */
   mp_link((A)) = d;
 } while (0)
@d mp_type(A)      (A)->type /* identifies what kind of value this is */
@d mp_name_type(A) (A)->name_type /* a clue to the name of this value */

@ @<MPlib internal header stuff@>=
#define NODE_BODY                       \
  mp_variable_type type;                \
  mp_name_type_type name_type;          \
  unsigned short has_number;		\
  struct mp_node_data *link
typedef struct mp_node_data {
  NODE_BODY;
  mp_value_data data; 
} mp_node_data;
typedef struct mp_node_data *mp_symbolic_node;

@ Users who wish to study the memory requirements of particular applications can
can use the special features that keep track of current and maximum memory usage. 
\MP\ will report these statistics when |mp_tracing_stats| is positive.

@d add_var_used(a) do {
   mp->var_used+=(a);
   if (mp->var_used>mp->var_used_max) mp->var_used_max=mp->var_used;
} while (0)

@<Glob...@>=
size_t var_used;        /* how much memory is in use */
size_t var_used_max;    /* how much memory was in use max */

@ These redirect to function to aid in debugging.

@c
#if DEBUG
#define mp_sym_info(A)       get_mp_sym_info(mp,(A))
#define set_mp_sym_info(A,B) do_set_mp_sym_info(mp,(A),(B))
#define mp_sym_sym(A)        get_mp_sym_sym(mp,(A))
#define set_mp_sym_sym(A,B)  do_set_mp_sym_sym(mp,(A),(mp_sym)(B))
static void do_set_mp_sym_info (MP mp, mp_node p, halfword v) {
  FUNCTION_TRACE3 ("do_set_mp_sym_info(%p,%d)\n", p, v);
  assert (p->type == mp_symbol_node);
  set_indep_value(p, v);
}
static halfword get_mp_sym_info (MP mp, mp_node p) {
  FUNCTION_TRACE3 ("%d = get_mp_sym_info(%p)\n", indep_value (p), p);
  assert (p->type == mp_symbol_node);
  return indep_value(p);
}
static void do_set_mp_sym_sym (MP mp, mp_node p, mp_sym v) {
  mp_symbolic_node pp = (mp_symbolic_node) p;
  FUNCTION_TRACE3 ("do_set_mp_sym_sym(%p,%p)\n", pp, v);
  assert (pp->type == mp_symbol_node);
  pp->data.sym = v;
}
static mp_sym get_mp_sym_sym (MP mp, mp_node p) {
  mp_symbolic_node pp = (mp_symbolic_node) p;
  FUNCTION_TRACE3 ("%p = get_mp_sym_sym(%p)\n", pp->data.sym, pp);
  assert (pp->type == mp_symbol_node);
  return pp->data.sym;
}
#else
#define mp_sym_info(A)        indep_value(A)
#define set_mp_sym_info(A,B)  set_indep_value(A, (B))
#define mp_sym_sym(A)        (A)->data.sym
#define set_mp_sym_sym(A,B)  (A)->data.sym = (mp_sym)(B)
#endif

@ @<Declarations@>=
#if DEBUG
static void do_set_mp_sym_info (MP mp, mp_node A, halfword B);
static halfword get_mp_sym_info (MP mp, mp_node p);
static void do_set_mp_sym_sym (MP mp, mp_node A, mp_sym B);
static mp_sym get_mp_sym_sym (MP mp, mp_node p);
#endif

@ The function |get_symbolic_node| returns a pointer to a new symbolic node whose
|link| field is null.
@^inner loop@>

@d symbolic_node_size sizeof(mp_node_data)
@c
static mp_node mp_get_symbolic_node (MP mp) {
  mp_symbolic_node p;
  if (mp->symbolic_nodes) {
    p = (mp_symbolic_node)mp->symbolic_nodes;
    mp->symbolic_nodes = p->link;
    mp->num_symbolic_nodes--;
    p->link = NULL;
  } else {
    p = malloc_node (symbolic_node_size);
    new_number(p->data.n);
    p->has_number = 1;
  }
  p->type = mp_symbol_node;
  p->name_type = mp_normal_sym;
  FUNCTION_TRACE2 ("%p = mp_get_symbolic_node()\n", p);
  return (mp_node) p;
}


@ Conversely, when some node |p| of size |s| is no longer needed,
the operation |free_node(p,s)| will make its words available, by inserting
|p| as a new empty node just before where |rover| now points.

A symbolic node is recycled by calling |free_symbolic_node|.

@c
void mp_free_node (MP mp, mp_node p, size_t siz) {  /* node liberation */
  FUNCTION_TRACE3 ("mp_free_node(%p,%d)\n", p, (int)siz);
  if (!p) return;
  mp->var_used -= siz;
  if (mp->math_mode > mp_math_double_mode) {
    if (p->has_number >= 1 && is_number(((mp_symbolic_node)p)->data.n)) {
      free_number(((mp_symbolic_node)p)->data.n); 
    }
    if (p->has_number == 2 && is_number(((mp_value_node)p)->subscript_)) {
      free_number(((mp_value_node)p)->subscript_); 
    }
    /* There was a quite large |switch| here first, but the |mp_dash_node|
       case was the only one that did anything ... */
    if (mp_type (p) == mp_dash_node_type) {
      free_number(((mp_dash_node)p)->start_x);
      free_number(((mp_dash_node)p)->stop_x);
      free_number(((mp_dash_node)p)->dash_y);
    }
  }
  xfree (p);
}
void mp_free_symbolic_node (MP mp, mp_node p) {  /* node liberation */
  FUNCTION_TRACE2 ("mp_free_symbolic_node(%p)\n", p);
  if (!p) return;
  if (mp->num_symbolic_nodes < max_num_symbolic_nodes) {
    p->link = mp->symbolic_nodes;
    mp->symbolic_nodes = p;
    mp->num_symbolic_nodes++;
    return;
  }
  mp->var_used -= symbolic_node_size;
  xfree (p);
}
void mp_free_value_node (MP mp, mp_node p) {  /* node liberation */
  FUNCTION_TRACE2 ("mp_free_value_node(%p)\n", p);
  if (!p) return;
  if (mp->num_value_nodes < max_num_value_nodes) {
    p->link = mp->value_nodes;
    mp->value_nodes = p;
    mp->num_value_nodes++;
    return;
  }
  mp->var_used -= value_node_size;
  assert(p->has_number == 2);
  if (mp->math_mode > mp_math_double_mode) {
    free_number(((mp_value_node)p)->data.n); 
    free_number(((mp_value_node)p)->subscript_); 
  }
  xfree (p);
}


@ @<Internal library declarations@>=
void mp_free_node (MP mp, mp_node p, size_t siz);
void mp_free_symbolic_node (MP mp, mp_node p);
void mp_free_value_node (MP mp, mp_node p);

@* Memory layout.
Some nodes are created statically, since static allocation is
more efficient than dynamic allocation when we can get away with it. 

@<Glob...@>=
mp_dash_node null_dash;
mp_value_node dep_head;
mp_node inf_val;
mp_node zero_val;
mp_node temp_val;
mp_node end_attr;
mp_node bad_vardef;
mp_node temp_head;
mp_node hold_head;
mp_node spec_head;

@ The following code gets the memory off to a good start.

@<Initialize table entries@>=
mp->spec_head = mp_get_symbolic_node (mp);
mp->last_pending = mp->spec_head;
mp->temp_head = mp_get_symbolic_node (mp);
mp->hold_head = mp_get_symbolic_node (mp);

@ @<Free table entries@>=
mp_free_symbolic_node (mp, mp->spec_head);
mp_free_symbolic_node (mp, mp->temp_head);
mp_free_symbolic_node (mp, mp->hold_head);

@ The procedure |flush_node_list(p)| frees an entire linked list of 
nodes that starts at a given position, until coming to a |NULL| pointer.
@^inner loop@>

@c
static void mp_flush_node_list (MP mp, mp_node p) {
  mp_node q;    /* the node being recycled */
  FUNCTION_TRACE2 ("mp_flush_node_list(%p)\n", p);
  while (p != NULL) {
    q = p;
    p = p->link;
    if (q->type != mp_symbol_node)
      mp_free_token_node (mp, q);
    else
      mp_free_symbolic_node (mp, q);
  }
}


@* The command codes.
Before we can go much further, we need to define symbolic names for the internal
code numbers that represent the various commands obeyed by \MP. These codes
are somewhat arbitrary, but not completely so. For example,
some codes have been made adjacent so that |case| statements in the
program need not consider cases that are widely spaced, or so that |case|
statements can be replaced by |if| statements. A command can begin an
expression if and only if its code lies between |min_primary_command| and
|max_primary_command|, inclusive. The first token of a statement that doesn't
begin with an expression has a command code between |min_command| and
|max_statement_command|, inclusive. Anything less than |min_command| is
eliminated during macro expansions, and anything no more than |max_pre_command|
is eliminated when expanding \TeX\ material.  Ranges such as
|min_secondary_command..max_secondary_command| are used when parsing
expressions, but the relative ordering within such a range is generally not
critical.

The ordering of the highest-numbered commands
(|comma<semicolon<end_group<stop|) is crucial for the parsing and
error-recovery methods of this program as is the ordering |if_test<fi_or_else|
for the smallest two commands.  The ordering is also important in the ranges
|numeric_token..plus_or_minus| and |left_brace..ampersand|.

At any rate, here is the list, for future reference.

@d mp_max_command_code mp_stop
@d mp_max_pre_command mp_mpx_break
@d mp_min_command (mp_defined_macro+1)
@d mp_max_statement_command mp_type_name
@d mp_min_primary_command mp_type_name
@d mp_min_suffix_token mp_internal_quantity
@d mp_max_suffix_token mp_numeric_token
@d mp_max_primary_command mp_plus_or_minus /* should also be |numeric_token+1| */
@d mp_min_tertiary_command mp_plus_or_minus
@d mp_max_tertiary_command mp_tertiary_binary
@d mp_min_expression_command mp_left_brace
@d mp_max_expression_command mp_equals
@d mp_min_secondary_command mp_and_command
@d mp_max_secondary_command mp_secondary_binary
@d mp_end_of_statement (cur_cmd()>mp_comma)


@<Enumeration types@>=
typedef enum {
mp_start_tex=1, /* begin \TeX\ material (\&{btex}, \&{verbatimtex}) */
mp_etex_marker, /* end \TeX\ material (\&{etex}) */
mp_mpx_break, /* stop reading an \.{MPX} file (\&{mpxbreak}) */
mp_if_test, /* conditional text (\&{if}) */
mp_fi_or_else, /* delimiters for conditionals (\&{elseif}, \&{else}, \&{fi}) */
mp_input, /* input a source file (\&{input}, \&{endinput}) */
mp_iteration, /* iterate (\&{for}, \&{forsuffixes}, \&{forever}, \&{endfor}) */
mp_repeat_loop, /* special command substituted for \&{endfor} */
mp_exit_test, /* premature exit from a loop (\&{exitif}) */
mp_relax, /* do nothing (\.{\char`\\}) */
mp_scan_tokens, /* put a string into the input buffer */
mp_runscript, /* put a script result string into the input buffer */
mp_expand_after, /* look ahead one token */
mp_defined_macro, /* a macro defined by the user */
mp_save_command, /* save a list of tokens (\&{save}) */
mp_interim_command, /* save an internal quantity (\&{interim}) */
mp_let_command, /* redefine a symbolic token (\&{let}) */
mp_new_internal, /* define a new internal quantity (\&{newinternal}) */
mp_macro_def, /* define a macro (\&{def}, \&{vardef}, etc.) */
mp_ship_out_command, /* output a character (\&{shipout}) */
mp_add_to_command, /* add to edges (\&{addto}) */
mp_bounds_command,  /* add bounding path to edges (\&{setbounds}, \&{clip}) */
mp_tfm_command, /* command for font metric info (\&{ligtable}, etc.) */
mp_protection_command, /* set protection flag (\&{outer}, \&{inner}) */
mp_show_command, /* diagnostic output (\&{show}, \&{showvariable}, etc.) */
mp_mode_command, /* set interaction level (\&{batchmode}, etc.) */
mp_random_seed, /* initialize random number generator (\&{randomseed}) */
mp_message_command, /* communicate to user (\&{message}, \&{errmessage}) */
mp_every_job_command, /* designate a starting token (\&{everyjob}) */
mp_delimiters, /* define a pair of delimiters (\&{delimiters}) */
mp_special_command, /* output special info (\&{special})
                       or font map info (\&{fontmapfile}, \&{fontmapline}) */
mp_write_command, /* write text to a file (\&{write}) */
mp_type_name, /* declare a type (\&{numeric}, \&{pair}, etc.) */
mp_left_delimiter, /* the left delimiter of a matching pair */
mp_begin_group, /* beginning of a group (\&{begingroup}) */
mp_nullary, /* an operator without arguments (e.g., \&{normaldeviate}) */
mp_unary, /* an operator with one argument (e.g., \&{sqrt}) */
mp_str_op, /* convert a suffix to a string (\&{str}) */
mp_cycle, /* close a cyclic path (\&{cycle}) */
mp_primary_binary, /* binary operation taking `\&{of}' (e.g., \&{point}) */
mp_capsule_token, /* a value that has been put into a token list */
mp_string_token, /* a string constant (e.g., |"hello"|) */
mp_internal_quantity, /* internal numeric parameter (e.g., \&{pausing}) */
mp_tag_token, /* a symbolic token without a primitive meaning */
mp_numeric_token, /* a numeric constant (e.g., \.{3.14159}) */
mp_plus_or_minus, /* either `\.+' or `\.-' */
mp_tertiary_secondary_macro, /* a macro defined by \&{secondarydef} */
mp_tertiary_binary, /* an operator at the tertiary level (e.g., `\.{++}') */
mp_left_brace, /* the operator `\.{\char`\{}' */
mp_path_join, /* the operator `\.{..}' */
mp_ampersand, /* the operator `\.\&' */
mp_expression_tertiary_macro, /* a macro defined by \&{tertiarydef} */
mp_expression_binary, /* an operator at the expression level (e.g., `\.<') */
mp_equals, /* the operator `\.=' */
mp_and_command, /* the operator `\&{and}' */
mp_secondary_primary_macro, /* a macro defined by \&{primarydef} */
mp_slash, /* the operator `\./' */
mp_secondary_binary, /* an operator at the binary level (e.g., \&{shifted}) */
mp_param_type, /* type of parameter (\&{primary}, \&{expr}, \&{suffix}, etc.) */
mp_controls, /* specify control points explicitly (\&{controls}) */
mp_tension, /* specify tension between knots (\&{tension}) */
mp_at_least, /* bounded tension value (\&{atleast}) */
mp_curl_command, /* specify curl at an end knot (\&{curl}) */
mp_macro_special, /* special macro operators (\&{quote}, \.{\#\AT!}, etc.) */
mp_right_delimiter, /* the right delimiter of a matching pair */
mp_left_bracket, /* the operator `\.[' */
mp_right_bracket, /* the operator `\.]' */
mp_right_brace, /* the operator `\.{\char`\}}' */
mp_with_option, /* option for filling (\&{withpen}, \&{withweight}, etc.) */
mp_thing_to_add,
  /* variant of \&{addto} (\&{contour}, \&{doublepath}, \&{also}) */
mp_of_token, /* the operator `\&{of}' */
mp_to_token, /* the operator `\&{to}' */
mp_step_token, /* the operator `\&{step}' */
mp_until_token, /* the operator `\&{until}' */
mp_within_token, /* the operator `\&{within}' */
mp_lig_kern_token,
  /* the operators `\&{kern}' and `\.{=:}' and `\.{=:\char'174}', etc. */
mp_assignment, /* the operator `\.{:=}' */
mp_skip_to, /* the operation `\&{skipto}' */
mp_bchar_label, /* the operator `\.{\char'174\char'174:}' */
mp_double_colon, /* the operator `\.{::}' */
mp_colon, /* the operator `\.:' */
@#
mp_comma, /* the operator `\.,', must be |colon+1| */
mp_semicolon, /* the operator `\.;', must be |comma+1| */
mp_end_group, /* end a group (\&{endgroup}), must be |semicolon+1| */
mp_stop, /* end a job (\&{end}, \&{dump}), must be |end_group+1| */
mp_outer_tag, /* protection code added to command code */
mp_undefined_cs, /* protection code added to command code */
} mp_command_code;

@ Variables and capsules in \MP\ have a variety of ``types,''
distinguished by the code numbers defined here. These numbers are also
not completely arbitrary.  Things that get expanded must have types
|>mp_independent|; a type remaining after expansion is numeric if and only if
its code number is at least |numeric_type|; objects containing numeric
parts must have types between |transform_type| and |pair_type|;
all other types must be smaller than |transform_type|; and among the types
that are not unknown or vacuous, the smallest two must be |boolean_type|
and |string_type| in that order.

@d unknown_tag 1 /* this constant is added to certain type codes below */
@d unknown_types mp_unknown_boolean: case mp_unknown_string:
  case mp_unknown_pen: case mp_unknown_picture: case mp_unknown_path

@<Enumeration types@>=
typedef enum {
  mp_undefined = 0,       /* no type has been declared */
  mp_vacuous,                   /* no expression was present */
  mp_boolean_type,              /* \&{boolean} with a known value */
  mp_unknown_boolean,
  mp_string_type,               /* \&{string} with a known value */
  mp_unknown_string,
  mp_pen_type,                  /* \&{pen} with a known value */
  mp_unknown_pen,
  mp_path_type,                 /* \&{path} with a known value */
  mp_unknown_path,
  mp_picture_type,              /* \&{picture} with a known value */
  mp_unknown_picture,
  mp_transform_type,            /* \&{transform} variable or capsule */
  mp_color_type,                /* \&{color} variable or capsule */
  mp_cmykcolor_type,            /* \&{cmykcolor} variable or capsule */
  mp_pair_type,                 /* \&{pair} variable or capsule */
  mp_numeric_type,              /* variable that has been declared \&{numeric} but not used */
  mp_known,                     /* \&{numeric} with a known value */
  mp_dependent,                 /* a linear combination with |fraction| coefficients */
  mp_proto_dependent,           /* a linear combination with |scaled| coefficients */
  mp_independent,               /* \&{numeric} with unknown value */
  mp_token_list,                /* variable name or suffix argument or text argument */
  mp_structured,                /* variable with subscripts and attributes */
  mp_unsuffixed_macro,          /* variable defined with \&{vardef} but no \.{\AT!\#} */
  mp_suffixed_macro,            /* variable defined with \&{vardef} and \.{\AT!\#} */
/* here are some generic node types */
  mp_symbol_node,
  mp_token_node_type,
  mp_value_node_type,
  mp_attr_node_type,
  mp_subscr_node_type,
  mp_pair_node_type,
  mp_transform_node_type,
  mp_color_node_type,
  mp_cmykcolor_node_type,
/* it is important that the next 7 items remain in this order, for export */
  mp_fill_node_type,
  mp_stroked_node_type,
  mp_text_node_type,
  mp_start_clip_node_type,
  mp_start_bounds_node_type,
  mp_stop_clip_node_type,
  mp_stop_bounds_node_type,
  mp_dash_node_type,
  mp_dep_node_type,
  mp_if_node_type,
  mp_edge_header_node_type,
} mp_variable_type;

@ @<Declarations@>=
static void mp_print_type (MP mp, quarterword t);

@ @<Basic printing procedures@>=
static const char *mp_type_string (quarterword t) {
  const char *s = NULL;
  switch (t) {
  case mp_undefined:
    s = "undefined";
    break;
  case mp_vacuous:
    s = "vacuous";
    break;
  case mp_boolean_type:
    s = "boolean";
    break;
  case mp_unknown_boolean:
    s = "unknown boolean";
    break;
  case mp_string_type:
    s = "string";
    break;
  case mp_unknown_string:
    s = "unknown string";
    break;
  case mp_pen_type:
    s = "pen";
    break;
  case mp_unknown_pen:
    s = "unknown pen";
    break;
  case mp_path_type:
    s = "path";
    break;
  case mp_unknown_path:
    s = "unknown path";
    break;
  case mp_picture_type:
    s = "picture";
    break;
  case mp_unknown_picture:
    s = "unknown picture";
    break;
  case mp_transform_type:
    s = "transform";
    break;
  case mp_color_type:
    s = "color";
    break;
  case mp_cmykcolor_type:
    s = "cmykcolor";
    break;
  case mp_pair_type:
    s = "pair";
    break;
  case mp_known:
    s = "known numeric";
    break;
  case mp_dependent:
    s = "dependent";
    break;
  case mp_proto_dependent:
    s = "proto-dependent";
    break;
  case mp_numeric_type:
    s = "numeric";
    break;
  case mp_independent:
    s = "independent";
    break;
  case mp_token_list:
    s = "token list";
    break;
  case mp_structured:
    s = "mp_structured";
    break;
  case mp_unsuffixed_macro:
    s = "unsuffixed macro";
    break;
  case mp_suffixed_macro:
    s = "suffixed macro";
    break;
  case mp_symbol_node:
    s = "symbol node";
    break;
  case mp_token_node_type:
    s = "token node";
    break;
  case mp_value_node_type:
    s = "value node";
    break;
  case mp_attr_node_type:
    s = "attribute node";
    break;
  case mp_subscr_node_type:
    s = "subscript node";
    break;
  case mp_pair_node_type:
    s = "pair node";
    break;
  case mp_transform_node_type:
    s = "transform node";
    break;
  case mp_color_node_type:
    s = "color node";
    break;
  case mp_cmykcolor_node_type:
    s = "cmykcolor node";
    break;
  case mp_fill_node_type:
    s = "fill node";
    break;
  case mp_stroked_node_type:
    s = "stroked node";
    break;
  case mp_text_node_type:
    s = "text node";
    break;
  case mp_start_clip_node_type:
    s = "start clip node";
    break;
  case mp_start_bounds_node_type:
    s = "start bounds node";
    break;
  case mp_stop_clip_node_type:
    s = "stop clip node";
    break;
  case mp_stop_bounds_node_type:
    s = "stop bounds node";
    break;
  case mp_dash_node_type:
    s = "dash node";
    break;
  case mp_dep_node_type:
    s = "dependency node";
    break;
  case mp_if_node_type:
    s = "if node";
    break;
  case mp_edge_header_node_type:
    s = "edge header node";
    break;
  default:
    {
        char ss[256];
    	mp_snprintf (ss, 256, "<unknown type %d>", t);
	s = strdup(ss);
    }
    break;
  }
  return s;
}
void mp_print_type (MP mp, quarterword t) {
  if (t >= 0 && t <= mp_edge_header_node_type)
    mp_print (mp, mp_type_string (t));
  else
    mp_print (mp, "unknown");
}


@ Values inside \MP\ are stored in non-symbolic nodes that have a |name_type|
as well as a |type|. The possibilities for |name_type| are defined
here; they will be explained in more detail later.

@<Enumeration types...@>=
typedef enum {
  mp_root = 0,  /* |name_type| at the top level of a variable */
  mp_saved_root,                /* same, when the variable has been saved */
  mp_structured_root,           /* |name_type| where a |mp_structured| branch occurs */
  mp_subscr,                    /* |name_type| in a subscript node */
  mp_attr,                      /* |name_type| in an attribute node */
  mp_x_part_sector,             /* |name_type| in the \&{xpart} of a node */
  mp_y_part_sector,             /* |name_type| in the \&{ypart} of a node */
  mp_xx_part_sector,            /* |name_type| in the \&{xxpart} of a node */
  mp_xy_part_sector,            /* |name_type| in the \&{xypart} of a node */
  mp_yx_part_sector,            /* |name_type| in the \&{yxpart} of a node */
  mp_yy_part_sector,            /* |name_type| in the \&{yypart} of a node */
  mp_red_part_sector,           /* |name_type| in the \&{redpart} of a node */
  mp_green_part_sector,         /* |name_type| in the \&{greenpart} of a node */
  mp_blue_part_sector,          /* |name_type| in the \&{bluepart} of a node */
  mp_cyan_part_sector,          /* |name_type| in the \&{redpart} of a node */
  mp_magenta_part_sector,       /* |name_type| in the \&{greenpart} of a node */
  mp_yellow_part_sector,        /* |name_type| in the \&{bluepart} of a node */
  mp_black_part_sector,         /* |name_type| in the \&{greenpart} of a node */
  mp_grey_part_sector,          /* |name_type| in the \&{bluepart} of a node */
  mp_capsule,                   /* |name_type| in stashed-away subexpressions */
  mp_token,                     /* |name_type| in a numeric token or string token */
  /* Symbolic nodes also have |name_type|, which is a different enumeration */
  mp_normal_sym,
  mp_internal_sym,              /* for values of internals */
  mp_macro_sym,                 /* for macro names */
  mp_expr_sym,                  /* for macro parameters if type |expr| */
  mp_suffix_sym,                /* for macro parameters if type |suffix| */
  mp_text_sym,                  /* for macro parameters if type |text| */
  @<Operation codes@>
} mp_name_type_type;

@ Primitive operations that produce values have a secondary identification
code in addition to their command code; it's something like genera and species.
For example, `\.*' has the command code |primary_binary|, and its
secondary identification is |times|. The secondary codes start such that
they don't overlap with the type codes; some type codes (e.g., |mp_string_type|)
are used as operators as well as type identifications.  The relative values
are not critical, except for |true_code..false_code|, |or_op..and_op|,
and |filled_op..bounded_op|.  The restrictions are that
|and_op-false_code=or_op-true_code|, that the ordering of
|x_part...blue_part| must match that of |x_part_sector..mp_blue_part_sector|,
and the ordering of |filled_op..bounded_op| must match that of the code
values they test for.

@d mp_min_of mp_substring_of

@<Operation codes@>= 
mp_true_code, /* operation code for \.{true} */
mp_false_code, /* operation code for \.{false} */
mp_null_picture_code, /* operation code for \.{nullpicture} */
mp_null_pen_code, /* operation code for \.{nullpen} */ 
mp_read_string_op, /* operation code for \.{readstring} */
mp_pen_circle, /* operation code for \.{pencircle} */
mp_normal_deviate, /* operation code for \.{normaldeviate} */
mp_read_from_op, /* operation code for \.{readfrom} */
mp_close_from_op, /* operation code for \.{closefrom} */
mp_odd_op, /* operation code for \.{odd} */
mp_known_op, /* operation code for \.{known} */
mp_unknown_op, /* operation code for \.{unknown} */
mp_not_op, /* operation code for \.{not} */
mp_decimal, /* operation code for \.{decimal} */
mp_reverse, /* operation code for \.{reverse} */
mp_make_path_op, /* operation code for \.{makepath} */
mp_make_pen_op, /* operation code for \.{makepen} */
mp_oct_op, /* operation code for \.{oct} */
mp_hex_op, /* operation code for \.{hex} */
mp_ASCII_op, /* operation code for \.{ASCII} */
mp_char_op, /* operation code for \.{char} */
mp_length_op, /* operation code for \.{length} */
mp_turning_op, /* operation code for \.{turningnumber} */
mp_color_model_part, /* operation code for \.{colormodel} */
mp_x_part, /* operation code for \.{xpart} */
mp_y_part, /* operation code for \.{ypart} */
mp_xx_part, /* operation code for \.{xxpart} */
mp_xy_part, /* operation code for \.{xypart} */
mp_yx_part, /* operation code for \.{yxpart} */
mp_yy_part, /* operation code for \.{yypart} */
mp_red_part, /* operation code for \.{redpart} */
mp_green_part, /* operation code for \.{greenpart} */
mp_blue_part, /* operation code for \.{bluepart} */
mp_cyan_part, /* operation code for \.{cyanpart} */
mp_magenta_part, /* operation code for \.{magentapart} */
mp_yellow_part, /* operation code for \.{yellowpart} */
mp_black_part, /* operation code for \.{blackpart} */
mp_grey_part, /* operation code for \.{greypart} */
mp_font_part, /* operation code for \.{fontpart} */
mp_text_part, /* operation code for \.{textpart} */
mp_path_part, /* operation code for \.{pathpart} */
mp_pen_part, /* operation code for \.{penpart} */
mp_dash_part, /* operation code for \.{dashpart} */
mp_prescript_part, /* operation code for \.{prescriptpart} */
mp_postscript_part, /* operation code for \.{postscriptpart} */
mp_sqrt_op, /* operation code for \.{sqrt} */
mp_m_exp_op, /* operation code for \.{mexp} */
mp_m_log_op, /* operation code for \.{mlog} */
mp_sin_d_op, /* operation code for \.{sind} */
mp_cos_d_op, /* operation code for \.{cosd} */
mp_floor_op, /* operation code for \.{floor} */
mp_uniform_deviate, /* operation code for \.{uniformdeviate} */
mp_char_exists_op, /* operation code for \.{charexists} */
mp_font_size, /* operation code for \.{fontsize} */
mp_ll_corner_op, /* operation code for \.{llcorner} */
mp_lr_corner_op, /* operation code for \.{lrcorner} */
mp_ul_corner_op, /* operation code for \.{ulcorner} */
mp_ur_corner_op, /* operation code for \.{urcorner} */
mp_arc_length, /* operation code for \.{arclength} */
mp_angle_op, /* operation code for \.{angle} */
mp_cycle_op, /* operation code for \.{cycle} */
mp_filled_op, /* operation code for \.{filled} */
mp_stroked_op, /* operation code for \.{stroked} */
mp_textual_op, /* operation code for \.{textual} */
mp_clipped_op, /* operation code for \.{clipped} */
mp_bounded_op, /* operation code for \.{bounded} */
mp_plus, /* operation code for \.+ */
mp_minus, /* operation code for \.- */
mp_times, /* operation code for \.* */
mp_over, /* operation code for \./ */
mp_pythag_add, /* operation code for \.{++} */
mp_pythag_sub, /* operation code for \.{+-+} */
mp_or_op, /* operation code for \.{or} */
mp_and_op, /* operation code for \.{and} */
mp_less_than, /* operation code for \.< */
mp_less_or_equal, /* operation code for \.{<=} */
mp_greater_than, /* operation code for \.> */
mp_greater_or_equal, /* operation code for \.{>=} */
mp_equal_to, /* operation code for \.= */
mp_unequal_to, /* operation code for \.{<>} */
mp_concatenate, /* operation code for \.\& */
mp_rotated_by, /* operation code for \.{rotated} */
mp_slanted_by, /* operation code for \.{slanted} */
mp_scaled_by, /* operation code for \.{scaled} */
mp_shifted_by, /* operation code for \.{shifted} */
mp_transformed_by, /* operation code for \.{transformed} */
mp_x_scaled, /* operation code for \.{xscaled} */
mp_y_scaled, /* operation code for \.{yscaled} */
mp_z_scaled, /* operation code for \.{zscaled} */
mp_in_font, /* operation code for \.{infont} */
mp_intersect, /* operation code for \.{intersectiontimes} */
mp_double_dot, /* operation code for improper \.{..} */
mp_substring_of, /* operation code for \.{substring} */
mp_subpath_of, /* operation code for \.{subpath} */
mp_direction_time_of, /* operation code for \.{directiontime} */
mp_point_of, /* operation code for \.{point} */
mp_precontrol_of, /* operation code for \.{precontrol} */
mp_postcontrol_of, /* operation code for \.{postcontrol} */
mp_pen_offset_of, /* operation code for \.{penoffset} */
mp_arc_time_of, /* operation code for \.{arctime} */
mp_version, /* operation code for \.{mpversion} */
mp_envelope_of, /* operation code for \.{envelope} */
mp_glyph_infont, /* operation code for \.{glyph} */
mp_kern_flag /* operation code for \.{kern} */

@ @c
static void mp_print_op (MP mp, quarterword c) {
  if (c <= mp_numeric_type) {
    mp_print_type (mp, c);
  } else {
    switch (c) {
    case mp_true_code:
      mp_print (mp, "true");
      break;
    case mp_false_code:
      mp_print (mp, "false");
      break;
    case mp_null_picture_code:
      mp_print (mp, "nullpicture");
      break;
    case mp_null_pen_code:
      mp_print (mp, "nullpen");
      break;
    case mp_read_string_op:
      mp_print (mp, "readstring");
      break;
    case mp_pen_circle:
      mp_print (mp, "pencircle");
      break;
    case mp_normal_deviate:
      mp_print (mp, "normaldeviate");
      break;
    case mp_read_from_op:
      mp_print (mp, "readfrom");
      break;
    case mp_close_from_op:
      mp_print (mp, "closefrom");
      break;
    case mp_odd_op:
      mp_print (mp, "odd");
      break;
    case mp_known_op:
      mp_print (mp, "known");
      break;
    case mp_unknown_op:
      mp_print (mp, "unknown");
      break;
    case mp_not_op:
      mp_print (mp, "not");
      break;
    case mp_decimal:
      mp_print (mp, "decimal");
      break;
    case mp_reverse:
      mp_print (mp, "reverse");
      break;
    case mp_make_path_op:
      mp_print (mp, "makepath");
      break;
    case mp_make_pen_op:
      mp_print (mp, "makepen");
      break;
    case mp_oct_op:
      mp_print (mp, "oct");
      break;
    case mp_hex_op:
      mp_print (mp, "hex");
      break;
    case mp_ASCII_op:
      mp_print (mp, "ASCII");
      break;
    case mp_char_op:
      mp_print (mp, "char");
      break;
    case mp_length_op:
      mp_print (mp, "length");
      break;
    case mp_turning_op:
      mp_print (mp, "turningnumber");
      break;
    case mp_x_part:
      mp_print (mp, "xpart");
      break;
    case mp_y_part:
      mp_print (mp, "ypart");
      break;
    case mp_xx_part:
      mp_print (mp, "xxpart");
      break;
    case mp_xy_part:
      mp_print (mp, "xypart");
      break;
    case mp_yx_part:
      mp_print (mp, "yxpart");
      break;
    case mp_yy_part:
      mp_print (mp, "yypart");
      break;
    case mp_red_part:
      mp_print (mp, "redpart");
      break;
    case mp_green_part:
      mp_print (mp, "greenpart");
      break;
    case mp_blue_part:
      mp_print (mp, "bluepart");
      break;
    case mp_cyan_part:
      mp_print (mp, "cyanpart");
      break;
    case mp_magenta_part:
      mp_print (mp, "magentapart");
      break;
    case mp_yellow_part:
      mp_print (mp, "yellowpart");
      break;
    case mp_black_part:
      mp_print (mp, "blackpart");
      break;
    case mp_grey_part:
      mp_print (mp, "greypart");
      break;
    case mp_color_model_part:
      mp_print (mp, "colormodel");
      break;
    case mp_font_part:
      mp_print (mp, "fontpart");
      break;
    case mp_text_part:
      mp_print (mp, "textpart");
      break;
    case mp_prescript_part:
      mp_print (mp, "prescriptpart");
      break;
    case mp_postscript_part:
      mp_print (mp, "postscriptpart");
      break;
    case mp_path_part:
      mp_print (mp, "pathpart");
      break;
    case mp_pen_part:
      mp_print (mp, "penpart");
      break;
    case mp_dash_part:
      mp_print (mp, "dashpart");
      break;
    case mp_sqrt_op:
      mp_print (mp, "sqrt");
      break;
    case mp_m_exp_op:
      mp_print (mp, "mexp");
      break;
    case mp_m_log_op:
      mp_print (mp, "mlog");
      break;
    case mp_sin_d_op:
      mp_print (mp, "sind");
      break;
    case mp_cos_d_op:
      mp_print (mp, "cosd");
      break;
    case mp_floor_op:
      mp_print (mp, "floor");
      break;
    case mp_uniform_deviate:
      mp_print (mp, "uniformdeviate");
      break;
    case mp_char_exists_op:
      mp_print (mp, "charexists");
      break;
    case mp_font_size:
      mp_print (mp, "fontsize");
      break;
    case mp_ll_corner_op:
      mp_print (mp, "llcorner");
      break;
    case mp_lr_corner_op:
      mp_print (mp, "lrcorner");
      break;
    case mp_ul_corner_op:
      mp_print (mp, "ulcorner");
      break;
    case mp_ur_corner_op:
      mp_print (mp, "urcorner");
      break;
    case mp_arc_length:
      mp_print (mp, "arclength");
      break;
    case mp_angle_op:
      mp_print (mp, "angle");
      break;
    case mp_cycle_op:
      mp_print (mp, "cycle");
      break;
    case mp_filled_op:
      mp_print (mp, "filled");
      break;
    case mp_stroked_op:
      mp_print (mp, "stroked");
      break;
    case mp_textual_op:
      mp_print (mp, "textual");
      break;
    case mp_clipped_op:
      mp_print (mp, "clipped");
      break;
    case mp_bounded_op:
      mp_print (mp, "bounded");
      break;
    case mp_plus:
      mp_print_char (mp, xord ('+'));
      break;
    case mp_minus:
      mp_print_char (mp, xord ('-'));
      break;
    case mp_times:
      mp_print_char (mp, xord ('*'));
      break;
    case mp_over:
      mp_print_char (mp, xord ('/'));
      break;
    case mp_pythag_add:
      mp_print (mp, "++");
      break;
    case mp_pythag_sub:
      mp_print (mp, "+-+");
      break;
    case mp_or_op:
      mp_print (mp, "or");
      break;
    case mp_and_op:
      mp_print (mp, "and");
      break;
    case mp_less_than:
      mp_print_char (mp, xord ('<'));
      break;
    case mp_less_or_equal:
      mp_print (mp, "<=");
      break;
    case mp_greater_than:
      mp_print_char (mp, xord ('>'));
      break;
    case mp_greater_or_equal:
      mp_print (mp, ">=");
      break;
    case mp_equal_to:
      mp_print_char (mp, xord ('='));
      break;
    case mp_unequal_to:
      mp_print (mp, "<>");
      break;
    case mp_concatenate:
      mp_print (mp, "&");
      break;
    case mp_rotated_by:
      mp_print (mp, "rotated");
      break;
    case mp_slanted_by:
      mp_print (mp, "slanted");
      break;
    case mp_scaled_by:
      mp_print (mp, "scaled");
      break;
    case mp_shifted_by:
      mp_print (mp, "shifted");
      break;
    case mp_transformed_by:
      mp_print (mp, "transformed");
      break;
    case mp_x_scaled:
      mp_print (mp, "xscaled");
      break;
    case mp_y_scaled:
      mp_print (mp, "yscaled");
      break;
    case mp_z_scaled:
      mp_print (mp, "zscaled");
      break;
    case mp_in_font:
      mp_print (mp, "infont");
      break;
    case mp_intersect:
      mp_print (mp, "intersectiontimes");
      break;
    case mp_substring_of:
      mp_print (mp, "substring");
      break;
    case mp_subpath_of:
      mp_print (mp, "subpath");
      break;
    case mp_direction_time_of:
      mp_print (mp, "directiontime");
      break;
    case mp_point_of:
      mp_print (mp, "point");
      break;
    case mp_precontrol_of:
      mp_print (mp, "precontrol");
      break;
    case mp_postcontrol_of:
      mp_print (mp, "postcontrol");
      break;
    case mp_pen_offset_of:
      mp_print (mp, "penoffset");
      break;
    case mp_arc_time_of:
      mp_print (mp, "arctime");
      break;
    case mp_version:
      mp_print (mp, "mpversion");
      break;
    case mp_envelope_of:
      mp_print (mp, "envelope");
      break;
    case mp_glyph_infont:
      mp_print (mp, "glyph");
      break;
    default:
      mp_print (mp, "..");
      break;
    }
  }
}


@ \MP\ also has a bunch of internal parameters that a user might want to
fuss with. Every such parameter has an identifying code number, defined here.

@<Types...@>=
enum mp_given_internal {
  mp_output_template = 1,       /* a string set up by \&{outputtemplate} */
  mp_output_filename,           /* the output file name, accessible as \&{outputfilename} */
  mp_output_format,             /* the output format set up by \&{outputformat} */
  mp_output_format_options,     /* the output format options set up by \&{outputformatoptions} */
  mp_number_system,             /* the number system as set up by \&{numbersystem} */
  mp_number_precision,          /* the number system precision as set up by \&{numberprecision} */
  mp_job_name,                  /* the perceived jobname, as set up from the options stucture, 
                                   the name of the input file, or by \&{jobname}  */
  mp_tracing_titles,            /* show titles online when they appear */
  mp_tracing_equations,         /* show each variable when it becomes known */
  mp_tracing_capsules,          /* show capsules too */
  mp_tracing_choices,           /* show the control points chosen for paths */
  mp_tracing_specs,             /* show path subdivision prior to filling with polygonal a pen */
  mp_tracing_commands,          /* show commands and operations before they are performed */
  mp_tracing_restores,          /* show when a variable or internal is restored */
  mp_tracing_macros,            /* show macros before they are expanded */
  mp_tracing_output,            /* show digitized edges as they are output */
  mp_tracing_stats,             /* show memory usage at end of job */
  mp_tracing_lost_chars,        /* show characters that aren't \&{infont} */
  mp_tracing_online,            /* show long diagnostics on terminal and in the log file */
  mp_year,                      /* the current year (e.g., 1984) */
  mp_month,                     /* the current month (e.g., 3 $\equiv$ March) */
  mp_day,                       /* the current day of the month */
  mp_time,                      /* the number of minutes past midnight when this job started */
  mp_hour,                      /* the number of hours past midnight when this job started */
  mp_minute,                    /* the number of minutes in that hour when this job started */
  mp_char_code,                 /* the number of the next character to be output */
  mp_char_ext,                  /* the extension code of the next character to be output */
  mp_char_wd,                   /* the width of the next character to be output */
  mp_char_ht,                   /* the height of the next character to be output */
  mp_char_dp,                   /* the depth of the next character to be output */
  mp_char_ic,                   /* the italic correction of the next character to be output */
  mp_design_size,               /* the unit of measure used for |mp_char_wd..mp_char_ic|, in points */
  mp_pausing,                   /* positive to display lines on the terminal before they are read */
  mp_showstopping,              /* positive to stop after each \&{show} command */
  mp_fontmaking,                /* positive if font metric output is to be produced */
  mp_linejoin,                  /* as in \ps: 0 for mitered, 1 for round, 2 for beveled */
  mp_linecap,                   /* as in \ps: 0 for butt, 1 for round, 2 for square */
  mp_miterlimit,                /* controls miter length as in \ps */
  mp_warning_check,             /* controls error message when variable value is large */
  mp_boundary_char,             /* the right boundary character for ligatures */
  mp_prologues,                 /* positive to output conforming PostScript using built-in fonts */
  mp_true_corners,              /* positive to make \&{llcorner} etc. ignore \&{setbounds} */
  mp_default_color_model,       /* the default color model for unspecified items */
  mp_restore_clip_color,
  mp_procset,                   /* wether or not create PostScript command shortcuts */
  mp_hppp,                      /* horizontal pixels per point (for png output) */
  mp_vppp,                      /* vertical pixels per point (for png output) */
  mp_gtroffmode,                /* whether the user specified |-troff| on the command line */
};
typedef struct {
  mp_value v;
  char *intname;
} mp_internal;


@ @<MPlib internal header stuff@>=
#define internal_value(A) mp->internal[(A)].v.data.n
#define set_internal_from_number(A,B) do { \
  number_clone (internal_value ((A)),(B));\
} while (0)
#define internal_string(A) (mp_string)mp->internal[(A)].v.data.str
#define set_internal_string(A,B) mp->internal[(A)].v.data.str=(B)
#define internal_name(A) mp->internal[(A)].intname
#define set_internal_name(A,B) mp->internal[(A)].intname=(B)
#define internal_type(A) (mp_variable_type)mp->internal[(A)].v.type
#define set_internal_type(A,B) mp->internal[(A)].v.type=(B)
#define set_internal_from_cur_exp(A) do { \
  if (internal_type ((A)) == mp_string_type) { \
      add_str_ref (cur_exp_str ()); \
      set_internal_string ((A), cur_exp_str ()); \
  } else { \
      set_internal_from_number ((A), cur_exp_value_number ()); \
  } \
} while (0)



@

@d max_given_internal mp_gtroffmode

@<Glob...@>=
mp_internal *internal;  /* the values of internal quantities */
int int_ptr;    /* the maximum internal quantity defined so far */
int max_internal;       /* current maximum number of internal quantities */

@ @<Option variables@>=
int troff_mode;

@ @<Allocate or initialize ...@>=
mp->max_internal = 2 * max_given_internal;
mp->internal = xmalloc ((mp->max_internal + 1), sizeof (mp_internal));
memset (mp->internal, 0,
        (size_t) (mp->max_internal + 1) * sizeof (mp_internal));
{
  int i;
  for (i = 1; i <= mp->max_internal; i++) {
    new_number(mp->internal[i].v.data.n);
  }
  for (i = 1; i <= max_given_internal; i++) {
    set_internal_type (i, mp_known);
  }
}
set_internal_type (mp_output_format, mp_string_type);
set_internal_type (mp_output_filename, mp_string_type);
set_internal_type (mp_output_format_options, mp_string_type);
set_internal_type (mp_output_template, mp_string_type);
set_internal_type (mp_number_system, mp_string_type);
set_internal_type (mp_job_name, mp_string_type);
mp->troff_mode = (opt->troff_mode > 0 ? true : false);

@ @<Exported function ...@>=
int mp_troff_mode (MP mp);

@ @c
int mp_troff_mode (MP mp) {
  return mp->troff_mode;
}


@ @<Set initial ...@>=
mp->int_ptr = max_given_internal;

@ The symbolic names for internal quantities are put into \MP's hash table
by using a routine called |primitive|, which will be defined later. Let us
enter them now, so that we don't have to list all those names again
anywhere else.

@<Put each of \MP's primitives into the hash table@>=
mp_primitive (mp, "tracingtitles", mp_internal_quantity, mp_tracing_titles);
@:tracingtitles_}{\&{tracingtitles} primitive@>;
mp_primitive (mp, "tracingequations", mp_internal_quantity, mp_tracing_equations);
@:mp_tracing_equations_}{\&{tracingequations} primitive@>;
mp_primitive (mp, "tracingcapsules", mp_internal_quantity, mp_tracing_capsules);
@:mp_tracing_capsules_}{\&{tracingcapsules} primitive@>;
mp_primitive (mp, "tracingchoices", mp_internal_quantity, mp_tracing_choices);
@:mp_tracing_choices_}{\&{tracingchoices} primitive@>;
mp_primitive (mp, "tracingspecs", mp_internal_quantity, mp_tracing_specs);
@:mp_tracing_specs_}{\&{tracingspecs} primitive@>;
mp_primitive (mp, "tracingcommands", mp_internal_quantity, mp_tracing_commands);
@:mp_tracing_commands_}{\&{tracingcommands} primitive@>;
mp_primitive (mp, "tracingrestores", mp_internal_quantity, mp_tracing_restores);
@:mp_tracing_restores_}{\&{tracingrestores} primitive@>;
mp_primitive (mp, "tracingmacros", mp_internal_quantity, mp_tracing_macros);
@:mp_tracing_macros_}{\&{tracingmacros} primitive@>;
mp_primitive (mp, "tracingoutput", mp_internal_quantity, mp_tracing_output);
@:mp_tracing_output_}{\&{tracingoutput} primitive@>;
mp_primitive (mp, "tracingstats", mp_internal_quantity, mp_tracing_stats);
@:mp_tracing_stats_}{\&{tracingstats} primitive@>;
mp_primitive (mp, "tracinglostchars", mp_internal_quantity, mp_tracing_lost_chars);
@:mp_tracing_lost_chars_}{\&{tracinglostchars} primitive@>;
mp_primitive (mp, "tracingonline", mp_internal_quantity, mp_tracing_online);
@:mp_tracing_online_}{\&{tracingonline} primitive@>;
mp_primitive (mp, "year", mp_internal_quantity, mp_year);
@:mp_year_}{\&{year} primitive@>;
mp_primitive (mp, "month", mp_internal_quantity, mp_month);
@:mp_month_}{\&{month} primitive@>;
mp_primitive (mp, "day", mp_internal_quantity, mp_day);
@:mp_day_}{\&{day} primitive@>;
mp_primitive (mp, "time", mp_internal_quantity, mp_time);
@:time_}{\&{time} primitive@>;
mp_primitive (mp, "hour", mp_internal_quantity, mp_hour);
@:hour_}{\&{hour} primitive@>;
mp_primitive (mp, "minute", mp_internal_quantity, mp_minute);
@:minute_}{\&{minute} primitive@>;
mp_primitive (mp, "charcode", mp_internal_quantity, mp_char_code);
@:mp_char_code_}{\&{charcode} primitive@>;
mp_primitive (mp, "charext", mp_internal_quantity, mp_char_ext);
@:mp_char_ext_}{\&{charext} primitive@>;
mp_primitive (mp, "charwd", mp_internal_quantity, mp_char_wd);
@:mp_char_wd_}{\&{charwd} primitive@>;
mp_primitive (mp, "charht", mp_internal_quantity, mp_char_ht);
@:mp_char_ht_}{\&{charht} primitive@>;
mp_primitive (mp, "chardp", mp_internal_quantity, mp_char_dp);
@:mp_char_dp_}{\&{chardp} primitive@>;
mp_primitive (mp, "charic", mp_internal_quantity, mp_char_ic);
@:mp_char_ic_}{\&{charic} primitive@>;
mp_primitive (mp, "designsize", mp_internal_quantity, mp_design_size);
@:mp_design_size_}{\&{designsize} primitive@>;
mp_primitive (mp, "pausing", mp_internal_quantity, mp_pausing);
@:mp_pausing_}{\&{pausing} primitive@>;
mp_primitive (mp, "showstopping", mp_internal_quantity, mp_showstopping);
@:mp_showstopping_}{\&{showstopping} primitive@>;
mp_primitive (mp, "fontmaking", mp_internal_quantity, mp_fontmaking);
@:mp_fontmaking_}{\&{fontmaking} primitive@>;
mp_primitive (mp, "linejoin", mp_internal_quantity, mp_linejoin);
@:mp_linejoin_}{\&{linejoin} primitive@>;
mp_primitive (mp, "linecap", mp_internal_quantity, mp_linecap);
@:mp_linecap_}{\&{linecap} primitive@>;
mp_primitive (mp, "miterlimit", mp_internal_quantity, mp_miterlimit);
@:mp_miterlimit_}{\&{miterlimit} primitive@>;
mp_primitive (mp, "warningcheck", mp_internal_quantity, mp_warning_check);
@:mp_warning_check_}{\&{warningcheck} primitive@>;
mp_primitive (mp, "boundarychar", mp_internal_quantity, mp_boundary_char);
@:mp_boundary_char_}{\&{boundarychar} primitive@>;
mp_primitive (mp, "prologues", mp_internal_quantity, mp_prologues);
@:mp_prologues_}{\&{prologues} primitive@>;
mp_primitive (mp, "truecorners", mp_internal_quantity, mp_true_corners);
@:mp_true_corners_}{\&{truecorners} primitive@>;
mp_primitive (mp, "mpprocset", mp_internal_quantity, mp_procset);
@:mp_procset_}{\&{mpprocset} primitive@>;
mp_primitive (mp, "troffmode", mp_internal_quantity, mp_gtroffmode);
@:troffmode_}{\&{troffmode} primitive@>;
mp_primitive (mp, "defaultcolormodel", mp_internal_quantity,
              mp_default_color_model);
@:mp_default_color_model_}{\&{defaultcolormodel} primitive@>;
mp_primitive (mp, "restoreclipcolor", mp_internal_quantity, mp_restore_clip_color);
@:mp_restore_clip_color_}{\&{restoreclipcolor} primitive@>;
mp_primitive (mp, "outputtemplate", mp_internal_quantity, mp_output_template);
@:mp_output_template_}{\&{outputtemplate} primitive@>;
mp_primitive (mp, "outputfilename", mp_internal_quantity, mp_output_filename);
@:mp_output_filename_}{\&{outputfilename} primitive@>;
mp_primitive (mp, "numbersystem", mp_internal_quantity, mp_number_system);
@:mp_number_system_}{\&{numbersystem} primitive@>;
mp_primitive (mp, "numberprecision", mp_internal_quantity, mp_number_precision);
@:mp_number_precision_}{\&{numberprecision} primitive@>;
mp_primitive (mp, "outputformat", mp_internal_quantity, mp_output_format);
@:mp_output_format_}{\&{outputformat} primitive@>;
mp_primitive (mp, "outputformatoptions", mp_internal_quantity, mp_output_format_options);
@:mp_output_format_options_}{\&{outputformatoptions} primitive@>;
mp_primitive (mp, "jobname", mp_internal_quantity, mp_job_name);
@:mp_job_name_}{\&{jobname} primitive@>
mp_primitive (mp, "hppp", mp_internal_quantity, mp_hppp);
@:mp_hppp_}{\&{hppp} primitive@>;
mp_primitive (mp, "vppp", mp_internal_quantity, mp_vppp);
@:mp_vppp_}{\&{vppp} primitive@>;
 

@ Colors can be specified in four color models. In the special
case of |no_model|, MetaPost does not output any color operator to
the postscript output.

Note: these values are passed directly on to |with_option|. This only
works because the other possible values passed to |with_option| are
8 and 10 respectively (from |with_pen| and |with_picture|).

There is a first state, that is only used for |gs_colormodel|. It flags
the fact that there has not been any kind of color specification by
the user so far in the game.

@<MPlib header stuff@>=
enum mp_color_model {
  mp_no_model = 1,
  mp_grey_model = 3,
  mp_rgb_model = 5,
  mp_cmyk_model = 7,
  mp_uninitialized_model = 9
};


@ @<Initialize table entries@>=
set_internal_from_number (mp_default_color_model, unity_t);
number_multiply_int (internal_value (mp_default_color_model), mp_rgb_model);
number_clone (internal_value (mp_restore_clip_color), unity_t);
number_clone (internal_value (mp_hppp), unity_t);
number_clone (internal_value (mp_vppp), unity_t);
set_internal_string (mp_output_template, mp_intern (mp, "%j.%c"));
set_internal_string (mp_output_filename, mp_intern (mp, ""));
set_internal_string (mp_output_format, mp_intern (mp, "eps"));
set_internal_string (mp_output_format_options, mp_intern (mp, ""));
set_internal_string (mp_number_system, mp_intern (mp, "scaled"));
set_internal_from_number (mp_number_precision, precision_default);
#if DEBUG
number_clone (internal_value (mp_tracing_titles), three_t);
number_clone (internal_value (mp_tracing_equations), three_t);
number_clone (internal_value (mp_tracing_capsules), three_t);
number_clone (internal_value (mp_tracing_choices), three_t);
number_clone (internal_value (mp_tracing_specs), three_t);
number_clone (internal_value (mp_tracing_commands), three_t);
number_clone (internal_value (mp_tracing_restores), three_t);
number_clone (internal_value (mp_tracing_macros), three_t);
number_clone (internal_value (mp_tracing_output), three_t);
number_clone (internal_value (mp_tracing_stats), three_t);
number_clone (internal_value (mp_tracing_lost_chars), three_t);
number_clone (internal_value (mp_tracing_online), three_t);
#endif

@ Well, we do have to list the names one more time, for use in symbolic
printouts.

@<Initialize table...@>=
set_internal_name (mp_tracing_titles, xstrdup ("tracingtitles"));
set_internal_name (mp_tracing_equations, xstrdup ("tracingequations"));
set_internal_name (mp_tracing_capsules, xstrdup ("tracingcapsules"));
set_internal_name (mp_tracing_choices, xstrdup ("tracingchoices"));
set_internal_name (mp_tracing_specs, xstrdup ("tracingspecs"));
set_internal_name (mp_tracing_commands, xstrdup ("tracingcommands"));
set_internal_name (mp_tracing_restores, xstrdup ("tracingrestores"));
set_internal_name (mp_tracing_macros, xstrdup ("tracingmacros"));
set_internal_name (mp_tracing_output, xstrdup ("tracingoutput"));
set_internal_name (mp_tracing_stats, xstrdup ("tracingstats"));
set_internal_name (mp_tracing_lost_chars, xstrdup ("tracinglostchars"));
set_internal_name (mp_tracing_online, xstrdup ("tracingonline"));
set_internal_name (mp_year, xstrdup ("year"));
set_internal_name (mp_month, xstrdup ("month"));
set_internal_name (mp_day, xstrdup ("day"));
set_internal_name (mp_time, xstrdup ("time"));
set_internal_name (mp_hour, xstrdup ("hour"));
set_internal_name (mp_minute, xstrdup ("minute"));
set_internal_name (mp_char_code, xstrdup ("charcode"));
set_internal_name (mp_char_ext, xstrdup ("charext"));
set_internal_name (mp_char_wd, xstrdup ("charwd"));
set_internal_name (mp_char_ht, xstrdup ("charht"));
set_internal_name (mp_char_dp, xstrdup ("chardp"));
set_internal_name (mp_char_ic, xstrdup ("charic"));
set_internal_name (mp_design_size, xstrdup ("designsize"));
set_internal_name (mp_pausing, xstrdup ("pausing"));
set_internal_name (mp_showstopping, xstrdup ("showstopping"));
set_internal_name (mp_fontmaking, xstrdup ("fontmaking"));
set_internal_name (mp_linejoin, xstrdup ("linejoin"));
set_internal_name (mp_linecap, xstrdup ("linecap"));
set_internal_name (mp_miterlimit, xstrdup ("miterlimit"));
set_internal_name (mp_warning_check, xstrdup ("warningcheck"));
set_internal_name (mp_boundary_char, xstrdup ("boundarychar"));
set_internal_name (mp_prologues, xstrdup ("prologues"));
set_internal_name (mp_true_corners, xstrdup ("truecorners"));
set_internal_name (mp_default_color_model, xstrdup ("defaultcolormodel"));
set_internal_name (mp_procset, xstrdup ("mpprocset"));
set_internal_name (mp_gtroffmode, xstrdup ("troffmode"));
set_internal_name (mp_restore_clip_color, xstrdup ("restoreclipcolor"));
set_internal_name (mp_output_template, xstrdup ("outputtemplate"));
set_internal_name (mp_output_filename, xstrdup ("outputfilename"));
set_internal_name (mp_output_format, xstrdup ("outputformat"));
set_internal_name (mp_output_format_options, xstrdup ("outputformatoptions"));
set_internal_name (mp_job_name, xstrdup ("jobname"));
set_internal_name (mp_number_system, xstrdup ("numbersystem"));
set_internal_name (mp_number_precision, xstrdup ("numberprecision"));
set_internal_name (mp_hppp, xstrdup ("hppp"));
set_internal_name (mp_vppp, xstrdup ("vppp"));

@ The following procedure, which is called just before \MP\ initializes its
input and output, establishes the initial values of the date and time.
@^system dependencies@>

Note that the values are |scaled| integers. Hence \MP\ can no longer
be used after the year 32767.

@c
static void mp_fix_date_and_time (MP mp) {
  time_t aclock = time ((time_t *) 0);
  struct tm *tmptr = localtime (&aclock);
  set_internal_from_number (mp_time, unity_t);
  number_multiply_int (internal_value(mp_time), (tmptr->tm_hour * 60 + tmptr->tm_min));
  set_internal_from_number (mp_hour, unity_t);
  number_multiply_int (internal_value(mp_hour), (tmptr->tm_hour));
  set_internal_from_number (mp_minute, unity_t);
  number_multiply_int (internal_value(mp_minute), (tmptr->tm_min));
  set_internal_from_number (mp_day, unity_t);
  number_multiply_int (internal_value(mp_day), (tmptr->tm_mday));
  set_internal_from_number (mp_month, unity_t);
  number_multiply_int (internal_value(mp_month), (tmptr->tm_mon + 1));
  set_internal_from_number (mp_year, unity_t);
  number_multiply_int (internal_value(mp_year), (tmptr->tm_year + 1900));
}


@ @<Declarations@>=
static void mp_fix_date_and_time (MP mp);

@ \MP\ is occasionally supposed to print diagnostic information that
goes only into the transcript file, unless |mp_tracing_online| is positive.
Now that we have defined |mp_tracing_online| we can define
two routines that adjust the destination of print commands:

@<Declarations@>=
static void mp_begin_diagnostic (MP mp);
static void mp_end_diagnostic (MP mp, boolean blank_line);
static void mp_print_diagnostic (MP mp, const char *s, const char *t,
                                 boolean nuline);

@ @<Basic printing...@>=
void mp_begin_diagnostic (MP mp) {                               /* prepare to do some tracing */
  mp->old_setting = mp->selector;
  if (number_nonpositive(internal_value (mp_tracing_online))
      && (mp->selector == term_and_log)) {
    decr (mp->selector);
    if (mp->history == mp_spotless)
      mp->history = mp_warning_issued;
  }
}
@#
void mp_end_diagnostic (MP mp, boolean blank_line) {
  /* restore proper conditions after tracing */
  mp_print_nl (mp, "");
  if (blank_line)
    mp_print_ln (mp);
  mp->selector = mp->old_setting;
}


@ 

@<Glob...@>=
unsigned int old_setting;

@ We will occasionally use |begin_diagnostic| in connection with line-number
printing, as follows. (The parameter |s| is typically |"Path"| or
|"Cycle spec"|, etc.)

@<Basic printing...@>=
void mp_print_diagnostic (MP mp, const char *s, const char *t, boolean nuline) {
  mp_begin_diagnostic (mp);
  if (nuline)
    mp_print_nl (mp, s);
  else
    mp_print (mp, s);
  mp_print (mp, " at line ");
  mp_print_int (mp, mp_true_line (mp));
  mp_print (mp, t);
  mp_print_char (mp, xord (':'));
}


@ The 256 |ASCII_code| characters are grouped into classes by means of
the |char_class| table. Individual class numbers have no semantic
or syntactic significance, except in a few instances defined here.
There's also |max_class|, which can be used as a basis for additional
class numbers in nonstandard extensions of \MP.

@d digit_class 0 /* the class number of \.{0123456789} */
@d period_class 1 /* the class number of `\..' */
@d space_class 2 /* the class number of spaces and nonstandard characters */
@d percent_class 3 /* the class number of `\.\%' */
@d string_class 4 /* the class number of `\."' */
@d right_paren_class 8 /* the class number of `\.)' */
@d isolated_classes 5: case 6: case 7: case 8 /* characters that make length-one tokens only */
@d letter_class 9 /* letters and the underline character */
@d mp_left_bracket_class 17 /* `\.[' */
@d mp_right_bracket_class 18 /* `\.]' */
@d invalid_class 20 /* bad character in the input */
@d max_class 20 /* the largest class number */

@<Glob...@>=
#define digit_class 0 /* the class number of \.{0123456789} */
int char_class[256];    /* the class numbers */

@ If changes are made to accommodate non-ASCII character sets, they should
follow the guidelines in Appendix~C of {\sl The {\logos METAFONT\/}book}.
@:METAFONTbook}{\sl The {\logos METAFONT\/}book@>
@^system dependencies@>

@<Set initial ...@>=
for (k = '0'; k <= '9'; k++)
  mp->char_class[k] = digit_class;
mp->char_class['.'] = period_class;
mp->char_class[' '] = space_class;
mp->char_class['%'] = percent_class;
mp->char_class['"'] = string_class;
mp->char_class[','] = 5;
mp->char_class[';'] = 6;
mp->char_class['('] = 7;
mp->char_class[')'] = right_paren_class;
for (k = 'A'; k <= 'Z'; k++)
  mp->char_class[k] = letter_class;
for (k = 'a'; k <= 'z'; k++)
  mp->char_class[k] = letter_class;
mp->char_class['_'] = letter_class;
mp->char_class['<'] = 10;
mp->char_class['='] = 10;
mp->char_class['>'] = 10;
mp->char_class[':'] = 10;
mp->char_class['|'] = 10;
mp->char_class['`'] = 11;
mp->char_class['\''] = 11;
mp->char_class['+'] = 12;
mp->char_class['-'] = 12;
mp->char_class['/'] = 13;
mp->char_class['*'] = 13;
mp->char_class['\\'] = 13;
mp->char_class['!'] = 14;
mp->char_class['?'] = 14;
mp->char_class['#'] = 15;
mp->char_class['&'] = 15;
mp->char_class['@@'] = 15;
mp->char_class['$'] = 15;
mp->char_class['^'] = 16;
mp->char_class['~'] = 16;
mp->char_class['['] = mp_left_bracket_class;
mp->char_class[']'] = mp_right_bracket_class;
mp->char_class['{'] = 19;
mp->char_class['}'] = 19;
for (k = 0; k < ' '; k++)
  mp->char_class[k] = invalid_class;
mp->char_class['\t'] = space_class;
mp->char_class['\f'] = space_class;
for (k = 127; k <= 255; k++)
  mp->char_class[k] = invalid_class;

@* The hash table.

Symbolic tokens are stored in and retrieved from an AVL tree. This
is not as fast as an actual hash table, but it is easily extensible.

A symbolic token contains a pointer to the |mp_string| that 
contains the string representation of the symbol, a |halfword| 
that holds the current command value of the token, and an 
|mp_value| for the associated equivalent. 

@d set_text(A)     do {
   FUNCTION_TRACE3 ("set_text(%p, %p)\n",(A),(B));
   (A)->text=(B) ;
} while (0)

@d set_eq_type(A,B)  do {
   FUNCTION_TRACE3 ("set_eq_type(%p, %d)\n",(A),(B));
   (A)->type=(B) ;
} while (0)

@d set_equiv(A,B)  do {
   FUNCTION_TRACE3 ("set_equiv(%p, %d)\n",(A),(B));
   (A)->v.data.node=NULL ;
   (A)->v.data.indep.serial=(B);
} while (0)

@d set_equiv_node(A,B)  do {
   FUNCTION_TRACE3 ("set_equiv_node(%p, %p)\n",(A),(B));
   (A)->v.data.node=(B) ;
   (A)->v.data.indep.serial=0;
} while (0)

@d set_equiv_sym(A,B)  do {
   FUNCTION_TRACE3 ("set_equiv_sym(%p, %p)\n",(A),(B));
   (A)->v.data.node=(mp_node)(B);
   (A)->v.data.indep.serial=0;
} while (0)

@ @c
#if DEBUG
#define text(A)         do_get_text(mp, (A))
#define eq_type(A)      do_get_eq_type(mp, (A))
#define equiv(A)        do_get_equiv(mp, (A))
#define equiv_node(A)   do_get_equiv_node(mp, (A))
#define equiv_sym(A)    do_get_equiv_sym(mp, (A))
static mp_string do_get_text (MP mp, mp_sym A) {
  FUNCTION_TRACE3 ("%d = do_get_text(%p)\n",A->text,A);
  return A->text;
}
static halfword do_get_eq_type (MP mp, mp_sym A) {
  FUNCTION_TRACE3 ("%d = do_get_eq_type(%p)\n",A->type,A);
  return A->type;
}
static halfword do_get_equiv (MP mp, mp_sym A) {
  FUNCTION_TRACE3 ("%d = do_get_equiv(%p)\n",A->v.data.indep.serial,A);
  return A->v.data.indep.serial;
}
static mp_node do_get_equiv_node (MP mp, mp_sym A) {
  FUNCTION_TRACE3 ("%p = do_get_equiv_node(%p)\n",A->v.data.node,A);
  return A->v.data.node;
}
static mp_sym do_get_equiv_sym (MP mp, mp_sym A) {
  FUNCTION_TRACE3 ("%p = do_get_equiv_sym(%p)\n",A->v.data.node,A);
  return (mp_sym)A->v.data.node;
}
#else
#define text(A)         (A)->text
#define eq_type(A)      (A)->type
#define equiv(A)        (A)->v.data.indep.serial
#define equiv_node(A)   (A)->v.data.node
#define equiv_sym(A)    (mp_sym)(A)->v.data.node
#endif

@ @<Declarations...@>=
#if DEBUG
static mp_string do_get_text (MP mp, mp_sym A);
static halfword do_get_eq_type (MP mp, mp_sym A);
static halfword do_get_equiv (MP mp, mp_sym A);
static mp_node do_get_equiv_node (MP mp, mp_sym A);
static mp_sym do_get_equiv_sym (MP mp, mp_sym A);
#endif

@ @<Types...@>=
typedef struct mp_symbol_entry {
  halfword type;
  mp_value v;
  mp_string text;
  void *parent;
} mp_symbol_entry;

@ @<Glob...@>=
integer st_count;       /* total number of known identifiers */
avl_tree symbols;       /* avl tree of symbolic tokens */
avl_tree frozen_symbols;        /* avl tree of frozen symbolic tokens */
mp_sym frozen_bad_vardef;
mp_sym frozen_colon;
mp_sym frozen_end_def;
mp_sym frozen_end_for;
mp_sym frozen_end_group;
mp_sym frozen_etex;
mp_sym frozen_fi;
mp_sym frozen_inaccessible;
mp_sym frozen_left_bracket;
mp_sym frozen_mpx_break;
mp_sym frozen_repeat_loop;
mp_sym frozen_right_delimiter;
mp_sym frozen_semicolon;
mp_sym frozen_slash;
mp_sym frozen_undefined;
mp_sym frozen_dump;


@ Here are the functions needed for the avl construction.

@<Declarations@>=
static int comp_symbols_entry (void *p, const void *pa, const void *pb);
static void *copy_symbols_entry (const void *p);
static void *delete_symbols_entry (void *p);


@ The avl comparison function is a straightword version of |strcmp|,
except that checks for the string lengths first.

@c
static int comp_symbols_entry (void *p, const void *pa, const void *pb) {
  const mp_symbol_entry *a = (const mp_symbol_entry *) pa;
  const mp_symbol_entry *b = (const mp_symbol_entry *) pb;
  (void) p;
  if (a->text->len != b->text->len) {
    return (a->text->len > b->text->len ? 1 : -1);
  }
  return strncmp ((const char *) a->text->str, (const char *) b->text->str,
                  a->text->len);
}


@ Copying a symbol happens when an item is inserted into an AVL tree.
The |text| and |mp_number| needs to be deep copied, every thing else 
can be reassigned.
 
@c
static void *copy_symbols_entry (const void *p) {
  MP mp;
  mp_sym ff;
  const mp_symbol_entry *fp;
  fp = (const mp_symbol_entry *) p;
  mp = (MP)fp->parent;
  ff = malloc (sizeof (mp_symbol_entry));
  if (ff == NULL)
    return NULL;
  ff->text = copy_strings_entry (fp->text); 
  if (ff->text == NULL)
    return NULL;
  ff->v = fp->v;
  ff->type = fp->type;
  ff->parent = mp;
  new_number(ff->v.data.n);
  number_clone(ff->v.data.n, fp->v.data.n);
  return ff;
}


@ In the current implementation, symbols are not freed until the
end of the run.

@c
static void *delete_symbols_entry (void *p) {
  MP mp;
  mp_sym ff = (mp_sym) p;
  mp = (MP)ff->parent;
  free_number(ff->v.data.n);
  mp_xfree (ff->text->str);
  mp_xfree (ff->text);
  mp_xfree (ff);
  return NULL;
}


@ @<Allocate or initialize ...@>=
mp->symbols = avl_create (comp_symbols_entry,
                          copy_symbols_entry,
                          delete_symbols_entry, malloc, free, NULL);
mp->frozen_symbols = avl_create (comp_symbols_entry,
                                 copy_symbols_entry,
                                 delete_symbols_entry, malloc, free, NULL);

@ @<Dealloc variables@>=
if (mp->symbols != NULL)
  avl_destroy (mp->symbols);
if (mp->frozen_symbols != NULL)
  avl_destroy (mp->frozen_symbols);

@ Actually creating symbols is done by |id_lookup|, but in order to
do so it needs a way to create a new, empty symbol structure.

@<Declarations@>=
static mp_sym new_symbols_entry (MP mp, unsigned char *nam, size_t len);

@ @c
static mp_sym new_symbols_entry (MP mp, unsigned char *nam, size_t len) {
  mp_sym ff;
  ff = mp_xmalloc (mp, 1, sizeof (mp_symbol_entry));
  memset (ff, 0, sizeof (mp_symbol_entry));
  ff->parent = mp;
  ff->text = mp_xmalloc (mp, 1, sizeof (mp_lstring));
  ff->text->str = nam;
  ff->text->len = len;
  ff->type = mp_tag_token;
  ff->v.type = mp_known;
  new_number(ff->v.data.n);
  FUNCTION_TRACE4 ("%p = new_symbols_entry(\"%s\",%d)\n", ff, nam, (int)len);
  return ff;
}


@ There is one global variable so that |id_lookup| does not always have to
create a new entry just for testing. This is not freed because it creates
a double-free thanks to the |NULL| init.

@<Global ...@>=
mp_sym id_lookup_test;

@ @<Initialize table entries@>=
mp->id_lookup_test = new_symbols_entry (mp, NULL, 0);

@ Certain symbols are ``frozen'' and not redefinable, since they are
used
in error recovery.

@<Initialize table entries@>=
mp->st_count = 0;
mp->frozen_bad_vardef = mp_frozen_primitive (mp, "a bad variable", mp_tag_token, 0);
mp->frozen_right_delimiter = mp_frozen_primitive (mp, ")", mp_right_delimiter, 0);
mp->frozen_inaccessible = mp_frozen_primitive (mp, " INACCESSIBLE", mp_tag_token, 0);
mp->frozen_undefined = mp_frozen_primitive (mp, " UNDEFINED", mp_tag_token, 0);

@ Here is the subroutine that searches the avl tree for an identifier
that matches a given string of length~|l| appearing in |buffer[j..
(j+l-1)]|. If the identifier is not found, it is inserted if
|insert_new| is |true|, and the corresponding symbol will be returned.

There are two variations on the lookup function: one for the normal
symbol table, and one for the table of error recovery symbols.

@d mp_id_lookup(A,B,C,D) mp_do_id_lookup ((A), mp->symbols, (B), (C), (D))

@c
static mp_sym mp_do_id_lookup (MP mp, avl_tree symbols, char *j,
                               size_t l, boolean insert_new) {
  /* search an avl tree */
  mp_sym str;
  mp->id_lookup_test->text->str = (unsigned char *)j;
  mp->id_lookup_test->text->len = l;
  str = (mp_sym) avl_find (mp->id_lookup_test, symbols);
  if (str == NULL && insert_new) {
    unsigned char *nam = (unsigned char *) mp_xstrldup (mp, j, l);
    mp_sym s = new_symbols_entry (mp, nam, l);
    mp->st_count++;
    assert (avl_ins (s, symbols, avl_false) > 0);
    str = (mp_sym) avl_find (s, symbols);
    delete_symbols_entry (s);
  }
  return str;
}
static mp_sym mp_frozen_id_lookup (MP mp, char *j, size_t l,
                                   boolean insert_new) {
  /* search the error recovery symbol table */
  return mp_do_id_lookup (mp, mp->frozen_symbols, j, l, insert_new);
}

/* see mp_print_sym  (mp_sym sym) */

double mp_get_numeric_value (MP mp, const char *s, size_t l) {
    char *ss = mp_xstrdup(mp,s);
    if (ss) {
     mp_sym sym = mp_id_lookup(mp,ss,l,false);
     if (sym != NULL) {
        if (mp_type(sym->v.data.node) == mp_known) {
	    mp_xfree (ss);
            return number_to_double(sym->v.data.node->data.n) ;
        }
     }
    }
    mp_xfree (ss);
    return 0 ;
}

int mp_get_boolean_value (MP mp, const char *s, size_t l) {
   char *ss = mp_xstrdup(mp,s);
   if (ss) {
    mp_sym sym = mp_id_lookup(mp,ss,l,false);
    if (sym != NULL) {
        if (mp_type(sym->v.data.node) == mp_boolean_type) {
            if (number_to_boolean (sym->v.data.node->data.n) == mp_true_code) {
 	        mp_xfree(ss);
                return 1 ;
            }
        }
     }
   }
   mp_xfree (ss);
   return 0;
}

char *mp_get_string_value (MP mp, const char *s, size_t l) {
   char *ss = mp_xstrdup(mp,s);
   if (ss) {
    mp_sym sym = mp_id_lookup(mp,ss,l,false);
    if (sym != NULL) {
        if (mp_type(sym->v.data.node) == mp_string_type) {
	    mp_xfree (ss);
            return (char *) sym->v.data.node->data.str->str;
        }
    }
   }
   mp_xfree (ss);
   return NULL;
}

@ @<Exported function headers@>=
double mp_get_numeric_value(MP mp,const char *s,size_t l);
int mp_get_boolean_value(MP mp,const char *s,size_t l);
char *mp_get_string_value(MP mp,const char *s,size_t l);

@ We need to put \MP's ``primitive'' symbolic tokens into the hash
table, together with their command code (which will be the |eq_type|)
and an operand (which will be the |equiv|). The |primitive| procedure
does this, in a way that no \MP\ user can. The global value |cur_sym|
contains the new |eqtb| pointer after |primitive| has acted.

@c
static void mp_primitive (MP mp, const char *ss, halfword c, halfword o) {
  char *s = mp_xstrdup (mp, ss);
  set_cur_sym (mp_id_lookup (mp, s, strlen (s), true));
  mp_xfree (s);
  set_eq_type (cur_sym(), c);
  set_equiv (cur_sym(), o);
}


@ Some other symbolic tokens only exist for error recovery.

@c
static mp_sym mp_frozen_primitive (MP mp, const char *ss, halfword c,
                                   halfword o) {
  char *s = mp_xstrdup (mp, ss);
  mp_sym str = mp_frozen_id_lookup (mp, s, strlen (ss), true);
  mp_xfree (s);
  str->type = c;
  str->v.data.indep.serial = o;
  return str;
}


@ This routine returns |true| if the argument is an un-redefinable symbol
because it is one of the error recovery tokens (as explained elsewhere,
|frozen_inaccessible| actuall is redefinable). 

@c
static boolean mp_is_frozen (MP mp, mp_sym sym) {
  mp_sym temp = mp_frozen_id_lookup (mp, (char *) sym->text->str, sym->text->len, false);
  if (temp==mp->frozen_inaccessible)
    return false;
  return (temp == sym);
}


@ Many of \MP's primitives need no |equiv|, since they are identifiable
by their |eq_type| alone. These primitives are loaded into the hash table
as follows:

@<Put each of \MP's primitives into the hash table@>=
mp_primitive (mp, "..", mp_path_join, 0);
@:.._}{\.{..} primitive@>;
mp_primitive (mp, "[", mp_left_bracket, 0);
mp->frozen_left_bracket = mp_frozen_primitive (mp, "[", mp_left_bracket, 0);
@:[ }{\.{[} primitive@>;
mp_primitive (mp, "]", mp_right_bracket, 0);
@:] }{\.{]} primitive@>;
mp_primitive (mp, "}", mp_right_brace, 0);
@:]]}{\.{\char`\}} primitive@>;
mp_primitive (mp, "{", mp_left_brace, 0);
@:][}{\.{\char`\{} primitive@>;
mp_primitive (mp, ":", mp_colon, 0);
mp->frozen_colon = mp_frozen_primitive (mp, ":", mp_colon, 0);
@:: }{\.{:} primitive@>;
mp_primitive (mp, "::", mp_double_colon, 0);
@::: }{\.{::} primitive@>;
mp_primitive (mp, "||:", mp_bchar_label, 0);
@:::: }{\.{\char'174\char'174:} primitive@>;
mp_primitive (mp, ":=", mp_assignment, 0);
@::=_}{\.{:=} primitive@>;
mp_primitive (mp, ",", mp_comma, 0);
@:, }{\., primitive@>;
mp_primitive (mp, ";", mp_semicolon, 0);
mp->frozen_semicolon = mp_frozen_primitive (mp, ";", mp_semicolon, 0);
@:; }{\.; primitive@>;
mp_primitive (mp, "\\", mp_relax, 0);
@:]]\\}{\.{\char`\\} primitive@>;
mp_primitive (mp, "addto", mp_add_to_command, 0);
@:add_to_}{\&{addto} primitive@>;
mp_primitive (mp, "atleast", mp_at_least, 0);
@:at_least_}{\&{atleast} primitive@>;
mp_primitive (mp, "begingroup", mp_begin_group, 0);
mp->bg_loc = cur_sym();
@:begin_group_}{\&{begingroup} primitive@>;
mp_primitive (mp, "controls", mp_controls, 0);
@:controls_}{\&{controls} primitive@>;
mp_primitive (mp, "curl", mp_curl_command, 0);
@:curl_}{\&{curl} primitive@>;
mp_primitive (mp, "delimiters", mp_delimiters, 0);
@:delimiters_}{\&{delimiters} primitive@>;
mp_primitive (mp, "endgroup", mp_end_group, 0);
mp->eg_loc = cur_sym();
mp->frozen_end_group = mp_frozen_primitive (mp, "endgroup", mp_end_group, 0);
@:endgroup_}{\&{endgroup} primitive@>;
mp_primitive (mp, "everyjob", mp_every_job_command, 0);
@:every_job_}{\&{everyjob} primitive@>;
mp_primitive (mp, "exitif", mp_exit_test, 0);
@:exit_if_}{\&{exitif} primitive@>;
mp_primitive (mp, "expandafter", mp_expand_after, 0);
@:expand_after_}{\&{expandafter} primitive@>;
mp_primitive (mp, "interim", mp_interim_command, 0);
@:interim_}{\&{interim} primitive@>;
mp_primitive (mp, "let", mp_let_command, 0);
@:let_}{\&{let} primitive@>;
mp_primitive (mp, "newinternal", mp_new_internal, 0);
@:new_internal_}{\&{newinternal} primitive@>;
mp_primitive (mp, "of", mp_of_token, 0);
@:of_}{\&{of} primitive@>;
mp_primitive (mp, "randomseed", mp_random_seed, 0);
@:mp_random_seed_}{\&{randomseed} primitive@>;
mp_primitive (mp, "save", mp_save_command, 0);
@:save_}{\&{save} primitive@>;
mp_primitive (mp, "scantokens", mp_scan_tokens, 0);
@:scan_tokens_}{\&{scantokens} primitive@>;

mp_primitive (mp, "runscript", mp_runscript, 0);
@:run_script_}{\&{runscript} primitive@>;

mp_primitive (mp, "shipout", mp_ship_out_command, 0);
@:ship_out_}{\&{shipout} primitive@>;
mp_primitive (mp, "skipto", mp_skip_to, 0);
@:skip_to_}{\&{skipto} primitive@>;
mp_primitive (mp, "special", mp_special_command, 0);
@:special}{\&{special} primitive@>;
mp_primitive (mp, "fontmapfile", mp_special_command, 1);
@:fontmapfile}{\&{fontmapfile} primitive@>;
mp_primitive (mp, "fontmapline", mp_special_command, 2);
@:fontmapline}{\&{fontmapline} primitive@>;
mp_primitive (mp, "step", mp_step_token, 0);
@:step_}{\&{step} primitive@>;
mp_primitive (mp, "str", mp_str_op, 0);
@:str_}{\&{str} primitive@>;
mp_primitive (mp, "tension", mp_tension, 0);
@:tension_}{\&{tension} primitive@>;
mp_primitive (mp, "to", mp_to_token, 0);
@:to_}{\&{to} primitive@>;
mp_primitive (mp, "until", mp_until_token, 0);
@:until_}{\&{until} primitive@>;
mp_primitive (mp, "within", mp_within_token, 0);
@:within_}{\&{within} primitive@>;
mp_primitive (mp, "write", mp_write_command, 0);
@:write_}{\&{write} primitive@>
 

@ Each primitive has a corresponding inverse, so that it is possible to
display the cryptic numeric contents of |eqtb| in symbolic form.
Every call of |primitive| in this program is therefore accompanied by some
straightforward code that forms part of the |print_cmd_mod| routine
explained below.

@<Cases of |print_cmd_mod| for symbolic printing of primitives@>=
case mp_add_to_command:
mp_print (mp, "addto");
break;
case mp_assignment:
mp_print (mp, ":=");
break;
case mp_at_least:
mp_print (mp, "atleast");
break;
case mp_bchar_label:
mp_print (mp, "||:");
break;
case mp_begin_group:
mp_print (mp, "begingroup");
break;
case mp_colon:
mp_print (mp, ":");
break;
case mp_comma:
mp_print (mp, ",");
break;
case mp_controls:
mp_print (mp, "controls");
break;
case mp_curl_command:
mp_print (mp, "curl");
break;
case mp_delimiters:
mp_print (mp, "delimiters");
break;
case mp_double_colon:
mp_print (mp, "::");
break;
case mp_end_group:
mp_print (mp, "endgroup");
break;
case mp_every_job_command:
mp_print (mp, "everyjob");
break;
case mp_exit_test:
mp_print (mp, "exitif");
break;
case mp_expand_after:
mp_print (mp, "expandafter");
break;
case mp_interim_command:
mp_print (mp, "interim");
break;
case mp_left_brace:
mp_print (mp, "{");
break;
case mp_left_bracket:
mp_print (mp, "[");
break;
case mp_let_command:
mp_print (mp, "let");
break;
case mp_new_internal:
mp_print (mp, "newinternal");
break;
case mp_of_token:
mp_print (mp, "of");
break;
case mp_path_join:
mp_print (mp, "..");
break;
case mp_random_seed:
mp_print (mp, "randomseed");
break;
case mp_relax:
mp_print_char (mp, xord ('\\'));
break;
case mp_right_brace:
mp_print_char (mp, xord ('}'));
break;
case mp_right_bracket:
mp_print_char (mp, xord (']'));
break;
case mp_save_command:
mp_print (mp, "save");
break;
case mp_scan_tokens:
mp_print (mp, "scantokens");
break;
case mp_runscript:
mp_print (mp, "runscript");
break;
case mp_semicolon:
mp_print_char (mp, xord (';'));
break;
case mp_ship_out_command:
mp_print (mp, "shipout");
break;
case mp_skip_to:
mp_print (mp, "skipto");
break;
case mp_special_command:
if (m == 2)
  mp_print (mp, "fontmapline");
else if (m == 1)
  mp_print (mp, "fontmapfile");
else
  mp_print (mp, "special");
break;
case mp_step_token:
mp_print (mp, "step");
break;
case mp_str_op:
mp_print (mp, "str");
break;
case mp_tension:
mp_print (mp, "tension");
break;
case mp_to_token:
mp_print (mp, "to");
break;
case mp_until_token:
mp_print (mp, "until");
break;
case mp_within_token:
mp_print (mp, "within");
break;
case mp_write_command:
mp_print (mp, "write");
break;

@ We will deal with the other primitives later, at some point in the program
where their |eq_type| and |equiv| values are more meaningful.  For example,
the primitives for macro definitions will be loaded when we consider the
routines that define macros. It is easy to find where each particular
primitive was treated by looking in the index at the end; for example, the
section where |"def"| entered |eqtb| is listed under `\&{def} primitive'.

@* Token lists. 

A \MP\ token is either symbolic or numeric or a string, or it denotes a macro 
parameter or capsule or an internal; so there are six corresponding ways to 
encode it internally:
@^token@>

(1)~A symbolic token for symbol |p| is represented by the pointer |p|, 
in the |sym_sym| field of a symbolic node in~|mem|. The |type| field is |symbol_node|;
and it has a |name_type| to differentiate various subtypes of symbolic tokens, 
which is usually |normal_sym|, but |macro_sym| for macro names.

(2)~A numeric token whose |scaled| value is~|v| is
represented in a non-symbolic node of~|mem|; the |type| field is |known|,
the |name_type| field is |token|, and the |value| field holds~|v|.

(3)~A string token is also represented in a non-symbolic node; the |type|
field is |mp_string_type|, the |name_type| field is |token|, and the
|value| field holds the corresponding |mp_string|.  

(4)~Capsules have |name_type=capsule|, and their |type| and |value| fields 
represent arbitrary values, with |type| different from |symbol_node| 
(in ways to be explained later).  

(5)~Macro parameters appear in |sym_info| fields of symbolic nodes.  The |type| 
field is |symbol_node|; the $k$th parameter is represented by |k| in |sym_info|;
and |expr_sym| in |name_type|, if it is of type \&{expr}, or |suffix_sym| if it 
is of type \&{suffix}, or by |text_sym| if it is of type \&{text}. 

(6)~The $k$th internal is also represented by |k| in |sym_info|; the |type| field is 
|symbol_node| as for the other symbolic tokens; and |internal_sym| is its |name_type|;

Actual values of the parameters and internals are kept in a separate
stack, as we will see later.

Note that the `\\{type}' field of a node has nothing to do with ``type'' in a
printer's sense. It's curious that the same word is used in such different ways.

@d token_node_size sizeof(mp_node_data) /* the number of words in a large token node */

@d set_value_sym(A,B) do_set_value_sym(mp, (mp_token_node)(A), (B))
@d set_value_number(A,B) do_set_value_number(mp, (mp_token_node)(A), (B))
@d set_value_node(A,B) do_set_value_node(mp, (mp_token_node)(A), (B))
@d set_value_str(A,B) do_set_value_str(mp, (mp_token_node)(A), (B))
@d set_value_knot(A,B) do_set_value_knot(mp, (mp_token_node)A, (B))

@d value_sym_NEW(A) (mp_sym)mp_link(A)
@d set_value_sym_NEW(A,B) set_mp_link(A,(mp_node)B)

@<MPlib internal header stuff@>=
typedef struct mp_node_data *mp_token_node;

@ @c
#if DEBUG
#define value_sym(A)    do_get_value_sym(mp,(mp_token_node)(A))
/* |#define value_number(A) do_get_value_number(mp,(mp_token_node)(A))| */
#define value_number(A) ((mp_token_node)(A))->data.n
#define value_node(A)   do_get_value_node(mp,(mp_token_node)(A))
#define value_str(A)    do_get_value_str(mp,(mp_token_node)(A))
#define value_knot(A)   do_get_value_knot(mp,(mp_token_node)(A))
#else
#define value_sym(A)    ((mp_token_node)(A))->data.sym
#define value_number(A) ((mp_token_node)(A))->data.n
#define value_node(A)   ((mp_token_node)(A))->data.node
#define value_str(A)    ((mp_token_node)(A))->data.str
#define value_knot(A)   ((mp_token_node)(A))->data.p
#endif
static void do_set_value_sym(MP mp, mp_token_node A, mp_sym B) {
   FUNCTION_TRACE3 ("set_value_sym(%p,%p)\n", (A),(B));
   A->data.sym=(B);
}
static void do_set_value_number(MP mp, mp_token_node A, mp_number B) {
   FUNCTION_TRACE3 ("set_value(%p,%s)\n", (A), number_tostring(B));
   A->data.p = NULL;
   A->data.str = NULL;
   A->data.node = NULL;
   number_clone (A->data.n, B);
}
static void do_set_value_str(MP mp, mp_token_node A, mp_string B) {
   FUNCTION_TRACE3 ("set_value_str(%p,%p)\n", (A),(B));
   assert (A->type != mp_structured);
   A->data.p = NULL;
   A->data.str = (B);
   add_str_ref((B));
   A->data.node = NULL;
   number_clone (A->data.n, zero_t);
}
static void do_set_value_node(MP mp, mp_token_node A, mp_node B) {
   /* store the value in a large token node */
   FUNCTION_TRACE3 ("set_value_node(%p,%p)\n", A,B);
   assert (A->type != mp_structured);
   A->data.p = NULL;
   A->data.str = NULL;
   A->data.node = B;
   number_clone (A->data.n, zero_t);
}
static void do_set_value_knot(MP mp, mp_token_node A, mp_knot B) {
   FUNCTION_TRACE3 ("set_value_knot(%p,%p)\n", (A),(B));
   assert (A->type != mp_structured);
   A->data.p = (B);
   A->data.str = NULL;
   A->data.node = NULL;
   number_clone (A->data.n, zero_t);
}


@ @c
#if DEBUG
static mp_sym do_get_value_sym (MP mp, mp_token_node A) {
  /* |A->type| can be structured in this case */
  FUNCTION_TRACE3 ("%p = get_value_sym(%p)\n", A->data.sym, A);
  return A->data.sym ;
}
static mp_node do_get_value_node (MP mp, mp_token_node A) {
  assert (A->type != mp_structured);
  FUNCTION_TRACE3 ("%p = get_value_node(%p)\n", A->data.node, A);
  return  A->data.node ;
}
static mp_string do_get_value_str (MP mp, mp_token_node A) {
  assert (A->type != mp_structured);
  FUNCTION_TRACE3 ("%p = get_value_str(%p)\n", A->data.str, A);
  return  A->data.str ;
}
static mp_knot do_get_value_knot (MP mp, mp_token_node A) {
  assert (A->type != mp_structured);
  FUNCTION_TRACE3 ("%p = get_value_knot(%p)\n", A->data.p, A);
  return  A->data.p ;
}
static mp_number do_get_value_number (MP mp, mp_token_node A) {
  assert (A->type != mp_structured);
  FUNCTION_TRACE3 ("%d = get_value_number(%p)\n", A->data.n.type, A);
  return  A->data.n ;
}
#endif

@ @<Declarations@>=
#if DEBUG
static mp_number do_get_value_number (MP mp, mp_token_node A);
static mp_sym    do_get_value_sym    (MP mp, mp_token_node A);
static mp_node   do_get_value_node   (MP mp, mp_token_node A);
static mp_string do_get_value_str    (MP mp, mp_token_node A) ;
static mp_knot   do_get_value_knot   (MP mp, mp_token_node A) ;
#endif
static void do_set_value_sym    (MP mp, mp_token_node A, mp_sym B);
static void do_set_value_number (MP mp, mp_token_node A, mp_number B);
static void do_set_value_node   (MP mp, mp_token_node A, mp_node B);
static void do_set_value_str    (MP mp, mp_token_node A, mp_string B);
static void do_set_value_knot   (MP mp, mp_token_node A, mp_knot B);

@
@c
static mp_node mp_get_token_node (MP mp) {
  mp_node p; 
  if (mp->token_nodes) {
    p = mp->token_nodes;
    mp->token_nodes = p->link;
    mp->num_token_nodes--;
    p->link = NULL;
  } else {
    p = malloc_node (token_node_size);
    new_number(p->data.n);
    p->has_number = 1;
  }
  p->type = mp_token_node_type;
  FUNCTION_TRACE2 ("%p = mp_get_token_node()\n", p);
  return (mp_node) p;
}

@ @c
static void mp_free_token_node (MP mp, mp_node p) {
  FUNCTION_TRACE2 ("mp_free_token_node(%p)\n", p);
  if (!p) return;
  if (mp->num_token_nodes < max_num_token_nodes) {
    p->link = mp->token_nodes;
    mp->token_nodes = p;
    mp->num_token_nodes++;
    return;
  }
  mp->var_used -= token_node_size;
  if (mp->math_mode > mp_math_double_mode) {
    free_number(((mp_value_node)p)->data.n); 
  }
  xfree (p);
}

@ @<Declarations@>=
static void mp_free_token_node (MP mp, mp_node p);

@ A numeric token is created by the following trivial routine.

@c
static mp_node mp_new_num_tok (MP mp, mp_number v) {
  mp_node p;    /* the new node */
  p = mp_get_token_node (mp);
  set_value_number (p, v);
  p->type = mp_known;
  p->name_type = mp_token;
  FUNCTION_TRACE3 ("%p = mp_new_num_tok(%p)\n", p, v);
  return p;
}


@ A token list is a singly linked list of nodes in |mem|, where
each node contains a token and a link.  Here's a subroutine that gets rid
of a token list when it is no longer needed.

@c
static void mp_flush_token_list (MP mp, mp_node p) {
  mp_node q;    /* the node being recycled */
  FUNCTION_TRACE2 ("mp_flush_token_list(%p)\n", p);
  while (p != NULL) {
    q = p;
    p = mp_link (p);
    if (mp_type (q) == mp_symbol_node) {
      mp_free_symbolic_node (mp, q);
    } else {
      switch (mp_type (q)) {
      case mp_vacuous:
      case mp_boolean_type:
      case mp_known:
        break;
      case mp_string_type:
        delete_str_ref (value_str (q));
        break;
      case unknown_types:
      case mp_pen_type:
      case mp_path_type:
      case mp_picture_type:
      case mp_pair_type:
      case mp_color_type:
      case mp_cmykcolor_type:
      case mp_transform_type:
      case mp_dependent:
      case mp_proto_dependent:
      case mp_independent:
        mp_recycle_value (mp, q);
        break;
      default:
        mp_confusion (mp, "token");
@:this can't happen token}{\quad token@>;
      }
      mp_free_token_node (mp, q);
    }
  }
}


@ The procedure |show_token_list|, which prints a symbolic form of
the token list that starts at a given node |p|, illustrates these
conventions. The token list being displayed should not begin with a reference
count. 

An additional parameter |q| is also given; this parameter is either NULL
or it points to a node in the token list where a certain magic computation
takes place that will be explained later. (Basically, |q| is non-NULL when
we are printing the two-line context information at the time of an error
message; |q| marks the place corresponding to where the second line
should begin.)

The generation will stop, and `\.{\char`\ ETC.}' will be printed, if the length
of printing exceeds a given limit~|l|; the length of printing upon entry is
assumed to be a given amount called |null_tally|. (Note that
|show_token_list| sometimes uses itself recursively to print
variable names within a capsule.)
@^recursion@>

Unusual entries are printed in the form of all-caps tokens
preceded by a space, e.g., `\.{\char`\ BAD}'.

@<Declarations@>=
static void mp_show_token_list (MP mp, mp_node p, mp_node q, integer l,
                                integer null_tally);

@ @c
void mp_show_token_list (MP mp, mp_node p, mp_node q, integer l,
                         integer null_tally) {
  quarterword cclass, c; /* the |char_class| of previous and new tokens */
  cclass = percent_class;
  mp->tally = null_tally;
  while ((p != NULL) && (mp->tally < l)) {
    if (p == q) {
      set_trick_count();
    }
    /* Display token |p| and set |c| to its class; but |return| if there are problems */
    c = letter_class;               /* the default */
    if (mp_type (p) != mp_symbol_node) {
      /* Display non-symbolic token */
      if (mp_name_type (p) == mp_token) {
        if (mp_type (p) == mp_known) {
          /* Display a numeric token */
          if (cclass == digit_class)
            mp_print_char (mp, xord (' '));
          if (number_negative (value_number (p))) {
            if (cclass == mp_left_bracket_class)
              mp_print_char (mp, xord (' '));
            mp_print_char (mp, xord ('['));
            print_number (value_number (p));
            mp_print_char (mp, xord (']'));
            c = mp_right_bracket_class;
          } else {
            print_number (value_number (p));
            c = digit_class;
          }
    
        } else if (mp_type (p) != mp_string_type) {
          mp_print (mp, " BAD");
        } else {
          mp_print_char (mp, xord ('"'));
          mp_print_str (mp, value_str (p));
          mp_print_char (mp, xord ('"'));
          c = string_class;
        }
      } else if ((mp_name_type (p) != mp_capsule) || (mp_type (p) < mp_vacuous)
                 || (mp_type (p) > mp_independent)) {
        mp_print (mp, " BAD");
      } else {
        mp_print_capsule (mp, p);
        c = right_paren_class;
      }
    
    } else {
      if (mp_name_type (p) == mp_expr_sym ||
          mp_name_type (p) == mp_suffix_sym || mp_name_type (p) == mp_text_sym) {
        integer r; /* temporary register */
        r = mp_sym_info (p);
        if (mp_name_type (p) == mp_expr_sym) {
          mp_print (mp, "(EXPR");
        } else if (mp_name_type (p) == mp_suffix_sym) {
          mp_print (mp, "(SUFFIX");
        } else {
          mp_print (mp, "(TEXT");
        }
        mp_print_int (mp, r);
        mp_print_char (mp, xord (')'));
        c = right_paren_class;
      } else {
        mp_sym sr = mp_sym_sym (p);
        if (sr == collective_subscript) {
          /* Display a collective subscript */
          if (cclass == mp_left_bracket_class)
            mp_print_char (mp, xord (' '));
          mp_print (mp, "[]");
          c = mp_right_bracket_class;
    
        } else {
          mp_string rr = text (sr);
          if (rr == NULL || rr->str == NULL) {
            mp_print (mp, " NONEXISTENT");
          } else {
            /* Print string |r| as a symbolic token and set |c| to its class */
            c = (quarterword) mp->char_class[(rr->str[0])];
            if (c == cclass) {
              switch (c) {
              case letter_class:
                mp_print_char (mp, xord ('.'));
                break;
              case isolated_classes:
                break;
              default:
                mp_print_char (mp, xord (' '));
                break;
              }
            }
            mp_print_str (mp, rr);
    
          }
        }
      }
    }
        
    cclass = c;
    p = mp_link (p);
  }
  if (p != NULL)
    mp_print (mp, " ETC.");
  return;
}


@ @<Declarations@>=
static void mp_print_capsule (MP mp, mp_node p);

@ @<Declare miscellaneous procedures that were declared |forward|@>=
void mp_print_capsule (MP mp, mp_node p) {
  mp_print_char (mp, xord ('('));
  mp_print_exp (mp, p, 0);
  mp_print_char (mp, xord (')'));
}


@ Macro definitions are kept in \MP's memory in the form of token lists
that have a few extra symbolic nodes at the beginning.

The first node contains a reference count that is used to tell when the
list is no longer needed. To emphasize the fact that a reference count is
present, we shall refer to the |sym_info| field of this special node as the
|ref_count| field.
@^reference counts@>

The next node or nodes after the reference count serve to describe the
formal parameters. They consist of zero or more parameter tokens followed
by a code for the type of macro.

/* reference count preceding a macro definition or picture header */
@d ref_count(A) indep_value(A)
@d set_ref_count(A,B) set_indep_value(A,B)
@d add_mac_ref(A)  set_ref_count((A),ref_count((A))+1) /* make a new reference to a macro list */
@d decr_mac_ref(A) set_ref_count((A),ref_count((A))-1) /* remove a reference to a macro list */

@<Types...@>=
typedef enum {
 mp_general_macro, /* preface to a macro defined with a parameter list */
 mp_primary_macro, /* preface to a macro with a \&{primary} parameter */
 mp_secondary_macro, /* preface to a macro with a \&{secondary} parameter */
 mp_tertiary_macro, /* preface to a macro with a \&{tertiary} parameter */
 mp_expr_macro, /* preface to a macro with an undelimited \&{expr} parameter */
 mp_of_macro, /* preface to a macro with undelimited `\&{expr} |x| \&{of}~|y|' parameters */
 mp_suffix_macro, /* preface to a macro with an undelimited \&{suffix} parameter */
 mp_text_macro, /* preface to a macro with an undelimited \&{text} parameter */
 mp_expr_param, /* used by \.{expr} primitive */
 mp_suffix_param, /* used by \.{suffix} primitive */
 mp_text_param /* used by \.{text} primitive */
} mp_macro_info;

@ @c
static void mp_delete_mac_ref (MP mp, mp_node p) {
  /* |p| points to the reference count of a macro list that is
     losing one reference */
  if (ref_count (p) == 0)
    mp_flush_token_list (mp, p);
  else
    decr_mac_ref (p);
}


@ The following subroutine displays a macro, given a pointer to its
reference count.

@c
static void mp_show_macro (MP mp, mp_node p, mp_node q, integer l) {
  mp_node r;    /* temporary storage */
  p = mp_link (p);              /* bypass the reference count */
  while (mp_name_type (p) != mp_macro_sym) {
    r = mp_link (p);
    mp_link (p) = NULL;
    mp_show_token_list (mp, p, NULL, l, 0);
    mp_link (p) = r;
    p = r;
    if (l > 0)
      l = l - mp->tally;
    else
      return;
  }                             /* control printing of `\.{ETC.}' */
@.ETC@>;
  mp->tally = 0;
  switch (mp_sym_info (p)) {
  case mp_general_macro:
    mp_print (mp, "->");
    break;
@.->@>
  case mp_primary_macro:
  case mp_secondary_macro:
  case mp_tertiary_macro:
    mp_print_char (mp, xord ('<'));
    mp_print_cmd_mod (mp, mp_param_type, mp_sym_info (p));
    mp_print (mp, ">->");
    break;
  case mp_expr_macro:
    mp_print (mp, "<expr>->");
    break;
  case mp_of_macro:
    mp_print (mp, "<expr>of<primary>->");
    break;
  case mp_suffix_macro:
    mp_print (mp, "<suffix>->");
    break;
  case mp_text_macro:
    mp_print (mp, "<text>->");
    break;
  }                             /* there are no other cases */
  mp_show_token_list (mp, mp_link (p), q, l - mp->tally, 0);
}


@* Data structures for variables.
The variables of \MP\ programs can be simple, like `\.x', or they can
combine the structural properties of arrays and records, like `\.{x20a.b}'.
A \MP\ user assigns a type to a variable like \.{x20a.b} by saying, for
example, `\.{boolean} \.{x[]a.b}'. It's time for us to study how such
things are represented inside of the computer.

Each variable value occupies two consecutive words, either in a non-symbolic
node called a value node, or as a non-symbolic subfield of a larger node.  One
of those two words is called the |value| field; it is an integer,
containing either a |scaled| numeric value or the representation of some
other type of quantity. (It might also be subdivided into halfwords, in
which case it is referred to by other names instead of |value|.) The other
word is broken into subfields called |type|, |name_type|, and |link|.  The
|type| field is a quarterword that specifies the variable's type, and
|name_type| is a quarterword from which \MP\ can reconstruct the
variable's name (sometimes by using the |link| field as well).  Thus, only
1.25 words are actually devoted to the value itself; the other
three-quarters of a word are overhead, but they aren't wasted because they
allow \MP\ to deal with sparse arrays and to provide meaningful diagnostics.

In this section we shall be concerned only with the structural aspects of
variables, not their values. Later parts of the program will change the
|type| and |value| fields, but we shall treat those fields as black boxes
whose contents should not be touched.

However, if the |type| field is |mp_structured|, there is no |value| field,
and the second word is broken into two pointer fields called |attr_head|
and |subscr_head|. Those fields point to additional nodes that
contain structural information, as we shall see.

TH Note: DEK and JDH had a nice theoretical split between |value|,
|attr| and |subscr| nodes, as documented above and further
below. However, all three types had a bad habit of transmuting into
each other in practice while pointers to them still lived on
elsewhere, so using three different C structures is simply not
workable. All three are now represented as a single C structure called
|mp_value_node|.

There is a potential union in this structure in the interest of space
saving: |subscript_| and |hashloc_| are mutually exclusive.

Actually, so are |attr_head_| + |subscr_head_| on one side and and
|value_| on the other, but because of all the access macros that are
used in the code base to get at values, those cannot be folded into a
union (yet); this would have required creating a similar union in
|mp_token_node| where it would only serve to confuse things.

Finally, |parent_| only applies in |attr| nodes (the ones that have
|hashloc_|), but creating an extra substructure inside the union just
for that does not save space and the extra complication in the
structure is not worth the minimal extra code clarification.

@d attr_head(A)   do_get_attr_head(mp,(mp_value_node)(A))
@d set_attr_head(A,B) do_set_attr_head(mp,(mp_value_node)(A),(mp_node)(B))

@d subscr_head(A)   do_get_subscr_head(mp,(mp_value_node)(A))
@d set_subscr_head(A,B) do_set_subscr_head(mp,(mp_value_node)(A),(mp_node)(B))

@<MPlib internal header stuff@>=
typedef struct mp_value_node_data {
  NODE_BODY;
  mp_value_data data;
  mp_number subscript_;
  mp_sym hashloc_;
  mp_node parent_;
  mp_node attr_head_;
  mp_node subscr_head_;
} mp_value_node_data;

@ @c
static mp_node do_get_attr_head (MP mp, mp_value_node A) {
  assert (A->type == mp_structured);
  FUNCTION_TRACE3 ("%p = get_attr_head(%p)\n", A->attr_head_, A);
  return A->attr_head_;
}
static mp_node do_get_subscr_head (MP mp, mp_value_node A) {
  assert (A->type == mp_structured);
  FUNCTION_TRACE3 ("%p = get_subscr_head(%p)\n", A->subscr_head_, A);
  return A->subscr_head_;
}
static void do_set_attr_head (MP mp, mp_value_node A, mp_node d) {
   FUNCTION_TRACE4 ("set_attr_head(%p,%p) on line %d\n", (A), d, __LINE__);
   assert (A->type == mp_structured);
   A->attr_head_ = d;
}
static void do_set_subscr_head (MP mp, mp_value_node A, mp_node d) {
   FUNCTION_TRACE4 ("set_subscr_head(%p,%p) on line %d\n", (A), d, __LINE__);
   assert (A->type == mp_structured);
   A->subscr_head_ = d;
}

@ @<Declarations@>=
static mp_node do_get_subscr_head (MP mp, mp_value_node A);
static mp_node do_get_attr_head (MP mp, mp_value_node A);
static void do_set_attr_head (MP mp, mp_value_node A, mp_node d);
static void do_set_subscr_head (MP mp, mp_value_node A, mp_node d);

@ It would have been nicer to make |mp_get_value_node| return
|mp_value_node| variables, but with |eqtb| as it stands that 
became messy: lots of typecasts. So, it returns a simple
|mp_node| for now.

@d value_node_size sizeof(struct mp_value_node_data)

@c
static mp_node mp_get_value_node (MP mp) {
  mp_value_node p; 
  if (mp->value_nodes) {
    p = (mp_value_node)mp->value_nodes;
    mp->value_nodes = p->link;
    mp->num_value_nodes--;
    p->link = NULL;
  } else {
    p = malloc_node (value_node_size);
    new_number(p->data.n);
    new_number(p->subscript_);
    p->has_number = 2;
  }
  mp_type (p) = mp_value_node_type;
  FUNCTION_TRACE2 ("%p = mp_get_value_node()\n", p);
  return (mp_node)p;
}
#if DEBUG > 1
static void debug_dump_value_node (mp_node x) {
  mp_value_node qq = (mp_value_node)x;
  fprintf (stdout, "\nnode %p:\n", qq);
  fprintf (stdout, "  type=%s\n", mp_type_string(qq->type));
  fprintf (stdout, "  name_type=%d\n", qq->name_type);
  fprintf (stdout, "  link=%p\n", qq->link);
  fprintf (stdout, "  data.n=%d\n", qq->data.n.type);
  if (is_number(qq->data.n)) {
    fprintf (stdout, "    data.n.data.val=%d\n",  qq->data.n.data.val);
    fprintf (stdout, "    data.n.data.dval=%f\n", qq->data.n.data.dval);
  }
  fprintf (stdout, "  data.str=%p\n", qq->data.str);
  if (qq->data.str != NULL) {
    fprintf (stdout, "    data.str->len=%d\n", (int)qq->data.str->len);
    fprintf (stdout, "    data.str->str=%s\n", qq->data.str->str);
  }
  fprintf (stdout, "  data.indep.serial=%d\n  data.indep.scale=%d\n", qq->data.indep.serial, 
                                                                       qq->data.indep.scale);
  fprintf (stdout, "  data.sym=%p\n", qq->data.sym);
  fprintf (stdout, "  data.p=%p\n", qq->data.p);
  fprintf (stdout, "  data.node=%p\n", qq->data.node);
  fprintf (stdout, "  subscript=%d\n", qq->subscript_.type);
  if (is_number(qq->subscript_)) {
    fprintf (stdout, "    subscript_.data.val=%d\n",  qq->subscript_.data.val);
    fprintf (stdout, "    subscript_.data.dval=%f\n", qq->subscript_.data.dval);
  }
  fprintf (stdout, "  hashloc=%p\n", qq->hashloc_);
  fprintf (stdout, "  parent=%p\n", qq->parent_);
  fprintf (stdout, "  attr_head=%p\n", qq->attr_head_);
  fprintf (stdout, "  subscr_head=%p\n\n", qq->subscr_head_);
}
#endif

@ @<Declarations@>=
static mp_node mp_get_value_node (MP mp);
#if DEBUG > 1
static void debug_dump_value_node (mp_node x);
#endif

@ An attribute node is three words long. Two of these words contain |type|
and |value| fields as described above, and the third word contains
additional information:  There is an |hashloc| field, which contains the
hash address of the token that names this attribute; and there's also a
|parent| field, which points to the value node of |mp_structured| type at the
next higher level (i.e., at the level to which this attribute is
subsidiary).  The |name_type| in an attribute node is `|attr|'.  The
|link| field points to the next attribute with the same parent; these are
arranged in increasing order, so that |hashloc(mp_link(p))>hashloc(p)|. The
final attribute node links to the constant |end_attr|, whose |hashloc|
field is greater than any legal hash address. The |attr_head| in the
parent points to a node whose |name_type| is |mp_structured_root|; this
node represents the NULL attribute, i.e., the variable that is relevant
when no attributes are attached to the parent. The |attr_head| node
has the fields of either
a value node, a subscript node, or an attribute node, depending on what
the parent would be if it were not structured; but the subscript and
attribute fields are ignored, so it effectively contains only the data of
a value node. The |link| field in this special node points to an attribute
node whose |hashloc| field is zero; the latter node represents a collective
subscript `\.{[]}' attached to the parent, and its |link| field points to
the first non-special attribute node (or to |end_attr| if there are none).

A subscript node likewise occupies three words, with |type| and |value| fields
plus extra information; its |name_type| is |subscr|. In this case the
third word is called the |subscript| field, which is a |scaled| integer.
The |link| field points to the subscript node with the next larger
subscript, if any; otherwise the |link| points to the attribute node
for collective subscripts at this level. We have seen that the latter node
contains an upward pointer, so that the parent can be deduced.

The |name_type| in a parent-less value node is |root|, and the |link|
is the hash address of the token that names this value.

In other words, variables have a hierarchical structure that includes
enough threads running around so that the program is able to move easily
between siblings, parents, and children. An example should be helpful:
(The reader is advised to draw a picture while reading the following
description, since that will help to firm up the ideas.)
Suppose that `\.x' and `\.{x.a}' and `\.{x[]b}' and `\.{x5}'
and `\.{x20b}' have been mentioned in a user's program, where
\.{x[]b} has been declared to be of \&{boolean} type. Let |h(x)|, |h(a)|,
and |h(b)| be the hash addresses of \.x, \.a, and~\.b. Then
|eq_type(h(x))=name| and |equiv(h(x))=p|, where |p|~is a non-symbolic value
node with |mp_name_type(p)=root| and |mp_link(p)=h(x)|. We have |type(p)=mp_structured|,
|attr_head(p)=q|, and |subscr_head(p)=r|, where |q| points to a value
node and |r| to a subscript node. (Are you still following this? Use
a pencil to draw a diagram.) The lone variable `\.x' is represented by
|type(q)| and |value(q)|; furthermore
|mp_name_type(q)=mp_structured_root| and |mp_link(q)=q1|, where |q1| points
to an attribute node representing `\.{x[]}'. Thus |mp_name_type(q1)=attr|,
|hashloc(q1)=collective_subscript=0|, |parent(q1)=p|,
|type(q1)=mp_structured|, |attr_head(q1)=qq|, and |subscr_head(q1)=qq1|;
|qq| is a  three-word ``attribute-as-value'' node with |type(qq)=numeric_type|
(assuming that \.{x5} is numeric, because |qq| represents `\.{x[]}' 
with no further attributes), |mp_name_type(qq)=structured_root|, 
|hashloc(qq)=0|, |parent(qq)=p|, and
|mp_link(qq)=qq1|. (Now pay attention to the next part.) Node |qq1| is
an attribute node representing `\.{x[][]}', which has never yet
occurred; its |type| field is |undefined|, and its |value| field is
undefined. We have |mp_name_type(qq1)=attr|, |hashloc(qq1)=collective_subscript|,
|parent(qq1)=q1|, and |mp_link(qq1)=qq2|. Since |qq2| represents
`\.{x[]b}', |type(qq2)=mp_unknown_boolean|; also |hashloc(qq2)=h(b)|,
|parent(qq2)=q1|, |mp_name_type(qq2)=attr|, |mp_link(qq2)=end_attr|.
(Maybe colored lines will help untangle your picture.)
 Node |r| is a subscript node with |type| and |value|
representing `\.{x5}'; |mp_name_type(r)=subscr|, |subscript(r)=5.0|,
and |mp_link(r)=r1| is another subscript node. To complete the picture,
see if you can guess what |mp_link(r1)| is; give up? It's~|q1|.
Furthermore |subscript(r1)=20.0|, |mp_name_type(r1)=subscr|,
|type(r1)=mp_structured|, |attr_head(r1)=qqq|, |subscr_head(r1)=qqq1|,
and we finish things off with three more nodes
|qqq|, |qqq1|, and |qqq2| hung onto~|r1|. (Perhaps you should start again
with a larger sheet of paper.) The value of variable \.{x20b}
appears in node~|qqq2|, as you can well imagine.

If the example in the previous paragraph doesn't make things crystal
clear, a glance at some of the simpler subroutines below will reveal how
things work out in practice.

The only really unusual thing about these conventions is the use of
collective subscript attributes. The idea is to avoid repeating a lot of
type information when many elements of an array are identical macros
(for which distinct values need not be stored) or when they don't have
all of the possible attributes. Branches of the structure below collective
subscript attributes do not carry actual values except for macro identifiers;
branches of the structure below subscript nodes do not carry significant
information in their collective subscript attributes.


@c
#if DEBUG
#define hashloc(A)       do_get_hashloc(mp,(mp_value_node)(A))
#define set_hashloc(A,B) do_set_hashloc (mp,(mp_value_node)A, B)
#define parent(A)        do_get_parent(mp, A)
#define set_parent(A,B)  do_set_parent (mp,(mp_value_node)A, B)
static mp_sym do_get_hashloc (MP mp, mp_value_node A) {
  assert((A)->type == mp_attr_node_type || (A)->name_type == mp_attr);
  return (A)->hashloc_;
}
static void do_set_hashloc (MP mp, mp_value_node A, mp_sym B) {
  FUNCTION_TRACE4 ("set_hashloc(%p,%p) on line %d\n", (A), (B), __LINE__);
   assert((A)->type == mp_attr_node_type || (A)->name_type == mp_attr);
   A->hashloc_ = B;
}
static mp_node do_get_parent (MP mp, mp_value_node A) {
  assert((A)->type == mp_attr_node_type || (A)->name_type == mp_attr);
  return (A)->parent_; /* pointer to |mp_structured| variable */
}
static void do_set_parent (MP mp, mp_value_node A, mp_node d) {
   assert((A)->type == mp_attr_node_type || (A)->name_type == mp_attr);
   FUNCTION_TRACE4 ("set_parent(%p,%p) on line %d\n", (A), d, __LINE__);
   A->parent_ = d;
}
#else
#define hashloc(A)       ((mp_value_node)(A))->hashloc_
#define set_hashloc(A,B) ((mp_value_node)(A))->hashloc_ = B
#define parent(A)        ((mp_value_node)(A))->parent_
#define set_parent(A,B)  ((mp_value_node)(A))->parent_ = B
#endif

@ 
@d mp_free_attr_node(a,b) do {
   assert((b)->type == mp_attr_node_type || (b)->name_type == mp_attr);
   mp_free_value_node(a,b);
} while (0)

@c
static mp_value_node mp_get_attr_node (MP mp) {
  mp_value_node p = (mp_value_node) mp_get_value_node (mp);
  mp_type (p) = mp_attr_node_type;
  return p;
}


@ Setting the |hashloc| field of |end_attr| to a value greater than
any legal hash address is done by assigning $-1$ typecasted to
|mp_sym|, hopefully resulting in all bits being set. On systems that
support negative pointer values or where typecasting $-1$ does not
result in all bits in a pointer being set, something else needs to be done.  
@^system dependencies@>

@<Initialize table...@>=
mp->end_attr = (mp_node) mp_get_attr_node (mp);
set_hashloc (mp->end_attr, (mp_sym)-1);
set_parent ((mp_value_node) mp->end_attr, NULL);

@ @<Free table...@>=
mp_free_attr_node (mp, mp->end_attr);

@
@d collective_subscript (void *)0 /* code for the attribute `\.{[]}' */
@d subscript(A) ((mp_value_node)(A))->subscript_
@d set_subscript(A,B) do_set_subscript (mp, (mp_value_node)(A), B)

@c
static void do_set_subscript (MP mp, mp_value_node A, mp_number B) {
  FUNCTION_TRACE3("set_subscript(%p,%p)\n", (A), (B));
  assert((A)->type == mp_subscr_node_type || (A)->name_type == mp_subscr);
  number_clone(A->subscript_,B); /* subscript of this variable */
}

@ 
@c
static mp_value_node mp_get_subscr_node (MP mp) {
  mp_value_node p = (mp_value_node) mp_get_value_node (mp);
  mp_type (p) = mp_subscr_node_type;
  return p;
}


@ Variables of type \&{pair} will have values that point to four-word
nodes containing two numeric values. The first of these values has
|name_type=mp_x_part_sector| and the second has |name_type=mp_y_part_sector|;
the |link| in the first points back to the node whose |value| points
to this four-word node.

@d x_part(A) ((mp_pair_node)(A))->x_part_ /* where the \&{xpart} is found in a pair node */
@d y_part(A) ((mp_pair_node)(A))->y_part_ /* where the \&{ypart} is found in a pair node */

@<MPlib internal header stuff@>=
typedef struct mp_pair_node_data {
  NODE_BODY;
  mp_node x_part_;
  mp_node y_part_;
} mp_pair_node_data;
typedef struct mp_pair_node_data *mp_pair_node;

@
@d pair_node_size sizeof(struct mp_pair_node_data) /* the number of words in a subscript node */

@c
static mp_node mp_get_pair_node (MP mp) {
  mp_node p;
  if (mp->pair_nodes) {
    p = mp->pair_nodes;
    mp->pair_nodes = p->link;
    mp->num_pair_nodes--;
    p->link = NULL;
  } else {
    p = malloc_node (pair_node_size);
  }
  mp_type (p) = mp_pair_node_type;
  FUNCTION_TRACE2("get_pair_node(): %p\n", p);
  return (mp_node) p;
}

@ @<Declarations@>=
void mp_free_pair_node (MP mp, mp_node p);

@ @c
void mp_free_pair_node (MP mp, mp_node p) {
  FUNCTION_TRACE2 ("mp_free_pair_node(%p)\n", p);
  if (!p) return;
  if (mp->num_pair_nodes < max_num_pair_nodes) {
    p->link = mp->pair_nodes;
    mp->pair_nodes = p;
    mp->num_pair_nodes++;
    return;
  }
  mp->var_used -= pair_node_size;
  xfree (p);
}


@ If |type(p)=mp_pair_type| or if |value(p)=NULL|, the procedure call |init_pair_node(p)| will 
allocate a pair node for~|p|.  The individual parts of such nodes are  initially of type 
|mp_independent|.

@c
static void mp_init_pair_node (MP mp, mp_node p) {
  mp_node q;    /* the new node */
  mp_type (p) = mp_pair_type;
  q = mp_get_pair_node (mp);
  y_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, y_part (q));   /* sets |type(q)| and |value(q)| */
  mp_name_type (y_part (q)) = (quarterword) (mp_y_part_sector);
  mp_link (y_part (q)) = p;
  x_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, x_part (q));   /* sets |type(q)| and |value(q)| */
  mp_name_type (x_part (q)) = (quarterword) (mp_x_part_sector);
  mp_link (x_part (q)) = p;
  set_value_node (p, q);
}


@ 
Variables of type \&{transform} are similar, but in this case their
|value| points to a 12-word node containing six values, identified by
|x_part_sector|, |y_part_sector|, |mp_xx_part_sector|, |mp_xy_part_sector|,
|mp_yx_part_sector|, and |mp_yy_part_sector|.

@d tx_part(A) ((mp_transform_node)(A))->tx_part_ /* where the \&{xpart} is found in a transform node */
@d ty_part(A) ((mp_transform_node)(A))->ty_part_ /* where the \&{ypart} is found in a transform node */
@d xx_part(A) ((mp_transform_node)(A))->xx_part_ /* where the \&{xxpart} is found in a transform node */
@d xy_part(A) ((mp_transform_node)(A))->xy_part_ /* where the \&{xypart} is found in a transform node */
@d yx_part(A) ((mp_transform_node)(A))->yx_part_ /* where the \&{yxpart} is found in a transform node */
@d yy_part(A) ((mp_transform_node)(A))->yy_part_ /* where the \&{yypart} is found in a transform node */

@<MPlib internal header stuff@>=
typedef struct mp_transform_node_data {
  NODE_BODY;
  mp_node tx_part_;
  mp_node ty_part_;
  mp_node xx_part_;
  mp_node yx_part_;
  mp_node xy_part_;
  mp_node yy_part_;
} mp_transform_node_data;
typedef struct mp_transform_node_data *mp_transform_node;

@
@d transform_node_size sizeof(struct mp_transform_node_data) /* the number of words in a subscript node */

@c
static mp_node mp_get_transform_node (MP mp) {
  mp_transform_node p = (mp_transform_node) malloc_node (transform_node_size);
  mp_type (p) = mp_transform_node_type;
  return (mp_node) p;
}


@ @c
static void mp_init_transform_node (MP mp, mp_node p) {
  mp_node q;    /* the new node */
  mp_type (p) = mp_transform_type;
  q = mp_get_transform_node (mp);       /* big node */
  yy_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, yy_part (q));  /* sets |type(q)| and |value(q)| */
  mp_name_type (yy_part (q)) = (quarterword) (mp_yy_part_sector);
  mp_link (yy_part (q)) = p;
  yx_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, yx_part (q));  /* sets |type(q)| and |value(q)| */
  mp_name_type (yx_part (q)) = (quarterword) (mp_yx_part_sector);
  mp_link (yx_part (q)) = p;
  xy_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, xy_part (q));  /* sets |type(q)| and |value(q)| */
  mp_name_type (xy_part (q)) = (quarterword) (mp_xy_part_sector);
  mp_link (xy_part (q)) = p;
  xx_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, xx_part (q));  /* sets |type(q)| and |value(q)| */
  mp_name_type (xx_part (q)) = (quarterword) (mp_xx_part_sector);
  mp_link (xx_part (q)) = p;
  ty_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, ty_part (q));  /* sets |type(q)| and |value(q)| */
  mp_name_type (ty_part (q)) = (quarterword) (mp_y_part_sector);
  mp_link (ty_part (q)) = p;
  tx_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, tx_part (q));  /* sets |type(q)| and |value(q)| */
  mp_name_type (tx_part (q)) = (quarterword) (mp_x_part_sector);
  mp_link (tx_part (q)) = p;
  set_value_node (p, q);
}


@
Variables of type \&{color} have 3~values in 6~words identified by |mp_red_part_sector|, 
|mp_green_part_sector|, and |mp_blue_part_sector|.

@d red_part(A) ((mp_color_node)(A))->red_part_ /* where the \&{redpart} is found in a color node */
@d green_part(A) ((mp_color_node)(A))->green_part_ /* where the \&{greenpart} is found in a color node */
@d blue_part(A) ((mp_color_node)(A))->blue_part_ /* where the \&{bluepart} is found in a color node */

@d grey_part(A) red_part(A) /* where the \&{greypart} is found in a color node */

@<MPlib internal header stuff@>=
typedef struct mp_color_node_data {
  NODE_BODY;
  mp_node red_part_;
  mp_node green_part_;
  mp_node blue_part_;
} mp_color_node_data;
typedef struct mp_color_node_data *mp_color_node;

@
@d color_node_size sizeof(struct mp_color_node_data) /* the number of words in a subscript node */

@c
static mp_node mp_get_color_node (MP mp) {
  mp_color_node p = (mp_color_node) malloc_node (color_node_size);
  mp_type (p) = mp_color_node_type;
  p->link = NULL;
  return (mp_node) p;
}


@ 
@c
static void mp_init_color_node (MP mp, mp_node p) {
  mp_node q;    /* the new node */
  mp_type (p) = mp_color_type;
  q = mp_get_color_node (mp);   /* big node */
  blue_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, blue_part (q));        /* sets |type(q)| and |value(q)| */
  mp_name_type (blue_part (q)) = (quarterword) (mp_blue_part_sector);
  mp_link (blue_part (q)) = p;
  green_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, green_part (q));       /* sets |type(q)| and |value(q)| */
  mp_name_type (y_part (q)) = (quarterword) (mp_green_part_sector);
  mp_link (green_part (q)) = p;
  red_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, red_part (q)); /* sets |type(q)| and |value(q)| */
  mp_name_type (red_part (q)) = (quarterword) (mp_red_part_sector);
  mp_link (red_part (q)) = p;
  set_value_node (p, q);
}


@ Finally, variables of type |cmykcolor|.

@d cyan_part(A)    ((mp_cmykcolor_node)(A))->cyan_part_ /* where the \&{cyanpart} is found in a color node */
@d magenta_part(A) ((mp_cmykcolor_node)(A))->magenta_part_ /* where the \&{magentapart} is found in a color node */
@d yellow_part(A)  ((mp_cmykcolor_node)(A))->yellow_part_ /* where the \&{yellowpart} is found in a color node */
@d black_part(A)   ((mp_cmykcolor_node)(A))->black_part_ /* where the \&{blackpart} is found in a color node */

@<MPlib internal header stuff@>=
typedef struct mp_cmykcolor_node_data {
  NODE_BODY;
  mp_node cyan_part_;
  mp_node magenta_part_;
  mp_node yellow_part_;
  mp_node black_part_;
} mp_cmykcolor_node_data;
typedef struct mp_cmykcolor_node_data *mp_cmykcolor_node;

@
@d cmykcolor_node_size sizeof(struct mp_cmykcolor_node_data) /* the number of words in a subscript node */

@c
static mp_node mp_get_cmykcolor_node (MP mp) {
  mp_cmykcolor_node p = (mp_cmykcolor_node) malloc_node (cmykcolor_node_size);
  mp_type (p) = mp_cmykcolor_node_type;
  p->link = NULL;
  return (mp_node) p;
}


@
@c
static void mp_init_cmykcolor_node (MP mp, mp_node p) {
  mp_node q;    /* the new node */
  mp_type (p) = mp_cmykcolor_type;
  q = mp_get_cmykcolor_node (mp);       /* big node */
  black_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, black_part (q));       /* sets |type(q)| and |value(q)| */
  mp_name_type (black_part (q)) = (quarterword) (mp_black_part_sector);
  mp_link (black_part (q)) = p;
  yellow_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, yellow_part (q));      /* sets |type(q)| and |value(q)| */
  mp_name_type (yellow_part (q)) = (quarterword) (mp_yellow_part_sector);
  mp_link (yellow_part (q)) = p;
  magenta_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, magenta_part (q));     /* sets |type(q)| and |value(q)| */
  mp_name_type (magenta_part (q)) = (quarterword) (mp_magenta_part_sector);
  mp_link (magenta_part (q)) = p;
  cyan_part (q) = mp_get_value_node (mp);
  mp_new_indep (mp, cyan_part (q));        /* sets |type(q)| and |value(q)| */
  mp_name_type (cyan_part (q)) = (quarterword) (mp_cyan_part_sector);
  mp_link (cyan_part (q)) = p;
  set_value_node (p, q);
}


@ When an entire structured variable is saved, the |root| indication
is temporarily replaced by |saved_root|.

Some variables have no name; they just are used for temporary storage
while expressions are being evaluated. We call them {\sl capsules}.

@ The |id_transform| function creates a capsule for the
identity transformation.

@c
static mp_node mp_id_transform (MP mp) {
  mp_node p, q; /* list manipulation registers */
  p = mp_get_value_node (mp);
  mp_name_type (p) = mp_capsule;
  set_value_number (p, zero_t);             /* todo: this was |null| */
  mp_init_transform_node (mp, p);
  q = value_node (p);
  mp_type (tx_part (q)) = mp_known;
  set_value_number (tx_part (q), zero_t);
  mp_type (ty_part (q)) = mp_known;
  set_value_number (ty_part (q), zero_t);
  mp_type (xy_part (q)) = mp_known;
  set_value_number (xy_part (q), zero_t);
  mp_type (yx_part (q)) = mp_known;
  set_value_number (yx_part (q), zero_t);
  mp_type (xx_part (q)) = mp_known;
  set_value_number (xx_part (q), unity_t);
  mp_type (yy_part (q)) = mp_known;
  set_value_number (yy_part (q), unity_t);
  return p;
}


@ Tokens are of type |tag_token| when they first appear, but they point
to |NULL| until they are first used as the root of a variable.
The following subroutine establishes the root node on such grand occasions.

@c
static void mp_new_root (MP mp, mp_sym x) {
  mp_node p;    /* the new node */
  p = mp_get_value_node (mp);
  mp_type (p) = mp_undefined;
  mp_name_type (p) = mp_root;
  set_value_sym (p, x);
  set_equiv_node (x, p);
}


@ These conventions for variable representation are illustrated by the
|print_variable_name| routine, which displays the full name of a
variable given only a pointer to its value.

@<Declarations@>=
static void mp_print_variable_name (MP mp, mp_node p);

@ @c
void mp_print_variable_name (MP mp, mp_node p) {
  mp_node q;    /* a token list that will name the variable's suffix */
  mp_node r;    /* temporary for token list creation */
  while (mp_name_type (p) >= mp_x_part_sector) {
    switch (mp_name_type (p)) {
    case mp_x_part_sector:      mp_print (mp, "xpart ");      break;
    case mp_y_part_sector:      mp_print (mp, "ypart ");      break;
    case mp_xx_part_sector:     mp_print (mp, "xxpart ");     break;
    case mp_xy_part_sector:     mp_print (mp, "xypart ");     break;
    case mp_yx_part_sector:     mp_print (mp, "yxpart ");     break;
    case mp_yy_part_sector:     mp_print (mp, "yypart ");     break;
    case mp_red_part_sector:    mp_print (mp, "redpart ");    break;
    case mp_green_part_sector:  mp_print (mp, "greenpart ");  break;
    case mp_blue_part_sector:   mp_print (mp, "bluepart ");   break;
    case mp_cyan_part_sector:   mp_print (mp, "cyanpart ");   break;
    case mp_magenta_part_sector:mp_print (mp, "magentapart ");break;
    case mp_yellow_part_sector: mp_print (mp, "yellowpart "); break;
    case mp_black_part_sector:  mp_print (mp, "blackpart ");  break;
    case mp_grey_part_sector:   mp_print (mp, "greypart ");   break;
    case mp_capsule:            mp_printf (mp, "%%CAPSULE%p",p); return; break;
    /* this is to please the compiler: the remaining cases are operation codes */
    default: break;
    }
    p = mp_link (p);
  }
  q = NULL;
  while (mp_name_type (p) > mp_saved_root) {
    /* Ascend one level, pushing a token onto list |q|
       and replacing |p| by its parent */
    if (mp_name_type (p) == mp_subscr) {
      r = mp_new_num_tok (mp, subscript (p));
      do {
        p = mp_link (p);
      } while (mp_name_type (p) != mp_attr);
    } else if (mp_name_type (p) == mp_structured_root) {
      p = mp_link (p);
      goto FOUND;
    } else {
      if (mp_name_type (p) != mp_attr)
        mp_confusion (mp, "var");
      r = mp_get_symbolic_node (mp);
      set_mp_sym_sym (r, hashloc (p)); /* the hash address */
    }
    set_mp_link (r, q);
    q = r;
  FOUND:
    p = parent ((mp_value_node) p);

  }
  /* now |link(p)| is the hash address of |p|, and
     |name_type(p)| is either |root| or |saved_root|. 
     Have to prepend a token to |q| for |show_token_list|. */
  r = mp_get_symbolic_node (mp);
  set_mp_sym_sym (r, value_sym (p));
  mp_link (r) = q;
  if (mp_name_type (p) == mp_saved_root)
    mp_print (mp, "(SAVED)");
  mp_show_token_list (mp, r, NULL, max_integer, mp->tally);
  mp_flush_token_list (mp, r);
}

@ The |interesting| function returns |true| if a given variable is not
in a capsule, or if the user wants to trace capsules.

@c
static boolean mp_interesting (MP mp, mp_node p) {
  mp_name_type_type t;        /* a |name_type| */
  if (number_positive(internal_value (mp_tracing_capsules))) {
    return true;
  } else {
    t = mp_name_type (p);
    if (t >= mp_x_part_sector && t != mp_capsule) {
      mp_node tt = value_node(mp_link(p)); 
      switch (t) {
      case mp_x_part_sector:
        t = mp_name_type (x_part (tt));
        break;
      case mp_y_part_sector:
        t = mp_name_type (y_part (tt));
        break;
      case mp_xx_part_sector:
        t = mp_name_type (xx_part (tt));
        break;
      case mp_xy_part_sector:
        t = mp_name_type (xy_part (tt));
        break;
      case mp_yx_part_sector:
        t = mp_name_type (yx_part (tt));
        break;
      case mp_yy_part_sector:
        t = mp_name_type (yy_part (tt));
        break;
      case mp_red_part_sector:
        t = mp_name_type (red_part (tt));
        break;
      case mp_green_part_sector:
        t = mp_name_type (green_part (tt));
        break;
      case mp_blue_part_sector:
        t = mp_name_type (blue_part (tt));
        break;
      case mp_cyan_part_sector:
        t = mp_name_type (cyan_part (tt));
        break;
      case mp_magenta_part_sector:
        t = mp_name_type (magenta_part (tt));
        break;
      case mp_yellow_part_sector:
        t = mp_name_type (yellow_part (tt));
        break;
      case mp_black_part_sector:
        t = mp_name_type (black_part (tt));
        break;
      case mp_grey_part_sector:
        t = mp_name_type (grey_part (tt));
        break;
      default:
        break;
      }
    }
  }
  return (t != mp_capsule);
}


@ Now here is a subroutine that converts an unstructured type into an
equivalent structured type, by inserting a |mp_structured| node that is
capable of growing. This operation is done only when |mp_name_type(p)=root|,
|subscr|, or |attr|.

The procedure returns a pointer to the new node that has taken node~|p|'s
place in the structure. Node~|p| itself does not move, nor are its
|value| or |type| fields changed in any way.

@c
static mp_node mp_new_structure (MP mp, mp_node p) {
  mp_node q, r = NULL;  /* list manipulation registers */
  mp_sym qq = NULL;
  switch (mp_name_type (p)) {
  case mp_root:
    {
      qq = value_sym (p);
      r = mp_get_value_node (mp);
      set_equiv_node (qq, r);
    }
    break;
  case mp_subscr:
    /* Link a new subscript node |r| in place of node |p| */
    {
      mp_node q_new;
      q = p;
      do {
        q = mp_link (q);
      } while (mp_name_type (q) != mp_attr);
      q = parent ((mp_value_node) q);
      r = mp->temp_head;
      set_mp_link (r, subscr_head (q));
      do {
        q_new = r;
        r = mp_link (r);
      } while (r != p);
      r = (mp_node) mp_get_subscr_node (mp);
      if (q_new == mp->temp_head) {
        set_subscr_head (q, r);
      } else {
        set_mp_link (q_new, r);
      }
      set_subscript (r, subscript (p));
    }

    break;
  case mp_attr:
    /* Link a new attribute node |r| in place of node |p| */
    /* If the attribute is |collective_subscript|, there are two pointers to
       node~|p|, so we must change both of them. */
    {
      mp_value_node rr;
      q = parent ((mp_value_node) p);
      r = attr_head (q);
      do {
        q = r;
        r = mp_link (r);
      } while (r != p);
      rr = mp_get_attr_node (mp);
      r = (mp_node) rr;
      set_mp_link (q, (mp_node) rr);
      set_hashloc (rr, hashloc (p));
      set_parent (rr, parent ((mp_value_node) p));
      if (hashloc (p) == collective_subscript) {
        q = mp->temp_head;
        set_mp_link (q, subscr_head (parent ((mp_value_node) p)));
        while (mp_link (q) != p)
          q = mp_link (q);
        if (q == mp->temp_head)
          set_subscr_head (parent ((mp_value_node) p), (mp_node) rr);
        else
          set_mp_link (q, (mp_node) rr);
      }
    }

    break;
  default:
    mp_confusion (mp, "struct");
    break;
  }
  set_mp_link (r, mp_link (p));
  set_value_sym (r, value_sym (p));
  mp_type (r) = mp_structured;
  mp_name_type (r) = mp_name_type (p);
  set_attr_head (r, p);
  mp_name_type (p) = mp_structured_root;
  {
    mp_value_node qqr = mp_get_attr_node (mp);
    set_mp_link (p, (mp_node) qqr);
    set_subscr_head (r, (mp_node) qqr);
    set_parent (qqr, r);
    mp_type (qqr) = mp_undefined;
    mp_name_type (qqr) = mp_attr;
    set_mp_link (qqr, mp->end_attr);
    set_hashloc (qqr, collective_subscript);
  }
  return r;
}

@ The |find_variable| routine is given a pointer~|t| to a nonempty token
list of suffixes; it returns a pointer to the corresponding non-symbolic
value. For example, if |t| points to token \.x followed by a numeric
token containing the value~7, |find_variable| finds where the value of
\.{x7} is stored in memory. This may seem a simple task, and it
usually is, except when \.{x7} has never been referenced before.
Indeed, \.x may never have even been subscripted before; complexities
arise with respect to updating the collective subscript information.

If a macro type is detected anywhere along path~|t|, or if the first
item on |t| isn't a |tag_token|, the value |NULL| is returned.
Otherwise |p| will be a non-NULL pointer to a node such that
|undefined<type(p)<mp_structured|.

@c
static mp_node mp_find_variable (MP mp, mp_node t) {
  mp_node p, q, r, s;   /* nodes in the ``value'' line */
  mp_sym p_sym;
  mp_node pp, qq, rr, ss;       /* nodes in the ``collective'' line */
@^inner loop@>;
  p_sym = mp_sym_sym (t);
  t = mp_link (t);
  if ((eq_type (p_sym) % mp_outer_tag) != mp_tag_token)
    return NULL;
  if (equiv_node (p_sym) == NULL)
    mp_new_root (mp, p_sym);
  p = equiv_node (p_sym);
  pp = p;
  while (t != NULL) {
    /* Make sure that both nodes |p| and |pp| are of |mp_structured| type */
    /* Although |pp| and |p| begin together, they diverge when a subscript occurs;
       |pp|~stays in the collective line while |p|~goes through actual subscript
       values. */
    if (mp_type (pp) != mp_structured) {
      if (mp_type (pp) > mp_structured)
        return NULL;
      ss = mp_new_structure (mp, pp);
      if (p == pp)
        p = ss;
      pp = ss;
    }                             /* now |type(pp)=mp_structured| */
    if (mp_type (p) != mp_structured) {   /* it cannot be |>mp_structured| */
      p = mp_new_structure (mp, p);       /* now |type(p)=mp_structured| */
    }

    if (mp_type (t) != mp_symbol_node) {
      /* Descend one level for the subscript |value(t)| */
      /* We want this part of the program to be reasonably fast, in case there are
        lots of subscripts at the same level of the data structure. Therefore
        we store an ``infinite'' value in the word that appears at the end of the
        subscript list, even though that word isn't part of a subscript node. */
      mp_number nn, save_subscript;      /* temporary storage */
      new_number (nn);
      new_number (save_subscript);
      number_clone (nn, value_number (t));
      pp = mp_link (attr_head (pp)); /* now |hashloc(pp)=collective_subscript| */
      q = mp_link (attr_head (p));
      number_clone (save_subscript, subscript (q));
      set_number_to_inf(subscript (q));
      s = mp->temp_head;
      set_mp_link (s, subscr_head (p));
      do {
        r = s;
        s = mp_link (s);
      } while (number_greater (nn, subscript (s)));
      if (number_equal(nn, subscript (s))) {
        p = s;
      } else {
        mp_value_node p1 = mp_get_subscr_node (mp);
        if (r == mp->temp_head)
          set_subscr_head (p, (mp_node) p1);
        else
          set_mp_link (r, (mp_node) p1);
        set_mp_link (p1, s);
        number_clone (subscript (p1), nn);
        mp_name_type (p1) = mp_subscr;
        mp_type (p1) = mp_undefined;
        p = (mp_node) p1;
      }
      number_clone (subscript (q), save_subscript);
      free_number (save_subscript);
      free_number (nn);
    } else {
      /* Descend one level for the attribute |mp_sym_info(t)| */
      mp_sym nn1 = mp_sym_sym (t);
      ss = attr_head (pp);
      do {
        rr = ss;
        ss = mp_link (ss);
      } while (nn1 > hashloc (ss));
      if (nn1 < hashloc (ss)) {
        qq = (mp_node) mp_get_attr_node (mp);
        set_mp_link (rr, qq);
        set_mp_link (qq, ss);
        set_hashloc (qq, nn1);
        mp_name_type (qq) = mp_attr;
        mp_type (qq) = mp_undefined;
        set_parent ((mp_value_node) qq, pp);
        ss = qq;
      }
      if (p == pp) {
        p = ss;
        pp = ss;
      } else {
        pp = ss;
        s = attr_head (p);
        do {
          r = s;
          s = mp_link (s);
        } while (nn1 > hashloc (s));
        if (nn1 == hashloc (s)) {
          p = s;
        } else {
          q = (mp_node) mp_get_attr_node (mp);
          set_mp_link (r, q);
          set_mp_link (q, s);
          set_hashloc (q, nn1);
          mp_name_type (q) = mp_attr;
          mp_type (q) = mp_undefined;
          set_parent ((mp_value_node) q, p);
          p = q;
        }
      }
    }
    t = mp_link (t);
  }
  if (mp_type (pp) >= mp_structured) {
    if (mp_type (pp) == mp_structured)
      pp = attr_head (pp);
    else
      return NULL;
  }
  if (mp_type (p) == mp_structured)
    p = attr_head (p);
  if (mp_type (p) == mp_undefined) {
    if (mp_type (pp) == mp_undefined) {
      mp_type (pp) = mp_numeric_type;
      set_value_number (pp, zero_t);
    }
    mp_type (p) = mp_type (pp);
    set_value_number (p, zero_t);
  }
  return p;
}


@ Variables lose their former values when they appear in a type declaration,
or when they are defined to be macros or \&{let} equal to something else.
A subroutine will be defined later that recycles the storage associated
with any particular |type| or |value|; our goal now is to study a higher
level process called |flush_variable|, which selectively frees parts of a
variable structure.

This routine has some complexity because of examples such as
`\hbox{\tt numeric x[]a[]b}'
which recycles all variables of the form \.{x[i]a[j]b} (and no others), while
`\hbox{\tt vardef x[]a[]=...}'
discards all variables of the form \.{x[i]a[j]} followed by an arbitrary
suffix, except for the collective node \.{x[]a[]} itself. The obvious way
to handle such examples is to use recursion; so that's what we~do.
@^recursion@>

Parameter |p| points to the root information of the variable;
parameter |t| points to a list of symbolic nodes that represent
suffixes, with |info=collective_subscript| for subscripts.

@<Declarations@>=
void mp_flush_cur_exp (MP mp, mp_value v);

@ @c
static void mp_flush_variable (MP mp, mp_node p, mp_node t,
                               boolean discard_suffixes) {
  mp_node q, r = NULL; /* list manipulation */
  mp_sym n;     /* attribute to match */
  while (t != NULL) {
    if (mp_type (p) != mp_structured) {
      return;
    }
    n = mp_sym_sym (t);
    t = mp_link (t);
    if (n == collective_subscript) {
      q = subscr_head (p);
      while (mp_name_type (q) == mp_subscr) {
        mp_flush_variable (mp, q, t, discard_suffixes);
        if (t == NULL) {
          if (mp_type (q) == mp_structured) {
            r = q;
          } else {
            if (r==NULL) 
   	      set_subscr_head (p, mp_link (q));
            else
              set_mp_link (r, mp_link (q));
            mp_free_value_node (mp, q);
          }
        } else {
          r = q;
        }
        q = (r==NULL ? subscr_head (p) : mp_link (r));
      }
    }
    p = attr_head (p);
    do {
      p = mp_link (p);
    } while (hashloc (p) < n);
    if (hashloc (p) != n) {
      return;
    }
  }
  if (discard_suffixes) {
    mp_flush_below_variable (mp, p);
  } else {
    if (mp_type (p) == mp_structured) {
      p = attr_head (p); 
    }
    mp_recycle_value (mp, p);
  }
}


@ The next procedure is simpler; it wipes out everything but |p| itself,
which becomes undefined.

@<Declarations@>=
static void mp_flush_below_variable (MP mp, mp_node p);

@ @c
void mp_flush_below_variable (MP mp, mp_node p) {
  mp_node q, r; /* list manipulation registers */
  FUNCTION_TRACE2 ("mp_flush_below_variable(%p)\n", p);
  if (mp_type (p) != mp_structured) {
    mp_recycle_value (mp, p);   /* this sets |type(p)=undefined| */
  } else {
    q = subscr_head (p);
    while (mp_name_type (q) == mp_subscr) {
      mp_flush_below_variable (mp, q);
      r = q;
      q = mp_link (q);
      mp_free_value_node (mp, r);
    }
    r = attr_head (p);
    q = mp_link (r);
    mp_recycle_value (mp, r);
    mp_free_value_node (mp, r);
    do {
      mp_flush_below_variable (mp, q);
      r = q;
      q = mp_link (q);
      mp_free_value_node (mp, r);
    } while (q != mp->end_attr);
    mp_type (p) = mp_undefined;
  }
}


@ Just before assigning a new value to a variable, we will recycle the
old value and make the old value undefined. The |und_type| routine
determines what type of undefined value should be given, based on
the current type before recycling.

@c
static quarterword mp_und_type (MP mp, mp_node p) {
  (void) mp;
  switch (mp_type (p)) {
  case mp_vacuous:
    return mp_undefined;
  case mp_boolean_type:
  case mp_unknown_boolean:
    return mp_unknown_boolean;
  case mp_string_type:
  case mp_unknown_string:
    return mp_unknown_string;
  case mp_pen_type:
  case mp_unknown_pen:
    return mp_unknown_pen;
  case mp_path_type:
  case mp_unknown_path:
    return mp_unknown_path;
  case mp_picture_type:
  case mp_unknown_picture:
    return mp_unknown_picture;
  case mp_transform_type:
  case mp_color_type:
  case mp_cmykcolor_type:
  case mp_pair_type:
  case mp_numeric_type:
    return mp_type (p);
  case mp_known:
  case mp_dependent:
  case mp_proto_dependent:
  case mp_independent:
    return mp_numeric_type;
  default:                     /* there are no other valid cases, but please the compiler */
    return 0;
  }
  return 0;
}


@ The |clear_symbol| routine is used when we want to redefine the equivalent
of a symbolic token. It must remove any variable structure or macro
definition that is currently attached to that symbol. If the |saving|
parameter is true, a subsidiary structure is saved instead of destroyed.

@c
static void mp_clear_symbol (MP mp, mp_sym p, boolean saving) {
  mp_node q;    /* |equiv(p)| */
  FUNCTION_TRACE3 ("mp_clear_symbol(%p,%d)\n", p, saving);
  q = equiv_node (p);
  switch (eq_type (p) % mp_outer_tag) {
  case mp_defined_macro:
  case mp_secondary_primary_macro:
  case mp_tertiary_secondary_macro:
  case mp_expression_tertiary_macro:
    if (!saving)
      mp_delete_mac_ref (mp, q);
    break;
  case mp_tag_token:
    if (q != NULL) {
      if (saving) {
        mp_name_type (q) = mp_saved_root;
      } else {
        mp_flush_below_variable (mp, q);
        mp_free_value_node (mp, q);
      }
    }
    break;
  default:
    break;
  }
  set_equiv (p, mp->frozen_undefined->v.data.indep.serial);
  set_eq_type (p, mp->frozen_undefined->type);
}


@* Saving and restoring equivalents.
The nested structure given by \&{begingroup} and \&{endgroup}
allows |eqtb| entries to be saved and restored, so that temporary changes
can be made without difficulty.  When the user requests a current value to
be saved, \MP\ puts that value into its ``save stack.'' An appearance of
\&{endgroup} ultimately causes the old values to be removed from the save
stack and put back in their former places.

The save stack is a linked list containing three kinds of entries,
distinguished by their |type| fields. If |p| points to a saved item,
then

\smallskip\hang
|p->type=0| stands for a group boundary; each \&{begingroup} contributes
such an item to the save stack and each \&{endgroup} cuts back the stack
until the most recent such entry has been removed.

\smallskip\hang
|p->type=mp_normal_sym| means that |p->value| holds the former
contents of |eqtb[q]| (saved in the |knot| field of the value, which
is otherwise unused for variables). Such save stack entries are generated by \&{save}
commands.

\smallskip\hang
|p->type=mp_internal_sym| means that |p->value| is a |mp_internal|
to be restored to internal parameter number~|q| (saved in the |serial| field of the value, which
is otherwise unused for internals). Such entries are generated by \&{interim} commands.

\smallskip\noindent
The global variable |save_ptr| points to the top item on the save stack.

@<Types...@>=
typedef struct mp_save_data {
  quarterword type;
  mp_internal value;
  struct mp_save_data *link;
} mp_save_data;

@ @<Glob...@>=
mp_save_data *save_ptr; /* the most recently saved item */

@ @<Set init...@>=
mp->save_ptr = NULL;

@ Saving a boundary item
@c
static void mp_save_boundary (MP mp) {
  mp_save_data *p;      /* temporary register */
  FUNCTION_TRACE1 ("mp_save_boundary ()\n");
  p = xmalloc (1, sizeof (mp_save_data));
  p->type = 0;
  p->link = mp->save_ptr;
  mp->save_ptr = p;
}


@ The |save_variable| routine is given a hash address |q|; it salts this
address in the save stack, together with its current equivalent,
then makes token~|q| behave as though it were brand new.

Nothing is stacked when |save_ptr=NULL|, however; there's no way to remove
things from the stack when the program is not inside a group, so there's
no point in wasting the space.

@c
static void mp_save_variable (MP mp, mp_sym q) {
  mp_save_data *p;      /* temporary register */
  FUNCTION_TRACE2 ("mp_save_variable (%p)\n", q);
  if (mp->save_ptr != NULL) {
    p = xmalloc (1, sizeof (mp_save_data));
    p->type = mp_normal_sym;
    p->link = mp->save_ptr;
    p->value.v.data.indep.scale = eq_type (q);
    p->value.v.data.indep.serial = equiv(q);
    p->value.v.data.node = equiv_node(q);
    p->value.v.data.p = (mp_knot)q;
    mp->save_ptr = p;
  }
  mp_clear_symbol (mp, q, (mp->save_ptr != NULL));
}
static void mp_unsave_variable (MP mp) {
  mp_sym q = (mp_sym)mp->save_ptr->value.v.data.p;
  if (number_positive(internal_value (mp_tracing_restores))) {
    mp_begin_diagnostic (mp);
    mp_print_nl (mp, "{restoring ");
    mp_print_text (q);
    mp_print_char (mp, xord ('}'));
    mp_end_diagnostic (mp, false);
  }
  mp_clear_symbol (mp, q, false);
  set_eq_type(q, mp->save_ptr->value.v.data.indep.scale);
  set_equiv  (q,mp->save_ptr->value.v.data.indep.serial);
  q->v.data.node = mp->save_ptr->value.v.data.node; 
  if (eq_type (q) % mp_outer_tag == mp_tag_token) {
    mp_node pp = q->v.data.node;
    if (pp != NULL)
      mp_name_type (pp) = mp_root;
  }
}

@ Similarly, |save_internal| is given the location |q| of an internal
quantity like |mp_tracing_pens|. It creates a save stack entry of the
third kind.

@c
static void mp_save_internal (MP mp, halfword q) {
  mp_save_data *p;      /* new item for the save stack */
  FUNCTION_TRACE2 ("mp_save_internal (%d)\n", q);
  if (mp->save_ptr != NULL) {
    p = xmalloc (1, sizeof (mp_save_data));
    p->type = mp_internal_sym;
    p->link = mp->save_ptr;
    p->value = mp->internal[q];
    p->value.v.data.indep.serial = q;
    new_number(p->value.v.data.n);
    number_clone(p->value.v.data.n, mp->internal[q].v.data.n);
    mp->save_ptr = p;
  }
}

static void mp_unsave_internal (MP mp) {
  halfword q = mp->save_ptr->value.v.data.indep.serial;
  mp_internal saved = mp->save_ptr->value;
  if (number_positive(internal_value (mp_tracing_restores))) {
    mp_begin_diagnostic (mp);
    mp_print_nl (mp, "{restoring ");
    mp_print (mp, internal_name (q));
    mp_print_char (mp, xord ('='));
    if (internal_type (q) == mp_known) {
      print_number (saved.v.data.n);
    } else if (internal_type (q) == mp_string_type) {
      char *s = mp_str (mp, saved.v.data.str);
      mp_print (mp, s);
    } else {
      mp_confusion (mp, "internal_restore");
    }
    mp_print_char (mp, xord ('}'));
    mp_end_diagnostic (mp, false);
  }
  free_number (mp->internal[q].v.data.n);
  mp->internal[q] = saved;
}

@ At the end of a group, the |unsave| routine restores all of the saved
equivalents in reverse order. This routine will be called only when there
is at least one boundary item on the save stack.

@c
static void mp_unsave (MP mp) {
  mp_save_data *p;      /* saved item */
  FUNCTION_TRACE1 ("mp_unsave ()\n");
  while (mp->save_ptr->type != 0) {
    if (mp->save_ptr->type == mp_internal_sym) {
      mp_unsave_internal(mp);
    } else {
      mp_unsave_variable(mp);
    }
    p = mp->save_ptr->link;
    xfree (mp->save_ptr);
    mp->save_ptr = p;
  }
  p = mp->save_ptr->link;
  xfree (mp->save_ptr);
  mp->save_ptr = p;
}


@* Data structures for paths.
When a \MP\ user specifies a path, \MP\ will create a list of knots
and control points for the associated cubic spline curves. If the
knots are $z_0$, $z_1$, \dots, $z_n$, there are control points
$z_k^+$ and $z_{k+1}^-$ such that the cubic splines between knots
$z_k$ and $z_{k+1}$ are defined by B\'ezier's formula
@:Bezier}{B\'ezier, Pierre Etienne@>
$$\eqalign{z(t)&=B(z_k,z_k^+,z_{k+1}^-,z_{k+1};t)\cr
&=(1-t)^3z_k+3(1-t)^2tz_k^++3(1-t)t^2z_{k+1}^-+t^3z_{k+1}\cr}$$
for |0<=t<=1|.

There is a 8-word node for each knot $z_k$, containing one word of
control information and six words for the |x| and |y| coordinates of
$z_k^-$ and $z_k$ and~$z_k^+$. The control information appears in the
|mp_left_type| and |mp_right_type| fields, which each occupy a quarter of
the first word in the node; they specify properties of the curve as it
enters and leaves the knot. There's also a halfword |link| field,
which points to the following knot, and a final supplementary word (of
which only a quarter is used).

If the path is a closed contour, knots 0 and |n| are identical;
i.e., the |link| in knot |n-1| points to knot~0. But if the path
is not closed, the |mp_left_type| of knot~0 and the |mp_right_type| of knot~|n|
are equal to |endpoint|. In the latter case the |link| in knot~|n| points
to knot~0, and the control points $z_0^-$ and $z_n^+$ are not used.

@d mp_next_knot(A)   (A)->next /* the next knot in this list */
@d mp_left_type(A)   (A)->data.types.left_type /* characterizes the path entering this knot */
@d mp_right_type(A)  (A)->data.types.right_type /* characterizes the path leaving this knot */
@d mp_prev_knot(A)   (A)->data.prev /* the previous knot in this list (only for pens) */
@d mp_knot_info(A)   (A)->data.info /* temporary info, used during splitting */

@<Exported types...@>=
typedef struct mp_knot_data *mp_knot;
typedef struct mp_knot_data {
  mp_number x_coord; /* the |x| coordinate of this knot */ 
  mp_number y_coord; /* the |y| coordinate of this knot */
  mp_number left_x; /* the |x| coordinate of previous control point */
  mp_number left_y; /* the |y| coordinate of previous control point */
  mp_number right_x; /* the |x| coordinate of next control point */
  mp_number right_y; /* the |y| coordinate of next control point */
  mp_knot next;
  union {
    struct {
      unsigned short left_type;
      unsigned short right_type;
    } types;
    mp_knot prev;
    signed int info;
  } data;
  unsigned char originator;
} mp_knot_data;


@ 
@d mp_gr_next_knot(A)   (A)->next /* the next knot in this list */

@<Exported types...@>=
typedef struct mp_gr_knot_data *mp_gr_knot;
typedef struct mp_gr_knot_data {
  double x_coord;
  double y_coord;
  double left_x;
  double left_y;
  double right_x;
  double right_y;
  mp_gr_knot next;
  union {
    struct {
      unsigned short left_type;
      unsigned short right_type;
    } types;
    mp_gr_knot prev;
    signed int info;
  } data;
  unsigned char originator;
} mp_gr_knot_data;


@ @<MPlib header stuff@>=
enum mp_knot_type {
  mp_endpoint = 0,      /* |mp_left_type| at path beginning and |mp_right_type| at path end */
  mp_explicit,                  /* |mp_left_type| or |mp_right_type| when control points are known */
  mp_given,                     /* |mp_left_type| or |mp_right_type| when a direction is given */
  mp_curl,                      /* |mp_left_type| or |mp_right_type| when a curl is desired */
  mp_open,                      /* |mp_left_type| or |mp_right_type| when \MP\ should choose the direction */
  mp_end_cycle
};

@ Before the B\'ezier control points have been calculated, the memory
space they will ultimately occupy is taken up by information that can be
used to compute them. There are four cases:

\yskip
\textindent{$\bullet$} If |mp_right_type=mp_open|, the curve should leave
the knot in the same direction it entered; \MP\ will figure out a
suitable direction.

\yskip
\textindent{$\bullet$} If |mp_right_type=mp_curl|, the curve should leave the
knot in a direction depending on the angle at which it enters the next
knot and on the curl parameter stored in |right_curl|.

\yskip
\textindent{$\bullet$} If |mp_right_type=mp_given|, the curve should leave the
knot in a nonzero direction stored as an |angle| in |right_given|.

\yskip
\textindent{$\bullet$} If |mp_right_type=mp_explicit|, the B\'ezier control
point for leaving this knot has already been computed; it is in the
|mp_right_x| and |mp_right_y| fields.

\yskip\noindent
The rules for |mp_left_type| are similar, but they refer to the curve entering
the knot, and to \\{left} fields instead of \\{right} fields.

Non-|explicit| control points will be chosen based on ``tension'' parameters
in the |left_tension| and |right_tension| fields. The
`\&{atleast}' option is represented by negative tension values.
@:at_least_}{\&{atleast} primitive@>

For example, the \MP\ path specification
$$\.{z0..z1..tension atleast 1..\{curl 2\}z2..z3\{-1,-2\}..tension
  3 and 4..p},$$
where \.p is the path `\.{z4..controls z45 and z54..z5}', will be represented
by the six knots
\def\lodash{\hbox to 1.1em{\thinspace\hrulefill\thinspace}}
$$\vbox{\halign{#\hfil&&\qquad#\hfil\cr
|mp_left_type|&\\{left} info&|x_coord,y_coord|&|mp_right_type|&\\{right} info\cr
\noalign{\yskip}
|endpoint|&\lodash$,\,$\lodash&$x_0,y_0$&|curl|&$1.0,1.0$\cr
|open|&\lodash$,1.0$&$x_1,y_1$&|open|&\lodash$,-1.0$\cr
|curl|&$2.0,-1.0$&$x_2,y_2$&|curl|&$2.0,1.0$\cr
|given|&$d,1.0$&$x_3,y_3$&|given|&$d,3.0$\cr
|open|&\lodash$,4.0$&$x_4,y_4$&|explicit|&$x_{45},y_{45}$\cr
|explicit|&$x_{54},y_{54}$&$x_5,y_5$&|endpoint|&\lodash$,\,$\lodash\cr}}$$
Here |d| is the |angle| obtained by calling |n_arg(-unity,-two)|.
Of course, this example is more complicated than anything a normal user
would ever write.

These types must satisfy certain restrictions because of the form of \MP's
path syntax:
(i)~|open| type never appears in the same node together with |endpoint|,
|given|, or |curl|.
(ii)~The |mp_right_type| of a node is |explicit| if and only if the
|mp_left_type| of the following node is |explicit|.
(iii)~|endpoint| types occur only at the ends, as mentioned above.

@d left_curl left_x /* curl information when entering this knot */
@d left_given left_x /* given direction when entering this knot */
@d left_tension left_y /* tension information when entering this knot */
@d right_curl right_x /* curl information when leaving this knot */
@d right_given right_x /* given direction when leaving this knot */
@d right_tension right_y /* tension information when leaving this knot */

@ Knots can be user-supplied, or they can be created by program code,
like the |split_cubic| function, or |copy_path|. The distinction is
needed for the cleanup routine that runs after |split_cubic|, because
it should only delete knots it has previously inserted, and never
anything that was user-supplied. In order to be able to differentiate
one knot from another, we will set |originator(p):=mp_metapost_user| when
it appeared in the actual metapost program, and
|originator(p):=mp_program_code| in all other cases.

@d mp_originator(A)   (A)->originator /* the creator of this knot */

@<Exported types@>=
enum mp_knot_originator {
  mp_program_code = 0,  /* not created by a user */
  mp_metapost_user              /* created by a user */
};

@ Here is a routine that prints a given knot list
in symbolic form. It illustrates the conventions discussed above,
and checks for anomalies that might arise while \MP\ is being debugged.

@<Declarations@>=
static void mp_pr_path (MP mp, mp_knot h);

@ @c
void mp_pr_path (MP mp, mp_knot h) {
  mp_knot p, q; /* for list traversal */
  p = h;
  do {
    q = mp_next_knot (p);
    if ((p == NULL) || (q == NULL)) {
      mp_print_nl (mp, "???");
      return;                   /* this won't happen */
@.???@>
    }
    @<Print information for adjacent knots |p| and |q|@>;
  DONE1:
    p = q;
    if (p && ((p != h) || (mp_left_type (h) != mp_endpoint))) {
      @<Print two dots, followed by |given| or |curl| if present@>;
    }
  } while (p != h);
  if (mp_left_type (h) != mp_endpoint)
    mp_print (mp, "cycle");
}


@ @<Print information for adjacent knots...@>=
mp_print_two (mp, p->x_coord, p->y_coord);
switch (mp_right_type (p)) {
case mp_endpoint:
  if (mp_left_type (p) == mp_open)
    mp_print (mp, "{open?}");   /* can't happen */
@.open?@>;
  if ((mp_left_type (q) != mp_endpoint) || (q != h))
    q = NULL;                   /* force an error */
  goto DONE1;
  break;
case mp_explicit:
  @<Print control points between |p| and |q|, then |goto done1|@>;
  break;
case mp_open:
  @<Print information for a curve that begins |open|@>;
  break;
case mp_curl:
case mp_given:
  @<Print information for a curve that begins |curl| or |given|@>;
  break;
default:
  mp_print (mp, "???");         /* can't happen */
@.???@>;
  break;
}
if (mp_left_type (q) <= mp_explicit) {
  mp_print (mp, "..control?");  /* can't happen */
@.control?@>
} else if ((!number_equal(p->right_tension, unity_t)) || (!number_equal(q->left_tension, unity_t))) {
  @<Print tension between |p| and |q|@>;
}

@ Since |n_sin_cos| produces |fraction| results, which we will print as if they
were |scaled|, the magnitude of a |given| direction vector will be~4096.

@<Print two dots...@>=
{
  mp_number n_sin, n_cos;
  new_fraction (n_sin);
  new_fraction (n_cos);
  mp_print_nl (mp, " ..");
  if (mp_left_type (p) == mp_given) {
    n_sin_cos (p->left_given, n_cos, n_sin);
    mp_print_char (mp, xord ('{'));
    print_number (n_cos);
    mp_print_char (mp, xord (','));
    print_number (n_sin);
    mp_print_char (mp, xord ('}'));
  } else if (mp_left_type (p) == mp_curl) {
    mp_print (mp, "{curl ");
    print_number (p->left_curl);
    mp_print_char (mp, xord ('}'));
  }
  free_number (n_sin);
  free_number (n_cos);
}


@ @<Print tension between |p| and |q|@>=
{
  mp_number v1;
  new_number (v1);
  mp_print (mp, "..tension ");
  if (number_negative(p->right_tension))
    mp_print (mp, "atleast");
  number_clone (v1, p->right_tension);
  number_abs (v1);
  print_number (v1);
  if (!number_equal(p->right_tension, q->left_tension)) {
    mp_print (mp, " and ");
    if (number_negative(q->left_tension))
      mp_print (mp, "atleast");
    number_clone (v1, p->left_tension);
    number_abs (v1);
    print_number (v1);
  }
  free_number (v1);
}


@ @<Print control points between |p| and |q|, then |goto done1|@>=
{
  mp_print (mp, "..controls ");
  mp_print_two (mp, p->right_x, p->right_y);
  mp_print (mp, " and ");
  if (mp_left_type (q) != mp_explicit) {
    mp_print (mp, "??");        /* can't happen */
@.??@>
  } else {
    mp_print_two (mp, q->left_x, q->left_y);
  }
  goto DONE1;
}


@ @<Print information for a curve that begins |open|@>=
if ((mp_left_type (p) != mp_explicit) && (mp_left_type (p) != mp_open)) {
  mp_print (mp, "{open?}");     /* can't happen */
@.open?@>
}

@ A curl of 1 is shown explicitly, so that the user sees clearly that
\MP's default curl is present.

@<Print information for a curve that begins |curl|...@>=
{
  if (mp_left_type (p) == mp_open)
    mp_print (mp, "??");        /* can't happen */
@.??@>;
  if (mp_right_type (p) == mp_curl) {
    mp_print (mp, "{curl ");
    print_number (p->right_curl);
  } else {
    mp_number n_sin, n_cos;
    new_fraction (n_sin);
    new_fraction (n_cos);
    n_sin_cos (p->right_given, n_cos, n_sin);
    mp_print_char (mp, xord ('{'));
    print_number (n_cos);
    mp_print_char (mp, xord (','));
    print_number (n_sin);
    free_number (n_sin);
    free_number (n_cos);
  }
  mp_print_char (mp, xord ('}'));
}


@ It is convenient to have another version of |pr_path| that prints the path
as a diagnostic message.

@<Declarations@>=
static void mp_print_path (MP mp, mp_knot h, const char *s, boolean nuline);

@ @c
void mp_print_path (MP mp, mp_knot h, const char *s, boolean nuline) {
  mp_print_diagnostic (mp, "Path", s, nuline);
  mp_print_ln (mp);
@.Path at line...@>;
  mp_pr_path (mp, h);
  mp_end_diagnostic (mp, true);
}


@ @<Declarations@>=
static mp_knot mp_new_knot (MP mp);

@ @c
static mp_knot mp_new_knot (MP mp) {
  mp_knot q;
  if (mp->knot_nodes) {
    q = mp->knot_nodes;
    mp->knot_nodes = q->next;
    mp->num_knot_nodes--;
  } else {
    q = mp_xmalloc (mp, 1, sizeof (struct mp_knot_data));
  }
  memset(q,0,sizeof (struct mp_knot_data));
  new_number(q->x_coord);
  new_number(q->y_coord);
  new_number(q->left_x);
  new_number(q->left_y);
  new_number(q->right_x);
  new_number(q->right_y);
  return q;
}


@ @<Declarations@>=
static mp_gr_knot mp_gr_new_knot (MP mp);

@ @c
static mp_gr_knot mp_gr_new_knot (MP mp) {
  mp_gr_knot q = mp_xmalloc (mp, 1, sizeof (struct mp_gr_knot_data));
  return q;
}


@ If we want to duplicate a knot node, we can say |copy_knot|:

@c
static mp_knot mp_copy_knot (MP mp, mp_knot p) {
  mp_knot q;
  if (mp->knot_nodes) {
    q  = mp->knot_nodes;
    mp->knot_nodes = q->next;
    mp->num_knot_nodes--;
  } else {
    q = mp_xmalloc (mp, 1, sizeof (struct mp_knot_data));
  }
  memcpy (q, p, sizeof (struct mp_knot_data));
  if (mp->math_mode > mp_math_double_mode) {
    new_number(q->x_coord);
    new_number(q->y_coord);
    new_number(q->left_x);
    new_number(q->left_y);
    new_number(q->right_x);
    new_number(q->right_y);
    number_clone(q->x_coord, p->x_coord);
    number_clone(q->y_coord, p->y_coord);
    number_clone(q->left_x, p->left_x);
    number_clone(q->left_y, p->left_y);
    number_clone(q->right_x, p->right_x);
    number_clone(q->right_y, p->right_y);
  }
  mp_next_knot (q) = NULL;
  return q;
}

@ If we want to export a knot node, we can say |export_knot|:

@c
static mp_gr_knot mp_export_knot (MP mp, mp_knot p) {
  mp_gr_knot q;    /* the copy */
  q = mp_gr_new_knot (mp);
  q->x_coord = number_to_double(p->x_coord);
  q->y_coord = number_to_double(p->y_coord);
  q->left_x  = number_to_double(p->left_x);
  q->left_y  = number_to_double(p->left_y);
  q->right_x = number_to_double(p->right_x);
  q->right_y = number_to_double(p->right_y);
  q->data.types.left_type = mp_left_type(p);
  q->data.types.right_type = mp_left_type(p);
  q->data.info = mp_knot_info(p);
  mp_gr_next_knot (q) = NULL;
  return q;
}


@ The |copy_path| routine makes a clone of a given path.

@c
static mp_knot mp_copy_path (MP mp, mp_knot p) {
  mp_knot q, pp, qq;    /* for list manipulation */
  if (p == NULL)
    return NULL;
  q = mp_copy_knot (mp, p);
  qq = q;
  pp = mp_next_knot (p);
  while (pp != p) {
    mp_next_knot (qq) = mp_copy_knot (mp, pp);
    qq = mp_next_knot (qq);
    pp = mp_next_knot (pp);
  }
  mp_next_knot (qq) = q;
  return q;
}

@ The |export_path| routine makes a clone of a given path
and converts the |value|s therein to |double|s.

@c
static mp_gr_knot mp_export_path (MP mp, mp_knot p) {
  mp_knot pp;    /* for list manipulation */
  mp_gr_knot q, qq;
  if (p == NULL)
    return NULL;
  q = mp_export_knot (mp, p);
  qq = q;
  pp = mp_next_knot (p);
  while (pp != p) {
    mp_gr_next_knot (qq) = mp_export_knot (mp, pp);
    qq = mp_gr_next_knot (qq);
    pp = mp_next_knot (pp);
  }
  mp_gr_next_knot (qq) = q;
  return q;
}

@ If we want to import a knot node, we can say |import_knot|:

@c
static mp_knot mp_import_knot (MP mp, mp_gr_knot p) {
  mp_knot q;    /* the copy */
  q = mp_new_knot (mp);
  set_number_from_double(q->x_coord, p->x_coord);
  set_number_from_double(q->y_coord, p->y_coord);
  set_number_from_double(q->left_x, p->left_x);
  set_number_from_double(q->left_y, p->left_y);
  set_number_from_double(q->right_x, p->right_x);
  set_number_from_double(q->right_y, p->right_y);
  mp_left_type(q) = p->data.types.left_type;
  mp_left_type(q) = p->data.types.right_type;
  mp_knot_info(q) = p->data.info;
  mp_next_knot (q) = NULL;
  return q;
}


@ The |import_path| routine makes a clone of a given path
and converts the |value|s therein to |scaled|s.

@c
static mp_knot mp_import_path (MP mp, mp_gr_knot p) {
  mp_gr_knot pp;    /* for list manipulation */
  mp_knot q, qq;
  if (p == NULL)
    return NULL;
  q = mp_import_knot (mp, p);
  qq = q;
  pp = mp_gr_next_knot (p);
  while (pp != p) {
    mp_next_knot (qq) = mp_import_knot (mp, pp);
    qq = mp_next_knot (qq);
    pp = mp_gr_next_knot (pp);
  }
  mp_next_knot (qq) = q;
  return q;
}


@ Just before |ship_out|, knot lists are exported for printing.

@ The |export_knot_list| routine therefore also makes a clone 
of a given path.

@c
static mp_gr_knot mp_export_knot_list (MP mp, mp_knot p) {
  mp_gr_knot q;    /* the exported copy */
  if (p == NULL)
    return NULL;
  q = mp_export_path (mp, p);
  return q;
}
static mp_knot mp_import_knot_list (MP mp, mp_gr_knot q) {
  mp_knot p;    /* the imported copy */
  if (q == NULL)
    return NULL;
  p = mp_import_path (mp, q);
  return p;
}

@ Similarly, there's a way to copy the {\sl reverse\/} of a path. This procedure
returns a pointer to the first node of the copy, if the path is a cycle,
but to the final node of a non-cyclic copy. The global
variable |path_tail| will point to the final node of the original path;
this trick makes it easier to implement `\&{doublepath}'.

All node types are assumed to be |endpoint| or |explicit| only.

@c
static mp_knot mp_htap_ypoc (MP mp, mp_knot p) {
  mp_knot q, pp, qq, rr;        /* for list manipulation */
  q = mp_new_knot (mp);         /* this will correspond to |p| */
  qq = q;
  pp = p;
  while (1) {
    mp_right_type (qq) = mp_left_type (pp);
    mp_left_type (qq) = mp_right_type (pp);
    number_clone (qq->x_coord, pp->x_coord);
    number_clone (qq->y_coord, pp->y_coord);
    number_clone (qq->right_x, pp->left_x);
    number_clone (qq->right_y, pp->left_y);
    number_clone (qq->left_x, pp->right_x);
    number_clone (qq->left_y, pp->right_y);
    mp_originator (qq) = mp_originator (pp);
    if (mp_next_knot (pp) == p) {
      mp_next_knot (q) = qq;
      mp->path_tail = pp;
      return q;
    }
    rr = mp_new_knot (mp);
    mp_next_knot (rr) = qq;
    qq = rr;
    pp = mp_next_knot (pp);
  }
}


@ @<Glob...@>=
mp_knot path_tail;      /* the node that links to the beginning of a path */

@ When a cyclic list of knot nodes is no longer needed, it can be recycled by
calling the following subroutine.

@<Declarations@>=
static void mp_toss_knot_list (MP mp, mp_knot p);
static void mp_toss_knot (MP mp, mp_knot p);
static void mp_free_knot (MP mp, mp_knot p);

@ @c
void mp_free_knot  (MP mp, mp_knot q) {
  free_number (q->x_coord); 
  free_number (q->y_coord); 
  free_number (q->left_x); 
  free_number (q->left_y); 
  free_number (q->right_x); 
  free_number (q->right_y);
  mp_xfree (q);
}
void mp_toss_knot (MP mp, mp_knot q) {
  if (mp->num_knot_nodes < max_num_knot_nodes) {
    q->next = mp->knot_nodes;
    mp->knot_nodes = q;
    mp->num_knot_nodes++;
    return;
  }
  if (mp->math_mode > mp_math_double_mode) {
    mp_free_knot(mp,q);
  } else {
    mp_xfree (q);
  }
}
void mp_toss_knot_list (MP mp, mp_knot p) {
  mp_knot q;    /* the node being freed */
  mp_knot r;    /* the next node */
  if (p == NULL)
    return;
  q = p;
  if (mp->math_mode > mp_math_double_mode) {
    do {
      r = mp_next_knot (q);
      mp_toss_knot(mp, q);
      q = r;
    } while (q != p);
  } else {
    do {
      r = mp_next_knot (q);
      if (mp->num_knot_nodes < max_num_knot_nodes) {
        q->next = mp->knot_nodes;
	mp->knot_nodes = q;
	mp->num_knot_nodes++;
      } else {
        mp_xfree (q);
      }
      q = r;
    } while (q != p);
  }
}


@* Choosing control points.
Now we must actually delve into one of \MP's more difficult routines,
the |make_choices| procedure that chooses angles and control points for
the splines of a curve when the user has not specified them explicitly.
The parameter to |make_choices| points to a list of knots and
path information, as described above.

A path decomposes into independent segments at ``breakpoint'' knots,
which are knots whose left and right angles are both prespecified in
some way (i.e., their |mp_left_type| and |mp_right_type| aren't both open).

@c
void mp_make_choices (MP mp, mp_knot knots) {
  mp_knot h;    /* the first breakpoint */
  mp_knot p, q; /* consecutive breakpoints being processed */
  @<Other local variables for |make_choices|@>;
  FUNCTION_TRACE1 ("make_choices()\n");
  check_arith();                  /* make sure that |arith_error=false| */
  if (number_positive(internal_value (mp_tracing_choices)))
    mp_print_path (mp, knots, ", before choices", true);
  @<If consecutive knots are equal, join them explicitly@>;
  @<Find the first breakpoint, |h|, on the path;
    insert an artificial breakpoint if the path is an unbroken cycle@>;
  p = h;
  do {
    @<Fill in the control points between |p| and the next breakpoint,
      then advance |p| to that breakpoint@>;
  } while (p != h);
  if (number_positive(internal_value (mp_tracing_choices)))
    mp_print_path (mp, knots, ", after choices", true);
  if (mp->arith_error) {
    @<Report an unexpected problem during the choice-making@>;
  }
}

@ @<Internal ...@>=
void mp_make_choices (MP mp, mp_knot knots);

@ @<Report an unexpected problem during the choice...@>=
{
  const char *hlp[] = {
         "The path that I just computed is out of range.",
         "So it will probably look funny. Proceed, for a laugh.",
          NULL };
  mp_back_error (mp, "Some number got too big", hlp, true);
@.Some number got too big@>;
  mp_get_x_next (mp);
  mp->arith_error = false;
}


@ Two knots in a row with the same coordinates will always be joined
by an explicit ``curve'' whose control points are identical with the
knots.

@<If consecutive knots are equal, join them explicitly@>=
p = knots;
do {
  q = mp_next_knot (p);
  if (number_equal (p->x_coord, q->x_coord) &&
      number_equal (p->y_coord, q->y_coord) && 
      mp_right_type (p) > mp_explicit) {
    mp_right_type (p) = mp_explicit;
    if (mp_left_type (p) == mp_open) {
      mp_left_type (p) = mp_curl;
      set_number_to_unity(p->left_curl);
    }
    mp_left_type (q) = mp_explicit;
    if (mp_right_type (q) == mp_open) {
      mp_right_type (q) = mp_curl;
      set_number_to_unity(q->right_curl);
    }
    number_clone (p->right_x, p->x_coord);
    number_clone (q->left_x, p->x_coord);
    number_clone (p->right_y, p->y_coord);
    number_clone (q->left_y, p->y_coord);
  }
  p = q;
} while (p != knots)

@ If there are no breakpoints, it is necessary to compute the direction
angles around an entire cycle. In this case the |mp_left_type| of the first
node is temporarily changed to |end_cycle|.

@<Find the first breakpoint, |h|, on the path...@>=
h = knots;
while (1) {
  if (mp_left_type (h) != mp_open)
    break;
  if (mp_right_type (h) != mp_open)
    break;
  h = mp_next_knot (h);
  if (h == knots) {
    mp_left_type (h) = mp_end_cycle;
    break;
  }
}


@ If |mp_right_type(p)<given| and |q=mp_link(p)|, we must have
|mp_right_type(p)=mp_left_type(q)=mp_explicit| or |endpoint|.

@<Fill in the control points between |p| and the next breakpoint...@>=
q = mp_next_knot (p);
if (mp_right_type (p) >= mp_given) {
  while ((mp_left_type (q) == mp_open) && (mp_right_type (q) == mp_open)) {
    q = mp_next_knot (q);
  }
  @<Fill in the control information between consecutive breakpoints |p| and |q|@>;
} else if (mp_right_type (p) == mp_endpoint) {
  @<Give reasonable values for the unused control points between |p| and~|q|@>;
}
p = q

@ This step makes it possible to transform an explicitly computed path without
checking the |mp_left_type| and |mp_right_type| fields.

@<Give reasonable values for the unused control points between |p| and~|q|@>=
{
  number_clone (p->right_x, p->x_coord);
  number_clone (p->right_y, p->y_coord);
  number_clone (q->left_x, q->x_coord);
  number_clone (q->left_y, q->y_coord);
}


@ Before we can go further into the way choices are made, we need to
consider the underlying theory. The basic ideas implemented in |make_choices|
are due to John Hobby, who introduced the notion of ``mock curvature''
@^Hobby, John Douglas@>
at a knot. Angles are chosen so that they preserve mock curvature when
a knot is passed, and this has been found to produce excellent results.

It is convenient to introduce some notations that simplify the necessary
formulas. Let $d_{k,k+1}=\vert z\k-z_k\vert$ be the (nonzero) distance
between knots |k| and |k+1|; and let
$${z\k-z_k\over z_k-z_{k-1}}={d_{k,k+1}\over d_{k-1,k}}e^{i\psi_k}$$
so that a polygonal line from $z_{k-1}$ to $z_k$ to $z\k$ turns left
through an angle of~$\psi_k$. We assume that $\vert\psi_k\vert\L180^\circ$.
The control points for the spline from $z_k$ to $z\k$ will be denoted by
$$\eqalign{z_k^+&=z_k+
  \textstyle{1\over3}\rho_k e^{i\theta_k}(z\k-z_k),\cr
 z\k^-&=z\k-
  \textstyle{1\over3}\sigma\k e^{-i\phi\k}(z\k-z_k),\cr}$$
where $\rho_k$ and $\sigma\k$ are nonnegative ``velocity ratios'' at the
beginning and end of the curve, while $\theta_k$ and $\phi\k$ are the
corresponding ``offset angles.'' These angles satisfy the condition
$$\theta_k+\phi_k+\psi_k=0,\eqno(*)$$
whenever the curve leaves an intermediate knot~|k| in the direction that
it enters.

@ Let $\alpha_k$ and $\beta\k$ be the reciprocals of the ``tension'' of
the curve at its beginning and ending points. This means that
$\rho_k=\alpha_k f(\theta_k,\phi\k)$ and $\sigma\k=\beta\k f(\phi\k,\theta_k)$,
where $f(\theta,\phi)$ is \MP's standard velocity function defined in
the |velocity| subroutine. The cubic spline $B(z_k^{\phantom+},z_k^+,
z\k^-,z\k^{\phantom+};t)$
has curvature
@^curvature@>
$${2\sigma\k\sin(\theta_k+\phi\k)-6\sin\theta_k\over\rho_k^2d_{k,k+1}}
\qquad{\rm and}\qquad
{2\rho_k\sin(\theta_k+\phi\k)-6\sin\phi\k\over\sigma\k^2d_{k,k+1}}$$
at |t=0| and |t=1|, respectively. The mock curvature is the linear
@^mock curvature@>
approximation to this true curvature that arises in the limit for
small $\theta_k$ and~$\phi\k$, if second-order terms are discarded.
The standard velocity function satisfies
$$f(\theta,\phi)=1+O(\theta^2+\theta\phi+\phi^2);$$
hence the mock curvatures are respectively
$${2\beta\k(\theta_k+\phi\k)-6\theta_k\over\alpha_k^2d_{k,k+1}}
\qquad{\rm and}\qquad
{2\alpha_k(\theta_k+\phi\k)-6\phi\k\over\beta\k^2d_{k,k+1}}.\eqno(**)$$

@ The turning angles $\psi_k$ are given, and equation $(*)$ above
determines $\phi_k$ when $\theta_k$ is known, so the task of
angle selection is essentially to choose appropriate values for each
$\theta_k$. When equation~$(*)$ is used to eliminate $\phi$~variables
from $(**)$, we obtain a system of linear equations of the form
$$A_k\theta_{k-1}+(B_k+C_k)\theta_k+D_k\theta\k=-B_k\psi_k-D_k\psi\k,$$
where
$$A_k={\alpha_{k-1}\over\beta_k^2d_{k-1,k}},
\qquad B_k={3-\alpha_{k-1}\over\beta_k^2d_{k-1,k}},
\qquad C_k={3-\beta\k\over\alpha_k^2d_{k,k+1}},
\qquad D_k={\beta\k\over\alpha_k^2d_{k,k+1}}.$$
The tensions are always $3\over4$ or more, hence each $\alpha$ and~$\beta$
will be at most $4\over3$. It follows that $B_k\G{5\over4}A_k$ and
$C_k\G{5\over4}D_k$; hence the equations are diagonally dominant;
hence they have a unique solution. Moreover, in most cases the tensions
are equal to~1, so that $B_k=2A_k$ and $C_k=2D_k$. This makes the
solution numerically stable, and there is an exponential damping
effect: The data at knot $k\pm j$ affects the angle at knot~$k$ by
a factor of~$O(2^{-j})$.

@ However, we still must consider the angles at the starting and ending
knots of a non-cyclic path. These angles might be given explicitly, or
they might be specified implicitly in terms of an amount of ``curl.''

Let's assume that angles need to be determined for a non-cyclic path
starting at $z_0$ and ending at~$z_n$. Then equations of the form
$$A_k\theta_{k-1}+(B_k+C_k)\theta_k+D_k\theta_{k+1}=R_k$$
have been given for $0<k<n$, and it will be convenient to introduce
equations of the same form for $k=0$ and $k=n$, where
$$A_0=B_0=C_n=D_n=0.$$
If $\theta_0$ is supposed to have a given value $E_0$, we simply
define $C_0=1$, $D_0=0$, and $R_0=E_0$. Otherwise a curl
parameter, $\gamma_0$, has been specified at~$z_0$; this means
that the mock curvature at $z_0$ should be $\gamma_0$ times the
mock curvature at $z_1$; i.e.,
$${2\beta_1(\theta_0+\phi_1)-6\theta_0\over\alpha_0^2d_{01}}
=\gamma_0{2\alpha_0(\theta_0+\phi_1)-6\phi_1\over\beta_1^2d_{01}}.$$
This equation simplifies to
$$(\alpha_0\chi_0+3-\beta_1)\theta_0+
 \bigl((3-\alpha_0)\chi_0+\beta_1\bigr)\theta_1=
 -\bigl((3-\alpha_0)\chi_0+\beta_1\bigr)\psi_1,$$
where $\chi_0=\alpha_0^2\gamma_0/\beta_1^2$; so we can set $C_0=
\chi_0\alpha_0+3-\beta_1$, $D_0=(3-\alpha_0)\chi_0+\beta_1$, $R_0=-D_0\psi_1$.
It can be shown that $C_0>0$ and $C_0B_1-A_1D_0>0$ when $\gamma_0\G0$,
hence the linear equations remain nonsingular.

Similar considerations apply at the right end, when the final angle $\phi_n$
may or may not need to be determined. It is convenient to let $\psi_n=0$,
hence $\theta_n=-\phi_n$. We either have an explicit equation $\theta_n=E_n$,
or we have
$$\bigl((3-\beta_n)\chi_n+\alpha_{n-1}\bigr)\theta_{n-1}+
(\beta_n\chi_n+3-\alpha_{n-1})\theta_n=0,\qquad
  \chi_n={\beta_n^2\gamma_n\over\alpha_{n-1}^2}.$$

When |make_choices| chooses angles, it must compute the coefficients of
these linear equations, then solve the equations. To compute the coefficients,
it is necessary to compute arctangents of the given turning angles~$\psi_k$.
When the equations are solved, the chosen directions $\theta_k$ are put
back into the form of control points by essentially computing sines and
cosines.

@ OK, we are ready to make the hard choices of |make_choices|.
Most of the work is relegated to an auxiliary procedure
called |solve_choices|, which has been introduced to keep
|make_choices| from being extremely long.

@<Fill in the control information between...@>=
@<Calculate the turning angles $\psi_k$ and the distances $d_{k,k+1}$;
  set $n$ to the length of the path@>;
@<Remove |open| types at the breakpoints@>;
mp_solve_choices (mp, p, q, n)
 

@ It's convenient to precompute quantities that will be needed several
times later. The values of |delta_x[k]| and |delta_y[k]| will be the
coordinates of $z\k-z_k$, and the magnitude of this vector will be
|delta[k]=@t$d_{k,k+1}$@>|. The path angle $\psi_k$ between $z_k-z_{k-1}$
and $z\k-z_k$ will be stored in |psi[k]|.

@<Glob...@>=
int path_size;  /* maximum number of knots between breakpoints of a path */
mp_number *delta_x;
mp_number *delta_y;
mp_number *delta;  /* knot differences */
mp_number *psi;     /* turning angles */

@ @<Dealloc variables@>=
{
  int k;
  for (k = 0; k<mp->path_size; k++) { 
    free_number (mp->delta_x[k]); 
    free_number (mp->delta_y[k]); 
    free_number (mp->delta[k]); 
    free_number (mp->psi[k]); 
  }
  xfree (mp->delta_x);
  xfree (mp->delta_y);
  xfree (mp->delta);
  xfree (mp->psi);
}

@ @<Other local variables for |make_choices|@>=
int k, n;       /* current and final knot numbers */
mp_knot s, t;   /* registers for list traversal */

@ @<Calculate the turning angles...@>=
{
  mp_number sine, cosine;  /* trig functions of various angles */
  new_fraction (sine);
  new_fraction (cosine);
RESTART:
  k = 0;
  s = p;
  n = mp->path_size;
  do {
    t = mp_next_knot (s);
    set_number_from_substraction(mp->delta_x[k], t->x_coord, s->x_coord);
    set_number_from_substraction(mp->delta_y[k], t->y_coord, s->y_coord);
    pyth_add (mp->delta[k], mp->delta_x[k], mp->delta_y[k]);
    if (k > 0) {
      mp_number arg1, arg2, r1, r2;
      new_number (arg1);
      new_number (arg2);
      new_fraction (r1);
      new_fraction (r2);
      make_fraction (r1, mp->delta_y[k - 1], mp->delta[k - 1]);
      number_clone (sine, r1);
      make_fraction (r2, mp->delta_x[k - 1], mp->delta[k - 1]);
      number_clone (cosine, r2);
      take_fraction (r1, mp->delta_x[k], cosine);
      take_fraction (r2, mp->delta_y[k], sine);
      set_number_from_addition (arg1, r1, r2); 
      take_fraction (r1, mp->delta_y[k], cosine);
      take_fraction (r2, mp->delta_x[k], sine);
      set_number_from_substraction (arg2, r1, r2);
      n_arg (mp->psi[k], arg1, arg2 );
      free_number (r1);
      free_number (r2);
      free_number (arg1);
      free_number (arg2);
    }
    incr (k);
    s = t;
    if (k == mp->path_size) {
      mp_reallocate_paths (mp, mp->path_size + (mp->path_size / 4));
      goto RESTART;             /* retry, loop size has changed */
    }
    if (s == q)
      n = k;
  } while (!((k >= n) && (mp_left_type (s) != mp_end_cycle)));
  if (k == n)
    set_number_to_zero(mp->psi[k]);
  else
    number_clone(mp->psi[k], mp->psi[1]);
  free_number (sine);
  free_number (cosine);
}


@ When we get to this point of the code, |mp_right_type(p)| is either
|given| or |curl| or |open|. If it is |open|, we must have
|mp_left_type(p)=mp_end_cycle| or |mp_left_type(p)=mp_explicit|. In the latter
case, the |open| type is converted to |given|; however, if the
velocity coming into this knot is zero, the |open| type is
converted to a |curl|, since we don't know the incoming direction.

Similarly, |mp_left_type(q)| is either |given| or |curl| or |open| or
|mp_end_cycle|. The |open| possibility is reduced either to |given| or to |curl|.

@<Remove |open| types at the breakpoints@>=
{
  mp_number delx, dely;      /* directions where |open| meets |explicit| */
  new_number(delx);
  new_number(dely);
  if (mp_left_type (q) == mp_open) {
    set_number_from_substraction(delx, q->right_x, q->x_coord);
    set_number_from_substraction(dely, q->right_y, q->y_coord);
    if (number_zero(delx) && number_zero(dely)) {
      mp_left_type (q) = mp_curl;
      set_number_to_unity(q->left_curl);
    } else {
      mp_left_type (q) = mp_given;
      n_arg (q->left_given, delx, dely);
    }
  }
  if ((mp_right_type (p) == mp_open) && (mp_left_type (p) == mp_explicit)) {
    set_number_from_substraction(delx, p->x_coord, p->left_x);
    set_number_from_substraction(dely, p->y_coord, p->left_y);
    if (number_zero(delx) && number_zero(dely)) {
      mp_right_type (p) = mp_curl;
      set_number_to_unity(p->right_curl);
    } else {
      mp_right_type (p) = mp_given;
      n_arg (p->right_given, delx, dely);
    }
  }
  free_number (delx);
  free_number (dely);
}

@ Linear equations need to be solved whenever |n>1|; and also when |n=1|
and exactly one of the breakpoints involves a curl. The simplest case occurs
when |n=1| and there is a curl at both breakpoints; then we simply draw
a straight line.

But before coding up the simple cases, we might as well face the general case,
since we must deal with it sooner or later, and since the general case
is likely to give some insight into the way simple cases can be handled best.

When there is no cycle, the linear equations to be solved form a tridiagonal
system, and we can apply the standard technique of Gaussian elimination
to convert that system to a sequence of equations of the form
$$\theta_0+u_0\theta_1=v_0,\quad
\theta_1+u_1\theta_2=v_1,\quad\ldots,\quad
\theta_{n-1}+u_{n-1}\theta_n=v_{n-1},\quad
\theta_n=v_n.$$
It is possible to do this diagonalization while generating the equations.
Once $\theta_n$ is known, it is easy to determine $\theta_{n-1}$, \dots,
$\theta_1$, $\theta_0$; thus, the equations will be solved.

The procedure is slightly more complex when there is a cycle, but the
basic idea will be nearly the same. In the cyclic case the right-hand
sides will be $v_k+w_k\theta_0$ instead of simply $v_k$, and we will start
the process off with $u_0=v_0=0$, $w_0=1$. The final equation will be not
$\theta_n=v_n$ but $\theta_n+u_n\theta_1=v_n+w_n\theta_0$; an appropriate
ending routine will take account of the fact that $\theta_n=\theta_0$ and
eliminate the $w$'s from the system, after which the solution can be
obtained as before.

When $u_k$, $v_k$, and $w_k$ are being computed, the three pointer
variables |r|, |s|,~|t| will point respectively to knots |k-1|, |k|,
and~|k+1|. The $u$'s and $w$'s are scaled by $2^{28}$, i.e., they are
of type |fraction|; the $\theta$'s and $v$'s are of type |angle|.

@<Glob...@>=
mp_number *theta;   /* values of $\theta_k$ */
mp_number *uu;   /* values of $u_k$ */
mp_number *vv;      /* values of $v_k$ */
mp_number *ww;   /* values of $w_k$ */

@ @<Dealloc variables@>=
{
  int k;
  for (k = 0; k<mp->path_size; k++) { 
    free_number (mp->theta[k]); 
    free_number (mp->uu[k]); 
    free_number (mp->vv[k]); 
    free_number (mp->ww[k]); 
  }
  xfree (mp->theta);
  xfree (mp->uu);
  xfree (mp->vv);
  xfree (mp->ww);
}

@ @<Declarations@>=
static void mp_reallocate_paths (MP mp, int l);

@ @c
void mp_reallocate_paths (MP mp, int l) {
  int k;
  XREALLOC (mp->delta_x, l, mp_number);
  XREALLOC (mp->delta_y, l, mp_number);
  XREALLOC (mp->delta, l, mp_number);
  XREALLOC (mp->psi, l, mp_number);
  XREALLOC (mp->theta, l, mp_number);
  XREALLOC (mp->uu, l, mp_number);
  XREALLOC (mp->vv, l, mp_number);
  XREALLOC (mp->ww, l, mp_number);
  for (k = mp->path_size; k<l; k++) { 
    new_number (mp->delta_x[k]); 
    new_number (mp->delta_y[k]); 
    new_number (mp->delta[k]); 
    new_angle (mp->psi[k]); 
    new_angle (mp->theta[k]); 
    new_fraction (mp->uu[k]); 
    new_angle (mp->vv[k]); 
    new_fraction (mp->ww[k]); 
  }
  mp->path_size = l;
}


@ Our immediate problem is to get the ball rolling by setting up the
first equation or by realizing that no equations are needed, and to fit
this initialization into a framework suitable for the overall computation.

@<Declarations@>=
static void mp_solve_choices (MP mp, mp_knot p, mp_knot q, halfword n);

@ @c
void mp_solve_choices (MP mp, mp_knot p, mp_knot q, halfword n) {
  int k;        /* current knot number */
  mp_knot r, s, t;      /* registers for list traversal */
  mp_number ff;
  new_fraction (ff);
  FUNCTION_TRACE2 ("solve_choices(%d)\n", n);
  k = 0;
  s = p;
  r = 0;
  while (1) {
    t = mp_next_knot (s);
    if (k == 0) {
      @<Get the linear equations started; or |return|
        with the control points in place, if linear equations
        needn't be solved@>
    } else {
      switch (mp_left_type (s)) {
      case mp_end_cycle:
      case mp_open:
        @<Set up equation to match mock curvatures
          at $z_k$; then |goto found| with $\theta_n$
          adjusted to equal $\theta_0$, if a cycle has ended@>;
        break;
      case mp_curl:
        @<Set up equation for a curl at $\theta_n$
          and |goto found|@>;
        break;
      case mp_given:
        @<Calculate the given value of $\theta_n$
          and |goto found|@>;
        break;
      }                         /* there are no other cases */
    }
    r = s;
    s = t;
    incr (k);
  }
FOUND:
  @<Finish choosing angles and assigning control points@>;
  free_number (ff);
}


@ On the first time through the loop, we have |k=0| and |r| is not yet
defined. The first linear equation, if any, will have $A_0=B_0=0$.

@<Get the linear equations started...@>=
switch (mp_right_type (s)) {
case mp_given:
  if (mp_left_type (t) == mp_given) {
    @<Reduce to simple case of two givens  and |return|@>
  } else {
    @<Set up the equation for a given value of $\theta_0$@>;
  }
  break;
case mp_curl:
  if (mp_left_type (t) == mp_curl) {
    @<Reduce to simple case of straight line and |return|@>
  } else {
    @<Set up the equation for a curl at $\theta_0$@>;
  }
  break;
case mp_open:
  set_number_to_zero(mp->uu[0]);
  set_number_to_zero(mp->vv[0]);
  number_clone(mp->ww[0], fraction_one_t);
  /* this begins a cycle */
  break;
}                               /* there are no other cases */


@ The general equation that specifies equality of mock curvature at $z_k$ is
$$A_k\theta_{k-1}+(B_k+C_k)\theta_k+D_k\theta\k=-B_k\psi_k-D_k\psi\k,$$
as derived above. We want to combine this with the already-derived equation
$\theta_{k-1}+u_{k-1}\theta_k=v_{k-1}+w_{k-1}\theta_0$ in order to obtain
a new equation
$\theta_k+u_k\theta\k=v_k+w_k\theta_0$. This can be done by dividing the
equation
$$(B_k-u_{k-1}A_k+C_k)\theta_k+D_k\theta\k=-B_k\psi_k-D_k\psi\k-A_kv_{k-1}
    -A_kw_{k-1}\theta_0$$
by $B_k-u_{k-1}A_k+C_k$. The trick is to do this carefully with
fixed-point arithmetic, avoiding the chance of overflow while retaining
suitable precision.

The calculations will be performed in several registers that
provide temporary storage for intermediate quantities.

@ @<Set up equation to match mock curvatures...@>=
{
  mp_number aa, bb, cc, acc;   /* temporary registers */
  mp_number dd, ee;  /* likewise, but |scaled| */
  new_fraction (aa);
  new_fraction (bb);
  new_fraction (cc);
  new_fraction (acc);
  new_number (dd);
  new_number (ee);
  @<Calculate the values $\\{aa}=A_k/B_k$, $\\{bb}=D_k/C_k$,
    $\\{dd}=(3-\alpha_{k-1})d_{k,k+1}$, $\\{ee}=(3-\beta\k)d_{k-1,k}$,
    and $\\{cc}=(B_k-u_{k-1}A_k)/B_k$@>;
  @<Calculate the ratio $\\{ff}=C_k/(C_k+B_k-u_{k-1}A_k)$@>;
  take_fraction (mp->uu[k], ff, bb);
  @<Calculate the values of $v_k$ and $w_k$@>;
  if (mp_left_type (s) == mp_end_cycle) {
    @<Adjust $\theta_n$ to equal $\theta_0$ and |goto found|@>;
  }
  free_number(aa);
  free_number(bb);
  free_number(cc);
  free_number(acc);
  free_number(dd);
  free_number(ee);
}


@ Since tension values are never less than 3/4, the values |aa| and
|bb| computed here are never more than 4/5.

@<Calculate the values $\\{aa}=...@>=
{
  mp_number absval;
  new_number (absval);
  number_clone (absval, r->right_tension);
  number_abs (absval);
if (number_equal (absval, unity_t)) {
  number_clone (aa, fraction_half_t);
  number_clone (dd, mp->delta[k]);
  number_double (dd);
} else {
  mp_number arg1, arg2, ret;
  new_number (arg2);
  new_number (arg1);
  number_clone (arg2, r->right_tension);
  number_abs (arg2);
  number_multiply_int (arg2, 3);
  number_substract (arg2, unity_t);
  make_fraction (aa, unity_t, arg2);
  number_clone (arg2, r->right_tension);
  number_abs (arg2);
  new_fraction (ret); 
  make_fraction (ret, unity_t, arg2);
  set_number_from_substraction (arg1, fraction_three_t, ret);
  take_fraction (arg2, mp->delta[k], arg1);
  number_clone (dd, arg2);
  free_number (ret);
  free_number (arg1);
  free_number (arg2);
}
  number_clone (absval, t->left_tension);
  number_abs (absval);
if (number_equal (absval, unity_t)) {
  number_clone (bb, fraction_half_t);
  number_clone (ee, mp->delta[k - 1]);
  number_double (ee);
} else {
  mp_number arg1, arg2, ret;
  new_number (arg1);
  new_number (arg2);
  number_clone (arg2, t->left_tension);
  number_abs (arg2);
  number_multiply_int (arg2, 3);
  number_substract (arg2, unity_t);
  make_fraction (bb, unity_t, arg2);
  number_clone (arg2, t->left_tension);
  number_abs (arg2);
  new_fraction(ret);
  make_fraction (ret, unity_t, arg2);
  set_number_from_substraction (arg1,fraction_three_t, ret);
  take_fraction (ee, mp->delta[k - 1], arg1);
  free_number (ret);
  free_number (arg1);
  free_number (arg2);
}
free_number (absval);
}
{
  mp_number r1;
  new_number (r1);
  take_fraction (r1, mp->uu[k - 1], aa);
  set_number_from_substraction (cc, fraction_one_t, r1);
  free_number (r1);
}

@ The ratio to be calculated in this step can be written in the form
$$\beta_k^2\cdot\\{ee}\over\beta_k^2\cdot\\{ee}+\alpha_k^2\cdot
  \\{cc}\cdot\\{dd},$$
because of the quantities just calculated. The values of |dd| and |ee|
will not be needed after this step has been performed.

@<Calculate the ratio $\\{ff}=C_k/(C_k+B_k-u_{k-1}A_k)$@>=
{
  mp_number rt, lt;
  mp_number arg2;
  new_number (arg2);
  number_clone (arg2, dd);
  take_fraction (dd, arg2, cc);
  new_number (lt);
  new_number (rt);
  number_clone (lt, s->left_tension);
  number_abs (lt);
  number_clone (rt, s->right_tension);
  number_abs (rt);
  if (!number_equal(lt, rt)) {                 /* $\beta_k^{-1}\ne\alpha_k^{-1}$ */
    mp_number r1;
    new_number (r1);
    if (number_less(lt, rt)) {
      make_fraction (r1, lt, rt);  /* $\alpha_k^2/\beta_k^2$ */
      take_fraction (ff, r1, r1);
      number_clone (r1, dd);
      take_fraction (dd, r1, ff);
    } else {
      make_fraction (r1, rt, lt);  /* $\beta_k^2/\alpha_k^2$ */
      take_fraction (ff, r1, r1);
      number_clone (r1, ee);
      take_fraction (ee, r1, ff);
    }
    free_number (r1);
  }
  free_number (rt);
  free_number (lt);
  set_number_from_addition (arg2, dd, ee);
  make_fraction (ff, ee, arg2);
  free_number (arg2);
}
 

@ The value of $u_{k-1}$ will be |<=1| except when $k=1$ and the previous
equation was specified by a curl. In that case we must use a special
method of computation to prevent overflow.

Fortunately, the calculations turn out to be even simpler in this ``hard''
case. The curl equation makes $w_0=0$ and $v_0=-u_0\psi_1$, hence
$-B_1\psi_1-A_1v_0=-(B_1-u_0A_1)\psi_1=-\\{cc}\cdot B_1\psi_1$.

@<Calculate the values of $v_k$ and $w_k$@>=
take_fraction (acc, mp->psi[k + 1], mp->uu[k]);
number_negate (acc);
if (mp_right_type (r) == mp_curl) {
  mp_number r1, arg2;
  new_fraction (r1);
  new_number (arg2);
  set_number_from_substraction (arg2, fraction_one_t, ff);
  take_fraction (r1, mp->psi[1], arg2);
  set_number_to_zero(mp->ww[k]);
  set_number_from_substraction(mp->vv[k], acc, r1);
  free_number (r1);
  free_number (arg2);
} else {
  mp_number arg1, r1;
  new_fraction (r1);
  new_number (arg1);
  set_number_from_substraction (arg1, fraction_one_t, ff);
  make_fraction (ff, arg1, cc);    /* this is $B_k/(C_k+B_k-u_{k-1}A_k)<5$ */
  free_number (arg1);
  take_fraction (r1, mp->psi[k], ff);
  number_substract (acc, r1);
  number_clone (r1, ff);
  take_fraction (ff, r1, aa);   /* this is $A_k/(C_k+B_k-u_{k-1}A_k)$ */
  take_fraction (r1, mp->vv[k - 1], ff);
  set_number_from_substraction(mp->vv[k], acc, r1 );
  if (number_zero(mp->ww[k - 1])) {
    set_number_to_zero(mp->ww[k]);
  } else {
    take_fraction (mp->ww[k], mp->ww[k - 1], ff);
    number_negate(mp->ww[k]);
  }
  free_number (r1);
}


@ When a complete cycle has been traversed, we have $\theta_k+u_k\theta\k=
v_k+w_k\theta_0$, for |1<=k<=n|. We would like to determine the value of
$\theta_n$ and reduce the system to the form $\theta_k+u_k\theta\k=v_k$
for |0<=k<n|, so that the cyclic case can be finished up just as if there
were no cycle.

The idea in the following code is to observe that
$$\eqalign{\theta_n&=v_n+w_n\theta_0-u_n\theta_1=\cdots\cr
&=v_n+w_n\theta_0-u_n\bigl(v_1+w_1\theta_0-u_1(v_2+\cdots
  -u_{n-2}(v_{n-1}+w_{n-1}\theta_0-u_{n-1}\theta_0))\bigr),\cr}$$
so we can solve for $\theta_n=\theta_0$.

@<Adjust $\theta_n$ to equal $\theta_0$ and |goto found|@>=
{
  mp_number arg2, r1;
  new_number (arg2);
  new_number (r1);
  set_number_to_zero (aa);
  number_clone (bb, fraction_one_t);            /* we have |k=n| */
  do {
    decr (k);
    if (k == 0)
      k = n;
    take_fraction (r1, aa, mp->uu[k]);
    set_number_from_substraction (aa, mp->vv[k], r1);
    take_fraction (r1, bb, mp->uu[k]);
    set_number_from_substraction (bb, mp->ww[k], r1);
  } while (k != n);             /* now $\theta_n=\\{aa}+\\{bb}\cdot\theta_n$ */
  set_number_from_substraction (arg2, fraction_one_t, bb);
  make_fraction (r1, aa, arg2);
  number_clone (aa, r1);
  number_clone(mp->theta[n], aa);
  number_clone(mp->vv[0], aa);
  for (k = 1; k < n; k++) {
    take_fraction (r1, aa, mp->ww[k]);
    number_add(mp->vv[k], r1);
  }
  free_number(arg2);
  free_number(r1);
  free_number(aa);
  free_number(bb);
  free_number(cc);
  free_number(acc);
  free_number(dd);
  free_number(ee);
  goto FOUND;
}


@ @c 
void mp_reduce_angle (MP mp, mp_number *a) {
  mp_number abs_a;
  FUNCTION_TRACE2 ("reduce_angle(%f)\n", number_to_double(*a));
  new_number(abs_a);
  number_clone(abs_a, *a);
  number_abs(abs_a);
  if ( number_greater(abs_a, one_eighty_deg_t)) {
    if (number_positive(*a)) {
      number_substract(*a, three_sixty_deg_t); 
    } else {
      number_add(*a, three_sixty_deg_t); 
    }
  }
  free_number(abs_a);
}

@ @<Declarations@>=
void mp_reduce_angle (MP mp, mp_number *a);


@ @<Calculate the given value of $\theta_n$...@>=
{
  mp_number narg;
  new_angle (narg);
  n_arg (narg, mp->delta_x[n - 1], mp->delta_y[n - 1]);
  set_number_from_substraction(mp->theta[n], s->left_given, narg);
  free_number (narg);
  mp_reduce_angle (mp, &mp->theta[n]);
  goto FOUND;
}


@ @<Set up the equation for a given value of $\theta_0$@>=
{
  mp_number narg;
  new_angle (narg);
  n_arg (narg, mp->delta_x[0], mp->delta_y[0]);
  set_number_from_substraction(mp->vv[0], s->right_given, narg);
  free_number (narg);
  mp_reduce_angle (mp, &mp->vv[0]);
  set_number_to_zero(mp->uu[0]);
  set_number_to_zero(mp->ww[0]);
}


@ @<Set up the equation for a curl at $\theta_0$@>=
{
  mp_number lt, rt, cc;  /* tension values */
  new_number (lt);
  new_number (rt);
  new_number (cc);
  number_clone (cc, s->right_curl);
  number_clone (lt, t->left_tension);
  number_abs(lt);
  number_clone (rt, s->right_tension);
  number_abs(rt);
  if (number_unity(rt) && number_unity(lt)) {
    mp_number arg1, arg2;
    new_number (arg1);
    new_number (arg2);
    number_clone (arg1, cc);
    number_double (arg1);
    number_add (arg1, unity_t);
    number_clone (arg2, cc);
    number_add (arg2, two_t);
    make_fraction (mp->uu[0], arg1, arg2);
    free_number (arg1);
    free_number (arg2);
  } else {
    mp_curl_ratio (mp, &mp->uu[0], cc, rt, lt);
  }
  take_fraction (mp->vv[0], mp->psi[1], mp->uu[0]);
  number_negate(mp->vv[0]);
  set_number_to_zero(mp->ww[0]);
  free_number (rt);
  free_number (lt);
  free_number (cc);
}


@ @<Set up equation for a curl at $\theta_n$...@>=
{
  mp_number lt, rt, cc;  /* tension values */
  new_number (lt);
  new_number (rt);
  new_number (cc);
  number_clone (cc, s->left_curl);
  number_clone (lt, s->left_tension);
  number_abs(lt);
  number_clone (rt, r->right_tension);
  number_abs(rt);
  if (number_unity(rt) && number_unity(lt)) {
    mp_number arg1, arg2;
    new_number (arg1);
    new_number (arg2);
    number_clone (arg1, cc);
    number_double (arg1);
    number_add (arg1, unity_t);
    number_clone (arg2, cc);
    number_add (arg2, two_t);
    make_fraction (ff, arg1, arg2);
    free_number (arg1);
    free_number (arg2);
  } else {
    mp_curl_ratio (mp, &ff, cc, lt, rt);
  }
  {
    mp_number arg1, arg2, r1;
    new_fraction (r1);
    new_fraction (arg1);
    new_number (arg2);
    take_fraction (arg1, mp->vv[n - 1], ff);
    take_fraction (r1, ff, mp->uu[n - 1]);
    set_number_from_substraction (arg2, fraction_one_t, r1);
    make_fraction (mp->theta[n], arg1, arg2);
    number_negate(mp->theta[n]);
    free_number (r1);
    free_number (arg1);
    free_number (arg2);
  }
  free_number (rt);
  free_number (lt);
  free_number (cc);
  goto FOUND;
}


@ The |curl_ratio| subroutine has three arguments, which our previous notation
encourages us to call $\gamma$, $\alpha^{-1}$, and $\beta^{-1}$. It is
a somewhat tedious program to calculate
$${(3-\alpha)\alpha^2\gamma+\beta^3\over
  \alpha^3\gamma+(3-\beta)\beta^2},$$
with the result reduced to 4 if it exceeds 4. (This reduction of curl
is necessary only if the curl and tension are both large.)
The values of $\alpha$ and $\beta$ will be at most~4/3.

@<Declarations@>=
static void mp_curl_ratio (MP mp, mp_number *ret, mp_number gamma, mp_number a_tension,
                                 mp_number b_tension);

@ @c
void mp_curl_ratio (MP mp, mp_number *ret, mp_number gamma_orig, mp_number a_tension, mp_number b_tension) {
  mp_number alpha, beta, gamma, num, denom, ff; /* registers */
  mp_number arg1;
  new_number (arg1);
  new_fraction (alpha);
  new_fraction (beta);
  new_fraction (gamma);
  new_fraction (ff);
  new_fraction (denom);
  new_fraction (num);
  make_fraction (alpha, unity_t, a_tension);
  make_fraction (beta, unity_t, b_tension);
  number_clone (gamma, gamma_orig);
  if (number_lessequal(alpha, beta)) {
    make_fraction (ff, alpha, beta);
    number_clone (arg1, ff);
    take_fraction (ff, arg1, arg1);
    number_clone (arg1, gamma);
    take_fraction (gamma, arg1, ff);
    convert_fraction_to_scaled (beta);
    take_fraction (denom, gamma, alpha);
    number_add (denom, three_t);
  } else {
    make_fraction (ff, beta, alpha);
    number_clone (arg1, ff);
    take_fraction (ff, arg1, arg1);
    take_fraction (arg1, beta, ff);
    convert_fraction_to_scaled (arg1);
    number_clone (beta, arg1);
    take_fraction (denom, gamma, alpha);
    set_number_from_div (arg1, ff, twelvebits_3);
    number_add (denom, arg1);
  }
  number_substract (denom, beta);
  set_number_from_substraction (arg1, fraction_three_t, alpha);
  take_fraction (num, gamma, arg1);
  number_add (num, beta);
  number_clone (arg1, denom);
  number_double (arg1);
  number_double (arg1); /* arg1 = 4*denom */
  if (number_greaterequal(num, arg1)) {
    number_clone(*ret, fraction_four_t);
  } else {
    make_fraction (*ret, num, denom);
  }
  free_number (alpha);
  free_number (beta);
  free_number (gamma);
  free_number (num);
  free_number (denom);
  free_number (ff);
  free_number (arg1);
}


@ We're in the home stretch now.

@<Finish choosing angles and assigning control points@>=
{
  mp_number r1;
  new_number (r1);
  for (k = n - 1; k >= 0; k--) {
    take_fraction (r1, mp->theta[k + 1], mp->uu[k]);
    set_number_from_substraction(mp->theta[k], mp->vv[k], r1);
  }
  free_number (r1);
}
s = p;
k = 0;
{
mp_number arg;
new_number (arg);
do {
  t = mp_next_knot (s);
  n_sin_cos (mp->theta[k], mp->ct, mp->st);
  number_clone (arg, mp->psi[k + 1]);
  number_negate (arg);
  number_substract (arg, mp->theta[k + 1]);
  n_sin_cos (arg, mp->cf, mp->sf);
  mp_set_controls (mp, s, t, k);
  incr (k);
  s = t;
} while (k != n);
free_number (arg);
}


@ The |set_controls| routine actually puts the control points into
a pair of consecutive nodes |p| and~|q|. Global variables are used to
record the values of $\sin\theta$, $\cos\theta$, $\sin\phi$, and
$\cos\phi$ needed in this calculation.

@<Glob...@>=
mp_number st;
mp_number ct;
mp_number sf;
mp_number cf;    /* sines and cosines */

@ @<Initialize table...@>=
new_fraction (mp->st);
new_fraction (mp->ct);
new_fraction (mp->sf);
new_fraction (mp->cf);

@ @<Dealloc ...@>=
free_number (mp->st);
free_number (mp->ct);
free_number (mp->sf);
free_number (mp->cf);


@ @<Declarations@>=
static void mp_set_controls (MP mp, mp_knot p, mp_knot q, integer k);

@ @c
void mp_set_controls (MP mp, mp_knot p, mp_knot q, integer k) {
  mp_number rr, ss;      /* velocities, divided by thrice the tension */
  mp_number lt, rt;        /* tensions */
  mp_number sine;        /* $\sin(\theta+\phi)$ */
  mp_number tmp;
  mp_number r1, r2;
  new_number(tmp);
  new_number (lt);
  new_number (rt);
  new_number (r1);
  new_number (r2);
  number_clone(lt, q->left_tension);
  number_abs(lt);
  number_clone(rt, p->right_tension);
  number_abs(rt);
  new_fraction (sine);
  new_fraction (rr);
  new_fraction (ss);
  velocity (rr, mp->st, mp->ct, mp->sf, mp->cf, rt);
  velocity (ss, mp->sf, mp->cf, mp->st, mp->ct, lt);
  if (number_negative(p->right_tension) || number_negative(q->left_tension)) {
    @<Decrease the velocities,
      if necessary, to stay inside the bounding triangle@>;
  }
  take_fraction (r1, mp->delta_x [k], mp->ct);
  take_fraction (r2, mp->delta_y [k], mp->st);
  number_substract (r1, r2);
  take_fraction (tmp, r1, rr);
  set_number_from_addition (p->right_x, p->x_coord, tmp);
  take_fraction (r1, mp->delta_y[k], mp->ct);
  take_fraction (r2, mp->delta_x[k], mp->st);
  number_add (r1, r2);
  take_fraction (tmp, r1, rr);
  set_number_from_addition (p->right_y, p->y_coord, tmp);
  take_fraction (r1, mp->delta_x[k], mp->cf);
  take_fraction (r2, mp->delta_y[k], mp->sf);
  number_add (r1, r2);
  take_fraction (tmp, r1, ss);
  set_number_from_substraction (q->left_x, q->x_coord, tmp);
  take_fraction (r1, mp->delta_y[k], mp->cf);
  take_fraction (r2, mp->delta_x[k], mp->sf);
  number_substract (r1, r2);
  take_fraction (tmp, r1, ss);
  set_number_from_substraction(q->left_y, q->y_coord, tmp);
  mp_right_type (p) = mp_explicit;
  mp_left_type (q) = mp_explicit;
  free_number (tmp);
  free_number (r1);
  free_number (r2);
  free_number (lt);
  free_number (rt);
  free_number (rr);
  free_number (ss);
  free_number (sine);
}


@ The boundedness conditions $\\{rr}\L\sin\phi\,/\sin(\theta+\phi)$ and
$\\{ss}\L\sin\theta\,/\sin(\theta+\phi)$ are to be enforced if $\sin\theta$,
$\sin\phi$, and $\sin(\theta+\phi)$ all have the same sign. Otherwise
there is no ``bounding triangle.''

@<Decrease the velocities, if necessary...@>=
if ((number_nonnegative(mp->st) && number_nonnegative(mp->sf)) || (number_nonpositive(mp->st) && number_nonpositive(mp->sf))) {
  mp_number r1, r2, arg1;
  mp_number ab_vs_cd;
  new_number (ab_vs_cd);
  new_fraction (r1);
  new_fraction (r2);
  new_number (arg1);
  number_clone (arg1, mp->st);
  number_abs (arg1);
  take_fraction (r1, arg1, mp->cf);
  number_clone (arg1, mp->sf);
  number_abs (arg1);
  take_fraction (r2, arg1, mp->ct);
  set_number_from_addition (sine, r1, r2);
  if (number_positive(sine)) {
    set_number_from_addition (arg1, fraction_one_t, unity_t);  /* safety factor */
    number_clone (r1, sine);
    take_fraction (sine, r1, arg1);
    if (number_negative(p->right_tension)) {
      number_clone (arg1, mp->sf);
      number_abs (arg1);
      ab_vs_cd (ab_vs_cd, arg1, fraction_one_t, rr, sine);
      if (number_negative(ab_vs_cd)) {
        number_clone (arg1, mp->sf);
        number_abs (arg1);
        make_fraction (rr, arg1, sine);
      }
    }
    if (number_negative(q->left_tension)) {
      number_clone (arg1, mp->st);
      number_abs (arg1);
      ab_vs_cd (ab_vs_cd, arg1, fraction_one_t, ss, sine);
      if (number_negative(ab_vs_cd)) {
        number_clone (arg1, mp->st);
        number_abs (arg1);
        make_fraction (ss, arg1, sine);
      }
    }
  }
  free_number (arg1);
  free_number (r1);
  free_number (r2);
  free_number (ab_vs_cd);
}

@ Only the simple cases remain to be handled.

@<Reduce to simple case of two givens and |return|@>=
{
  mp_number arg1;
  mp_number narg;
  new_angle (narg);
  n_arg (narg, mp->delta_x[0], mp->delta_y[0]);
  new_number (arg1);
  set_number_from_substraction (arg1, p->right_given, narg);
  n_sin_cos (arg1, mp->ct, mp->st);
  set_number_from_substraction (arg1, q->left_given, narg);
  n_sin_cos (arg1, mp->cf, mp->sf);
  number_negate (mp->sf);
  mp_set_controls (mp, p, q, 0);
  free_number (narg);
  free_number (arg1);
  free_number (ff);
  return;
}


@ @<Reduce to simple case of straight line and |return|@>=
{
  mp_number lt, rt;  /* tension values */
  mp_right_type (p) = mp_explicit;
  mp_left_type (q) = mp_explicit;
  new_number (lt);
  new_number (rt);
  number_clone (lt, q->left_tension);
  number_abs(lt);
  number_clone (rt, p->right_tension);
  number_abs(rt);
  if (number_unity(rt)) {
    mp_number arg2;
    new_number (arg2);
    if (number_nonnegative(mp->delta_x[0])) {
      set_number_from_addition (arg2, mp->delta_x[0], epsilon_t);
    } else {
      set_number_from_substraction (arg2, mp->delta_x[0], epsilon_t);
    }
    number_int_div (arg2, 3);
    set_number_from_addition (p->right_x, p->x_coord, arg2);
    if (number_nonnegative(mp->delta_y[0])) {
      set_number_from_addition (arg2, mp->delta_y[0], epsilon_t);
    } else {
      set_number_from_substraction (arg2, mp->delta_y[0], epsilon_t);
    }
    number_int_div (arg2, 3);
    set_number_from_addition (p->right_y, p->y_coord, arg2);
    free_number (arg2);
  } else {
    mp_number arg2, r1;
    new_fraction (r1);
    new_number (arg2);
    number_clone (arg2, rt);
    number_multiply_int (arg2, 3);
    make_fraction (ff, unity_t, arg2);  /* $\alpha/3$ */
    free_number (arg2);
    take_fraction (r1, mp->delta_x[0], ff);
    set_number_from_addition (p->right_x, p->x_coord, r1);
    take_fraction (r1, mp->delta_y[0], ff);
    set_number_from_addition (p->right_y, p->y_coord, r1);
  }
  if (number_unity(lt)) {
    mp_number arg2;
    new_number (arg2);
    if (number_nonnegative(mp->delta_x[0])) {
      set_number_from_addition (arg2, mp->delta_x[0], epsilon_t);
    } else {
      set_number_from_substraction (arg2, mp->delta_x[0], epsilon_t);
    }
    number_int_div (arg2, 3);
    set_number_from_substraction (q->left_x, q->x_coord, arg2);
    if (number_nonnegative(mp->delta_y[0])) {
      set_number_from_addition (arg2, mp->delta_y[0], epsilon_t);
    } else {
      set_number_from_substraction (arg2, mp->delta_y[0], epsilon_t);
    }
    number_int_div (arg2, 3);
    set_number_from_substraction (q->left_y, q->y_coord, arg2);
    free_number (arg2);
  } else {
    mp_number arg2, r1;
    new_fraction (r1);
    new_number (arg2);
    number_clone (arg2, lt);
    number_multiply_int (arg2, 3);
    make_fraction (ff, unity_t, arg2);  /* $\beta/3$ */
    free_number (arg2);
    take_fraction (r1, mp->delta_x[0], ff);
    set_number_from_substraction(q->left_x, q->x_coord, r1);
    take_fraction (r1, mp->delta_y[0], ff);
    set_number_from_substraction(q->left_y, q->y_coord, r1);
    free_number (r1);
  }
  free_number (ff);
  free_number (lt);
  free_number (rt);
  return;
}

@ Various subroutines that are useful for the new (1.770) exported 
api for solving path choices

@c
#define TOO_LARGE(a) (fabs((a))>4096.0)
#define PI 3.1415926535897932384626433832795028841971 

static int out_of_range(MP mp, double a) 
{
    mp_number t;
    new_number (t);
    set_number_from_double(t,fabs(a));
    if (number_greaterequal(t,inf_t)) {
       free_number (t);
       return 1;
    }
    free_number (t);
    return 0;
}

static int mp_link_knotpair (MP mp, mp_knot p, mp_knot q);
static int mp_link_knotpair (MP mp, mp_knot p, mp_knot q)
{
    if (p==NULL ||q==NULL) return 0;
    p->next = q;
    set_number_from_double(p->right_tension, 1.0);
    if (mp_right_type(p)==mp_endpoint) {
	mp_right_type(p) = mp_open;
    }
    set_number_from_double(q->left_tension, 1.0);
    if (mp_left_type(q) == mp_endpoint) {
	mp_left_type(q) = mp_open;
    }
    return 1;
}

int mp_close_path_cycle (MP mp, mp_knot p, mp_knot q)
{
    return mp_link_knotpair(mp,p,q);
}

int mp_close_path (MP mp, mp_knot q, mp_knot first)
{
    if (q==NULL || first==NULL) return 0;
    q->next = first;
    mp_right_type(q) = mp_endpoint;
    set_number_from_double(q->right_tension, 1.0);
    mp_left_type(first) = mp_endpoint;
    set_number_from_double(first->left_tension, 1.0);
    return 1;
}

mp_knot mp_create_knot (MP mp)
{
    mp_knot q = mp_new_knot(mp);
    mp_left_type(q) = mp_endpoint;
    mp_right_type(q) = mp_endpoint;
    return q;
}

int mp_set_knot (MP mp, mp_knot p, double x, double y)
{
    if (out_of_range(mp, x)) return 0;
    if (out_of_range(mp, y)) return 0;
    if (p==NULL) return 0;
    set_number_from_double(p->x_coord, x);
    set_number_from_double(p->y_coord, y);
    return 1;
}

mp_knot mp_append_knot (MP mp, mp_knot p, double x, double y)
{
    mp_knot q = mp_create_knot(mp);
    if (q==NULL) return NULL;
    if (!mp_set_knot(mp, q, x, y)) {
	free(q);
	return NULL;
    }
    if (p == NULL) return q;
    if (!mp_link_knotpair(mp, p,q)) {
	free(q);
	return NULL;
    }
    return q;
}

int mp_set_knot_curl (MP mp, mp_knot q, double value) {
    if (q==NULL) return 0;
    if (TOO_LARGE(value)) return 0;
    mp_right_type(q)=mp_curl; 
    set_number_from_double(q->right_curl, value);
    if (mp_left_type(q)==mp_open) {
	mp_left_type(q)=mp_curl; 
	set_number_from_double(q->left_curl, value);
    }
    return 1;
}

int mp_set_knot_left_curl (MP mp, mp_knot q, double value) {
    if (q==NULL) return 0;
    if (TOO_LARGE(value)) return 0;
    mp_left_type(q)=mp_curl; 
    set_number_from_double(q->left_curl, value);
    if (mp_right_type(q)==mp_open) {
	mp_right_type(q)=mp_curl; 
	set_number_from_double(q->right_curl, value);
    }
    return 1;
}

int mp_set_knot_right_curl (MP mp, mp_knot q, double value) {
    if (q==NULL) return 0;
    if (TOO_LARGE(value)) return 0;
    mp_right_type(q)=mp_curl; 
    set_number_from_double(q->right_curl, value);
    if (mp_left_type(q)==mp_open) {
	mp_left_type(q)=mp_curl; 
	set_number_from_double(q->left_curl, value);
    }
    return 1;
}

int mp_set_knotpair_curls (MP mp, mp_knot p, mp_knot q, double t1, double t2) {
    if (p==NULL || q==NULL) return 0;
    if (mp_set_knot_curl(mp, p, t1))
	return mp_set_knot_curl(mp, q, t2);
    return 0;
}

int mp_set_knotpair_tensions (MP mp, mp_knot p, mp_knot q, double t1, double t2) {
    if (p==NULL || q==NULL) return 0;
    if (TOO_LARGE(t1)) return 0;
    if (TOO_LARGE(t2)) return 0;
    if ((fabs(t1)<0.75)) return 0;
    if ((fabs(t2)<0.75)) return 0;
    set_number_from_double(p->right_tension, t1);
    set_number_from_double(q->left_tension, t2);
    return 1;
}

int mp_set_knot_left_tension (MP mp, mp_knot p, double t1) {
    if (p==NULL) return 0;
    if (TOO_LARGE(t1)) return 0;
    if ((fabs(t1)<0.75)) return 0;
    set_number_from_double(p->left_tension, t1);
    return 1;
}

int mp_set_knot_right_tension (MP mp, mp_knot p, double t1) {
    if (p==NULL) return 0;
    if (TOO_LARGE(t1)) return 0;
    if ((fabs(t1)<0.75)) return 0;
    set_number_from_double(p->right_tension, t1);
    return 1;
}

int mp_set_knotpair_controls (MP mp, mp_knot p, mp_knot q, double x1, double y1, double x2, double y2) {
    if (p==NULL || q==NULL) return 0;
    if (out_of_range(mp, x1)) return 0;
    if (out_of_range(mp, y1)) return 0;
    if (out_of_range(mp, x2)) return 0;
    if (out_of_range(mp, y2)) return 0;
    mp_right_type(p)=mp_explicit; 
    set_number_from_double(p->right_x, x1);
    set_number_from_double(p->right_y, y1);
    mp_left_type(q)=mp_explicit; 
    set_number_from_double(q->left_x, x2);
    set_number_from_double(q->left_y, y2);
    return 1;
}

int mp_set_knot_left_control (MP mp, mp_knot p, double x1, double y1) {
    if (p==NULL) return 0;
    if (out_of_range(mp, x1)) return 0;
    if (out_of_range(mp, y1)) return 0;
    mp_left_type(p)=mp_explicit; 
    set_number_from_double(p->left_x, x1);
    set_number_from_double(p->left_y, y1);
    return 1;
}

int mp_set_knot_right_control (MP mp, mp_knot p, double x1, double y1) {
    if (p==NULL) return 0;
    if (out_of_range(mp, x1)) return 0;
    if (out_of_range(mp, y1)) return 0;
    mp_right_type(p)=mp_explicit; 
    set_number_from_double(p->right_x, x1);
    set_number_from_double(p->right_y, y1);
    return 1;
}

int mp_set_knot_direction (MP mp, mp_knot q, double x, double y) {
    double value = 0;
    if (q==NULL) return 0;
    if (TOO_LARGE(x)) return 0;
    if (TOO_LARGE(y)) return 0;
    if (!(x==0 && y == 0))
	value = atan2 (y, x) * (180.0 / PI)  * 16.0;
    mp_right_type(q)=mp_given; 
    set_number_from_double(q->right_curl, value);
    if (mp_left_type(q)==mp_open) {
	mp_left_type(q)=mp_given; 
	set_number_from_double(q->left_curl, value);
    }
    return 1;
}

int mp_set_knotpair_directions (MP mp, mp_knot p, mp_knot q, double x1, double y1, double x2, double y2) {
    if (p==NULL || q==NULL) return 0;
    if (mp_set_knot_direction(mp,p, x1, y1))
	return mp_set_knot_direction(mp,q, x2, y2);
    return 0;
}

@
@c
static int path_needs_fixing (mp_knot source);
static int path_needs_fixing (mp_knot source) {
    mp_knot sourcehead = source;
    do {
	source = source->next;
    } while (source && source != sourcehead);
    if (!source) {
	return 1;
    }
    return 0;
}

int mp_solve_path (MP mp, mp_knot first)
{
    int saved_arith_error = mp->arith_error;
    jmp_buf *saved_jump_buf = mp->jump_buf;
    int retval = 1;
    if (first==NULL) return 0;
    if (path_needs_fixing(first)) return 0;
    mp->jump_buf = malloc(sizeof(jmp_buf));
    if (mp->jump_buf == NULL || setjmp(*(mp->jump_buf)) != 0) {   
       return 0; 
    }    
    mp->arith_error = 0;
    mp_make_choices(mp, first);
    if (mp->arith_error)
 	retval = 0;
    mp->arith_error = saved_arith_error;
    free(mp->jump_buf);
    mp->jump_buf = saved_jump_buf;
    return retval;
}

void mp_free_path (MP mp, mp_knot p) {
    mp_toss_knot_list(mp, p);
}

@ @<Exported function headers@>=
int mp_close_path_cycle (MP mp, mp_knot p, mp_knot q);
int mp_close_path (MP mp, mp_knot q, mp_knot first);
mp_knot mp_create_knot (MP mp);
int mp_set_knot (MP mp, mp_knot p, double x, double y);
mp_knot mp_append_knot (MP mp, mp_knot p, double x, double y);
int mp_set_knot_curl (MP mp, mp_knot q, double value);
int mp_set_knot_left_curl (MP mp, mp_knot q, double value);
int mp_set_knot_right_curl (MP mp, mp_knot q, double value);
int mp_set_knotpair_curls (MP mp, mp_knot p, mp_knot q, double t1, double t2) ;
int mp_set_knotpair_tensions (MP mp, mp_knot p, mp_knot q, double t1, double t2) ;
int mp_set_knot_left_tension (MP mp, mp_knot p, double t1);
int mp_set_knot_right_tension (MP mp, mp_knot p, double t1);
int mp_set_knot_left_control (MP mp, mp_knot p, double t1, double t2);
int mp_set_knot_right_control (MP mp, mp_knot p, double t1, double t2);
int mp_set_knotpair_controls (MP mp, mp_knot p, mp_knot q, double x1, double y1, double x2, double y2) ;
int mp_set_knot_direction (MP mp, mp_knot q, double x, double y) ;
int mp_set_knotpair_directions (MP mp, mp_knot p, mp_knot q, double x1, double y1, double x2, double y2) ;
int mp_solve_path (MP mp, mp_knot first);
void mp_free_path (MP mp, mp_knot p);

@ Simple accessors for |mp_knot|.

@c
mp_number mp_knot_x_coord(MP mp, mp_knot p) { return p->x_coord; }
mp_number mp_knot_y_coord(MP mp, mp_knot p) { return p->y_coord; }
mp_number mp_knot_left_x (MP mp, mp_knot p) { return p->left_x;  }
mp_number mp_knot_left_y (MP mp, mp_knot p) { return p->left_y;  }
mp_number mp_knot_right_x(MP mp, mp_knot p) { return p->right_x;  }
mp_number mp_knot_right_y(MP mp, mp_knot p) { return p->right_y;  }
int mp_knot_right_type(MP mp, mp_knot p) { return mp_right_type(p);}
int mp_knot_left_type (MP mp, mp_knot p) { return mp_left_type(p);}
mp_knot mp_knot_next (MP mp, mp_knot p)  { return p->next; }
double mp_number_as_double(MP mp, mp_number n) {
  return number_to_double(n);
}

@ @<Exported function headers@>=
#define mp_knot_left_curl mp_knot_left_x
#define mp_knot_left_given mp_knot_left_x
#define mp_knot_left_tension mp_knot_left_y
#define mp_knot_right_curl mp_knot_right_x
#define mp_knot_right_given mp_knot_right_x
#define mp_knot_right_tension mp_knot_right_y
mp_number mp_knot_x_coord(MP mp, mp_knot p);
mp_number mp_knot_y_coord(MP mp, mp_knot p);
mp_number mp_knot_left_x(MP mp, mp_knot p);
mp_number mp_knot_left_y(MP mp, mp_knot p);
mp_number mp_knot_right_x(MP mp, mp_knot p);
mp_number mp_knot_right_y(MP mp, mp_knot p);
int mp_knot_right_type(MP mp, mp_knot p);
int mp_knot_left_type(MP mp, mp_knot p);
mp_knot mp_knot_next(MP mp, mp_knot p);
double mp_number_as_double(MP mp, mp_number n);


@* Measuring paths.
\MP's \&{llcorner}, \&{lrcorner}, \&{ulcorner}, and \&{urcorner} operators
allow the user to measure the bounding box of anything that can go into a
picture.  It's easy to get rough bounds on the $x$ and $y$ extent of a path
by just finding the bounding box of the knots and the control points. We
need a more accurate version of the bounding box, but we can still use the
easy estimate to save time by focusing on the interesting parts of the path.

@ Computing an accurate bounding box involves a theme that will come up again
and again. Given a Bernshte{\u\i}n polynomial
@^Bernshte{\u\i}n, Serge{\u\i} Natanovich@>
$$B(z_0,z_1,\ldots,z_n;t)=\sum_k{n\choose k}t^k(1-t)^{n-k}z_k,$$
we can conveniently bisect its range as follows:

\smallskip
\textindent{1)} Let $z_k^{(0)}=z_k$, for |0<=k<=n|.

\smallskip
\textindent{2)} Let $z_k^{(j+1)}={1\over2}(z_k^{(j)}+z\k^{(j)})$, for
|0<=k<n-j|, for |0<=j<n|.

\smallskip\noindent
Then
$$B(z_0,z_1,\ldots,z_n;t)=B(z_0^{(0)},z_0^{(1)},\ldots,z_0^{(n)};2t)
 =B(z_0^{(n)},z_1^{(n-1)},\ldots,z_n^{(0)};2t-1).$$
This formula gives us the coefficients of polynomials to use over the ranges
$0\L t\L{1\over2}$ and ${1\over2}\L t\L1$.

@ Here is a routine that computes the $x$ or $y$ coordinate of the point on
a cubic corresponding to the |fraction| value~|t|.

@c
static void mp_eval_cubic (MP mp, mp_number *r, mp_knot p, mp_knot q, quarterword c,
                             mp_number t) {
  mp_number x1, x2, x3;    /* intermediate values */
  new_number(x1);
  new_number(x2);
  new_number(x3);
  if (c == mp_x_code) {
    set_number_from_of_the_way(x1, t, p->x_coord, p->right_x);
    set_number_from_of_the_way(x2, t, p->right_x, q->left_x);
    set_number_from_of_the_way(x3, t, q->left_x, q->x_coord);
  } else {
    set_number_from_of_the_way(x1, t, p->y_coord, p->right_y);
    set_number_from_of_the_way(x2, t, p->right_y, q->left_y);
    set_number_from_of_the_way(x3, t, q->left_y, q->y_coord);
  }
  set_number_from_of_the_way(x1, t, x1, x2);
  set_number_from_of_the_way(x2, t, x2, x3);
  set_number_from_of_the_way(*r, t, x1, x2);
  free_number (x1);
  free_number (x2);
  free_number (x3);
}


@ The actual bounding box information is stored in global variables.
Since it is convenient to address the $x$ and $y$ information
separately, we define arrays indexed by |x_code..y_code| and use
macros to give them more convenient names.

@<Types...@>=
enum mp_bb_code {
  mp_x_code = 0,        /* index for |minx| and |maxx| */
  mp_y_code                     /* index for |miny| and |maxy| */
};

@ 
@d mp_minx mp->bbmin[mp_x_code]
@d mp_maxx mp->bbmax[mp_x_code]
@d mp_miny mp->bbmin[mp_y_code]
@d mp_maxy mp->bbmax[mp_y_code]

@<Glob...@>=
mp_number bbmin[mp_y_code + 1];
mp_number bbmax[mp_y_code + 1];
/* the result of procedures that compute bounding box information */

@ @<Initialize table ...@>=
{ 
  int i;
  for (i=0;i<=mp_y_code;i++) {
    new_number(mp->bbmin[i]);
    new_number(mp->bbmax[i]);
  }
}

@ @<Dealloc...@>=
{ 
  int i;
  for (i=0;i<=mp_y_code;i++) {
    free_number(mp->bbmin[i]);
    free_number(mp->bbmax[i]);
  }
}


@ Now we're ready for the key part of the bounding box computation.
The |bound_cubic| procedure updates |bbmin[c]| and |bbmax[c]| based on
$$B(\hbox{|knot_coord(p)|}, \hbox{|right_coord(p)|},
    \hbox{|left_coord(q)|}, \hbox{|knot_coord(q)|};t)
$$
for $0<t\le1$.  In other words, the procedure adjusts the bounds to
accommodate |knot_coord(q)| and any extremes over the range $0<t<1$.
The |c| parameter is |x_code| or |y_code|.

@c
static void mp_bound_cubic (MP mp, mp_knot p, mp_knot q, quarterword c) {
  boolean wavy; /* whether we need to look for extremes */
  mp_number del1, del2, del3, del, dmax;  /* proportional to the control
                         points of a quadratic derived from a cubic */
  mp_number t, tt;       /* where a quadratic crosses zero */
  mp_number x;     /* a value that |bbmin[c]| and |bbmax[c]| must accommodate */
  new_number (x);
  new_fraction (t);
  new_fraction (tt);
  if (c == mp_x_code) {
    number_clone(x, q->x_coord);
  } else {
    number_clone(x, q->y_coord);
  }
  new_number(del1);
  new_number(del2);
  new_number(del3);
  new_number(del);
  new_number(dmax);
  @<Adjust |bbmin[c]| and |bbmax[c]| to accommodate |x|@>;
  @<Check the control points against the bounding box and set |wavy:=true|
    if any of them lie outside@>;
  if (wavy) {
    if (c == mp_x_code) {
      set_number_from_substraction(del1, p->right_x, p->x_coord);
      set_number_from_substraction(del2, q->left_x, p->right_x);
      set_number_from_substraction(del3, q->x_coord, q->left_x);
    } else {
      set_number_from_substraction(del1, p->right_y, p->y_coord);
      set_number_from_substraction(del2, q->left_y, p->right_y);
      set_number_from_substraction(del3, q->y_coord, q->left_y);
    }
    @<Scale up |del1|, |del2|, and |del3| for greater accuracy;
      also set |del| to the first nonzero element of |(del1,del2,del3)|@>;
    if (number_negative(del)) {
      number_negate (del1);
      number_negate (del2);
      number_negate (del3);
    }
    crossing_point (t, del1, del2, del3);
    if (number_less(t, fraction_one_t)) {
      @<Test the extremes of the cubic against the bounding box@>;
    }
  }
  free_number (del3);
  free_number (del2);
  free_number (del1);
  free_number (del);
  free_number (dmax);
  free_number (x);
  free_number (t);
  free_number (tt);
}


@ @<Adjust |bbmin[c]| and |bbmax[c]| to accommodate |x|@>=
if (number_less(x, mp->bbmin[c]))
  number_clone(mp->bbmin[c], x);
if (number_greater(x, mp->bbmax[c]))
  number_clone(mp->bbmax[c], x)

@ @<Check the control points against the bounding box and set...@>=
wavy = true;
if (c == mp_x_code) {
  if (number_lessequal(mp->bbmin[c], p->right_x))
    if (number_lessequal (p->right_x, mp->bbmax[c]))
      if (number_lessequal(mp->bbmin[c], q->left_x))
        if (number_lessequal (q->left_x, mp->bbmax[c]))
          wavy = false;
} else {
  if (number_lessequal(mp->bbmin[c], p->right_y))
    if (number_lessequal (p->right_y, mp->bbmax[c]))
      if (number_lessequal(mp->bbmin[c], q->left_y))
        if (number_lessequal (q->left_y, mp->bbmax[c]))
          wavy = false;
}


@ If |del1=del2=del3=0|, it's impossible to obey the title of this
section. We just set |del=0| in that case.

@<Scale up |del1|, |del2|, and |del3| for greater accuracy...@>=
if (number_nonzero(del1)) {
  number_clone (del, del1);
} else if (number_nonzero(del2)) {
  number_clone (del, del2);
} else {
  number_clone (del, del3);
}
if (number_nonzero(del)) {
  mp_number absval1; 
  new_number(absval1);
  number_clone (dmax, del1);
  number_abs (dmax);
  number_clone (absval1, del2);
  number_abs(absval1);
  if (number_greater(absval1, dmax)) {
    number_clone(dmax, absval1);
  }
  number_clone (absval1, del3);
  number_abs(absval1);
  if (number_greater(absval1, dmax)) {
    number_clone(dmax, absval1);
  }
  while (number_less(dmax, fraction_half_t)) {
    number_double(dmax);
    number_double(del1);
    number_double(del2);
    number_double(del3);
  }
  free_number (absval1);
}

@ Since |crossing_point| has tried to choose |t| so that
$B(|del1|,|del2|,|del3|;\tau)$ crosses zero at $\tau=|t|$ with negative
slope, the value of |del2| computed below should not be positive.
But rounding error could make it slightly positive in which case we
must cut it to zero to avoid confusion.

@<Test the extremes of the cubic against the bounding box@>=
{
  mp_eval_cubic (mp, &x, p, q, c, t);
  @<Adjust |bbmin[c]| and |bbmax[c]| to accommodate |x|@>;
  set_number_from_of_the_way(del2, t, del2, del3);
  /* now |0,del2,del3| represent the derivative on the remaining interval */
  if (number_positive(del2))
    set_number_to_zero(del2);
  {
    mp_number arg2, arg3;
    new_number(arg2);
    new_number(arg3);
    number_clone(arg2, del2);
    number_negate(arg2);
    number_clone(arg3, del3);
    number_negate(arg3);
    crossing_point (tt, zero_t, arg2, arg3);
    free_number (arg2);
    free_number (arg3);
  }
  if (number_less(tt, fraction_one_t)) {
    @<Test the second extreme against the bounding box@>;
  }
}


@ @<Test the second extreme against the bounding box@>=
{
  mp_number arg;
  new_number (arg);
  set_number_from_of_the_way (arg, t, tt, fraction_one_t);
  mp_eval_cubic (mp, &x, p, q, c, arg);
  free_number (arg);
  @<Adjust |bbmin[c]| and |bbmax[c]| to accommodate |x|@>;
}


@ Finding the bounding box of a path is basically a matter of applying
|bound_cubic| twice for each pair of adjacent knots.

@c
static void mp_path_bbox (MP mp, mp_knot h) {
  mp_knot p, q; /* a pair of adjacent knots */
  number_clone(mp_minx, h->x_coord);
  number_clone(mp_miny, h->y_coord);
  number_clone (mp_maxx, mp_minx);
  number_clone (mp_maxy, mp_miny);
  p = h;
  do {
    if (mp_right_type (p) == mp_endpoint)
      return;
    q = mp_next_knot (p);
    mp_bound_cubic (mp, p, q, mp_x_code);
    mp_bound_cubic (mp, p, q, mp_y_code);
    p = q;
  } while (p != h);
}


@ Another important way to measure a path is to find its arc length.  This
is best done by using the general bisection algorithm to subdivide the path
until obtaining ``well behaved'' subpaths whose arc lengths can be approximated
by simple means.

Since the arc length is the integral with respect to time of the magnitude of
the velocity, it is natural to use Simpson's rule for the approximation.
@^Simpson's rule@>
If $\dot B(t)$ is the spline velocity, Simpson's rule gives
$$ \vb\dot B(0)\vb + 4\vb\dot B({1\over2})\vb + \vb\dot B(1)\vb \over 6 $$
for the arc length of a path of length~1.  For a cubic spline
$B(z_0,z_1,z_2,z_3;t)$, the time derivative $\dot B(t)$ is
$3B(dz_0,dz_1,dz_2;t)$, where $dz_i=z_{i+1}-z_i$.  Hence the arc length
approximation is
$$ {\vb dz_0\vb \over 2} + 2\vb dz_{02}\vb + {\vb dz_2\vb \over 2}, $$
where
$$ dz_{02}={1\over2}\left({dz_0+dz_1\over 2}+{dz_1+dz_2\over 2}\right)$$
is the result of the bisection algorithm.

@ The remaining problem is how to decide when a subpath is ``well behaved.''
This could be done via the theoretical error bound for Simpson's rule,
@^Simpson's rule@>
but this is impractical because it requires an estimate of the fourth
derivative of the quantity being integrated.  It is much easier to just perform
a bisection step and see how much the arc length estimate changes.  Since the
error for Simpson's rule is proportional to the fourth power of the sample
spacing, the remaining error is typically about $1\over16$ of the amount of
the change.  We say ``typically'' because the error has a pseudo-random behavior
that could cause the two estimates to agree when each contain large errors.

To protect against disasters such as undetected cusps, the bisection process
should always continue until all the $dz_i$ vectors belong to a single
$90^\circ$ sector.  This ensures that no point on the spline can have velocity
less than 70\% of the minimum of $\vb dz_0\vb$, $\vb dz_1\vb$ and $\vb dz_2\vb$.
If such a spline happens to produce an erroneous arc length estimate that
is little changed by bisection, the amount of the error is likely to be fairly
small.  We will try to arrange things so that freak accidents of this type do
not destroy the inverse relationship between the \&{arclength} and
\&{arctime} operations.
@:arclength_}{\&{arclength} primitive@>
@:arctime_}{\&{arctime} primitive@>

@ The \&{arclength} and \&{arctime} operations are both based on a recursive
@^recursion@>
function that finds the arc length of a cubic spline given $dz_0$, $dz_1$,
$dz_2$. This |arc_test| routine also takes an arc length goal |a_goal| and
returns the time when the arc length reaches |a_goal| if there is such a time.
Thus the return value is either an arc length less than |a_goal| or, if the
arc length would be at least |a_goal|, it returns a time value decreased by
|two|.  This allows the caller to use the sign of the result to distinguish
between arc lengths and time values.  On certain types of overflow, it is
possible for |a_goal| and the result of |arc_test| both to be |EL_GORDO|.
Otherwise, the result is always less than |a_goal|.

Rather than halving the control point coordinates on each recursive call to
|arc_test|, it is better to keep them proportional to velocity on the original
curve and halve the results instead.  This means that recursive calls can
potentially use larger error tolerances in their arc length estimates.  How
much larger depends on to what extent the errors behave as though they are
independent of each other.  To save computing time, we use optimistic assumptions
and increase the tolerance by a factor of about $\sqrt2$ for each recursive
call.

In addition to the tolerance parameter, |arc_test| should also have parameters
for ${1\over3}\vb\dot B(0)\vb$, ${2\over3}\vb\dot B({1\over2})\vb$, and
${1\over3}\vb\dot B(1)\vb$.  These quantities are relatively expensive to compute
and they are needed in different instances of |arc_test|.

@c
static void mp_arc_test (MP mp, mp_number *ret, mp_number dx0, mp_number dy0, mp_number dx1,
                           mp_number dy1, mp_number dx2, mp_number dy2, mp_number v0,
                           mp_number v02, mp_number v2, mp_number a_goal, mp_number tol_orig) {
  boolean simple;       /* are the control points confined to a $90^\circ$ sector? */
  mp_number dx01, dy01, dx12, dy12, dx02, dy02;    /* bisection results */
  mp_number v002, v022; /* twice the velocity magnitudes at $t={1\over4}$ and $t={3\over4}$ */
  mp_number arc;   /* best arc length estimate before recursion */
  mp_number arc1;    /* arc length estimate for the first half */
  mp_number simply; 
  mp_number tol; 
  new_number (arc );
  new_number (arc1);
  new_number (dx01);
  new_number (dy01);
  new_number (dx12); 
  new_number (dy12);
  new_number (dx02);
  new_number (dy02);
  new_number (v002);
  new_number (v022);
  new_number (simply);
  new_number (tol);
  number_clone(tol, tol_orig);
  @<Bisect the B\'ezier quadratic given by |dx0|, |dy0|, |dx1|, |dy1|,
    |dx2|, |dy2|@>;
  @<Initialize |v002|, |v022|, and the arc length estimate |arc|; if it overflows
    set |arc_test| and |return|@>;
  @<Test if the control points are confined to one quadrant or rotating them
    $45^\circ$ would put them in one quadrant.  Then set |simple| appropriately@>;

  set_number_from_addition(simply, v0, v2);
  number_halfp (simply);
  number_negate (simply);
  number_add (simply, arc);
  number_substract (simply, v02);
  number_abs (simply);

 if (simple && number_lessequal(simply, tol)) {
    if (number_less(arc, a_goal)){
      number_clone(*ret, arc);
    } else {
      @<Estimate when the arc length reaches |a_goal| and set |arc_test| to
         that time minus |two|@>;
    }
  } else {
    @<Use one or two recursive calls to compute the |arc_test| function@>;
  }
DONE:
  free_number (arc);
  free_number (arc1);
  free_number (dx01);
  free_number (dy01);
  free_number (dx12); 
  free_number (dy12);
  free_number (dx02);
  free_number (dy02);
  free_number (v002);
  free_number (v022);
  free_number (simply);
  free_number (tol);
}


@ The |tol| value should by multiplied by $\sqrt 2$ before making recursive
calls, but $1.5$ is an adequate approximation.  It is best to avoid using
|make_fraction| in this inner loop.
@^inner loop@>

@<Use one or two recursive calls to compute the |arc_test| function@>=
{
  mp_number a_new, a_aux;    /* the sum of these gives the |a_goal| */
  mp_number a, b;    /* results of recursive calls */
  mp_number half_v02; /* |halfp(v02)|, a recursion argument */
  new_number(a_new);
  new_number(a_aux);
  new_number(half_v02);
  @<Set |a_new| and |a_aux| so their sum is |2*a_goal| and |a_new| is as
    large as possible@>;
  {
    mp_number halfp_tol;
    new_number(halfp_tol);
    number_clone (halfp_tol, tol);
    number_halfp (halfp_tol);
    number_add(tol, halfp_tol);
    free_number (halfp_tol);
  }
  number_clone(half_v02, v02);
  number_halfp(half_v02);
  new_number (a);
  mp_arc_test (mp, &a, dx0, dy0, dx01, dy01, dx02, dy02, 
                              v0, v002, half_v02, a_new, tol);
  if (number_negative(a)) {
    set_number_to_unity(*ret);
    number_double(*ret); /* two */
    number_substract(*ret, a); /* two - a */
    number_halfp(*ret);
    number_negate(*ret); /* -halfp(two - a) */
  } else {
    @<Update |a_new| to reduce |a_new+a_aux| by |a|@>;
    new_number (b);
    mp_arc_test (mp, &b, dx02, dy02, dx12, dy12, dx2, dy2,
                         half_v02, v022, v2, a_new, tol);
    if (number_negative(b)) {
      mp_number tmp ;
      new_number (tmp);
      number_clone(tmp, b);
      number_negate(tmp);
      number_halfp(tmp);
      number_negate(tmp);
      number_clone(*ret, tmp);
      set_number_to_unity(tmp);
      number_halfp(tmp);
      number_substract(*ret, tmp); /* (-(halfp(-b)) - 1/2) */
      free_number (tmp);
    } else {
      set_number_from_substraction(*ret, b, a);
      number_half(*ret);
      set_number_from_addition(*ret, a, *ret); /* (a + half(b - a)) */
    }
    free_number (b);
  }
  free_number (half_v02);
  free_number (a_aux);
  free_number (a_new);
  free_number (a);
  goto DONE;
}


@ @<Set |a_new| and |a_aux| so their sum is |2*a_goal| and |a_new| is...@>=
set_number_to_inf(a_aux);
number_substract(a_aux, a_goal);
if (number_greater(a_goal, a_aux)) {
  set_number_from_substraction(a_aux, a_goal, a_aux);
  set_number_to_inf(a_new);
} else {
  set_number_from_addition(a_new, a_goal, a_goal);
  set_number_to_zero(a_aux);
}


@ There is no need to maintain |a_aux| at this point so we use it as a temporary
to force the additions and subtractions to be done in an order that avoids
overflow.

@<Update |a_new| to reduce |a_new+a_aux| by |a|@>=
if (number_greater(a, a_aux)) {
  number_substract(a_aux, a);
  number_add(a_new, a_aux);
}

@ This code assumes all {\it dx} and {\it dy} variables have magnitude less than
|fraction_four|.  To simplify the rest of the |arc_test| routine, we strengthen
this assumption by requiring the norm of each $({\it dx},{\it dy})$ pair to obey
this bound.  Note that recursive calls will maintain this invariant.

@<Bisect the B\'ezier quadratic given by |dx0|, |dy0|, |dx1|, |dy1|,...@>=
set_number_from_addition(dx01, dx0, dx1);
number_half(dx01);
set_number_from_addition(dx12, dx1, dx2);
number_half(dx12);
set_number_from_addition(dx02, dx01, dx12);
number_half(dx02);
set_number_from_addition(dy01, dy0, dy1);
number_half(dy01);
set_number_from_addition(dy12, dy1, dy2);
number_half(dy12);
set_number_from_addition(dy02, dy01, dy12);
number_half(dy02);

@ We should be careful to keep |arc<EL_GORDO| so that calling |arc_test| with
|a_goal=EL_GORDO| is guaranteed to yield the arc length.

@<Initialize |v002|, |v022|, and the arc length estimate |arc|;...@>=
{
  mp_number tmp, arg1, arg2 ;
  new_number (tmp);
  new_number (arg1);
  new_number (arg2);
  set_number_from_addition(arg1, dx0, dx02);
  number_half(arg1);
  number_add(arg1, dx01);
  set_number_from_addition(arg2, dy0, dy02);
  number_half(arg2);
  number_add(arg2, dy01);
  pyth_add (v002, arg1, arg2);

  set_number_from_addition(arg1, dx02, dx2);
  number_half(arg1);
  number_add(arg1, dx12);
  set_number_from_addition(arg2, dy02, dy2);
  number_half(arg2);
  number_add(arg2, dy12);
  pyth_add (v022, arg1, arg2);
  free_number(arg1);
  free_number(arg2);

  number_clone (tmp, v02);
  number_add_scaled (tmp, 2);
  number_halfp (tmp);

  set_number_from_addition(arc1, v0, tmp);
  number_halfp (arc1);
  number_substract (arc1, v002);
  number_half (arc1);
  set_number_from_addition(arc1, v002, arc1);

  set_number_from_addition(arc, v2, tmp);
  number_halfp (arc);
  number_substract (arc, v022);
  number_half (arc);
  set_number_from_addition(arc, v022, arc);

  /* reuse |tmp| for the next |if| test: */
  set_number_to_inf(tmp);
  number_substract(tmp,arc1);
  if (number_less(arc, tmp)) {
    free_number (tmp);
    number_add(arc, arc1);
  } else {
    free_number (tmp);
    mp->arith_error = true;
    if (number_infinite(a_goal)) {
      set_number_to_inf(*ret);
    } else {
      set_number_to_unity(*ret);
      number_double(*ret);
      number_negate(*ret); /* -two */
    }
    goto DONE;
  }
}


@ @<Test if the control points are confined to one quadrant or rotating...@>=
simple = ((number_nonnegative(dx0) && number_nonnegative(dx1) && number_nonnegative(dx2)) ||
          (number_nonpositive(dx0) && number_nonpositive(dx1) && number_nonpositive(dx2)));
if (simple) {
  simple = (number_nonnegative(dy0) && number_nonnegative(dy1) && number_nonnegative(dy2)) ||
           (number_nonpositive(dy0) && number_nonpositive(dy1) && number_nonpositive(dy2));
}
if (!simple) {
  simple = (number_greaterequal(dx0, dy0) && number_greaterequal(dx1, dy1) && number_greaterequal(dx2, dy2)) ||
           (number_lessequal(dx0, dy0) && number_lessequal(dx1, dy1) && number_lessequal(dx2, dy2));
  if (simple) {
    mp_number neg_dx0, neg_dx1, neg_dx2;
    new_number(neg_dx0);
    new_number(neg_dx1);
    new_number(neg_dx2);
    number_clone(neg_dx0, dx0);
    number_clone(neg_dx1, dx1);
    number_clone(neg_dx2, dx2);
    number_negate(neg_dx0);
    number_negate(neg_dx1);
    number_negate(neg_dx2);
    simple = 
      (number_greaterequal(neg_dx0, dy0) && number_greaterequal(neg_dx1, dy1) && number_greaterequal(neg_dx2, dy2)) ||
      (number_lessequal(neg_dx0, dy0) && number_lessequal(neg_dx1, dy1) && number_lessequal(neg_dx2, dy2));
    free_number (neg_dx0);
    free_number (neg_dx1);
    free_number (neg_dx2);
  }
}

@ Since Simpson's rule is based on approximating the integrand by a parabola,
@^Simpson's rule@>
it is appropriate to use the same approximation to decide when the integral
reaches the intermediate value |a_goal|.  At this point
$$\eqalign{
    {\vb\dot B(0)\vb\over 3} &= \hbox{|v0|}, \qquad
    {\vb\dot B({1\over4})\vb\over 3} = {\hbox{|v002|}\over 2}, \qquad
    {\vb\dot B({1\over2})\vb\over 3} = {\hbox{|v02|}\over 2}, \cr
    {\vb\dot B({3\over4})\vb\over 3} &= {\hbox{|v022|}\over 2}, \qquad
    {\vb\dot B(1)\vb\over 3} = \hbox{|v2|} \cr
}
$$
and
$$ {\vb\dot B(t)\vb\over 3} \approx
  \cases{B\left(\hbox{|v0|},
      \hbox{|v002|}-{1\over 2}\hbox{|v0|}-{1\over 4}\hbox{|v02|},
      {1\over 2}\hbox{|v02|}; 2t \right)&
    if $t\le{1\over 2}$\cr
  B\left({1\over 2}\hbox{|v02|},
      \hbox{|v022|}-{1\over 4}\hbox{|v02|}-{1\over 2}\hbox{|v2|},
      \hbox{|v2|}; 2t-1 \right)&
    if $t\ge{1\over 2}$.\cr}
 \eqno (*)
$$
We can integrate $\vb\dot B(t)\vb$ by using
$$\int 3B(a,b,c;\tau)\,dt =
  {B(0,a,a+b,a+b+c;\tau) + {\rm constant} \over {d\tau\over dt}}.
$$

This construction allows us to find the time when the arc length reaches
|a_goal| by solving a cubic equation of the form
$$ B(0,a,a+b,a+b+c;\tau) = x, $$
where $\tau$ is $2t$ or $2t+1$, $x$ is |a_goal| or |a_goal-arc1|, and $a$, $b$,
and $c$ are the Bernshte{\u\i}n coefficients from $(*)$ divided by
@^Bernshte{\u\i}n, Serge{\u\i} Natanovich@>
$d\tau\over dt$.  We shall define a function |solve_rising_cubic| that finds
$\tau$ given $a$, $b$, $c$, and $x$.

@<Estimate when the arc length reaches |a_goal| and set |arc_test| to...@>=
{
  mp_number tmp;
  mp_number tmp2;
  mp_number tmp3;
  mp_number tmp4;  
  mp_number tmp5;  
  new_number (tmp); 
  new_number (tmp2);
  new_number (tmp3);
  new_number (tmp4);
  new_number (tmp5);
  number_clone(tmp, v02);
  number_add_scaled(tmp, 2);
  number_half(tmp);
  number_half(tmp); /* (v02+2) / 4 */
  if (number_lessequal(a_goal, arc1)) {
    number_clone(tmp2, v0);
    number_halfp(tmp2);
    set_number_from_substraction(tmp3, arc1, tmp2);
    number_substract(tmp3, tmp);
    mp_solve_rising_cubic (mp, &tmp5, tmp2, tmp3, tmp, a_goal);
    number_halfp (tmp5);
    set_number_to_unity(tmp3);
    number_substract(tmp5, tmp3);
    number_substract(tmp5, tmp3);
    number_clone(*ret, tmp5);
  } else {
    number_clone(tmp2, v2);
    number_halfp(tmp2);
    set_number_from_substraction(tmp3, arc, arc1);
    number_substract(tmp3, tmp);
    number_substract(tmp3, tmp2);
    set_number_from_substraction(tmp4, a_goal, arc1);
    mp_solve_rising_cubic (mp, &tmp5, tmp, tmp3, tmp2, tmp4);
    number_halfp(tmp5);
    set_number_to_unity(tmp2);
    set_number_to_unity(tmp3);
    number_half(tmp2);
    number_substract(tmp2, tmp3);
    number_substract(tmp2, tmp3);
    set_number_from_addition(*ret, tmp2, tmp5);
  }
  free_number (tmp);
  free_number (tmp2);
  free_number (tmp3);
  free_number (tmp4);
  free_number (tmp5);
  goto DONE;
}


@ Here is the |solve_rising_cubic| routine that finds the time~$t$ when
$$ B(0, a, a+b, a+b+c; t) = x. $$
This routine is based on |crossing_point| but is simplified by the
assumptions that $B(a,b,c;t)\ge0$ for $0\le t\le1$ and that |0<=x<=a+b+c|.
If rounding error causes this condition to be violated slightly, we just ignore
it and proceed with binary search.  This finds a time when the function value
reaches |x| and the slope is positive.

@<Declarations@>=
static void mp_solve_rising_cubic (MP mp, mp_number *ret, mp_number a, mp_number b, mp_number c, mp_number x);

@ @c
void mp_solve_rising_cubic (MP mp, mp_number *ret, mp_number a_orig, mp_number b_orig, mp_number c_orig, mp_number x_orig) {
  mp_number abc;
  mp_number a, b, c, x;      /* local versions of arguments */
  mp_number ab, bc, ac;    /* bisection results */
  mp_number t;    /* $2^k+q$ where unscaled answer is in $[q2^{-k},(q+1)2^{-k})$ */
  mp_number xx;   /* temporary for updating |x| */
  mp_number neg_x; /* temporary for an |if| */
  if (number_negative(a_orig) || number_negative(c_orig))
    mp_confusion (mp, "rising?");
@:this can't happen rising?}{\quad rising?@>;
  new_number (t); 
  new_number (abc);
  new_number (a);
  new_number (b);
  new_number (c);
  new_number (x);
  number_clone(a, a_orig);
  number_clone(b, b_orig);
  number_clone(c, c_orig);
  number_clone(x, x_orig);
  new_number (ab);
  new_number (bc);
  new_number (ac);
  new_number (xx);
  new_number (neg_x);
  set_number_from_addition(abc, a, b);
  number_add(abc, c);
  if (number_nonpositive(x)) {
    set_number_to_zero(*ret);
  } else if (number_greaterequal(x, abc)) {
    set_number_to_unity(*ret);
  } else {
    number_clone (t, epsilon_t);
    @<Rescale if necessary to make sure |a|, |b|, and |c| are all less than
      |EL_GORDO div 3|@>;
    do {
      number_add (t, t);
      @<Subdivide the B\'ezier quadratic defined by |a|, |b|, |c|@>;
      number_clone(xx,x);
      number_substract(xx, a);
      number_substract(xx, ab);
      number_substract(xx, ac);
      number_clone(neg_x, x);
      number_negate(neg_x);
      if (number_less(xx, neg_x)) {
        number_double(x);
        number_clone(b, ab);
        number_clone(c, ac);
      } else {
        number_add(x, xx);
        number_clone(a, ac);
        number_clone(b, bc);
        number_add (t, epsilon_t);
      }
    } while (number_less (t, unity_t));
    set_number_from_substraction(*ret, t, unity_t);
  }
  free_number (abc);
  free_number (t);
  free_number (a);
  free_number (b);
  free_number (c);
  free_number (ab);
  free_number (bc);
  free_number (ac);
  free_number (xx);
  free_number (x);
  free_number (neg_x);
}


@ @<Subdivide the B\'ezier quadratic defined by |a|, |b|, |c|@>=
set_number_from_addition(ab, a, b);
number_half(ab);
set_number_from_addition(bc, b, c);
number_half(bc);
set_number_from_addition(ac, ab, bc);
number_half(ac);

@ The upper bound on |a|, |b|, and |c|:

@d one_third_inf_t  ((math_data *)mp->math)->one_third_inf_t

@<Rescale if necessary to make sure |a|, |b|, and |c| are all less than...@>=
while (number_greater(a, one_third_inf_t) || 
       number_greater(b, one_third_inf_t) || 
       number_greater(c, one_third_inf_t)) {
  number_halfp(a);
  number_half(b);
  number_halfp(c);
  number_halfp(x);
}


@ It is convenient to have a simpler interface to |arc_test| that requires no
unnecessary arguments and ensures that each $({\it dx},{\it dy})$ pair has
length less than |fraction_four|.

@c
static void mp_do_arc_test (MP mp, mp_number *ret, mp_number dx0, mp_number dy0, mp_number dx1,
                              mp_number dy1, mp_number dx2, mp_number dy2, mp_number a_goal) {
  mp_number v0, v1, v2;    /* length of each $({\it dx},{\it dy})$ pair */
  mp_number v02;   /* twice the norm of the quadratic at $t={1\over2}$ */
  new_number (v0);
  new_number (v1);
  new_number (v2);
  pyth_add (v0, dx0, dy0);
  pyth_add (v1, dx1, dy1);
  pyth_add (v2, dx2, dy2);
  if ((number_greaterequal(v0, fraction_four_t)) || 
      (number_greaterequal(v1, fraction_four_t)) || 
      (number_greaterequal(v2, fraction_four_t))) {
    mp->arith_error = true;
    if (number_infinite(a_goal)) {
      set_number_to_inf(*ret);
    } else {
      set_number_to_unity(*ret);
      number_double(*ret);
      number_negate(*ret);
    }
  } else {
    mp_number arg1, arg2;
    new_number (v02);
    new_number (arg1);
    new_number (arg2);
    set_number_from_addition(arg1, dx0, dx2);
    number_half(arg1);
    number_add(arg1, dx1);
    set_number_from_addition(arg2, dy0, dy2);
    number_half(arg2);
    number_add(arg2, dy1);
    pyth_add (v02, arg1, arg2);
    free_number(arg1);
    free_number(arg2);
    mp_arc_test (mp, ret, dx0, dy0, dx1, dy1, dx2, dy2, v0, v02, v2, a_goal, arc_tol_k);
    free_number (v02);
  }
  free_number (v0);
  free_number (v1);
  free_number (v2);
}


@ Now it is easy to find the arc length of an entire path.

@c
static void mp_get_arc_length (MP mp, mp_number *ret, mp_knot h) {
  mp_knot p, q; /* for traversing the path */
  mp_number a;  /* current arc length */
  mp_number a_tot; /* total arc length */
  mp_number arg1, arg2, arg3, arg4, arg5, arg6;
  mp_number arcgoal; 
  p = h;
  new_number (a_tot);
  new_number (arg1);
  new_number (arg2);
  new_number (arg3);
  new_number (arg4);
  new_number (arg5);
  new_number (arg6);
  new_number (a);
  new_number(arcgoal);
  set_number_to_inf(arcgoal);
  while (mp_right_type (p) != mp_endpoint) {
    q = mp_next_knot (p);
    set_number_from_substraction(arg1, p->right_x, p->x_coord);
    set_number_from_substraction(arg2, p->right_y, p->y_coord);
    set_number_from_substraction(arg3, q->left_x,  p->right_x);
    set_number_from_substraction(arg4, q->left_y,  p->right_y);
    set_number_from_substraction(arg5, q->x_coord, q->left_x);
    set_number_from_substraction(arg6, q->y_coord, q->left_y);
    mp_do_arc_test (mp, &a, arg1, arg2, arg3, arg4, arg5, arg6, arcgoal);
    slow_add (a_tot, a, a_tot);
    if (q == h)
      break;
    else
      p = q;
  }
  free_number (arcgoal);
  free_number (a);
  free_number (arg1);
  free_number (arg2);
  free_number (arg3);
  free_number (arg4);
  free_number (arg5);
  free_number (arg6);
  check_arith();
  number_clone (*ret, a_tot);
  free_number (a_tot);
}


@ The inverse operation of finding the time on a path~|h| when the arc length
reaches some value |arc0| can also be accomplished via |do_arc_test|.  Some care
is required to handle very large times or negative times on cyclic paths.  For
non-cyclic paths, |arc0| values that are negative or too large cause
|get_arc_time| to return 0 or the length of path~|h|.

If |arc0| is greater than the arc length of a cyclic path~|h|, the result is a
time value greater than the length of the path.  Since it could be much greater,
we must be prepared to compute the arc length of path~|h| and divide this into
|arc0| to find how many multiples of the length of path~|h| to add.

@c
static void mp_get_arc_time (MP mp, mp_number *ret, mp_knot h, mp_number arc0_orig) {
  mp_knot p, q; /* for traversing the path */
  mp_number t_tot; /* accumulator for the result */
  mp_number t;     /* the result of |do_arc_test| */
  mp_number arc, arc0;   /* portion of |arc0| not used up so far */
  mp_number arg1, arg2, arg3, arg4, arg5, arg6; /* |do_arc_test| arguments */
  if (number_negative(arc0_orig)) {
    @<Deal with a negative |arc0_orig| value and |return|@>;
  }
  new_number (t_tot);
  new_number (arc0);
  number_clone(arc0, arc0_orig);
  if (number_infinite(arc0)) {
    number_add_scaled (arc0, -1);
  }
  new_number (arc);
  number_clone(arc, arc0);
  p = h;
  new_number (arg1);
  new_number (arg2);
  new_number (arg3);
  new_number (arg4);
  new_number (arg5);
  new_number (arg6);
  new_number (t);
  while ((mp_right_type (p) != mp_endpoint) && number_positive(arc)) {
    q = mp_next_knot (p);
    set_number_from_substraction(arg1, p->right_x, p->x_coord);
    set_number_from_substraction(arg2, p->right_y, p->y_coord);
    set_number_from_substraction(arg3, q->left_x,  p->right_x);
    set_number_from_substraction(arg4, q->left_y,  p->right_y);
    set_number_from_substraction(arg5, q->x_coord, q->left_x);
    set_number_from_substraction(arg6, q->y_coord, q->left_y);
    mp_do_arc_test (mp, &t, arg1, arg2, arg3, arg4, arg5, arg6, arc);
    @<Update |arc| and |t_tot| after |do_arc_test| has just returned |t|@>;
    if (q == h) {
      @<Update |t_tot| and |arc| to avoid going around the cyclic
        path too many times but set |arith_error:=true| and |goto done| on
        overflow@>;
    }
    p = q;
  }
  check_arith();
  number_clone (*ret, t_tot);
RETURN:
  free_number (t_tot);
  free_number (t);
  free_number (arc);
  free_number (arc0);
  free_number (arg1);
  free_number (arg2);
  free_number (arg3);
  free_number (arg4);
  free_number (arg5);
  free_number (arg6);
}


@ @<Update |arc| and |t_tot| after |do_arc_test| has just returned |t|@>=
if (number_negative(t)) {
  number_add (t_tot, t);
  number_add (t_tot, two_t);
  set_number_to_zero(arc);
} else {
  number_add (t_tot, unity_t);
  number_substract(arc, t);
}


@ @<Deal with a negative |arc0_orig| value and |return|@>=
{
  if (mp_left_type (h) == mp_endpoint) {
    set_number_to_zero (*ret);
  } else {
    mp_number neg_arc0;
    p = mp_htap_ypoc (mp, h);
    new_number(neg_arc0);
    number_clone(neg_arc0, arc0_orig);
    number_negate(neg_arc0);
    mp_get_arc_time (mp, ret, p, neg_arc0);
    number_negate(*ret);
    mp_toss_knot_list (mp, p);
    free_number (neg_arc0);
  }
  check_arith();
  return;
}


@ @<Update |t_tot| and |arc| to avoid going around the cyclic...@>=
if (number_positive(arc)) {
  mp_number n, n1, d1, v1;
  new_number (n);
  new_number (n1);
  new_number (d1);
  new_number (v1);

  set_number_from_substraction (d1, arc0, arc); /* d1 = arc0 - arc */
  set_number_from_div (n1, arc, d1); /* n1 = (arc / d1) */
  number_clone (n, n1);
  set_number_from_mul (n1, n1, d1); /* n1 = (n1 * d1) */
  number_substract (arc, n1); /* arc = arc - n1 */

  number_clone (d1, inf_t);         /* reuse d1 */
  number_clone (v1, n);             /* v1 = n */
  number_add (v1, epsilon_t);       /* v1 = n1+1 */
  set_number_from_div (d1, d1, v1); /* |d1 = EL_GORDO / v1| */
  if (number_greater (t_tot, d1)) {
    mp->arith_error = true;
    check_arith();
    set_number_to_inf(*ret);
    free_number (n);
    free_number (n1);
    free_number (d1);
    free_number (v1);
    goto RETURN;
  }
  set_number_from_mul (t_tot, t_tot, v1);
  free_number (n);
  free_number (n1);
  free_number (d1);
  free_number (v1);
}

@* Data structures for pens.
A Pen in \MP\ can be either elliptical or polygonal.  Elliptical pens result
in \ps\ \&{stroke} commands, while anything drawn with a polygonal pen is
@:stroke}{\&{stroke} command@>
converted into an area fill as described in the next part of this program.
The mathematics behind this process is based on simple aspects of the theory
of tracings developed by Leo Guibas, Lyle Ramshaw, and Jorge Stolfi
[``A kinematic framework for computational geometry,'' Proc.\ IEEE Symp.\
Foundations of Computer Science {\bf 24} (1983), 100--111].

Polygonal pens are created from paths via \MP's \&{makepen} primitive.
@:makepen_}{\&{makepen} primitive@>
This path representation is almost sufficient for our purposes except that
a pen path should always be a convex polygon with the vertices in
counter-clockwise order.
Since we will need to scan pen polygons both forward and backward, a pen
should be represented as a doubly linked ring of knot nodes.  There is
room for the extra back pointer because we do not need the
|mp_left_type| or |mp_right_type| fields.  In fact, we don't need the |left_x|,
|left_y|, |right_x|, or |right_y| fields either but we leave these alone
so that certain procedures can operate on both pens and paths.  In particular,
pens can be copied using |copy_path| and recycled using |toss_knot_list|.

@ The |make_pen| procedure turns a path into a pen by initializing
the |prev_knot| pointers and making sure the knots form a convex polygon.
Thus each cubic in the given path becomes a straight line and the control
points are ignored.  If the path is not cyclic, the ends are connected by a
straight line.

@d copy_pen(A) mp_make_pen(mp, mp_copy_path(mp, (A)),false)

@c
static mp_knot mp_make_pen (MP mp, mp_knot h, boolean need_hull) {
  mp_knot p, q; /* two consecutive knots */
  q = h;
  do {
    p = q;
    q = mp_next_knot (q);
    mp_prev_knot (q) = p;
  } while (q != h);
  if (need_hull) {
    h = mp_convex_hull (mp, h);
    @<Make sure |h| isn't confused with an elliptical pen@>;
  }
  return h;
}


@ The only information required about an elliptical pen is the overall
transformation that has been applied to the original \&{pencircle}.
@:pencircle_}{\&{pencircle} primitive@>
Since it suffices to keep track of how the three points $(0,0)$, $(1,0)$,
and $(0,1)$ are transformed, an elliptical pen can be stored in a single
knot node and transformed as if it were a path.

@d pen_is_elliptical(A) ((A)==mp_next_knot((A)))

@c
static mp_knot mp_get_pen_circle (MP mp, mp_number diam) {
  mp_knot h;    /* the knot node to return */
  h = mp_new_knot (mp);
  mp_next_knot (h) = h;
  mp_prev_knot (h) = h;
  mp_originator (h) = mp_program_code;
  set_number_to_zero(h->x_coord);
  set_number_to_zero(h->y_coord);
  number_clone(h->left_x, diam);
  set_number_to_zero(h->left_y);
  set_number_to_zero(h->right_x);
  number_clone(h->right_y, diam);
  return h;
}


@ If the polygon being returned by |make_pen| has only one vertex, it will
be interpreted as an elliptical pen.  This is no problem since a degenerate
polygon can equally well be thought of as a degenerate ellipse.  We need only
initialize the |left_x|, |left_y|, |right_x|, and |right_y| fields.

@<Make sure |h| isn't confused with an elliptical pen@>=
if (pen_is_elliptical (h)) {
  number_clone(h->left_x, h->x_coord);
  number_clone(h->left_y, h->y_coord);
  number_clone(h->right_x, h->x_coord);
  number_clone(h->right_y, h->y_coord);
}

@ Printing a polygonal pen is very much like printing a path

@<Declarations@>=
static void mp_pr_pen (MP mp, mp_knot h);

@ @c
void mp_pr_pen (MP mp, mp_knot h) {
  mp_knot p, q; /* for list traversal */
  if (pen_is_elliptical (h)) {
    @<Print the elliptical pen |h|@>;
  } else {
    p = h;
    do {
      mp_print_two (mp, p->x_coord, p->y_coord);
      mp_print_nl (mp, " .. ");
      @<Advance |p| making sure the links are OK and |return| if there is
        a problem@>;
    } while (p != h);
    mp_print (mp, "cycle");
  }
}


@ @<Advance |p| making sure the links are OK and |return| if there is...@>=
q = mp_next_knot (p);
if ((q == NULL) || (mp_prev_knot (q) != p)) {
  mp_print_nl (mp, "???");
  return;                       /* this won't happen */
@.???@>
}
p = q

@ @<Print the elliptical pen |h|@>=
{
  mp_number v1;
  new_number (v1);
  mp_print (mp, "pencircle transformed (");
  print_number (h->x_coord);
  mp_print_char (mp, xord (','));
  print_number (h->y_coord);
  mp_print_char (mp, xord (','));
  set_number_from_substraction (v1, h->left_x, h->x_coord);
  print_number (v1);
  mp_print_char (mp, xord (','));
  set_number_from_substraction (v1, h->right_x, h->x_coord);
  print_number (v1);
  mp_print_char (mp, xord (','));
  set_number_from_substraction (v1, h->left_y, h->y_coord);
  print_number (v1);
  mp_print_char (mp, xord (','));
  set_number_from_substraction (v1, h->right_y, h->y_coord);
  print_number (v1);
  mp_print_char (mp, xord (')'));
  free_number (v1);
}


@ Here us another version of |pr_pen| that prints the pen as a diagnostic
message.

@<Declarations@>=
static void mp_print_pen (MP mp, mp_knot h, const char *s, boolean nuline);

@ @c
void mp_print_pen (MP mp, mp_knot h, const char *s, boolean nuline) {
  mp_print_diagnostic (mp, "Pen", s, nuline);
  mp_print_ln (mp);
@.Pen at line...@>;
  mp_pr_pen (mp, h);
  mp_end_diagnostic (mp, true);
}


@ Making a polygonal pen into a path involves restoring the |mp_left_type| and
|mp_right_type| fields and setting the control points so as to make a polygonal
path.

@c
static void mp_make_path (MP mp, mp_knot h) {
  mp_knot p;    /* for traversing the knot list */
  quarterword k;        /* a loop counter */
  @<Other local variables in |make_path|@>;
  FUNCTION_TRACE1 ("make_path()\n");
  if (pen_is_elliptical (h)) {
    FUNCTION_TRACE1 ("make_path(elliptical)\n");
    @<Make the elliptical pen |h| into a path@>;
  } else {
    p = h;
    do {
      mp_left_type (p) = mp_explicit;
      mp_right_type (p) = mp_explicit;
      @<copy the coordinates of knot |p| into its control points@>;
      p = mp_next_knot (p);
    } while (p != h);
  }
}


@ @<copy the coordinates of knot |p| into its control points@>=
number_clone (p->left_x, p->x_coord);
number_clone (p->left_y, p->y_coord);
number_clone (p->right_x, p->x_coord);
number_clone (p->right_y, p->y_coord)
 

@ We need an eight knot path to get a good approximation to an ellipse.

@<Make the elliptical pen |h| into a path@>=
{
  mp_number center_x, center_y;      /* translation parameters for an elliptical pen */
  mp_number width_x, width_y;        /* the effect of a unit change in $x$ */
  mp_number height_x, height_y;      /* the effect of a unit change in $y$ */
  mp_number dx, dy;  /* the vector from knot |p| to its right control point */
  new_number (center_x);
  new_number (center_y);
  new_number (width_x);
  new_number (width_y);
  new_number (height_x);
  new_number (height_y);
  new_number (dx);
  new_number (dy);
  @<Extract the transformation parameters from the elliptical pen~|h|@>;
  p = h;
  for (k = 0; k <= 7; k++) {
    @<Initialize |p| as the |k|th knot of a circle of unit diameter,
      transforming it appropriately@>;
    if (k == 7)
      mp_next_knot (p) = h;
    else
      mp_next_knot (p) = mp_new_knot (mp);
    p = mp_next_knot (p);
  }
  free_number (dx);
  free_number (dy);
  free_number (center_x);
  free_number (center_y);
  free_number (width_x);
  free_number (width_y);
  free_number (height_x);
  free_number (height_y);
}


@ @<Extract the transformation parameters from the elliptical pen~|h|@>=
number_clone (center_x, h->x_coord);
number_clone (center_y, h->y_coord);
set_number_from_substraction (width_x, h->left_x, center_x);
set_number_from_substraction (width_y, h->left_y, center_y);
set_number_from_substraction (height_x, h->right_x, center_x);
set_number_from_substraction (height_y, h->right_y, center_y);

@ @<Other local variables in |make_path|@>=
integer kk;
  /* |k| advanced $270^\circ$ around the ring (cf. $\sin\theta=\cos(\theta+270)$) */

@ The only tricky thing here are the tables |half_cos| and |d_cos| used to
find the point $k/8$ of the way around the circle and the direction vector
to use there.

@<Initialize |p| as the |k|th knot of a circle of unit diameter,...@>=
kk = (k + 6) % 8;
{ 
  mp_number r1, r2;
  new_fraction (r1);
  new_fraction (r2);
  take_fraction (r1, mp->half_cos[k], width_x);
  take_fraction (r2, mp->half_cos[kk], height_x);
  number_add (r1, r2);
  set_number_from_addition (p->x_coord, center_x, r1);
  take_fraction (r1, mp->half_cos[k],  width_y);
  take_fraction (r2, mp->half_cos[kk], height_y);
  number_add (r1, r2);
  set_number_from_addition (p->y_coord, center_y, r1);
  take_fraction (r1, mp->d_cos[kk], width_x);
  take_fraction (r2, mp->d_cos[k], height_x);
  number_clone (dx, r1);
  number_negate (dx);
  number_add (dx, r2);
  take_fraction (r1, mp->d_cos[kk], width_y);
  take_fraction (r2, mp->d_cos[k], height_y);
  number_clone (dy, r1);
  number_negate (dy);
  number_add (dy, r2);
  set_number_from_addition (p->right_x, p->x_coord, dx);
  set_number_from_addition (p->right_y, p->y_coord, dy);
  set_number_from_substraction (p->left_x, p->x_coord, dx);
  set_number_from_substraction (p->left_y, p->y_coord, dy);
  free_number (r1);
  free_number (r2);
}
mp_left_type (p) = mp_explicit;
mp_right_type (p) = mp_explicit;
mp_originator (p) = mp_program_code

@ @<Glob...@>=
mp_number half_cos[8];   /* ${1\over2}\cos(45k)$ */
mp_number d_cos[8];      /* a magic constant times $\cos(45k)$ */

@ The magic constant for |d_cos| is the distance between $({1\over2},0)$ and
$({1\over4}\sqrt2,{1\over4}\sqrt2)$ times the result of the |velocity|
function for $\theta=\phi=22.5^\circ$.  This comes out to be
$$ d = {\sqrt{2-\sqrt2}\over 3+3\cos22.5^\circ}
  \approx 0.132608244919772.
$$

@<Set init...@>=
for (k = 0; k <= 7; k++) {
  new_fraction (mp->half_cos[k]);
  new_fraction (mp->d_cos[k]);
}
number_clone (mp->half_cos[0], fraction_half_t);
number_clone (mp->half_cos[1], twentysixbits_sqrt2_t);  
number_clone (mp->half_cos[2], zero_t);
number_clone (mp->d_cos[0], twentyeightbits_d_t);
number_clone (mp->d_cos[1], twentysevenbits_sqrt2_d_t);
number_clone (mp->d_cos[2], zero_t);
for (k = 3; k <= 4; k++) {
  number_clone (mp->half_cos[k], mp->half_cos[4 - k]);
  number_negate (mp->half_cos[k]);
  number_clone (mp->d_cos[k], mp->d_cos[4 - k]);
  number_negate (mp->d_cos[k]);
}
for (k = 5; k <= 7; k++) {
  number_clone (mp->half_cos[k], mp->half_cos[8 - k]);
  number_clone (mp->d_cos[k], mp->d_cos[8 - k]);
}

@ @<Dealloc...@>=
for (k = 0; k <= 7; k++) {
  free_number (mp->half_cos[k]);
  free_number (mp->d_cos[k]);
}


@ The |convex_hull| function forces a pen polygon to be convex when it is
returned by |make_pen| and after any subsequent transformation where rounding
error might allow the convexity to be lost.
The convex hull algorithm used here is described by F.~P. Preparata and
M.~I. Shamos [{\sl Computational Geometry}, Springer-Verlag, 1985].

@<Declarations@>=
static mp_knot mp_convex_hull (MP mp, mp_knot h);

@ @c
mp_knot mp_convex_hull (MP mp, mp_knot h) {                               /* Make a polygonal pen convex */
  mp_knot l, r; /* the leftmost and rightmost knots */
  mp_knot p, q; /* knots being scanned */
  mp_knot s;    /* the starting point for an upcoming scan */
  mp_number dx, dy;        /* a temporary pointer */
  mp_knot ret;
  new_number (dx);
  new_number (dy);
  if (pen_is_elliptical (h)) {
    ret = h;
  } else {
    @<Set |l| to the leftmost knot in polygon~|h|@>;
    @<Set |r| to the rightmost knot in polygon~|h|@>;
    if (l != r) {
      s = mp_next_knot (r);
      @<Find any knots on the path from |l| to |r| above the |l|-|r| line and
        move them past~|r|@>;
      @<Find any knots on the path from |s| to |l| below the |l|-|r| line and
        move them past~|l|@>;
      @<Sort the path from |l| to |r| by increasing $x$@>;
      @<Sort the path from |r| to |l| by decreasing $x$@>;
    }
    if (l != mp_next_knot (l)) {
      @<Do a Gramm scan and remove vertices where there is no left turn@>;
    }
    ret = l;
  }
  free_number (dx);
  free_number (dy);
  return ret;
}


@ All comparisons are done primarily on $x$ and secondarily on $y$.

@<Set |l| to the leftmost knot in polygon~|h|@>=
l = h;
p = mp_next_knot (h);
while (p != h) {
  if (number_lessequal (p->x_coord, l->x_coord))
    if ((number_less (p->x_coord, l->x_coord)) || 
        (number_less (p->y_coord, l->y_coord)))
      l = p;
  p = mp_next_knot (p);
}


@ @<Set |r| to the rightmost knot in polygon~|h|@>=
r = h;
p = mp_next_knot (h);
while (p != h) {
  if (number_greaterequal(p->x_coord, r->x_coord))
    if (number_greater (p->x_coord, r->x_coord) || 
        number_greater (p->y_coord, r->y_coord))
      r = p;
  p = mp_next_knot (p);
}


@ @<Find any knots on the path from |l| to |r| above the |l|-|r| line...@>=
{
  mp_number ab_vs_cd;
  mp_number arg1, arg2;
  new_number (arg1);
  new_number (arg2);
  new_number (ab_vs_cd);
  set_number_from_substraction (dx, r->x_coord, l->x_coord);
  set_number_from_substraction (dy, r->y_coord, l->y_coord);
  p = mp_next_knot (l);
  while (p != r) {
    q = mp_next_knot (p);
    set_number_from_substraction (arg1, p->y_coord, l->y_coord);
    set_number_from_substraction (arg2, p->x_coord, l->x_coord);
    ab_vs_cd (ab_vs_cd, dx, arg1, dy, arg2);
    if (number_positive(ab_vs_cd))
      mp_move_knot (mp, p, r);
    p = q;
  }
  free_number (ab_vs_cd);
  free_number (arg1);
  free_number (arg2);
}


@ The |move_knot| procedure removes |p| from a doubly linked list and inserts
it after |q|.

@ @<Declarations@>=
static void mp_move_knot (MP mp, mp_knot p, mp_knot q);

@ @c
void mp_move_knot (MP mp, mp_knot p, mp_knot q) {
  (void) mp;
  mp_next_knot (mp_prev_knot (p)) = mp_next_knot (p);
  mp_prev_knot (mp_next_knot (p)) = mp_prev_knot (p);
  mp_prev_knot (p) = q;
  mp_next_knot (p) = mp_next_knot (q);
  mp_next_knot (q) = p;
  mp_prev_knot (mp_next_knot (p)) = p;
}


@ @<Find any knots on the path from |s| to |l| below the |l|-|r| line...@>=
{
  mp_number ab_vs_cd;
  mp_number arg1, arg2;
  new_number (ab_vs_cd);
  new_number (arg1);
  new_number (arg2);
  p = s;
  while (p != l) {
    q = mp_next_knot (p);
    set_number_from_substraction (arg1, p->y_coord, l->y_coord);
    set_number_from_substraction (arg2, p->x_coord, l->x_coord);
    ab_vs_cd (ab_vs_cd, dx, arg1, dy, arg2);
    if (number_negative(ab_vs_cd))
      mp_move_knot (mp, p, l);
    p = q;
  }
  free_number (ab_vs_cd);
  free_number (arg1);
  free_number (arg2);
}


@ The list is likely to be in order already so we just do linear insertions.
Secondary comparisons on $y$ ensure that the sort is consistent with the
choice of |l| and |r|.

@<Sort the path from |l| to |r| by increasing $x$@>=
p = mp_next_knot (l);
while (p != r) {
  q = mp_prev_knot (p);
  while (number_greater(q->x_coord, p->x_coord))
    q = mp_prev_knot (q);
  while (number_equal(q->x_coord, p->x_coord)) {
    if (number_greater(q->y_coord, p->y_coord))
      q = mp_prev_knot (q);
    else
      break;
  }
  if (q == mp_prev_knot (p)) {
    p = mp_next_knot (p);
  } else {
    p = mp_next_knot (p);
    mp_move_knot (mp, mp_prev_knot (p), q);
  }
}


@ @<Sort the path from |r| to |l| by decreasing $x$@>=
p = mp_next_knot (r);
while (p != l) {
  q = mp_prev_knot (p);
  while (number_less(q->x_coord, p->x_coord))
    q = mp_prev_knot (q);
  while (number_equal(q->x_coord, p->x_coord)) {
    if (number_less (q->y_coord, p->y_coord))
      q = mp_prev_knot (q);
    else
      break;
  }
  if (q == mp_prev_knot (p)) {
    p = mp_next_knot (p);
  } else {
    p = mp_next_knot (p);
    mp_move_knot (mp, mp_prev_knot (p), q);
  }
}


@ The condition involving |ab_vs_cd| tests if there is not a left turn
at knot |q|.  There usually will be a left turn so we streamline the case
where the |then| clause is not executed.

@<Do a Gramm scan and remove vertices where there...@>=
{
  mp_number ab_vs_cd;
  mp_number arg1, arg2;
  new_number (arg1);
  new_number (arg2);
  new_number (ab_vs_cd);
  p = l;
  q = mp_next_knot (l);
  while (1) {
    set_number_from_substraction (dx, q->x_coord, p->x_coord);
    set_number_from_substraction (dy, q->y_coord, p->y_coord);
    p = q;
    q = mp_next_knot (q);
    if (p == l)
      break;
    if (p != r) {
      set_number_from_substraction (arg1, q->y_coord, p->y_coord);
      set_number_from_substraction (arg2, q->x_coord, p->x_coord);
      ab_vs_cd (ab_vs_cd, dx, arg1, dy, arg2);
      if (number_nonpositive(ab_vs_cd)) {
        @<Remove knot |p| and back up |p| and |q| but don't go past |l|@>;
      }
    }
  }
  free_number (ab_vs_cd);
  free_number (arg1);
  free_number (arg2);
}


@ @<Remove knot |p| and back up |p| and |q| but don't go past |l|@>=
{
  s = mp_prev_knot (p);
  mp_xfree (p);
  mp_next_knot (s) = q;
  mp_prev_knot (q) = s;
  if (s == l) {
    p = s;
  } else {
    p = mp_prev_knot (s);
    q = s;
  }
}


@ The |find_offset| procedure sets global variables |(cur_x,cur_y)| to the
offset associated with the given direction |(x,y)|.  If two different offsets
apply, it chooses one of them.

@c
static void mp_find_offset (MP mp, mp_number x_orig, mp_number y_orig, mp_knot h) {
  mp_knot p, q; /* consecutive knots */
  if (pen_is_elliptical (h)) {
    mp_fraction xx, yy;      /* untransformed offset for an elliptical pen */
    mp_number wx, wy, hx, hy; /* the transformation matrix for an elliptical pen */
    mp_fraction d;   /* a temporary register */
    new_fraction(xx);
    new_fraction(yy);
    new_number(wx);
    new_number(wy);
    new_number(hx);
    new_number(hy);
    new_fraction(d);
    @<Find the offset for |(x,y)| on the elliptical pen~|h|@>
    free_number (xx);
    free_number (yy);
    free_number (wx);
    free_number (wy);
    free_number (hx);
    free_number (hy);
    free_number (d);
  } else {
    mp_number ab_vs_cd;
    mp_number arg1, arg2;
    new_number (arg1);
    new_number (arg2);
    new_number (ab_vs_cd);
    q = h;
    do {
      p = q;
      q = mp_next_knot (q);
      set_number_from_substraction (arg1, q->x_coord, p->x_coord);
      set_number_from_substraction (arg2, q->y_coord, p->y_coord);
      ab_vs_cd (ab_vs_cd, arg1, y_orig, arg2, x_orig);
    } while (number_negative(ab_vs_cd));    
    do {
      p = q;
      q = mp_next_knot (q);
      set_number_from_substraction (arg1, q->x_coord, p->x_coord);
      set_number_from_substraction (arg2, q->y_coord, p->y_coord);
      ab_vs_cd (ab_vs_cd, arg1, y_orig, arg2, x_orig);
    } while (number_positive(ab_vs_cd));
    number_clone (mp->cur_x, p->x_coord);
    number_clone (mp->cur_y, p->y_coord);
    free_number (ab_vs_cd);
    free_number (arg1);
    free_number (arg2);
  }
}


@ @<Glob...@>=
mp_number cur_x;
mp_number cur_y;   /* all-purpose return value registers */

@ @<Initialize table entries@>=
new_number (mp->cur_x);
new_number (mp->cur_y);

@ @<Dealloc...@>=
free_number (mp->cur_x);
free_number (mp->cur_y);

@ @<Find the offset for |(x,y)| on the elliptical pen~|h|@>=
if (number_zero(x_orig) && number_zero(y_orig)) {
  number_clone(mp->cur_x, h->x_coord);
  number_clone(mp->cur_y, h->y_coord);
} else {
  mp_number x, y, abs_x, abs_y;
  new_number(x);
  new_number(y);
  new_number(abs_x);
  new_number(abs_y);
  number_clone(x, x_orig);
  number_clone(y, y_orig);
  @<Find the non-constant part of the transformation for |h|@>;
  number_clone(abs_x, x);
  number_clone(abs_y, y);
  number_abs(abs_x);
  number_abs(abs_y);
  while (number_less(abs_x, fraction_half_t) && number_less(abs_y, fraction_half_t)) {
    number_double(x);
    number_double(y);
    number_clone(abs_x, x);
    number_clone(abs_y, y);
    number_abs(abs_x);
    number_abs(abs_y);
  }
  @<Make |(xx,yy)| the offset on the untransformed \&{pencircle} for the
    untransformed version of |(x,y)|@>;
  {
    mp_number r1, r2;
    new_fraction (r1);
    new_fraction (r2);
    take_fraction (r1, xx, wx);
    take_fraction (r2, yy, hx);
    number_add(r1, r2);
    set_number_from_addition(mp->cur_x, h->x_coord, r1);
    take_fraction (r1, xx, wy);
    take_fraction (r2, yy, hy);
    number_add(r1, r2);
    set_number_from_addition(mp->cur_y, h->y_coord, r1);
    free_number (r1);
    free_number (r2);
  }
  free_number(abs_x);
  free_number(abs_y);
  free_number(x);
  free_number(y);
}


@ @<Find the non-constant part of the transformation for |h|@>=
{
  set_number_from_substraction(wx, h->left_x,  h->x_coord);
  set_number_from_substraction(wy, h->left_y,  h->y_coord);
  set_number_from_substraction(hx, h->right_x, h->x_coord);
  set_number_from_substraction(hy, h->right_y, h->y_coord);
}
 

@ @<Make |(xx,yy)| the offset on the untransformed \&{pencircle} for the...@>=
{
  mp_number r1, r2, arg1;
  new_number (arg1);
  new_fraction (r1);
  new_fraction (r2);
  take_fraction (r1, x, hy);
  number_clone (arg1, hx);
  number_negate (arg1);
  take_fraction (r2, y, arg1);
  number_add (r1, r2);
  number_negate (r1);
  number_clone(yy, r1);
  number_clone (arg1, wy);
  number_negate (arg1);
  take_fraction (r1, x, arg1);
  take_fraction (r2, y, wx);
  number_add (r1, r2);
  number_clone(xx, r1);
  free_number (arg1);
  free_number (r1);
  free_number (r2);
}
pyth_add (d, xx, yy);
if (number_positive(d)) {
  mp_number ret;
  new_fraction (ret);
  make_fraction (ret, xx, d);
  number_half(ret);
  number_clone(xx, ret);
  make_fraction (ret, yy, d);
  number_half(ret);
  number_clone(yy, ret);
  free_number (ret);
}

@ Finding the bounding box of a pen is easy except if the pen is elliptical.
But we can handle that case by just calling |find_offset| twice.  The answer
is stored in the global variables |minx|, |maxx|, |miny|, and |maxy|.

@c
static void mp_pen_bbox (MP mp, mp_knot h) {
  mp_knot p;    /* for scanning the knot list */
  if (pen_is_elliptical (h)) {
    @<Find the bounding box of an elliptical pen@>;
  } else {
    number_clone (mp_minx, h->x_coord);
    number_clone (mp_maxx, mp_minx);
    number_clone (mp_miny, h->y_coord);
    number_clone (mp_maxy, mp_miny);
    p = mp_next_knot (h);
    while (p != h) {
      if (number_less (p->x_coord, mp_minx))
        number_clone (mp_minx, p->x_coord);
      if (number_less (p->y_coord, mp_miny))
        number_clone (mp_miny, p->y_coord);
      if (number_greater (p->x_coord, mp_maxx))
        number_clone (mp_maxx, p->x_coord);
      if (number_greater (p->y_coord, mp_maxy))
        number_clone (mp_maxy, p->y_coord);
      p = mp_next_knot (p);
    }
  }
}


@ @<Find the bounding box of an elliptical pen@>=
{
  mp_number arg1, arg2;
  new_number(arg1);
  new_fraction (arg2);
  number_clone(arg2, fraction_one_t);
  mp_find_offset (mp, arg1, arg2, h);
  number_clone (mp_maxx, mp->cur_x);
  number_clone (mp_minx, h->x_coord);
  number_double (mp_minx);
  number_substract (mp_minx, mp->cur_x);
  number_negate (arg2);
  mp_find_offset (mp, arg2, arg1, h);
  number_clone (mp_maxy, mp->cur_y);
  number_clone (mp_miny, h->y_coord);
  number_double (mp_miny);
  number_substract (mp_miny, mp->cur_y);
  free_number(arg1);
  free_number(arg2);
}


@* Numerical values.

This first set goes into the header

@<MPlib internal header stuff@>=
#define mp_fraction mp_number
#define mp_angle mp_number
#define new_number(A) (((math_data *)(mp->math))->allocate)(mp, &(A), mp_scaled_type)
#define new_fraction(A) (((math_data *)(mp->math))->allocate)(mp, &(A), mp_fraction_type)
#define new_angle(A) (((math_data *)(mp->math))->allocate)(mp, &(A), mp_angle_type)
#define free_number(A) (((math_data *)(mp->math))->free)(mp, &(A))

@ 
@d set_precision()                     (((math_data *)(mp->math))->set_precision)(mp) 
@d free_math()                         (((math_data *)(mp->math))->free_math)(mp) 
@d scan_numeric_token(A)               (((math_data *)(mp->math))->scan_numeric)(mp, A) 
@d scan_fractional_token(A)            (((math_data *)(mp->math))->scan_fractional)(mp, A) 
@d set_number_from_of_the_way(A,t,B,C) (((math_data *)(mp->math))->from_oftheway)(mp, &(A),t,B,C) 
@d set_number_from_int(A,B)	       (((math_data *)(mp->math))->from_int)(&(A),B)
@d set_number_from_scaled(A,B)	       (((math_data *)(mp->math))->from_scaled)(&(A),B)
@d set_number_from_boolean(A,B)	       (((math_data *)(mp->math))->from_boolean)(&(A),B)
@d set_number_from_double(A,B)	       (((math_data *)(mp->math))->from_double)(&(A),B)
@d set_number_from_addition(A,B,C)     (((math_data *)(mp->math))->from_addition)(&(A),B,C)
@d set_number_from_substraction(A,B,C) (((math_data *)(mp->math))->from_substraction)(&(A),B,C)
@d set_number_from_div(A,B,C)          (((math_data *)(mp->math))->from_div)(&(A),B,C)
@d set_number_from_mul(A,B,C)          (((math_data *)(mp->math))->from_mul)(&(A),B,C)
@d number_int_div(A,C)                 (((math_data *)(mp->math))->from_int_div)(&(A),A,C)
@d set_number_from_int_mul(A,B,C)      (((math_data *)(mp->math))->from_int_mul)(&(A),B,C)
@#
@d set_number_to_unity(A)	       (((math_data *)(mp->math))->clone)(&(A), unity_t)
@d set_number_to_zero(A)	       (((math_data *)(mp->math))->clone)(&(A), zero_t)
@d set_number_to_inf(A)		       (((math_data *)(mp->math))->clone)(&(A), inf_t)
@d set_number_to_neg_inf(A)	       do { set_number_to_inf(A); number_negate (A); } while (0)
@#
@d init_randoms(A)                     (((math_data *)(mp->math))->init_randoms)(mp,A)
@d print_number(A)                     (((math_data *)(mp->math))->print)(mp,A)
@d number_tostring(A)                  (((math_data *)(mp->math))->tostring)(mp,A)
@d make_scaled(R,A,B)                  (((math_data *)(mp->math))->make_scaled)(mp,&(R),A,B)
@d take_scaled(R,A,B)                  (((math_data *)(mp->math))->take_scaled)(mp,&(R),A,B)
@d make_fraction(R,A,B)                (((math_data *)(mp->math))->make_fraction)(mp,&(R),A,B)
@d take_fraction(R,A,B)                (((math_data *)(mp->math))->take_fraction)(mp,&(R),A,B)
@d pyth_add(R,A,B)                     (((math_data *)(mp->math))->pyth_add)(mp,&(R),A,B)
@d pyth_sub(R,A,B)                     (((math_data *)(mp->math))->pyth_sub)(mp,&(R),A,B)
@d n_arg(R,A,B)                        (((math_data *)(mp->math))->n_arg)(mp,&(R),A,B)
@d m_log(R,A)                          (((math_data *)(mp->math))->m_log)(mp,&(R),A)
@d m_exp(R,A)                          (((math_data *)(mp->math))->m_exp)(mp,&(R),A)
@d velocity(R,A,B,C,D,E)               (((math_data *)(mp->math))->velocity)(mp,&(R),A,B,C,D,E)
@d ab_vs_cd(R,A,B,C,D)                 (((math_data *)(mp->math))->ab_vs_cd)(mp,&(R),A,B,C,D)
@d crossing_point(R,A,B,C)             (((math_data *)(mp->math))->crossing_point)(mp,&(R),A,B,C)
@d n_sin_cos(A,S,C)                    (((math_data *)(mp->math))->sin_cos)(mp,A,&(S),&(C))
@d square_rt(A,S)                      (((math_data *)(mp->math))->sqrt)(mp,&(A),S)
@d slow_add(R,A,B)                     (((math_data *)(mp->math))->slow_add)(mp,&(R),A,B)
@d round_unscaled(A)		       (((math_data *)(mp->math))->round_unscaled)(A)		       
@d floor_scaled(A)		       (((math_data *)(mp->math))->floor_scaled)(&(A))
@d fraction_to_round_scaled(A)         (((math_data *)(mp->math))->fraction_to_round_scaled)(&(A))
@d number_to_int(A)		       (((math_data *)(mp->math))->to_int)(A)
@d number_to_boolean(A)		       (((math_data *)(mp->math))->to_boolean)(A)
@d number_to_scaled(A)		       (((math_data *)(mp->math))->to_scaled)(A)		       
@d number_to_double(A)		       (((math_data *)(mp->math))->to_double)(A)		       
@d number_negate(A)		       (((math_data *)(mp->math))->negate)(&(A))		       
@d number_add(A,B)		       (((math_data *)(mp->math))->add)(&(A),B)		       
@d number_substract(A,B)	       (((math_data *)(mp->math))->substract)(&(A),B)	       
@d number_half(A)		       (((math_data *)(mp->math))->half)(&(A))		       
@d number_halfp(A)		       (((math_data *)(mp->math))->halfp)(&(A))		       
@d number_double(A)		       (((math_data *)(mp->math))->do_double)(&(A))		       
@d number_add_scaled(A,B)	       (((math_data *)(mp->math))->add_scaled)(&(A),B)	       
@d number_multiply_int(A,B)	       (((math_data *)(mp->math))->multiply_int)(&(A),B)	       
@d number_divide_int(A,B)	       (((math_data *)(mp->math))->divide_int)(&(A),B)	       
@d number_abs(A)		       (((math_data *)(mp->math))->abs)(&(A))		       
@d number_modulo(A,B)		       (((math_data *)(mp->math))->modulo)(&(A), B)		       
@d number_nonequalabs(A,B)	       (((math_data *)(mp->math))->nonequalabs)(A,B)	       
@d number_odd(A)		       (((math_data *)(mp->math))->odd)(A)		       
@d number_equal(A,B)		       (((math_data *)(mp->math))->equal)(A,B)		       
@d number_greater(A,B)		       (((math_data *)(mp->math))->greater)(A,B)		       
@d number_less(A,B)		       (((math_data *)(mp->math))->less)(A,B)		       
@d number_clone(A,B)		       (((math_data *)(mp->math))->clone)(&(A),B)		       
@d number_swap(A,B)		       (((math_data *)(mp->math))->swap)(&(A),&(B));
@d convert_scaled_to_angle(A)          (((math_data *)(mp->math))->scaled_to_angle)(&(A));
@d convert_angle_to_scaled(A)          (((math_data *)(mp->math))->angle_to_scaled)(&(A));
@d convert_fraction_to_scaled(A)       (((math_data *)(mp->math))->fraction_to_scaled)(&(A));
@d convert_scaled_to_fraction(A)       (((math_data *)(mp->math))->scaled_to_fraction)(&(A));
@#
@d number_zero(A)		       number_equal(A, zero_t)		       
@d number_infinite(A)		       number_equal(A, inf_t)		       
@d number_unity(A)		       number_equal(A, unity_t)
@d number_negative(A)		       number_less(A, zero_t)
@d number_nonnegative(A)	       (!number_negative(A))
@d number_positive(A)		       number_greater(A, zero_t)		       
@d number_nonpositive(A)	       (!number_positive(A))
@d number_nonzero(A)		       (!number_zero(A))	       
@d number_greaterequal(A,B)	       (!number_less(A,B))
@d number_lessequal(A,B)	       (!number_greater(A,B))

@* Edge structures.
Now we come to \MP's internal scheme for representing pictures.
The representation is very different from \MF's edge structures
because \MP\ pictures contain \ps\ graphics objects instead of pixel
images.  However, the basic idea is somewhat similar in that shapes
are represented via their boundaries.

The main purpose of edge structures is to keep track of graphical objects
until it is time to translate them into \ps.  Since \MP\ does not need to
know anything about an edge structure other than how to translate it into
\ps\ and how to find its bounding box, edge structures can be just linked
lists of graphical objects.  \MP\ has no easy way to determine whether
two such objects overlap, but it suffices to draw the first one first and
let the second one overwrite it if necessary.

@<MPlib header stuff@>=
enum mp_graphical_object_code {
  @<Graphical object codes@>
  mp_final_graphic
};

@ Let's consider the types of graphical objects one at a time.
First of all, a filled contour is represented by a eight-word node.  The first
word contains |type| and |link| fields, and the next six words contain a
pointer to a cyclic path and the value to use for \ps' \&{currentrgbcolor}
parameter.  If a pen is used for filling |pen_p|, |ljoin| and |miterlim|
give the relevant information.

@d mp_path_p(A) (A)->path_p_  /* a pointer to the path that needs filling */
@d mp_pen_p(A) (A)->pen_p_  /* a pointer to the pen to fill or stroke with */
@d mp_color_model(A) ((mp_fill_node)(A))->color_model_ /*  the color model  */
@d cyan red
@d grey red
@d magenta green
@d yellow blue
@d mp_pre_script(A) ((mp_fill_node)(A))->pre_script_
@d mp_post_script(A) ((mp_fill_node)(A))->post_script_

@<MPlib internal header stuff@>=
typedef struct mp_fill_node_data {
  NODE_BODY;
  halfword color_model_;
  mp_number red;
  mp_number green;
  mp_number blue;
  mp_number black;
  mp_string pre_script_;
  mp_string post_script_;
  mp_knot path_p_;
  mp_knot pen_p_;
  unsigned char ljoin;
  mp_number miterlim;
} mp_fill_node_data;
typedef struct mp_fill_node_data *mp_fill_node;

@ @<Graphical object codes@>=
mp_fill_code = 1,

@ Make a fill node for cyclic path |p| and color black.

@d fill_node_size sizeof(struct mp_fill_node_data)

@c
static mp_node mp_new_fill_node (MP mp, mp_knot p) {
  mp_fill_node t = malloc_node (fill_node_size);
  mp_type (t) = mp_fill_node_type;
  mp_path_p (t) = p;
  mp_pen_p (t) = NULL;          /* |NULL| means don't use a pen */
  new_number(t->red);
  new_number(t->green);
  new_number(t->blue);
  new_number(t->black);
  new_number(t->miterlim);
  clear_color (t);
  mp_color_model (t) = mp_uninitialized_model;
  mp_pre_script (t) = NULL;
  mp_post_script (t) = NULL;
  /* Set the |ljoin| and |miterlim| fields in object |t| */
  if (number_greater(internal_value (mp_linejoin), unity_t))
    t->ljoin = 2;
  else if (number_positive(internal_value (mp_linejoin)))
    t->ljoin = 1;
  else
    t->ljoin = 0;
  if (number_less(internal_value (mp_miterlimit), unity_t)) {
    set_number_to_unity(t->miterlim);
  } else {
    number_clone(t->miterlim,internal_value (mp_miterlimit));
  }
  return (mp_node) t;
}

@ @c
static void mp_free_fill_node (MP mp, mp_fill_node p) {
  mp_toss_knot_list (mp, mp_path_p (p));
  if (mp_pen_p (p) != NULL)
    mp_toss_knot_list (mp, mp_pen_p (p));
  if (mp_pre_script (p) != NULL)
    delete_str_ref (mp_pre_script (p));
  if (mp_post_script (p) != NULL)
    delete_str_ref (mp_post_script (p));
  free_number(p->red);
  free_number(p->green);
  free_number(p->blue);
  free_number(p->black);
  free_number(p->miterlim);
  mp_free_node (mp, (mp_node)p, fill_node_size);
}



@ A stroked path is represented by an eight-word node that is like a filled
contour node except that it contains the current \&{linecap} value, a scale
factor for the dash pattern, and a pointer that is non-NULL if the stroke
is to be dashed.  The purpose of the scale factor is to allow a picture to
be transformed without touching the picture that |dash_p| points to.

@d mp_dash_p(A) ((mp_stroked_node)(A))->dash_p_  /* a pointer to the edge structure that gives the dash pattern */

@<MPlib internal header stuff@>=
typedef struct mp_stroked_node_data {
  NODE_BODY;
  halfword color_model_;
  mp_number red;
  mp_number green;
  mp_number blue;
  mp_number black;
  mp_string pre_script_;
  mp_string post_script_;
  mp_knot path_p_;
  mp_knot pen_p_;
  unsigned char ljoin;
  mp_number miterlim;
  unsigned char lcap;
  mp_node dash_p_;
  mp_number dash_scale;
} mp_stroked_node_data;
typedef struct mp_stroked_node_data *mp_stroked_node;


@ @<Graphical object codes@>=
mp_stroked_code = 2,

@  Make a stroked node for path |p| with |mp_pen_p(p)| temporarily |NULL|.

@d stroked_node_size sizeof(struct mp_stroked_node_data)

@c
static mp_node mp_new_stroked_node (MP mp, mp_knot p) {
  mp_stroked_node t = malloc_node (stroked_node_size);
  mp_type (t) = mp_stroked_node_type;
  mp_path_p (t) = p;
  mp_pen_p (t) = NULL;
  mp_dash_p (t) = NULL;
  new_number(t->dash_scale);
  set_number_to_unity(t->dash_scale);
  new_number(t->red);
  new_number(t->green);
  new_number(t->blue);
  new_number(t->black);
  new_number(t->miterlim);
  clear_color(t);
  mp_pre_script (t) = NULL;
  mp_post_script (t) = NULL;
  /* Set the |ljoin| and |miterlim| fields in object |t| */
  if (number_greater(internal_value (mp_linejoin), unity_t))
    t->ljoin = 2;
  else if (number_positive(internal_value (mp_linejoin)))
    t->ljoin = 1;
  else
    t->ljoin = 0;
  if (number_less(internal_value (mp_miterlimit), unity_t)) {
    set_number_to_unity(t->miterlim);
  } else {
    number_clone(t->miterlim,internal_value (mp_miterlimit));
  }
  if (number_greater(internal_value (mp_linecap), unity_t))
    t->lcap = 2;
  else if (number_positive(internal_value (mp_linecap)))
    t->lcap = 1;
  else
    t->lcap = 0;
  return (mp_node) t;
}

@ @c
static mp_edge_header_node mp_free_stroked_node (MP mp, mp_stroked_node p) {
  mp_edge_header_node e = NULL;
  mp_toss_knot_list (mp, mp_path_p (p));
  if (mp_pen_p (p) != NULL)
    mp_toss_knot_list (mp, mp_pen_p (p));
  if (mp_pre_script (p) != NULL)
    delete_str_ref (mp_pre_script (p));
  if (mp_post_script (p) != NULL)
    delete_str_ref (mp_post_script (p));
  e = (mp_edge_header_node)mp_dash_p (p);
  free_number(p->dash_scale);
  free_number(p->red);
  free_number(p->green);
  free_number(p->blue);
  free_number(p->black);
  free_number(p->miterlim);
  mp_free_node (mp, (mp_node)p, stroked_node_size);
  return e;
}

@ When a dashed line is computed in a transformed coordinate system, the dash
lengths get scaled like the pen shape and we need to compensate for this.  Since
there is no unique scale factor for an arbitrary transformation, we use the
the square root of the determinant.  The properties of the determinant make it
easier to maintain the |dash_scale|.  The computation is fairly straight-forward
except for the initialization of the scale factor |s|.  The factor of 64 is
needed because |square_rt| scales its result by $2^8$ while we need $2^{14}$
to counteract the effect of |take_fraction|.

@ @c
void mp_sqrt_det (MP mp, mp_number *ret, mp_number a_orig, mp_number b_orig, mp_number c_orig, mp_number d_orig) {
  mp_number a,b,c,d; 
  mp_number maxabs;        /* $max(|a|,|b|,|c|,|d|)$ */
  unsigned s;   /* amount by which the result of |square_rt| needs to be scaled */
  new_number(a);
  new_number(b);
  new_number(c);
  new_number(d);
  new_number(maxabs);
  number_clone(a, a_orig);
  number_clone(b, b_orig);
  number_clone(c, c_orig);
  number_clone(d, d_orig);
  /* Initialize |maxabs| */
  {
    mp_number tmp;
    new_number (tmp);
    number_clone(maxabs, a);
    number_abs(maxabs);
    number_clone(tmp, b);
    number_abs(tmp);
    if (number_greater(tmp, maxabs))
      number_clone(maxabs, tmp);
    number_clone(tmp, c);
    number_abs(tmp);
    if (number_greater(tmp, maxabs))
      number_clone(maxabs, tmp);
    number_clone(tmp, d);
    number_abs(tmp);
    if (number_greater(tmp, maxabs))
      number_clone(maxabs, tmp);
    free_number(tmp);
  }
   

  s = 64;
  while ((number_less(maxabs, fraction_one_t)) && (s > 1)) {
    number_double(a);
    number_double(b);
    number_double(c);
    number_double(d);
    number_double(maxabs);
    s = s/2;
  }
  { 
    mp_number r1, r2;
    new_fraction (r1);
    new_fraction (r2);
    take_fraction (r1, a, d);
    take_fraction (r2, b, c);
    number_substract (r1, r2);
    number_abs (r1);
    square_rt(*ret, r1);
    number_multiply_int(*ret, s);
    free_number (r1);
    free_number (r2);
  }
  free_number(a);
  free_number(b);
  free_number(c);
  free_number(d);
  free_number(maxabs);
}
@#
static void mp_get_pen_scale (MP mp, mp_number *ret, mp_knot p) {
  if (p == NULL) {
    set_number_to_zero(*ret);
  } else {
    mp_number a,b,c,d;
    new_number(a);
    new_number(b);
    new_number(c);
    new_number(d);
    set_number_from_substraction(a, p->left_x, p->x_coord);
    set_number_from_substraction(b, p->right_x, p->x_coord);
    set_number_from_substraction(c, p->left_y,  p->y_coord);
    set_number_from_substraction(d, p->right_y, p->y_coord);
    mp_sqrt_det (mp, ret, a, b, c, d);
    free_number(a);
    free_number(b);
    free_number(c);
    free_number(d);
  }
}


@ @<Declarations@>=
static void mp_sqrt_det (MP mp, mp_number *ret, mp_number a, mp_number b, mp_number c, mp_number d);

@ When a picture contains text, this is represented by a fourteen-word node
where the color information and |type| and |link| fields are augmented by
additional fields that describe the text and  how it is transformed.
The |path_p| and |mp_pen_p| pointers are replaced by a number that identifies
the font and a string number that gives the text to be displayed.
The |width|, |height|, and |depth| fields
give the dimensions of the text at its design size, and the remaining six
words give a transformation to be applied to the text.  The |new_text_node|
function initializes everything to default values so that the text comes out
black with its reference point at the origin.

@d mp_text_p(A) ((mp_text_node)(A))->text_p_  /* a string pointer for the text to display */
@d mp_font_n(A) ((mp_text_node)(A))->font_n_ /* the font number */

@<MPlib internal header stuff@>=
typedef struct mp_text_node_data {
  NODE_BODY;
  halfword color_model_;
  mp_number red;
  mp_number green;
  mp_number blue;
  mp_number black;
  mp_string pre_script_;
  mp_string post_script_;
  mp_string text_p_;
  halfword font_n_;
  mp_number width;
  mp_number height;
  mp_number depth;
  mp_number tx;
  mp_number ty;
  mp_number txx;
  mp_number txy;
  mp_number tyx;
  mp_number tyy;
} mp_text_node_data;
typedef struct mp_text_node_data *mp_text_node;

@ @<Graphical object codes@>=
mp_text_code = 3,

@  Make a text node for font |f| and text string |s|.

@d text_node_size sizeof(struct mp_text_node_data)

@c
static mp_node mp_new_text_node (MP mp, char *f, mp_string s) {
  mp_text_node t = malloc_node (text_node_size);
  mp_type (t) = mp_text_node_type;
  mp_text_p (t) = s;
  add_str_ref(s); 
  mp_font_n (t) = (halfword) mp_find_font (mp, f);      /* this identifies the font */
  new_number(t->red);
  new_number(t->green);
  new_number(t->blue);
  new_number(t->black);
  new_number(t->width);
  new_number(t->height);
  new_number(t->depth);
  clear_color (t);
  mp_pre_script (t) = NULL;
  mp_post_script (t) = NULL;
  new_number(t->tx);
  new_number(t->ty);
  new_number(t->txx);
  new_number(t->txy);
  new_number(t->tyx);
  new_number(t->tyy);
  /* |tx_val (t) = 0; ty_val (t) = 0;| */
  /* |txy_val (t) = 0; tyx_val (t) = 0;| */
  set_number_to_unity(t->txx);
  set_number_to_unity(t->tyy);
  mp_set_text_box (mp, t);    /* this finds the bounding box */
  return (mp_node) t;
}

@ @c
static void mp_free_text_node (MP mp, mp_text_node p) {
  /* |delete_str_ref (mp_text_p (p));| */ /* gives errors */
  if (mp_pre_script (p) != NULL)
    delete_str_ref (mp_pre_script (p));
  if (mp_post_script (p) != NULL)
    delete_str_ref (mp_post_script (p));
  free_number(p->red);
  free_number(p->green);
  free_number(p->blue);
  free_number(p->black);
  free_number(p->width);
  free_number(p->height);
  free_number(p->depth);
  free_number(p->tx);
  free_number(p->ty);
  free_number(p->txx);
  free_number(p->txy);
  free_number(p->tyx);
  free_number(p->tyy);
  mp_free_node (mp, (mp_node)p, text_node_size);
}

@ The last two types of graphical objects that can occur in an edge structure
are clipping paths and \&{setbounds} paths.  These are slightly more difficult
@:set_bounds_}{\&{setbounds} primitive@>
to implement because we must keep track of exactly what is being clipped or
bounded when pictures get merged together.  For this reason, each clipping or
\&{setbounds} operation is represented by a pair of nodes:  first comes a
node whose |path_p| gives the relevant path, then there is the list
of objects to clip or bound followed by a closing node.

@d has_color(A) (mp_type((A))<mp_start_clip_node_type)
  /* does a graphical object have color fields? */
@d has_pen(A) (mp_type((A))<mp_text_node_type)
  /* does a graphical object have a |mp_pen_p| field? */
@d is_start_or_stop(A) (mp_type((A))>=mp_start_clip_node_type)
@d is_stop(A) (mp_type((A))>=mp_stop_clip_node_type)

@<MPlib internal header stuff@>=
typedef struct mp_start_clip_node_data {
  NODE_BODY;
  mp_knot path_p_;
} mp_start_clip_node_data;
typedef struct mp_start_clip_node_data *mp_start_clip_node;
typedef struct mp_start_bounds_node_data {
  NODE_BODY;
  mp_knot path_p_;
} mp_start_bounds_node_data;
typedef struct mp_start_bounds_node_data *mp_start_bounds_node;
typedef struct mp_stop_clip_node_data {
  NODE_BODY;
} mp_stop_clip_node_data;
typedef struct mp_stop_clip_node_data *mp_stop_clip_node;
typedef struct mp_stop_bounds_node_data {
  NODE_BODY;
} mp_stop_bounds_node_data;
typedef struct mp_stop_bounds_node_data *mp_stop_bounds_node;


@ @<Graphical object codes@>=
mp_start_clip_code = 4,         /* |type| of a node that starts clipping */
  mp_start_bounds_code = 5,     /* |type| of a node that gives a \&{setbounds} path */
  mp_stop_clip_code = 6,        /* |type| of a node that stops clipping */
  mp_stop_bounds_code = 7,      /* |type| of a node that stops \&{setbounds} */
  

@ 

@d start_clip_size sizeof(struct mp_start_clip_node_data)
@d stop_clip_size sizeof(struct mp_stop_clip_node_data)
@d start_bounds_size sizeof(struct mp_start_bounds_node_data)
@d stop_bounds_size sizeof(struct mp_stop_bounds_node_data)

@c
static mp_node mp_new_bounds_node (MP mp, mp_knot p, quarterword c) {
  /* make a node of type |c| where |p| is the clipping or \&{setbounds} path */
  if (c == mp_start_clip_node_type) {
    mp_start_clip_node t;       /* the new node */
    t = (mp_start_clip_node) malloc_node (start_clip_size);
    t->path_p_ = p;
    mp_type (t) = c;
    t->link = NULL;
    return (mp_node) t;
  } else if (c == mp_start_bounds_node_type) {
    mp_start_bounds_node t;     /* the new node */
    t = (mp_start_bounds_node) malloc_node (start_bounds_size);
    t->path_p_ = p;
    mp_type (t) = c;
    t->link = NULL;
    return (mp_node) t;
  } else if (c == mp_stop_clip_node_type) {
    mp_stop_clip_node t;        /* the new node */
    t = (mp_stop_clip_node) malloc_node (stop_clip_size);
    mp_type (t) = c;
    t->link = NULL;
    return (mp_node) t;
  } else if (c == mp_stop_bounds_node_type) {
    mp_stop_bounds_node t;      /* the new node */
    t = (mp_stop_bounds_node) malloc_node (stop_bounds_size);
    mp_type (t) = c;
    t->link = NULL;
    return (mp_node) t;
  } else {
    assert (0);
  }
  return NULL;
}


@ @c
static void mp_free_start_clip_node (MP mp, mp_start_clip_node p) {
  mp_toss_knot_list (mp, mp_path_p (p));
  mp_free_node (mp, (mp_node)p, start_clip_size);
}
static void mp_free_start_bounds_node (MP mp, mp_start_bounds_node p) {
  mp_toss_knot_list (mp, mp_path_p (p));
  mp_free_node (mp, (mp_node)p, start_bounds_size);
}
static void mp_free_stop_clip_node (MP mp, mp_stop_clip_node p) {
  mp_free_node (mp, (mp_node)p, stop_clip_size);
}
static void mp_free_stop_bounds_node (MP mp, mp_stop_bounds_node p) {
  mp_free_node (mp, (mp_node)p, stop_bounds_size);
}


@ All the essential information in an edge structure is encoded as a linked list
of graphical objects as we have just seen, but it is helpful to add some
redundant information.  A single edge structure might be used as a dash pattern
many times, and it would be nice to avoid scanning the same structure
repeatedly.  Thus, an edge structure known to be a suitable dash pattern
has a header that gives a list of dashes in a sorted order designed for rapid
translation into \ps.

Each dash is represented by a three-word node containing the initial and final
$x$~coordinates as well as the usual |link| field.  The |link| fields points to
the dash node with the next higher $x$-coordinates and the final link points
to a special location called |null_dash|.  (There should be no overlap between
dashes).  Since the $y$~coordinate of the dash pattern is needed to determine
the period of repetition, this needs to be stored in the edge header along
with a pointer to the list of dash nodes.

The |dash_info| is explained below.

@d dash_list(A) (mp_dash_node)(((mp_dash_node)(A))->link)  /* in an edge header this points to the first dash node */
@d set_dash_list(A,B) ((mp_dash_node)(A))->link=(mp_node)((B))  /* in an edge header this points to the first dash node */

@<MPlib internal header stuff@>=
typedef struct mp_dash_node_data {
  NODE_BODY;
  mp_number start_x; /* the starting $x$~coordinate in a dash node */
  mp_number stop_x; /* the ending $x$~coordinate in a dash node */
  mp_number dash_y; /* $y$ value for the dash list in an edge header */
  mp_node dash_info_;
} mp_dash_node_data;

@ @<Types...@>=
typedef struct mp_dash_node_data *mp_dash_node;

@ @<Initialize table entries@>=
mp->null_dash = mp_get_dash_node (mp);

@ @<Free table entries@>=
mp_free_node (mp, (mp_node)mp->null_dash, dash_node_size);

@ 
@d dash_node_size sizeof(struct mp_dash_node_data)

@c
static mp_dash_node mp_get_dash_node (MP mp) {
  mp_dash_node p = (mp_dash_node) malloc_node (dash_node_size);
  p->has_number = 0;
  new_number(p->start_x);
  new_number(p->stop_x);
  new_number(p->dash_y);
  mp_type (p) = mp_dash_node_type;
  return p;
}


@ It is also convenient for an edge header to contain the bounding
box information needed by the \&{llcorner} and \&{urcorner} operators
so that this does not have to be recomputed unnecessarily.  This is done by
adding fields for the $x$~and $y$ extremes as well as a pointer that indicates
how far the bounding box computation has gotten.  Thus if the user asks for
the bounding box and then adds some more text to the picture before asking
for more bounding box information, the second computation need only look at
the additional text.

When the bounding box has not been computed, the |bblast| pointer points
to a dummy link at the head of the graphical object list while the |minx_val|
and |miny_val| fields contain |EL_GORDO| and the |maxx_val| and |maxy_val|
fields contain |-EL_GORDO|.

Since the bounding box of pictures containing objects of type
|mp_start_bounds_node| depends on the value of \&{truecorners}, the bounding box
@:mp_true_corners_}{\&{truecorners} primitive@>
data might not be valid for all values of this parameter.  Hence, the |bbtype|
field is needed to keep track of this.

@d bblast(A) ((mp_edge_header_node)(A))->bblast_  /* last item considered in bounding box computation */
@d edge_list(A)  ((mp_edge_header_node)(A))->list_ /* where the object list begins in an edge header */

@<MPlib internal header stuff@>=
typedef struct mp_edge_header_node_data {
  NODE_BODY;
  mp_number start_x;
  mp_number stop_x;
  mp_number dash_y;
  mp_node dash_info_;
  mp_number minx;
  mp_number miny;
  mp_number maxx;
  mp_number maxy;
  mp_node bblast_;
  int bbtype; /* tells how bounding box data depends on \&{truecorners} */
  mp_node list_;
  mp_node obj_tail_;    /* explained below */
  halfword ref_count_;  /* explained below */
} mp_edge_header_node_data;
typedef struct mp_edge_header_node_data *mp_edge_header_node;

@
@d no_bounds 0  /* |bbtype| value when bounding box data is valid for all \&{truecorners} values */
@d bounds_set 1  /* |bbtype| value when bounding box data is for \&{truecorners}${}\le 0$ */
@d bounds_unset 2  /* |bbtype| value when bounding box data is for \&{truecorners}${}>0$ */
@c
static void mp_init_bbox (MP mp, mp_edge_header_node h) {
  /* Initialize the bounding box information in edge structure |h| */
  (void) mp;
  bblast (h) = edge_list (h);
  h->bbtype = no_bounds;
  set_number_to_inf(h->minx);
  set_number_to_inf(h->miny);
  set_number_to_neg_inf(h->maxx);
  set_number_to_neg_inf(h->maxy);
}


@ The only other entries in an edge header are a reference count in the first
word and a pointer to the tail of the object list in the last word.

@d obj_tail(A) ((mp_edge_header_node)(A))->obj_tail_  /* points to the last entry in the object list */
@d edge_ref_count(A) ((mp_edge_header_node)(A))->ref_count_

@d edge_header_size sizeof(struct mp_edge_header_node_data)

@c
static mp_edge_header_node mp_get_edge_header_node (MP mp) {
  mp_edge_header_node p = (mp_edge_header_node) malloc_node (edge_header_size);
  mp_type (p) = mp_edge_header_node_type;
  new_number(p->start_x);
  new_number(p->stop_x);
  new_number(p->dash_y);
  new_number(p->minx);
  new_number(p->miny);
  new_number(p->maxx);
  new_number(p->maxy);
  p->list_ = mp_get_token_node (mp);   /* or whatever, just a need a link handle */
  return p;
}
static void mp_init_edges (MP mp, mp_edge_header_node h) {
  /* initialize an edge header to NULL values */
  set_dash_list (h, mp->null_dash);
  obj_tail (h) = edge_list (h);
  mp_link (edge_list (h)) = NULL;
  edge_ref_count (h) = 0;
  mp_init_bbox (mp, h);
}


@ Here is how edge structures are deleted.  The process can be recursive because
of the need to dereference edge structures that are used as dash patterns.
@^recursion@>

@d add_edge_ref(A) incr(edge_ref_count((A)))
@d delete_edge_ref(A) { 
   if ( edge_ref_count((A))==0 ) 
     mp_toss_edges(mp, (mp_edge_header_node)(A));
   else 
     decr(edge_ref_count((A))); 
   }

@<Declarations@>=
static void mp_flush_dash_list (MP mp, mp_edge_header_node h);
static mp_edge_header_node mp_toss_gr_object (MP mp, mp_node p);
static void mp_toss_edges (MP mp, mp_edge_header_node h);

@ @c
void mp_toss_edges (MP mp, mp_edge_header_node h) {
  mp_node p, q; /* pointers that scan the list being recycled */
  mp_edge_header_node r;    /* an edge structure that object |p| refers to */
  mp_flush_dash_list (mp, h);
  q = mp_link (edge_list (h));
  while ((q != NULL)) {
    p = q;
    q = mp_link (q);
    r = mp_toss_gr_object (mp, p);
    if (r != NULL)
      delete_edge_ref (r);
  }
  free_number(h->start_x);
  free_number(h->stop_x);
  free_number(h->dash_y);
  free_number(h->minx);
  free_number(h->miny);
  free_number(h->maxx);
  free_number(h->maxy);
  mp_free_token_node (mp, h->list_);
  mp_free_node (mp, (mp_node)h, edge_header_size);
}
void mp_flush_dash_list (MP mp, mp_edge_header_node h) {
  mp_dash_node p, q; /* pointers that scan the list being recycled */
  q = dash_list (h);
  while (q != mp->null_dash) { /* todo: NULL check should not be needed */
    p = q;
    q = (mp_dash_node)mp_link (q);
    mp_free_node (mp, (mp_node)p, dash_node_size);
  }
  set_dash_list (h,mp->null_dash);
}
mp_edge_header_node mp_toss_gr_object (MP mp, mp_node p) {
  /* returns an edge structure that needs to be dereferenced */
  mp_edge_header_node e = NULL;     /* the edge structure to return */
  switch (mp_type (p)) {
  case mp_fill_node_type:
    mp_free_fill_node (mp, (mp_fill_node)p);
    break;
  case mp_stroked_node_type:
    e = mp_free_stroked_node (mp, (mp_stroked_node)p);
    break;
  case mp_text_node_type:
    mp_free_text_node(mp, (mp_text_node)p);
    break;
  case mp_start_clip_node_type:
    mp_free_start_clip_node(mp, (mp_start_clip_node)p);
    break;
  case mp_start_bounds_node_type:
    mp_free_start_bounds_node(mp, (mp_start_bounds_node)p);
    break;
  case mp_stop_clip_node_type:
    mp_free_stop_clip_node(mp, (mp_stop_clip_node)p);
    break;
  case mp_stop_bounds_node_type:
    mp_free_stop_bounds_node(mp, (mp_stop_bounds_node)p);
    break;
  default:  /* there are no other valid cases, but please the compiler */
    break;
  }
  return e;
}


@ If we use |add_edge_ref| to ``copy'' edge structures, the real copying needs
to be done before making a significant change to an edge structure.  Much of
the work is done in a separate routine |copy_objects| that copies a list of
graphical objects into a new edge header.

@c
static mp_edge_header_node mp_private_edges (MP mp, mp_edge_header_node h) {
  /* make a private copy of the edge structure headed by |h| */
  mp_edge_header_node hh;   /* the edge header for the new copy */
  mp_dash_node p, pp;        /* pointers for copying the dash list */
  assert (mp_type (h) == mp_edge_header_node_type);
  if (edge_ref_count (h) == 0) {
    return h;
  } else {
    decr (edge_ref_count (h));
    hh = (mp_edge_header_node)mp_copy_objects (mp, mp_link (edge_list (h)), NULL);
    @<Copy the dash list from |h| to |hh|@>;
    @<Copy the bounding box information from |h| to |hh| and make |bblast(hh)|
      point into the new object list@>;
    return hh;
  }
}


@ Here we use the fact that |dash_list(hh)=mp_link(hh)|.
@^data structure assumptions@>

@<Copy the dash list from |h| to |hh|@>=
pp = (mp_dash_node)hh;
p = dash_list (h);
while ((p != mp->null_dash)) {
  mp_link (pp) = (mp_node)mp_get_dash_node (mp);
  pp = (mp_dash_node)mp_link (pp);
  number_clone(pp->start_x, p->start_x);
  number_clone(pp->stop_x, p->stop_x);
  p = (mp_dash_node)mp_link (p);
}
mp_link (pp) = (mp_node)mp->null_dash;
number_clone(hh->dash_y, h->dash_y )
 

@ |h| is an edge structure

@c
static mp_dash_object *mp_export_dashes (MP mp, mp_stroked_node q, mp_number w) {
  mp_dash_object *d;
  mp_dash_node p, h;
  mp_number scf;   /* scale factor */
  mp_number dashoff;
  double *dashes = NULL;
  int num_dashes = 1;
  h = (mp_dash_node)mp_dash_p (q);
  if (h == NULL || dash_list (h) == mp->null_dash)
    return NULL;
  new_number (scf);
  p = dash_list (h);
  mp_get_pen_scale (mp, &scf, mp_pen_p (q));
  if (number_zero(scf)) {
    if (number_zero(w)) {
      number_clone(scf, q->dash_scale);
    } else {
      free_number(scf);
      return NULL;
    }
  } else {
    mp_number ret;
    new_number (ret);
    make_scaled (ret, w, scf);
    take_scaled (scf, ret, q->dash_scale);
    free_number (ret);
  }
  number_clone(w, scf);
  d = xmalloc (1, sizeof (mp_dash_object));
  add_var_used (sizeof (mp_dash_object));
  set_number_from_addition(mp->null_dash->start_x, p->start_x, h->dash_y);
  {
    mp_number ret, arg1;
    new_number (ret);
    new_number (arg1);
    new_number (dashoff);
    while (p != mp->null_dash) {
      dashes = xrealloc (dashes, (num_dashes + 2), sizeof (double));
      set_number_from_substraction (arg1, p->stop_x, p->start_x);
      take_scaled (ret, arg1, scf);
      dashes[(num_dashes - 1)] = number_to_double (ret);
      set_number_from_substraction (arg1, ((mp_dash_node)mp_link (p))->start_x, p->stop_x);
      take_scaled (ret, arg1, scf);
      dashes[(num_dashes)] = number_to_double (ret);
      dashes[(num_dashes + 1)] = -1.0;      /* terminus */
      num_dashes += 2;
      p = (mp_dash_node)mp_link (p);
    }
    d->array = dashes;
    mp_dash_offset (mp, &dashoff, h);
    take_scaled (ret, dashoff, scf);
    d->offset = number_to_double(ret);
    free_number (ret);
    free_number (arg1);
  }
  free_number (dashoff);
  free_number(scf);
  return d;
}


@ @<Copy the bounding box information from |h| to |hh|...@>=
number_clone(hh->minx, h->minx);
number_clone(hh->miny, h->miny);
number_clone(hh->maxx, h->maxx);
number_clone(hh->maxy, h->maxy);
hh->bbtype = h->bbtype;
p = (mp_dash_node)edge_list (h);
pp = (mp_dash_node)edge_list (hh);
while ((p != (mp_dash_node)bblast (h))) {
  if (p == NULL)
    mp_confusion (mp, "bblast");
@:this can't happen bblast}{\quad bblast@>;
  p = (mp_dash_node)mp_link (p);
  pp = (mp_dash_node)mp_link (pp);
}
bblast (hh) = (mp_node)pp

@ Here is the promised routine for copying graphical objects into a new edge
structure.  It starts copying at object~|p| and stops just before object~|q|.
If |q| is NULL, it copies the entire sublist headed at |p|.  The resulting edge
structure requires further initialization by |init_bbox|.

@<Declarations@>=
static mp_edge_header_node mp_copy_objects (MP mp, mp_node p, mp_node q);

@ @c
mp_edge_header_node mp_copy_objects (MP mp, mp_node p, mp_node q) {
  mp_edge_header_node hh;   /* the new edge header */
  mp_node pp;   /* the last newly copied object */
  quarterword k = 0;  /* temporary register */
  hh = mp_get_edge_header_node (mp);
  set_dash_list (hh, mp->null_dash);
  edge_ref_count (hh) = 0;
  pp = edge_list (hh);
  while (p != q) {
    @<Make |mp_link(pp)| point to a copy of object |p|, and update |p| and |pp|@>;
  }
  obj_tail (hh) = pp;
  mp_link (pp) = NULL;
  return hh;
}


@ @<Make |mp_link(pp)| point to a copy of object |p|, and update |p| and |pp|@>=
{
  switch (mp_type (p)) {
  case mp_start_clip_node_type:
    k = start_clip_size;
    break;
  case mp_start_bounds_node_type:
    k = start_bounds_size;
    break;
  case mp_fill_node_type:
    k = fill_node_size;
    break;
  case mp_stroked_node_type:
    k = stroked_node_size;
    break;
  case mp_text_node_type:
    k = text_node_size;
    break;
  case mp_stop_clip_node_type:
    k = stop_clip_size;
    break;
  case mp_stop_bounds_node_type:
    k = stop_bounds_size;
    break;
  default:                     /* there are no other valid cases, but please the compiler */
    break;
  }
  mp_link (pp) = malloc_node ((size_t) k);       /* |gr_object| */
  pp = mp_link (pp);
  memcpy (pp, p, (size_t) k);
  pp->link = NULL;
  @<Fix anything in graphical object |pp| that should differ from the
    corresponding field in |p|@>;
  p = mp_link (p);
}


@ @<Fix anything in graphical object |pp| that should differ from the...@>=
switch (mp_type (p)) {
case mp_start_clip_node_type:
  {
    mp_start_clip_node tt = (mp_start_clip_node)pp;
    mp_start_clip_node t =  (mp_start_clip_node)p;
    mp_path_p (tt) = mp_copy_path (mp, mp_path_p (t));
  }
  break;
case mp_start_bounds_node_type:
  {
    mp_start_bounds_node tt = (mp_start_bounds_node)pp;
    mp_start_bounds_node t =  (mp_start_bounds_node)p;
    mp_path_p (tt) = mp_copy_path (mp, mp_path_p (t));
  }
  break;
case mp_fill_node_type:
  {
    mp_fill_node tt = (mp_fill_node)pp;
    mp_fill_node t =  (mp_fill_node)p;
    new_number(tt->red);    number_clone(tt->red,    t->red);
    new_number(tt->green);  number_clone(tt->green,  t->green);
    new_number(tt->blue);   number_clone(tt->blue,   t->blue);
    new_number(tt->black);  number_clone(tt->black,  t->black);
    new_number(tt->miterlim); number_clone(tt->miterlim,t->miterlim);
    mp_path_p (tt) =  mp_copy_path (mp, mp_path_p (t));
    if (mp_pre_script (p) != NULL)
      add_str_ref (mp_pre_script (p));
    if (mp_post_script (p) != NULL)
      add_str_ref (mp_post_script (p));
    if (mp_pen_p (t) != NULL)
      mp_pen_p (tt) = copy_pen (mp_pen_p (t));
  }
  break;
case mp_stroked_node_type:
  {
    mp_stroked_node tt = (mp_stroked_node)pp;
    mp_stroked_node t =  (mp_stroked_node)p;
    new_number(tt->red);        number_clone(tt->red,    t->red);
    new_number(tt->green);      number_clone(tt->green,  t->green);
    new_number(tt->blue);       number_clone(tt->blue,   t->blue);
    new_number(tt->black);      number_clone(tt->black,  t->black);
    new_number(tt->miterlim);   number_clone(tt->miterlim,t->miterlim);
    new_number(tt->dash_scale); number_clone(tt->dash_scale,t->dash_scale);
    if (mp_pre_script (p) != NULL)
      add_str_ref (mp_pre_script (p));
    if (mp_post_script (p) != NULL)
      add_str_ref (mp_post_script (p));
    mp_path_p (tt) =  mp_copy_path (mp, mp_path_p (t));
    mp_pen_p (tt) =  copy_pen (mp_pen_p (t));
    if (mp_dash_p (p) != NULL)
      add_edge_ref (mp_dash_p (pp));
  }
  break;
case mp_text_node_type:
  { 
    mp_text_node tt = (mp_text_node)pp;
    mp_text_node t = (mp_text_node)p;
    new_number(tt->red);    number_clone(tt->red,    t->red);
    new_number(tt->green);  number_clone(tt->green,  t->green);
    new_number(tt->blue);   number_clone(tt->blue,   t->blue);
    new_number(tt->black);  number_clone(tt->black,  t->black);
    new_number(tt->width);  number_clone(tt->width,  t->width);
    new_number(tt->height); number_clone(tt->height, t->height);
    new_number(tt->depth);  number_clone(tt->depth,  t->depth);
    new_number(tt->tx);  number_clone(tt->tx,  t->tx);
    new_number(tt->ty);  number_clone(tt->ty,  t->ty);
    new_number(tt->txx); number_clone(tt->txx, t->txx);
    new_number(tt->tyx); number_clone(tt->tyx, t->tyx);
    new_number(tt->txy); number_clone(tt->txy, t->txy);
    new_number(tt->tyy); number_clone(tt->tyy, t->tyy);
    if (mp_pre_script (p) != NULL)
      add_str_ref (mp_pre_script (p));
    if (mp_post_script (p) != NULL)
      add_str_ref (mp_post_script (p));
    add_str_ref (mp_text_p (pp));
  }
  break;
case mp_stop_clip_node_type:
case mp_stop_bounds_node_type:
  break;
default:                       /* there are no other valid cases, but please the compiler */
  break;
}


@ Here is one way to find an acceptable value for the second argument to
|copy_objects|.  Given a non-NULL graphical object list, |skip_1component|
skips past one picture component, where a ``picture component'' is a single
graphical object, or a start bounds or start clip object and everything up
through the matching stop bounds or stop clip object.  

@c
static mp_node mp_skip_1component (MP mp, mp_node p) {
  integer lev;  /* current nesting level */
  lev = 0;
  (void) mp;
  do {
    if (is_start_or_stop (p)) {
      if (is_stop (p))
        decr (lev);
      else
        incr (lev);
    }
    p = mp_link (p);
  } while (lev != 0);
  return p;
}


@ Here is a diagnostic routine for printing an edge structure in symbolic form.

@<Declarations@>=
static void mp_print_edges (MP mp, mp_node h, const char *s, boolean nuline);

@ @c
void mp_print_edges (MP mp, mp_node h, const char *s, boolean nuline) {
  mp_node p;    /* a graphical object to be printed */
  mp_number scf;   /* a scale factor for the dash pattern */
  boolean ok_to_dash;   /* |false| for polygonal pen strokes */
  new_number (scf);
  mp_print_diagnostic (mp, "Edge structure", s, nuline);
  p = edge_list (h);
  while (mp_link (p) != NULL) {
    p = mp_link (p);
    mp_print_ln (mp);
    switch (mp_type (p)) {
      @<Cases for printing graphical object node |p|@>;
    default:
      mp_print (mp, "[unknown object type!]");
      break;
    }
  }
  mp_print_nl (mp, "End edges");
  if (p != obj_tail (h))
    mp_print (mp, "?");
@.End edges?@>;
  mp_end_diagnostic (mp, true);
  free_number (scf);
}


@ @<Cases for printing graphical object node |p|@>=
case mp_fill_node_type:
mp_print (mp, "Filled contour ");
mp_print_obj_color (mp, p);
mp_print_char (mp, xord (':'));
mp_print_ln (mp);
mp_pr_path (mp, mp_path_p ((mp_fill_node) p));
mp_print_ln (mp);
if ((mp_pen_p ((mp_fill_node) p) != NULL)) {
  @<Print join type for graphical object |p|@>;
  mp_print (mp, " with pen");
  mp_print_ln (mp);
  mp_pr_pen (mp, mp_pen_p ((mp_fill_node) p));
}
break;

@ @<Print join type for graphical object |p|@>=
switch (((mp_stroked_node)p)->ljoin) {
case 0:
  mp_print (mp, "mitered joins limited ");
  print_number (((mp_stroked_node)p)->miterlim);
  break;
case 1:
  mp_print (mp, "round joins");
  break;
case 2:
  mp_print (mp, "beveled joins");
  break;
default:
  mp_print (mp, "?? joins");
@.??@>;
  break;
}


@ For stroked nodes, we need to print |lcap_val(p)| as well.

@<Print join and cap types for stroked node |p|@>=
switch (((mp_stroked_node)p)->lcap ) {
case 0:
  mp_print (mp, "butt");
  break;
case 1:
  mp_print (mp, "round");
  break;
case 2:
  mp_print (mp, "square");
  break;
default:
  mp_print (mp, "??");
  break;
@.??@>
}
mp_print (mp, " ends, ");
@<Print join type for graphical object |p|@>
 

@ Here is a routine that prints the color of a graphical object if it isn't
black (the default color).

@<Declarations@>=
static void mp_print_obj_color (MP mp, mp_node p);

@ @c
void mp_print_obj_color (MP mp, mp_node p) {
  mp_stroked_node p0 = (mp_stroked_node) p;
  if (mp_color_model (p) == mp_grey_model) {
    if (number_positive(p0->grey)) {
      mp_print (mp, "greyed ");
      mp_print_char (mp, xord ('('));
      print_number (p0->grey);
      mp_print_char (mp, xord (')'));
    };
  } else if (mp_color_model (p) == mp_cmyk_model) {
    if (number_positive(p0->cyan) || number_positive(p0->magenta) ||
        number_positive(p0->yellow) || number_positive(p0->black)) {
      mp_print (mp, "processcolored ");
      mp_print_char (mp, xord ('('));
      print_number (p0->cyan);
      mp_print_char (mp, xord (','));
      print_number (p0->magenta);
      mp_print_char (mp, xord (','));
      print_number (p0->yellow);
      mp_print_char (mp, xord (','));
      print_number (p0->black);
      mp_print_char (mp, xord (')'));
    };
  } else if (mp_color_model (p) == mp_rgb_model) {
    if (number_positive(p0->red) || number_positive(p0->green) || 
	number_positive(p0->blue)) {
      mp_print (mp, "colored ");
      mp_print_char (mp, xord ('('));
      print_number (p0->red);
      mp_print_char (mp, xord (','));
      print_number (p0->green);
      mp_print_char (mp, xord (','));
      print_number (p0->blue);
      mp_print_char (mp, xord (')'));
    };
  }
}


@ @<Cases for printing graphical object node |p|@>=
case mp_stroked_node_type:
mp_print (mp, "Filled pen stroke ");
mp_print_obj_color (mp, p);
mp_print_char (mp, xord (':'));
mp_print_ln (mp);
mp_pr_path (mp, mp_path_p ((mp_stroked_node) p));
if (mp_dash_p (p) != NULL) {
  mp_print_nl (mp, "dashed (");
  @<Finish printing the dash pattern that |p| refers to@>;
}
mp_print_ln (mp);
@<Print join and cap types for stroked node |p|@>;
mp_print (mp, " with pen");
mp_print_ln (mp);
if (mp_pen_p ((mp_stroked_node) p) == NULL) {
  mp_print (mp, "???");         /* shouldn't happen */
@.???@>
} else {
  mp_pr_pen (mp, mp_pen_p ((mp_stroked_node) p));
}
break;

@ Normally, the  |dash_list| field in an edge header is set to |null_dash|
when it is not known to define a suitable dash pattern.  This is disallowed
here because the |mp_dash_p| field should never point to such an edge header.
Note that memory is allocated for |start_x(null_dash)| and we are free to
give it any convenient value.

@<Finish printing the dash pattern that |p| refers to@>=
{
mp_dash_node ppd, hhd;
ok_to_dash = pen_is_elliptical (mp_pen_p ((mp_stroked_node) p));
if (!ok_to_dash)
  set_number_to_unity (scf);
else
  number_clone(scf, ((mp_stroked_node) p)->dash_scale);
hhd = (mp_dash_node)mp_dash_p (p);
ppd = dash_list (hhd);
if ((ppd == mp->null_dash) || number_negative(hhd->dash_y)) {
  mp_print (mp, " ??");
} else {
  mp_number dashoff;
  mp_number ret, arg1;
  new_number (ret);
  new_number (arg1);
  new_number (dashoff);
  set_number_from_addition(mp->null_dash->start_x, ppd->start_x, hhd->dash_y );
  while (ppd != mp->null_dash) {
    mp_print (mp, "on ");
    set_number_from_substraction (arg1, ppd->stop_x, ppd->start_x);
    take_scaled (ret, arg1, scf);
    print_number ( ret);
    mp_print (mp, " off ");
    set_number_from_substraction (arg1, ((mp_dash_node)mp_link (ppd))->start_x, ppd->stop_x);
    take_scaled (ret, arg1, scf);
    print_number (ret);
    ppd = (mp_dash_node)mp_link (ppd);
    if (ppd != mp->null_dash)
      mp_print_char (mp, xord (' '));
  }
  mp_print (mp, ") shifted ");
  mp_dash_offset (mp, &dashoff, hhd);
  take_scaled (ret, dashoff, scf);
  number_negate (ret);
  print_number (ret);
  free_number (dashoff);
  free_number (ret);
  free_number (arg1);
  if (!ok_to_dash || number_zero(hhd->dash_y) )
    mp_print (mp, " (this will be ignored)");
}
}

@ @<Declarations@>=
static void mp_dash_offset (MP mp, mp_number *x, mp_dash_node h);

@ @c
void mp_dash_offset (MP mp, mp_number *x, mp_dash_node h) {
  if (dash_list (h) == mp->null_dash || number_negative(h->dash_y ))
    mp_confusion (mp, "dash0");
@:this can't happen dash0}{\quad dash0@>;
  if (number_zero(h->dash_y)) {
    set_number_to_zero(*x); 
  } else {
    number_clone (*x, (dash_list (h))->start_x );
    number_modulo (*x, h->dash_y);
    number_negate (*x);
    if (number_negative(*x))
      number_add(*x, h->dash_y);
  }
}


@ @<Cases for printing graphical object node |p|@>=
case mp_text_node_type:
{
mp_text_node p0 = (mp_text_node)p;
mp_print_char (mp, xord ('"'));
mp_print_str (mp, mp_text_p (p));
mp_print (mp, "\" infont \"");
mp_print (mp, mp->font_name[mp_font_n (p)]);
mp_print_char (mp, xord ('"'));
mp_print_ln (mp);
mp_print_obj_color (mp, p);
mp_print (mp, "transformed ");
mp_print_char (mp, xord ('('));
print_number (p0->tx);
mp_print_char (mp, xord (','));
print_number (p0->ty);
mp_print_char (mp, xord (','));
print_number (p0->txx);
mp_print_char (mp, xord (','));
print_number (p0->txy);
mp_print_char (mp, xord (','));
print_number (p0->tyx);
mp_print_char (mp, xord (','));
print_number (p0->tyy);
mp_print_char (mp, xord (')'));
}
break;

@ @<Cases for printing graphical object node |p|@>=
case mp_start_clip_node_type:
mp_print (mp, "clipping path:");
mp_print_ln (mp);
mp_pr_path (mp, mp_path_p ((mp_start_clip_node) p));
break;
case mp_stop_clip_node_type:
mp_print (mp, "stop clipping");
break;

@ @<Cases for printing graphical object node |p|@>=
case mp_start_bounds_node_type:
mp_print (mp, "setbounds path:");
mp_print_ln (mp);
mp_pr_path (mp, mp_path_p ((mp_start_bounds_node) p));
break;
case mp_stop_bounds_node_type:
mp_print (mp, "end of setbounds");
break;

@ To initialize the |dash_list| field in an edge header~|h|, we need a
subroutine that scans an edge structure and tries to interpret it as a dash
pattern.  This can only be done when there are no filled regions or clipping
paths and all the pen strokes have the same color.  The first step is to let
$y_0$ be the initial $y$~coordinate of the first pen stroke.  Then we implicitly
project all the pen stroke paths onto the line $y=y_0$ and require that there
be no retracing.  If the resulting paths cover a range of $x$~coordinates of
length $\Delta x$, we set |dash_y(h)| to the length of the dash pattern by
finding the maximum of $\Delta x$ and the absolute value of~$y_0$.

@c
static mp_edge_header_node mp_make_dashes (MP mp, mp_edge_header_node h) { /* returns |h| or |NULL| */
  mp_node p;    /* this scans the stroked nodes in the object list */
  mp_node p0;   /* if not |NULL| this points to the first stroked node */
  mp_knot pp, qq, rr;   /* pointers into |mp_path_p(p)| */
  mp_dash_node d, dd;        /* pointers used to create the dash list */
  mp_number y0;
  @<Other local variables in |make_dashes|@>;
  if (dash_list (h) != mp->null_dash)
    return h;
  new_number (y0);                       /* the initial $y$ coordinate */
  p0 = NULL;
  p = mp_link (edge_list (h));
  while (p != NULL) {
    if (mp_type (p) != mp_stroked_node_type) {
      @<Compain that the edge structure contains a node of the wrong type
        and |goto not_found|@>;
    }
    pp = mp_path_p ((mp_stroked_node) p);
    if (p0 == NULL) {
      p0 = p;
      number_clone(y0, pp->y_coord);
    }
    @<Make |d| point to a new dash node created from stroke |p| and path |pp|
      or |goto not_found| if there is an error@>;
    @<Insert |d| into the dash list and |goto not_found| if there is an error@>;
    p = mp_link (p);
  }
  if (dash_list (h) == mp->null_dash)
    goto NOT_FOUND;             /* No error message */
  @<Scan |dash_list(h)| and deal with any dashes that are themselves dashed@>;
  @<Set |dash_y(h)| and merge the first and last dashes if necessary@>;
  free_number (y0);
  return h;
NOT_FOUND:
  free_number (y0);
  @<Flush the dash list, recycle |h| and return |NULL|@>;
}


@ @<Compain that the edge structure contains a node of the wrong type...@>=
{
  const char *hlp[] = {
         "When you say `dashed p', picture p should not contain any",
         "text, filled regions, or clipping paths.  This time it did",
         "so I'll just make it a solid line instead.",
         NULL };
  mp_back_error (mp, "Picture is too complicated to use as a dash pattern", hlp, true);
  mp_get_x_next (mp);
  goto NOT_FOUND;
}


@ A similar error occurs when monotonicity fails.

@<Declarations@>=
static void mp_x_retrace_error (MP mp);

@ @c
void mp_x_retrace_error (MP mp) {
  const char *hlp[] = {
         "When you say `dashed p', every path in p should be monotone",
         "in x and there must be no overlapping.  This failed",
         "so I'll just make it a solid line instead.",
         NULL };
  mp_back_error (mp, "Picture is too complicated to use as a dash pattern", hlp, true);
  mp_get_x_next (mp);
}


@ We stash |p| in |dash_info(d)| if |mp_dash_p(p)<>0| so that subsequent processing can
handle the case where the pen stroke |p| is itself dashed.

@d dash_info(A) ((mp_dash_node)(A))->dash_info_  /* in an edge header this points to the first dash node */

@<Make |d| point to a new dash node created from stroke |p| and path...@>=
@<Make sure |p| and |p0| are the same color and |goto not_found| if there is
  an error@>;
rr = pp;
if (mp_next_knot (pp) != pp) {
  do {
    qq = rr;
    rr = mp_next_knot (rr);
    @<Check for retracing between knots |qq| and |rr| and |goto not_found|
      if there is a problem@>;
  } while (mp_right_type (rr) != mp_endpoint);
}
d = (mp_dash_node)mp_get_dash_node (mp);
if (mp_dash_p (p) == NULL)
  dash_info (d) = NULL;
else
  dash_info (d) = p;
if (number_less (pp->x_coord, rr->x_coord)) {
  number_clone(d->start_x, pp->x_coord);
  number_clone(d->stop_x, rr->x_coord);
} else {
  number_clone(d->start_x, rr->x_coord);
  number_clone(d->stop_x, pp->x_coord);
}


@ We also need to check for the case where the segment from |qq| to |rr| is
monotone in $x$ but is reversed relative to the path from |pp| to |qq|.

@<Check for retracing between knots |qq| and |rr| and |goto not_found|...@>=
{
  mp_number x0, x1, x2, x3;  /* $x$ coordinates of the segment from |qq| to |rr| */
  new_number(x0);
  new_number(x1);
  new_number(x2);
  new_number(x3);
  number_clone(x0, qq->x_coord);
  number_clone(x1, qq->right_x);
  number_clone(x2, rr->left_x);
  number_clone(x3, rr->x_coord);
  if (number_greater(x0, x1) || number_greater(x1, x2) || number_greater(x2, x3)) {
    if (number_less(x0, x1) || number_less(x1, x2) || number_less(x2, x3)) {
      mp_number a1, a2, a3, a4;
      mp_number test;
      new_number(test);
      new_number(a1);
      new_number(a2);
      new_number(a3);
      new_number(a4);
      set_number_from_substraction(a1, x2, x1);
      set_number_from_substraction(a2, x2, x1);
      set_number_from_substraction(a3, x1, x0);
      set_number_from_substraction(a4, x3, x2);
      ab_vs_cd (test, a1, a2, a3, a4);
      free_number(a1);
      free_number(a2);
      free_number(a3);
      free_number(a4);
      if (number_positive(test)) {
        mp_x_retrace_error (mp);
        free_number(x0);
        free_number(x1);
        free_number(x2);
        free_number(x3);
        free_number(test);
        goto NOT_FOUND;
      }
      free_number(test);
    }
  }
  if (number_greater(pp->x_coord, x0) || number_greater(x0, x3)) {
    if (number_less (pp->x_coord, x0) || number_less(x0, x3)) {
      mp_x_retrace_error (mp);
      free_number(x0);
      free_number(x1);
      free_number(x2);
      free_number(x3);
      goto NOT_FOUND;
    }
  }
  free_number(x0);
  free_number(x1);
  free_number(x2);
  free_number(x3);
}

@ @<Make sure |p| and |p0| are the same color and |goto not_found|...@>=
if (!number_equal(((mp_stroked_node)p)->red, ((mp_stroked_node)p0)->red) || 
    !number_equal(((mp_stroked_node)p)->black, ((mp_stroked_node)p0)->black) ||
    !number_equal(((mp_stroked_node)p)->green, ((mp_stroked_node)p0)->green) || 
    !number_equal(((mp_stroked_node)p)->blue, ((mp_stroked_node)p0)->blue)
    ) {
  const char *hlp[] = {
         "When you say `dashed p', everything in picture p should",
         "be the same color.  I can\'t handle your color changes",
         "so I'll just make it a solid line instead.",
         NULL };
  mp_back_error (mp, "Picture is too complicated to use as a dash pattern", hlp, true);
  mp_get_x_next (mp);
  goto NOT_FOUND;
}

@ @<Insert |d| into the dash list and |goto not_found| if there is an error@>=
number_clone(mp->null_dash->start_x, d->stop_x);
dd = (mp_dash_node)h;                         /* this makes |mp_link(dd)=dash_list(h)| */
while (number_less(((mp_dash_node)mp_link (dd))->start_x, d->stop_x ))
  dd = (mp_dash_node)mp_link (dd);
if (dd != (mp_dash_node)h) {
  if (number_greater(dd->stop_x, d->start_x)) {
    mp_x_retrace_error (mp);
    goto NOT_FOUND;
  };
}
mp_link (d) = mp_link (dd);
mp_link (dd) = (mp_node)d

@ @<Set |dash_y(h)| and merge the first and last dashes if necessary@>=
d = dash_list (h);
while ((mp_link (d) != (mp_node)mp->null_dash))
  d = (mp_dash_node)mp_link (d);
dd = dash_list (h);
set_number_from_substraction(h->dash_y, d->stop_x, dd->start_x);
{
  mp_number absval;
  new_number (absval);
  number_clone (absval, y0);
  number_abs (absval);
  if (number_greater (absval, h->dash_y) ) {
    number_clone(h->dash_y, absval);
  } else if (d != dd) {
    set_dash_list (h, mp_link (dd));
    set_number_from_addition(d->stop_x, dd->stop_x, h->dash_y);
    mp_free_node (mp, (mp_node)dd, dash_node_size);
  }
  free_number (absval);
}
@ We get here when the argument is a NULL picture or when there is an error.
Recovering from an error involves making |dash_list(h)| empty to indicate
that |h| is not known to be a valid dash pattern.  We also dereference |h|
since it is not being used for the return value.

@<Flush the dash list, recycle |h| and return |NULL|@>=
mp_flush_dash_list (mp, h);
delete_edge_ref (h);
return NULL

@ Having carefully saved the dashed stroked nodes in the
corresponding dash nodes, we must be prepared to break up these dashes into
smaller dashes.

@<Scan |dash_list(h)| and deal with any dashes that are themselves dashed@>=
{
mp_number hsf;     /* the dash pattern from |hh| gets scaled by this */
new_number (hsf);
d = (mp_dash_node)h;                          /* now |mp_link(d)=dash_list(h)| */
while (mp_link (d) != (mp_node)mp->null_dash) {
  ds = dash_info (mp_link (d));
  if (ds == NULL) {
    d = (mp_dash_node)mp_link (d);
  } else {
    hh = (mp_edge_header_node)mp_dash_p (ds);
    number_clone(hsf, ((mp_stroked_node)ds)->dash_scale);
    if (hh == NULL)
      mp_confusion (mp, "dash1");
@:this can't happen dash0}{\quad dash1@>;
    /* clang: dereference null pointer 'hh' */ assert(hh); 
    if (number_zero(((mp_dash_node)hh)->dash_y )) {
      d = (mp_dash_node)mp_link (d);
    } else {
      if (dash_list (hh) == NULL)
        mp_confusion (mp, "dash1");
@:this can't happen dash0}{\quad dash1@>;
      @<Replace |mp_link(d)| by a dashed version as determined by edge header
          |hh| and scale factor |ds|@>;
    }
  }
}
free_number (hsf);
}

@ @<Other local variables in |make_dashes|@>=
mp_dash_node dln;    /* |mp_link(d)| */
mp_edge_header_node hh;     /* an edge header that tells how to break up |dln| */
mp_node ds;     /* the stroked node from which |hh| and |hsf| are derived */

@ @<Replace |mp_link(d)| by a dashed version as determined by edge header...@>=
{
  mp_number xoff;    /* added to $x$ values in |dash_list(hh)| to match |dln| */
  mp_number dashoff;
  mp_number r1, r2;
  new_number (r1);
  new_number (r2);
  dln = (mp_dash_node)mp_link (d);
  dd = dash_list (hh); 
  /* clang: dereference null pointer 'dd' */ assert(dd); 
  new_number (xoff);
  new_number (dashoff);
  mp_dash_offset (mp, &dashoff, (mp_dash_node)hh);
  take_scaled (r1, hsf, dd->start_x);
  take_scaled (r2, hsf, dashoff);
  number_add (r1, r2);
  set_number_from_substraction(xoff, dln->start_x, r1);
  free_number (dashoff);
  take_scaled (r1, hsf, dd->start_x);
  take_scaled (r2, hsf, hh->dash_y);
  set_number_from_addition(mp->null_dash->start_x, r1, r2);
  number_clone(mp->null_dash->stop_x, mp->null_dash->start_x);
  @<Advance |dd| until finding the first dash that overlaps |dln| when
    offset by |xoff|@>;
  while (number_lessequal(dln->start_x, dln->stop_x)) {
    @<If |dd| has `fallen off the end', back up to the beginning and fix |xoff|@>;
    @<Insert a dash between |d| and |dln| for the overlap with the offset version
      of |dd|@>;
    dd = (mp_dash_node)mp_link (dd);
    take_scaled (r1, hsf, dd->start_x);
    set_number_from_addition(dln->start_x , xoff, r1);
  }
  free_number(xoff);
  free_number (r1);
  free_number (r2);
  mp_link (d) = mp_link (dln);
  mp_free_node (mp, (mp_node)dln, dash_node_size);
}
 

@ The name of this module is a bit of a lie because we just find the
first |dd| where |take_scaled (hsf, stop_x(dd))| is large enough to make an
overlap possible.  It could be that the unoffset version of dash |dln| falls
in the gap between |dd| and its predecessor.

@<Advance |dd| until finding the first dash that overlaps |dln| when...@>=
{
  mp_number r1;
  new_number (r1);
  take_scaled (r1, hsf, dd->stop_x);
  number_add (r1, xoff);
  while (number_less(r1, dln->start_x)) {
    dd = (mp_dash_node)mp_link (dd);
    take_scaled (r1, hsf, dd->stop_x);
    number_add (r1, xoff);
  }
  free_number (r1);
}

@ @<If |dd| has `fallen off the end', back up to the beginning and fix...@>=
if (dd == mp->null_dash) {
  mp_number ret;
  new_number (ret);
  dd = dash_list (hh);
  take_scaled (ret, hsf, hh->dash_y);
  number_add(xoff, ret);
  free_number (ret);
}

@ At this point we already know that |start_x(dln)<=xoff+take_scaled(hsf,stop_x(dd))|.

@<Insert a dash between |d| and |dln| for the overlap with the offset...@>=
{
  mp_number r1;
  new_number (r1);
  take_scaled (r1, hsf, dd->start_x);
  number_add (r1, xoff);
  if (number_lessequal(r1, dln->stop_x)) {
    mp_link (d) = (mp_node)mp_get_dash_node (mp);
    d = (mp_dash_node)mp_link (d);
    mp_link (d) = (mp_node)dln;
    take_scaled (r1, hsf, dd->start_x );
    number_add (r1, xoff);
    if (number_greater(dln->start_x, r1))
      number_clone(d->start_x, dln->start_x);
    else {
      number_clone(d->start_x, r1);
    }
    take_scaled (r1, hsf, dd->stop_x);
    number_add (r1, xoff);
    if (number_less(dln->stop_x, r1))
      number_clone(d->stop_x, dln->stop_x );
    else {
      number_clone(d->stop_x, r1);
    }
  }
  free_number (r1);
}

@ The next major task is to update the bounding box information in an edge
header~|h|. This is done via a procedure |adjust_bbox| that enlarges an edge
header's bounding box to accommodate the box computed by |path_bbox| or
|pen_bbox|. (This is stored in global variables |minx|, |miny|, |maxx|, and
|maxy|.)

@c
static void mp_adjust_bbox (MP mp, mp_edge_header_node h) {
  if (number_less (mp_minx, h->minx))
    number_clone(h->minx, mp_minx);
  if (number_less (mp_miny, h->miny))
    number_clone(h->miny, mp_miny);
  if (number_greater (mp_maxx, h->maxx))
    number_clone(h->maxx, mp_maxx);
  if (number_greater (mp_maxy, h->maxy))
    number_clone(h->maxy, mp_maxy);
}


@ Here is a special routine for updating the bounding box information in
edge header~|h| to account for the squared-off ends of a non-cyclic path~|p|
that is to be stroked with the pen~|pp|.

@c
static void mp_box_ends (MP mp, mp_knot p, mp_knot pp, mp_edge_header_node h) {
  mp_knot q;    /* a knot node adjacent to knot |p| */
  mp_fraction dx, dy;      /* a unit vector in the direction out of the path at~|p| */
  mp_number d;     /* a factor for adjusting the length of |(dx,dy)| */
  mp_number z;     /* a coordinate being tested against the bounding box */
  mp_number xx, yy;        /* the extreme pen vertex in the |(dx,dy)| direction */
  integer i;    /* a loop counter */
  new_fraction(dx);
  new_fraction(dy);
  new_number(xx);
  new_number(yy);
  new_number(z);
  new_number(d);
  if (mp_right_type (p) != mp_endpoint) {
    q = mp_next_knot (p);
    while (1) {
      @<Make |(dx,dy)| the final direction for the path segment from
        |q| to~|p|; set~|d|@>;
      pyth_add (d, dx, dy);
      if (number_positive(d)) {
        @<Normalize the direction |(dx,dy)| and find the pen offset |(xx,yy)|@>;
        for (i = 1; i <= 2; i++) {
          @<Use |(dx,dy)| to generate a vertex of the square end cap and
             update the bounding box to accommodate it@>;
          number_negate(dx);
          number_negate(dy);
        }
      }
      if (mp_right_type (p) == mp_endpoint) {
        goto DONE;
      } else {
        @<Advance |p| to the end of the path and make |q| the previous knot@>;
      }
    }
  }
DONE:
  free_number (dx);
  free_number (dy);
  free_number (xx);
  free_number (yy);
  free_number (z);
  free_number (d);
}


@ @<Make |(dx,dy)| the final direction for the path segment from...@>=
if (q == mp_next_knot (p)) {
  set_number_from_substraction(dx, p->x_coord, p->right_x);
  set_number_from_substraction(dy, p->y_coord, p->right_y);
  if (number_zero(dx) && number_zero(dy)) {
    set_number_from_substraction(dx, p->x_coord, q->left_x);
    set_number_from_substraction(dy, p->y_coord, q->left_y);
  }
} else {
  set_number_from_substraction(dx, p->x_coord, p->left_x);
  set_number_from_substraction(dy, p->y_coord, p->left_y);
  if (number_zero(dx) && number_zero(dy)) {
    set_number_from_substraction(dx, p->x_coord, q->right_x);
    set_number_from_substraction(dy, p->y_coord, q->right_y);
  }
}
set_number_from_substraction(dx, p->x_coord, q->x_coord);
set_number_from_substraction(dy, p->y_coord, q->y_coord);
 

@ @<Normalize the direction |(dx,dy)| and find the pen offset |(xx,yy)|@>=
{
  mp_number arg1, r;
  new_fraction (r);
  new_number(arg1);
  make_fraction (r, dx, d);
  number_clone(dx, r);
  make_fraction (r, dy, d);
  number_clone(dy, r);
  free_number (r);
  number_clone(arg1, dy);
  number_negate(arg1);
  mp_find_offset (mp, arg1, dx, pp);
  free_number(arg1);
  number_clone(xx, mp->cur_x);
  number_clone(yy, mp->cur_y);
}

@ @<Use |(dx,dy)| to generate a vertex of the square end cap and...@>=
{
  mp_number r1, r2, arg1;
  new_number (arg1);
  new_fraction (r1);
  new_fraction (r2);
  mp_find_offset (mp, dx, dy, pp);
  set_number_from_substraction (arg1, xx, mp->cur_x);
  take_fraction (r1, arg1, dx);
  set_number_from_substraction (arg1, yy, mp->cur_y);
  take_fraction (r2, arg1, dy); 
  set_number_from_addition(d, r1, r2);
  if ((number_negative(d) && (i == 1)) || (number_positive(d) && (i == 2)))
    mp_confusion (mp, "box_ends");
@:this can't happen box ends}{\quad\\{box\_ends}@>;
  take_fraction (r1, d, dx);
  set_number_from_addition(z, p->x_coord, mp->cur_x);
  number_add (z, r1);
  if (number_less(z, h->minx))
    number_clone(h->minx, z);
  if (number_greater(z, h->maxx))
    number_clone(h->maxx, z);
  take_fraction (r1, d, dy);
  set_number_from_addition(z, p->y_coord, mp->cur_y);
  number_add (z, r1);
  if (number_less(z, h->miny))
    number_clone(h->miny, z);
  if (number_greater(z, h->maxy))
    number_clone(h->maxy, z);
  free_number (r1);
  free_number (r2);
  free_number (arg1);
}

@ @<Advance |p| to the end of the path and make |q| the previous knot@>=
do {
  q = p;
  p = mp_next_knot (p);
} while (mp_right_type (p) != mp_endpoint)

@ The major difficulty in finding the bounding box of an edge structure is the
effect of clipping paths.  We treat them conservatively by only clipping to the
clipping path's bounding box, but this still
requires recursive calls to |set_bbox| in order to find the bounding box of
@^recursion@>
the objects to be clipped.  Such calls are distinguished by the fact that the
boolean parameter |top_level| is false.

@c
void mp_set_bbox (MP mp, mp_edge_header_node h, boolean top_level) {
  mp_node p;    /* a graphical object being considered */
  integer lev;  /* nesting level for |mp_start_bounds_node| nodes */
  /* Wipe out any existing bounding box information if |bbtype(h)| is
     incompatible with |internal[mp_true_corners]| */
  switch (h->bbtype ) {
  case no_bounds:
    break;
  case bounds_set:
    if (number_positive(internal_value (mp_true_corners)))
      mp_init_bbox (mp, h);
    break;
  case bounds_unset:
    if (number_nonpositive(internal_value (mp_true_corners)))
      mp_init_bbox (mp, h);
    break;
  } /* there are no other cases */

  while (mp_link (bblast (h)) != NULL) {
    p = mp_link (bblast (h));
    bblast (h) = p;
    switch (mp_type (p)) {
    case mp_stop_clip_node_type:
      if (top_level)
        mp_confusion (mp, "bbox");
      else
        return;
@:this can't happen bbox}{\quad bbox@>;
      break;
      @<Other cases for updating the bounding box based on the type of object |p|@>;
    default:                   /* there are no other valid cases, but please the compiler */
      break;
    }
  }
  if (!top_level)
    mp_confusion (mp, "bbox");
}


@ @<Declarations@>=
static void mp_set_bbox (MP mp, mp_edge_header_node h, boolean top_level);


@ @<Other cases for updating the bounding box...@>=
case mp_fill_node_type:
  mp_path_bbox (mp, mp_path_p ((mp_fill_node) p));
  if (mp_pen_p ((mp_fill_node) p) != NULL) {
    mp_number x0a, y0a, x1a, y1a;
    new_number (x0a);
    new_number (y0a);
    new_number (x1a);
    new_number (y1a);
    number_clone (x0a, mp_minx);
    number_clone (y0a, mp_miny);
    number_clone (x1a, mp_maxx);
    number_clone (y1a, mp_maxy);
    mp_pen_bbox (mp, mp_pen_p ((mp_fill_node) p));
    number_add (mp_minx, x0a);
    number_add (mp_miny, y0a);
    number_add (mp_maxx, x1a);
    number_add (mp_maxy, y1a);
    free_number (x0a);
    free_number (y0a);
    free_number (x1a);
    free_number (y1a);
  }
  mp_adjust_bbox (mp, h);
break;

@ @<Other cases for updating the bounding box...@>=
case mp_start_bounds_node_type:
  if (number_positive (internal_value (mp_true_corners))) {
    h->bbtype = bounds_unset;
  } else {
    h->bbtype = bounds_set;
    mp_path_bbox (mp, mp_path_p ((mp_start_bounds_node) p));
    mp_adjust_bbox (mp, h);
    @<Scan to the matching |mp_stop_bounds_node| node and update |p| and
      |bblast(h)|@>;
  }
  break;
case mp_stop_bounds_node_type:
  if (number_nonpositive (internal_value (mp_true_corners)))
    mp_confusion (mp, "bbox2");
@:this can't happen bbox2}{\quad bbox2@>;
  break;

@ @<Scan to the matching |mp_stop_bounds_node| node and update |p| and...@>=
lev = 1;
while (lev != 0) {
  if (mp_link (p) == NULL)
    mp_confusion (mp, "bbox2");
@:this can't happen bbox2}{\quad bbox2@>;
  /* clang: dereference null pointer */ assert(mp_link(p));   
  p = mp_link (p);
  if (mp_type (p) == mp_start_bounds_node_type)
    incr (lev);
  else if (mp_type (p) == mp_stop_bounds_node_type)
    decr (lev);
}
bblast (h) = p

@ It saves a lot of grief here to be slightly conservative and not account for
omitted parts of dashed lines.  We also don't worry about the material omitted
when using butt end caps.  The basic computation is for round end caps and
|box_ends| augments it for square end caps.

@<Other cases for updating the bounding box...@>=
case mp_stroked_node_type:
mp_path_bbox (mp, mp_path_p ((mp_stroked_node) p));
{
    mp_number x0a, y0a, x1a, y1a;
    new_number (x0a);
    new_number (y0a);
    new_number (x1a);
    new_number (y1a);
    number_clone (x0a, mp_minx);
    number_clone (y0a, mp_miny);
    number_clone (x1a, mp_maxx);
    number_clone (y1a, mp_maxy);
    mp_pen_bbox (mp, mp_pen_p ((mp_stroked_node) p));
    number_add (mp_minx, x0a);
    number_add (mp_miny, y0a);
    number_add (mp_maxx, x1a);
    number_add (mp_maxy, y1a);
    free_number (x0a);
    free_number (y0a);
    free_number (x1a);
    free_number (y1a);
}
mp_adjust_bbox (mp, h);
if ((mp_left_type (mp_path_p ((mp_stroked_node) p)) == mp_endpoint)
    && (((mp_stroked_node) p)->lcap == 2))
  mp_box_ends (mp, mp_path_p ((mp_stroked_node) p),
             mp_pen_p ((mp_stroked_node) p), h);
break;

@ The height width and depth information stored in a text node determines a
rectangle that needs to be transformed according to the transformation
parameters stored in the text node.

@<Other cases for updating the bounding box...@>=
case mp_text_node_type:
{
  mp_number x0a, y0a, x1a, y1a, arg1;
  mp_text_node p0 = (mp_text_node)p;
  new_number (x0a);
  new_number (x1a);
  new_number (y0a);
  new_number (y1a);
  new_number (arg1);
  number_clone (arg1, p0->depth);
  number_negate (arg1);
  take_scaled (x1a, p0->txx, p0->width);
  take_scaled (y0a, p0->txy, arg1);
  take_scaled (y1a, p0->txy, p0->height);
  number_clone (mp_minx, p0->tx);
  number_clone (mp_maxx, mp_minx);
  if (number_less(y0a, y1a)) {
    number_add (mp_minx, y0a);
    number_add (mp_maxx, y1a);
  } else {
    number_add (mp_minx, y1a);
    number_add (mp_maxx, y0a);
  }
  if (number_negative(x1a))
    number_add (mp_minx, x1a);
  else
    number_add (mp_maxx, x1a);
  take_scaled (x1a, p0->tyx, p0->width);
  number_clone (arg1, p0->depth);
  number_negate (arg1);
  take_scaled (y0a, p0->tyy, arg1);
  take_scaled (y1a, p0->tyy, p0->height);
  number_clone (mp_miny, p0->ty);
  number_clone (mp_maxy, mp_miny);
  if (number_less (y0a, y1a)) {
    number_add (mp_miny, y0a);
    number_add (mp_maxy, y1a);
  } else {
    number_add (mp_miny, y1a);
    number_add (mp_maxy, y0a);
  }
  if (number_negative(x1a))
    number_add (mp_miny, x1a);
  else
    number_add (mp_maxy, x1a);
  mp_adjust_bbox (mp, h);
  free_number (x0a);
  free_number (y0a);
  free_number (x1a);
  free_number (y1a);
  free_number (arg1);
}
break;

@ This case involves a recursive call that advances |bblast(h)| to the node of
type |mp_stop_clip_node| that matches |p|.

@<Other cases for updating the bounding box...@>=
case mp_start_clip_node_type:
{
  mp_number sminx, sminy, smaxx, smaxy;
  /* for saving the bounding box during recursive calls */
  mp_number x0a, y0a, x1a, y1a;
    new_number (x0a);
    new_number (y0a);
    new_number (x1a);
    new_number (y1a);
    new_number (sminx);
    new_number (sminy);
    new_number (smaxx);
    new_number (smaxy);
mp_path_bbox (mp, mp_path_p ((mp_start_clip_node) p));
number_clone (x0a, mp_minx);
number_clone (y0a, mp_miny);
number_clone (x1a, mp_maxx);
number_clone (y1a, mp_maxy);
number_clone (sminx, h->minx);
number_clone (sminy, h->miny);
number_clone (smaxx, h->maxx);
number_clone (smaxy, h->maxy);
@<Reinitialize the bounding box in header |h| and call |set_bbox| recursively
    starting at |mp_link(p)|@>;
@<Clip the bounding box in |h| to the rectangle given by |x0a|, |x1a|,
    |y0a|, |y1a|@>;
number_clone (mp_minx, sminx);
number_clone (mp_miny, sminy);
number_clone (mp_maxx, smaxx);
number_clone (mp_maxy, smaxy);
mp_adjust_bbox (mp, h);
    free_number (sminx);
    free_number (sminy);
    free_number (smaxx);
    free_number (smaxy);
    free_number (x0a);
    free_number (y0a);
    free_number (x1a);
    free_number (y1a);
}
break;

@ @<Reinitialize the bounding box in header |h| and call |set_bbox|...@>=
set_number_to_inf(h->minx);
set_number_to_inf(h->miny);
set_number_to_neg_inf(h->maxx);
set_number_to_neg_inf(h->maxy);
mp_set_bbox (mp, h, false)
 

@ @<Clip the bounding box in |h| to the rectangle given by |x0a|, |x1a|,...@>=
if (number_less(h->minx, x0a))
  number_clone(h->minx, x0a);
if (number_less(h->miny, y0a))
  number_clone(h->miny, y0a);
if (number_greater(h->maxx, x1a))
  number_clone(h->maxx, x1a);
if (number_greater(h->maxy, y1a))
  number_clone(h->maxy, y1a);

@* Finding an envelope.
When \MP\ has a path and a polygonal pen, it needs to express the desired
shape in terms of things \ps\ can understand.  The present task is to compute
a new path that describes the region to be filled.  It is convenient to
define this as a two step process where the first step is determining what
offset to use for each segment of the path.

@ Given a pointer |c| to a cyclic path,
and a pointer~|h| to the first knot of a pen polygon,
the |offset_prep| routine changes the path into cubics that are
associated with particular pen offsets. Thus if the cubic between |p|
and~|q| is associated with the |k|th offset and the cubic between |q| and~|r|
has offset |l| then |mp_info(q)=zero_off+l-k|. (The constant |zero_off| is added
to because |l-k| could be negative.)

After overwriting the type information with offset differences, we no longer
have a true path so we refer to the knot list returned by |offset_prep| as an
``envelope spec.''
@^envelope spec@>
Since an envelope spec only determines relative changes in pen offsets,
|offset_prep| sets a global variable |spec_offset| to the relative change from
|h| to the first offset.

@d zero_off 16384 /* added to offset changes to make them positive */

@<Glob...@>=
integer spec_offset;    /* number of pen edges between |h| and the initial offset */

@ @c
static mp_knot mp_offset_prep (MP mp, mp_knot c, mp_knot h) {
  int n;   /* the number of vertices in the pen polygon */
  mp_knot c0, p, q, q0, r, w, ww;       /* for list manipulation */
  int k_needed;     /* amount to be added to |mp_info(p)| when it is computed */
  mp_knot w0;   /* a pointer to pen offset to use just before |p| */
  mp_number dxin, dyin;    /* the direction into knot |p| */
  int turn_amt;     /* change in pen offsets for the current cubic */
  mp_number max_coef;       /* used while scaling */
  mp_number ss;
  @<Other local variables for |offset_prep|@>;
  new_number(max_coef);
  new_number(dxin);
  new_number(dyin);
  new_number(dx0);
  new_number(dy0);
  new_number(x0);
  new_number(y0);
  new_number(x1);
  new_number(y1);
  new_number(x2);
  new_number(y2);
  new_number(du);
  new_number(dv);
  new_number(dx);
  new_number(dy);
  new_number(x0a);
  new_number(y0a);
  new_number(x1a);
  new_number(y1a);
  new_number(x2a);
  new_number(y2a);
  new_number(t0);
  new_number(t1);
  new_number(t2);
  new_number(u0);
  new_number(u1);
  new_number(v0);
  new_number(v1);
  new_fraction (ss);
  new_fraction (s);
  new_fraction (t);
  @<Initialize the pen size~|n|@>;
  @<Initialize the incoming direction and pen offset at |c|@>;
  p = c;
  c0 = c;
  k_needed = 0;
  do {
    q = mp_next_knot (p);
    @<Split the cubic between |p| and |q|, if necessary, into cubics
      associated with single offsets, after which |q| should
      point to the end of the final such cubic@>;
  NOT_FOUND:
    @<Advance |p| to node |q|, removing any ``dead'' cubics that
      might have been introduced by the splitting process@>;
  } while (q != c);
  @<Fix the offset change in |mp_knot_info(c)| and set |c| to the return value of
    |offset_prep|@>;
  free_number (ss);
  free_number (s);
  free_number (dxin);
  free_number (dyin);
  free_number (dx0);
  free_number (dy0);
  free_number (x0);
  free_number (y0);
  free_number (x1);
  free_number (y1);
  free_number (x2);
  free_number (y2);
  free_number (max_coef);
  free_number (du);
  free_number (dv);
  free_number (dx);
  free_number (dy);
  free_number (x0a);
  free_number (y0a);
  free_number (x1a);
  free_number (y1a);
  free_number (x2a);
  free_number (y2a);
  free_number (t0);
  free_number (t1);
  free_number (t2);
  free_number (u0);
  free_number (u1);
  free_number (v0);
  free_number (v1);
  free_number (t);
  return c;
}


@ We shall want to keep track of where certain knots on the cyclic path
wind up in the envelope spec.  It doesn't suffice just to keep pointers to
knot nodes because some nodes are deleted while removing dead cubics.  Thus
|offset_prep| updates the following pointers

@<Glob...@>=
mp_knot spec_p1;
mp_knot spec_p2;        /* pointers to distinguished knots */

@ @<Set init...@>=
mp->spec_p1 = NULL;
mp->spec_p2 = NULL;

@ @<Initialize the pen size~|n|@>=
n = 0;
p = h;
do {
  incr (n);
  p = mp_next_knot (p);
} while (p != h)

@ Since the true incoming direction isn't known yet, we just pick a direction
consistent with the pen offset~|h|.  If this is wrong, it can be corrected
later.

@<Initialize the incoming direction and pen offset at |c|@>=
{
  mp_knot hn = mp_next_knot (h);
  mp_knot hp = mp_prev_knot (h);
  set_number_from_substraction(dxin, hn->x_coord, hp->x_coord);
  set_number_from_substraction(dyin, hn->y_coord, hp->y_coord);
  if (number_zero(dxin) && number_zero(dyin)) {
    set_number_from_substraction(dxin, hp->y_coord, h->y_coord);
    set_number_from_substraction(dyin, h->x_coord, hp->x_coord);
  }
}
w0 = h

@ We must be careful not to remove the only cubic in a cycle.

But we must also be careful for another reason. If the user-supplied
path starts with a set of degenerate cubics, the target node |q| can
be collapsed to the initial node |p| which might be the same as the
initial node |c| of the curve. This would cause the |offset_prep| routine
to bail out too early, causing distress later on. (See for example
the testcase reported by Bogus\l{}aw Jackowski in tracker id 267, case 52c
on Sarovar.)

@<Advance |p| to node |q|, removing any ``dead'' cubics...@>=
q0 = q;
do {
  r = mp_next_knot (p);
  if (number_equal (p->x_coord, p->right_x) &&
      number_equal (p->y_coord, p->right_y) &&
      number_equal (p->x_coord, r->left_x) && 
      number_equal (p->y_coord, r->left_y) &&
      number_equal (p->x_coord, r->x_coord) && 
      number_equal (p->y_coord, r->y_coord) &&
      r != p && r != q) {
    @<Remove the cubic following |p| and update the data structures
        to merge |r| into |p|@>;
  }
  p = r;
} while (p != q);
/* Check if we removed too much */
if ((q != q0) && (q != c || c == c0))
  q = mp_next_knot (q)
   

@ @<Remove the cubic following |p| and update the data structures...@>=
{
  k_needed = mp_knot_info (p) - zero_off;
  if (r == q) {
    q = p;
  } else {
    mp_knot_info (p) = k_needed + mp_knot_info (r);
    k_needed = 0;
  }
  if (r == c) {
    mp_knot_info (p) = mp_knot_info (c);
    c = p;
  }
  if (r == mp->spec_p1)
    mp->spec_p1 = p;
  if (r == mp->spec_p2)
    mp->spec_p2 = p;
  r = p;
  mp_remove_cubic (mp, p);
}


@ Not setting the |info| field of the newly created knot allows the splitting
routine to work for paths.

@<Declarations@>=
static void mp_split_cubic (MP mp, mp_knot p, mp_number t);

@ @c
void mp_split_cubic (MP mp, mp_knot p, mp_number t) {                               /* splits the cubic after |p| */
  mp_number v;     /* an intermediate value */
  mp_knot q, r; /* for list manipulation */
  q = mp_next_knot (p);
  r = mp_new_knot (mp);
  mp_next_knot (p) = r;
  mp_next_knot (r) = q;
  mp_originator (r) = mp_program_code;
  mp_left_type (r) = mp_explicit;
  mp_right_type (r) = mp_explicit;
  new_number(v);
  set_number_from_of_the_way (v,          t, p->right_x, q->left_x);
  set_number_from_of_the_way (p->right_x, t, p->x_coord, p->right_x);
  set_number_from_of_the_way (q->left_x,  t, q->left_x, q->x_coord);
  set_number_from_of_the_way (r->left_x,  t, p->right_x, v);
  set_number_from_of_the_way (r->right_x, t, v, q->left_x);
  set_number_from_of_the_way (r->x_coord, t, r->left_x, r->right_x);
  set_number_from_of_the_way (v,          t, p->right_y, q->left_y);
  set_number_from_of_the_way (p->right_y, t, p->y_coord, p->right_y);
  set_number_from_of_the_way (q->left_y,  t, q->left_y, q->y_coord);
  set_number_from_of_the_way (r->left_y,  t, p->right_y, v);
  set_number_from_of_the_way (r->right_y, t, v, q->left_y);
  set_number_from_of_the_way (r->y_coord, t, r->left_y, r->right_y);
  free_number (v);
}


@ This does not set |mp_knot_info(p)| or |mp_right_type(p)|.

@<Declarations@>=
static void mp_remove_cubic (MP mp, mp_knot p);

@ @c
void mp_remove_cubic (MP mp, mp_knot p) {                               /* removes the dead cubic following~|p| */
  mp_knot q;    /* the node that disappears */
  (void) mp;
  q = mp_next_knot (p);
  mp_next_knot (p) = mp_next_knot (q);
  number_clone (p->right_x, q->right_x);
  number_clone (p->right_y, q->right_y);
  mp_xfree (q);
}


@ Let $d\prec d'$ mean that the counter-clockwise angle from $d$ to~$d'$ is
strictly between zero and $180^\circ$.  Then we can define $d\preceq d'$ to
mean that the angle could be zero or $180^\circ$. If $w_k=(u_k,v_k)$ is the
$k$th pen offset, the $k$th pen edge direction is defined by the formula
$$d_k=(u\k-u_k,\,v\k-v_k).$$
When listed by increasing $k$, these directions occur in counter-clockwise
order so that $d_k\preceq d\k$ for all~$k$.
The goal of |offset_prep| is to find an offset index~|k| to associate with
each cubic, such that the direction $d(t)$ of the cubic satisfies
$$d_{k-1}\preceq d(t)\preceq d_k\qquad\hbox{for $0\le t\le 1$.}\eqno(*)$$
We may have to split a cubic into many pieces before each
piece corresponds to a unique offset.

@<Split the cubic between |p| and |q|, if necessary, into cubics...@>=
mp_knot_info (p) = zero_off + k_needed;
k_needed = 0;
@<Prepare for derivative computations;
  |goto not_found| if the current cubic is dead@>;
@<Find the initial direction |(dx,dy)|@>;
@<Update |mp_knot_info(p)| and find the offset $w_k$ such that
  $d_{k-1}\preceq(\\{dx},\\{dy})\prec d_k$; also advance |w0| for
  the direction change at |p|@>;
@<Find the final direction |(dxin,dyin)|@>;
@<Decide on the net change in pen offsets and set |turn_amt|@>;
@<Complete the offset splitting process@>;
w0 = mp_pen_walk (mp, w0, turn_amt)
 

@ @<Declarations@>=
static mp_knot mp_pen_walk (MP mp, mp_knot w, integer k);

@ @c
mp_knot mp_pen_walk (MP mp, mp_knot w, integer k) {
  /* walk |k| steps around a pen from |w| */
  (void) mp;
  while (k > 0) {
    w = mp_next_knot (w);
    decr (k);
  }
  while (k < 0) {
    w = mp_prev_knot (w);
    incr (k);
  }
  return w;
}


@ The direction of a cubic $B(z_0,z_1,z_2,z_3;t)=\bigl(x(t),y(t)\bigr)$ can be
calculated from the quadratic polynomials
${1\over3}x'(t)=B(x_1-x_0,x_2-x_1,x_3-x_2;t)$ and
${1\over3}y'(t)=B(y_1-y_0,y_2-y_1,y_3-y_2;t)$.
Since we may be calculating directions from several cubics
split from the current one, it is desirable to do these calculations
without losing too much precision. ``Scaled up'' values of the
derivatives, which will be less tainted by accumulated errors than
derivatives found from the cubics themselves, are maintained in
local variables |x0|, |x1|, and |x2|, representing $X_0=2^l(x_1-x_0)$,
$X_1=2^l(x_2-x_1)$, and $X_2=2^l(x_3-x_2)$; similarly |y0|, |y1|, and~|y2|
represent $Y_0=2^l(y_1-y_0)$, $Y_1=2^l(y_2-y_1)$, and $Y_2=2^l(y_3-y_2)$.

@<Other local variables for |offset_prep|@>=
mp_number x0, x1, x2, y0, y1, y2; /* representatives of derivatives */
mp_number t0, t1, t2;     /* coefficients of polynomial for slope testing */
mp_number du, dv, dx, dy; /* for directions of the pen and the curve */
mp_number dx0, dy0;       /* initial direction for the first cubic in the curve */
mp_number x0a, x1a, x2a, y0a, y1a, y2a;   /* intermediate values */
mp_number t;     /* where the derivative passes through zero */
mp_number s;     /* a temporary value */

@ @<Prepare for derivative computations...@>=
set_number_from_substraction(x0, p->right_x, p->x_coord);
set_number_from_substraction(x2, q->x_coord, q->left_x);
set_number_from_substraction(x1, q->left_x, p->right_x);
set_number_from_substraction(y0, p->right_y, p->y_coord);
set_number_from_substraction(y2, q->y_coord, q->left_y);
set_number_from_substraction(y1, q->left_y, p->right_y);
{
  mp_number absval; 
  new_number (absval);
  number_clone(absval, x1);
  number_abs(absval);
  number_clone(max_coef, x0);
  number_abs (max_coef);
  if (number_greater(absval, max_coef)) {
    number_clone(max_coef, absval);
  }
  number_clone(absval, x2);
  number_abs(absval);
  if (number_greater(absval, max_coef)) {
    number_clone(max_coef, absval);
  }
  number_clone(absval, y0);
  number_abs(absval);
  if (number_greater(absval, max_coef)) {
    number_clone(max_coef, absval);
  }
  number_clone(absval, y1);
  number_abs(absval);
  if (number_greater(absval, max_coef)) {
    number_clone(max_coef, absval);
  }
  number_clone(absval, y2);
  number_abs(absval);
  if (number_greater(absval, max_coef)) {
    number_clone(max_coef, absval);
  }
  if (number_zero(max_coef)) {
    goto NOT_FOUND;
  }
  free_number (absval);
}
while (number_less(max_coef, fraction_half_t)) {
  number_double (max_coef);
  number_double (x0);
  number_double (x1);
  number_double (x2);
  number_double (y0);
  number_double (y1);
  number_double (y2);
}


@ Let us first solve a special case of the problem: Suppose we
know an index~$k$ such that either (i)~$d(t)\succeq d_{k-1}$ for all~$t$
and $d(0)\prec d_k$, or (ii)~$d(t)\preceq d_k$ for all~$t$ and
$d(0)\succ d_{k-1}$.
Then, in a sense, we're halfway done, since one of the two relations
in $(*)$ is satisfied, and the other couldn't be satisfied for
any other value of~|k|.

Actually, the conditions can be relaxed somewhat since a relation such as
$d(t)\succeq d_{k-1}$ restricts $d(t)$ to a half plane when all that really
matters is whether $d(t)$ crosses the ray in the $d_{k-1}$ direction from
the origin.  The condition for case~(i) becomes $d_{k-1}\preceq d(0)\prec d_k$
and $d(t)$ never crosses the $d_{k-1}$ ray in the clockwise direction.
Case~(ii) is similar except $d(t)$ cannot cross the $d_k$ ray in the
counterclockwise direction.

The |fin_offset_prep| subroutine solves the stated subproblem.
It has a parameter called |rise| that is |1| in
case~(i), |-1| in case~(ii). Parameters |x0| through |y2| represent
the derivative of the cubic following |p|.
The |w| parameter should point to offset~$w_k$ and |mp_info(p)| should already
be set properly.  The |turn_amt| parameter gives the absolute value of the
overall net change in pen offsets.

@<Declarations@>=
static void mp_fin_offset_prep (MP mp, mp_knot p, mp_knot w, mp_number
                                x0, mp_number x1, mp_number x2, mp_number y0,
                                mp_number y1, mp_number y2, integer rise,
                                integer turn_amt);

@ @c
void mp_fin_offset_prep (MP mp, mp_knot p, mp_knot w, mp_number
                         x0, mp_number x1, mp_number x2, mp_number y0, mp_number y1,
                         mp_number y2, integer rise, integer turn_amt) {
  mp_knot ww;   /* for list manipulation */
  mp_number du, dv;        /* for slope calculation */
  mp_number t0, t1, t2;   /* test coefficients */
  mp_number t;   /* place where the derivative passes a critical slope */
  mp_number s;   /* slope or reciprocal slope */
  mp_number v;    /* intermediate value for updating |x0..y2| */
  mp_knot q;    /* original |mp_next_knot(p)| */
  q = mp_next_knot (p);
  new_number(du);
  new_number(dv);
  new_number(v);
  new_number(t0);
  new_number(t1);
  new_number(t2);
  new_fraction(s);
  new_fraction(t);
  while (1) {
    if (rise > 0)
      ww = mp_next_knot (w);    /* a pointer to $w\k$ */
    else
      ww = mp_prev_knot (w);    /* a pointer to $w_{k-1}$ */
    @<Compute test coefficients |(t0,t1,t2)|
      for $d(t)$ versus $d_k$ or $d_{k-1}$@>;
    crossing_point (t, t0, t1, t2);
    if (number_greaterequal(t, fraction_one_t)) {
      if (turn_amt > 0)
        number_clone(t, fraction_one_t);
      else
        goto RETURN;
    }
    @<Split the cubic at $t$,
      and split off another cubic if the derivative crosses back@>;
    w = ww;
  }
RETURN:
  free_number (s);
  free_number (t);
  free_number (du);
  free_number (dv);
  free_number (v);
  free_number (t0);
  free_number (t1);
  free_number (t2);
}


@ We want $B(\\{t0},\\{t1},\\{t2};t)$ to be the dot product of $d(t)$ with a
$-90^\circ$ rotation of the vector from |w| to |ww|.  This makes the resulting
function cross from positive to negative when $d_{k-1}\preceq d(t)\preceq d_k$
begins to fail.

@<Compute test coefficients |(t0,t1,t2)| for $d(t)$ versus...@>=
{
  mp_number abs_du, abs_dv;
  new_number (abs_du);
  new_number (abs_dv);
  set_number_from_substraction(du, ww->x_coord, w->x_coord);
  set_number_from_substraction(dv, ww->y_coord, w->y_coord);
  number_clone(abs_du, du);
  number_abs(abs_du);
  number_clone(abs_dv, dv);
  number_abs(abs_dv);
  if (number_greaterequal(abs_du, abs_dv)) {
    mp_number r1;
    new_fraction (r1);
    make_fraction (s, dv, du);
    take_fraction (r1, x0, s);
    set_number_from_substraction(t0, r1, y0);
    take_fraction (r1, x1, s);
    set_number_from_substraction(t1, r1, y1);
    take_fraction (r1, x2, s);
    set_number_from_substraction(t2, r1, y2);
    if (number_negative(du)) {
      number_negate (t0);
      number_negate (t1);
      number_negate (t2);
    }
    free_number (r1);
  } else {
    mp_number r1;
    new_fraction (r1);
    make_fraction (s, du, dv);
    take_fraction (r1, y0, s);
    set_number_from_substraction(t0, x0, r1);
    take_fraction (r1, y1, s);
    set_number_from_substraction(t1, x1, r1);
    take_fraction (r1, y2, s);
    set_number_from_substraction(t2, x2, r1);
    if (number_negative(dv)) {
      number_negate (t0);
      number_negate (t1);
      number_negate (t2);
    }
    free_number (r1);
  }
  free_number (abs_du);
  free_number (abs_dv);
  if (number_negative(t0))
    set_number_to_zero(t0); /* should be positive without rounding error */
}
    

@ The curve has crossed $d_k$ or $d_{k-1}$; its initial segment satisfies
$(*)$, and it might cross again and return towards $s_{k-1}$ or $s_k$,
respectively, yielding another solution of $(*)$.

@<Split the cubic at $t$, and split off another...@>=
{
  mp_split_cubic (mp, p, t);
  p = mp_next_knot (p);
  mp_knot_info (p) = zero_off + rise;
  decr (turn_amt);
  set_number_from_of_the_way(v,  t, x0, x1);
  set_number_from_of_the_way(x1, t, x1, x2);
  set_number_from_of_the_way(x0, t, v, x1);
  set_number_from_of_the_way(v,  t, y0, y1);
  set_number_from_of_the_way(y1, t, y1, y2);
  set_number_from_of_the_way(y0, t, v, y1);
  if (turn_amt < 0) {
    mp_number arg1, arg2, arg3;
    new_number (arg1);
    new_number (arg2);
    new_number (arg3);
    set_number_from_of_the_way(t1, t, t1, t2);
    if (number_positive(t1))
      set_number_to_zero(t1);  /* without rounding error, |t1| would be |<=0| */
    number_clone(arg2, t1);
    number_negate(arg2);
    number_clone(arg3, t2);
    number_negate(arg3);
    crossing_point (t, arg1, arg2, arg3);
    free_number (arg1);
    free_number (arg2);
    free_number (arg3);
    if (number_greater(t, fraction_one_t))
      number_clone(t, fraction_one_t);
    incr (turn_amt);
    if (number_equal(t,fraction_one_t) && (mp_next_knot (p) != q)) {
      mp_knot_info (mp_next_knot (p)) = mp_knot_info (mp_next_knot (p)) - rise;
    } else {
      mp_split_cubic (mp, p, t);
      mp_knot_info (mp_next_knot (p)) = zero_off - rise;
      set_number_from_of_the_way(v,  t, x1, x2);
      set_number_from_of_the_way(x1, t, x0, x1);
      set_number_from_of_the_way(x2, t, x1, v);
      set_number_from_of_the_way(v,  t, y1, y2);
      set_number_from_of_the_way(y1, t, y0, y1);
      set_number_from_of_the_way(y2, t, y1, v);
    }
  }
}


@ Now we must consider the general problem of |offset_prep|, when
nothing is known about a given cubic. We start by finding its
direction in the vicinity of |t=0|.

If $z'(t)=0$, the given cubic is numerically unstable but |offset_prep|
has not yet introduced any more numerical errors.  Thus we can compute
the true initial direction for the given cubic, even if it is almost
degenerate.

@<Find the initial direction |(dx,dy)|@>=
number_clone(dx, x0);
number_clone(dy, y0);
if (number_zero(dx) && number_zero(dy)) {
  number_clone(dx, x1);
  number_clone(dy, y1);
  if (number_zero(dx) && number_zero(dy)) {
    number_clone(dx, x2);
    number_clone(dy, y2);
  }
}
if (p == c) {
  number_clone(dx0, dx);
  number_clone(dy0, dy);
}

@ @<Find the final direction |(dxin,dyin)|@>=
number_clone(dxin, x2);
number_clone(dyin, y2);
if (number_zero(dxin) && number_zero(dyin)) {
  number_clone(dxin, x1);
  number_clone(dyin, y1);
  if (number_zero(dxin) && number_zero(dyin)) {
    number_clone(dxin, x0);
    number_clone(dyin, y0);
  }
}

@ The next step is to bracket the initial direction between consecutive
edges of the pen polygon.  We must be careful to turn clockwise only if
this makes the turn less than $180^\circ$. (A $180^\circ$ turn must be
counter-clockwise in order to make \&{doublepath} envelopes come out
@:double_path_}{\&{doublepath} primitive@>
right.) This code depends on |w0| being the offset for |(dxin,dyin)|.

@<Update |mp_knot_info(p)| and find the offset $w_k$ such that...@>=
{
  mp_number ab_vs_cd;
  new_number (ab_vs_cd);
  ab_vs_cd (ab_vs_cd, dy, dxin, dx, dyin);
  turn_amt = mp_get_turn_amt (mp, w0, dx, dy, number_nonnegative(ab_vs_cd));
  free_number (ab_vs_cd);
  w = mp_pen_walk (mp, w0, turn_amt);
  w0 = w;
  mp_knot_info (p) = mp_knot_info (p) + turn_amt;
}

@ Decide how many pen offsets to go away from |w| in order to find the offset
for |(dx,dy)|, going counterclockwise if |ccw| is |true|.  This assumes that
|w| is the offset for some direction $(x',y')$ from which the angle to |(dx,dy)|
in the sense determined by |ccw| is less than or equal to $180^\circ$.

If the pen polygon has only two edges, they could both be parallel
to |(dx,dy)|.  In this case, we must be careful to stop after crossing the first
such edge in order to avoid an infinite loop.

@<Declarations@>=
static integer mp_get_turn_amt (MP mp, mp_knot w, mp_number dx,
                                mp_number dy, boolean ccw);

@ @c
integer mp_get_turn_amt (MP mp, mp_knot w, mp_number dx, mp_number dy, boolean ccw) {
  mp_knot ww;   /* a neighbor of knot~|w| */
  integer s;    /* turn amount so far */
  mp_number t;    /* |ab_vs_cd| result */
  mp_number arg1, arg2;
  s = 0;
  new_number (arg1);
  new_number (arg2);
  new_number (t);
  if (ccw) {
    ww = mp_next_knot (w);
    do {
      set_number_from_substraction (arg1, ww->x_coord, w->x_coord);
      set_number_from_substraction (arg2, ww->y_coord, w->y_coord);
      ab_vs_cd (t, dy, arg1, dx, arg2);
      if (number_negative(t))
        break;
      incr (s);
      w = ww;
      ww = mp_next_knot (ww);
    } while (number_positive(t));
  } else {
    ww = mp_prev_knot (w);
    set_number_from_substraction (arg1, w->x_coord, ww->x_coord);
    set_number_from_substraction (arg2, w->y_coord, ww->y_coord);
    ab_vs_cd (t, dy, arg1, dx, arg2);
    while (number_negative(t)) {
      decr (s);
      w = ww;
      ww = mp_prev_knot (ww);
      set_number_from_substraction (arg1, w->x_coord, ww->x_coord);
      set_number_from_substraction (arg2, w->y_coord, ww->y_coord);
      ab_vs_cd (t, dy, arg1, dx, arg2);
    }
  }
  free_number (t);
  free_number (arg1);
  free_number (arg2);
  return s;
}


@ When we're all done, the final offset is |w0| and the final curve direction
is |(dxin,dyin)|.  With this knowledge of the incoming direction at |c|, we
can correct |mp_info(c)| which was erroneously based on an incoming offset
of~|h|.

@d fix_by(A) mp_knot_info(c)=mp_knot_info(c)+(A)

@<Fix the offset change in |mp_knot_info(c)| and set |c| to the return value of...@>=
mp->spec_offset = mp_knot_info (c) - zero_off;
if (mp_next_knot (c) == c) {
mp_knot_info (c) = zero_off + n;
} else {
  mp_number ab_vs_cd;
  new_number (ab_vs_cd);
  fix_by (k_needed);
  while (w0 != h) {
    fix_by (1);
    w0 = mp_next_knot (w0);
  }
  while (mp_knot_info (c) <= zero_off - n)
    fix_by (n);
  while (mp_knot_info (c) > zero_off)
    fix_by (-n);
  ab_vs_cd (ab_vs_cd, dy0, dxin, dx0, dyin);
  if ((mp_knot_info (c) != zero_off)   && number_nonnegative(ab_vs_cd))
    fix_by (n);
  free_number (ab_vs_cd);
}


@ Finally we want to reduce the general problem to situations that
|fin_offset_prep| can handle. We split the cubic into at most three parts
with respect to $d_{k-1}$, and apply |fin_offset_prep| to each part.

@<Complete the offset splitting process@>=
ww = mp_prev_knot (w);
@<Compute test coeff...@>;
@<Find the first |t| where $d(t)$ crosses $d_{k-1}$ or set
  |t:=fraction_one+1|@>;
if (number_greater(t, fraction_one_t)) {
  mp_fin_offset_prep (mp, p, w, x0, x1, x2, y0, y1, y2, 1, turn_amt);
} else {
  mp_split_cubic (mp, p, t);
  r = mp_next_knot (p);
  set_number_from_of_the_way(x1a, t, x0, x1);
  set_number_from_of_the_way(x1,  t, x1, x2);
  set_number_from_of_the_way(x2a, t, x1a, x1);
  set_number_from_of_the_way(y1a, t, y0, y1);
  set_number_from_of_the_way(y1,  t, y1, y2);
  set_number_from_of_the_way(y2a, t, y1a, y1);
  mp_fin_offset_prep (mp, p, w, x0, x1a, x2a, y0, y1a, y2a, 1, 0);
  number_clone(x0, x2a);
  number_clone(y0, y2a);
  mp_knot_info (r) = zero_off - 1;
  if (turn_amt >= 0) {
    mp_number arg1, arg2, arg3;
    new_number(arg1);
    new_number(arg2);
    new_number(arg3);
    set_number_from_of_the_way(t1, t, t1, t2);
    if (number_positive(t1))
      set_number_to_zero(t1);
    number_clone(arg2, t1);
    number_negate(arg2);
    number_clone(arg3, t2);
    number_negate(arg3);
    crossing_point (t, arg1, arg2, arg3);
    free_number (arg1);
    free_number (arg2);
    free_number (arg3);
    if (number_greater(t, fraction_one_t))
      number_clone (t, fraction_one_t);
    @<Split off another rising cubic for |fin_offset_prep|@>;
    mp_fin_offset_prep (mp, r, ww, x0, x1, x2, y0, y1, y2, -1, 0);
  } else {
    mp_fin_offset_prep (mp, r, ww, x0, x1, x2, y0, y1, y2, -1, (-1 - turn_amt));
  }
}


@ @<Split off another rising cubic for |fin_offset_prep|@>=
mp_split_cubic (mp, r, t);
mp_knot_info (mp_next_knot (r)) = zero_off + 1;
set_number_from_of_the_way(x1a, t, x1, x2);
set_number_from_of_the_way(x1,  t, x0, x1);
set_number_from_of_the_way(x0a, t, x1, x1a);
set_number_from_of_the_way(y1a, t, y1, y2);
set_number_from_of_the_way(y1,  t, y0, y1);
set_number_from_of_the_way(y0a, t, y1, y1a);
mp_fin_offset_prep (mp, mp_next_knot (r), w, x0a, x1a, x2, y0a, y1a, y2, 1,  turn_amt);
number_clone(x2, x0a);
number_clone(y2, y0a)

@ At this point, the direction of the incoming pen edge is |(-du,-dv)|.
When the component of $d(t)$ perpendicular to |(-du,-dv)| crosses zero, we
need to decide whether the directions are parallel or antiparallel.  We
can test this by finding the dot product of $d(t)$ and |(-du,-dv)|, but this
should be avoided when the value of |turn_amt| already determines the
answer.  If |t2<0|, there is one crossing and it is antiparallel only if
|turn_amt>=0|.  If |turn_amt<0|, there should always be at least one
crossing and the first crossing cannot be antiparallel.

@<Find the first |t| where $d(t)$ crosses $d_{k-1}$ or set...@>=
crossing_point (t, t0, t1, t2);
if (turn_amt >= 0) {
  if (number_negative(t2)) {
    number_clone(t, fraction_one_t);
    number_add_scaled (t, 1);
  } else {
    mp_number tmp, arg1, r1;
    new_fraction (r1);
    new_number(tmp);
    new_number(arg1);
    set_number_from_of_the_way(u0, t, x0, x1);
    set_number_from_of_the_way(u1, t, x1, x2);
    set_number_from_of_the_way(tmp, t, u0, u1);
    number_clone (arg1, du);
    number_abs (arg1);
    take_fraction (ss, arg1, tmp);
    set_number_from_of_the_way(v0, t, y0, y1);
    set_number_from_of_the_way(v1, t, y1, y2);
    set_number_from_of_the_way(tmp, t, v0, v1);
    number_clone (arg1, dv);
    number_abs (arg1);
    take_fraction (r1, arg1, tmp);
    number_add (ss, r1);
    free_number (tmp);
    if (number_negative(ss)) {
      number_clone(t, fraction_one_t);
      number_add_scaled (t, 1);
    }
    free_number(arg1);
    free_number(r1);
  }
} else if (number_greater(t, fraction_one_t)) {
  number_clone (t, fraction_one_t);
}

@ @<Other local variables for |offset_prep|@>=
mp_number u0, u1, v0, v1; /* intermediate values for $d(t)$ calculation */
int d_sign;     /* sign of overall change in direction for this cubic */

@ If the cubic almost has a cusp, it is a numerically ill-conditioned
problem to decide which way it loops around but that's OK as long we're
consistent.  To make \&{doublepath} envelopes work properly, reversing
the path should always change the sign of |turn_amt|.

@<Decide on the net change in pen offsets and set |turn_amt|@>=
{
  mp_number ab_vs_cd;
  new_number (ab_vs_cd);
  ab_vs_cd (ab_vs_cd, dx, dyin, dxin, dy);
  if (number_negative (ab_vs_cd))
    d_sign = -1;
  else if (number_zero (ab_vs_cd))
    d_sign = 0;
  else
    d_sign = 1;
  free_number (ab_vs_cd);
}
if (d_sign == 0) {
  @<Check rotation direction based on node position@>
}
if (d_sign == 0) {
  if (number_zero(dx)) {
    if (number_positive(dy))
      d_sign = 1;
    else
      d_sign = -1;
  } else {
    if (number_positive(dx))
      d_sign = 1;
    else
      d_sign = -1;
  }
}
@<Make |ss| negative if and only if the total change in direction is
  more than $180^\circ$@>;
turn_amt = mp_get_turn_amt (mp, w, dxin, dyin, (d_sign > 0));
if (number_negative(ss))
  turn_amt = turn_amt - d_sign * n

@ We check rotation direction by looking at the vector connecting the current
node with the next. If its angle with incoming and outgoing tangents has the
same sign, we pick this as |d_sign|, since it means we have a flex, not a cusp.
Otherwise we proceed to the cusp code.

@<Check rotation direction based on node position@>=
{
  mp_number ab_vs_cd1, ab_vs_cd2, t; 
  new_number (ab_vs_cd1);
  new_number (ab_vs_cd2);
  new_number (t);
  set_number_from_substraction(u0, q->x_coord, p->x_coord);
  set_number_from_substraction(u1, q->y_coord, p->y_coord);
  ab_vs_cd (ab_vs_cd1, dx, u1, u0, dy);
  ab_vs_cd (ab_vs_cd2, u0, dyin, dxin, u1);
  set_number_from_addition (t, ab_vs_cd1, ab_vs_cd2);
  number_half (t);
  if (number_negative (t))
    d_sign = -1;
  else if (number_zero (t))
    d_sign = 0;
  else
    d_sign = 1;
  free_number (t);
  free_number (ab_vs_cd1);
  free_number (ab_vs_cd2);
}

@ In order to be invariant under path reversal, the result of this computation
should not change when |x0|, |y0|, $\ldots$ are all negated and |(x0,y0)| is
then swapped with |(x2,y2)|.  We make use of the identities
|take_fraction(-a,-b)=take_fraction(a,b)| and
|t_of_the_way(-a,-b)=-(t_of_the_way(a,b))|.

@<Make |ss| negative if and only if the total change in direction is...@>=
{
  mp_number r1, r2, arg1;
  new_number (arg1);
  new_fraction (r1);
  new_fraction (r2);
  take_fraction (r1, x0, y2);
  take_fraction (r2, x2, y0);
  number_half (r1);
  number_half (r2);
  set_number_from_substraction(t0, r1, r2);
  set_number_from_addition (arg1, y0, y2);
  take_fraction (r1, x1, arg1);
  set_number_from_addition (arg1, x0, x2);
  take_fraction (r1, y1, arg1);
  number_half (r1);
  number_half (r2);
  set_number_from_substraction(t1, r1, r2);
  free_number (arg1);
  free_number (r1);
  free_number (r2);
}
if (number_zero(t0))
  set_number_from_scaled(t0, d_sign);                  /* path reversal always negates |d_sign| */
if (number_positive(t0)) {
  mp_number arg3;
  new_number(arg3);
  number_clone(arg3, t0);
  number_negate(arg3);
  crossing_point (t, t0, t1, arg3);
  free_number (arg3);
  set_number_from_of_the_way(u0, t, x0, x1);
  set_number_from_of_the_way(u1, t, x1, x2);
  set_number_from_of_the_way(v0, t, y0, y1);
  set_number_from_of_the_way(v1, t, y1, y2);
} else {
  mp_number arg1;
  new_number(arg1);
  number_clone(arg1, t0);
  number_negate(arg1);
  crossing_point (t, arg1, t1, t0);
  free_number (arg1);
  set_number_from_of_the_way(u0, t, x2, x1);
  set_number_from_of_the_way(u1, t, x1, x0);
  set_number_from_of_the_way(v0, t, y2, y1);
  set_number_from_of_the_way(v1, t, y1, y0);
}
{ 
  mp_number tmp1, tmp2, r1, r2, arg1;
  new_fraction (r1);
  new_fraction (r2);
  new_number(arg1);
  new_number(tmp1);
  new_number(tmp2);
  set_number_from_of_the_way(tmp1, t, u0, u1);
  set_number_from_of_the_way(tmp2, t, v0, v1);
  set_number_from_addition(arg1, x0, x2);
  take_fraction (r1, arg1, tmp1);
  set_number_from_addition(arg1, y0, y2);
  take_fraction (r2, arg1, tmp2);
  set_number_from_addition (ss, r1, r2);
  free_number (arg1);
  free_number (r1);
  free_number (r2);
  free_number (tmp1);
  free_number (tmp2);
}
 

@ Here's a routine that prints an envelope spec in symbolic form.  It assumes
that the |cur_pen| has not been walked around to the first offset.

@c
static void mp_print_spec (MP mp, mp_knot cur_spec, mp_knot cur_pen,
                           const char *s) {
  mp_knot p, q; /* list traversal */
  mp_knot w;    /* the current pen offset */
  mp_print_diagnostic (mp, "Envelope spec", s, true);
  p = cur_spec;
  w = mp_pen_walk (mp, cur_pen, mp->spec_offset);
  mp_print_ln (mp);
  mp_print_two (mp, cur_spec->x_coord, cur_spec->y_coord);
  mp_print (mp, " % beginning with offset ");
  mp_print_two (mp, w->x_coord, w->y_coord);
  do {
    while (1) {
      q = mp_next_knot (p);
      @<Print the cubic between |p| and |q|@>;
      p = q;
      if ((p == cur_spec) || (mp_knot_info (p) != zero_off))
        break;
    }
    if (mp_knot_info (p) != zero_off) {
      @<Update |w| as indicated by |mp_knot_info(p)| and print an explanation@>;
    }
  } while (p != cur_spec);
  mp_print_nl (mp, " & cycle");
  mp_end_diagnostic (mp, true);
}


@ @<Update |w| as indicated by |mp_knot_info(p)| and print an explanation@>=
{
  w = mp_pen_walk (mp, w, (mp_knot_info (p) - zero_off));
  mp_print (mp, " % ");
  if (mp_knot_info (p) > zero_off)
    mp_print (mp, "counter");
  mp_print (mp, "clockwise to offset ");
  mp_print_two (mp, w->x_coord, w->y_coord);
}


@ @<Print the cubic between |p| and |q|@>=
{
  mp_print_nl (mp, "   ..controls ");
  mp_print_two (mp, p->right_x,  p->right_y);
  mp_print (mp, " and ");
  mp_print_two (mp, q->left_x, q->left_y);
  mp_print_nl (mp, " ..");
  mp_print_two (mp, q->x_coord, q->y_coord);
}


@ Once we have an envelope spec, the remaining task to construct the actual
envelope by offsetting each cubic as determined by the |info| fields in
the knots.  First we use |offset_prep| to convert the |c| into an envelope
spec. Then we add the offsets so that |c| becomes a cyclic path that represents
the envelope.

The |ljoin| and |miterlim| parameters control the treatment of points where the
pen offset changes, and |lcap| controls the endpoints of a \&{doublepath}.
The endpoints are easily located because |c| is given in undoubled form
and then doubled in this procedure.  We use |spec_p1| and |spec_p2| to keep
track of the endpoints and treat them like very sharp corners.
Butt end caps are treated like beveled joins; round end caps are treated like
round joins; and square end caps are achieved by setting |join_type:=3|.

None of these parameters apply to inside joins where the convolution tracing
has retrograde lines.  In such cases we use a simple connect-the-endpoints
approach that is achieved by setting |join_type:=2|.

@c
static mp_knot mp_make_envelope (MP mp, mp_knot c, mp_knot h, quarterword ljoin,
                                 quarterword lcap, mp_number miterlim) {
  mp_knot p, q, r, q0;  /* for manipulating the path */
  mp_knot w, w0;        /* the pen knot for the current offset */
  halfword k, k0;       /* controls pen edge insertion */
  mp_number qx, qy;        /* unshifted coordinates of |q| */
  mp_fraction dxin, dyin, dxout, dyout;      /* directions at |q| when square or mitered */
  int join_type = 0;    /* codes |0..3| for mitered, round, beveled, or square */
  @<Other local variables for |make_envelope|@>;
  new_number (max_ht);
  new_number (tmp);
  new_fraction(dxin);
  new_fraction(dyin);
  new_fraction(dxout);
  new_fraction(dyout);
  mp->spec_p1 = NULL;
  mp->spec_p2 = NULL;
  new_number(qx);
  new_number(qy);
  @<If endpoint, double the path |c|, and set |spec_p1| and |spec_p2|@>;
  @<Use |offset_prep| to compute the envelope spec then walk |h| around to
    the initial offset@>;
  w = h;
  p = c;
  do {
    q = mp_next_knot (p);
    q0 = q;
    number_clone (qx, q->x_coord);
    number_clone (qy, q->y_coord);
    k = mp_knot_info (q);
    k0 = k;
    w0 = w;
    if (k != zero_off) {
      @<Set |join_type| to indicate how to handle offset changes at~|q|@>;
    }
    @<Add offset |w| to the cubic from |p| to |q|@>;
    while (k != zero_off) {
      @<Step |w| and move |k| one step closer to |zero_off|@>;
      if ((join_type == 1) || (k == zero_off)) {
        mp_number xtot, ytot; 
        new_number(xtot);
        new_number(ytot);
        set_number_from_addition (xtot, qx, w->x_coord);
        set_number_from_addition (ytot, qy, w->y_coord);
        q = mp_insert_knot (mp, q, xtot, ytot);
      }
    }
    if (q != mp_next_knot (p)) {
      @<Set |p=mp_link(p)| and add knots between |p| and |q| as
        required by |join_type|@>;
    }
    p = q;
  } while (q0 != c);
  free_number (max_ht);
  free_number (tmp);
  free_number (qx);
  free_number (qy);
  free_number (dxin);
  free_number (dyin);
  free_number (dxout);
  free_number (dyout);
  return c;
}


@ @<Use |offset_prep| to compute the envelope spec then walk |h| around to...@>=
c = mp_offset_prep (mp, c, h);
if (number_positive(internal_value (mp_tracing_specs)))
  mp_print_spec (mp, c, h, "");
h = mp_pen_walk (mp, h, mp->spec_offset)
 

@ Mitered and squared-off joins depend on path directions that are difficult to
compute for degenerate cubics.  The envelope spec computed by |offset_prep| can
have degenerate cubics only if the entire cycle collapses to a single
degenerate cubic.  Setting |join_type:=2| in this case makes the computed
envelope degenerate as well.

@<Set |join_type| to indicate how to handle offset changes at~|q|@>=
if (k < zero_off) {
  join_type = 2;
} else {
  if ((q != mp->spec_p1) && (q != mp->spec_p2))
    join_type = ljoin;
  else if (lcap == 2)
    join_type = 3;
  else
    join_type = 2 - lcap;
  if ((join_type == 0) || (join_type == 3)) {
    @<Set the incoming and outgoing directions at |q|; in case of
      degeneracy set |join_type:=2|@>;
    if (join_type == 0) {
      @<If |miterlim| is less than the secant of half the angle at |q|
        then set |join_type:=2|@>;
    }
  }
}


@ @<If |miterlim| is less than the secant of half the angle at |q|...@>=
{
  mp_number r1, r2;
  new_fraction (r1);
  new_fraction (r2);
  take_fraction (r1, dxin, dxout);
  take_fraction (r2, dyin, dyout);
  number_add (r1, r2);
  number_half (r1);
  number_add (r1, fraction_half_t);
  take_fraction (tmp, miterlim, r1);
  if (number_less(tmp, unity_t)) {
    mp_number ret;
    new_number (ret);
    take_scaled (ret, miterlim, tmp);
    if (number_less(ret, unity_t))
      join_type = 2;
    free_number (ret);
  }
  free_number (r1);
  free_number (r2);
}


@ @<Other local variables for |make_envelope|@>=
mp_number tmp;     /* a temporary value */

@ The coordinates of |p| have already been shifted unless |p| is the first
knot in which case they get shifted at the very end.

@<Add offset |w| to the cubic from |p| to |q|@>=
number_add (p->right_x, w->x_coord);
number_add (p->right_y, w->y_coord);
number_add (q->left_x,  w->x_coord);
number_add (q->left_y,  w->y_coord);
number_add (q->x_coord, w->x_coord);
number_add (q->y_coord, w->y_coord);
mp_left_type (q) = mp_explicit;
mp_right_type (q) = mp_explicit

@ @<Step |w| and move |k| one step closer to |zero_off|@>=
if (k > zero_off) {
  w = mp_next_knot (w);
  decr (k);
} else {
  w = mp_prev_knot (w);
  incr (k);
}


@ The cubic from |q| to the new knot at |(x,y)| becomes a line segment and
the |mp_right_x| and |mp_right_y| fields of |r| are set from |q|.  This is done in
case the cubic containing these control points is ``yet to be examined.''

@<Declarations@>=
static mp_knot mp_insert_knot (MP mp, mp_knot q, mp_number x, mp_number y);

@ @c
mp_knot mp_insert_knot (MP mp, mp_knot q, mp_number x, mp_number y) {
  /* returns the inserted knot */
  mp_knot r;    /* the new knot */
  r = mp_new_knot (mp);
  mp_next_knot (r) = mp_next_knot (q);
  mp_next_knot (q) = r;
  number_clone (r->right_x, q->right_x);
  number_clone (r->right_y, q->right_y);
  number_clone (r->x_coord, x);
  number_clone (r->y_coord, y);
  number_clone (q->right_x, q->x_coord);
  number_clone (q->right_y, q->y_coord);
  number_clone (r->left_x, r->x_coord);
  number_clone (r->left_y, r->y_coord);
  mp_left_type (r) = mp_explicit;
  mp_right_type (r) = mp_explicit;
  mp_originator (r) = mp_program_code;
  return r;
}


@ After setting |p:=mp_link(p)|, either |join_type=1| or |q=mp_link(p)|.

@<Set |p=mp_link(p)| and add knots between |p| and |q| as...@>=
{
  p = mp_next_knot (p);
  if ((join_type == 0) || (join_type == 3)) {
    if (join_type == 0) {
      @<Insert a new knot |r| between |p| and |q| as required for a mitered join@>
    } else {
      @<Make |r| the last of two knots inserted between |p| and |q| to form a
        squared join@>;
    }
    if (r != NULL) {
      number_clone (r->right_x, r->x_coord);
      number_clone (r->right_y, r->y_coord);
    }
  }
}


@ For very small angles, adding a knot is unnecessary and would cause numerical
problems, so we just set |r:=NULL| in that case.

@d near_zero_angle_k ((math_data *)mp->math)->near_zero_angle_t

@<Insert a new knot |r| between |p| and |q| as required for a mitered join@>=
{
  mp_number det;   /* a determinant used for mitered join calculations */
  mp_number absdet;
  mp_number r1, r2;
  new_fraction (r1);
  new_fraction (r2);
  new_fraction (det);
  new_fraction (absdet);
  take_fraction (r1, dyout, dxin);
  take_fraction (r2, dxout, dyin);
  set_number_from_substraction(det, r1, r2);
  number_clone (absdet, det);
  number_abs (absdet);
  if (number_less (absdet, near_zero_angle_k)) {
    r = NULL;                   /* sine $<10^{-4}$ */
  } else {
    mp_number xtot, ytot, xsub, ysub;
    new_fraction(xsub);
    new_fraction(ysub);
    new_number(xtot);
    new_number(ytot);
    set_number_from_substraction (tmp, q->x_coord, p->x_coord);
    take_fraction (r1, tmp, dyout);
    set_number_from_substraction (tmp, q->y_coord, p->y_coord);
    take_fraction (r2, tmp, dxout);
    set_number_from_substraction (tmp, r1, r2);
    make_fraction (r1, tmp, det);
    number_clone (tmp, r1);
    take_fraction (xsub, tmp, dxin);
    take_fraction (ysub, tmp, dyin);
    set_number_from_addition(xtot, p->x_coord, xsub);
    set_number_from_addition(ytot, p->y_coord, ysub);
    r = mp_insert_knot (mp, p, xtot, ytot);
    free_number (xtot);
    free_number (ytot);
    free_number (xsub);
    free_number (ysub);
  }
  free_number (r1);
  free_number (r2);
  free_number (det);
  free_number (absdet);
}


@ @<Make |r| the last of two knots inserted between |p| and |q| to form a...@>=
{
  mp_number ht_x, ht_y;    /* perpendicular to the segment from |p| to |q| */
  mp_number ht_x_abs, ht_y_abs;    /* absolutes */
  mp_number xtot, ytot, xsub, ysub;
  new_fraction(xsub);
  new_fraction(ysub);
  new_number(xtot);
  new_number(ytot);
  new_fraction (ht_x);
  new_fraction (ht_y);
  new_fraction (ht_x_abs);
  new_fraction (ht_y_abs);
  set_number_from_substraction(ht_x, w->y_coord, w0->y_coord);
  set_number_from_substraction(ht_y, w0->x_coord, w->x_coord);
  number_clone (ht_x_abs, ht_x);
  number_clone (ht_y_abs, ht_y);
  number_abs (ht_x_abs);
  number_abs (ht_y_abs);
  while (number_less(ht_x_abs, fraction_half_t) && number_less(ht_y_abs, fraction_half_t)) {
    number_double(ht_x);
    number_double(ht_y);
    number_clone (ht_x_abs, ht_x);
    number_clone (ht_y_abs, ht_y);
    number_abs (ht_x_abs);
    number_abs (ht_y_abs);
  }
  @<Scan the pen polygon between |w0| and |w| and make |max_ht| the range dot
    product with |(ht_x,ht_y)|@>;
  {
    mp_number  r1 ,r2;
    new_fraction (r1);
    new_fraction (r2);
    take_fraction (r1, dxin, ht_x);
    take_fraction (r2, dyin, ht_y);
    number_add (r1, r2);
    make_fraction (tmp, max_ht, r1);
    free_number (r1);
    free_number (r2);   
  }
  take_fraction (xsub, tmp, dxin);
  take_fraction (ysub, tmp, dyin);
  set_number_from_addition(xtot, p->x_coord, xsub);
  set_number_from_addition(ytot, p->y_coord, ysub);
  r = mp_insert_knot (mp, p, xtot, ytot);
  /* clang: value never read */ assert(r);
  {
    mp_number  r1 ,r2;
    new_fraction (r1);
    new_fraction (r2);
    take_fraction (r1, dxout, ht_x);
    take_fraction (r2, dyout, ht_y);
    number_add (r1, r2);
    make_fraction (tmp, max_ht, r1);
    free_number (r1);
    free_number (r2);   
  }
  take_fraction (xsub, tmp, dxout);
  take_fraction (ysub, tmp, dyout);
  set_number_from_addition(xtot, q->x_coord, xsub);
  set_number_from_addition(ytot, q->y_coord, ysub);
  r = mp_insert_knot (mp, p, xtot, ytot);
  free_number (xsub);
  free_number (ysub);
  free_number (xtot);
  free_number (ytot);
  free_number (ht_x);
  free_number (ht_y);
  free_number (ht_x_abs);
  free_number (ht_y_abs);
}


@ @<Other local variables for |make_envelope|@>=
mp_number max_ht;  /* maximum height of the pen polygon above the |w0|-|w| line */
halfword kk;    /* keeps track of the pen vertices being scanned */
mp_knot ww;     /* the pen vertex being tested */

@ The dot product of the vector from |w0| to |ww| with |(ht_x,ht_y)| ranges
from zero to |max_ht|.

@<Scan the pen polygon between |w0| and |w| and make |max_ht| the range...@>=
set_number_to_zero (max_ht);
kk = zero_off;
ww = w;
while (1) {
  @<Step |ww| and move |kk| one step closer to |k0|@>;
  if (kk == k0)
    break;
  {
    mp_number r1, r2;
    new_fraction (r1);
    new_fraction (r2);
    set_number_from_substraction (tmp, ww->x_coord, w0->x_coord);
    take_fraction (r1, tmp, ht_x);
    set_number_from_substraction (tmp, ww->y_coord, w0->y_coord);
    take_fraction (r2, tmp, ht_y);
    set_number_from_addition (tmp, r1,  r2);
    free_number (r1);
    free_number (r2);
  }
  if (number_greater(tmp, max_ht))
    number_clone(max_ht, tmp);
}


@ @<Step |ww| and move |kk| one step closer to |k0|@>=
if (kk > k0) {
  ww = mp_next_knot (ww);
  decr (kk);
} else {
  ww = mp_prev_knot (ww);
  incr (kk);
}


@ @<If endpoint, double the path |c|, and set |spec_p1| and |spec_p2|@>=
if (mp_left_type (c) == mp_endpoint) {
  mp->spec_p1 = mp_htap_ypoc (mp, c);
  mp->spec_p2 = mp->path_tail;
  mp_originator (mp->spec_p1) = mp_program_code;
  mp_next_knot (mp->spec_p2) = mp_next_knot (mp->spec_p1);
  mp_next_knot (mp->spec_p1) = c;
  mp_remove_cubic (mp, mp->spec_p1);
  c = mp->spec_p1;
  if (c != mp_next_knot (c)) {
    mp_originator (mp->spec_p2) = mp_program_code;
    mp_remove_cubic (mp, mp->spec_p2);
  } else {
    @<Make |c| look like a cycle of length one@>;
  }
}

@ @<Make |c| look like a cycle of length one@>=
{
  mp_left_type (c) = mp_explicit;
  mp_right_type (c) = mp_explicit;
  number_clone(c->left_x, c->x_coord);
  number_clone(c->left_y, c->y_coord);
  number_clone(c->right_x, c->x_coord);
  number_clone(c->right_y, c->y_coord);
}


@ In degenerate situations we might have to look at the knot preceding~|q|.
That knot is |p| but if |p<>c|, its coordinates have already been offset by |w|.

@<Set the incoming and outgoing directions at |q|; in case of...@>=
{
  set_number_from_substraction(dxin, q->x_coord, q->left_x);
  set_number_from_substraction(dyin, q->y_coord, q->left_y);
  if (number_zero(dxin) && number_zero(dyin)) {
    set_number_from_substraction(dxin, q->x_coord, p->right_x);
    set_number_from_substraction(dyin, q->y_coord, p->right_y);
    if (number_zero(dxin) && number_zero(dyin)) {
      set_number_from_substraction(dxin, q->x_coord, p->x_coord);
      set_number_from_substraction(dyin, q->y_coord, p->y_coord);
      if (p != c) {                 /* the coordinates of |p| have been offset by |w| */
        number_add(dxin, w->x_coord);
        number_add(dyin, w->y_coord);
      }
    }
  }
  pyth_add (tmp, dxin, dyin);
  if (number_zero(tmp)) {
    join_type = 2;
  } else {
    mp_number r1;
    new_fraction (r1);
    make_fraction (r1, dxin, tmp);
    number_clone(dxin, r1);
    make_fraction (r1, dyin, tmp);
    number_clone(dyin, r1);
    free_number (r1);
    @<Set the outgoing direction at |q|@>;
  }
}


@ If |q=c| then the coordinates of |r| and the control points between |q|
and~|r| have already been offset by |h|.

@<Set the outgoing direction at |q|@>=
{
  set_number_from_substraction(dxout, q->right_x, q->x_coord);
  set_number_from_substraction(dyout, q->right_y, q->y_coord);
  if (number_zero(dxout) && number_zero(dyout)) {
    r = mp_next_knot (q);
    set_number_from_substraction(dxout, r->left_x, q->x_coord);
    set_number_from_substraction(dyout, r->left_y, q->y_coord);
    if (number_zero(dxout) && number_zero(dyout)) {
      set_number_from_substraction(dxout, r->x_coord, q->x_coord);
      set_number_from_substraction(dyout, r->y_coord, q->y_coord);
    }
  }
  if (q == c) {
    number_substract(dxout, h->x_coord);
    number_substract(dyout, h->y_coord);
  }
  pyth_add (tmp, dxout, dyout);
  if (number_zero(tmp)) {
    /* |mp_confusion (mp, "degenerate spec");| */
@:this can't happen degerate spec}{\quad degenerate spec@>;
    /* But apparently, it actually can happen. The test case is this:  

  path p;
  linejoin := mitered;
  p:= (10,0)..(0,10)..(-10,0)..(0,-10)..cycle;
  addto currentpicture contour p withpen pensquare;

  The reason for failure here is the addition of |r != q| in revision 1757 
  in ``Advance |p| to node |q|, removing any ``dead'' cubics'', which itself 
  was needed to fix a bug with disappearing knots in a path that was rotated 
  exactly 45 degrees (luatex.org bug 530).
     */
  } else {
    mp_number r1;
    new_fraction (r1);
    make_fraction (r1, dxout, tmp);
    number_clone(dxout, r1);
    make_fraction (r1, dyout, tmp);
    number_clone(dyout, r1);
    free_number (r1);
  }
}
 

@* Direction and intersection times.
A path of length $n$ is defined parametrically by functions $x(t)$ and
$y(t)$, for |0<=t<=n|; we can regard $t$ as the ``time'' at which the path
reaches the point $\bigl(x(t),y(t)\bigr)$.  In this section of the program
we shall consider operations that determine special times associated with
given paths: the first time that a path travels in a given direction, and
a pair of times at which two paths cross each other.

@ Let's start with the easier task. The function |find_direction_time| is
given a direction |(x,y)| and a path starting at~|h|. If the path never
travels in direction |(x,y)|, the direction time will be~|-1|; otherwise
it will be nonnegative.

Certain anomalous cases can arise: If |(x,y)=(0,0)|, so that the given
direction is undefined, the direction time will be~0. If $\bigl(x'(t),
y'(t)\bigr)=(0,0)$, so that the path direction is undefined, it will be
assumed to match any given direction at time~|t|.

The routine solves this problem in nondegenerate cases by rotating the path
and the given direction so that |(x,y)=(1,0)|; i.e., the main task will be
to find when a given path first travels ``due east.''

@c
static void mp_find_direction_time (MP mp, mp_number *ret, mp_number x_orig, mp_number y_orig, mp_knot h) {
  mp_number max;   /* $\max\bigl(\vert x\vert,\vert y\vert\bigr)$ */
  mp_knot p, q; /* for list traversal */
  mp_number n;     /* the direction time at knot |p| */
  mp_number tt;    /* the direction time within a cubic */
  mp_number x, y;
  mp_number abs_x, abs_y;
  /* Other local variables for |find_direction_time| */
  mp_number x1, x2, x3, y1, y2, y3;  /* multiples of rotated derivatives */
  mp_number phi;       /* angles of exit and entry at a knot */
  mp_number t;     /* temp storage */
  mp_number ab_vs_cd;
  new_number(max);
  new_number(x1);
  new_number(x2);
  new_number(x3);
  new_number(y1);
  new_number(y2);
  new_number(y3);
  new_fraction(t);
  new_angle(phi);
  new_number (ab_vs_cd);
  set_number_to_zero (*ret); /* just in case */
  new_number (x);
  new_number (y);
  new_number (abs_x);
  new_number (abs_y);
  new_number (n);
  new_fraction (tt); 
  number_clone (x, x_orig);
  number_clone (y, y_orig);
  number_clone (abs_x, x_orig);
  number_clone (abs_y, y_orig);
  number_abs (abs_x);
  number_abs (abs_y);
  /* Normalize the given direction for better accuracy;
     but |return| with zero result if it's zero */
  if (number_less(abs_x, abs_y)) {
    mp_number r1;
    new_fraction (r1);
    make_fraction (r1, x, abs_y);
    number_clone(x, r1);
    free_number (r1);
    if (number_positive(y)) {
      number_clone(y, fraction_one_t);
    } else {
      number_clone(y, fraction_one_t);
      number_negate(y);
    }
  } else if (number_zero(x)) {
    goto FREE;
  } else {
    mp_number r1;
    new_fraction (r1);
    make_fraction (r1, y, abs_x);
    number_clone(y, r1);
    free_number (r1);
    if (number_positive(x)) {
      number_clone(x, fraction_one_t);
    } else {
      number_clone(x, fraction_one_t);
      number_negate(x);
    }
  }

  p = h;
  while (1) {
    if (mp_right_type (p) == mp_endpoint)
      break;
    q = mp_next_knot (p);
    @<Rotate the cubic between |p| and |q|; then
      |goto found| if the rotated cubic travels due east at some time |tt|;
      but |break| if an entire cyclic path has been traversed@>;
    p = q;
    number_add(n, unity_t);
  }
  set_number_to_unity (*ret);
  number_negate(*ret);
  goto FREE;
FOUND:
  set_number_from_addition (*ret, n, tt);
  goto FREE;
FREE: 
  free_number (x);
  free_number (y);
  free_number (abs_x);
  free_number (abs_y);
  /* Free local variables for |find_direction_time| */
  free_number (x1);
  free_number (x2);
  free_number (x3);
  free_number (y1);
  free_number (y2);
  free_number (y3);
  free_number (t);
  free_number (phi);
  free_number (ab_vs_cd);

  free_number (n);
  free_number (max);
  free_number (tt);
}



@ Since we're interested in the tangent directions, we work with the
derivative $${1\over3}B'(x_0,x_1,x_2,x_3;t)=
B(x_1-x_0,x_2-x_1,x_3-x_2;t)$$ instead of
$B(x_0,x_1,x_2,x_3;t)$ itself. The derived coefficients are also scale-d up
in order to achieve better accuracy.

The given path may turn abruptly at a knot, and it might pass the critical
tangent direction at such a time. Therefore we remember the direction |phi|
in which the previous rotated cubic was traveling. (The value of |phi| will be
undefined on the first cubic, i.e., when |n=0|.)

@d we_found_it { 
  number_clone (tt, t);
  fraction_to_round_scaled (tt);
  goto FOUND; 
}

@<Rotate the cubic between |p| and |q|; then...@>=
set_number_to_zero(tt);
/* Set local variables |x1,x2,x3| and |y1,y2,y3| to multiples of the control
   points of the rotated derivatives */
{
  mp_number absval; 
  new_number (absval);
  set_number_from_substraction(x1, p->right_x, p->x_coord);
  set_number_from_substraction(x2, q->left_x,  p->right_x);
  set_number_from_substraction(x3, q->x_coord, q->left_x);
  set_number_from_substraction(y1, p->right_y, p->y_coord);
  set_number_from_substraction(y2, q->left_y,  p->right_y);
  set_number_from_substraction(y3, q->y_coord, q->left_y);
  number_clone(absval, x2);
  number_abs(absval);
  number_clone(max, x1);
  number_abs(max);
  if (number_greater(absval, max)) {
    number_clone(max, absval);
  }
  number_clone(absval, x3);
  number_abs(absval);
  if (number_greater(absval, max)) {
    number_clone(max, absval);
  }
  number_clone(absval, y1);
  number_abs(absval);
  if (number_greater(absval, max)) {
    number_clone(max, absval);
  }
  number_clone(absval, y2);
  number_abs(absval);
  if (number_greater(absval, max)) {
    number_clone(max, absval);
  }
  number_clone(absval, y3);
  number_abs(absval);
  if (number_greater(absval, max)) {
    number_clone(max, absval);
  }
  free_number (absval);
  if (number_zero(max))
    goto FOUND;
  while (number_less (max, fraction_half_t)) {
    number_double(max);
    number_double(x1);
    number_double(x2);
    number_double(x3);
    number_double(y1);
    number_double(y2);
    number_double(y3);
  }
  number_clone(t, x1);
  {
     mp_number r1, r2;
     new_fraction (r1);
     new_fraction (r2);
     take_fraction (r1, x1, x);
     take_fraction (r2, y1, y);
     set_number_from_addition(x1, r1, r2);
     take_fraction (r1, y1, x);
     take_fraction (r2, t, y);
     set_number_from_substraction(y1, r1, r2);
     number_clone(t, x2);
     take_fraction (r1, x2, x); 
     take_fraction (r2, y2, y);
     set_number_from_addition(x2, r1, r2);
     take_fraction (r1, y2, x); 
     take_fraction (r2, t, y);
     set_number_from_substraction(y2, r1, r2);
     number_clone(t, x3);
     take_fraction (r1, x3 ,x); 
     take_fraction (r2, y3, y);
     set_number_from_addition(x3, r1, r2);
     take_fraction (r1, y3, x); 
     take_fraction (r2, t, y);
     set_number_from_substraction(y3, r1, r2);
     free_number (r1);
     free_number (r2);
  }
}
if (number_zero(y1))
  if (number_zero(x1) || number_positive(x1))
    goto FOUND;
if (number_positive(n)) {
  /* Exit to |found| if an eastward direction occurs at knot |p| */
  mp_number theta;
  mp_number tmp;
  new_angle (theta);
  n_arg (theta, x1, y1);
  new_angle (tmp);
  set_number_from_substraction (tmp, theta, one_eighty_deg_t);

  if (number_nonnegative(theta) && number_nonpositive(phi) && number_greaterequal(phi, tmp)) {
    free_number (tmp);
    free_number (theta);
    goto FOUND;
  }
  set_number_from_addition (tmp, theta, one_eighty_deg_t);
  if (number_nonpositive(theta) && number_nonnegative(phi) && number_lessequal(phi, tmp)) {
    free_number (tmp);
    free_number (theta);
    goto FOUND;
  }
  free_number (tmp);
  free_number (theta);

  if (p == h)
    break;
}
if (number_nonzero(x3) || number_nonzero(y3)) {
  n_arg (phi, x3, y3);
}
/* Exit to |found| if the curve whose derivatives are specified by
   |x1,x2,x3,y1,y2,y3| travels eastward at some time~|tt| */
/* In this step we want to use the |crossing_point| routine to find the
roots of the quadratic equation $B(y_1,y_2,y_3;t)=0$.
Several complications arise: If the quadratic equation has a double root,
the curve never crosses zero, and |crossing_point| will find nothing;
this case occurs iff $y_1y_3=y_2^2$ and $y_1y_2<0$. If the quadratic
equation has simple roots, or only one root, we may have to negate it
so that $B(y_1,y_2,y_3;t)$ crosses from positive to negative at its first root.
And finally, we need to do special things if $B(y_1,y_2,y_3;t)$ is
identically zero. */
if (number_negative(x1))
  if (number_negative(x2))
    if (number_negative(x3))
      goto DONE;
{
  ab_vs_cd (ab_vs_cd, y1, y3, y2, y2);
  if (number_zero(ab_vs_cd)) {
    /* Handle the test for eastward directions when $y_1y_3=y_2^2$;
      either |goto found| or |goto done| */
{
  ab_vs_cd (ab_vs_cd, y1, y2, zero_t, zero_t);
  if (number_negative(ab_vs_cd)) {
    mp_number tmp, arg2;
    new_number(tmp);
    new_number(arg2);
    set_number_from_substraction (arg2, y1, y2);
    make_fraction (t, y1, arg2);
    free_number (arg2);
    set_number_from_of_the_way(x1, t, x1, x2);
    set_number_from_of_the_way(x2, t, x2, x3);
    set_number_from_of_the_way(tmp, t, x1, x2);
    if (number_zero(tmp) || number_positive(tmp)) {
      free_number (tmp);
      we_found_it;
    }
    free_number (tmp);
  } else if (number_zero(y3)) {
    if (number_zero(y1)) {
      /* Exit to |found| if the derivative $B(x_1,x_2,x_3;t)$ becomes |>=0| */
/* At this point we know that the derivative of |y(t)| is identically zero,
and that |x1<0|; but either |x2>=0| or |x3>=0|, so there's some hope of
traveling east. */
{
  mp_number arg1, arg2, arg3;
  new_number (arg1);
  new_number (arg2);
  new_number (arg3);
  number_clone(arg1, x1);
  number_negate(arg1);
  number_clone(arg2, x2);
  number_negate(arg2);
  number_clone(arg3, x3);
  number_negate(arg3);
  crossing_point (t, arg1, arg2, arg3);
  free_number (arg1);
  free_number (arg2);
  free_number (arg3);
  if (number_lessequal (t, fraction_one_t))
    we_found_it;
  ab_vs_cd (ab_vs_cd, x1, x3, x2, x2);
  if (number_nonpositive(ab_vs_cd)) {
    mp_number arg2;
    new_number (arg2);
    set_number_from_substraction (arg2, x1, x2);
    make_fraction (t, x1, arg2);
    free_number (arg2);
    we_found_it;
  }
}



    } else if (number_zero(x3) || number_positive(x3)) {
      set_number_to_unity(tt);
      goto FOUND;
    }
  }
  goto DONE;
}


  }
}
if (number_zero(y1) || number_negative(y1)) {
  if (number_negative(y1)) {
    number_negate(y1);
    number_negate(y2);
    number_negate(y3);
  } else if (number_positive(y2)) {
    number_negate(y2);
    number_negate(y3);
  }
}
/* Check the places where $B(y_1,y_2,y_3;t)=0$ to see if
  $B(x_1,x_2,x_3;t)\ge0$ */
/* The quadratic polynomial $B(y_1,y_2,y_3;t)$ begins |>=0| and has at most
two roots, because we know that it isn't identically zero.

It must be admitted that the |crossing_point| routine is not perfectly accurate;
rounding errors might cause it to find a root when $y_1y_3>y_2^2$, or to
miss the roots when $y_1y_3<y_2^2$. The rotation process is itself
subject to rounding errors. Yet this code optimistically tries to
do the right thing.

*/
crossing_point (t, y1, y2, y3);
if (number_greater (t, fraction_one_t))
  goto DONE;
set_number_from_of_the_way(y2, t, y2, y3);
set_number_from_of_the_way(x1, t, x1, x2);
set_number_from_of_the_way(x2, t, x2, x3);
set_number_from_of_the_way(x1, t, x1, x2);
if (number_zero(x1) || number_positive(x1))
  we_found_it;
if (number_positive(y2))
  set_number_to_zero(y2);
number_clone(tt, t);
{
  mp_number arg1, arg2, arg3;
  new_number (arg1);
  new_number (arg2);
  new_number (arg3);
  number_clone(arg2, y2);
  number_negate(arg2);
  number_clone(arg3, y3);
  number_negate(arg3);
  crossing_point (t, arg1, arg2, arg3);
  free_number (arg1);
  free_number (arg2);
  free_number (arg3);
}
if (number_greater (t, fraction_one_t))
  goto DONE;
{
  mp_number tmp;
  new_number(tmp);
  set_number_from_of_the_way(x1, t, x1, x2);
  set_number_from_of_the_way(x2, t, x2, x3);
  set_number_from_of_the_way(tmp, t, x1, x2);
  if (number_nonnegative(tmp)) {
    free_number (tmp);
    set_number_from_of_the_way (t, t, tt, fraction_one_t);
    we_found_it;
  }
  free_number (tmp);
}
DONE:


@ The intersection of two cubics can be found by an interesting variant
of the general bisection scheme described in the introduction to
|crossing_point|.\
Given $w(t)=B(w_0,w_1,w_2,w_3;t)$ and $z(t)=B(z_0,z_1,z_2,z_3;t)$,
we wish to find a pair of times $(t_1,t_2)$ such that $w(t_1)=z(t_2)$,
if an intersection exists. First we find the smallest rectangle that
encloses the points $\{w_0,w_1,w_2,w_3\}$ and check that it overlaps
the smallest rectangle that encloses
$\{z_0,z_1,z_2,z_3\}$; if not, the cubics certainly don't intersect.
But if the rectangles do overlap, we bisect the intervals, getting
new cubics $w'$ and~$w''$, $z'$~and~$z''$; the intersection routine first
tries for an intersection between $w'$ and~$z'$, then (if unsuccessful)
between $w'$ and~$z''$, then (if still unsuccessful) between $w''$ and~$z'$,
finally (if thrice unsuccessful) between $w''$ and~$z''$. After $l$~successful
levels of bisection we will have determined the intersection times $t_1$
and~$t_2$ to $l$~bits of accuracy.

\def\submin{_{\rm min}} \def\submax{_{\rm max}}
As before, it is better to work with the numbers $W_k=2^l(w_k-w_{k-1})$
and $Z_k=2^l(z_k-z_{k-1})$ rather than the coefficients $w_k$ and $z_k$
themselves. We also need one other quantity, $\Delta=2^l(w_0-z_0)$,
to determine when the enclosing rectangles overlap. Here's why:
The $x$~coordinates of~$w(t)$ are between $u\submin$ and $u\submax$,
and the $x$~coordinates of~$z(t)$ are between $x\submin$ and $x\submax$,
if we write $w_k=(u_k,v_k)$ and $z_k=(x_k,y_k)$ and $u\submin=
\min(u_0,u_1,u_2,u_3)$, etc. These intervals of $x$~coordinates
overlap if and only if $u\submin\L x\submax$ and
$x\submin\L u\submax$. Letting
$$U\submin=\min(0,U_1,U_1+U_2,U_1+U_2+U_3),\;
  U\submax=\max(0,U_1,U_1+U_2,U_1+U_2+U_3),$$
we have $2^lu\submin=2^lu_0+U\submin$, etc.; the condition for overlap
reduces to
$$X\submin-U\submax\L 2^l(u_0-x_0)\L X\submax-U\submin.$$
Thus we want to maintain the quantity $2^l(u_0-x_0)$; similarly,
the quantity $2^l(v_0-y_0)$ accounts for the $y$~coordinates. The
coordinates of $\Delta=2^l(w_0-z_0)$ must stay bounded as $l$ increases,
because of the overlap condition; i.e., we know that $X\submin$,
$X\submax$, and their relatives are bounded, hence $X\submax-
U\submin$ and $X\submin-U\submax$ are bounded.

@ Incidentally, if the given cubics intersect more than once, the process
just sketched will not necessarily find the lexicographically smallest pair
$(t_1,t_2)$. The solution actually obtained will be smallest in ``shuffled
order''; i.e., if $t_1=(.a_1a_2\ldots a_{16})_2$ and
$t_2=(.b_1b_2\ldots b_{16})_2$, then we will minimize
$a_1b_1a_2b_2\ldots a_{16}b_{16}$, not
$a_1a_2\ldots a_{16}b_1b_2\ldots b_{16}$.
Shuffled order agrees with lexicographic order if all pairs of solutions
$(t_1,t_2)$ and $(t_1',t_2')$ have the property that $t_1<t_1'$ iff
$t_2<t_2'$; but in general, lexicographic order can be quite different,
and the bisection algorithm would be substantially less efficient if it were
constrained by lexicographic order.

For example, suppose that an overlap has been found for $l=3$ and
$(t_1,t_2)= (.101,.011)$ in binary, but that no overlap is produced by
either of the alternatives $(.1010,.0110)$, $(.1010,.0111)$ at level~4.
Then there is probably an intersection in one of the subintervals
$(.1011,.011x)$; but lexicographic order would require us to explore
$(.1010,.1xxx)$ and $(.1011,.00xx)$ and $(.1011,.010x)$ first. We wouldn't
want to store all of the subdivision data for the second path, so the
subdivisions would have to be regenerated many times. Such inefficiencies
would be associated with every `1' in the binary representation of~$t_1$.

@ The subdivision process introduces rounding errors, hence we need to
make a more liberal test for overlap. It is not hard to show that the
computed values of $U_i$ differ from the truth by at most~$l$, on
level~$l$, hence $U\submin$ and $U\submax$ will be at most $3l$ in error.
If $\beta$ is an upper bound on the absolute error in the computed
components of $\Delta=(|delx|,|dely|)$ on level~$l$, we will replace
the test `$X\submin-U\submax\L|delx|$' by the more liberal test
`$X\submin-U\submax\L|delx|+|tol|$', where $|tol|=6l+\beta$.

More accuracy is obtained if we try the algorithm first with |tol=0|;
the more liberal tolerance is used only if an exact approach fails.
It is convenient to do this double-take by letting `3' in the preceding
paragraph be a parameter, which is first 0, then 3.

@<Glob...@>=
unsigned int tol_step;  /* either 0 or 3, usually */

@ We shall use an explicit stack to implement the recursive bisection
method described above. The |bisect_stack| array will contain numerous 5-word
packets like $(U_1,U_2,U_3,U\submin,U\submax)$, as well as 20-word packets
comprising the 5-word packets for $U$, $V$, $X$, and~$Y$.

The following macros define the allocation of stack positions to
the quantities needed for bisection-intersection.

@d stack_1(A) mp->bisect_stack[(A)] /* $U_1$, $V_1$, $X_1$, or $Y_1$ */
@d stack_2(A) mp->bisect_stack[(A)+1] /* $U_2$, $V_2$, $X_2$, or $Y_2$ */
@d stack_3(A) mp->bisect_stack[(A)+2] /* $U_3$, $V_3$, $X_3$, or $Y_3$ */
@d stack_min(A) mp->bisect_stack[(A)+3]
  /* $U\submin$, $V\submin$, $X\submin$, or $Y\submin$ */
@d stack_max(A) mp->bisect_stack[(A)+4]
  /* $U\submax$, $V\submax$, $X\submax$, or $Y\submax$ */
@d int_packets 20 /* number of words to represent $U_k$, $V_k$, $X_k$, and $Y_k$ */
@#
@d u_packet(A) ((A)-5)
@d v_packet(A) ((A)-10)
@d x_packet(A) ((A)-15)
@d y_packet(A) ((A)-20)
@d l_packets (mp->bisect_ptr-int_packets)
@d r_packets mp->bisect_ptr
@d ul_packet u_packet(l_packets) /* base of $U'_k$ variables */
@d vl_packet v_packet(l_packets) /* base of $V'_k$ variables */
@d xl_packet x_packet(l_packets) /* base of $X'_k$ variables */
@d yl_packet y_packet(l_packets) /* base of $Y'_k$ variables */
@d ur_packet u_packet(r_packets) /* base of $U''_k$ variables */
@d vr_packet v_packet(r_packets) /* base of $V''_k$ variables */
@d xr_packet x_packet(r_packets) /* base of $X''_k$ variables */
@d yr_packet y_packet(r_packets) /* base of $Y''_k$ variables */
@#
@d u1l stack_1(ul_packet) /* $U'_1$ */
@d u2l stack_2(ul_packet) /* $U'_2$ */
@d u3l stack_3(ul_packet) /* $U'_3$ */
@d v1l stack_1(vl_packet) /* $V'_1$ */
@d v2l stack_2(vl_packet) /* $V'_2$ */
@d v3l stack_3(vl_packet) /* $V'_3$ */
@d x1l stack_1(xl_packet) /* $X'_1$ */
@d x2l stack_2(xl_packet) /* $X'_2$ */
@d x3l stack_3(xl_packet) /* $X'_3$ */
@d y1l stack_1(yl_packet) /* $Y'_1$ */
@d y2l stack_2(yl_packet) /* $Y'_2$ */
@d y3l stack_3(yl_packet) /* $Y'_3$ */
@d u1r stack_1(ur_packet) /* $U''_1$ */
@d u2r stack_2(ur_packet) /* $U''_2$ */
@d u3r stack_3(ur_packet) /* $U''_3$ */
@d v1r stack_1(vr_packet) /* $V''_1$ */
@d v2r stack_2(vr_packet) /* $V''_2$ */
@d v3r stack_3(vr_packet) /* $V''_3$ */
@d x1r stack_1(xr_packet) /* $X''_1$ */
@d x2r stack_2(xr_packet) /* $X''_2$ */
@d x3r stack_3(xr_packet) /* $X''_3$ */
@d y1r stack_1(yr_packet) /* $Y''_1$ */
@d y2r stack_2(yr_packet) /* $Y''_2$ */
@d y3r stack_3(yr_packet) /* $Y''_3$ */
@#
@d stack_dx mp->bisect_stack[mp->bisect_ptr] /* stacked value of |delx| */
@d stack_dy mp->bisect_stack[mp->bisect_ptr+1] /* stacked value of |dely| */
@d stack_tol mp->bisect_stack[mp->bisect_ptr+2] /* stacked value of |tol| */
@d stack_uv mp->bisect_stack[mp->bisect_ptr+3] /* stacked value of |uv| */
@d stack_xy mp->bisect_stack[mp->bisect_ptr+4] /* stacked value of |xy| */
@d int_increment (int_packets+int_packets+5) /* number of stack words per level */

@<Glob...@>=
mp_number *bisect_stack;
integer bisect_ptr;

@ @<Allocate or initialize ...@>=
mp->bisect_stack = xmalloc ((bistack_size + 1), sizeof (mp_number));
{
  int i;
  for (i=0;i<bistack_size + 1;i++) {
    new_number (mp->bisect_stack[i]);
  }
}

@ @<Dealloc variables@>=
{
  int i;
  for (i=0;i<bistack_size + 1;i++) {
    free_number (mp->bisect_stack[i]);
  }
}
xfree (mp->bisect_stack);

@ @<Check the ``constant''...@>=
if (int_packets + 17 * int_increment > bistack_size)
  mp->bad = 19;

@ Computation of the min and max is a tedious but fairly fast sequence of
instructions; exactly four comparisons are made in each branch.

@d set_min_max(A)
  debug_number (stack_1(A));
  debug_number (stack_3(A));
  debug_number (stack_2(A));
  debug_number (stack_min(A));
  debug_number (stack_max(A));
  if ( number_negative(stack_1((A))) ) {
    if ( number_nonnegative (stack_3((A))) ) {
      if ( number_negative (stack_2((A))) ) 
        set_number_from_addition (stack_min((A)), stack_1((A)), stack_2((A)));
      else 
        number_clone (stack_min((A)), stack_1((A)));
      set_number_from_addition (stack_max((A)), stack_1((A)), stack_2((A)));
      number_add (stack_max((A)), stack_3((A)));
      if ( number_negative (stack_max((A))) ) 
        set_number_to_zero (stack_max((A)));
    } else { 
      set_number_from_addition (stack_min((A)), stack_1((A)), stack_2((A)));
      number_add (stack_min((A)), stack_3((A)));
      if ( number_greater (stack_min((A)), stack_1((A)))) 
        number_clone (stack_min((A)), stack_1((A)));
      set_number_from_addition (stack_max((A)), stack_1((A)), stack_2((A)));
      if ( number_negative (stack_max((A))) ) 
        set_number_to_zero (stack_max((A)));
    }
  } else if ( number_nonpositive (stack_3((A)))) {
    if ( number_positive (stack_2((A))) ) 
      set_number_from_addition (stack_max((A)), stack_1((A)), stack_2((A)));
    else 
      number_clone (stack_max((A)), stack_1((A)));
    set_number_from_addition (stack_min((A)), stack_1((A)), stack_2((A)));
    number_add (stack_min((A)), stack_3((A)));
    if ( number_positive (stack_min((A))) ) 
      set_number_to_zero (stack_min((A)));
  } else  { 
    set_number_from_addition (stack_max((A)), stack_1((A)), stack_2((A)));
    number_add (stack_max((A)), stack_3((A)));
    if ( number_less (stack_max((A)), stack_1((A)))) 
      number_clone (stack_max((A)), stack_1((A)));
    set_number_from_addition (stack_min((A)), stack_1((A)), stack_2((A)));
    if ( number_positive (stack_min((A))) ) 
      set_number_to_zero (stack_min((A)));
  }

@ It's convenient to keep the current values of $l$, $t_1$, and $t_2$ in
the integer form $2^l+2^lt_1$ and $2^l+2^lt_2$. The |cubic_intersection|
routine uses global variables |cur_t| and |cur_tt| for this purpose;
after successful completion, |cur_t| and |cur_tt| will contain |unity|
plus the |scaled| values of $t_1$ and~$t_2$.

The values of |cur_t| and |cur_tt| will be set to zero if |cubic_intersection|
finds no intersection. The routine gives up and gives an approximate answer
if it has backtracked
more than 5000 times (otherwise there are cases where several minutes
of fruitless computation would be possible).

@d max_patience 5000

@<Glob...@>=
mp_number cur_t;
mp_number cur_tt; /* controls and results of |cubic_intersection| */
integer time_to_go;     /* this many backtracks before giving up */
mp_number max_t;  /* maximum of $2^{l+1}$ so far achieved */

@ @<Initialize table ...@>=
new_number (mp->cur_t);
new_number (mp->cur_tt);
new_number (mp->max_t);

@ @<Dealloc ...@>=
free_number (mp->cur_t);
free_number (mp->cur_tt);
free_number (mp->max_t);

@ The given cubics $B(w_0,w_1,w_2,w_3;t)$ and
$B(z_0,z_1,z_2,z_3;t)$ are specified in adjacent knot nodes |(p,mp_link(p))|
and |(pp,mp_link(pp))|, respectively.
 
@d half(A) ((A)/2)
 
@c
static void mp_cubic_intersection (MP mp, mp_knot p, mp_knot pp) {
  mp_knot q, qq;        /* |mp_link(p)|, |mp_link(pp)| */
  mp->time_to_go = max_patience;
  set_number_from_scaled (mp->max_t, 2);
  @<Initialize for intersections at level zero@>;
CONTINUE:
  while (1) {
    if (number_to_scaled (mp->delx) - mp->tol <=
        number_to_scaled (stack_max (x_packet (mp->xy))) - number_to_scaled (stack_min (u_packet (mp->uv))))
      if (number_to_scaled (mp->delx) + mp->tol >=
          number_to_scaled (stack_min (x_packet (mp->xy))) - number_to_scaled (stack_max (u_packet (mp->uv))))
        if (number_to_scaled (mp->dely) - mp->tol <=
            number_to_scaled (stack_max (y_packet (mp->xy))) - number_to_scaled (stack_min (v_packet (mp->uv))))
          if (number_to_scaled (mp->dely) + mp->tol >=
              number_to_scaled (stack_min (y_packet (mp->xy))) - number_to_scaled (stack_max (v_packet (mp->uv)))) {
            if (number_to_scaled (mp->cur_t) >= number_to_scaled (mp->max_t)) {
              if (number_equal(mp->max_t, two_t)) {   /* we've done 17 bisections */
                set_number_from_scaled (mp->cur_t, ((number_to_scaled (mp->cur_t) + 1)/2));
                set_number_from_scaled (mp->cur_tt, ((number_to_scaled (mp->cur_tt) + 1)/2));
                return;
              }
              number_double(mp->max_t);
              number_clone (mp->appr_t, mp->cur_t);
              number_clone (mp->appr_tt, mp->cur_tt);
            }
            @<Subdivide for a new level of intersection@>;
            goto CONTINUE;
          }
    if (mp->time_to_go > 0) {
      decr (mp->time_to_go);
    } else {
      while (number_less (mp->appr_t, unity_t)) {
        number_double(mp->appr_t);
        number_double(mp->appr_tt);
      }
      number_clone (mp->cur_t, mp->appr_t);
      number_clone (mp->cur_tt, mp->appr_tt);
      return;
    }
  NOT_FOUND:
    /* Advance to the next pair |(cur_t,cur_tt)| */
    if (odd (number_to_scaled (mp->cur_tt))) {
      if (odd (number_to_scaled (mp->cur_t))) {
        /* Descend to the previous level and |goto not_found| */
        {
          set_number_from_scaled (mp->cur_t, half (number_to_scaled (mp->cur_t)));
          set_number_from_scaled (mp->cur_tt, half (number_to_scaled (mp->cur_tt)));
          if (number_to_scaled (mp->cur_t) == 0)
            return;
          mp->bisect_ptr -= int_increment;
          mp->three_l -= (integer) mp->tol_step;
          number_clone (mp->delx, stack_dx);
          number_clone (mp->dely, stack_dy);
          mp->tol = number_to_scaled (stack_tol);
          mp->uv = number_to_scaled (stack_uv);
          mp->xy = number_to_scaled (stack_xy);
          goto NOT_FOUND;
        }
        
      } else {
        set_number_from_scaled (mp->cur_t, number_to_scaled (mp->cur_t) + 1);
        number_add (mp->delx, stack_1 (u_packet (mp->uv)));
        number_add (mp->delx, stack_2 (u_packet (mp->uv)));
        number_add (mp->delx, stack_3 (u_packet (mp->uv)));
        number_add (mp->dely, stack_1 (v_packet (mp->uv)));
        number_add (mp->dely, stack_2 (v_packet (mp->uv)));
        number_add (mp->dely, stack_3 (v_packet (mp->uv)));
        mp->uv = mp->uv + int_packets;      /* switch from |l_packets| to |r_packets| */
        set_number_from_scaled (mp->cur_tt, number_to_scaled (mp->cur_tt) - 1);
        mp->xy = mp->xy - int_packets;
        number_add (mp->delx, stack_1 (x_packet (mp->xy)));
        number_add (mp->delx, stack_2 (x_packet (mp->xy)));
        number_add (mp->delx, stack_3 (x_packet (mp->xy)));
        number_add (mp->dely, stack_1 (y_packet (mp->xy)));
        number_add (mp->dely, stack_2 (y_packet (mp->xy)));
        number_add (mp->dely, stack_3 (y_packet (mp->xy)));
      }
    } else {
      set_number_from_scaled (mp->cur_tt, number_to_scaled (mp->cur_tt) + 1);
      mp->tol = mp->tol + mp->three_l;
      number_substract (mp->delx, stack_1 (x_packet (mp->xy)));
      number_substract (mp->delx, stack_2 (x_packet (mp->xy)));
      number_substract (mp->delx, stack_3 (x_packet (mp->xy)));
      number_substract (mp->dely, stack_1 (y_packet (mp->xy)));
      number_substract (mp->dely, stack_2 (y_packet (mp->xy)));
      number_substract (mp->dely, stack_3 (y_packet (mp->xy)));
      mp->xy = mp->xy + int_packets;        /* switch from |l_packets| to |r_packets| */
    }
  }
}


@ The following variables are global, although they are used only by
|cubic_intersection|, because it is necessary on some machines to
split |cubic_intersection| up into two procedures.

@<Glob...@>=
mp_number delx;
mp_number dely;   /* the components of $\Delta=2^l(w_0-z_0)$ */
integer tol;    /* bound on the uncertainty in the overlap test */
integer uv;
integer xy;     /* pointers to the current packets of interest */
integer three_l;        /* |tol_step| times the bisection level */
mp_number appr_t;
mp_number appr_tt;        /* best approximations known to the answers */

@ @<Initialize table ...@>=
new_number (mp->delx);
new_number (mp->dely);
new_number (mp->appr_t);
new_number (mp->appr_tt);

@ @<Dealloc...@>=
free_number (mp->delx);
free_number (mp->dely);
free_number (mp->appr_t);
free_number (mp->appr_tt);

@ We shall assume that the coordinates are sufficiently non-extreme that
integer overflow will not occur.
@^overflow in arithmetic@>

@<Initialize for intersections at level zero@>=
q = mp_next_knot (p);
qq = mp_next_knot (pp);
mp->bisect_ptr = int_packets;
set_number_from_substraction (u1r, p->right_x, p->x_coord);
set_number_from_substraction (u2r, q->left_x, p->right_x);
set_number_from_substraction (u3r, q->x_coord, q->left_x);
set_min_max (ur_packet);
set_number_from_substraction (v1r, p->right_y, p->y_coord );
set_number_from_substraction (v2r, q->left_y, p->right_y);
set_number_from_substraction (v3r, q->y_coord, q->left_y );
set_min_max (vr_packet);
set_number_from_substraction (x1r, pp->right_x, pp->x_coord );
set_number_from_substraction (x2r, qq->left_x, pp->right_x );
set_number_from_substraction (x3r, qq->x_coord, qq->left_x );
set_min_max (xr_packet);
set_number_from_substraction (y1r, pp->right_y, pp->y_coord );
set_number_from_substraction (y2r, qq->left_y, pp->right_y);
set_number_from_substraction (y3r, qq->y_coord, qq->left_y);
set_min_max (yr_packet);
set_number_from_substraction (mp->delx, p->x_coord, pp->x_coord );
set_number_from_substraction (mp->dely, p->y_coord, pp->y_coord );
mp->tol = 0;
mp->uv = r_packets;
mp->xy = r_packets;
mp->three_l = 0;
set_number_from_scaled (mp->cur_t, 1);
set_number_from_scaled (mp->cur_tt, 1)

@ 

@<Subdivide for a new level of intersection@>=
number_clone (stack_dx, mp->delx);
number_clone (stack_dy, mp->dely);
set_number_from_scaled (stack_tol, mp->tol);
set_number_from_scaled (stack_uv, mp->uv);
set_number_from_scaled (stack_xy, mp->xy);
mp->bisect_ptr = mp->bisect_ptr + int_increment;
number_double (mp->cur_t);
number_double (mp->cur_tt);
number_clone (u1l, stack_1 (u_packet (mp->uv)));
number_clone (u3r, stack_3 (u_packet (mp->uv)));
set_number_from_addition (u2l, u1l, stack_2 (u_packet (mp->uv))); number_half (u2l);
set_number_from_addition (u2r, u3r, stack_2 (u_packet (mp->uv))); number_half (u2r);
set_number_from_addition (u3l, u2l, u2r); number_half (u3l);
number_clone (u1r, u3l);
set_min_max (ul_packet);
set_min_max (ur_packet);
number_clone (v1l, stack_1 (v_packet (mp->uv)));
number_clone (v3r, stack_3 (v_packet (mp->uv)));
set_number_from_addition (v2l, v1l, stack_2 (v_packet (mp->uv))); number_half(v2l);
set_number_from_addition (v2r, v3r, stack_2 (v_packet (mp->uv))); number_half(v2r);
set_number_from_addition (v3l, v2l, v2r); number_half(v3l);
number_clone (v1r, v3l);
set_min_max (vl_packet);
set_min_max (vr_packet);
number_clone (x1l, stack_1 (x_packet (mp->xy)));
number_clone (x3r, stack_3 (x_packet (mp->xy)));
set_number_from_addition (x2l, x1l, stack_2 (x_packet (mp->xy))); number_half(x2l);
set_number_from_addition (x2r, x3r, stack_2 (x_packet (mp->xy))); number_half(x2r);
set_number_from_addition (x3l, x2l, x2r); number_half(x3l);
number_clone (x1r, x3l);
set_min_max (xl_packet);
set_min_max (xr_packet);
number_clone (y1l, stack_1 (y_packet (mp->xy)));
number_clone (y3r, stack_3 (y_packet (mp->xy)));
set_number_from_addition (y2l, y1l, stack_2 (y_packet (mp->xy))); number_half (y2l);
set_number_from_addition (y2r, y3r, stack_2 (y_packet (mp->xy))); number_half (y2r);
set_number_from_addition (y3l, y2l, y2r); number_half (y3l);
number_clone (y1r, y3l);
set_min_max (yl_packet);
set_min_max (yr_packet);
mp->uv = l_packets;
mp->xy = l_packets;
number_double(mp->delx);
number_double(mp->dely);
mp->tol = mp->tol - mp->three_l + (integer) mp->tol_step;
mp->tol += mp->tol;
mp->three_l = mp->three_l + (integer) mp->tol_step

@ The |path_intersection| procedure is much simpler.
It invokes |cubic_intersection| in lexicographic order until finding a
pair of cubics that intersect. The final intersection times are placed in
|cur_t| and~|cur_tt|.

@c
static void mp_path_intersection (MP mp, mp_knot h, mp_knot hh) {
  mp_knot p, pp;        /* link registers that traverse the given paths */
  mp_number n, nn;        /* integer parts of intersection times, minus |unity| */
  @<Change one-point paths into dead cycles@>;
  new_number (n);
  new_number (nn);
  mp->tol_step = 0;
  do {
    set_number_to_unity(n);
    number_negate (n);
    p = h;
    do {
      if (mp_right_type (p) != mp_endpoint) {
        set_number_to_unity(nn);
        number_negate (nn);
        pp = hh;
        do {
          if (mp_right_type (pp) != mp_endpoint) {
            mp_cubic_intersection (mp, p, pp);
            if (number_positive (mp->cur_t)) {
              number_add (mp->cur_t, n);
              number_add (mp->cur_tt, nn);
              goto DONE;
            }
          }
          number_add(nn, unity_t);
          pp = mp_next_knot (pp);
        } while (pp != hh);
      }
      number_add(n, unity_t);
      p = mp_next_knot (p);
    } while (p != h);
    mp->tol_step = mp->tol_step + 3;
  } while (mp->tol_step <= 3);
  number_clone (mp->cur_t, unity_t);
  number_negate (mp->cur_t);
  number_clone (mp->cur_tt, unity_t);
  number_negate (mp->cur_tt);
DONE:
  free_number (n);
  free_number (nn);
}


@ @<Change one-point paths...@>=
if (mp_right_type (h) == mp_endpoint) {
  number_clone (h->right_x, h->x_coord);
  number_clone (h->left_x, h->x_coord);
  number_clone (h->right_y, h->y_coord);
  number_clone (h->left_y, h->y_coord);
  mp_right_type (h) = mp_explicit;
}
if (mp_right_type (hh) == mp_endpoint) {
  number_clone (hh->right_x, hh->x_coord);
  number_clone (hh->left_x, hh->x_coord);
  number_clone (hh->right_y, hh->y_coord);
  number_clone (hh->left_y, hh->y_coord);
  mp_right_type (hh) = mp_explicit;
}

@* Dynamic linear equations.
\MP\ users define variables implicitly by stating equations that should be
satisfied; the computer is supposed to be smart enough to solve those equations.
And indeed, the computer tries valiantly to do so, by distinguishing five
different types of numeric values:

\smallskip\hang
|type(p)=mp_known| is the nice case, when |value(p)| is the |scaled| value
of the variable whose address is~|p|.

\smallskip\hang
|type(p)=mp_dependent| means that |value(p)| is not present, but |dep_list(p)|
points to a {\sl dependency list\/} that expresses the value of variable~|p|
as a |scaled| number plus a sum of independent variables with |fraction|
coefficients.

\smallskip\hang
|type(p)=mp_independent| means that |indep_value(p)=s|, where |s>0| is a ``serial
number'' reflecting the time this variable was first used in an equation;
and there is an extra field |indep_scale(p)=m|, with |0<=m<64|, each dependent 
variable that refers to this one is actually referring to the future value of
this variable times~$2^m$. (Usually |m=0|, but higher degrees of
scaling are sometimes needed to keep the coefficients in dependency lists
from getting too large. The value of~|m| will always be even.)

\smallskip\hang
|type(p)=mp_numeric_type| means that variable |p| hasn't appeared in an
equation before, but it has been explicitly declared to be numeric.

\smallskip\hang
|type(p)=undefined| means that variable |p| hasn't appeared before.

\smallskip\noindent
We have actually discussed these five types in the reverse order of their
history during a computation: Once |known|, a variable never again
becomes |dependent|; once |dependent|, it almost never again becomes
|mp_independent|; once |mp_independent|, it never again becomes |mp_numeric_type|;
and once |mp_numeric_type|, it never again becomes |undefined| (except
of course when the user specifically decides to scrap the old value
and start again). A backward step may, however, take place: Sometimes
a |dependent| variable becomes |mp_independent| again, when one of the
independent variables it depends on is reverting to |undefined|.

@d indep_scale(A) ((mp_value_node)(A))->data.indep.scale
@d set_indep_scale(A,B) ((mp_value_node)(A))->data.indep.scale=(B)
@d indep_value(A) ((mp_value_node)(A))->data.indep.serial
@d set_indep_value(A,B) ((mp_value_node)(A))->data.indep.serial=(B)


@c 
void mp_new_indep(MP mp, mp_node p)  { /* create a new independent variable */
  if ( mp->serial_no>=max_integer ) {
    mp_fatal_error(mp, "variable instance identifiers exhausted");
  }
  mp_type(p)=mp_independent;
  mp->serial_no=mp->serial_no+1;
  set_indep_scale(p,0);
  set_indep_value(p,mp->serial_no);
}

@ @<Declarations@>=
void mp_new_indep(MP mp, mp_node p);


@ @<Glob...@>=
integer serial_no;      /* the most recent serial number */

@ But how are dependency lists represented? It's simple: The linear combination
$\alpha_1v_1+\cdots+\alpha_kv_k+\beta$ appears in |k+1| value nodes. If
|q=dep_list(p)| points to this list, and if |k>0|, then |dep_value(q)=
@t$\alpha_1$@>| (which is a |fraction|); |dep_info(q)| points to the location
of $\alpha_1$; and |mp_link(p)| points to the dependency list
$\alpha_2v_2+\cdots+\alpha_kv_k+\beta$. On the other hand if |k=0|,
then |dep_value(q)=@t$\beta$@>| (which is |scaled|) and |dep_info(q)=NULL|.
The independent variables $v_1$, \dots,~$v_k$ have been sorted so that
they appear in decreasing order of their |value| fields (i.e., of
their serial numbers). \ (It is convenient to use decreasing order,
since |value(NULL)=0|. If the independent variables were not sorted by
serial number but by some other criterion, such as their location in |mem|,
the equation-solving mechanism would be too system-dependent, because
the ordering can affect the computed results.)

The |link| field in the node that contains the constant term $\beta$ is
called the {\sl final link\/} of the dependency list. \MP\ maintains
a doubly-linked master list of all dependency lists, in terms of a permanently
allocated node
in |mem| called |dep_head|. If there are no dependencies, we have
|mp_link(dep_head)=dep_head| and |prev_dep(dep_head)=dep_head|;
otherwise |mp_link(dep_head)| points to the first dependent variable, say~|p|,
and |prev_dep(p)=dep_head|. We have |type(p)=mp_dependent|, and |dep_list(p)|
points to its dependency list. If the final link of that dependency list
occurs in location~|q|, then |mp_link(q)| points to the next dependent
variable (say~|r|); and we have |prev_dep(r)=q|, etc.

Dependency nodes sometimes mutate into value nodes and vice versa, so their
structures have to match.

@d dep_value(A) ((mp_value_node)(A))->data.n
@d set_dep_value(A,B) do_set_dep_value(mp,(A),(B)) 
@d dep_info(A) get_dep_info(mp, (A))
@d set_dep_info(A,B) do {
   mp_value_node d = (mp_value_node)(B);
   FUNCTION_TRACE4("set_dep_info(%p,%p) on %d\n",(A),d,__LINE__);
  ((mp_value_node)(A))->parent_ = (mp_node)d;
} while (0)
@d dep_list(A) ((mp_value_node)(A))->attr_head_  /* half of the |value| field in a |dependent| variable */
@d set_dep_list(A,B) do {
   mp_value_node d = (mp_value_node)(B);
   FUNCTION_TRACE4("set_dep_list(%p,%p) on %d\n",(A),d,__LINE__);
   dep_list((A)) = (mp_node)d;
} while (0)
@d prev_dep(A) ((mp_value_node)(A))->subscr_head_ /* the other half; makes a doubly linked list */
@d set_prev_dep(A,B) do {
   mp_value_node d = (mp_value_node)(B);
   FUNCTION_TRACE4("set_prev_dep(%p,%p) on %d\n",(A),d,__LINE__);
   prev_dep((A)) = (mp_node)d;
} while (0)

@c
static mp_node get_dep_info (MP mp, mp_value_node p) {
  mp_node d;
  d = p->parent_;               /* half of the |value| field in a |dependent| variable */
  FUNCTION_TRACE3 ("%p = dep_info(%p)\n", d, p);
  return d;
}
static void do_set_dep_value (MP mp, mp_value_node p, mp_number q) {
   number_clone (p->data.n, q);  /* half of the |value| field in a |dependent| variable */
   FUNCTION_TRACE3("set_dep_value(%p,%d)\n", p, q);
   p->attr_head_ = NULL;
   p->subscr_head_ = NULL;
}

@ @<Declarations...@>=
static mp_node get_dep_info (MP mp, mp_value_node p);

@ 

@c
static mp_value_node mp_get_dep_node (MP mp) {
  mp_value_node p = (mp_value_node) mp_get_value_node (mp);
  mp_type (p) = mp_dep_node_type;
  return p;
}
static void mp_free_dep_node (MP mp, mp_value_node p) {
  mp_free_value_node (mp, (mp_node) p);
}


@ @<Declarations...@>=
static void mp_free_dep_node (MP mp, mp_value_node p);

@ @<Initialize table entries@>=
mp->serial_no = 0;
mp->dep_head = mp_get_dep_node (mp);
set_mp_link (mp->dep_head, (mp_node) mp->dep_head);
set_prev_dep (mp->dep_head, (mp_node) mp->dep_head);
set_dep_info (mp->dep_head, NULL);
set_dep_list (mp->dep_head, NULL);

@ @<Free table entries@>=
mp_free_dep_node (mp, mp->dep_head);

@ Actually the description above contains a little white lie. There's
another kind of variable called |mp_proto_dependent|, which is
just like a |dependent| one except that the $\alpha$ coefficients
in its dependency list are |scaled| instead of being fractions.
Proto-dependency lists are mixed with dependency lists in the
nodes reachable from |dep_head|.

@ Here is a procedure that prints a dependency list in symbolic form.
The second parameter should be either |dependent| or |mp_proto_dependent|,
to indicate the scaling of the coefficients.

@<Declarations@>=
static void mp_print_dependency (MP mp, mp_value_node p, quarterword t);

@ @c
void mp_print_dependency (MP mp, mp_value_node p, quarterword t) {
  mp_number v;    /* a coefficient */
  mp_value_node pp;     /* for list manipulation */
  mp_node q;
  pp = p;
  new_number (v);
  while (true) {
    number_clone (v, dep_value (p));
    number_abs (v);
    q = dep_info (p);
    if (q == NULL) {            /* the constant term */
      if (number_nonzero(v) || (p == pp)) {
        if (number_positive(dep_value (p)))
          if (p != pp)
            mp_print_char (mp, xord ('+'));
        print_number (dep_value (p));
      }
      return;
    }
    /* Print the coefficient, unless it's $\pm1.0$ */
    if (number_negative(dep_value (p)))
      mp_print_char (mp, xord ('-'));
    else if (p != pp)
      mp_print_char (mp, xord ('+'));
    if (t == mp_dependent) {
      fraction_to_round_scaled (v);
    }
    if (!number_equal (v, unity_t))
      print_number (v);
   
    if (mp_type (q) != mp_independent)
      mp_confusion (mp, "dep");
    mp_print_variable_name (mp, q);
    set_number_from_scaled (v, indep_scale(q));
    while (number_positive (v)) {
      mp_print (mp, "*4");
      number_add_scaled (v, -2);
    }
    p = (mp_value_node) mp_link (p);
  }
}



@ The maximum absolute value of a coefficient in a given dependency list
is returned by the following simple function.

@c
static void mp_max_coef (MP mp, mp_number *x, mp_value_node p) {
  mp_number (absv);
  new_number (absv);
  set_number_to_zero (*x);
  while (dep_info (p) != NULL) {
    number_clone (absv, dep_value (p));
    number_abs (absv);
    if (number_greater (absv, *x)) {
      number_clone (*x, absv);
    }
    p = (mp_value_node) mp_link (p);
  }
  free_number (absv);
}


@ One of the main operations needed on dependency lists is to add a multiple
of one list to the other; we call this |p_plus_fq|, where |p| and~|q| point
to dependency lists and |f| is a fraction.

If the coefficient of any independent variable becomes |coef_bound| or
more, in absolute value, this procedure changes the type of that variable
to `|independent_needing_fix|', and sets the global variable |fix_needed|
to~|true|. The value of $|coef_bound|=\mu$ is chosen so that
$\mu^2+\mu<8$; this means that the numbers we deal with won't
get too large. (Instead of the ``optimum'' $\mu=(\sqrt{33}-1)/2\approx
2.3723$, the safer value 7/3 is taken as the threshold.)

The changes mentioned in the preceding paragraph are actually done only if
the global variable |watch_coefs| is |true|. But it usually is; in fact,
it is |false| only when \MP\ is making a dependency list that will soon
be equated to zero.

Several procedures that act on dependency lists, including |p_plus_fq|,
set the global variable |dep_final| to the final (constant term) node of
the dependency list that they produce.

@d independent_needing_fix 0

@<Glob...@>=
boolean fix_needed;     /* does at least one |independent| variable need scaling? */
boolean watch_coefs;    /* should we scale coefficients that exceed |coef_bound|? */
mp_value_node dep_final;        /* location of the constant term and final link */

@ @<Set init...@>=
mp->fix_needed = false;
mp->watch_coefs = true;

@ The |p_plus_fq| procedure has a fourth parameter, |t|, that should be
set to |mp_proto_dependent| if |p| is a proto-dependency list. In this
case |f| will be |scaled|, not a |fraction|. Similarly, the fifth parameter~|tt|
should be |mp_proto_dependent| if |q| is a proto-dependency list.

List |q| is unchanged by the operation; but list |p| is totally destroyed.

The final link of the dependency list or proto-dependency list returned
by |p_plus_fq| is the same as the original final link of~|p|. Indeed, the
constant term of the result will be located in the same |mem| location
as the original constant term of~|p|.

Coefficients of the result are assumed to be zero if they are less than
a certain threshold. This compensates for inevitable rounding errors,
and tends to make more variables `|known|'. The threshold is approximately
$10^{-5}$ in the case of normal dependency lists, $10^{-4}$ for
proto-dependencies.

@d fraction_threshold_k ((math_data *)mp->math)->fraction_threshold_t
@d half_fraction_threshold_k ((math_data *)mp->math)->half_fraction_threshold_t
@d scaled_threshold_k ((math_data *)mp->math)->scaled_threshold_t
@d half_scaled_threshold_k ((math_data *)mp->math)->half_scaled_threshold_t

@<Declarations@>=
static mp_value_node mp_p_plus_fq (MP mp, mp_value_node p, mp_number f,
                                   mp_value_node q, mp_variable_type t,
                                   mp_variable_type tt);

@ @c
static mp_value_node mp_p_plus_fq (MP mp, mp_value_node p, mp_number f,
                                   mp_value_node q, mp_variable_type t,
                                   mp_variable_type tt) {
  mp_node pp, qq;       /* |dep_info(p)| and |dep_info(q)|, respectively */
  mp_value_node r, s;   /* for list manipulation */
  mp_number threshold, half_threshold;    /* defines a neighborhood of zero */
  mp_number v, vv; /* temporary registers */
  new_number (v);
  new_number (vv);
  new_number (threshold); 
  new_number (half_threshold); 
  if (t == mp_dependent) {
    number_clone (threshold, fraction_threshold_k);
    number_clone (half_threshold, half_fraction_threshold_k);
  } else {
    number_clone (threshold, scaled_threshold_k);
    number_clone (half_threshold, half_scaled_threshold_k);
  }
  r = (mp_value_node) mp->temp_head;
  pp = dep_info (p);
  qq = dep_info (q);
  while (1) {
    if (pp == qq) {
      if (pp == NULL) {
        break;
      } else {
        /* Contribute a term from |p|, plus |f| times the
          corresponding term from |q| */
        mp_number r1;
        mp_number absv;
        new_fraction (r1);
        new_number (absv);
        if (tt == mp_dependent) {
          take_fraction (r1, f, dep_value (q));
        } else {
          take_scaled (r1, f, dep_value (q));
        }
        set_number_from_addition (v, dep_value (p), r1);
        free_number (r1);
        set_dep_value (p, v);
        s = p;
        p = (mp_value_node) mp_link (p);
        number_clone (absv, v);
        number_abs(absv);
        if (number_less (absv, threshold)) {
          mp_free_dep_node (mp, s);
        } else {
          if (number_greaterequal (absv, coef_bound_k) && mp->watch_coefs) {
            mp_type (qq) = independent_needing_fix;
	    /* If we set this , then we can drop (mp_type(pp) == independent_needing_fix && mp->fix_needed) later */
	    /* set_number_from_scaled (value_number (qq), indep_value(qq)); */
            mp->fix_needed = true;
          }
          set_mp_link (r, (mp_node) s);
          r = s;
        }
        free_number (absv);
        pp = dep_info (p);
        q = (mp_value_node) mp_link (q);
        qq = dep_info (q);
      }

    } else {
      if (pp == NULL) 
        set_number_to_neg_inf(v);
      else if (mp_type(pp) == mp_independent || (mp_type(pp) == independent_needing_fix && mp->fix_needed))
        set_number_from_scaled(v, indep_value(pp));
      else
        number_clone (v, value_number (pp));
      if (qq == NULL) 
        set_number_to_neg_inf(vv);
      else if (mp_type(qq) == mp_independent || (mp_type(qq) == independent_needing_fix && mp->fix_needed))
        set_number_from_scaled(vv, indep_value(qq));
      else
        number_clone (vv, value_number (qq));
      if (number_less (v, vv)) {
        /* Contribute a term from |q|, multiplied by~|f| */
        mp_number absv;
        new_number (absv);
        {
          mp_number r1;
          mp_number arg1, arg2;
          new_fraction (r1);
          new_number (arg1);
          new_number (arg2);
          number_clone (arg1, f);
          number_clone (arg2, dep_value (q));
          if (tt == mp_dependent) {
            take_fraction (r1, arg1, arg2);
          } else {
            take_scaled (r1, arg1, arg2);
          }
          number_clone (v, r1);
          free_number (r1);
          free_number (arg1);
          free_number (arg2);
        }
        number_clone (absv, v);
        number_abs(absv);
        if (number_greater (absv, half_threshold)) {
          s = mp_get_dep_node (mp);
          set_dep_info (s, qq);
          set_dep_value (s, v);
          if (number_greaterequal(absv, coef_bound_k) && mp->watch_coefs) {
          /* clang:  dereference of a null pointer ('qq') */ assert(qq);
           mp_type (qq) = independent_needing_fix;
            mp->fix_needed = true;
          }
          set_mp_link (r, (mp_node) s);
          r = s;
        }
        q = (mp_value_node) mp_link (q);
        qq = dep_info (q);
        free_number (absv);
      
      } else {
        set_mp_link (r, (mp_node) p);
        r = p;
        p = (mp_value_node) mp_link (p);
        pp = dep_info (p);
      }
    }
  }
  {
    mp_number r1;
    mp_number arg1, arg2;
    new_fraction (r1);
    new_number (arg1);
    new_number (arg2);
    number_clone (arg1, dep_value (q));
    number_clone (arg2, f);
    if (t == mp_dependent) {
      take_fraction (r1, arg1, arg2);
    } else {
      take_scaled (r1, arg1, arg2);
    }
    slow_add (arg1, dep_value (p), r1);
    set_dep_value (p, arg1);
    free_number (r1);
    free_number (arg1);
    free_number (arg2);
  }
  set_mp_link (r, (mp_node) p);
  mp->dep_final = p;
  free_number (threshold);
  free_number (half_threshold);
  free_number (v);
  free_number (vv);
  return (mp_value_node) mp_link (mp->temp_head);
}


@ It is convenient to have another subroutine for the special case
of |p_plus_fq| when |f=1.0|. In this routine lists |p| and |q| are
both of the same type~|t| (either |dependent| or |mp_proto_dependent|).

@c
static mp_value_node mp_p_plus_q (MP mp, mp_value_node p, mp_value_node q,
                                  mp_variable_type t) {
  mp_node pp, qq;       /* |dep_info(p)| and |dep_info(q)|, respectively */
  mp_value_node s;      /* for list manipulation */
  mp_value_node r;      /* for list manipulation */
  mp_number threshold;    /* defines a neighborhood of zero */
  mp_number v, vv;        /* temporary register */
  new_number (v);
  new_number (vv);
  new_number (threshold);
  if (t == mp_dependent)
    number_clone (threshold, fraction_threshold_k);
  else
    number_clone (threshold, scaled_threshold_k);
  r = (mp_value_node) mp->temp_head;
  pp = dep_info (p);
  qq = dep_info (q);
  while (1) {
    if (pp == qq) {
      if (pp == NULL) {
        break;
      } else {
        /* Contribute a term from |p|, plus the corresponding term from |q| */
        mp_number test;
        new_number (test);
        set_number_from_addition (v, dep_value (p), dep_value (q)); 
        set_dep_value (p, v);
        s = p;
        p = (mp_value_node) mp_link (p);
        pp = dep_info (p);
        number_clone (test, v);
        number_abs(test);
        if (number_less (test, threshold)) {
          mp_free_dep_node (mp, s);
        } else {
          if (number_greaterequal(test, coef_bound_k) && mp->watch_coefs) {
            mp_type (qq) = independent_needing_fix;
	    /* If we set this , then we can drop (mp_type(pp) == independent_needing_fix && mp->fix_needed) later */
	    /* set_number_from_scaled (value_number (qq), indep_value(qq)); */
            mp->fix_needed = true;
          }
          set_mp_link (r, (mp_node) s);
          r = s;
        }
        free_number (test);
        q = (mp_value_node) mp_link (q);
        qq = dep_info (q);
      }

    } else {
      if (pp == NULL) 
        set_number_to_zero (v);
      else if (mp_type(pp) == mp_independent || (mp_type(pp) == independent_needing_fix && mp->fix_needed))
        set_number_from_scaled (v, indep_value(pp));
      else
        number_clone (v, value_number (pp));
      if (qq == NULL) 
        set_number_to_zero (vv);
      else if (mp_type(qq) == mp_independent || (mp_type(qq) == independent_needing_fix && mp->fix_needed))
        set_number_from_scaled (vv, indep_value(qq));
      else
        number_clone (vv, value_number (qq));
      if (number_less (v, vv)) {
        s = mp_get_dep_node (mp);
        set_dep_info (s, qq);
        set_dep_value (s, dep_value (q));
        q = (mp_value_node) mp_link (q);
        qq = dep_info (q);
        set_mp_link (r, (mp_node) s);
        r = s;
      } else {
        set_mp_link (r, (mp_node) p);
        r = p;
        p = (mp_value_node) mp_link (p);
        pp = dep_info (p);
      }
    }
  }
  {
    mp_number r1;
    new_number (r1);
    slow_add (r1, dep_value (p), dep_value (q));
    set_dep_value (p, r1);
    free_number (r1);
  }
  set_mp_link (r, (mp_node) p);
  mp->dep_final = p;
  free_number (v);
  free_number (vv);
  free_number (threshold);
  return (mp_value_node) mp_link (mp->temp_head);
}

@ A somewhat simpler routine will multiply a dependency list
by a given constant~|v|. The constant is either a |fraction| less than
|fraction_one|, or it is |scaled|. In the latter case we might be forced to
convert a dependency list to a proto-dependency list.
Parameters |t0| and |t1| are the list types before and after;
they should agree unless |t0=mp_dependent| and |t1=mp_proto_dependent|
and |v_is_scaled=true|.

@c
static mp_value_node mp_p_times_v (MP mp, mp_value_node p, mp_number v,
                                   quarterword t0, quarterword t1,
                                   boolean v_is_scaled) {
  mp_value_node r, s;   /* for list manipulation */
  mp_number w;    /* tentative coefficient */
  mp_number threshold;
  boolean scaling_down;
  new_number (threshold);
  new_number (w);
  if (t0 != t1)
    scaling_down = true;
  else
    scaling_down = (!v_is_scaled);
  if (t1 == mp_dependent)
    number_clone (threshold, half_fraction_threshold_k);
  else
    number_clone (threshold, half_scaled_threshold_k);
  r = (mp_value_node) mp->temp_head;
  while (dep_info (p) != NULL) {
    mp_number test;
    new_number (test);
    if (scaling_down) {
      take_fraction (w, v, dep_value (p));
    } else {
      take_scaled (w, v, dep_value (p));
    }
    number_clone (test, w);
    number_abs(test);
    if (number_lessequal (test, threshold)) {
      s = (mp_value_node) mp_link (p);
      mp_free_dep_node (mp, p);
      p = s;
    } else {
      if (number_greaterequal(test, coef_bound_k)) {
        mp->fix_needed = true;
        mp_type (dep_info (p)) = independent_needing_fix;
      }
      set_mp_link (r, (mp_node) p);
      r = p;
      set_dep_value (p, w);
      p = (mp_value_node) mp_link (p);
    }
    free_number (test);
  }
  set_mp_link (r, (mp_node) p);
  {
    mp_number r1;
    new_number (r1);
    if (v_is_scaled) {
      take_scaled (r1, dep_value (p), v);
    } else {
      take_fraction (r1, dep_value (p), v);
    }
    set_dep_value (p, r1);
    free_number (r1);
  }
  free_number (w);
  free_number (threshold);
  return (mp_value_node) mp_link (mp->temp_head);
}


@ Similarly, we sometimes need to divide a dependency list
by a given |scaled| constant.

@<Declarations@>=
static mp_value_node mp_p_over_v (MP mp, mp_value_node p, mp_number v, quarterword
                                  t0, quarterword t1);

@ 
@d p_over_v_threshold_k ((math_data *)mp->math)->p_over_v_threshold_t

@c
mp_value_node mp_p_over_v (MP mp, mp_value_node p, mp_number v_orig, quarterword
                           t0, quarterword t1) {
  mp_value_node r, s;   /* for list manipulation */
  mp_number w;    /* tentative coefficient */
  mp_number threshold;
  mp_number v;
  boolean scaling_down;
  new_number (v);
  new_number (w);
  new_number (threshold);
  number_clone (v, v_orig);
  if (t0 != t1)
    scaling_down = true;
  else
    scaling_down = false;
  if (t1 == mp_dependent)
    number_clone (threshold, half_fraction_threshold_k);
  else
    number_clone (threshold, half_scaled_threshold_k);
  r = (mp_value_node) mp->temp_head;
  while (dep_info (p) != NULL) {
    if (scaling_down) {
      mp_number x, absv;
      new_number (x);
      new_number (absv);
      number_clone (absv, v);
      number_abs (absv);
      if (number_less (absv, p_over_v_threshold_k)) {
        number_clone (x, v);
        convert_scaled_to_fraction (x);
        make_scaled (w, dep_value (p), x);
      } else {
        number_clone (x, dep_value (p));
        fraction_to_round_scaled (x);
        make_scaled (w, x, v);
      }
      free_number (x);
      free_number (absv);
    } else {
      make_scaled (w, dep_value (p), v);
    }
    {
    mp_number test;
    new_number (test);
    number_clone (test, w);
    number_abs(test);
    if (number_lessequal (test, threshold)) {
      s = (mp_value_node) mp_link (p);
      mp_free_dep_node (mp, p);
      p = s;
    } else {
      if (number_greaterequal (test, coef_bound_k)) {
        mp->fix_needed = true;
        mp_type (dep_info (p)) = independent_needing_fix;
      }
      set_mp_link (r, (mp_node) p);
      r = p;
      set_dep_value (p, w);
      p = (mp_value_node) mp_link (p);
    }
    free_number (test);
    }
  }
  set_mp_link (r, (mp_node) p);
  {
    mp_number ret;
    new_number (ret);
    make_scaled (ret, dep_value (p), v);
    set_dep_value (p, ret);
    free_number (ret);
  }
  free_number (v);
  free_number (w);
  free_number (threshold);
  return (mp_value_node) mp_link (mp->temp_head);
}


@ Here's another utility routine for dependency lists. When an independent
variable becomes dependent, we want to remove it from all existing
dependencies. The |p_with_x_becoming_q| function computes the
dependency list of~|p| after variable~|x| has been replaced by~|q|.

This procedure has basically the same calling conventions as |p_plus_fq|:
List~|q| is unchanged; list~|p| is destroyed; the constant node and the
final link are inherited from~|p|; and the fourth parameter tells whether
or not |p| is |mp_proto_dependent|. However, the global variable |dep_final|
is not altered if |x| does not occur in list~|p|.

@c
static mp_value_node mp_p_with_x_becoming_q (MP mp, mp_value_node p,
                                             mp_node x, mp_node q,
                                             quarterword t) {
  mp_value_node r, s;   /* for list manipulation */
  integer sx;   /* serial number of |x| */
  s = p;
  r = (mp_value_node) mp->temp_head;
  sx = indep_value (x);
  while (dep_info (s) != NULL && indep_value (dep_info (s)) > sx) {
    r = s;
    s = (mp_value_node) mp_link (s);
  }
  if (dep_info (s) == NULL || dep_info (s) != x) {
    return p;
  } else {
    mp_value_node ret;
    mp_number v1;
    new_number (v1);
    set_mp_link (mp->temp_head, (mp_node) p);
    set_mp_link (r, mp_link (s));
    number_clone (v1, dep_value (s));
    mp_free_dep_node (mp, s);
    ret = mp_p_plus_fq (mp, (mp_value_node) mp_link (mp->temp_head), v1,
                         (mp_value_node) q, t, mp_dependent);
    free_number (v1);
    return ret;
  }
}


@ Here's a simple procedure that reports an error when a variable
has just received a known value that's out of the required range.

@<Declarations@>=
static void mp_val_too_big (MP mp, mp_number x);

@ @c
static void mp_val_too_big (MP mp, mp_number x) {
  if (number_positive (internal_value (mp_warning_check))) {
    char msg[256];
    const char *hlp[] = {
           "The equation I just processed has given some variable a",
           "value outside of the safetyp range. Continue and I'll try",
           "to cope with that big value; but it might be dangerous.",
           "(Set warningcheck:=0 to suppress this message.)",
           NULL };
    mp_snprintf (msg, 256, "Value is too large (%s)", number_tostring(x));
    mp_error (mp, msg, hlp, true);
  }
}

@ When a dependent variable becomes known, the following routine
removes its dependency list. Here |p| points to the variable, and
|q| points to the dependency list (which is one node long).

@<Declarations@>=
static void mp_make_known (MP mp, mp_value_node p, mp_value_node q);

@ @c
void mp_make_known (MP mp, mp_value_node p, mp_value_node q) {
  mp_variable_type t;   /* the previous type */
  mp_number absp;
  new_number (absp);
  set_prev_dep (mp_link (q), prev_dep (p));
  set_mp_link (prev_dep (p), mp_link (q));
  t = mp_type (p);
  mp_type (p) = mp_known;
  set_value_number (p, dep_value (q));
  mp_free_dep_node (mp, q);
  number_clone (absp, value_number (p));
  number_abs (absp);
  if (number_greaterequal (absp, warning_limit_t))
    mp_val_too_big (mp, value_number (p));
  if ((number_positive(internal_value (mp_tracing_equations)))
      && mp_interesting (mp, (mp_node) p)) {
    mp_begin_diagnostic (mp);
    mp_print_nl (mp, "#### ");
    mp_print_variable_name (mp, (mp_node) p);
    mp_print_char (mp, xord ('='));
    print_number (value_number (p));
    mp_end_diagnostic (mp, false);
  }
  if (cur_exp_node () == (mp_node) p && mp->cur_exp.type == t) {
    mp->cur_exp.type = mp_known;
    set_cur_exp_value_number (value_number (p));
    mp_free_value_node (mp, (mp_node) p);
  }
  free_number (absp);
}


@ The |fix_dependencies| routine is called into action when |fix_needed|
has been triggered. The program keeps a list~|s| of independent variables
whose coefficients must be divided by~4.

In unusual cases, this fixup process might reduce one or more coefficients
to zero, so that a variable will become known more or less by default.

@<Declarations@>=
static void mp_fix_dependencies (MP mp);

@ 
@d independent_being_fixed 1 /* this variable already appears in |s| */
@c
static void mp_fix_dependencies (MP mp) {
  mp_value_node p, q, r, s, t;  /* list manipulation registers */
  mp_node x;    /* an independent variable */
  r = (mp_value_node) mp_link (mp->dep_head);
  s = NULL;
  while (r != mp->dep_head) {
    t = r;
    /* Run through the dependency list for variable |t|, fixing
      all nodes, and ending with final link~|q| */
    while (1) {
      if (t==r) {
        q = (mp_value_node) dep_list(t);
      } else {
        q = (mp_value_node) mp_link (r);
      }
      x = dep_info (q);
      if (x == NULL)
        break;
      if (mp_type (x) <= independent_being_fixed) {
        if (mp_type (x) < independent_being_fixed) {
          p = mp_get_dep_node (mp);
          set_mp_link (p, (mp_node) s);
          s = p;
          set_dep_info (s, x);
          mp_type (x) = independent_being_fixed;
        }
        set_dep_value (q, dep_value (q));
        number_divide_int (dep_value (q), 4);
        if (number_zero(dep_value (q))) {
          set_mp_link (r, mp_link (q));
          mp_free_dep_node (mp, q);
          q = r;
        }
      }
      r = q;
    }

    r = (mp_value_node) mp_link (q);
    if (q == (mp_value_node) dep_list (t))
      mp_make_known (mp, t, q);
  }
  while (s != NULL) {
    p = (mp_value_node) mp_link (s);
    x = dep_info (s);
    mp_free_dep_node (mp, s);
    s = p;
    mp_type (x) = mp_independent;
    set_indep_scale (x, indep_scale (x) + 2);
  }
  mp->fix_needed = false;
}


@ The |new_dep| routine installs a dependency list~|p| based on the value node~|q|,
linking it into the list of all known dependencies. It replaces |q| with the new
dependency node. We assume that |dep_final| points to the final node of list~|p|.

@c
static void mp_new_dep (MP mp, mp_node q, mp_variable_type newtype,
                        mp_value_node p) {
  mp_node r;    /* what used to be the first dependency */
  FUNCTION_TRACE4 ("mp_new_dep(%p,%d,%p)\n", q, newtype, p);
  mp_type (q) = newtype;
  set_dep_list (q, p);
  set_prev_dep (q, (mp_node) mp->dep_head);
  r = mp_link (mp->dep_head);
  set_mp_link (mp->dep_final, r);
  set_prev_dep (r, (mp_node) mp->dep_final);
  set_mp_link (mp->dep_head, q);
}


@ Here is one of the ways a dependency list gets started.
The |const_dependency| routine produces a list that has nothing but
a constant term.

@c
static mp_value_node mp_const_dependency (MP mp, mp_number v) {
  mp->dep_final = mp_get_dep_node (mp);
  set_dep_value (mp->dep_final, v);
  set_dep_info (mp->dep_final, NULL);
  FUNCTION_TRACE3 ("%p = mp_const_dependency(%d)\n", mp->dep_final, number_to_scaled (v));
  return mp->dep_final;
}


@ And here's a more interesting way to start a dependency list from scratch:
The parameter to |single_dependency| is the location of an
independent variable~|x|, and the result is the simple dependency list
`|x+0|'.

In the unlikely event that the given independent variable has been doubled so
often that we can't refer to it with a nonzero coefficient,
|single_dependency| returns the simple list `0'.  This case can be
recognized by testing that the returned list pointer is equal to
|dep_final|.

@d two_to_the(A) (1<<(unsigned)(A))

@c
static mp_value_node mp_single_dependency (MP mp, mp_node p) {
  mp_value_node q, rr;  /* the new dependency list */
  integer m;    /* the number of doublings */
  m = indep_scale (p);
  if (m > 28) {
    q = mp_const_dependency (mp, zero_t);
  } else {
    q = mp_get_dep_node (mp);
    set_dep_value (q, zero_t);
    set_number_from_scaled (dep_value (q), (integer) two_to_the (28 - m));
    set_dep_info (q, p);
    rr = mp_const_dependency (mp, zero_t);
    set_mp_link (q, (mp_node) rr);
  }
  FUNCTION_TRACE3 ("%p = mp_single_dependency(%p)\n", q, p);
  return q;
}


@ We sometimes need to make an exact copy of a dependency list.

@c
static mp_value_node mp_copy_dep_list (MP mp, mp_value_node p) {
  mp_value_node q;      /* the new dependency list */
  FUNCTION_TRACE2 ("mp_copy_dep_list(%p)\n", p);
  q = mp_get_dep_node (mp);
  mp->dep_final = q;
  while (1) {
    set_dep_info (mp->dep_final, dep_info (p));
    set_dep_value (mp->dep_final, dep_value (p));
    if (dep_info (mp->dep_final) == NULL)
      break;
    set_mp_link (mp->dep_final, (mp_node) mp_get_dep_node (mp));
    mp->dep_final = (mp_value_node) mp_link (mp->dep_final);
    p = (mp_value_node) mp_link (p);
  }
  return q;
}


@ But how do variables normally become known? Ah, now we get to the heart of the
equation-solving mechanism. The |linear_eq| procedure is given a |dependent|
or |mp_proto_dependent| list,~|p|, in which at least one independent variable
appears. It equates this list to zero, by choosing an independent variable
with the largest coefficient and making it dependent on the others. The
newly dependent variable is eliminated from all current dependencies,
thereby possibly making other dependent variables known.

The given list |p| is, of course, totally destroyed by all this processing.

@c
static mp_value_node find_node_with_largest_coefficient(MP mp, mp_value_node p, mp_number *v);
static void display_new_dependency (MP mp, mp_value_node p, mp_node x, integer n);
static void change_to_known (MP mp, mp_value_node p, mp_node x, mp_value_node final_node, integer n);
static mp_value_node divide_p_by_minusv_removing_q (MP mp, mp_value_node p, mp_value_node q, 
     				    mp_value_node *final_node, mp_number v, quarterword t);
static mp_value_node divide_p_by_2_n (MP mp, mp_value_node p, integer n);
static void mp_linear_eq (MP mp, mp_value_node p, quarterword t) {
  mp_value_node r;   /* for link manipulation */
  mp_node x;    /* the variable that loses its independence */
  integer n;    /* the number of times |x| had been halved */
  mp_number v;    /* the coefficient of |x| in list |p| */
  mp_value_node prev_r; /* lags one step behind |r| */
  mp_value_node final_node;     /* the constant term of the new dependency list */
  mp_value_node qq;
  new_number (v);
  FUNCTION_TRACE3 ("mp_linear_eq(%p,%d)\n", p, t);
  qq = find_node_with_largest_coefficient(mp, p, &v);
  x = dep_info (qq);
  n = indep_scale (x);
  p = divide_p_by_minusv_removing_q(mp, p, qq, &final_node, v, t);
  if (number_positive (internal_value (mp_tracing_equations))) {
    display_new_dependency(mp,p,(mp_node)x,n);
  }
  prev_r = (mp_value_node) mp->dep_head;
  r = (mp_value_node) mp_link (mp->dep_head);
  while (r != mp->dep_head) {
    mp_value_node s = (mp_value_node) dep_list (r);
    mp_value_node q = mp_p_with_x_becoming_q (mp, s, x, (mp_node) p, mp_type (r));
    if (dep_info (q) == NULL) {
      mp_make_known (mp, r, q);
    } else {
      set_dep_list (r, q);
      do {
        q = (mp_value_node) mp_link (q);
      } while (dep_info (q) != NULL);
      prev_r = q;
    }
    r = (mp_value_node) mp_link (prev_r);
  }
  if (n > 0) {
    p = divide_p_by_2_n(mp, p, n);  
  }
  change_to_known(mp,p,(mp_node)x,final_node,n);
  if (mp->fix_needed)
    mp_fix_dependencies (mp);
  free_number (v);
}


@ 
@c
static mp_value_node find_node_with_largest_coefficient(MP mp, mp_value_node p, mp_number *v) {
  mp_number vabs; /* its absolute value of v*/
  mp_number rabs; /* the absolute value of |dep_value(r)| */
  mp_value_node q = p;
  mp_value_node r = (mp_value_node) mp_link (p);
  new_number (vabs);
  new_number (rabs);
  number_clone (*v, dep_value (q));
  while (dep_info (r) != NULL) {
     number_clone (vabs, *v);
     number_abs (vabs);
     number_clone (rabs, dep_value (r));
     number_abs (rabs);
     if (number_greater (rabs, vabs)) {
       q = r;
       number_clone (*v, dep_value (r));
    }
    r = (mp_value_node) mp_link (r);
  }
  free_number (vabs);
  free_number (rabs);
  return q;
}


@ Here we want to change the coefficients from |scaled| to |fraction|,
except in the constant term. In the common case of a trivial equation
like `\.{x=3.14}', we will have |v=-fraction_one|, |q=p|, and |t=mp_dependent|.

@c
static mp_value_node divide_p_by_minusv_removing_q (MP mp, mp_value_node p, mp_value_node q, 
       		     				   mp_value_node *final_node, mp_number v, quarterword t) {
  mp_value_node r;   /* for link manipulation */
  mp_value_node s;
  s = (mp_value_node) mp->temp_head;
  set_mp_link (s, (mp_node) p);
  r = p;
  do {
    if (r == q) {
      set_mp_link (s, mp_link (r));
      mp_free_dep_node (mp, r);
    } else {
      mp_number w;    /* a tentative coefficient */
      mp_number absw;
      new_number (w);
      new_number (absw);
      make_fraction (w, dep_value (r), v);
      number_clone (absw, w);
      number_abs (absw);
      if (number_lessequal (absw, half_fraction_threshold_k)) {
        set_mp_link (s, mp_link (r));
        mp_free_dep_node (mp, r);
      } else {
        number_negate (w);
        set_dep_value (r, w);
        s = r; 
      }
      free_number(w);
      free_number (absw);
    }
    r = (mp_value_node) mp_link (s);
  } while (dep_info (r) != NULL);

  if (t == mp_proto_dependent) {
    mp_number ret;
    new_number (ret);
    make_scaled (ret, dep_value (r), v);
    number_negate (ret);
    set_dep_value (r, ret);
    free_number (ret);
  } else if (number_to_scaled (v) != -number_to_scaled (fraction_one_t)) {
    mp_number ret;
    new_fraction (ret);
    make_fraction (ret, dep_value (r), v);
    number_negate (ret);
    set_dep_value (r, ret);
    free_number (ret);
  }
  *final_node = r;
  return (mp_value_node) mp_link (mp->temp_head);
}
 

@ 
@c
static void display_new_dependency (MP mp, mp_value_node p, mp_node x, integer n) {
  if (mp_interesting (mp, x)) {
    int w0;
    mp_begin_diagnostic (mp);
    mp_print_nl (mp, "## ");
    mp_print_variable_name (mp, x);
    w0 = n;
    while (w0 > 0) {
      mp_print (mp, "*4");
      w0 = w0 - 2;
    }
    mp_print_char (mp, xord ('='));
    mp_print_dependency (mp, p, mp_dependent);
    mp_end_diagnostic (mp, false);
  }
}

@ The |n > 0| test is repeated here because it is of vital importance to the
function's functioning.

@c
static mp_value_node divide_p_by_2_n (MP mp, mp_value_node p, integer n) {
  mp_value_node pp = NULL;
  if (n > 0) {
    /* Divide list |p| by $2^n$ */
    mp_value_node r;
    mp_value_node s;
    mp_number absw;
    mp_number w;    /* a tentative coefficient */
    new_number (w);
    new_number (absw);
    s = (mp_value_node) mp->temp_head;
    set_mp_link (mp->temp_head, (mp_node) p);
    r = p;  
    do {
      if (n > 30) {
        set_number_to_zero (w);
      } else {
        number_clone (w, dep_value (r));
        number_divide_int (w, two_to_the (n));
      }
      number_clone (absw, w);
      number_abs (absw);
      if (number_lessequal(absw, half_fraction_threshold_k) && (dep_info (r) != NULL)) {
        set_mp_link (s, mp_link (r));
        mp_free_dep_node (mp, r);
      } else {
        set_dep_value (r, w);
        s = r;
      }
      r = (mp_value_node) mp_link (s);
    } while (dep_info (s) != NULL);
    pp = (mp_value_node) mp_link (mp->temp_head);
    free_number (absw);
    free_number (w);
  }
  return pp;
}

@
@c
static void change_to_known (MP mp, mp_value_node p, mp_node x, mp_value_node final_node, integer n) {
  if (dep_info (p) == NULL) {
    mp_number absx;
    new_number (absx);
    mp_type (x) = mp_known;
    set_value_number (x, dep_value (p));
    number_clone (absx, value_number (x));
    number_abs (absx);
    if (number_greaterequal (absx, warning_limit_t))
      mp_val_too_big (mp, value_number (x));
    free_number (absx);
    mp_free_dep_node (mp, p);
    if (cur_exp_node () == x && mp->cur_exp.type == mp_independent) {
      set_cur_exp_value_number (value_number (x));
      mp->cur_exp.type = mp_known;
      mp_free_value_node (mp, x);
    }
  } else {
    mp->dep_final = final_node;
    mp_new_dep (mp, x, mp_dependent, p);
    if (cur_exp_node () == x && mp->cur_exp.type == mp_independent) {
      mp->cur_exp.type = mp_dependent;
    }
  }
}

@* Dynamic nonlinear equations.
Variables of numeric type are maintained by the general scheme of
independent, dependent, and known values that we have just studied;
and the components of pair and transform variables are handled in the
same way. But \MP\ also has five other types of values: \&{boolean},
\&{string}, \&{pen}, \&{path}, and \&{picture}; what about them?

Equations are allowed between nonlinear quantities, but only in a
simple form. Two variables that haven't yet been assigned values are
either equal to each other, or they're not.

Before a boolean variable has received a value, its type is |mp_unknown_boolean|;
similarly, there are variables whose type is |mp_unknown_string|, |mp_unknown_pen|,
|mp_unknown_path|, and |mp_unknown_picture|. In such cases the value is either
|NULL| (which means that no other variables are equivalent to this one), or
it points to another variable of the same undefined type. The pointers in the
latter case form a cycle of nodes, which we shall call a ``ring.''
Rings of undefined variables may include capsules, which arise as
intermediate results within expressions or as \&{expr} parameters to macros.

When one member of a ring receives a value, the same value is given to
all the other members. In the case of paths and pictures, this implies
making separate copies of a potentially large data structure; users should
restrain their enthusiasm for such generality, unless they have lots and
lots of memory space.

@ The following procedure is called when a capsule node is being
added to a ring (e.g., when an unknown variable is mentioned in an expression).

@c
static mp_node mp_new_ring_entry (MP mp, mp_node p) {
  mp_node q;    /* the new capsule node */
  q = mp_get_value_node (mp);
  mp_name_type (q) = mp_capsule;
  mp_type (q) = mp_type (p);
  if (value_node (p) == NULL)
    set_value_node (q, p);
  else
    set_value_node (q, value_node (p));
  set_value_node (p, q);
  return q;
}


@ Conversely, we might delete a capsule or a variable before it becomes known.
The following procedure simply detaches a quantity from its ring,
without recycling the storage.

@<Declarations@>=
static void mp_ring_delete (MP mp, mp_node p);

@ @c
void mp_ring_delete (MP mp, mp_node p) {
  mp_node q;
  (void) mp;
  q = value_node (p);
  if (q != NULL && q != p) {
    while (value_node (q) != p)
      q = value_node (q);
    set_value_node (q, value_node (p));
  }
}


@ Eventually there might be an equation that assigns values to all of the
variables in a ring. The |nonlinear_eq| subroutine does the necessary
propagation of values.

If the parameter |flush_p| is |true|, node |p| itself needn't receive a
value, it will soon be recycled.

@c
static void mp_nonlinear_eq (MP mp, mp_value v, mp_node p, boolean flush_p) {
  mp_variable_type t;   /* the type of ring |p| */
  mp_node q, r; /* link manipulation registers */
  t = (mp_type (p) - unknown_tag);
  q = value_node (p);
  if (flush_p)
    mp_type (p) = mp_vacuous;
  else
    p = q;
  do {
    r = value_node (q);
    mp_type (q) = t;
    switch (t) {
    case mp_boolean_type:
      set_value_number (q, v.data.n);
      break;
    case mp_string_type:
      set_value_str (q, v.data.str);
      add_str_ref (v.data.str);
      break;
    case mp_pen_type:
      set_value_knot (q, copy_pen (v.data.p));
      break;
    case mp_path_type:
      set_value_knot (q, mp_copy_path (mp, v.data.p));
      break;
    case mp_picture_type:
      set_value_node (q, v.data.node);
      add_edge_ref (v.data.node);
      break;
    default:
      break;
    }                           /* there ain't no more cases */
    q = r;
  } while (q != p);
}


@ If two members of rings are equated, and if they have the same type,
the |ring_merge| procedure is called on to make them equivalent.

@c
static void mp_ring_merge (MP mp, mp_node p, mp_node q) {
  mp_node r;    /* traverses one list */
  r = value_node (p);
  while (r != p) {
    if (r == q) {
      exclaim_redundant_equation(mp);
      return;
    };
    r = value_node (r);
  }
  r = value_node (p);
  set_value_node (p, value_node (q));
  set_value_node (q, r);
}


@ @c
static void exclaim_redundant_equation (MP mp) {
  const char *hlp[] = {
         "I already knew that this equation was true.",
         "But perhaps no harm has been done; let's continue.",
         NULL };
  mp_back_error (mp, "Redundant equation", hlp, true);
  mp_get_x_next (mp);
}

@ @<Declarations@>=
static void exclaim_redundant_equation (MP mp);

@* Introduction to the syntactic routines.
Let's pause a moment now and try to look at the Big Picture.
The \MP\ program consists of three main parts: syntactic routines,
semantic routines, and output routines. The chief purpose of the
syntactic routines is to deliver the user's input to the semantic routines,
while parsing expressions and locating operators and operands. The
semantic routines act as an interpreter responding to these operators,
which may be regarded as commands. And the output routines are
periodically called on to produce compact font descriptions that can be
used for typesetting or for making interim proof drawings. We have
discussed the basic data structures and many of the details of semantic
operations, so we are good and ready to plunge into the part of \MP\ that
actually controls the activities.

Our current goal is to come to grips with the |get_next| procedure,
which is the keystone of \MP's input mechanism. Each call of |get_next|
sets the value of three variables |cur_cmd|, |cur_mod|, and |cur_sym|,
representing the next input token.
$$\vbox{\halign{#\hfil\cr
  \hbox{|cur_cmd| denotes a command code from the long list of codes
   given earlier;}\cr
  \hbox{|cur_mod| denotes a modifier or operand of the command code;}\cr
  \hbox{|cur_sym| is the hash address of the symbolic token that was
   just scanned,}\cr
  \hbox{\qquad or zero in the case of a numeric or string
   or capsule token.}\cr}}$$
Underlying this external behavior of |get_next| is all the machinery
necessary to convert from character files to tokens. At a given time we
may be only partially finished with the reading of several files (for
which \&{input} was specified), and partially finished with the expansion
of some user-defined macros and/or some macro parameters, and partially
finished reading some text that the user has inserted online,
and so on. When reading a character file, the characters must be
converted to tokens; comments and blank spaces must
be removed, numeric and string tokens must be evaluated.

To handle these situations, which might all be present simultaneously,
\MP\ uses various stacks that hold information about the incomplete
activities, and there is a finite state control for each level of the
input mechanism. These stacks record the current state of an implicitly
recursive process, but the |get_next| procedure is not recursive.

@d cur_cmd() (unsigned)(mp->cur_mod_->type)
@d set_cur_cmd(A) mp->cur_mod_->type=(A)
@d cur_mod_int() number_to_int (mp->cur_mod_->data.n) /* operand of current command */
@d cur_mod() number_to_scaled (mp->cur_mod_->data.n) /* operand of current command */
@d cur_mod_number() mp->cur_mod_->data.n /* operand of current command */
@d set_cur_mod(A) set_number_from_scaled (mp->cur_mod_->data.n, (A))
@d set_cur_mod_number(A) number_clone (mp->cur_mod_->data.n, (A))
@d cur_mod_node() mp->cur_mod_->data.node
@d set_cur_mod_node(A) mp->cur_mod_->data.node=(A)
@d cur_mod_str() mp->cur_mod_->data.str
@d set_cur_mod_str(A) mp->cur_mod_->data.str=(A)
@d cur_sym() mp->cur_mod_->data.sym
@d set_cur_sym(A) mp->cur_mod_->data.sym=(A)
@d cur_sym_mod() mp->cur_mod_->name_type
@d set_cur_sym_mod(A) mp->cur_mod_->name_type=(A)

@<Glob...@>=
mp_node cur_mod_;         /* current command, symbol, and its operands */

@ @<Initialize table...@>=
mp->cur_mod_ = mp_get_symbolic_node(mp);

@ @<Free table...@>=
mp_free_symbolic_node(mp, mp->cur_mod_);

@ The |print_cmd_mod| routine prints a symbolic interpretation of a
command code and its modifier.
It consists of a rather tedious sequence of print
commands, and most of it is essentially an inverse to the |primitive|
routine that enters a \MP\ primitive into |hash| and |eqtb|. Therefore almost
all of this procedure appears elsewhere in the program, together with the
corresponding |primitive| calls.

@<Declarations@>=
static void mp_print_cmd_mod (MP mp, integer c, integer m);

@ @c
void mp_print_cmd_mod (MP mp, integer c, integer m) {
  switch (c) {
    @<Cases of |print_cmd_mod| for symbolic printing of primitives@>
  default:
    mp_print (mp, "[unknown command code!]");
    break;
  }
}


@ Here is a procedure that displays a given command in braces, in the
user's transcript file.

@d show_cur_cmd_mod mp_show_cmd_mod(mp, cur_cmd(),cur_mod())

@c
static void mp_show_cmd_mod (MP mp, integer c, integer m) {
  mp_begin_diagnostic (mp);
  mp_print_nl (mp, "{");
  mp_print_cmd_mod (mp, c, m);
  mp_print_char (mp, xord ('}'));
  mp_end_diagnostic (mp, false);
}


@* Input stacks and states.
The state of \MP's input mechanism appears in the input stack, whose
entries are records with five fields, called |index|, |start|, |loc|,
|limit|, and |name|. The top element of this stack is maintained in a
global variable for which no subscripting needs to be done; the other
elements of the stack appear in an array. Hence the stack is declared thus:

@<Types...@>=
typedef struct {
  char *long_name_field;
  halfword start_field, loc_field, limit_field;
  mp_node nstart_field, nloc_field;
  mp_string name_field;
  quarterword index_field;
} in_state_record;

@ @<Glob...@>=
in_state_record *input_stack;
integer input_ptr;      /* first unused location of |input_stack| */
integer max_in_stack;   /* largest value of |input_ptr| when pushing */
in_state_record cur_input;      /* the ``top'' input state */
int stack_size; /* maximum number of simultaneous input sources */

@ @<Allocate or initialize ...@>=
mp->stack_size = 16;
mp->input_stack = xmalloc ((mp->stack_size + 1), sizeof (in_state_record));

@ @<Dealloc variables@>=
xfree (mp->input_stack);

@ We've already defined the special variable |loc==cur_input.loc_field|
in our discussion of basic input-output routines. The other components of
|cur_input| are defined in the same way:

@d iindex mp->cur_input.index_field /* reference for buffer information */
@d start mp->cur_input.start_field /* starting position in |buffer| */
@d limit mp->cur_input.limit_field /* end of current line in |buffer| */
@d name mp->cur_input.name_field /* name of the current file */

@ Let's look more closely now at the five control variables
(|index|,~|start|,~|loc|,~|limit|,~|name|),
assuming that \MP\ is reading a line of characters that have been input
from some file or from the user's terminal. There is an array called
|buffer| that acts as a stack of all lines of characters that are
currently being read from files, including all lines on subsidiary
levels of the input stack that are not yet completed. \MP\ will return to
the other lines when it is finished with the present input file.

(Incidentally, on a machine with byte-oriented addressing, it would be
appropriate to combine |buffer| with the |str_pool| array,
letting the buffer entries grow downward from the top of the string pool
and checking that these two tables don't bump into each other.)

The line we are currently working on begins in position |start| of the
buffer; the next character we are about to read is |buffer[loc]|; and
|limit| is the location of the last character present. We always have
|loc<=limit|. For convenience, |buffer[limit]| has been set to |"%"|, so
that the end of a line is easily sensed.

The |name| variable is a string number that designates the name of
the current file, if we are reading an ordinary text file.  Special codes
|is_term..max_spec_src| indicate other sources of input text.

@d is_term (mp_string)0 /* |name| value when reading from the terminal for normal input */
@d is_read (mp_string)1 /* |name| value when executing a \&{readstring} or \&{readfrom} */
@d is_scantok (mp_string)2 /* |name| value when reading text generated by \&{scantokens} */
@d max_spec_src is_scantok

@ Additional information about the current line is available via the
|index| variable, which counts how many lines of characters are present
in the buffer below the current level. We have |index=0| when reading
from the terminal and prompting the user for each line; then if the user types,
e.g., `\.{input figs}', we will have |index=1| while reading
the file \.{figs.mp}. However, it does not follow that |index| is the
same as the input stack pointer, since many of the levels on the input
stack may come from token lists and some |index| values may correspond
to \.{MPX} files that are not currently on the stack.

The global variable |in_open| is equal to the highest |index| value counting
\.{MPX} files but excluding token-list input levels.  Thus, the number of
partially read lines in the buffer is |in_open+1| and we have |in_open>=index|
when we are not reading a token list.

If we are not currently reading from the terminal,
we are reading from the file variable |input_file[index]|. We use
the notation |terminal_input| as a convenient abbreviation for |name=is_term|,
and |cur_file| as an abbreviation for |input_file[index]|.

When \MP\ is not reading from the terminal, the global variable |line| contains
the line number in the current file, for use in error messages. More precisely,
|line| is a macro for |line_stack[index]| and the |line_stack| array gives
the line number for each file in the |input_file| array.

When an \.{MPX} file is opened the file name is stored in the |mpx_name|
array so that the name doesn't get lost when the file is temporarily removed
from the input stack.
Thus when |input_file[k]| is an \.{MPX} file, its name is |mpx_name[k]|
and it contains translated \TeX\ pictures for |input_file[k-1]|.
Since this is not an \.{MPX} file, we have
$$ \hbox{|mpx_name[k-1]<=absent|}. $$
This |name| field is set to |finished| when |input_file[k]| is completely
read.

If more information about the input state is needed, it can be
included in small arrays like those shown here. For example,
the current page or segment number in the input file might be put
into a variable |page|, that is really a macro for the current entry
in `\ignorespaces|page_stack:array[0..max_in_open] of integer|\unskip'
by analogy with |line_stack|.
@^system dependencies@>

@d terminal_input (name==is_term) /* are we reading from the terminal? */
@d cur_file mp->input_file[iindex] /* the current |void *| variable */
@d line mp->line_stack[iindex] /* current line number in the current source file */
@d in_ext mp->inext_stack[iindex] /* a string used to construct \.{MPX} file names */
@d in_name mp->iname_stack[iindex] /* a string used to construct \.{MPX} file names */
@d in_area mp->iarea_stack[iindex] /* another string for naming \.{MPX} files */
@d absent (mp_string)1 /* |name_field| value for unused |mpx_in_stack| entries */
@d mpx_reading (mp->mpx_name[iindex]>absent)
  /* when reading a file, is it an \.{MPX} file? */
@d mpx_finished 0
  /* |name_field| value when the corresponding \.{MPX} file is finished */

@<Glob...@>=
integer in_open;        /* the number of lines in the buffer, less one */
integer in_open_max;    /* highest value of |in_open| ever seen */
unsigned int open_parens;       /* the number of open text files */
void **input_file;
integer *line_stack;    /* the line number for each file */
char **inext_stack;     /* used for naming \.{MPX} files */
char **iname_stack;     /* used for naming \.{MPX} files */
char **iarea_stack;     /* used for naming \.{MPX} files */
mp_string *mpx_name;

@ @<Declarations@>=
static void mp_reallocate_input_stack (MP mp, int newsize);

@ @c
static void mp_reallocate_input_stack (MP mp, int newsize) {
  int k;
  int n = newsize +1;
  XREALLOC (mp->input_file, n, void *);
  XREALLOC (mp->line_stack, n, integer);
  XREALLOC (mp->inext_stack, n, char *);
  XREALLOC (mp->iname_stack, n, char *);
  XREALLOC (mp->iarea_stack, n, char *);
  XREALLOC (mp->mpx_name, n, mp_string);
  for (k = mp->max_in_open; k <= n; k++) {
    mp->input_file[k] = NULL;
    mp->line_stack[k] = 0;
    mp->inext_stack[k] = NULL;
    mp->iname_stack[k] = NULL;
    mp->iarea_stack[k] = NULL;
    mp->mpx_name[k] = NULL;
  }
  mp->max_in_open = newsize;
}


@ This has to be more than |file_bottom|, so:
@<Allocate or ...@>=
mp_reallocate_input_stack (mp, file_bottom+4);

@ @<Dealloc variables@>=
{
  int l;
  for (l = 0; l <= mp->max_in_open; l++) {
    xfree (mp->inext_stack[l]);
    xfree (mp->iname_stack[l]);
    xfree (mp->iarea_stack[l]);
  }
}
xfree (mp->input_file);
xfree (mp->line_stack);
xfree (mp->inext_stack);
xfree (mp->iname_stack);
xfree (mp->iarea_stack);
xfree (mp->mpx_name);


@ However, all this discussion about input state really applies only to the
case that we are inputting from a file. There is another important case,
namely when we are currently getting input from a token list. In this case
|iindex>max_in_open|, and the conventions about the other state variables
are different:

\yskip\hang|nloc| is a pointer to the current node in the token list, i.e.,
the node that will be read next. If |nloc=NULL|, the token list has been
fully read.

\yskip\hang|start| points to the first node of the token list; this node
may or may not contain a reference count, depending on the type of token
list involved.

\yskip\hang|token_type|, which takes the place of |iindex| in the
discussion above, is a code number that explains what kind of token list
is being scanned.

\yskip\hang|name| points to the |eqtb| address of the control sequence
being expanded, if the current token list is a macro not defined by
\&{vardef}. Macros defined by \&{vardef} have |name=NULL|; their name
can be deduced by looking at their first two parameters.

\yskip\hang|param_start|, which takes the place of |limit|, tells where
the parameters of the current macro or loop text begin in the |param_stack|.

\yskip\noindent The |token_type| can take several values, depending on
where the current token list came from:

\yskip
\indent|forever_text|, if the token list being scanned is the body of
a \&{forever} loop;

\indent|loop_text|, if the token list being scanned is the body of
a \&{for} or \&{forsuffixes} loop;

\indent|parameter|, if a \&{text} or \&{suffix} parameter is being scanned;

\indent|backed_up|, if the token list being scanned has been inserted as
`to be read again'.

\indent|inserted|, if the token list being scanned has been inserted as
part of error recovery;

\indent|macro|, if the expansion of a user-defined symbolic token is being
scanned.

\yskip\noindent
The token list begins with a reference count if and only if |token_type=
macro|.
@^reference counts@>

@d nloc mp->cur_input.nloc_field /* location of next node node */
@d nstart mp->cur_input.nstart_field /* location of next node node */

@d token_type iindex /* type of current token list */
@d token_state (iindex<=macro) /* are we scanning a token list? */
@d file_state (iindex>macro) /* are we scanning a file line? */
@d param_start limit /* base of macro parameters in |param_stack| */
@d forever_text 0 /* |token_type| code for loop texts */
@d loop_text 1 /* |token_type| code for loop texts */
@d parameter 2 /* |token_type| code for parameter texts */
@d backed_up 3 /* |token_type| code for texts to be reread */
@d inserted 4 /* |token_type| code for inserted texts */
@d macro 5 /* |token_type| code for macro replacement texts */
@d file_bottom 6 /* lowest file code */

@ The |param_stack| is an auxiliary array used to hold pointers to the token
lists for parameters at the current level and subsidiary levels of input.
This stack grows at a different rate from the others, and is dynamically reallocated
when needed.

@<Glob...@>=
mp_node *param_stack;   /* token list pointers for parameters */
integer param_ptr;      /* first unused entry in |param_stack| */
integer max_param_stack;        /* largest value of |param_ptr| */

@ @<Allocate or initialize ...@>=
mp->param_stack = xmalloc ((mp->param_size + 1), sizeof (mp_node));

@ @c
static void mp_check_param_size (MP mp, int k) {
  while (k >= mp->param_size) {
    XREALLOC (mp->param_stack, (k + k / 4), mp_node);
    mp->param_size = k + k / 4;
  }
}


@ @<Dealloc variables@>=
xfree (mp->param_stack);

@ Notice that the |line| isn't valid when |token_state| is true because it
depends on |iindex|.  If we really need to know the line number for the
topmost file in the iindex stack we use the following function.  If a page
number or other information is needed, this routine should be modified to
compute it as well.
@^system dependencies@>

@<Declarations@>=
static integer mp_true_line (MP mp);

@ @c
integer mp_true_line (MP mp) {
  int k;        /* an index into the input stack */
  if (file_state && (name > max_spec_src)) {
    return line;
  } else {
    k = mp->input_ptr;
    while ((k > 0) &&
           ((mp->input_stack[(k - 1)].index_field < file_bottom) ||
            (mp->input_stack[(k - 1)].name_field <= max_spec_src))) {
      decr (k);
    }
    return (k > 0 ? mp->line_stack[(k - 1) + file_bottom] : 0);
  }
}


@ Thus, the ``current input state'' can be very complicated indeed; there
can be many levels and each level can arise in a variety of ways. The
|show_context| procedure, which is used by \MP's error-reporting routine to
print out the current input state on all levels down to the most recent
line of characters from an input file, illustrates most of these conventions.
The global variable |file_ptr| contains the lowest level that was
displayed by this procedure.

@<Glob...@>=
integer file_ptr;       /* shallowest level shown by |show_context| */

@ The status at each level is indicated by printing two lines, where the first
line indicates what was read so far and the second line shows what remains
to be read. The context is cropped, if necessary, so that the first line
contains at most |half_error_line| characters, and the second contains
at most |error_line|. Non-current input levels whose |token_type| is
`|backed_up|' are shown only if they have not been fully read.

@c
void mp_show_context (MP mp) {                               /* prints where the scanner is */
  unsigned old_setting; /* saved |selector| setting */
  @<Local variables for formatting calculations@>;
  mp->file_ptr = mp->input_ptr;
  mp->input_stack[mp->file_ptr] = mp->cur_input;
  /* store current state */
  while (1) {
    mp->cur_input = mp->input_stack[mp->file_ptr];      /* enter into the context */
    @<Display the current context@>;
    if (file_state)
      if ((name > max_spec_src) || (mp->file_ptr == 0))
        break;
    decr (mp->file_ptr);
  }
  mp->cur_input = mp->input_stack[mp->input_ptr];       /* restore original state */
}


@ @<Display the current context@>=
if ((mp->file_ptr == mp->input_ptr) || file_state ||
    (token_type != backed_up) || (nloc != NULL)) {
  /* we omit backed-up token lists that have already been read */
  mp->tally = 0;                /* get ready to count characters */
  old_setting = mp->selector;
  if (file_state) {
    @<Print location of current line@>;
    @<Pseudoprint the line@>;
  } else {
    @<Print type of token list@>;
    @<Pseudoprint the token list@>;
  }
  mp->selector = old_setting;   /* stop pseudoprinting */
  @<Print two lines using the tricky pseudoprinted information@>;
}

@ This routine should be changed, if necessary, to give the best possible
indication of where the current line resides in the input file.
For example, on some systems it is best to print both a page and line number.
@^system dependencies@>

@<Print location of current line@>=
if (name > max_spec_src) {
  mp_print_nl (mp, "l.");
  mp_print_int (mp, mp_true_line (mp));
} else if (terminal_input) {
  if (mp->file_ptr == 0)
    mp_print_nl (mp, "<*>");
  else
    mp_print_nl (mp, "<insert>");
} else if (name == is_scantok) {
  mp_print_nl (mp, "<scantokens>");
} else {
  mp_print_nl (mp, "<read>");
}
mp_print_char (mp, xord (' '))
 

@ Can't use case statement here because the |token_type| is not
a constant expression.

@<Print type of token list@>=
{
  if (token_type == forever_text) {
    mp_print_nl (mp, "<forever> ");
  } else if (token_type == loop_text) {
    @<Print the current loop value@>;
  } else if (token_type == parameter) {
    mp_print_nl (mp, "<argument> ");
  } else if (token_type == backed_up) {
    if (nloc == NULL)
      mp_print_nl (mp, "<recently read> ");
    else
      mp_print_nl (mp, "<to be read again> ");
  } else if (token_type == inserted) {
    mp_print_nl (mp, "<inserted text> ");
  } else if (token_type == macro) {
    mp_print_ln (mp);
    if (name != NULL)
      mp_print_str (mp, name);
    else
      @<Print the name of a \&{vardef}'d macro@>;
    mp_print (mp, "->");
  } else {
    mp_print_nl (mp, "?");      /* this should never happen */
@.?\relax@>
  }
}


@ The parameter that corresponds to a loop text is either a token list
(in the case of \&{forsuffixes}) or a ``capsule'' (in the case of \&{for}).
We'll discuss capsules later; for now, all we need to know is that
the |link| field in a capsule parameter is |void| and that
|print_exp(p,0)| displays the value of capsule~|p| in abbreviated form.

@<Print the current loop value@>=
{
  mp_node pp;
  mp_print_nl (mp, "<for(");
  pp = mp->param_stack[param_start];
  if (pp != NULL) {
    if (mp_link (pp) == MP_VOID)
      mp_print_exp (mp, pp, 0); /* we're in a \&{for} loop */
    else
      mp_show_token_list (mp, pp, NULL, 20, mp->tally);
  }
  mp_print (mp, ")> ");
}


@ The first two parameters of a macro defined by \&{vardef} will be token
lists representing the macro's prefix and ``at point.'' By putting these
together, we get the macro's full name.

@<Print the name of a \&{vardef}'d macro@>=
{
  mp_node pp = mp->param_stack[param_start];
  if (pp == NULL) {
    mp_show_token_list (mp, mp->param_stack[param_start + 1], NULL, 20,
                        mp->tally);
  } else {
    mp_node qq = pp;
    while (mp_link (qq) != NULL)
      qq = mp_link (qq);
    mp_link (qq) = mp->param_stack[param_start + 1];
    mp_show_token_list (mp, pp, NULL, 20, mp->tally);
    mp_link (qq) = NULL;
  }
}


@ Now it is necessary to explain a little trick. We don't want to store a long
string that corresponds to a token list, because that string might take up
lots of memory; and we are printing during a time when an error message is
being given, so we dare not do anything that might overflow one of \MP's
tables. So `pseudoprinting' is the answer: We enter a mode of printing
that stores characters into a buffer of length |error_line|, where character
$k+1$ is placed into \hbox{|trick_buf[k mod error_line]|} if
|k<trick_count|, otherwise character |k| is dropped. Initially we set
|tally:=0| and |trick_count:=1000000|; then when we reach the
point where transition from line 1 to line 2 should occur, we
set |first_count:=tally| and |trick_count:=@tmax@>(error_line,
tally+1+error_line-half_error_line)|. At the end of the
pseudoprinting, the values of |first_count|, |tally|, and
|trick_count| give us all the information we need to print the two lines,
and all of the necessary text is in |trick_buf|.

Namely, let |l| be the length of the descriptive information that appears
on the first line. The length of the context information gathered for that
line is |k=first_count|, and the length of the context information
gathered for line~2 is $m=\min(|tally|, |trick_count|)-k$. If |l+k<=h|,
where |h=half_error_line|, we print |trick_buf[0..k-1]| after the
descriptive information on line~1, and set |n:=l+k|; here |n| is the
length of line~1. If $l+k>h$, some cropping is necessary, so we set |n:=h|
and print `\.{...}' followed by
$$\hbox{|trick_buf[(l+k-h+3)..k-1]|,}$$
where subscripts of |trick_buf| are circular modulo |error_line|. The
second line consists of |n|~spaces followed by |trick_buf[k..(k+m-1)]|,
unless |n+m>error_line|; in the latter case, further cropping is done.
This is easier to program than to explain.

@<Local variables for formatting...@>=
int i;  /* index into |buffer| */
integer l;      /* length of descriptive information on line 1 */
integer m;      /* context information gathered for line 2 */
int n;  /* length of line 1 */
integer p;      /* starting or ending place in |trick_buf| */
integer q;      /* temporary index */

@ The following code tells the print routines to gather
the desired information.

@d begin_pseudoprint { 
  l=mp->tally; mp->tally=0; mp->selector=pseudo;
  mp->trick_count=1000000;
}
@d set_trick_count() {
  mp->first_count=mp->tally;
  mp->trick_count=mp->tally+1+mp->error_line-mp->half_error_line;
  if ( mp->trick_count<mp->error_line ) mp->trick_count=mp->error_line;
}

@ And the following code uses the information after it has been gathered.

@<Print two lines using the tricky pseudoprinted information@>=
if (mp->trick_count == 1000000)
  set_trick_count();
  /* |set_trick_count| must be performed */
if (mp->tally < mp->trick_count)
  m = mp->tally - mp->first_count;
else
  m = mp->trick_count - mp->first_count;        /* context on line 2 */
if (l + mp->first_count <= mp->half_error_line) {
  p = 0;
  n = l + mp->first_count;
} else {
  mp_print (mp, "...");
  p = l + mp->first_count - mp->half_error_line + 3;
  n = mp->half_error_line;
}
for (q = p; q <= mp->first_count - 1; q++) {
  mp_print_char (mp, mp->trick_buf[q % mp->error_line]);
}
mp_print_ln (mp);
for (q = 1; q <= n; q++) {
  mp_print_char (mp, xord (' '));       /* print |n| spaces to begin line~2 */
}
if (m + n <= mp->error_line)
  p = mp->first_count + m;
else
  p = mp->first_count + (mp->error_line - n - 3);
for (q = mp->first_count; q <= p - 1; q++) {
  mp_print_char (mp, mp->trick_buf[q % mp->error_line]);
}
if (m + n > mp->error_line)
  mp_print (mp, "...")
   

@ But the trick is distracting us from our current goal, which is to
understand the input state. So let's concentrate on the data structures that
are being pseudoprinted as we finish up the |show_context| procedure.

@<Pseudoprint the line@>=
begin_pseudoprint;
if (limit > 0) {
  for (i = start; i <= limit - 1; i++) {
    if (i == loc)
      set_trick_count();
    mp_print_char (mp, mp->buffer[i]);
  }
}

@ @<Pseudoprint the token list@>=
begin_pseudoprint;
if (token_type != macro)
  mp_show_token_list (mp, nstart, nloc, 100000, 0);
else
  mp_show_macro (mp, nstart, nloc, 100000)
   

@* Maintaining the input stacks.
The following subroutines change the input status in commonly needed ways.

First comes |push_input|, which stores the current state and creates a
new level (having, initially, the same properties as the old).

@d push_input  { /* enter a new input level, save the old */
  if ( mp->input_ptr>mp->max_in_stack ) {
    mp->max_in_stack=mp->input_ptr;
    if ( mp->input_ptr==mp->stack_size ) {
      int l = (mp->stack_size+(mp->stack_size/4));
      XREALLOC(mp->input_stack, l, in_state_record);
      mp->stack_size = l;
    }         
  }
  mp->input_stack[mp->input_ptr]=mp->cur_input; /* stack the record */
  incr(mp->input_ptr);
}

@ And of course what goes up must come down.

@d pop_input { /* leave an input level, re-enter the old */
    decr(mp->input_ptr); mp->cur_input=mp->input_stack[mp->input_ptr];
  }

@ Here is a procedure that starts a new level of token-list input, given
a token list |p| and its type |t|. If |t=macro|, the calling routine should
set |name|, reset~|loc|, and increase the macro's reference count.

@d back_list(A) mp_begin_token_list(mp, (A), (quarterword)backed_up) /* backs up a simple token list */

@c
static void mp_begin_token_list (MP mp, mp_node p, quarterword t) {
  push_input;
  nstart = p;
  token_type = t;
  param_start = mp->param_ptr;
  nloc = p;
}


@ When a token list has been fully scanned, the following computations
should be done as we leave that level of input.
@^inner loop@>

@c
static void mp_end_token_list (MP mp) {                               /* leave a token-list input level */
  mp_node p;    /* temporary register */
  if (token_type >= backed_up) {        /* token list to be deleted */
    if (token_type <= inserted) {
      mp_flush_token_list (mp, nstart);
      goto DONE;
    } else {
      mp_delete_mac_ref (mp, nstart);   /* update reference count */
    }
  }
  while (mp->param_ptr > param_start) { /* parameters must be flushed */
    decr (mp->param_ptr);
    p = mp->param_stack[mp->param_ptr];
    if (p != NULL) {
      if (mp_link (p) == MP_VOID) {        /* it's an \&{expr} parameter */
        mp_recycle_value (mp, p);
        mp_free_value_node (mp, p);
      } else {
        mp_flush_token_list (mp, p);    /* it's a \&{suffix} or \&{text} parameter */
      }
    }
  }
DONE:
  pop_input;
  check_interrupt;
}


@ The contents of |cur_cmd,cur_mod,cur_sym| are placed into an equivalent
token by the |cur_tok| routine.
@^inner loop@>

@c
@<Declare the procedure called |make_exp_copy|@>;
static mp_node mp_cur_tok (MP mp) {
  mp_node p;    /* a new token node */
  if (cur_sym() == NULL && cur_sym_mod() == 0) {
    if (cur_cmd() == mp_capsule_token) {
      mp_number save_exp_num; /* possible |cur_exp| numerical to be restored */
      mp_value save_exp = mp->cur_exp;  /* |cur_exp| to be restored */
      new_number (save_exp_num);
      number_clone (save_exp_num, cur_exp_value_number());
      mp_make_exp_copy (mp, cur_mod_node());
      p = mp_stash_cur_exp (mp);
      mp_link (p) = NULL;
      mp->cur_exp = save_exp;
      number_clone (mp->cur_exp.data.n, save_exp_num);
      free_number (save_exp_num);
    } else {
      p = mp_get_token_node (mp);
      mp_name_type (p) = mp_token;
      if (cur_cmd() == mp_numeric_token) {
        set_value_number (p, cur_mod_number());
        mp_type (p) = mp_known;
      } else {
        set_value_str (p, cur_mod_str());
        mp_type (p) = mp_string_type;
      }
    }
  } else {
    p = mp_get_symbolic_node (mp);
    set_mp_sym_sym (p, cur_sym());
    mp_name_type (p) = cur_sym_mod();
  }
  return p;
}


@ Sometimes \MP\ has read too far and wants to ``unscan'' what it has
seen. The |back_input| procedure takes care of this by putting the token
just scanned back into the input stream, ready to be read again.
If |cur_sym<>0|, the values of |cur_cmd| and |cur_mod| are irrelevant.

@<Declarations@>=
static void mp_back_input (MP mp);

@ @c
void mp_back_input (MP mp) {                               /* undoes one token of input */
  mp_node p;    /* a token list of length one */
  p = mp_cur_tok (mp);
  while (token_state && (nloc == NULL))
    mp_end_token_list (mp);     /* conserve stack space */
  back_list (p);
}


@ The |back_error| routine is used when we want to restore or replace an
offending token just before issuing an error message.  We disable interrupts
during the call of |back_input| so that the help message won't be lost.

@<Declarations@>=
static void mp_back_error (MP mp, const char *msg, const char **hlp, boolean deletions_allowed) ;

@ @c
static void mp_back_error (MP mp, const char *msg, const char **hlp, boolean deletions_allowed) { 
  /* back up one token and call |error| */
  mp->OK_to_interrupt = false;
  mp_back_input (mp);
  mp->OK_to_interrupt = true;
  mp_error (mp, msg, hlp, deletions_allowed);
}
static void mp_ins_error (MP mp, const char *msg, const char **hlp, boolean deletions_allowed) {
  /* back up one inserted token and call |error| */
  mp->OK_to_interrupt = false;
  mp_back_input (mp);
  token_type = (quarterword) inserted;
  mp->OK_to_interrupt = true;
  mp_error (mp, msg, hlp, deletions_allowed);
}


@ The |begin_file_reading| procedure starts a new level of input for lines
of characters to be read from a file, or as an insertion from the
terminal. It does not take care of opening the file, nor does it set |loc|
or |limit| or |line|.
@^system dependencies@>

@c
void mp_begin_file_reading (MP mp) {
  if (mp->in_open == (mp->max_in_open-1))
    mp_reallocate_input_stack (mp, (mp->max_in_open + mp->max_in_open / 4));
  if (mp->first == mp->buf_size)
    mp_reallocate_buffer (mp, (mp->buf_size + mp->buf_size / 4));
  mp->in_open++;
  push_input;
  iindex = (quarterword) mp->in_open;
  if (mp->in_open_max < mp->in_open)
    mp->in_open_max = mp->in_open;
  mp->mpx_name[iindex] = absent;
  start = (halfword) mp->first;
  name = is_term;               /* |terminal_input| is now |true| */
}


@ Conversely, the variables must be downdated when such a level of input
is finished.  Any associated \.{MPX} file must also be closed and popped
off the file stack. While finishing preloading, it is possible that the file
does not actually end with 'dump', so we capture that case here as well.

@c
static void mp_end_file_reading (MP mp) {
  if (mp->reading_preload && mp->input_ptr == 0) {
      set_cur_sym(mp->frozen_dump);
      mp_back_input (mp);
      return;
  }
  if (mp->in_open > iindex) {
    if ((mp->mpx_name[mp->in_open] == absent) || (name <= max_spec_src)) {
      mp_confusion (mp, "endinput");
@:this can't happen endinput}{\quad endinput@>;
    } else {
      (mp->close_file) (mp, mp->input_file[mp->in_open]);       /* close an \.{MPX} file */
      delete_str_ref (mp->mpx_name[mp->in_open]);
      decr (mp->in_open);
    }
  }
  mp->first = (size_t) start;
  if (iindex != mp->in_open)
    mp_confusion (mp, "endinput");
  if (name > max_spec_src) {
    (mp->close_file) (mp, cur_file);
    xfree (in_ext);
    xfree (in_name);
    xfree (in_area);
  }
  pop_input;
  decr (mp->in_open);
}


@ Here is a function that tries to resume input from an \.{MPX} file already
associated with the current input file.  It returns |false| if this doesn't
work.

@c
static boolean mp_begin_mpx_reading (MP mp) {
  if (mp->in_open != iindex + 1) {
    return false;
  } else {
    if (mp->mpx_name[mp->in_open] <= absent)
      mp_confusion (mp, "mpx");
    if (mp->first == mp->buf_size)
      mp_reallocate_buffer (mp, (mp->buf_size + (mp->buf_size / 4)));
    push_input;
    iindex = (quarterword) mp->in_open;
    start = (halfword) mp->first;
    name = mp->mpx_name[mp->in_open];
    add_str_ref (name);
    /* Put an empty line in the input buffer */
    /* We want to make it look as though we have just read a blank line
       without really doing so. */
    mp->last = mp->first;
    limit = (halfword) mp->last;
    /* simulate |input_ln| and |firm_up_the_line| */
    mp->buffer[limit] = xord ('%');
    mp->first = (size_t) (limit + 1);
    loc = start;
    return true;
  }
}


@ This procedure temporarily stops reading an \.{MPX} file.

@c
static void mp_end_mpx_reading (MP mp) {
  if (mp->in_open != iindex)
    mp_confusion (mp, "mpx");
@:this can't happen mpx}{\quad mpx@>;
  if (loc < limit) {
    /* Complain that we are not at the end of a line in the \.{MPX} file */
    /* Here we enforce a restriction that simplifies the input stacks considerably.
       This should not inconvenience the user because \.{MPX} files are generated
       by an auxiliary program called \.{DVItoMP}. */
    const char *hlp[] = {
         "This file contains picture expressions for btex...etex",
         "blocks.  Such files are normally generated automatically",
         "but this one seems to be messed up.  I'm going to ignore",
         "the rest of this line.",
         NULL };
    mp_error (mp, "`mpxbreak' must be at the end of a line", hlp, true);
  }
  mp->first = (size_t) start;
  pop_input;
}

@ In order to keep the stack from overflowing during a long sequence of
inserted `\.{show}' commands, the following routine removes completed
error-inserted lines from memory.

@c
void mp_clear_for_error_prompt (MP mp) {
  while (file_state && terminal_input && (mp->input_ptr > 0) && (loc == limit))
    mp_end_file_reading (mp);
  mp_print_ln (mp);
  clear_terminal();
}


@ To get \MP's whole input mechanism going, we perform the following
actions.

@<Initialize the input routines@>=
{
  mp->input_ptr = 0;
  mp->max_in_stack = file_bottom;
  mp->in_open = file_bottom;
  mp->open_parens = 0;
  mp->max_buf_stack = 0;
  mp->param_ptr = 0;
  mp->max_param_stack = 0;
  mp->first = 0;
  start = 0;
  iindex = file_bottom;
  line = 0;
  name = is_term;
  mp->mpx_name[file_bottom] = absent;
  mp->force_eof = false;
  if (!mp_init_terminal (mp))
    mp_jump_out (mp);
  limit = (halfword) mp->last;
  mp->first = mp->last + 1;
  /* |init_terminal| has set |loc| and |last| */
}


@* Getting the next token.
The heart of \MP's input mechanism is the |get_next| procedure, which
we shall develop in the next few sections of the program. Perhaps we
shouldn't actually call it the ``heart,'' however; it really acts as \MP's
eyes and mouth, reading the source files and gobbling them up. And it also
helps \MP\ to regurgitate stored token lists that are to be processed again.

The main duty of |get_next| is to input one token and to set |cur_cmd|
and |cur_mod| to that token's command code and modifier. Furthermore, if
the input token is a symbolic token, that token's |hash| address
is stored in |cur_sym|; otherwise |cur_sym| is set to zero.

Underlying this simple description is a certain amount of complexity
because of all the cases that need to be handled.
However, the inner loop of |get_next| is reasonably short and fast.

@ Before getting into |get_next|, we need to consider a mechanism by which
\MP\ helps keep errors from propagating too far. Whenever the program goes
into a mode where it keeps calling |get_next| repeatedly until a certain
condition is met, it sets |scanner_status| to some value other than |normal|.
Then if an input file ends, or if an `\&{outer}' symbol appears,
an appropriate error recovery will be possible.

The global variable |warning_info| helps in this error recovery by providing
additional information. For example, |warning_info| might indicate the
name of a macro whose replacement text is being scanned.

@d normal 0 /* |scanner_status| at ``quiet times'' */
@d skipping 1 /* |scanner_status| when false conditional text is being skipped */
@d flushing 2 /* |scanner_status| when junk after a statement is being ignored */
@d absorbing 3 /* |scanner_status| when a \&{text} parameter is being scanned */
@d var_defining 4 /* |scanner_status| when a \&{vardef} is being scanned */
@d op_defining 5 /* |scanner_status| when a macro \&{def} is being scanned */
@d loop_defining 6 /* |scanner_status| when a \&{for} loop is being scanned */

@<Glob...@>=
#define tex_flushing 7 /* |scanner_status| when skipping \TeX\ material */
integer scanner_status; /* are we scanning at high speed? */
mp_sym warning_info;    /* if so, what else do we need to know,
                           in case an error occurs? */
integer warning_line;
mp_node warning_info_node;

@ @<Initialize the input routines@>=
mp->scanner_status = normal;

@ The following subroutine
is called when an `\&{outer}' symbolic token has been scanned or
when the end of a file has been reached. These two cases are distinguished
by |cur_sym|, which is zero at the end of a file.

@c
static boolean mp_check_outer_validity (MP mp) {
  mp_node p;    /* points to inserted token list */
  if (mp->scanner_status == normal) {
    return true;
  } else if (mp->scanner_status == tex_flushing) {
    @<Check if the file has ended while flushing \TeX\ material and set the
      result value for |check_outer_validity|@>;
  } else {
    @<Back up an outer symbolic token so that it can be reread@>;
    if (mp->scanner_status > skipping) {
      @<Tell the user what has run away and try to recover@>;
    } else {
      char msg[256];
      const char *hlp[] = {
             "A forbidden `outer' token occurred in skipped text.",
             "This kind of error happens when you say `if...' and forget",
             "the matching `fi'. I've inserted a `fi'; this might work.",
             NULL };
      mp_snprintf(msg, 256, "Incomplete if; all text was ignored after line %d", (int)mp->warning_line);
@.Incomplete if...@>;
      if (cur_sym() == NULL) {
        hlp[0] = "The file ended while I was skipping conditional text.";
      } 
      set_cur_sym (mp->frozen_fi);
      mp_ins_error (mp, msg, hlp, false);
    }
    return false;
  }
}


@ @<Check if the file has ended while flushing \TeX\ material and set...@>=
if (cur_sym() != NULL) {
  return true;
} else {
  char msg[256];
  const char *hlp[] = {
         "The file ended while I was looking for the `etex' to",
         "finish this TeX material.  I've inserted `etex' now.",
          NULL };
  mp_snprintf(msg, 256, "TeX mode didn't end; all text was ignored after line %d", (int)mp->warning_line);
  set_cur_sym(mp->frozen_etex);
  mp_ins_error (mp, msg, hlp, false);
  return false;
}


@ @<Back up an outer symbolic token so that it can be reread@>=
if (cur_sym() != NULL) {
  p = mp_get_symbolic_node (mp);
  set_mp_sym_sym (p, cur_sym());
  mp_name_type (p) = cur_sym_mod();
  back_list (p);                /* prepare to read the symbolic token again */
}

@ @<Tell the user what has run away...@>=
{
  char msg[256];
  const char *msg_start = NULL;
  const char *hlp[] = {
         "I suspect you have forgotten an `enddef',",
         "causing me to read past where you wanted me to stop.",
         "I'll try to recover; but if the error is serious,",
         "you'd better type `E' or `X' now and fix your file.",
         NULL };
  mp_runaway (mp);              /* print the definition-so-far */
  if (cur_sym() == NULL) {
    msg_start = "File ended while scanning";
@.File ended while scanning...@>
  } else {
    msg_start = "Forbidden token found while scanning";
@.Forbidden token found...@>
  }
  switch (mp->scanner_status) {
    @<Complete the error message,
      and set |cur_sym| to a token that might help recover from the error@>
  }                             /* there are no other cases */
  mp_ins_error (mp, msg, hlp, true);
}


@ As we consider various kinds of errors, it is also appropriate to
change the first line of the help message just given; |help_line[3]|
points to the string that might be changed.

@<Complete the error message,...@>=
case flushing:
  mp_snprintf (msg, 256, "%s to the end of the statement", msg_start);
  hlp[0] = "A previous error seems to have propagated,";
  set_cur_sym(mp->frozen_semicolon);
  break;
case absorbing:
  mp_snprintf (msg, 256, "%s a text argument", msg_start);
  hlp[0] = "It seems that a right delimiter was left out,";
  if (mp->warning_info == NULL) {
    set_cur_sym(mp->frozen_end_group);
  } else {
    set_cur_sym(mp->frozen_right_delimiter);
    /* the next line makes sure that the inserted delimiter will
      match the delimiter that already was read. */
    set_equiv_sym (cur_sym(), mp->warning_info);
  }
  break;
case var_defining:
  {
    mp_string s;
    int old_setting = mp->selector;
    mp->selector = new_string;
    mp_print_variable_name (mp, mp->warning_info_node);
    s = mp_make_string (mp);
    mp->selector = old_setting;
    mp_snprintf (msg, 256, "%s the definition of %s", msg_start, s->str);
    delete_str_ref(s);
  }
  set_cur_sym(mp->frozen_end_def);
  break;
case op_defining:
  {
    char *s = mp_str(mp, text(mp->warning_info));
    mp_snprintf (msg, 256, "%s the definition of %s", msg_start, s);
  }
  set_cur_sym(mp->frozen_end_def);
  break;
case loop_defining:
  {
    char *s = mp_str(mp, text(mp->warning_info));
    mp_snprintf (msg, 256, "%s the text of a %s loop", msg_start, s);
  }
  hlp[0] = "I suspect you have forgotten an `endfor',";
  set_cur_sym(mp->frozen_end_for);
break;

@ The |runaway| procedure displays the first part of the text that occurred
when \MP\ began its special |scanner_status|, if that text has been saved.

@<Declarations@>=
static void mp_runaway (MP mp);

@ @c
void mp_runaway (MP mp) {
  if (mp->scanner_status > flushing) {
    mp_print_nl (mp, "Runaway ");
    switch (mp->scanner_status) {
    case absorbing:
      mp_print (mp, "text?");
      break;
    case var_defining:
    case op_defining:
      mp_print (mp, "definition?");
      break;
    case loop_defining:
      mp_print (mp, "loop?");
      break;
    }                           /* there are no other cases */
    mp_print_ln (mp);
    mp_show_token_list (mp, mp_link (mp->hold_head), NULL, mp->error_line - 10,
                        0);
  }
}


@ We need to mention a procedure that may be called by |get_next|.

@<Declarations@>=
static void mp_firm_up_the_line (MP mp);

@ And now we're ready to take the plunge into |get_next| itself.
Note that the behavior depends on the |scanner_status| because percent signs
and double quotes need to be passed over when skipping TeX material.

@c
void mp_get_next (MP mp) {
  /* sets |cur_cmd|, |cur_mod|, |cur_sym| to next token */
  mp_sym cur_sym_;    /* speed up access */
RESTART:
  set_cur_sym(NULL);
  set_cur_sym_mod(0);
  if (file_state) {
    int k;        /* an index into |buffer| */
    ASCII_code c; /* the current character in the buffer */
    int cclass;    /* its class number */
    /* Input from external file; |goto restart| if no input found,
       or |return| if a non-symbolic token is found */
    /* A percent sign appears in |buffer[limit]|; this makes it unnecessary
       to have a special test for end-of-line. */
  SWITCH:
    c = mp->buffer[loc];
    incr (loc);
    cclass = mp->char_class[c];
    switch (cclass) {
    case digit_class:
      scan_numeric_token((c - '0'));
      return;
      break;
    case period_class:
      cclass = mp->char_class[mp->buffer[loc]];
      if (cclass > period_class) {
        goto SWITCH;
      } else if (cclass < period_class) {  /* |class=digit_class| */
        scan_fractional_token(0);
        return;
      }
      break;
    case space_class:
      goto SWITCH;
      break;
    case percent_class:
      if (mp->scanner_status == tex_flushing) {
        if (loc < limit)
          goto SWITCH;
      }
      /* Move to next line of file, or |goto restart| if there is no next line */
      switch (move_to_next_line(mp)) {
      case 1:  goto RESTART;       break;
      case 2:  goto COMMON_ENDING; break;
      default: break;
      }
      check_interrupt;
      goto SWITCH;
      break;
    case string_class:
      if (mp->scanner_status == tex_flushing) {
        goto SWITCH;
      } else {
        if (mp->buffer[loc] == '"') {
          set_cur_mod_str(mp_rts(mp,""));
        } else {
          k = loc;
          mp->buffer[limit + 1] = xord ('"');
          do {
            incr (loc);
          } while (mp->buffer[loc] != '"');
          if (loc > limit) {
            /* Decry the missing string delimiter and |goto restart| */
            /* We go to |restart| after this error message, not to |SWITCH|,
               because the |clear_for_error_prompt| routine might have reinstated
               |token_state| after |error| has finished. */
            const char *hlp[] =  {
             "Strings should finish on the same line as they began.",
             "I've deleted the partial string; you might want to",
             "insert another by typing, e.g., `I\"new string\"'.",
             NULL };
            loc = limit;  /* the next character to be read on this line will be |"%"| */
            mp_error (mp, "Incomplete string token has been flushed", hlp, false);
            goto RESTART;
          }
          str_room ((size_t) (loc - k));
          do {
            append_char (mp->buffer[k]);
            incr (k);
          } while (k != loc);
          set_cur_mod_str(mp_make_string (mp));
        }
        incr (loc);
        set_cur_cmd((mp_variable_type)mp_string_token);
        return;
      }
      break;
    case isolated_classes:
      k = loc - 1;
      goto FOUND;
      break;
    case invalid_class:
      if (mp->scanner_status == tex_flushing) {
        goto SWITCH;
      } else {
        /* Decry the invalid character and |goto restart| */
        /* We go to |restart| instead of to |SWITCH|, because we might enter
        |token_state| after the error has been dealt with
        (cf.\ |clear_for_error_prompt|). */
        const char *hlp[] = {
           "A funny symbol that I can\'t read has just been input.",
           "Continue, and I'll forget that it ever happened.",
           NULL };
        mp_error(mp, "Text line contains an invalid character", hlp, false);
        goto RESTART;
      }
      break;
    default:
      break;                      /* letters, etc. */
    }
    k = loc - 1;
    while (mp->char_class[mp->buffer[loc]] == cclass)
      incr (loc);
  FOUND:
    set_cur_sym(mp_id_lookup (mp, (char *) (mp->buffer + k), (size_t) (loc - k), true));

  } else {
    /* Input from token list; |goto restart| if end of list or
       if a parameter needs to be expanded,
       or |return| if a non-symbolic token is found */
    if (nloc != NULL && mp_type (nloc) == mp_symbol_node) { /* symbolic token */
      int cur_sym_mod_ = mp_name_type (nloc);
      halfword cur_info = mp_sym_info (nloc);
      set_cur_sym(mp_sym_sym (nloc));
      set_cur_sym_mod(cur_sym_mod_);
      nloc = mp_link (nloc);        /* move to next */
      if (cur_sym_mod_ == mp_expr_sym) {
        set_cur_cmd((mp_variable_type)mp_capsule_token);
        set_cur_mod_node(mp->param_stack[param_start + cur_info]);
        set_cur_sym_mod(0);
        set_cur_sym(NULL);
        return;
      } else if (cur_sym_mod_ == mp_suffix_sym || cur_sym_mod_ == mp_text_sym) {
        mp_begin_token_list (mp,
                             mp->param_stack[param_start + cur_info],
                             (quarterword) parameter);
        goto RESTART;
      }
    } else if (nloc != NULL) {
      /* Get a stored numeric or string or capsule token and |return| */
      if (mp_name_type (nloc) == mp_token) {
        if (mp_type (nloc) == mp_known) {
          set_cur_mod_number(value_number (nloc));
          set_cur_cmd((mp_variable_type)mp_numeric_token);
        } else {
          set_cur_mod_str(value_str (nloc));
          set_cur_cmd((mp_variable_type)mp_string_token);
          add_str_ref (cur_mod_str());
        }
      } else {
        set_cur_mod_node(nloc);
        set_cur_cmd((mp_variable_type)mp_capsule_token);
      }
      nloc = mp_link (nloc);
      return;
    } else {                        /* we are done with this token list */
      mp_end_token_list (mp);
      goto RESTART;                 /* resume previous level */
    }
  }
COMMON_ENDING:
  /* When a symbolic token is declared to be `\&{outer}', its command code
     is increased by |outer_tag|. */
  cur_sym_ = cur_sym();
  set_cur_cmd(eq_type (cur_sym_));
  set_cur_mod(equiv (cur_sym_));
  set_cur_mod_node(equiv_node (cur_sym_));
  if (cur_cmd() >= mp_outer_tag) {
    if (mp_check_outer_validity (mp))
      set_cur_cmd(cur_cmd() - mp_outer_tag);
    else
      goto RESTART;
  }
}

@ The global variable |force_eof| is normally |false|; it is set |true|
by an \&{endinput} command.

@<Glob...@>=
boolean force_eof;      /* should the next \&{input} be aborted early? */

@ @<Declarations@>=
static int move_to_next_line (MP mp);

@ @c
static int move_to_next_line (MP mp) {
  if (name > max_spec_src) {
    /* Read next line of file into |buffer|, or return 1
      (|goto restart|) if the file has ended */
    /* We must decrement |loc| in order to leave the buffer in a valid state
       when an error condition causes us to |goto restart| without calling
      |end_file_reading|. */
    {
      incr (line);
      mp->first = (size_t) start;
      if (!mp->force_eof) {
        if (mp_input_ln (mp, cur_file))     /* not end of file */
          mp_firm_up_the_line (mp); /* this sets |limit| */
        else
          mp->force_eof = true;
      };
      if (mp->force_eof) {
        mp->force_eof = false;
        decr (loc);
        if (mpx_reading) {
          /* Complain that the \.{MPX} file ended unexpectly; then set
            |cur_sym:=mp->frozen_mpx_break| and |goto comon_ending| */
	  /* We should never actually come to the end of an \.{MPX} file because such
             files should have an \&{mpxbreak} after the translation of the last
             \&{btex}$\,\ldots\,$\&{etex} block. */
          const char *hlp[] =  {"The file had too few picture expressions for btex...etex",
           "blocks.  Such files are normally generated automatically",
           "but this one got messed up.  You might want to insert a",
           "picture expression now.",
            NULL }; 
          mp->mpx_name[iindex] = mpx_finished;
          mp_error (mp, "mpx file ended unexpectedly", hlp, false);
          set_cur_sym(mp->frozen_mpx_break);
          return 2;
        } else {
          mp_print_char (mp, xord (')'));
          decr (mp->open_parens);
          update_terminal();          /* show user that file has been read */
          mp_end_file_reading (mp); /* resume previous level */
          if (mp_check_outer_validity (mp))
            return 1;
          else
            return 1;
        }
      }
      mp->buffer[limit] = xord ('%');
      mp->first = (size_t) (limit + 1);
      loc = start;                  /* ready to read */
    }
    

  } else {
    if (mp->input_ptr > 0) {
      /* text was inserted during error recovery or by \&{scantokens} */
      mp_end_file_reading (mp);
      /* goto RESTART */
      return 1;               /* resume previous level */
    }
    if (mp->job_name == NULL
        && (mp->selector < log_only || mp->selector >= write_file))
      mp_open_log_file (mp);
    if (mp->interaction > mp_nonstop_mode) {
      if (limit == start)         /* previous line was empty */
        mp_print_nl (mp, "(Please type a command or say `end')");
      mp_print_ln (mp);
      mp->first = (size_t) start;
      prompt_input ("*");         /* input on-line into |buffer| */
      limit = (halfword) mp->last;
      mp->buffer[limit] = xord ('%');
      mp->first = (size_t) (limit + 1);
      loc = start;
    } else {
      mp_fatal_error (mp, "*** (job aborted, no legal end found)");
      /* nonstop mode, which is intended for overnight batch processing,
         never waits for on-line input */
    }
  }
  return 0;
}


@ If the user has set the |mp_pausing| parameter to some positive value,
and if nonstop mode has not been selected, each line of input is displayed
on the terminal and the transcript file, followed by `\.{=>}'.
\MP\ waits for a response. If the response is NULL (i.e., if nothing is
typed except perhaps a few blank spaces), the original
line is accepted as it stands; otherwise the line typed is
used instead of the line in the file.

@c
void mp_firm_up_the_line (MP mp) {
  size_t k;     /* an index into |buffer| */
  limit = (halfword) mp->last;
  if ((!mp->noninteractive)
      && (number_positive (internal_value (mp_pausing)))
      && (mp->interaction > mp_nonstop_mode)) {
    wake_up_terminal();
    mp_print_ln (mp);
    if (start < limit) {
      for (k = (size_t) start; k < (size_t) limit; k++) {
        mp_print_char (mp, mp->buffer[k]);
      }
    }
    mp->first = (size_t) limit;
    prompt_input ("=>");        /* wait for user response */
@.=>@>;
    if (mp->last > mp->first) {
      for (k = mp->first; k < mp->last; k++) {  /* move line down in buffer */
        mp->buffer[k + (size_t) start - mp->first] = mp->buffer[k];
      }
      limit = (halfword) ((size_t) start + mp->last - mp->first);
    }
  }
}


@* Dealing with \TeX\ material.
The \&{btex}$\,\ldots\,$\&{etex} and \&{verbatimtex}$\,\ldots\,$\&{etex}
features need to be implemented at a low level in the scanning process
so that \MP\ can stay in synch with the a preprocessor that treats
blocks of \TeX\ material as they occur in the input file without trying
to expand \MP\ macros.  Thus we need a special version of |get_next|
that does not expand macros and such but does handle \&{btex},
\&{verbatimtex}, etc.

The special version of |get_next| is called |get_t_next|.  It works by flushing
\&{btex}$\,\ldots\,$\&{etex} and \&{verbatimtex}\allowbreak
$\,\ldots\,$\&{etex} blocks, switching to the \.{MPX} file when it sees
\&{btex}, and switching back when it sees \&{mpxbreak}.

@d btex_code 0
@d verbatim_code 1

@ @<Put each...@>=
mp_primitive (mp, "btex", mp_start_tex, btex_code);
@:btex_}{\&{btex} primitive@>;
mp_primitive (mp, "verbatimtex", mp_start_tex, verbatim_code);
@:verbatimtex_}{\&{verbatimtex} primitive@>;
mp_primitive (mp, "etex", mp_etex_marker, 0);
mp->frozen_etex = mp_frozen_primitive (mp, "etex", mp_etex_marker, 0);
@:etex_}{\&{etex} primitive@>;
mp_primitive (mp, "mpxbreak", mp_mpx_break, 0);
mp->frozen_mpx_break = mp_frozen_primitive (mp, "mpxbreak", mp_mpx_break, 0);
@:mpx_break_}{\&{mpxbreak} primitive@>
 

@ @<Cases of |print_cmd...@>=
case mp_start_tex:
if (m == btex_code)
  mp_print (mp, "btex");
else
  mp_print (mp, "verbatimtex");
break;
case mp_etex_marker:
mp_print (mp, "etex");
break;
case mp_mpx_break:
mp_print (mp, "mpxbreak");
break;

@ Actually, |get_t_next| is a macro that avoids procedure overhead except
in the unusual case where \&{btex}, \&{verbatimtex}, \&{etex}, or \&{mpxbreak}
is encountered.

@d get_t_next(a) do {
  mp_get_next (mp);
  if (cur_cmd() <= mp_max_pre_command)
    mp_t_next (mp);
} while (0)

@c
@ @<Declarations@>=
static void mp_t_next (MP mp);
static void mp_start_mpx_input (MP mp);

@ @c
static void mp_t_next (MP mp) {
  int old_status;       /* saves the |scanner_status| */
  integer old_info;     /* saves the |warning_info| */
  while (cur_cmd() <= mp_max_pre_command) {
    if (cur_cmd() == mp_mpx_break) {
      if (!file_state || (mp->mpx_name[iindex] == absent)) {
        @<Complain about a misplaced \&{mpxbreak}@>;
      } else {
        mp_end_mpx_reading (mp);
        goto TEX_FLUSH;
      }
    } else if (cur_cmd() == mp_start_tex) {
      if (token_state || (name <= max_spec_src)) {
        @<Complain that we are not reading a file@>;
      } else if (mpx_reading) {
        @<Complain that \.{MPX} files cannot contain \TeX\ material@>;
      } else if ((cur_mod() != verbatim_code) &&
                 (mp->mpx_name[iindex] != mpx_finished)) {
        if (!mp_begin_mpx_reading (mp))
          mp_start_mpx_input (mp);
      } else {
        goto TEX_FLUSH;
      }
    } else {
      @<Complain about a misplaced \&{etex}@>;
    }
    goto COMMON_ENDING;
  TEX_FLUSH:
    @<Flush the \TeX\ material@>;
  COMMON_ENDING:
    mp_get_next (mp);
  }
}


@ We could be in the middle of an operation such as skipping false conditional
text when \TeX\ material is encountered, so we must be careful to save the
|scanner_status|.

@<Flush the \TeX\ material@>=
old_status = mp->scanner_status;
old_info = mp->warning_line;
mp->scanner_status = tex_flushing;
mp->warning_line = line;
do {
  mp_get_next (mp);
} while (cur_cmd() != mp_etex_marker);
mp->scanner_status = old_status;
mp->warning_line = old_info

@ @<Complain that \.{MPX} files cannot contain \TeX\ material@>=
{
  const char *hlp[] = {
         "This file contains picture expressions for btex...etex",
         "blocks.  Such files are normally generated automatically",
         "but this one seems to be messed up.  I'll just keep going",
         "and hope for the best.",
         NULL };
  mp_error (mp, "An mpx file cannot contain btex or verbatimtex blocks", hlp, true);
}


@ @<Complain that we are not reading a file@>=
{
  const char *hlp[] = {
         "I'll have to ignore this preprocessor command because it",
         "only works when there is a file to preprocess.  You might",
         "want to delete everything up to the next `etex`.",
         NULL };
  mp_error (mp, "You can only use `btex' or `verbatimtex' in a file", hlp, true);
}


@ @<Complain about a misplaced \&{mpxbreak}@>=
{
  const char *hlp[] = {
         "I'll ignore this preprocessor command because it",
         "doesn't belong here",
         NULL };
  mp_error (mp, "Misplaced mpxbreak", hlp, true);
}


@ @<Complain about a misplaced \&{etex}@>=
{
  const char *hlp[] = { 
         "There is no btex or verbatimtex for this to match",
          NULL };
  mp_error (mp, "Extra etex will be ignored", hlp, true);
}


@* Scanning macro definitions.
\MP\ has a variety of ways to tuck tokens away into token lists for later
use: Macros can be defined with \&{def}, \&{vardef}, \&{primarydef}, etc.;
repeatable code can be defined with \&{for}, \&{forever}, \&{forsuffixes}.
All such operations are handled by the routines in this part of the program.

The modifier part of each command code is zero for the ``ending delimiters''
like \&{enddef} and \&{endfor}.

@d start_def 1 /* command modifier for \&{def} */
@d var_def 2 /* command modifier for \&{vardef} */
@d end_def 0 /* command modifier for \&{enddef} */
@d start_forever 1 /* command modifier for \&{forever} */
@d start_for 2 /* command modifier for \&{forever} */
@d start_forsuffixes 3 /* command modifier for \&{forever} */
@d end_for 0 /* command modifier for \&{endfor} */

@<Put each...@>=
mp_primitive (mp, "def", mp_macro_def, start_def);
@:def_}{\&{def} primitive@>;
mp_primitive (mp, "vardef", mp_macro_def, var_def);
@:var_def_}{\&{vardef} primitive@>;
mp_primitive (mp, "primarydef", mp_macro_def, mp_secondary_primary_macro);
@:primary_def_}{\&{primarydef} primitive@>;
mp_primitive (mp, "secondarydef", mp_macro_def, mp_tertiary_secondary_macro);
@:secondary_def_}{\&{secondarydef} primitive@>;
mp_primitive (mp, "tertiarydef", mp_macro_def, mp_expression_tertiary_macro);
@:tertiary_def_}{\&{tertiarydef} primitive@>;
mp_primitive (mp, "enddef", mp_macro_def, end_def);
mp->frozen_end_def = mp_frozen_primitive (mp, "enddef", mp_macro_def, end_def);
@:end_def_}{\&{enddef} primitive@>;
mp_primitive (mp, "for", mp_iteration, start_for);
@:for_}{\&{for} primitive@>;
mp_primitive (mp, "forsuffixes", mp_iteration, start_forsuffixes);
@:for_suffixes_}{\&{forsuffixes} primitive@>;
mp_primitive (mp, "forever", mp_iteration, start_forever);
@:forever_}{\&{forever} primitive@>;
mp_primitive (mp, "endfor", mp_iteration, end_for);
mp->frozen_end_for = mp_frozen_primitive (mp, "endfor", mp_iteration, end_for);
@:end_for_}{\&{endfor} primitive@>
 

@ @<Cases of |print_cmd...@>=
case mp_macro_def:
if (m <= var_def) {
  if (m == start_def)
    mp_print (mp, "def");
  else if (m < start_def)
    mp_print (mp, "enddef");
  else
    mp_print (mp, "vardef");
} else if (m == mp_secondary_primary_macro) {
  mp_print (mp, "primarydef");
} else if (m == mp_tertiary_secondary_macro) {
  mp_print (mp, "secondarydef");
} else {
  mp_print (mp, "tertiarydef");
}
break;
case mp_iteration:
if (m == start_forever)
  mp_print (mp, "forever");
else if (m == end_for)
  mp_print (mp, "endfor");
else if (m == start_for)
  mp_print (mp, "for");
else
  mp_print (mp, "forsuffixes");
break;

@ Different macro-absorbing operations have different syntaxes, but they
also have a lot in common. There is a list of special symbols that are to
be replaced by parameter tokens; there is a special command code that
ends the definition; the quotation conventions are identical.  Therefore
it makes sense to have most of the work done by a single subroutine. That
subroutine is called |scan_toks|.

The first parameter to |scan_toks| is the command code that will
terminate scanning (either |macro_def| or |iteration|).

The second parameter, |subst_list|, points to a (possibly empty) list
of non-symbolic nodes whose |info| and |value| fields specify symbol tokens
before and after replacement. The list will be returned to free storage
by |scan_toks|.

The third parameter is simply appended to the token list that is built.
And the final parameter tells how many of the special operations
\.{\#\AT!}, \.{\AT!}, and \.{\AT!\#} are to be replaced by suffix parameters.
When such parameters are present, they are called \.{(SUFFIX0)},
\.{(SUFFIX1)}, and \.{(SUFFIX2)}.

@<Types...@>=
typedef struct mp_subst_list_item {
  mp_name_type_type info_mod;
  quarterword value_mod;
  mp_sym info;
  halfword value_data;
  struct mp_subst_list_item *link;
} mp_subst_list_item;

@
@c
static mp_node mp_scan_toks (MP mp, mp_command_code terminator,
                             mp_subst_list_item * subst_list, mp_node tail_end,
                             quarterword suffix_count) {
  mp_node p;    /* tail of the token list being built */
  mp_subst_list_item *q = NULL; /* temporary for link management */
  integer balance;      /* left delimiters minus right delimiters */
  halfword cur_data;
  quarterword cur_data_mod = 0;
  p = mp->hold_head;
  balance = 1;
  mp_link (mp->hold_head) = NULL;
  while (1) {
    get_t_next (mp);
    cur_data = -1;
    if (cur_sym() != NULL) {
      @<Substitute for |cur_sym|, if it's on the |subst_list|@>;
      if (cur_cmd() == terminator) {
        @<Adjust the balance; |break| if it's zero@>;
      } else if (cur_cmd() == mp_macro_special) {
        /* Handle quoted symbols, \.{\#\AT!}, \.{\AT!}, or \.{\AT!\#} */
        if (cur_mod() == quote) {
          get_t_next (mp);
        } else if (cur_mod() <= suffix_count) {
          cur_data = cur_mod() - 1;
          cur_data_mod = mp_suffix_sym;
        }
      }
    }
    if (cur_data != -1) {
      mp_node pp = mp_get_symbolic_node (mp);
      set_mp_sym_info (pp, cur_data);
      mp_name_type (pp) = cur_data_mod;
      mp_link (p) = pp;
    } else {
      mp_link (p) = mp_cur_tok (mp);
    }
    p = mp_link (p);
  }
  mp_link (p) = tail_end;
  while (subst_list) {
    q = subst_list->link;
    xfree (subst_list);
    subst_list = q;
  }
  return mp_link (mp->hold_head);
}

@
@c
void mp_print_sym  (mp_sym sym) {
  printf("{type = %d, v = {type = %d, data = {indep = {scale = %d, serial = %d}, n = %d, str = %p, sym = %p, node = %p, p = %p}}, text = %p}\n", sym->type, sym->v.type, (int)sym->v.data.indep.scale, (int)sym->v.data.indep.serial,
    sym->v.data.n.type, sym->v.data.str, sym->v.data.sym, sym->v.data.node, sym->v.data.p, sym->text);
  if (is_number(sym->v.data.n)) {
      mp_number n = sym->v.data.n;
      printf("{data = {dval = %f, val = %d}, type = %d}\n", n.data.dval, n.data.val, n.type);
  }
  if (sym->text != NULL) {
     mp_string t = sym->text;
     printf ("{str = %p \"%s\", len = %d, refs = %d}\n", t->str, t->str, (int)t->len, t->refs);
  }
}

@
@<Declarations@>=
void mp_print_sym  (mp_sym sym) ;

@ @<Substitute for |cur_sym|...@>=
{
  q = subst_list;
  while (q != NULL) {
    if (q->info == cur_sym() && q->info_mod == cur_sym_mod()) {
      cur_data = q->value_data;
      cur_data_mod = q->value_mod;
      set_cur_cmd((mp_variable_type)mp_relax);
      break;
    }
    q = q->link;
  }
}


@ @<Adjust the balance; |break| if it's zero@>=
if (cur_mod() > 0) {
  incr (balance);
} else {
  decr (balance);
  if (balance == 0)
    break;
}


@ Four commands are intended to be used only within macro texts: \&{quote},
\.{\#\AT!}, \.{\AT!}, and \.{\AT!\#}. They are variants of a single command
code called |macro_special|.

@d quote 0 /* |macro_special| modifier for \&{quote} */
@d macro_prefix 1 /* |macro_special| modifier for \.{\#\AT!} */
@d macro_at 2 /* |macro_special| modifier for \.{\AT!} */
@d macro_suffix 3 /* |macro_special| modifier for \.{\AT!\#} */

@<Put each...@>=
mp_primitive (mp, "quote", mp_macro_special, quote);
@:quote_}{\&{quote} primitive@>;
mp_primitive (mp, "#@@", mp_macro_special, macro_prefix);
@:]]]\#\AT!_}{\.{\#\AT!} primitive@>;
mp_primitive (mp, "@@", mp_macro_special, macro_at);
@:]]]\AT!_}{\.{\AT!} primitive@>;
mp_primitive (mp, "@@#", mp_macro_special, macro_suffix);
@:]]]\AT!\#_}{\.{\AT!\#} primitive@>
 

@ @<Cases of |print_cmd...@>=
case mp_macro_special:
switch (m) {
case macro_prefix:
  mp_print (mp, "#@@");
  break;
case macro_at:
  mp_print_char (mp, xord ('@@'));
  break;
case macro_suffix:
  mp_print (mp, "@@#");
  break;
default:
  mp_print (mp, "quote");
  break;
}
break;

@ Here is a routine that's used whenever a token will be redefined. If
the user's token is unredefinable, the `|mp->frozen_inaccessible|' token is
substituted; the latter is redefinable but essentially impossible to use,
hence \MP's tables won't get fouled up.

@c
static void mp_get_symbol (MP mp) {                               /* sets |cur_sym| to a safe symbol */
RESTART:
  get_t_next (mp);
  if ((cur_sym() == NULL) || mp_is_frozen(mp, cur_sym())) {
    const char *hlp[] = {
           "Sorry: You can\'t redefine a number, string, or expr.",
           "I've inserted an inaccessible symbol so that your",
           "definition will be completed without mixing me up too badly.",
           NULL };
    if (cur_sym() != NULL)
      hlp[0] = "Sorry: You can\'t redefine my error-recovery tokens.";
    else if (cur_cmd() == mp_string_token)
      delete_str_ref (cur_mod_str());
    set_cur_sym(mp->frozen_inaccessible);
    mp_ins_error (mp, "Missing symbolic token inserted", hlp, true);
@.Missing symbolic token...@>;
    goto RESTART;
  }
}


@ Before we actually redefine a symbolic token, we need to clear away its
former value, if it was a variable. The following stronger version of
|get_symbol| does that.

@c
static void mp_get_clear_symbol (MP mp) {
  mp_get_symbol (mp);
  mp_clear_symbol (mp, cur_sym(), false);
}


@ Here's another little subroutine; it checks that an equals sign
or assignment sign comes along at the proper place in a macro definition.

@c
static void mp_check_equals (MP mp) {
  if (cur_cmd() != mp_equals)
    if (cur_cmd() != mp_assignment) {
      const char *hlp[] = {
             "The next thing in this `def' should have been `=',",
             "because I've already looked at the definition heading.",
             "But don't worry; I'll pretend that an equals sign",
             "was present. Everything from here to `enddef'",
             "will be the replacement text of this macro.",
             NULL };
      mp_back_error (mp, "Missing `=' has been inserted", hlp, true);
@.Missing `='@>;
    }
}


@ A \&{primarydef}, \&{secondarydef}, or \&{tertiarydef} is rather easily
handled now that we have |scan_toks|.  In this case there are
two parameters, which will be \.{EXPR0} and \.{EXPR1}.

@c
static void mp_make_op_def (MP mp) {
  mp_command_code m;       /* the type of definition */
  mp_node q, r; /* for list manipulation */
  mp_subst_list_item *qm = NULL, *qn = NULL;
  m = cur_mod();
  mp_get_symbol (mp);
  qm = xmalloc (1, sizeof (mp_subst_list_item));
  qm->link = NULL;
  qm->info = cur_sym();
  qm->info_mod = cur_sym_mod();
  qm->value_data = 0;
  qm->value_mod = mp_expr_sym;
  mp_get_clear_symbol (mp);
  mp->warning_info = cur_sym();
  mp_get_symbol (mp);
  qn = xmalloc (1, sizeof (mp_subst_list_item));
  qn->link = qm;
  qn->info = cur_sym();
  qn->info_mod = cur_sym_mod();
  qn->value_data = 1;
  qn->value_mod = mp_expr_sym;
  get_t_next (mp);
  mp_check_equals (mp);
  mp->scanner_status = op_defining;
  q = mp_get_symbolic_node (mp);
  set_ref_count (q, 0);
  r = mp_get_symbolic_node (mp);
  mp_link (q) = r;
  set_mp_sym_info (r, mp_general_macro);
  mp_name_type (r) = mp_macro_sym;
  mp_link (r) = mp_scan_toks (mp, mp_macro_def, qn, NULL, 0);
  mp->scanner_status = normal;
  set_eq_type (mp->warning_info, m);
  set_equiv_node (mp->warning_info, q);
  mp_get_x_next (mp);
}


@ Parameters to macros are introduced by the keywords \&{expr},
\&{suffix}, \&{text}, \&{primary}, \&{secondary}, and \&{tertiary}.

@<Put each...@>=
mp_primitive (mp, "expr", mp_param_type, mp_expr_param);
@:expr_}{\&{expr} primitive@>;
mp_primitive (mp, "suffix", mp_param_type, mp_suffix_param);
@:suffix_}{\&{suffix} primitive@>;
mp_primitive (mp, "text", mp_param_type, mp_text_param);
@:text_}{\&{text} primitive@>;
mp_primitive (mp, "primary", mp_param_type, mp_primary_macro);
@:primary_}{\&{primary} primitive@>;
mp_primitive (mp, "secondary", mp_param_type, mp_secondary_macro);
@:secondary_}{\&{secondary} primitive@>;
mp_primitive (mp, "tertiary", mp_param_type, mp_tertiary_macro);
@:tertiary_}{\&{tertiary} primitive@>
 

@ @<Cases of |print_cmd...@>=
case mp_param_type:
if (m == mp_expr_param)
  mp_print (mp, "expr");
else if (m == mp_suffix_param)
  mp_print (mp, "suffix");
else if (m == mp_text_param)
  mp_print (mp, "text");
else if (m == mp_primary_macro)
  mp_print (mp, "primary");
else if (m == mp_secondary_macro)
  mp_print (mp, "secondary");
else
  mp_print (mp, "tertiary");
break;

@ Let's turn next to the more complex processing associated with \&{def}
and \&{vardef}. When the following procedure is called, |cur_mod|
should be either |start_def| or |var_def|.

Note that although the macro scanner allows |def = := enddef| and
|def := = enddef|; |def = = enddef| and |def := := enddef| will generate
an error because by the time the second of the two identical tokens is 
seen, its meaning has already become undefined.

@c
static void mp_scan_def (MP mp) {
  int m;        /* the type of definition */
  int n;        /* the number of special suffix parameters */
  int k;        /* the total number of parameters */
  int c;        /* the kind of macro we're defining */
  mp_subst_list_item *r = NULL, *rp = NULL;     /* parameter-substitution list */
  mp_node q;    /* tail of the macro token list */
  mp_node p;    /* temporary storage */
  quarterword sym_type; /* |expr_sym|, |suffix_sym|, or |text_sym| */
  mp_sym l_delim, r_delim;      /* matching delimiters */
  m = cur_mod();
  c = mp_general_macro;
  mp_link (mp->hold_head) = NULL;
  q = mp_get_symbolic_node (mp);
  set_ref_count (q, 0);
  r = NULL;
  /* Scan the token or variable to be defined;
    set |n|, |scanner_status|, and |warning_info| */
  if (m == start_def) {
    mp_get_clear_symbol (mp);
    mp->warning_info = cur_sym();
    get_t_next (mp);
    mp->scanner_status = op_defining;
    n = 0;
    set_eq_type (mp->warning_info, mp_defined_macro);
    set_equiv_node (mp->warning_info, q);
  } else { /* |var_def| */
    p = mp_scan_declared_variable (mp);
    mp_flush_variable (mp, equiv_node (mp_sym_sym (p)), mp_link (p), true);
    mp->warning_info_node = mp_find_variable (mp, p);
    mp_flush_node_list (mp, p);
    if (mp->warning_info_node == NULL) {
      /* Change to `\.{a bad variable}' */
      const char *hlp[] = {
         "After `vardef a' you can\'t say `vardef a.b'.",
         "So I'll have to discard this definition.",
         NULL };
      mp_error (mp, "This variable already starts with a macro", hlp, true);
      mp->warning_info_node = mp->bad_vardef;
    }
    mp->scanner_status = var_defining;
    n = 2;
    if (cur_cmd() == mp_macro_special && cur_mod() == macro_suffix) {    /* \.{\AT!\#} */
      n = 3;
      get_t_next (mp);
    }
    mp_type (mp->warning_info_node) = (quarterword) (mp_unsuffixed_macro - 2 + n);
    /* |mp_suffixed_macro=mp_unsuffixed_macro+1| */
    set_value_node (mp->warning_info_node, q);
  }

  k = n;
  if (cur_cmd() == mp_left_delimiter) {
    /* Absorb delimited parameters, putting them into lists |q| and |r| */
    do {
      l_delim = cur_sym();
      r_delim = equiv_sym (cur_sym());
      get_t_next (mp);
      if ((cur_cmd() == mp_param_type) && (cur_mod() == mp_expr_param)) {
        sym_type = mp_expr_sym;
      } else if ((cur_cmd() == mp_param_type) && (cur_mod() == mp_suffix_param)) {
        sym_type = mp_suffix_sym;
      } else if ((cur_cmd() == mp_param_type) && (cur_mod() == mp_text_param)) {
        sym_type = mp_text_sym;
      } else {
        const char *hlp[] = { "You should've had `expr' or `suffix' or `text' here.", NULL };
        mp_back_error (mp, "Missing parameter type; `expr' will be assumed", hlp, true);
        sym_type = mp_expr_sym;
      }
      /* Absorb parameter tokens for type |sym_type| */
      do {
        mp_link (q) = mp_get_symbolic_node (mp);
        q = mp_link (q);
        mp_name_type (q) = sym_type;
        set_mp_sym_info (q, k);
        mp_get_symbol (mp);
        rp = xmalloc (1, sizeof (mp_subst_list_item));
        rp->link = NULL;
        rp->value_data = k;
        rp->value_mod = sym_type;
        rp->info = cur_sym();
        rp->info_mod = cur_sym_mod();
        mp_check_param_size (mp, k);
        incr (k);
        rp->link = r;
        r = rp;
        get_t_next (mp);
      } while (cur_cmd() == mp_comma);
    
      mp_check_delimiter (mp, l_delim, r_delim);
      get_t_next (mp);
    } while (cur_cmd() == mp_left_delimiter);

  }
  if (cur_cmd() == mp_param_type) {
    /* Absorb undelimited parameters, putting them into list |r| */
    rp = xmalloc (1, sizeof (mp_subst_list_item));
    rp->link = NULL;
    rp->value_data = k;
    if (cur_mod() == mp_expr_param) {
      rp->value_mod = mp_expr_sym;
      c = mp_expr_macro;
    } else if (cur_mod() == mp_suffix_param) {
      rp->value_mod = mp_suffix_sym;
      c = mp_suffix_macro;
    } else if (cur_mod() == mp_text_param) {
      rp->value_mod = mp_text_sym;
      c = mp_text_macro;
    } else {
      c = cur_mod();
      rp->value_mod = mp_expr_sym;
    }
    mp_check_param_size (mp, k);
    incr (k);
    mp_get_symbol (mp);
    rp->info = cur_sym();
    rp->info_mod = cur_sym_mod();
    rp->link = r;
    r = rp;
    get_t_next (mp);
    if (c == mp_expr_macro) {
      if (cur_cmd() == mp_of_token) {
        c = mp_of_macro;
        rp = xmalloc (1, sizeof (mp_subst_list_item));
        rp->link = NULL;
        mp_check_param_size (mp, k);
        rp->value_data = k;
        rp->value_mod = mp_expr_sym;
        mp_get_symbol (mp);
        rp->info = cur_sym();
        rp->info_mod = cur_sym_mod();
        rp->link = r;
        r = rp;
        get_t_next (mp);
      }
    }
  }
  mp_check_equals (mp);
  p = mp_get_symbolic_node (mp);
  set_mp_sym_info (p, c);
  mp_name_type (p) = mp_macro_sym;
  mp_link (q) = p;
  /* Attach the replacement text to the tail of node |p| */
  /* We don't put `|mp->frozen_end_group|' into the replacement text of
     a \&{vardef}, because the user may want to redefine `\.{endgroup}'. */
  if (m == start_def) {
    mp_link (p) = mp_scan_toks (mp, mp_macro_def, r, NULL, (quarterword) n);
  } else {
    mp_node qq = mp_get_symbolic_node (mp);
    set_mp_sym_sym (qq, mp->bg_loc);
    mp_link (p) = qq;
    p = mp_get_symbolic_node (mp);
    set_mp_sym_sym (p, mp->eg_loc);
    mp_link (qq) = mp_scan_toks (mp, mp_macro_def, r, p, (quarterword) n);
  }
  if (mp->warning_info_node == mp->bad_vardef)
    mp_flush_token_list (mp, value_node (mp->bad_vardef));
  mp->scanner_status = normal;
  mp_get_x_next (mp);
}

@ @<Glob...@>=
mp_sym bg_loc;
mp_sym eg_loc;  /* hash addresses of `\.{begingroup}' and `\.{endgroup}' */

@ @<Initialize table entries@>=
mp->bad_vardef = mp_get_value_node (mp);
mp_name_type (mp->bad_vardef) = mp_root;
set_value_sym (mp->bad_vardef, mp->frozen_bad_vardef);

@ @<Free table entries@>=
mp_free_value_node (mp, mp->bad_vardef);


@* Expanding the next token.
Only a few command codes |<min_command| can possibly be returned by
|get_t_next|; in increasing order, they are
|if_test|, |fi_or_else|, |input|, |iteration|, |repeat_loop|,
|exit_test|, |relax|, |scan_tokens|, |run_script|, |expand_after|, and |defined_macro|.

\MP\ usually gets the next token of input by saying |get_x_next|. This is
like |get_t_next| except that it keeps getting more tokens until
finding |cur_cmd>=min_command|. In other words, |get_x_next| expands
macros and removes conditionals or iterations or input instructions that
might be present.

It follows that |get_x_next| might invoke itself recursively. In fact,
there is massive recursion, since macro expansion can involve the
scanning of arbitrarily complex expressions, which in turn involve
macro expansion and conditionals, etc.
@^recursion@>

Therefore it's necessary to declare a whole bunch of |forward|
procedures at this point, and to insert some other procedures
that will be invoked by |get_x_next|.

@<Declarations@>=
static void mp_scan_primary (MP mp);
static void mp_scan_secondary (MP mp);
static void mp_scan_tertiary (MP mp);
static void mp_scan_expression (MP mp);
static void mp_scan_suffix (MP mp);
static void mp_pass_text (MP mp);
static void mp_conditional (MP mp);
static void mp_start_input (MP mp);
static void mp_begin_iteration (MP mp);
static void mp_resume_iteration (MP mp);
static void mp_stop_iteration (MP mp);

@ A recursion depth counter is used to discover infinite recursions.
(Near) infinite recursion is a problem because it translates into 
C function calls that eat up the available call stack. A better solution
would be to depend on signal trapping, but that is problematic when
Metapost is used as a library. 

@<Global...@>=
int expand_depth_count; /* current expansion depth */
int expand_depth;       /* current expansion depth */

@ The limit is set at |10000|, which should be enough to allow 
normal usages of metapost while preventing the most obvious 
crashes on most all operating systems, but the value can be
raised if the runtime system allows a larger C stack.
@^system dependencies@>

@<Set initial...@>=
mp->expand_depth = 10000;

@ Even better would be if the system allows discovery of the amount of
space available on the call stack.
@^system dependencies@>

In any case, when the limit is crossed, that is a fatal error.

@d check_expansion_depth()  if (++mp->expand_depth_count >= mp->expand_depth)
                              mp_expansion_depth_error(mp)

@c
static void mp_expansion_depth_error (MP mp) {
    const char *hlp[] = {
         "Recursive macro expansion cannot be unlimited because of runtime",
         "stack constraints. The limit is 10000 recursion levels in total.",
         NULL };
    if ( mp->interaction==mp_error_stop_mode )
      mp->interaction=mp_scroll_mode; /* no more interaction */
    if ( mp->log_opened ) 
      mp_error(mp, "Maximum expansion depth reached", hlp, true);
    mp->history=mp_fatal_error_stop; 
    mp_jump_out(mp);
}


@ An auxiliary subroutine called |expand| is used by |get_x_next|
when it has to do exotic expansion commands.

@c
static void mp_expand (MP mp) {
  size_t k;     /* something that we hope is |<=buf_size| */
  size_t j;     /* index into |str_pool| */
  check_expansion_depth();
  if (number_greater (internal_value (mp_tracing_commands), unity_t))
    if (cur_cmd() != mp_defined_macro)
      show_cur_cmd_mod;
  switch (cur_cmd()) {
  case mp_if_test:
    mp_conditional (mp);        /* this procedure is discussed in Part 36 below */
    break;
  case mp_fi_or_else:
    @<Terminate the current conditional and skip to \&{fi}@>;
    break;
  case mp_input:
    @<Initiate or terminate input from a file@>;
    break;
  case mp_iteration:
    if (cur_mod() == end_for) {
      @<Scold the user for having an extra \&{endfor}@>;
    } else {
      mp_begin_iteration (mp);  /* this procedure is discussed in Part 37 below */
    }
    break;
  case mp_repeat_loop:
    @<Repeat a loop@>;
    break;
  case mp_exit_test:
    @<Exit a loop if the proper time has come@>;
    break;
  case mp_relax:
    break;
  case mp_expand_after:
    @<Expand the token after the next token@>;
    break;
  case mp_scan_tokens:
    @<Put a string into the input buffer@>;
    break;
  case mp_runscript:
    @<Put a script result string into the input buffer@>;
    break;
  case mp_defined_macro:
    mp_macro_call (mp, cur_mod_node(), NULL, cur_sym());
    break;
  default:
    break; /* make the compiler happy */
  };                            /* there are no other cases */
  mp->expand_depth_count--;
}


@ @<Scold the user...@>=
{
  const char *hlp[] = {
         "I'm not currently working on a for loop,",
         "so I had better not try to end anything.",
         NULL };
  mp_error (mp, "Extra `endfor'", hlp, true);
@.Extra `endfor'@>;
}


@ The processing of \&{input} involves the |start_input| subroutine,
which will be declared later; the processing of \&{endinput} is trivial.

@<Put each...@>=
mp_primitive (mp, "input", mp_input, 0);
@:input_}{\&{input} primitive@>;
mp_primitive (mp, "endinput", mp_input, 1);
@:end_input_}{\&{endinput} primitive@>
 

@ @<Cases of |print_cmd_mod|...@>=
case mp_input:
if (m == 0)
  mp_print (mp, "input");
else
  mp_print (mp, "endinput");
break;

@ @<Initiate or terminate input...@>=
if (cur_mod() > 0)
  mp->force_eof = true;
else
  mp_start_input (mp)
   

@ We'll discuss the complicated parts of loop operations later. For now
it suffices to know that there's a global variable called |loop_ptr|
that will be |NULL| if no loop is in progress.

@<Repeat a loop@>=
{
  while (token_state && (nloc == NULL))
    mp_end_token_list (mp);     /* conserve stack space */
  if (mp->loop_ptr == NULL) {
    const char *hlp[] = {
           "I'm confused; after exiting from a loop, I still seem",
           "to want to repeat it. I'll try to forget the problem.",
           NULL };
    mp_error (mp, "Lost loop", hlp, true);
@.Lost loop@>;
  } else {
    mp_resume_iteration (mp);   /* this procedure is in Part 37 below */
  }
}


@ @<Exit a loop if the proper time has come@>=
{
  mp_get_boolean (mp);
  if (number_greater (internal_value (mp_tracing_commands),  unity_t))
    mp_show_cmd_mod (mp, mp_nullary, cur_exp_value_boolean ());
  if (cur_exp_value_boolean () == mp_true_code) {
    if (mp->loop_ptr == NULL) {
      const char *hlp[] = {
          "Why say `exitif' when there's nothing to exit from?", 
          NULL };
      if (cur_cmd() == mp_semicolon)
        mp_error (mp, "No loop is in progress", hlp, true);
      else
        mp_back_error (mp, "No loop is in progress", hlp, true);
@.No loop is in progress@>;
    } else {
      @<Exit prematurely from an iteration@>;
    }
  } else if (cur_cmd() != mp_semicolon) {
    const char *hlp[] = {
           "After `exitif <boolean exp>' I expect to see a semicolon.",
           "I shall pretend that one was there.",
           NULL };
    mp_back_error (mp, "Missing `;' has been inserted", hlp, true);
@.Missing `;'@>;
  }
}


@ Here we use the fact that |forever_text| is the only |token_type| that
is less than |loop_text|.

@<Exit prematurely...@>=
{
  mp_node p = NULL;
  do {
    if (file_state) {
      mp_end_file_reading (mp);
    } else {
      if (token_type <= loop_text)
        p = nstart;
      mp_end_token_list (mp);
    }
  } while (p == NULL);
  if (p != mp->loop_ptr->info)
    mp_fatal_error (mp, "*** (loop confusion)");
@.loop confusion@>;
  mp_stop_iteration (mp);       /* this procedure is in Part 34 below */
}


@ @<Expand the token after the next token@>=
{
  mp_node p;
  get_t_next (mp);
  p = mp_cur_tok (mp);
  get_t_next (mp);
  if (cur_cmd() < mp_min_command)
    mp_expand (mp);
  else
    mp_back_input (mp);
  back_list (p);
}


@ @<Put a string into the input buffer@>=
{
  mp_get_x_next (mp);
  mp_scan_primary (mp);
  if (mp->cur_exp.type != mp_string_type) {
    mp_value new_expr;
    const char *hlp[] = {
           "I'm going to flush this expression, since",
           "scantokens should be followed by a known string.",
           NULL };
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    mp_disp_err (mp, NULL);
    mp_back_error (mp, "Not a string", hlp, true);
@.Not a string@>;
    mp_get_x_next (mp);
    mp_flush_cur_exp (mp, new_expr);
  } else {
    mp_back_input (mp);
    if (cur_exp_str ()->len > 0)
      @<Pretend we're reading a new one-line file@>;
  }
}

@ @<Put a script result string into the input buffer@>=
{
    mp_get_x_next (mp);
    mp_scan_primary (mp);
    if (mp->cur_exp.type != mp_string_type) {
        mp_value new_expr;
        const char *hlp[] = {
           "I'm going to flush this expression, since",
           "runscript should be followed by a known string.",
           NULL };
        memset(&new_expr,0,sizeof(mp_value));
        new_number(new_expr.data.n);
        mp_disp_err (mp, NULL);
        mp_back_error (mp, "Not a string", hlp, true);
@.Not a string@>;
        mp_get_x_next (mp);
        mp_flush_cur_exp (mp, new_expr);
    } else {
        mp_back_input (mp);
        if (cur_exp_str ()->len > 0) {
            mp_value new_expr;
            char *s = mp->run_script(mp,(const char*) cur_exp_str()->str) ;
            if (s != NULL) {
                size_t size = strlen(s);
                memset(&new_expr,0,sizeof(mp_value));
                new_number(new_expr.data.n);
                mp_begin_file_reading (mp);
                name = is_scantok;
                mp->last = mp->first;
                k = mp->first + size;
                if (k >= mp->max_buf_stack) {
                    while (k >= mp->buf_size) {
                        mp_reallocate_buffer (mp, (mp->buf_size + (mp->buf_size / 4)));
                    }
                    mp->max_buf_stack = k + 1;
                }
                limit = (halfword) k;
                (void) memcpy ((mp->buffer + mp->first), s, size);
                free(s);
                mp->buffer[limit] = xord ('%');
                mp->first = (size_t) (limit + 1);
                loc = start;
                mp_flush_cur_exp (mp, new_expr);
            }
        }
    }
}

@ @<Pretend we're reading a new one-line file@>=
{
  mp_value new_expr;
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  mp_begin_file_reading (mp);
  name = is_scantok;
  k = mp->first + (size_t) cur_exp_str ()->len;
  if (k >= mp->max_buf_stack) {
    while (k >= mp->buf_size) {
      mp_reallocate_buffer (mp, (mp->buf_size + (mp->buf_size / 4)));
    }
    mp->max_buf_stack = k + 1;
  }
  j = 0;
  limit = (halfword) k;
  while (mp->first < (size_t) limit) {
    mp->buffer[mp->first] = *(cur_exp_str ()->str + j);
    j++;
    incr (mp->first);
  }
  mp->buffer[limit] = xord ('%');
  mp->first = (size_t) (limit + 1);
  loc = start;
  mp_flush_cur_exp (mp, new_expr);
}


@ Here finally is |get_x_next|.

The expression scanning routines to be considered later
communicate via the global quantities |cur_type| and |cur_exp|;
we must be very careful to save and restore these quantities while
macros are being expanded.
@^inner loop@>

@<Declarations@>=
static void mp_get_x_next (MP mp);

@ @c
void mp_get_x_next (MP mp) {
  mp_node save_exp;     /* a capsule to save |cur_type| and |cur_exp| */
  get_t_next (mp);
  if (cur_cmd() < mp_min_command) {
    save_exp = mp_stash_cur_exp (mp);
    do {
      if (cur_cmd() == mp_defined_macro)
        mp_macro_call (mp, cur_mod_node(), NULL, cur_sym());
      else
        mp_expand (mp);
      get_t_next (mp);
    } while (cur_cmd() < mp_min_command);
    mp_unstash_cur_exp (mp, save_exp);  /* that restores |cur_type| and |cur_exp| */
  }
}


@ Now let's consider the |macro_call| procedure, which is used to start up
all user-defined macros. Since the arguments to a macro might be expressions,
|macro_call| is recursive.
@^recursion@>

The first parameter to |macro_call| points to the reference count of the
token list that defines the macro. The second parameter contains any
arguments that have already been parsed (see below).  The third parameter
points to the symbolic token that names the macro. If the third parameter
is |NULL|, the macro was defined by \&{vardef}, so its name can be
reconstructed from the prefix and ``at'' arguments found within the
second parameter.

What is this second parameter? It's simply a linked list of symbolic items,
whose |info| fields point to the arguments. In other words, if |arg_list=NULL|,
no arguments have been scanned yet; otherwise |mp_info(arg_list)| points to
the first scanned argument, and |mp_link(arg_list)| points to the list of
further arguments (if any).

Arguments of type \&{expr} are so-called capsules, which we will
discuss later when we concentrate on expressions; they can be
recognized easily because their |link| field is |void|. Arguments of type
\&{suffix} and \&{text} are token lists without reference counts.

@ After argument scanning is complete, the arguments are moved to the
|param_stack|. (They can't be put on that stack any sooner, because
the stack is growing and shrinking in unpredictable ways as more arguments
are being acquired.)  Then the macro body is fed to the scanner; i.e.,
the replacement text of the macro is placed at the top of the \MP's
input stack, so that |get_t_next| will proceed to read it next.

@<Declarations@>=
static void mp_macro_call (MP mp, mp_node def_ref, mp_node arg_list,
                           mp_sym macro_name);

@ @c
void mp_macro_call (MP mp, mp_node def_ref, mp_node arg_list, mp_sym macro_name) {
  /* invokes a user-defined control sequence */
  mp_node r;    /* current node in the macro's token list */
  mp_node p, q; /* for list manipulation */
  integer n;    /* the number of arguments */
  mp_node tail = 0;     /* tail of the argument list */
  mp_sym l_delim = NULL, r_delim = NULL;        /* a delimiter pair */
  r = mp_link (def_ref);
  add_mac_ref (def_ref);
  if (arg_list == NULL) {
    n = 0;
  } else {
    @<Determine the number |n| of arguments already supplied,
    and set |tail| to the tail of |arg_list|@>;
  }
  if (number_positive (internal_value (mp_tracing_macros))) {
    @<Show the text of the macro being expanded, and the existing arguments@>;
  }
  @<Scan the remaining arguments, if any; set |r| to the first token
    of the replacement text@>;
  @<Feed the arguments and replacement text to the scanner@>;
}


@ @<Show the text of the macro...@>=
mp_begin_diagnostic (mp);
mp_print_ln (mp);
mp_print_macro_name (mp, arg_list, macro_name);
if (n == 3)
  mp_print (mp, "@@#");         /* indicate a suffixed macro */
mp_show_macro (mp, def_ref, NULL, 100000);
if (arg_list != NULL) {
  n = 0;
  p = arg_list;
  do {
    q = (mp_node)mp_sym_sym (p);
    mp_print_arg (mp, q, n, 0, 0);
    incr (n);
    p = mp_link (p);
  } while (p != NULL);
}
mp_end_diagnostic (mp, false)
 

@ @<Declarations@>=
static void mp_print_macro_name (MP mp, mp_node a, mp_sym n);

@ @c
void mp_print_macro_name (MP mp, mp_node a, mp_sym n) {
  mp_node p, q; /* they traverse the first part of |a| */
  if (n != NULL) {
    mp_print_text (n);
  } else {
    p = (mp_node)mp_sym_sym (a);
    if (p == NULL) {
      mp_print_text (mp_sym_sym ((mp_node)mp_sym_sym (mp_link (a))));
    } else {
      q = p;
      while (mp_link (q) != NULL)
        q = mp_link (q);
      mp_link (q) = (mp_node)mp_sym_sym (mp_link (a));
      mp_show_token_list (mp, p, NULL, 1000, 0);
      mp_link (q) = NULL;
    }
  }
}


@ @<Declarations@>=
static void mp_print_arg (MP mp, mp_node q, integer n, halfword b,
                          quarterword bb);

@ @c
void mp_print_arg (MP mp, mp_node q, integer n, halfword b, quarterword bb) {
  if (q && mp_link (q) == MP_VOID) {
    mp_print_nl (mp, "(EXPR");
  } else {
    if ((bb < mp_text_sym) && (b != mp_text_macro))
      mp_print_nl (mp, "(SUFFIX");
    else
      mp_print_nl (mp, "(TEXT");
  }
  mp_print_int (mp, n);
  mp_print (mp, ")<-");
  if (q && mp_link (q) == MP_VOID)
    mp_print_exp (mp, q, 1);
  else
    mp_show_token_list (mp, q, NULL, 1000, 0);
}


@ @<Determine the number |n| of arguments already supplied...@>=
{
  n = 1;
  tail = arg_list;
  while (mp_link (tail) != NULL) {
    incr (n);
    tail = mp_link (tail);
  }
}


@ @<Scan the remaining arguments, if any; set |r|...@>=
set_cur_cmd(mp_comma + 1);        /* anything |<>comma| will do */
while (mp_name_type (r) == mp_expr_sym ||
       mp_name_type (r) == mp_suffix_sym || mp_name_type (r) == mp_text_sym) {
  @<Scan the delimited argument represented by |mp_sym_info(r)|@>;
  r = mp_link (r);
}
if (cur_cmd() == mp_comma) {
  char msg[256];
  const char *hlp[] = {
         "I'm going to assume that the comma I just read was a",
         "right delimiter, and then I'll begin expanding the macro.",
         "You might want to delete some tokens before continuing.",
         NULL };
  mp_string rname;
  int old_setting = mp->selector;
  mp->selector = new_string;
  mp_print_macro_name (mp, arg_list, macro_name);
  rname = mp_make_string(mp);
  mp->selector = old_setting;
  mp_snprintf (msg, 256, "Too many arguments to %s; Missing `%s' has been inserted",
	       mp_str(mp, rname), mp_str(mp, text(r_delim)));
  delete_str_ref(rname);
@.Too many arguments...@>;
@.Missing `)'...@>;
  mp_error (mp, msg, hlp, true);
}
if (mp_sym_info (r) != mp_general_macro) {
  @<Scan undelimited argument(s)@>;
}
r = mp_link (r)
 

@ At this point, the reader will find it advisable to review the explanation
of token list format that was presented earlier, paying special attention to
the conventions that apply only at the beginning of a macro's token list.

On the other hand, the reader will have to take the expression-parsing
aspects of the following program on faith; we will explain |cur_type|
and |cur_exp| later. (Several things in this program depend on each other,
and it's necessary to jump into the circle somewhere.)

@<Scan the delimited argument represented by |mp_sym_info(r)|@>=
if (cur_cmd() != mp_comma) {
  mp_get_x_next (mp);
  if (cur_cmd() != mp_left_delimiter) {
    char msg[256];
    const char *hlp[] = {
           "That macro has more parameters than you thought.",
           "I'll continue by pretending that each missing argument",
           "is either zero or null.",
           NULL };
    mp_string sname;
    int old_setting = mp->selector;
    mp->selector = new_string;    
    mp_print_macro_name (mp, arg_list, macro_name);
    sname = mp_make_string(mp);
    mp->selector = old_setting;    
    mp_snprintf (msg, 256, "Missing argument to %s", mp_str(mp, sname));
@.Missing argument...@>;
    delete_str_ref(sname);    
    if (mp_name_type (r) == mp_suffix_sym || mp_name_type (r) == mp_text_sym) {
      set_cur_exp_value_number (zero_t);  /* todo: this was |null| */
      mp->cur_exp.type = mp_token_list;
    } else {
      set_cur_exp_value_number (zero_t);
      mp->cur_exp.type = mp_known;
    }
    mp_back_error (mp, msg, hlp, true);
    set_cur_cmd((mp_variable_type)mp_right_delimiter);
    goto FOUND;
  }
  l_delim = cur_sym();
  r_delim = equiv_sym (cur_sym());
}
@<Scan the argument represented by |mp_sym_info(r)|@>;
if (cur_cmd() != mp_comma)
  @<Check that the proper right delimiter was present@>;
FOUND:
@<Append the current expression to |arg_list|@>
 

@ @<Check that the proper right delim...@>=
if ((cur_cmd() != mp_right_delimiter) || (equiv_sym (cur_sym()) != l_delim)) {
  if (mp_name_type (mp_link (r)) == mp_expr_sym ||
      mp_name_type (mp_link (r)) == mp_suffix_sym ||
      mp_name_type (mp_link (r)) == mp_text_sym) {
    const char *hlp[] = {
           "I've finished reading a macro argument and am about to",
           "read another; the arguments weren't delimited correctly.",
           "You might want to delete some tokens before continuing.",
           NULL };
    mp_back_error (mp, "Missing `,' has been inserted", hlp, true);
@.Missing `,'@>;
    set_cur_cmd((mp_variable_type)mp_comma);
  } else {
    char msg[256];
    const char *hlp[] = {
           "I've gotten to the end of the macro parameter list.",
           "You might want to delete some tokens before continuing.",
           NULL };
    mp_snprintf(msg, 256, "Missing `%s' has been inserted", mp_str(mp, text(r_delim)));
@.Missing `)'@>;
    mp_back_error (mp, msg, hlp, true);
  }
}

@ A \&{suffix} or \&{text} parameter will have been scanned as
a token list pointed to by |cur_exp|, in which case we will have
|cur_type=token_list|.

@<Append the current expression to |arg_list|@>=
{
  p = mp_get_symbolic_node (mp);
  if (mp->cur_exp.type == mp_token_list)
    set_mp_sym_sym (p, mp->cur_exp.data.node);
  else
    set_mp_sym_sym (p, mp_stash_cur_exp (mp));
  if (number_positive (internal_value (mp_tracing_macros))) {
    mp_begin_diagnostic (mp);
    mp_print_arg (mp, (mp_node)mp_sym_sym (p), n, mp_sym_info (r), mp_name_type (r));
    mp_end_diagnostic (mp, false);
  }
  if (arg_list == NULL) {
    arg_list = p;
  } else {
    mp_link (tail) = p;
  }
  tail = p;
  incr (n);
}


@ @<Scan the argument represented by |mp_sym_info(r)|@>=
if (mp_name_type (r) == mp_text_sym) {
  mp_scan_text_arg (mp, l_delim, r_delim);
} else {
  mp_get_x_next (mp);
  if (mp_name_type (r) == mp_suffix_sym)
    mp_scan_suffix (mp);
  else
    mp_scan_expression (mp);
}


@ The parameters to |scan_text_arg| are either a pair of delimiters
or zero; the latter case is for undelimited text arguments, which
end with the first semicolon or \&{endgroup} or \&{end} that is not
contained in a group.

@<Declarations@>=
static void mp_scan_text_arg (MP mp, mp_sym l_delim, mp_sym r_delim);

@ @c
void mp_scan_text_arg (MP mp, mp_sym l_delim, mp_sym r_delim) {
  integer balance;      /* excess of |l_delim| over |r_delim| */
  mp_node p;    /* list tail */
  mp->warning_info = l_delim;
  mp->scanner_status = absorbing;
  p = mp->hold_head;
  balance = 1;
  mp_link (mp->hold_head) = NULL;
  while (1) {
    get_t_next (mp);
    if (l_delim == NULL) {
      @<Adjust the balance for an undelimited argument; |break| if done@>;
    } else {
      @<Adjust the balance for a delimited argument; |break| if done@>;
    }
    mp_link (p) = mp_cur_tok (mp);
    p = mp_link (p);
  }
  set_cur_exp_node (mp_link (mp->hold_head));
  mp->cur_exp.type = mp_token_list;
  mp->scanner_status = normal;
}


@ @<Adjust the balance for a delimited argument...@>=
if (cur_cmd() == mp_right_delimiter) {
  if (equiv_sym (cur_sym()) == l_delim) {
    decr (balance);
    if (balance == 0)
      break;
  }
} else if (cur_cmd() == mp_left_delimiter) {
  if (equiv_sym (cur_sym()) == r_delim)
    incr (balance);
}

@ @<Adjust the balance for an undelimited...@>=
if (mp_end_of_statement) {         /* |cur_cmd=semicolon|, |end_group|, or |stop| */
  if (balance == 1) {
    break;
  } else {
    if (cur_cmd() == mp_end_group)
      decr (balance);
  }
} else if (cur_cmd() == mp_begin_group) {
  incr (balance);
}

@ @<Scan undelimited argument(s)@>=
{
  if (mp_sym_info (r) < mp_text_macro) {
    mp_get_x_next (mp);
    if (mp_sym_info (r) != mp_suffix_macro) {
      if ((cur_cmd() == mp_equals) || (cur_cmd() == mp_assignment))
        mp_get_x_next (mp);
    }
  }
  switch (mp_sym_info (r)) {
  case mp_primary_macro:
    mp_scan_primary (mp);
    break;
  case mp_secondary_macro:
    mp_scan_secondary (mp);
    break;
  case mp_tertiary_macro:
    mp_scan_tertiary (mp);
    break;
  case mp_expr_macro:
    mp_scan_expression (mp);
    break;
  case mp_of_macro:
    @<Scan an expression followed by `\&{of} $\langle$primary$\rangle$'@>;
    break;
  case mp_suffix_macro:
    @<Scan a suffix with optional delimiters@>;
    break;
  case mp_text_macro:
    mp_scan_text_arg (mp, NULL, NULL);
    break;
  }                             /* there are no other cases */
  mp_back_input (mp);
  @<Append the current expression to |arg_list|@>;
}


@ @<Scan an expression followed by `\&{of} $\langle$primary$\rangle$'@>=
{
  mp_scan_expression (mp);
  p = mp_get_symbolic_node (mp);
  set_mp_sym_sym (p, mp_stash_cur_exp (mp));
  if (number_positive (internal_value (mp_tracing_macros))) {
    mp_begin_diagnostic (mp);
    mp_print_arg (mp, (mp_node)mp_sym_sym (p), n, 0, 0);
    mp_end_diagnostic (mp, false);
  }
  if (arg_list == NULL)
    arg_list = p;
  else
    mp_link (tail) = p;
  tail = p;
  incr (n);
  if (cur_cmd() != mp_of_token) {
    char msg[256];
    mp_string sname;
    const char *hlp[] = { 
        "I've got the first argument; will look now for the other.",
        NULL };
    int old_setting = mp->selector;
    mp->selector = new_string;
    mp_print_macro_name (mp, arg_list, macro_name);
    sname = mp_make_string(mp);
    mp->selector = old_setting;
    mp_snprintf (msg, 256, "Missing `of' has been inserted for %s", mp_str(mp, sname));
    delete_str_ref(sname);
@.Missing `of'@>;
    mp_back_error (mp, msg, hlp, true);
  }
  mp_get_x_next (mp);
  mp_scan_primary (mp);
}


@ @<Scan a suffix with optional delimiters@>=
{
  if (cur_cmd() != mp_left_delimiter) {
    l_delim = NULL;
  } else {
    l_delim = cur_sym();
    r_delim = equiv_sym (cur_sym());
    mp_get_x_next (mp);
  }
  mp_scan_suffix (mp);
  if (l_delim != NULL) {
    if ((cur_cmd() != mp_right_delimiter) || (equiv_sym (cur_sym()) != l_delim)) {
      char msg[256];
      const char *hlp[] = { 
        "I've gotten to the end of the macro parameter list.",
        "You might want to delete some tokens before continuing.",
        NULL };
      mp_snprintf(msg, 256, "Missing `%s' has been inserted", mp_str (mp, text (r_delim)));
@.Missing `)'@>;
      mp_back_error (mp, msg, hlp, true);
    }
    mp_get_x_next (mp);
  }
}


@ Before we put a new token list on the input stack, it is wise to clean off
all token lists that have recently been depleted. Then a user macro that ends
with a call to itself will not require unbounded stack space.

@<Feed the arguments and replacement text to the scanner@>=
while (token_state && (nloc == NULL))
  mp_end_token_list (mp);       /* conserve stack space */
if (mp->param_ptr + n > mp->max_param_stack) {
  mp->max_param_stack = mp->param_ptr + n;
  mp_check_param_size (mp, mp->max_param_stack);
@:MetaPost capacity exceeded parameter stack size}{\quad parameter stack size@>
}
mp_begin_token_list (mp, def_ref, (quarterword) macro);
if (macro_name)
  name = text (macro_name);
else
  name = NULL;
nloc = r;
if (n > 0) {
  p = arg_list;
  do {
    mp->param_stack[mp->param_ptr] = (mp_node)mp_sym_sym (p);
    incr (mp->param_ptr);
    p = mp_link (p);
  } while (p != NULL);
  mp_flush_node_list (mp, arg_list);
}

@ It's sometimes necessary to put a single argument onto |param_stack|.
The |stack_argument| subroutine does this.

@c
static void mp_stack_argument (MP mp, mp_node p) {
  if (mp->param_ptr == mp->max_param_stack) {
    incr (mp->max_param_stack);
    mp_check_param_size (mp, mp->max_param_stack);
  }
  mp->param_stack[mp->param_ptr] = p;
  incr (mp->param_ptr);
}


@* Conditional processing.
Let's consider now the way \&{if} commands are handled.

Conditions can be inside conditions, and this nesting has a stack
that is independent of other stacks.
Four global variables represent the top of the condition stack:
|cond_ptr| points to pushed-down entries, if~any; |cur_if| tells whether
we are processing \&{if} or \&{elseif}; |if_limit| specifies
the largest code of a |fi_or_else| command that is syntactically legal;
and |if_line| is the line number at which the current conditional began.

If no conditions are currently in progress, the condition stack has the
special state |cond_ptr=NULL|, |if_limit=normal|, |cur_if=0|, |if_line=0|.
Otherwise |cond_ptr| points to a non-symbolic node; the |type|, |name_type|, and
|link| fields of the first word contain |if_limit|, |cur_if|, and
|cond_ptr| at the next level, and the second word contains the
corresponding |if_line|.

@d if_line_field(A) ((mp_if_node)(A))->if_line_field_
@d if_code 1 /* code for \&{if} being evaluated */
@d fi_code 2 /* code for \&{fi} */
@d else_code 3 /* code for \&{else} */
@d else_if_code 4 /* code for \&{elseif} */

@<MPlib internal header stuff@>=
typedef struct mp_if_node_data {
  NODE_BODY;
  int if_line_field_;
} mp_if_node_data;
typedef struct mp_if_node_data *mp_if_node;

@
@d if_node_size sizeof(struct mp_if_node_data) /* number of words in stack entry for conditionals */

@c
static mp_node mp_get_if_node (MP mp) {
  mp_if_node p = (mp_if_node) malloc_node (if_node_size);
  mp_type (p) = mp_if_node_type;
  return (mp_node) p;
}


@ @<Glob...@>=
mp_node cond_ptr;       /* top of the condition stack */
integer if_limit;       /* upper bound on |fi_or_else| codes */
quarterword cur_if;     /* type of conditional being worked on */
integer if_line;        /* line where that conditional began */

@ @<Set init...@>=
mp->cond_ptr = NULL;
mp->if_limit = normal;
mp->cur_if = 0;
mp->if_line = 0;

@ @<Put each...@>=
mp_primitive (mp, "if", mp_if_test, if_code);
@:if_}{\&{if} primitive@>;
mp_primitive (mp, "fi", mp_fi_or_else, fi_code);
mp->frozen_fi = mp_frozen_primitive (mp, "fi", mp_fi_or_else, fi_code);
@:fi_}{\&{fi} primitive@>;
mp_primitive (mp, "else", mp_fi_or_else, else_code);
@:else_}{\&{else} primitive@>;
mp_primitive (mp, "elseif", mp_fi_or_else, else_if_code);
@:else_if_}{\&{elseif} primitive@>
 

@ @<Cases of |print_cmd_mod|...@>=
case mp_if_test:
case mp_fi_or_else:
switch (m) {
case if_code:
  mp_print (mp, "if");
  break;
case fi_code:
  mp_print (mp, "fi");
  break;
case else_code:
  mp_print (mp, "else");
  break;
default:
  mp_print (mp, "elseif");
  break;
}
break;

@ Here is a procedure that ignores text until coming to an \&{elseif},
\&{else}, or \&{fi} at level zero of $\&{if}\ldots\&{fi}$
nesting. After it has acted, |cur_mod| will indicate the token that
was found.

\MP's smallest two command codes are |if_test| and |fi_or_else|; this
makes the skipping process a bit simpler.

@c
void mp_pass_text (MP mp) {
  integer l = 0;
  mp->scanner_status = skipping;
  mp->warning_line = mp_true_line (mp);
  while (1) {
    get_t_next (mp);
    if (cur_cmd() <= mp_fi_or_else) {
      if (cur_cmd() < mp_fi_or_else) {
        incr (l);
      } else {
        if (l == 0)
          break;
        if (cur_mod() == fi_code)
          decr (l);
      }
    } else {
      @<Decrease the string reference count,
       if the current token is a string@>;
    }
  }
  mp->scanner_status = normal;
}


@ @<Decrease the string reference count...@>=
if (cur_cmd() == mp_string_token) {
  delete_str_ref (cur_mod_str());
}

@ When we begin to process a new \&{if}, we set |if_limit:=if_code|; then
if \&{elseif} or \&{else} or \&{fi} occurs before the current \&{if}
condition has been evaluated, a colon will be inserted.
A construction like `\.{if fi}' would otherwise get \MP\ confused.

@<Push the condition stack@>=
{
  p = mp_get_if_node (mp);
  mp_link (p) = mp->cond_ptr;
  mp_type (p) = (quarterword) mp->if_limit;
  mp_name_type (p) = mp->cur_if;
  if_line_field (p) = mp->if_line;
  mp->cond_ptr = p;
  mp->if_limit = if_code;
  mp->if_line = mp_true_line (mp);
  mp->cur_if = if_code;
}


@ @<Pop the condition stack@>=
{
  mp_node p = mp->cond_ptr;
  mp->if_line = if_line_field (p);
  mp->cur_if = mp_name_type (p);
  mp->if_limit = mp_type (p);
  mp->cond_ptr = mp_link (p);
  mp_free_node (mp, p, if_node_size);
}


@ Here's a procedure that changes the |if_limit| code corresponding to
a given value of |cond_ptr|.

@c
static void mp_change_if_limit (MP mp, quarterword l, mp_node p) {
  mp_node q;
  if (p == mp->cond_ptr) {
    mp->if_limit = l;           /* that's the easy case */
  } else {
    q = mp->cond_ptr;
    while (1) {
      if (q == NULL)
        mp_confusion (mp, "if");
@:this can't happen if}{\quad if@>;
      /* clang: dereference of null pointer */ assert(q);
      if (mp_link (q) == p) {
        mp_type (q) = l;
        return;
      }
      q = mp_link (q);
    }
  }
}


@ The user is supposed to put colons into the proper parts of conditional
statements. Therefore, \MP\ has to check for their presence.

@c
static void mp_check_colon (MP mp) {
  if (cur_cmd() != mp_colon) {
    const char *hlp[] = {
           "There should've been a colon after the condition.",
           "I shall pretend that one was there.",
           NULL }; 
    mp_back_error (mp, "Missing `:' has been inserted", hlp, true);
@.Missing `:'@>;
  }
}


@ A condition is started when the |get_x_next| procedure encounters
an |if_test| command; in that case |get_x_next| calls |conditional|,
which is a recursive procedure.
@^recursion@>

@c
void mp_conditional (MP mp) {
  mp_node save_cond_ptr;        /* |cond_ptr| corresponding to this conditional */
  int new_if_limit;     /* future value of |if_limit| */
  mp_node p;    /* temporary register */
  @<Push the condition stack@>;
  save_cond_ptr = mp->cond_ptr;
RESWITCH:
  mp_get_boolean (mp);
  new_if_limit = else_if_code;
  if (number_greater (internal_value (mp_tracing_commands), unity_t)) {
    @<Display the boolean value of |cur_exp|@>;
  }
FOUND:
  mp_check_colon (mp);
  if (cur_exp_value_boolean () == mp_true_code) {
    mp_change_if_limit (mp, (quarterword) new_if_limit, save_cond_ptr);
    return;                     /* wait for \&{elseif}, \&{else}, or \&{fi} */
  };
  @<Skip to \&{elseif} or \&{else} or \&{fi}, then |goto done|@>;
DONE:
  mp->cur_if = (quarterword) cur_mod();
  mp->if_line = mp_true_line (mp);
  if (cur_mod() == fi_code) {
    @<Pop the condition stack@>
  } else if (cur_mod() == else_if_code) {
    goto RESWITCH;
  } else {
    set_cur_exp_value_boolean (mp_true_code);
    new_if_limit = fi_code;
    mp_get_x_next (mp);
    goto FOUND;
  }
}


@ In a construction like `\&{if} \&{if} \&{true}: $0=1$: \\{foo}
\&{else}: \\{bar} \&{fi}', the first \&{else}
that we come to after learning that the \&{if} is false is not the
\&{else} we're looking for. Hence the following curious logic is needed.

@<Skip to \&{elseif}...@>=
while (1) {
  mp_pass_text (mp);
  if (mp->cond_ptr == save_cond_ptr)
    goto DONE;
  else if (cur_mod() == fi_code)
    @<Pop the condition stack@>;
}


@ @<Display the boolean value...@>=
{
  mp_begin_diagnostic (mp);
  if (cur_exp_value_boolean () == mp_true_code)
    mp_print (mp, "{true}");
  else
    mp_print (mp, "{false}");
  mp_end_diagnostic (mp, false);
}


@ The processing of conditionals is complete except for the following
code, which is actually part of |get_x_next|. It comes into play when
\&{elseif}, \&{else}, or \&{fi} is scanned.

@<Terminate the current conditional and skip to \&{fi}@>=
if (cur_mod() > mp->if_limit) {
  if (mp->if_limit == if_code) {        /* condition not yet evaluated */
    const char *hlp[] = { "Something was missing here", NULL };
    mp_back_input (mp);
    set_cur_sym(mp->frozen_colon);
    mp_ins_error (mp, "Missing `:' has been inserted", hlp, true);
@.Missing `:'@>;
  } else {
    const char *hlp[] =  {"I'm ignoring this; it doesn't match any if.", NULL};
    if (cur_mod() == fi_code) {
       mp_error(mp, "Extra fi", hlp, true);
@.Extra fi@>;
    } else if (cur_mod() == else_code) {
       mp_error(mp, "Extra else", hlp, true);
@.Extra else@>
    } else {
       mp_error(mp, "Extra elseif", hlp, true);
@.Extra elseif@>
    }
  }
} else {
  while (cur_mod() != fi_code)
    mp_pass_text (mp);          /* skip to \&{fi} */
  @<Pop the condition stack@>;
}


@* Iterations.
To bring our treatment of |get_x_next| to a close, we need to consider what
\MP\ does when it sees \&{for}, \&{forsuffixes}, and \&{forever}.

There's a global variable |loop_ptr| that keeps track of the \&{for} loops
that are currently active. If |loop_ptr=NULL|, no loops are in progress;
otherwise |loop_ptr.info| points to the iterative text of the current
(innermost) loop, and |loop_ptr.link| points to the data for any other
loops that enclose the current one.

A loop-control node also has two other fields, called |type| and
|list|, whose contents depend on the type of loop:

\yskip\indent|loop_ptr.type=NULL| means that the link of |loop_ptr.list|
points to a list of symbolic nodes whose |info| fields point to the
remaining argument values of a suffix list and expression list.
In this case, an extra field |loop_ptr.start_list| is needed to
make sure that |resume_operation| skips ahead.

\yskip\indent|loop_ptr.type=MP_VOID| means that the current loop is
`\&{forever}'.

\yskip\indent|loop_ptr.type=PROGRESSION_FLAG| means that
|loop_ptr.value|, |loop_ptr.step_size|, and |loop_ptr.final_value|
contain the data for an arithmetic progression.

\yskip\indent|loop_ptr.type=p>PROGRESSION_FLAG| means that |p| points to an edge
header and |loop_ptr.list| points into the graphical object list for
that edge header.

@d PROGRESSION_FLAG (mp_node)(2) /* |NULL+2| */
  /* |loop_type| value when |loop_list| points to a progression node */

@<Types...@>=
typedef struct mp_loop_data {
  mp_node info; /* iterative text of this loop */
  mp_node type; /* the special type of this loop, or a pointer into
                   mem */
  mp_node list; /* the remaining list elements */
  mp_node list_start;   /* head fo the list of elements */
  mp_number value; /* current arithmetic value */
  mp_number step_size;     /* arithmetic step size */
  mp_number final_value;   /* end arithmetic value */
  struct mp_loop_data *link;    /* the enclosing loop, if any */
} mp_loop_data;

@ @<Glob...@>=
mp_loop_data *loop_ptr; /* top of the loop-control-node stack */

@ @<Set init...@>=
mp->loop_ptr = NULL;

@ If the expressions that define an arithmetic progression in a
\&{for} loop don't have known numeric values, the |bad_for| subroutine
screams at the user.

@c
static void mp_bad_for (MP mp, const char *s) {
  char msg[256];
  mp_value new_expr;
  const char *hlp[] = {"When you say `for x=a step b until c',",
         "the initial value `a' and the step size `b'",
         "and the final value `c' must have known numeric values.",
         "I'm zeroing this one. Proceed, with fingers crossed.",
         NULL };
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  mp_disp_err (mp, NULL); 
  /* show the bad expression above the message */
  mp_snprintf(msg, 256, "Improper %s has been replaced by 0", s);  
@.Improper...replaced by 0@>;
  mp_back_error (mp, msg, hlp, true);
  mp_get_x_next (mp);
  mp_flush_cur_exp (mp, new_expr);
}


@ Here's what \MP\ does when \&{for}, \&{forsuffixes}, or \&{forever}
has just been scanned. (This code requires slight familiarity with
expression-parsing routines that we have not yet discussed; but it
seems to belong in the present part of the program, even though the
original author didn't write it until later. The reader may wish to
come back to it.)

@c
void mp_begin_iteration (MP mp) {
  halfword m;   /* |start_for| (\&{for}) or |start_forsuffixes|
                   (\&{forsuffixes}) */
  mp_sym n;     /* hash address of the current symbol */
  mp_loop_data *s;      /* the new loop-control node */
  mp_subst_list_item *p = NULL; /* substitution list for |scan_toks|
                                 */
  mp_node q;    /* link manipulation register */
  m = cur_mod();
  n = cur_sym();
  s = xmalloc (1, sizeof (mp_loop_data));
  s->type = s->list = s->info = s->list_start = NULL;
  s->link = NULL;
  new_number (s->value);
  new_number (s->step_size);
  new_number (s->final_value);
  if (m == start_forever) {
    s->type = MP_VOID;
    p = NULL;
    mp_get_x_next (mp);
  } else {
    mp_get_symbol (mp);
    p = xmalloc (1, sizeof (mp_subst_list_item));
    p->link = NULL;
    p->info = cur_sym();
    p->info_mod = cur_sym_mod();
    p->value_data = 0;
    if (m == start_for) {
      p->value_mod = mp_expr_sym;
    } else {                    /* |start_forsuffixes| */
      p->value_mod = mp_suffix_sym;
    }
    mp_get_x_next (mp);
    if (cur_cmd() == mp_within_token) {
      @<Set up a picture iteration@>;
    } else {
      @<Check for the assignment in a loop header@>;
      @<Scan the values to be used in the loop@>;
    }
  }
  @<Check for the presence of a colon@>;
  @<Scan the loop text and put it on the loop control stack@>;
  mp_resume_iteration (mp);
}


@ @<Check for the assignment in a loop header@>=
if ((cur_cmd() != mp_equals) && (cur_cmd() != mp_assignment)) {
  const char *hlp[] = {
         "The next thing in this loop should have been `=' or `:='.",
         "But don't worry; I'll pretend that an equals sign",
         "was present, and I'll look for the values next.",
         NULL };
  mp_back_error (mp, "Missing `=' has been inserted", hlp, true);
@.Missing `='@>;
}

@ @<Check for the presence of a colon@>=
if (cur_cmd() != mp_colon) {
  const char *hlp[] = {
         "The next thing in this loop should have been a `:'.",
         "So I'll pretend that a colon was present;",
         "everything from here to `endfor' will be iterated.",
         NULL };
  mp_back_error (mp, "Missing `:' has been inserted", hlp, true);
@.Missing `:'@>;
}

@ We append a special |mp->frozen_repeat_loop| token in place of the
`\&{endfor}' at the end of the loop. This will come through \MP's
scanner at the proper time to cause the loop to be repeated.

(If the user tries some shenanigan like `\&{for} $\ldots$ \&{let}
\&{endfor}', he will be foiled by the |get_symbol| routine, which
keeps frozen tokens unchanged. Furthermore the
|mp->frozen_repeat_loop| is an \&{outer} token, so it won't be lost
accidentally.)

@ @<Scan the loop text...@>=
q = mp_get_symbolic_node (mp);
set_mp_sym_sym (q, mp->frozen_repeat_loop);
mp->scanner_status = loop_defining;
mp->warning_info = n;
s->info = mp_scan_toks (mp, mp_iteration, p, q, 0);
mp->scanner_status = normal;
s->link = mp->loop_ptr;
mp->loop_ptr = s

@ @<Initialize table...@>=
mp->frozen_repeat_loop =
mp_frozen_primitive (mp, " ENDFOR", mp_repeat_loop + mp_outer_tag, 0);

@ The loop text is inserted into \MP's scanning apparatus by the
|resume_iteration| routine.

@c
void mp_resume_iteration (MP mp) {
  mp_node p, q; /* link registers */
  p = mp->loop_ptr->type;
  if (p == PROGRESSION_FLAG) {
    set_cur_exp_value_number (mp->loop_ptr->value);
    if (@<The arithmetic progression has ended@>) {
      mp_stop_iteration (mp);
      return;
    }
    mp->cur_exp.type = mp_known;
    q = mp_stash_cur_exp (mp);  /* make |q| an \&{expr} argument */
    set_number_from_addition (mp->loop_ptr->value, cur_exp_value_number (), mp->loop_ptr->step_size);   
                                                                       /* set |value(p)| for the next iteration */
    /* detect numeric overflow */
    if (number_positive(mp->loop_ptr->step_size) &&
        number_less(mp->loop_ptr->value,  cur_exp_value_number ())) {
      if (number_positive(mp->loop_ptr->final_value)) {
        number_clone (mp->loop_ptr->value, mp->loop_ptr->final_value);
        number_add_scaled (mp->loop_ptr->final_value, -1);
      } else {
        number_clone (mp->loop_ptr->value, mp->loop_ptr->final_value);
        number_add_scaled (mp->loop_ptr->value, 1);
      }
    } else if (number_negative(mp->loop_ptr->step_size) &&
              number_greater (mp->loop_ptr->value, cur_exp_value_number ())) {
      if (number_negative (mp->loop_ptr->final_value)) {
        number_clone (mp->loop_ptr->value, mp->loop_ptr->final_value);
        number_add_scaled (mp->loop_ptr->final_value, 1);
      } else {
        number_clone (mp->loop_ptr->value, mp->loop_ptr->final_value);
        number_add_scaled (mp->loop_ptr->value, -1);
      }
    }
  } else if (p == NULL) {
    p = mp->loop_ptr->list;
    if (p != NULL && p == mp->loop_ptr->list_start) {
      q = p;
      p = mp_link (p);
      mp_free_symbolic_node (mp, q);
      mp->loop_ptr->list = p;
    }
    if (p == NULL) {
      mp_stop_iteration (mp);
      return;
    }
    mp->loop_ptr->list = mp_link (p);
    q = (mp_node)mp_sym_sym (p);
    mp_free_symbolic_node (mp, p);
  } else if (p == MP_VOID) {
    mp_begin_token_list (mp, mp->loop_ptr->info, (quarterword) forever_text);
    return;
  } else {
    @<Make |q| a capsule containing the next picture component from
      |loop_list(loop_ptr)| or |goto not_found|@>;
  }
  mp_begin_token_list (mp, mp->loop_ptr->info, (quarterword) loop_text);
  mp_stack_argument (mp, q);
  if (number_greater (internal_value (mp_tracing_commands), unity_t)) {
    @<Trace the start of a loop@>;
  }
  return;
NOT_FOUND:
  mp_stop_iteration (mp);
}


@ @<The arithmetic progression has ended@>=
(number_positive(mp->loop_ptr->step_size) && number_greater(cur_exp_value_number (), mp->loop_ptr->final_value))
|| 
(number_negative(mp->loop_ptr->step_size) && number_less(cur_exp_value_number (), mp->loop_ptr->final_value))
 

@ @<Trace the start of a loop@>=
{
  mp_begin_diagnostic (mp);
  mp_print_nl (mp, "{loop value=");
@.loop value=n@>;
  if ((q != NULL) && (mp_link (q) == MP_VOID))
    mp_print_exp (mp, q, 1);
  else
    mp_show_token_list (mp, q, NULL, 50, 0);
  mp_print_char (mp, xord ('}'));
  mp_end_diagnostic (mp, false);
}


@ @<Make |q| a capsule containing the next picture component
from...@>=
{
  q = mp->loop_ptr->list;
  if (q == NULL)
    goto NOT_FOUND;
    if ( ! is_start_or_stop(q) ) 
      q=mp_link(q);
    else if ( ! is_stop(q) ) 
      q=mp_skip_1component(mp, q);
    else 
      goto NOT_FOUND;

  set_cur_exp_node ((mp_node)mp_copy_objects (mp, mp->loop_ptr->list, q));
  mp_init_bbox (mp, (mp_edge_header_node)cur_exp_node ());
  mp->cur_exp.type = mp_picture_type;
  mp->loop_ptr->list = q;
  q = mp_stash_cur_exp (mp);
}


@ A level of loop control disappears when |resume_iteration| has
decided not to resume, or when an \&{exitif} construction has removed
the loop text from the input stack.

@c
void mp_stop_iteration (MP mp) {
  mp_node p, q; /* the usual */
  mp_loop_data *tmp;    /* for free() */
  p = mp->loop_ptr->type;
  if (p == PROGRESSION_FLAG) {
    mp_free_symbolic_node (mp, mp->loop_ptr->list);
  } else if (p == NULL) {
    q = mp->loop_ptr->list;
    while (q != NULL) {
      p = (mp_node)mp_sym_sym (q);
      if (p != NULL) {
        if (mp_link (p) == MP_VOID) {      /* it's an \&{expr} parameter */
          mp_recycle_value (mp, p);
          mp_free_value_node (mp, p);
        } else {
          mp_flush_token_list (mp, p);  /* it's a \&{suffix} or \&{text}
                                           parameter */
        }
      }
      p = q;
      q = mp_link (q);
      mp_free_symbolic_node (mp, p);
    }
  } else if (p > PROGRESSION_FLAG) {
    delete_edge_ref (p);
  }
  tmp = mp->loop_ptr;
  mp->loop_ptr = tmp->link;
  mp_flush_token_list (mp, tmp->info);
  free_number (tmp->value);
  free_number (tmp->step_size);
  free_number (tmp->final_value);
  xfree (tmp);
}


@ Now that we know all about loop control, we can finish up the
missing portion of |begin_iteration| and we'll be done.

The following code is performed after the `\.=' has been scanned in a
\&{for} construction (if |m=start_for|) or a \&{forsuffixes}
construction (if |m=start_forsuffixes|).

@<Scan the values to be used in the loop@>=
s->type = NULL;
s->list = mp_get_symbolic_node (mp);
s->list_start = s->list;
q = s->list;
do {
  mp_get_x_next (mp);
  if (m != start_for) {
    mp_scan_suffix (mp);
  } else {
    if (cur_cmd() >= mp_colon)
      if (cur_cmd() <= mp_comma)
        goto CONTINUE;
    mp_scan_expression (mp);
    if (cur_cmd() == mp_step_token)
      if (q == s->list) {
        @<Prepare for step-until construction and |break|@>;
      }
    set_cur_exp_node (mp_stash_cur_exp (mp));
  }
  mp_link (q) = mp_get_symbolic_node (mp);
  q = mp_link (q);
  set_mp_sym_sym (q, mp->cur_exp.data.node);
  if (m == start_for)
    mp_name_type (q) = mp_expr_sym;
  else if (m == start_forsuffixes)
    mp_name_type (q) = mp_suffix_sym;
  mp->cur_exp.type = mp_vacuous;
CONTINUE:
  ;
} while (cur_cmd() == mp_comma)

@ @<Prepare for step-until construction and |break|@>=
{
  if (mp->cur_exp.type != mp_known)
    mp_bad_for (mp, "initial value");
  number_clone (s->value, cur_exp_value_number ());
  mp_get_x_next (mp);
  mp_scan_expression (mp);
  if (mp->cur_exp.type != mp_known)
    mp_bad_for (mp, "step size");
  number_clone (s->step_size, cur_exp_value_number ());
  if (cur_cmd() != mp_until_token) {
    const char *hlp[] = {
           "I assume you meant to say `until' after `step'.",
           "So I'll look for the final value and colon next.",
           NULL };
    mp_back_error (mp, "Missing `until' has been inserted", hlp, true);
@.Missing `until'@>;
  }
  mp_get_x_next (mp);
  mp_scan_expression (mp);
  if (mp->cur_exp.type != mp_known)
    mp_bad_for (mp, "final value");
  number_clone (s->final_value, cur_exp_value_number ());
  s->type = PROGRESSION_FLAG;
  break;
}


@ The last case is when we have just seen ``\&{within}'', and we need to
parse a picture expression and prepare to iterate over it.

@<Set up a picture iteration@>=
{
  mp_get_x_next (mp);
  mp_scan_expression (mp);
  @<Make sure the current expression is a known picture@>;
  s->type = mp->cur_exp.data.node;
  mp->cur_exp.type = mp_vacuous;
  q = mp_link (edge_list (mp->cur_exp.data.node));
  if (q != NULL)
    if (is_start_or_stop (q))
      if (mp_skip_1component (mp, q) == NULL)
        q = mp_link (q);
  s->list = q;
}


@ @<Make sure the current expression is a known picture@>=
if (mp->cur_exp.type != mp_picture_type) {
  mp_value new_expr;
  const char *hlp[] = { "When you say `for x in p', p must be a known picture.", NULL };
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  new_expr.data.node = (mp_node)mp_get_edge_header_node (mp);
  mp_disp_err (mp, NULL);
  mp_back_error (mp,"Improper iteration spec has been replaced by nullpicture", hlp, true);
  mp_get_x_next (mp);
  mp_flush_cur_exp (mp, new_expr);
  mp_init_edges (mp, (mp_edge_header_node)mp->cur_exp.data.node);
  mp->cur_exp.type = mp_picture_type;
}

@* File names.
It's time now to fret about file names.  Besides the fact that different
operating systems treat files in different ways, we must cope with the
fact that completely different naming conventions are used by different
groups of people. The following programs show what is required for one
particular operating system; similar routines for other systems are not
difficult to devise.
@^system dependencies@>

\MP\ assumes that a file name has three parts: the name proper; its
``extension''; and a ``file area'' where it is found in an external file
system.  The extension of an input file is assumed to be
`\.{.mp}' unless otherwise specified; it is `\.{.log}' on the
transcript file that records each run of \MP; it is `\.{.tfm}' on the font
metric files that describe characters in any fonts created by \MP; it is
`\.{.ps}' or `.{\it nnn}' for some number {\it nnn} on the \ps\ output files.
The file area can be arbitrary on input files, but files are usually
output to the user's current area.  If an input file cannot be
found on the specified area, \MP\ will look for it on a special system
area; this special area is intended for commonly used input files.

Simple uses of \MP\ refer only to file names that have no explicit
extension or area. For example, a person usually says `\.{input} \.{cmr10}'
instead of `\.{input} \.{cmr10.new}'. Simple file
names are best, because they make the \MP\ source files portable;
whenever a file name consists entirely of letters and digits, it should be
treated in the same way by all implementations of \MP. However, users
need the ability to refer to other files in their environment, especially
when responding to error messages concerning unopenable files; therefore
we want to let them use the syntax that appears in their favorite
operating system.

@ \MP\ uses the same conventions that have proved to be satisfactory for
\TeX\ and \MF. In order to isolate the system-dependent aspects of file names,
@^system dependencies@>
the system-independent parts of \MP\ are expressed in terms
of three system-dependent
procedures called |begin_name|, |more_name|, and |end_name|. In
essence, if the user-specified characters of the file name are $c_1\ldots c_n$,
the system-independent driver program does the operations
$$|begin_name|;\,|more_name|(c_1);\,\ldots\,;\,|more_name|(c_n);
\,|end_name|.$$
These three procedures communicate with each other via global variables.
Afterwards the file name will appear in the string pool as three strings
called |cur_name|\penalty10000\hskip-.05em,
|cur_area|, and |cur_ext|; the latter two are NULL (i.e.,
|""|), unless they were explicitly specified by the user.

Actually the situation is slightly more complicated, because \MP\ needs
to know when the file name ends. The |more_name| routine is a function
(with side effects) that returns |true| on the calls |more_name|$(c_1)$,
\dots, |more_name|$(c_{n-1})$. The final call |more_name|$(c_n)$
returns |false|; or, it returns |true| and $c_n$ is the last character
on the current input line. In other words,
|more_name| is supposed to return |true| unless it is sure that the
file name has been completely scanned; and |end_name| is supposed to be able
to finish the assembly of |cur_name|, |cur_area|, and |cur_ext| regardless of
whether $|more_name|(c_n)$ returned |true| or |false|.

@<Glob...@>=
char *cur_name; /* name of file just scanned */
char *cur_area; /* file area just scanned, or \.{""} */
char *cur_ext;  /* file extension just scanned, or \.{""} */

@ It is easier to maintain reference counts if we assign initial values.

@<Set init...@>=
mp->cur_name = xstrdup ("");
mp->cur_area = xstrdup ("");
mp->cur_ext = xstrdup ("");

@ @<Dealloc variables@>=
xfree (mp->cur_area);
xfree (mp->cur_name);
xfree (mp->cur_ext);

@ The file names we shall deal with for illustrative purposes have the
following structure:  If the name contains `\.>' or `\.:', the file area
consists of all characters up to and including the final such character;
otherwise the file area is null.  If the remaining file name contains
`\..', the file extension consists of all such characters from the first
remaining `\..' to the end, otherwise the file extension is null.
@^system dependencies@>

We can scan such file names easily by using two global variables that keep track
of the occurrences of area and extension delimiters.  

@<Glob...@>=
integer area_delimiter;
  /* most recent `\.>' or `\.:' relative to |str_start[str_ptr]| */
integer ext_delimiter;  /* the relevant `\..', if any */
boolean quoted_filename;        /* whether the filename is wrapped in " markers */

@ Here now is the first of the system-dependent routines for file name scanning.
@^system dependencies@>

@<Declarations@>=
static void mp_begin_name (MP mp);
static boolean mp_more_name (MP mp, ASCII_code c);
static void mp_end_name (MP mp);

@ @c
void mp_begin_name (MP mp) {
  xfree (mp->cur_name);
  xfree (mp->cur_area);
  xfree (mp->cur_ext);
  mp->area_delimiter = -1;
  mp->ext_delimiter = -1;
  mp->quoted_filename = false;
}


@ And here's the second.
@^system dependencies@>

@c
#ifndef IS_DIR_SEP
#define IS_DIR_SEP(c) (c=='/' || c=='\\')
#endif
boolean mp_more_name (MP mp, ASCII_code c) {
  if (c == '"') {
    mp->quoted_filename = !mp->quoted_filename;
  } else if ((c == ' ' || c == '\t') && (mp->quoted_filename == false)) {
    return false;
  } else {
    if (IS_DIR_SEP (c)) {
      mp->area_delimiter = (integer) mp->cur_length;
      mp->ext_delimiter = -1;
    } else if (c == '.') {
      mp->ext_delimiter = (integer) mp->cur_length;
    }
    append_char (c);            /* contribute |c| to the current string */
  }
  return true;
}


@ The third.
@^system dependencies@>

@d copy_pool_segment(A,B,C) { 
      A = xmalloc(C+1,sizeof(char)); 
      (void)memcpy(A,(char *)(mp->cur_string+B),C);  
      A[C] = 0;}

@c
void mp_end_name (MP mp) {
  size_t s = 0; /* length of area, name, and extension */
  size_t len;
  /* "my/w.mp" */
  if (mp->area_delimiter < 0) {
    mp->cur_area = xstrdup ("");
  } else {
    len = (size_t) mp->area_delimiter - s + 1;
    copy_pool_segment (mp->cur_area, s, len);
    s += len;
  }
  if (mp->ext_delimiter < 0) {
    mp->cur_ext = xstrdup ("");
    len = (unsigned) (mp->cur_length - s);
  } else {
    copy_pool_segment (mp->cur_ext, mp->ext_delimiter,
                       (mp->cur_length - (size_t) mp->ext_delimiter));
    len = (size_t) mp->ext_delimiter - s;
  }
  copy_pool_segment (mp->cur_name, s, len);
  mp_reset_cur_string (mp);
}


@ Conversely, here is a routine that takes three strings and prints a file
name that might have produced them. (The routine is system dependent, because
some operating systems put the file area last instead of first.)
@^system dependencies@>

@<Basic printing...@>=
static void mp_print_file_name (MP mp, char *n, char *a, char *e) {
  boolean must_quote = false;
  if (((a != NULL) && (strchr (a, ' ') != NULL)) ||
      ((n != NULL) && (strchr (n, ' ') != NULL)) ||
      ((e != NULL) && (strchr (e, ' ') != NULL)))
    must_quote = true;
  if (must_quote)
    mp_print_char (mp, (ASCII_code) '"');
  mp_print (mp, a);
  mp_print (mp, n);
  mp_print (mp, e);
  if (must_quote)
    mp_print_char (mp, (ASCII_code) '"');
}


@ Another system-dependent routine is needed to convert three internal
\MP\ strings
to the |name_of_file| value that is used to open files. The present code
allows both lowercase and uppercase letters in the file name.
@^system dependencies@>

@d append_to_name(A) { mp->name_of_file[k++]=(char)xchr(xord((ASCII_code)(A))); }

@ @c
void mp_pack_file_name (MP mp, const char *n, const char *a, const char *e) {
  integer k;    /* number of positions filled in |name_of_file| */
  const char *j;        /* a character  index */
  size_t slen;
  k = 0;
  assert (n != NULL);
  xfree (mp->name_of_file);
  slen = strlen (n) + 1;
  if (a != NULL)
    slen += strlen (a);
  if (e != NULL)
    slen += strlen (e);
  mp->name_of_file = xmalloc (slen, 1);
  if (a != NULL) {
    for (j = a; *j != '\0'; j++) {
      append_to_name (*j);
    }
  }
  for (j = n; *j != '\0'; j++) {
    append_to_name (*j);
  }
  if (e != NULL) {
    for (j = e; *j != '\0'; j++) {
      append_to_name (*j);
    }
  }
  mp->name_of_file[k] = 0;
}


@ @<Internal library declarations@>=
void mp_pack_file_name (MP mp, const char *n, const char *a, const char *e);

@ @<Option variables@>=
char *mem_name; /* for commandline */

@ Stripping a |.mem| extension here is for backward compatibility.

@<Find and load preload file, if required@>=
if (!opt->ini_version) {
  mp->mem_name = xstrdup (opt->mem_name);
  if (mp->mem_name) {
    size_t l = strlen (mp->mem_name);
    if (l > 4) {
      char *test = strstr (mp->mem_name, ".mem");
      if (test == mp->mem_name + l - 4) {
       *test = 0;
      }
    }
  }
  if (mp->mem_name != NULL) {
    if (!mp_open_mem_file (mp)) {
      mp->history = mp_fatal_error_stop;
      mp_jump_out (mp);
    }
  }
}



@ @<Dealloc variables@>=
xfree (mp->mem_name);

@ This part of the program becomes active when a ``virgin'' \MP\ is
trying to get going, just after the preliminary initialization.  
The buffer contains the first line of input in |buffer[loc..(last-1)]|, 
where |loc<last| and |buffer[loc]<>""|.

@<Declarations@>=
static boolean mp_open_mem_name (MP mp);
static boolean mp_open_mem_file (MP mp);

@ @c
boolean mp_open_mem_name (MP mp) {
  if (mp->mem_name != NULL) {
    size_t l = strlen (mp->mem_name);
    char *s = xstrdup (mp->mem_name);
    if (l > 4) {
      char *test = strstr (s, ".mp");
      if (test == NULL || test != s + l - 4) {
        s = xrealloc (s, l + 5, 1);
        strcat (s, ".mp");
      }
    } else {
      s = xrealloc (s, l + 5, 1);
      strcat (s, ".mp");
    }
    s = (mp->find_file) (mp, s, "r", mp_filetype_program);
    xfree(mp->name_of_file);
    if (s == NULL)
      return false;
    mp->name_of_file = xstrdup(s);
    mp->mem_file = (mp->open_file) (mp, s, "r", mp_filetype_program);
    free (s);
    if (mp->mem_file)
      return true;
  }
  return false;
}
boolean mp_open_mem_file (MP mp) {
  if (mp->mem_file != NULL)
    return true;
  if (mp_open_mem_name (mp))
    return true;
  if (mp_xstrcmp (mp->mem_name, "plain")) {
    wake_up_terminal();
    wterm ("Sorry, I can\'t find the '");
    wterm (mp->mem_name);
    wterm ("' preload file; will try 'plain'.");
    wterm_cr;
@.Sorry, I can't find...@>;
    update_terminal();
    /* now pull out all the stops: try for the system \.{plain} file */
    xfree (mp->mem_name);
    mp->mem_name = xstrdup ("plain");
    if (mp_open_mem_name (mp))
      return true;
  }
  wake_up_terminal();
  wterm_ln ("I can't find the 'plain' preload file!\n");
@.I can't find PLAIN...@>
@.plain@>;
  return false;
}


@ Operating systems often make it possible to determine the exact name (and
possible version number) of a file that has been opened. The following routine,
which simply makes a \MP\ string from the value of |name_of_file|, should
ideally be changed to deduce the full name of file~|f|, which is the file
most recently opened, if it is possible to do this.
@^system dependencies@>

@ @c
static mp_string mp_make_name_string (MP mp) {
  int k;        /* index into |name_of_file| */
  int name_length = (int) strlen (mp->name_of_file);
  str_room (name_length);
  for (k = 0; k < name_length; k++) {
    append_char (xord ((ASCII_code) mp->name_of_file[k]));
  }
  return mp_make_string (mp);
}


@ Now let's consider the ``driver''
routines by which \MP\ deals with file names
in a system-independent manner.  First comes a procedure that looks for a
file name in the input by taking the information from the input buffer.
(We can't use |get_next|, because the conversion to tokens would
destroy necessary information.)

This procedure doesn't allow semicolons or percent signs to be part of
file names, because of other conventions of \MP.
{\sl The {\logos METAFONT\/}book} doesn't
use semicolons or percents immediately after file names, but some users
no doubt will find it natural to do so; therefore system-dependent
changes to allow such characters in file names should probably
be made with reluctance, and only when an entire file name that
includes special characters is ``quoted'' somehow.
@^system dependencies@>

@c
static void mp_scan_file_name (MP mp) {
  mp_begin_name (mp);
  while (mp->buffer[loc] == ' ')
    incr (loc);
  while (1) {
    if ((mp->buffer[loc] == ';') || (mp->buffer[loc] == '%'))
      break;
    if (!mp_more_name (mp, mp->buffer[loc]))
      break;
    incr (loc);
  }
  mp_end_name (mp);
}


@ Here is another version that takes its input from a string.

@<Declare subroutines for parsing file names@>=
void mp_str_scan_file (MP mp, mp_string s);

@ @c
void mp_str_scan_file (MP mp, mp_string s) {
  size_t p, q;  /* current position and stopping point */
  mp_begin_name (mp);
  p = 0;
  q = s->len;
  while (p < q) {
    if (!mp_more_name (mp, *(s->str + p)))
      break;
    incr (p);
  }
  mp_end_name (mp);
}


@ And one that reads from a |char*|.

@<Declare subroutines for parsing file names@>=
extern void mp_ptr_scan_file (MP mp, char *s);

@ @c
void mp_ptr_scan_file (MP mp, char *s) {
  char *p, *q;  /* current position and stopping point */
  mp_begin_name (mp);
  p = s;
  q = p + strlen (s);
  while (p < q) {
    if (!mp_more_name (mp, (ASCII_code) (*p)))
      break;
    p++;
  }
  mp_end_name (mp);
}


@ The option variable |job_name| contains the file name that was first
\&{input} by the user. This name is used to initialize the |job_name| global
as well as the |mp_job_name| internal, and is extended by `\.{.log}' and 
`\.{ps}' and `\.{.mem}' and `\.{.tfm}' in order to make the names of \MP's 
output files.

@<Glob...@>=
boolean log_opened;     /* has the transcript file been opened? */
char *log_name; /* full name of the log file */

@ @<Option variables@>=
char *job_name; /* principal file name */

@ Initially |job_name=NULL|; it becomes nonzero as soon as the true name is known.
We have |job_name=NULL| if and only if the `\.{log}' file has not been opened,
except of course for a short time just after |job_name| has become nonzero.

@<Allocate or ...@>=
mp->job_name = mp_xstrdup (mp, opt->job_name);
/*|
if (mp->job_name != NULL) {
  char *s = mp->job_name + strlen (mp->job_name);
  while (s > mp->job_name) {
    if (*s == '.') {
      *s = '\0';
    }
    s--;
  }
}
|*/
if (opt->noninteractive) {
  if (mp->job_name == NULL)
    mp->job_name = mp_xstrdup (mp, mp->mem_name);
}
mp->log_opened = false;

@ Cannot do this earlier because at the |<Allocate or ...>|, the string
pool is not yet initialized.

@<Fix up |mp->internal[mp_job_name]|@>=
if (mp->job_name != NULL) {
  if (internal_string (mp_job_name) != 0)
    delete_str_ref (internal_string (mp_job_name));
  set_internal_string (mp_job_name, mp_rts (mp, mp->job_name));
}

@ @<Dealloc variables@>=
xfree (mp->job_name);

@ Here is a routine that manufactures the output file names, assuming that
|job_name<>0|. It ignores and changes the current settings of |cur_area|
and |cur_ext|.

@d pack_cur_name mp_pack_file_name(mp, mp->cur_name,mp->cur_area,mp->cur_ext)

@<Internal library ...@>=
void mp_pack_job_name (MP mp, const char *s);

@ @c
void mp_pack_job_name (MP mp, const char *s) {                               /* |s = ".log"|, |".mem"|, |".ps"|, or .\\{nnn} */
  xfree (mp->cur_name);
  mp->cur_name = xstrdup (mp->job_name);
  xfree (mp->cur_area);
  mp->cur_area = xstrdup ("");
  xfree (mp->cur_ext);
  mp->cur_ext = xstrdup (s);
  pack_cur_name;
}


@ If some trouble arises when \MP\ tries to open a file, the following
routine calls upon the user to supply another file name. Parameter~|s|
is used in the error message to identify the type of file; parameter~|e|
is the default extension if none is given. Upon exit from the routine,
variables |cur_name|, |cur_area|, |cur_ext|, and |name_of_file| are
ready for another attempt at file opening.

@<Internal library ...@>=
void mp_prompt_file_name (MP mp, const char *s, const char *e);

@ @c
void mp_prompt_file_name (MP mp, const char *s, const char *e) {
  size_t k;     /* index into |buffer| */
  char *saved_cur_name;
  if (mp->interaction == mp_scroll_mode)
    wake_up_terminal();
  if (strcmp (s, "input file name") == 0) {
    mp_print_err (mp, "I can\'t open file `");
@.I can't find file x@>
  } else {
    mp_print_err (mp, "I can\'t write on file `");
@.I can't write on file x@>
  }
  if (strcmp (s, "file name for output") == 0) {
    mp_print (mp, mp->name_of_file);
  } else {
    mp_print_file_name (mp, mp->cur_name, mp->cur_area, mp->cur_ext);
  }
  mp_print (mp, "'.");
  if (strcmp (e, "") == 0)
    mp_show_context (mp);
  mp_print_nl (mp, "Please type another ");
  mp_print (mp, s);
@.Please type...@>;
  if (mp->noninteractive || mp->interaction < mp_scroll_mode)
    mp_fatal_error (mp, "*** (job aborted, file error in nonstop mode)");
@.job aborted, file error...@>;
  saved_cur_name = xstrdup (mp->cur_name);
  clear_terminal();
  prompt_input (": ");
  @<Scan file name in the buffer@>;
  if (strcmp (mp->cur_ext, "") == 0)
    mp->cur_ext = xstrdup (e);
  if (strlen (mp->cur_name) == 0) {
    mp->cur_name = saved_cur_name;
  } else {
    xfree (saved_cur_name);
  }
  pack_cur_name;
}


@ @<Scan file name in the buffer@>=
{
  mp_begin_name (mp);
  k = mp->first;
  while ((mp->buffer[k] == ' ') && (k < mp->last))
    incr (k);
  while (1) {
    if (k == mp->last)
      break;
    if (!mp_more_name (mp, mp->buffer[k]))
      break;
    incr (k);
  }
  mp_end_name (mp);
}


@ The |open_log_file| routine is used to open the transcript file and to help
it catch up to what has previously been printed on the terminal.

@c
void mp_open_log_file (MP mp) {
  unsigned old_setting; /* previous |selector| setting */
  int k;        /* index into |months| and |buffer| */
  int l;        /* end of first input line */
  integer m;    /* the current month */
  const char *months = "JANFEBMARAPRMAYJUNJULAUGSEPOCTNOVDEC";
  /* abbreviations of month names */
  if (mp->log_opened)
    return;
  old_setting = mp->selector;
  if (mp->job_name == NULL) {
    mp->job_name = xstrdup ("mpout");
    @<Fix up |mp->internal[mp_job_name]|@>;
  }
  mp_pack_job_name (mp, ".log");
  while (!mp_open_out (mp, &mp->log_file, mp_filetype_log)) {
    @<Try to get a different log file name@>;
  }
  mp->log_name = xstrdup (mp->name_of_file);
  mp->selector = log_only;
  mp->log_opened = true;
  @<Print the banner line, including the date and time@>;
  mp->input_stack[mp->input_ptr] = mp->cur_input;
  /* make sure bottom level is in memory */
  if (!mp->noninteractive) {
    mp_print_nl (mp, "**");
@.**@>;
    l = mp->input_stack[0].limit_field - 1;     /* last position of first line */
    for (k = 0; k <= l; k++)
      mp_print_char (mp, mp->buffer[k]);
    mp_print_ln (mp);           /* now the transcript file contains the first line of input */
  }
  mp->selector = old_setting + 2;       /* |log_only| or |term_and_log| */
}


@ @<Dealloc variables@>=
xfree (mp->log_name);

@ Sometimes |open_log_file| is called at awkward moments when \MP\ is
unable to print error messages or even to |show_context|.
The |prompt_file_name| routine can result in a |fatal_error|, but the |error|
routine will not be invoked because |log_opened| will be false.

The normal idea of |mp_batch_mode| is that nothing at all should be written
on the terminal. However, in the unusual case that
no log file could be opened, we make an exception and allow
an explanatory message to be seen.

Incidentally, the program always refers to the log file as a `\.{transcript
file}', because some systems cannot use the extension `\.{.log}' for
this file.

@<Try to get a different log file name@>=
{
  mp->selector = term_only;
  mp_prompt_file_name (mp, "transcript file name", ".log");
}


@ @<Print the banner...@>=
{
  wlog (mp->banner);
  mp_print (mp, "  ");
  mp_print_int (mp, round_unscaled (internal_value (mp_day)));
  mp_print_char (mp, xord (' '));
  m = round_unscaled (internal_value (mp_month));
  for (k = 3 * m - 3; k < 3 * m; k++) {
    wlog_chr ((unsigned char) months[k]);
  }
  mp_print_char (mp, xord (' '));
  mp_print_int (mp, round_unscaled (internal_value (mp_year)));
  mp_print_char (mp, xord (' '));
  mp_print_dd (mp, round_unscaled (internal_value (mp_hour)));
  mp_print_char (mp, xord (':'));
  mp_print_dd (mp, round_unscaled (internal_value (mp_minute)));
}


@ The |try_extension| function tries to open an input file determined by
|cur_name|, |cur_area|, and the argument |ext|.  It returns |false| if it
can't find the file in |cur_area| or the appropriate system area.

@c
static boolean mp_try_extension (MP mp, const char *ext) {
  mp_pack_file_name (mp, mp->cur_name, mp->cur_area, ext);
  in_name = xstrdup (mp->cur_name);
  in_area = xstrdup (mp->cur_area);
  in_ext = xstrdup (ext);
  if (mp_open_in (mp, &cur_file, mp_filetype_program)) {
    return true;
  } else {
    mp_pack_file_name (mp, mp->cur_name, NULL, ext);
    return mp_open_in (mp, &cur_file, mp_filetype_program);
  }
}


@ Let's turn now to the procedure that is used to initiate file reading
when an `\.{input}' command is being processed.

@c
void mp_start_input (MP mp) {                               /* \MP\ will \.{input} something */
  char *fname = NULL;
  @<Put the desired file name in |(cur_name,cur_ext,cur_area)|@>;
  while (1) {
    mp_begin_file_reading (mp); /* set up |cur_file| and new level of input */
    if (strlen (mp->cur_ext) == 0) {
      if (mp_try_extension (mp, ".mp"))
        break;
      else if (mp_try_extension (mp, ""))
        break;
      else if (mp_try_extension (mp, ".mf"))
        break;
    } else if (mp_try_extension (mp, mp->cur_ext)) {
      break;
    }
    mp_end_file_reading (mp);   /* remove the level that didn't work */
    mp_prompt_file_name (mp, "input file name", "");
  }
  name = mp_make_name_string (mp);
  fname = xstrdup (mp->name_of_file);
  if (mp->job_name == NULL) {
    mp->job_name = xstrdup (mp->cur_name);
    @<Fix up |mp->internal[mp_job_name]|@>;
  }
  if (!mp->log_opened) {
    mp_open_log_file (mp);
  }                             /* |open_log_file| doesn't |show_context|, so |limit|
                                   and |loc| needn't be set to meaningful values yet */
  if (((int) mp->term_offset + (int) strlen (fname)) > (mp->max_print_line - 2))
    mp_print_ln (mp);
  else if ((mp->term_offset > 0) || (mp->file_offset > 0))
    mp_print_char (mp, xord (' '));
  mp_print_char (mp, xord ('('));
  incr (mp->open_parens);
  mp_print (mp, fname);
  xfree (fname);
  update_terminal();
  @<Flush |name| and replace it with |cur_name| if it won't be needed@>;
  @<Read the first line of the new file@>;
}


@ This code should be omitted if |make_name_string| returns something other
than just a copy of its argument and the full file name is needed for opening
\.{MPX} files or implementing the switch-to-editor option.
@^system dependencies@>

@<Flush |name| and replace it with |cur_name| if it won't be needed@>=
mp_flush_string (mp, name);
name = mp_rts (mp, mp->cur_name);
xfree (mp->cur_name)
 

@ If the file is empty, it is considered to contain a single blank line,
so there is no need to test the return value.

@<Read the first line...@>=
{
  line = 1;
  (void) mp_input_ln (mp, cur_file);
  mp_firm_up_the_line (mp);
  mp->buffer[limit] = xord ('%');
  mp->first = (size_t) (limit + 1);
  loc = start;
}


@ @<Put the desired file name in |(cur_name,cur_ext,cur_area)|@>=
while (token_state && (nloc == NULL))
  mp_end_token_list (mp);
if (token_state) {
  const char *hlp[] = {
         "Sorry...I've converted what follows to tokens,",
         "possibly garbaging the name you gave.",
         "Please delete the tokens and insert the name again.",
         NULL };
  mp_error (mp, "File names can't appear within macros", hlp, true);
@.File names can't...@>;
}
if (file_state) {
  mp_scan_file_name (mp);
} else {
  xfree (mp->cur_name);
  mp->cur_name = xstrdup ("");
  xfree (mp->cur_ext);
  mp->cur_ext = xstrdup ("");
  xfree (mp->cur_area);
  mp->cur_area = xstrdup ("");
}


@ The following simple routine starts reading the \.{MPX} file associated
with the current input file.

@c
void mp_start_mpx_input (MP mp) {
  char *origname = NULL;        /* a copy of nameoffile */
  mp_pack_file_name (mp, in_name, in_area, in_ext);
  origname = xstrdup (mp->name_of_file);
  mp_pack_file_name (mp, in_name, in_area, ".mpx");
  if (!(mp->run_make_mpx) (mp, origname, mp->name_of_file))
    goto NOT_FOUND;
  mp_begin_file_reading (mp);
  if (!mp_open_in (mp, &cur_file, mp_filetype_program)) {
    mp_end_file_reading (mp);
    goto NOT_FOUND;
  }
  name = mp_make_name_string (mp);
  mp->mpx_name[iindex] = name;
  add_str_ref (name);
  @<Read the first line of the new file@>;
  xfree (origname);
  return;
NOT_FOUND:
  @<Explain that the \.{MPX} file can't be read and |succumb|@>;
  xfree (origname);
}


@ This should ideally be changed to do whatever is necessary to create the
\.{MPX} file given by |name_of_file| if it does not exist or if it is out
of date.  This requires invoking \.{MPtoTeX} on the |origname| and passing
the results through \TeX\ and \.{DVItoMP}.  (It is possible to use a
completely different typesetting program if suitable postprocessor is
available to perform the function of \.{DVItoMP}.)
@^system dependencies@>

@ @<Exported types@>=
typedef int (*mp_makempx_cmd) (MP mp, char *origname, char *mtxname);

@ @<Option variables@>=
mp_makempx_cmd run_make_mpx;

@ @<Allocate or initialize ...@>=
set_callback_option (run_make_mpx);

@ @<Declarations@>=
static int mp_run_make_mpx (MP mp, char *origname, char *mtxname);

@ The default does nothing.
@c
int mp_run_make_mpx (MP mp, char *origname, char *mtxname) {
  (void) mp;
  (void) origname;
  (void) mtxname;
  return false;
}


@ @<Explain that the \.{MPX} file can't be read and |succumb|@>=
{
  const char *hlp[] = {
       "The two files given above are one of your source files",
       "and an auxiliary file I need to read to find out what your",
       "btex..etex blocks mean. If you don't know why I had trouble,",
       "try running it manually through MPtoTeX, TeX, and DVItoMP",
        NULL };
  if (mp->interaction == mp_error_stop_mode)
    wake_up_terminal();
  mp_print_nl (mp, ">> ");
  mp_print (mp, origname);
  mp_print_nl (mp, ">> ");
  mp_print (mp, mp->name_of_file);
  xfree (origname);
  if ( mp->interaction==mp_error_stop_mode )
    mp->interaction=mp_scroll_mode; /* no more interaction */
  if ( mp->log_opened ) 
    mp_error(mp, "! Unable to read mpx file", hlp, true);
  mp->history=mp_fatal_error_stop; 
  mp_jump_out(mp); /* irrecoverable error */
}

@ The last file-opening commands are for files accessed via the \&{readfrom}
@:read_from_}{\&{readfrom} primitive@>
operator and the \&{write} command.  Such files are stored in separate arrays.
@:write_}{\&{write} primitive@>

@<Types in the outer block@>=
typedef unsigned int readf_index;       /* |0..max_read_files| */
typedef unsigned int write_index;       /* |0..max_write_files| */

@ @<Glob...@>=
readf_index max_read_files;     /* maximum number of simultaneously open \&{readfrom} files */
void **rd_file; /* \&{readfrom} files */
char **rd_fname;        /* corresponding file name or 0 if file not open */
readf_index read_files; /* number of valid entries in the above arrays */
write_index max_write_files;    /* maximum number of simultaneously open \&{write} */
void **wr_file; /* \&{write} files */
char **wr_fname;        /* corresponding file name or 0 if file not open */
write_index write_files;        /* number of valid entries in the above arrays */

@ @<Allocate or initialize ...@>=
mp->max_read_files = 8;
mp->rd_file = xmalloc ((mp->max_read_files + 1), sizeof (void *));
mp->rd_fname = xmalloc ((mp->max_read_files + 1), sizeof (char *));
memset (mp->rd_fname, 0, sizeof (char *) * (mp->max_read_files + 1));
mp->max_write_files = 8;
mp->wr_file = xmalloc ((mp->max_write_files + 1), sizeof (void *));
mp->wr_fname = xmalloc ((mp->max_write_files + 1), sizeof (char *));
memset (mp->wr_fname, 0, sizeof (char *) * (mp->max_write_files + 1));


@ This routine starts reading the file named by string~|s| without setting
|loc|, |limit|, or |name|.  It returns |false| if the file is empty or cannot
be opened.  Otherwise it updates |rd_file[n]| and |rd_fname[n]|.

@c
static boolean mp_start_read_input (MP mp, char *s, readf_index n) {
  mp_ptr_scan_file (mp, s);
  pack_cur_name;
  mp_begin_file_reading (mp);
  if (!mp_open_in (mp, &mp->rd_file[n], (int) (mp_filetype_text + n)))
    goto NOT_FOUND;
  if (!mp_input_ln (mp, mp->rd_file[n])) {
    (mp->close_file) (mp, mp->rd_file[n]);
    goto NOT_FOUND;
  }
  mp->rd_fname[n] = xstrdup (s);
  return true;
NOT_FOUND:
  mp_end_file_reading (mp);
  return false;
}


@ Open |wr_file[n]| using file name~|s| and update |wr_fname[n]|.

@<Declarations@>=
static void mp_open_write_file (MP mp, char *s, readf_index n);

@ @c
void mp_open_write_file (MP mp, char *s, readf_index n) {
  mp_ptr_scan_file (mp, s);
  pack_cur_name;
  while (!mp_open_out (mp, &mp->wr_file[n], (int) (mp_filetype_text + n)))
    mp_prompt_file_name (mp, "file name for write output", "");
  mp->wr_fname[n] = xstrdup (s);
}


@* Introduction to the parsing routines.
We come now to the central nervous system that sparks many of \MP's activities.
By evaluating expressions, from their primary constituents to ever larger
subexpressions, \MP\ builds the structures that ultimately define complete
pictures or fonts of type.

Four mutually recursive subroutines are involved in this process: We call them
$$\hbox{|scan_primary|, |scan_secondary|, |scan_tertiary|,
and |scan_expression|.}$$
@^recursion@>
Each of them is parameterless and begins with the first token to be scanned
already represented in |cur_cmd|, |cur_mod|, and |cur_sym|. After execution,
the value of the primary or secondary or tertiary or expression that was
found will appear in the global variables |cur_type| and |cur_exp|. The
token following the expression will be represented in |cur_cmd|, |cur_mod|,
and |cur_sym|.

Technically speaking, the parsing algorithms are ``LL(1),'' more or less;
backup mechanisms have been added in order to provide reasonable error
recovery.

@d cur_exp_value_boolean() number_to_int (mp->cur_exp.data.n)
@d cur_exp_value_number() mp->cur_exp.data.n
@d cur_exp_node() mp->cur_exp.data.node
@d cur_exp_str() mp->cur_exp.data.str
@d cur_exp_knot() mp->cur_exp.data.p

@d set_cur_exp_value_scaled(A) do {
    if (cur_exp_str()) {
        delete_str_ref(cur_exp_str());
    }
    set_number_from_scaled (mp->cur_exp.data.n, (A));
    cur_exp_node() = NULL;
    cur_exp_str() = NULL;
    cur_exp_knot() = NULL;
  } while (0)
@d set_cur_exp_value_boolean(A) do {
    if (cur_exp_str()) {
        delete_str_ref(cur_exp_str());
    }
    set_number_from_int (mp->cur_exp.data.n, (A));
    cur_exp_node() = NULL;
    cur_exp_str() = NULL;
    cur_exp_knot() = NULL;
  } while (0)
@d set_cur_exp_value_number(A) do {
    if (cur_exp_str()) {
        delete_str_ref(cur_exp_str());
    }
    number_clone (mp->cur_exp.data.n, (A));
    cur_exp_node() = NULL;
    cur_exp_str() = NULL;
    cur_exp_knot() = NULL;
  } while (0)
@d set_cur_exp_node(A) do {
    if (cur_exp_str()) {
        delete_str_ref(cur_exp_str());
    }
    cur_exp_node() = A;
    cur_exp_str() = NULL;
    cur_exp_knot() = NULL;
    set_number_to_zero (mp->cur_exp.data.n);
  } while (0)
@d set_cur_exp_str(A) do {
    if (cur_exp_str()) {
        delete_str_ref(cur_exp_str());
    }
    cur_exp_str() = A;
    add_str_ref(cur_exp_str());
    cur_exp_node() = NULL;
    cur_exp_knot() = NULL;
    set_number_to_zero (mp->cur_exp.data.n);
  } while (0)
@d set_cur_exp_knot(A) do {
    if (cur_exp_str()) {
        delete_str_ref(cur_exp_str());
    }
    cur_exp_knot() = A;
    cur_exp_node() = NULL;
    cur_exp_str() = NULL;
    set_number_to_zero (mp->cur_exp.data.n);
  } while (0)
  

@ @<Glob...@>=
mp_value cur_exp;       /* the value of the expression just found */

@ @<Set init...@>=
memset (&mp->cur_exp.data, 0, sizeof (mp_value));
new_number(mp->cur_exp.data.n);

@ @<Free table ...@>=
free_number(mp->cur_exp.data.n);

@ Many different kinds of expressions are possible, so it is wise to have
precise descriptions of what |cur_type| and |cur_exp| mean in all cases:

\smallskip\hang
|cur_type=mp_vacuous| means that this expression didn't turn out to have a
value at all, because it arose from a \&{begingroup}$\,\ldots\,$\&{endgroup}
construction in which there was no expression before the \&{endgroup}.
In this case |cur_exp| has some irrelevant value.

\smallskip\hang
|cur_type=mp_boolean_type| means that |cur_exp| is either |true_code|
or |false_code|.

\smallskip\hang
|cur_type=mp_unknown_boolean| means that |cur_exp| points to a capsule
node that is in 
a ring of equivalent booleans whose value has not yet been defined.

\smallskip\hang
|cur_type=mp_string_type| means that |cur_exp| is a string number (i.e., an
integer in the range |0<=cur_exp<str_ptr|). That string's reference count
includes this particular reference.

\smallskip\hang
|cur_type=mp_unknown_string| means that |cur_exp| points to a capsule
node that is in
a ring of equivalent strings whose value has not yet been defined.

\smallskip\hang
|cur_type=mp_pen_type| means that |cur_exp| points to a node in a pen.  Nobody
else points to any of the nodes in this pen.  The pen may be polygonal or
elliptical.

\smallskip\hang
|cur_type=mp_unknown_pen| means that |cur_exp| points to a capsule
node that is in
a ring of equivalent pens whose value has not yet been defined.

\smallskip\hang
|cur_type=mp_path_type| means that |cur_exp| points to a the first node of
a path; nobody else points to this particular path. The control points of
the path will have been chosen.

\smallskip\hang
|cur_type=mp_unknown_path| means that |cur_exp| points to a capsule
node that is in
a ring of equivalent paths whose value has not yet been defined.

\smallskip\hang
|cur_type=mp_picture_type| means that |cur_exp| points to an edge header node.
There may be other pointers to this particular set of edges.  The header node
contains a reference count that includes this particular reference.

\smallskip\hang
|cur_type=mp_unknown_picture| means that |cur_exp| points to a capsule
node that is in
a ring of equivalent pictures whose value has not yet been defined.

\smallskip\hang
|cur_type=mp_transform_type| means that |cur_exp| points to a |mp_transform_type|
capsule node. The |value| part of this capsule
points to a transform node that contains six numeric values,
each of which is |independent|, |dependent|, |mp_proto_dependent|, or |known|.

\smallskip\hang
|cur_type=mp_color_type| means that |cur_exp| points to a |color_type|
capsule node. The |value| part of this capsule
points to a color node that contains three numeric values,
each of which is |independent|, |dependent|, |mp_proto_dependent|, or |known|.

\smallskip\hang
|cur_type=mp_cmykcolor_type| means that |cur_exp| points to a |mp_cmykcolor_type|
capsule node. The |value| part of this capsule
points to a color node that contains four numeric values,
each of which is |independent|, |dependent|, |mp_proto_dependent|, or |known|.

\smallskip\hang
|cur_type=mp_pair_type| means that |cur_exp| points to a capsule
node whose type is |mp_pair_type|. The |value| part of this capsule
points to a pair node that contains two numeric values,
each of which is |independent|, |dependent|, |mp_proto_dependent|, or |known|.

\smallskip\hang
|cur_type=mp_known| means that |cur_exp| is a |scaled| value.

\smallskip\hang
|cur_type=mp_dependent| means that |cur_exp| points to a capsule node whose type
is |dependent|. The |dep_list| field in this capsule points to the associated
dependency list.

\smallskip\hang
|cur_type=mp_proto_dependent| means that |cur_exp| points to a |mp_proto_dependent|
capsule node. The |dep_list| field in this capsule
points to the associated dependency list.

\smallskip\hang
|cur_type=independent| means that |cur_exp| points to a capsule node
whose type is |independent|. This somewhat unusual case can arise, for
example, in the expression
`$x+\&{begingroup}\penalty0\,\&{string}\,x; 0\,\&{endgroup}$'.

\smallskip\hang
|cur_type=mp_token_list| means that |cur_exp| points to a linked list of
tokens. 

\smallskip\noindent
The possible settings of |cur_type| have been listed here in increasing
numerical order. Notice that |cur_type| will never be |mp_numeric_type| or
|suffixed_macro| or |mp_unsuffixed_macro|, although variables of those types
are allowed.  Conversely, \MP\ has no variables of type |mp_vacuous| or
|token_list|.

@ Capsules are non-symbolic nodes that have a similar meaning
to |cur_type| and |cur_exp|. Such nodes have |name_type=capsule|,
and their |type| field is one of the possibilities for |cur_type| listed above.
Also |link<=void| in capsules that aren't part of a token list.

The |value| field of a capsule is, in most cases, the value that
corresponds to its |type|, as |cur_exp| corresponds to |cur_type|.
However, when |cur_exp| would point to a capsule,
no extra layer of indirection is present; the |value|
field is what would have been called |value(cur_exp)| if it had not been
encapsulated.  Furthermore, if the type is |dependent| or
|mp_proto_dependent|, the |value| field of a capsule is replaced by
|dep_list| and |prev_dep| fields, since dependency lists in capsules are
always part of the general |dep_list| structure.

The |get_x_next| routine is careful not to change the values of |cur_type|
and |cur_exp| when it gets an expanded token. However, |get_x_next| might
call a macro, which might parse an expression, which might execute lots of
commands in a group; hence it's possible that |cur_type| might change
from, say, |mp_unknown_boolean| to |mp_boolean_type|, or from |dependent| to
|known| or |independent|, during the time |get_x_next| is called. The
programs below are careful to stash sensitive intermediate results in
capsules, so that \MP's generality doesn't cause trouble.

Here's a procedure that illustrates these conventions. It takes
the contents of $(|cur_type|\kern-.3pt,|cur_exp|\kern-.3pt)$
and stashes them away in a
capsule. It is not used when |cur_type=mp_token_list|.
After the operation, |cur_type=mp_vacuous|; hence there is no need to
copy path lists or to update reference counts, etc.

The special link |MP_VOID| is put on the capsule returned by
|stash_cur_exp|, because this procedure is used to store macro parameters
that must be easily distinguishable from token lists.

@<Declare the stashing/unstashing routines@>=
static mp_node mp_stash_cur_exp (MP mp) {
  mp_node p;    /* the capsule that will be returned */
  mp_variable_type exp_type = mp->cur_exp.type;
  switch (exp_type) {
  case unknown_types:
  case mp_transform_type:
  case mp_color_type:
  case mp_pair_type:
  case mp_dependent:
  case mp_proto_dependent:
  case mp_independent:
  case mp_cmykcolor_type:
    p = cur_exp_node ();
    break;
    /* |case mp_path_type: case mp_pen_type: case mp_string_type:| */
  default:
    p = mp_get_value_node (mp);
    mp_name_type (p) = mp_capsule;
    mp_type (p) = mp->cur_exp.type;
    set_value_number (p, cur_exp_value_number ());    /* this also resets the rest to 0/NULL */
    if (cur_exp_str ())  {
      set_value_str (p, cur_exp_str ());
    } else if (cur_exp_knot ()) {
      set_value_knot (p, cur_exp_knot ());
    } else if (cur_exp_node ()) {
      set_value_node (p, cur_exp_node ());
    }
    break;
  }
  mp->cur_exp.type = mp_vacuous;
  mp_link (p) = MP_VOID;
  return p;
}


@ The inverse of |stash_cur_exp| is the following procedure, which
deletes an unnecessary capsule and puts its contents into |cur_type|
and |cur_exp|.

The program steps of \MP\ can be divided into two categories: those in
which |cur_type| and |cur_exp| are ``alive'' and those in which they are
``dead,'' in the sense that |cur_type| and |cur_exp| contain relevant
information or not. It's important not to ignore them when they're alive,
and it's important not to pay attention to them when they're dead.

There's also an intermediate category: If |cur_type=mp_vacuous|, then
|cur_exp| is irrelevant, hence we can proceed without caring if |cur_type|
and |cur_exp| are alive or dead. In such cases we say that |cur_type|
and |cur_exp| are {\sl dormant}. It is permissible to call |get_x_next|
only when they are alive or dormant.

The \\{stash} procedure above assumes that |cur_type| and |cur_exp|
are alive or dormant. The \\{unstash} procedure assumes that they are
dead or dormant; it resuscitates them.

@<Declare the stashing/unstashing...@>=
static void mp_unstash_cur_exp (MP mp, mp_node p);

@ @c
void mp_unstash_cur_exp (MP mp, mp_node p) {
  mp->cur_exp.type = mp_type (p);
  switch (mp->cur_exp.type) {
  case unknown_types:
  case mp_transform_type:
  case mp_color_type:
  case mp_pair_type:
  case mp_dependent:
  case mp_proto_dependent:
  case mp_independent:
  case mp_cmykcolor_type:
    set_cur_exp_node (p);
    break;
  case mp_token_list: /* this is how symbols are stashed */
    set_cur_exp_node (value_node(p));
    mp_free_value_node (mp, p);
    break;
  case mp_path_type:
  case mp_pen_type:
    set_cur_exp_knot (value_knot (p));
    mp_free_value_node (mp, p);
    break;
  case mp_string_type:
    set_cur_exp_str (value_str (p));
    mp_free_value_node (mp, p);
    break;
  case mp_picture_type:
    set_cur_exp_node (value_node (p));
    mp_free_value_node (mp, p);
    break;
  case mp_boolean_type:
  case mp_known:
    set_cur_exp_value_number (value_number (p));
    mp_free_value_node (mp, p);
    break;
  default:
    set_cur_exp_value_number (value_number (p));
    if (value_knot(p)) {
      set_cur_exp_knot (value_knot (p));
    } else if (value_node(p)) {
      set_cur_exp_node (value_node (p));
    } else if (value_str(p)) {
      set_cur_exp_str (value_str (p));
    } 
    mp_free_value_node (mp, p);
    break;
  }
}


@ The following procedure prints the values of expressions in an
abbreviated format. If its first parameter |p| is NULL, the value of
|(cur_type,cur_exp)| is displayed; otherwise |p| should be a capsule
containing the desired value. The second parameter controls the amount of
output. If it is~0, dependency lists will be abbreviated to
`\.{linearform}' unless they consist of a single term.  If it is greater
than~1, complicated structures (pens, pictures, and paths) will be displayed
in full.
@.linearform@>

@<Declarations@>=
@<Declare the procedure called |print_dp|@>;
@<Declare the stashing/unstashing routines@>;
static void mp_print_exp (MP mp, mp_node p, quarterword verbosity);

@ @c
void mp_print_exp (MP mp, mp_node p, quarterword verbosity) {
  boolean restore_cur_exp;      /* should |cur_exp| be restored? */
  mp_variable_type t;   /* the type of the expression */
  mp_number vv;       /* the value of the expression */
  mp_node v = NULL;
  new_number (vv);
  if (p != NULL) {
    restore_cur_exp = false;
  } else {
    p = mp_stash_cur_exp (mp);
    restore_cur_exp = true;
  }
  t = mp_type (p);
  if (t < mp_dependent) {       /* no dep list, could be a capsule */
    if (t != mp_vacuous && t != mp_known && value_node (p) != NULL)
      v = value_node (p);
    else
      number_clone (vv, value_number (p));
  } else if (t < mp_independent) {
    v = (mp_node) dep_list ((mp_value_node) p);
  }
  @<Print an abbreviated value of |v| or |vv| with format depending on |t|@>;
  if (restore_cur_exp)
    mp_unstash_cur_exp (mp, p);
  free_number (vv);
}


@ @<Print an abbreviated value of |v| or |vv| with format depending on |t|@>=
switch (t) {
case mp_vacuous:
  mp_print (mp, "vacuous");
  break;
case mp_boolean_type:
  if (number_to_boolean (vv) == mp_true_code)
    mp_print (mp, "true");
  else
    mp_print (mp, "false");
  break;
case unknown_types:
case mp_numeric_type:
  @<Display a variable that's been declared but not defined@>;
  break;
case mp_string_type:
  mp_print_char (mp, xord ('"'));
  mp_print_str (mp, value_str (p));
  mp_print_char (mp, xord ('"'));
  break;
case mp_pen_type:
case mp_path_type:
case mp_picture_type:
  @<Display a complex type@>;
  break;
case mp_transform_type:
  if (number_zero (vv) && v == NULL)
    mp_print_type (mp, t);
  else
    @<Display a transform node@>;
  break;
case mp_color_type:
  if (number_zero (vv) && v == NULL)
    mp_print_type (mp, t);
  else
    @<Display a color node@>;
  break;
case mp_pair_type:
  if (number_zero (vv) && v == NULL)
    mp_print_type (mp, t);
  else
    @<Display a pair node@>;
  break;
case mp_cmykcolor_type:
  if (number_zero (vv) && v == NULL)
    mp_print_type (mp, t);
  else
    @<Display a cmykcolor node@>;
  break;
case mp_known:
  print_number (vv);
  break;
case mp_dependent:
case mp_proto_dependent:
  mp_print_dp (mp, t, (mp_value_node) v, verbosity);
  break;
case mp_independent:
  mp_print_variable_name (mp, p);
  break;
default:
  mp_confusion (mp, "exp");
  break;
@:this can't happen exp}{\quad exp@>
}


@ @<Display big node item |v|@>=
{
  if (mp_type (v) == mp_known)
    print_number (value_number (v));
  else if (mp_type (v) == mp_independent)
    mp_print_variable_name (mp, v);
  else
    mp_print_dp (mp, mp_type (v), (mp_value_node) dep_list ((mp_value_node) v),
                 verbosity);
}


@ In these cases, |v| starts as the big node.

@<Display a pair node@>=
{
  mp_node vvv = v;
  mp_print_char (mp, xord ('('));
  /* clang: dereference of null pointer */ assert(vvv);
  v = x_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = y_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (')'));
}


@ @<Display a transform node@>=
{
  mp_node vvv = v;
  mp_print_char (mp, xord ('('));
  /* clang: dereference of null pointer */ assert(vvv);
  v = tx_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = ty_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = xx_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = xy_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = yx_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = yy_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (')'));
}


@ @<Display a color node@>=
{
  mp_node vvv = v;
  mp_print_char (mp, xord ('('));
  /* clang: dereference of null pointer */ assert(vvv);
  v = red_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = green_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = blue_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (')'));
}


@ @<Display a cmykcolor node@>=
{
  mp_node vvv = v;
  mp_print_char (mp, xord ('('));
  /* clang: dereference of null pointer */ assert(vvv);
  v = cyan_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = magenta_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = yellow_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (','));
  v = black_part (vvv);
  @<Display big node item |v|@>;
  mp_print_char (mp, xord (')'));
}


@ Values of type \&{picture}, \&{path}, and \&{pen} are displayed verbosely
in the log file only, unless the user has given a positive value to
\\{tracingonline}.

@<Display a complex type@>=
if (verbosity <= 1) {
  mp_print_type (mp, t);
} else {
  if (mp->selector == term_and_log)
    if (number_nonpositive (internal_value (mp_tracing_online))) {
      mp->selector = term_only;
      mp_print_type (mp, t);
      mp_print (mp, " (see the transcript file)");
      mp->selector = term_and_log;
    };
  switch (t) {
  case mp_pen_type:
    mp_print_pen (mp, value_knot (p), "", false);
    break;
  case mp_path_type:
    mp_print_path (mp, value_knot (p), "", false);
    break;
  case mp_picture_type:
    mp_print_edges (mp, v, "", false);
    break;
  default:
    break;
  }
}


@ @<Declare the procedure called |print_dp|@>=
static void mp_print_dp (MP mp, quarterword t, mp_value_node p,
                         quarterword verbosity) {
  mp_value_node q;      /* the node following |p| */
  q = (mp_value_node) mp_link (p);
  if ((dep_info (q) == NULL) || (verbosity > 0))
    mp_print_dependency (mp, p, t);
  else
    mp_print (mp, "linearform");
}


@ The displayed name of a variable in a ring will not be a capsule unless
the ring consists entirely of capsules.

@<Display a variable that's been declared but not defined@>=
{
  mp_print_type (mp, t);
  if (v != NULL) {
    mp_print_char (mp, xord (' '));
    while ((mp_name_type (v) == mp_capsule) && (v != p))
      v = value_node (v);
    mp_print_variable_name (mp, v);
  };
}


@ When errors are detected during parsing, it is often helpful to
display an expression just above the error message, using |disp_err| 
just before |mp_error|.

@<Declarations@>=
static void mp_disp_err (MP mp, mp_node p);

@ @c
void mp_disp_err (MP mp, mp_node p) {
  if (mp->interaction == mp_error_stop_mode)
    wake_up_terminal();
  mp_print_nl (mp, ">> ");
@.>>@>;
  mp_print_exp (mp, p, 1);      /* ``medium verbose'' printing of the expression */
}


@ If |cur_type| and |cur_exp| contain relevant information that should
be recycled, we will use the following procedure, which changes |cur_type|
to |known| and stores a given value in |cur_exp|. We can think of |cur_type|
and |cur_exp| as either alive or dormant after this has been done,
because |cur_exp| will not contain a pointer value.

@ @c
void mp_flush_cur_exp (MP mp, mp_value v) {
  if (is_number(mp->cur_exp.data.n)) {
    free_number(mp->cur_exp.data.n);
  }
  switch (mp->cur_exp.type) {
  case unknown_types:
  case mp_transform_type:
  case mp_color_type:
  case mp_pair_type:
  case mp_dependent:
  case mp_proto_dependent:
  case mp_independent:
  case mp_cmykcolor_type:
    mp_recycle_value (mp, cur_exp_node ());
    mp_free_value_node (mp, cur_exp_node ());
    break;
  case mp_string_type:
    delete_str_ref (cur_exp_str ());
    break;
  case mp_pen_type:
  case mp_path_type:
    mp_toss_knot_list (mp, cur_exp_knot ());
    break;
  case mp_picture_type:
    delete_edge_ref (cur_exp_node ());
    break;
  default:
    break;
  }
  mp->cur_exp = v;
  mp->cur_exp.type = mp_known;
}


@ There's a much more general procedure that is capable of releasing
the storage associated with any non-symbolic value packet.

@<Declarations@>=
static void mp_recycle_value (MP mp, mp_node p);

@ @c
static void mp_recycle_value (MP mp, mp_node p) {
  mp_variable_type t;   /* a type code */
  FUNCTION_TRACE2 ("mp_recycle_value(%p)\n", p);
  t = mp_type (p);
  switch (t) {
  case mp_vacuous:
  case mp_boolean_type:
  case mp_known:
  case mp_numeric_type:
    break;
  case unknown_types:
    mp_ring_delete (mp, p);
    break;
  case mp_string_type:
    delete_str_ref (value_str (p));
    break;
  case mp_path_type:
  case mp_pen_type:
    mp_toss_knot_list (mp, value_knot (p));
    break;
  case mp_picture_type:
    delete_edge_ref (value_node (p));
    break;
  case mp_cmykcolor_type:
    if (value_node (p) != NULL) {
      mp_recycle_value (mp, cyan_part (value_node (p)));
      mp_recycle_value (mp, magenta_part (value_node (p)));
      mp_recycle_value (mp, yellow_part (value_node (p)));
      mp_recycle_value (mp, black_part (value_node (p)));
      mp_free_value_node (mp, cyan_part (value_node (p)));
      mp_free_value_node (mp, magenta_part (value_node (p)));
      mp_free_value_node (mp, black_part (value_node (p)));
      mp_free_value_node (mp, yellow_part (value_node (p)));
      mp_free_node (mp, value_node (p), cmykcolor_node_size);
    }
    break;
  case mp_pair_type:
    if (value_node (p) != NULL) {
      mp_recycle_value (mp, x_part (value_node (p)));
      mp_recycle_value (mp, y_part (value_node (p)));
      mp_free_value_node (mp, x_part (value_node (p)));
      mp_free_value_node (mp, y_part (value_node (p)));
      mp_free_pair_node (mp, value_node (p));
    }
    break;
  case mp_color_type:
    if (value_node (p) != NULL) {
      mp_recycle_value (mp, red_part (value_node (p)));
      mp_recycle_value (mp, green_part (value_node (p)));
      mp_recycle_value (mp, blue_part (value_node (p)));
      mp_free_value_node (mp, red_part (value_node (p)));
      mp_free_value_node (mp, green_part (value_node (p)));
      mp_free_value_node (mp, blue_part (value_node (p)));
      mp_free_node (mp, value_node (p), color_node_size);
    }
    break;
  case mp_transform_type:
    if (value_node (p) != NULL) {
      mp_recycle_value (mp, tx_part (value_node (p)));
      mp_recycle_value (mp, ty_part (value_node (p)));
      mp_recycle_value (mp, xx_part (value_node (p)));
      mp_recycle_value (mp, xy_part (value_node (p)));
      mp_recycle_value (mp, yx_part (value_node (p)));
      mp_recycle_value (mp, yy_part (value_node (p)));
      mp_free_value_node (mp, tx_part (value_node (p)));
      mp_free_value_node (mp, ty_part (value_node (p)));
      mp_free_value_node (mp, xx_part (value_node (p)));
      mp_free_value_node (mp, xy_part (value_node (p)));
      mp_free_value_node (mp, yx_part (value_node (p)));
      mp_free_value_node (mp, yy_part (value_node (p)));
      mp_free_node (mp, value_node (p), transform_node_size);
    }
    break;
  case mp_dependent:
  case mp_proto_dependent:
    /* Recycle a dependency list */
    {
      mp_value_node qq = (mp_value_node) dep_list ((mp_value_node) p);
      while (dep_info (qq) != NULL)
        qq = (mp_value_node) mp_link (qq);
      set_mp_link (prev_dep ((mp_value_node) p), mp_link (qq));
      set_prev_dep (mp_link (qq), prev_dep ((mp_value_node) p));
      set_mp_link (qq, NULL);
      mp_flush_node_list (mp, (mp_node) dep_list ((mp_value_node) p));
    }
    break;
  case mp_independent:
    @<Recycle an independent variable@>;
    break;
  case mp_token_list:
  case mp_structured:
    mp_confusion (mp, "recycle");
    break;
  case mp_unsuffixed_macro:
  case mp_suffixed_macro:
    mp_delete_mac_ref (mp, value_node (p));
    break;
  default: /* there are no other valid cases, but please the compiler */
    break;
  } 
  mp_type (p) = mp_undefined;
}

@ When an independent variable disappears, it simply fades away, unless
something depends on it. In the latter case, a dependent variable whose
coefficient of dependence is maximal will take its place.
The relevant algorithm is due to Ignacio~A. Zabala, who implemented it
as part of his Ph.n->data. thesis (Stanford University, December 1982).
@^Zabala Salelles, Ignacio Andr\'es@>

For example, suppose that variable $x$ is being recycled, and that the
only variables depending on~$x$ are $y=2x+a$ and $z=x+b$. In this case
we want to make $y$ independent and $z=.5y-.5a+b$; no other variables
will depend on~$y$. If $\\{tracingequations}>0$ in this situation,
we will print `\.{\#\#\# -2x=-y+a}'.

There's a slight complication, however: An independent variable $x$
can occur both in dependency lists and in proto-dependency lists.
This makes it necessary to be careful when deciding which coefficient
is maximal.

Furthermore, this complication is not so slight when
a proto-dependent variable is chosen to become independent. For example,
suppose that $y=2x+100a$ is proto-dependent while $z=x+b$ is dependent;
then we must change $z=.5y-50a+b$ to a proto-dependency, because of the
large coefficient `50'.

In order to deal with these complications without wasting too much time,
we shall link together the occurrences of~$x$ among all the linear
dependencies, maintaining separate lists for the dependent and
proto-dependent cases.

@<Recycle an independent variable@>=
{
  mp_value_node q, r, s;
  mp_node pp;   /* link manipulation register */
  mp_number v ;        /* a value */
  mp_number test;      /* a temporary value */
  new_number (test);
  new_number (v);
  if (t < mp_dependent) 	 
    number_clone (v, value_number (p));
  set_number_to_zero(mp->max_c[mp_dependent]);
  set_number_to_zero(mp->max_c[mp_proto_dependent]);
  mp->max_link[mp_dependent] = NULL;
  mp->max_link[mp_proto_dependent] = NULL;
  q = (mp_value_node) mp_link (mp->dep_head);
  while (q != mp->dep_head) {
    s = (mp_value_node) mp->temp_head;
    set_mp_link (s, dep_list (q));
    while (1) {
      r = (mp_value_node) mp_link (s);
      if (dep_info (r) == NULL)
        break;
      if (dep_info (r) != p) {
        s = r;
      } else {
        t = mp_type (q);
        if (mp_link (s) == dep_list (q)) {      /* reset the |dep_list| */
          set_dep_list (q, mp_link (r));
        }
        set_mp_link (s, mp_link (r));
        set_dep_info (r, (mp_node) q);
        number_clone (test, dep_value (r));
        number_abs (test);
        if (number_greater (test, mp->max_c[t])) {
          /* Record a new maximum coefficient of type |t| */
          if (number_positive(mp->max_c[t])) {
            set_mp_link (mp->max_ptr[t], (mp_node) mp->max_link[t]);
            mp->max_link[t] = mp->max_ptr[t];
          }
          number_clone (mp->max_c[t], test);
          mp->max_ptr[t] = r;
        } else {
          set_mp_link (r, (mp_node) mp->max_link[t]);
          mp->max_link[t] = r;
        }
      }
    }
    q = (mp_value_node) mp_link (r);
  }
  if (number_positive(mp->max_c[mp_dependent]) || number_positive(mp->max_c[mp_proto_dependent])) {
    /* Choose a dependent variable to take the place of the disappearing
       independent variable, and change all remaining dependencies
       accordingly */
    mp_number test, ret; /* temporary use */
    new_number (ret);
    new_number (test);
    number_clone (test, mp->max_c[mp_dependent]);
    number_divide_int (test, 4096);
    if (number_greaterequal(test, mp->max_c[mp_proto_dependent]))
      t = mp_dependent;
    else
      t = mp_proto_dependent;

    /* Let |s=max_ptr[t]|. At this point we have $|value|(s)=\pm|max_c|[t]$,
       and |dep_info(s)| points to the dependent variable~|pp| of type~|t| from
       whose dependency list we have removed node~|s|. We must reinsert
       node~|s| into the dependency list, with coefficient $-1.0$, and with
       |pp| as the new independent variable. Since |pp| will have a larger serial
       number than any other variable, we can put node |s| at the head of the
       list. */
    /* Determine the dependency list |s| to substitute for the independent
       variable~|p| */

    s = mp->max_ptr[t];
    pp = (mp_node) dep_info (s);
    number_clone (v, dep_value (s));
    if (t == mp_dependent) {
      set_dep_value (s, fraction_one_t);
    } else {
      set_dep_value (s, unity_t);
    }
    number_negate(dep_value(s));
    r = (mp_value_node) dep_list ((mp_value_node) pp);
    set_mp_link (s, (mp_node) r);
    while (dep_info (r) != NULL)
      r = (mp_value_node) mp_link (r);
    q = (mp_value_node) mp_link (r);
    set_mp_link (r, NULL);
    set_prev_dep (q, prev_dep ((mp_value_node) pp));
    set_mp_link (prev_dep ((mp_value_node) pp), (mp_node) q);
    mp_new_indep (mp, pp);
    if (cur_exp_node () == pp && mp->cur_exp.type == t)
      mp->cur_exp.type = mp_independent;
    if (number_positive (internal_value (mp_tracing_equations))) {
      /* Show the transformed dependency */
      if (mp_interesting (mp, p)) {
        mp_begin_diagnostic (mp);
        mp_show_transformed_dependency(mp, v, t, p);
        mp_print_dependency (mp, s, t);
        mp_end_diagnostic (mp, false);
      }
    }

    t = (quarterword) (mp_dependent + mp_proto_dependent - t);    /* complement |t| */
    if (number_positive(mp->max_c[t])) {
      /* we need to pick up an unchosen dependency */
      set_mp_link (mp->max_ptr[t], (mp_node) mp->max_link[t]);
      mp->max_link[t] = mp->max_ptr[t];
    }
    /* Finally, there are dependent and proto-dependent variables whose
       dependency lists must be brought up to date. */
    if (t != mp_dependent) {
      /* Substitute new dependencies in place of |p| */
      for (t = mp_dependent; t <= mp_proto_dependent; t=t+1) {
        r = mp->max_link[t];
        while (r != NULL) {
          q = (mp_value_node) dep_info (r);
          number_clone (test, v);
          number_negate (test);
          make_fraction (ret, dep_value (r), test);
          set_dep_list (q, mp_p_plus_fq (mp, (mp_value_node) dep_list (q), ret, s, t, mp_dependent));
          if (dep_list (q) == (mp_node) mp->dep_final)
            mp_make_known (mp, q, mp->dep_final);
          q = r;
          r = (mp_value_node) mp_link (r);
          mp_free_dep_node (mp, q);
        }
      }
    } else {
      /* Substitute new proto-dependencies in place of |p| */
      for (t = mp_dependent; t <= mp_proto_dependent; t=t+1) {
        r = mp->max_link[t];
        while (r != NULL) {
          q = (mp_value_node) dep_info (r);
          if (t == mp_dependent) {    /* for safety's sake, we change |q| to |mp_proto_dependent| */
            if (cur_exp_node () == (mp_node) q && mp->cur_exp.type == mp_dependent)
              mp->cur_exp.type = mp_proto_dependent;
            set_dep_list (q, mp_p_over_v (mp, (mp_value_node) dep_list (q),
                                           unity_t, mp_dependent,
                                           mp_proto_dependent));
            mp_type (q) = mp_proto_dependent;
            fraction_to_round_scaled (dep_value (r));
          }
          number_clone (test, v);
          number_negate (test);
          make_scaled (ret, dep_value (r), test);
          set_dep_list (q, mp_p_plus_fq (mp, (mp_value_node) dep_list (q),
                                             ret, s,
                                             mp_proto_dependent,
                                             mp_proto_dependent));
          if (dep_list (q) == (mp_node) mp->dep_final)
            mp_make_known (mp, q, mp->dep_final);
          q = r;
          r = (mp_value_node) mp_link (r);
          mp_free_dep_node (mp, q);
        }
      }
    }
    mp_flush_node_list (mp, (mp_node) s);
    if (mp->fix_needed)
      mp_fix_dependencies (mp);
    check_arith();
    free_number (ret);
  }
  free_number (v);
  free_number(test);
}

@ @<Declarations@>=
static void mp_show_transformed_dependency(MP mp, mp_number v, mp_variable_type t, mp_node p);

@ @c
static void mp_show_transformed_dependency(MP mp, mp_number v, mp_variable_type t, mp_node p)
{
  mp_number vv;   /* for temp use */
  new_number (vv);
  mp_print_nl (mp, "### ");
  if (number_positive(v))
    mp_print_char (mp, xord ('-'));
  if (t == mp_dependent) {
    number_clone (vv, mp->max_c[mp_dependent]);
    fraction_to_round_scaled (vv);
  } else {
    number_clone (vv, mp->max_c[mp_proto_dependent]);
  }
  if (!number_equal(vv, unity_t)) {
    print_number (vv);
  }
  mp_print_variable_name (mp, p);
  while (indep_scale (p) > 0) {
    mp_print (mp, "*4");
    set_indep_scale(p, indep_scale(p)-2);
  }
  if (t == mp_dependent)
    mp_print_char (mp, xord ('='));
  else
    mp_print (mp, " = ");
  free_number (vv);
}


@ The code for independency removal makes use of three non-symbolic arrays.

@<Glob...@>=
mp_number max_c[mp_proto_dependent + 1];  /* max coefficient magnitude */
mp_value_node max_ptr[mp_proto_dependent + 1];  /* where |p| occurs with |max_c| */
mp_value_node max_link[mp_proto_dependent + 1]; /* other occurrences of |p| */


@ @<Initialize table ... @>=
{
  int i;
  for (i=0;i<mp_proto_dependent + 1;i++) {
    new_number (mp->max_c[i]);
  }
}

@ @<Dealloc...@>=
{
  int i;
  for (i=0;i<mp_proto_dependent + 1;i++) {
    free_number (mp->max_c[i]);
  }
}

@ A global variable |var_flag| is set to a special command code
just before \MP\ calls |scan_expression|, if the expression should be
treated as a variable when this command code immediately follows. For
example, |var_flag| is set to |assignment| at the beginning of a
statement, because we want to know the {\sl location\/} of a variable at
the left of `\.{:=}', not the {\sl value\/} of that variable.

The |scan_expression| subroutine calls |scan_tertiary|,
which calls |scan_secondary|, which calls |scan_primary|, which sets
|var_flag:=0|. In this way each of the scanning routines ``knows''
when it has been called with a special |var_flag|, but |var_flag| is
usually zero.

A variable preceding a command that equals |var_flag| is converted to a
token list rather than a value. Furthermore, an `\.{=}' sign following an
expression with |var_flag=assignment| is not considered to be a relation
that produces boolean expressions.


@<Glob...@>=
int var_flag;   /* command that wants a variable */

@ @<Set init...@>=
mp->var_flag = 0;

@* Parsing primary expressions.
The first parsing routine, |scan_primary|, is also the most complicated one,
since it involves so many different cases. But each case---with one
exception---is fairly simple by itself.

When |scan_primary| begins, the first token of the primary to be scanned
should already appear in |cur_cmd|, |cur_mod|, and |cur_sym|. The values
of |cur_type| and |cur_exp| should be either dead or dormant, as explained
earlier. If |cur_cmd| is not between |min_primary_command| and
|max_primary_command|, inclusive, a syntax error will be signaled.

Later we'll come to procedures that perform actual operations like
addition, square root, and so on; our purpose now is to do the parsing.
But we might as well mention those future procedures now, so that the
suspense won't be too bad:

\smallskip
|do_nullary(c)| does primitive operations that have no operands (e.g.,
`\&{true}' or `\&{pencircle}');

\smallskip
|do_unary(c)| applies a primitive operation to the current expression;

\smallskip
|do_binary(p,c)| applies a primitive operation to the capsule~|p|
and the current expression.

@<Declare the basic parsing subroutines@>=
static void check_for_mediation (MP mp);
void mp_scan_primary (MP mp) {
  mp_command_code my_var_flag;      /* initial value of |my_var_flag| */
  my_var_flag = mp->var_flag;
  mp->var_flag = 0;
RESTART:
  check_arith();
  /* Supply diagnostic information, if requested */
  if (mp->interrupt != 0) {
    if (mp->OK_to_interrupt) {
      mp_back_input (mp);
      check_interrupt;
      mp_get_x_next (mp);
    }
  }
  switch (cur_cmd()) {
  case mp_left_delimiter:
  {
    /* Scan a delimited primary */
    mp_node p, q, r;      /* for list manipulation */
    mp_sym l_delim, r_delim;      /* hash addresses of a delimiter pair */
    l_delim = cur_sym();
    r_delim = equiv_sym (cur_sym());
    mp_get_x_next (mp);
    mp_scan_expression (mp);
    if ((cur_cmd() == mp_comma) && (mp->cur_exp.type >= mp_known)) {
      /* Scan the rest of a delimited set of numerics */
      /* This code uses the fact that |red_part| and |green_part|
         are synonymous with |x_part| and |y_part|. */
      p = mp_stash_cur_exp (mp);
      mp_get_x_next (mp);
      mp_scan_expression (mp);
      /* Make sure the second part of a pair or color has a numeric type */
      if (mp->cur_exp.type < mp_known) {
        const char *hlp[] = {
               "I've started to scan a pair `(a,b)' or a color `(a,b,c)';",
               "but after finding a nice `a' I found a `b' that isn't",
               "of numeric type. So I've changed that part to zero.",
               "(The b that I didn't like appears above the error message.)",
               NULL };
        mp_value new_expr;
        memset(&new_expr,0,sizeof(mp_value));
        mp_disp_err(mp, NULL);
        new_number(new_expr.data.n);
        set_number_to_zero(new_expr.data.n);
        mp_back_error (mp,"Nonnumeric ypart has been replaced by 0", hlp, true);
        mp_get_x_next (mp);
        mp_flush_cur_exp (mp, new_expr);
      }

      q = mp_get_value_node (mp);
      mp_name_type (q) = mp_capsule;
      if (cur_cmd() == mp_comma) {
        mp_init_color_node (mp, q);
        r = value_node (q);
        mp_stash_in (mp, y_part (r));
        mp_unstash_cur_exp (mp, p);
        mp_stash_in (mp, x_part (r));
        /* Scan the last of a triplet of numerics */
        mp_get_x_next (mp);
        mp_scan_expression (mp);
        if (mp->cur_exp.type < mp_known) {
          mp_value new_expr;
          const char *hlp[] = {
              "I've just scanned a color `(a,b,c)' or cmykcolor(a,b,c,d); but the `c'",
              "isn't of numeric type. So I've changed that part to zero.",
              "(The c that I didn't like appears above the error message.)",
              NULL };
          memset(&new_expr,0,sizeof(mp_value));
          mp_disp_err(mp, NULL);
          new_number(new_expr.data.n);
          set_number_to_zero(new_expr.data.n);
          mp_back_error (mp,"Nonnumeric third part has been replaced by 0", hlp, true);
          mp_get_x_next (mp);
          mp_flush_cur_exp (mp, new_expr);
        }
        mp_stash_in (mp, blue_part (r));

        if (cur_cmd() == mp_comma) {
          mp_node t;      /* a token */
          mp_init_cmykcolor_node (mp, q);
          t = value_node (q);
          mp_type (cyan_part (t)) = mp_type (red_part (r));
          set_value_number (cyan_part (t), value_number (red_part (r)));
          mp_type (magenta_part (t)) = mp_type (green_part (r));
          set_value_number (magenta_part (t), value_number (green_part (r)));
          mp_type (yellow_part (t)) = mp_type (blue_part (r));
          set_value_number (yellow_part (t), value_number (blue_part (r)));
          mp_recycle_value (mp, r);
          r = t;
          /* Scan the last of a quartet of numerics */
          mp_get_x_next (mp);
          mp_scan_expression (mp);
          if (mp->cur_exp.type < mp_known) {
            const char *hlp[] = {
                   "I've just scanned a cmykcolor `(c,m,y,k)'; but the `k' isn't",
                   "of numeric type. So I've changed that part to zero.",
                   "(The k that I didn't like appears above the error message.)",
                   NULL };
            mp_value new_expr;
            memset(&new_expr,0,sizeof(mp_value));
            new_number(new_expr.data.n);
            mp_disp_err(mp, NULL); 
            set_number_to_zero(new_expr.data.n);
            mp_back_error (mp,"Nonnumeric blackpart has been replaced by 0", hlp, true);
            mp_get_x_next (mp);
            mp_flush_cur_exp (mp, new_expr);
          }
          mp_stash_in (mp, black_part (r));      

        }
      } else {
        mp_init_pair_node (mp, q);
        r = value_node (q);
        mp_stash_in (mp, y_part (r));
        mp_unstash_cur_exp (mp, p);
        mp_stash_in (mp, x_part (r));
      }
      mp_check_delimiter (mp, l_delim, r_delim);
      mp->cur_exp.type = mp_type (q);
      set_cur_exp_node (q);

    } else {
      mp_check_delimiter (mp, l_delim, r_delim);
    }
  }
    break;
  case mp_begin_group:
    /* Scan a grouped primary */
    /* The local variable |group_line| keeps track of the line
       where a \&{begingroup} command occurred; this will be useful
       in an error message if the group doesn't actually end. */
    {
      integer group_line;     /* where a group began */
      group_line = mp_true_line (mp);
      if (number_positive (internal_value (mp_tracing_commands)))
        show_cur_cmd_mod;
      mp_save_boundary (mp);
      do {
        mp_do_statement (mp);       /* ends with |cur_cmd>=semicolon| */
      } while (cur_cmd() == mp_semicolon);
      if (cur_cmd() != mp_end_group) {
        char msg[256];
        const char *hlp[] = {
               "I saw a `begingroup' back there that hasn't been matched",
               "by `endgroup'. So I've inserted `endgroup' now.",
               NULL };
        mp_snprintf(msg, 256, "A group begun on line %d never ended", (int)group_line);
        mp_back_error (mp, msg, hlp, true);
        set_cur_cmd((mp_variable_type)mp_end_group);
      }
      mp_unsave (mp);
      /* this might change |cur_type|, if independent variables are recycled */
      if (number_positive (internal_value (mp_tracing_commands)))
        show_cur_cmd_mod;
    }
    break;
  case mp_string_token:
    /* Scan a string constant */
    mp->cur_exp.type = mp_string_type;
    set_cur_exp_str (cur_mod_str());
    break;
  case mp_numeric_token:
  {
    /* Scan a primary that starts with a numeric token */
    /* A numeric token might be a primary by itself, or it might be the
       numerator of a fraction composed solely of numeric tokens, or it might
       multiply the primary that follows (provided that the primary doesn't begin
       with a plus sign or a minus sign). The code here uses the facts that
       |max_primary_command=plus_or_minus| and
       |max_primary_command-1=numeric_token|. If a fraction is found that is less
       than unity, we try to retain higher precision when we use it in scalar
       multiplication. */
    mp_number num, denom;      /* for primaries that are fractions, like `1/2' */
    new_number (num);
    new_number (denom);
    set_cur_exp_value_number (cur_mod_number());
    mp->cur_exp.type = mp_known;
    mp_get_x_next (mp);
    if (cur_cmd() != mp_slash) {
      set_number_to_zero(num);
      set_number_to_zero(denom);
    } else {
      mp_get_x_next (mp);
      if (cur_cmd() != mp_numeric_token) {
        mp_back_input (mp);
        set_cur_cmd((mp_variable_type)mp_slash);
        set_cur_mod(mp_over);
        set_cur_sym(mp->frozen_slash);
        free_number (num);
        free_number (denom);
        goto DONE;
      }
      number_clone (num, cur_exp_value_number ());
      number_clone (denom, cur_mod_number());
      if (number_zero(denom)) {
        /* Protest division by zero */
        const char *hlp[] = { "I'll pretend that you meant to divide by 1.", NULL };
        mp_error (mp, "Division by zero", hlp, true);
      } else {
        mp_number ret;
        new_number (ret);
        make_scaled (ret, num, denom);
        set_cur_exp_value_number (ret);
        free_number (ret);
      }
      check_arith();
      mp_get_x_next (mp);
    }
    if (cur_cmd() >= mp_min_primary_command) {
      if (cur_cmd() < mp_numeric_token) {  /* in particular, |cur_cmd<>plus_or_minus| */
        mp_node p;      /* for list manipulation */
        mp_number absnum, absdenom;
        new_number (absnum);
        new_number (absdenom);
        p = mp_stash_cur_exp (mp);
        mp_scan_primary (mp);
        number_clone (absnum, num);
        number_abs (absnum);
        number_clone (absdenom, denom);
        number_abs (absdenom);
        if (number_greaterequal(absnum, absdenom) || (mp->cur_exp.type < mp_color_type)) {
          mp_do_binary (mp, p, mp_times);
        } else {
          mp_frac_mult (mp, num, denom);
          mp_free_value_node (mp, p);
        }
        free_number (absnum);
        free_number (absdenom);
      }
    }
    free_number (num);
    free_number (denom);
    goto DONE;
  }
    break;
  case mp_nullary:
    /* Scan a nullary operation */
    mp_do_nullary (mp, (quarterword) cur_mod());
    break;
  case mp_unary:
  case mp_type_name:
  case mp_cycle:
  case mp_plus_or_minus:
  {
    /* Scan a unary operation */
    quarterword c;        /* a primitive operation code */
    c = (quarterword) cur_mod();
    mp_get_x_next (mp);
    mp_scan_primary (mp);
    mp_do_unary (mp, c);
    goto DONE;
  }
    break;
  case mp_primary_binary:
  {
    /* Scan a binary operation with `\&{of}' between its operands */
    mp_node p;      /* for list manipulation */
    quarterword c;        /* a primitive operation code */
    c = (quarterword) cur_mod();
    mp_get_x_next (mp);
    mp_scan_expression (mp);
    if (cur_cmd() != mp_of_token) {
      char msg[256];
      mp_string sname;
      const char *hlp[] = {
          "I've got the first argument; will look now for the other.",
          NULL };
      int old_setting = mp->selector;
      mp->selector = new_string;
      mp_print_cmd_mod (mp, mp_primary_binary, c);
      mp->selector = old_setting;
      sname = mp_make_string(mp);
      mp_snprintf (msg, 256, "Missing `of' has been inserted for %s", mp_str(mp, sname));
      delete_str_ref(sname);
      mp_back_error (mp, msg, hlp, true);
    }
    p = mp_stash_cur_exp (mp);
    mp_get_x_next (mp);
    mp_scan_primary (mp);
    mp_do_binary (mp, p, c);
    goto DONE;
  }
    break;
  case mp_str_op:
    /* Convert a suffix to a string */
    mp_get_x_next (mp);
    mp_scan_suffix (mp);
    mp->old_setting = mp->selector;
    mp->selector = new_string;
    mp_show_token_list (mp, cur_exp_node (), NULL, 100000, 0);
    mp_flush_token_list (mp, cur_exp_node ());
    set_cur_exp_str (mp_make_string (mp));
    mp->selector = mp->old_setting;
    mp->cur_exp.type = mp_string_type;
    goto DONE;
    break;
  case mp_internal_quantity:
    /* Scan an internal numeric quantity */
    /* If an internal quantity appears all by itself on the left of an
       assignment, we return a token list of length one, containing the address
       of the internal quantity, with |name_type| equal to |mp_internal_sym|. 
       (This accords with the conventions of the save stack, as described earlier.) */
    {
      halfword qq = cur_mod();
      if (my_var_flag == mp_assignment) {
        mp_get_x_next (mp);
        if (cur_cmd() == mp_assignment) {
          set_cur_exp_node (mp_get_symbolic_node (mp));
          set_mp_sym_info (cur_exp_node (), qq);
          mp_name_type (cur_exp_node ()) = mp_internal_sym;
          mp->cur_exp.type = mp_token_list;
          goto DONE;
        }
        mp_back_input (mp);
      }
      if (internal_type (qq) == mp_string_type) {
        set_cur_exp_str (internal_string (qq));
      } else {
        set_cur_exp_value_number (internal_value (qq));
      }
      mp->cur_exp.type = internal_type (qq);
    }
    break;
  case mp_capsule_token:
    mp_make_exp_copy (mp, cur_mod_node());
    break;
  case mp_tag_token:
    @<Scan a variable primary; |goto restart| if it turns out to be a macro@>;
    break;
  default:
    mp_bad_exp (mp, "A primary");
    goto RESTART;
    break;
  }
  mp_get_x_next (mp);           /* the routines |goto done| if they don't want this */
DONE:
  check_for_mediation (mp);
}

@  Expressions of the form `\.{a[b,c]}' are converted into
`\.{b+a*(c-b)}', without checking the types of \.b~or~\.c,
provided that \.a is numeric.

@<Declare the basic parsing subroutines@>=
static void check_for_mediation (MP mp) {
  mp_node p, q, r;      /* for list manipulation */
  if (cur_cmd() == mp_left_bracket) {
    if (mp->cur_exp.type >= mp_known) {
      /* Scan a mediation construction */
      p = mp_stash_cur_exp (mp);
      mp_get_x_next (mp);
      mp_scan_expression (mp);
      if (cur_cmd() != mp_comma) {
        /* Put the left bracket and the expression back to be rescanned */
        /* The left bracket that we thought was introducing a subscript might have
           actually been the left bracket in a mediation construction like `\.{x[a,b]}'.
           So we don't issue an error message at this point; but we do want to back up
           so as to avoid any embarrassment about our incorrect assumption. */
        mp_back_input (mp);           /* that was the token following the current expression */
        mp_back_expr (mp);
        set_cur_cmd((mp_variable_type)mp_left_bracket);
        set_cur_mod_number(zero_t);
        set_cur_sym(mp->frozen_left_bracket);
        mp_unstash_cur_exp (mp, p);
      } else {
        q = mp_stash_cur_exp (mp);
        mp_get_x_next (mp);
        mp_scan_expression (mp);
        if (cur_cmd() != mp_right_bracket) {
          const char *hlp[] = {
                 "I've scanned an expression of the form `a[b,c',",
                 "so a right bracket should have come next.",
                 "I shall pretend that one was there.",
                 NULL };
          mp_back_error (mp, "Missing `]' has been inserted", hlp, true);
        }
        r = mp_stash_cur_exp (mp);
        mp_make_exp_copy (mp, q);
        mp_do_binary (mp, r, mp_minus);
        mp_do_binary (mp, p, mp_times);
        mp_do_binary (mp, q, mp_plus);
        mp_get_x_next (mp);
      }
    }
  }
}


@ Errors at the beginning of expressions are flagged by |bad_exp|.

@c
static void mp_bad_exp (MP mp, const char *s) {
  char msg[256];
  int save_flag;
  const char *hlp[] = { 
         "I'm afraid I need some sort of value in order to continue,",
         "so I've tentatively inserted `0'. You may want to",
         "delete this zero and insert something else;",
         "see Chapter 27 of The METAFONTbook for an example.",
         NULL };
@:METAFONTbook}{\sl The {\logos METAFONT\/}book@>;
  {
     mp_string cm;
     int old_selector = mp->selector;
     mp->selector = new_string;
     mp_print_cmd_mod (mp, cur_cmd(), cur_mod());
     mp->selector = old_selector;
     cm = mp_make_string(mp);
     mp_snprintf(msg, 256, "%s expression can't begin with `%s'", s, mp_str(mp, cm));
     delete_str_ref(cm);
  }
  mp_back_input (mp);
  set_cur_sym(NULL);
  set_cur_cmd((mp_variable_type)mp_numeric_token);
  set_cur_mod_number (zero_t);
  mp_ins_error (mp, msg, hlp, true);
  save_flag = mp->var_flag;
  mp->var_flag = 0;
  mp_get_x_next (mp);
  mp->var_flag = save_flag;
}


@ The |stash_in| subroutine puts the current (numeric) expression into a field
within a ``big node.''

@c
static void mp_stash_in (MP mp, mp_node p) {
  mp_value_node q;      /* temporary register */
  mp_type (p) = mp->cur_exp.type;
  if (mp->cur_exp.type == mp_known) {
    set_value_number (p, cur_exp_value_number ());
  } else {
    if (mp->cur_exp.type == mp_independent) {
      /* Stash an independent |cur_exp| into a big node */
      /* In rare cases the current expression can become |independent|. There
         may be many dependency lists pointing to such an independent capsule,
	 so we can't simply move it into place within a big node. Instead,
	 we copy it, then recycle it. */
      q = mp_single_dependency (mp, cur_exp_node ());
      if (q == mp->dep_final) {
        mp_type (p) = mp_known;
        set_value_number (p, zero_t);
        mp_free_dep_node (mp, q);
      } else {
        mp_new_dep (mp, p, mp_dependent, q);
      }
      mp_recycle_value (mp, cur_exp_node ());
      mp_free_value_node (mp, cur_exp_node ());
    } else {
      set_dep_list ((mp_value_node) p,
                    dep_list ((mp_value_node) cur_exp_node ()));
      set_prev_dep ((mp_value_node) p,
                    prev_dep ((mp_value_node) cur_exp_node ()));
      set_mp_link (prev_dep ((mp_value_node) p), p);
      mp_free_dep_node (mp, (mp_value_node) cur_exp_node ());
    }
  }
  mp->cur_exp.type = mp_vacuous;
}

@ The most difficult part of |scan_primary| has been saved for last, since
it was necessary to build up some confidence first. We can now face the task
of scanning a variable.

As we scan a variable, we build a token list containing the relevant
names and subscript values, simultaneously following along in the
``collective'' structure to see if we are actually dealing with a macro
instead of a value.

The local variables |pre_head| and |post_head| will point to the beginning
of the prefix and suffix lists; |tail| will point to the end of the list
that is currently growing.

Another local variable, |tt|, contains partial information about the
declared type of the variable-so-far. If |tt>=mp_unsuffixed_macro|, the
relation |tt=mp_type(q)| will always hold. If |tt=undefined|, the routine
doesn't bother to update its information about type. And if
|undefined<tt<mp_unsuffixed_macro|, the precise value of |tt| isn't critical.

@ @<Scan a variable primary...@>=
{
  mp_node p, q;      /* for list manipulation */
  mp_node t;      /* a token */
  mp_node pre_head, post_head, tail; /* prefix and suffix list variables */
  quarterword tt; /* approximation to the type of the variable-so-far */
  mp_node macro_ref = 0;  /* reference count for a suffixed macro */
  pre_head = mp_get_symbolic_node (mp);
  tail = pre_head;
  post_head = NULL;
  tt = mp_vacuous;
  while (1) {
    t = mp_cur_tok (mp);
    mp_link (tail) = t;
    if (tt != mp_undefined) {
      /* Find the approximate type |tt| and corresponding~|q| */
      /* Every time we call |get_x_next|, there's a chance that the variable we've
         been looking at will disappear. Thus, we cannot safely keep |q| pointing
         into the variable structure; we need to start searching from the root each 
         time. */
      mp_sym qq;
      p = mp_link (pre_head);
      qq = mp_sym_sym (p);
      tt = mp_undefined;
      if (eq_type (qq) % mp_outer_tag == mp_tag_token) {
        q = equiv_node (qq);
        if (q == NULL)
          goto DONE2;
        while (1) {
          p = mp_link (p);
          if (p == NULL) {
            tt = mp_type (q);
            goto DONE2;
          }
          if (mp_type (q) != mp_structured)
            goto DONE2;
          q = mp_link (attr_head (q));      /* the |collective_subscript| attribute */
          if (mp_type (p) == mp_symbol_node) {      /* it's not a subscript */
            do {
              q = mp_link (q);
            } while (!(hashloc (q) >= mp_sym_sym (p)));
            if (hashloc (q) > mp_sym_sym (p))
              goto DONE2;
          }
        }
      }
    DONE2:

      if (tt >= mp_unsuffixed_macro) {
        /* Either begin an unsuffixed macro call or
          prepare for a suffixed one */
        mp_link (tail) = NULL;
        if (tt > mp_unsuffixed_macro) {       /* |tt=mp_suffixed_macro| */
          post_head = mp_get_symbolic_node (mp);
          tail = post_head;
          mp_link (tail) = t;
          tt = mp_undefined;
          macro_ref = value_node (q);
          add_mac_ref (macro_ref);
        } else {
          /* Set up unsuffixed macro call and |goto restart| */
          /* The only complication associated with macro calling is that the prefix
             and ``at'' parameters must be packaged in an appropriate list of lists. */
          p = mp_get_symbolic_node (mp);
          set_mp_sym_sym (pre_head, mp_link (pre_head));
          mp_link (pre_head) = p;
          set_mp_sym_sym (p, t);
          mp_macro_call (mp, value_node (q), pre_head, NULL);
          mp_get_x_next (mp);
          goto RESTART;
        }
      }
    }
    mp_get_x_next (mp);
    tail = t;
    if (cur_cmd() == mp_left_bracket) {
      /* Scan for a subscript; replace |cur_cmd| by |numeric_token| if found */
      mp_get_x_next (mp);
      mp_scan_expression (mp);
      if (cur_cmd() != mp_right_bracket) {
        /* Put the left bracket and the expression back to be rescanned */
        /* The left bracket that we thought was introducing a subscript might have
           actually been the left bracket in a mediation construction like `\.{x[a,b]}'.
           So we don't issue an error message at this point; but we do want to back up
           so as to avoid any embarrassment about our incorrect assumption. */
        mp_back_input (mp);           /* that was the token following the current expression */
        mp_back_expr (mp);
        set_cur_cmd((mp_variable_type)mp_left_bracket);
        set_cur_mod_number(zero_t);
        set_cur_sym(mp->frozen_left_bracket);

      } else {
        if (mp->cur_exp.type != mp_known)
          mp_bad_subscript (mp);
        set_cur_cmd((mp_variable_type)mp_numeric_token);
        set_cur_mod_number(cur_exp_value_number ());
        set_cur_sym(NULL);
      }
    }
    if (cur_cmd() > mp_max_suffix_token)
      break;
    if (cur_cmd() < mp_min_suffix_token)
      break;
  } /* now |cur_cmd| is |internal_quantity|, |tag_token|, or |numeric_token| */
   /* Handle unusual cases that masquerade as variables, and |goto restart| or
      |goto done| if appropriate; otherwise make a copy of the variable and |goto done| */
   /* If the variable does exist, we also need to check
      for a few other special cases before deciding that a plain old ordinary
      variable has, indeed, been scanned. */
  if (post_head != NULL) {
    /* Set up suffixed macro call and |goto restart| */
    /* If the ``variable'' that turned out to be a suffixed macro no longer exists,
       we don't care, because we have reserved a pointer (|macro_ref|) to its
       token list. */
    mp_back_input (mp);
    p = mp_get_symbolic_node (mp);
    q = mp_link (post_head);
    set_mp_sym_sym (pre_head, mp_link (pre_head));
    mp_link (pre_head) = post_head;
    set_mp_sym_sym (post_head, q);
    mp_link (post_head) = p;
    set_mp_sym_sym (p, mp_link (q));
    mp_link (q) = NULL;
    mp_macro_call (mp, macro_ref, pre_head, NULL);
    decr_mac_ref (macro_ref);
    mp_get_x_next (mp);
    goto RESTART;
  }
  q = mp_link (pre_head);
  mp_free_symbolic_node (mp, pre_head);
  if (cur_cmd() == my_var_flag) {
    mp->cur_exp.type = mp_token_list;
    set_cur_exp_node (q);
    goto DONE;
  }
  p = mp_find_variable (mp, q);
  if (p != NULL) {
    mp_make_exp_copy (mp, p);
  } else {
    mp_value new_expr;
    const char *hlp[] = {
      "While I was evaluating the suffix of this variable,",
      "something was redefined, and it's no longer a variable!",
      "In order to get back on my feet, I've inserted `0' instead.",
      NULL };
    char *msg = mp_obliterated (mp, q);
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    set_number_to_zero(new_expr.data.n);
    mp_back_error (mp, msg, hlp, true);
    free(msg);
    mp_get_x_next (mp);
    mp_flush_cur_exp (mp, new_expr);
  }
  mp_flush_node_list (mp, q);
  goto DONE;
}


@ Here's a routine that puts the current expression back to be read again.

@c
static void mp_back_expr (MP mp) {
  mp_node p;    /* capsule token */
  p = mp_stash_cur_exp (mp);
  mp_link (p) = NULL;
  back_list (p);
}


@ Unknown subscripts lead to the following error message.

@c
static void mp_bad_subscript (MP mp) {
  mp_value new_expr;
  const char *hlp[] = { 
         "A bracketed subscript must have a known numeric value;",
         "unfortunately, what I found was the value that appears just",
         "above this error message. So I'll try a zero subscript.",
         NULL };
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  mp_disp_err(mp, NULL);
  mp_error (mp, "Improper subscript has been replaced by zero", hlp, true);
@.Improper subscript...@>;
  mp_flush_cur_exp (mp, new_expr);
}


@ How do things stand now? Well, we have scanned an entire variable name,
including possible subscripts and/or attributes; |cur_cmd|, |cur_mod|, and
|cur_sym| represent the token that follows. If |post_head=NULL|, a
token list for this variable name starts at |mp_link(pre_head)|, with all
subscripts evaluated. But if |post_head<>NULL|, the variable turned out
to be a suffixed macro; |pre_head| is the head of the prefix list, while
|post_head| is the head of a token list containing both `\.{\AT!}' and
the suffix.

Our immediate problem is to see if this variable still exists. (Variable
structures can change drastically whenever we call |get_x_next|; users
aren't supposed to do this, but the fact that it is possible means that
we must be cautious.)

The following procedure creates an error message for when a variable
unexpectedly disappears. 

@c
static char *mp_obliterated (MP mp, mp_node q) {
  char msg[256];
  mp_string sname;
  int old_setting = mp->selector;
  mp->selector = new_string;
  mp_show_token_list (mp, q, NULL, 1000, 0);
  sname = mp_make_string(mp);
  mp->selector = old_setting;
  mp_snprintf(msg, 256, "Variable %s has been obliterated", mp_str(mp, sname));
@.Variable...obliterated@>;
  delete_str_ref(sname);
  return xstrdup(msg);
}


@ Our remaining job is simply to make a copy of the value that has been
found. Some cases are harder than others, but complexity arises solely
because of the multiplicity of possible cases.

@<Declare the procedure called |make_exp_copy|@>=
@<Declare subroutines needed by |make_exp_copy|@>;
static void mp_make_exp_copy (MP mp, mp_node p) {
  mp_node t;    /* register(s) for list manipulation */
  mp_value_node q;
RESTART:
  mp->cur_exp.type = mp_type (p);
  switch (mp->cur_exp.type) {
  case mp_vacuous:
  case mp_boolean_type:
  case mp_known:
    set_cur_exp_value_number (value_number (p));
    break;
  case unknown_types:
    t = mp_new_ring_entry (mp, p);
    set_cur_exp_node (t);
    break;
  case mp_string_type:
    set_cur_exp_str (value_str (p));
    break;
  case mp_picture_type:
    set_cur_exp_node (value_node (p));
    add_edge_ref (cur_exp_node ());
    break;
  case mp_pen_type:
    set_cur_exp_knot (copy_pen (value_knot (p)));
    break;
  case mp_path_type:
    set_cur_exp_knot (mp_copy_path (mp, value_knot (p)));
    break;
  case mp_transform_type:
  case mp_color_type:
  case mp_cmykcolor_type:
  case mp_pair_type:
    /* Copy the big node |p| */
    /* The most tedious case arises when the user refers to a
       \&{pair}, \&{color}, or \&{transform} variable; we must copy several fields,
       each of which can be |independent|, |dependent|, |mp_proto_dependent|,
       or |known|. */
    if (value_node (p) == NULL) {
      switch (mp_type (p)) {
      case mp_pair_type:
        mp_init_pair_node (mp, p);
        break;
      case mp_color_type:
        mp_init_color_node (mp, p);
        break;
      case mp_cmykcolor_type:
        mp_init_cmykcolor_node (mp, p);
        break;
      case mp_transform_type:
        mp_init_transform_node (mp, p);
        break;
      default:                   /* there are no other valid cases, but please the compiler */
        break;
      }
    }
    t = mp_get_value_node (mp);
    mp_name_type (t) = mp_capsule;
    q = (mp_value_node)value_node (p);
    switch (mp->cur_exp.type) {
    case mp_pair_type:
      mp_init_pair_node (mp, t);
      mp_install (mp, y_part (value_node (t)), y_part (q));
      mp_install (mp, x_part (value_node (t)), x_part (q));
      break;
    case mp_color_type:
      mp_init_color_node (mp, t);
      mp_install (mp, blue_part (value_node (t)),  blue_part (q));
      mp_install (mp, green_part (value_node (t)), green_part (q));
      mp_install (mp, red_part (value_node (t)),   red_part (q));
      break;
    case mp_cmykcolor_type:
      mp_init_cmykcolor_node (mp, t);
      mp_install (mp, black_part (value_node (t)),   black_part (q));
      mp_install (mp, yellow_part (value_node (t)),  yellow_part (q));
      mp_install (mp, magenta_part (value_node (t)), magenta_part (q));
      mp_install (mp, cyan_part (value_node (t)),    cyan_part (q));
      break;
    case mp_transform_type:
      mp_init_transform_node (mp, t);
      mp_install (mp, yy_part (value_node (t)), yy_part (q));
      mp_install (mp, yx_part (value_node (t)), yx_part (q));
      mp_install (mp, xy_part (value_node (t)), xy_part (q));
      mp_install (mp, xx_part (value_node (t)), xx_part (q));
      mp_install (mp, ty_part (value_node (t)), ty_part (q));
      mp_install (mp, tx_part (value_node (t)), tx_part (q));
      break;
    default:  /* there are no other valid cases, but please the compiler */
      break;
    }
    set_cur_exp_node (t);
    break;
  case mp_dependent:
  case mp_proto_dependent:
    mp_encapsulate (mp,
                    mp_copy_dep_list (mp,
                                      (mp_value_node) dep_list ((mp_value_node)
                                                                p)));
    break;
  case mp_numeric_type:
    mp_new_indep (mp, p);
    goto RESTART;
    break;
  case mp_independent:
    q = mp_single_dependency (mp, p);
    if (q == mp->dep_final) {
      mp->cur_exp.type = mp_known;
      set_cur_exp_value_number (zero_t);
      mp_free_dep_node (mp, q);
    } else {
      mp->cur_exp.type = mp_dependent;
      mp_encapsulate (mp, q);
    }
    break;
  default:
    mp_confusion (mp, "copy");
@:this can't happen copy}{\quad copy@>;
    break;
  }
}


@ The |encapsulate| subroutine assumes that |dep_final| is the
tail of dependency list~|p|.

@<Declare subroutines needed by |make_exp_copy|@>=
static void mp_encapsulate (MP mp, mp_value_node p) {
  mp_node q = mp_get_value_node (mp);
  FUNCTION_TRACE2 ("mp_encapsulate(%p)\n", p);
  mp_name_type (q) = mp_capsule;
  mp_new_dep (mp, q, mp->cur_exp.type, p);
  set_cur_exp_node (q);
}

@ The |install| procedure copies a numeric field~|q| into field~|r| of
a big node that will be part of a capsule.

@<Declare subroutines needed by |make_exp_copy|@>=
static void mp_install (MP mp, mp_node r, mp_node q) {
  mp_value_node p;      /* temporary register */
  if (mp_type (q) == mp_known) {
    mp_type (r) = mp_known;
    set_value_number (r, value_number (q));
  } else if (mp_type (q) == mp_independent) {
    p = mp_single_dependency (mp, q);
    if (p == mp->dep_final) {
      mp_type (r) = mp_known;
      set_value_number (r, zero_t);
      mp_free_dep_node (mp, p);
    } else {
      mp_new_dep (mp, r, mp_dependent, p);
    }
  } else {
    mp_new_dep (mp, r, mp_type (q),
                mp_copy_dep_list (mp, (mp_value_node) dep_list ((mp_value_node)
                                                                q)));
  }
}


@ Here is a comparatively simple routine that is used to scan the
\&{suffix} parameters of a macro.

@<Declare the basic parsing subroutines@>=
static void mp_scan_suffix (MP mp) {
  mp_node h, t; /* head and tail of the list being built */
  mp_node p;    /* temporary register */
  h = mp_get_symbolic_node (mp);
  t = h;
  while (1) {
    if (cur_cmd() == mp_left_bracket) {
      /* Scan a bracketed subscript and set |cur_cmd:=numeric_token| */
      mp_get_x_next (mp);
      mp_scan_expression (mp);
      if (mp->cur_exp.type != mp_known)
        mp_bad_subscript (mp);
      if (cur_cmd() != mp_right_bracket) {
        const char *hlp[] = {
               "I've seen a `[' and a subscript value, in a suffix,",
               "so a right bracket should have come next.",
               "I shall pretend that one was there.",
               NULL };
        mp_back_error (mp, "Missing `]' has been inserted", hlp, true);
      }
      set_cur_cmd((mp_variable_type)mp_numeric_token);
      set_cur_mod_number(cur_exp_value_number ());

    }
    if (cur_cmd() == mp_numeric_token) {
      mp_number arg1;
      new_number (arg1);
      number_clone (arg1, cur_mod_number());
      p = mp_new_num_tok (mp, arg1);
      free_number (arg1);
    } else if ((cur_cmd() == mp_tag_token) || (cur_cmd() == mp_internal_quantity)) {
      p = mp_get_symbolic_node (mp);
      set_mp_sym_sym (p, cur_sym());
      mp_name_type (p) = cur_sym_mod();
    } else {
      break;
    }
    mp_link (t) = p;
    t = p;
    mp_get_x_next (mp);
  }
  set_cur_exp_node (mp_link (h));
  mp_free_symbolic_node (mp, h);
  mp->cur_exp.type = mp_token_list;
}

@* Parsing secondary and higher expressions.

After the intricacies of |scan_primary|\kern-1pt,
the |scan_secondary| routine is
refreshingly simple. It's not trivial, but the operations are relatively
straightforward; the main difficulty is, again, that expressions and data
structures might change drastically every time we call |get_x_next|, so a
cautious approach is mandatory. For example, a macro defined by
\&{primarydef} might have disappeared by the time its second argument has
been scanned; we solve this by increasing the reference count of its token
list, so that the macro can be called even after it has been clobbered.

@<Declare the basic parsing subroutines@>=
static void mp_scan_secondary (MP mp) {
  mp_node p;    /* for list manipulation */
  halfword c, d;        /* operation codes or modifiers */
  mp_node cc = NULL;
  mp_sym mac_name = NULL;      /* token defined with \&{primarydef} */
RESTART:
  if ((cur_cmd() < mp_min_primary_command) ||
      (cur_cmd() > mp_max_primary_command))
    mp_bad_exp (mp, "A secondary");
@.A secondary expression...@>;
  mp_scan_primary (mp);
CONTINUE:
  if (cur_cmd() <= mp_max_secondary_command &&
      cur_cmd() >= mp_min_secondary_command) {
    p = mp_stash_cur_exp (mp);
    d = cur_cmd();
    c = cur_mod();
    if (d == mp_secondary_primary_macro) {
      cc = cur_mod_node();
      mac_name = cur_sym();
      add_mac_ref (cc);
    }
    mp_get_x_next (mp);
    mp_scan_primary (mp);
    if (d != mp_secondary_primary_macro) {
      mp_do_binary (mp, p, c);
    } else {
      mp_back_input (mp);
      mp_binary_mac (mp, p, cc, mac_name);
      decr_mac_ref (cc);
      mp_get_x_next (mp);
      goto RESTART;
    }
    goto CONTINUE;
  }
}


@ The following procedure calls a macro that has two parameters,
|p| and |cur_exp|.

@c
static void mp_binary_mac (MP mp, mp_node p, mp_node c, mp_sym n) {
  mp_node q, r; /* nodes in the parameter list */
  q = mp_get_symbolic_node (mp);
  r = mp_get_symbolic_node (mp);
  mp_link (q) = r;
  set_mp_sym_sym (q, p);
  set_mp_sym_sym (r, mp_stash_cur_exp (mp));
  mp_macro_call (mp, c, q, n);
}


@ The next procedure, |scan_tertiary|, is pretty much the same deal.

@<Declare the basic parsing subroutines@>=
static void mp_scan_tertiary (MP mp) {
  mp_node p;    /* for list manipulation */
  halfword c, d;        /* operation codes or modifiers */
  mp_node cc = NULL;
  mp_sym mac_name = NULL;      /* token defined with \&{secondarydef} */
RESTART:
  if ((cur_cmd() < mp_min_primary_command) ||
      (cur_cmd() > mp_max_primary_command))
    mp_bad_exp (mp, "A tertiary");
@.A tertiary expression...@>;
  mp_scan_secondary (mp);
CONTINUE:
  if (cur_cmd() <= mp_max_tertiary_command) {
    if (cur_cmd() >= mp_min_tertiary_command) {
      p = mp_stash_cur_exp (mp);
      c = cur_mod();
      d = cur_cmd();
      if (d == mp_tertiary_secondary_macro) {
        cc = cur_mod_node();
        mac_name = cur_sym();
        add_mac_ref (cc);
      }
      mp_get_x_next (mp);
      mp_scan_secondary (mp);
      if (d != mp_tertiary_secondary_macro) {
        mp_do_binary (mp, p, c);
      } else {
        mp_back_input (mp);
        mp_binary_mac (mp, p, cc, mac_name);
        decr_mac_ref (cc);
        mp_get_x_next (mp);
        goto RESTART;
      }
      goto CONTINUE;
    }
  }
}


@ Finally we reach the deepest level in our quartet of parsing routines.
This one is much like the others; but it has an extra complication from
paths, which materialize here.

@<Declare the basic parsing subroutines@>=
static int mp_scan_path (MP mp);
static void mp_scan_expression (MP mp) {
  int my_var_flag;      /* initial value of |var_flag| */
  my_var_flag = mp->var_flag;
  check_expansion_depth();
RESTART:
  if ((cur_cmd() < mp_min_primary_command) ||
      (cur_cmd() > mp_max_primary_command))
    mp_bad_exp (mp, "An");
@.An expression...@>;
  mp_scan_tertiary (mp);
CONTINUE:
  if (cur_cmd() <= mp_max_expression_command) {
    if (cur_cmd() >= mp_min_expression_command) {
      if ((cur_cmd() != mp_equals) || (my_var_flag != mp_assignment)) {
        mp_node p;    /* for list manipulation */
        mp_node cc = NULL;
        halfword c; 
        halfword d;        /* operation codes or modifiers */
        mp_sym mac_name;      /* token defined with \&{tertiarydef} */
        mac_name = NULL;
        p = mp_stash_cur_exp (mp);
        d = cur_cmd();
        c = cur_mod();
        if (d == mp_expression_tertiary_macro) {
          cc = cur_mod_node();
          mac_name = cur_sym();
          add_mac_ref (cc);
        }
        if ((d < mp_ampersand) || ((d == mp_ampersand) &&
                                ((mp_type (p) == mp_pair_type)
                                 || (mp_type (p) == mp_path_type)))) {
          /* Scan a path construction operation;  but |return| if |p| has the wrong type */
          
          mp_unstash_cur_exp (mp, p);
          if (!mp_scan_path(mp)) {
            mp->expand_depth_count--;
            return;
          }
        } else {
          mp_get_x_next (mp);
          mp_scan_tertiary (mp);
          if (d != mp_expression_tertiary_macro) {
            mp_do_binary (mp, p, c);
          } else {
            mp_back_input (mp);
            mp_binary_mac (mp, p, cc, mac_name);
            decr_mac_ref (cc);
            mp_get_x_next (mp);
            goto RESTART;
          }
        }
        goto CONTINUE;
      }
    }
  }
  mp->expand_depth_count--;
}


@ The reader should review the data structure conventions for paths before
hoping to understand the next part of this code.

@d min_tension three_quarter_unit_t

@<Declare the basic parsing subroutines@>=
static void force_valid_tension_setting(MP mp) {
  if ((mp->cur_exp.type != mp_known) || number_less(cur_exp_value_number (), min_tension)) {
    mp_value new_expr;
    const char *hlp[] = { 
               "The expression above should have been a number >=3/4.", 
                NULL };
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    mp_disp_err(mp, NULL);
    number_clone(new_expr.data.n, unity_t);
    mp_back_error (mp, "Improper tension has been set to 1", hlp, true);
    mp_get_x_next (mp);
    mp_flush_cur_exp (mp, new_expr);
  }
}
static int mp_scan_path (MP mp) {
  mp_knot path_p, path_q, r;
  mp_knot pp, qq;
  halfword d;        /* operation code or modifier */
  boolean cycle_hit; /* did a path expression just end with `\&{cycle}'? */
  mp_number x, y;    /* explicit coordinates or tension at a path join */
  int t;             /* knot type following a path join */
  t = 0;
  cycle_hit = false;
  /* Convert the left operand, |p|, into a partial path ending at~|q|;
    but |return| if |p| doesn't have a suitable type */
  if (mp->cur_exp.type == mp_pair_type)
    path_p = mp_pair_to_knot (mp);
  else if (mp->cur_exp.type == mp_path_type)
    path_p = cur_exp_knot ();
  else
    return 0;
  path_q = path_p;
  while (mp_next_knot (path_q) != path_p)
    path_q = mp_next_knot (path_q);
  if (mp_left_type (path_p) != mp_endpoint) {   /* open up a cycle */
    r = mp_copy_knot (mp, path_p);
    mp_next_knot (path_q) = r;
    path_q = r;
  }
  mp_left_type (path_p) = mp_open;
  mp_right_type (path_q) = mp_open;

  new_number (y);
  new_number (x);

CONTINUE_PATH:
  /* Determine the path join parameters;
    but |goto finish_path| if there's only a direction specifier */
  /* At this point |cur_cmd| is either |ampersand|, |left_brace|, or |path_join|. */

  if (cur_cmd() == mp_left_brace) {
    /* Put the pre-join direction information into node |q| */
    /* At this point |mp_right_type(q)| is usually |open|, but it may have been
       set to some other value by a previous operation. We must maintain
       the value of |mp_right_type(q)| in cases such as
       `\.{..\{curl2\}z\{0,0\}..}'. */
    t = mp_scan_direction (mp);
    if (t != mp_open) {
      mp_right_type (path_q) = (unsigned short) t;
      number_clone(path_q->right_given, cur_exp_value_number ());
      if (mp_left_type (path_q) == mp_open) {
        mp_left_type (path_q) = (unsigned short) t;
        number_clone(path_q->left_given, cur_exp_value_number ());
      }   /* note that |left_given(q)=left_curl(q)| */
    }
  }
  d = cur_cmd();
  if (d == mp_path_join) {
    /* Determine the tension and/or control points */
    mp_get_x_next (mp);
    if (cur_cmd() == mp_tension) {
      /* Set explicit tensions */
      mp_get_x_next (mp);
      set_number_from_scaled (y, cur_cmd());
      if (cur_cmd() == mp_at_least)
        mp_get_x_next (mp);
      mp_scan_primary (mp);
      force_valid_tension_setting(mp);
      if (number_to_scaled (y) == mp_at_least) {
       if (is_number(cur_exp_value_number())) 
         number_negate (cur_exp_value_number());
      }
      number_clone(path_q->right_tension, cur_exp_value_number ());
      if (cur_cmd() == mp_and_command) {
        mp_get_x_next (mp);
        set_number_from_scaled (y, cur_cmd());
        if (cur_cmd() == mp_at_least)
          mp_get_x_next (mp);
        mp_scan_primary (mp);
        force_valid_tension_setting(mp);
        if (number_to_scaled (y) == mp_at_least) {
          if (is_number(cur_exp_value_number())) 
            number_negate (cur_exp_value_number());
        }
      }
      number_clone (y, cur_exp_value_number ());
  
    } else if (cur_cmd() == mp_controls) {
      /* Set explicit control points */
      mp_right_type (path_q) = mp_explicit;
      t = mp_explicit;
      mp_get_x_next (mp);
      mp_scan_primary (mp);
      mp_known_pair (mp);
      number_clone (path_q->right_x, mp->cur_x);
      number_clone (path_q->right_y, mp->cur_y);
      if (cur_cmd() != mp_and_command) {
        number_clone (x, path_q->right_x);
        number_clone (y, path_q->right_y);
      } else {
        mp_get_x_next (mp);
        mp_scan_primary (mp);
        mp_known_pair (mp);
        number_clone (x, mp->cur_x);
        number_clone (y, mp->cur_y);
      }
    
    } else {
      set_number_to_unity(path_q->right_tension);
      set_number_to_unity(y);
      mp_back_input (mp);         /* default tension */
      goto DONE;
    };
    if (cur_cmd() != mp_path_join) {
      const char *hlp[] = { "A path join command should end with two dots.", NULL};
      mp_back_error (mp, "Missing `..' has been inserted", hlp, true);
    }
  DONE:
    ;
  } else if (d != mp_ampersand) {
    goto FINISH_PATH;
  }
  mp_get_x_next (mp);
  if (cur_cmd() == mp_left_brace) {
    /* Put the post-join direction information into |x| and |t| */
    /* Since |left_tension| and |mp_left_y| share the same position in knot nodes,
       and since |left_given| is similarly equivalent to |left_x|, we use
       |x| and |y| to hold the given direction and tension information when
       there are no explicit control points. */
    t = mp_scan_direction (mp);
    if (mp_right_type (path_q) != mp_explicit)
      number_clone (x, cur_exp_value_number ());
    else
      t = mp_explicit;            /* the direction information is superfluous */

  } else if (mp_right_type (path_q) != mp_explicit) {
    t = mp_open;
    set_number_to_zero(x);
  }
  
  if (cur_cmd() == mp_cycle) {
    /* Get ready to close a cycle */
    /* If a person tries to define an entire path by saying `\.{(x,y)\&cycle}',
       we silently change the specification to `\.{(x,y)..cycle}', since a cycle
       shouldn't have length zero. */
    cycle_hit = true;
    mp_get_x_next (mp);
    pp = path_p;
    qq = path_p;
    if (d == mp_ampersand) {
      if (path_p == path_q) {
        d = mp_path_join;
        set_number_to_unity(path_q->right_tension);
        set_number_to_unity(y);
      }
    }
  } else {
    mp_scan_tertiary (mp);
    /* Convert the right operand, |cur_exp|,
      into a partial path from |pp| to~|qq| */
    if (mp->cur_exp.type != mp_path_type)
      pp = mp_pair_to_knot (mp);
    else
      pp = cur_exp_knot ();
    qq = pp;
    while (mp_next_knot (qq) != pp)
      qq = mp_next_knot (qq);
    if (mp_left_type (pp) != mp_endpoint) {       /* open up a cycle */
      r = mp_copy_knot (mp, pp);
      mp_next_knot (qq) = r;
      qq = r;
    }
    mp_left_type (pp) = mp_open;
    mp_right_type (qq) = mp_open;
  }
  /* Join the partial paths and reset |p| and |q| to the head and tail
    of the result */
  if (d == mp_ampersand) {
    if (!(number_equal (path_q->x_coord, pp->x_coord)) ||
        !(number_equal (path_q->y_coord, pp->y_coord))) {
      const char *hlp[] = {
             "When you join paths `p&q', the ending point of p",
             "must be exactly equal to the starting point of q.",
             "So I'm going to pretend that you said `p..q' instead.",
             NULL };
      mp_back_error (mp, "Paths don't touch; `&' will be changed to `..'", hlp, true);
@.Paths don't touch@>;
      mp_get_x_next (mp);
      d = mp_path_join;
      set_number_to_unity (path_q->right_tension);
      set_number_to_unity (y);
    }
  }
  /* Plug an opening in |mp_right_type(pp)|, if possible */
  if (mp_right_type (pp) == mp_open) {
    if ((t == mp_curl) || (t == mp_given)) {
      mp_right_type (pp) = (unsigned short) t;
      number_clone (pp->right_given, x);
    }
  }
  if (d == mp_ampersand) {
    /* Splice independent paths together */
    if (mp_left_type (path_q) == mp_open)
      if (mp_right_type (path_q) == mp_open) {
        mp_left_type (path_q) = mp_curl;
        set_number_to_unity(path_q->left_curl);
      }
    if (mp_right_type (pp) == mp_open)
      if (t == mp_open) {
        mp_right_type (pp) = mp_curl;
        set_number_to_unity(pp->right_curl);
      }
    mp_right_type (path_q) = mp_right_type (pp);
    mp_next_knot (path_q) = mp_next_knot (pp);
    number_clone (path_q->right_x, pp->right_x);
    number_clone (path_q->right_y, pp->right_y);
    mp_xfree (pp);
    if (qq == pp)
      qq = path_q;

  } else {
    /* Plug an opening in |mp_right_type(q)|, if possible */
    if (mp_right_type (path_q) == mp_open) {
      if ((mp_left_type (path_q) == mp_curl) || (mp_left_type (path_q) == mp_given)) {
        mp_right_type (path_q) = mp_left_type (path_q);
        number_clone(path_q->right_given, path_q->left_given);
      }
    }

    mp_next_knot (path_q) = pp;
    number_clone (pp->left_y, y);
    if (t != mp_open) {
      number_clone (pp->left_x, x);
      mp_left_type (pp) = (unsigned short) t;
    };
  }
  path_q = qq;

  if (cur_cmd() >= mp_min_expression_command)
    if (cur_cmd() <= mp_ampersand)
      if (!cycle_hit)
        goto CONTINUE_PATH;
FINISH_PATH:
  /* Choose control points for the path and put the result into |cur_exp| */
  if (cycle_hit) {
    if (d == mp_ampersand)
      path_p = path_q;
  } else {
    mp_left_type (path_p) = mp_endpoint;
    if (mp_right_type (path_p) == mp_open) {
      mp_right_type (path_p) = mp_curl;
      set_number_to_unity(path_p->right_curl);
    }
    mp_right_type (path_q) = mp_endpoint;
    if (mp_left_type (path_q) == mp_open) {
      mp_left_type (path_q) = mp_curl;
      set_number_to_unity(path_q->left_curl);
    }
    mp_next_knot (path_q) = path_p;
  }
  mp_make_choices (mp, path_p);
  mp->cur_exp.type = mp_path_type;
  set_cur_exp_knot (path_p);

  free_number (x);
  free_number (y);
  return 1;
}


@ A pair of numeric values is changed into a knot node for a one-point path
when \MP\ discovers that the pair is part of a path.

@c
static mp_knot mp_pair_to_knot (MP mp) {                               /* convert a pair to a knot with two endpoints */
  mp_knot q;    /* the new node */
  q = mp_new_knot(mp);
  mp_left_type (q) = mp_endpoint;
  mp_right_type (q) = mp_endpoint;
  mp_originator (q) = mp_metapost_user;
  mp_next_knot (q) = q;
  mp_known_pair (mp);
  number_clone (q->x_coord, mp->cur_x);
  number_clone (q->y_coord, mp->cur_y);
  return q;
}


@ The |known_pair| subroutine sets |cur_x| and |cur_y| to the components
of the current expression, assuming that the current expression is a
pair of known numerics. Unknown components are zeroed, and the
current expression is flushed.

@<Declarations@>=
static void mp_known_pair (MP mp);

@ @c
void mp_known_pair (MP mp) {
  mp_value new_expr;
  mp_node p;    /* the pair node */
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  if (mp->cur_exp.type != mp_pair_type) {
    const char *hlp[] = { 
           "I need x and y numbers for this part of the path.",
           "The value I found (see above) was no good;",
           "so I'll try to keep going by using zero instead.",
           "(Chapter 27 of The METAFONTbook explains that",
           "you might want to type `I ??" "?' now.)",
           NULL };
    mp_disp_err(mp, NULL);
    mp_back_error (mp, "Undefined coordinates have been replaced by (0,0)", hlp, true);
    mp_get_x_next (mp);
    mp_flush_cur_exp (mp, new_expr);
    set_number_to_zero(mp->cur_x);
    set_number_to_zero(mp->cur_y);
  } else {
    p = value_node (cur_exp_node ());
    /* Make sure that both |x| and |y| parts of |p| are known;
       copy them into |cur_x| and |cur_y| */
    if (mp_type (x_part (p)) == mp_known) {
      number_clone(mp->cur_x, value_number (x_part (p)));
    } else {
      const char *hlp[] = { 
             "I need a `known' x value for this part of the path.",
             "The value I found (see above) was no good;",
             "so I'll try to keep going by using zero instead.",
             "(Chapter 27 of The METAFONTbook explains that",
             "you might want to type `I ??" "?' now.)",
             NULL };
      mp_disp_err (mp, x_part (p));
      mp_back_error (mp, "Undefined x coordinate has been replaced by 0", hlp, true);
      mp_get_x_next (mp);
      mp_recycle_value (mp, x_part (p));
      set_number_to_zero(mp->cur_x);
    }
    if (mp_type (y_part (p)) == mp_known) {
      number_clone(mp->cur_y, value_number (y_part (p)));
    } else {
      const char *hlp[] = { 
             "I need a `known' y value for this part of the path.",
             "The value I found (see above) was no good;",
             "so I'll try to keep going by using zero instead.",
             "(Chapter 27 of The METAFONTbook explains that",
             "you might want to type `I ??" "?' now.)",
             NULL };
      mp_disp_err (mp, y_part (p));
      mp_back_error (mp, "Undefined y coordinate has been replaced by 0", hlp, true);
      mp_get_x_next (mp);
      mp_recycle_value (mp, y_part (p));
      set_number_to_zero(mp->cur_y);
    }
    mp_flush_cur_exp (mp, new_expr);
  }
}

@ The |scan_direction| subroutine looks at the directional information
that is enclosed in braces, and also scans ahead to the following character.
A type code is returned, either |open| (if the direction was $(0,0)$),
or |curl| (if the direction was a curl of known value |cur_exp|), or
|given| (if the direction is given by the |angle| value that now
appears in |cur_exp|).

There's nothing difficult about this subroutine, but the program is rather
lengthy because a variety of potential errors need to be nipped in the bud.

@c
static quarterword mp_scan_direction (MP mp) {
  int t;        /* the type of information found */
  mp_get_x_next (mp);
  if (cur_cmd() == mp_curl_command) {
    /* Scan a curl specification */
    mp_get_x_next (mp);
    mp_scan_expression (mp);
    if ((mp->cur_exp.type != mp_known) || (number_negative(cur_exp_value_number ()))) {
      mp_value new_expr;
      const char *hlp[] = { "A curl must be a known, nonnegative number.", NULL };
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      set_number_to_unity(new_expr.data.n);
      mp_disp_err(mp, NULL);
      mp_back_error (mp, "Improper curl has been replaced by 1", hlp, true);
      mp_get_x_next (mp);
      mp_flush_cur_exp (mp, new_expr);
    }
    t = mp_curl;

  } else {
    /* Scan a given direction */
    mp_scan_expression (mp);
    if (mp->cur_exp.type > mp_pair_type) {
      /* Get given directions separated by commas */
      mp_number xx;
      new_number(xx);
      if (mp->cur_exp.type != mp_known) {
        mp_value new_expr;
        const char *hlp[] = { 
               "I need a `known' x value for this part of the path.",
               "The value I found (see above) was no good;",
               "so I'll try to keep going by using zero instead.",
               "(Chapter 27 of The METAFONTbook explains that",
               "you might want to type `I ??" "?' now.)",
               NULL };
        memset(&new_expr,0,sizeof(mp_value));
        new_number(new_expr.data.n);
        set_number_to_zero(new_expr.data.n);
        mp_disp_err(mp, NULL);
        mp_back_error (mp, "Undefined x coordinate has been replaced by 0", hlp, true);
        mp_get_x_next (mp);
        mp_flush_cur_exp (mp, new_expr);
      }
      number_clone(xx, cur_exp_value_number ());
      if (cur_cmd() != mp_comma) {
        const char *hlp[] = {
               "I've got the x coordinate of a path direction;",
               "will look for the y coordinate next.",
               NULL };
        mp_back_error (mp, "Missing `,' has been inserted", hlp, true);
      }
      mp_get_x_next (mp);
      mp_scan_expression (mp);
      if (mp->cur_exp.type != mp_known) {
        mp_value new_expr;
        const char *hlp[] = { 
               "I need a `known' y value for this part of the path.",
               "The value I found (see above) was no good;",
               "so I'll try to keep going by using zero instead.",
               "(Chapter 27 of The METAFONTbook explains that",
               "you might want to type `I ??" "?' now.)",
               NULL };
        memset(&new_expr,0,sizeof(mp_value));
        new_number(new_expr.data.n);
        set_number_to_zero(new_expr.data.n);
        mp_disp_err(mp, NULL);
        mp_back_error (mp, "Undefined y coordinate has been replaced by 0", hlp, true);
        mp_get_x_next (mp);
        mp_flush_cur_exp (mp, new_expr);
      }
      number_clone(mp->cur_y, cur_exp_value_number ());
      number_clone(mp->cur_x, xx);
      free_number(xx);
  
    } else {
      mp_known_pair (mp);
    }
    if (number_zero(mp->cur_x) && number_zero(mp->cur_y))
      t = mp_open;
    else {
      mp_number narg;
      new_angle (narg); 
      n_arg (narg, mp->cur_x, mp->cur_y);
      t = mp_given;
      set_cur_exp_value_number (narg);
      free_number (narg);
    }
  }
  if (cur_cmd() != mp_right_brace) {
    const char *hlp[] = {
           "I've scanned a direction spec for part of a path,",
           "so a right brace should have come next.",
           "I shall pretend that one was there.",
           NULL };
    mp_back_error (mp, "Missing `}' has been inserted", hlp, true);
  }
  mp_get_x_next (mp);
  return (quarterword) t;
}


@ Finally, we sometimes need to scan an expression whose value is
supposed to be either |true_code| or |false_code|.

@d mp_get_boolean(mp) do {
  mp_get_x_next (mp);
  mp_scan_expression (mp);
  if (mp->cur_exp.type != mp_boolean_type) {
    do_boolean_error(mp);
  }
} while (0)

@<Declare the basic parsing subroutines@>=
static void do_boolean_error (MP mp) {
  mp_value new_expr;
  const char *hlp[] = { 
         "The expression shown above should have had a definite",
         "true-or-false value. I'm changing it to `false'.",
         NULL };
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  mp_disp_err(mp, NULL);
  set_number_from_boolean (new_expr.data.n, mp_false_code);
  mp_back_error (mp, "Undefined condition will be treated as `false'", hlp, true);
  mp_get_x_next (mp);
  mp_flush_cur_exp (mp, new_expr);
  mp->cur_exp.type = mp_boolean_type;
}

@ @<Declarations@>=
static void do_boolean_error (MP mp);

@* Doing the operations.
The purpose of parsing is primarily to permit people to avoid piles of
parentheses. But the real work is done after the structure of an expression
has been recognized; that's when new expressions are generated. We
turn now to the guts of \MP, which handles individual operators that
have come through the parsing mechanism.

We'll start with the easy ones that take no operands, then work our way
up to operators with one and ultimately two arguments. In other words,
we will write the three procedures |do_nullary|, |do_unary|, and |do_binary|
that are invoked periodically by the expression scanners.

First let's make sure that all of the primitive operators are in the
hash table. Although |scan_primary| and its relatives made use of the
\\{cmd} code for these operators, the \\{do} routines base everything
on the \\{mod} code. For example, |do_binary| doesn't care whether the
operation it performs is a |primary_binary| or |secondary_binary|, etc.

@<Put each...@>=
mp_primitive (mp, "true", mp_nullary, mp_true_code);
@:true_}{\&{true} primitive@>;
mp_primitive (mp, "false", mp_nullary, mp_false_code);
@:false_}{\&{false} primitive@>;
mp_primitive (mp, "nullpicture", mp_nullary, mp_null_picture_code);
@:null_picture_}{\&{nullpicture} primitive@>;
mp_primitive (mp, "nullpen", mp_nullary, mp_null_pen_code);
@:null_pen_}{\&{nullpen} primitive@>;
mp_primitive (mp, "readstring", mp_nullary, mp_read_string_op);
@:read_string_}{\&{readstring} primitive@>;
mp_primitive (mp, "pencircle", mp_nullary, mp_pen_circle);
@:pen_circle_}{\&{pencircle} primitive@>;
mp_primitive (mp, "normaldeviate", mp_nullary, mp_normal_deviate);
@:normal_deviate_}{\&{normaldeviate} primitive@>;
mp_primitive (mp, "readfrom", mp_unary, mp_read_from_op);
@:read_from_}{\&{readfrom} primitive@>;
mp_primitive (mp, "closefrom", mp_unary, mp_close_from_op);
@:close_from_}{\&{closefrom} primitive@>;
mp_primitive (mp, "odd", mp_unary, mp_odd_op);
@:odd_}{\&{odd} primitive@>;
mp_primitive (mp, "known", mp_unary, mp_known_op);
@:known_}{\&{known} primitive@>;
mp_primitive (mp, "unknown", mp_unary, mp_unknown_op);
@:unknown_}{\&{unknown} primitive@>;
mp_primitive (mp, "not", mp_unary, mp_not_op);
@:not_}{\&{not} primitive@>;
mp_primitive (mp, "decimal", mp_unary, mp_decimal);
@:decimal_}{\&{decimal} primitive@>;
mp_primitive (mp, "reverse", mp_unary, mp_reverse);
@:reverse_}{\&{reverse} primitive@>;
mp_primitive (mp, "makepath", mp_unary, mp_make_path_op);
@:make_path_}{\&{makepath} primitive@>;
mp_primitive (mp, "makepen", mp_unary, mp_make_pen_op);
@:make_pen_}{\&{makepen} primitive@>;
mp_primitive (mp, "oct", mp_unary, mp_oct_op);
@:oct_}{\&{oct} primitive@>;
mp_primitive (mp, "hex", mp_unary, mp_hex_op);
@:hex_}{\&{hex} primitive@>;
mp_primitive (mp, "ASCII", mp_unary, mp_ASCII_op);
@:ASCII_}{\&{ASCII} primitive@>;
mp_primitive (mp, "char", mp_unary, mp_char_op);
@:char_}{\&{char} primitive@>;
mp_primitive (mp, "length", mp_unary, mp_length_op);
@:length_}{\&{length} primitive@>;
mp_primitive (mp, "turningnumber", mp_unary, mp_turning_op);
@:turning_number_}{\&{turningnumber} primitive@>;
mp_primitive (mp, "xpart", mp_unary, mp_x_part);
@:x_part_}{\&{xpart} primitive@>;
mp_primitive (mp, "ypart", mp_unary, mp_y_part);
@:y_part_}{\&{ypart} primitive@>;
mp_primitive (mp, "xxpart", mp_unary, mp_xx_part);
@:xx_part_}{\&{xxpart} primitive@>;
mp_primitive (mp, "xypart", mp_unary, mp_xy_part);
@:xy_part_}{\&{xypart} primitive@>;
mp_primitive (mp, "yxpart", mp_unary, mp_yx_part);
@:yx_part_}{\&{yxpart} primitive@>;
mp_primitive (mp, "yypart", mp_unary, mp_yy_part);
@:yy_part_}{\&{yypart} primitive@>;
mp_primitive (mp, "redpart", mp_unary, mp_red_part);
@:red_part_}{\&{redpart} primitive@>;
mp_primitive (mp, "greenpart", mp_unary, mp_green_part);
@:green_part_}{\&{greenpart} primitive@>;
mp_primitive (mp, "bluepart", mp_unary, mp_blue_part);
@:blue_part_}{\&{bluepart} primitive@>;
mp_primitive (mp, "cyanpart", mp_unary, mp_cyan_part);
@:cyan_part_}{\&{cyanpart} primitive@>;
mp_primitive (mp, "magentapart", mp_unary, mp_magenta_part);
@:magenta_part_}{\&{magentapart} primitive@>;
mp_primitive (mp, "yellowpart", mp_unary, mp_yellow_part);
@:yellow_part_}{\&{yellowpart} primitive@>;
mp_primitive (mp, "blackpart", mp_unary, mp_black_part);
@:black_part_}{\&{blackpart} primitive@>;
mp_primitive (mp, "greypart", mp_unary, mp_grey_part);
@:grey_part_}{\&{greypart} primitive@>;
mp_primitive (mp, "colormodel", mp_unary, mp_color_model_part);
@:color_model_part_}{\&{colormodel} primitive@>;
mp_primitive (mp, "fontpart", mp_unary, mp_font_part);
@:font_part_}{\&{fontpart} primitive@>;
mp_primitive (mp, "textpart", mp_unary, mp_text_part);
@:text_part_}{\&{textpart} primitive@>;
mp_primitive (mp, "prescriptpart", mp_unary, mp_prescript_part);
@:prescript_part_}{\&{prescriptpart} primitive@>;
mp_primitive (mp, "postscriptpart", mp_unary, mp_postscript_part);
@:postscript_part_}{\&{postscriptpart} primitive@>;
mp_primitive (mp, "pathpart", mp_unary, mp_path_part);
@:path_part_}{\&{pathpart} primitive@>;
mp_primitive (mp, "penpart", mp_unary, mp_pen_part);
@:pen_part_}{\&{penpart} primitive@>;
mp_primitive (mp, "dashpart", mp_unary, mp_dash_part);
@:dash_part_}{\&{dashpart} primitive@>;
mp_primitive (mp, "sqrt", mp_unary, mp_sqrt_op);
@:sqrt_}{\&{sqrt} primitive@>;
mp_primitive (mp, "mexp", mp_unary, mp_m_exp_op);
@:m_exp_}{\&{mexp} primitive@>;
mp_primitive (mp, "mlog", mp_unary, mp_m_log_op);
@:m_log_}{\&{mlog} primitive@>;
mp_primitive (mp, "sind", mp_unary, mp_sin_d_op);
@:sin_d_}{\&{sind} primitive@>;
mp_primitive (mp, "cosd", mp_unary, mp_cos_d_op);
@:cos_d_}{\&{cosd} primitive@>;
mp_primitive (mp, "floor", mp_unary, mp_floor_op);
@:floor_}{\&{floor} primitive@>;
mp_primitive (mp, "uniformdeviate", mp_unary, mp_uniform_deviate);
@:uniform_deviate_}{\&{uniformdeviate} primitive@>;
mp_primitive (mp, "charexists", mp_unary, mp_char_exists_op);
@:char_exists_}{\&{charexists} primitive@>;
mp_primitive (mp, "fontsize", mp_unary, mp_font_size);
@:font_size_}{\&{fontsize} primitive@>;
mp_primitive (mp, "llcorner", mp_unary, mp_ll_corner_op);
@:ll_corner_}{\&{llcorner} primitive@>;
mp_primitive (mp, "lrcorner", mp_unary, mp_lr_corner_op);
@:lr_corner_}{\&{lrcorner} primitive@>;
mp_primitive (mp, "ulcorner", mp_unary, mp_ul_corner_op);
@:ul_corner_}{\&{ulcorner} primitive@>;
mp_primitive (mp, "urcorner", mp_unary, mp_ur_corner_op);
@:ur_corner_}{\&{urcorner} primitive@>;
mp_primitive (mp, "arclength", mp_unary, mp_arc_length);
@:arc_length_}{\&{arclength} primitive@>;
mp_primitive (mp, "angle", mp_unary, mp_angle_op);
@:angle_}{\&{angle} primitive@>;
mp_primitive (mp, "cycle", mp_cycle, mp_cycle_op);
@:cycle_}{\&{cycle} primitive@>;
mp_primitive (mp, "stroked", mp_unary, mp_stroked_op);
@:stroked_}{\&{stroked} primitive@>;
mp_primitive (mp, "filled", mp_unary, mp_filled_op);
@:filled_}{\&{filled} primitive@>;
mp_primitive (mp, "textual", mp_unary, mp_textual_op);
@:textual_}{\&{textual} primitive@>;
mp_primitive (mp, "clipped", mp_unary, mp_clipped_op);
@:clipped_}{\&{clipped} primitive@>;
mp_primitive (mp, "bounded", mp_unary, mp_bounded_op);
@:bounded_}{\&{bounded} primitive@>;
mp_primitive (mp, "+", mp_plus_or_minus, mp_plus);
@:+ }{\.{+} primitive@>;
mp_primitive (mp, "-", mp_plus_or_minus, mp_minus);
@:- }{\.{-} primitive@>;
mp_primitive (mp, "*", mp_secondary_binary, mp_times);
@:* }{\.{*} primitive@>;
mp_primitive (mp, "/", mp_slash, mp_over);
mp->frozen_slash = mp_frozen_primitive (mp, "/", mp_slash, mp_over);
@:/ }{\.{/} primitive@>;
mp_primitive (mp, "++", mp_tertiary_binary, mp_pythag_add);
@:++_}{\.{++} primitive@>;
mp_primitive (mp, "+-+", mp_tertiary_binary, mp_pythag_sub);
@:+-+_}{\.{+-+} primitive@>;
mp_primitive (mp, "or", mp_tertiary_binary, mp_or_op);
@:or_}{\&{or} primitive@>;
mp_primitive (mp, "and", mp_and_command, mp_and_op);
@:and_}{\&{and} primitive@>;
mp_primitive (mp, "<", mp_expression_binary, mp_less_than);
@:< }{\.{<} primitive@>;
mp_primitive (mp, "<=", mp_expression_binary, mp_less_or_equal);
@:<=_}{\.{<=} primitive@>;
mp_primitive (mp, ">", mp_expression_binary, mp_greater_than);
@:> }{\.{>} primitive@>;
mp_primitive (mp, ">=", mp_expression_binary, mp_greater_or_equal);
@:>=_}{\.{>=} primitive@>;
mp_primitive (mp, "=", mp_equals, mp_equal_to);
@:= }{\.{=} primitive@>;
mp_primitive (mp, "<>", mp_expression_binary, mp_unequal_to);
@:<>_}{\.{<>} primitive@>;
mp_primitive (mp, "substring", mp_primary_binary, mp_substring_of);
@:substring_}{\&{substring} primitive@>;
mp_primitive (mp, "subpath", mp_primary_binary, mp_subpath_of);
@:subpath_}{\&{subpath} primitive@>;
mp_primitive (mp, "directiontime", mp_primary_binary, mp_direction_time_of);
@:direction_time_}{\&{directiontime} primitive@>;
mp_primitive (mp, "point", mp_primary_binary, mp_point_of);
@:point_}{\&{point} primitive@>;
mp_primitive (mp, "precontrol", mp_primary_binary, mp_precontrol_of);
@:precontrol_}{\&{precontrol} primitive@>;
mp_primitive (mp, "postcontrol", mp_primary_binary, mp_postcontrol_of);
@:postcontrol_}{\&{postcontrol} primitive@>;
mp_primitive (mp, "penoffset", mp_primary_binary, mp_pen_offset_of);
@:pen_offset_}{\&{penoffset} primitive@>;
mp_primitive (mp, "arctime", mp_primary_binary, mp_arc_time_of);
@:arc_time_of_}{\&{arctime} primitive@>;
mp_primitive (mp, "mpversion", mp_nullary, mp_version);
@:mp_verison_}{\&{mpversion} primitive@>;
mp_primitive (mp, "&", mp_ampersand, mp_concatenate);
@:!!!}{\.{\&} primitive@>;
mp_primitive (mp, "rotated", mp_secondary_binary, mp_rotated_by);
@:rotated_}{\&{rotated} primitive@>;
mp_primitive (mp, "slanted", mp_secondary_binary, mp_slanted_by);
@:slanted_}{\&{slanted} primitive@>;
mp_primitive (mp, "scaled", mp_secondary_binary, mp_scaled_by);
@:scaled_}{\&{scaled} primitive@>;
mp_primitive (mp, "shifted", mp_secondary_binary, mp_shifted_by);
@:shifted_}{\&{shifted} primitive@>;
mp_primitive (mp, "transformed", mp_secondary_binary, mp_transformed_by);
@:transformed_}{\&{transformed} primitive@>;
mp_primitive (mp, "xscaled", mp_secondary_binary, mp_x_scaled);
@:x_scaled_}{\&{xscaled} primitive@>;
mp_primitive (mp, "yscaled", mp_secondary_binary, mp_y_scaled);
@:y_scaled_}{\&{yscaled} primitive@>;
mp_primitive (mp, "zscaled", mp_secondary_binary, mp_z_scaled);
@:z_scaled_}{\&{zscaled} primitive@>;
mp_primitive (mp, "infont", mp_secondary_binary, mp_in_font);
@:in_font_}{\&{infont} primitive@>;
mp_primitive (mp, "intersectiontimes", mp_tertiary_binary, mp_intersect);
@:intersection_times_}{\&{intersectiontimes} primitive@>;
mp_primitive (mp, "envelope", mp_primary_binary, mp_envelope_of);
@:envelope_}{\&{envelope} primitive@>;
mp_primitive (mp, "glyph", mp_primary_binary, mp_glyph_infont);
@:glyph_infont_}{\&{envelope} primitive@>
 

@ @<Cases of |print_cmd...@>=
case mp_nullary:
case mp_unary:
case mp_primary_binary:
case mp_secondary_binary:
case mp_tertiary_binary:
case mp_expression_binary:
case mp_cycle:
case mp_plus_or_minus:
case mp_slash:
case mp_ampersand:
case mp_equals:
case mp_and_command:
mp_print_op (mp, (quarterword) m);
break;

@ OK, let's look at the simplest \\{do} procedure first.

@c
@<Declare nullary action procedure@>;
static void mp_do_nullary (MP mp, quarterword c) {
  check_arith();
  if (number_greater (internal_value (mp_tracing_commands), two_t))
    mp_show_cmd_mod (mp, mp_nullary, c);
  switch (c) {
  case mp_true_code:
  case mp_false_code:
    mp->cur_exp.type = mp_boolean_type;
    set_cur_exp_value_boolean (c);
    break;
  case mp_null_picture_code:
    mp->cur_exp.type = mp_picture_type;
    set_cur_exp_node ((mp_node)mp_get_edge_header_node (mp));
    mp_init_edges (mp, (mp_edge_header_node)cur_exp_node ());
    break;
  case mp_null_pen_code:
    mp->cur_exp.type = mp_pen_type;
    set_cur_exp_knot (mp_get_pen_circle (mp, zero_t));
    break;
  case mp_normal_deviate:
    {
      mp_number r;
      new_number (r);
      mp_norm_rand (mp, &r);
      mp->cur_exp.type = mp_known;
      set_cur_exp_value_number (r);
      free_number (r);
    }
    break;
  case mp_pen_circle:
    mp->cur_exp.type = mp_pen_type;
    set_cur_exp_knot (mp_get_pen_circle (mp, unity_t));
    break;
  case mp_version:
    mp->cur_exp.type = mp_string_type;
    set_cur_exp_str (mp_intern (mp, metapost_version));
    break;
  case mp_read_string_op:
    /* Read a string from the terminal */
    if (mp->noninteractive || mp->interaction <= mp_nonstop_mode)
      mp_fatal_error (mp, "*** (cannot readstring in nonstop modes)");
    mp_begin_file_reading (mp);
    name = is_read;
    limit = start;
    prompt_input ("");
    mp_finish_read (mp);
    break;
  }                             /* there are no other cases */
  check_arith();
}


@ @<Declare nullary action procedure@>=
static void mp_finish_read (MP mp) {                               /* copy |buffer| line to |cur_exp| */
  size_t k;
  str_room (((int) mp->last - (int) start));
  for (k = (size_t) start; k < mp->last; k++) {
    append_char (mp->buffer[k]);
  }
  mp_end_file_reading (mp);
  mp->cur_exp.type = mp_string_type;
  set_cur_exp_str (mp_make_string (mp));
}


@ Things get a bit more interesting when there's an operand. The
operand to |do_unary| appears in |cur_type| and |cur_exp|.

This complicated if test makes sure that any |bounds| or |clip|
picture objects that get passed into \&{within} do not raise an 
error when queried using the color part primitives (this is needed
for backward compatibility) .

@d cur_pic_item mp_link(edge_list(cur_exp_node()))
@d pict_color_type(A) ((cur_pic_item!=NULL) &&
         ((!has_color(cur_pic_item)) 
          ||
         (((mp_color_model(cur_pic_item)==A)
          ||
          ((mp_color_model(cur_pic_item)==mp_uninitialized_model) &&
           (number_to_scaled (internal_value(mp_default_color_model))/number_to_scaled (unity_t))==(A))))))

@d boolean_reset(A) if ( (A) ) set_cur_exp_value_boolean(mp_true_code); else set_cur_exp_value_boolean(mp_false_code)

@d type_range(A,B) { 
  if ( (mp->cur_exp.type>=(A)) && (mp->cur_exp.type<=(B)) ) 
    set_number_from_boolean (new_expr.data.n, mp_true_code);
  else 
    set_number_from_boolean (new_expr.data.n, mp_false_code);
  mp_flush_cur_exp(mp, new_expr);
  mp->cur_exp.type=mp_boolean_type;
  }
@d type_test(A) { 
  if ( mp->cur_exp.type==(mp_variable_type)(A) ) 
    set_number_from_boolean (new_expr.data.n, mp_true_code);
  else 
    set_number_from_boolean (new_expr.data.n, mp_false_code);
  mp_flush_cur_exp(mp, new_expr);
  mp->cur_exp.type=mp_boolean_type;
  }


@c
@<Declare unary action procedures@>;
static void mp_do_unary (MP mp, quarterword c) {
  mp_node p;      /* for list manipulation */
  mp_value new_expr;
  check_arith();
  if (number_greater (internal_value (mp_tracing_commands), two_t)) {
    /* Trace the current unary operation */
    mp_begin_diagnostic (mp);
    mp_print_nl (mp, "{");
    mp_print_op (mp, c);
    mp_print_char (mp, xord ('('));
    mp_print_exp (mp, NULL, 0);   /* show the operand, but not verbosely */
    mp_print (mp, ")}");
    mp_end_diagnostic (mp, false);
  }
  switch (c) {
  case mp_plus:
    if (mp->cur_exp.type < mp_color_type)
      mp_bad_unary (mp, mp_plus);
    break;
  case mp_minus:
    negate_cur_expr(mp);
    break;
  case mp_not_op:
    if (mp->cur_exp.type != mp_boolean_type) {
      mp_bad_unary (mp, mp_not_op);
    } else {
      halfword bb;
      if (cur_exp_value_boolean () == mp_true_code)
        bb = mp_false_code;
      else
        bb = mp_true_code;
      set_cur_exp_value_boolean (bb);
    }
    break;
  case mp_sqrt_op:
  case mp_m_exp_op:
  case mp_m_log_op:
  case mp_sin_d_op:
  case mp_cos_d_op:
  case mp_floor_op:
  case mp_uniform_deviate:
  case mp_odd_op:
  case mp_char_exists_op:
    if (mp->cur_exp.type != mp_known) {
      mp_bad_unary (mp, c);
    } else {
      switch (c) {
      case mp_sqrt_op:
        {
          mp_number r1;
          new_number (r1);
          square_rt (r1, cur_exp_value_number ());
          set_cur_exp_value_number  (r1);
          free_number (r1);
        }
        break;
      case mp_m_exp_op:
        {
          mp_number r1;
          new_number (r1);
          m_exp (r1, cur_exp_value_number ());
          set_cur_exp_value_number (r1);
          free_number (r1);
        }
        break;
      case mp_m_log_op:
        {
          mp_number r1;
          new_number (r1);
          m_log (r1, cur_exp_value_number ());
          set_cur_exp_value_number (r1);
          free_number (r1);
        }
        break;
      case mp_sin_d_op:
      case mp_cos_d_op:
        {
          mp_number n_sin, n_cos, arg1, arg2;
          new_number (arg1);
          new_number (arg2);
          new_fraction (n_sin);
          new_fraction (n_cos); /* results computed by |n_sin_cos| */
          number_clone (arg1, cur_exp_value_number()); 
          number_clone (arg2, unity_t);
          number_multiply_int (arg2, 360);
          number_modulo (arg1, arg2);
          convert_scaled_to_angle (arg1);
          n_sin_cos (arg1, n_cos, n_sin);
          if (c == mp_sin_d_op) {
            fraction_to_round_scaled (n_sin);
            set_cur_exp_value_number (n_sin);
          } else {
            fraction_to_round_scaled (n_cos);
            set_cur_exp_value_number (n_cos);
          }
          free_number (arg1);
          free_number (arg2);
          free_number (n_sin);
          free_number (n_cos);
        }
        break;
      case mp_floor_op:
        {
          mp_number vvx;
          new_number (vvx);
          number_clone (vvx, cur_exp_value_number ());
          floor_scaled (vvx);
          set_cur_exp_value_number (vvx);
          free_number (vvx);
        }
        break;
      case mp_uniform_deviate:
        {
          mp_number vvx;
          new_number (vvx);
          mp_unif_rand (mp, &vvx, cur_exp_value_number ());
          set_cur_exp_value_number (vvx);
          free_number (vvx);
        }
        break;
      case mp_odd_op:
        {
          integer vvx = odd (round_unscaled (cur_exp_value_number ()));
          boolean_reset (vvx);
          mp->cur_exp.type = mp_boolean_type;
        }
        break;
      case mp_char_exists_op:
        /* Determine if a character has been shipped out */
        set_cur_exp_value_scaled (round_unscaled (cur_exp_value_number ()) % 256);
        if (number_negative(cur_exp_value_number ())) {
          halfword vv = number_to_scaled(cur_exp_value_number ());
          set_cur_exp_value_scaled (vv + 256);
        }
        boolean_reset (mp->char_exists[number_to_scaled(cur_exp_value_number ())]);
        mp->cur_exp.type = mp_boolean_type;
        break;
      }                             /* there are no other cases */
    }
    break;
  case mp_angle_op:
    if (mp_nice_pair (mp, cur_exp_node (), mp->cur_exp.type)) {
      mp_number narg;
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      new_angle (narg);
      p = value_node (cur_exp_node ());
      n_arg (narg, value_number (x_part (p)), value_number (y_part (p)));
      number_clone (new_expr.data.n, narg);
      convert_angle_to_scaled (new_expr.data.n);
      free_number (narg);
      mp_flush_cur_exp (mp, new_expr);
    } else {
      mp_bad_unary (mp, mp_angle_op);
    }
    break;
  case mp_x_part:
  case mp_y_part:
    if ((mp->cur_exp.type == mp_pair_type)
        || (mp->cur_exp.type == mp_transform_type))
      mp_take_part (mp, c);
    else if (mp->cur_exp.type == mp_picture_type)
      mp_take_pict_part (mp, c);
    else
      mp_bad_unary (mp, c);
    break;
  case mp_xx_part:
  case mp_xy_part:
  case mp_yx_part:
  case mp_yy_part:
    if (mp->cur_exp.type == mp_transform_type)
      mp_take_part (mp, c);
    else if (mp->cur_exp.type == mp_picture_type)
      mp_take_pict_part (mp, c);
    else
      mp_bad_unary (mp, c);
    break;
  case mp_red_part:
  case mp_green_part:
  case mp_blue_part:
    if (mp->cur_exp.type == mp_color_type)
      mp_take_part (mp, c);
    else if (mp->cur_exp.type == mp_picture_type) {
      if pict_color_type
        (mp_rgb_model) mp_take_pict_part (mp, c);
      else
        mp_bad_color_part (mp, c);
    } else
      mp_bad_unary (mp, c);
    break;
  case mp_cyan_part:
  case mp_magenta_part:
  case mp_yellow_part:
  case mp_black_part:
    if (mp->cur_exp.type == mp_cmykcolor_type)
      mp_take_part (mp, c);
    else if (mp->cur_exp.type == mp_picture_type) {
      if pict_color_type
        (mp_cmyk_model) mp_take_pict_part (mp, c);
      else
        mp_bad_color_part (mp, c);
    } else
      mp_bad_unary (mp, c);
    break;
  case mp_grey_part:
    if (mp->cur_exp.type == mp_known); 
    else if (mp->cur_exp.type == mp_picture_type) {
      if pict_color_type
        (mp_grey_model) mp_take_pict_part (mp, c);
      else
        mp_bad_color_part (mp, c);
    } else
      mp_bad_unary (mp, c);
    break;
  case mp_color_model_part:
    if (mp->cur_exp.type == mp_picture_type)
      mp_take_pict_part (mp, c);
    else
      mp_bad_unary (mp, c);
    break;
  case mp_font_part:
  case mp_text_part:
  case mp_path_part:
  case mp_pen_part:
  case mp_dash_part:
  case mp_prescript_part:
  case mp_postscript_part:
    if (mp->cur_exp.type == mp_picture_type)
      mp_take_pict_part (mp, c);
    else
      mp_bad_unary (mp, c);
    break;
  case mp_char_op:
    if (mp->cur_exp.type != mp_known) {
      mp_bad_unary (mp, mp_char_op);
    } else {
      int vv = round_unscaled (cur_exp_value_number ()) % 256;
      set_cur_exp_value_scaled (vv);
      mp->cur_exp.type = mp_string_type;
      if (number_negative(cur_exp_value_number ())) {
        vv = number_to_scaled(cur_exp_value_number ()) + 256;
        set_cur_exp_value_scaled (vv);
      }
      {
        unsigned char ss[2];
        ss[0] = (unsigned char) number_to_scaled(cur_exp_value_number ());
        ss[1] = '\0';
        set_cur_exp_str (mp_rtsl (mp, (char *) ss, 1));
      }
    }
    break;
  case mp_decimal:
    if (mp->cur_exp.type != mp_known) {
      mp_bad_unary (mp, mp_decimal);
    } else {
      mp->old_setting = mp->selector;
      mp->selector = new_string;
      print_number (cur_exp_value_number ());
      set_cur_exp_str (mp_make_string (mp));
      mp->selector = mp->old_setting;
      mp->cur_exp.type = mp_string_type;
    }
    break;
  case mp_oct_op:
  case mp_hex_op:
  case mp_ASCII_op:
    if (mp->cur_exp.type != mp_string_type)
      mp_bad_unary (mp, c);
    else
      mp_str_to_num (mp, c);
    break;
  case mp_font_size:
    if (mp->cur_exp.type != mp_string_type) {
      mp_bad_unary (mp, mp_font_size);
    } else {
      /* Find the design size of the font whose name is |cur_exp| */
      /* One simple application of |find_font| is the implementation of the |font_size|
         operator that gets the design size for a given font name. */
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      set_number_from_scaled (new_expr.data.n, 
               (mp->font_dsize[mp_find_font (mp, mp_str (mp, cur_exp_str ()))] + 8) / 16);
      mp_flush_cur_exp (mp, new_expr);
    }
    break;
  case mp_length_op:
    /* The length operation is somewhat unusual in that it applies to a variety
       of different types of operands. */
    switch (mp->cur_exp.type) {
    case mp_string_type:
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      number_clone (new_expr.data.n, unity_t);
      number_multiply_int(new_expr.data.n, cur_exp_str ()->len);
      mp_flush_cur_exp (mp, new_expr);
      break;
    case mp_path_type:
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      mp_path_length (mp, &new_expr.data.n);
      mp_flush_cur_exp (mp, new_expr);
      break;
    case mp_known:
      set_cur_exp_value_number (cur_exp_value_number ());
      number_abs (cur_exp_value_number ());
      break;
    case mp_picture_type:
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      mp_pict_length (mp, &new_expr.data.n);
      mp_flush_cur_exp (mp, new_expr);
      break;
    default:
      if (mp_nice_pair (mp, cur_exp_node (), mp->cur_exp.type)) {
        memset(&new_expr,0,sizeof(mp_value));
        new_number(new_expr.data.n);
        pyth_add (new_expr.data.n, value_number (x_part (value_node (cur_exp_node ()))),
                                   value_number (y_part (value_node (cur_exp_node ()))));
        mp_flush_cur_exp (mp, new_expr);
      } else
        mp_bad_unary (mp, c);
      break;
    }
    break;
  case mp_turning_op:
    if (mp->cur_exp.type == mp_pair_type) {
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      set_number_to_zero(new_expr.data.n);
      mp_flush_cur_exp (mp, new_expr);
    } else if (mp->cur_exp.type != mp_path_type) {
      mp_bad_unary (mp, mp_turning_op);
    } else if (mp_left_type (cur_exp_knot ()) == mp_endpoint) {
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      new_expr.data.p = NULL;
      mp_flush_cur_exp (mp, new_expr);      /* not a cyclic path */
    } else {
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      mp_turn_cycles_wrapper (mp, &new_expr.data.n, cur_exp_knot ());
      mp_flush_cur_exp (mp, new_expr);
    }
    break;
  case mp_boolean_type:
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    type_range (mp_boolean_type, mp_unknown_boolean);
    break;
  case mp_string_type:
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    type_range (mp_string_type, mp_unknown_string);
    break;
  case mp_pen_type:
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    type_range (mp_pen_type, mp_unknown_pen);
    break;
  case mp_path_type:
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    type_range (mp_path_type, mp_unknown_path);
    break;
  case mp_picture_type:
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    type_range (mp_picture_type, mp_unknown_picture);
    break;
  case mp_transform_type:
  case mp_color_type:
  case mp_cmykcolor_type:
  case mp_pair_type:
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    type_test (c);
    break;
  case mp_numeric_type:
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    type_range (mp_known, mp_independent);
    break;
  case mp_known_op:
  case mp_unknown_op:
    mp_test_known (mp, c);
    break;
  case mp_cycle_op:
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    if (mp->cur_exp.type != mp_path_type)
      set_number_from_boolean (new_expr.data.n, mp_false_code);
    else if (mp_left_type (cur_exp_knot ()) != mp_endpoint)
      set_number_from_boolean (new_expr.data.n, mp_true_code);
    else
      set_number_from_boolean (new_expr.data.n, mp_false_code);
    mp_flush_cur_exp (mp, new_expr);
    mp->cur_exp.type = mp_boolean_type;
    break;
  case mp_arc_length:
    if (mp->cur_exp.type == mp_pair_type)
      mp_pair_to_path (mp);
    if (mp->cur_exp.type != mp_path_type) {
      mp_bad_unary (mp, mp_arc_length);
    } else {
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      mp_get_arc_length (mp, &new_expr.data.n, cur_exp_knot ());
      mp_flush_cur_exp (mp, new_expr);
    }
    break;
  case mp_filled_op:
  case mp_stroked_op:
  case mp_textual_op:
  case mp_clipped_op:
  case mp_bounded_op:
    /* Here we use the fact that |c-filled_op+fill_code| is the desired graphical
    object |type|. */
@^data structure assumptions@>
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    if (mp->cur_exp.type != mp_picture_type) {
      set_number_from_boolean (new_expr.data.n, mp_false_code);
    } else if (mp_link (edge_list (cur_exp_node ())) == NULL) {
      set_number_from_boolean (new_expr.data.n, mp_false_code);
    } else if (mp_type (mp_link (edge_list (cur_exp_node ()))) ==
               (mp_variable_type) (c + mp_fill_node_type - mp_filled_op)) {
      set_number_from_boolean (new_expr.data.n, mp_true_code);
    } else {
      set_number_from_boolean (new_expr.data.n, mp_false_code);
    }
    mp_flush_cur_exp (mp, new_expr);
    mp->cur_exp.type = mp_boolean_type;
    break;
  case mp_make_pen_op:
    if (mp->cur_exp.type == mp_pair_type)
      mp_pair_to_path (mp);
    if (mp->cur_exp.type != mp_path_type)
      mp_bad_unary (mp, mp_make_pen_op);
    else {
      mp->cur_exp.type = mp_pen_type;
      set_cur_exp_knot (mp_make_pen (mp, cur_exp_knot (), true));
    }
    break;
  case mp_make_path_op:
    if (mp->cur_exp.type != mp_pen_type) {
      mp_bad_unary (mp, mp_make_path_op);
    } else {
      mp->cur_exp.type = mp_path_type;
      mp_make_path (mp, cur_exp_knot ());
    }
    break;
  case mp_reverse:
    if (mp->cur_exp.type == mp_path_type) {
      mp_knot pk = mp_htap_ypoc (mp, cur_exp_knot ());
      if (mp_right_type (pk) == mp_endpoint)
        pk = mp_next_knot (pk);
      mp_toss_knot_list (mp, cur_exp_knot ());
      set_cur_exp_knot (pk);
    } else if (mp->cur_exp.type == mp_pair_type) {
      mp_pair_to_path (mp);
    } else {
      mp_bad_unary (mp, mp_reverse);
    }
    break;
  case mp_ll_corner_op:
    if (!mp_get_cur_bbox (mp))
      mp_bad_unary (mp, mp_ll_corner_op);
    else
      mp_pair_value (mp, mp_minx, mp_miny);
    break;
  case mp_lr_corner_op:
    if (!mp_get_cur_bbox (mp))
      mp_bad_unary (mp, mp_lr_corner_op);
    else
      mp_pair_value (mp,  mp_maxx, mp_miny);
    break;
  case mp_ul_corner_op:
    if (!mp_get_cur_bbox (mp))
      mp_bad_unary (mp, mp_ul_corner_op);
    else
      mp_pair_value (mp, mp_minx, mp_maxy);
    break;
  case mp_ur_corner_op:
    if (!mp_get_cur_bbox (mp))
      mp_bad_unary (mp, mp_ur_corner_op);
    else
      mp_pair_value (mp, mp_maxx,  mp_maxy);
    break;
  case mp_read_from_op:
  case mp_close_from_op:
    if (mp->cur_exp.type != mp_string_type)
      mp_bad_unary (mp, c);
    else
      mp_do_read_or_close (mp, c);
    break;

  }                             /* there are no other cases */
  check_arith();
}


@ The |nice_pair| function returns |true| if both components of a pair
are known.

@<Declare unary action procedures@>=
static boolean mp_nice_pair (MP mp, mp_node p, quarterword t) {
  (void) mp;
  if (t == mp_pair_type) {
    p = value_node (p);
    if (mp_type (x_part (p)) == mp_known)
      if (mp_type (y_part (p)) == mp_known)
        return true;
  }
  return false;
}


@ The |nice_color_or_pair| function is analogous except that it also accepts
fully known colors.

@<Declare unary action procedures@>=
static boolean mp_nice_color_or_pair (MP mp, mp_node p, quarterword t) {
  mp_node q;
  (void) mp;
  switch (t) {
  case mp_pair_type:
    q = value_node (p);
    if (mp_type (x_part (q)) == mp_known)
      if (mp_type (y_part (q)) == mp_known)
        return true;
    break;
  case mp_color_type:
    q = value_node (p);
    if (mp_type (red_part (q)) == mp_known)
      if (mp_type (green_part (q)) == mp_known)
        if (mp_type (blue_part (q)) == mp_known)
          return true;
    break;
  case mp_cmykcolor_type:
    q = value_node (p);
    if (mp_type (cyan_part (q)) == mp_known)
      if (mp_type (magenta_part (q)) == mp_known)
        if (mp_type (yellow_part (q)) == mp_known)
          if (mp_type (black_part (q)) == mp_known)
            return true;
    break;
  }
  return false;
}


@ @<Declare unary action...@>=
static void mp_print_known_or_unknown_type (MP mp, quarterword t, mp_node v) {
  mp_print_char (mp, xord ('('));
  if (t > mp_known)
    mp_print (mp, "unknown numeric");
  else {
    if ((t == mp_pair_type) || (t == mp_color_type) || (t == mp_cmykcolor_type))
      if (!mp_nice_color_or_pair (mp, v, t))
        mp_print (mp, "unknown ");
    mp_print_type (mp, t);
  }
  mp_print_char (mp, xord (')'));
}


@ @<Declare unary action...@>=
static void mp_bad_unary (MP mp, quarterword c) {
  char msg[256];
  mp_string sname;
  int old_setting = mp->selector;
  const char *hlp[] = {
         "I'm afraid I don't know how to apply that operation to that",
         "particular type. Continue, and I'll simply return the",
         "argument (shown above) as the result of the operation.",
         NULL };
  mp->selector = new_string;
  mp_print_op (mp, c);
  mp_print_known_or_unknown_type (mp, mp->cur_exp.type, cur_exp_node ());
  sname = mp_make_string(mp);
  mp->selector = old_setting;
  mp_snprintf (msg, 256, "Not implemented: %s", mp_str(mp, sname));
  delete_str_ref(sname);
  mp_disp_err(mp, NULL);
  mp_back_error (mp, msg, hlp, true);
@.Not implemented...@>;
  mp_get_x_next (mp);
}



@ Negation is easy except when the current expression
is of type |independent|, or when it is a pair with one or more
|independent| components.

@<Declare unary action...@>=
static void mp_negate_dep_list (MP mp, mp_value_node p) {
  (void) mp;
  while (1) {
    number_negate (dep_value (p));
    if (dep_info (p) == NULL)
      return;
    p = (mp_value_node) mp_link (p);
  }
}


@ It is tempting to argue that the negative of an independent variable
is an independent variable, hence we don't have to do anything when
negating it. The fallacy is that other dependent variables pointing
to the current expression must change the sign of their
coefficients if we make no change to the current expression.

Instead, we work around the problem by copying the current expression
and recycling it afterwards (cf.~the |stash_in| routine).

@d negate_value(A) if (mp_type (A) == mp_known) {
        set_value_number(A, (value_number (A))); /* to clear the rest */
        number_negate (value_number (A));
      } else {
        mp_negate_dep_list (mp, (mp_value_node) dep_list ((mp_value_node) A));
      }

@<Declare unary action...@>=
static void negate_cur_expr(MP mp) {
  mp_node p, q, r;      /* for list manipulation */
  switch (mp->cur_exp.type) {
  case mp_color_type:
  case mp_cmykcolor_type:
  case mp_pair_type:
  case mp_independent:
    q = cur_exp_node ();
    mp_make_exp_copy (mp, q);
    if (mp->cur_exp.type == mp_dependent) {
      mp_negate_dep_list (mp, (mp_value_node) dep_list ((mp_value_node)
                                                        cur_exp_node ()));
    } else if (mp->cur_exp.type <= mp_pair_type) {
      /* |mp_color_type| |mp_cmykcolor_type|, or |mp_pair_type| */
      p = value_node (cur_exp_node ());
      switch (mp->cur_exp.type) {
      case mp_pair_type:
        r = x_part (p);
        negate_value (r);
        r = y_part (p);
        negate_value (r);
        break;
      case mp_color_type:
        r = red_part (p);
        negate_value (r);
        r = green_part (p);
        negate_value (r);
        r = blue_part (p);
        negate_value (r);
        break;
      case mp_cmykcolor_type:
        r = cyan_part (p);
        negate_value (r);
        r = magenta_part (p);
        negate_value (r);
        r = yellow_part (p);
        negate_value (r);
        r = black_part (p);
        negate_value (r);
        break;
      default:                   /* there are no other valid cases, but please the compiler */
        break;
      }
    }                             /* if |cur_type=mp_known| then |cur_exp=0| */
    mp_recycle_value (mp, q);
    mp_free_value_node (mp, q);
    break;
  case mp_dependent:
  case mp_proto_dependent:
    mp_negate_dep_list (mp, (mp_value_node) dep_list ((mp_value_node)
                                                      cur_exp_node ()));
    break;
  case mp_known:
    if (is_number(cur_exp_value_number())) 
      number_negate (cur_exp_value_number());
    break;
  default:
    mp_bad_unary (mp, mp_minus);
    break;
  }
}

@ If the current expression is a pair, but the context wants it to
be a path, we call |pair_to_path|.

@<Declare unary action...@>=
static void mp_pair_to_path (MP mp) {
  set_cur_exp_knot (mp_pair_to_knot (mp));
  mp->cur_exp.type = mp_path_type;
}



@ @<Declarations@>=
static void mp_bad_color_part (MP mp, quarterword c);

@ @c
static void mp_bad_color_part (MP mp, quarterword c) {
  mp_node p;    /* the big node */
  mp_value new_expr;
  char msg[256];
  int old_setting;
  mp_string sname;
  const char *hlp[] = {
     "You can only ask for the redpart, greenpart, bluepart of a rgb object,",
     "the cyanpart, magentapart, yellowpart or blackpart of a cmyk object, ",
     "or the greypart of a grey object. No mixing and matching, please.",
     NULL };
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  p = mp_link (edge_list (cur_exp_node ()));
  mp_disp_err(mp, NULL);
  old_setting = mp->selector;
  mp->selector = new_string;
  mp_print_op (mp, c);
  sname = mp_make_string(mp);
  mp->selector = old_setting;
@.Wrong picture color model...@>;
  if (mp_color_model (p) == mp_grey_model)
    mp_snprintf (msg, 256, "Wrong picture color model: %s of grey object", mp_str(mp, sname));
  else if (mp_color_model (p) == mp_cmyk_model)
    mp_snprintf (msg, 256, "Wrong picture color model: %s of cmyk object", mp_str(mp, sname));
  else if (mp_color_model (p) == mp_rgb_model)
    mp_snprintf (msg, 256, "Wrong picture color model: %s of rgb object", mp_str(mp, sname));
  else if (mp_color_model (p) == mp_no_model)
    mp_snprintf (msg, 256, "Wrong picture color model: %s of marking object", mp_str(mp, sname));
  else
    mp_snprintf (msg, 256, "Wrong picture color model: %s of defaulted object", mp_str(mp, sname));
  delete_str_ref(sname);
  mp_error (mp, msg, hlp, true);
  if (c == mp_black_part)
    number_clone (new_expr.data.n, unity_t);
  else
    set_number_to_zero(new_expr.data.n);
  mp_flush_cur_exp (mp, new_expr);
}


@ In the following procedure, |cur_exp| points to a capsule, which points to
a big node. We want to delete all but one part of the big node.

@<Declare unary action...@>=
static void mp_take_part (MP mp, quarterword c) {
  mp_node p;    /* the big node */
  p = value_node (cur_exp_node ());
  set_value_node (mp->temp_val, p);
  mp_type (mp->temp_val) = mp->cur_exp.type;
  mp_link (p) = mp->temp_val;
  mp_free_value_node (mp, cur_exp_node ()); 
  switch (c) {
  case mp_x_part:
    if (mp->cur_exp.type == mp_pair_type)
      mp_make_exp_copy (mp, x_part (p));
    else
      mp_make_exp_copy (mp, tx_part (p));
    break;
  case mp_y_part:
    if (mp->cur_exp.type == mp_pair_type)
      mp_make_exp_copy (mp, y_part (p));
    else
      mp_make_exp_copy (mp, ty_part (p));
    break;
  case mp_xx_part:
    mp_make_exp_copy (mp, xx_part (p));
    break;
  case mp_xy_part:
    mp_make_exp_copy (mp, xy_part (p));
    break;
  case mp_yx_part:
    mp_make_exp_copy (mp, yx_part (p));
    break;
  case mp_yy_part:
    mp_make_exp_copy (mp, yy_part (p));
    break;
  case mp_red_part:
    mp_make_exp_copy (mp, red_part (p));
    break;
  case mp_green_part:
    mp_make_exp_copy (mp, green_part (p));
    break;
  case mp_blue_part:
    mp_make_exp_copy (mp, blue_part (p));
    break;
  case mp_cyan_part:
    mp_make_exp_copy (mp, cyan_part (p));
    break;
  case mp_magenta_part:
    mp_make_exp_copy (mp, magenta_part (p));
    break;
  case mp_yellow_part:
    mp_make_exp_copy (mp, yellow_part (p));
    break;
  case mp_black_part:
    mp_make_exp_copy (mp, black_part (p));
    break;
  }
  mp_recycle_value (mp, mp->temp_val);
}


@ @<Initialize table entries@>=
mp->temp_val = mp_get_value_node (mp);
mp_name_type (mp->temp_val) = mp_capsule;

@ @<Free table entries@>=
mp_free_value_node (mp, mp->temp_val);


@ @<Declarations@>=
static mp_edge_header_node mp_scale_edges (MP mp, mp_number se_sf, mp_edge_header_node se_pic);

@ @<Declare unary action...@>=
static void mp_take_pict_part (MP mp, quarterword c) {
  mp_node p;    /* first graphical object in |cur_exp| */
  mp_value new_expr;
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  p = mp_link (edge_list (cur_exp_node ()));
  if (p != NULL) {
    switch (c) {
    case mp_x_part:
    case mp_y_part:
    case mp_xx_part:
    case mp_xy_part:
    case mp_yx_part:
    case mp_yy_part:
      if (mp_type (p) == mp_text_node_type) {
        mp_text_node p0 = (mp_text_node)p;
        switch (c) {
        case mp_x_part:
          number_clone(new_expr.data.n, p0->tx);
          break;
        case mp_y_part:
          number_clone(new_expr.data.n, p0->ty);
          break;
        case mp_xx_part:
          number_clone(new_expr.data.n, p0->txx);
          break;
        case mp_xy_part:
          number_clone(new_expr.data.n, p0->txy);
          break;
        case mp_yx_part:
          number_clone(new_expr.data.n, p0->tyx);
          break;
        case mp_yy_part:
          number_clone(new_expr.data.n, p0->tyy);
          break;
        }
        mp_flush_cur_exp (mp, new_expr);
      } else
        goto NOT_FOUND;
      break;
    case mp_red_part:
    case mp_green_part:
    case mp_blue_part:
      if (has_color (p)) {
        switch (c) {
        case mp_red_part:
          number_clone(new_expr.data.n,((mp_stroked_node)p)->red);
          break;
        case mp_green_part:
          number_clone(new_expr.data.n,((mp_stroked_node)p)->green);
          break;
        case mp_blue_part:
          number_clone(new_expr.data.n,((mp_stroked_node)p)->blue);
          break;
        }
        mp_flush_cur_exp (mp, new_expr);
      } else
        goto NOT_FOUND;
      break;
    case mp_cyan_part:
    case mp_magenta_part:
    case mp_yellow_part:
    case mp_black_part:
      if (has_color (p)) {
        if (mp_color_model (p) == mp_uninitialized_model && c == mp_black_part) {
          set_number_to_unity(new_expr.data.n);
        } else {
          switch (c) {
          case mp_cyan_part:
            number_clone(new_expr.data.n,((mp_stroked_node)p)->cyan);
            break;
          case mp_magenta_part:
            number_clone(new_expr.data.n,((mp_stroked_node)p)->magenta);
            break;
          case mp_yellow_part:
            number_clone(new_expr.data.n,((mp_stroked_node)p)->yellow);
            break;
          case mp_black_part:
            number_clone(new_expr.data.n,((mp_stroked_node)p)->black);
            break;
          }
        }
        mp_flush_cur_exp (mp, new_expr);
      } else
        goto NOT_FOUND;
      break;
    case mp_grey_part:
      if (has_color (p)) {
        number_clone(new_expr.data.n,((mp_stroked_node)p)->grey);
        mp_flush_cur_exp (mp, new_expr);
      } else
        goto NOT_FOUND;
      break;
    case mp_color_model_part:
      if (has_color (p)) {
        if (mp_color_model (p) == mp_uninitialized_model) {
          number_clone (new_expr.data.n, internal_value (mp_default_color_model));
        } else {
          number_clone (new_expr.data.n, unity_t);
          number_multiply_int (new_expr.data.n, mp_color_model (p));
        }
        mp_flush_cur_exp (mp, new_expr);
      } else
        goto NOT_FOUND;
      break;
    case mp_text_part:
      if (mp_type (p) != mp_text_node_type)
        goto NOT_FOUND;
      else {
        new_expr.data.str = mp_text_p (p);
        add_str_ref (new_expr.data.str);
        mp_flush_cur_exp (mp, new_expr);
        mp->cur_exp.type = mp_string_type;
      };
      break;
    case mp_prescript_part:
      if (!has_color (p)) {
        goto NOT_FOUND;
      } else {
        if (mp_pre_script(p)) {
          new_expr.data.str = mp_pre_script(p);
          add_str_ref (new_expr.data.str);
        } else {
          new_expr.data.str = mp_rts(mp,"");
        }
        mp_flush_cur_exp (mp, new_expr);
        mp->cur_exp.type = mp_string_type;
      };
      break;
    case mp_postscript_part:
      if (!has_color (p)) {
        goto NOT_FOUND;
      } else {
        if (mp_post_script(p)) {
          new_expr.data.str = mp_post_script(p);
          add_str_ref (new_expr.data.str);
        } else {
          new_expr.data.str = mp_rts(mp,"");
        }
        mp_flush_cur_exp (mp, new_expr);
        mp->cur_exp.type = mp_string_type;
      };
      break;
    case mp_font_part:
      if (mp_type (p) != mp_text_node_type)
        goto NOT_FOUND;
      else {
        new_expr.data.str = mp_rts (mp, mp->font_name[mp_font_n (p)]);
        add_str_ref (new_expr.data.str);
        mp_flush_cur_exp (mp, new_expr);
        mp->cur_exp.type = mp_string_type;
      };
      break;
    case mp_path_part:
      if (mp_type (p) == mp_text_node_type) {
        goto NOT_FOUND;
      } else if (is_stop (p)) {
        mp_confusion (mp, "pict");
      } else {
        new_expr.data.node = NULL;
        switch (mp_type (p)) {
        case mp_fill_node_type:
          new_expr.data.p = mp_copy_path (mp, mp_path_p ((mp_fill_node) p));
          break;
        case mp_stroked_node_type:
          new_expr.data.p = mp_copy_path (mp, mp_path_p ((mp_stroked_node) p));
          break;
        case mp_start_bounds_node_type:
          new_expr.data.p = mp_copy_path (mp, mp_path_p ((mp_start_bounds_node) p));
          break;
        case mp_start_clip_node_type:
          new_expr.data.p = mp_copy_path (mp, mp_path_p ((mp_start_clip_node) p));
          break;
        default:
          assert (0);
          break;
        }
        mp_flush_cur_exp (mp, new_expr);
        mp->cur_exp.type = mp_path_type;
      }
      break;
    case mp_pen_part:
      if (!has_pen (p)) {
        goto NOT_FOUND;
      } else {
        switch (mp_type (p)) {
        case mp_fill_node_type:
          if (mp_pen_p ((mp_fill_node) p) == NULL)
            goto NOT_FOUND;
          else {
            new_expr.data.p = copy_pen (mp_pen_p ((mp_fill_node) p));
            mp_flush_cur_exp (mp, new_expr);
            mp->cur_exp.type = mp_pen_type;
          }
          break;
        case mp_stroked_node_type:
          if (mp_pen_p ((mp_stroked_node) p) == NULL)
            goto NOT_FOUND;
          else {
            new_expr.data.p = copy_pen (mp_pen_p ((mp_stroked_node) p));
            mp_flush_cur_exp (mp, new_expr);
            mp->cur_exp.type = mp_pen_type;
          }
          break;
        default:
          assert (0);
          break;
        }
      }
      break;
    case mp_dash_part:
      if (mp_type (p) != mp_stroked_node_type) {
        goto NOT_FOUND;
      } else {
        if (mp_dash_p (p) == NULL) {
          goto NOT_FOUND;
        } else {
          add_edge_ref (mp_dash_p (p));
          new_expr.data.node = (mp_node)mp_scale_edges (mp, ((mp_stroked_node)p)->dash_scale, 
                                                            (mp_edge_header_node)mp_dash_p (p));
          mp_flush_cur_exp (mp, new_expr);
          mp->cur_exp.type = mp_picture_type;
        }
      }
      break;
    }                           /* all cases have been enumerated */
    return;
  };
NOT_FOUND:
  /* Convert the current expression to a NULL value appropriate for |c| */
  switch (c) {
  case mp_text_part:
  case mp_font_part:
  case mp_prescript_part:
  case mp_postscript_part:
    new_expr.data.str = mp_rts(mp,"");
    mp_flush_cur_exp (mp, new_expr);
    mp->cur_exp.type = mp_string_type;
    break;
  case mp_path_part:
    new_expr.data.p = mp_new_knot (mp);
    mp_flush_cur_exp (mp, new_expr);
    mp_left_type (cur_exp_knot ()) = mp_endpoint;
    mp_right_type (cur_exp_knot ()) = mp_endpoint;
    mp_next_knot (cur_exp_knot ()) = cur_exp_knot ();
    set_number_to_zero(cur_exp_knot ()->x_coord);
    set_number_to_zero(cur_exp_knot ()->y_coord);
    mp_originator (cur_exp_knot ()) = mp_metapost_user;
    mp->cur_exp.type = mp_path_type;
    break;
  case mp_pen_part:
    new_expr.data.p = mp_get_pen_circle (mp, zero_t);
    mp_flush_cur_exp (mp, new_expr);
    mp->cur_exp.type = mp_pen_type;
    break;
  case mp_dash_part:
    new_expr.data.node = (mp_node)mp_get_edge_header_node (mp);
    mp_flush_cur_exp (mp, new_expr);
    mp_init_edges (mp, (mp_edge_header_node)cur_exp_node ());
    mp->cur_exp.type = mp_picture_type;
    break;
  default:
    set_number_to_zero(new_expr.data.n);
    mp_flush_cur_exp (mp, new_expr);
    break;
  }
}

@ @<Declare unary action...@>=
static void mp_str_to_num (MP mp, quarterword c) {  /* converts a string to a number */
  integer n;    /* accumulator */
  ASCII_code m; /* current character */
  unsigned k;   /* index into |str_pool| */
  int b;        /* radix of conversion */
  boolean bad_char;     /* did the string contain an invalid digit? */
  mp_value new_expr;
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  if (c == mp_ASCII_op) {
    if (cur_exp_str ()->len == 0)
      n = -1;
    else
      n = cur_exp_str ()->str[0];
  } else {
    if (c == mp_oct_op)
      b = 8;
    else
      b = 16;
    n = 0;
    bad_char = false;
    for (k = 0; k < cur_exp_str ()->len; k++) {
      m = (ASCII_code) (*(cur_exp_str ()->str + k));
      if ((m >= '0') && (m <= '9'))
        m = (ASCII_code) (m - '0');
      else if ((m >= 'A') && (m <= 'F'))
        m = (ASCII_code) (m - 'A' + 10);
      else if ((m >= 'a') && (m <= 'f'))
        m = (ASCII_code) (m - 'a' + 10);
      else {
        bad_char = true;
        m = 0;
      };
      if ((int) m >= b) {
        bad_char = true;
        m = 0;
      };
      if (n < 32768 / b)
        n = n * b + m;
      else
        n = 32767;
    }
    /* Give error messages if |bad_char| or |n>=4096| */
    if (bad_char) {
      const char *hlp[] = {"I zeroed out characters that weren't hex digits.", NULL};
      if (c == mp_oct_op) {
        hlp[0] = "I zeroed out characters that weren't in the range 0..7.";
      }
      mp_disp_err(mp, NULL);
      mp_back_error (mp, "String contains illegal digits", hlp, true);
      mp_get_x_next (mp);
    }
    if ((n > 4095)) { /* todo, this is scaled specific */
      if (number_positive (internal_value (mp_warning_check))) {
        char msg[256];
        const char *hlp[] = {
               "I have trouble with numbers greater than 4095; watch out.",
               "(Set warningcheck:=0 to suppress this message.)",
               NULL };
        mp_snprintf (msg, 256,"Number too large (%d)", (int)n);
        mp_back_error (mp, msg, hlp, true);
        mp_get_x_next (mp);
      }
    }
  }
  number_clone (new_expr.data.n, unity_t);
  number_multiply_int(new_expr.data.n, n);
  mp_flush_cur_exp (mp, new_expr);
}

@ @<Declare unary action...@>=
static void mp_path_length (MP mp, mp_number *n) {                               /* computes the length of the current path */
  mp_knot p;    /* traverser */
  set_number_to_zero (*n);
  p = cur_exp_knot ();
  if (mp_left_type (p) == mp_endpoint) {
    number_substract(*n, unity_t); /* -unity */
  }
  do {
    p = mp_next_knot (p);
    number_add(*n, unity_t);
  } while (p != cur_exp_knot ());
}


@ @<Declare unary action...@>=
static void mp_pict_length (MP mp, mp_number *n) {
  /* counts interior components in picture |cur_exp| */
  mp_node p;    /* traverser */
  set_number_to_zero (*n);
  p = mp_link (edge_list (cur_exp_node ()));
  if (p != NULL) {
    if (is_start_or_stop (p))
      if (mp_skip_1component (mp, p) == NULL)
        p = mp_link (p);
    while (p != NULL) {
      if ( ! is_start_or_stop(p) ) 
        p = mp_link(p);
      else if ( ! is_stop(p)) 
        p = mp_skip_1component(mp, p);
      else 
        return;
      number_add(*n, unity_t);
    }
  }
}


@ The function |an_angle| returns the value of the |angle| primitive, or $0$ if the
argument is |origin|.

@<Declare unary action...@>=
static void mp_an_angle (MP mp, mp_number *ret, mp_number xpar, mp_number ypar) {
  set_number_to_zero (*ret);
  if ((!(number_zero(xpar) && number_zero(ypar)))) {
    n_arg (*ret, xpar, ypar);
  }
}


@ The actual turning number is (for the moment) computed in a C function
that receives eight integers corresponding to the four controlling points,
and returns a single angle.  Besides those, we have to account for discrete
moves at the actual points.

@d mp_floor(a) ((a)>=0 ? (int)(a) : -(int)(-(a)))
@d bezier_error (720*(256*256*16))+1
@d mp_sign(v) ((v)>0 ? 1 : ((v)<0 ? -1 : 0 ))
@d mp_out(A) (double)((A)/16)

@<Declare unary action...@>=
static void mp_bezier_slope (MP mp, mp_number *ret, mp_number AX, mp_number AY, mp_number BX,
                              mp_number BY, mp_number CX, mp_number CY, mp_number DX,
                              mp_number DY);

@ @c
static void mp_bezier_slope (MP mp, mp_number *ret, mp_number AX, mp_number AY, mp_number BX,
                              mp_number BY, mp_number CX, mp_number CY, mp_number DX,
                              mp_number DY) {
  double a, b, c;
  mp_number deltax, deltay;
  double ax, ay, bx, by, cx, cy, dx, dy;
  mp_number xi, xo, xm;
  double res = 0;
  ax = number_to_double (AX);
  ay = number_to_double (AY);
  bx = number_to_double (BX);
  by = number_to_double (BY);
  cx = number_to_double (CX);
  cy = number_to_double (CY);
  dx = number_to_double (DX);
  dy = number_to_double (DY);
  new_number (deltax);
  new_number (deltay);
  set_number_from_substraction(deltax, BX, AX);
  set_number_from_substraction(deltay, BY, AY);
  if (number_zero(deltax) && number_zero(deltay)) {
    set_number_from_substraction(deltax, CX, AX);
    set_number_from_substraction(deltay, CY, AY);
  }
  if (number_zero(deltax) && number_zero(deltay)) {
    set_number_from_substraction(deltax, DX, AX);
    set_number_from_substraction(deltay, DY, AY);
  }
  new_number (xi);
  new_number (xm);
  new_number (xo);
  mp_an_angle (mp, &xi, deltax, deltay);
  set_number_from_substraction(deltax, CX, BX);
  set_number_from_substraction(deltay, CY, BY);
  mp_an_angle (mp, &xm, deltax, deltay); /* !!! never used? */
  set_number_from_substraction(deltax, DX, CX);
  set_number_from_substraction(deltay, DY, CY);
  if (number_zero(deltax) && number_zero(deltay)) {
    set_number_from_substraction(deltax, DX, BX);
    set_number_from_substraction(deltay, DY, BY);
  }
  if (number_zero(deltax) && number_zero(deltay)) {
    set_number_from_substraction(deltax, DX, AX);
    set_number_from_substraction(deltay, DY, AY);
  }
  mp_an_angle (mp, &xo, deltax, deltay);
  a = (bx - ax) * (cy - by) - (cx - bx) * (by - ay);    /* a = (bp-ap)x(cp-bp); */
  b = (bx - ax) * (dy - cy) - (by - ay) * (dx - cx);;   /* b = (bp-ap)x(dp-cp); */
  c = (cx - bx) * (dy - cy) - (dx - cx) * (cy - by);    /* c = (cp-bp)x(dp-cp); */
  if ((a == 0) && (c == 0)) {
    res = (b == 0 ? 0 : (mp_out (number_to_double(xo)) - mp_out (number_to_double(xi))));
  } else if ((a == 0) || (c == 0)) {
    if ((mp_sign (b) == mp_sign (a)) || (mp_sign (b) == mp_sign (c))) {
      res = mp_out (number_to_double(xo)) - mp_out (number_to_double(xi));  /* ? */
      if (res < -180.0)
        res += 360.0;
      else if (res > 180.0)
        res -= 360.0;
    } else {
      res = mp_out (number_to_double(xo)) - mp_out (number_to_double(xi));  /* ? */
    }
  } else if ((mp_sign (a) * mp_sign (c)) < 0) {
    res = mp_out (number_to_double(xo)) - mp_out (number_to_double(xi));    /* ? */
    if (res < -180.0)
      res += 360.0;
    else if (res > 180.0)
      res -= 360.0;
  } else {
    if (mp_sign (a) == mp_sign (b)) {
      res = mp_out (number_to_double(xo)) - mp_out (number_to_double(xi));  /* ? */
      if (res < -180.0)
        res += 360.0;
      else if (res > 180.0)
        res -= 360.0;
    } else {
      if ((b * b) == (4 * a * c)) {
        res = (double) bezier_error;
      } else if ((b * b) < (4 * a * c)) {
        res = mp_out (number_to_double(xo)) - mp_out (number_to_double(xi));        /* ? */
        if (res <= 0.0 && res > -180.0)
          res += 360.0;
        else if (res >= 0.0 && res < 180.0)
          res -= 360.0;
      } else {
        res = mp_out (number_to_double(xo)) - mp_out (number_to_double(xi));
        if (res < -180.0)
          res += 360.0;
        else if (res > 180.0)
          res -= 360.0;
      }
    }
  }
  free_number (deltax);
  free_number (deltay);
  free_number (xi);
  free_number (xo);
  free_number (xm);
  set_number_from_double(*ret, res);
  convert_scaled_to_angle (*ret);
}


@
@d p_nextnext mp_next_knot(mp_next_knot(p))
@d p_next mp_next_knot(p)

@<Declare unary action...@>=
static void mp_turn_cycles (MP mp, mp_number *turns, mp_knot c) {
  mp_angle res, ang;       /*  the angles of intermediate results  */
  mp_knot p;    /*  for running around the path  */
  mp_number xp, yp;       /*  coordinates of next point  */
  mp_number x, y; /*  helper coordinates  */
  mp_number arg1, arg2;
  mp_angle in_angle, out_angle;    /*  helper angles */
  mp_angle seven_twenty_deg_t, neg_one_eighty_deg_t;
  unsigned old_setting; /* saved |selector| setting */
  set_number_to_zero(*turns);
  new_number(arg1);
  new_number(arg2);
  new_number(xp);
  new_number(yp);
  new_number(x);
  new_number(y);
  new_angle(in_angle);
  new_angle(out_angle);
  new_angle(ang);
  new_angle(res);
  new_angle(seven_twenty_deg_t);
  new_angle(neg_one_eighty_deg_t);
  number_clone(seven_twenty_deg_t, three_sixty_deg_t);
  number_double(seven_twenty_deg_t);
  number_clone(neg_one_eighty_deg_t, one_eighty_deg_t);
  number_negate(neg_one_eighty_deg_t);
  p = c;
  old_setting = mp->selector;
  mp->selector = term_only;
  if (number_greater (internal_value (mp_tracing_commands), unity_t)) {
    mp_begin_diagnostic (mp);
    mp_print_nl (mp, "");
    mp_end_diagnostic (mp, false);
  }
  do {
    number_clone (xp, p_next->x_coord);
    number_clone (yp, p_next->y_coord);
    mp_bezier_slope (mp, &ang, p->x_coord,  p->y_coord, p->right_x, p->right_y, 
                         p_next->left_x, p_next->left_y, xp, yp);
    if (number_greater(ang, seven_twenty_deg_t)) {
      mp_error (mp, "Strange path", NULL, true);
      mp->selector = old_setting;
      set_number_to_zero(*turns);
      goto DONE;
    }
    number_add(res, ang);
    if (number_greater(res, one_eighty_deg_t)) {
      number_substract(res, three_sixty_deg_t);
      number_add(*turns, unity_t);
    }
    if (number_lessequal(res, neg_one_eighty_deg_t)) {
      number_add(res, three_sixty_deg_t);
      number_substract(*turns, unity_t);
    }
    /*  incoming angle at next point  */
    number_clone (x, p_next->left_x);
    number_clone (y, p_next->left_y);
    if (number_equal(xp, x) && number_equal(yp, y)) {
      number_clone (x, p->right_x);
      number_clone (y, p->right_y);
    }
    if (number_equal(xp, x) && number_equal(yp, y)) {
      number_clone (x, p->x_coord);
      number_clone (y, p->y_coord);
    }
    set_number_from_substraction(arg1, xp, x);
    set_number_from_substraction(arg2, yp, y);
    mp_an_angle (mp, &in_angle, arg1, arg2);
    /*  outgoing angle at next point  */
    number_clone (x, p_next->right_x);
    number_clone (y, p_next->right_y);
    if (number_equal(xp, x) && number_equal(yp, y)) {
      number_clone (x, p_nextnext->left_x);
      number_clone (y, p_nextnext->left_y);
    }
    if (number_equal(xp, x) && number_equal(yp, y)) {
      number_clone (x, p_nextnext->x_coord);
      number_clone (y, p_nextnext->y_coord);
    }
    set_number_from_substraction(arg1, x, xp);
    set_number_from_substraction(arg2, y, yp);
    mp_an_angle (mp, &out_angle, arg1, arg2);
    set_number_from_substraction(ang, out_angle, in_angle);
    mp_reduce_angle (mp, &ang);
    if (number_nonzero(ang)) {
      number_add(res, ang);
      if (number_greaterequal(res, one_eighty_deg_t)) {
        number_substract(res, three_sixty_deg_t);
        number_add(*turns, unity_t);
      }
      if (number_lessequal(res, neg_one_eighty_deg_t)) {
        number_add(res, three_sixty_deg_t);
        number_substract(*turns, unity_t);
      }
    }
    p = mp_next_knot (p);
  } while (p != c);
  mp->selector = old_setting;
DONE:
  free_number(xp);
  free_number(yp);
  free_number(x);
  free_number(y);
  free_number(seven_twenty_deg_t);
  free_number(neg_one_eighty_deg_t);
  free_number(in_angle);
  free_number(out_angle);
  free_number(ang);
  free_number(res);
  free_number(arg1);
  free_number(arg2);
}

@ @<Declare unary action...@>=
static void mp_turn_cycles_wrapper (MP mp, mp_number *ret, mp_knot c) {
  if (mp_next_knot (c) == c) { 
    /* one-knot paths always have a turning number of 1 */
    set_number_to_unity(*ret);
  } else {
    mp_turn_cycles (mp, ret, c);
  }
}

@ @<Declare unary action procedures@>=
static void mp_test_known (MP mp, quarterword c) {
  int b;        /* is the current expression known? */
  mp_node p;    /* location in a big node */
  mp_value new_expr;
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  b = mp_false_code;
  switch (mp->cur_exp.type) {
  case mp_vacuous:
  case mp_boolean_type:
  case mp_string_type:
  case mp_pen_type:
  case mp_path_type:
  case mp_picture_type:
  case mp_known:
    b = mp_true_code;
    break;
  case mp_transform_type:
    p = value_node (cur_exp_node ());
    if (mp_type (tx_part (p)) != mp_known)
      break;
    if (mp_type (ty_part (p)) != mp_known)
      break;
    if (mp_type (xx_part (p)) != mp_known)
      break;
    if (mp_type (xy_part (p)) != mp_known)
      break;
    if (mp_type (yx_part (p)) != mp_known)
      break;
    if (mp_type (yy_part (p)) != mp_known)
      break;
    b = mp_true_code;
    break;
  case mp_color_type:
    p = value_node (cur_exp_node ());
    if (mp_type (red_part (p)) != mp_known)
      break;
    if (mp_type (green_part (p)) != mp_known)
      break;
    if (mp_type (blue_part (p)) != mp_known)
      break;
    b = mp_true_code;
    break;
  case mp_cmykcolor_type:
    p = value_node (cur_exp_node ());
    if (mp_type (cyan_part (p)) != mp_known)
      break;
    if (mp_type (magenta_part (p)) != mp_known)
      break;
    if (mp_type (yellow_part (p)) != mp_known)
      break;
    if (mp_type (black_part (p)) != mp_known)
      break;
    b = mp_true_code;
    break;
  case mp_pair_type:
    p = value_node (cur_exp_node ());
    if (mp_type (x_part (p)) != mp_known)
      break;
    if (mp_type (y_part (p)) != mp_known)
      break;
    b = mp_true_code;
    break;
  default:
    break;
  }
  if (c == mp_known_op) {
    set_number_from_boolean (new_expr.data.n, b);
  } else {
    if (b==mp_true_code) {
      set_number_from_boolean (new_expr.data.n, mp_false_code);
    } else {
      set_number_from_boolean (new_expr.data.n, mp_true_code);
    }
  }
  mp_flush_cur_exp (mp, new_expr);
  cur_exp_node() = NULL; /* !! do not replace with |set_cur_exp_node()| !! */
  mp->cur_exp.type = mp_boolean_type;
}

@ The |pair_value| routine changes the current expression to a
given ordered pair of values.

@<Declare unary action procedures@>=
static void mp_pair_value (MP mp, mp_number x, mp_number y) {
  mp_node p;    /* a pair node */
  mp_value new_expr;
  mp_number x1, y1;
  new_number(x1);
  new_number(y1);
  number_clone (x1, x);
  number_clone (y1, y);
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  p = mp_get_value_node (mp);
  new_expr.type = mp_type (p);
  new_expr.data.node = p;
  mp_flush_cur_exp (mp, new_expr);
  mp->cur_exp.type = mp_pair_type;
  mp_name_type (p) = mp_capsule;
  mp_init_pair_node (mp, p);
  p = value_node (p);
  mp_type (x_part (p)) = mp_known;
  set_value_number (x_part (p), x1);
  mp_type (y_part (p)) = mp_known;
  set_value_number (y_part (p), y1);
  free_number(x1);
  free_number(y1);
}


@ Here is a function that sets |minx|, |maxx|, |miny|, |maxy| to the bounding
box of the current expression.  The boolean result is |false| if the expression
has the wrong type.

@<Declare unary action procedures@>=
static boolean mp_get_cur_bbox (MP mp) {
  switch (mp->cur_exp.type) {
  case mp_picture_type:
  {
    mp_edge_header_node p0 = (mp_edge_header_node)cur_exp_node ();
    mp_set_bbox (mp, p0, true);
    if (number_greater(p0->minx, p0->maxx)) {
      set_number_to_zero(mp_minx);
      set_number_to_zero(mp_maxx);
      set_number_to_zero(mp_miny);
      set_number_to_zero(mp_maxy);
    } else {
      number_clone (mp_minx, p0->minx);
      number_clone (mp_maxx, p0->maxx);
      number_clone (mp_miny, p0->miny);
      number_clone (mp_maxy, p0->maxy);
    }
  }
    break;
  case mp_path_type:
    mp_path_bbox (mp, cur_exp_knot ());
    break;
  case mp_pen_type:
    mp_pen_bbox (mp, cur_exp_knot ());
    break;
  default:
    return false;
  }
  return true;
}


@ Here is a routine that interprets |cur_exp| as a file name and tries to read
a line from the file or to close the file.

@<Declare unary action procedures@>=
static void mp_do_read_or_close (MP mp, quarterword c) {
  mp_value new_expr;
  readf_index n, n0;    /* indices for searching |rd_fname| */
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  /* Find the |n| where |rd_fname[n]=cur_exp|; if |cur_exp| must be inserted,
     call |start_read_input| and |goto found| or |not_found| */
  /* Free slots in the |rd_file| and |rd_fname| arrays are marked with NULL's in
     |rd_fname|. */
  {
    char *fn;
    n = mp->read_files;
    n0 = mp->read_files;
    fn = mp_xstrdup (mp, mp_str (mp, cur_exp_str ()));
    while (mp_xstrcmp (fn, mp->rd_fname[n]) != 0) {
      if (n > 0) {
        decr (n);
      } else if (c == mp_close_from_op) {
        goto CLOSE_FILE;
      } else {
        if (n0 == mp->read_files) {
          if (mp->read_files < mp->max_read_files) {
            incr (mp->read_files);
          } else {
            void **rd_file;
            char **rd_fname;
            readf_index l, k;
            l = mp->max_read_files + (mp->max_read_files / 4);
            rd_file = xmalloc ((l + 1), sizeof (void *));
            rd_fname = xmalloc ((l + 1), sizeof (char *));
            for (k = 0; k <= l; k++) {
              if (k <= mp->max_read_files) {
                rd_file[k] = mp->rd_file[k];
                rd_fname[k] = mp->rd_fname[k];
              } else {
                rd_file[k] = 0;
                rd_fname[k] = NULL;
              }
            }
            xfree (mp->rd_file);
            xfree (mp->rd_fname);
            mp->max_read_files = l;
            mp->rd_file = rd_file;
            mp->rd_fname = rd_fname;
          }
        }
        n = n0;
        if (mp_start_read_input (mp, fn, n))
          goto FOUND;
        else
          goto NOT_FOUND;
      }
      if (mp->rd_fname[n] == NULL) {
        n0 = n;
      }
    }
    if (c == mp_close_from_op) {
      (mp->close_file) (mp, mp->rd_file[n]);
      goto NOT_FOUND;
    }
  }
  mp_begin_file_reading (mp);
  name = is_read;
  if (mp_input_ln (mp, mp->rd_file[n]))
    goto FOUND;
  mp_end_file_reading (mp);
NOT_FOUND:
  /* Record the end of file and set |cur_exp| to a dummy value  */
  xfree (mp->rd_fname[n]);
  mp->rd_fname[n] = NULL;
  if (n == mp->read_files - 1)
    mp->read_files = n;
  if (c == mp_close_from_op)
    goto CLOSE_FILE;
  new_expr.data.str = mp->eof_line;
  add_str_ref (new_expr.data.str);
  mp_flush_cur_exp (mp, new_expr);
  mp->cur_exp.type = mp_string_type;
  return;
CLOSE_FILE:
  mp_flush_cur_exp (mp, new_expr);
  mp->cur_exp.type = mp_vacuous;
  return;
FOUND:
  mp_flush_cur_exp (mp, new_expr);
  mp_finish_read (mp);
}

@ The string denoting end-of-file is a one-byte string at position zero, by definition.
I have to cheat a little here because 

@<Glob...@>=
mp_string eof_line;

@ @<Set init...@>=
mp->eof_line = mp_rtsl (mp, "\0", 1);
mp->eof_line->refs = MAX_STR_REF;

@ Finally, we have the operations that combine a capsule~|p|
with the current expression.

Several of the binary operations are potentially complicated by the
fact that |independent| values can sneak into capsules. For example,
we've seen an instance of this difficulty in the unary operation
of negation. In order to reduce the number of cases that need to be
handled, we first change the two operands (if necessary)
to rid them of |independent| components. The original operands are
put into capsules called |old_p| and |old_exp|, which will be
recycled after the binary operation has been safely carried out.

@d binary_return  { mp_finish_binary(mp, old_p, old_exp); return; }

@c
@<Declare binary action procedures@>;
static void mp_finish_binary (MP mp, mp_node old_p, mp_node old_exp) {
  check_arith();
  /* Recycle any sidestepped |independent| capsules */
  if (old_p != NULL) {
    mp_recycle_value (mp, old_p);
    mp_free_value_node (mp, old_p);
  }
  if (old_exp != NULL) {
    mp_recycle_value (mp, old_exp);
    mp_free_value_node (mp, old_exp);
  }
}
static void mp_do_binary (MP mp, mp_node p, integer c) {
  mp_node q, r, rr;     /* for list manipulation */
  mp_node old_p, old_exp;       /* capsules to recycle */
  mp_value new_expr;
  check_arith();
  if (number_greater (internal_value (mp_tracing_commands), two_t)) {
    /* Trace the current binary operation */
    mp_begin_diagnostic (mp);
    mp_print_nl (mp, "{(");
    mp_print_exp (mp, p, 0);      /* show the operand, but not verbosely */
    mp_print_char (mp, xord (')'));
    mp_print_op (mp, (quarterword) c);
    mp_print_char (mp, xord ('('));
    mp_print_exp (mp, NULL, 0);
    mp_print (mp, ")}");
    mp_end_diagnostic (mp, false);
  }
  /* Sidestep |independent| cases in capsule |p| */
  /* A big node is considered to be ``tarnished'' if it contains at least one
     independent component. We will define a simple function called `|tarnished|'
     that returns |NULL| if and only if its argument is not tarnished. */
  switch (mp_type (p)) {
  case mp_transform_type:
  case mp_color_type:
  case mp_cmykcolor_type:
  case mp_pair_type:
    old_p = mp_tarnished (mp, p);
    break;
  case mp_independent:
    old_p = MP_VOID;
    break;
  default:
    old_p = NULL;
    break;
  }
  if (old_p != NULL) {
    q = mp_stash_cur_exp (mp);
    old_p = p;
    mp_make_exp_copy (mp, old_p);
    p = mp_stash_cur_exp (mp);
    mp_unstash_cur_exp (mp, q);
  }

  /* Sidestep |independent| cases in the current expression */
  switch (mp->cur_exp.type) {
  case mp_transform_type:
  case mp_color_type:
  case mp_cmykcolor_type:
  case mp_pair_type:
    old_exp = mp_tarnished (mp, cur_exp_node ());
    break;
  case mp_independent:
    old_exp = MP_VOID;
    break;
  default:
    old_exp = NULL;
    break;
  }
  if (old_exp != NULL) {
    old_exp = cur_exp_node ();
    mp_make_exp_copy (mp, old_exp);
  }
  
  switch (c) {
  case mp_plus:
  case mp_minus:
    /* Add or subtract the current expression from |p| */
    if ((mp->cur_exp.type < mp_color_type) || (mp_type (p) < mp_color_type)) {
      mp_bad_binary (mp, p, (quarterword) c);
    } else {
      quarterword cc = (quarterword)c;
      if ((mp->cur_exp.type > mp_pair_type) && (mp_type (p) > mp_pair_type)) {
        mp_add_or_subtract (mp, p, NULL, cc);
      } else {
        if (mp->cur_exp.type != mp_type (p)) {
          mp_bad_binary (mp, p, cc);
        } else {
          q = value_node (p);
          r = value_node (cur_exp_node ());
          switch (mp->cur_exp.type) {
          case mp_pair_type:
            mp_add_or_subtract (mp, x_part (q), x_part (r),cc);
            mp_add_or_subtract (mp, y_part (q), y_part (r),cc);
            break;
          case mp_color_type:
            mp_add_or_subtract (mp, red_part (q), red_part (r),cc);
            mp_add_or_subtract (mp, green_part (q), green_part (r),cc);
            mp_add_or_subtract (mp, blue_part (q), blue_part (r),cc);
            break;
          case mp_cmykcolor_type:
            mp_add_or_subtract (mp, cyan_part (q), cyan_part (r),cc);
            mp_add_or_subtract (mp, magenta_part (q), magenta_part (r),cc);
            mp_add_or_subtract (mp, yellow_part (q), yellow_part (r),cc);
            mp_add_or_subtract (mp, black_part (q), black_part (r),cc);
            break;
          case mp_transform_type:
            mp_add_or_subtract (mp, tx_part (q), tx_part (r),cc);
            mp_add_or_subtract (mp, ty_part (q), ty_part (r),cc);
            mp_add_or_subtract (mp, xx_part (q), xx_part (r),cc);
            mp_add_or_subtract (mp, xy_part (q), xy_part (r),cc);
            mp_add_or_subtract (mp, yx_part (q), yx_part (r),cc);
            mp_add_or_subtract (mp, yy_part (q), yy_part (r),cc);
            break;
          default:                 /* there are no other valid cases, but please the compiler */
            break;
          }
        }
      }
    }
    break;
  case mp_less_than:
  case mp_less_or_equal:
  case mp_greater_than:
  case mp_greater_or_equal:
  case mp_equal_to:
  case mp_unequal_to:
    check_arith();                    /* at this point |arith_error| should be |false|? */
    if ((mp->cur_exp.type > mp_pair_type) && (mp_type (p) > mp_pair_type)) {
      mp_add_or_subtract (mp, p, NULL, mp_minus);      /* |cur_exp:=(p)-cur_exp| */
    } else if (mp->cur_exp.type != mp_type (p)) {
      mp_bad_binary (mp, p, (quarterword) c);
      goto DONE;
    } else if (mp->cur_exp.type == mp_string_type) {
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      set_number_from_scaled (new_expr.data.n, mp_str_vs_str (mp, value_str (p), cur_exp_str ()));
      mp_flush_cur_exp (mp, new_expr);
    } else if ((mp->cur_exp.type == mp_unknown_string) ||
               (mp->cur_exp.type == mp_unknown_boolean)) {
      /* Check if unknowns have been equated */
      /* When two unknown strings are in the same ring, we know that they are
         equal. Otherwise, we don't know whether they are equal or not, so we
         make no change. */
      q = value_node (cur_exp_node ());
      while ((q != cur_exp_node ()) && (q != p))
        q = value_node (q);
      if (q == p) {
        memset(&new_expr,0,sizeof(mp_value));
        new_number(new_expr.data.n);
        set_cur_exp_node (NULL);
        mp_flush_cur_exp (mp, new_expr);
      }
    
    } else if ((mp->cur_exp.type <= mp_pair_type)
               && (mp->cur_exp.type >= mp_transform_type)) {
      /* Reduce comparison of big nodes to comparison of scalars */
      /* In the following, the |while| loops exist just so that |break| can be used,
         each loop runs exactly once. */
      quarterword part_type;
      q = value_node (p);
      r = value_node (cur_exp_node ());
      part_type = 0;
      switch (mp->cur_exp.type) {
      case mp_pair_type:
        while (part_type==0) {
          rr = x_part (r);
          part_type = mp_x_part;
          mp_add_or_subtract (mp, x_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || ! number_zero(value_number (rr)))
            break;
          rr = y_part (r);
          part_type = mp_y_part;
          mp_add_or_subtract (mp, y_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
        }
        mp_take_part (mp, part_type);
        break;
      case mp_color_type:
        while (part_type==0) {
          rr = red_part (r);
          part_type = mp_red_part;
          mp_add_or_subtract (mp, red_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || ! number_zero(value_number (rr)))
            break;
          rr = green_part (r);
          part_type = mp_green_part;
          mp_add_or_subtract (mp, green_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = blue_part (r);
          part_type = mp_blue_part;
          mp_add_or_subtract (mp, blue_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
        }
        mp_take_part (mp, part_type);
        break;
      case mp_cmykcolor_type:
        while (part_type==0) {
          rr = cyan_part (r);
          part_type = mp_cyan_part;
          mp_add_or_subtract (mp, cyan_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = magenta_part (r);
          part_type = mp_magenta_part;
          mp_add_or_subtract (mp, magenta_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = yellow_part (r);
          part_type = mp_yellow_part;
          mp_add_or_subtract (mp, yellow_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = black_part (r);
          part_type = mp_black_part;
          mp_add_or_subtract (mp, black_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
        }
        mp_take_part (mp, part_type);
        break;
      case mp_transform_type:
        while (part_type==0) {
          rr = tx_part (r);
          part_type = mp_x_part;
          mp_add_or_subtract (mp, tx_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = ty_part (r);
          part_type = mp_y_part;
          mp_add_or_subtract (mp, ty_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = xx_part (r);
          part_type = mp_xx_part;
          mp_add_or_subtract (mp, xx_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = xy_part (r);
          part_type = mp_xy_part;
          mp_add_or_subtract (mp, xy_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = yx_part (r);
          part_type = mp_yx_part;
          mp_add_or_subtract (mp, yx_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
          rr = yy_part (r);
          part_type = mp_yy_part;
          mp_add_or_subtract (mp, yy_part (q), rr, mp_minus);
          if (mp_type (rr) != mp_known || !number_zero(value_number (rr)))
            break;
        }
        mp_take_part (mp, part_type);
        break;
      default:
        assert (0);                 /* todo: |mp->cur_exp.type>mp_transform_node_type| ? */
        break;
      }
    
    } else if (mp->cur_exp.type == mp_boolean_type) {
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      set_number_from_boolean (new_expr.data.n, number_to_scaled(cur_exp_value_number ()) - 
                                                number_to_scaled (value_number (p)));
      mp_flush_cur_exp (mp, new_expr);
    } else {
      mp_bad_binary (mp, p, (quarterword) c);
      goto DONE;
    }
    /* Compare the current expression with zero */
    if (mp->cur_exp.type != mp_known) {
      const char *hlp[] = {
          "Oh dear. I can\'t decide if the expression above is positive,",
          "negative, or zero. So this comparison test won't be `true'.",
          NULL  };
      if (mp->cur_exp.type < mp_known) {
        mp_disp_err (mp, p);
        hlp[0]  = "The quantities shown above have not been equated.";
        hlp[1]  = NULL;
      }
      mp_disp_err(mp, NULL);
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      set_number_from_boolean (new_expr.data.n, mp_false_code);
      mp_back_error (mp,"Unknown relation will be considered false", hlp, true);
    @.Unknown relation...@>;
      mp_get_x_next (mp);
      mp_flush_cur_exp (mp, new_expr);
    } else {
      switch (c) {
      case mp_less_than:
        boolean_reset (number_negative(cur_exp_value_number ()));
        break;
      case mp_less_or_equal:
        boolean_reset (number_nonpositive(cur_exp_value_number ()));
        break;
      case mp_greater_than:
        boolean_reset (number_positive(cur_exp_value_number ()));
        break;
      case mp_greater_or_equal:
        boolean_reset (number_nonnegative(cur_exp_value_number ()));
        break;
      case mp_equal_to:
        boolean_reset (number_zero(cur_exp_value_number ()));
        break;
      case mp_unequal_to:
        boolean_reset (number_nonzero(cur_exp_value_number ()));
        break;
      };                            /* there are no other cases */
    }
    mp->cur_exp.type = mp_boolean_type;
  DONE:
    mp->arith_error = false;        /* ignore overflow in comparisons */
    break;
  case mp_and_op:
  case mp_or_op:
    /* Here we use the sneaky fact that |and_op-false_code=or_op-true_code| */
    if ((mp_type (p) != mp_boolean_type) || (mp->cur_exp.type != mp_boolean_type))
      mp_bad_binary (mp, p, (quarterword) c);
    else if (number_to_boolean (p->data.n) == c + mp_false_code - mp_and_op) {
      set_cur_exp_value_boolean (number_to_boolean (p->data.n));
    }
    break;
  case mp_times:
    if ((mp->cur_exp.type < mp_color_type) || (mp_type (p) < mp_color_type)) {
      mp_bad_binary (mp, p, mp_times);
    } else if ((mp->cur_exp.type == mp_known) || (mp_type (p) == mp_known)) {
      /* Multiply when at least one operand is known */
      mp_number vv;
      new_fraction (vv);
      if (mp_type (p) == mp_known) {
        number_clone(vv, value_number (p));
        mp_free_value_node (mp, p);
      } else {
        number_clone(vv, cur_exp_value_number ());
        mp_unstash_cur_exp (mp, p);
      }
      if (mp->cur_exp.type == mp_known) {
        mp_number ret;
        new_number (ret);
        take_scaled (ret, cur_exp_value_number (), vv);
        set_cur_exp_value_number (ret);
        free_number (ret);
      } else if (mp->cur_exp.type == mp_pair_type) {
        mp_dep_mult (mp, (mp_value_node) x_part (value_node (cur_exp_node ())), vv, true);
        mp_dep_mult (mp, (mp_value_node) y_part (value_node (cur_exp_node ())), vv, true);
      } else if (mp->cur_exp.type == mp_color_type) {
        mp_dep_mult (mp, (mp_value_node) red_part (value_node (cur_exp_node ())), vv, true);
        mp_dep_mult (mp, (mp_value_node) green_part (value_node (cur_exp_node ())), vv, true);
        mp_dep_mult (mp, (mp_value_node) blue_part (value_node (cur_exp_node ())), vv, true);
      } else if (mp->cur_exp.type == mp_cmykcolor_type) {
        mp_dep_mult (mp, (mp_value_node) cyan_part (value_node (cur_exp_node ())), vv, true);
        mp_dep_mult (mp, (mp_value_node) magenta_part (value_node (cur_exp_node ())), vv, true);
        mp_dep_mult (mp, (mp_value_node) yellow_part (value_node (cur_exp_node ())), vv, true);
        mp_dep_mult (mp, (mp_value_node) black_part (value_node (cur_exp_node ())),  vv, true);
      } else {
        mp_dep_mult (mp, NULL, vv, true);
      }
      free_number (vv);
      binary_return;
    
    } else if ((mp_nice_color_or_pair (mp, p, mp_type (p))
                && (mp->cur_exp.type > mp_pair_type))
               || (mp_nice_color_or_pair (mp, cur_exp_node (), mp->cur_exp.type)
                   && (mp_type (p) > mp_pair_type))) {
      mp_hard_times (mp, p);
      binary_return;
    } else {
      mp_bad_binary (mp, p, mp_times);
    }
    break;
  case mp_over:
    if ((mp->cur_exp.type != mp_known) || (mp_type (p) < mp_color_type)) {
      mp_bad_binary (mp, p, mp_over);
    } else {
      mp_number v_n;
      new_number (v_n); 
      number_clone (v_n, cur_exp_value_number ());
      mp_unstash_cur_exp (mp, p);
      if (number_zero(v_n)) {
        /* Squeal about division by zero */
        const char *hlp[] = {
             "You're trying to divide the quantity shown above the error",
             "message by zero. I'm going to divide it by one instead.",
             NULL };
        mp_disp_err(mp, NULL);
        mp_back_error (mp, "Division by zero", hlp, true);
        mp_get_x_next (mp);
    
      } else {
        if (mp->cur_exp.type == mp_known) {
          mp_number ret;
          new_number (ret);
          make_scaled (ret, cur_exp_value_number (), v_n);
          set_cur_exp_value_number (ret);
          free_number (ret);
        } else if (mp->cur_exp.type == mp_pair_type) {
          mp_dep_div (mp, (mp_value_node) x_part (value_node (cur_exp_node ())),
                      v_n);
          mp_dep_div (mp, (mp_value_node) y_part (value_node (cur_exp_node ())),
                      v_n);
        } else if (mp->cur_exp.type == mp_color_type) {
          mp_dep_div (mp,
                      (mp_value_node) red_part (value_node (cur_exp_node ())),
                      v_n);
          mp_dep_div (mp,
                      (mp_value_node) green_part (value_node (cur_exp_node ())),
                      v_n);
          mp_dep_div (mp,
                      (mp_value_node) blue_part (value_node (cur_exp_node ())),
                      v_n);
        } else if (mp->cur_exp.type == mp_cmykcolor_type) {
          mp_dep_div (mp,
                      (mp_value_node) cyan_part (value_node (cur_exp_node ())),
                      v_n);
          mp_dep_div (mp, (mp_value_node)
                      magenta_part (value_node (cur_exp_node ())), v_n);
          mp_dep_div (mp, (mp_value_node)
                      yellow_part (value_node (cur_exp_node ())), v_n);
          mp_dep_div (mp,
                      (mp_value_node) black_part (value_node (cur_exp_node ())),
                      v_n);
        } else {
          mp_dep_div (mp, NULL, v_n);
        }
      }
      free_number(v_n);
      binary_return;
    }
    break;
  case mp_pythag_add:
  case mp_pythag_sub:
    if ((mp->cur_exp.type == mp_known) && (mp_type (p) == mp_known)) {
      mp_number r;
      new_number (r);
      if (c == mp_pythag_add) {
        pyth_add (r, value_number (p), cur_exp_value_number ());
      } else {
        pyth_sub (r, value_number (p), cur_exp_value_number ());
      }
      set_cur_exp_value_number (r);
      free_number (r);
    } else
      mp_bad_binary (mp, p, (quarterword) c);
    break;
  case mp_rotated_by:
  case mp_slanted_by:
  case mp_scaled_by:
  case mp_shifted_by:
  case mp_transformed_by:
  case mp_x_scaled:
  case mp_y_scaled:
  case mp_z_scaled:
    /* The next few sections of the program deal with affine transformations
    of coordinate data. */
    if (mp_type (p) == mp_path_type) {
      path_trans ((quarterword) c, p);
      binary_return;
    } else if (mp_type (p) == mp_pen_type) {
      pen_trans ((quarterword) c, p);
      set_cur_exp_knot (mp_convex_hull (mp, cur_exp_knot ()));
      /* rounding error could destroy convexity */
      binary_return;
    } else if ((mp_type (p) == mp_pair_type) || (mp_type (p) == mp_transform_type)) {
      mp_big_trans (mp, p, (quarterword) c);
    } else if (mp_type (p) == mp_picture_type) {
      mp_do_edges_trans (mp, p, (quarterword) c);
      binary_return;
    } else {
      mp_bad_binary (mp, p, (quarterword) c);
    }
    break;
  case mp_concatenate:
    if ((mp->cur_exp.type == mp_string_type) && (mp_type (p) == mp_string_type)) {
      mp_string str = mp_cat (mp, value_str (p), cur_exp_str());
      delete_str_ref (cur_exp_str ()) ;
      set_cur_exp_str (str);
    } else
      mp_bad_binary (mp, p, mp_concatenate);
    break;
  case mp_substring_of:
    if (mp_nice_pair (mp, p, mp_type (p)) && (mp->cur_exp.type == mp_string_type)) {
      mp_string str = mp_chop_string (mp, 
                          cur_exp_str (),
                          round_unscaled (value_number (x_part (value_node(p)))), 
                          round_unscaled (value_number (y_part (value_node(p)))));
      delete_str_ref (cur_exp_str ()) ;
      set_cur_exp_str (str);
    } else
      mp_bad_binary (mp, p, mp_substring_of);
    break;
  case mp_subpath_of:
    if (mp->cur_exp.type == mp_pair_type)
      mp_pair_to_path (mp);
    if (mp_nice_pair (mp, p, mp_type (p)) && (mp->cur_exp.type == mp_path_type))
      mp_chop_path (mp, value_node (p));
    else
      mp_bad_binary (mp, p, mp_subpath_of);
    break;
  case mp_point_of:
  case mp_precontrol_of:
  case mp_postcontrol_of:
    if (mp->cur_exp.type == mp_pair_type)
      mp_pair_to_path (mp);
    if ((mp->cur_exp.type == mp_path_type) && (mp_type (p) == mp_known))
      mp_find_point (mp, value_number (p), (quarterword) c);
    else
      mp_bad_binary (mp, p, (quarterword) c);
    break;
  case mp_pen_offset_of:
    if ((mp->cur_exp.type == mp_pen_type) && mp_nice_pair (mp, p, mp_type (p)))
      mp_set_up_offset (mp, value_node (p));
    else
      mp_bad_binary (mp, p, mp_pen_offset_of);
    break;
  case mp_direction_time_of:
    if (mp->cur_exp.type == mp_pair_type)
      mp_pair_to_path (mp);
    if ((mp->cur_exp.type == mp_path_type) && mp_nice_pair (mp, p, mp_type (p)))
      mp_set_up_direction_time (mp, value_node (p));
    else
      mp_bad_binary (mp, p, mp_direction_time_of);
    break;
  case mp_envelope_of:
    if ((mp_type (p) != mp_pen_type) || (mp->cur_exp.type != mp_path_type))
      mp_bad_binary (mp, p, mp_envelope_of);
    else
      mp_set_up_envelope (mp, p);
    break;
  case mp_glyph_infont:
    if ((mp_type (p) != mp_string_type &&
         mp_type (p) != mp_known) || (mp->cur_exp.type != mp_string_type))
      mp_bad_binary (mp, p, mp_glyph_infont);
    else
      mp_set_up_glyph_infont (mp, p);
    break;
  case mp_arc_time_of:
    if (mp->cur_exp.type == mp_pair_type)
      mp_pair_to_path (mp);
    if ((mp->cur_exp.type == mp_path_type) && (mp_type (p) == mp_known)) {
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      mp_get_arc_time (mp, &new_expr.data.n, cur_exp_knot (), value_number (p));
      mp_flush_cur_exp (mp, new_expr);
    } else {
      mp_bad_binary (mp, p, (quarterword) c);
    }
    break;
  case mp_intersect:
    if (mp_type (p) == mp_pair_type) {
      q = mp_stash_cur_exp (mp);
      mp_unstash_cur_exp (mp, p);
      mp_pair_to_path (mp);
      p = mp_stash_cur_exp (mp);
      mp_unstash_cur_exp (mp, q);
    }
    if (mp->cur_exp.type == mp_pair_type)
      mp_pair_to_path (mp);
    if ((mp->cur_exp.type == mp_path_type) && (mp_type (p) == mp_path_type)) {
      mp_number arg1, arg2;
      new_number (arg1);
      new_number (arg2);
      mp_path_intersection (mp, value_knot (p), cur_exp_knot ());
      number_clone (arg1, mp->cur_t);
      number_clone (arg2, mp->cur_tt);
      mp_pair_value (mp, arg1, arg2);
      free_number (arg1);
      free_number (arg2);
    } else {
      mp_bad_binary (mp, p, mp_intersect);
    }
    break;
  case mp_in_font:
    if ((mp->cur_exp.type != mp_string_type) || mp_type (p) != mp_string_type) {
      mp_bad_binary (mp, p, mp_in_font);
    } else {
      mp_do_infont (mp, p);
      binary_return;
    }
    break;
  }                            /* there are no other cases */
  mp_recycle_value (mp, p);
  mp_free_value_node (mp, p);        /* |return| to avoid this */
  mp_finish_binary (mp, old_p, old_exp);
}


@ @<Declare binary action...@>=
static void mp_bad_binary (MP mp, mp_node p, quarterword c) {
  char msg[256];
  mp_string sname;
  int old_setting = mp->selector;
  const char *hlp[] = {
         "I'm afraid I don't know how to apply that operation to that",
         "combination of types. Continue, and I'll return the second",
         "argument (see above) as the result of the operation.",
         NULL };
  mp->selector = new_string;
  if (c >= mp_min_of)
    mp_print_op (mp, c);
  mp_print_known_or_unknown_type (mp, mp_type (p), p);
  if (c >= mp_min_of)
    mp_print (mp, "of");
  else
    mp_print_op (mp, c);
  mp_print_known_or_unknown_type (mp, mp->cur_exp.type, cur_exp_node ());
  sname = mp_make_string(mp);
  mp->selector = old_setting;
  mp_snprintf (msg, 256, "Not implemented: %s", mp_str(mp, sname));
@.Not implemented...@>;
  delete_str_ref(sname);
  mp_disp_err (mp, p);
  mp_disp_err (mp, NULL);
  mp_back_error (mp, msg, hlp, true);
  mp_get_x_next (mp);
}
static void mp_bad_envelope_pen (MP mp) {
  const char *hlp[] = {
         "I'm afraid I don't know how to apply that operation to that",
         "combination of types. Continue, and I'll return the second",
         "argument (see above) as the result of the operation.",
         NULL };
  mp_disp_err (mp, NULL);
  mp_disp_err (mp, NULL);
  mp_back_error (mp, "Not implemented: envelope(elliptical pen)of(path)", hlp, true);
@.Not implemented...@>;
  mp_get_x_next (mp);
}

@ @<Declare binary action...@>=
static mp_node mp_tarnished (MP mp, mp_node p) {
  mp_node q;    /* beginning of the big node */
  mp_node r;    /* moving value node pointer */
  (void) mp;
  q = value_node (p);
  switch (mp_type (p)) {
  case mp_pair_type:
    r = x_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = y_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    break;
  case mp_color_type:
    r = red_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = green_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = blue_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    break;
  case mp_cmykcolor_type:
    r = cyan_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = magenta_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = yellow_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = black_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    break;
  case mp_transform_type:
    r = tx_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = ty_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = xx_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = xy_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = yx_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    r = yy_part (q);
    if (mp_type (r) == mp_independent)
      return MP_VOID;
    break;
  default:                     /* there are no other valid cases, but please the compiler */
    break;
  }
  return NULL;
}

@ The first argument to |add_or_subtract| is the location of a value node
in a capsule or pair node that will soon be recycled. The second argument
is either a location within a pair or transform node of |cur_exp|,
or it is NULL (which means that |cur_exp| itself should be the second
argument).  The third argument is either |plus| or |minus|.

The sum or difference of the numeric quantities will replace the second
operand.  Arithmetic overflow may go undetected; users aren't supposed to
be monkeying around with really big values.
@^overflow in arithmetic@>

@<Declare binary action...@>=
@<Declare the procedure called |dep_finish|@>;
static void mp_add_or_subtract (MP mp, mp_node p, mp_node q, quarterword c) {
  mp_variable_type s, t;        /* operand types */
  mp_value_node r;      /* dependency list traverser */
  mp_value_node v = NULL;       /* second operand value for dep lists */
  mp_number vv;       /* second operand value for known values */
  new_number (vv);
  if (q == NULL) {
    t = mp->cur_exp.type;
    if (t < mp_dependent)
      number_clone (vv, cur_exp_value_number ());
    else
      v = (mp_value_node) dep_list ((mp_value_node) cur_exp_node ());
  } else {
    t = mp_type (q);
    if (t < mp_dependent)
      number_clone (vv, value_number (q));
    else
      v = (mp_value_node) dep_list ((mp_value_node) q);
  }
  if (t == mp_known) {
    mp_value_node qq = (mp_value_node) q;
    if (c == mp_minus)
      number_negate (vv);
    if (mp_type (p) == mp_known) {
      slow_add (vv, value_number (p), vv);
      if (q == NULL)
        set_cur_exp_value_number (vv);
      else
        set_value_number (q, vv);
      free_number (vv);
      return;
    }
    /* Add a known value to the constant term of |dep_list(p)| */
    r = (mp_value_node) dep_list ((mp_value_node) p);
    while (dep_info (r) != NULL)
      r = (mp_value_node) mp_link (r);
    slow_add (vv, dep_value (r), vv);
    set_dep_value (r, vv);
    if (qq == NULL) {
      qq = mp_get_dep_node (mp);
      set_cur_exp_node ((mp_node) qq);
      mp->cur_exp.type = mp_type (p);
      mp_name_type (qq) = mp_capsule;
      /* clang: never read: |q = (mp_node) qq;| */
    }
    set_dep_list (qq, dep_list ((mp_value_node) p));
    mp_type (qq) = mp_type (p);
    set_prev_dep (qq, prev_dep ((mp_value_node) p));
    mp_link (prev_dep ((mp_value_node) p)) = (mp_node) qq;
    mp_type (p) = mp_known;     /* this will keep the recycler from collecting non-garbage */
  } else {
    if (c == mp_minus)
      mp_negate_dep_list (mp, v);
    /* Add operand |p| to the dependency list |v| */
    /* We prefer |dependent| lists to |mp_proto_dependent| ones, because it is
       nice to retain the extra accuracy of |fraction| coefficients.
       But we have to handle both kinds, and mixtures too. */
    if (mp_type (p) == mp_known) {
      /* Add the known |value(p)| to the constant term of |v| */
      while (dep_info (v) != NULL) {
        v = (mp_value_node) mp_link (v);
      }
      slow_add (vv, value_number (p), dep_value (v));
      set_dep_value (v, vv);
    } else {
      s = mp_type (p);
      r = (mp_value_node) dep_list ((mp_value_node) p);
      if (t == mp_dependent) {
        if (s == mp_dependent) {
          mp_number ret1, ret2;
          new_fraction (ret1);
          new_fraction (ret2);
          mp_max_coef (mp, &ret1, r);
          mp_max_coef (mp, &ret2, v);
          number_add (ret1, ret2);
          free_number (ret2);
          if (number_less (ret1, coef_bound_k)) {
            v = mp_p_plus_q (mp, v, r, mp_dependent);
            free_number (ret1);
            goto DONE;
          }
          free_number (ret1);
        }                           /* |fix_needed| will necessarily be false */
        t = mp_proto_dependent;
        v = mp_p_over_v (mp, v, unity_t, mp_dependent, mp_proto_dependent);
      }
      if (s == mp_proto_dependent)
        v = mp_p_plus_q (mp, v, r, mp_proto_dependent);
      else
        v = mp_p_plus_fq (mp, v, unity_t, r, mp_proto_dependent, mp_dependent);
    DONE:
      /* Output the answer, |v| (which might have become |known|) */
      if (q != NULL) {
        mp_dep_finish (mp, v, (mp_value_node) q, t);
      } else {
        mp->cur_exp.type = t;
        mp_dep_finish (mp, v, NULL, t);
      }
    }
  }
  free_number (vv);
}


@ Here's the current situation: The dependency list |v| of type |t|
should either be put into the current expression (if |q=NULL|) or
into location |q| within a pair node (otherwise). The destination (|cur_exp|
or |q|) formerly held a dependency list with the same
final pointer as the list |v|.

@<Declare the procedure called |dep_finish|@>=
static void mp_dep_finish (MP mp, mp_value_node v, mp_value_node q,
                           quarterword t) {
  mp_value_node p;      /* the destination */
  if (q == NULL)
    p = (mp_value_node) cur_exp_node ();
  else
    p = q;
  set_dep_list (p, v);
  mp_type (p) = t;
  if (dep_info (v) == NULL) {
    mp_number vv;    /* the value, if it is |known| */
    new_number (vv);
    number_clone (vv, value_number (v));
    if (q == NULL) {
      mp_value new_expr;
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      number_clone (new_expr.data.n, vv);
      mp_flush_cur_exp (mp, new_expr);
    } else {
      mp_recycle_value (mp, (mp_node) p);
      mp_type (q) = mp_known;
      set_value_number (q, vv);
    }
    free_number (vv);
  } else if (q == NULL) {
    mp->cur_exp.type = t;
  }
  if (mp->fix_needed)
    mp_fix_dependencies (mp);
}

@ @<Declare binary action...@>=
static void mp_dep_mult (MP mp, mp_value_node p, mp_number v, boolean v_is_scaled) {
  mp_value_node q;      /* the dependency list being multiplied by |v| */
  quarterword s, t;     /* its type, before and after */
  if (p == NULL) {
    q = (mp_value_node) cur_exp_node ();
  } else if (mp_type (p) != mp_known) {
    q = p;
  } else {
    {
      mp_number r1, arg1;
      new_number (arg1);
      number_clone (arg1, dep_value (p));
      if (v_is_scaled) {
        new_number (r1);
        take_scaled (r1, arg1, v);
      } else {
        new_fraction (r1);
        take_fraction (r1, arg1, v);
      }
      set_dep_value (p, r1);
      free_number (r1);
      free_number (arg1);
    }
    return;
  }
  t = mp_type (q);
  q = (mp_value_node) dep_list (q);
  s = t;
  if (t == mp_dependent) {
    if (v_is_scaled) {
      mp_number ab_vs_cd;
      mp_number arg1, arg2;
      new_number (ab_vs_cd);
      new_number (arg2);
      new_fraction (arg1);
      mp_max_coef (mp, &arg1, q);
      number_clone (arg2, v);
      number_abs (arg2);
      ab_vs_cd (ab_vs_cd, arg1, arg2, coef_bound_minus_1, unity_t);
      free_number (arg1);
      free_number (arg2);
      if (number_nonnegative(ab_vs_cd)) {
        t = mp_proto_dependent;
      }
      free_number (ab_vs_cd);
    }
  }
  q = mp_p_times_v (mp, q, v, s, t, v_is_scaled);
  mp_dep_finish (mp, q, p, t);
}


@ Here is a routine that is similar to |times|; but it is invoked only
internally, when |v| is a |fraction| whose magnitude is at most~1,
and when |cur_type>=mp_color_type|.

@c
static void mp_frac_mult (MP mp, mp_number n, mp_number d) {
  /* multiplies |cur_exp| by |n/d| */
  mp_node old_exp;      /* a capsule to recycle */
  mp_number v;   /* |n/d| */
  new_fraction (v);
  if (number_greater (internal_value (mp_tracing_commands), two_t)) {
    @<Trace the fraction multiplication@>;
  }
  switch (mp->cur_exp.type) {
  case mp_transform_type:
  case mp_color_type:
  case mp_cmykcolor_type:
  case mp_pair_type:
    old_exp = mp_tarnished (mp, cur_exp_node ());
    break;
  case mp_independent:
    old_exp = MP_VOID;
    break;
  default:
    old_exp = NULL;
    break;
  }
  if (old_exp != NULL) {
    old_exp = cur_exp_node ();
    mp_make_exp_copy (mp, old_exp);
  }
  make_fraction (v, n, d);
  if (mp->cur_exp.type == mp_known) {
    mp_number r1, arg1;
    new_fraction (r1);
    new_number (arg1);
    number_clone (arg1, cur_exp_value_number ());
    take_fraction (r1, arg1, v);
    set_cur_exp_value_number (r1);
    free_number (r1);
    free_number (arg1);
  } else if (mp->cur_exp.type == mp_pair_type) {
    mp_dep_mult (mp, (mp_value_node) x_part (value_node (cur_exp_node ())), v, false);
    mp_dep_mult (mp, (mp_value_node) y_part (value_node (cur_exp_node ())), v, false);
  } else if (mp->cur_exp.type == mp_color_type) {
    mp_dep_mult (mp, (mp_value_node) red_part (value_node (cur_exp_node ())), v, false);
    mp_dep_mult (mp, (mp_value_node) green_part (value_node (cur_exp_node ())), v, false);
    mp_dep_mult (mp, (mp_value_node) blue_part (value_node (cur_exp_node ())), v, false);
  } else if (mp->cur_exp.type == mp_cmykcolor_type) {
    mp_dep_mult (mp, (mp_value_node) cyan_part (value_node (cur_exp_node ())), v, false);
    mp_dep_mult (mp, (mp_value_node) magenta_part (value_node (cur_exp_node ())), v, false);
    mp_dep_mult (mp, (mp_value_node) yellow_part (value_node (cur_exp_node ())), v, false);
    mp_dep_mult (mp, (mp_value_node) black_part (value_node (cur_exp_node ())), v, false);
  } else {
    mp_dep_mult (mp, NULL, v, false);
  }
  if (old_exp != NULL) {
    mp_recycle_value (mp, old_exp);
    mp_free_value_node (mp, old_exp);
  }
  free_number (v);
}


@ @<Trace the fraction multiplication@>=
{
  mp_begin_diagnostic (mp);
  mp_print_nl (mp, "{(");
  print_number (n);
  mp_print_char (mp, xord ('/'));
  print_number (d);
  mp_print (mp, ")*(");
  mp_print_exp (mp, NULL, 0);
  mp_print (mp, ")}");
  mp_end_diagnostic (mp, false);
}


@ The |hard_times| routine multiplies a nice color or pair by a dependency list.

@<Declare binary action procedures@>=
static void mp_hard_times (MP mp, mp_node p) {
  mp_value_node q;      /* a copy of the dependent variable |p| */
  mp_value_node pp;     /* for typecasting p */
  mp_node r;    /* a component of the big node for the nice color or pair */
  mp_number v;     /* the known value for |r| */
  new_number (v);
  if (mp_type (p) <= mp_pair_type) {
    q = (mp_value_node) mp_stash_cur_exp (mp);
    mp_unstash_cur_exp (mp, p);
    p = (mp_node) q;
  }                             /* now |cur_type=mp_pair_type| or |cur_type=mp_color_type| or |cur_type=mp_cmykcolor_type| */
  pp = (mp_value_node) p;
  if (mp->cur_exp.type == mp_pair_type) {
    r = x_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
    r = y_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
  } else if (mp->cur_exp.type == mp_color_type) {
    r = red_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
    r = green_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
    r = blue_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
  } else if (mp->cur_exp.type == mp_cmykcolor_type) {
    r = cyan_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
    r = yellow_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
    r = magenta_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
    r = black_part (value_node (cur_exp_node ()));
    number_clone(v, value_number (r));
    mp_new_dep (mp, r, mp_type (pp),
                mp_copy_dep_list (mp, (mp_value_node) dep_list (pp)));
    mp_dep_mult (mp, (mp_value_node) r, v, true);
  }
  free_number (v);
}

@ @<Declare binary action...@>=
static void mp_dep_div (MP mp, mp_value_node p, mp_number v) {
  mp_value_node q;      /* the dependency list being divided by |v| */
  quarterword s, t;     /* its type, before and after */
  if (p == NULL)
    q = (mp_value_node) cur_exp_node ();
  else if (mp_type (p) != mp_known)
    q = p;
  else {
    mp_number ret;
    new_number (ret);
    make_scaled (ret, value_number (p), v);
    set_value_number (p, ret);
    free_number (ret);
    return;
  }
  t = mp_type (q);
  q = (mp_value_node) dep_list (q);
  s = t;
  if (t == mp_dependent) {
      mp_number ab_vs_cd;
      mp_number arg1, arg2;
      new_number (ab_vs_cd);
      new_number (arg2);
      new_fraction (arg1);
      mp_max_coef (mp, &arg1, q);
      number_clone (arg2, v);
      number_abs (arg2);
      ab_vs_cd (ab_vs_cd, arg1, unity_t, coef_bound_minus_1, arg2);
      free_number (arg1);
      free_number (arg2);
      if (number_nonnegative(ab_vs_cd)) {
        t = mp_proto_dependent;
      }
      free_number (ab_vs_cd);
  }
  q = mp_p_over_v (mp, q, v, s, t);
  mp_dep_finish (mp, q, p, t);
}

@ Let |c| be one of the eight transform operators. The procedure call
|set_up_trans(c)| first changes |cur_exp| to a transform that corresponds to
|c| and the original value of |cur_exp|. (In particular, |cur_exp| doesn't
change at all if |c=transformed_by|.)

Then, if all components of the resulting transform are |known|, they are
moved to the global variables |txx|, |txy|, |tyx|, |tyy|, |tx|, |ty|;
and |cur_exp| is changed to the known value zero.

@<Declare binary action...@>=
static void mp_set_up_trans (MP mp, quarterword c) {
  mp_node p, q, r;      /* list manipulation registers */
  mp_value new_expr;
  memset(&new_expr,0,sizeof(mp_value));
  if ((c != mp_transformed_by) || (mp->cur_exp.type != mp_transform_type)) {
    /* Put the current transform into |cur_exp| */
    const char *hlp[] = {
           "The expression shown above has the wrong type,",
           "so I can\'t transform anything using it.",
           "Proceed, and I'll omit the transformation.",
           NULL };
    p = mp_stash_cur_exp (mp);
    set_cur_exp_node (mp_id_transform (mp));
    mp->cur_exp.type = mp_transform_type;
    q = value_node (cur_exp_node ());
    switch (c) {
      @<For each of the eight cases, change the relevant fields of |cur_exp|
      and |goto done|;
      but do nothing if capsule |p| doesn't have the appropriate type@>;
    };                            /* there are no other cases */
    mp_disp_err (mp, p);
    mp_back_error (mp, "Improper transformation argument", hlp, true);
    mp_get_x_next (mp);
  DONE:
    mp_recycle_value (mp, p);
    mp_free_value_node (mp, p);

  }
  /* If the current transform is entirely known, stash it in global variables;
    otherwise |return| */
  q = value_node (cur_exp_node ());
  if (mp_type (tx_part (q)) != mp_known)
    return;
  if (mp_type (ty_part (q)) != mp_known)
    return;
  if (mp_type (xx_part (q)) != mp_known)
    return;
  if (mp_type (xy_part (q)) != mp_known)
    return;
  if (mp_type (yx_part (q)) != mp_known)
    return;
  if (mp_type (yy_part (q)) != mp_known)
    return;
  number_clone(mp->txx, value_number (xx_part (q)));
  number_clone(mp->txy, value_number (xy_part (q)));
  number_clone(mp->tyx, value_number (yx_part (q)));
  number_clone(mp->tyy, value_number (yy_part (q)));
  number_clone(mp->tx, value_number (tx_part (q)));
  number_clone(mp->ty, value_number (ty_part (q)));
  new_number(new_expr.data.n);
  set_number_to_zero (new_expr.data.n);
  mp_flush_cur_exp (mp, new_expr);
}


@ @<Glob...@>=
mp_number txx;
mp_number txy;
mp_number tyx;
mp_number tyy;
mp_number tx;
mp_number ty;      /* current transform coefficients */

@ @<Initialize table...@>=
new_number(mp->txx);
new_number(mp->txy);
new_number(mp->tyx);
new_number(mp->tyy);
new_number(mp->tx);
new_number(mp->ty);

@ @<Free table...@>=
free_number(mp->txx);
free_number(mp->txy);
free_number(mp->tyx);
free_number(mp->tyy);
free_number(mp->tx);
free_number(mp->ty);


@ @<For each of the eight cases...@>=
case mp_rotated_by:
if (mp_type (p) == mp_known)
  @<Install sines and cosines, then |goto done|@>;
break;
case mp_slanted_by:
if (mp_type (p) > mp_pair_type) {
  mp_install (mp, xy_part (q), p);
  goto DONE;
}
break;
case mp_scaled_by:
if (mp_type (p) > mp_pair_type) {
  mp_install (mp, xx_part (q), p);
  mp_install (mp, yy_part (q), p);
  goto DONE;
}
break;
case mp_shifted_by:
if (mp_type (p) == mp_pair_type) {
  r = value_node (p);
  mp_install (mp, tx_part (q), x_part (r));
  mp_install (mp, ty_part (q), y_part (r));
  goto DONE;
}
break;
case mp_x_scaled:
if (mp_type (p) > mp_pair_type) {
  mp_install (mp, xx_part (q), p);
  goto DONE;
}
break;
case mp_y_scaled:
if (mp_type (p) > mp_pair_type) {
  mp_install (mp, yy_part (q), p);
  goto DONE;
}
break;
case mp_z_scaled:
if (mp_type (p) == mp_pair_type)
  @<Install a complex multiplier, then |goto done|@>;
break;
case mp_transformed_by:
break;


@ @<Install sines and cosines, then |goto done|@>=
{
  mp_number n_sin, n_cos, arg1, arg2;
  new_number (arg1);
  new_number (arg2);
  new_fraction (n_sin);
  new_fraction (n_cos); /* results computed by |n_sin_cos| */
  number_clone (arg2, unity_t);
  number_clone (arg1, value_number (p));
  number_multiply_int (arg2, 360);
  number_modulo (arg1, arg2);
  convert_scaled_to_angle (arg1);
  n_sin_cos (arg1, n_cos, n_sin);
  fraction_to_round_scaled (n_sin);
  fraction_to_round_scaled (n_cos);
  set_value_number (xx_part (q), n_cos);
  set_value_number (yx_part (q), n_sin);
  set_value_number (xy_part (q), value_number (yx_part (q)));
  number_negate (value_number (xy_part (q)));
  set_value_number (yy_part (q), value_number (xx_part (q)));
  free_number (arg1);
  free_number (arg2);
  free_number (n_sin);
  free_number (n_cos);
  goto DONE;
}


@ @<Install a complex multiplier, then |goto done|@>=
{
  r = value_node (p);
  mp_install (mp, xx_part (q), x_part (r));
  mp_install (mp, yy_part (q), x_part (r));
  mp_install (mp, yx_part (q), y_part (r));
  if (mp_type (y_part (r)) == mp_known) {
    set_value_number (y_part (r), value_number (y_part (r)));
    number_negate (value_number (y_part (r)));
  } else {
    mp_negate_dep_list (mp, (mp_value_node) dep_list ((mp_value_node)
                                                      y_part (r)));
  }
  mp_install (mp, xy_part (q), y_part (r));
  goto DONE;
}


@ Procedure |set_up_known_trans| is like |set_up_trans|, but it
insists that the transformation be entirely known.

@<Declare binary action...@>=
static void mp_set_up_known_trans (MP mp, quarterword c) {
  mp_set_up_trans (mp, c);
  if (mp->cur_exp.type != mp_known) {
    mp_value new_expr;
    const char *hlp[] = { 
           "I'm unable to apply a partially specified transformation",
           "except to a fully known pair or transform.",
           "Proceed, and I'll omit the transformation.",
           NULL };
    memset(&new_expr,0,sizeof(mp_value));
    new_number(new_expr.data.n);
    mp_disp_err(mp, NULL);
    set_number_to_zero (new_expr.data.n);
    mp_back_error (mp,"Transform components aren't all known", hlp, true);
    mp_get_x_next (mp);
    mp_flush_cur_exp (mp, new_expr);
    set_number_to_unity(mp->txx);
    set_number_to_zero(mp->txy);
    set_number_to_zero(mp->tyx);
    set_number_to_unity(mp->tyy);
    set_number_to_zero(mp->tx);
    set_number_to_zero(mp->ty);
  }
}


@ Here's a procedure that applies the transform |txx..ty| to a pair of
coordinates in locations |p| and~|q|.

@<Declare binary action...@>=
static void mp_number_trans (MP mp, mp_number *p, mp_number *q) {
  mp_number r1, r2, v;
  new_number (r1);
  new_number (r2);
  new_number (v);
  take_scaled (r1, *p, mp->txx);
  take_scaled (r2, *q, mp->txy);
  number_add (r1, r2);
  set_number_from_addition(v, r1, mp->tx);
  take_scaled (r1, *p, mp->tyx);
  take_scaled (r2, *q, mp->tyy);
  number_add (r1, r2);
  set_number_from_addition(*q, r1, mp->ty);
  number_clone(*p,v);
  free_number (r1);
  free_number (r2);
  free_number(v);
}


@ The simplest transformation procedure applies a transform to all
coordinates of a path.  The |path_trans(c)(p)| macro applies
a transformation defined by |cur_exp| and the transform operator |c|
to the path~|p|.

@d path_trans(A,B) { mp_set_up_known_trans(mp, (A)); 
                     mp_unstash_cur_exp(mp, (B)); 
                     mp_do_path_trans(mp, cur_exp_knot()); }

@<Declare binary action...@>=
static void mp_do_path_trans (MP mp, mp_knot p) {
  mp_knot q;    /* list traverser */
  q = p;
  do {
    if (mp_left_type (q) != mp_endpoint)
      mp_number_trans (mp, &q->left_x, &q->left_y);
    mp_number_trans (mp, &q->x_coord, &q->y_coord);
    if (mp_right_type (q) != mp_endpoint)
      mp_number_trans (mp, &q->right_x, &q->right_y);
    q = mp_next_knot (q);
  } while (q != p);
}


@ Transforming a pen is very similar, except that there are no |mp_left_type|
and |mp_right_type| fields.

@d pen_trans(A,B) { mp_set_up_known_trans(mp, (A)); 
                    mp_unstash_cur_exp(mp, (B)); 
                    mp_do_pen_trans(mp, cur_exp_knot()); }

@<Declare binary action...@>=
static void mp_do_pen_trans (MP mp, mp_knot p) {
  mp_knot q;    /* list traverser */
  if (pen_is_elliptical (p)) {
    mp_number_trans (mp, &p->left_x, &p->left_y);
    mp_number_trans (mp, &p->right_x, &p->right_y);
  }
  q = p;
  do {
    mp_number_trans (mp, &q->x_coord, &q->y_coord);
    q = mp_next_knot (q);
  } while (q != p);
}


@ The next transformation procedure applies to edge structures. It will do
any transformation, but the results may be substandard if the picture contains
text that uses downloaded bitmap fonts.  The binary action procedure is
|do_edges_trans|, but we also need a function that just scales a picture.
That routine is |scale_edges|.  Both it and the underlying routine |edges_trans|
should be thought of as procedures that update an edge structure |h|, except
that they have to return a (possibly new) structure because of the need to call
|private_edges|.

@<Declare binary action...@>=
static mp_edge_header_node mp_edges_trans (MP mp, mp_edge_header_node h) {
  mp_node q;    /* the object being transformed */
  mp_dash_node r, s; /* for list manipulation */
  mp_number sx, sy;        /* saved transformation parameters */
  mp_number sqdet; /* square root of determinant for |dash_scale| */
  mp_number sgndet;       /* sign of the determinant */
  h = mp_private_edges (mp, h);
  new_number(sx);
  new_number(sy);
  new_number(sqdet);
  new_number(sgndet);
  mp_sqrt_det (mp, &sqdet, mp->txx, mp->txy, mp->tyx, mp->tyy);
  ab_vs_cd (sgndet, mp->txx, mp->tyy, mp->txy, mp->tyx);
  if (dash_list (h) != mp->null_dash) {
    @<Try to transform the dash list of |h|@>;
  }
  @<Make the bounding box of |h| unknown if it can't be updated properly
    without scanning the whole structure@>;
  q = mp_link (edge_list (h));
  while (q != NULL) {
    @<Transform graphical object |q|@>;
    q = mp_link (q);
  }
  free_number (sx);
  free_number (sy);
  free_number (sqdet);
  free_number(sgndet);
  return h;
}
static void mp_do_edges_trans (MP mp, mp_node p, quarterword c) {
  mp_set_up_known_trans (mp, c);
  set_value_node (p, (mp_node)mp_edges_trans (mp, (mp_edge_header_node)value_node (p)));
  mp_unstash_cur_exp (mp, p);
}
static mp_edge_header_node mp_scale_edges (MP mp, mp_number se_sf, mp_edge_header_node se_pic) {
  number_clone(mp->txx, se_sf);
  number_clone(mp->tyy, se_sf);
  set_number_to_zero(mp->txy);
  set_number_to_zero(mp->tyx);
  set_number_to_zero(mp->tx);
  set_number_to_zero(mp->ty);
  return mp_edges_trans (mp, se_pic);
}


@ @<Try to transform the dash list of |h|@>=
if (number_nonzero(mp->txy) || number_nonzero(mp->tyx) ||
    number_nonzero(mp->ty) || number_nonequalabs (mp->txx, mp->tyy)) {
  mp_flush_dash_list (mp, h);
} else {
  mp_number abs_tyy, ret;
  new_number (abs_tyy);
  if (number_negative(mp->txx)) {
    @<Reverse the dash list of |h|@>;
  }
  @<Scale the dash list by |txx| and shift it by |tx|@>;
  number_clone(abs_tyy, mp->tyy);
  number_abs (abs_tyy);
  new_number (ret);
  take_scaled (ret, h->dash_y, abs_tyy);
  number_clone(h->dash_y, ret);
  free_number (ret);
  free_number (abs_tyy);
}


@ @<Reverse the dash list of |h|@>=
{
  r = dash_list (h);
  set_dash_list (h, mp->null_dash);
  while (r != mp->null_dash) {
    s = r;
    r = (mp_dash_node)mp_link (r);
    number_swap(s->start_x, s->stop_x );
    mp_link (s) = (mp_node)dash_list (h);
    set_dash_list (h, s);
  }
}


@ @<Scale the dash list by |txx| and shift it by |tx|@>=
r = dash_list (h);
{
  mp_number arg1;
  new_number (arg1);
  while (r != mp->null_dash) {
    take_scaled (arg1, r->start_x, mp->txx);
    set_number_from_addition(r->start_x, arg1, mp->tx);
    take_scaled (arg1, r->stop_x, mp->txx);
    set_number_from_addition(r->stop_x, arg1, mp->tx);
    r = (mp_dash_node)mp_link (r);
  }
  free_number (arg1);
}


@ @<Make the bounding box of |h| unknown if it can't be updated properly...@>=
if (number_zero(mp->txx) && number_zero(mp->tyy)) {
  @<Swap the $x$ and $y$ parameters in the bounding box of |h|@>;
} else if (number_nonzero(mp->txy) || number_nonzero(mp->tyx)) {
  mp_init_bbox (mp, h);
  goto DONE1;
}
if (number_lessequal (h->minx, h->maxx)) {
  @<Scale the bounding box by |txx+txy| and |tyx+tyy|; then shift by
   |(tx,ty)|@>;
}
DONE1:


@ @<Swap the $x$ and $y$ parameters in the bounding box of |h|@>=
{
  number_swap(h->minx, h->miny);
  number_swap(h->maxx, h->maxy);
}


@ The sum ``|txx+txy|'' is whichever of |txx| or |txy| is nonzero.  The other
sum is similar.

@<Scale the bounding box by |txx+txy| and |tyx+tyy|; then shift...@>=
{
  mp_number tot, ret;
  new_number(tot);
  new_number (ret);
  set_number_from_addition(tot,mp->txx,mp->txy);
  take_scaled (ret, h->minx, tot);
  set_number_from_addition(h->minx,ret, mp->tx);
  take_scaled (ret, h->maxx, tot);
  set_number_from_addition(h->maxx,ret, mp->tx);

  set_number_from_addition(tot,mp->tyx,mp->tyy);
  take_scaled (ret, h->miny, tot);
  set_number_from_addition(h->miny, ret, mp->ty);
  take_scaled (ret, h->maxy, tot);
  set_number_from_addition(h->maxy, ret, mp->ty);

  set_number_from_addition(tot, mp->txx, mp->txy);
  if (number_negative(tot)) {
    number_swap(h->minx, h->maxx);
  }
  set_number_from_addition(tot, mp->tyx, mp->tyy);
  if (number_negative(tot)) {
    number_swap(h->miny, h->maxy);
  }
  free_number (ret);  
  free_number (tot);
}


@ Now we ready for the main task of transforming the graphical objects in edge
structure~|h|.

@<Transform graphical object |q|@>=
switch (mp_type (q)) {
case mp_fill_node_type:
  {
    mp_fill_node qq = (mp_fill_node) q;
    mp_do_path_trans (mp, mp_path_p (qq));
    @<Transform |mp_pen_p(qq)|, making sure polygonal pens stay counter-clockwise@>;
  }
  break;
case mp_stroked_node_type:
  {
    mp_stroked_node qq = (mp_stroked_node) q;
    mp_do_path_trans (mp, mp_path_p (qq));
    @<Transform |mp_pen_p(qq)|, making sure polygonal pens stay counter-clockwise@>;
  }
  break;
case mp_start_clip_node_type:
  mp_do_path_trans (mp, mp_path_p ((mp_start_clip_node) q));
  break;
case mp_start_bounds_node_type:
  mp_do_path_trans (mp, mp_path_p ((mp_start_bounds_node) q));
  break;
case mp_text_node_type:
  @<Transform the compact transformation@>;
  break;
case mp_stop_clip_node_type:
case mp_stop_bounds_node_type:
  break;
default:                       /* there are no other valid cases, but please the compiler */
  break;
}


@ Note that the shift parameters |(tx,ty)| apply only to the path being stroked.
The |dash_scale| has to be adjusted  to scale the dash lengths in |mp_dash_p(q)|
since the \ps\ output procedures will try to compensate for the transformation
we are applying to |mp_pen_p(q)|.  Since this compensation is based on the square
root of the determinant, |sqdet| is the appropriate factor.

We pass the mptrap test only if |dash_scale| is not adjusted, nowadays
(backend is changed?)

@<Transform |mp_pen_p(qq)|, making sure...@>=
if (mp_pen_p (qq) != NULL) {
  number_clone(sx, mp->tx);
  number_clone(sy, mp->ty);
  set_number_to_zero(mp->tx);
  set_number_to_zero(mp->ty);
  mp_do_pen_trans (mp, mp_pen_p (qq));
  if (number_nonzero(sqdet)
      && ((mp_type (q) == mp_stroked_node_type) && (mp_dash_p (q) != NULL))) {
    mp_number ret;
    new_number (ret);
    take_scaled (ret, ((mp_stroked_node)q)->dash_scale, sqdet);
    number_clone(((mp_stroked_node)q)->dash_scale, ret);
    free_number (ret);
  }
  if (!pen_is_elliptical (mp_pen_p (qq)))
    if (number_negative(sgndet))
      mp_pen_p (qq) = mp_make_pen (mp, mp_copy_path (mp, mp_pen_p (qq)), true);
  /* this unreverses the pen */
  number_clone(mp->tx, sx);
  number_clone(mp->ty, sy);
}

@ @<Transform the compact transformation@>=
mp_number_trans (mp, &((mp_text_node)q)->tx, &((mp_text_node)q)->ty);
number_clone(sx, mp->tx);
number_clone(sy, mp->ty);
set_number_to_zero(mp->tx);
set_number_to_zero(mp->ty);
mp_number_trans (mp, &((mp_text_node)q)->txx, &((mp_text_node)q)->tyx);
mp_number_trans (mp, &((mp_text_node)q)->txy, &((mp_text_node)q)->tyy);
number_clone(mp->tx, sx);
number_clone(mp->ty, sy)

@ The hard cases of transformation occur when big nodes are involved,
and when some of their components are unknown.

@<Declare binary action...@>=
@<Declare subroutines needed by |big_trans|@>;
static void mp_big_trans (MP mp, mp_node p, quarterword c) {
  mp_node q, r, pp, qq; /* list manipulation registers */
  q = value_node (p);
  if (mp_type (q) == mp_pair_node_type) {
    if (mp_type (x_part (q)) != mp_known ||
        mp_type (y_part (q)) != mp_known) {
      @<Transform an unknown big node and |return|@>;
    }
  } else {                      /* |mp_transform_type| */
    if (mp_type (tx_part (q)) != mp_known ||
        mp_type (ty_part (q)) != mp_known ||
        mp_type (xx_part (q)) != mp_known ||
        mp_type (xy_part (q)) != mp_known ||
        mp_type (yx_part (q)) != mp_known ||
        mp_type (yy_part (q)) != mp_known) {
      @<Transform an unknown big node and |return|@>;
    }
  }
  @<Transform a known big node@>;
}                               /* node |p| will now be recycled by |do_binary| */


@ @<Transform an unknown big node and |return|@>=
{
  mp_set_up_known_trans (mp, c);
  mp_make_exp_copy (mp, p);
  r = value_node (cur_exp_node ());
  if (mp->cur_exp.type == mp_transform_type) {
    mp_bilin1 (mp, yy_part (r), mp->tyy, xy_part (q), mp->tyx, zero_t);
    mp_bilin1 (mp, yx_part (r), mp->tyy, xx_part (q), mp->tyx, zero_t);
    mp_bilin1 (mp, xy_part (r), mp->txx, yy_part (q), mp->txy, zero_t);
    mp_bilin1 (mp, xx_part (r), mp->txx, yx_part (q), mp->txy, zero_t);
  }
  mp_bilin1 (mp, y_part (r), mp->tyy, x_part (q), mp->tyx, mp->ty);
  mp_bilin1 (mp, x_part (r), mp->txx, y_part (q), mp->txy, mp->tx);
  return;
}


@ Let |p| point to a value field inside a big node of |cur_exp|,
and let |q| point to a another value field. The |bilin1| procedure
replaces |p| by $p\cdot t+q\cdot u+\delta$.

@<Declare subroutines needed by |big_trans|@>=
static void mp_bilin1 (MP mp, mp_node p, mp_number t, mp_node q,
                       mp_number u, mp_number delta_orig) {
  mp_number delta;
  new_number (delta);
  number_clone (delta, delta_orig);
  if (!number_equal(t, unity_t)) {
    mp_dep_mult (mp, (mp_value_node) p, t, true);
  }
  if (number_nonzero(u)) {
    if (mp_type (q) == mp_known) {
      mp_number tmp;
      new_number (tmp);
      take_scaled (tmp, value_number (q), u);
      number_add (delta, tmp);
      free_number (tmp);
    } else {
      /* Ensure that |type(p)=mp_proto_dependent| */
      if (mp_type (p) != mp_proto_dependent) {
        if (mp_type (p) == mp_known) {
          mp_new_dep (mp, p, mp_type (p), mp_const_dependency (mp, value_number (p)));
        } else {
          set_dep_list ((mp_value_node) p,
            mp_p_times_v (mp,
                                    (mp_value_node) dep_list ((mp_value_node)
                                                              p), unity_t,
                                    mp_dependent, mp_proto_dependent, true));
        }
        mp_type (p) = mp_proto_dependent;
      }
      set_dep_list ((mp_value_node) p,
        mp_p_plus_fq (mp,
                                (mp_value_node) dep_list ((mp_value_node) p), u,
                                (mp_value_node) dep_list ((mp_value_node) q),
                                mp_proto_dependent, mp_type (q)));
    }
  }
  if (mp_type (p) == mp_known) {
    set_value_number (p, value_number (p));
    number_add (value_number (p), delta);
  } else {
    mp_number tmp;
    mp_value_node r;    /* list traverser */
    new_number (tmp);
    r = (mp_value_node) dep_list ((mp_value_node) p);
    while (dep_info (r) != NULL)
      r = (mp_value_node) mp_link (r);
    number_clone (tmp, value_number(r));
    number_add (delta, tmp);
    if (r != (mp_value_node) dep_list ((mp_value_node) p))
      set_value_number (r, delta);
    else {
      mp_recycle_value (mp, p);
      mp_type (p) = mp_known;
      set_value_number (p, delta);
    }
    free_number (tmp);
  }
  if (mp->fix_needed)
    mp_fix_dependencies (mp);
  free_number (delta);
}


@ @<Transform a known big node@>=
mp_set_up_trans (mp, c);
if (mp->cur_exp.type == mp_known) {
  @<Transform known by known@>;
} else {
  pp = mp_stash_cur_exp (mp);
  qq = value_node (pp);
  mp_make_exp_copy (mp, p);
  r = value_node (cur_exp_node ());
  if (mp->cur_exp.type == mp_transform_type) {
    mp_bilin2 (mp, yy_part (r), yy_part (qq), value_number (xy_part (q)),
               yx_part (qq), NULL);
    mp_bilin2 (mp, yx_part (r), yy_part (qq), value_number (xx_part (q)),
               yx_part (qq), NULL);
    mp_bilin2 (mp, xy_part (r), xx_part (qq), value_number (yy_part (q)),
               xy_part (qq), NULL);
    mp_bilin2 (mp, xx_part (r), xx_part (qq), value_number (yx_part (q)),
               xy_part (qq), NULL);
  }
  mp_bilin2 (mp, y_part (r), yy_part (qq), value_number (x_part (q)),
             yx_part (qq), y_part (qq));
  mp_bilin2 (mp, x_part (r), xx_part (qq), value_number (y_part (q)),
             xy_part (qq), x_part (qq));
  mp_recycle_value (mp, pp);
  mp_free_value_node (mp, pp);
}


@ Let |p| be a |mp_proto_dependent| value whose dependency list ends
at |dep_final|. The following procedure adds |v| times another
numeric quantity to~|p|.

@<Declare subroutines needed by |big_trans|@>=
static void mp_add_mult_dep (MP mp, mp_value_node p, mp_number v, mp_node r) {
  if (mp_type (r) == mp_known) {
    mp_number ret;
    new_number (ret);
    take_scaled (ret, value_number (r), v);
    set_dep_value (mp->dep_final, dep_value (mp->dep_final));
    number_add (dep_value (mp->dep_final), ret);
    free_number (ret);
  } else {
    set_dep_list (p,
      mp_p_plus_fq (mp, (mp_value_node) dep_list (p), v,
                              (mp_value_node) dep_list ((mp_value_node) r),
                              mp_proto_dependent, mp_type (r)));
    if (mp->fix_needed)
      mp_fix_dependencies (mp);
  }
}


@ The |bilin2| procedure is something like |bilin1|, but with known
and unknown quantities reversed. Parameter |p| points to a value field
within the big node for |cur_exp|; and |type(p)=mp_known|. Parameters
|t| and~|u| point to value fields elsewhere; so does parameter~|q|,
unless it is |NULL| (which stands for zero). Location~|p| will be
replaced by $p\cdot t+v\cdot u+q$.

@<Declare subroutines needed by |big_trans|@>=
static void mp_bilin2 (MP mp, mp_node p, mp_node t, mp_number v,
                       mp_node u, mp_node q) {
  mp_number vv;    /* temporary storage for |value(p)| */
  new_number (vv);
  number_clone (vv, value_number (p));
  mp_new_dep (mp, p, mp_proto_dependent, mp_const_dependency (mp, zero_t));  /* this sets |dep_final| */
  if (number_nonzero(vv)) {
    mp_add_mult_dep (mp, (mp_value_node) p, vv, t);     /* |dep_final| doesn't change */
  }
  if (number_nonzero(v)) {
    mp_number arg1;
    new_number (arg1);
    number_clone (arg1, v);
    mp_add_mult_dep (mp, (mp_value_node) p, arg1, u);
    free_number (arg1);
  }
  if (q != NULL)
    mp_add_mult_dep (mp, (mp_value_node) p, unity_t, q);
  if (dep_list ((mp_value_node) p) == (mp_node) mp->dep_final) {
    number_clone (vv, dep_value (mp->dep_final));
    mp_recycle_value (mp, p);
    mp_type (p) = mp_known;
    set_value_number (p, vv);
  }
  free_number (vv);
}


@ @<Transform known by known@>=
{
  mp_make_exp_copy (mp, p);
  r = value_node (cur_exp_node ());
  if (mp->cur_exp.type == mp_transform_type) {
    mp_bilin3 (mp, yy_part (r), mp->tyy, value_number (xy_part (q)), mp->tyx, zero_t);
    mp_bilin3 (mp, yx_part (r), mp->tyy, value_number (xx_part (q)), mp->tyx, zero_t);
    mp_bilin3 (mp, xy_part (r), mp->txx, value_number (yy_part (q)), mp->txy, zero_t);
    mp_bilin3 (mp, xx_part (r), mp->txx, value_number (yx_part (q)), mp->txy, zero_t);
  }
  mp_bilin3 (mp, y_part (r), mp->tyy, value_number (x_part (q)), mp->tyx, mp->ty);
  mp_bilin3 (mp, x_part (r), mp->txx, value_number (y_part (q)), mp->txy, mp->tx);
}


@ Finally, in |bilin3| everything is |known|.

@<Declare subroutines needed by |big_trans|@>=
static void mp_bilin3 (MP mp, mp_node p, mp_number t,
                       mp_number v, mp_number u, mp_number delta_orig) {
  mp_number delta;
  mp_number tmp;
  new_number (tmp);
  new_number (delta);
  number_clone (delta, delta_orig);
  if (!number_equal(t, unity_t)) {
    take_scaled (tmp, value_number (p), t);
  } else {
    number_clone (tmp, value_number (p));
  }
  number_add (delta, tmp);
  if (number_nonzero(u)) {
    mp_number ret;
    new_number (ret);
    take_scaled (ret, v, u);
    set_value_number (p, delta);
    number_add (value_number (p), ret);
    free_number (ret);
  } else
    set_value_number (p, delta);
  free_number (tmp);
  free_number (delta);
}


@ @<Declare binary action...@>=
static void mp_chop_path (MP mp, mp_node p) {
  mp_knot q;    /* a knot in the original path */
  mp_knot pp, qq, rr, ss;       /* link variables for copies of path nodes */
  mp_number a, b;    /* indices for chopping */
  mp_number l;
  boolean reversed;     /* was |a>b|? */
  new_number (a);
  new_number (b);
  new_number (l);
  mp_path_length (mp, &l);
  number_clone (a, value_number (x_part (p)));
  number_clone (b, value_number (y_part (p)));
  if (number_lessequal(a, b)) {
    reversed = false;
  } else {
    reversed = true;
    number_swap (a, b);
  }
  /* Dispense with the cases |a<0| and/or |b>l| */
  if (number_negative(a)) {
    if (mp_left_type (cur_exp_knot ()) == mp_endpoint) {
      set_number_to_zero(a);
      if (number_negative(b))
        set_number_to_zero(b);
    } else {
      do {
        number_add (a, l);
        number_add (b, l);
      } while (number_negative(a));            /* a cycle always has length |l>0| */
    }
  }
  if (number_greater (b, l)) {
    if (mp_left_type (cur_exp_knot ()) == mp_endpoint) {
      number_clone (b, l);
      if (number_greater (a, l))
        number_clone(a, l);
    } else {
      while (number_greaterequal (a, l)) {
        number_substract (a, l);
        number_substract (b, l);
      }
    }
  }

  q = cur_exp_knot ();
  while (number_greaterequal(a, unity_t)) {
    q = mp_next_knot (q);
    number_substract(a, unity_t);
    number_substract(b, unity_t);
  }
  if (number_equal(b, a)) {
    /* Construct a path from |pp| to |qq| of length zero */
    if (number_positive (a)) {
      mp_number arg1;
      new_number (arg1);
      number_clone (arg1, a);
      convert_scaled_to_fraction (arg1);
      mp_split_cubic (mp, q, arg1);
      free_number (arg1);
      q = mp_next_knot (q);
    }
    pp = mp_copy_knot (mp, q);
    qq = pp;

  } else {
    /* Construct a path from |pp| to |qq| of length $\lceil b\rceil$ */
    pp = mp_copy_knot (mp, q);
    qq = pp;
    do {
      q = mp_next_knot (q);
      rr = qq;
      qq = mp_copy_knot (mp, q);
      mp_next_knot (rr) = qq;
      number_substract (b, unity_t);
    } while (number_positive (b));
    if (number_positive (a)) {
      mp_number arg1;
      new_number (arg1);
      ss = pp;
      number_clone (arg1, a);
      convert_scaled_to_fraction (arg1);
      mp_split_cubic (mp, ss, arg1);
      free_number (arg1);
      pp = mp_next_knot (ss);
      mp_toss_knot (mp, ss);
      if (rr == ss) {
        mp_number arg1, arg2;
        new_number (arg1);
        new_number (arg2);
        set_number_from_substraction (arg1, unity_t, a);
        number_clone (arg2, b);
        make_scaled (b, arg2, arg1);
        free_number (arg1);
        free_number (arg2);
        rr = pp;
      }
    }
    if (number_negative (b)) {
      mp_number arg1;
      new_number (arg1);
      set_number_from_addition (arg1, b, unity_t);
      convert_scaled_to_fraction (arg1);
      mp_split_cubic (mp, rr, arg1);
      free_number (arg1);
      mp_toss_knot (mp, qq);
      qq = mp_next_knot (rr);
    }

  }
  mp_left_type (pp) = mp_endpoint;
  mp_right_type (qq) = mp_endpoint;
  mp_next_knot (qq) = pp;
  mp_toss_knot_list (mp, cur_exp_knot ());
  if (reversed) {
    set_cur_exp_knot (mp_next_knot (mp_htap_ypoc (mp, pp)));
    mp_toss_knot_list (mp, pp);
  } else {
    set_cur_exp_knot (pp);
  }
  free_number (l);
  free_number (a);
  free_number (b);
}


@ @<Declare binary action...@>=
static void mp_set_up_offset (MP mp, mp_node p) {
  mp_find_offset (mp, value_number (x_part (p)), value_number (y_part (p)),
                  cur_exp_knot ());
  mp_pair_value (mp, mp->cur_x, mp->cur_y);
}
static void mp_set_up_direction_time (MP mp, mp_node p) {
  mp_value new_expr;
  memset(&new_expr,0,sizeof(mp_value));
  new_number (new_expr.data.n);
  mp_find_direction_time (mp, &new_expr.data.n, value_number (x_part (p)),
                                              value_number (y_part (p)),
                                              cur_exp_knot ());
  mp_flush_cur_exp (mp, new_expr);
}
static void mp_set_up_envelope (MP mp, mp_node p) {
  unsigned char ljoin, lcap;
  mp_number miterlim;
  mp_knot q = mp_copy_path (mp, cur_exp_knot ());       /* the original path */
  new_number(miterlim);
  /* TODO: accept elliptical pens for straight paths */
  if (pen_is_elliptical (value_knot (p))) {
    mp_bad_envelope_pen (mp);
    set_cur_exp_knot (q);
    mp->cur_exp.type = mp_path_type;
    return;
  }
  if (number_greater (internal_value (mp_linejoin), unity_t))
    ljoin = 2;
  else if (number_positive (internal_value (mp_linejoin)))
    ljoin = 1;
  else
    ljoin = 0;
  if (number_greater (internal_value (mp_linecap), unity_t))
    lcap = 2;
  else if (number_positive (internal_value (mp_linecap)))
    lcap = 1;
  else
    lcap = 0;
  if (number_less (internal_value (mp_miterlimit), unity_t))
    set_number_to_unity(miterlim);
  else
    number_clone(miterlim, internal_value (mp_miterlimit));
  set_cur_exp_knot (mp_make_envelope
                    (mp, q, value_knot (p), ljoin, lcap, miterlim));
  mp->cur_exp.type = mp_path_type;
}


@ This is pretty straightfoward. The one silly thing is that
the output of |mp_ps_do_font_charstring| has to be un-exported.

@<Declare binary action...@>=
static void mp_set_up_glyph_infont (MP mp, mp_node p) {
  mp_edge_object *h = NULL;
  mp_ps_font *f = NULL;
  char *n = mp_str (mp, cur_exp_str ());
  f = mp_ps_font_parse (mp, (int) mp_find_font (mp, n));
  if (f != NULL) {
    if (mp_type (p) == mp_known) {
      int v = round_unscaled (value_number (p));
      if (v < 0 || v > 255) {
        char msg[256];
        mp_snprintf (msg, 256, "glyph index too high (%d)", v);
        mp_error (mp, msg, NULL, true);
      } else {
        h = mp_ps_font_charstring (mp, f, v);
      }
    } else {
      n = mp_str (mp, value_str (p));
      h = mp_ps_do_font_charstring (mp, f, n);
    }
    mp_ps_font_free (mp, f);
  }
  if (h != NULL) {
    set_cur_exp_node ((mp_node)mp_gr_import (mp, h));
  } else {
    set_cur_exp_node ((mp_node)mp_get_edge_header_node (mp));
    mp_init_edges (mp, (mp_edge_header_node)cur_exp_node ());
  }
  mp->cur_exp.type = mp_picture_type;
}


@ @<Declare binary action...@>=
static void mp_find_point (MP mp, mp_number v_orig, quarterword c) {
  mp_knot p;    /* the path */
  mp_number n;     /* its length */
  mp_number v;
  new_number (v);
  new_number (n);
  number_clone (v, v_orig);
  p = cur_exp_knot ();
  if (mp_left_type (p) == mp_endpoint) {
    set_number_to_unity (n);
    number_negate (n);
  } else {
    set_number_to_zero (n);
  }
  do {
    p = mp_next_knot (p);
    number_add (n, unity_t);
  } while (p != cur_exp_knot ());
  if (number_zero (n)) {
    set_number_to_zero(v);
  } else if (number_negative(v)) {
    if (mp_left_type (p) == mp_endpoint) {
      set_number_to_zero(v);
    } else  {
      /* |v = n - 1 - ((-v - 1) % n)
          == - ((-v - 1) % n) - 1 + n| */
      number_negate (v);
      number_add_scaled (v, -1);
      number_modulo (v, n);
      number_negate (v);
      number_add_scaled (v, -1);
      number_add (v, n);
    }
  } else if (number_greater(v, n)) {
    if (mp_left_type (p) == mp_endpoint)
      number_clone (v, n);
    else
      number_modulo (v, n);
  }
  p = cur_exp_knot ();
  while (number_greaterequal(v, unity_t)) {
    p = mp_next_knot (p);
    number_substract (v, unity_t);
  }
  if (number_nonzero(v)) {
    /* Insert a fractional node by splitting the cubic */
    convert_scaled_to_fraction (v);
    mp_split_cubic (mp, p, v);
    p = mp_next_knot (p);
  }
  /* Set the current expression to the desired path coordinates */
  switch (c) {
  case mp_point_of:
    mp_pair_value (mp, p->x_coord, p->y_coord);
    break;
  case mp_precontrol_of:
    if (mp_left_type (p) == mp_endpoint)
      mp_pair_value (mp, p->x_coord, p->y_coord);
    else
      mp_pair_value (mp, p->left_x, p->left_y);
    break;
  case mp_postcontrol_of:
    if (mp_right_type (p) == mp_endpoint)
      mp_pair_value (mp, p->x_coord, p->y_coord);
    else
      mp_pair_value (mp, p->right_x, p->right_y);
    break;
  }  /* there are no other cases */
  free_number (v);
  free_number (n);
}

@ Function |new_text_node| owns the reference count for its second argument
(the text string) but not its first (the font name).

@<Declare binary action...@>=
static void mp_do_infont (MP mp, mp_node p) {
  mp_edge_header_node q;
  mp_value new_expr;
  memset(&new_expr,0,sizeof(mp_value));
  new_number(new_expr.data.n);
  q = mp_get_edge_header_node (mp);
  mp_init_edges (mp, q);
  add_str_ref (cur_exp_str());
  mp_link (obj_tail (q)) =
    mp_new_text_node (mp, mp_str (mp, cur_exp_str ()), value_str (p));
  obj_tail (q) = mp_link (obj_tail (q));
  mp_free_value_node (mp, p);
  new_expr.data.node = (mp_node)q;
  mp_flush_cur_exp (mp, new_expr);
  mp->cur_exp.type = mp_picture_type;
}


@* Statements and commands.
The chief executive of \MP\ is the |do_statement| routine, which
contains the master switch that causes all the various pieces of \MP\
to do their things, in the right order.

In a sense, this is the grand climax of the program: It applies all the
tools that we have worked so hard to construct. In another sense, this is
the messiest part of the program: It necessarily refers to other pieces
of code all over the place, so that a person can't fully understand what is
going on without paging back and forth to be reminded of conventions that
are defined elsewhere. We are now at the hub of the web.

The structure of |do_statement| itself is quite simple.  The first token
of the statement is fetched using |get_x_next|.  If it can be the first
token of an expression, we look for an equation, an assignment, or a
title. Otherwise we use a \&{case} construction to branch at high speed to
the appropriate routine for various and sundry other types of commands,
each of which has an ``action procedure'' that does the necessary work.

The program uses the fact that
$$\hbox{|min_primary_command=max_statement_command=type_name|}$$
to interpret a statement that starts with, e.g., `\&{string}',
as a type declaration rather than a boolean expression.

@c
static void worry_about_bad_statement (MP mp);
static void flush_unparsable_junk_after_statement (MP mp);
void mp_do_statement (MP mp) {                               /* governs \MP's activities */
  mp->cur_exp.type = mp_vacuous;
  mp_get_x_next (mp);
  if (cur_cmd() > mp_max_primary_command) {
    worry_about_bad_statement (mp);
  } else if (cur_cmd() > mp_max_statement_command) {
    /* Do an equation, assignment, title, or
     `$\langle\,$expression$\,\rangle\,$\&{endgroup}'; */
    /* The most important statements begin with expressions */
    mp_value new_expr;
    mp->var_flag = mp_assignment;
    mp_scan_expression (mp);
    if (cur_cmd() < mp_end_group) {
      if (cur_cmd() == mp_equals)
        mp_do_equation (mp);
      else if (cur_cmd() == mp_assignment)
        mp_do_assignment (mp);
      else if (mp->cur_exp.type == mp_string_type) {
        /* Do a title */
        if (number_positive (internal_value (mp_tracing_titles))) {
          mp_print_nl (mp, "");
          mp_print_str (mp, cur_exp_str ());
          update_terminal();
        }
      } else if (mp->cur_exp.type != mp_vacuous) {
        const char *hlp[] = {
             "I couldn't find an `=' or `:=' after the",
             "expression that is shown above this error message,",
             "so I guess I'll just ignore it and carry on.",
             NULL };
        mp_disp_err(mp, NULL);
        mp_back_error (mp, "Isolated expression", hlp, true);
        mp_get_x_next (mp);
      }
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      set_number_to_zero (new_expr.data.n);
      mp_flush_cur_exp (mp, new_expr);
      mp->cur_exp.type = mp_vacuous;
    }
  } else {
    /* Do a statement that doesn't begin with an expression */
    /* If |do_statement| ends with |cur_cmd=end_group|, we should have
       |cur_type=mp_vacuous| unless the statement was simply an expression;
       in the latter case, |cur_type| and |cur_exp| should represent that
       expression. */
    if (number_positive (internal_value (mp_tracing_commands)))
      show_cur_cmd_mod;
    switch (cur_cmd()) {
    case mp_type_name:
      mp_do_type_declaration (mp);
      break;
    case mp_macro_def:
      if (cur_mod() > var_def)
        mp_make_op_def (mp);
      else if (cur_mod() > end_def)
        mp_scan_def (mp);
      break;
    case mp_random_seed:
      mp_do_random_seed (mp);
      break;
    case mp_mode_command:
      mp_print_ln (mp);
      mp->interaction = cur_mod();
      initialize_print_selector();
      if (mp->log_opened)
        mp->selector = mp->selector + 2;
      mp_get_x_next (mp);
      break;
    case mp_protection_command:
      mp_do_protection (mp);
      break;
    case mp_delimiters:
      mp_def_delims (mp);
      break;
    case mp_save_command:
      do {
        mp_get_symbol (mp);
        mp_save_variable (mp, cur_sym());
        mp_get_x_next (mp);
      } while (cur_cmd() == mp_comma);
      break;
    case mp_interim_command:
      mp_do_interim (mp);
      break;
    case mp_let_command:
      mp_do_let (mp);
      break;
    case mp_new_internal:
      mp_do_new_internal (mp);
      break;
    case mp_show_command:
      mp_do_show_whatever (mp);
      break;
    case mp_add_to_command:
      mp_do_add_to (mp);
      break;
    case mp_bounds_command:
      mp_do_bounds (mp);
      break;
    case mp_ship_out_command:
      mp_do_ship_out (mp);
      break;
    case mp_every_job_command:
      mp_get_symbol (mp);
      mp->start_sym = cur_sym();
      mp_get_x_next (mp);
      break;
    case mp_message_command:
      mp_do_message (mp);
      break;
    case mp_write_command:
      mp_do_write (mp);
      break;
    case mp_tfm_command:
      mp_do_tfm_command (mp);
      break;
    case mp_special_command:
      if (cur_mod() == 0)
        mp_do_special (mp);
      else if (cur_mod() == 1)
        mp_do_mapfile (mp);
      else
        mp_do_mapline (mp);
      break;
    default:
      break; /* make the compiler happy */
    }
    mp->cur_exp.type = mp_vacuous;
  }
  if (cur_cmd() < mp_semicolon)
    flush_unparsable_junk_after_statement(mp);
  mp->error_count = 0;
}


@ @<Declarations@>=
@<Declare action procedures for use by |do_statement|@>
 

@ The only command codes |>max_primary_command| that can be present
at the beginning of a statement are |semicolon| and higher; these
occur when the statement is null.

@c
static void worry_about_bad_statement (MP mp) {
  if (cur_cmd() < mp_semicolon) {
    char msg[256];
    mp_string sname;
    int old_setting = mp->selector;
    const char *hlp[] = {
           "I was looking for the beginning of a new statement.",
           "If you just proceed without changing anything, I'll ignore",
           "everything up to the next `;'. Please insert a semicolon",
           "now in front of anything that you don't want me to delete.",
           "(See Chapter 27 of The METAFONTbook for an example.)",
           NULL };
    mp->selector = new_string;
    mp_print_cmd_mod (mp, cur_cmd(), cur_mod());
    sname = mp_make_string(mp);
    mp->selector = old_setting;
    mp_snprintf (msg, 256, "A statement can't begin with `%s'", mp_str(mp, sname));
    delete_str_ref(sname);
    mp_back_error (mp, msg, hlp, true);
    mp_get_x_next (mp);
  }
}


@ The help message printed here says that everything is flushed up to
a semicolon, but actually the commands |end_group| and |stop| will
also terminate a statement.

@c
static void flush_unparsable_junk_after_statement (MP mp)
{
  const char *hlp[] = {
         "I've just read as much of that statement as I could fathom,",
         "so a semicolon should have been next. It's very puzzling...",
         "but I'll try to get myself back together, by ignoring",
         "everything up to the next `;'. Please insert a semicolon",
         "now in front of anything that you don't want me to delete.",
         "(See Chapter 27 of The METAFONTbook for an example.)",
          NULL };
  mp_back_error (mp, "Extra tokens will be flushed", hlp, true);
  mp->scanner_status = flushing;
  do {
    get_t_next (mp);
    if (cur_cmd() == mp_string_token) {
      delete_str_ref (cur_mod_str());
    }
  } while (!mp_end_of_statement);  /* |cur_cmd=semicolon|, |end_group|, or |stop| */
  mp->scanner_status = normal;
}



@ Equations and assignments are performed by the pair of mutually recursive
@^recursion@>
routines |do_equation| and |do_assignment|. These routines are called when
|cur_cmd=equals| and when |cur_cmd=assignment|, respectively; the left-hand
side is in |cur_type| and |cur_exp|, while the right-hand side is yet
to be scanned. After the routines are finished, |cur_type| and |cur_exp|
will be equal to the right-hand side (which will normally be equal
to the left-hand side).

@<Declarations@>=
@<Declare the procedure called |make_eq|@>;
static void mp_do_equation (MP mp);

@ @c
static void trace_equation (MP mp, mp_node lhs) {
  mp_begin_diagnostic (mp);
  mp_print_nl (mp, "{(");
  mp_print_exp (mp, lhs, 0);
  mp_print (mp, ")=(");
  mp_print_exp (mp, NULL, 0);
  mp_print (mp, ")}");
  mp_end_diagnostic (mp, false);
}
void mp_do_equation (MP mp) {
  mp_node lhs;  /* capsule for the left-hand side */
  lhs = mp_stash_cur_exp (mp);
  mp_get_x_next (mp);
  mp->var_flag = mp_assignment;
  mp_scan_expression (mp);
  if (cur_cmd() == mp_equals)
    mp_do_equation (mp);
  else if (cur_cmd() == mp_assignment)
    mp_do_assignment (mp);
  if (number_greater (internal_value (mp_tracing_commands), two_t)) {
    trace_equation(mp, lhs);
  }
  if (mp->cur_exp.type == mp_unknown_path) {
    if (mp_type (lhs) == mp_pair_type) {
      mp_node p;    /* temporary register */
      p = mp_stash_cur_exp (mp);
      mp_unstash_cur_exp (mp, lhs);
      lhs = p;
    }  /* in this case |make_eq| will change the pair to a path */
  }
  mp_make_eq (mp, lhs); /* equate |lhs| to |(cur_type,cur_exp)| */
}


@ And |do_assignment| is similar to |do_equation|:

@<Declarations@>=
static void mp_do_assignment (MP mp);

@ @c
static void bad_lhs (MP mp) {
  const char *hlp[] = {
         "I didn't find a variable name at the left of the `:=',",
         "so I'm going to pretend that you said `=' instead.",
         NULL };
  mp_disp_err(mp, NULL);
  mp_error (mp, "Improper `:=' will be changed to `='", hlp, true);
  mp_do_equation (mp);
}
static void bad_internal_assignment (MP mp, mp_node lhs) {
  char msg[256];
  const char *hlp[] = {
           "I can\'t set this internal quantity to anything but a known",
           "numeric value, so I'll have to ignore this assignment.",
           NULL };
  mp_disp_err(mp, NULL);
  if (internal_type (mp_sym_info (lhs)) == mp_known) {
    mp_snprintf (msg, 256, "Internal quantity `%s' must receive a known numeric value",
                 internal_name (mp_sym_info (lhs)));
  } else {
    mp_snprintf (msg, 256, "Internal quantity `%s' must receive a known string",
              internal_name (mp_sym_info (lhs)));
    hlp[1] = "string, so I'll have to ignore this assignment.";
  }
  mp_back_error (mp, msg, hlp, true);
  mp_get_x_next (mp);
}
static void forbidden_internal_assignment (MP mp, mp_node lhs) {
  char msg[256];
  const char *hlp[] = {
           "I can\'t set this internal quantity to anything just yet",
           "(it is read-only), so I'll have to ignore this assignment.",
           NULL };
  mp_snprintf (msg, 256, "Internal quantity `%s' is read-only",
               internal_name (mp_sym_info (lhs)));
  mp_back_error (mp, msg, hlp, true);
  mp_get_x_next (mp);
}
static void bad_internal_assignment_precision (MP mp, mp_node lhs, mp_number min, mp_number max) {
  char msg[256];
  char s[256];
  const char *hlp[] = {
       "Precision values are limited by the current numbersystem.",
       NULL,
       NULL } ;
  mp_snprintf (msg, 256, "Bad '%s' has been ignored", internal_name (mp_sym_info (lhs)));
  mp_snprintf (s, 256, "Currently I am using '%s'; the allowed precision range is [%s,%s].", 
               mp_str (mp, internal_string (mp_number_system)), number_tostring(min), number_tostring(max));
  hlp[1] = s;
  mp_back_error (mp, msg, hlp, true);
  mp_get_x_next (mp);
}
static void bad_expression_assignment (MP mp, mp_node lhs) {
  const char *hlp[] = { 
       "It seems you did a nasty thing---probably by accident,",
       "but nevertheless you nearly hornswoggled me...",
       "While I was evaluating the right-hand side of this",
       "command, something happened, and the left-hand side",
       "is no longer a variable! So I won't change anything.",
       NULL };
  char *msg = mp_obliterated (mp, lhs);
  mp_back_error (mp, msg, hlp, true);
  free(msg);
  mp_get_x_next (mp);
}
static void trace_assignment (MP mp, mp_node lhs) {
  mp_begin_diagnostic (mp);
  mp_print_nl (mp, "{");
  if (mp_name_type (lhs) == mp_internal_sym)
    mp_print (mp, internal_name (mp_sym_info (lhs)));
  else
    mp_show_token_list (mp, lhs, NULL, 1000, 0);
  mp_print (mp, ":=");
  mp_print_exp (mp, NULL, 0);
  mp_print_char (mp, xord ('}'));
  mp_end_diagnostic (mp, false);
}
void mp_do_assignment (MP mp) {
  if (mp->cur_exp.type != mp_token_list) {
    bad_lhs(mp);
  } else {
    mp_node lhs;  /* token list for the left-hand side */
    lhs = cur_exp_node ();
    mp->cur_exp.type = mp_vacuous;
    mp_get_x_next (mp);
    mp->var_flag = mp_assignment;
    mp_scan_expression (mp);
    if (cur_cmd() == mp_equals)
      mp_do_equation (mp);
    else if (cur_cmd() == mp_assignment)
      mp_do_assignment (mp);
    if (number_greater (internal_value (mp_tracing_commands), two_t)) {
      trace_assignment (mp, lhs);
    }
    if (mp_name_type (lhs) == mp_internal_sym) {
      /* Assign the current expression to an internal variable */
      if ((mp->cur_exp.type == mp_known || mp->cur_exp.type == mp_string_type)
          && (internal_type (mp_sym_info (lhs)) == mp->cur_exp.type)) {
	  if(mp_sym_info (lhs) == mp_number_system) {
             forbidden_internal_assignment (mp, lhs);
          } else if (mp_sym_info (lhs) == mp_number_precision) {
	     if (!(mp->cur_exp.type == mp_known && 
               (!number_less(cur_exp_value_number(), precision_min)) &&
               (!number_greater(cur_exp_value_number(), precision_max))
               )) {
	       bad_internal_assignment_precision(mp, lhs, precision_min, precision_max);
             } else {
	       set_internal_from_cur_exp(mp_sym_info (lhs));
               set_precision();
             }
          } else {
	     set_internal_from_cur_exp(mp_sym_info (lhs));
          }
      } else {
        bad_internal_assignment (mp, lhs);
      }
    } else {
      /* Assign the current expression to the variable |lhs| */
      mp_node p;    /* where the left-hand value is stored */
      mp_node q;    /* temporary capsule for the right-hand value */
      p = mp_find_variable (mp, lhs);
      if (p != NULL) {
        q = mp_stash_cur_exp (mp);
        mp->cur_exp.type = mp_und_type (mp, p);
        mp_recycle_value (mp, p);
        mp_type (p) = mp->cur_exp.type;
        set_value_number (p, zero_t);
        mp_make_exp_copy (mp, p);
        p = mp_stash_cur_exp (mp);
        mp_unstash_cur_exp (mp, q);
        mp_make_eq (mp, p);
      } else {
        bad_expression_assignment(mp, lhs);
      }
    }
    mp_flush_node_list (mp, lhs);
  }
}


@ And now we get to the nitty-gritty. The |make_eq| procedure is given
a pointer to a capsule that is to be equated to the current expression.

@<Declare the procedure called |make_eq|@>=
static void mp_make_eq (MP mp, mp_node lhs);

@ 
@c
static void announce_bad_equation (MP mp, mp_node lhs) {
  char msg[256];
  const char *hlp[] = {
       "I'm sorry, but I don't know how to make such things equal.",
       "(See the two expressions just above the error message.)",
       NULL  };
  mp_snprintf(msg, 256, "Equation cannot be performed (%s=%s)", 
	(mp_type (lhs) <= mp_pair_type ? mp_type_string (mp_type (lhs)) : "numeric"),
	(mp->cur_exp.type <= mp_pair_type ? mp_type_string (mp->cur_exp.type) : "numeric"));
  mp_disp_err (mp, lhs);
  mp_disp_err(mp, NULL);
  mp_back_error (mp, msg, hlp, true);
  mp_get_x_next (mp);
}
static void exclaim_inconsistent_equation (MP mp) {
  const char *hlp[] = {
       "The equation I just read contradicts what was said before.",
       "But don't worry; continue and I'll just ignore it.",
        NULL };
  mp_back_error (mp,"Inconsistent equation", hlp, true);
  mp_get_x_next (mp);
}
static void exclaim_redundant_or_inconsistent_equation (MP mp) {
  const char *hlp[] = {
       "An equation between already-known quantities can't help.",
       "But don't worry; continue and I'll just ignore it.",
        NULL };
  mp_back_error (mp, "Redundant or inconsistent equation", hlp, true);
  mp_get_x_next (mp);
}
static void report_redundant_or_inconsistent_equation (MP mp, mp_node lhs, mp_number v) {
  if (mp->cur_exp.type <= mp_string_type) {
    if (mp->cur_exp.type == mp_string_type) {
      if (mp_str_vs_str (mp, value_str (lhs), cur_exp_str ()) != 0) {
        exclaim_inconsistent_equation(mp);
      } else {
        exclaim_redundant_equation(mp);
      }
    } else if (!number_equal (v, cur_exp_value_number ())) {
      exclaim_inconsistent_equation(mp);
    } else {
      exclaim_redundant_equation(mp);
    }
  } else {
    exclaim_redundant_or_inconsistent_equation (mp);
  }
}

void mp_make_eq (MP mp, mp_node lhs) {
  mp_value new_expr;
  mp_variable_type t;   /* type of the left-hand side */
  mp_number v;        /* value of the left-hand side */
  memset(&new_expr,0,sizeof(mp_value));
  new_number (v);
RESTART:
  t = mp_type (lhs);
  if (t <= mp_pair_type)
    number_clone (v, value_number (lhs));
  /* For each type |t|, make an equation or complain if |cur_type|
     is incompatible with~|t| */
  switch (t) {
  case mp_boolean_type:
  case mp_string_type:
  case mp_pen_type:
  case mp_path_type:
  case mp_picture_type:
    if (mp->cur_exp.type == t + unknown_tag) {
      new_number(new_expr.data.n);
      if (t==mp_boolean_type) {
        number_clone (new_expr.data.n, v);
      } else if (t==mp_string_type) {
        new_expr.data.str = value_str(lhs);
      } else if (t==mp_picture_type) {
        new_expr.data.node = value_node(lhs);
      } else { /* pen or path */
        new_expr.data.p = value_knot(lhs);
      }
      mp_nonlinear_eq (mp, new_expr, cur_exp_node (), false);
      mp_unstash_cur_exp (mp, cur_exp_node ());
    } else if (mp->cur_exp.type == t) {
      report_redundant_or_inconsistent_equation(mp, lhs, v);
    } else {
      announce_bad_equation(mp, lhs);
    }
    break;
  case unknown_types:
    if (mp->cur_exp.type == t - unknown_tag) {
      mp_nonlinear_eq (mp, mp->cur_exp, lhs, true);
    } else if (mp->cur_exp.type == t) {
      mp_ring_merge (mp, lhs, cur_exp_node ());
    } else if (mp->cur_exp.type == mp_pair_type) {
      if (t == mp_unknown_path) {
        mp_pair_to_path (mp);
        goto RESTART;
      }
    } else {
      announce_bad_equation(mp, lhs);
    }
    break;
  case mp_transform_type:
  case mp_color_type:
  case mp_cmykcolor_type:
  case mp_pair_type:
    if (mp->cur_exp.type == t) {
      /* Do multiple equations */
      mp_node q = value_node (cur_exp_node ());
      mp_node p = value_node (lhs);
      switch (t) {
      case mp_transform_type:
        mp_try_eq (mp, yy_part (p), yy_part (q));
        mp_try_eq (mp, yx_part (p), yx_part (q));
        mp_try_eq (mp, xy_part (p), xy_part (q));
        mp_try_eq (mp, xx_part (p), xx_part (q));
        mp_try_eq (mp, ty_part (p), ty_part (q));
        mp_try_eq (mp, tx_part (p), tx_part (q));
        break;
      case mp_color_type:
        mp_try_eq (mp, blue_part (p),  blue_part (q));
        mp_try_eq (mp, green_part (p), green_part (q));
        mp_try_eq (mp, red_part (p),   red_part (q));
        break;
      case mp_cmykcolor_type:
        mp_try_eq (mp, black_part (p),   black_part (q));
        mp_try_eq (mp, yellow_part (p),  yellow_part (q));
        mp_try_eq (mp, magenta_part (p), magenta_part (q));
        mp_try_eq (mp, cyan_part (p),    cyan_part (q));
        break;
      case mp_pair_type:
        mp_try_eq (mp, y_part (p), y_part (q));
        mp_try_eq (mp, x_part (p), x_part (q));
        break;
      default:  /* there are no other valid cases, but please the compiler */
        break;
      }
    } else {
      announce_bad_equation(mp, lhs);
    }
    break;
  case mp_known:
  case mp_dependent:
  case mp_proto_dependent:
  case mp_independent:
    if (mp->cur_exp.type >= mp_known) {
      mp_try_eq (mp, lhs, NULL);
    } else {
      announce_bad_equation(mp, lhs);
    }
    break;
  case mp_vacuous:
    announce_bad_equation(mp, lhs);
    break;
  default: /* there are no other valid cases, but please the compiler */
    announce_bad_equation(mp, lhs);
    break;
  } 
  check_arith();
  mp_recycle_value (mp, lhs);
  free_number (v);
  mp_free_value_node (mp, lhs);
}

@ The first argument to |try_eq| is the location of a value node
in a capsule that will soon be recycled. The second argument is
either a location within a pair or transform node pointed to by
|cur_exp|, or it is |NULL| (which means that |cur_exp| itself
serves as the second argument). The idea is to leave |cur_exp| unchanged,
but to equate the two operands.

@<Declarations@>=
static void mp_try_eq (MP mp, mp_node l, mp_node r);

@ 
@d equation_threshold_k ((math_data *)mp->math)->equation_threshold_t

@c
static void deal_with_redundant_or_inconsistent_equation(MP mp, mp_value_node p, mp_node r) {
  mp_number absp;
  new_number (absp);
  number_clone (absp, value_number (p));
  number_abs (absp);
  if (number_greater (absp, equation_threshold_k)) {   /* off by .001 or more */
    char msg[256];    
    const char *hlp[] = {
           "The equation I just read contradicts what was said before.",
           "But don't worry; continue and I'll just ignore it.",
           NULL };
    mp_snprintf (msg, 256, "Inconsistent equation (off by %s)", number_tostring (value_number (p)));
    mp_back_error (mp, msg, hlp, true);
    mp_get_x_next (mp);
  } else if (r == NULL) {
    exclaim_redundant_equation(mp);
  }
  free_number (absp);
  mp_free_dep_node (mp, p);
}

void mp_try_eq (MP mp, mp_node l, mp_node r) {
  mp_value_node p;      /* dependency list for right operand minus left operand */
  mp_variable_type t;   /* the type of list |p| */
  mp_value_node q;      /* the constant term of |p| is here */
  mp_value_node pp;     /* dependency list for right operand */
  mp_variable_type tt;  /* the type of list |pp| */
  boolean copied;       /* have we copied a list that ought to be recycled? */
  /* Remove the left operand from its container, negate it, and
     put it into dependency list~|p| with constant term~|q| */
  t = mp_type (l);
  if (t == mp_known) {
    mp_number arg1;
    new_number (arg1);
    number_clone (arg1, value_number(l));
    number_negate (arg1);
    t = mp_dependent;
    p = mp_const_dependency (mp, arg1);
    q = p;
    free_number (arg1);
  } else if (t == mp_independent) {
    t = mp_dependent;
    p = mp_single_dependency (mp, l);
    number_negate(dep_value (p));
    q = mp->dep_final;
  } else {
    mp_value_node ll = (mp_value_node) l;
    p = (mp_value_node) dep_list (ll);
    q = p;
    while (1) {
      number_negate(dep_value (q));
      if (dep_info (q) == NULL)
        break;
      q = (mp_value_node) mp_link (q);
    }
    mp_link (prev_dep (ll)) = mp_link (q);
    set_prev_dep ((mp_value_node) mp_link (q), prev_dep (ll));
    mp_type (ll) = mp_known;
  }
  
  /* Add the right operand to list |p| */
  if (r == NULL) {
    if (mp->cur_exp.type == mp_known) {
      number_add (value_number (q), cur_exp_value_number ());
      goto DONE1;
    } else {
      tt = mp->cur_exp.type;
      if (tt == mp_independent)
        pp = mp_single_dependency (mp, cur_exp_node ());
      else
        pp = (mp_value_node) dep_list ((mp_value_node) cur_exp_node ());
    }
  } else {
    if (mp_type (r) == mp_known) {
      number_add (dep_value (q), value_number (r));
      goto DONE1;
    } else {
      tt = mp_type (r);
      if (tt == mp_independent)
        pp = mp_single_dependency (mp, r);
      else
        pp = (mp_value_node) dep_list ((mp_value_node) r);
    }
  }
  if (tt != mp_independent) {
    copied = false;
  } else {
    copied = true;
    tt = mp_dependent;
  }
  /* Add dependency list |pp| of type |tt| to dependency list~|p| of type~|t| */
  mp->watch_coefs = false;
  if (t == tt) {
    p = mp_p_plus_q (mp, p, pp, (quarterword) t);
  } else if (t == mp_proto_dependent) {
    p = mp_p_plus_fq (mp, p, unity_t, pp, mp_proto_dependent, mp_dependent);
  } else {
    mp_number x;
    new_number (x);
    q = p;
    while (dep_info (q) != NULL) {
      number_clone (x, dep_value (q));
      fraction_to_round_scaled (x);
      set_dep_value (q, x);
      q = (mp_value_node) mp_link (q);
    }
    free_number (x);
    t = mp_proto_dependent;
    p = mp_p_plus_q (mp, p, pp, (quarterword) t);
  }
  mp->watch_coefs = true;

  if (copied)
    mp_flush_node_list (mp, (mp_node) pp);
  DONE1:

  if (dep_info (p) == NULL) {
    deal_with_redundant_or_inconsistent_equation(mp, p, r);
  } else {
    mp_linear_eq (mp, p, (quarterword) t);
    if (r == NULL && mp->cur_exp.type != mp_known) {
      if (mp_type (cur_exp_node ()) == mp_known) {
        mp_node pp = cur_exp_node ();
        set_cur_exp_value_number (value_number (pp));
        mp->cur_exp.type = mp_known;
        mp_free_value_node (mp, pp);
      }
    }
  }
}

@ Our next goal is to process type declarations. For this purpose it's
convenient to have a procedure that scans a $\langle\,$declared
variable$\,\rangle$ and returns the corresponding token list. After the
following procedure has acted, the token after the declared variable
will have been scanned, so it will appear in |cur_cmd|, |cur_mod|,
and~|cur_sym|.

@<Declarations@>=
static mp_node mp_scan_declared_variable (MP mp);

@ @c
mp_node mp_scan_declared_variable (MP mp) {
  mp_sym x;     /* hash address of the variable's root */
  mp_node h, t; /* head and tail of the token list to be returned */
  mp_get_symbol (mp);
  x = cur_sym();
  if (cur_cmd() != mp_tag_token)
    mp_clear_symbol (mp, x, false);
  h = mp_get_symbolic_node (mp);
  set_mp_sym_sym (h, x);
  t = h;
  while (1) {
    mp_get_x_next (mp);
    if (cur_sym() == NULL)
      break;
    if (cur_cmd() != mp_tag_token) {
      if (cur_cmd() != mp_internal_quantity) {
        if (cur_cmd() == mp_left_bracket) {
          /* Descend past a collective subscript */
	  /* If the subscript isn't collective, we don't accept it as part of the
  	     declared variable. */
	   mp_sym ll = cur_sym();      /* hash address of left bracket */
	   mp_get_x_next (mp);
	   if (cur_cmd() == mp_right_bracket) {
	     set_cur_sym(collective_subscript);
	   } else {
	     mp_back_input (mp);
	     set_cur_sym(ll);
	     set_cur_cmd((mp_variable_type)mp_left_bracket);
	     break;  
	   }
        } else {
          break;
        }
      }
    }
    mp_link (t) = mp_get_symbolic_node (mp);
    t = mp_link (t);
    set_mp_sym_sym (t, cur_sym());
    mp_name_type (t) = cur_sym_mod();
  }
  if ((eq_type (x) % mp_outer_tag) != mp_tag_token)
    mp_clear_symbol (mp, x, false);
  if (equiv_node (x) == NULL)
    mp_new_root (mp, x);
  return h;
}


@ Type declarations are introduced by the following primitive operations.

@<Put each...@>=
mp_primitive (mp, "numeric", mp_type_name, mp_numeric_type);
@:numeric_}{\&{numeric} primitive@>;
mp_primitive (mp, "string", mp_type_name, mp_string_type);
@:string_}{\&{string} primitive@>;
mp_primitive (mp, "boolean", mp_type_name, mp_boolean_type);
@:boolean_}{\&{boolean} primitive@>;
mp_primitive (mp, "path", mp_type_name, mp_path_type);
@:path_}{\&{path} primitive@>;
mp_primitive (mp, "pen", mp_type_name, mp_pen_type);
@:pen_}{\&{pen} primitive@>;
mp_primitive (mp, "picture", mp_type_name, mp_picture_type);
@:picture_}{\&{picture} primitive@>;
mp_primitive (mp, "transform", mp_type_name, mp_transform_type);
@:transform_}{\&{transform} primitive@>;
mp_primitive (mp, "color", mp_type_name, mp_color_type);
@:color_}{\&{color} primitive@>;
mp_primitive (mp, "rgbcolor", mp_type_name, mp_color_type);
@:color_}{\&{rgbcolor} primitive@>;
mp_primitive (mp, "cmykcolor", mp_type_name, mp_cmykcolor_type);
@:color_}{\&{cmykcolor} primitive@>;
mp_primitive (mp, "pair", mp_type_name, mp_pair_type);
@:pair_}{\&{pair} primitive@>
 

@ @<Cases of |print_cmd...@>=
case mp_type_name:
mp_print_type (mp, (quarterword) m);
break;

@ Now we are ready to handle type declarations, assuming that a
|type_name| has just been scanned.

@<Declare action procedures for use by |do_statement|@>=
static void mp_do_type_declaration (MP mp);

@ @c
static void flush_spurious_symbols_after_declared_variable(MP mp);
void mp_do_type_declaration (MP mp) {
  integer t;        /* the type being declared */
  mp_node p;    /* token list for a declared variable */
  mp_node q;    /* value node for the variable */
  if (cur_mod() >= mp_transform_type)
    t = (quarterword) cur_mod();
  else
    t = (quarterword) (cur_mod() + unknown_tag);
  do {
    p = mp_scan_declared_variable (mp);
    mp_flush_variable (mp, equiv_node (mp_sym_sym (p)), mp_link (p), false);
    q = mp_find_variable (mp, p);
    if (q != NULL) {
      mp_type (q) = t;
      set_value_number (q, zero_t);         /* todo: this was |null| */
    } else {
      const char *hlp[] = {
             "You can't use, e.g., `numeric foo[]' after `vardef foo'.",
             "Proceed, and I'll ignore the illegal redeclaration.",
             NULL };
      mp_back_error (mp, "Declared variable conflicts with previous vardef", hlp, true);
      mp_get_x_next (mp);
    }
    mp_flush_node_list (mp, p);
    if (cur_cmd() < mp_comma) {
      flush_spurious_symbols_after_declared_variable(mp);
    }
  } while (!mp_end_of_statement);
}


@
@c
static void flush_spurious_symbols_after_declared_variable (MP mp)
{
  const char *hlp[] = { 
         "Variables in declarations must consist entirely of",
         "names and collective subscripts, e.g., `x[]a'.",
         "Are you trying to use a reserved word in a variable name?",
         "I'm going to discard the junk I found here,",
         "up to the next comma or the end of the declaration.",
         NULL };
  if (cur_cmd() == mp_numeric_token)
    hlp[2] = "Explicit subscripts like `x15a' aren't permitted.";
  mp_back_error (mp, "Illegal suffix of declared variable will be flushed", hlp, true);
  mp_get_x_next (mp);
  mp->scanner_status = flushing;
  do {
    get_t_next (mp);
    @<Decrease the string reference count...@>;
  } while (cur_cmd() < mp_comma); /* break on either |end_of_statement| or |comma| */
  mp->scanner_status = normal;
}


@ \MP's |main_control| procedure just calls |do_statement| repeatedly
until coming to the end of the user's program.
Each execution of |do_statement| concludes with
|cur_cmd=semicolon|, |end_group|, or |stop|.

@c
static void mp_main_control (MP mp) {
  do {
    mp_do_statement (mp);
    if (cur_cmd() == mp_end_group) {
      mp_value new_expr;
      const char *hlp[] = { 
             "I'm not currently working on a `begingroup',",
             "so I had better not try to end anything.",
             NULL };
      memset(&new_expr,0,sizeof(mp_value));
      new_number(new_expr.data.n);
      mp_error (mp, "Extra `endgroup'", hlp, true);
      mp_flush_cur_exp (mp, new_expr);
    }
  } while (cur_cmd() != mp_stop);
}
int mp_run (MP mp) {
  if (mp->history < mp_fatal_error_stop) {
    xfree (mp->jump_buf);
    mp->jump_buf = malloc (sizeof (jmp_buf));
    if (mp->jump_buf == NULL || setjmp (*(mp->jump_buf)) != 0)
      return mp->history;
    mp_main_control (mp);       /* come to life */
    mp_final_cleanup (mp);      /* prepare for death */
    mp_close_files_and_terminate (mp);
  }
  return mp->history;
}


@ This function allows setting of internals from an external
source (like the command line or a controlling application).

It accepts two |char *|'s, even for numeric assignments when
it calls |atoi| to get an integer from the start of the string.

@c
void mp_set_internal (MP mp, char *n, char *v, int isstring) {
  size_t l = strlen (n);
  char err[256];
  const char *errid = NULL;
  if (l > 0) {
    mp_sym p = mp_id_lookup (mp, n, l, false);
    if (p == NULL) {
      errid = "variable does not exist";
    } else {
      if (eq_type (p) == mp_internal_quantity) {
        if ((internal_type (equiv (p)) == mp_string_type) && (isstring)) {
          set_internal_string (equiv (p), mp_rts (mp, v));
        } else if ((internal_type (equiv (p)) == mp_known) && (!isstring)) {
          int test = atoi (v);
          if (test > 16383) {
            errid = "value is too large";
          } else if (test < -16383) {
            errid = "value is too small";
          } else {
            set_internal_from_number (equiv (p), unity_t);
            number_multiply_int (internal_value(equiv (p)), test);
          }
        } else {
          errid = "value has the wrong type";
        }
      } else {
        errid = "variable is not an internal";
      }
    }
  }
  if (errid != NULL) {
    if (isstring) {
      mp_snprintf (err, 256, "%s=\"%s\": %s, assignment ignored.", n, v, errid);
    } else {
      mp_snprintf (err, 256, "%s=%d: %s, assignment ignored.", n, atoi (v),
                   errid);
    }
    mp_warn (mp, err);
  }
}


@ @<Exported function headers@>=
void mp_set_internal (MP mp, char *n, char *v, int isstring);

@ For |mp_execute|, we need to define a structure to store the
redirected input and output. This structure holds the five relevant
streams: the three informational output streams, the PostScript
generation stream, and the input stream. These streams have many
things in common, so it makes sense to give them their own structure
definition. 

\item{fptr} is a virtual file pointer
\item{data} is the data this stream holds
\item{cur}  is a cursor pointing into |data| 
\item{size} is the allocated length of the data stream
\item{used} is the actual length of the data stream

There are small differences between input and output: |term_in| never
uses |used|, whe