/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "configdata.h"

#include <QSettings>
#if defined( Q_WS_WIN )
#include <QApplication>
#include <QDir>
#endif

ConfigData* configData = NULL;

ConfigData::ConfigData( QSettings* settings, bool isLocal ) :
    m_settings( settings ),
    m_isLocal( isLocal )
{
    loadSettings();
}

ConfigData::~ConfigData()
{
    delete m_settings;
}

void ConfigData::loadSettings()
{
    m_settings->beginGroup( "GeneralSettings" );
    m_dockWindow = m_settings->value( "DockWindow", false ).toBool();
    m_showAtStartup = (RestoreOption)m_settings->value( "ShowAtStartup", (int)RestoreAuto ).toInt();
    QVariant reconnect = m_settings->value( "ReconnectAtStartup" );
    if ( reconnect.toString() == QLatin1String( "true" ) )
        m_reconnectAtStartup = RestoreAuto;
    else
        m_reconnectAtStartup = (RestoreOption)reconnect.toInt();
    m_language = m_settings->value( "Language", QString() ).toString();
    m_windowLayout = (WindowLayout)m_settings->value( "WindowLayout", (int)LayoutThreePanes ).toInt();
    m_detailsLayout = (WindowLayout)m_settings->value( "DetailsLayout", (int)LayoutSinglePane ).toInt();
    m_settings->endGroup();

    m_settings->beginGroup( "ConnectionSettings" );
    QNetworkProxy::ProxyType proxyType = (QNetworkProxy::ProxyType)m_settings->value( "ProxyType", (int)QNetworkProxy::NoProxy ).toInt();
    QString proxyHost = m_settings->value( "ProxyHost", QString() ).toString();
    quint16 proxyPort = (quint16)m_settings->value( "ProxyPort", 0 ).toInt();
    QString proxyUser = m_settings->value( "ProxyUser", QString() ).toString();
    QString proxyPassword = m_settings->value( "ProxyPassword", QString() ).toString();
    m_proxy = QNetworkProxy( proxyType, proxyHost, proxyPort, proxyUser, proxyPassword );
    m_settings->endGroup();

    m_settings->beginGroup( "AdvancedSettings" );
    m_updateInterval = m_settings->value( "UpdateInterval", 5 ).toInt();
    m_defaultAttachmentAction = (AttachmentAction)m_settings->value( "DefaultAttachmentAction", (int)ActionAsk ).toInt();
    m_attachmentsCacheSize = m_settings->value( "AttachmentsCacheSize", 10 ).toInt();
    m_warnAttachmentSize = m_settings->value( "WarnAttachmentSize", 2 ).toInt();
    m_warnCommentSize = m_settings->value( "WarnCommentSize", 25 ).toInt();
    m_settings->endGroup();
}

void ConfigData::saveSettings()
{
    m_settings->beginGroup( "GeneralSettings" );
    m_settings->setValue( "DockWindow", m_dockWindow );
    m_settings->setValue( "ShowAtStartup", (int)m_showAtStartup );
    m_settings->setValue( "ReconnectAtStartup", (int)m_reconnectAtStartup );
    m_settings->setValue( "Language", m_language );
    m_settings->setValue( "WindowLayout", (int)m_windowLayout );
    m_settings->setValue( "DetailsLayout", (int)m_detailsLayout );
    m_settings->endGroup();

    m_settings->beginGroup( "ConnectionSettings" );
    m_settings->setValue( "ProxyType", (int)m_proxy.type() );
    m_settings->setValue( "ProxyHost", m_proxy.hostName() );
    m_settings->setValue( "ProxyPort", (int)m_proxy.port() );
    m_settings->setValue( "ProxyUser", m_proxy.user() );
    m_settings->setValue( "ProxyPassword", m_proxy.password() );
    m_settings->endGroup();

    m_settings->beginGroup( "AdvancedSettings" );
    m_settings->setValue( "UpdateInterval", m_updateInterval );
    m_settings->setValue( "DefaultAttachmentAction", (int)m_defaultAttachmentAction );
    m_settings->setValue( "AttachmentsCacheSize", m_attachmentsCacheSize );
    m_settings->setValue( "WarnAttachmentSize", m_warnAttachmentSize );
    m_settings->setValue( "WarnCommentSize", m_warnCommentSize );
    m_settings->endGroup();

    emit settingsChanged();
}

#if defined( Q_WS_WIN )

bool ConfigData::checkAutoStart()
{
    if ( m_isLocal )
        return false;

    QSettings autoStart( "HKEY_CURRENT_USER\\Software\\Microsoft\\Windows\\CurrentVersion\\Run", QSettings::NativeFormat );
    return autoStart.contains( "WebIssues" );
}

void ConfigData::setAutoStart( bool start )
{
    if ( m_isLocal )
        return;

    QSettings autoStart( "HKEY_CURRENT_USER\\Software\\Microsoft\\Windows\\CurrentVersion\\Run", QSettings::NativeFormat );
    if ( start )
        autoStart.setValue( "WebIssues", '"' + QDir::toNativeSeparators( QApplication::applicationFilePath() ) + '"' );
    else
        autoStart.remove( "WebIssues" );
}

#endif
