/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef PREFERENCESBATCH_H
#define PREFERENCESBATCH_H

#include "commands/abstractbatch.h"
#include "preferencesdata.h"
#include "batchjob.h"

class Reply;

/**
* Batch for loading preferences of a user.
*
* This batch sends a <tt>LIST PREFERENCES</tt> command for the given user.
* In order to load preferences for the current user, use UpdateBatch instead.
*/
class LoadPreferencesBatch : public AbstractBatch, private PreferencesData
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param userId Identifier of the user.
    */
    LoadPreferencesBatch( int userId );

    /**
    * Destructor.
    */
    ~LoadPreferencesBatch();

public:
    /**
    * Create the object storing the loaded preferences data.
    */
    PreferencesData* preferencesData() const;

public: // overrides
    Command* fetchNext();

private:
    typedef BatchJob<LoadPreferencesBatch> Job;
    typedef BatchJobQueue<LoadPreferencesBatch> JobQueue;

private:
    Command* listPreferencesJob( const Job& job );

private slots:
    void listPreferencesReply( const Reply& reply );

private:
    JobQueue m_queue;
};

/**
* Batch for setting preferences of a user.
*
* This batch sends the <tt>SET PREFERENCE</tt> commands with new preferences
* for the given user. If current user's preferences are changed, they are
* automatically updated.
*/
class SavePreferencesBatch : public AbstractBatch, private PreferencesData
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param userId Identifier of the user.
    * @param data The preferences to set.
    */
    SavePreferencesBatch( int userId, const PreferencesData& data );

    /**
    * Destructor.
    */
    ~SavePreferencesBatch();

public: // overrides
    Command* fetchNext();

private:
    Command* createSetCommand();

private slots:
    void setUpdate();

private:
    int m_userId;

    QMapIterator<QString, QString> m_iterator;

    bool m_update;
};

#endif
