/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#if defined( HAVE_OPENSSL )

#include "sslcertificatesdialog.h"

#include <QLayout>
#include <QLabel>
#include <QGroupBox>
#include <QPushButton>
#include <QListWidget>
#include <QDialogButtonBox>

#include "attributes/datetimehelper.h"
#include "widgets/propertypanel.h"
#include "iconloader.h"

SslCertificatesDialog::SslCertificatesDialog( QWidget* parent ) : QDialog( parent )
{
    setWindowTitle( tr( "Certificate Information" ) );

    QVBoxLayout* mainLayout = new QVBoxLayout( this );

    QGroupBox* listBox = new QGroupBox( tr( "Certification Path" ), this );
    QVBoxLayout* listLayout = new QVBoxLayout( listBox );

    m_list = new QListWidget( listBox );

    connect( m_list, SIGNAL( currentRowChanged( int ) ), this, SLOT( showCertificate( int ) ) );

    listLayout->addWidget( m_list );
    mainLayout->addWidget( listBox );

    QGroupBox* subjectBox = new QGroupBox( tr( "Issued To" ), this );
    QVBoxLayout* subjectLayout = new QVBoxLayout( subjectBox );
    subjectLayout->setMargin( 0 );

    m_subjectPanel = new PropertyPanel( subjectBox );
    m_subjectPanel->addProperty( "CN", tr( "Common Name" ) );
    m_subjectPanel->addProperty( "O", tr( "Organization" ) );
    m_subjectPanel->addProperty( "OU", tr( "Organizational Unit" ) );

    subjectLayout->addWidget( m_subjectPanel );
    mainLayout->addWidget( subjectBox );

    QGroupBox* issuerBox = new QGroupBox( tr( "Issued By" ), this );
    QVBoxLayout* issuerLayout = new QVBoxLayout( issuerBox );
    issuerLayout->setMargin( 0 );

    m_issuerPanel = new PropertyPanel( issuerBox );
    m_issuerPanel->addProperty( "CN", tr( "Common Name" ) );
    m_issuerPanel->addProperty( "O", tr( "Organization" ) );
    m_issuerPanel->addProperty( "OU", tr( "Organizational Unit" ) );

    issuerLayout->addWidget( m_issuerPanel );
    mainLayout->addWidget( issuerBox );

    QGroupBox* detailsBox = new QGroupBox( tr( "Certificate Details" ), this );
    QVBoxLayout* detailsLayout = new QVBoxLayout( detailsBox );
    detailsLayout->setMargin( 0 );

    m_detailsPanel = new PropertyPanel( detailsBox );
    m_detailsPanel->addProperty( "effectiveDate", tr( "Issued On" ) );
    m_detailsPanel->addProperty( "expiryDate", tr( "Expires On" ) );
    m_detailsPanel->addProperty( "SHA1", tr( "SHA1 Fingerprint" ) );
    m_detailsPanel->addProperty( "MD5", tr( "MD5 Fingerprint" ) );

    detailsLayout->addWidget( m_detailsPanel );
    mainLayout->addWidget( detailsBox );

    QDialogButtonBox* buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok, Qt::Horizontal, this );
    buttonBox->setCenterButtons( true );
    mainLayout->addWidget( buttonBox );

    connect( buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );

    resize( 450, 500 );
}

SslCertificatesDialog::~SslCertificatesDialog()
{
}

void SslCertificatesDialog::setCertificates( const QList<QSslCertificate>& certificates )
{
    m_certificates = certificates;

    m_list->clear();

    for ( int i = 0; i < certificates.count(); i++ ) {
        QListWidgetItem* item = new QListWidgetItem( m_list );
        item->setText( certificates.at( i ).subjectInfo( QSslCertificate::CommonName ) );
    }

    m_list->setCurrentRow( 0 );
}

void SslCertificatesDialog::showCertificate( int index )
{
    if ( index >= 0 && index < m_certificates.count() ) {
        QSslCertificate certificate = m_certificates.at( index );

        m_subjectPanel->setValue( "CN", certificate.subjectInfo( QSslCertificate::CommonName ) );
        m_subjectPanel->setValue( "O", certificate.subjectInfo( QSslCertificate::Organization ) );
        m_subjectPanel->setValue( "OU", certificate.subjectInfo( QSslCertificate::OrganizationalUnitName ) );

        m_issuerPanel->setValue( "CN", certificate.issuerInfo( QSslCertificate::CommonName ) );
        m_issuerPanel->setValue( "O", certificate.issuerInfo( QSslCertificate::Organization ) );
        m_issuerPanel->setValue( "OU", certificate.issuerInfo( QSslCertificate::OrganizationalUnitName ) );

        m_detailsPanel->setValue( "effectiveDate", formatDate( certificate.effectiveDate() ) );
        m_detailsPanel->setValue( "expiryDate", formatDate( certificate.expiryDate() ) );
        m_detailsPanel->setValue( "SHA1", formatBinary( certificate.digest( QCryptographicHash::Sha1 ) ) );
        m_detailsPanel->setValue( "MD5", formatBinary( certificate.digest( QCryptographicHash::Md5 ) ) );
    } else {
        m_subjectPanel->clearValues();
        m_issuerPanel->clearValues();
        m_detailsPanel->clearValues();
    }
}

QString SslCertificatesDialog::formatDate( const QDateTime& dateTime )
{
    if ( !dateTime.isValid() )
        return QString();
    return DateTimeHelper::formatDate( dateTime.date() );
}

QString SslCertificatesDialog::formatBinary( const QByteArray& binary )
{
    QString result;
    for ( int i = 0; i < binary.length(); i++ ) {
        if ( i > 0 )
            result += QLatin1Char( ':' );
        result += QString( "%1" ).arg( (int)(unsigned char)binary.at( i ), 2, 16, QLatin1Char( '0' ) );
    }
    return result;
}

#endif // defined( HAVE_OPENSSL )
