/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <librevenge-stream/librevenge-stream.h>
#include <libmwaw/libmwaw.hxx>
#include <libodfgen/libodfgen.hxx>

#include "OutputFileHelper.hxx"
#include "StringDocumentHandler.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "mwaw2odf"

class OdfOutputFileHelper : public OutputFileHelper
{
public:
	enum OdfType { Odg, Odp, Ods, Odt };
	OdfOutputFileHelper(const char *outFileName,const char *password) :
		OutputFileHelper(outFileName, password) {};
	~OdfOutputFileHelper() {};

	bool convertOdtDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdtGenerator collector;
		collector.registerEmbeddedObjectHandler("image/mwaw-odg", &handleEmbeddedMWAWGraphicObject);
		collector.registerEmbeddedObjectHandler("image/mwaw-ods", &handleEmbeddedMWAWSpreadsheetObject);
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, metaHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&metaHandler, ODF_META_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		if (MWAWDocument::MWAW_R_OK != MWAWDocument::parse(&input, &collector))
			return false;
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		static const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.text";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("meta.xml", metaHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}

	bool convertOdgDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdgGenerator collector;
		collector.registerEmbeddedObjectHandler("image/mwaw-odg", &handleEmbeddedMWAWGraphicObject);
		collector.registerEmbeddedObjectHandler("image/mwaw-ods", &handleEmbeddedMWAWSpreadsheetObject);
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, metaHandler, settingsHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&metaHandler, ODF_META_XML);
			collector.addDocumentHandler(&settingsHandler, ODF_SETTINGS_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		if (MWAWDocument::MWAW_R_OK != MWAWDocument::parse(&input, &collector))
			return false;
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		static const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.graphics";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("meta.xml", metaHandler.cstr()) ||
		        !writeChildFile("settings.xml", settingsHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}

	bool convertOdpDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdpGenerator collector;
		collector.registerEmbeddedObjectHandler("image/mwaw-odg", &handleEmbeddedMWAWGraphicObject);
		collector.registerEmbeddedObjectHandler("image/mwaw-ods", &handleEmbeddedMWAWSpreadsheetObject);
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, metaHandler, settingsHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&metaHandler, ODF_META_XML);
			collector.addDocumentHandler(&settingsHandler, ODF_SETTINGS_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		if (MWAWDocument::MWAW_R_OK != MWAWDocument::parse(&input, &collector))
			return false;
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		static const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.presentation";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("meta.xml", metaHandler.cstr()) ||
		        !writeChildFile("settings.xml", settingsHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}

	bool convertOdsDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdsGenerator collector;
		collector.registerEmbeddedObjectHandler("image/mwaw-odg", &handleEmbeddedMWAWGraphicObject);
		collector.registerEmbeddedObjectHandler("image/mwaw-ods", &handleEmbeddedMWAWSpreadsheetObject);
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, metaHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&metaHandler, ODF_META_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		if (MWAWDocument::MWAW_R_OK != MWAWDocument::parse(&input, &collector))
			return false;
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		static const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.spreadsheet";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("meta.xml", metaHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}

	static bool isSupportedFormat(librevenge::RVNGInputStream &input, OdfType &odfType)
	{
		MWAWDocument::Type type;
		MWAWDocument::Kind kind;
		MWAWDocument::Confidence confidence = MWAWDocument::isFileFormatSupported(&input, type, kind);
		if (confidence != MWAWDocument::MWAW_C_EXCELLENT)
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Mac Classic document.\n");
			return false;
		}
		if (kind == MWAWDocument::MWAW_K_SPREADSHEET || kind == MWAWDocument::MWAW_K_DATABASE)
			odfType = Ods;
		else if (kind == MWAWDocument::MWAW_K_DRAW || kind == MWAWDocument::MWAW_K_PAINT)
			odfType = Odg;
		else if (kind == MWAWDocument::MWAW_K_TEXT)
			odfType = Odt;
		else if (kind == MWAWDocument::MWAW_K_PRESENTATION)
			odfType = Odp;
		else
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Mac Classic text document.\n");
			return false;
		}

		return true;
	}
private:
	static bool handleEmbeddedMWAWGraphicObject(const librevenge::RVNGBinaryData &data, OdfDocumentHandler *pHandler,  const OdfStreamType streamType)
	{
		OdgGenerator exporter;
		exporter.addDocumentHandler(pHandler, streamType);
		return MWAWDocument::decodeGraphic(data, &exporter);
	}
	static bool handleEmbeddedMWAWSpreadsheetObject(const librevenge::RVNGBinaryData &data, OdfDocumentHandler *pHandler,  const OdfStreamType streamType)
	{
		OdsGenerator exporter;
		exporter.registerEmbeddedObjectHandler("image/mwaw-odg", &handleEmbeddedMWAWGraphicObject);
		exporter.addDocumentHandler(pHandler, streamType);
		return MWAWDocument::decodeSpreadsheet(data, &exporter);
	}
};

int printUsage(char const *name)
{
	UsageHelper usage(name, "converts Mac documents to ODF.");
	usage.addToDescription("If OUTPUT is omitted, the result is printed as Flat ODF to standard output.\n");
	usage.addStdoutOption();
	usage.addExtensionOption();
	return usage.printUsage();
}

int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(TOOLNAME);

	char *szInputFile = 0;
	char *szOutFile = 0;
	bool stdOutput = false;
	bool printExtension=false;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--stdout"))
			stdOutput = true;
		else if (!strcmp(argv[i], "--extension") || !strcmp(argv[i], "-x"))
			printExtension = true;
		else if (!strcmp(argv[i], "--help"))
			return printUsage(TOOLNAME);
		else if (!strcmp(argv[i], "--version"))
			return UsageHelper::printVersion(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile)
		return printUsage(TOOLNAME);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	OdfOutputFileHelper::OdfType type;
	librevenge::RVNGFileStream input(szInputFile);
	if (!OdfOutputFileHelper::isSupportedFormat(input, type))
		return 1;
	if (printExtension)
	{
		if (type == OdfOutputFileHelper::Odt)
			printf("odt\n");
		else if (type == OdfOutputFileHelper::Odg)
			printf("odg\n");
		else if (type == OdfOutputFileHelper::Odp)
			printf("odp\n");
		else if (type == OdfOutputFileHelper::Ods)
			printf("ods\n");
		else
			printf("unknown\n");
		return 0;
	}
	OdfOutputFileHelper helper(szOutFile, 0);
	bool ok=false;
	if (type == OdfOutputFileHelper::Odt)
		ok=helper.convertOdtDocument(input, szOutFile==0);
	else if (type == OdfOutputFileHelper::Odg)
		ok=helper.convertOdgDocument(input, szOutFile==0);
	else if (type == OdfOutputFileHelper::Odp)
		ok=helper.convertOdpDocument(input, szOutFile==0);
	else if (type == OdfOutputFileHelper::Ods)
		ok=helper.convertOdsDocument(input, szOutFile==0);
	if (!ok)
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
