/*
 * Copyright (c) 2007-2010 by The Broad Institute, Inc. and the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * This software is licensed under the terms of the GNU Lesser General Public License (LGPL), Version 2.1 which
 * is available at http://www.opensource.org/licenses/lgpl-2.1.php.
 *
 * THE SOFTWARE IS PROVIDED "AS IS." THE BROAD AND MIT MAKE NO REPRESENTATIONS OR WARRANTIES OF
 * ANY KIND CONCERNING THE SOFTWARE, EXPRESS OR IMPLIED, INCLUDING, WITHOUT LIMITATION, WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, NONINFRINGEMENT, OR THE ABSENCE OF LATENT
 * OR OTHER DEFECTS, WHETHER OR NOT DISCOVERABLE.  IN NO EVENT SHALL THE BROAD OR MIT, OR THEIR
 * RESPECTIVE TRUSTEES, DIRECTORS, OFFICERS, EMPLOYEES, AND AFFILIATES BE LIABLE FOR ANY DAMAGES OF
 * ANY KIND, INCLUDING, WITHOUT LIMITATION, INCIDENTAL OR CONSEQUENTIAL DAMAGES, ECONOMIC
 * DAMAGES OR INJURY TO PROPERTY AND LOST PROFITS, REGARDLESS OF WHETHER THE BROAD OR MIT SHALL
 * BE ADVISED, SHALL HAVE OTHER REASON TO KNOW, OR IN FACT SHALL KNOW OF THE POSSIBILITY OF THE
 * FOREGOING.
 */
package org.broad.igv.sam;

import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMSequenceRecord;
import net.sf.samtools.util.CloseableIterator;
import org.apache.log4j.Logger;
import org.broad.igv.Globals;
import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.Genome;
import org.broad.igv.sam.AlignmentTrack.SortOption;
import org.broad.igv.sam.reader.AlignmentQueryReader;
import org.broad.igv.sam.reader.SamListReader;
import org.broad.igv.sam.reader.SamQueryReaderFactory;
import org.broad.igv.session.ViewContext;
import org.broad.igv.track.MultiFileWrapper;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.WaitCursorManager;
import org.broad.igv.util.ResourceLocator;

import javax.swing.*;
import java.io.IOException;
import java.util.*;

public class AlignmentDataManager {

    private static Logger log = Logger.getLogger(AlignmentDataManager.class);
    private AlignmentInterval loadedInterval = null;
    private List<Row> alignmentRows;
    HashMap<String, String> chrMappings = new HashMap();
    boolean isLoading = false;
    AlignmentQueryReader reader;

    CoverageTrack coverageTrack;

    private static final int DEFAULT_DEPTH = 10;


    public static AlignmentDataManager getDataManager(ResourceLocator locator) {
        return new AlignmentDataManager(locator);
    }

    public AlignmentQueryReader getReader() {
        return reader;
    }

    private AlignmentDataManager(ResourceLocator locator) {

        if (locator.getPath().endsWith(".sam.list")) {
            MultiFileWrapper mfw = MultiFileWrapper.parse(locator);
            reader = new SamListReader(mfw.getLocators());
        } else {
            reader = SamQueryReaderFactory.getReader(locator);
        }
        initChrMap();
    }

    private void initChrMap() {
        Genome genome = ViewContext.getInstance().getGenome();
        if (genome != null) {
            Set<String> seqNames = reader.getSequenceNames();
            if (seqNames != null) {
                for (String chr : seqNames) {
                    String alias = genome.getChromosomeAlias(chr);
                    chrMappings.put(alias, chr);
                }

            }
        }
    }


    public boolean hasIndex() {
        return reader.hasIndex();
    }

    public int getMaxDepth() {
        return loadedInterval == null ? DEFAULT_DEPTH :
                (loadedInterval.getMaxCount() == 0 ? DEFAULT_DEPTH : loadedInterval.getMaxCount());
    }

    public void setCoverageTrack(CoverageTrack coverageTrack) {
        this.coverageTrack = coverageTrack;
    }

    public AlignmentInterval getLoadedInterval() {
        return loadedInterval;
    }


    /**
     * Sort alignment rows such that alignments that intersect from the
     * center appear left to right by start position
     */
    public void sortRows(SortOption option) {
        if (getAlignmentRows() == null) {
            return;
        }

        double center = ViewContext.getInstance().getCenter();
        for (Row row : getAlignmentRows()) {
            if (option == SortOption.NUCELOTIDE) {

            }
            row.updateScore(option, center, loadedInterval);
        }

        Collections.sort(getAlignmentRows(), new Comparator<Row>() {

            public int compare(Row arg0, Row arg1) {
                if (arg0.getScore() > arg1.getScore()) {
                    return 1;
                } else if (arg0.getScore() > arg1.getScore()) {
                    return -1;
                }
                return 0;
            }
        });
    }

    public void packAlignments() {
        RowIterator iter = new RowIterator();
        final PreferenceManager.SAMPreferences prefs = PreferenceManager.getInstance().getSAMPreferences();
        final int maxLevels = prefs.getMaxLevels();
        final int qualityThreshold = prefs.getQualityThreshold();

        alignmentRows = AlignmentLoader.loadAndPackAlignments(
                iter,
                prefs.isShowDuplicates(),
                qualityThreshold,
                maxLevels,
                getLoadedInterval().getEnd());
    }


    public synchronized void preloadData(final String chr, final int start, final int end, int zoom) {

        final PreferenceManager.SAMPreferences prefs = PreferenceManager.getInstance().getSAMPreferences();
        final float maxWindow = prefs.getMaxVisibleRange() * 1000;
        final int maxLevels = prefs.getMaxLevels();
        final int qualityThreshold = prefs.getQualityThreshold();


        int window = end - start;
        if (chr.equals(Globals.CHR_ALL) || (window > maxWindow)) {
            return;
        }

        // If the requested interval is outside the currently loaded range load
        final String genomeId = ViewContext.getInstance().getGenomeId();
        if (getLoadedInterval() == null || !loadedInterval.contains(genomeId, chr, start, end) && !isLoading) {
            isLoading = true;

            WaitCursorManager.CursorToken token = WaitCursorManager.showWaitCursor();
            try {


                long t0 = System.currentTimeMillis();

                //if(log.isDebugEnabled()) {
                //    log.debug("Enter preloadData: " + chr + ":" + start + "-" + end);
                //    for(StackTraceElement tmp : Thread.currentThread().getStackTrace()) {
                //         log.debug("\t" + tmp);
                //    }
                //}

                alignmentRows = new ArrayList(maxLevels);

                // Expand start and end to facilitate panning, but by no more than
                // 1 screen or 8kb, whichever is less
                // DON'T expand mitochondria

                int expandLength = isMitochondria(chr) ? 0 : Math.min(8000, end - start) / 2;
                int intervalStart = Math.max(0, start - expandLength);
                int intervalEnd = end + expandLength;
                CloseableIterator<Alignment> iter = null;
                try {

                    String sequence = chrMappings.containsKey(chr) ? chrMappings.get(chr) : chr;

                    iter = reader.query(sequence, intervalStart, intervalEnd, false);

                    alignmentRows = AlignmentLoader.loadAndPackAlignments(iter,
                            prefs.isShowDuplicates(),
                            qualityThreshold,
                            maxLevels,
                            intervalEnd);

                    setLoadedInterval(new AlignmentInterval(genomeId, chr, intervalStart, intervalEnd));

                    // Caclulate coverage.  TODO -- refactor/move this
                    for (AlignmentDataManager.Row row : getAlignmentRows()) {
                        for (Alignment a : row.alignments) {
                            getLoadedInterval().incCounts(a);
                        }
                    }


                    if (coverageTrack != null) {
                        coverageTrack.rescale();
                    }

                    IGVMainFrame.getInstance().doResizeTrackPanels();
                    IGVMainFrame.getInstance().repaintDataAndHeaderPanels();


                } catch (Exception exception) {
                    log.error("Error loading alignments", exception);
                    JOptionPane.showMessageDialog(IGVMainFrame.getInstance(), "Error reading file: " + exception.getMessage());
                } finally {
                    isLoading = false;
                    if (iter != null) {
                        iter.close();
                    }
                }

            }
            finally {
                WaitCursorManager.removeWaitCursor(token);
            }

            // Don't block the swing dispatch thread
            //if (SwingUtilities.isEventDispatchThread()) {
            //    try {
            //
            //        LongRunningTask.submit(runnable).get();   // <= This will force wait until the task is complete
            //    } catch (Exception ex) {
            //        log.error("Error loading alignments", ex);
            //    }
            //} else {
            //    runnable.run();
            //}


        }
    }

    private boolean isMitochondria(String chr) {
        return chr.equals("M") || chr.equals("chrM") ||
                chr.equals("MT") || chr.equals("chrMT");
    }

    /**
     * @return the alignmentRows
     */
    public List<Row> getAlignmentRows() {
        return alignmentRows;
    }


    /**
     * @param loadedInterval the loadedInterval to set
     */
    public void setLoadedInterval(AlignmentInterval loadedInterval) {
        this.loadedInterval = loadedInterval;
    }


    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        if (reader != null) {
            try {
                reader.close();
            } catch (IOException ex) {
                log.error("Error closing AlignmentQueryReader. ", ex);
            }
        }

    }

    public static class Row {
        int nextIdx;
        private double score = 0;
        List<Alignment> alignments;
        private int start;
        private int lastEnd;

        public Row() {
            nextIdx = 0;
            this.alignments = new ArrayList(100);
        }

        public void addAlignment(Alignment alignment) {
            if (alignments.isEmpty()) {
                this.start = alignment.getStart();
            }
            alignments.add(alignment);
            lastEnd = alignment.getEnd();

        }

        public void updateScore(SortOption option, double center, AlignmentInterval loadedInterval) {

            int adjustedCenter = (int) center;
            Alignment centerAlignment = getFeatureContaining(alignments, adjustedCenter);
            if (centerAlignment == null) {
                setScore(Double.MAX_VALUE);
            } else {
                switch (option) {
                    case START:
                        setScore(centerAlignment.getStart());
                        break;
                    case STRAND:
                        setScore(centerAlignment.isNegativeStrand() ? -1 : 1);
                        break;
                    case NUCELOTIDE:
                        byte base = centerAlignment.getBase(adjustedCenter);
                        byte ref = loadedInterval.getReference(adjustedCenter);
                        if (base == 'N' || base == 'n') {
                            setScore(Integer.MAX_VALUE - 1);
                        } else if (base == ref) {
                            setScore(Integer.MAX_VALUE);
                        } else {
                            int count = loadedInterval.getCount(adjustedCenter, base);
                            byte phred = centerAlignment.getPhred(adjustedCenter);
                            float score = -(count + (phred / 100.0f));
                            setScore(score);
                        }
                        break;
                    case QUALITY:
                        setScore(-centerAlignment.getMappingQuality());
                        break;
                    case SAMPLE:
                        String sample = centerAlignment.getSample();
                        score = sample == null ? 0 : sample.hashCode();
                        setScore(score);
                        break;
                    case READ_GROUP:
                        String readGroup = centerAlignment.getReadGroup();
                        score = readGroup == null ? 0 : readGroup.hashCode();
                        setScore(score);
                        break;
                }
            }
        }

        // Used for iterating over all alignments, e.g. for packing

        public Alignment nextAlignment() {
            if (nextIdx < alignments.size()) {
                Alignment tmp = alignments.get(nextIdx);
                nextIdx++;
                return tmp;
            } else {
                return null;
            }
        }

        public int getNextStartPos() {
            if (nextIdx < alignments.size()) {
                return alignments.get(nextIdx).getAlignmentStart();
            } else {
                return Integer.MAX_VALUE;
            }
        }

        public boolean hasNext() {
            return nextIdx < alignments.size();
        }

        public void resetIdx() {
            nextIdx = 0;
        }

        /**
         * @return the score
         */
        public double getScore() {
            return score;
        }

        /**
         * @param score the score to set
         */
        public void setScore(double score) {
            this.score = score;
        }

        public int getStart() {
            return start;
        }

        public int getLastEnd() {
            return lastEnd;
        }
    }

    /**
     * An alignment iterator that iterates over packed rows.  Used for
     * "repacking".   Using the iterator avoids the need to copy alignments
     * from the rows
     */
    class RowIterator implements CloseableIterator<Alignment> {

        PriorityQueue<Row> rows;
        Alignment nextAlignment;

        RowIterator() {
            rows = new PriorityQueue(5, new Comparator<Row>() {

                public int compare(Row o1, Row o2) {
                    return o1.getNextStartPos() - o2.getNextStartPos();
                }
            });

            for (Row r : getAlignmentRows()) {
                r.resetIdx();
                rows.add(r);
            }

            advance();
        }

        public void close() {
            // Ignored
        }

        public boolean hasNext() {
            return nextAlignment != null;
        }

        public Alignment next() {
            Alignment tmp = nextAlignment;
            if (tmp != null) {
                advance();
            }
            return tmp;
        }

        private void advance() {

            nextAlignment = null;
            Row nextRow = null;
            while (nextAlignment == null && !rows.isEmpty()) {
                while ((nextRow = rows.poll()) != null) {
                    if (nextRow.hasNext()) {
                        nextAlignment = nextRow.nextAlignment();
                        break;
                    }
                }
            }
            if (nextRow != null && nextAlignment != null) {
                rows.add(nextRow);
            }
        }

        public void remove() {
            // ignore
        }
    }

    private static Alignment getFeatureContaining(
            List<Alignment> features, int right) {

        int leftBounds = 0;
        int rightBounds = features.size() - 1;
        int idx = features.size() / 2;
        int lastIdx = -1;

        while (idx != lastIdx) {
            lastIdx = idx;
            Alignment f = features.get(idx);
            if (f.contains(right)) {
                return f;
            }

            if (f.getStart() > right) {
                rightBounds = idx;
                idx = (leftBounds + idx) / 2;
            } else {
                leftBounds = idx;
                idx = (rightBounds + idx) / 2;

            }

        }
        // Check the extremes
        if (features.get(0).contains(right)) {
            return features.get(0);
        }

        if (features.get(rightBounds).contains(right)) {
            return features.get(rightBounds);
        }

        return null;
    }
}

