/*
 * Copyright (c) 2007-2010 by The Broad Institute, Inc. and the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * This software is licensed under the terms of the GNU Lesser General Public License (LGPL), Version 2.1 which
 * is available at http://www.opensource.org/licenses/lgpl-2.1.php.
 *
 * THE SOFTWARE IS PROVIDED "AS IS." THE BROAD AND MIT MAKE NO REPRESENTATIONS OR WARRANTIES OF
 * ANY KIND CONCERNING THE SOFTWARE, EXPRESS OR IMPLIED, INCLUDING, WITHOUT LIMITATION, WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, NONINFRINGEMENT, OR THE ABSENCE OF LATENT
 * OR OTHER DEFECTS, WHETHER OR NOT DISCOVERABLE.  IN NO EVENT SHALL THE BROAD OR MIT, OR THEIR
 * RESPECTIVE TRUSTEES, DIRECTORS, OFFICERS, EMPLOYEES, AND AFFILIATES BE LIABLE FOR ANY DAMAGES OF
 * ANY KIND, INCLUDING, WITHOUT LIMITATION, INCIDENTAL OR CONSEQUENTIAL DAMAGES, ECONOMIC
 * DAMAGES OR INJURY TO PROPERTY AND LOST PROFITS, REGARDLESS OF WHETHER THE BROAD OR MIT SHALL
 * BE ADVISED, SHALL HAVE OTHER REASON TO KNOW, OR IN FACT SHALL KNOW OF THE POSSIBILITY OF THE
 * FOREGOING.
 */
/*
 * FeatureTrackH5.java
 *
 * Created on November 12, 2007, 8:22 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.broad.igv.track;

import org.apache.log4j.Logger;
import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.FeatureUtils;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.renderer.DataRange;
import org.broad.igv.renderer.DataRenderer;
import org.broad.igv.renderer.XYPlotRenderer;
import org.broad.igv.session.ViewContext;
import org.broad.igv.util.ResourceLocator;

import java.awt.*;
import java.util.List;
import java.util.Map;

/**
 * Represents a track of numeric data
 *
 * @author jrobinso
 */
public abstract class DataTrack extends AbstractTrack {

    private static Logger log = Logger.getLogger(DataTrack.class);
    private DataRenderer renderer;
    private boolean autoscale;


    public DataTrack(ResourceLocator locator, String id, String name) {
        super(locator, id, name);
        autoscale = PreferenceManager.getInstance().getChartPreferences().isAutoscale();
    }

    public boolean isAutoscale() {
        return autoscale;
    }

    public void setAutoscale(boolean autoscale) {
        this.autoscale = autoscale;
    }

    public void preloadData(String chrName, int start, int end, int zoom) {
        getSummaryScores(chrName, start, end, zoom);
    }

    public void render(RenderContext context, Rectangle rect) {

        List<LocusScore> scores = getSummaryScores(context.getChr(),(int) context.getOrigin(),
                (int) context.getEndLocation() + 1, context.getZoom());

        List<LocusScore> inViewScores = scores;
        if (autoscale) {
            double origin = context.getOrigin();
            double end = origin + context.getScale() * rect.width;
            InViewInterval inter = computeScale(origin, end, scores);
            if (inter.endIdx > inter.startIdx) {
                inViewScores = scores.subList(inter.startIdx, inter.endIdx);

                DataRange dr = getDataRange();
                float min = Math.min(0, inter.dataMin);
                float base = Math.max(min, dr.getBaseline());
                float max = inter.dataMax;
                // Pathological case where min ~= max  (no data in view)
                if(max - min <= (2*Float.MIN_VALUE)) {
                    max = min + 1;
                }

                DataRange newDR = new DataRange(min, base, max, dr.isDrawBaseline());
                newDR.setType(dr.getType());
                setDataRange(newDR);
            }

        }

        getRenderer().render(this, inViewScores, context, rect);
    }


    public void setRendererClass(Class rc) {
        try {
            renderer = (DataRenderer) rc.newInstance();
        } catch (Exception ex) {
            log.error("Error instatiating renderer ", ex);
        }
    }


    public DataRenderer getRenderer() {
        if (renderer == null) {
            setRendererClass(getDefaultRendererClass());
        }
        return renderer;
    }


    public String getValueStringAt(String chr, double position, int y) {
        StringBuffer buf = new StringBuffer();
        buf.append(getName() + "<br>");
        if ((getDataRange() != null) && (getRenderer() instanceof XYPlotRenderer)) {
            buf.append("Data scale: " + getDataRange().getMinimum() + " - " + getDataRange().getMaximum() + "<br>");
        }

        LocusScore score = getLocusScoreAt(chr, position);
        buf.append( (score == null) ? "" : score.getValueString(position, getWindowFunction()));
        return buf.toString();
    }


    public LocusScore getLocusScoreAt(String chr, double position) {
        int zoom = Math.max(0, ViewContext.getInstance().getZoom());
        List<LocusScore> scores = getSummaryScores(chr, (int) position - 10, (int) position + 10, zoom);

        // give a 2 pixel window, otherwise very narrow features will be missed.
        double bpPerPixel = ViewContext.getInstance().getScale();
        double minWidth = 2 * bpPerPixel;    /* * */

        if (scores == null) {
            return null;
        } else {
            return FeatureUtils.getFeatureAt(position, minWidth, scores);
        }
    }


    abstract public List<LocusScore> getSummaryScores(String chr, int startLocation,
                                                      int endLocation, int zoom);

    public boolean handleClick(int x, int y) {

        // Ignore
        return false;
    }

    @Override
    public void setColor(Color color) {
        super.setColor(color);
    }


    @Override
    public void setAltColor(Color color) {
        super.setAltColor(color);

    }


    @Override
    public void setMidColor(Color color) {
        super.setMidColor(color);

    }

    public boolean isRegionScoreType(RegionScoreType type) {
        return ((getTrackType() == TrackType.GENE_EXPRESSION) && (type == RegionScoreType.EXPRESSION)) ||
                (((getTrackType() == TrackType.COPY_NUMBER) || (getTrackType() == TrackType.CNV) ||
                        (getTrackType() == TrackType.ALLELE_SPECIFIC_COPY_NUMBER)) &&
                        ((type == RegionScoreType.AMPLIFICATION) ||
                                (type == RegionScoreType.DELETION))) ||
                (type == RegionScoreType.SCORE);
    }


    private InViewInterval computeScale(double origin, double end, List<LocusScore> scores) {

        InViewInterval interval = new InViewInterval();

        if (scores.size() == 1) {
            interval.dataMax = Math.max(0, scores.get(0).getScore());
            interval.dataMin = Math.min(0, scores.get(0).getScore());
        } else {
            interval.startIdx = 0;
            interval.endIdx = scores.size();
            for (int i = 1; i < scores.size(); i++) {
                if (scores.get(i).getEnd() >= origin) {
                    interval.startIdx = i - 1;
                    break;
                }
            }

            for (int i = interval.startIdx + 1; i < scores.size(); i++) {
                LocusScore locusScore = scores.get(i);
                interval.dataMax = Math.max(interval.dataMax, locusScore.getScore());
                interval.dataMin = Math.min(interval.dataMin, locusScore.getScore());
                if (locusScore.getStart() > end) {
                    interval.endIdx = i;
                    break;
                }
            }
        }

        return interval;
    }

    @Override
    public Map<String, String> getPersistentState() {
        Map<String, String> properties = super.getPersistentState();
        properties.put("autoscale", String.valueOf(autoscale));
        return properties;
    }


    @Override
    public void restorePersistentState(Map<String, String> attributes) {
        super.restorePersistentState(attributes);
        String as = attributes.get("autoscale");
        if (as != null) {
            try {
                autoscale = Boolean.parseBoolean(as);

            }
            catch (Exception e) {
                log.error("Error restoring session.  Invalid autoscale value: " + autoscale);

            }
        }
    }


    class InViewInterval {
        int startIdx;
        int endIdx;
        float dataMax = 0;
        float dataMin = 0;
    }

}
