/*
 * Copyright (c) 2007-2011 by The Broad Institute, Inc. and the Massachusetts Institute of
 * Technology.  All Rights Reserved.
 *
 * This software is licensed under the terms of the GNU Lesser General Public License (LGPL),
 * Version 2.1 which is available at http://www.opensource.org/licenses/lgpl-2.1.php.
 *
 * THE SOFTWARE IS PROVIDED "AS IS." THE BROAD AND MIT MAKE NO REPRESENTATIONS OR
 * WARRANTES OF ANY KIND CONCERNING THE SOFTWARE, EXPRESS OR IMPLIED, INCLUDING,
 * WITHOUT LIMITATION, WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, NONINFRINGEMENT, OR THE ABSENCE OF LATENT OR OTHER DEFECTS, WHETHER
 * OR NOT DISCOVERABLE.  IN NO EVENT SHALL THE BROAD OR MIT, OR THEIR RESPECTIVE
 * TRUSTEES, DIRECTORS, OFFICERS, EMPLOYEES, AND AFFILIATES BE LIABLE FOR ANY DAMAGES
 * OF ANY KIND, INCLUDING, WITHOUT LIMITATION, INCIDENTAL OR CONSEQUENTIAL DAMAGES,
 * ECONOMIC DAMAGES OR INJURY TO PROPERTY AND LOST PROFITS, REGARDLESS OF WHETHER
 * THE BROAD OR MIT SHALL BE ADVISED, SHALL HAVE OTHER REASON TO KNOW, OR IN FACT
 * SHALL KNOW OF THE POSSIBILITY OF THE FOREGOING.
 */

package org.broad.igv.util.sigma;

import org.broad.igv.track.TribbleFeatureSource;
import org.broad.tribble.Feature;
import org.broad.tribble.iterators.CloseableTribbleIterator;
import org.broad.tribble.util.variantcontext.Genotype;
import org.broad.tribble.util.variantcontext.VariantContext;
import org.broad.tribble.vcf.VCFHeader;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.*;

/**
 * @author jrobinso
 * @date Feb 3, 2011
 */
public class AlleleRatioCounter {

    static String chr = "chr1";
    static int roiStart = 152668595;
    static int roiEnd = 154746948;

    Collection<String> samples;
    TribbleFeatureSource source;
    VCFHeader header;


    public static void main(String[] args) throws IOException {

        AlleleRatioCounter counter = new AlleleRatioCounter(args[0]);
        counter.walkVariants(null, null);
    }

    public AlleleRatioCounter(String path) throws IOException {


        source = new TribbleFeatureSource(path);

        header = (VCFHeader) source.getHeader();

        samples = header.getGenotypeSamples();


    }


    public void walkVariants(PrintWriter bedWriter, PrintWriter excelWriter) throws IOException {


        Map<String, PrintWriter> writers = new HashMap();


        CloseableTribbleIterator<Feature> features = source.getFeatures(chr, roiStart, roiEnd);

        for (Feature feature : features) {


            VariantContext variant = (VariantContext) feature;
            int pos = variant.getStart();

            // Loop through the families
            for (String s : samples) {


                Genotype genotype = variant.getGenotype(s);

                if (genotype.isHet()) {

                    String ad = genotype.getAttributeAsString("AD");
                    String [] tmp = ad.split(",");
                    int varAllele = Integer.parseInt(tmp[1]);
                    double refAllele = Math.max(0.0001, Integer.parseInt(tmp[0])); 
                    float ratio = (float) Math.log10(varAllele / refAllele);
                    PrintWriter pw = writers.get(s);
                    if(pw == null) {
                        pw = new PrintWriter(new BufferedWriter(new FileWriter(s + ".alleleRatio.txt")));
                        writers.put(s, pw);
                    }
                    pw.println(pos + "\t" + ratio);

                }
            }
        }
        for(PrintWriter pw : writers.values()) {
            pw.close();
        }
    }


}
