/*
 * Decompiled with CFR 0.152.
 */
package picard.sam;

import htsjdk.samtools.ReservedTagConstants;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileWriter;
import htsjdk.samtools.SAMFileWriterFactory;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMUtils;
import htsjdk.samtools.fastq.FastqReader;
import htsjdk.samtools.fastq.FastqRecord;
import htsjdk.samtools.util.FastqQualityFormat;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Iso8601Date;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.QualityEncodingDetector;
import htsjdk.samtools.util.SolexaQualityConverter;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.Option;
import picard.cmdline.Usage;

public class FastqToSam
extends CommandLineProgram {
    private static final Log LOG = Log.getInstance(FastqToSam.class);
    @Usage
    public String USAGE = "Extracts read sequences and qualities from the input fastq file and writes them into the output file in unaligned BAM format. Input files can be in GZip format (end in .gz).\n";
    @Option(shortName="F1", doc="Input fastq file (optionally gzipped) for single end data, or first read in paired end data.")
    public File FASTQ;
    @Option(shortName="F2", doc="Input fastq file (optionally gzipped) for the second read of paired end data.", optional=true)
    public File FASTQ2;
    @Option(shortName="V", doc="A value describing how the quality values are encoded in the fastq.  Either Solexa for pre-pipeline 1.3 style scores (solexa scaling + 66), Illumina for pipeline 1.3 and above (phred scaling + 64) or Standard for phred scaled scores with a character shift of 33.  If this value is not specified, the quality format will be detected automatically.", optional=true)
    public FastqQualityFormat QUALITY_FORMAT;
    @Option(doc="Output SAM/BAM file. ", shortName="O")
    public File OUTPUT;
    @Option(shortName="RG", doc="Read group name")
    public String READ_GROUP_NAME = "A";
    @Option(shortName="SM", doc="Sample name to insert into the read group header")
    public String SAMPLE_NAME;
    @Option(shortName="LB", doc="The library name to place into the LB attribute in the read group header", optional=true)
    public String LIBRARY_NAME;
    @Option(shortName="PU", doc="The platform unit (often run_barcode.lane) to insert into the read group header", optional=true)
    public String PLATFORM_UNIT;
    @Option(shortName="PL", doc="The platform type (e.g. illumina, solid) to insert into the read group header", optional=true)
    public String PLATFORM;
    @Option(shortName="CN", doc="The sequencing center from which the data originated", optional=true)
    public String SEQUENCING_CENTER;
    @Option(shortName="PI", doc="Predicted median insert size, to insert into the read group header", optional=true)
    public Integer PREDICTED_INSERT_SIZE;
    @Option(doc="Comment(s) to include in the merged output file's header.", optional=true, shortName="CO")
    public List<String> COMMENT = new ArrayList<String>();
    @Option(shortName="DS", doc="Inserted into the read group header", optional=true)
    public String DESCRIPTION;
    @Option(shortName="DT", doc="Date the run was produced, to insert into the read group header", optional=true)
    public Iso8601Date RUN_DATE;
    @Option(shortName="SO", doc="The sort order for the output sam/bam file.")
    public SAMFileHeader.SortOrder SORT_ORDER = SAMFileHeader.SortOrder.queryname;
    @Option(doc="Minimum quality allowed in the input fastq.  An exception will be thrown if a quality is less than this value.")
    public int MIN_Q = 0;
    @Option(doc="Maximum quality allowed in the input fastq.  An exception will be thrown if a quality is greater than this value.")
    public int MAX_Q = 93;
    @Option(doc="If true and this is an unpaired fastq any occurance of '/1' will be removed from the end of a read name.")
    public Boolean STRIP_UNPAIRED_MATE_NUMBER = false;
    @Option(doc="Allow (and ignore) empty lines")
    public Boolean ALLOW_AND_IGNORE_EMPTY_LINES = false;
    private static final SolexaQualityConverter solexaQualityConverter = SolexaQualityConverter.getSingleton();

    public static void main(String[] argv) {
        System.exit(new FastqToSam().instanceMain(argv));
    }

    @Override
    protected int doWork() {
        QualityEncodingDetector detector = new QualityEncodingDetector();
        FastqReader reader = new FastqReader(this.FASTQ, this.ALLOW_AND_IGNORE_EMPTY_LINES.booleanValue());
        if (this.FASTQ2 == null) {
            detector.add(10000L, new FastqReader[]{reader});
        } else {
            FastqReader reader2 = new FastqReader(this.FASTQ2, this.ALLOW_AND_IGNORE_EMPTY_LINES.booleanValue());
            detector.add(10000L, new FastqReader[]{reader, reader2});
            reader2.close();
        }
        reader.close();
        this.QUALITY_FORMAT = detector.generateBestGuess(QualityEncodingDetector.FileContext.FASTQ, this.QUALITY_FORMAT);
        if (detector.isDeterminationAmbiguous()) {
            LOG.warn(new Object[]{"Making ambiguous determination about fastq's quality encoding; more than one format possible based on observed qualities."});
        }
        LOG.info(new Object[]{String.format("Auto-detected quality format as: %s.", this.QUALITY_FORMAT)});
        int readCount = this.FASTQ2 == null ? this.doUnpaired() : this.doPaired();
        LOG.info(new Object[]{"Processed " + readCount + " fastq reads"});
        return 0;
    }

    protected int doUnpaired() {
        IOUtil.assertFileIsReadable((File)this.FASTQ);
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        FastqReader freader = new FastqReader(this.FASTQ, this.ALLOW_AND_IGNORE_EMPTY_LINES.booleanValue());
        SAMFileHeader header = this.createFileHeader();
        SAMFileWriter writer = new SAMFileWriterFactory().makeSAMOrBAMWriter(header, false, this.OUTPUT);
        int readCount = 0;
        ProgressLogger progress = new ProgressLogger(LOG);
        while (freader.hasNext()) {
            FastqRecord frec = freader.next();
            SAMRecord srec = this.createSamRecord(header, this.getReadName(frec.getReadHeader(), false), frec, false);
            srec.setReadPairedFlag(false);
            writer.addAlignment(srec);
            progress.record(srec);
            ++readCount;
        }
        writer.close();
        return readCount;
    }

    protected int doPaired() {
        IOUtil.assertFileIsReadable((File)this.FASTQ);
        IOUtil.assertFileIsReadable((File)this.FASTQ2);
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        FastqReader freader1 = new FastqReader(this.FASTQ, this.ALLOW_AND_IGNORE_EMPTY_LINES.booleanValue());
        FastqReader freader2 = new FastqReader(this.FASTQ2, this.ALLOW_AND_IGNORE_EMPTY_LINES.booleanValue());
        SAMFileHeader header = this.createFileHeader();
        SAMFileWriter writer = new SAMFileWriterFactory().makeSAMOrBAMWriter(header, false, this.OUTPUT);
        int readCount = 0;
        ProgressLogger progress = new ProgressLogger(LOG);
        while (freader1.hasNext() && freader2.hasNext()) {
            FastqRecord frec1 = freader1.next();
            FastqRecord frec2 = freader2.next();
            String frec1Name = this.getReadName(frec1.getReadHeader(), true);
            String frec2Name = this.getReadName(frec2.getReadHeader(), true);
            String baseName = this.getBaseName(frec1Name, frec2Name, freader1, freader2);
            SAMRecord srec1 = this.createSamRecord(header, baseName, frec1, true);
            srec1.setFirstOfPairFlag(true);
            srec1.setSecondOfPairFlag(false);
            writer.addAlignment(srec1);
            progress.record(srec1);
            SAMRecord srec2 = this.createSamRecord(header, baseName, frec2, true);
            srec2.setFirstOfPairFlag(false);
            srec2.setSecondOfPairFlag(true);
            writer.addAlignment(srec2);
            progress.record(srec2);
            ++readCount;
        }
        writer.close();
        if (freader1.hasNext() || freader2.hasNext()) {
            throw new PicardException("Input paired fastq files must be the same length");
        }
        return readCount;
    }

    private SAMRecord createSamRecord(SAMFileHeader header, String baseName, FastqRecord frec, boolean paired) {
        SAMRecord srec = new SAMRecord(header);
        srec.setReadName(baseName);
        srec.setReadString(frec.getReadString());
        srec.setReadUnmappedFlag(true);
        srec.setAttribute(ReservedTagConstants.READ_GROUP_ID, (Object)this.READ_GROUP_NAME);
        byte[] quals = StringUtil.stringToBytes((String)frec.getBaseQualityString());
        this.convertQuality(quals, this.QUALITY_FORMAT);
        for (byte qual : quals) {
            int uQual = qual & 0xFF;
            if (uQual >= this.MIN_Q && uQual <= this.MAX_Q) continue;
            throw new PicardException("Base quality " + uQual + " is not in the range " + this.MIN_Q + ".." + this.MAX_Q + " for read " + frec.getReadHeader());
        }
        srec.setBaseQualities(quals);
        if (paired) {
            srec.setReadPairedFlag(true);
            srec.setMateUnmappedFlag(true);
        }
        return srec;
    }

    private SAMFileHeader createFileHeader() {
        SAMReadGroupRecord rgroup = new SAMReadGroupRecord(this.READ_GROUP_NAME);
        rgroup.setSample(this.SAMPLE_NAME);
        if (this.LIBRARY_NAME != null) {
            rgroup.setLibrary(this.LIBRARY_NAME);
        }
        if (this.PLATFORM != null) {
            rgroup.setPlatform(this.PLATFORM);
        }
        if (this.PLATFORM_UNIT != null) {
            rgroup.setPlatformUnit(this.PLATFORM_UNIT);
        }
        if (this.SEQUENCING_CENTER != null) {
            rgroup.setSequencingCenter(this.SEQUENCING_CENTER);
        }
        if (this.PREDICTED_INSERT_SIZE != null) {
            rgroup.setPredictedMedianInsertSize(this.PREDICTED_INSERT_SIZE);
        }
        if (this.DESCRIPTION != null) {
            rgroup.setDescription(this.DESCRIPTION);
        }
        if (this.RUN_DATE != null) {
            rgroup.setRunDate((Date)this.RUN_DATE);
        }
        SAMFileHeader header = new SAMFileHeader();
        header.addReadGroup(rgroup);
        for (String comment : this.COMMENT) {
            header.addComment(comment);
        }
        header.setSortOrder(this.SORT_ORDER);
        return header;
    }

    void convertQuality(byte[] quals, FastqQualityFormat version) {
        switch (version) {
            case Standard: {
                SAMUtils.fastqToPhred((byte[])quals);
                break;
            }
            case Solexa: {
                solexaQualityConverter.convertSolexaQualityCharsToPhredBinary(quals);
                break;
            }
            case Illumina: {
                solexaQualityConverter.convertSolexa_1_3_QualityCharsToPhredBinary(quals);
            }
        }
    }

    String getBaseName(String readName1, String readName2, FastqReader freader1, FastqReader freader2) {
        String[] toks = this.getReadNameTokens(readName1, 1, freader1);
        String baseName1 = toks[0];
        String num1 = toks[1];
        toks = this.getReadNameTokens(readName2, 2, freader2);
        String baseName2 = toks[0];
        String num2 = toks[1];
        if (!baseName1.equals(baseName2)) {
            throw new PicardException(String.format("In paired mode, read name 1 (%s) does not match read name 2 (%s)", baseName1, baseName2));
        }
        boolean num1Blank = StringUtil.isBlank((String)num1);
        boolean num2Blank = StringUtil.isBlank((String)num2);
        if (num1Blank || num2Blank) {
            if (!num1Blank) {
                throw new PicardException(this.error(freader1, "Pair 1 number is missing (" + readName1 + "). Both pair numbers must be present or neither."));
            }
            if (!num2Blank) {
                throw new PicardException(this.error(freader2, "Pair 2 number is missing (" + readName2 + "). Both pair numbers must be present or neither."));
            }
        } else {
            if (!num1.equals("1")) {
                throw new PicardException(this.error(freader1, "Pair 1 number must be 1 (" + readName1 + ")"));
            }
            if (!num2.equals("2")) {
                throw new PicardException(this.error(freader2, "Pair 2 number must be 2 (" + readName2 + ")"));
            }
        }
        return baseName1;
    }

    private String[] getReadNameTokens(String readName, int pairNum, FastqReader freader) {
        if (readName.equals("")) {
            throw new PicardException(this.error(freader, "Pair read name " + pairNum + " cannot be empty: " + readName));
        }
        int idx = readName.lastIndexOf("/");
        String[] result = new String[2];
        if (idx == -1) {
            result[0] = readName;
            result[1] = null;
        } else {
            result[1] = readName.substring(idx + 1, readName.length());
            if (!result[1].equals("1") && !result[1].equals("2")) {
                result[0] = readName;
                result[1] = null;
            } else {
                result[0] = readName.substring(0, idx);
            }
        }
        return result;
    }

    private String error(FastqReader freader, String str) {
        return str + " at line " + freader.getLineNumber() + " in file " + freader.getFile().getAbsolutePath();
    }

    private String getReadName(String fastqHeader, boolean paired) {
        String readName;
        int idx = fastqHeader.indexOf(" ");
        String string = readName = idx == -1 ? fastqHeader : fastqHeader.substring(0, idx);
        while (this.STRIP_UNPAIRED_MATE_NUMBER.booleanValue() && !paired && readName.endsWith("/1")) {
            readName = readName.substring(0, readName.length() - 2);
        }
        return readName;
    }

    @Override
    protected String[] customCommandLineValidation() {
        if (this.MIN_Q < 0) {
            return new String[]{"MIN_Q must be >= 0"};
        }
        if (this.MAX_Q > 93) {
            return new String[]{"MAX_Q must be <= 93"};
        }
        return null;
    }
}

