/**
* UGENE - Integrated Bioinformatics Tools.
* Copyright (C) 2008-2013 UniPro <ugene@unipro.ru>
* http://ugene.unipro.ru
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
* MA 02110-1301, USA.
*/

#ifndef _U2_DNA_SEQUENCE_GENERATOR_H_
#define _U2_DNA_SEQUENCE_GENERATOR_H_

#include <U2Core/Task.h>
#include <U2Core/DNASequence.h>


namespace U2 {

class GObject;
class Document;
class MAlignment;
class DNAAlphabet;
class DocumentFormat;
class LoadDocumentTask;
class SaveDocumentTask;

class DNASequenceGeneratorConfig {
public:
    DNASequenceGeneratorConfig()
        : addToProj(false), saveDoc(true), format(NULL), alphabet(NULL), length(0), numSeqs(1), useRef(true), window() {}

    DNAAlphabet* getAlphabet() const { assert(alphabet); return alphabet; }

    bool useReference() const { return useRef; }

    const QString& getReferenceUrl() const { return refUrl; }

    const QMap<char, qreal>& getContent() const { return content; }

    int getLength() const { return length; }

    int getNumberOfSequences() const { return numSeqs; }

    QString getSequenceName() const { return sequenceName; }

    QString getOutUrlString() const { return outUrl; }

    DocumentFormat* getDocumentFormat() const { return format; }

    bool addToProj;
    bool saveDoc;
    // output url
    QString outUrl;
    // output sequence base name
    QString sequenceName;
    // output document format
    DocumentFormat* format;
    // output sequence alphabet
    DNAAlphabet* alphabet;
    // output sequence length
    int length;
    // number of sequences to generate
    int numSeqs;
    // use content from reference or specified manually
    bool useRef;
    // reference file url
    QString refUrl;
    // char frequencies
    QMap<char, qreal> content;
    //window size
    int window;
    //seed to initialize qrand
    int seed;
};

class DNASequenceGenerator {
public:
    static const QString ID;

    static QString prepareReferenceFileFilter();

    static void generateSequence(const QMap<char, qreal>& charFreqs, int length, QByteArray& result);

    static void evaluateBaseContent(const DNASequence& sequence, QMap<char, qreal>& result);

    static void evaluateBaseContent(const MAlignment& ma, QMap<char, qreal>& result);
};

class EvaluateBaseContentTask : public Task {
    Q_OBJECT
public:
    EvaluateBaseContentTask(GObject* obj);

    void run();

    QMap<char, qreal> getResult() const { return result; }

    DNAAlphabet* getAlphabet() const { return alp; }

private:
    GObject* _obj;
    DNAAlphabet* alp;
    QMap<char, qreal> result;
};

class GenerateDNASequenceTask : public Task {
    Q_OBJECT
public:
    GenerateDNASequenceTask(const QMap<char, qreal>& baseContent_, int length_, int window_, int count_, int seed_);

    void run();

    QList< QByteArray > getResult() const { return result; }

private:
    QMap<char, qreal> baseContent;
    int length;
    int window;
    int count;
    int seed;
    QList< QByteArray > result;
};

class DNASequenceGeneratorTask : public Task {
    Q_OBJECT
public:
    DNASequenceGeneratorTask(const DNASequenceGeneratorConfig& cfg_);
    QList<Task*> onSubTaskFinished(Task* subTask);
    QList<DNASequence> getSequences() const { return results; }
private:
    static EvaluateBaseContentTask* createEvaluationTask(Document* doc, QString& err);
private:
    DNASequenceGeneratorConfig cfg;
    LoadDocumentTask* loadRefTask;
    EvaluateBaseContentTask* evalTask;
    GenerateDNASequenceTask* generateTask;
    SaveDocumentTask* saveTask;
    QList<DNASequence> results;
};

} //namespace

#endif
