# The example numbers in these specs are ported from Microsoft's C++ STL.
# The following is their license:
#
#   Copyright 2018 Ulf Adams
#   Copyright (c) Microsoft Corporation. All rights reserved.
#
#   Boost Software License - Version 1.0 - August 17th, 2003
#
#   Permission is hereby granted, free of charge, to any person or organization
#   obtaining a copy of the software and accompanying documentation covered by
#   this license (the "Software") to use, reproduce, display, distribute,
#   execute, and transmit the Software, and to prepare derivative works of the
#   Software, and to permit third-parties to whom the Software is furnished to
#   do so, all subject to the following:
#
#   The copyright notices in the Software and this entire statement, including
#   the above license grant, this restriction and the following disclaimer,
#   must be included in all copies of the Software, in whole or in part, and
#   all derivative works of the Software, unless such copies or derivative
#   works are solely in the form of machine-executable object code generated by
#   a source language processor.
#
#   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
#   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#   FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
#   SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
#   FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
#   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
#   DEALINGS IN THE SOFTWARE.
#
# This file contains test cases derived from:
#
# * https://github.com/ulfjack/ryu
# * https://github.com/microsoft/STL/tree/main/tests/std/tests/P0067R5_charconv

require "spec"
require "./spec_helper"
require "../support/string"
require "../support/number"

# Tests that `v.to_s` is the same as the *v* literal is written in the source
# code, except possibly omitting the `_f32` suffix for `Float32` literals.
private macro it_converts_to_s(v)
  it {{ "converts #{v} to \"#{v.id.gsub(/_f32$/, "")}\"" }} do
    assert_prints {{ v }}.to_s, "{{ v.id.gsub(/_f32$/, "") }}"
  end
end

# Tests that `v.to_s == str`.
private macro it_converts_to_s(v, str)
  it {{ "converts #{v.id.gsub(/^hexfloat\("(.*)"\)$/, "\\1")} to #{str}" }} do
    assert_prints ({{ v }}).to_s, {{ str }}
  end
end

describe "Float32#to_s" do
  context "special cases" do
    it_converts_to_s 0.0_f32
    it_converts_to_s -0.0_f32
    it_converts_to_s Float32::INFINITY, "Infinity"
    it_converts_to_s -Float32::INFINITY, "-Infinity"
    it_converts_to_s 0x7FC0_0000_u32.unsafe_as(Float32), "NaN"
    it_converts_to_s 0xFFC0_0000_u32.unsafe_as(Float32), "NaN"
    it_converts_to_s Float32::MIN_POSITIVE, "1.1754944e-38"
    it_converts_to_s Float32::MAX, "3.4028235e+38"
    it_converts_to_s hexfloat("0x1.ffffffp-127_f32"), "1.1754942e-38" # largest denormal
    it_converts_to_s 1.0e-45_f32                                      # smallest denormal
  end

  context "Ryu f2s_test.cc BoundaryRoundEven" do
    it_converts_to_s 3.355445e7_f32, "33554450.0"
    it_converts_to_s 8.999999e9_f32, "9000000000.0"
    it_converts_to_s 3.4366717e10_f32, "34366720000.0"
  end

  context "Ryu f2s_test.cc ExactValueRoundEven" do
    it_converts_to_s 3.0540412e5_f32, "305404.12"
    it_converts_to_s 8.0990312e3_f32, "8099.0312"
  end

  context "Ryu f2s_test.cc LotsOfTrailingZeros" do
    it_converts_to_s 2.4414062e-4_f32, "0.00024414062"
    it_converts_to_s 2.4414062e-3_f32, "0.0024414062"
    it_converts_to_s 4.3945312e-3_f32, "0.0043945312"
    it_converts_to_s 6.3476562e-3_f32, "0.0063476562"
  end

  context "Ryu f2s_test.cc Regression" do
    it_converts_to_s 4.7223665e21_f32, "4.7223665e+21"
    it_converts_to_s 8388608.0_f32, "8388608.0"
    it_converts_to_s 1.6777216e7_f32, "16777216.0"
    it_converts_to_s 3.3554436e7_f32, "33554436.0"
    it_converts_to_s 6.7131496e7_f32, "67131496.0"
    it_converts_to_s 1.9310392e-38_f32, "1.9310392e-38"
    it_converts_to_s -2.47e-43_f32, "-2.47e-43"
    it_converts_to_s 1.993244e-38_f32, "1.993244e-38"
    it_converts_to_s 4103.9003_f32, "4103.9004"
    it_converts_to_s 5.3399997e9_f32, "5339999700.0"
    it_converts_to_s 6.0898e-39_f32, "6.0898e-39"
    it_converts_to_s 0.0010310042_f32, "0.0010310042"
    it_converts_to_s 2.8823261e17_f32, "2.882326e+17"
    it_converts_to_s 7.038531e-26_f32, "7.038531e-26"
    it_converts_to_s 9.2234038e17_f32, "9.223404e+17"
    it_converts_to_s 6.7108872e7_f32, "67108870.0"
    it_converts_to_s 1.0e-44_f32, "1.0e-44"
    it_converts_to_s 2.816025e14_f32, "281602500000000.0"
    it_converts_to_s 9.223372e18_f32, "9.223372e+18"
    it_converts_to_s 1.5846085e29_f32, "1.5846086e+29"
    it_converts_to_s 1.1811161e19_f32, "1.1811161e+19"
    it_converts_to_s 5.368709e18_f32, "5.368709e+18"
    it_converts_to_s 4.6143165e18_f32, "4.6143166e+18"
    it_converts_to_s 0.007812537_f32, "0.007812537"
    it_converts_to_s 1.4e-45_f32, "1.0e-45"
    it_converts_to_s 1.18697724e20_f32, "1.18697725e+20"
    it_converts_to_s 1.00014165e-36_f32, "1.00014165e-36"
    it_converts_to_s 200.0_f32, "200.0"
    it_converts_to_s 3.3554432e7_f32, "33554432.0"
  end

  context "Ryu f2s_test.cc LooksLikePow5" do
    it_converts_to_s hexfloat("0x1.2a05f2p+59_f32"), "6.7108864e+17"
    it_converts_to_s hexfloat("0x1.2a05f2p+60_f32"), "1.3421773e+18"
    it_converts_to_s hexfloat("0x1.2a05f2p+61_f32"), "2.6843546e+18"
  end

  it_converts_to_s 1.0_f32
  it_converts_to_s 1.2_f32
  it_converts_to_s 1.23_f32
  it_converts_to_s 1.234_f32
  it_converts_to_s 1.2345_f32
  it_converts_to_s 1.23456_f32
  it_converts_to_s 1.234567_f32
  it_converts_to_s 1.2345678_f32
  it_converts_to_s 1.23456789_f32, "1.2345679"
  it_converts_to_s 1.23456735e-36_f32, "1.23456735e-36"

  it_converts_to_s 1.0_f32
  it_converts_to_s 12.0_f32
  it_converts_to_s 123.0_f32
  it_converts_to_s 1234.0_f32
  it_converts_to_s 12345.0_f32
  it_converts_to_s 123456.0_f32
  it_converts_to_s 1234567.0_f32
  it_converts_to_s 12345678.0_f32
  it_converts_to_s 123456789.0_f32, "123456790.0"
  it_converts_to_s 1234567890.0_f32, "1234568000.0"
  it_converts_to_s 1234567895.0_f32, "1234568000.0"
  it_converts_to_s 12345678901.0_f32, "12345679000.0"
  it_converts_to_s 123456789012.0_f32, "123456790000.0"
  it_converts_to_s 1234567890123.0_f32, "1234568000000.0"
  it_converts_to_s 12345678901234.0_f32, "12345679000000.0"
  it_converts_to_s 123456789012345.0_f32, "123456790000000.0"
  it_converts_to_s 1234567890123456.0_f32, "1.234568e+15"

  it_converts_to_s 1.0_f32
  it_converts_to_s 10.0_f32
  it_converts_to_s 100.0_f32
  it_converts_to_s 1000.0_f32
  it_converts_to_s 10000.0_f32
  it_converts_to_s 100000.0_f32
  it_converts_to_s 1000000.0_f32
  it_converts_to_s 10000000.0_f32
  it_converts_to_s 100000000.0_f32
  it_converts_to_s 1000000000.0_f32
  it_converts_to_s 10000000000.0_f32
  it_converts_to_s 100000000000.0_f32
  it_converts_to_s 1000000000000.0_f32
  it_converts_to_s 10000000000000.0_f32
  it_converts_to_s 100000000000000.0_f32
  it_converts_to_s 1000000000000000.0_f32, "1.0e+15"

  it_converts_to_s 1000000010000000.0_f32, "1.0e+15"
  it_converts_to_s 1000000100000000.0_f32, "1.0000001e+15"
  it_converts_to_s 1000001000000000.0_f32, "1.000001e+15"
  it_converts_to_s 1000010000000000.0_f32, "1.00001e+15"
  it_converts_to_s 1000100000000000.0_f32, "1.0001e+15"
  it_converts_to_s 1001000000000000.0_f32, "1.001e+15"
  it_converts_to_s 1010000000000000.0_f32, "1.01e+15"
  it_converts_to_s 1100000000000000.0_f32, "1.1e+15"

  it_converts_to_s 8.0_f32
  it_converts_to_s 64.0_f32
  it_converts_to_s 512.0_f32
  it_converts_to_s 8192.0_f32
  it_converts_to_s 65536.0_f32
  it_converts_to_s 524288.0_f32
  it_converts_to_s 8388608.0_f32
  it_converts_to_s 67108864.0_f32

  it_converts_to_s 8000.0_f32
  it_converts_to_s 64000.0_f32
  it_converts_to_s 512000.0_f32
  it_converts_to_s 8192000.0_f32
  it_converts_to_s 65536000.0_f32
  it_converts_to_s 524288000.0_f32
  it_converts_to_s 8388608000.0_f32
  it_converts_to_s 67108864000.0_f32

  context "all exponents" do
    it_converts_to_s 1.729e-45_f32, "1.0e-45"
    it_converts_to_s 1.729e-44_f32, "1.7e-44"
    it_converts_to_s 1.729e-43_f32, "1.72e-43"
    it_converts_to_s 1.729e-42_f32, "1.729e-42"
    it_converts_to_s 1.729e-41_f32, "1.729e-41"
    it_converts_to_s 1.729e-40_f32, "1.729e-40"
    it_converts_to_s 1.729e-39_f32, "1.729e-39"
    it_converts_to_s 1.729e-38_f32, "1.729e-38"
    it_converts_to_s 1.729e-37_f32, "1.729e-37"
    it_converts_to_s 1.729e-36_f32, "1.729e-36"
    it_converts_to_s 1.729e-35_f32, "1.729e-35"
    it_converts_to_s 1.729e-34_f32, "1.729e-34"
    it_converts_to_s 1.729e-33_f32, "1.729e-33"
    it_converts_to_s 1.729e-32_f32, "1.729e-32"
    it_converts_to_s 1.729e-31_f32, "1.729e-31"
    it_converts_to_s 1.729e-30_f32, "1.729e-30"
    it_converts_to_s 1.729e-29_f32, "1.729e-29"
    it_converts_to_s 1.729e-28_f32, "1.729e-28"
    it_converts_to_s 1.729e-27_f32, "1.729e-27"
    it_converts_to_s 1.729e-26_f32, "1.729e-26"
    it_converts_to_s 1.729e-25_f32, "1.729e-25"
    it_converts_to_s 1.729e-24_f32, "1.729e-24"
    it_converts_to_s 1.729e-23_f32, "1.729e-23"
    it_converts_to_s 1.729e-22_f32, "1.729e-22"
    it_converts_to_s 1.729e-21_f32, "1.729e-21"
    it_converts_to_s 1.729e-20_f32, "1.729e-20"
    it_converts_to_s 1.729e-19_f32, "1.729e-19"
    it_converts_to_s 1.729e-18_f32, "1.729e-18"
    it_converts_to_s 1.729e-17_f32, "1.729e-17"
    it_converts_to_s 1.729e-16_f32, "1.729e-16"
    it_converts_to_s 1.729e-15_f32, "1.729e-15"
    it_converts_to_s 1.729e-14_f32, "1.729e-14"
    it_converts_to_s 1.729e-13_f32, "1.729e-13"
    it_converts_to_s 1.729e-12_f32, "1.729e-12"
    it_converts_to_s 1.729e-11_f32, "1.729e-11"
    it_converts_to_s 1.729e-10_f32, "1.729e-10"
    it_converts_to_s 1.729e-9_f32, "1.729e-9"
    it_converts_to_s 1.729e-8_f32, "1.729e-8"
    it_converts_to_s 1.729e-7_f32, "1.729e-7"
    it_converts_to_s 1.729e-6_f32, "1.729e-6"
    it_converts_to_s 1.729e-5_f32, "1.729e-5"

    it_converts_to_s 1.729e-4_f32, "0.0001729"
    it_converts_to_s 1.729e-3_f32, "0.001729"
    it_converts_to_s 1.729e-2_f32, "0.01729"
    it_converts_to_s 1.729e-1_f32, "0.1729"
    it_converts_to_s 1.729e+0_f32, "1.729"
    it_converts_to_s 1.729e+1_f32, "17.29"
    it_converts_to_s 1.729e+2_f32, "172.9"
    it_converts_to_s 1.729e+3_f32, "1729.0"
    it_converts_to_s 1.729e+4_f32, "17290.0"
    it_converts_to_s 1.729e+5_f32, "172900.0"
    it_converts_to_s 1.729e+6_f32, "1729000.0"
    it_converts_to_s 1.729e+7_f32, "17290000.0"
    it_converts_to_s 1.729e+8_f32, "172900000.0"
    it_converts_to_s 1.729e+9_f32, "1729000000.0"
    it_converts_to_s 1.729e+10_f32, "17290000000.0"
    it_converts_to_s 1.729e+11_f32, "172900000000.0"
    it_converts_to_s 1.729e+12_f32, "1729000000000.0"
    it_converts_to_s 1.729e+13_f32, "17290000000000.0"
    it_converts_to_s 1.729e+14_f32, "172900000000000.0"

    it_converts_to_s 1.729e15_f32, "1.729e+15"
    it_converts_to_s 1.729e16_f32, "1.729e+16"
    it_converts_to_s 1.729e17_f32, "1.729e+17"
    it_converts_to_s 1.729e18_f32, "1.729e+18"
    it_converts_to_s 1.729e19_f32, "1.729e+19"
    it_converts_to_s 1.729e20_f32, "1.729e+20"
    it_converts_to_s 1.729e21_f32, "1.729e+21"
    it_converts_to_s 1.729e22_f32, "1.729e+22"
    it_converts_to_s 1.729e23_f32, "1.729e+23"
    it_converts_to_s 1.729e24_f32, "1.729e+24"
    it_converts_to_s 1.729e25_f32, "1.729e+25"
    it_converts_to_s 1.729e26_f32, "1.729e+26"
    it_converts_to_s 1.729e27_f32, "1.729e+27"
    it_converts_to_s 1.729e28_f32, "1.729e+28"
    it_converts_to_s 1.729e29_f32, "1.729e+29"
    it_converts_to_s 1.729e30_f32, "1.729e+30"
    it_converts_to_s 1.729e31_f32, "1.729e+31"
    it_converts_to_s 1.729e32_f32, "1.729e+32"
    it_converts_to_s 1.729e33_f32, "1.729e+33"
    it_converts_to_s 1.729e34_f32, "1.729e+34"
    it_converts_to_s 1.729e35_f32, "1.729e+35"
    it_converts_to_s 1.729e36_f32, "1.729e+36"
    it_converts_to_s 1.729e37_f32, "1.729e+37"
    it_converts_to_s 1.729e38_f32, "1.729e+38"
  end

  context "one-digit cases, where the decimal point can't appear between digits like \"17.29\"" do
    it_converts_to_s 7e-3_f32, "0.007"
    it_converts_to_s 7e-2_f32, "0.07"
    it_converts_to_s 7e-1_f32, "0.7"
    it_converts_to_s 7e+0_f32, "7.0"
    it_converts_to_s 7e+1_f32, "70.0"
    it_converts_to_s 7e+2_f32, "700.0"
    it_converts_to_s 7e+3_f32, "7000.0"
  end

  context "highly-trimmed powers of 2" do
    it_converts_to_s hexfloat("0x1p118_f32"), "3.32307e+35"
    it_converts_to_s hexfloat("0x1p119_f32"), "6.64614e+35"
  end

  context "odd mantissas (unaffected by shifting)" do
    it_converts_to_s 3355443e1_f32, "33554430.0"
    it_converts_to_s 671087e2_f32, "67108700.0"
    it_converts_to_s 134217e3_f32, "134217000.0"
    it_converts_to_s 26843e4_f32, "268430000.0"
    it_converts_to_s 5367e5_f32, "536700000.0"
    it_converts_to_s 1073e6_f32, "1073000000.0"
    it_converts_to_s 213e7_f32, "2130000000.0"
    it_converts_to_s 41e8_f32, "4100000000.0"
    it_converts_to_s 7e9_f32, "7000000000.0"
    it_converts_to_s 1e10_f32, "10000000000.0"
    it_converts_to_s 3355445e1_f32, "33554450.0"
    it_converts_to_s 671089e2_f32, "67108900.0"
    it_converts_to_s 134219e3_f32, "134219000.0"
    it_converts_to_s 26845e4_f32, "268450000.0"
    it_converts_to_s 5369e5_f32, "536900000.0"
    it_converts_to_s 1075e6_f32, "1075000000.0"
    it_converts_to_s 215e7_f32, "2150000000.0"
    it_converts_to_s 43e8_f32, "4300000000.0"
    it_converts_to_s 9e9_f32, "9000000000.0"
    it_converts_to_s 3e10_f32, "30000000000.0"
    it_converts_to_s 5495808e5_f32, "549580800000.0"
    it_converts_to_s 5497856e5_f32, "549785600000.0"
  end

  context "Grisu failures" do
    it_converts_to_s 85_f32 / 512_f32, "0.16601562"
  end

  # https://www.exploringbinary.com/the-shortest-decimal-string-that-round-trips-may-not-be-the-nearest/
  context "anomalous values" do
    it_converts_to_s hexfloat("0x1p90_f32"), "1.2379401e+27"
    it_converts_to_s hexfloat("0x1p87_f32"), "1.5474251e+26"
    it_converts_to_s hexfloat("0x1p-96_f32"), "1.2621775e-29"
  end
end

describe "Float64#to_s" do
  context "special cases" do
    it_converts_to_s 0.0
    it_converts_to_s -0.0
    it_converts_to_s Float64::INFINITY, "Infinity"
    it_converts_to_s -Float64::INFINITY, "-Infinity"
    it_converts_to_s 0x7FF8_0000_0000_0000_u64.unsafe_as(Float64), "NaN"
    it_converts_to_s 0xFFF8_0000_0000_0000_u64.unsafe_as(Float64), "NaN"
    it_converts_to_s Float64::MIN_POSITIVE, "2.2250738585072014e-308"
    it_converts_to_s Float64::MAX, "1.7976931348623157e+308"
    it_converts_to_s hexfloat("0x1.fffffffffffffp-1023"), "2.225073858507201e-308" # largest denormal
    it_converts_to_s 5.0e-324                                                      # smallest denormal
  end

  context "Ryu d2s_test.cc LotsOfTrailingZeros" do
    it_converts_to_s 2.98023223876953125e-8, "2.9802322387695312e-8"
  end

  context "Ryu d2s_test.cc Regression" do
    it_converts_to_s -2.109808898695963e16, "-2.109808898695963e+16"
    it_converts_to_s 4.940656e-318, "4.940656e-318"
    it_converts_to_s 1.18575755e-316, "1.18575755e-316"
    it_converts_to_s 2.989102097996e-312, "2.989102097996e-312"
    it_converts_to_s 9.0608011534336e15, "9.0608011534336e+15"
    it_converts_to_s 4.708356024711512e18, "4.708356024711512e+18"
    it_converts_to_s 9.409340012568248e18, "9.409340012568248e+18"
  end

  context "Ryu d2s_test.cc LooksLikePow5" do
    it_converts_to_s hexfloat("0x1.0f0cf064dd592p+132"), "5.764607523034235e+39"
    it_converts_to_s hexfloat("0x1.0f0cf064dd592p+133"), "1.152921504606847e+40"
    it_converts_to_s hexfloat("0x1.0f0cf064dd592p+134"), "2.305843009213694e+40"
  end

  context "Ryu d2s_test.cc 32-bit Chunking" do
    it_converts_to_s 4.294967294, "4.294967294"
    it_converts_to_s 4.294967295, "4.294967295"
    it_converts_to_s 4.294967296, "4.294967296"
    it_converts_to_s 4.294967297, "4.294967297"
    it_converts_to_s 4.294967298, "4.294967298"
  end

  context "Ryu d2s_test.cc MinMaxShift" do
    it_converts_to_s hexfloat("0x1.0000000000000p-1019"), "1.7800590868057611e-307"
    it_converts_to_s hexfloat("0x1.fffffffffffffp-1016"), "2.8480945388892175e-306"
    it_converts_to_s hexfloat("0x1.0000000000000p-982"), "2.446494580089078e-296"
    it_converts_to_s hexfloat("0x1.fffffffffffffp-982"), "4.8929891601781557e-296"
    it_converts_to_s hexfloat("0x1.0000000000000p+54"), "1.8014398509481984e+16"
    it_converts_to_s hexfloat("0x1.fffffffffffffp+54"), "3.6028797018963964e+16"
    it_converts_to_s hexfloat("0x1.0000000000000p-716"), "2.900835519859558e-216"
    it_converts_to_s hexfloat("0x1.fffffffffffffp-716"), "5.801671039719115e-216"
    it_converts_to_s hexfloat("0x1.fa7161a4d6e0cp-89"), "3.196104012172126e-27"
  end

  context "Ryu d2s_test.cc SmallIntegers" do
    it_converts_to_s 9007199254740991.0, "9.007199254740991e+15"
    it_converts_to_s 9007199254740992.0, "9.007199254740992e+15"
  end

  it_converts_to_s 1.0
  it_converts_to_s 1.2
  it_converts_to_s 1.23
  it_converts_to_s 1.234
  it_converts_to_s 1.2345
  it_converts_to_s 1.23456
  it_converts_to_s 1.234567
  it_converts_to_s 1.2345678
  it_converts_to_s 1.23456789
  it_converts_to_s 1.234567895
  it_converts_to_s 1.2345678901
  it_converts_to_s 1.23456789012
  it_converts_to_s 1.234567890123
  it_converts_to_s 1.2345678901234
  it_converts_to_s 1.23456789012345
  it_converts_to_s 1.234567890123456
  it_converts_to_s 1.2345678901234567

  it_converts_to_s 1.0
  it_converts_to_s 12.0
  it_converts_to_s 123.0
  it_converts_to_s 1234.0
  it_converts_to_s 12345.0
  it_converts_to_s 123456.0
  it_converts_to_s 1234567.0
  it_converts_to_s 12345678.0
  it_converts_to_s 123456789.0
  it_converts_to_s 1234567890.0
  it_converts_to_s 1234567895.0
  it_converts_to_s 12345678901.0
  it_converts_to_s 123456789012.0
  it_converts_to_s 1234567890123.0
  it_converts_to_s 12345678901234.0
  it_converts_to_s 123456789012345.0
  it_converts_to_s 1234567890123456.0, "1.234567890123456e+15"

  it_converts_to_s 1.0
  it_converts_to_s 10.0
  it_converts_to_s 100.0
  it_converts_to_s 1000.0
  it_converts_to_s 10000.0
  it_converts_to_s 100000.0
  it_converts_to_s 1000000.0
  it_converts_to_s 10000000.0
  it_converts_to_s 100000000.0
  it_converts_to_s 1000000000.0
  it_converts_to_s 10000000000.0
  it_converts_to_s 100000000000.0
  it_converts_to_s 1000000000000.0
  it_converts_to_s 10000000000000.0
  it_converts_to_s 100000000000000.0
  it_converts_to_s 1000000000000000.0, "1.0e+15"

  it_converts_to_s 1000000000000001.0, "1.000000000000001e+15"
  it_converts_to_s 1000000000000010.0, "1.00000000000001e+15"
  it_converts_to_s 1000000000000100.0, "1.0000000000001e+15"
  it_converts_to_s 1000000000001000.0, "1.000000000001e+15"
  it_converts_to_s 1000000000010000.0, "1.00000000001e+15"
  it_converts_to_s 1000000000100000.0, "1.0000000001e+15"
  it_converts_to_s 1000000001000000.0, "1.000000001e+15"
  it_converts_to_s 1000000010000000.0, "1.00000001e+15"
  it_converts_to_s 1000000100000000.0, "1.0000001e+15"
  it_converts_to_s 1000001000000000.0, "1.000001e+15"
  it_converts_to_s 1000010000000000.0, "1.00001e+15"
  it_converts_to_s 1000100000000000.0, "1.0001e+15"
  it_converts_to_s 1001000000000000.0, "1.001e+15"
  it_converts_to_s 1010000000000000.0, "1.01e+15"
  it_converts_to_s 1100000000000000.0, "1.1e+15"

  it_converts_to_s 8.0
  it_converts_to_s 64.0
  it_converts_to_s 512.0
  it_converts_to_s 8192.0
  it_converts_to_s 65536.0
  it_converts_to_s 524288.0
  it_converts_to_s 8388608.0
  it_converts_to_s 67108864.0
  it_converts_to_s 536870912.0
  it_converts_to_s 8589934592.0
  it_converts_to_s 68719476736.0
  it_converts_to_s 549755813888.0
  it_converts_to_s 8796093022208.0
  it_converts_to_s 70368744177664.0
  it_converts_to_s 562949953421312.0
  it_converts_to_s 9007199254740992.0, "9.007199254740992e+15"

  it_converts_to_s 8000.0
  it_converts_to_s 64000.0
  it_converts_to_s 512000.0
  it_converts_to_s 8192000.0
  it_converts_to_s 65536000.0
  it_converts_to_s 524288000.0
  it_converts_to_s 8388608000.0
  it_converts_to_s 67108864000.0
  it_converts_to_s 536870912000.0
  it_converts_to_s 8589934592000.0
  it_converts_to_s 68719476736000.0
  it_converts_to_s 549755813888000.0
  it_converts_to_s 8796093022208000.0, "8.796093022208e+15"

  context "all exponents" do
    it_converts_to_s 7.29e-324, "5.0e-324"
    it_converts_to_s 1.729e-323, "1.5e-323"
    it_converts_to_s 1.729e-322, "1.73e-322"
    it_converts_to_s 1.729e-321, "1.73e-321"
    it_converts_to_s 1.729e-320, "1.729e-320"
    it_converts_to_s 1.729e-319, "1.729e-319"
    it_converts_to_s 1.729e-318, "1.729e-318"
    it_converts_to_s 1.729e-317, "1.729e-317"
    it_converts_to_s 1.729e-316, "1.729e-316"
    it_converts_to_s 1.729e-315, "1.729e-315"
    it_converts_to_s 1.729e-314, "1.729e-314"
    it_converts_to_s 1.729e-313, "1.729e-313"
    it_converts_to_s 1.729e-312, "1.729e-312"
    it_converts_to_s 1.729e-311, "1.729e-311"
    it_converts_to_s 1.729e-310, "1.729e-310"
    it_converts_to_s 1.729e-309, "1.729e-309"
    it_converts_to_s 1.729e-308, "1.729e-308"
    it_converts_to_s 1.729e-307, "1.729e-307"
    it_converts_to_s 1.729e-306, "1.729e-306"
    it_converts_to_s 1.729e-305, "1.729e-305"
    it_converts_to_s 1.729e-304, "1.729e-304"
    it_converts_to_s 1.729e-303, "1.729e-303"
    it_converts_to_s 1.729e-302, "1.729e-302"
    it_converts_to_s 1.729e-301, "1.729e-301"
    it_converts_to_s 1.729e-300, "1.729e-300"
    it_converts_to_s 1.729e-299, "1.729e-299"
    it_converts_to_s 1.729e-298, "1.729e-298"
    it_converts_to_s 1.729e-297, "1.729e-297"
    it_converts_to_s 1.729e-296, "1.729e-296"
    it_converts_to_s 1.729e-295, "1.729e-295"
    it_converts_to_s 1.729e-294, "1.729e-294"
    it_converts_to_s 1.729e-293, "1.729e-293"
    it_converts_to_s 1.729e-292, "1.729e-292"
    it_converts_to_s 1.729e-291, "1.729e-291"
    it_converts_to_s 1.729e-290, "1.729e-290"
    it_converts_to_s 1.729e-289, "1.729e-289"
    it_converts_to_s 1.729e-288, "1.729e-288"
    it_converts_to_s 1.729e-287, "1.729e-287"
    it_converts_to_s 1.729e-286, "1.729e-286"
    it_converts_to_s 1.729e-285, "1.729e-285"
    it_converts_to_s 1.729e-284, "1.729e-284"
    it_converts_to_s 1.729e-283, "1.729e-283"
    it_converts_to_s 1.729e-282, "1.729e-282"
    it_converts_to_s 1.729e-281, "1.729e-281"
    it_converts_to_s 1.729e-280, "1.729e-280"
    it_converts_to_s 1.729e-279, "1.729e-279"
    it_converts_to_s 1.729e-278, "1.729e-278"
    it_converts_to_s 1.729e-277, "1.729e-277"
    it_converts_to_s 1.729e-276, "1.729e-276"
    it_converts_to_s 1.729e-275, "1.729e-275"
    it_converts_to_s 1.729e-274, "1.729e-274"
    it_converts_to_s 1.729e-273, "1.729e-273"
    it_converts_to_s 1.729e-272, "1.729e-272"
    it_converts_to_s 1.729e-271, "1.729e-271"
    it_converts_to_s 1.729e-270, "1.729e-270"
    it_converts_to_s 1.729e-269, "1.729e-269"
    it_converts_to_s 1.729e-268, "1.729e-268"
    it_converts_to_s 1.729e-267, "1.729e-267"
    it_converts_to_s 1.729e-266, "1.729e-266"
    it_converts_to_s 1.729e-265, "1.729e-265"
    it_converts_to_s 1.729e-264, "1.729e-264"
    it_converts_to_s 1.729e-263, "1.729e-263"
    it_converts_to_s 1.729e-262, "1.729e-262"
    it_converts_to_s 1.729e-261, "1.729e-261"
    it_converts_to_s 1.729e-260, "1.729e-260"
    it_converts_to_s 1.729e-259, "1.729e-259"
    it_converts_to_s 1.729e-258, "1.729e-258"
    it_converts_to_s 1.729e-257, "1.729e-257"
    it_converts_to_s 1.729e-256, "1.729e-256"
    it_converts_to_s 1.729e-255, "1.729e-255"
    it_converts_to_s 1.729e-254, "1.729e-254"
    it_converts_to_s 1.729e-253, "1.729e-253"
    it_converts_to_s 1.729e-252, "1.729e-252"
    it_converts_to_s 1.729e-251, "1.729e-251"
    it_converts_to_s 1.729e-250, "1.729e-250"
    it_converts_to_s 1.729e-249, "1.729e-249"
    it_converts_to_s 1.729e-248, "1.729e-248"
    it_converts_to_s 1.729e-247, "1.729e-247"
    it_converts_to_s 1.729e-246, "1.729e-246"
    it_converts_to_s 1.729e-245, "1.729e-245"
    it_converts_to_s 1.729e-244, "1.729e-244"
    it_converts_to_s 1.729e-243, "1.729e-243"
    it_converts_to_s 1.729e-242, "1.729e-242"
    it_converts_to_s 1.729e-241, "1.729e-241"
    it_converts_to_s 1.729e-240, "1.729e-240"
    it_converts_to_s 1.729e-239, "1.729e-239"
    it_converts_to_s 1.729e-238, "1.729e-238"
    it_converts_to_s 1.729e-237, "1.729e-237"
    it_converts_to_s 1.729e-236, "1.729e-236"
    it_converts_to_s 1.729e-235, "1.729e-235"
    it_converts_to_s 1.729e-234, "1.729e-234"
    it_converts_to_s 1.729e-233, "1.729e-233"
    it_converts_to_s 1.729e-232, "1.729e-232"
    it_converts_to_s 1.729e-231, "1.729e-231"
    it_converts_to_s 1.729e-230, "1.729e-230"
    it_converts_to_s 1.729e-229, "1.729e-229"
    it_converts_to_s 1.729e-228, "1.729e-228"
    it_converts_to_s 1.729e-227, "1.729e-227"
    it_converts_to_s 1.729e-226, "1.729e-226"
    it_converts_to_s 1.729e-225, "1.729e-225"
    it_converts_to_s 1.729e-224, "1.729e-224"
    it_converts_to_s 1.729e-223, "1.729e-223"
    it_converts_to_s 1.729e-222, "1.729e-222"
    it_converts_to_s 1.729e-221, "1.729e-221"
    it_converts_to_s 1.729e-220, "1.729e-220"
    it_converts_to_s 1.729e-219, "1.729e-219"
    it_converts_to_s 1.729e-218, "1.729e-218"
    it_converts_to_s 1.729e-217, "1.729e-217"
    it_converts_to_s 1.729e-216, "1.729e-216"
    it_converts_to_s 1.729e-215, "1.729e-215"
    it_converts_to_s 1.729e-214, "1.729e-214"
    it_converts_to_s 1.729e-213, "1.729e-213"
    it_converts_to_s 1.729e-212, "1.729e-212"
    it_converts_to_s 1.729e-211, "1.729e-211"
    it_converts_to_s 1.729e-210, "1.729e-210"
    it_converts_to_s 1.729e-209, "1.729e-209"
    it_converts_to_s 1.729e-208, "1.729e-208"
    it_converts_to_s 1.729e-207, "1.729e-207"
    it_converts_to_s 1.729e-206, "1.729e-206"
    it_converts_to_s 1.729e-205, "1.729e-205"
    it_converts_to_s 1.729e-204, "1.729e-204"
    it_converts_to_s 1.729e-203, "1.729e-203"
    it_converts_to_s 1.729e-202, "1.729e-202"
    it_converts_to_s 1.729e-201, "1.729e-201"
    it_converts_to_s 1.729e-200, "1.729e-200"
    it_converts_to_s 1.729e-199, "1.729e-199"
    it_converts_to_s 1.729e-198, "1.729e-198"
    it_converts_to_s 1.729e-197, "1.729e-197"
    it_converts_to_s 1.729e-196, "1.729e-196"
    it_converts_to_s 1.729e-195, "1.729e-195"
    it_converts_to_s 1.729e-194, "1.729e-194"
    it_converts_to_s 1.729e-193, "1.729e-193"
    it_converts_to_s 1.729e-192, "1.729e-192"
    it_converts_to_s 1.729e-191, "1.729e-191"
    it_converts_to_s 1.729e-190, "1.729e-190"
    it_converts_to_s 1.729e-189, "1.729e-189"
    it_converts_to_s 1.729e-188, "1.729e-188"
    it_converts_to_s 1.729e-187, "1.729e-187"
    it_converts_to_s 1.729e-186, "1.729e-186"
    it_converts_to_s 1.729e-185, "1.729e-185"
    it_converts_to_s 1.729e-184, "1.729e-184"
    it_converts_to_s 1.729e-183, "1.729e-183"
    it_converts_to_s 1.729e-182, "1.729e-182"
    it_converts_to_s 1.729e-181, "1.729e-181"
    it_converts_to_s 1.729e-180, "1.729e-180"
    it_converts_to_s 1.729e-179, "1.729e-179"
    it_converts_to_s 1.729e-178, "1.729e-178"
    it_converts_to_s 1.729e-177, "1.729e-177"
    it_converts_to_s 1.729e-176, "1.729e-176"
    it_converts_to_s 1.729e-175, "1.729e-175"
    it_converts_to_s 1.729e-174, "1.729e-174"
    it_converts_to_s 1.729e-173, "1.729e-173"
    it_converts_to_s 1.729e-172, "1.729e-172"
    it_converts_to_s 1.729e-171, "1.729e-171"
    it_converts_to_s 1.729e-170, "1.729e-170"
    it_converts_to_s 1.729e-169, "1.729e-169"
    it_converts_to_s 1.729e-168, "1.729e-168"
    it_converts_to_s 1.729e-167, "1.729e-167"
    it_converts_to_s 1.729e-166, "1.729e-166"
    it_converts_to_s 1.729e-165, "1.729e-165"
    it_converts_to_s 1.729e-164, "1.729e-164"
    it_converts_to_s 1.729e-163, "1.729e-163"
    it_converts_to_s 1.729e-162, "1.729e-162"
    it_converts_to_s 1.729e-161, "1.729e-161"
    it_converts_to_s 1.729e-160, "1.729e-160"
    it_converts_to_s 1.729e-159, "1.729e-159"
    it_converts_to_s 1.729e-158, "1.729e-158"
    it_converts_to_s 1.729e-157, "1.729e-157"
    it_converts_to_s 1.729e-156, "1.729e-156"
    it_converts_to_s 1.729e-155, "1.729e-155"
    it_converts_to_s 1.729e-154, "1.729e-154"
    it_converts_to_s 1.729e-153, "1.729e-153"
    it_converts_to_s 1.729e-152, "1.729e-152"
    it_converts_to_s 1.729e-151, "1.729e-151"
    it_converts_to_s 1.729e-150, "1.729e-150"
    it_converts_to_s 1.729e-149, "1.729e-149"
    it_converts_to_s 1.729e-148, "1.729e-148"
    it_converts_to_s 1.729e-147, "1.729e-147"
    it_converts_to_s 1.729e-146, "1.729e-146"
    it_converts_to_s 1.729e-145, "1.729e-145"
    it_converts_to_s 1.729e-144, "1.729e-144"
    it_converts_to_s 1.729e-143, "1.729e-143"
    it_converts_to_s 1.729e-142, "1.729e-142"
    it_converts_to_s 1.729e-141, "1.729e-141"
    it_converts_to_s 1.729e-140, "1.729e-140"
    it_converts_to_s 1.729e-139, "1.729e-139"
    it_converts_to_s 1.729e-138, "1.729e-138"
    it_converts_to_s 1.729e-137, "1.729e-137"
    it_converts_to_s 1.729e-136, "1.729e-136"
    it_converts_to_s 1.729e-135, "1.729e-135"
    it_converts_to_s 1.729e-134, "1.729e-134"
    it_converts_to_s 1.729e-133, "1.729e-133"
    it_converts_to_s 1.729e-132, "1.729e-132"
    it_converts_to_s 1.729e-131, "1.729e-131"
    it_converts_to_s 1.729e-130, "1.729e-130"
    it_converts_to_s 1.729e-129, "1.729e-129"
    it_converts_to_s 1.729e-128, "1.729e-128"
    it_converts_to_s 1.729e-127, "1.729e-127"
    it_converts_to_s 1.729e-126, "1.729e-126"
    it_converts_to_s 1.729e-125, "1.729e-125"
    it_converts_to_s 1.729e-124, "1.729e-124"
    it_converts_to_s 1.729e-123, "1.729e-123"
    it_converts_to_s 1.729e-122, "1.729e-122"
    it_converts_to_s 1.729e-121, "1.729e-121"
    it_converts_to_s 1.729e-120, "1.729e-120"
    it_converts_to_s 1.729e-119, "1.729e-119"
    it_converts_to_s 1.729e-118, "1.729e-118"
    it_converts_to_s 1.729e-117, "1.729e-117"
    it_converts_to_s 1.729e-116, "1.729e-116"
    it_converts_to_s 1.729e-115, "1.729e-115"
    it_converts_to_s 1.729e-114, "1.729e-114"
    it_converts_to_s 1.729e-113, "1.729e-113"
    it_converts_to_s 1.729e-112, "1.729e-112"
    it_converts_to_s 1.729e-111, "1.729e-111"
    it_converts_to_s 1.729e-110, "1.729e-110"
    it_converts_to_s 1.729e-109, "1.729e-109"
    it_converts_to_s 1.729e-108, "1.729e-108"
    it_converts_to_s 1.729e-107, "1.729e-107"
    it_converts_to_s 1.729e-106, "1.729e-106"
    it_converts_to_s 1.729e-105, "1.729e-105"
    it_converts_to_s 1.729e-104, "1.729e-104"
    it_converts_to_s 1.729e-103, "1.729e-103"
    it_converts_to_s 1.729e-102, "1.729e-102"
    it_converts_to_s 1.729e-101, "1.729e-101"
    it_converts_to_s 1.729e-100, "1.729e-100"
    it_converts_to_s 1.729e-99, "1.729e-99"
    it_converts_to_s 1.729e-98, "1.729e-98"
    it_converts_to_s 1.729e-97, "1.729e-97"
    it_converts_to_s 1.729e-96, "1.729e-96"
    it_converts_to_s 1.729e-95, "1.729e-95"
    it_converts_to_s 1.729e-94, "1.729e-94"
    it_converts_to_s 1.729e-93, "1.729e-93"
    it_converts_to_s 1.729e-92, "1.729e-92"
    it_converts_to_s 1.729e-91, "1.729e-91"
    it_converts_to_s 1.729e-90, "1.729e-90"
    it_converts_to_s 1.729e-89, "1.729e-89"
    it_converts_to_s 1.729e-88, "1.729e-88"
    it_converts_to_s 1.729e-87, "1.729e-87"
    it_converts_to_s 1.729e-86, "1.729e-86"
    it_converts_to_s 1.729e-85, "1.729e-85"
    it_converts_to_s 1.729e-84, "1.729e-84"
    it_converts_to_s 1.729e-83, "1.729e-83"
    it_converts_to_s 1.729e-82, "1.729e-82"
    it_converts_to_s 1.729e-81, "1.729e-81"
    it_converts_to_s 1.729e-80, "1.729e-80"
    it_converts_to_s 1.729e-79, "1.729e-79"
    it_converts_to_s 1.729e-78, "1.729e-78"
    it_converts_to_s 1.729e-77, "1.729e-77"
    it_converts_to_s 1.729e-76, "1.729e-76"
    it_converts_to_s 1.729e-75, "1.729e-75"
    it_converts_to_s 1.729e-74, "1.729e-74"
    it_converts_to_s 1.729e-73, "1.729e-73"
    it_converts_to_s 1.729e-72, "1.729e-72"
    it_converts_to_s 1.729e-71, "1.729e-71"
    it_converts_to_s 1.729e-70, "1.729e-70"
    it_converts_to_s 1.729e-69, "1.729e-69"
    it_converts_to_s 1.729e-68, "1.729e-68"
    it_converts_to_s 1.729e-67, "1.729e-67"
    it_converts_to_s 1.729e-66, "1.729e-66"
    it_converts_to_s 1.729e-65, "1.729e-65"
    it_converts_to_s 1.729e-64, "1.729e-64"
    it_converts_to_s 1.729e-63, "1.729e-63"
    it_converts_to_s 1.729e-62, "1.729e-62"
    it_converts_to_s 1.729e-61, "1.729e-61"
    it_converts_to_s 1.729e-60, "1.729e-60"
    it_converts_to_s 1.729e-59, "1.729e-59"
    it_converts_to_s 1.729e-58, "1.729e-58"
    it_converts_to_s 1.729e-57, "1.729e-57"
    it_converts_to_s 1.729e-56, "1.729e-56"
    it_converts_to_s 1.729e-55, "1.729e-55"
    it_converts_to_s 1.729e-54, "1.729e-54"
    it_converts_to_s 1.729e-53, "1.729e-53"
    it_converts_to_s 1.729e-52, "1.729e-52"
    it_converts_to_s 1.729e-51, "1.729e-51"
    it_converts_to_s 1.729e-50, "1.729e-50"
    it_converts_to_s 1.729e-49, "1.729e-49"
    it_converts_to_s 1.729e-48, "1.729e-48"
    it_converts_to_s 1.729e-47, "1.729e-47"
    it_converts_to_s 1.729e-46, "1.729e-46"
    it_converts_to_s 1.729e-45, "1.729e-45"
    it_converts_to_s 1.729e-44, "1.729e-44"
    it_converts_to_s 1.729e-43, "1.729e-43"
    it_converts_to_s 1.729e-42, "1.729e-42"
    it_converts_to_s 1.729e-41, "1.729e-41"
    it_converts_to_s 1.729e-40, "1.729e-40"
    it_converts_to_s 1.729e-39, "1.729e-39"
    it_converts_to_s 1.729e-38, "1.729e-38"
    it_converts_to_s 1.729e-37, "1.729e-37"
    it_converts_to_s 1.729e-36, "1.729e-36"
    it_converts_to_s 1.729e-35, "1.729e-35"
    it_converts_to_s 1.729e-34, "1.729e-34"
    it_converts_to_s 1.729e-33, "1.729e-33"
    it_converts_to_s 1.729e-32, "1.729e-32"
    it_converts_to_s 1.729e-31, "1.729e-31"
    it_converts_to_s 1.729e-30, "1.729e-30"
    it_converts_to_s 1.729e-29, "1.729e-29"
    it_converts_to_s 1.729e-28, "1.729e-28"
    it_converts_to_s 1.729e-27, "1.729e-27"
    it_converts_to_s 1.729e-26, "1.729e-26"
    it_converts_to_s 1.729e-25, "1.729e-25"
    it_converts_to_s 1.729e-24, "1.729e-24"
    it_converts_to_s 1.729e-23, "1.729e-23"
    it_converts_to_s 1.729e-22, "1.729e-22"
    it_converts_to_s 1.729e-21, "1.729e-21"
    it_converts_to_s 1.729e-20, "1.729e-20"
    it_converts_to_s 1.729e-19, "1.729e-19"
    it_converts_to_s 1.729e-18, "1.729e-18"
    it_converts_to_s 1.729e-17, "1.729e-17"
    it_converts_to_s 1.729e-16, "1.729e-16"
    it_converts_to_s 1.729e-15, "1.729e-15"
    it_converts_to_s 1.729e-14, "1.729e-14"
    it_converts_to_s 1.729e-13, "1.729e-13"
    it_converts_to_s 1.729e-12, "1.729e-12"
    it_converts_to_s 1.729e-11, "1.729e-11"
    it_converts_to_s 1.729e-10, "1.729e-10"
    it_converts_to_s 1.729e-9, "1.729e-9"
    it_converts_to_s 1.729e-8, "1.729e-8"
    it_converts_to_s 1.729e-7, "1.729e-7"
    it_converts_to_s 1.729e-6, "1.729e-6"
    it_converts_to_s 1.729e-5, "1.729e-5"

    it_converts_to_s 1.729e-4, "0.0001729"
    it_converts_to_s 1.729e-3, "0.001729"
    it_converts_to_s 1.729e-2, "0.01729"
    it_converts_to_s 1.729e-1, "0.1729"
    it_converts_to_s 1.729e+0, "1.729"
    it_converts_to_s 1.729e+1, "17.29"
    it_converts_to_s 1.729e+2, "172.9"
    it_converts_to_s 1.729e+3, "1729.0"
    it_converts_to_s 1.729e+4, "17290.0"
    it_converts_to_s 1.729e+5, "172900.0"
    it_converts_to_s 1.729e+6, "1729000.0"
    it_converts_to_s 1.729e+7, "17290000.0"
    it_converts_to_s 1.729e+8, "172900000.0"
    it_converts_to_s 1.729e+9, "1729000000.0"
    it_converts_to_s 1.729e+10, "17290000000.0"
    it_converts_to_s 1.729e+11, "172900000000.0"
    it_converts_to_s 1.729e+12, "1729000000000.0"
    it_converts_to_s 1.729e+13, "17290000000000.0"
    it_converts_to_s 1.729e+14, "172900000000000.0"

    it_converts_to_s 1.729e+15, "1.729e+15"
    it_converts_to_s 1.729e+16, "1.729e+16"
    it_converts_to_s 1.729e+17, "1.729e+17"
    it_converts_to_s 1.729e+18, "1.729e+18"
    it_converts_to_s 1.729e+19, "1.729e+19"
    it_converts_to_s 1.729e+20, "1.729e+20"
    it_converts_to_s 1.729e+21, "1.729e+21"
    it_converts_to_s 1.729e+22, "1.729e+22"
    it_converts_to_s 1.729e+23, "1.729e+23"
    it_converts_to_s 1.729e+24, "1.729e+24"
    it_converts_to_s 1.729e+25, "1.729e+25"
    it_converts_to_s 1.729e+26, "1.729e+26"
    it_converts_to_s 1.729e+27, "1.729e+27"
    it_converts_to_s 1.729e+28, "1.729e+28"
    it_converts_to_s 1.729e+29, "1.729e+29"
    it_converts_to_s 1.729e+30, "1.729e+30"
    it_converts_to_s 1.729e+31, "1.729e+31"
    it_converts_to_s 1.729e+32, "1.729e+32"
    it_converts_to_s 1.729e+33, "1.729e+33"
    it_converts_to_s 1.729e+34, "1.729e+34"
    it_converts_to_s 1.729e+35, "1.729e+35"
    it_converts_to_s 1.729e+36, "1.729e+36"
    it_converts_to_s 1.729e+37, "1.729e+37"
    it_converts_to_s 1.729e+38, "1.729e+38"
    it_converts_to_s 1.729e+39, "1.729e+39"
    it_converts_to_s 1.729e+40, "1.729e+40"
    it_converts_to_s 1.729e+41, "1.729e+41"
    it_converts_to_s 1.729e+42, "1.729e+42"
    it_converts_to_s 1.729e+43, "1.729e+43"
    it_converts_to_s 1.729e+44, "1.729e+44"
    it_converts_to_s 1.729e+45, "1.729e+45"
    it_converts_to_s 1.729e+46, "1.729e+46"
    it_converts_to_s 1.729e+47, "1.729e+47"
    it_converts_to_s 1.729e+48, "1.729e+48"
    it_converts_to_s 1.729e+49, "1.729e+49"
    it_converts_to_s 1.729e+50, "1.729e+50"
    it_converts_to_s 1.729e+51, "1.729e+51"
    it_converts_to_s 1.729e+52, "1.729e+52"
    it_converts_to_s 1.729e+53, "1.729e+53"
    it_converts_to_s 1.729e+54, "1.729e+54"
    it_converts_to_s 1.729e+55, "1.729e+55"
    it_converts_to_s 1.729e+56, "1.729e+56"
    it_converts_to_s 1.729e+57, "1.729e+57"
    it_converts_to_s 1.729e+58, "1.729e+58"
    it_converts_to_s 1.729e+59, "1.729e+59"
    it_converts_to_s 1.729e+60, "1.729e+60"
    it_converts_to_s 1.729e+61, "1.729e+61"
    it_converts_to_s 1.729e+62, "1.729e+62"
    it_converts_to_s 1.729e+63, "1.729e+63"
    it_converts_to_s 1.729e+64, "1.729e+64"
    it_converts_to_s 1.729e+65, "1.729e+65"
    it_converts_to_s 1.729e+66, "1.729e+66"
    it_converts_to_s 1.729e+67, "1.729e+67"
    it_converts_to_s 1.729e+68, "1.729e+68"
    it_converts_to_s 1.729e+69, "1.729e+69"
    it_converts_to_s 1.729e+70, "1.729e+70"
    it_converts_to_s 1.729e+71, "1.729e+71"
    it_converts_to_s 1.729e+72, "1.729e+72"
    it_converts_to_s 1.729e+73, "1.729e+73"
    it_converts_to_s 1.729e+74, "1.729e+74"
    it_converts_to_s 1.729e+75, "1.729e+75"
    it_converts_to_s 1.729e+76, "1.729e+76"
    it_converts_to_s 1.729e+77, "1.729e+77"
    it_converts_to_s 1.729e+78, "1.729e+78"
    it_converts_to_s 1.729e+79, "1.729e+79"
    it_converts_to_s 1.729e+80, "1.729e+80"
    it_converts_to_s 1.729e+81, "1.729e+81"
    it_converts_to_s 1.729e+82, "1.729e+82"
    it_converts_to_s 1.729e+83, "1.729e+83"
    it_converts_to_s 1.729e+84, "1.729e+84"
    it_converts_to_s 1.729e+85, "1.729e+85"
    it_converts_to_s 1.729e+86, "1.729e+86"
    it_converts_to_s 1.729e+87, "1.729e+87"
    it_converts_to_s 1.729e+88, "1.729e+88"
    it_converts_to_s 1.729e+89, "1.729e+89"
    it_converts_to_s 1.729e+90, "1.729e+90"
    it_converts_to_s 1.729e+91, "1.729e+91"
    it_converts_to_s 1.729e+92, "1.729e+92"
    it_converts_to_s 1.729e+93, "1.729e+93"
    it_converts_to_s 1.729e+94, "1.729e+94"
    it_converts_to_s 1.729e+95, "1.729e+95"
    it_converts_to_s 1.729e+96, "1.729e+96"
    it_converts_to_s 1.729e+97, "1.729e+97"
    it_converts_to_s 1.729e+98, "1.729e+98"
    it_converts_to_s 1.729e+99, "1.729e+99"
    it_converts_to_s 1.729e+100, "1.729e+100"
    it_converts_to_s 1.729e+101, "1.729e+101"
    it_converts_to_s 1.729e+102, "1.729e+102"
    it_converts_to_s 1.729e+103, "1.729e+103"
    it_converts_to_s 1.729e+104, "1.729e+104"
    it_converts_to_s 1.729e+105, "1.729e+105"
    it_converts_to_s 1.729e+106, "1.729e+106"
    it_converts_to_s 1.729e+107, "1.729e+107"
    it_converts_to_s 1.729e+108, "1.729e+108"
    it_converts_to_s 1.729e+109, "1.729e+109"
    it_converts_to_s 1.729e+110, "1.729e+110"
    it_converts_to_s 1.729e+111, "1.729e+111"
    it_converts_to_s 1.729e+112, "1.729e+112"
    it_converts_to_s 1.729e+113, "1.729e+113"
    it_converts_to_s 1.729e+114, "1.729e+114"
    it_converts_to_s 1.729e+115, "1.729e+115"
    it_converts_to_s 1.729e+116, "1.729e+116"
    it_converts_to_s 1.729e+117, "1.729e+117"
    it_converts_to_s 1.729e+118, "1.729e+118"
    it_converts_to_s 1.729e+119, "1.729e+119"
    it_converts_to_s 1.729e+120, "1.729e+120"
    it_converts_to_s 1.729e+121, "1.729e+121"
    it_converts_to_s 1.729e+122, "1.729e+122"
    it_converts_to_s 1.729e+123, "1.729e+123"
    it_converts_to_s 1.729e+124, "1.729e+124"
    it_converts_to_s 1.729e+125, "1.729e+125"
    it_converts_to_s 1.729e+126, "1.729e+126"
    it_converts_to_s 1.729e+127, "1.729e+127"
    it_converts_to_s 1.729e+128, "1.729e+128"
    it_converts_to_s 1.729e+129, "1.729e+129"
    it_converts_to_s 1.729e+130, "1.729e+130"
    it_converts_to_s 1.729e+131, "1.729e+131"
    it_converts_to_s 1.729e+132, "1.729e+132"
    it_converts_to_s 1.729e+133, "1.729e+133"
    it_converts_to_s 1.729e+134, "1.729e+134"
    it_converts_to_s 1.729e+135, "1.729e+135"
    it_converts_to_s 1.729e+136, "1.729e+136"
    it_converts_to_s 1.729e+137, "1.729e+137"
    it_converts_to_s 1.729e+138, "1.729e+138"
    it_converts_to_s 1.729e+139, "1.729e+139"
    it_converts_to_s 1.729e+140, "1.729e+140"
    it_converts_to_s 1.729e+141, "1.729e+141"
    it_converts_to_s 1.729e+142, "1.729e+142"
    it_converts_to_s 1.729e+143, "1.729e+143"
    it_converts_to_s 1.729e+144, "1.729e+144"
    it_converts_to_s 1.729e+145, "1.729e+145"
    it_converts_to_s 1.729e+146, "1.729e+146"
    it_converts_to_s 1.729e+147, "1.729e+147"
    it_converts_to_s 1.729e+148, "1.729e+148"
    it_converts_to_s 1.729e+149, "1.729e+149"
    it_converts_to_s 1.729e+150, "1.729e+150"
    it_converts_to_s 1.729e+151, "1.729e+151"
    it_converts_to_s 1.729e+152, "1.729e+152"
    it_converts_to_s 1.729e+153, "1.729e+153"
    it_converts_to_s 1.729e+154, "1.729e+154"
    it_converts_to_s 1.729e+155, "1.729e+155"
    it_converts_to_s 1.729e+156, "1.729e+156"
    it_converts_to_s 1.729e+157, "1.729e+157"
    it_converts_to_s 1.729e+158, "1.729e+158"
    it_converts_to_s 1.729e+159, "1.729e+159"
    it_converts_to_s 1.729e+160, "1.729e+160"
    it_converts_to_s 1.729e+161, "1.729e+161"
    it_converts_to_s 1.729e+162, "1.729e+162"
    it_converts_to_s 1.729e+163, "1.729e+163"
    it_converts_to_s 1.729e+164, "1.729e+164"
    it_converts_to_s 1.729e+165, "1.729e+165"
    it_converts_to_s 1.729e+166, "1.729e+166"
    it_converts_to_s 1.729e+167, "1.729e+167"
    it_converts_to_s 1.729e+168, "1.729e+168"
    it_converts_to_s 1.729e+169, "1.729e+169"
    it_converts_to_s 1.729e+170, "1.729e+170"
    it_converts_to_s 1.729e+171, "1.729e+171"
    it_converts_to_s 1.729e+172, "1.729e+172"
    it_converts_to_s 1.729e+173, "1.729e+173"
    it_converts_to_s 1.729e+174, "1.729e+174"
    it_converts_to_s 1.729e+175, "1.729e+175"
    it_converts_to_s 1.729e+176, "1.729e+176"
    it_converts_to_s 1.729e+177, "1.729e+177"
    it_converts_to_s 1.729e+178, "1.729e+178"
    it_converts_to_s 1.729e+179, "1.729e+179"
    it_converts_to_s 1.729e+180, "1.729e+180"
    it_converts_to_s 1.729e+181, "1.729e+181"
    it_converts_to_s 1.729e+182, "1.729e+182"
    it_converts_to_s 1.729e+183, "1.729e+183"
    it_converts_to_s 1.729e+184, "1.729e+184"
    it_converts_to_s 1.729e+185, "1.729e+185"
    it_converts_to_s 1.729e+186, "1.729e+186"
    it_converts_to_s 1.729e+187, "1.729e+187"
    it_converts_to_s 1.729e+188, "1.729e+188"
    it_converts_to_s 1.729e+189, "1.729e+189"
    it_converts_to_s 1.729e+190, "1.729e+190"
    it_converts_to_s 1.729e+191, "1.729e+191"
    it_converts_to_s 1.729e+192, "1.729e+192"
    it_converts_to_s 1.729e+193, "1.729e+193"
    it_converts_to_s 1.729e+194, "1.729e+194"
    it_converts_to_s 1.729e+195, "1.729e+195"
    it_converts_to_s 1.729e+196, "1.729e+196"
    it_converts_to_s 1.729e+197, "1.729e+197"
    it_converts_to_s 1.729e+198, "1.729e+198"
    it_converts_to_s 1.729e+199, "1.729e+199"
    it_converts_to_s 1.729e+200, "1.729e+200"
    it_converts_to_s 1.729e+201, "1.729e+201"
    it_converts_to_s 1.729e+202, "1.729e+202"
    it_converts_to_s 1.729e+203, "1.729e+203"
    it_converts_to_s 1.729e+204, "1.729e+204"
    it_converts_to_s 1.729e+205, "1.729e+205"
    it_converts_to_s 1.729e+206, "1.729e+206"
    it_converts_to_s 1.729e+207, "1.729e+207"
    it_converts_to_s 1.729e+208, "1.729e+208"
    it_converts_to_s 1.729e+209, "1.729e+209"
    it_converts_to_s 1.729e+210, "1.729e+210"
    it_converts_to_s 1.729e+211, "1.729e+211"
    it_converts_to_s 1.729e+212, "1.729e+212"
    it_converts_to_s 1.729e+213, "1.729e+213"
    it_converts_to_s 1.729e+214, "1.729e+214"
    it_converts_to_s 1.729e+215, "1.729e+215"
    it_converts_to_s 1.729e+216, "1.729e+216"
    it_converts_to_s 1.729e+217, "1.729e+217"
    it_converts_to_s 1.729e+218, "1.729e+218"
    it_converts_to_s 1.729e+219, "1.729e+219"
    it_converts_to_s 1.729e+220, "1.729e+220"
    it_converts_to_s 1.729e+221, "1.729e+221"
    it_converts_to_s 1.729e+222, "1.729e+222"
    it_converts_to_s 1.729e+223, "1.729e+223"
    it_converts_to_s 1.729e+224, "1.729e+224"
    it_converts_to_s 1.729e+225, "1.729e+225"
    it_converts_to_s 1.729e+226, "1.729e+226"
    it_converts_to_s 1.729e+227, "1.729e+227"
    it_converts_to_s 1.729e+228, "1.729e+228"
    it_converts_to_s 1.729e+229, "1.729e+229"
    it_converts_to_s 1.729e+230, "1.729e+230"
    it_converts_to_s 1.729e+231, "1.729e+231"
    it_converts_to_s 1.729e+232, "1.729e+232"
    it_converts_to_s 1.729e+233, "1.729e+233"
    it_converts_to_s 1.729e+234, "1.729e+234"
    it_converts_to_s 1.729e+235, "1.729e+235"
    it_converts_to_s 1.729e+236, "1.729e+236"
    it_converts_to_s 1.729e+237, "1.729e+237"
    it_converts_to_s 1.729e+238, "1.729e+238"
    it_converts_to_s 1.729e+239, "1.729e+239"
    it_converts_to_s 1.729e+240, "1.729e+240"
    it_converts_to_s 1.729e+241, "1.729e+241"
    it_converts_to_s 1.729e+242, "1.729e+242"
    it_converts_to_s 1.729e+243, "1.729e+243"
    it_converts_to_s 1.729e+244, "1.729e+244"
    it_converts_to_s 1.729e+245, "1.729e+245"
    it_converts_to_s 1.729e+246, "1.729e+246"
    it_converts_to_s 1.729e+247, "1.729e+247"
    it_converts_to_s 1.729e+248, "1.729e+248"
    it_converts_to_s 1.729e+249, "1.729e+249"
    it_converts_to_s 1.729e+250, "1.729e+250"
    it_converts_to_s 1.729e+251, "1.729e+251"
    it_converts_to_s 1.729e+252, "1.729e+252"
    it_converts_to_s 1.729e+253, "1.729e+253"
    it_converts_to_s 1.729e+254, "1.729e+254"
    it_converts_to_s 1.729e+255, "1.729e+255"
    it_converts_to_s 1.729e+256, "1.729e+256"
    it_converts_to_s 1.729e+257, "1.729e+257"
    it_converts_to_s 1.729e+258, "1.729e+258"
    it_converts_to_s 1.729e+259, "1.729e+259"
    it_converts_to_s 1.729e+260, "1.729e+260"
    it_converts_to_s 1.729e+261, "1.729e+261"
    it_converts_to_s 1.729e+262, "1.729e+262"
    it_converts_to_s 1.729e+263, "1.729e+263"
    it_converts_to_s 1.729e+264, "1.729e+264"
    it_converts_to_s 1.729e+265, "1.729e+265"
    it_converts_to_s 1.729e+266, "1.729e+266"
    it_converts_to_s 1.729e+267, "1.729e+267"
    it_converts_to_s 1.729e+268, "1.729e+268"
    it_converts_to_s 1.729e+269, "1.729e+269"
    it_converts_to_s 1.729e+270, "1.729e+270"
    it_converts_to_s 1.729e+271, "1.729e+271"
    it_converts_to_s 1.729e+272, "1.729e+272"
    it_converts_to_s 1.729e+273, "1.729e+273"
    it_converts_to_s 1.729e+274, "1.729e+274"
    it_converts_to_s 1.729e+275, "1.729e+275"
    it_converts_to_s 1.729e+276, "1.729e+276"
    it_converts_to_s 1.729e+277, "1.729e+277"
    it_converts_to_s 1.729e+278, "1.729e+278"
    it_converts_to_s 1.729e+279, "1.729e+279"
    it_converts_to_s 1.729e+280, "1.729e+280"
    it_converts_to_s 1.729e+281, "1.729e+281"
    it_converts_to_s 1.729e+282, "1.729e+282"
    it_converts_to_s 1.729e+283, "1.729e+283"
    it_converts_to_s 1.729e+284, "1.729e+284"
    it_converts_to_s 1.729e+285, "1.729e+285"
    it_converts_to_s 1.729e+286, "1.729e+286"
    it_converts_to_s 1.729e+287, "1.729e+287"
    it_converts_to_s 1.729e+288, "1.729e+288"
    it_converts_to_s 1.729e+289, "1.729e+289"
    it_converts_to_s 1.729e+290, "1.729e+290"
    it_converts_to_s 1.729e+291, "1.729e+291"
    it_converts_to_s 1.729e+292, "1.729e+292"
    it_converts_to_s 1.729e+293, "1.729e+293"
    it_converts_to_s 1.729e+294, "1.729e+294"
    it_converts_to_s 1.729e+295, "1.729e+295"
    it_converts_to_s 1.729e+296, "1.729e+296"
    it_converts_to_s 1.729e+297, "1.729e+297"
    it_converts_to_s 1.729e+298, "1.729e+298"
    it_converts_to_s 1.729e+299, "1.729e+299"
    it_converts_to_s 1.729e+300, "1.729e+300"
    it_converts_to_s 1.729e+301, "1.729e+301"
    it_converts_to_s 1.729e+302, "1.729e+302"
    it_converts_to_s 1.729e+303, "1.729e+303"
    it_converts_to_s 1.729e+304, "1.729e+304"
    it_converts_to_s 1.729e+305, "1.729e+305"
    it_converts_to_s 1.729e+306, "1.729e+306"
    it_converts_to_s 1.729e+307, "1.729e+307"
    it_converts_to_s 1.729e+308, "1.729e+308"
  end

  context "one-digit cases, where the decimal point can't appear between digits like \"17.29\"" do
    it_converts_to_s 7e-3, "0.007"
    it_converts_to_s 7e-2, "0.07"
    it_converts_to_s 7e-1, "0.7"
    it_converts_to_s 7e+0, "7.0"
    it_converts_to_s 7e+1, "70.0"
    it_converts_to_s 7e+2, "700.0"
    it_converts_to_s 7e+3, "7000.0"
  end

  context "highly-trimmed powers of 2" do
    it_converts_to_s hexfloat("0x1p959"), "4.8726570057e+288"
    it_converts_to_s hexfloat("0x1p960"), "9.7453140114e+288"
  end

  context "odd mantissas (unaffected by shifting)" do
    it_converts_to_s 1801439850948197e1, "1.801439850948197e+16"
    it_converts_to_s 360287970189639e2, "3.60287970189639e+16"
    it_converts_to_s 72057594037927e3, "7.2057594037927e+16"
    it_converts_to_s 14411518807585e4, "1.4411518807585e+17"
    it_converts_to_s 2882303761517e5, "2.882303761517e+17"
    it_converts_to_s 576460752303e6, "5.76460752303e+17"
    it_converts_to_s 115292150459e7, "1.15292150459e+18"
    it_converts_to_s 23058430091e8, "2.3058430091e+18"
    it_converts_to_s 4611686017e9, "4.611686017e+18"
    it_converts_to_s 922337203e10, "9.22337203e+18"
    it_converts_to_s 184467439e11, "1.84467439e+19"
    it_converts_to_s 36893487e12, "3.6893487e+19"
    it_converts_to_s 7378697e13, "7.378697e+19"
    it_converts_to_s 1475739e14, "1.475739e+20"
    it_converts_to_s 295147e15, "2.95147e+20"
    it_converts_to_s 59029e16, "5.9029e+20"
    it_converts_to_s 11805e17, "1.1805e+21"
    it_converts_to_s 2361e18, "2.361e+21"
    it_converts_to_s 471e19, "4.71e+21"
    it_converts_to_s 93e20, "9.3e+21"
    it_converts_to_s 17e21, "1.7e+22"
    it_converts_to_s 3e22, "3.0e+22"
    it_converts_to_s 1801439850948199e1, "1.801439850948199e+16"
    it_converts_to_s 360287970189641e2, "3.60287970189641e+16"
    it_converts_to_s 72057594037929e3, "7.2057594037929e+16"
    it_converts_to_s 14411518807587e4, "1.4411518807587e+17"
    it_converts_to_s 2882303761519e5, "2.882303761519e+17"
    it_converts_to_s 576460752305e6, "5.76460752305e+17"
    it_converts_to_s 115292150461e7, "1.15292150461e+18"
    it_converts_to_s 23058430093e8, "2.3058430093e+18"
    it_converts_to_s 4611686019e9, "4.611686019e+18"
    it_converts_to_s 922337205e10, "9.22337205e+18"
    it_converts_to_s 184467441e11, "1.84467441e+19"
    it_converts_to_s 36893489e12, "3.6893489e+19"
    it_converts_to_s 7378699e13, "7.378699e+19"
    it_converts_to_s 1475741e14, "1.475741e+20"
    it_converts_to_s 295149e15, "2.95149e+20"
    it_converts_to_s 59031e16, "5.9031e+20"
    it_converts_to_s 11807e17, "1.1807e+21"
    it_converts_to_s 2363e18, "2.363e+21"
    it_converts_to_s 473e19, "4.73e+21"
    it_converts_to_s 95e20, "9.5e+21"
    it_converts_to_s 19e21, "1.9e+22"
    it_converts_to_s 5e22, "5.0e+22"
    it_converts_to_s 302230528e15, "3.02230528e+23"
    it_converts_to_s 302232576e15, "3.02232576e+23"
    it_converts_to_s 81123342286848e18, "8.1123342286848e+31"
    it_converts_to_s 81192061763584e18, "8.1192061763584e+31"
  end

  context "Grisu failures" do
    it_converts_to_s hexfloat("0x1.e0ffed391517ep-186"), "1.9156918820264798e-56"
    it_converts_to_s hexfloat("0x1.a6c767640cd71p+879"), "6.6564021122018745e+264"
    it_converts_to_s 4.91e-6
    it_converts_to_s 5.547e-6
    it_converts_to_s 1.0e+23
    it_converts_to_s 3.5844466002796428e+298
  end

  # https://www.exploringbinary.com/the-shortest-decimal-string-that-round-trips-may-not-be-the-nearest/
  context "anomalous values" do
    it_converts_to_s hexfloat("0x1p976"), "6.386688990511104e+293"
    it_converts_to_s hexfloat("0x1p896"), "5.282945311356653e+269"
    it_converts_to_s hexfloat("0x1p863"), "6.150157786156811e+259"
    it_converts_to_s hexfloat("0x1p803"), "5.334411546303884e+241"
    it_converts_to_s hexfloat("0x1p710"), "5.386379163185535e+213"
    it_converts_to_s hexfloat("0x1p594"), "6.483618076376552e+178"
    it_converts_to_s hexfloat("0x1p574"), "6.183260036827614e+172"
    it_converts_to_s hexfloat("0x1p554"), "5.896816288783659e+166"
    it_converts_to_s hexfloat("0x1p544"), "5.758609657015292e+163"
    it_converts_to_s hexfloat("0x1p534"), "5.623642243178996e+160"
    it_converts_to_s hexfloat("0x1p481"), "6.243497100631985e+144"
    it_converts_to_s hexfloat("0x1p405"), "8.263199609878108e+121"
    it_converts_to_s hexfloat("0x1p398"), "6.455624695217272e+119"
    it_converts_to_s hexfloat("0x1p378"), "6.156563468186638e+113"
    it_converts_to_s hexfloat("0x1p345"), "7.167183174968974e+103"
    it_converts_to_s hexfloat("0x1p305"), "6.518515124270356e+91"
    it_converts_to_s hexfloat("0x1p275"), "6.070840288205404e+82"
    it_converts_to_s hexfloat("0x1p182"), "6.129982163463556e+54"
    it_converts_to_s hexfloat("0x1p172"), "5.986310706507379e+51"
    it_converts_to_s hexfloat("0x1p132"), "5.444517870735016e+39"
    it_converts_to_s hexfloat("0x1p122"), "5.316911983139664e+36"
    it_converts_to_s hexfloat("0x1p89"), "6.189700196426902e+26"
    it_converts_to_s hexfloat("0x1p-24"), "5.960464477539063e-8"
    it_converts_to_s hexfloat("0x1p-44"), "5.684341886080802e-14"
    it_converts_to_s hexfloat("0x1p-77"), "6.617444900424222e-24"
    it_converts_to_s hexfloat("0x1p-97"), "6.310887241768095e-30"
    it_converts_to_s hexfloat("0x1p-140"), "7.174648137343064e-43"
    it_converts_to_s hexfloat("0x1p-296"), "7.854549544476363e-90"
    it_converts_to_s hexfloat("0x1p-366"), "6.653062250012736e-111"
    it_converts_to_s hexfloat("0x1p-383"), "5.075883674631299e-116"
    it_converts_to_s hexfloat("0x1p-489"), "6.256509672447191e-148"
    it_converts_to_s hexfloat("0x1p-496"), "4.887898181599368e-150"
    it_converts_to_s hexfloat("0x1p-509"), "5.966672584960166e-154"
    it_converts_to_s hexfloat("0x1p-549"), "5.426657103235053e-166"
    it_converts_to_s hexfloat("0x1p-652"), "5.351097043477547e-197"
    it_converts_to_s hexfloat("0x1p-662"), "5.225680706521042e-200"
    it_converts_to_s hexfloat("0x1p-695"), "6.083493012144512e-210"
    it_converts_to_s hexfloat("0x1p-705"), "5.940911144672375e-213"
    it_converts_to_s hexfloat("0x1p-778"), "6.290184345309701e-235"
    it_converts_to_s hexfloat("0x1p-788"), "6.142758149716505e-238"
    it_converts_to_s hexfloat("0x1p-791"), "7.678447687145631e-239"
    it_converts_to_s hexfloat("0x1p-808"), "5.858190679279809e-244"
    it_converts_to_s hexfloat("0x1p-921"), "5.641232424577593e-278"
    it_converts_to_s hexfloat("0x1p-957"), "8.209073602596753e-289"
    it_converts_to_s hexfloat("0x1p-1007"), "7.291122019556398e-304"
    it_converts_to_s hexfloat("0x1p-1017"), "7.120236347223045e-307"
  end

  context "almost-but-not-quite-anomalous values" do
    it_converts_to_s hexfloat("0x1p966"), "6.237000967296e+290"
    it_converts_to_s hexfloat("0x1p956"), "6.090821257125e+287"
    it_converts_to_s hexfloat("0x1p890"), "8.25460204899477e+267"
    it_converts_to_s hexfloat("0x1p740"), "5.78358058743443e+222"
    it_converts_to_s hexfloat("0x1p149"), "7.1362384635298e+44"
    it_converts_to_s hexfloat("0x1p-499"), "6.10987272699921e-151"
    it_converts_to_s hexfloat("0x1p-569"), "5.17526350329881e-172"
    it_converts_to_s hexfloat("0x1p-645"), "6.84940421565126e-195"
  end
end
