<?php
/**
* DAViCal extensions to AWL Session handling
*
* @package   davical
* @subpackage   DAViCalSession
* @author Andrew McMillan <andrew@mcmillan.net.nz>
* @copyright Catalyst .Net Ltd, Morphoss Ltd <http://www.morphoss.com/>
* @license   http://gnu.org/copyleft/gpl.html GNU GPL v2
*/

/**
* @global resource $session
* @name $session
* The session object is global.
*/
$session = 1;  // Fake initialisation

// The Session object uses some (optional) configurable SQL to load
// the records related to the logged-on user...  (the where clause gets added).
// It's very important that someone not be able to externally control this,
// so we make it a function rather than a variable.
/**
* @todo Make this a defined constant
*/
function local_session_sql() {
  $sql = <<<EOSQL
SELECT session.*, usr.*, principal.*
        FROM session JOIN usr USING(user_no) JOIN principal USING(user_no)
EOSQL;
  return $sql;
}

/**
* We extend the AWL Session class.
*/
require('Session.php');
include_once('DAVResource.php');


@Session::_CheckLogout();

/**
* A class for creating and holding session information.
*
* @package   davical
*/
class DAViCalSession extends Session
{

  public $principal_id;
  private $privilege_resources = array();

  /**
  * Create a new DAViCalSession object.
  *
  * We create a Session and extend it with some additional useful DAViCal
  * related information.
  *
  * @param string $sid A session identifier.
  */
  function __construct( $sid='' ) {
    $this->principal_id = null;
    parent::__construct($sid);
  }


  /**
  * Internal function used to assign the session details to a user's new session.
  * @param object $u The user+session object we (probably) read from the database.
  */
  function AssignSessionDetails( $u ) {
    if ( !isset($u->principal_id) ) {
      // If they don't have a principal_id set then we should re-read from our local database
      $qry = new AwlQuery('SELECT * FROM dav_principal WHERE username = :username', array(':username' => $u->username) );
      if ( $qry->Exec() && $qry->rows() == 1 ) {
        $u = $qry->Fetch();
      }
    }

    parent::AssignSessionDetails( $u );
    $this->GetRoles();
    if ( function_exists('awl_set_locale') && isset($this->locale) && $this->locale != '' ) {
      awl_set_locale($this->locale);
    }
  }


  /**
  * Method used to get the user's roles
  */
  function GetRoles () {
    $this->roles = array();
    $sql = 'SELECT role_name FROM roles JOIN role_member ON roles.role_no=role_member.role_no WHERE user_no = '.$this->user_no;
    $qry = new AwlQuery( $sql );
    if ( $qry->Exec('DAViCalSession') && $qry->rows() > 0 ) {
      while( $role = $qry->Fetch() ) {
        $this->roles[$role->role_name] = 1;
      }
    }
    // inherit the Admin role
    $sql = 'SELECT role_name FROM (((group_member JOIN dav_principal first_dav_principal ON group_member.group_id=first_dav_principal.principal_id) JOIN role_member ON first_dav_principal.user_no=role_member.user_no) JOIN roles ON roles.role_no=role_member.role_no) JOIN dav_principal second_dav_principal ON group_member.member_id=second_dav_principal.principal_id WHERE second_dav_principal.user_no = '.$this->user_no;
    $qry = new AwlQuery( $sql );
    if ( $qry->Exec('DAViCalSession') && $qry->rows() > 0 ) {
      while( $role = $qry->Fetch() ) {
        if($role->role_name=='Admin')
          $this->roles['Admin'] = 1;
      }
    }
  }


  /**
   * Does the user have the privileges to do what is requested.
   * @param $do_what mixed The request privilege name, or array of privilege names, to be checked.
   * @param $path string The path we want that permission for
   * @param $any boolean Whether we accept any of the privileges. The default is true, unless the requested privilege is 'all', when it is false.
   * @return boolean Whether they do have one of those privileges against the specified path.
   */
  function HavePrivilegeTo( $do_what, $path, $any = null ) {
    if ( $this->AllowedTo('Admin') ) return true;
    if ( !isset($this->privilege_resources[$path]) ) {
      $this->privilege_resources[$path] = new DAVResource($path);
    }
    $resource = $this->privilege_resources[$path];
    if ( isset($resource) && $resource->Exists() ) {
      return $resource->HavePrivilegeTo($do_what,$any);
    }
    return false;
  }


  /**
   * Renders some HTML for a basic login panel
   *
   * @return string The HTML to display a login panel.
   */
  function RenderLoginPanel() {
    global $c;
    $action_target = htmlspecialchars(preg_replace('/\?logout.*$/','',$_SERVER['REQUEST_URI']));
    dbg_error_log( "Login", " RenderLoginPanel: action_target='%s'", $action_target );
    $userprompt = translate("User Name");
    $pwprompt = translate("Password");
    $gobutton = htmlspecialchars(translate("GO!"));
    $gotitle = htmlspecialchars(translate("Enter your username and password then click here to log in."));
    $temppwprompt = translate("If you have forgotten your password then");
    $temppwbutton = htmlspecialchars(translate("Help! I've forgotten my password!"));
    if (isset($c->password_change_override) ) {
      $temppw_html = '<a href="' . $c->password_change_override['href'] . '">' . $c->password_change_override['label'] . '</a>';
    } else {
      $temppwtitle = htmlspecialchars(translate("Enter a username, if you know it, and click here, to be e-mailed a temporary password."));
      $temppw_html = '<input type="submit" value="' . $temppwbutton . '" title="' . $temppwtitle . '" name="lostpass" class="submit" />';
    }
    $html = <<<EOTEXT
<div id="logon">
<form action="$action_target" method="post">
<table>
<tr>
<th class="prompt">$userprompt:</th>
<td class="entry">
<input class="text" type="text" name="username" size="12" /></td>
</tr>
<tr>
<th class="prompt">$pwprompt:</th>
<td class="entry">
<input class="password" type="password" name="password" size="12" />
</td>
</tr>
<tr>
<th class="prompt">&nbsp;</th>
<td class="entry">
<input type="submit" value="$gobutton" title="$gotitle" name="submit" class="submit" />
</td>
</tr>
</table>
<p>
$temppwprompt: $temppw_html
</p>
</form>
</div>

EOTEXT;
    return $html;
  }


  /**
  * Checks that this user is logged in, and presents a login screen if they aren't.
  *
  * The function can optionally confirm whether they are a member of one of a list
  * of roles, and deny access if they are not a member of any of them.
  *
  * @param string $roles The list of roles that the user must be a member of one of to be allowed to proceed.
  * @return boolean Whether or not the user is logged in and is a member of one of the required roles.
  */
  function LoginRequired( $roles = '' ) {
    global $c, $session, $main_menu, $sub_menu, $tab_menu;

    $current_domain = (isset($_SERVER['SERVER_NAME'])?$_SERVER['SERVER_NAME']:$_SERVER['SERVER_ADDR']);
    if ( (isset($c->restrict_admin_domain) && $c->restrict_admin_domain != $current_domain)
      || (isset($c->restrict_admin_port) && $c->restrict_admin_port != $_SERVER['SERVER_PORT'] ) ) {
      header('Location: caldav.php');
      dbg_error_log( 'LOG WARNING', 'Access to "%s" via "%s:%d" rejected.', $_SERVER['REQUEST_URI'], $current_domain, $_SERVER['SERVER_PORT'] );
      @ob_flush(); exit(0);
    }
    if ( isset($c->restrict_admin_roles) && $roles == '' ) $roles = $c->restrict_admin_roles;
    if ( $this->logged_in && $roles == '' ) return;

    /**
     * We allow basic auth to apply also, if present, though we check everything else first...
     */
    if ( isset($_SERVER['PHP_AUTH_USER']) && !$this->logged_in && $_SERVER['PHP_AUTH_USER'] != "" && $_SERVER['PHP_AUTH_PW'] != "" && ! $_COOKIE['NoAutoLogin'] ) {
      if ( $this->Login($_SERVER['PHP_AUTH_USER'],$_SERVER['PHP_AUTH_PW'],false)) {
        setcookie('NoAutoLogin',1,0);
        return;
      }
    }
    if ( ! $this->logged_in ) {
      $c->messages[] = i18n('You must log in to use this system.');
      include_once('page-header.php');
      if ( function_exists('local_index_not_logged_in') ) {
        local_index_not_logged_in();
      }
      else {
        if ( $this->login_failed ) {
          $c->messages[] = i18n('Invalid user name or password.');
        }
        echo '<h1>'.translate('Log On Please')."</h1>\n";
        echo '<p>'.translate('For access to the')
                  .' '.translate($c->system_name).' '
                  .translate('you should log on with the username and password that have been issued to you.')
            ."</p>\n";
        echo '<p>'.translate('If you would like to request access, please e-mail').' '.$c->admin_email."</p>\n";
        echo $this->RenderLoginPanel();
      }
    }
    else {
      $valid_roles = explode(',', $roles);
      foreach( $valid_roles AS $k => $v ) {
        if ( $this->AllowedTo($v) ) return;
      }
      $c->messages[] = i18n('You are not authorised to use this function.');
      include_once('page-header.php');
    }

    include('page-footer.php');
    @ob_flush(); exit(0);
  }
}

$session = new DAViCalSession();
$session->_CheckLogin();

