# frozen_string_literal: true

# Somehow in the course of converting primary keys from int to bigint or
# when partitioning CI tables, a number of sequences somehow got
# assigned to the p_ci_builds table. This migration fixes the sequence
# ownership based on known good values from db/structure.sql.
class FixSequenceOwnersForCiBuilds < Gitlab::Database::Migration[2.2]
  milestone '17.3'
  disable_ddl_transaction!

  # grep 'ALTER SEQUENCE' db/structure.sql | sed "s/ALTER SEQUENCE /['/" | sed "s/ OWNED BY /', '/" | sed "s/;/'],/"
  SEQUENCES = [
    ['abuse_events_id_seq', 'abuse_events.id'],
    ['abuse_report_assignees_id_seq', 'abuse_report_assignees.id'],
    ['abuse_report_events_id_seq', 'abuse_report_events.id'],
    ['abuse_report_notes_id_seq', 'abuse_report_notes.id'],
    ['abuse_report_user_mentions_id_seq', 'abuse_report_user_mentions.id'],
    ['abuse_reports_id_seq', 'abuse_reports.id'],
    ['abuse_trust_scores_id_seq', 'abuse_trust_scores.id'],
    ['achievements_id_seq', 'achievements.id'],
    ['activity_pub_releases_subscriptions_id_seq', 'activity_pub_releases_subscriptions.id'],
    ['agent_activity_events_id_seq', 'agent_activity_events.id'],
    ['agent_group_authorizations_id_seq', 'agent_group_authorizations.id'],
    ['agent_project_authorizations_id_seq', 'agent_project_authorizations.id'],
    ['agent_user_access_group_authorizations_id_seq', 'agent_user_access_group_authorizations.id'],
    ['agent_user_access_project_authorizations_id_seq', 'agent_user_access_project_authorizations.id'],
    ['ai_agent_version_attachments_id_seq', 'ai_agent_version_attachments.id'],
    ['ai_agent_versions_id_seq', 'ai_agent_versions.id'],
    ['ai_agents_id_seq', 'ai_agents.id'],
    ['ai_feature_settings_id_seq', 'ai_feature_settings.id'],
    ['ai_self_hosted_models_id_seq', 'ai_self_hosted_models.id'],
    ['ai_vectorizable_files_id_seq', 'ai_vectorizable_files.id'],
    ['alert_management_alert_assignees_id_seq', 'alert_management_alert_assignees.id'],
    ['alert_management_alert_metric_images_id_seq', 'alert_management_alert_metric_images.id'],
    ['alert_management_alert_user_mentions_id_seq', 'alert_management_alert_user_mentions.id'],
    ['alert_management_alerts_id_seq', 'alert_management_alerts.id'],
    ['alert_management_http_integrations_id_seq', 'alert_management_http_integrations.id'],
    ['allowed_email_domains_id_seq', 'allowed_email_domains.id'],
    ['analytics_cycle_analytics_group_stages_id_seq', 'analytics_cycle_analytics_group_stages.id'],
    ['analytics_cycle_analytics_group_value_streams_id_seq', 'analytics_cycle_analytics_group_value_streams.id'],
    ['analytics_cycle_analytics_stage_event_hashes_id_seq', 'analytics_cycle_analytics_stage_event_hashes.id'],
    ['analytics_dashboards_pointers_id_seq', 'analytics_dashboards_pointers.id'],
    ['analytics_devops_adoption_segments_id_seq', 'analytics_devops_adoption_segments.id'],
    ['analytics_devops_adoption_snapshots_id_seq', 'analytics_devops_adoption_snapshots.id'],
    ['analytics_usage_trends_measurements_id_seq', 'analytics_usage_trends_measurements.id'],
    ['appearances_id_seq', 'appearances.id'],
    ['application_setting_terms_id_seq', 'application_setting_terms.id'],
    ['application_settings_id_seq', 'application_settings.id'],
    ['approval_group_rules_groups_id_seq', 'approval_group_rules_groups.id'],
    ['approval_group_rules_id_seq', 'approval_group_rules.id'],
    ['approval_group_rules_protected_branches_id_seq', 'approval_group_rules_protected_branches.id'],
    ['approval_group_rules_users_id_seq', 'approval_group_rules_users.id'],
    ['approval_merge_request_rule_sources_id_seq', 'approval_merge_request_rule_sources.id'],
    ['approval_merge_request_rules_approved_approvers_id_seq', 'approval_merge_request_rules_approved_approvers.id'],
    ['approval_merge_request_rules_groups_id_seq', 'approval_merge_request_rules_groups.id'],
    ['approval_merge_request_rules_id_seq', 'approval_merge_request_rules.id'],
    ['approval_merge_request_rules_users_id_seq', 'approval_merge_request_rules_users.id'],
    ['approval_policy_rules_id_seq', 'approval_policy_rules.id'],
    ['approval_project_rules_groups_id_seq', 'approval_project_rules_groups.id'],
    ['approval_project_rules_id_seq', 'approval_project_rules.id'],
    ['approval_project_rules_users_id_seq', 'approval_project_rules_users.id'],
    ['approvals_id_seq', 'approvals.id'],
    ['approver_groups_id_seq', 'approver_groups.id'],
    ['approvers_id_seq', 'approvers.id'],
    ['atlassian_identities_user_id_seq', 'atlassian_identities.user_id'],
    ['audit_events_amazon_s3_configurations_id_seq', 'audit_events_amazon_s3_configurations.id'],
    ['audit_events_external_audit_event_destinations_id_seq', 'audit_events_external_audit_event_destinations.id'],
    ['audit_events_google_cloud_logging_configurations_id_seq', 'audit_events_google_cloud_logging_configurations.id'],
    ['audit_events_group_external_streaming_destinations_id_seq',
      'audit_events_group_external_streaming_destinations.id'],
    ['audit_events_group_streaming_event_type_filters_id_seq', 'audit_events_group_streaming_event_type_filters.id'],
    ['audit_events_id_seq', 'audit_events.id'],
    ['audit_events_instance_amazon_s3_configurations_id_seq', 'audit_events_instance_amazon_s3_configurations.id'],
    ['audit_events_instance_external_audit_event_destinations_id_seq',
      'audit_events_instance_external_audit_event_destinations.id'],
    ['audit_events_instance_external_streaming_destinations_id_seq',
      'audit_events_instance_external_streaming_destinations.id'],
    ['audit_events_instance_google_cloud_logging_configuration_id_seq',
      'audit_events_instance_google_cloud_logging_configurations.id'],
    ['audit_events_instance_streaming_event_type_filters_id_seq',
      'audit_events_instance_streaming_event_type_filters.id'],
    ['audit_events_streaming_event_type_filters_id_seq', 'audit_events_streaming_event_type_filters.id'],
    ['audit_events_streaming_group_namespace_filters_id_seq', 'audit_events_streaming_group_namespace_filters.id'],
    ['audit_events_streaming_headers_id_seq', 'audit_events_streaming_headers.id'],
    ['audit_events_streaming_http_group_namespace_filters_id_seq',
      'audit_events_streaming_http_group_namespace_filters.id'],
    ['audit_events_streaming_http_instance_namespace_filters_id_seq',
      'audit_events_streaming_http_instance_namespace_filters.id'],
    ['audit_events_streaming_instance_event_type_filters_id_seq',
      'audit_events_streaming_instance_event_type_filters.id'],
    ['audit_events_streaming_instance_namespace_filters_id_seq',
      'audit_events_streaming_instance_namespace_filters.id'],
    ['authentication_events_id_seq', 'authentication_events.id'],
    ['automation_rules_id_seq', 'automation_rules.id'],
    ['award_emoji_id_seq', 'award_emoji.id'],
    ['background_migration_jobs_id_seq', 'background_migration_jobs.id'],
    ['badges_id_seq', 'badges.id'],
    ['batched_background_migration_job_transition_logs_id_seq', 'batched_background_migration_job_transition_logs.id'],
    ['batched_background_migration_jobs_id_seq', 'batched_background_migration_jobs.id'],
    ['batched_background_migrations_id_seq', 'batched_background_migrations.id'],
    ['board_assignees_id_seq', 'board_assignees.id'],
    ['board_group_recent_visits_id_seq', 'board_group_recent_visits.id'],
    ['board_labels_id_seq', 'board_labels.id'],
    ['board_project_recent_visits_id_seq', 'board_project_recent_visits.id'],
    ['board_user_preferences_id_seq', 'board_user_preferences.id'],
    ['boards_epic_board_labels_id_seq', 'boards_epic_board_labels.id'],
    ['boards_epic_board_positions_id_seq', 'boards_epic_board_positions.id'],
    ['boards_epic_board_recent_visits_id_seq', 'boards_epic_board_recent_visits.id'],
    ['boards_epic_boards_id_seq', 'boards_epic_boards.id'],
    ['boards_epic_list_user_preferences_id_seq', 'boards_epic_list_user_preferences.id'],
    ['boards_epic_lists_id_seq', 'boards_epic_lists.id'],
    ['boards_epic_user_preferences_id_seq', 'boards_epic_user_preferences.id'],
    ['boards_id_seq', 'boards.id'],
    ['broadcast_messages_id_seq', 'broadcast_messages.id'],
    ['bulk_import_batch_trackers_id_seq', 'bulk_import_batch_trackers.id'],
    ['bulk_import_configurations_id_seq', 'bulk_import_configurations.id'],
    ['bulk_import_entities_id_seq', 'bulk_import_entities.id'],
    ['bulk_import_export_batches_id_seq', 'bulk_import_export_batches.id'],
    ['bulk_import_export_uploads_id_seq', 'bulk_import_export_uploads.id'],
    ['bulk_import_exports_id_seq', 'bulk_import_exports.id'],
    ['bulk_import_failures_id_seq', 'bulk_import_failures.id'],
    ['bulk_import_trackers_id_seq', 'bulk_import_trackers.id'],
    ['bulk_imports_id_seq', 'bulk_imports.id'],
    ['catalog_resource_components_id_seq', 'catalog_resource_components.id'],
    ['catalog_resource_versions_id_seq', 'catalog_resource_versions.id'],
    ['catalog_resources_id_seq', 'catalog_resources.id'],
    ['catalog_verified_namespaces_id_seq', 'catalog_verified_namespaces.id'],
    ['chat_names_id_seq', 'chat_names.id'],
    ['chat_teams_id_seq', 'chat_teams.id'],
    ['ci_build_needs_id_seq', 'ci_build_needs.id'],
    ['ci_build_pending_states_id_seq', 'ci_build_pending_states.id'],
    ['ci_build_trace_chunks_id_seq', 'ci_build_trace_chunks.id'],
    ['ci_builds_id_seq', 'p_ci_builds.id'],
    ['ci_builds_metadata_id_seq', 'p_ci_builds_metadata.id'],
    ['ci_builds_runner_session_id_seq', 'ci_builds_runner_session.id'],
    ['ci_daily_build_group_report_results_id_seq', 'ci_daily_build_group_report_results.id'],
    ['ci_deleted_objects_id_seq', 'ci_deleted_objects.id'],
    ['ci_freeze_periods_id_seq', 'ci_freeze_periods.id'],
    ['ci_group_variables_id_seq', 'ci_group_variables.id'],
    ['ci_instance_variables_id_seq', 'ci_instance_variables.id'],
    ['ci_job_artifacts_id_seq', 'p_ci_job_artifacts.id'],
    ['ci_job_token_group_scope_links_id_seq', 'ci_job_token_group_scope_links.id'],
    ['ci_job_token_project_scope_links_id_seq', 'ci_job_token_project_scope_links.id'],
    ['ci_job_variables_id_seq', 'ci_job_variables.id'],
    ['ci_minutes_additional_packs_id_seq', 'ci_minutes_additional_packs.id'],
    ['ci_namespace_mirrors_id_seq', 'ci_namespace_mirrors.id'],
    ['ci_namespace_monthly_usages_id_seq', 'ci_namespace_monthly_usages.id'],
    ['ci_pending_builds_id_seq', 'ci_pending_builds.id'],
    ['ci_pipeline_artifacts_id_seq', 'ci_pipeline_artifacts.id'],
    ['ci_pipeline_chat_data_id_seq', 'ci_pipeline_chat_data.id'],
    ['ci_pipeline_messages_id_seq', 'ci_pipeline_messages.id'],
    ['ci_pipeline_schedule_variables_id_seq', 'ci_pipeline_schedule_variables.id'],
    ['ci_pipeline_schedules_id_seq', 'ci_pipeline_schedules.id'],
    ['ci_pipeline_variables_id_seq', 'p_ci_pipeline_variables.id'],
    ['ci_pipelines_id_seq', 'ci_pipelines.id'],
    ['ci_platform_metrics_id_seq', 'ci_platform_metrics.id'],
    ['ci_project_mirrors_id_seq', 'ci_project_mirrors.id'],
    ['ci_project_monthly_usages_id_seq', 'ci_project_monthly_usages.id'],
    ['ci_refs_id_seq', 'ci_refs.id'],
    ['ci_resource_groups_id_seq', 'ci_resource_groups.id'],
    ['ci_resources_id_seq', 'ci_resources.id'],
    ['ci_runner_machines_id_seq', 'ci_runner_machines.id'],
    ['ci_runner_namespaces_id_seq', 'ci_runner_namespaces.id'],
    ['ci_runner_projects_id_seq', 'ci_runner_projects.id'],
    ['ci_runners_id_seq', 'ci_runners.id'],
    ['ci_running_builds_id_seq', 'ci_running_builds.id'],
    ['ci_secure_file_states_ci_secure_file_id_seq', 'ci_secure_file_states.ci_secure_file_id'],
    ['ci_secure_files_id_seq', 'ci_secure_files.id'],
    ['ci_sources_pipelines_id_seq', 'ci_sources_pipelines.id'],
    ['ci_sources_projects_id_seq', 'ci_sources_projects.id'],
    ['ci_stages_id_seq', 'p_ci_stages.id'],
    ['ci_subscriptions_projects_id_seq', 'ci_subscriptions_projects.id'],
    ['ci_trigger_requests_id_seq', 'ci_trigger_requests.id'],
    ['ci_triggers_id_seq', 'ci_triggers.id'],
    ['ci_unit_test_failures_id_seq', 'ci_unit_test_failures.id'],
    ['ci_unit_tests_id_seq', 'ci_unit_tests.id'],
    ['ci_variables_id_seq', 'ci_variables.id'],
    ['cloud_connector_access_id_seq', 'cloud_connector_access.id'],
    ['cluster_agent_tokens_id_seq', 'cluster_agent_tokens.id'],
    ['cluster_agents_id_seq', 'cluster_agents.id'],
    ['cluster_enabled_grants_id_seq', 'cluster_enabled_grants.id'],
    ['cluster_groups_id_seq', 'cluster_groups.id'],
    ['cluster_platforms_kubernetes_id_seq', 'cluster_platforms_kubernetes.id'],
    ['cluster_projects_id_seq', 'cluster_projects.id'],
    ['cluster_providers_aws_id_seq', 'cluster_providers_aws.id'],
    ['cluster_providers_gcp_id_seq', 'cluster_providers_gcp.id'],
    ['clusters_id_seq', 'clusters.id'],
    ['clusters_kubernetes_namespaces_id_seq', 'clusters_kubernetes_namespaces.id'],
    ['commit_user_mentions_id_seq', 'commit_user_mentions.id'],
    ['compliance_framework_security_policies_id_seq', 'compliance_framework_security_policies.id'],
    ['compliance_management_frameworks_id_seq', 'compliance_management_frameworks.id'],
    ['container_registry_protection_rules_id_seq', 'container_registry_protection_rules.id'],
    ['container_repositories_id_seq', 'container_repositories.id'],
    ['content_blocked_states_id_seq', 'content_blocked_states.id'],
    ['conversational_development_index_metrics_id_seq', 'conversational_development_index_metrics.id'],
    ['country_access_logs_id_seq', 'country_access_logs.id'],
    ['coverage_fuzzing_corpuses_id_seq', 'coverage_fuzzing_corpuses.id'],
    ['csv_issue_imports_id_seq', 'csv_issue_imports.id'],
    ['custom_emoji_id_seq', 'custom_emoji.id'],
    ['custom_software_licenses_id_seq', 'custom_software_licenses.id'],
    ['customer_relations_contacts_id_seq', 'customer_relations_contacts.id'],
    ['customer_relations_organizations_id_seq', 'customer_relations_organizations.id'],
    ['dast_pre_scan_verification_steps_id_seq', 'dast_pre_scan_verification_steps.id'],
    ['dast_pre_scan_verifications_id_seq', 'dast_pre_scan_verifications.id'],
    ['dast_profile_schedules_id_seq', 'dast_profile_schedules.id'],
    ['dast_profiles_id_seq', 'dast_profiles.id'],
    ['dast_profiles_tags_id_seq', 'dast_profiles_tags.id'],
    ['dast_scanner_profiles_id_seq', 'dast_scanner_profiles.id'],
    ['dast_site_profile_secret_variables_id_seq', 'dast_site_profile_secret_variables.id'],
    ['dast_site_profiles_id_seq', 'dast_site_profiles.id'],
    ['dast_site_tokens_id_seq', 'dast_site_tokens.id'],
    ['dast_site_validations_id_seq', 'dast_site_validations.id'],
    ['dast_sites_id_seq', 'dast_sites.id'],
    ['dependency_list_export_parts_id_seq', 'dependency_list_export_parts.id'],
    ['dependency_list_exports_id_seq', 'dependency_list_exports.id'],
    ['dependency_proxy_blobs_id_seq', 'dependency_proxy_blobs.id'],
    ['dependency_proxy_group_settings_id_seq', 'dependency_proxy_group_settings.id'],
    ['dependency_proxy_manifests_id_seq', 'dependency_proxy_manifests.id'],
    ['deploy_keys_projects_id_seq', 'deploy_keys_projects.id'],
    ['deploy_tokens_id_seq', 'deploy_tokens.id'],
    ['deployment_approvals_id_seq', 'deployment_approvals.id'],
    ['deployments_id_seq', 'deployments.id'],
    ['description_versions_id_seq', 'description_versions.id'],
    ['design_management_designs_id_seq', 'design_management_designs.id'],
    ['design_management_designs_versions_id_seq', 'design_management_designs_versions.id'],
    ['design_management_repositories_id_seq', 'design_management_repositories.id'],
    ['design_management_versions_id_seq', 'design_management_versions.id'],
    ['design_user_mentions_id_seq', 'design_user_mentions.id'],
    ['detached_partitions_id_seq', 'detached_partitions.id'],
    ['diff_note_positions_id_seq', 'diff_note_positions.id'],
    ['dingtalk_tracker_data_id_seq', 'dingtalk_tracker_data.id'],
    ['dora_configurations_id_seq', 'dora_configurations.id'],
    ['dora_daily_metrics_id_seq', 'dora_daily_metrics.id'],
    ['dora_performance_scores_id_seq', 'dora_performance_scores.id'],
    ['draft_notes_id_seq', 'draft_notes.id'],
    ['duo_workflows_checkpoints_id_seq', 'duo_workflows_checkpoints.id'],
    ['duo_workflows_workflows_id_seq', 'duo_workflows_workflows.id'],
    ['early_access_program_tracking_events_id_seq', 'early_access_program_tracking_events.id'],
    ['elastic_index_settings_id_seq', 'elastic_index_settings.id'],
    ['elastic_reindexing_slices_id_seq', 'elastic_reindexing_slices.id'],
    ['elastic_reindexing_subtasks_id_seq', 'elastic_reindexing_subtasks.id'],
    ['elastic_reindexing_tasks_id_seq', 'elastic_reindexing_tasks.id'],
    ['emails_id_seq', 'emails.id'],
    ['environments_id_seq', 'environments.id'],
    ['epic_issues_id_seq', 'epic_issues.id'],
    ['epic_metrics_id_seq', 'epic_metrics.id'],
    ['epic_user_mentions_id_seq', 'epic_user_mentions.id'],
    ['epics_id_seq', 'epics.id'],
    ['error_tracking_client_keys_id_seq', 'error_tracking_client_keys.id'],
    ['error_tracking_error_events_id_seq', 'error_tracking_error_events.id'],
    ['error_tracking_errors_id_seq', 'error_tracking_errors.id'],
    ['events_id_seq', 'events.id'],
    ['evidences_id_seq', 'evidences.id'],
    ['external_approval_rules_id_seq', 'external_approval_rules.id'],
    ['external_pull_requests_id_seq', 'external_pull_requests.id'],
    ['external_status_checks_id_seq', 'external_status_checks.id'],
    ['external_status_checks_protected_branches_id_seq', 'external_status_checks_protected_branches.id'],
    ['feature_gates_id_seq', 'feature_gates.id'],
    ['features_id_seq', 'features.id'],
    ['fork_network_members_id_seq', 'fork_network_members.id'],
    ['fork_networks_id_seq', 'fork_networks.id'],
    ['geo_cache_invalidation_events_id_seq', 'geo_cache_invalidation_events.id'],
    ['geo_event_log_id_seq', 'geo_event_log.id'],
    ['geo_events_id_seq', 'geo_events.id'],
    ['geo_node_namespace_links_id_seq', 'geo_node_namespace_links.id'],
    ['geo_node_statuses_id_seq', 'geo_node_statuses.id'],
    ['geo_nodes_id_seq', 'geo_nodes.id'],
    ['ghost_user_migrations_id_seq', 'ghost_user_migrations.id'],
    ['gitlab_subscription_histories_id_seq', 'gitlab_subscription_histories.id'],
    ['gitlab_subscriptions_id_seq', 'gitlab_subscriptions.id'],
    ['gpg_key_subkeys_id_seq', 'gpg_key_subkeys.id'],
    ['gpg_keys_id_seq', 'gpg_keys.id'],
    ['gpg_signatures_id_seq', 'gpg_signatures.id'],
    ['grafana_integrations_id_seq', 'grafana_integrations.id'],
    ['group_crm_settings_group_id_seq', 'group_crm_settings.group_id'],
    ['group_custom_attributes_id_seq', 'group_custom_attributes.id'],
    ['group_deploy_keys_groups_id_seq', 'group_deploy_keys_groups.id'],
    ['group_deploy_keys_id_seq', 'group_deploy_keys.id'],
    ['group_deploy_tokens_id_seq', 'group_deploy_tokens.id'],
    ['group_group_links_id_seq', 'group_group_links.id'],
    ['group_import_states_group_id_seq', 'group_import_states.group_id'],
    ['group_repository_storage_moves_id_seq', 'group_repository_storage_moves.id'],
    ['group_saved_replies_id_seq', 'group_saved_replies.id'],
    ['group_ssh_certificates_id_seq', 'group_ssh_certificates.id'],
    ['group_wiki_repository_states_id_seq', 'group_wiki_repository_states.id'],
    ['groups_visits_id_seq', 'groups_visits.id'],
    ['historical_data_id_seq', 'historical_data.id'],
    ['identities_id_seq', 'identities.id'],
    ['import_export_uploads_id_seq', 'import_export_uploads.id'],
    ['import_failures_id_seq', 'import_failures.id'],
    ['import_source_user_placeholder_references_id_seq', 'import_source_user_placeholder_references.id'],
    ['import_source_users_id_seq', 'import_source_users.id'],
    ['incident_management_escalation_policies_id_seq', 'incident_management_escalation_policies.id'],
    ['incident_management_escalation_rules_id_seq', 'incident_management_escalation_rules.id'],
    ['incident_management_issuable_escalation_statuses_id_seq', 'incident_management_issuable_escalation_statuses.id'],
    ['incident_management_oncall_participants_id_seq', 'incident_management_oncall_participants.id'],
    ['incident_management_oncall_rotations_id_seq', 'incident_management_oncall_rotations.id'],
    ['incident_management_oncall_schedules_id_seq', 'incident_management_oncall_schedules.id'],
    ['incident_management_oncall_shifts_id_seq', 'incident_management_oncall_shifts.id'],
    ['incident_management_pending_alert_escalations_id_seq', 'incident_management_pending_alert_escalations.id'],
    ['incident_management_pending_issue_escalations_id_seq', 'incident_management_pending_issue_escalations.id'],
    ['incident_management_timeline_event_tag_links_id_seq', 'incident_management_timeline_event_tag_links.id'],
    ['incident_management_timeline_event_tags_id_seq', 'incident_management_timeline_event_tags.id'],
    ['incident_management_timeline_events_id_seq', 'incident_management_timeline_events.id'],
    ['index_statuses_id_seq', 'index_statuses.id'],
    ['insights_id_seq', 'insights.id'],
    ['instance_audit_events_streaming_headers_id_seq', 'instance_audit_events_streaming_headers.id'],
    ['integrations_id_seq', 'integrations.id'],
    ['internal_ids_id_seq', 'internal_ids.id'],
    ['ip_restrictions_id_seq', 'ip_restrictions.id'],
    ['issuable_metric_images_id_seq', 'issuable_metric_images.id'],
    ['issuable_resource_links_id_seq', 'issuable_resource_links.id'],
    ['issuable_severities_id_seq', 'issuable_severities.id'],
    ['issuable_slas_id_seq', 'issuable_slas.id'],
    ['issue_assignment_events_id_seq', 'issue_assignment_events.id'],
    ['issue_customer_relations_contacts_id_seq', 'issue_customer_relations_contacts.id'],
    ['issue_email_participants_id_seq', 'issue_email_participants.id'],
    ['issue_emails_id_seq', 'issue_emails.id'],
    ['issue_links_id_seq', 'issue_links.id'],
    ['issue_metrics_id_seq', 'issue_metrics.id'],
    ['issue_tracker_data_id_seq', 'issue_tracker_data.id'],
    ['issue_user_mentions_id_seq', 'issue_user_mentions.id'],
    ['issues_id_seq', 'issues.id'],
    ['iterations_cadences_id_seq', 'iterations_cadences.id'],
    ['jira_connect_installations_id_seq', 'jira_connect_installations.id'],
    ['jira_connect_subscriptions_id_seq', 'jira_connect_subscriptions.id'],
    ['jira_imports_id_seq', 'jira_imports.id'],
    ['jira_tracker_data_id_seq', 'jira_tracker_data.id'],
    ['keys_id_seq', 'keys.id'],
    ['label_links_id_seq', 'label_links.id'],
    ['label_priorities_id_seq', 'label_priorities.id'],
    ['labels_id_seq', 'labels.id'],
    ['ldap_group_links_id_seq', 'ldap_group_links.id'],
    ['lfs_file_locks_id_seq', 'lfs_file_locks.id'],
    ['lfs_object_states_lfs_object_id_seq', 'lfs_object_states.lfs_object_id'],
    ['lfs_objects_id_seq', 'lfs_objects.id'],
    ['lfs_objects_projects_id_seq', 'lfs_objects_projects.id'],
    ['licenses_id_seq', 'licenses.id'],
    ['list_user_preferences_id_seq', 'list_user_preferences.id'],
    ['lists_id_seq', 'lists.id'],
    ['loose_foreign_keys_deleted_records_id_seq', 'loose_foreign_keys_deleted_records.id'],
    ['member_approvals_id_seq', 'member_approvals.id'],
    ['member_roles_id_seq', 'member_roles.id'],
    ['members_id_seq', 'members.id'],
    ['merge_request_assignees_id_seq', 'merge_request_assignees.id'],
    ['merge_request_assignment_events_id_seq', 'merge_request_assignment_events.id'],
    ['merge_request_blocks_id_seq', 'merge_request_blocks.id'],
    ['merge_request_cleanup_schedules_merge_request_id_seq', 'merge_request_cleanup_schedules.merge_request_id'],
    ['merge_request_context_commits_id_seq', 'merge_request_context_commits.id'],
    ['merge_request_diff_commit_users_id_seq', 'merge_request_diff_commit_users.id'],
    ['merge_request_diff_details_merge_request_diff_id_seq', 'merge_request_diff_details.merge_request_diff_id'],
    ['merge_request_diffs_id_seq', 'merge_request_diffs.id'],
    ['merge_request_metrics_id_seq', 'merge_request_metrics.id'],
    ['merge_request_predictions_merge_request_id_seq', 'merge_request_predictions.merge_request_id'],
    ['merge_request_requested_changes_id_seq', 'merge_request_requested_changes.id'],
    ['merge_request_reviewers_id_seq', 'merge_request_reviewers.id'],
    ['merge_request_user_mentions_id_seq', 'merge_request_user_mentions.id'],
    ['merge_requests_closing_issues_id_seq', 'merge_requests_closing_issues.id'],
    ['merge_requests_compliance_violations_id_seq', 'merge_requests_compliance_violations.id'],
    ['merge_requests_id_seq', 'merge_requests.id'],
    ['merge_trains_id_seq', 'merge_trains.id'],
    ['metrics_dashboard_annotations_id_seq', 'metrics_dashboard_annotations.id'],
    ['metrics_users_starred_dashboards_id_seq', 'metrics_users_starred_dashboards.id'],
    ['milestones_id_seq', 'milestones.id'],
    ['ml_candidate_metadata_id_seq', 'ml_candidate_metadata.id'],
    ['ml_candidate_metrics_id_seq', 'ml_candidate_metrics.id'],
    ['ml_candidate_params_id_seq', 'ml_candidate_params.id'],
    ['ml_candidates_id_seq', 'ml_candidates.id'],
    ['ml_experiment_metadata_id_seq', 'ml_experiment_metadata.id'],
    ['ml_experiments_id_seq', 'ml_experiments.id'],
    ['ml_model_metadata_id_seq', 'ml_model_metadata.id'],
    ['ml_model_version_metadata_id_seq', 'ml_model_version_metadata.id'],
    ['ml_model_versions_id_seq', 'ml_model_versions.id'],
    ['ml_models_id_seq', 'ml_models.id'],
    ['namespace_admin_notes_id_seq', 'namespace_admin_notes.id'],
    ['namespace_bans_id_seq', 'namespace_bans.id'],
    ['namespace_commit_emails_id_seq', 'namespace_commit_emails.id'],
    ['namespace_import_users_id_seq', 'namespace_import_users.id'],
    ['namespace_statistics_id_seq', 'namespace_statistics.id'],
    ['namespaces_id_seq', 'namespaces.id'],
    ['namespaces_storage_limit_exclusions_id_seq', 'namespaces_storage_limit_exclusions.id'],
    ['namespaces_sync_events_id_seq', 'namespaces_sync_events.id'],
    ['note_diff_files_id_seq', 'note_diff_files.id'],
    ['note_metadata_note_id_seq', 'note_metadata.note_id'],
    ['notes_id_seq', 'notes.id'],
    ['notification_settings_id_seq', 'notification_settings.id'],
    ['oauth_access_grants_id_seq', 'oauth_access_grants.id'],
    ['oauth_access_tokens_id_seq', 'oauth_access_tokens.id'],
    ['oauth_applications_id_seq', 'oauth_applications.id'],
    ['oauth_device_grants_id_seq', 'oauth_device_grants.id'],
    ['oauth_openid_requests_id_seq', 'oauth_openid_requests.id'],
    ['onboarding_progresses_id_seq', 'onboarding_progresses.id'],
    ['operations_feature_flag_scopes_id_seq', 'operations_feature_flag_scopes.id'],
    ['operations_feature_flags_clients_id_seq', 'operations_feature_flags_clients.id'],
    ['operations_feature_flags_id_seq', 'operations_feature_flags.id'],
    ['operations_feature_flags_issues_id_seq', 'operations_feature_flags_issues.id'],
    ['operations_scopes_id_seq', 'operations_scopes.id'],
    ['operations_strategies_id_seq', 'operations_strategies.id'],
    ['operations_strategies_user_lists_id_seq', 'operations_strategies_user_lists.id'],
    ['operations_user_lists_id_seq', 'operations_user_lists.id'],
    ['organization_users_id_seq', 'organization_users.id'],
    ['organizations_id_seq', 'organizations.id'],
    ['p_batched_git_ref_updates_deletions_id_seq', 'p_batched_git_ref_updates_deletions.id'],
    ['p_catalog_resource_component_usages_id_seq', 'p_catalog_resource_component_usages.id'],
    ['p_catalog_resource_sync_events_id_seq', 'p_catalog_resource_sync_events.id'],
    ['p_ci_build_tags_id_seq', 'p_ci_build_tags.id'],
    ['p_ci_builds_execution_configs_id_seq', 'p_ci_builds_execution_configs.id'],
    ['p_ci_job_annotations_id_seq', 'p_ci_job_annotations.id'],
    ['packages_build_infos_id_seq', 'packages_build_infos.id'],
    ['packages_composer_cache_files_id_seq', 'packages_composer_cache_files.id'],
    ['packages_conan_file_metadata_id_seq', 'packages_conan_file_metadata.id'],
    ['packages_conan_metadata_id_seq', 'packages_conan_metadata.id'],
    ['packages_debian_group_architectures_id_seq', 'packages_debian_group_architectures.id'],
    ['packages_debian_group_component_files_id_seq', 'packages_debian_group_component_files.id'],
    ['packages_debian_group_components_id_seq', 'packages_debian_group_components.id'],
    ['packages_debian_group_distribution_keys_id_seq', 'packages_debian_group_distribution_keys.id'],
    ['packages_debian_group_distributions_id_seq', 'packages_debian_group_distributions.id'],
    ['packages_debian_project_architectures_id_seq', 'packages_debian_project_architectures.id'],
    ['packages_debian_project_component_files_id_seq', 'packages_debian_project_component_files.id'],
    ['packages_debian_project_components_id_seq', 'packages_debian_project_components.id'],
    ['packages_debian_project_distribution_keys_id_seq', 'packages_debian_project_distribution_keys.id'],
    ['packages_debian_project_distributions_id_seq', 'packages_debian_project_distributions.id'],
    ['packages_debian_publications_id_seq', 'packages_debian_publications.id'],
    ['packages_dependencies_id_seq', 'packages_dependencies.id'],
    ['packages_dependency_links_id_seq', 'packages_dependency_links.id'],
    ['packages_maven_metadata_id_seq', 'packages_maven_metadata.id'],
    ['packages_npm_metadata_caches_id_seq', 'packages_npm_metadata_caches.id'],
    ['packages_nuget_symbols_id_seq', 'packages_nuget_symbols.id'],
    ['packages_package_file_build_infos_id_seq', 'packages_package_file_build_infos.id'],
    ['packages_package_files_id_seq', 'packages_package_files.id'],
    ['packages_packages_id_seq', 'packages_packages.id'],
    ['packages_protection_rules_id_seq', 'packages_protection_rules.id'],
    ['packages_rpm_repository_files_id_seq', 'packages_rpm_repository_files.id'],
    ['packages_tags_id_seq', 'packages_tags.id'],
    ['pages_deployment_states_pages_deployment_id_seq', 'pages_deployment_states.pages_deployment_id'],
    ['pages_deployments_id_seq', 'pages_deployments.id'],
    ['pages_domain_acme_orders_id_seq', 'pages_domain_acme_orders.id'],
    ['pages_domains_id_seq', 'pages_domains.id'],
    ['path_locks_id_seq', 'path_locks.id'],
    ['personal_access_tokens_id_seq', 'personal_access_tokens.id'],
    ['plan_limits_id_seq', 'plan_limits.id'],
    ['plans_id_seq', 'plans.id'],
    ['pm_advisories_id_seq', 'pm_advisories.id'],
    ['pm_affected_packages_id_seq', 'pm_affected_packages.id'],
    ['pm_checkpoints_id_seq', 'pm_checkpoints.id'],
    ['pm_licenses_id_seq', 'pm_licenses.id'],
    ['pm_package_version_licenses_id_seq', 'pm_package_version_licenses.id'],
    ['pm_package_versions_id_seq', 'pm_package_versions.id'],
    ['pm_packages_id_seq', 'pm_packages.id'],
    ['pool_repositories_id_seq', 'pool_repositories.id'],
    ['postgres_async_foreign_key_validations_id_seq', 'postgres_async_foreign_key_validations.id'],
    ['postgres_async_indexes_id_seq', 'postgres_async_indexes.id'],
    ['postgres_reindex_actions_id_seq', 'postgres_reindex_actions.id'],
    ['postgres_reindex_queued_actions_id_seq', 'postgres_reindex_queued_actions.id'],
    ['programming_languages_id_seq', 'programming_languages.id'],
    ['project_aliases_id_seq', 'project_aliases.id'],
    ['project_auto_devops_id_seq', 'project_auto_devops.id'],
    ['project_build_artifacts_size_refreshes_id_seq', 'project_build_artifacts_size_refreshes.id'],
    ['project_ci_cd_settings_id_seq', 'project_ci_cd_settings.id'],
    ['project_ci_feature_usages_id_seq', 'project_ci_feature_usages.id'],
    ['project_compliance_framework_settings_id_seq', 'project_compliance_framework_settings.id'],
    ['project_compliance_framework_settings_project_id_seq', 'project_compliance_framework_settings.project_id'],
    ['project_compliance_standards_adherence_id_seq', 'project_compliance_standards_adherence.id'],
    ['project_custom_attributes_id_seq', 'project_custom_attributes.id'],
    ['project_daily_statistics_id_seq', 'project_daily_statistics.id'],
    ['project_data_transfers_id_seq', 'project_data_transfers.id'],
    ['project_deploy_tokens_id_seq', 'project_deploy_tokens.id'],
    ['project_export_jobs_id_seq', 'project_export_jobs.id'],
    ['project_features_id_seq', 'project_features.id'],
    ['project_group_links_id_seq', 'project_group_links.id'],
    ['project_import_data_id_seq', 'project_import_data.id'],
    ['project_incident_management_settings_project_id_seq', 'project_incident_management_settings.project_id'],
    ['project_mirror_data_id_seq', 'project_mirror_data.id'],
    ['project_relation_export_uploads_id_seq', 'project_relation_export_uploads.id'],
    ['project_relation_exports_id_seq', 'project_relation_exports.id'],
    ['project_repositories_id_seq', 'project_repositories.id'],
    ['project_repository_storage_moves_id_seq', 'project_repository_storage_moves.id'],
    ['project_saved_replies_id_seq', 'project_saved_replies.id'],
    ['project_security_settings_project_id_seq', 'project_security_settings.project_id'],
    ['project_states_id_seq', 'project_states.id'],
    ['project_statistics_id_seq', 'project_statistics.id'],
    ['project_topics_id_seq', 'project_topics.id'],
    ['project_wiki_repositories_id_seq', 'project_wiki_repositories.id'],
    ['projects_id_seq', 'projects.id'],
    ['projects_sync_events_id_seq', 'projects_sync_events.id'],
    ['projects_visits_id_seq', 'projects_visits.id'],
    ['prometheus_alert_events_id_seq', 'prometheus_alert_events.id'],
    ['prometheus_alerts_id_seq', 'prometheus_alerts.id'],
    ['prometheus_metrics_id_seq', 'prometheus_metrics.id'],
    ['protected_branch_merge_access_levels_id_seq', 'protected_branch_merge_access_levels.id'],
    ['protected_branch_push_access_levels_id_seq', 'protected_branch_push_access_levels.id'],
    ['protected_branch_unprotect_access_levels_id_seq', 'protected_branch_unprotect_access_levels.id'],
    ['protected_branches_id_seq', 'protected_branches.id'],
    ['protected_environment_approval_rules_id_seq', 'protected_environment_approval_rules.id'],
    ['protected_environment_deploy_access_levels_id_seq', 'protected_environment_deploy_access_levels.id'],
    ['protected_environments_id_seq', 'protected_environments.id'],
    ['protected_tag_create_access_levels_id_seq', 'protected_tag_create_access_levels.id'],
    ['protected_tags_id_seq', 'protected_tags.id'],
    ['push_rules_id_seq', 'push_rules.id'],
    ['raw_usage_data_id_seq', 'raw_usage_data.id'],
    ['redirect_routes_id_seq', 'redirect_routes.id'],
    ['related_epic_links_id_seq', 'related_epic_links.id'],
    ['relation_import_trackers_id_seq', 'relation_import_trackers.id'],
    ['release_links_id_seq', 'release_links.id'],
    ['releases_id_seq', 'releases.id'],
    ['remote_development_agent_configs_id_seq', 'remote_development_agent_configs.id'],
    ['remote_development_namespace_cluster_agent_mappings_id_seq',
      'remote_development_namespace_cluster_agent_mappings.id'],
    ['remote_mirrors_id_seq', 'remote_mirrors.id'],
    ['required_code_owners_sections_id_seq', 'required_code_owners_sections.id'],
    ['requirements_id_seq', 'requirements.id'],
    ['requirements_management_test_reports_id_seq', 'requirements_management_test_reports.id'],
    ['resource_iteration_events_id_seq', 'resource_iteration_events.id'],
    ['resource_label_events_id_seq', 'resource_label_events.id'],
    ['resource_link_events_id_seq', 'resource_link_events.id'],
    ['resource_milestone_events_id_seq', 'resource_milestone_events.id'],
    ['resource_state_events_id_seq', 'resource_state_events.id'],
    ['resource_weight_events_id_seq', 'resource_weight_events.id'],
    ['reviews_id_seq', 'reviews.id'],
    ['routes_id_seq', 'routes.id'],
    ['saml_group_links_id_seq', 'saml_group_links.id'],
    ['saml_providers_id_seq', 'saml_providers.id'],
    ['saved_replies_id_seq', 'saved_replies.id'],
    ['sbom_component_versions_id_seq', 'sbom_component_versions.id'],
    ['sbom_components_id_seq', 'sbom_components.id'],
    ['sbom_occurrences_id_seq', 'sbom_occurrences.id'],
    ['sbom_occurrences_vulnerabilities_id_seq', 'sbom_occurrences_vulnerabilities.id'],
    ['sbom_source_packages_id_seq', 'sbom_source_packages.id'],
    ['sbom_sources_id_seq', 'sbom_sources.id'],
    ['scan_execution_policy_rules_id_seq', 'scan_execution_policy_rules.id'],
    ['scan_result_policies_id_seq', 'scan_result_policies.id'],
    ['scan_result_policy_violations_id_seq', 'scan_result_policy_violations.id'],
    ['scim_identities_id_seq', 'scim_identities.id'],
    ['scim_oauth_access_tokens_id_seq', 'scim_oauth_access_tokens.id'],
    ['search_indices_id_seq', 'search_indices.id'],
    ['search_namespace_index_assignments_id_seq', 'search_namespace_index_assignments.id'],
    ['security_findings_id_seq', 'security_findings.id'],
    ['security_orchestration_policy_configurations_id_seq', 'security_orchestration_policy_configurations.id'],
    ['security_orchestration_policy_rule_schedules_id_seq', 'security_orchestration_policy_rule_schedules.id'],
    ['security_policies_id_seq', 'security_policies.id'],
    ['security_policy_project_links_id_seq', 'security_policy_project_links.id'],
    ['security_scans_id_seq', 'security_scans.id'],
    ['security_training_providers_id_seq', 'security_training_providers.id'],
    ['security_trainings_id_seq', 'security_trainings.id'],
    ['self_managed_prometheus_alert_events_id_seq', 'self_managed_prometheus_alert_events.id'],
    ['sent_notifications_id_seq', 'sent_notifications.id'],
    ['sentry_issues_id_seq', 'sentry_issues.id'],
    ['service_access_tokens_id_seq', 'service_access_tokens.id'],
    ['shards_id_seq', 'shards.id'],
    ['slack_api_scopes_id_seq', 'slack_api_scopes.id'],
    ['slack_integrations_id_seq', 'slack_integrations.id'],
    ['slack_integrations_scopes_id_seq', 'slack_integrations_scopes.id'],
    ['smartcard_identities_id_seq', 'smartcard_identities.id'],
    ['snippet_repository_storage_moves_id_seq', 'snippet_repository_storage_moves.id'],
    ['snippet_user_mentions_id_seq', 'snippet_user_mentions.id'],
    ['snippets_id_seq', 'snippets.id'],
    ['software_license_policies_id_seq', 'software_license_policies.id'],
    ['software_licenses_id_seq', 'software_licenses.id'],
    ['spam_logs_id_seq', 'spam_logs.id'],
    ['sprints_id_seq', 'sprints.id'],
    ['ssh_signatures_id_seq', 'ssh_signatures.id'],
    ['status_check_responses_id_seq', 'status_check_responses.id'],
    ['status_page_published_incidents_id_seq', 'status_page_published_incidents.id'],
    ['status_page_settings_project_id_seq', 'status_page_settings.project_id'],
    ['subscription_add_on_purchases_id_seq', 'subscription_add_on_purchases.id'],
    ['subscription_add_ons_id_seq', 'subscription_add_ons.id'],
    ['subscription_user_add_on_assignments_id_seq', 'subscription_user_add_on_assignments.id'],
    ['subscriptions_id_seq', 'subscriptions.id'],
    ['suggestions_id_seq', 'suggestions.id'],
    ['system_access_microsoft_applications_id_seq', 'system_access_microsoft_applications.id'],
    ['system_access_microsoft_graph_access_tokens_id_seq', 'system_access_microsoft_graph_access_tokens.id'],
    ['system_note_metadata_id_seq', 'system_note_metadata.id'],
    ['taggings_id_seq', 'taggings.id'],
    ['tags_id_seq', 'tags.id'],
    ['target_branch_rules_id_seq', 'target_branch_rules.id'],
    ['term_agreements_id_seq', 'term_agreements.id'],
    ['terraform_state_versions_id_seq', 'terraform_state_versions.id'],
    ['terraform_states_id_seq', 'terraform_states.id'],
    ['timelog_categories_id_seq', 'timelog_categories.id'],
    ['timelogs_id_seq', 'timelogs.id'],
    ['todos_id_seq', 'todos.id'],
    ['token_with_ivs_id_seq', 'token_with_ivs.id'],
    ['topics_id_seq', 'topics.id'],
    ['trending_projects_id_seq', 'trending_projects.id'],
    ['upcoming_reconciliations_id_seq', 'upcoming_reconciliations.id'],
    ['upload_states_upload_id_seq', 'upload_states.upload_id'],
    ['uploads_id_seq', 'uploads.id'],
    ['user_achievements_id_seq', 'user_achievements.id'],
    ['user_agent_details_id_seq', 'user_agent_details.id'],
    ['user_broadcast_message_dismissals_id_seq', 'user_broadcast_message_dismissals.id'],
    ['user_callouts_id_seq', 'user_callouts.id'],
    ['user_canonical_emails_id_seq', 'user_canonical_emails.id'],
    ['user_custom_attributes_id_seq', 'user_custom_attributes.id'],
    ['user_details_user_id_seq', 'user_details.user_id'],
    ['user_group_callouts_id_seq', 'user_group_callouts.id'],
    ['user_namespace_callouts_id_seq', 'user_namespace_callouts.id'],
    ['user_permission_export_uploads_id_seq', 'user_permission_export_uploads.id'],
    ['user_preferences_id_seq', 'user_preferences.id'],
    ['user_project_callouts_id_seq', 'user_project_callouts.id'],
    ['user_statuses_user_id_seq', 'user_statuses.user_id'],
    ['user_synced_attributes_metadata_id_seq', 'user_synced_attributes_metadata.id'],
    ['users_id_seq', 'users.id'],
    ['users_ops_dashboard_projects_id_seq', 'users_ops_dashboard_projects.id'],
    ['users_star_projects_id_seq', 'users_star_projects.id'],
    ['users_statistics_id_seq', 'users_statistics.id'],
    ['value_stream_dashboard_counts_id_seq', 'value_stream_dashboard_counts.id'],
    ['virtual_registries_packages_maven_registries_id_seq', 'virtual_registries_packages_maven_registries.id'],
    ['virtual_registries_packages_maven_registry_upstreams_id_seq',
      'virtual_registries_packages_maven_registry_upstreams.id'],
    ['virtual_registries_packages_maven_upstreams_id_seq', 'virtual_registries_packages_maven_upstreams.id'],
    ['vs_code_settings_id_seq', 'vs_code_settings.id'],
    ['vulnerabilities_id_seq', 'vulnerabilities.id'],
    ['vulnerability_export_parts_id_seq', 'vulnerability_export_parts.id'],
    ['vulnerability_exports_id_seq', 'vulnerability_exports.id'],
    ['vulnerability_external_issue_links_id_seq', 'vulnerability_external_issue_links.id'],
    ['vulnerability_feedback_id_seq', 'vulnerability_feedback.id'],
    ['vulnerability_finding_evidences_id_seq', 'vulnerability_finding_evidences.id'],
    ['vulnerability_finding_links_id_seq', 'vulnerability_finding_links.id'],
    ['vulnerability_finding_signatures_id_seq', 'vulnerability_finding_signatures.id'],
    ['vulnerability_findings_remediations_id_seq', 'vulnerability_findings_remediations.id'],
    ['vulnerability_flags_id_seq', 'vulnerability_flags.id'],
    ['vulnerability_historical_statistics_id_seq', 'vulnerability_historical_statistics.id'],
    ['vulnerability_identifiers_id_seq', 'vulnerability_identifiers.id'],
    ['vulnerability_issue_links_id_seq', 'vulnerability_issue_links.id'],
    ['vulnerability_merge_request_links_id_seq', 'vulnerability_merge_request_links.id'],
    ['vulnerability_occurrence_identifiers_id_seq', 'vulnerability_occurrence_identifiers.id'],
    ['vulnerability_occurrence_pipelines_id_seq', 'vulnerability_occurrence_pipelines.id'],
    ['vulnerability_occurrences_id_seq', 'vulnerability_occurrences.id'],
    ['vulnerability_reads_id_seq', 'vulnerability_reads.id'],
    ['vulnerability_remediations_id_seq', 'vulnerability_remediations.id'],
    ['vulnerability_scanners_id_seq', 'vulnerability_scanners.id'],
    ['vulnerability_state_transitions_id_seq', 'vulnerability_state_transitions.id'],
    ['vulnerability_statistics_id_seq', 'vulnerability_statistics.id'],
    ['vulnerability_user_mentions_id_seq', 'vulnerability_user_mentions.id'],
    ['web_hook_logs_id_seq', 'web_hook_logs.id'],
    ['web_hooks_id_seq', 'web_hooks.id'],
    ['webauthn_registrations_id_seq', 'webauthn_registrations.id'],
    ['wiki_page_meta_id_seq', 'wiki_page_meta.id'],
    ['wiki_page_slugs_id_seq', 'wiki_page_slugs.id'],
    ['wiki_repository_states_id_seq', 'wiki_repository_states.id'],
    ['work_item_hierarchy_restrictions_id_seq', 'work_item_hierarchy_restrictions.id'],
    ['work_item_parent_links_id_seq', 'work_item_parent_links.id'],
    ['work_item_related_link_restrictions_id_seq', 'work_item_related_link_restrictions.id'],
    ['work_item_types_id_seq', 'work_item_types.id'],
    ['work_item_widget_definitions_id_seq', 'work_item_widget_definitions.id'],
    ['workspace_variables_id_seq', 'workspace_variables.id'],
    ['workspaces_id_seq', 'workspaces.id'],
    ['x509_certificates_id_seq', 'x509_certificates.id'],
    ['x509_commit_signatures_id_seq', 'x509_commit_signatures.id'],
    ['x509_issuers_id_seq', 'x509_issuers.id'],
    ['xray_reports_id_seq', 'xray_reports.id'],
    ['zentao_tracker_data_id_seq', 'zentao_tracker_data.id'],
    ['zoekt_enabled_namespaces_id_seq', 'zoekt_enabled_namespaces.id'],
    ['zoekt_indices_id_seq', 'zoekt_indices.id'],
    ['zoekt_nodes_id_seq', 'zoekt_nodes.id'],
    ['zoekt_replicas_id_seq', 'zoekt_replicas.id'],
    ['zoekt_repositories_id_seq', 'zoekt_repositories.id'],
    ['zoekt_shards_id_seq', 'zoekt_shards.id'],
    ['zoekt_tasks_id_seq', 'zoekt_tasks.id'],
    ['zoom_meetings_id_seq', 'zoom_meetings.id']
  ].to_h

  def up
    # First find only the sequences owned either by ci_builds or p_ci_builds.
    sequences = find_sequences_owned_by_ci_builds.to_a.pluck('seq_name') - ['ci_builds_id_seq']

    sequences.each do |seq_name|
      owner = SEQUENCES[seq_name]

      # Ignore unknown sequences: it's possible that a sequence was
      # assigned the wrong owner due to a bug in `sequences_owned_by`
      # (fixed by https://gitlab.com/gitlab-org/gitlab/-/merge_requests/160528),
      # and so it didn't get dropped automatically when its associated
      # table was dropped: https://gitlab.com/gitlab-org/gitlab/-/issues/474293
      next unless owner

      with_lock_retries do
        execute(<<~SQL.squish)
          ALTER SEQUENCE #{quote_column_name(seq_name)} OWNED BY #{quote_table_name(owner)};
        SQL
      end
    end
  end

  def down; end

  private

  def find_sequences_owned_by_ci_builds
    select_all(<<-SQL)
      SELECT s.relname AS seq_name,
             t.relname AS table_name,
             a.attname AS column_name
      FROM pg_class s
      JOIN pg_depend d ON d.objid = s.oid
      AND d.classid = 'pg_class'::regclass
      AND d.refclassid = 'pg_class'::regclass
      JOIN pg_catalog.pg_class t ON t.oid = d.refobjid
      JOIN pg_catalog.pg_namespace n ON n.oid = t.relnamespace
      JOIN pg_catalog.pg_attribute a ON a.attrelid = t.oid
      AND a.attnum = d.refobjsubid
      WHERE s.relkind = 'S'
        AND (t.relname = 'ci_builds'
             OR t.relname = 'p_ci_builds');
    SQL
  end
end
