// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package forgejo

import (
	"context"
	"fmt"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"

	forgejo_sdk "code.forgejo.org/f3/gof3/v3/forges/forgejo/sdk"
)

type comment struct {
	common

	forgejoComment *forgejo_sdk.Comment
}

var _ f3_tree.ForgeDriverInterface = &comment{}

func newComment() generic.NodeDriverInterface {
	return &comment{}
}

func (o *comment) SetNative(comment any) {
	o.forgejoComment = comment.(*forgejo_sdk.Comment)
}

func (o *comment) GetNativeID() string {
	return fmt.Sprintf("%d", o.forgejoComment.ID)
}

func (o *comment) NewFormat() f3.Interface {
	node := o.GetNode()
	return node.GetTree().(f3_tree.TreeInterface).NewFormat(node.GetKind())
}

func (o *comment) ToFormat() f3.Interface {
	if o.forgejoComment == nil {
		return o.NewFormat()
	}

	return &f3.Comment{
		Common:   f3.NewCommon(o.GetNativeID()),
		PosterID: f3_tree.NewUserReference(o.forgejoComment.Poster.ID),
		Content:  o.forgejoComment.Body,
		Created:  o.forgejoComment.Created,
		Updated:  o.forgejoComment.Updated,
	}
}

func (o *comment) FromFormat(content f3.Interface) {
	comment := content.(*f3.Comment)
	o.forgejoComment = &forgejo_sdk.Comment{
		ID: util.ParseInt(comment.GetID()),
		Poster: &forgejo_sdk.User{
			ID: comment.PosterID.GetIDAsInt(),
		},
		Body:    comment.Content,
		Created: comment.Created,
		Updated: comment.Updated,
	}
}

func (o *comment) Get(ctx context.Context) bool {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	comment, resp, err := o.getClient().GetIssueComment(owner, project, node.GetID().Int64())
	if resp.StatusCode == 404 {
		return false
	}
	if err != nil {
		panic(fmt.Errorf("comment %v %w", o, err))
	}
	o.forgejoComment = comment
	return true
}

func (o *comment) Put(ctx context.Context) id.NodeID {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())
	commentable := f3_tree.GetCommentableID(o.GetNode())

	o.maybeSudoID(ctx, o.forgejoComment.Poster.ID)
	defer o.notSudo()

	comment, _, err := o.getClient().CreateIssueComment(owner, project, commentable, forgejo_sdk.CreateIssueCommentOption{
		Body: o.forgejoComment.Body,
	})
	if err != nil {
		panic(err)
	}
	o.forgejoComment = comment
	return id.NewNodeID(o.GetNativeID())
}

func (o *comment) Patch(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	_, _, err := o.getClient().EditIssueComment(owner, project, node.GetID().Int64(), forgejo_sdk.EditIssueCommentOption{
		Body: o.forgejoComment.Body,
	})
	if err != nil {
		panic(err)
	}
}

func (o *comment) Delete(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	resp, err := o.getClient().DeleteIssueComment(owner, project, node.GetID().Int64())
	if resp.StatusCode != 204 {
		panic(fmt.Errorf("unexpected status code deleting %s %d %v", node.GetID().String(), resp.StatusCode, resp))
	}
	if err != nil {
		panic(err)
	}
}
