#!/usr/bin/env python

"""
Date processing functions.

Copyright (C) 2014, 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from bisect import bisect_left
from datetime import date, datetime, timedelta
from os.path import exists
from pytz import timezone, UnknownTimeZoneError
import re

# iCalendar date and datetime parsing (from DateSupport in MoinSupport).

_date_icalendar_regexp_str = ur'(?P<year>[0-9]{4})(?P<month>[0-9]{2})(?P<day>[0-9]{2})'
date_icalendar_regexp_str = _date_icalendar_regexp_str + '$'

datetime_icalendar_regexp_str = _date_icalendar_regexp_str + \
    ur'(?:' \
    ur'T(?P<hour>[0-2][0-9])(?P<minute>[0-5][0-9])(?P<second>[0-6][0-9])' \
    ur'(?P<utc>Z)?' \
    ur')?$'

_duration_time_icalendar_regexp_str = \
    ur'T' \
    ur'(?:' \
    ur'([0-9]+H)(?:([0-9]+M)([0-9]+S)?)?' \
    ur'|' \
    ur'([0-9]+M)([0-9]+S)?' \
    ur'|' \
    ur'([0-9]+S)' \
    ur')'

duration_icalendar_regexp_str = ur'P' \
    ur'(?:' \
    ur'([0-9]+W)' \
    ur'|' \
    ur'(?:%s)' \
    ur'|' \
    ur'([0-9]+D)(?:%s)?' \
    ur')$' % (_duration_time_icalendar_regexp_str, _duration_time_icalendar_regexp_str)

match_date_icalendar = re.compile(date_icalendar_regexp_str, re.UNICODE).match
match_datetime_icalendar = re.compile(datetime_icalendar_regexp_str, re.UNICODE).match
match_duration_icalendar = re.compile(duration_icalendar_regexp_str, re.UNICODE).match

# Datetime formatting.

def format_datetime(dt):

    "Format 'dt' as an iCalendar-compatible string."

    if not dt:
        return None
    elif isinstance(dt, datetime):
        if dt.tzname() == "UTC":
            return dt.strftime("%Y%m%dT%H%M%SZ")
        else:
            return dt.strftime("%Y%m%dT%H%M%S")
    else:
        return dt.strftime("%Y%m%d")

def format_time(dt):

    "Format the time portion of 'dt' as an iCalendar-compatible string."

    if not dt:
        return None
    elif isinstance(dt, datetime):
        if dt.tzname() == "UTC":
            return dt.strftime("%H%M%SZ")
        else:
            return dt.strftime("%H%M%S")
    else:
        return None

def format_duration(td):

    "Format the timedelta 'td' as an iCalendar-compatible string."

    if not td:
        return None
    else:
        day_portion = td.days and "%dD" % td.days or ""
        time_portion = td.seconds and "T%dS" % td.seconds or ""
        if not day_portion and not time_portion:
            time_portion = "T0S"
        return "P%s%s" % (day_portion, time_portion)

# Parsing of datetime and related information.

def get_datetime(value, attr=None):

    """
    Return a datetime object from the given 'value' in iCalendar format, using
    the 'attr' mapping (if specified) to control the conversion.
    """

    if not value:
        return None

    if len(value) > 9 and (not attr or attr.get("VALUE") in (None, "DATE-TIME")):
        m = match_datetime_icalendar(value)
        if m:
            year, month, day, hour, minute, second = map(m.group, [
                "year", "month", "day", "hour", "minute", "second"
                ])

            if hour and minute and second:
                dt = datetime(
                    int(year), int(month), int(day), int(hour), int(minute), int(second)
                    )

                # Impose the indicated timezone.
                # NOTE: This needs an ambiguity policy for DST changes.

                return to_timezone(dt, m.group("utc") and "UTC" or attr and attr.get("TZID") or None)

        return None

    # Permit dates even if the VALUE is not set to DATE.

    if not attr or attr.get("VALUE") in (None, "DATE"):
        m = match_date_icalendar(value)
        if m:
            year, month, day = map(m.group, ["year", "month", "day"])
            return date(int(year), int(month), int(day))

    return None

def get_duration(value):

    """
    Return a duration for the given 'value' as a timedelta object.
    Where no valid duration is specified, None is returned.
    """

    if not value:
        return None

    m = match_duration_icalendar(value)
    if m:
        weeks, days, hours, minutes, seconds = 0, 0, 0, 0, 0
        for s in m.groups():
            if not s: continue
            if s[-1] == "W": weeks += int(s[:-1])
            elif s[-1] == "D": days += int(s[:-1])
            elif s[-1] == "H": hours += int(s[:-1])
            elif s[-1] == "M": minutes += int(s[:-1])
            elif s[-1] == "S": seconds += int(s[:-1])
        return timedelta(
            int(weeks) * 7 + int(days),
            (int(hours) * 60 + int(minutes)) * 60 + int(seconds)
            )
    else:
        return None

def get_period(value, attr=None):

    """
    Return a tuple of the form (start, end) for the given 'value' in iCalendar
    format, using the 'attr' mapping (if specified) to control the conversion.
    """

    if not value or attr and attr.get("VALUE") and attr.get("VALUE") != "PERIOD":
        return None

    t = value.split("/")
    if len(t) != 2:
        return None

    dtattr = {}
    if attr:
        dtattr.update(attr)
        if dtattr.has_key("VALUE"):
            del dtattr["VALUE"]

    start = get_datetime(t[0], dtattr)
    if t[1].startswith("P"):
        end = start + get_duration(t[1])
    else:
        end = get_datetime(t[1], dtattr)

    return start, end

# Time zone conversions and retrieval.

def ends_on_same_day(dt, end, tzid):

    """
    Return whether 'dt' ends on the same day as 'end', testing the date
    components of 'dt' and 'end' against each other, but also testing whether
    'end' is the actual end of the day in which 'dt' is positioned.

    Since time zone transitions may occur within a day, 'tzid' is required to
    determine the end of the day in which 'dt' is positioned, using the zone
    appropriate at that point in time, not necessarily the zone applying to
    'dt'.
    """

    return (
        to_timezone(dt, tzid).date() == to_timezone(end, tzid).date() or
        end == get_end_of_day(dt, tzid)
        )

def get_default_timezone():

    "Return the system time regime."

    filename = "/etc/timezone"

    if exists(filename):
        f = open(filename)
        try:
            return f.read().strip()
        finally:
            f.close()
    else:
        return None

def get_end_of_day(dt, tzid):

    """
    Get the end of the day in which 'dt' is positioned, using the given 'tzid'
    to obtain a datetime in the appropriate time zone. Where time zone
    transitions occur within a day, the zone of 'dt' may not be the eventual
    zone of the returned object.
    """

    return get_start_of_day(dt + timedelta(1), tzid)

def get_start_of_day(dt, tzid):

    """
    Get the start of the day in which 'dt' is positioned, using the given 'tzid'
    to obtain a datetime in the appropriate time zone. Where time zone
    transitions occur within a day, the zone of 'dt' may not be the eventual
    zone of the returned object.
    """

    start = datetime(dt.year, dt.month, dt.day, 0, 0)
    return to_timezone(start, tzid)

def get_start_of_next_day(dt, tzid):

    """
    Get the start of the day after the day in which 'dt' is positioned. This
    function is intended to extend either dates or datetimes to the end of a
    day for the purpose of generating a missing end date or datetime for an
    event.

    If 'dt' is a date and not a datetime, a plain date object for the next day
    will be returned.

    If 'dt' is a datetime, the given 'tzid' is used to obtain a datetime in the
    appropriate time zone. Where time zone transitions occur within a day, the
    zone of 'dt' may not be the eventual zone of the returned object.
    """

    if isinstance(dt, datetime):
        return get_end_of_day(dt, tzid)
    else:
        return dt + timedelta(1)

def get_datetime_tzid(dt):

    "Return the time zone identifier from 'dt' or None if unknown."

    if not isinstance(dt, datetime):
        return None
    elif dt.tzname() == "UTC":
        return "UTC"
    elif dt.tzinfo and hasattr(dt.tzinfo, "zone"):
        return dt.tzinfo.zone
    else:
        return None

def get_period_tzid(start, end):

    "Return the time zone identifier for 'start' and 'end' or None if unknown."

    if isinstance(start, datetime) or isinstance(end, datetime):
        return get_datetime_tzid(start) or get_datetime_tzid(end)
    else:
        return None

def to_date(dt):

    "Return the date of 'dt'."

    return date(dt.year, dt.month, dt.day)

def to_datetime(dt, tzid):

    """
    Return a datetime for 'dt', using the start of day for dates, and using the
    'tzid' for the conversion.
    """

    if isinstance(dt, datetime):
        return to_timezone(dt, tzid)
    else:
        return get_start_of_day(dt, tzid)

def to_utc_datetime(dt, tzid=None):

    """
    Return a datetime corresponding to 'dt' in the UTC time zone. If 'tzid'
    is specified, dates and floating datetimes are converted to UTC datetimes
    using the time zone information; otherwise, such dates and datetimes remain
    unconverted.
    """

    if not dt:
        return None
    elif get_datetime_tzid(dt):
        return to_timezone(dt, "UTC")
    elif tzid:
        return to_timezone(to_datetime(dt, tzid), "UTC")
    else:
        return dt

def to_timezone(dt, tzid):

    """
    Return a datetime corresponding to 'dt' in the time regime having the given
    'tzid'.
    """

    try:
        tz = tzid and timezone(tzid) or None
    except UnknownTimeZoneError:
        tz = None
    return to_tz(dt, tz)

def to_tz(dt, tz):

    "Return a datetime corresponding to 'dt' employing the pytz.timezone 'tz'."

    if tz is not None and isinstance(dt, datetime):
        if not dt.tzinfo:
            return tz.localize(dt)
        else:
            return dt.astimezone(tz)
    else:
        return dt

# iCalendar-related conversions.

def end_date_from_calendar(dt):

    """
    Change end dates to refer to the actual dates, not the iCalendar "next day"
    dates.
    """

    if not isinstance(dt, datetime):
        return dt - timedelta(1)
    else:
        return dt

def end_date_to_calendar(dt):

    """
    Change end dates to refer to the iCalendar "next day" dates, not the actual
    dates.
    """

    if not isinstance(dt, datetime):
        return dt + timedelta(1)
    else:
        return dt

def get_datetime_attributes(dt, tzid=None):

    """
    Return attributes for the 'dt' date or datetime object with 'tzid'
    indicating the time zone if not otherwise defined.
    """

    if isinstance(dt, datetime):
        attr = {"VALUE" : "DATE-TIME"}
        tzid = get_datetime_tzid(dt) or tzid
        if tzid:
            attr["TZID"] = tzid
        return attr
    else:
        return {"VALUE" : "DATE"}

def get_datetime_item(dt, tzid=None):

    """
    Return an iCalendar-compatible string and attributes for 'dt' using any
    specified 'tzid' to assert a particular time zone if not otherwise defined.
    """

    if not dt:
        return None, None
    if not get_datetime_tzid(dt):
        dt = to_timezone(dt, tzid)
    value = format_datetime(dt)
    attr = get_datetime_attributes(dt, tzid)
    return value, attr

def get_period_attributes(start, end, tzid=None):

    """
    Return attributes for the 'start' and 'end' datetime objects with 'tzid'
    indicating the time zone if not otherwise defined.
    """

    attr = {"VALUE" : "PERIOD"}
    tzid = get_period_tzid(start, end) or tzid
    if tzid:
        attr["TZID"] = tzid
    return attr

def get_period_item(start, end, tzid=None):

    """
    Return an iCalendar-compatible string and attributes for 'start', 'end' and
    'tzid'.
    """

    if start and end:
        attr = get_period_attributes(start, end, tzid)
        start_value = format_datetime(to_timezone(start, attr.get("TZID")))
        end_value = format_datetime(to_timezone(end, attr.get("TZID")))
        return "%s/%s" % (start_value, end_value), attr
    elif start:
        attr = get_datetime_attributes(start, tzid)
        start_value = format_datetime(to_timezone(start, attr.get("TZID")))
        return start_value, attr
    else:
        return None, None

def get_timestamp(offset=None):

    "Return the current time as an iCalendar-compatible string."

    offset = offset or timedelta(0)
    return format_datetime(to_timezone(datetime.utcnow(), "UTC") + offset)

def get_date(offset=None):

    """
    Return the current date, offset by the given timedelta 'offset' if
    specified. The returned date will not be positioned in any time zone.
    """

    offset = offset or timedelta(0)
    return date.today() + offset

def get_time(offset=None):

    """
    Return the current time, offset by the given timedelta 'offset' if
    specified. The returned time will be in the UTC time zone.
    """

    offset = offset or timedelta(0)
    return to_timezone(datetime.utcnow(), "UTC") + offset

def get_tzid(dtstart_attr, dtend_attr):

    """
    Return any time regime details from the given 'dtstart_attr' and
    'dtend_attr' attribute collections.
    """

    return dtstart_attr and dtstart_attr.get("TZID") or dtend_attr and dtend_attr.get("TZID") or None

def get_recurrence_start(recurrenceid):

    """
    Return 'recurrenceid' in a form suitable for comparison with period start
    dates or datetimes. The 'recurrenceid' should be an identifier normalised to
    a UTC datetime or employing a date or floating datetime representation where
    no time zone information was originally provided.
    """

    return get_datetime(recurrenceid)

def get_recurrence_start_point(recurrenceid, tzid):

    """
    Return 'recurrenceid' in a form suitable for comparison with free/busy start
    datetimes, using 'tzid' to convert recurrence identifiers that are dates.
    The 'recurrenceid' should be an identifier normalised to a UTC datetime or
    employing a date or floating datetime representation where no time zone
    information was originally provided.
    """

    return to_utc_datetime(get_datetime(recurrenceid), tzid)

# Time corrections.

class ValidityError(Exception):
    pass

def check_permitted_values(dt, permitted_values):

    "Check the datetime 'dt' against the 'permitted_values' list."

    if not isinstance(dt, datetime):
        raise ValidityError

    hours, minutes, seconds = permitted_values
    errors = []

    if hours and dt.hour not in hours:
        errors.append("hour")
    if minutes and dt.minute not in minutes:
        errors.append("minute")
    if seconds and dt.second not in seconds:
        errors.append("second")

    return errors

def correct_datetime(dt, permitted_values):

    "Correct 'dt' using the given 'permitted_values' details."

    carry, hour, minute, second = correct_value((dt.hour, dt.minute, dt.second), permitted_values)
    return datetime(dt.year, dt.month, dt.day, hour, minute, second, dt.microsecond, dt.tzinfo) + \
           (carry and timedelta(1) or timedelta(0))

def correct_value(value, permitted_values):

    """
    Correct the given (hour, minute, second) tuple 'value' according to the
    'permitted_values' details.
    """

    limits = 23, 59, 59

    corrected = []
    reset = False

    # Find invalid values and reset all following values.

    for v, values, limit in zip(value, permitted_values, limits):
        if reset:
            if values:
                v = values[0]
            else:
                v = 0

        elif values and v not in values:
            reset = True

        corrected.append(v)

    value = corrected
    corrected = []
    carry = 0

    # Find invalid values and update them to the next valid value, updating more
    # significant values if the next valid value is the first in the appropriate
    # series.

    for v, values, limit in zip(value, permitted_values, limits)[::-1]:
        if carry:
            v += 1
            if v > limit:
                if values:
                    v = values[0]
                else:
                    v = 0
                corrected.append(v)
                continue
            else:
                carry = 0

        if values:
            i = bisect_left(values, v)
            if i < len(values):
                v = values[i]
            else:
                v = values[0]
                carry = 1

        corrected.append(v)

    return [carry] + corrected[::-1]

# vim: tabstop=4 expandtab shiftwidth=4
