//! Device type detection and ID extraction.

use crate::sysfs::{self, ReadAttrError};
use crate::{BlockDeviceId, Error};
use nix::sys::stat::{major, minor, stat, SFlag};
use std::ffi::OsString;
use std::fs;
use std::os::unix::ffi::OsStrExt;
use std::path::{Path, PathBuf};

/// Block device types.
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum BlockDeviceType {
    /// MMC/SD/eMMC - identified by CID.
    Mmc,
    /// NVMe - identified by controller serial number.
    Nvme,
}

impl BlockDeviceType {
    /// Return a string representation of the device type.
    pub fn as_str(&self) -> &'static str {
        match self {
            BlockDeviceType::Mmc => "mmc",
            BlockDeviceType::Nvme => "nvme",
        }
    }
}

/// Get the block device ID from a path.
///
/// The path may be a block device or a symlink to one (symlinks are resolved).
pub fn get_block_device_id_by_path(path: &Path) -> Result<BlockDeviceId, Error> {
    // Use stat on the path - doesn't require read permission on the device
    let stat = stat(path).map_err(|e| Error::Io(e.into()))?;

    let file_type = SFlag::from_bits_truncate(stat.st_mode);

    // Verify it's a block device
    if !file_type.contains(SFlag::S_IFBLK) {
        return Err(Error::NotABlockDevice);
    }

    // Extract major and minor device numbers
    let rdev_major = major(stat.st_rdev) as u32;
    let rdev_minor = minor(stat.st_rdev) as u32;

    // Find sysfs path for this device
    let sysfs_path = get_sysfs_block_path(rdev_major, rdev_minor)?;

    // Check if this is a partition and get parent device path
    let sysfs_path = resolve_to_parent_device(&sysfs_path);

    // Detect device type and get ID
    get_device_id(&sysfs_path)
}

/// Get the sysfs block device path from major:minor numbers.
fn get_sysfs_block_path(major: u32, minor: u32) -> Result<PathBuf, Error> {
    let sysfs_dev = format!("/sys/dev/block/{}:{}", major, minor);
    let resolved = fs::canonicalize(&sysfs_dev)?;
    Ok(resolved)
}

/// If the path is a partition, return the parent device path.
fn resolve_to_parent_device(sysfs_path: &Path) -> PathBuf {
    let partition_file = sysfs_path.join("partition");
    if partition_file.exists() {
        // It's a partition - go up to parent
        if let Some(parent) = sysfs_path.parent() {
            return parent.to_path_buf();
        }
    }
    sysfs_path.to_path_buf()
}

/// Detect device type and extract the unique identifier.
fn get_device_id(sysfs_path: &Path) -> Result<BlockDeviceId, Error> {
    let basename = sysfs_path
        .file_name()
        .and_then(|s| s.to_str())
        .unwrap_or("");

    // MMC/SD/eMMC - check for device/cid
    if let Some(id) = try_mmc_id(sysfs_path)? {
        return Ok(BlockDeviceId {
            device_type: BlockDeviceType::Mmc,
            id,
        });
    }

    // NVMe - check device name and read device/serial
    if basename.starts_with("nvme") {
        if let Some(id) = try_nvme_id(sysfs_path)? {
            return Ok(BlockDeviceId {
                device_type: BlockDeviceType::Nvme,
                id,
            });
        }
    }

    // Unsupported device type
    Err(Error::UnsupportedDevice)
}

/// Try to read MMC CID.
fn try_mmc_id(sysfs_path: &Path) -> Result<Option<OsString>, Error> {
    let cid_path = sysfs_path.join("device/cid");
    if !cid_path.exists() {
        return Ok(None);
    }

    match sysfs::read_attr(&cid_path) {
        Ok(id) if id.as_bytes().is_empty() => Err(Error::NoIdentifierAvailable),
        Ok(id) => Ok(Some(id)),
        Err(ReadAttrError::Truncated) => Err(Error::IdentifierTruncated),
        Err(ReadAttrError::Io(e)) => Err(Error::Io(e)),
    }
}

/// Try to read NVMe controller serial.
fn try_nvme_id(sysfs_path: &Path) -> Result<Option<OsString>, Error> {
    // The device symlink points to the controller; serial is there directly
    let serial_path = sysfs_path.join("device/serial");
    if !serial_path.exists() {
        return Ok(None);
    }

    match sysfs::read_attr(&serial_path) {
        Ok(id) if id.as_bytes().is_empty() => Err(Error::NoIdentifierAvailable),
        Ok(id) => Ok(Some(id)),
        Err(ReadAttrError::Truncated) => Err(Error::IdentifierTruncated),
        Err(ReadAttrError::Io(e)) => Err(Error::Io(e)),
    }
}
