//! C-compatible FFI exports.
//!
//! This module provides a C-compatible interface for the block device ID
//! functionality, suitable for use from C/C++ programs via the cdylib output.

use crate::{get_id_by_path, Error};
use libc::{c_char, c_int, size_t};
use std::ffi::CStr;
use std::path::Path;

/// Library version string from Cargo.toml, null-terminated.
const VERSION: &[u8] = concat!(env!("CARGO_PKG_VERSION"), "\0").as_bytes();

/// Return the library version string.
///
/// # Returns
///
/// A pointer to a static, null-terminated version string (e.g., "0.1.0").
/// Never returns NULL.
///
/// # Example
///
/// ```c
/// printf("libblockdeviceid version: %s\n", bdi_version());
/// ```
#[no_mangle]
pub extern "C" fn bdi_version() -> *const c_char {
    VERSION.as_ptr() as *const c_char
}

/// Retrieve the innate unique identifier for a block device.
///
/// # Arguments
///
/// * `path` - Null-terminated path to a block device or symlink to one.
/// * `out_id` - Buffer to receive the null-terminated identifier string.
/// * `out_id_len` - Size of the output buffer in bytes.
///
/// # Returns
///
/// * `0` - Success; out_id contains the null-terminated identifier.
/// * `-ENOTBLK` - Path does not refer to a block device.
/// * `-ENOTSUP` - Block device type not supported (loop, dm, or unrecognised).
/// * `-ENODATA` - Device type recognised but no innate identifier available.
/// * `-EOVERFLOW` - Identifier too large for the provided buffer.
/// * `-EINVAL` - Invalid argument (null pointer or zero-length buffer).
/// * Other negative errno values on system call failures.
///
/// # Example
///
/// ```c
/// char id[129];
/// int ret = bdi_get_id("/dev/nvme0n1", id, sizeof(id));
/// if (ret == 0) {
///     printf("ID: %s\n", id);
/// }
/// ```
///
/// # Safety
///
/// * `path` must be a valid, null-terminated C string.
/// * `out_id` must point to a buffer of at least `out_id_len` bytes.
#[no_mangle]
pub unsafe extern "C" fn bdi_get_id(
    path: *const c_char,
    out_id: *mut c_char,
    out_id_len: size_t,
) -> c_int {
    // Validate arguments
    if path.is_null() || out_id.is_null() || out_id_len == 0 {
        return -(libc::EINVAL as c_int);
    }

    // Convert path to Rust string
    let path_cstr = match CStr::from_ptr(path).to_str() {
        Ok(s) => s,
        Err(_) => return -(libc::EINVAL as c_int),
    };

    let path = Path::new(path_cstr);

    // Get the device ID
    match get_id_by_path(path) {
        Ok(result) => {
            let id_bytes = result.id.as_encoded_bytes();

            // Check if ID fits in buffer (need room for null terminator)
            if id_bytes.len() >= out_id_len {
                return -(libc::EOVERFLOW as c_int);
            }

            // Copy ID to output buffer
            std::ptr::copy_nonoverlapping(id_bytes.as_ptr(), out_id.cast(), id_bytes.len());

            // Null-terminate
            *out_id.add(id_bytes.len()) = 0;

            0 // Success
        }
        Err(e) => match e {
            Error::NotABlockDevice => -(libc::ENOTBLK as c_int),
            Error::UnsupportedDevice => -(libc::ENOTSUP as c_int),
            Error::NoIdentifierAvailable => -(libc::ENODATA as c_int),
            Error::IdentifierTruncated => -(libc::EOVERFLOW as c_int),
            Error::Io(io_err) => {
                if let Some(code) = io_err.raw_os_error() {
                    -(code as c_int)
                } else {
                    -(libc::EIO as c_int)
                }
            }
        },
    }
}
