#!/bin/bash -e

# ====================
#  check arguments
# ====================

USAGE_MESSAGE="usage: $0 network_id network_type model_major_version model_minor_version [input_format_json]"

ARG_NETWORK_ID=$1
ARG_NETWORK_TYPE=$2
ARG_MODEL_MAJOR_VERSION=$3
ARG_MODEL_MINOR_VERSION=$4
ARG_INPUT_FORMAT_FILE_PATH=$5

if [ $# -lt 4 -o $# -gt 5 ] ; then
  echo $USAGE_MESSAGE
  exit
fi

# ====================
#  check environment variables
# ====================

if [ "$PKG_PACKER_OUTPUT_DIR" = "" ] ; then
  PKG_PACKER_OUTPUT_DIR=packer_output
fi

if [ "$PKG_POST_CONV_OUT_DIR" = "" ] ; then
  PKG_POST_CONV_OUT_DIR=post_conv_output
fi

if [ "$PKG_TOOL_DIR" = "" ] ; then
  PKG_TOOL_DIR="../../tool"
fi

# ====================
#  read versions and check restrictions
# ====================

function check_version_string_dec() {
  local version_name=$1
  local version_string=$2
  local version_length=$3

  if [ ${#version_string} != ${version_length} ]; then
    echo "Invalid $version_name"
    exit
  elif ! [[ $version_string =~ ^[0-9]+$ ]]; then
    echo "Invalid $version_string: can contain only [0-9]"
    exit
  fi
}

check_version_string_dec "Network ID"   $ARG_NETWORK_ID   6

check_version_string_dec "Network Type"   $ARG_NETWORK_TYPE   1

check_version_string_dec "Model Major Version"   $ARG_MODEL_MAJOR_VERSION   2

check_version_string_dec "Model Minor Version"   $ARG_MODEL_MINOR_VERSION  2


function check_file_exists() {
  local file_name=$1

  if ! [ -f "$file_name" ]; then
    echo " Unable to find $file_name : No such file"
    exit
  fi
  echo "$file_name File exists!!"
}

if [ ! $ARG_INPUT_FORMAT_FILE_PATH ] ; then
  OPT_INPUT_FORMAT_FILE_PATH=""
else
  check_file_exists $ARG_INPUT_FORMAT_FILE_PATH
  OPT_INPUT_FORMAT_FILE_PATH=$ARG_INPUT_FORMAT_FILE_PATH
fi

# ====================
#  Execute PostConverter
# ====================
pushd $PKG_PACKER_OUTPUT_DIR

# ====================
#  Check if required files exists
# ====================
echo "Check PackerOut Files"
packer_output_file_list=(sdpsA.bin sdpsC.bin cfgA.bin cfgC.bin dnnParams.xml l2staticData.bin manifest.json)
for file_name in  "${packer_output_file_list[@]}"
do
  check_file_exists "./$file_name"
done

echo "Check PackerOut Files Done!!"


java -jar ${PKG_TOOL_DIR}/PostConverter/PostConverter.jar dnnParams.xml $ARG_NETWORK_ID $ARG_NETWORK_TYPE $ARG_MODEL_MAJOR_VERSION $ARG_MODEL_MINOR_VERSION manifest.json $OPT_INPUT_FORMAT_FILE_PATH 

popd

# ====================
#  Make output directory
# ====================
rm -rf $PKG_POST_CONV_OUT_DIR
rm -f $PKG_POST_CONV_OUT_DIR.zip
mkdir $PKG_POST_CONV_OUT_DIR

# ====================
#  Copy files to output directory
# ====================
function copy_file() {
  local file_name=$1

  if ! [ -f "$PKG_PACKER_OUTPUT_DIR/$file_name" ]; then
    echo " Unable to find $file_name : No such file"
    exit
  fi

  cp "$PKG_PACKER_OUTPUT_DIR/$file_name" "$PKG_POST_CONV_OUT_DIR/$file_name"
}  

post_conv_output_file_list=(sdpsA.bin sdpsC.bin cfgA.bin cfgC.bin l2staticData.bin ap_params.bin dnn_params.bin network_info.txt deployManifest.json)
for file_name in  "${post_conv_output_file_list[@]}"
do
  copy_file $file_name
done

# ====================
#  Make zip
# ====================
zip -r $PKG_POST_CONV_OUT_DIR.zip $PKG_POST_CONV_OUT_DIR
echo "$PKG_POST_CONV_OUT_DIR.zip generated!!"
