#!/bin/bash -e

# ====================
#  check arguments
# ====================

USAGE_MESSAGE="usage: $0 parameter_filename"

ARG_PARAM_FILE=$1

if [ $# -ne 1 ] ; then
  echo $USAGE_MESSAGE
  exit
fi

# ====================
#  check environment variables
# ====================

if [ ! $PKG_DIR ] ; then
  PKG_DIR="./package"
fi

if [ "$PKG_PACKAGER_DIR" = "" ] ; then
  PKG_PACKAGER_DIR="."
fi

if [ "$PKG_POST_CONV_OUTPUT_DIR" = "" ] ; then
  PKG_POST_CONV_OUTPUT_DIR=post_conv_output
fi

if [ "$PKG_DEPLOY_MANIFEST_JSON" = "" ] ; then
  PKG_DEPLOY_MANIFEST_JSON=deployManifest.json
fi


# ====================
#  read versions from manifest
# ====================

function get_full_version_str_from_manifest_json() {
  local manifest_dir=$1
  local field_name=$2

  local version_str=`cat ${manifest_dir}/${PKG_DEPLOY_MANIFEST_JSON} | jq "${field_name}" | awk '{if (gsub(/"/,"")) print}' | perl -pe '($_)=/([0-9]+([.][0-9]+)+)/' | awk '{split($0,a,"."); out=sprintf("%02d%02d%02d",a[1],a[2],a[3]); print out}'`

  echo ${version_str}
}

function get_string_from_manifest_json() {
  local manifest_dir=$1
  local field_name=$2

  local string_val=`cat ${manifest_dir}/${PKG_DEPLOY_MANIFEST_JSON} | jq "${field_name}" | awk '{if (gsub(/"/,"")) print}'`

  echo "${string_val}"
}

converter_version=`get_full_version_str_from_manifest_json ${PKG_POST_CONV_OUTPUT_DIR} '."Converter Version"'`
network_id=`get_string_from_manifest_json ${PKG_POST_CONV_OUTPUT_DIR} '."Network ID"'`
model_major_ver=`get_string_from_manifest_json ${PKG_POST_CONV_OUTPUT_DIR} '."Major Version"'`
model_minor_ver=`get_string_from_manifest_json ${PKG_POST_CONV_OUTPUT_DIR} '."Minor Version"'`
network_info_file=`get_string_from_manifest_json ${PKG_POST_CONV_OUTPUT_DIR} '.NetworkInfo.File'`

# ====================
#  check restrictions
# ====================

function check_version_string_hex() {
  local version_name=$1
  local version_string=$2
  local version_length=$3

  if [ ${#version_string} != "${version_length}" ]; then
    echo "Invalid $version_name"
    exit
  elif ! [[ $version_string =~ ^[0-9a-fA-F]+$ ]]; then
    echo "Invalid $version_string: can contain only [0-9]"
    exit
  fi
}

function check_version_string_dec() {
  local version_name=$1
  local version_string=$2
  local version_length=$3

  if [ ${#version_string} != "${version_length}" ]; then
    echo "Invalid $version_name"
    exit
  elif ! [[ $version_string =~ ^[0-9]+$ ]]; then
    echo "Invalid $version_string: can contain only [0-9]"
    exit
  fi
}

echo "[Converter Version]" $converter_version
echo "[Network ID]" $network_id
echo "[Model Major Version]"  $model_major_ver
echo "[Model Minor Version]"  $model_minor_ver

check_version_string_dec "Converter Version"     $converter_version   6
check_version_string_dec "Network ID"            $network_id          6
check_version_string_dec "Model Major Version"   $model_major_ver     2
check_version_string_dec "Model Minor Version"   $model_minor_ver     2

if ! [[ "${network_info_file}" =~ ^[^\;\|\&\`\(\)\$\<\>\*\?\{\}\[!/]+$ ]]; then
    echo "Invalid network info file name: cannot contain '/' or any special characters"
    exit
fi
if ! [ -f "${PKG_POST_CONV_OUTPUT_DIR}/${network_info_file}" ]; then
    echo "Invalid network info file name: no such file"
    exit
fi

# ====================
#  set version
# ====================

OPT_VERSION=${converter_version}${network_id}${model_major_ver}${model_minor_ver}

# ====================
#  execute packager
# ====================
mkdir -p ${PKG_DIR}

if [ ! $PKG_SKIP_NETWORK_CAT ] ; then
  rm -f ${PKG_DIR}/network_*.rpk
fi

${PKG_PACKAGER_DIR}/rpk_packager -v ${OPT_VERSION} -o ${PKG_DIR}/network -r "${PKG_POST_CONV_OUTPUT_DIR}/${network_info_file}" ${ARG_PARAM_FILE}
