/*
    ScratchSRTED.c - part of a pigpio associated library to handle generic 4-pin ultrasonic
    rangefinder devices such as the HC-SR04. 
    Compiles with Joan's SRTED code to make a dynamic library for Scratch usage - though
    it's just C and could be used by anyone wanting a simple SR04 interface
    
    set up with 
    int  ultrasonicInit(int pi, int triggerPin, int echoPin);
    where 
    id = the pi identity previously provided whwn starting the pigpiod connection
    triggerPin is the gpio number of the pin connected to 'Trig'
    echoPin is the gpio number of the pin connected to 'Echo'
    The return value is NULL if the init call fails, or 1 to indicate success.
    
    The sensor will be triggered 10 times per second and the latest reading which
    appears to be a good value will be kept for easy access.
    
    read the most recent value in cm with
    int ultrasonicDistance(int pi);
    
    
    close down the threads etc with
    closeUltrasonic(int pi);
*/

#include <pigpiod_if2.h>

#include "SRTED.h"

int latestDistance = -1;
SRTED_t *sr = NULL;

void cbf(SRTED_data_t r)
{
    if (r.status == SRTE_GOOD) {
        latestDistance = r.range_cms;
    }
}

int ultrasonicInit(int pi, int triggerPin, int echoPin) {
   if (pi >= 0) {
        // only one sensor at a time for now
        if (sr != NULL) return (int)NULL;
        
        sr = SRTED(pi, triggerPin, echoPin, cbf);
        if ( sr == NULL) return (int)NULL;
        SRTED_auto_read(sr, 0.1);
   }
   return 1;
}

void ultrasonicClose(int pi) {
    // don't try to cancel a non-existent thread etc
    if (sr == NULL) return;

    SRTED_cancel(sr);
    sr = NULL;
}

int ultrasonicDistance(int pi) {
    return latestDistance;
}