/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef SRC_TRACE_PROCESSOR_TABLES_ANDROID_TABLES_H_
#define SRC_TRACE_PROCESSOR_TABLES_ANDROID_TABLES_H_

#include "src/trace_processor/tables/macros.h"

namespace perfetto {
namespace trace_processor {
namespace tables {

// Log entries from Android logcat.
//
// NOTE: this table is not sorted by timestamp. This is why we omit the
// sorted flag on the ts column.
//
// @param ts timestamp of log entry.
// @param utid thread writing the log entry {@joinable thread.utid}.
// @param prio priority of the log. 3=DEBUG, 4=INFO, 5=WARN, 6=ERROR.
// @param tag tag of the log entry.
// @param msg content of the log entry.
// @tablegroup Events
#define PERFETTO_TP_ANDROID_LOG_TABLE_DEF(NAME, PARENT, C) \
  NAME(AndroidLogTable, "android_logs")                    \
  PERFETTO_TP_ROOT_TABLE(PARENT, C)                        \
  C(int64_t, ts)                                           \
  C(uint32_t, utid)                                        \
  C(uint32_t, prio)                                        \
  C(base::Optional<StringPool::Id>, tag)                   \
  C(StringPool::Id, msg)

PERFETTO_TP_TABLE(PERFETTO_TP_ANDROID_LOG_TABLE_DEF);

// A table presenting all game modes and interventions
// of games installed on the system.
// This is generated by the game_mode_intervention data-source.
// @param package_name name of the pakcage, e.g. com.google.android.gm.
// @param uid UID processes of this package runs as.
// @param current_mode current game mode the game is running at.
// @param standard_mode_supported bool whether standard mode is supported.
// @param standard_mode_downscale resolution downscaling factor of standard
// mode.
// @param standard_mode_use_angle bool whether ANGLE is used in standard mode.
// @param standard_mode_fps frame rate that the game is throttled at in standard
// mode.
// @param perf_mode_supported bool whether performance mode is supported.
// @param perf_mode_downscale resolution downscaling factor of performance mode.
// @param perf_mode_use_angle bool whether ANGLE is used in performance mode.
// @param perf_mode_fps frame rate that the game is throttled at in performance
// mode.
// @param battery_mode_supported bool whether battery mode is supported.
// @param battery_mode_downscale resolution downscaling factor of battery mode.
// @param battery_mode_use_angle bool whether ANGLE is used in battery mode.
// @param battery_mode_fps frame rate that the game is throttled at in battery
// mode.
#define PERFETTO_TP_ANDROID_GAME_INTERVENTION_LIST_DEF(NAME, PARENT, C)    \
  NAME(AndroidGameInterventionListTable, "android_game_intervention_list") \
  PERFETTO_TP_ROOT_TABLE(PARENT, C)                                        \
  C(StringPool::Id, package_name)                                          \
  C(int64_t, uid)                                                          \
  C(int32_t, current_mode)                                                 \
  C(int32_t, standard_mode_supported)                                      \
  C(base::Optional<double>, standard_mode_downscale)                       \
  C(base::Optional<int32_t>, standard_mode_use_angle)                      \
  C(base::Optional<double>, standard_mode_fps)                             \
  C(int32_t, perf_mode_supported)                                          \
  C(base::Optional<double>, perf_mode_downscale)                           \
  C(base::Optional<int32_t>, perf_mode_use_angle)                          \
  C(base::Optional<double>, perf_mode_fps)                                 \
  C(int32_t, battery_mode_supported)                                       \
  C(base::Optional<double>, battery_mode_downscale)                        \
  C(base::Optional<int32_t>, battery_mode_use_angle)                       \
  C(base::Optional<double>, battery_mode_fps)

PERFETTO_TP_TABLE(PERFETTO_TP_ANDROID_GAME_INTERVENTION_LIST_DEF);

// Dumpsys entries from Android dumpstate.
//
// @param section name of the dumpstate section.
// @param service name of the dumpsys service. Only present when
// dumpstate=="dumpsys", NULL otherwise.
// @param line line-by-line contents of the section/service, one row per line.
// @tablegroup Events
#define PERFETTO_TP_ANDROID_DUMPSTATE_TABLE_DEF(NAME, PARENT, C) \
  NAME(AndroidDumpstateTable, "android_dumpstate")               \
  PERFETTO_TP_ROOT_TABLE(PARENT, C)                              \
  C(base::Optional<StringPool::Id>, section)                     \
  C(base::Optional<StringPool::Id>, service)                     \
  C(StringPool::Id, line)

PERFETTO_TP_TABLE(PERFETTO_TP_ANDROID_DUMPSTATE_TABLE_DEF);

}  // namespace tables
}  // namespace trace_processor
}  // namespace perfetto

#endif  // SRC_TRACE_PROCESSOR_TABLES_ANDROID_TABLES_H_
