/*============================================================================
Copyright (c) 2025 Raspberry Pi
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the copyright holder nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
============================================================================*/

#include <string.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

static char *to_utf (char *txt)
{
    if (!txt) return NULL;
    int len = strlen (txt);
    if (g_utf8_validate (txt, len, NULL)) return txt;
    else return g_convert (txt, len, "UTF-8", "ISO-8859-1", NULL, NULL, NULL);
}

static void show_help (void)
{
    printf ("Usage:\n");
    printf ("  zenoty [OPTION…]\n");
    printf ("\n");
    printf ("Application Options:\n");
    printf ("  --error                                           Display error dialog\n");
    printf ("  --info                                            Display info dialog\n");
    printf ("  --question                                        Display question dialog\n");
    printf ("  --warning                                         Display warning dialog\n");
    printf ("\n");
    printf ("Detail Options:\n");
    printf ("--title=<text>                                      Text to display on taskbar\n");
    printf ("--icon-name=<icon>                                  Name of icon to display on taskbar\n");
    printf ("--text=<text>                                       Text to display in window\n");
    printf ("--ok-label=<text>                                   Text to display on OK button - default is 'OK'\n");
    printf ("--cancel-label=<text>                               Text to display on Cancel button - default is 'Cancel'\n");
    printf ("--width=<value>                                     Approximate maximum width of window in pixels\n");
}

/*----------------------------------------------------------------------------*/
/* Main function                                                              */
/*----------------------------------------------------------------------------*/

int main (int argc, char *argv[])
{
    GtkWidget *msg_dlg, *msg_msg, *msg_btn, *msg_btn2;
    GtkBuilder *builder;
    const char *ok = _("OK");
    const char *cancel = _("Cancel");
    char *text = NULL;
    char *title = NULL;
    char *ok_label = (char *) ok;
    char *cancel_label = (char *) cancel;
    char *icon_name;
    int nbuttons = 1;
    int width = 0;

    setlocale (LC_ALL, "");
    bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    textdomain (GETTEXT_PACKAGE);

    // GTK setup
    gtk_init (&argc, &argv);

    // parse the arguments
    if (argc < 2)
    {
        show_help ();
        exit (0);
    }
    int arg = 1;
    while (arg < argc)
    {
        if (!strcmp ("--question", argv[arg]))
        {
            g_set_prgname ("zenoty-dialog-question");
            nbuttons = 2;
        }
        if (!strcmp ("--error", argv[arg])) g_set_prgname ("zenoty-dialog-error");
        if (!strcmp ("--info", argv[arg])) g_set_prgname ("zenoty-dialog-information");
        if (!strcmp ("--warning", argv[arg])) g_set_prgname ("zenoty-dialog-warning");
        if (!strncmp ("--title=", argv[arg], 8)) title = argv[arg] + 8;
        if (!strncmp ("--text=", argv[arg], 7)) text = argv[arg] + 7;
        if (!strncmp ("--ok-label=", argv[arg], 11)) ok_label = argv[arg] + 11;
        if (!strncmp ("--cancel-label=", argv[arg], 15)) cancel_label = argv[arg] + 15;
        if (!strncmp ("--width=", argv[arg], 8) && sscanf (argv[arg] + 8, "%d", &width) != 1) width = 0;
        if (!strncmp ("--icon-name=", argv[arg], 12))
        {
            icon_name = g_strdup_printf ("zenoty-%s", argv[arg] + 12);
            g_set_prgname (icon_name);
            g_free (icon_name);
        }
        if (!strncmp ("--help", argv[arg], 6))
        {
            show_help ();
            exit (0);
        }
        arg++;
    }

    // create the dialog
    builder = gtk_builder_new_from_file (PACKAGE_DATA_DIR "/ui/zenoty.ui");

    msg_dlg = (GtkWidget *) gtk_builder_get_object (builder, "modal");
    gtk_widget_show (msg_dlg);
    if (title) gtk_window_set_title (GTK_WINDOW (msg_dlg), to_utf (title));

    msg_msg = (GtkWidget *) gtk_builder_get_object (builder, "modal_msg");
    if (text) gtk_label_set_text (GTK_LABEL (msg_msg), g_strcompress (to_utf (text)));
    if (width) gtk_label_set_max_width_chars (GTK_LABEL (msg_msg), width / 10);
    gtk_widget_show (msg_msg);

    msg_btn = (GtkWidget *) gtk_builder_get_object (builder, "modal_ok");
    if (ok_label) gtk_button_set_label (GTK_BUTTON (msg_btn), to_utf (ok_label));
    gtk_widget_show (msg_btn);

    msg_btn2 = (GtkWidget *) gtk_builder_get_object (builder, "modal_cancel");
    if (nbuttons > 1) 
    {
        if (cancel_label) gtk_button_set_label (GTK_BUTTON (msg_btn2), to_utf (cancel_label));
        gtk_widget_show (msg_btn2);
    }
    else gtk_widget_hide (msg_btn2);

    return gtk_dialog_run (GTK_DIALOG (msg_dlg));
}

/* End of file                                                                */
/*----------------------------------------------------------------------------*/
