/* --- BEGIN COPYRIGHT BLOCK ---
 * Copyright (C) 2015  Red Hat
 * see files 'COPYING' and 'COPYING.openssl' for use and warranty
 * information
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Additional permission under GPLv3 section 7:
 * 
 * If you modify this Program, or any covered work, by linking or
 * combining it with OpenSSL, or a modified version of OpenSSL licensed
 * under the OpenSSL license
 * (https://www.openssl.org/source/license.html), the licensors of this
 * Program grant you additional permission to convey the resulting
 * work. Corresponding Source for a non-source form of such a
 * combination shall include the source code for the parts that are
 * licensed under the OpenSSL license as well as that of the covered
 * work.
 * --- END COPYRIGHT BLOCK ---
 */
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>

#include "nspr.h"
#include "plgetopt.h"
#include "plstr.h"
#include "nunc-stans.h"
#include "ns_tls.h"

#include "test-protocol.h"

static FILE *logfp;

static void
setup_logging()
{
    logfp = stdout;
}

static void
shutdown_logging()
{
    logfp = NULL;
}

static void
do_vlogging(int lvl, const char *fmt, va_list varg)
{
    fprintf(logfp, "%d ", lvl);
    vfprintf(logfp, fmt, varg);
}

void
do_logging(int lvl, const char *fmt, ...)
{
    va_list varg;

    va_start(varg, fmt);
    do_vlogging(lvl, fmt, varg);
    va_end(varg);
}

static void
test_shutdown(struct ns_job_t *job)
{
    do_logging(LOG_INFO, "Received shutdown signal\n");

    /* Stop our listener job first. */
    ns_job_done((struct ns_job_t *)ns_job_get_data(job));

    /* Shutdown the thread pool. */
    ns_thrpool_shutdown(ns_job_get_tp(job));

    /* NGK - this might leak, but we can't really finish the job if the event loop is gone! */
    /* Finish the shutdown signal job. */
    ns_job_done(job);
}

int
main (int argc, char **argv)
{
    ns_thrpool_t *tp;
    struct ns_sec_ctx_t *sc = NULL;
    char *progname = argv[0];
    struct ns_thrpool_config ns_config;
#ifdef USE_SSL
    char *secdir, *certname, *prefix;
#endif

    (void)progname; /* TODO */
    setup_logging();
    struct ns_job_t *listen_job = NULL;
    argv++;

    ns_thrpool_config_init(&ns_config);
    if (getenv("MAX_THREADS")) {
        ns_config.max_threads = atoi(getenv("MAX_THREADS"));
    } else {
        ns_config.max_threads = 3;
    }
    ns_config.log_fct = do_vlogging;
    tp = ns_thrpool_new(&ns_config);

#ifdef USE_SSL /* NGK - disabled ssl for testing */
    secdir = *argv++;
    certname = *argv++;
    prefix = NULL;
    sc = ns_tls_init(secdir, prefix, certname, PR_FALSE);
#endif

    /* arguments are listen ports and/or unix domain listen file names */
    for (; argv && *argv; ++argv) {
        PRFileDesc *listenfd = NULL;
        PRNetAddr netaddr;
        int portnum = atoi(*argv);
        PRSocketOptionData prsod = {PR_SockOpt_Nonblocking, {PR_TRUE}};

        if (portnum == 0) { /* unix socket */
            netaddr.local.family = PR_AF_LOCAL;
            PL_strncpyz(netaddr.local.path, *argv, sizeof(netaddr.local.path));
        } else {
            netaddr.inet.family = PR_AF_INET;
            netaddr.inet.port = PR_htons(portnum);
            netaddr.inet.ip = PR_htonl(PR_INADDR_ANY);
        }
        listenfd = PR_OpenTCPSocket(PR_NetAddrFamily(&netaddr));
        PR_SetSocketOption(listenfd, &prsod);
        prsod.option = PR_SockOpt_Reuseaddr;
        PR_SetSocketOption(listenfd, &prsod);
        /*        add_sec_layer(&listenfd, sc);*/
        PR_Bind(listenfd, &netaddr);
        PR_Listen(listenfd, 32);

        /* start the listener job */
        listen_job = test_protocol_init(tp, listenfd, sc);

        /* set up our signal handler for handling shutdown */
        ns_add_signal_job(tp, SIGTERM, NS_JOB_THREAD, test_shutdown, listen_job, NULL);
    }

    if (ns_thrpool_wait(tp)) {
        /* error */
    }

    ns_thrpool_destroy(tp);

/* NGK - disabled SSL for testing
    ns_tls_done(sc);
*/
    do_logging(LOG_INFO, "Shutdown.  Exiting.\n");

    shutdown_logging();
    return (0);
}
