/**
* @license Apache-2.0
*
* Copyright (c) 2020 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var resolve = require( 'path' ).resolve;
var tape = require( 'tape' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var lpad = require( '@stdlib/string/left-pad' );
var repeat = require( '@stdlib/string/repeat' );
var pow = require( '@stdlib/math/base/special/pow' );
var toBinaryString = require( './../../../../float64/base/to-binary-string' );
var tryRequire = require( '@stdlib/utils/try-require' );


// VARIABLES //

var getHighWord = tryRequire( resolve( __dirname, './../lib/native.js' ) );
var opts = {
	'skip': ( getHighWord instanceof Error )
};


// TESTS //

tape( 'main export is a function', opts, function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof getHighWord, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns an integer', opts, function test( t ) {
	var w = getHighWord( pow( 2.0, 53 ) );

	t.equal( typeof w, 'number', 'returns a number' );
	t.equal( w%1, 0, 'returns an integer' );

	t.end();
});

tape( 'the function extracts a higher order word from a floating-point number', opts, function test( t ) {
	var expected;
	var values;
	var high;
	var v;
	var w;
	var i;

	values = [
		5.0,
		pow( 2.0, 53 ),
		1.0e308,
		-1.0e308,
		-3.14,
		1.0e-324,
		4.94e-324,
		1.234567890123456789,
		-4.94e-324,
		6.333333333333333333e-310,
		-0.0,
		0.0,
		100.0,
		1.0/10.0,
		0.625,
		1.0/3.0,
		5.0e-240,
		-5.0e-240,
		10.0,
		15.0,
		-10.0,
		-15.0,
		pow( 2.0, -42 ),
		-pow( 2.0, 100 ),
		1.0,
		-1.0,
		1.5,
		1111111111111.111111111,
		-1111111111111.111111111,
		pow( 2.0, 54 ),
		pow( 2.0, 53 ) + 1.0,
		pow( 2.0, 53 ) + 2.0,
		pow( 2.0, 55 ),
		pow( 2.0, 56 ) - 1.0,
		-pow( 2.0, 57 ) + 5.0,
		3.0*pow( 2.0, 53 ),
		8.0*pow( 2.0, 54 )
	];

	for ( i = 0; i < values.length; i++ ) {
		v = values[ i ];
		w = getHighWord( v );

		// Convert to a binary string:
		high = w.toString( 2 );
		high = lpad( high, 32, '0' );

		expected = toBinaryString( v ).substring( 0, 32 );
		t.equal( high, expected, 'high equals expected bit string for ' + v );
	}
	t.end();
});

tape( 'if provided `NaN`, the sign bit may be either 0 or 1 and the exponent must be all 1s (IEEE 754-1985)', opts, function test( t ) {
	var high;
	var w;

	w = getHighWord( NaN );

	// Convert to a binary string:
	high = w.toString( 2 );
	high = lpad( high, 32, '0' );

	// Sign: +-1.
	t.ok( high[0] === '1' || high[0] === '0', 'sign is either 1 or 0' );

	// Exponent all 1s.
	t.equal( high.substring( 1, 12 ), '11111111111', 'exponent all 1s' );

	t.end();
});

tape( 'if provided `+infinity`, the higher order word corresponds to a sign of 0, all 1s in the exponent, and all 0s in the fraction (IEEE 754-1985)', opts, function test( t ) {
	var high;
	var w;

	high = '0'; // sign
	high += repeat( '1', 11 ); // exponent
	high += repeat( '0', 20 ); // fraction
	high = parseInt( high, 2 );

	w = getHighWord( PINF );
	t.equal( w, high, 'equals high word' );
	t.end();
});

tape( 'if provided `-infinity`, the higher order word corresponds to a sign of 1, all 1s in the exponent, and all 0s in the fraction (IEEE 754-1985)', opts, function test( t ) {
	var high;
	var w;

	high = '1'; // sign
	high += repeat( '1', 11 ); // exponent
	high += repeat( '0', 20 ); // fraction
	high = parseInt( high, 2 );

	w = getHighWord( NINF );
	t.equal( w, high, 'equals high word' );
	t.end();
});
